/*  fft.c
 *
 *  FFT functions of xdemorse-fft application
 */

/*
 *  xdemorse: An application to decode Morse code signals to text
 *
 *  Copyright (C) 2002  Neoklis Kyriazis
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "xdemorse.h"

/* fft in/out buffers */
int
  *fft_in_r,
  *fft_out_r,
  *fft_out_i;

/* Scaled-up, integer, sin/cos tables */
static int
  *isin,
  *icos;

/*------------------------------------------------------------------------*/

/* Ifft_init()
 *
 * Initializes Ifft()
 */
  void
Ifft_init( int fft_size )
{
  int i;
  double w, tpi;

  /* Allocate fft buffers */
  i = sizeof(int) * fft_size;
  mem_alloc( (void *)&fft_in_r, i );
  mem_alloc( (void *)&isin, i );
  mem_alloc( (void *)&icos, i );
  i = sizeof(int) * fft_size / 2;
  mem_alloc( (void *)&fft_out_r, i );
  mem_alloc( (void *)&fft_out_i, i );

  /* Make sin/cos tables */
  tpi = 2.0 * M_PI;
  for( i = 0; i < fft_size; i++ )
  {
	w = tpi * (double)i / (double)fft_size;
	isin[i] = (int)(32.0 * sin(w) + 0.5);
	icos[i] = (int)(32.0 * cos(w) + 0.5);
  }

} /* Ifft_init() */

/*------------------------------------------------------------------------*/

/* Ifft()
 *
 * Simple, integer-only, FFT function
 */
  void
Ifft( int fft_size )
{
  int i, j, w;
  /* In-phase and quadrature summation */
  int sum_i, sum_q;
  int k = fft_size/2;

  /* Calculate output bins */
  for( i = 0; i < k; i++ )
  {
	sum_i = sum_q = w = 0;

	/* Summate input values */
	for( j = 0; j < fft_size; j++ )
	{
	  w += i;
	  if( w >= fft_size )
		w -= fft_size;
	  sum_i += fft_in_r[j] * isin[w];
	  sum_q += fft_in_r[j] * icos[w];
	}

	/* Normalized summations to bins */
	fft_out_r[i] = sum_i/fft_size;
	fft_out_i[i] = sum_q/fft_size;

  } /* for( i = 0; i < k; i++ ) */

} /* Ifft() */

/*------------------------------------------------------------------------*/

gboolean mem_alloc( void **ptr, int req )
{
  *ptr = malloc( req );
  if( *ptr == NULL )
  {
	perror( "xdemorse: Memory allocation request:" );
	Error_Dialog(
		"Memory allocation request failed"
		"Quit xdemorse and correct" );
	return( FALSE );
  }
  bzero( *ptr, req );

  return( TRUE );
} /* End of void mem_alloc() */

/*------------------------------------------------------------------------*/

