/* pathjumpui.hh
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2012-2015 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef PATHJUMPUI_HH
#define PATHJUMPUI_HH

#include "wdefines.h"
#include <memory>
#include <string>
#include <list>
#include <vector>
#include "nwc_virtualdir.hh"
#include "stringmatcher_flexiblematch.hh"
#include <aguix/fieldlistview.h>

class DeepPathStore;
class Button;
class FieldListView;
class Text;
class AGUIX;
class SolidButton;
class AContainer;
class Worker;
class ChooseButton;
class CycleButton;
class Worker;

class PathJumpUI
{
public:
    PathJumpUI( AGUIX &aguix, DeepPathStore &path_store, Worker &worker );
    ~PathJumpUI();

    int mainLoop();
    std::string getSelectedPath();

    void setCurrentDirname( const std::string &dirname );
    void setCurrentBasename( const std::string &basename );

    std::unique_ptr< NWC::Dir > getResultsAsDir();
private:
    AGUIX &m_aguix;
    Worker &m_worker;
    DeepPathStore &m_path_store;

    std::string m_dirname, m_basename;
    std::string m_filter;
    std::string m_selected_path;
    
    std::unique_ptr<class AWindow> m_win;
    FieldListView *m_lv;
    FieldListView *m_class_counts_lv;
    Text *m_infixtext;
    Button *m_hold_status_b;
    Text *m_hold_filter_text;
    Button *m_okb;
    Button *m_cancelb;
    Button *m_cleanupb;
    Button *m_panelize_b;
    Button *m_checkb;
    AContainer *m_co1;
    AContainer *m_hold_co;
    ChooseButton *m_show_all_cb;
    ChooseButton *m_ignore_date_cb;
    CycleButton *m_filter_mode_cyb;
    Button *m_plus1h_b;
    Button *m_plus1d_b;
    Button *m_plus1w_b;
    Button *m_age_reset_b;
    Text *m_age_text;

    time_t m_entry_filter_age;

    typedef enum {
        PATH_HIDE,
        PATH_SHOW,
        BOOKMARK_HIDE,
        BOOKMARK_SHOW,
    } entry_visibility_t;

    struct pathjump_entry {
        entry_visibility_t m_type;
        int m_blockcount;
        std::string m_path;
        bool m_post_filtering;
        bool m_always_ignore;
        time_t m_last_access;

        pathjump_entry() = delete;
        pathjump_entry( entry_visibility_t type,
                        int blockcount,
                        const std::string &path,
                        time_t last_access ) : m_type( type ),
                                               m_blockcount( blockcount ),
                                               m_path( path ),
                                               m_post_filtering( true ),
                                               m_always_ignore( false ),
                                               m_last_access( last_access )
        {}

        static bool compare( const pathjump_entry &lhs,
                             const pathjump_entry &rhs,
                             bool ignore_time )
        {
            if ( lhs.m_blockcount < rhs.m_blockcount ) return true;
            else if ( lhs.m_blockcount == rhs.m_blockcount ) {
                if ( ignore_time ) {
                    if ( lhs.m_path < rhs.m_path ) return true;
                } else {
                    if ( lhs.m_last_access > rhs.m_last_access ) return true;
                    else if ( lhs.m_last_access == rhs.m_last_access ) {
                        if ( lhs.m_path < rhs.m_path ) return true;
                    }
                }
            }

            return false;
        }
        
        bool operator<( const pathjump_entry &rhs ) const
        {
            return compare( *this, rhs, true );
        }
    };

    std::list< pathjump_entry > m_entries;
    AContainer *m_breadcrumb_co;
    std::vector< Button * > m_breadcrumb_buttons;
    int m_current_depth;
    std::vector< std::string > m_current_components;
    std::list< std::string > m_bookmarks;

    bool m_show_hidden_entries;

    typedef enum {
        SHOW_ALL,
        SHOW_ONLY_SUBDIRS,
        SHOW_ONLY_DIRECT_SUBDIRS
    } entry_filter_mode_t;

    entry_filter_mode_t m_entry_filter_mode;

    std::string m_current_entry;

    bool m_lv_dirty;

    std::vector< std::pair< std::string, time_t > > m_entry_fullnames;

    struct hold_status {
        hold_status() : active( false ),
                        base_show_hidden_state( false )
        {}

        bool active;
        std::string current_filter;
        bool base_show_hidden_state;
    } m_hold_status;

    std::list< std::pair< int, int > > m_match_classes;

    static int s_pathjump_number;

    void showData();
    void maximizeWin();

    int apply_filter( const std::string &filter );
    int update_breadcrumb();

    void highlight_best_hit( const std::string &filter );

    int find_best_matching_depth( const std::string &filter );
    int find_best_matching_depth( const std::string &path,
                                  const std::string &filter );
    int build_current_path_components();
    int build_current_path_components( const std::string &path );

    int post_filtering();

    int find_entry( const std::string &name, const std::string &filter );
    int set_entry( const std::string &name );

    int cleanup();

    int resetHeldEntries();
    void resetHeldEntriesAndUpdateUI();
    int holdCurrentEntries( int top_x );
    int holdCurrentEntriesAndUpdateUI( int top_x );
    int calculateMatchClasses();

    int handleMatchClassRow( int row );
    int handleMainLVRowChange( int row );

    void showHoldHelp();

    void updateLVData( int row, int field,
                       struct FieldListView::on_demand_data &data );

    void checkExistance();

    void updateAgeText();
    
    struct {
        StringMatcherFlexibleMatch matcher;
        size_t prefix_filter_length;
    } m_ondemand_info;

    time_t m_current_time;
    bool m_ignore_date_for_sorting;
    time_t m_oldest_time;

    std::string m_previous_apply_filter;
};

#endif
