///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/undo/UndoManager.h>
#include <core/data/ObjectLoadStream.h>
#include <core/data/ObjectSaveStream.h>
#include <core/gui/properties/StringPropertyUI.h>
#include <base/linalg/Tensor.h>
#include "DataChannel.h"
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(DataChannel, RefTarget)
DEFINE_PROPERTY_FIELD(DataChannel, "SerializeData", _serializeData)
SET_PROPERTY_FIELD_LABEL(DataChannel, _serializeData, "Serialize data")

/******************************************************************************
* Serialization constructor.
******************************************************************************/
DataChannel::DataChannel(bool isLoading)
	: RefTarget(isLoading), _perAtomSize(0), _id(UserDataChannel), _dataTypeSize(0),
	_numAtoms(0), _type(QMetaType::Void), _isVisible(true), _componentCount(0), _serializeData(true)
{
	INIT_PROPERTY_FIELD(DataChannel, _serializeData);
	OVITO_ASSERT_MSG(isLoading == true, "DataChannel constructor", "This constructor should not be used to create new data channels.");
}

/******************************************************************************
* Normal constructor.
******************************************************************************/
DataChannel::DataChannel(int dataType, size_t dataTypeSize, size_t componentCount)
	: RefTarget(false), _numAtoms(0), _type(dataType), _dataTypeSize(dataTypeSize),
	_isVisible(true), _id(UserDataChannel), _perAtomSize(dataTypeSize*componentCount),
	_componentCount(componentCount), _serializeData(true)
{
	INIT_PROPERTY_FIELD(DataChannel, _serializeData);
	OVITO_ASSERT(_dataTypeSize > 0);
	OVITO_ASSERT(_componentCount > 0);
	if(componentCount > 1) {
		for(size_t i=1; i<=componentCount; i++)
			_componentNames << QString::number(i);
	}
}

/******************************************************************************
* Special constructor to create a standard data channel.
******************************************************************************/
DataChannel::DataChannel(DataChannelIdentifier which, size_t componentCount)
	: RefTarget(false), _numAtoms(0), _isVisible(true), _id(which), _serializeData(true)
{
	INIT_PROPERTY_FIELD(DataChannel, _serializeData);
	switch(which) {
	case DataChannel::AtomTypeChannel:
	case DataChannel::CNATypeChannel:
	case DataChannel::SelectionChannel:
	case DataChannel::ClusterChannel:
	case DataChannel::CoordinationChannel:
	case DataChannel::AtomIndexChannel:
		_type = qMetaTypeId<int>();
		_dataTypeSize = sizeof(int);
		_componentCount = 1;
		break;
	case DataChannel::PositionChannel:
	case DataChannel::ColorChannel:
	case DataChannel::DisplacementChannel:
	case DataChannel::VelocityChannel:
	case DataChannel::ForceChannel:
		_type = qMetaTypeId<FloatType>();
		_dataTypeSize = sizeof(FloatType);
		_componentCount = 3;
		OVITO_ASSERT(_dataTypeSize * _componentCount == sizeof(Vector3));
		break;
	case DataChannel::PotentialEnergyChannel:
	case DataChannel::KineticEnergyChannel:
	case DataChannel::TotalEnergyChannel:
	case DataChannel::RadiusChannel:
	case DataChannel::MassChannel:
	case DataChannel::TransparencyChannel:
		_type = qMetaTypeId<FloatType>();
		_dataTypeSize = sizeof(FloatType);
		_componentCount = 1;
		break;
	case DataChannel::StressTensorChannel:
	case DataChannel::StrainTensorChannel:
		_type = qMetaTypeId<FloatType>();
		_dataTypeSize = sizeof(FloatType);
		_componentCount = 6;
		OVITO_ASSERT(_dataTypeSize * _componentCount == sizeof(SymmetricTensor2));
		break;
	case DataChannel::DeformationGradientChannel:
		_type = qMetaTypeId<FloatType>();
		_dataTypeSize = sizeof(FloatType);
		_componentCount = 9;
		OVITO_ASSERT(_dataTypeSize * _componentCount == sizeof(Matrix3));
		break;
	case DataChannel::OrientationChannel:
		_type = qMetaTypeId<FloatType>();
		_dataTypeSize = sizeof(FloatType);
		_componentCount = 4;
		OVITO_ASSERT(_dataTypeSize * _componentCount == sizeof(Quaternion));
		break;
	case DataChannel::PeriodicImageChannel:
		_type = qMetaTypeId<int>();
		_dataTypeSize = sizeof(int);
		_componentCount = 3;
		break;
	case DataChannel::BondsChannel:
		_type = qMetaTypeId<int>();
		_dataTypeSize = sizeof(int);
		_componentCount = componentCount;
		componentCount = 0;
		break;
	default:
		OVITO_ASSERT_MSG(false, "DataChannel constructor", "Invalid standard data channel identifier");
		throw Exception(tr("This is not a valid standard data channel identifier: %1").arg(which));
	}
	OVITO_ASSERT_MSG(componentCount == 0, "DataChannel::DataChannel(DataChannelIdentifier)", "Cannot specify component count for a standard data channel with a fixed component count.");

	_perAtomSize = _componentCount * _dataTypeSize;
	_componentNames = standardChannelComponentNames(which, _componentCount);
	_name = standardChannelName(which);
}

/******************************************************************************
* Changes the number of components per atom.
******************************************************************************/
void DataChannel::setComponentCount(size_t count)
{
	_componentCount = count;
	_perAtomSize = _componentCount * _dataTypeSize;
	// Resize component names array.
	if(_id != UserDataChannel) {
		OVITO_ASSERT_MSG(standardChannelComponentCount(_id) == 0, "DataChannel::setComponentCount()", "Cannot change component count of a standard channel with a fixed number of components.");
		_componentNames = standardChannelComponentNames(_id, _componentCount);
	}
	else {
		if(_componentNames.size() > _componentCount)
			_componentNames = _componentNames.mid(0, _componentCount);
		else
			while(_componentNames.size() < _componentCount)
				_componentNames.append(QString());
	}
	// Re-allocate memory.
	resize(size());
}

/******************************************************************************
* Returns the number of AtomsObjects that have a reference to this data channel.
******************************************************************************/
int DataChannel::channelUsageCount() const
{
	int count = 0;
	Q_FOREACH(RefMaker* dependent, getDependents())
		if(dynamic_object_cast<AtomsObject>(dependent))
			count++;
	return count;
}

/******************************************************************************
* Sets the channels's name.
******************************************************************************/
void DataChannel::setName(const QString& name)
{
	if(_name == name) return;

	// Make the property change undoable.
	if(UNDO_MANAGER.isRecording())
		UNDO_MANAGER.addOperation(new SimplePropertyChangeOperation(this, "name"));

	_name = name;
	notifyDependents(REFTARGET_CHANGED);
	notifyDependents(SCHEMATIC_TITLE_CHANGED);
}

/******************************************************************************
* Sets the visibility flag for this data channel.
******************************************************************************/
void DataChannel::setVisible(bool visible)
{
	if(visible == this->_isVisible) return;

	// Make the property change undoable.
	if(UNDO_MANAGER.isRecording())
		UNDO_MANAGER.addOperation(new SimplePropertyChangeOperation(this, "isVisible"));

	this->_isVisible = visible;
	notifyDependents(REFTARGET_CHANGED);
}

/******************************************************************************
* Saves the class' contents to the given stream.
******************************************************************************/
void DataChannel::saveToStream(ObjectSaveStream& stream)
{
	RefTarget::saveToStream(stream);
	stream.beginChunk(0x10000000);
	stream.writeEnum(_type);
	stream << QByteArray(QMetaType::typeName(_type));
	stream.writeSizeT(_dataTypeSize);
	stream.writeSizeT(_perAtomSize);
	if(serializeData())
		stream.writeSizeT(_numAtoms);
	else
		stream.writeSizeT(0);
	stream.writeSizeT(_componentCount);
	stream << _componentNames;
	if(serializeData())
		stream << _dataBuffer;
	else
		stream << QByteArray();
	stream << _name;
	stream << _isVisible;
	stream.writeEnum(_id);
	stream.endChunk();
}

/******************************************************************************
* Loads the class' contents from the given stream.
******************************************************************************/
void DataChannel::loadFromStream(ObjectLoadStream& stream)
{
	RefTarget::loadFromStream(stream);
	stream.expectChunk(0x10000000);
	stream.readEnum(_type);
	QByteArray typeName;
	stream >> typeName;
	_type = QMetaType::type(typeName.constData());
	OVITO_ASSERT_MSG(_type != 0, "DataChannel::loadFromStream", QString("The meta data type '%1' seems to be no longer defined.").arg(QString(typeName)).toAscii().constData());
	OVITO_ASSERT(typeName == QMetaType::typeName(_type));
	stream.readSizeT(_dataTypeSize);
	stream.readSizeT(_perAtomSize);
	stream.readSizeT(_numAtoms);
	stream.readSizeT(_componentCount);
	stream >> _componentNames;
	stream >> _dataBuffer;
	stream >> _name;
	stream >> _isVisible;
	stream.readEnum(_id);
	stream.closeChunk();

	// Do floating-point precision conversion from 32 bit to 64 bit.
	if(_type == qMetaTypeId<float>() && qMetaTypeId<FloatType>() == qMetaTypeId<double>()) {
		OVITO_ASSERT(sizeof(FloatType) == sizeof(double));
		OVITO_ASSERT(_dataTypeSize == sizeof(float));
		_perAtomSize *= sizeof(double) / sizeof(float);
		_dataTypeSize = sizeof(double);
		_type = qMetaTypeId<FloatType>();
		QByteArray newBuffer;
		newBuffer.resize(_perAtomSize * _numAtoms);
		double* dst = (double*)newBuffer.data();
		const float* src = (const float*)_dataBuffer.constData();
		for(size_t c = _numAtoms * _componentCount; c--; )
			*dst++ = (double)*src++;
	}
	// Do floating-point precision conversion from 64 bit to 32 bit.
	if(_type == qMetaTypeId<double>() && qMetaTypeId<FloatType>() == qMetaTypeId<float>()) {
		OVITO_ASSERT(sizeof(FloatType) == sizeof(float));
		OVITO_ASSERT(_dataTypeSize == sizeof(double));
		_perAtomSize /= sizeof(double) / sizeof(float);
		_dataTypeSize = sizeof(float);
		_type = qMetaTypeId<FloatType>();
		QByteArray newBuffer;
		newBuffer.resize(_perAtomSize * _numAtoms);
		float* dst = (float*)newBuffer.data();
		const double* src = (const double*)_dataBuffer.constData();
		for(size_t c = _numAtoms * _componentCount; c--; )
			*dst++ = (float)*src++;
	}

	OVITO_ASSERT(_dataBuffer.size() == _numAtoms * _perAtomSize);
}

/******************************************************************************
* Creates a copy of this object.
******************************************************************************/
RefTarget::SmartPtr DataChannel::clone(bool deepCopy, CloneHelper& cloneHelper)
{
	// Let the base class create an instance of this class.
	DataChannel::SmartPtr clone = static_object_cast<DataChannel>(RefTarget::clone(deepCopy, cloneHelper));

	// Copy channel name.
	clone->_name = this->_name;
	// Copy internal data.
	clone->_type = this->_type;
	clone->_dataTypeSize = this->_dataTypeSize;
	clone->_perAtomSize = this->_perAtomSize;
	clone->_numAtoms = this->_numAtoms;
	clone->_componentCount = this->_componentCount;
	clone->_componentNames = this->_componentNames;
	clone->_dataBuffer = this->_dataBuffer;
	clone->_isVisible = this->_isVisible;
	clone->_id = this->_id;

	return clone;
}

/******************************************************************************
* Resizes the array to the given size.
******************************************************************************/
void DataChannel::resize(size_t newSize)
{
	OVITO_ASSERT(newSize >= 0 && newSize < 2000000000);
	_dataBuffer.resize(newSize * _perAtomSize);

	// Initialize new elements to zero.
	if(newSize > _numAtoms) {
		memset(_dataBuffer.data() + _numAtoms*_perAtomSize, 0, (newSize-_numAtoms)*_perAtomSize);
	}
	_numAtoms = newSize;
}

/******************************************************************************
* Copies the contents from the given source channel into this channel.
* Atoms for which the bit in the given mask is set are skipped.
******************************************************************************/
void DataChannel::filterCopy(DataChannel* source, const dynamic_bitset<>& mask)
{
	OVITO_ASSERT(source->size() == mask.size());
	OVITO_ASSERT(perAtomSize() == source->perAtomSize());
	size_t oldAtomsCount = source->size();

	// Optimize filter operation for the most common channel types.
	if(perAtomSize() == sizeof(FloatType)) {
		// Single float
		const FloatType* src = (const FloatType*)source->constData();
		FloatType* dst = (FloatType*)data();
		for(size_t i = 0; i < oldAtomsCount; ++i, ++src) {
			if(!mask.test(i)) *dst++ = *src;
		}
	}
	else if(perAtomSize() == sizeof(int)) {
		// Single integer
		const int* src = (const int* )source->constData();
		int* dst = (int*)data();
		for(size_t i = 0; i < oldAtomsCount; ++i, ++src) {
			if(!mask.test(i)) *dst++ = *src;
		}
	}
	else if(perAtomSize() == sizeof(Point3)) {
		// Triple float
		const Point3* src = (const Point3*)source->constData();
		Point3* dst = (Point3*)data();
		for(size_t i = 0; i < oldAtomsCount; ++i, ++src) {
			if(!mask.test(i)) *dst++ = *src;
		}
	}
	else {
		// General case:
		const char* src = source->constData();
		char* dst = data();
		for(size_t i = 0; i < oldAtomsCount; i++, src += _perAtomSize) {
			if(!mask.test(i)) {
				memcpy(dst, src, _perAtomSize);
				dst += _perAtomSize;
			}
		}
	}
}

/******************************************************************************
* Returns the name string used by default for the given standard data channel.
******************************************************************************/
QString DataChannel::standardChannelName(DataChannelIdentifier which)
{
	switch(which) {
	case DataChannel::AtomTypeChannel: return tr("Atom Type");
	case DataChannel::SelectionChannel: return tr("Selection");
	case DataChannel::ClusterChannel: return tr("Cluster");
	case DataChannel::CoordinationChannel: return tr("Coordination");
	case DataChannel::PositionChannel: return tr("Position");
	case DataChannel::ColorChannel: return tr("Color");
	case DataChannel::DisplacementChannel: return tr("Displacement");
	case DataChannel::VelocityChannel: return tr("Velocity");
	case DataChannel::PotentialEnergyChannel: return tr("Potential Energy");
	case DataChannel::KineticEnergyChannel: return tr("Kinetic Energy");
	case DataChannel::TotalEnergyChannel: return tr("Total Energy");
	case DataChannel::RadiusChannel: return tr("Radius");
	case DataChannel::CNATypeChannel: return tr("CNA Atom Type");
	case DataChannel::AtomIndexChannel: return tr("Atom Index");
	case DataChannel::StressTensorChannel: return tr("Stress Tensor");
	case DataChannel::StrainTensorChannel: return tr("Strain Tensor");
	case DataChannel::DeformationGradientChannel: return tr("Deformation Gradient");
	case DataChannel::OrientationChannel: return tr("Orientation");
	case DataChannel::ForceChannel: return tr("Force");
	case DataChannel::MassChannel: return tr("Mass");
	case DataChannel::PeriodicImageChannel: return tr("Periodic Image");
	case DataChannel::TransparencyChannel: return tr("Transparency");
	case DataChannel::BondsChannel: return tr("Bonds");
	default:
		OVITO_ASSERT_MSG(false, "DataChannel::standardChannelName", "Invalid standard data channel identifier");
		throw Exception(tr("This is not a valid standard data channel identifier: %1").arg(which));
	}
}

/******************************************************************************
* Returns the data type used by the given standard data channel.
******************************************************************************/
int DataChannel::standardChannelType(DataChannelIdentifier which)
{
	switch(which) {
	case DataChannel::AtomTypeChannel:
	case DataChannel::CNATypeChannel:
	case DataChannel::SelectionChannel:
	case DataChannel::ClusterChannel:
	case DataChannel::CoordinationChannel:
	case DataChannel::AtomIndexChannel:
	case DataChannel::PeriodicImageChannel:
	case DataChannel::BondsChannel:
		return qMetaTypeId<int>();
	case DataChannel::PositionChannel:
	case DataChannel::ColorChannel:
	case DataChannel::DisplacementChannel:
	case DataChannel::VelocityChannel:
	case DataChannel::PotentialEnergyChannel:
	case DataChannel::KineticEnergyChannel:
	case DataChannel::TotalEnergyChannel:
	case DataChannel::RadiusChannel:
	case DataChannel::StressTensorChannel:
	case DataChannel::StrainTensorChannel:
	case DataChannel::DeformationGradientChannel:
	case DataChannel::OrientationChannel:
	case DataChannel::ForceChannel:
	case DataChannel::MassChannel:
	case DataChannel::TransparencyChannel:
		return qMetaTypeId<FloatType>();
	default:
		OVITO_ASSERT_MSG(false, "DataChannel::standardChannelType", "Invalid standard data channel identifier");
		throw Exception(tr("This is not a valid standard data channel identifier: %1").arg(which));
	}
}

/******************************************************************************
* Returns a list with the names and identifiers of all defined standard data channels.
******************************************************************************/
QMap<QString, DataChannel::DataChannelIdentifier> DataChannel::standardChannelList()
{
	static QMap<QString, DataChannelIdentifier> table;
	if(table.empty()) {
		table.insert(standardChannelName(AtomTypeChannel), AtomTypeChannel);
		table.insert(standardChannelName(SelectionChannel), SelectionChannel);
		table.insert(standardChannelName(ClusterChannel), ClusterChannel);
		table.insert(standardChannelName(CoordinationChannel), CoordinationChannel);
		table.insert(standardChannelName(PositionChannel), PositionChannel);
		table.insert(standardChannelName(ColorChannel), ColorChannel);
		table.insert(standardChannelName(DisplacementChannel), DisplacementChannel);
		table.insert(standardChannelName(VelocityChannel), VelocityChannel);
		table.insert(standardChannelName(PotentialEnergyChannel), PotentialEnergyChannel);
		table.insert(standardChannelName(KineticEnergyChannel), KineticEnergyChannel);
		table.insert(standardChannelName(TotalEnergyChannel), TotalEnergyChannel);
		table.insert(standardChannelName(RadiusChannel), RadiusChannel);
		table.insert(standardChannelName(CNATypeChannel), CNATypeChannel);
		table.insert(standardChannelName(AtomIndexChannel), AtomIndexChannel);
		table.insert(standardChannelName(StressTensorChannel), StressTensorChannel);
		table.insert(standardChannelName(StrainTensorChannel), StrainTensorChannel);
		table.insert(standardChannelName(DeformationGradientChannel), DeformationGradientChannel);
		table.insert(standardChannelName(OrientationChannel), OrientationChannel);
		table.insert(standardChannelName(ForceChannel), ForceChannel);
		table.insert(standardChannelName(MassChannel), MassChannel);
		table.insert(standardChannelName(PeriodicImageChannel), PeriodicImageChannel);
		table.insert(standardChannelName(TransparencyChannel), TransparencyChannel);
		table.insert(standardChannelName(BondsChannel), BondsChannel);
	}
	return table;
}

/******************************************************************************
* Returns the number of vector components per atom used by the given standard data channel.
******************************************************************************/
size_t DataChannel::standardChannelComponentCount(DataChannelIdentifier which)
{
	switch(which) {
	case DataChannel::AtomTypeChannel:
	case DataChannel::CNATypeChannel:
	case DataChannel::SelectionChannel:
	case DataChannel::ClusterChannel:
	case DataChannel::CoordinationChannel:
	case DataChannel::AtomIndexChannel:
	case DataChannel::PotentialEnergyChannel:
	case DataChannel::KineticEnergyChannel:
	case DataChannel::TotalEnergyChannel:
	case DataChannel::RadiusChannel:
	case DataChannel::MassChannel:
	case DataChannel::TransparencyChannel:
		return 1;
	case DataChannel::PositionChannel:
	case DataChannel::ColorChannel:
	case DataChannel::DisplacementChannel:
	case DataChannel::VelocityChannel:
	case DataChannel::ForceChannel:
	case DataChannel::PeriodicImageChannel:
		return 3;
	case DataChannel::StressTensorChannel:
	case DataChannel::StrainTensorChannel:
		return 6;
	case DataChannel::DeformationGradientChannel:
		return 9;
	case DataChannel::OrientationChannel:
		return 4;
	// Variable number of components:
	case DataChannel::BondsChannel:
		return 0;
	default:
		OVITO_ASSERT_MSG(false, "DataChannel::standardChannelComponentCount", "Invalid standard data channel identifier");
		throw Exception(tr("This is not a valid standard data channel identifier: %1").arg(which));
	}
}

/******************************************************************************
* Returns the list of component names for the given standard channel.
******************************************************************************/
QStringList DataChannel::standardChannelComponentNames(DataChannelIdentifier which, size_t componentCount)
{
	const static QStringList emptyList;
	const static QStringList xyzList = QStringList() << "X" << "Y" << "Z";
	const static QStringList rgbList = QStringList() << "R" << "G" << "B";
	const static QStringList symmetricTensorList = QStringList() << "XX" << "YY" << "ZZ" << "XY" << "XZ" << "YZ";
	const static QStringList matrix3List = QStringList() << "11" << "21" << "31" << "12" << "22" << "32" << "13" << "23" << "33";
	const static QStringList quaternionList = QStringList() << "X" << "Y" << "Z" << "W";

	switch(which) {
	case DataChannel::AtomTypeChannel:
	case DataChannel::CNATypeChannel:
	case DataChannel::SelectionChannel:
	case DataChannel::ClusterChannel:
	case DataChannel::CoordinationChannel:
	case DataChannel::AtomIndexChannel:
	case DataChannel::PotentialEnergyChannel:
	case DataChannel::KineticEnergyChannel:
	case DataChannel::TotalEnergyChannel:
	case DataChannel::RadiusChannel:
	case DataChannel::MassChannel:
	case DataChannel::TransparencyChannel:
		return emptyList;
	case DataChannel::PositionChannel:
	case DataChannel::DisplacementChannel:
	case DataChannel::VelocityChannel:
	case DataChannel::ForceChannel:
	case DataChannel::PeriodicImageChannel:
		return xyzList;
	case DataChannel::ColorChannel:
		return rgbList;
	case DataChannel::StressTensorChannel:
	case DataChannel::StrainTensorChannel:
		return symmetricTensorList;
	case DataChannel::DeformationGradientChannel:
		return matrix3List;
	case DataChannel::OrientationChannel:
		return quaternionList;
	case DataChannel::BondsChannel:
	{
		QStringList componentNames;
		for(size_t i=1; i<=componentCount; i++)
			componentNames << QString::number(i);
		return componentNames;
	}
	default:
		OVITO_ASSERT_MSG(false, "DataChannel::standardChannelComponentNames", "Invalid standard data channel identifier");
		throw Exception(tr("This is not a valid standard data channel identifier: %1").arg(which));
	}
}

IMPLEMENT_PLUGIN_CLASS(DataChannelEditor, PropertiesEditor)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void DataChannelEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Data channel properties"), rolloutParams);

    // Create the rollout contents.
	QVBoxLayout* layout = new QVBoxLayout(rollout);
	layout->setContentsMargins(4,4,4,4);
	layout->setSpacing(0);

    // Data channel name
    StringPropertyUI* channelNameUI = new StringPropertyUI(this, "name");
    layout->addWidget(new QLabel(tr("Channel Name:")));
    channelNameUI->setEnabled(false);	// This is a read-only property
    layout->addWidget(channelNameUI->textBox());
}

};	// End of namespace AtomViz
