/*
 *  Copyright (c) 2007-2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ModulesManager.h"

#include "Debug.h"
#include "Module.h"

// LLVM
#include <llvm/Support/MutexGuard.h>
#ifdef LLVM_27_OR_28
#include <llvm/System/Path.h>
#include <llvm/System/Mutex.h>
#else
#include <llvm/Support/Path.h>
#include <llvm/Support/Mutex.h>
#endif

#include <GTLCore/Macros_p.h>

using namespace OpenCTL;

#include <map>
#include <list>

struct ModulesManager::Private {
  std::map<GTLCore::String, Module*> modules;
  std::list<GTLCore::String> directories;
  llvm::sys::Mutex mutex;
};

STATIC_POINTER(ModulesManager, s_instance);

ModulesManager::ModulesManager() : d(new Private) {
  addDirectory(_OPENCTL_CTL_STD_LIB_SRC_DIR_);
  addDirectory(_OPENCTL_CTL_SHARE_DIR_);
}

ModulesManager::~ModulesManager()
{
  for(std::map<GTLCore::String, Module*>::iterator it = d->modules.begin();
      it != d->modules.end(); ++it)
  {
    delete it->second;
  }
  delete d;
}

Module* ModulesManager::module(const GTLCore::String& name)
{
  std::map<GTLCore::String, Module*>::iterator it = d->modules.find(name);
  if( it == d->modules.end())
  {
    return 0;
  }
  return it->second;
}

Module* ModulesManager::loadModule(const GTLCore::String& name)
{
  llvm::MutexGuard mg(d->mutex);
  Module* m = module( name );
  if( m )
  { // The module is already loaded and in the registry, return it
    return m;
  }
  GTLCore::String sourceName = name + ".ctl";
  for( std::list<GTLCore::String>::iterator it = d->directories.begin();
       it != d->directories.end(); ++it )
  {
    llvm::sys::Path path( it->c_str() );
    path.appendComponent( (const std::string&)sourceName);
    OCTL_DEBUG("try " << path.c_str() );
    if(path.exists() and path.canRead())
    {
      m = new Module();
      registerModule( name, m );
      m->loadFromFile( path.c_str());
      break;
    }
  }
  
  return m;
}

void ModulesManager::registerModule(const GTLCore::String& name, Module* m )
{
  llvm::MutexGuard mg(d->mutex);
  d->modules[ name ] = m;
}

void ModulesManager::addDirectory(const GTLCore::String& directory)
{
  d->directories.push_back( directory );
}

ModulesManager* ModulesManager::instance()
{
  if(not s_instance)
  {
    s_instance = new ModulesManager();
  }
  return s_instance;
}
