/*
 * Windows 936 codepage (Chinese Simplified) functions
 *
 * Copyright (C) 2008-2020, Joachim Metz <joachim.metz@gmail.com>
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <common.h>
#include <types.h>

#include "libuna_codepage_windows_936.h"
#include "libuna_libcerror.h"
#include "libuna_types.h"

/* Extended ASCII to Unicode character lookup table for Windows 936 codepage
 * Unknown are filled with the Unicode replacement character 0xfffd
 */
const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8140[ 192 ] = {
	0x4e02, 0x4e04, 0x4e05, 0x4e06, 0x4e0f, 0x4e12, 0x4e17, 0x4e1f,
	0x4e20, 0x4e21, 0x4e23, 0x4e26, 0x4e29, 0x4e2e, 0x4e2f, 0x4e31,
	0x4e33, 0x4e35, 0x4e37, 0x4e3c, 0x4e40, 0x4e41, 0x4e42, 0x4e44,
	0x4e46, 0x4e4a, 0x4e51, 0x4e55, 0x4e57, 0x4e5a, 0x4e5b, 0x4e62,
	0x4e63, 0x4e64, 0x4e65, 0x4e67, 0x4e68, 0x4e6a, 0x4e6b, 0x4e6c,
	0x4e6d, 0x4e6e, 0x4e6f, 0x4e72, 0x4e74, 0x4e75, 0x4e76, 0x4e77,
	0x4e78, 0x4e79, 0x4e7a, 0x4e7b, 0x4e7c, 0x4e7d, 0x4e7f, 0x4e80,
	0x4e81, 0x4e82, 0x4e83, 0x4e84, 0x4e85, 0x4e87, 0x4e8a, 0xfffd,
	0x4e90, 0x4e96, 0x4e97, 0x4e99, 0x4e9c, 0x4e9d, 0x4e9e, 0x4ea3,
	0x4eaa, 0x4eaf, 0x4eb0, 0x4eb1, 0x4eb4, 0x4eb6, 0x4eb7, 0x4eb8,
	0x4eb9, 0x4ebc, 0x4ebd, 0x4ebe, 0x4ec8, 0x4ecc, 0x4ecf, 0x4ed0,
	0x4ed2, 0x4eda, 0x4edb, 0x4edc, 0x4ee0, 0x4ee2, 0x4ee6, 0x4ee7,
	0x4ee9, 0x4eed, 0x4eee, 0x4eef, 0x4ef1, 0x4ef4, 0x4ef8, 0x4ef9,
	0x4efa, 0x4efc, 0x4efe, 0x4f00, 0x4f02, 0x4f03, 0x4f04, 0x4f05,
	0x4f06, 0x4f07, 0x4f08, 0x4f0b, 0x4f0c, 0x4f12, 0x4f13, 0x4f14,
	0x4f15, 0x4f16, 0x4f1c, 0x4f1d, 0x4f21, 0x4f23, 0x4f28, 0x4f29,
	0x4f2c, 0x4f2d, 0x4f2e, 0x4f31, 0x4f33, 0x4f35, 0x4f37, 0x4f39,
	0x4f3b, 0x4f3e, 0x4f3f, 0x4f40, 0x4f41, 0x4f42, 0x4f44, 0x4f45,
	0x4f47, 0x4f48, 0x4f49, 0x4f4a, 0x4f4b, 0x4f4c, 0x4f52, 0x4f54,
	0x4f56, 0x4f61, 0x4f62, 0x4f66, 0x4f68, 0x4f6a, 0x4f6b, 0x4f6d,
	0x4f6e, 0x4f71, 0x4f72, 0x4f75, 0x4f77, 0x4f78, 0x4f79, 0x4f7a,
	0x4f7d, 0x4f80, 0x4f81, 0x4f82, 0x4f85, 0x4f86, 0x4f87, 0x4f8a,
	0x4f8c, 0x4f8e, 0x4f90, 0x4f92, 0x4f93, 0x4f95, 0x4f96, 0x4f98,
	0x4f99, 0x4f9a, 0x4f9c, 0x4f9e, 0x4f9f, 0x4fa1, 0x4fa2, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8240[ 192 ] = {
	0x4fa4, 0x4fab, 0x4fad, 0x4fb0, 0x4fb1, 0x4fb2, 0x4fb3, 0x4fb4,
	0x4fb6, 0x4fb7, 0x4fb8, 0x4fb9, 0x4fba, 0x4fbb, 0x4fbc, 0x4fbd,
	0x4fbe, 0x4fc0, 0x4fc1, 0x4fc2, 0x4fc6, 0x4fc7, 0x4fc8, 0x4fc9,
	0x4fcb, 0x4fcc, 0x4fcd, 0x4fd2, 0x4fd3, 0x4fd4, 0x4fd5, 0x4fd6,
	0x4fd9, 0x4fdb, 0x4fe0, 0x4fe2, 0x4fe4, 0x4fe5, 0x4fe7, 0x4feb,
	0x4fec, 0x4ff0, 0x4ff2, 0x4ff4, 0x4ff5, 0x4ff6, 0x4ff7, 0x4ff9,
	0x4ffb, 0x4ffc, 0x4ffd, 0x4fff, 0x5000, 0x5001, 0x5002, 0x5003,
	0x5004, 0x5005, 0x5006, 0x5007, 0x5008, 0x5009, 0x500a, 0xfffd,
	0x500b, 0x500e, 0x5010, 0x5011, 0x5013, 0x5015, 0x5016, 0x5017,
	0x501b, 0x501d, 0x501e, 0x5020, 0x5022, 0x5023, 0x5024, 0x5027,
	0x502b, 0x502f, 0x5030, 0x5031, 0x5032, 0x5033, 0x5034, 0x5035,
	0x5036, 0x5037, 0x5038, 0x5039, 0x503b, 0x503d, 0x503f, 0x5040,
	0x5041, 0x5042, 0x5044, 0x5045, 0x5046, 0x5049, 0x504a, 0x504b,
	0x504d, 0x5050, 0x5051, 0x5052, 0x5053, 0x5054, 0x5056, 0x5057,
	0x5058, 0x5059, 0x505b, 0x505d, 0x505e, 0x505f, 0x5060, 0x5061,
	0x5062, 0x5063, 0x5064, 0x5066, 0x5067, 0x5068, 0x5069, 0x506a,
	0x506b, 0x506d, 0x506e, 0x506f, 0x5070, 0x5071, 0x5072, 0x5073,
	0x5074, 0x5075, 0x5078, 0x5079, 0x507a, 0x507c, 0x507d, 0x5081,
	0x5082, 0x5083, 0x5084, 0x5086, 0x5087, 0x5089, 0x508a, 0x508b,
	0x508c, 0x508e, 0x508f, 0x5090, 0x5091, 0x5092, 0x5093, 0x5094,
	0x5095, 0x5096, 0x5097, 0x5098, 0x5099, 0x509a, 0x509b, 0x509c,
	0x509d, 0x509e, 0x509f, 0x50a0, 0x50a1, 0x50a2, 0x50a4, 0x50a6,
	0x50aa, 0x50ab, 0x50ad, 0x50ae, 0x50af, 0x50b0, 0x50b1, 0x50b3,
	0x50b4, 0x50b5, 0x50b6, 0x50b7, 0x50b8, 0x50b9, 0x50bc, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8340[ 192 ] = {
	0x50bd, 0x50be, 0x50bf, 0x50c0, 0x50c1, 0x50c2, 0x50c3, 0x50c4,
	0x50c5, 0x50c6, 0x50c7, 0x50c8, 0x50c9, 0x50ca, 0x50cb, 0x50cc,
	0x50cd, 0x50ce, 0x50d0, 0x50d1, 0x50d2, 0x50d3, 0x50d4, 0x50d5,
	0x50d7, 0x50d8, 0x50d9, 0x50db, 0x50dc, 0x50dd, 0x50de, 0x50df,
	0x50e0, 0x50e1, 0x50e2, 0x50e3, 0x50e4, 0x50e5, 0x50e8, 0x50e9,
	0x50ea, 0x50eb, 0x50ef, 0x50f0, 0x50f1, 0x50f2, 0x50f4, 0x50f6,
	0x50f7, 0x50f8, 0x50f9, 0x50fa, 0x50fc, 0x50fd, 0x50fe, 0x50ff,
	0x5100, 0x5101, 0x5102, 0x5103, 0x5104, 0x5105, 0x5108, 0xfffd,
	0x5109, 0x510a, 0x510c, 0x510d, 0x510e, 0x510f, 0x5110, 0x5111,
	0x5113, 0x5114, 0x5115, 0x5116, 0x5117, 0x5118, 0x5119, 0x511a,
	0x511b, 0x511c, 0x511d, 0x511e, 0x511f, 0x5120, 0x5122, 0x5123,
	0x5124, 0x5125, 0x5126, 0x5127, 0x5128, 0x5129, 0x512a, 0x512b,
	0x512c, 0x512d, 0x512e, 0x512f, 0x5130, 0x5131, 0x5132, 0x5133,
	0x5134, 0x5135, 0x5136, 0x5137, 0x5138, 0x5139, 0x513a, 0x513b,
	0x513c, 0x513d, 0x513e, 0x5142, 0x5147, 0x514a, 0x514c, 0x514e,
	0x514f, 0x5150, 0x5152, 0x5153, 0x5157, 0x5158, 0x5159, 0x515b,
	0x515d, 0x515e, 0x515f, 0x5160, 0x5161, 0x5163, 0x5164, 0x5166,
	0x5167, 0x5169, 0x516a, 0x516f, 0x5172, 0x517a, 0x517e, 0x517f,
	0x5183, 0x5184, 0x5186, 0x5187, 0x518a, 0x518b, 0x518e, 0x518f,
	0x5190, 0x5191, 0x5193, 0x5194, 0x5198, 0x519a, 0x519d, 0x519e,
	0x519f, 0x51a1, 0x51a3, 0x51a6, 0x51a7, 0x51a8, 0x51a9, 0x51aa,
	0x51ad, 0x51ae, 0x51b4, 0x51b8, 0x51b9, 0x51ba, 0x51be, 0x51bf,
	0x51c1, 0x51c2, 0x51c3, 0x51c5, 0x51c8, 0x51ca, 0x51cd, 0x51ce,
	0x51d0, 0x51d2, 0x51d3, 0x51d4, 0x51d5, 0x51d6, 0x51d7, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8440[ 192 ] = {
	0x51d8, 0x51d9, 0x51da, 0x51dc, 0x51de, 0x51df, 0x51e2, 0x51e3,
	0x51e5, 0x51e6, 0x51e7, 0x51e8, 0x51e9, 0x51ea, 0x51ec, 0x51ee,
	0x51f1, 0x51f2, 0x51f4, 0x51f7, 0x51fe, 0x5204, 0x5205, 0x5209,
	0x520b, 0x520c, 0x520f, 0x5210, 0x5213, 0x5214, 0x5215, 0x521c,
	0x521e, 0x521f, 0x5221, 0x5222, 0x5223, 0x5225, 0x5226, 0x5227,
	0x522a, 0x522c, 0x522f, 0x5231, 0x5232, 0x5234, 0x5235, 0x523c,
	0x523e, 0x5244, 0x5245, 0x5246, 0x5247, 0x5248, 0x5249, 0x524b,
	0x524e, 0x524f, 0x5252, 0x5253, 0x5255, 0x5257, 0x5258, 0xfffd,
	0x5259, 0x525a, 0x525b, 0x525d, 0x525f, 0x5260, 0x5262, 0x5263,
	0x5264, 0x5266, 0x5268, 0x526b, 0x526c, 0x526d, 0x526e, 0x5270,
	0x5271, 0x5273, 0x5274, 0x5275, 0x5276, 0x5277, 0x5278, 0x5279,
	0x527a, 0x527b, 0x527c, 0x527e, 0x5280, 0x5283, 0x5284, 0x5285,
	0x5286, 0x5287, 0x5289, 0x528a, 0x528b, 0x528c, 0x528d, 0x528e,
	0x528f, 0x5291, 0x5292, 0x5294, 0x5295, 0x5296, 0x5297, 0x5298,
	0x5299, 0x529a, 0x529c, 0x52a4, 0x52a5, 0x52a6, 0x52a7, 0x52ae,
	0x52af, 0x52b0, 0x52b4, 0x52b5, 0x52b6, 0x52b7, 0x52b8, 0x52b9,
	0x52ba, 0x52bb, 0x52bc, 0x52bd, 0x52c0, 0x52c1, 0x52c2, 0x52c4,
	0x52c5, 0x52c6, 0x52c8, 0x52ca, 0x52cc, 0x52cd, 0x52ce, 0x52cf,
	0x52d1, 0x52d3, 0x52d4, 0x52d5, 0x52d7, 0x52d9, 0x52da, 0x52db,
	0x52dc, 0x52dd, 0x52de, 0x52e0, 0x52e1, 0x52e2, 0x52e3, 0x52e5,
	0x52e6, 0x52e7, 0x52e8, 0x52e9, 0x52ea, 0x52eb, 0x52ec, 0x52ed,
	0x52ee, 0x52ef, 0x52f1, 0x52f2, 0x52f3, 0x52f4, 0x52f5, 0x52f6,
	0x52f7, 0x52f8, 0x52fb, 0x52fc, 0x52fd, 0x5301, 0x5302, 0x5303,
	0x5304, 0x5307, 0x5309, 0x530a, 0x530b, 0x530c, 0x530e, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8540[ 192 ] = {
	0x5311, 0x5312, 0x5313, 0x5314, 0x5318, 0x531b, 0x531c, 0x531e,
	0x531f, 0x5322, 0x5324, 0x5325, 0x5327, 0x5328, 0x5329, 0x532b,
	0x532c, 0x532d, 0x532f, 0x5330, 0x5331, 0x5332, 0x5333, 0x5334,
	0x5335, 0x5336, 0x5337, 0x5338, 0x533c, 0x533d, 0x5340, 0x5342,
	0x5344, 0x5346, 0x534b, 0x534c, 0x534d, 0x5350, 0x5354, 0x5358,
	0x5359, 0x535b, 0x535d, 0x5365, 0x5368, 0x536a, 0x536c, 0x536d,
	0x5372, 0x5376, 0x5379, 0x537b, 0x537c, 0x537d, 0x537e, 0x5380,
	0x5381, 0x5383, 0x5387, 0x5388, 0x538a, 0x538e, 0x538f, 0xfffd,
	0x5390, 0x5391, 0x5392, 0x5393, 0x5394, 0x5396, 0x5397, 0x5399,
	0x539b, 0x539c, 0x539e, 0x53a0, 0x53a1, 0x53a4, 0x53a7, 0x53aa,
	0x53ab, 0x53ac, 0x53ad, 0x53af, 0x53b0, 0x53b1, 0x53b2, 0x53b3,
	0x53b4, 0x53b5, 0x53b7, 0x53b8, 0x53b9, 0x53ba, 0x53bc, 0x53bd,
	0x53be, 0x53c0, 0x53c3, 0x53c4, 0x53c5, 0x53c6, 0x53c7, 0x53ce,
	0x53cf, 0x53d0, 0x53d2, 0x53d3, 0x53d5, 0x53da, 0x53dc, 0x53dd,
	0x53de, 0x53e1, 0x53e2, 0x53e7, 0x53f4, 0x53fa, 0x53fe, 0x53ff,
	0x5400, 0x5402, 0x5405, 0x5407, 0x540b, 0x5414, 0x5418, 0x5419,
	0x541a, 0x541c, 0x5422, 0x5424, 0x5425, 0x542a, 0x5430, 0x5433,
	0x5436, 0x5437, 0x543a, 0x543d, 0x543f, 0x5441, 0x5442, 0x5444,
	0x5445, 0x5447, 0x5449, 0x544c, 0x544d, 0x544e, 0x544f, 0x5451,
	0x545a, 0x545d, 0x545e, 0x545f, 0x5460, 0x5461, 0x5463, 0x5465,
	0x5467, 0x5469, 0x546a, 0x546b, 0x546c, 0x546d, 0x546e, 0x546f,
	0x5470, 0x5474, 0x5479, 0x547a, 0x547e, 0x547f, 0x5481, 0x5483,
	0x5485, 0x5487, 0x5488, 0x5489, 0x548a, 0x548d, 0x5491, 0x5493,
	0x5497, 0x5498, 0x549c, 0x549e, 0x549f, 0x54a0, 0x54a1, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8640[ 192 ] = {
	0x54a2, 0x54a5, 0x54ae, 0x54b0, 0x54b2, 0x54b5, 0x54b6, 0x54b7,
	0x54b9, 0x54ba, 0x54bc, 0x54be, 0x54c3, 0x54c5, 0x54ca, 0x54cb,
	0x54d6, 0x54d8, 0x54db, 0x54e0, 0x54e1, 0x54e2, 0x54e3, 0x54e4,
	0x54eb, 0x54ec, 0x54ef, 0x54f0, 0x54f1, 0x54f4, 0x54f5, 0x54f6,
	0x54f7, 0x54f8, 0x54f9, 0x54fb, 0x54fe, 0x5500, 0x5502, 0x5503,
	0x5504, 0x5505, 0x5508, 0x550a, 0x550b, 0x550c, 0x550d, 0x550e,
	0x5512, 0x5513, 0x5515, 0x5516, 0x5517, 0x5518, 0x5519, 0x551a,
	0x551c, 0x551d, 0x551e, 0x551f, 0x5521, 0x5525, 0x5526, 0xfffd,
	0x5528, 0x5529, 0x552b, 0x552d, 0x5532, 0x5534, 0x5535, 0x5536,
	0x5538, 0x5539, 0x553a, 0x553b, 0x553d, 0x5540, 0x5542, 0x5545,
	0x5547, 0x5548, 0x554b, 0x554c, 0x554d, 0x554e, 0x554f, 0x5551,
	0x5552, 0x5553, 0x5554, 0x5557, 0x5558, 0x5559, 0x555a, 0x555b,
	0x555d, 0x555e, 0x555f, 0x5560, 0x5562, 0x5563, 0x5568, 0x5569,
	0x556b, 0x556f, 0x5570, 0x5571, 0x5572, 0x5573, 0x5574, 0x5579,
	0x557a, 0x557d, 0x557f, 0x5585, 0x5586, 0x558c, 0x558d, 0x558e,
	0x5590, 0x5592, 0x5593, 0x5595, 0x5596, 0x5597, 0x559a, 0x559b,
	0x559e, 0x55a0, 0x55a1, 0x55a2, 0x55a3, 0x55a4, 0x55a5, 0x55a6,
	0x55a8, 0x55a9, 0x55aa, 0x55ab, 0x55ac, 0x55ad, 0x55ae, 0x55af,
	0x55b0, 0x55b2, 0x55b4, 0x55b6, 0x55b8, 0x55ba, 0x55bc, 0x55bf,
	0x55c0, 0x55c1, 0x55c2, 0x55c3, 0x55c6, 0x55c7, 0x55c8, 0x55ca,
	0x55cb, 0x55ce, 0x55cf, 0x55d0, 0x55d5, 0x55d7, 0x55d8, 0x55d9,
	0x55da, 0x55db, 0x55de, 0x55e0, 0x55e2, 0x55e7, 0x55e9, 0x55ed,
	0x55ee, 0x55f0, 0x55f1, 0x55f4, 0x55f6, 0x55f8, 0x55f9, 0x55fa,
	0x55fb, 0x55fc, 0x55ff, 0x5602, 0x5603, 0x5604, 0x5605, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8740[ 192 ] = {
	0x5606, 0x5607, 0x560a, 0x560b, 0x560d, 0x5610, 0x5611, 0x5612,
	0x5613, 0x5614, 0x5615, 0x5616, 0x5617, 0x5619, 0x561a, 0x561c,
	0x561d, 0x5620, 0x5621, 0x5622, 0x5625, 0x5626, 0x5628, 0x5629,
	0x562a, 0x562b, 0x562e, 0x562f, 0x5630, 0x5633, 0x5635, 0x5637,
	0x5638, 0x563a, 0x563c, 0x563d, 0x563e, 0x5640, 0x5641, 0x5642,
	0x5643, 0x5644, 0x5645, 0x5646, 0x5647, 0x5648, 0x5649, 0x564a,
	0x564b, 0x564f, 0x5650, 0x5651, 0x5652, 0x5653, 0x5655, 0x5656,
	0x565a, 0x565b, 0x565d, 0x565e, 0x565f, 0x5660, 0x5661, 0xfffd,
	0x5663, 0x5665, 0x5666, 0x5667, 0x566d, 0x566e, 0x566f, 0x5670,
	0x5672, 0x5673, 0x5674, 0x5675, 0x5677, 0x5678, 0x5679, 0x567a,
	0x567d, 0x567e, 0x567f, 0x5680, 0x5681, 0x5682, 0x5683, 0x5684,
	0x5687, 0x5688, 0x5689, 0x568a, 0x568b, 0x568c, 0x568d, 0x5690,
	0x5691, 0x5692, 0x5694, 0x5695, 0x5696, 0x5697, 0x5698, 0x5699,
	0x569a, 0x569b, 0x569c, 0x569d, 0x569e, 0x569f, 0x56a0, 0x56a1,
	0x56a2, 0x56a4, 0x56a5, 0x56a6, 0x56a7, 0x56a8, 0x56a9, 0x56aa,
	0x56ab, 0x56ac, 0x56ad, 0x56ae, 0x56b0, 0x56b1, 0x56b2, 0x56b3,
	0x56b4, 0x56b5, 0x56b6, 0x56b8, 0x56b9, 0x56ba, 0x56bb, 0x56bd,
	0x56be, 0x56bf, 0x56c0, 0x56c1, 0x56c2, 0x56c3, 0x56c4, 0x56c5,
	0x56c6, 0x56c7, 0x56c8, 0x56c9, 0x56cb, 0x56cc, 0x56cd, 0x56ce,
	0x56cf, 0x56d0, 0x56d1, 0x56d2, 0x56d3, 0x56d5, 0x56d6, 0x56d8,
	0x56d9, 0x56dc, 0x56e3, 0x56e5, 0x56e6, 0x56e7, 0x56e8, 0x56e9,
	0x56ea, 0x56ec, 0x56ee, 0x56ef, 0x56f2, 0x56f3, 0x56f6, 0x56f7,
	0x56f8, 0x56fb, 0x56fc, 0x5700, 0x5701, 0x5702, 0x5705, 0x5707,
	0x570b, 0x570c, 0x570d, 0x570e, 0x570f, 0x5710, 0x5711, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8840[ 192 ] = {
	0x5712, 0x5713, 0x5714, 0x5715, 0x5716, 0x5717, 0x5718, 0x5719,
	0x571a, 0x571b, 0x571d, 0x571e, 0x5720, 0x5721, 0x5722, 0x5724,
	0x5725, 0x5726, 0x5727, 0x572b, 0x5731, 0x5732, 0x5734, 0x5735,
	0x5736, 0x5737, 0x5738, 0x573c, 0x573d, 0x573f, 0x5741, 0x5743,
	0x5744, 0x5745, 0x5746, 0x5748, 0x5749, 0x574b, 0x5752, 0x5753,
	0x5754, 0x5755, 0x5756, 0x5758, 0x5759, 0x5762, 0x5763, 0x5765,
	0x5767, 0x576c, 0x576e, 0x5770, 0x5771, 0x5772, 0x5774, 0x5775,
	0x5778, 0x5779, 0x577a, 0x577d, 0x577e, 0x577f, 0x5780, 0xfffd,
	0x5781, 0x5787, 0x5788, 0x5789, 0x578a, 0x578d, 0x578e, 0x578f,
	0x5790, 0x5791, 0x5794, 0x5795, 0x5796, 0x5797, 0x5798, 0x5799,
	0x579a, 0x579c, 0x579d, 0x579e, 0x579f, 0x57a5, 0x57a8, 0x57aa,
	0x57ac, 0x57af, 0x57b0, 0x57b1, 0x57b3, 0x57b5, 0x57b6, 0x57b7,
	0x57b9, 0x57ba, 0x57bb, 0x57bc, 0x57bd, 0x57be, 0x57bf, 0x57c0,
	0x57c1, 0x57c4, 0x57c5, 0x57c6, 0x57c7, 0x57c8, 0x57c9, 0x57ca,
	0x57cc, 0x57cd, 0x57d0, 0x57d1, 0x57d3, 0x57d6, 0x57d7, 0x57db,
	0x57dc, 0x57de, 0x57e1, 0x57e2, 0x57e3, 0x57e5, 0x57e6, 0x57e7,
	0x57e8, 0x57e9, 0x57ea, 0x57eb, 0x57ec, 0x57ee, 0x57f0, 0x57f1,
	0x57f2, 0x57f3, 0x57f5, 0x57f6, 0x57f7, 0x57fb, 0x57fc, 0x57fe,
	0x57ff, 0x5801, 0x5803, 0x5804, 0x5805, 0x5808, 0x5809, 0x580a,
	0x580c, 0x580e, 0x580f, 0x5810, 0x5812, 0x5813, 0x5814, 0x5816,
	0x5817, 0x5818, 0x581a, 0x581b, 0x581c, 0x581d, 0x581f, 0x5822,
	0x5823, 0x5825, 0x5826, 0x5827, 0x5828, 0x5829, 0x582b, 0x582c,
	0x582d, 0x582e, 0x582f, 0x5831, 0x5832, 0x5833, 0x5834, 0x5836,
	0x5837, 0x5838, 0x5839, 0x583a, 0x583b, 0x583c, 0x583d, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8940[ 192 ] = {
	0x583e, 0x583f, 0x5840, 0x5841, 0x5842, 0x5843, 0x5845, 0x5846,
	0x5847, 0x5848, 0x5849, 0x584a, 0x584b, 0x584e, 0x584f, 0x5850,
	0x5852, 0x5853, 0x5855, 0x5856, 0x5857, 0x5859, 0x585a, 0x585b,
	0x585c, 0x585d, 0x585f, 0x5860, 0x5861, 0x5862, 0x5863, 0x5864,
	0x5866, 0x5867, 0x5868, 0x5869, 0x586a, 0x586d, 0x586e, 0x586f,
	0x5870, 0x5871, 0x5872, 0x5873, 0x5874, 0x5875, 0x5876, 0x5877,
	0x5878, 0x5879, 0x587a, 0x587b, 0x587c, 0x587d, 0x587f, 0x5882,
	0x5884, 0x5886, 0x5887, 0x5888, 0x588a, 0x588b, 0x588c, 0xfffd,
	0x588d, 0x588e, 0x588f, 0x5890, 0x5891, 0x5894, 0x5895, 0x5896,
	0x5897, 0x5898, 0x589b, 0x589c, 0x589d, 0x58a0, 0x58a1, 0x58a2,
	0x58a3, 0x58a4, 0x58a5, 0x58a6, 0x58a7, 0x58aa, 0x58ab, 0x58ac,
	0x58ad, 0x58ae, 0x58af, 0x58b0, 0x58b1, 0x58b2, 0x58b3, 0x58b4,
	0x58b5, 0x58b6, 0x58b7, 0x58b8, 0x58b9, 0x58ba, 0x58bb, 0x58bd,
	0x58be, 0x58bf, 0x58c0, 0x58c2, 0x58c3, 0x58c4, 0x58c6, 0x58c7,
	0x58c8, 0x58c9, 0x58ca, 0x58cb, 0x58cc, 0x58cd, 0x58ce, 0x58cf,
	0x58d0, 0x58d2, 0x58d3, 0x58d4, 0x58d6, 0x58d7, 0x58d8, 0x58d9,
	0x58da, 0x58db, 0x58dc, 0x58dd, 0x58de, 0x58df, 0x58e0, 0x58e1,
	0x58e2, 0x58e3, 0x58e5, 0x58e6, 0x58e7, 0x58e8, 0x58e9, 0x58ea,
	0x58ed, 0x58ef, 0x58f1, 0x58f2, 0x58f4, 0x58f5, 0x58f7, 0x58f8,
	0x58fa, 0x58fb, 0x58fc, 0x58fd, 0x58fe, 0x58ff, 0x5900, 0x5901,
	0x5903, 0x5905, 0x5906, 0x5908, 0x5909, 0x590a, 0x590b, 0x590c,
	0x590e, 0x5910, 0x5911, 0x5912, 0x5913, 0x5917, 0x5918, 0x591b,
	0x591d, 0x591e, 0x5920, 0x5921, 0x5922, 0x5923, 0x5926, 0x5928,
	0x592c, 0x5930, 0x5932, 0x5933, 0x5935, 0x5936, 0x593b, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8a40[ 192 ] = {
	0x593d, 0x593e, 0x593f, 0x5940, 0x5943, 0x5945, 0x5946, 0x594a,
	0x594c, 0x594d, 0x5950, 0x5952, 0x5953, 0x5959, 0x595b, 0x595c,
	0x595d, 0x595e, 0x595f, 0x5961, 0x5963, 0x5964, 0x5966, 0x5967,
	0x5968, 0x5969, 0x596a, 0x596b, 0x596c, 0x596d, 0x596e, 0x596f,
	0x5970, 0x5971, 0x5972, 0x5975, 0x5977, 0x597a, 0x597b, 0x597c,
	0x597e, 0x597f, 0x5980, 0x5985, 0x5989, 0x598b, 0x598c, 0x598e,
	0x598f, 0x5990, 0x5991, 0x5994, 0x5995, 0x5998, 0x599a, 0x599b,
	0x599c, 0x599d, 0x599f, 0x59a0, 0x59a1, 0x59a2, 0x59a6, 0xfffd,
	0x59a7, 0x59ac, 0x59ad, 0x59b0, 0x59b1, 0x59b3, 0x59b4, 0x59b5,
	0x59b6, 0x59b7, 0x59b8, 0x59ba, 0x59bc, 0x59bd, 0x59bf, 0x59c0,
	0x59c1, 0x59c2, 0x59c3, 0x59c4, 0x59c5, 0x59c7, 0x59c8, 0x59c9,
	0x59cc, 0x59cd, 0x59ce, 0x59cf, 0x59d5, 0x59d6, 0x59d9, 0x59db,
	0x59de, 0x59df, 0x59e0, 0x59e1, 0x59e2, 0x59e4, 0x59e6, 0x59e7,
	0x59e9, 0x59ea, 0x59eb, 0x59ed, 0x59ee, 0x59ef, 0x59f0, 0x59f1,
	0x59f2, 0x59f3, 0x59f4, 0x59f5, 0x59f6, 0x59f7, 0x59f8, 0x59fa,
	0x59fc, 0x59fd, 0x59fe, 0x5a00, 0x5a02, 0x5a0a, 0x5a0b, 0x5a0d,
	0x5a0e, 0x5a0f, 0x5a10, 0x5a12, 0x5a14, 0x5a15, 0x5a16, 0x5a17,
	0x5a19, 0x5a1a, 0x5a1b, 0x5a1d, 0x5a1e, 0x5a21, 0x5a22, 0x5a24,
	0x5a26, 0x5a27, 0x5a28, 0x5a2a, 0x5a2b, 0x5a2c, 0x5a2d, 0x5a2e,
	0x5a2f, 0x5a30, 0x5a33, 0x5a35, 0x5a37, 0x5a38, 0x5a39, 0x5a3a,
	0x5a3b, 0x5a3d, 0x5a3e, 0x5a3f, 0x5a41, 0x5a42, 0x5a43, 0x5a44,
	0x5a45, 0x5a47, 0x5a48, 0x5a4b, 0x5a4c, 0x5a4d, 0x5a4e, 0x5a4f,
	0x5a50, 0x5a51, 0x5a52, 0x5a53, 0x5a54, 0x5a56, 0x5a57, 0x5a58,
	0x5a59, 0x5a5b, 0x5a5c, 0x5a5d, 0x5a5e, 0x5a5f, 0x5a60, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8b40[ 192 ] = {
	0x5a61, 0x5a63, 0x5a64, 0x5a65, 0x5a66, 0x5a68, 0x5a69, 0x5a6b,
	0x5a6c, 0x5a6d, 0x5a6e, 0x5a6f, 0x5a70, 0x5a71, 0x5a72, 0x5a73,
	0x5a78, 0x5a79, 0x5a7b, 0x5a7c, 0x5a7d, 0x5a7e, 0x5a80, 0x5a81,
	0x5a82, 0x5a83, 0x5a84, 0x5a85, 0x5a86, 0x5a87, 0x5a88, 0x5a89,
	0x5a8a, 0x5a8b, 0x5a8c, 0x5a8d, 0x5a8e, 0x5a8f, 0x5a90, 0x5a91,
	0x5a93, 0x5a94, 0x5a95, 0x5a96, 0x5a97, 0x5a98, 0x5a99, 0x5a9c,
	0x5a9d, 0x5a9e, 0x5a9f, 0x5aa0, 0x5aa1, 0x5aa2, 0x5aa3, 0x5aa4,
	0x5aa5, 0x5aa6, 0x5aa7, 0x5aa8, 0x5aa9, 0x5aab, 0x5aac, 0xfffd,
	0x5aad, 0x5aae, 0x5aaf, 0x5ab0, 0x5ab1, 0x5ab4, 0x5ab6, 0x5ab7,
	0x5ab9, 0x5aba, 0x5abb, 0x5abc, 0x5abd, 0x5abf, 0x5ac0, 0x5ac3,
	0x5ac4, 0x5ac5, 0x5ac6, 0x5ac7, 0x5ac8, 0x5aca, 0x5acb, 0x5acd,
	0x5ace, 0x5acf, 0x5ad0, 0x5ad1, 0x5ad3, 0x5ad5, 0x5ad7, 0x5ad9,
	0x5ada, 0x5adb, 0x5add, 0x5ade, 0x5adf, 0x5ae2, 0x5ae4, 0x5ae5,
	0x5ae7, 0x5ae8, 0x5aea, 0x5aec, 0x5aed, 0x5aee, 0x5aef, 0x5af0,
	0x5af2, 0x5af3, 0x5af4, 0x5af5, 0x5af6, 0x5af7, 0x5af8, 0x5af9,
	0x5afa, 0x5afb, 0x5afc, 0x5afd, 0x5afe, 0x5aff, 0x5b00, 0x5b01,
	0x5b02, 0x5b03, 0x5b04, 0x5b05, 0x5b06, 0x5b07, 0x5b08, 0x5b0a,
	0x5b0b, 0x5b0c, 0x5b0d, 0x5b0e, 0x5b0f, 0x5b10, 0x5b11, 0x5b12,
	0x5b13, 0x5b14, 0x5b15, 0x5b18, 0x5b19, 0x5b1a, 0x5b1b, 0x5b1c,
	0x5b1d, 0x5b1e, 0x5b1f, 0x5b20, 0x5b21, 0x5b22, 0x5b23, 0x5b24,
	0x5b25, 0x5b26, 0x5b27, 0x5b28, 0x5b29, 0x5b2a, 0x5b2b, 0x5b2c,
	0x5b2d, 0x5b2e, 0x5b2f, 0x5b30, 0x5b31, 0x5b33, 0x5b35, 0x5b36,
	0x5b38, 0x5b39, 0x5b3a, 0x5b3b, 0x5b3c, 0x5b3d, 0x5b3e, 0x5b3f,
	0x5b41, 0x5b42, 0x5b43, 0x5b44, 0x5b45, 0x5b46, 0x5b47, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8c40[ 192 ] = {
	0x5b48, 0x5b49, 0x5b4a, 0x5b4b, 0x5b4c, 0x5b4d, 0x5b4e, 0x5b4f,
	0x5b52, 0x5b56, 0x5b5e, 0x5b60, 0x5b61, 0x5b67, 0x5b68, 0x5b6b,
	0x5b6d, 0x5b6e, 0x5b6f, 0x5b72, 0x5b74, 0x5b76, 0x5b77, 0x5b78,
	0x5b79, 0x5b7b, 0x5b7c, 0x5b7e, 0x5b7f, 0x5b82, 0x5b86, 0x5b8a,
	0x5b8d, 0x5b8e, 0x5b90, 0x5b91, 0x5b92, 0x5b94, 0x5b96, 0x5b9f,
	0x5ba7, 0x5ba8, 0x5ba9, 0x5bac, 0x5bad, 0x5bae, 0x5baf, 0x5bb1,
	0x5bb2, 0x5bb7, 0x5bba, 0x5bbb, 0x5bbc, 0x5bc0, 0x5bc1, 0x5bc3,
	0x5bc8, 0x5bc9, 0x5bca, 0x5bcb, 0x5bcd, 0x5bce, 0x5bcf, 0xfffd,
	0x5bd1, 0x5bd4, 0x5bd5, 0x5bd6, 0x5bd7, 0x5bd8, 0x5bd9, 0x5bda,
	0x5bdb, 0x5bdc, 0x5be0, 0x5be2, 0x5be3, 0x5be6, 0x5be7, 0x5be9,
	0x5bea, 0x5beb, 0x5bec, 0x5bed, 0x5bef, 0x5bf1, 0x5bf2, 0x5bf3,
	0x5bf4, 0x5bf5, 0x5bf6, 0x5bf7, 0x5bfd, 0x5bfe, 0x5c00, 0x5c02,
	0x5c03, 0x5c05, 0x5c07, 0x5c08, 0x5c0b, 0x5c0c, 0x5c0d, 0x5c0e,
	0x5c10, 0x5c12, 0x5c13, 0x5c17, 0x5c19, 0x5c1b, 0x5c1e, 0x5c1f,
	0x5c20, 0x5c21, 0x5c23, 0x5c26, 0x5c28, 0x5c29, 0x5c2a, 0x5c2b,
	0x5c2d, 0x5c2e, 0x5c2f, 0x5c30, 0x5c32, 0x5c33, 0x5c35, 0x5c36,
	0x5c37, 0x5c43, 0x5c44, 0x5c46, 0x5c47, 0x5c4c, 0x5c4d, 0x5c52,
	0x5c53, 0x5c54, 0x5c56, 0x5c57, 0x5c58, 0x5c5a, 0x5c5b, 0x5c5c,
	0x5c5d, 0x5c5f, 0x5c62, 0x5c64, 0x5c67, 0x5c68, 0x5c69, 0x5c6a,
	0x5c6b, 0x5c6c, 0x5c6d, 0x5c70, 0x5c72, 0x5c73, 0x5c74, 0x5c75,
	0x5c76, 0x5c77, 0x5c78, 0x5c7b, 0x5c7c, 0x5c7d, 0x5c7e, 0x5c80,
	0x5c83, 0x5c84, 0x5c85, 0x5c86, 0x5c87, 0x5c89, 0x5c8a, 0x5c8b,
	0x5c8e, 0x5c8f, 0x5c92, 0x5c93, 0x5c95, 0x5c9d, 0x5c9e, 0x5c9f,
	0x5ca0, 0x5ca1, 0x5ca4, 0x5ca5, 0x5ca6, 0x5ca7, 0x5ca8, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8d40[ 192 ] = {
	0x5caa, 0x5cae, 0x5caf, 0x5cb0, 0x5cb2, 0x5cb4, 0x5cb6, 0x5cb9,
	0x5cba, 0x5cbb, 0x5cbc, 0x5cbe, 0x5cc0, 0x5cc2, 0x5cc3, 0x5cc5,
	0x5cc6, 0x5cc7, 0x5cc8, 0x5cc9, 0x5cca, 0x5ccc, 0x5ccd, 0x5cce,
	0x5ccf, 0x5cd0, 0x5cd1, 0x5cd3, 0x5cd4, 0x5cd5, 0x5cd6, 0x5cd7,
	0x5cd8, 0x5cda, 0x5cdb, 0x5cdc, 0x5cdd, 0x5cde, 0x5cdf, 0x5ce0,
	0x5ce2, 0x5ce3, 0x5ce7, 0x5ce9, 0x5ceb, 0x5cec, 0x5cee, 0x5cef,
	0x5cf1, 0x5cf2, 0x5cf3, 0x5cf4, 0x5cf5, 0x5cf6, 0x5cf7, 0x5cf8,
	0x5cf9, 0x5cfa, 0x5cfc, 0x5cfd, 0x5cfe, 0x5cff, 0x5d00, 0xfffd,
	0x5d01, 0x5d04, 0x5d05, 0x5d08, 0x5d09, 0x5d0a, 0x5d0b, 0x5d0c,
	0x5d0d, 0x5d0f, 0x5d10, 0x5d11, 0x5d12, 0x5d13, 0x5d15, 0x5d17,
	0x5d18, 0x5d19, 0x5d1a, 0x5d1c, 0x5d1d, 0x5d1f, 0x5d20, 0x5d21,
	0x5d22, 0x5d23, 0x5d25, 0x5d28, 0x5d2a, 0x5d2b, 0x5d2c, 0x5d2f,
	0x5d30, 0x5d31, 0x5d32, 0x5d33, 0x5d35, 0x5d36, 0x5d37, 0x5d38,
	0x5d39, 0x5d3a, 0x5d3b, 0x5d3c, 0x5d3f, 0x5d40, 0x5d41, 0x5d42,
	0x5d43, 0x5d44, 0x5d45, 0x5d46, 0x5d48, 0x5d49, 0x5d4d, 0x5d4e,
	0x5d4f, 0x5d50, 0x5d51, 0x5d52, 0x5d53, 0x5d54, 0x5d55, 0x5d56,
	0x5d57, 0x5d59, 0x5d5a, 0x5d5c, 0x5d5e, 0x5d5f, 0x5d60, 0x5d61,
	0x5d62, 0x5d63, 0x5d64, 0x5d65, 0x5d66, 0x5d67, 0x5d68, 0x5d6a,
	0x5d6d, 0x5d6e, 0x5d70, 0x5d71, 0x5d72, 0x5d73, 0x5d75, 0x5d76,
	0x5d77, 0x5d78, 0x5d79, 0x5d7a, 0x5d7b, 0x5d7c, 0x5d7d, 0x5d7e,
	0x5d7f, 0x5d80, 0x5d81, 0x5d83, 0x5d84, 0x5d85, 0x5d86, 0x5d87,
	0x5d88, 0x5d89, 0x5d8a, 0x5d8b, 0x5d8c, 0x5d8d, 0x5d8e, 0x5d8f,
	0x5d90, 0x5d91, 0x5d92, 0x5d93, 0x5d94, 0x5d95, 0x5d96, 0x5d97,
	0x5d98, 0x5d9a, 0x5d9b, 0x5d9c, 0x5d9e, 0x5d9f, 0x5da0, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8e40[ 192 ] = {
	0x5da1, 0x5da2, 0x5da3, 0x5da4, 0x5da5, 0x5da6, 0x5da7, 0x5da8,
	0x5da9, 0x5daa, 0x5dab, 0x5dac, 0x5dad, 0x5dae, 0x5daf, 0x5db0,
	0x5db1, 0x5db2, 0x5db3, 0x5db4, 0x5db5, 0x5db6, 0x5db8, 0x5db9,
	0x5dba, 0x5dbb, 0x5dbc, 0x5dbd, 0x5dbe, 0x5dbf, 0x5dc0, 0x5dc1,
	0x5dc2, 0x5dc3, 0x5dc4, 0x5dc6, 0x5dc7, 0x5dc8, 0x5dc9, 0x5dca,
	0x5dcb, 0x5dcc, 0x5dce, 0x5dcf, 0x5dd0, 0x5dd1, 0x5dd2, 0x5dd3,
	0x5dd4, 0x5dd5, 0x5dd6, 0x5dd7, 0x5dd8, 0x5dd9, 0x5dda, 0x5ddc,
	0x5ddf, 0x5de0, 0x5de3, 0x5de4, 0x5dea, 0x5dec, 0x5ded, 0xfffd,
	0x5df0, 0x5df5, 0x5df6, 0x5df8, 0x5df9, 0x5dfa, 0x5dfb, 0x5dfc,
	0x5dff, 0x5e00, 0x5e04, 0x5e07, 0x5e09, 0x5e0a, 0x5e0b, 0x5e0d,
	0x5e0e, 0x5e12, 0x5e13, 0x5e17, 0x5e1e, 0x5e1f, 0x5e20, 0x5e21,
	0x5e22, 0x5e23, 0x5e24, 0x5e25, 0x5e28, 0x5e29, 0x5e2a, 0x5e2b,
	0x5e2c, 0x5e2f, 0x5e30, 0x5e32, 0x5e33, 0x5e34, 0x5e35, 0x5e36,
	0x5e39, 0x5e3a, 0x5e3e, 0x5e3f, 0x5e40, 0x5e41, 0x5e43, 0x5e46,
	0x5e47, 0x5e48, 0x5e49, 0x5e4a, 0x5e4b, 0x5e4d, 0x5e4e, 0x5e4f,
	0x5e50, 0x5e51, 0x5e52, 0x5e53, 0x5e56, 0x5e57, 0x5e58, 0x5e59,
	0x5e5a, 0x5e5c, 0x5e5d, 0x5e5f, 0x5e60, 0x5e63, 0x5e64, 0x5e65,
	0x5e66, 0x5e67, 0x5e68, 0x5e69, 0x5e6a, 0x5e6b, 0x5e6c, 0x5e6d,
	0x5e6e, 0x5e6f, 0x5e70, 0x5e71, 0x5e75, 0x5e77, 0x5e79, 0x5e7e,
	0x5e81, 0x5e82, 0x5e83, 0x5e85, 0x5e88, 0x5e89, 0x5e8c, 0x5e8d,
	0x5e8e, 0x5e92, 0x5e98, 0x5e9b, 0x5e9d, 0x5ea1, 0x5ea2, 0x5ea3,
	0x5ea4, 0x5ea8, 0x5ea9, 0x5eaa, 0x5eab, 0x5eac, 0x5eae, 0x5eaf,
	0x5eb0, 0x5eb1, 0x5eb2, 0x5eb4, 0x5eba, 0x5ebb, 0x5ebc, 0x5ebd,
	0x5ebf, 0x5ec0, 0x5ec1, 0x5ec2, 0x5ec3, 0x5ec4, 0x5ec5, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8f40[ 192 ] = {
	0x5ec6, 0x5ec7, 0x5ec8, 0x5ecb, 0x5ecc, 0x5ecd, 0x5ece, 0x5ecf,
	0x5ed0, 0x5ed4, 0x5ed5, 0x5ed7, 0x5ed8, 0x5ed9, 0x5eda, 0x5edc,
	0x5edd, 0x5ede, 0x5edf, 0x5ee0, 0x5ee1, 0x5ee2, 0x5ee3, 0x5ee4,
	0x5ee5, 0x5ee6, 0x5ee7, 0x5ee9, 0x5eeb, 0x5eec, 0x5eed, 0x5eee,
	0x5eef, 0x5ef0, 0x5ef1, 0x5ef2, 0x5ef3, 0x5ef5, 0x5ef8, 0x5ef9,
	0x5efb, 0x5efc, 0x5efd, 0x5f05, 0x5f06, 0x5f07, 0x5f09, 0x5f0c,
	0x5f0d, 0x5f0e, 0x5f10, 0x5f12, 0x5f14, 0x5f16, 0x5f19, 0x5f1a,
	0x5f1c, 0x5f1d, 0x5f1e, 0x5f21, 0x5f22, 0x5f23, 0x5f24, 0xfffd,
	0x5f28, 0x5f2b, 0x5f2c, 0x5f2e, 0x5f30, 0x5f32, 0x5f33, 0x5f34,
	0x5f35, 0x5f36, 0x5f37, 0x5f38, 0x5f3b, 0x5f3d, 0x5f3e, 0x5f3f,
	0x5f41, 0x5f42, 0x5f43, 0x5f44, 0x5f45, 0x5f46, 0x5f47, 0x5f48,
	0x5f49, 0x5f4a, 0x5f4b, 0x5f4c, 0x5f4d, 0x5f4e, 0x5f4f, 0x5f51,
	0x5f54, 0x5f59, 0x5f5a, 0x5f5b, 0x5f5c, 0x5f5e, 0x5f5f, 0x5f60,
	0x5f63, 0x5f65, 0x5f67, 0x5f68, 0x5f6b, 0x5f6e, 0x5f6f, 0x5f72,
	0x5f74, 0x5f75, 0x5f76, 0x5f78, 0x5f7a, 0x5f7d, 0x5f7e, 0x5f7f,
	0x5f83, 0x5f86, 0x5f8d, 0x5f8e, 0x5f8f, 0x5f91, 0x5f93, 0x5f94,
	0x5f96, 0x5f9a, 0x5f9b, 0x5f9d, 0x5f9e, 0x5f9f, 0x5fa0, 0x5fa2,
	0x5fa3, 0x5fa4, 0x5fa5, 0x5fa6, 0x5fa7, 0x5fa9, 0x5fab, 0x5fac,
	0x5faf, 0x5fb0, 0x5fb1, 0x5fb2, 0x5fb3, 0x5fb4, 0x5fb6, 0x5fb8,
	0x5fb9, 0x5fba, 0x5fbb, 0x5fbe, 0x5fbf, 0x5fc0, 0x5fc1, 0x5fc2,
	0x5fc7, 0x5fc8, 0x5fca, 0x5fcb, 0x5fce, 0x5fd3, 0x5fd4, 0x5fd5,
	0x5fda, 0x5fdb, 0x5fdc, 0x5fde, 0x5fdf, 0x5fe2, 0x5fe3, 0x5fe5,
	0x5fe6, 0x5fe8, 0x5fe9, 0x5fec, 0x5fef, 0x5ff0, 0x5ff2, 0x5ff3,
	0x5ff4, 0x5ff6, 0x5ff7, 0x5ff9, 0x5ffa, 0x5ffc, 0x6007, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9040[ 192 ] = {
	0x6008, 0x6009, 0x600b, 0x600c, 0x6010, 0x6011, 0x6013, 0x6017,
	0x6018, 0x601a, 0x601e, 0x601f, 0x6022, 0x6023, 0x6024, 0x602c,
	0x602d, 0x602e, 0x6030, 0x6031, 0x6032, 0x6033, 0x6034, 0x6036,
	0x6037, 0x6038, 0x6039, 0x603a, 0x603d, 0x603e, 0x6040, 0x6044,
	0x6045, 0x6046, 0x6047, 0x6048, 0x6049, 0x604a, 0x604c, 0x604e,
	0x604f, 0x6051, 0x6053, 0x6054, 0x6056, 0x6057, 0x6058, 0x605b,
	0x605c, 0x605e, 0x605f, 0x6060, 0x6061, 0x6065, 0x6066, 0x606e,
	0x6071, 0x6072, 0x6074, 0x6075, 0x6077, 0x607e, 0x6080, 0xfffd,
	0x6081, 0x6082, 0x6085, 0x6086, 0x6087, 0x6088, 0x608a, 0x608b,
	0x608e, 0x608f, 0x6090, 0x6091, 0x6093, 0x6095, 0x6097, 0x6098,
	0x6099, 0x609c, 0x609e, 0x60a1, 0x60a2, 0x60a4, 0x60a5, 0x60a7,
	0x60a9, 0x60aa, 0x60ae, 0x60b0, 0x60b3, 0x60b5, 0x60b6, 0x60b7,
	0x60b9, 0x60ba, 0x60bd, 0x60be, 0x60bf, 0x60c0, 0x60c1, 0x60c2,
	0x60c3, 0x60c4, 0x60c7, 0x60c8, 0x60c9, 0x60cc, 0x60cd, 0x60ce,
	0x60cf, 0x60d0, 0x60d2, 0x60d3, 0x60d4, 0x60d6, 0x60d7, 0x60d9,
	0x60db, 0x60de, 0x60e1, 0x60e2, 0x60e3, 0x60e4, 0x60e5, 0x60ea,
	0x60f1, 0x60f2, 0x60f5, 0x60f7, 0x60f8, 0x60fb, 0x60fc, 0x60fd,
	0x60fe, 0x60ff, 0x6102, 0x6103, 0x6104, 0x6105, 0x6107, 0x610a,
	0x610b, 0x610c, 0x6110, 0x6111, 0x6112, 0x6113, 0x6114, 0x6116,
	0x6117, 0x6118, 0x6119, 0x611b, 0x611c, 0x611d, 0x611e, 0x6121,
	0x6122, 0x6125, 0x6128, 0x6129, 0x612a, 0x612c, 0x612d, 0x612e,
	0x612f, 0x6130, 0x6131, 0x6132, 0x6133, 0x6134, 0x6135, 0x6136,
	0x6137, 0x6138, 0x6139, 0x613a, 0x613b, 0x613c, 0x613d, 0x613e,
	0x6140, 0x6141, 0x6142, 0x6143, 0x6144, 0x6145, 0x6146, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9140[ 192 ] = {
	0x6147, 0x6149, 0x614b, 0x614d, 0x614f, 0x6150, 0x6152, 0x6153,
	0x6154, 0x6156, 0x6157, 0x6158, 0x6159, 0x615a, 0x615b, 0x615c,
	0x615e, 0x615f, 0x6160, 0x6161, 0x6163, 0x6164, 0x6165, 0x6166,
	0x6169, 0x616a, 0x616b, 0x616c, 0x616d, 0x616e, 0x616f, 0x6171,
	0x6172, 0x6173, 0x6174, 0x6176, 0x6178, 0x6179, 0x617a, 0x617b,
	0x617c, 0x617d, 0x617e, 0x617f, 0x6180, 0x6181, 0x6182, 0x6183,
	0x6184, 0x6185, 0x6186, 0x6187, 0x6188, 0x6189, 0x618a, 0x618c,
	0x618d, 0x618f, 0x6190, 0x6191, 0x6192, 0x6193, 0x6195, 0xfffd,
	0x6196, 0x6197, 0x6198, 0x6199, 0x619a, 0x619b, 0x619c, 0x619e,
	0x619f, 0x61a0, 0x61a1, 0x61a2, 0x61a3, 0x61a4, 0x61a5, 0x61a6,
	0x61aa, 0x61ab, 0x61ad, 0x61ae, 0x61af, 0x61b0, 0x61b1, 0x61b2,
	0x61b3, 0x61b4, 0x61b5, 0x61b6, 0x61b8, 0x61b9, 0x61ba, 0x61bb,
	0x61bc, 0x61bd, 0x61bf, 0x61c0, 0x61c1, 0x61c3, 0x61c4, 0x61c5,
	0x61c6, 0x61c7, 0x61c9, 0x61cc, 0x61cd, 0x61ce, 0x61cf, 0x61d0,
	0x61d3, 0x61d5, 0x61d6, 0x61d7, 0x61d8, 0x61d9, 0x61da, 0x61db,
	0x61dc, 0x61dd, 0x61de, 0x61df, 0x61e0, 0x61e1, 0x61e2, 0x61e3,
	0x61e4, 0x61e5, 0x61e7, 0x61e8, 0x61e9, 0x61ea, 0x61eb, 0x61ec,
	0x61ed, 0x61ee, 0x61ef, 0x61f0, 0x61f1, 0x61f2, 0x61f3, 0x61f4,
	0x61f6, 0x61f7, 0x61f8, 0x61f9, 0x61fa, 0x61fb, 0x61fc, 0x61fd,
	0x61fe, 0x6200, 0x6201, 0x6202, 0x6203, 0x6204, 0x6205, 0x6207,
	0x6209, 0x6213, 0x6214, 0x6219, 0x621c, 0x621d, 0x621e, 0x6220,
	0x6223, 0x6226, 0x6227, 0x6228, 0x6229, 0x622b, 0x622d, 0x622f,
	0x6230, 0x6231, 0x6232, 0x6235, 0x6236, 0x6238, 0x6239, 0x623a,
	0x623b, 0x623c, 0x6242, 0x6244, 0x6245, 0x6246, 0x624a, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9240[ 192 ] = {
	0x624f, 0x6250, 0x6255, 0x6256, 0x6257, 0x6259, 0x625a, 0x625c,
	0x625d, 0x625e, 0x625f, 0x6260, 0x6261, 0x6262, 0x6264, 0x6265,
	0x6268, 0x6271, 0x6272, 0x6274, 0x6275, 0x6277, 0x6278, 0x627a,
	0x627b, 0x627d, 0x6281, 0x6282, 0x6283, 0x6285, 0x6286, 0x6287,
	0x6288, 0x628b, 0x628c, 0x628d, 0x628e, 0x628f, 0x6290, 0x6294,
	0x6299, 0x629c, 0x629d, 0x629e, 0x62a3, 0x62a6, 0x62a7, 0x62a9,
	0x62aa, 0x62ad, 0x62ae, 0x62af, 0x62b0, 0x62b2, 0x62b3, 0x62b4,
	0x62b6, 0x62b7, 0x62b8, 0x62ba, 0x62be, 0x62c0, 0x62c1, 0xfffd,
	0x62c3, 0x62cb, 0x62cf, 0x62d1, 0x62d5, 0x62dd, 0x62de, 0x62e0,
	0x62e1, 0x62e4, 0x62ea, 0x62eb, 0x62f0, 0x62f2, 0x62f5, 0x62f8,
	0x62f9, 0x62fa, 0x62fb, 0x6300, 0x6303, 0x6304, 0x6305, 0x6306,
	0x630a, 0x630b, 0x630c, 0x630d, 0x630f, 0x6310, 0x6312, 0x6313,
	0x6314, 0x6315, 0x6317, 0x6318, 0x6319, 0x631c, 0x6326, 0x6327,
	0x6329, 0x632c, 0x632d, 0x632e, 0x6330, 0x6331, 0x6333, 0x6334,
	0x6335, 0x6336, 0x6337, 0x6338, 0x633b, 0x633c, 0x633e, 0x633f,
	0x6340, 0x6341, 0x6344, 0x6347, 0x6348, 0x634a, 0x6351, 0x6352,
	0x6353, 0x6354, 0x6356, 0x6357, 0x6358, 0x6359, 0x635a, 0x635b,
	0x635c, 0x635d, 0x6360, 0x6364, 0x6365, 0x6366, 0x6368, 0x636a,
	0x636b, 0x636c, 0x636f, 0x6370, 0x6372, 0x6373, 0x6374, 0x6375,
	0x6378, 0x6379, 0x637c, 0x637d, 0x637e, 0x637f, 0x6381, 0x6383,
	0x6384, 0x6385, 0x6386, 0x638b, 0x638d, 0x6391, 0x6393, 0x6394,
	0x6395, 0x6397, 0x6399, 0x639a, 0x639b, 0x639c, 0x639d, 0x639e,
	0x639f, 0x63a1, 0x63a4, 0x63a6, 0x63ab, 0x63af, 0x63b1, 0x63b2,
	0x63b5, 0x63b6, 0x63b9, 0x63bb, 0x63bd, 0x63bf, 0x63c0, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9340[ 192 ] = {
	0x63c1, 0x63c2, 0x63c3, 0x63c5, 0x63c7, 0x63c8, 0x63ca, 0x63cb,
	0x63cc, 0x63d1, 0x63d3, 0x63d4, 0x63d5, 0x63d7, 0x63d8, 0x63d9,
	0x63da, 0x63db, 0x63dc, 0x63dd, 0x63df, 0x63e2, 0x63e4, 0x63e5,
	0x63e6, 0x63e7, 0x63e8, 0x63eb, 0x63ec, 0x63ee, 0x63ef, 0x63f0,
	0x63f1, 0x63f3, 0x63f5, 0x63f7, 0x63f9, 0x63fa, 0x63fb, 0x63fc,
	0x63fe, 0x6403, 0x6404, 0x6406, 0x6407, 0x6408, 0x6409, 0x640a,
	0x640d, 0x640e, 0x6411, 0x6412, 0x6415, 0x6416, 0x6417, 0x6418,
	0x6419, 0x641a, 0x641d, 0x641f, 0x6422, 0x6423, 0x6424, 0xfffd,
	0x6425, 0x6427, 0x6428, 0x6429, 0x642b, 0x642e, 0x642f, 0x6430,
	0x6431, 0x6432, 0x6433, 0x6435, 0x6436, 0x6437, 0x6438, 0x6439,
	0x643b, 0x643c, 0x643e, 0x6440, 0x6442, 0x6443, 0x6449, 0x644b,
	0x644c, 0x644d, 0x644e, 0x644f, 0x6450, 0x6451, 0x6453, 0x6455,
	0x6456, 0x6457, 0x6459, 0x645a, 0x645b, 0x645c, 0x645d, 0x645f,
	0x6460, 0x6461, 0x6462, 0x6463, 0x6464, 0x6465, 0x6466, 0x6468,
	0x646a, 0x646b, 0x646c, 0x646e, 0x646f, 0x6470, 0x6471, 0x6472,
	0x6473, 0x6474, 0x6475, 0x6476, 0x6477, 0x647b, 0x647c, 0x647d,
	0x647e, 0x647f, 0x6480, 0x6481, 0x6483, 0x6486, 0x6488, 0x6489,
	0x648a, 0x648b, 0x648c, 0x648d, 0x648e, 0x648f, 0x6490, 0x6493,
	0x6494, 0x6497, 0x6498, 0x649a, 0x649b, 0x649c, 0x649d, 0x649f,
	0x64a0, 0x64a1, 0x64a2, 0x64a3, 0x64a5, 0x64a6, 0x64a7, 0x64a8,
	0x64aa, 0x64ab, 0x64af, 0x64b1, 0x64b2, 0x64b3, 0x64b4, 0x64b6,
	0x64b9, 0x64bb, 0x64bd, 0x64be, 0x64bf, 0x64c1, 0x64c3, 0x64c4,
	0x64c6, 0x64c7, 0x64c8, 0x64c9, 0x64ca, 0x64cb, 0x64cc, 0x64cf,
	0x64d1, 0x64d3, 0x64d4, 0x64d5, 0x64d6, 0x64d9, 0x64da, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9440[ 192 ] = {
	0x64db, 0x64dc, 0x64dd, 0x64df, 0x64e0, 0x64e1, 0x64e3, 0x64e5,
	0x64e7, 0x64e8, 0x64e9, 0x64ea, 0x64eb, 0x64ec, 0x64ed, 0x64ee,
	0x64ef, 0x64f0, 0x64f1, 0x64f2, 0x64f3, 0x64f4, 0x64f5, 0x64f6,
	0x64f7, 0x64f8, 0x64f9, 0x64fa, 0x64fb, 0x64fc, 0x64fd, 0x64fe,
	0x64ff, 0x6501, 0x6502, 0x6503, 0x6504, 0x6505, 0x6506, 0x6507,
	0x6508, 0x650a, 0x650b, 0x650c, 0x650d, 0x650e, 0x650f, 0x6510,
	0x6511, 0x6513, 0x6514, 0x6515, 0x6516, 0x6517, 0x6519, 0x651a,
	0x651b, 0x651c, 0x651d, 0x651e, 0x651f, 0x6520, 0x6521, 0xfffd,
	0x6522, 0x6523, 0x6524, 0x6526, 0x6527, 0x6528, 0x6529, 0x652a,
	0x652c, 0x652d, 0x6530, 0x6531, 0x6532, 0x6533, 0x6537, 0x653a,
	0x653c, 0x653d, 0x6540, 0x6541, 0x6542, 0x6543, 0x6544, 0x6546,
	0x6547, 0x654a, 0x654b, 0x654d, 0x654e, 0x6550, 0x6552, 0x6553,
	0x6554, 0x6557, 0x6558, 0x655a, 0x655c, 0x655f, 0x6560, 0x6561,
	0x6564, 0x6565, 0x6567, 0x6568, 0x6569, 0x656a, 0x656d, 0x656e,
	0x656f, 0x6571, 0x6573, 0x6575, 0x6576, 0x6578, 0x6579, 0x657a,
	0x657b, 0x657c, 0x657d, 0x657e, 0x657f, 0x6580, 0x6581, 0x6582,
	0x6583, 0x6584, 0x6585, 0x6586, 0x6588, 0x6589, 0x658a, 0x658d,
	0x658e, 0x658f, 0x6592, 0x6594, 0x6595, 0x6596, 0x6598, 0x659a,
	0x659d, 0x659e, 0x65a0, 0x65a2, 0x65a3, 0x65a6, 0x65a8, 0x65aa,
	0x65ac, 0x65ae, 0x65b1, 0x65b2, 0x65b3, 0x65b4, 0x65b5, 0x65b6,
	0x65b7, 0x65b8, 0x65ba, 0x65bb, 0x65be, 0x65bf, 0x65c0, 0x65c2,
	0x65c7, 0x65c8, 0x65c9, 0x65ca, 0x65cd, 0x65d0, 0x65d1, 0x65d3,
	0x65d4, 0x65d5, 0x65d8, 0x65d9, 0x65da, 0x65db, 0x65dc, 0x65dd,
	0x65de, 0x65df, 0x65e1, 0x65e3, 0x65e4, 0x65ea, 0x65eb, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9540[ 192 ] = {
	0x65f2, 0x65f3, 0x65f4, 0x65f5, 0x65f8, 0x65f9, 0x65fb, 0x65fc,
	0x65fd, 0x65fe, 0x65ff, 0x6601, 0x6604, 0x6605, 0x6607, 0x6608,
	0x6609, 0x660b, 0x660d, 0x6610, 0x6611, 0x6612, 0x6616, 0x6617,
	0x6618, 0x661a, 0x661b, 0x661c, 0x661e, 0x6621, 0x6622, 0x6623,
	0x6624, 0x6626, 0x6629, 0x662a, 0x662b, 0x662c, 0x662e, 0x6630,
	0x6632, 0x6633, 0x6637, 0x6638, 0x6639, 0x663a, 0x663b, 0x663d,
	0x663f, 0x6640, 0x6642, 0x6644, 0x6645, 0x6646, 0x6647, 0x6648,
	0x6649, 0x664a, 0x664d, 0x664e, 0x6650, 0x6651, 0x6658, 0xfffd,
	0x6659, 0x665b, 0x665c, 0x665d, 0x665e, 0x6660, 0x6662, 0x6663,
	0x6665, 0x6667, 0x6669, 0x666a, 0x666b, 0x666c, 0x666d, 0x6671,
	0x6672, 0x6673, 0x6675, 0x6678, 0x6679, 0x667b, 0x667c, 0x667d,
	0x667f, 0x6680, 0x6681, 0x6683, 0x6685, 0x6686, 0x6688, 0x6689,
	0x668a, 0x668b, 0x668d, 0x668e, 0x668f, 0x6690, 0x6692, 0x6693,
	0x6694, 0x6695, 0x6698, 0x6699, 0x669a, 0x669b, 0x669c, 0x669e,
	0x669f, 0x66a0, 0x66a1, 0x66a2, 0x66a3, 0x66a4, 0x66a5, 0x66a6,
	0x66a9, 0x66aa, 0x66ab, 0x66ac, 0x66ad, 0x66af, 0x66b0, 0x66b1,
	0x66b2, 0x66b3, 0x66b5, 0x66b6, 0x66b7, 0x66b8, 0x66ba, 0x66bb,
	0x66bc, 0x66bd, 0x66bf, 0x66c0, 0x66c1, 0x66c2, 0x66c3, 0x66c4,
	0x66c5, 0x66c6, 0x66c7, 0x66c8, 0x66c9, 0x66ca, 0x66cb, 0x66cc,
	0x66cd, 0x66ce, 0x66cf, 0x66d0, 0x66d1, 0x66d2, 0x66d3, 0x66d4,
	0x66d5, 0x66d6, 0x66d7, 0x66d8, 0x66da, 0x66de, 0x66df, 0x66e0,
	0x66e1, 0x66e2, 0x66e3, 0x66e4, 0x66e5, 0x66e7, 0x66e8, 0x66ea,
	0x66eb, 0x66ec, 0x66ed, 0x66ee, 0x66ef, 0x66f1, 0x66f5, 0x66f6,
	0x66f8, 0x66fa, 0x66fb, 0x66fd, 0x6701, 0x6702, 0x6703, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9640[ 192 ] = {
	0x6704, 0x6705, 0x6706, 0x6707, 0x670c, 0x670e, 0x670f, 0x6711,
	0x6712, 0x6713, 0x6716, 0x6718, 0x6719, 0x671a, 0x671c, 0x671e,
	0x6720, 0x6721, 0x6722, 0x6723, 0x6724, 0x6725, 0x6727, 0x6729,
	0x672e, 0x6730, 0x6732, 0x6733, 0x6736, 0x6737, 0x6738, 0x6739,
	0x673b, 0x673c, 0x673e, 0x673f, 0x6741, 0x6744, 0x6745, 0x6747,
	0x674a, 0x674b, 0x674d, 0x6752, 0x6754, 0x6755, 0x6757, 0x6758,
	0x6759, 0x675a, 0x675b, 0x675d, 0x6762, 0x6763, 0x6764, 0x6766,
	0x6767, 0x676b, 0x676c, 0x676e, 0x6771, 0x6774, 0x6776, 0xfffd,
	0x6778, 0x6779, 0x677a, 0x677b, 0x677d, 0x6780, 0x6782, 0x6783,
	0x6785, 0x6786, 0x6788, 0x678a, 0x678c, 0x678d, 0x678e, 0x678f,
	0x6791, 0x6792, 0x6793, 0x6794, 0x6796, 0x6799, 0x679b, 0x679f,
	0x67a0, 0x67a1, 0x67a4, 0x67a6, 0x67a9, 0x67ac, 0x67ae, 0x67b1,
	0x67b2, 0x67b4, 0x67b9, 0x67ba, 0x67bb, 0x67bc, 0x67bd, 0x67be,
	0x67bf, 0x67c0, 0x67c2, 0x67c5, 0x67c6, 0x67c7, 0x67c8, 0x67c9,
	0x67ca, 0x67cb, 0x67cc, 0x67cd, 0x67ce, 0x67d5, 0x67d6, 0x67d7,
	0x67db, 0x67df, 0x67e1, 0x67e3, 0x67e4, 0x67e6, 0x67e7, 0x67e8,
	0x67ea, 0x67eb, 0x67ed, 0x67ee, 0x67f2, 0x67f5, 0x67f6, 0x67f7,
	0x67f8, 0x67f9, 0x67fa, 0x67fb, 0x67fc, 0x67fe, 0x6801, 0x6802,
	0x6803, 0x6804, 0x6806, 0x680d, 0x6810, 0x6812, 0x6814, 0x6815,
	0x6818, 0x6819, 0x681a, 0x681b, 0x681c, 0x681e, 0x681f, 0x6820,
	0x6822, 0x6823, 0x6824, 0x6825, 0x6826, 0x6827, 0x6828, 0x682b,
	0x682c, 0x682d, 0x682e, 0x682f, 0x6830, 0x6831, 0x6834, 0x6835,
	0x6836, 0x683a, 0x683b, 0x683f, 0x6847, 0x684b, 0x684d, 0x684f,
	0x6852, 0x6856, 0x6857, 0x6858, 0x6859, 0x685a, 0x685b, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9740[ 192 ] = {
	0x685c, 0x685d, 0x685e, 0x685f, 0x686a, 0x686c, 0x686d, 0x686e,
	0x686f, 0x6870, 0x6871, 0x6872, 0x6873, 0x6875, 0x6878, 0x6879,
	0x687a, 0x687b, 0x687c, 0x687d, 0x687e, 0x687f, 0x6880, 0x6882,
	0x6884, 0x6887, 0x6888, 0x6889, 0x688a, 0x688b, 0x688c, 0x688d,
	0x688e, 0x6890, 0x6891, 0x6892, 0x6894, 0x6895, 0x6896, 0x6898,
	0x6899, 0x689a, 0x689b, 0x689c, 0x689d, 0x689e, 0x689f, 0x68a0,
	0x68a1, 0x68a3, 0x68a4, 0x68a5, 0x68a9, 0x68aa, 0x68ab, 0x68ac,
	0x68ae, 0x68b1, 0x68b2, 0x68b4, 0x68b6, 0x68b7, 0x68b8, 0xfffd,
	0x68b9, 0x68ba, 0x68bb, 0x68bc, 0x68bd, 0x68be, 0x68bf, 0x68c1,
	0x68c3, 0x68c4, 0x68c5, 0x68c6, 0x68c7, 0x68c8, 0x68ca, 0x68cc,
	0x68ce, 0x68cf, 0x68d0, 0x68d1, 0x68d3, 0x68d4, 0x68d6, 0x68d7,
	0x68d9, 0x68db, 0x68dc, 0x68dd, 0x68de, 0x68df, 0x68e1, 0x68e2,
	0x68e4, 0x68e5, 0x68e6, 0x68e7, 0x68e8, 0x68e9, 0x68ea, 0x68eb,
	0x68ec, 0x68ed, 0x68ef, 0x68f2, 0x68f3, 0x68f4, 0x68f6, 0x68f7,
	0x68f8, 0x68fb, 0x68fd, 0x68fe, 0x68ff, 0x6900, 0x6902, 0x6903,
	0x6904, 0x6906, 0x6907, 0x6908, 0x6909, 0x690a, 0x690c, 0x690f,
	0x6911, 0x6913, 0x6914, 0x6915, 0x6916, 0x6917, 0x6918, 0x6919,
	0x691a, 0x691b, 0x691c, 0x691d, 0x691e, 0x6921, 0x6922, 0x6923,
	0x6925, 0x6926, 0x6927, 0x6928, 0x6929, 0x692a, 0x692b, 0x692c,
	0x692e, 0x692f, 0x6931, 0x6932, 0x6933, 0x6935, 0x6936, 0x6937,
	0x6938, 0x693a, 0x693b, 0x693c, 0x693e, 0x6940, 0x6941, 0x6943,
	0x6944, 0x6945, 0x6946, 0x6947, 0x6948, 0x6949, 0x694a, 0x694b,
	0x694c, 0x694d, 0x694e, 0x694f, 0x6950, 0x6951, 0x6952, 0x6953,
	0x6955, 0x6956, 0x6958, 0x6959, 0x695b, 0x695c, 0x695f, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9840[ 192 ] = {
	0x6961, 0x6962, 0x6964, 0x6965, 0x6967, 0x6968, 0x6969, 0x696a,
	0x696c, 0x696d, 0x696f, 0x6970, 0x6972, 0x6973, 0x6974, 0x6975,
	0x6976, 0x697a, 0x697b, 0x697d, 0x697e, 0x697f, 0x6981, 0x6983,
	0x6985, 0x698a, 0x698b, 0x698c, 0x698e, 0x698f, 0x6990, 0x6991,
	0x6992, 0x6993, 0x6996, 0x6997, 0x6999, 0x699a, 0x699d, 0x699e,
	0x699f, 0x69a0, 0x69a1, 0x69a2, 0x69a3, 0x69a4, 0x69a5, 0x69a6,
	0x69a9, 0x69aa, 0x69ac, 0x69ae, 0x69af, 0x69b0, 0x69b2, 0x69b3,
	0x69b5, 0x69b6, 0x69b8, 0x69b9, 0x69ba, 0x69bc, 0x69bd, 0xfffd,
	0x69be, 0x69bf, 0x69c0, 0x69c2, 0x69c3, 0x69c4, 0x69c5, 0x69c6,
	0x69c7, 0x69c8, 0x69c9, 0x69cb, 0x69cd, 0x69cf, 0x69d1, 0x69d2,
	0x69d3, 0x69d5, 0x69d6, 0x69d7, 0x69d8, 0x69d9, 0x69da, 0x69dc,
	0x69dd, 0x69de, 0x69e1, 0x69e2, 0x69e3, 0x69e4, 0x69e5, 0x69e6,
	0x69e7, 0x69e8, 0x69e9, 0x69ea, 0x69eb, 0x69ec, 0x69ee, 0x69ef,
	0x69f0, 0x69f1, 0x69f3, 0x69f4, 0x69f5, 0x69f6, 0x69f7, 0x69f8,
	0x69f9, 0x69fa, 0x69fb, 0x69fc, 0x69fe, 0x6a00, 0x6a01, 0x6a02,
	0x6a03, 0x6a04, 0x6a05, 0x6a06, 0x6a07, 0x6a08, 0x6a09, 0x6a0b,
	0x6a0c, 0x6a0d, 0x6a0e, 0x6a0f, 0x6a10, 0x6a11, 0x6a12, 0x6a13,
	0x6a14, 0x6a15, 0x6a16, 0x6a19, 0x6a1a, 0x6a1b, 0x6a1c, 0x6a1d,
	0x6a1e, 0x6a20, 0x6a22, 0x6a23, 0x6a24, 0x6a25, 0x6a26, 0x6a27,
	0x6a29, 0x6a2b, 0x6a2c, 0x6a2d, 0x6a2e, 0x6a30, 0x6a32, 0x6a33,
	0x6a34, 0x6a36, 0x6a37, 0x6a38, 0x6a39, 0x6a3a, 0x6a3b, 0x6a3c,
	0x6a3f, 0x6a40, 0x6a41, 0x6a42, 0x6a43, 0x6a45, 0x6a46, 0x6a48,
	0x6a49, 0x6a4a, 0x6a4b, 0x6a4c, 0x6a4d, 0x6a4e, 0x6a4f, 0x6a51,
	0x6a52, 0x6a53, 0x6a54, 0x6a55, 0x6a56, 0x6a57, 0x6a5a, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9940[ 192 ] = {
	0x6a5c, 0x6a5d, 0x6a5e, 0x6a5f, 0x6a60, 0x6a62, 0x6a63, 0x6a64,
	0x6a66, 0x6a67, 0x6a68, 0x6a69, 0x6a6a, 0x6a6b, 0x6a6c, 0x6a6d,
	0x6a6e, 0x6a6f, 0x6a70, 0x6a72, 0x6a73, 0x6a74, 0x6a75, 0x6a76,
	0x6a77, 0x6a78, 0x6a7a, 0x6a7b, 0x6a7d, 0x6a7e, 0x6a7f, 0x6a81,
	0x6a82, 0x6a83, 0x6a85, 0x6a86, 0x6a87, 0x6a88, 0x6a89, 0x6a8a,
	0x6a8b, 0x6a8c, 0x6a8d, 0x6a8f, 0x6a92, 0x6a93, 0x6a94, 0x6a95,
	0x6a96, 0x6a98, 0x6a99, 0x6a9a, 0x6a9b, 0x6a9c, 0x6a9d, 0x6a9e,
	0x6a9f, 0x6aa1, 0x6aa2, 0x6aa3, 0x6aa4, 0x6aa5, 0x6aa6, 0xfffd,
	0x6aa7, 0x6aa8, 0x6aaa, 0x6aad, 0x6aae, 0x6aaf, 0x6ab0, 0x6ab1,
	0x6ab2, 0x6ab3, 0x6ab4, 0x6ab5, 0x6ab6, 0x6ab7, 0x6ab8, 0x6ab9,
	0x6aba, 0x6abb, 0x6abc, 0x6abd, 0x6abe, 0x6abf, 0x6ac0, 0x6ac1,
	0x6ac2, 0x6ac3, 0x6ac4, 0x6ac5, 0x6ac6, 0x6ac7, 0x6ac8, 0x6ac9,
	0x6aca, 0x6acb, 0x6acc, 0x6acd, 0x6ace, 0x6acf, 0x6ad0, 0x6ad1,
	0x6ad2, 0x6ad3, 0x6ad4, 0x6ad5, 0x6ad6, 0x6ad7, 0x6ad8, 0x6ad9,
	0x6ada, 0x6adb, 0x6adc, 0x6add, 0x6ade, 0x6adf, 0x6ae0, 0x6ae1,
	0x6ae2, 0x6ae3, 0x6ae4, 0x6ae5, 0x6ae6, 0x6ae7, 0x6ae8, 0x6ae9,
	0x6aea, 0x6aeb, 0x6aec, 0x6aed, 0x6aee, 0x6aef, 0x6af0, 0x6af1,
	0x6af2, 0x6af3, 0x6af4, 0x6af5, 0x6af6, 0x6af7, 0x6af8, 0x6af9,
	0x6afa, 0x6afb, 0x6afc, 0x6afd, 0x6afe, 0x6aff, 0x6b00, 0x6b01,
	0x6b02, 0x6b03, 0x6b04, 0x6b05, 0x6b06, 0x6b07, 0x6b08, 0x6b09,
	0x6b0a, 0x6b0b, 0x6b0c, 0x6b0d, 0x6b0e, 0x6b0f, 0x6b10, 0x6b11,
	0x6b12, 0x6b13, 0x6b14, 0x6b15, 0x6b16, 0x6b17, 0x6b18, 0x6b19,
	0x6b1a, 0x6b1b, 0x6b1c, 0x6b1d, 0x6b1e, 0x6b1f, 0x6b25, 0x6b26,
	0x6b28, 0x6b29, 0x6b2a, 0x6b2b, 0x6b2c, 0x6b2d, 0x6b2e, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9a40[ 192 ] = {
	0x6b2f, 0x6b30, 0x6b31, 0x6b33, 0x6b34, 0x6b35, 0x6b36, 0x6b38,
	0x6b3b, 0x6b3c, 0x6b3d, 0x6b3f, 0x6b40, 0x6b41, 0x6b42, 0x6b44,
	0x6b45, 0x6b48, 0x6b4a, 0x6b4b, 0x6b4d, 0x6b4e, 0x6b4f, 0x6b50,
	0x6b51, 0x6b52, 0x6b53, 0x6b54, 0x6b55, 0x6b56, 0x6b57, 0x6b58,
	0x6b5a, 0x6b5b, 0x6b5c, 0x6b5d, 0x6b5e, 0x6b5f, 0x6b60, 0x6b61,
	0x6b68, 0x6b69, 0x6b6b, 0x6b6c, 0x6b6d, 0x6b6e, 0x6b6f, 0x6b70,
	0x6b71, 0x6b72, 0x6b73, 0x6b74, 0x6b75, 0x6b76, 0x6b77, 0x6b78,
	0x6b7a, 0x6b7d, 0x6b7e, 0x6b7f, 0x6b80, 0x6b85, 0x6b88, 0xfffd,
	0x6b8c, 0x6b8e, 0x6b8f, 0x6b90, 0x6b91, 0x6b94, 0x6b95, 0x6b97,
	0x6b98, 0x6b99, 0x6b9c, 0x6b9d, 0x6b9e, 0x6b9f, 0x6ba0, 0x6ba2,
	0x6ba3, 0x6ba4, 0x6ba5, 0x6ba6, 0x6ba7, 0x6ba8, 0x6ba9, 0x6bab,
	0x6bac, 0x6bad, 0x6bae, 0x6baf, 0x6bb0, 0x6bb1, 0x6bb2, 0x6bb6,
	0x6bb8, 0x6bb9, 0x6bba, 0x6bbb, 0x6bbc, 0x6bbd, 0x6bbe, 0x6bc0,
	0x6bc3, 0x6bc4, 0x6bc6, 0x6bc7, 0x6bc8, 0x6bc9, 0x6bca, 0x6bcc,
	0x6bce, 0x6bd0, 0x6bd1, 0x6bd8, 0x6bda, 0x6bdc, 0x6bdd, 0x6bde,
	0x6bdf, 0x6be0, 0x6be2, 0x6be3, 0x6be4, 0x6be5, 0x6be6, 0x6be7,
	0x6be8, 0x6be9, 0x6bec, 0x6bed, 0x6bee, 0x6bf0, 0x6bf1, 0x6bf2,
	0x6bf4, 0x6bf6, 0x6bf7, 0x6bf8, 0x6bfa, 0x6bfb, 0x6bfc, 0x6bfe,
	0x6bff, 0x6c00, 0x6c01, 0x6c02, 0x6c03, 0x6c04, 0x6c08, 0x6c09,
	0x6c0a, 0x6c0b, 0x6c0c, 0x6c0e, 0x6c12, 0x6c17, 0x6c1c, 0x6c1d,
	0x6c1e, 0x6c20, 0x6c23, 0x6c25, 0x6c2b, 0x6c2c, 0x6c2d, 0x6c31,
	0x6c33, 0x6c36, 0x6c37, 0x6c39, 0x6c3a, 0x6c3b, 0x6c3c, 0x6c3e,
	0x6c3f, 0x6c43, 0x6c44, 0x6c45, 0x6c48, 0x6c4b, 0x6c4c, 0x6c4d,
	0x6c4e, 0x6c4f, 0x6c51, 0x6c52, 0x6c53, 0x6c56, 0x6c58, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9b40[ 192 ] = {
	0x6c59, 0x6c5a, 0x6c62, 0x6c63, 0x6c65, 0x6c66, 0x6c67, 0x6c6b,
	0x6c6c, 0x6c6d, 0x6c6e, 0x6c6f, 0x6c71, 0x6c73, 0x6c75, 0x6c77,
	0x6c78, 0x6c7a, 0x6c7b, 0x6c7c, 0x6c7f, 0x6c80, 0x6c84, 0x6c87,
	0x6c8a, 0x6c8b, 0x6c8d, 0x6c8e, 0x6c91, 0x6c92, 0x6c95, 0x6c96,
	0x6c97, 0x6c98, 0x6c9a, 0x6c9c, 0x6c9d, 0x6c9e, 0x6ca0, 0x6ca2,
	0x6ca8, 0x6cac, 0x6caf, 0x6cb0, 0x6cb4, 0x6cb5, 0x6cb6, 0x6cb7,
	0x6cba, 0x6cc0, 0x6cc1, 0x6cc2, 0x6cc3, 0x6cc6, 0x6cc7, 0x6cc8,
	0x6ccb, 0x6ccd, 0x6cce, 0x6ccf, 0x6cd1, 0x6cd2, 0x6cd8, 0xfffd,
	0x6cd9, 0x6cda, 0x6cdc, 0x6cdd, 0x6cdf, 0x6ce4, 0x6ce6, 0x6ce7,
	0x6ce9, 0x6cec, 0x6ced, 0x6cf2, 0x6cf4, 0x6cf9, 0x6cff, 0x6d00,
	0x6d02, 0x6d03, 0x6d05, 0x6d06, 0x6d08, 0x6d09, 0x6d0a, 0x6d0d,
	0x6d0f, 0x6d10, 0x6d11, 0x6d13, 0x6d14, 0x6d15, 0x6d16, 0x6d18,
	0x6d1c, 0x6d1d, 0x6d1f, 0x6d20, 0x6d21, 0x6d22, 0x6d23, 0x6d24,
	0x6d26, 0x6d28, 0x6d29, 0x6d2c, 0x6d2d, 0x6d2f, 0x6d30, 0x6d34,
	0x6d36, 0x6d37, 0x6d38, 0x6d3a, 0x6d3f, 0x6d40, 0x6d42, 0x6d44,
	0x6d49, 0x6d4c, 0x6d50, 0x6d55, 0x6d56, 0x6d57, 0x6d58, 0x6d5b,
	0x6d5d, 0x6d5f, 0x6d61, 0x6d62, 0x6d64, 0x6d65, 0x6d67, 0x6d68,
	0x6d6b, 0x6d6c, 0x6d6d, 0x6d70, 0x6d71, 0x6d72, 0x6d73, 0x6d75,
	0x6d76, 0x6d79, 0x6d7a, 0x6d7b, 0x6d7d, 0x6d7e, 0x6d7f, 0x6d80,
	0x6d81, 0x6d83, 0x6d84, 0x6d86, 0x6d87, 0x6d8a, 0x6d8b, 0x6d8d,
	0x6d8f, 0x6d90, 0x6d92, 0x6d96, 0x6d97, 0x6d98, 0x6d99, 0x6d9a,
	0x6d9c, 0x6da2, 0x6da5, 0x6dac, 0x6dad, 0x6db0, 0x6db1, 0x6db3,
	0x6db4, 0x6db6, 0x6db7, 0x6db9, 0x6dba, 0x6dbb, 0x6dbc, 0x6dbd,
	0x6dbe, 0x6dc1, 0x6dc2, 0x6dc3, 0x6dc8, 0x6dc9, 0x6dca, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9c40[ 192 ] = {
	0x6dcd, 0x6dce, 0x6dcf, 0x6dd0, 0x6dd2, 0x6dd3, 0x6dd4, 0x6dd5,
	0x6dd7, 0x6dda, 0x6ddb, 0x6ddc, 0x6ddf, 0x6de2, 0x6de3, 0x6de5,
	0x6de7, 0x6de8, 0x6de9, 0x6dea, 0x6ded, 0x6def, 0x6df0, 0x6df2,
	0x6df4, 0x6df5, 0x6df6, 0x6df8, 0x6dfa, 0x6dfd, 0x6dfe, 0x6dff,
	0x6e00, 0x6e01, 0x6e02, 0x6e03, 0x6e04, 0x6e06, 0x6e07, 0x6e08,
	0x6e09, 0x6e0b, 0x6e0f, 0x6e12, 0x6e13, 0x6e15, 0x6e18, 0x6e19,
	0x6e1b, 0x6e1c, 0x6e1e, 0x6e1f, 0x6e22, 0x6e26, 0x6e27, 0x6e28,
	0x6e2a, 0x6e2c, 0x6e2e, 0x6e30, 0x6e31, 0x6e33, 0x6e35, 0xfffd,
	0x6e36, 0x6e37, 0x6e39, 0x6e3b, 0x6e3c, 0x6e3d, 0x6e3e, 0x6e3f,
	0x6e40, 0x6e41, 0x6e42, 0x6e45, 0x6e46, 0x6e47, 0x6e48, 0x6e49,
	0x6e4a, 0x6e4b, 0x6e4c, 0x6e4f, 0x6e50, 0x6e51, 0x6e52, 0x6e55,
	0x6e57, 0x6e59, 0x6e5a, 0x6e5c, 0x6e5d, 0x6e5e, 0x6e60, 0x6e61,
	0x6e62, 0x6e63, 0x6e64, 0x6e65, 0x6e66, 0x6e67, 0x6e68, 0x6e69,
	0x6e6a, 0x6e6c, 0x6e6d, 0x6e6f, 0x6e70, 0x6e71, 0x6e72, 0x6e73,
	0x6e74, 0x6e75, 0x6e76, 0x6e77, 0x6e78, 0x6e79, 0x6e7a, 0x6e7b,
	0x6e7c, 0x6e7d, 0x6e80, 0x6e81, 0x6e82, 0x6e84, 0x6e87, 0x6e88,
	0x6e8a, 0x6e8b, 0x6e8c, 0x6e8d, 0x6e8e, 0x6e91, 0x6e92, 0x6e93,
	0x6e94, 0x6e95, 0x6e96, 0x6e97, 0x6e99, 0x6e9a, 0x6e9b, 0x6e9d,
	0x6e9e, 0x6ea0, 0x6ea1, 0x6ea3, 0x6ea4, 0x6ea6, 0x6ea8, 0x6ea9,
	0x6eab, 0x6eac, 0x6ead, 0x6eae, 0x6eb0, 0x6eb3, 0x6eb5, 0x6eb8,
	0x6eb9, 0x6ebc, 0x6ebe, 0x6ebf, 0x6ec0, 0x6ec3, 0x6ec4, 0x6ec5,
	0x6ec6, 0x6ec8, 0x6ec9, 0x6eca, 0x6ecc, 0x6ecd, 0x6ece, 0x6ed0,
	0x6ed2, 0x6ed6, 0x6ed8, 0x6ed9, 0x6edb, 0x6edc, 0x6edd, 0x6ee3,
	0x6ee7, 0x6eea, 0x6eeb, 0x6eec, 0x6eed, 0x6eee, 0x6eef, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9d40[ 192 ] = {
	0x6ef0, 0x6ef1, 0x6ef2, 0x6ef3, 0x6ef5, 0x6ef6, 0x6ef7, 0x6ef8,
	0x6efa, 0x6efb, 0x6efc, 0x6efd, 0x6efe, 0x6eff, 0x6f00, 0x6f01,
	0x6f03, 0x6f04, 0x6f05, 0x6f07, 0x6f08, 0x6f0a, 0x6f0b, 0x6f0c,
	0x6f0d, 0x6f0e, 0x6f10, 0x6f11, 0x6f12, 0x6f16, 0x6f17, 0x6f18,
	0x6f19, 0x6f1a, 0x6f1b, 0x6f1c, 0x6f1d, 0x6f1e, 0x6f1f, 0x6f21,
	0x6f22, 0x6f23, 0x6f25, 0x6f26, 0x6f27, 0x6f28, 0x6f2c, 0x6f2e,
	0x6f30, 0x6f32, 0x6f34, 0x6f35, 0x6f37, 0x6f38, 0x6f39, 0x6f3a,
	0x6f3b, 0x6f3c, 0x6f3d, 0x6f3f, 0x6f40, 0x6f41, 0x6f42, 0xfffd,
	0x6f43, 0x6f44, 0x6f45, 0x6f48, 0x6f49, 0x6f4a, 0x6f4c, 0x6f4e,
	0x6f4f, 0x6f50, 0x6f51, 0x6f52, 0x6f53, 0x6f54, 0x6f55, 0x6f56,
	0x6f57, 0x6f59, 0x6f5a, 0x6f5b, 0x6f5d, 0x6f5f, 0x6f60, 0x6f61,
	0x6f63, 0x6f64, 0x6f65, 0x6f67, 0x6f68, 0x6f69, 0x6f6a, 0x6f6b,
	0x6f6c, 0x6f6f, 0x6f70, 0x6f71, 0x6f73, 0x6f75, 0x6f76, 0x6f77,
	0x6f79, 0x6f7b, 0x6f7d, 0x6f7e, 0x6f7f, 0x6f80, 0x6f81, 0x6f82,
	0x6f83, 0x6f85, 0x6f86, 0x6f87, 0x6f8a, 0x6f8b, 0x6f8f, 0x6f90,
	0x6f91, 0x6f92, 0x6f93, 0x6f94, 0x6f95, 0x6f96, 0x6f97, 0x6f98,
	0x6f99, 0x6f9a, 0x6f9b, 0x6f9d, 0x6f9e, 0x6f9f, 0x6fa0, 0x6fa2,
	0x6fa3, 0x6fa4, 0x6fa5, 0x6fa6, 0x6fa8, 0x6fa9, 0x6faa, 0x6fab,
	0x6fac, 0x6fad, 0x6fae, 0x6faf, 0x6fb0, 0x6fb1, 0x6fb2, 0x6fb4,
	0x6fb5, 0x6fb7, 0x6fb8, 0x6fba, 0x6fbb, 0x6fbc, 0x6fbd, 0x6fbe,
	0x6fbf, 0x6fc1, 0x6fc3, 0x6fc4, 0x6fc5, 0x6fc6, 0x6fc7, 0x6fc8,
	0x6fca, 0x6fcb, 0x6fcc, 0x6fcd, 0x6fce, 0x6fcf, 0x6fd0, 0x6fd3,
	0x6fd4, 0x6fd5, 0x6fd6, 0x6fd7, 0x6fd8, 0x6fd9, 0x6fda, 0x6fdb,
	0x6fdc, 0x6fdd, 0x6fdf, 0x6fe2, 0x6fe3, 0x6fe4, 0x6fe5, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9e40[ 192 ] = {
	0x6fe6, 0x6fe7, 0x6fe8, 0x6fe9, 0x6fea, 0x6feb, 0x6fec, 0x6fed,
	0x6ff0, 0x6ff1, 0x6ff2, 0x6ff3, 0x6ff4, 0x6ff5, 0x6ff6, 0x6ff7,
	0x6ff8, 0x6ff9, 0x6ffa, 0x6ffb, 0x6ffc, 0x6ffd, 0x6ffe, 0x6fff,
	0x7000, 0x7001, 0x7002, 0x7003, 0x7004, 0x7005, 0x7006, 0x7007,
	0x7008, 0x7009, 0x700a, 0x700b, 0x700c, 0x700d, 0x700e, 0x700f,
	0x7010, 0x7012, 0x7013, 0x7014, 0x7015, 0x7016, 0x7017, 0x7018,
	0x7019, 0x701c, 0x701d, 0x701e, 0x701f, 0x7020, 0x7021, 0x7022,
	0x7024, 0x7025, 0x7026, 0x7027, 0x7028, 0x7029, 0x702a, 0xfffd,
	0x702b, 0x702c, 0x702d, 0x702e, 0x702f, 0x7030, 0x7031, 0x7032,
	0x7033, 0x7034, 0x7036, 0x7037, 0x7038, 0x703a, 0x703b, 0x703c,
	0x703d, 0x703e, 0x703f, 0x7040, 0x7041, 0x7042, 0x7043, 0x7044,
	0x7045, 0x7046, 0x7047, 0x7048, 0x7049, 0x704a, 0x704b, 0x704d,
	0x704e, 0x7050, 0x7051, 0x7052, 0x7053, 0x7054, 0x7055, 0x7056,
	0x7057, 0x7058, 0x7059, 0x705a, 0x705b, 0x705c, 0x705d, 0x705f,
	0x7060, 0x7061, 0x7062, 0x7063, 0x7064, 0x7065, 0x7066, 0x7067,
	0x7068, 0x7069, 0x706a, 0x706e, 0x7071, 0x7072, 0x7073, 0x7074,
	0x7077, 0x7079, 0x707a, 0x707b, 0x707d, 0x7081, 0x7082, 0x7083,
	0x7084, 0x7086, 0x7087, 0x7088, 0x708b, 0x708c, 0x708d, 0x708f,
	0x7090, 0x7091, 0x7093, 0x7097, 0x7098, 0x709a, 0x709b, 0x709e,
	0x709f, 0x70a0, 0x70a1, 0x70a2, 0x70a3, 0x70a4, 0x70a5, 0x70a6,
	0x70a7, 0x70a8, 0x70a9, 0x70aa, 0x70b0, 0x70b2, 0x70b4, 0x70b5,
	0x70b6, 0x70ba, 0x70be, 0x70bf, 0x70c4, 0x70c5, 0x70c6, 0x70c7,
	0x70c9, 0x70cb, 0x70cc, 0x70cd, 0x70ce, 0x70cf, 0x70d0, 0x70d1,
	0x70d2, 0x70d3, 0x70d4, 0x70d5, 0x70d6, 0x70d7, 0x70da, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9f40[ 192 ] = {
	0x70dc, 0x70dd, 0x70de, 0x70e0, 0x70e1, 0x70e2, 0x70e3, 0x70e5,
	0x70ea, 0x70ee, 0x70f0, 0x70f1, 0x70f2, 0x70f3, 0x70f4, 0x70f5,
	0x70f6, 0x70f8, 0x70fa, 0x70fb, 0x70fc, 0x70fe, 0x70ff, 0x7100,
	0x7101, 0x7102, 0x7103, 0x7104, 0x7105, 0x7106, 0x7107, 0x7108,
	0x710b, 0x710c, 0x710d, 0x710e, 0x710f, 0x7111, 0x7112, 0x7114,
	0x7117, 0x711b, 0x711c, 0x711d, 0x711e, 0x711f, 0x7120, 0x7121,
	0x7122, 0x7123, 0x7124, 0x7125, 0x7127, 0x7128, 0x7129, 0x712a,
	0x712b, 0x712c, 0x712d, 0x712e, 0x7132, 0x7133, 0x7134, 0xfffd,
	0x7135, 0x7137, 0x7138, 0x7139, 0x713a, 0x713b, 0x713c, 0x713d,
	0x713e, 0x713f, 0x7140, 0x7141, 0x7142, 0x7143, 0x7144, 0x7146,
	0x7147, 0x7148, 0x7149, 0x714b, 0x714d, 0x714f, 0x7150, 0x7151,
	0x7152, 0x7153, 0x7154, 0x7155, 0x7156, 0x7157, 0x7158, 0x7159,
	0x715a, 0x715b, 0x715d, 0x715f, 0x7160, 0x7161, 0x7162, 0x7163,
	0x7165, 0x7169, 0x716a, 0x716b, 0x716c, 0x716d, 0x716f, 0x7170,
	0x7171, 0x7174, 0x7175, 0x7176, 0x7177, 0x7179, 0x717b, 0x717c,
	0x717e, 0x717f, 0x7180, 0x7181, 0x7182, 0x7183, 0x7185, 0x7186,
	0x7187, 0x7188, 0x7189, 0x718b, 0x718c, 0x718d, 0x718e, 0x7190,
	0x7191, 0x7192, 0x7193, 0x7195, 0x7196, 0x7197, 0x719a, 0x719b,
	0x719c, 0x719d, 0x719e, 0x71a1, 0x71a2, 0x71a3, 0x71a4, 0x71a5,
	0x71a6, 0x71a7, 0x71a9, 0x71aa, 0x71ab, 0x71ad, 0x71ae, 0x71af,
	0x71b0, 0x71b1, 0x71b2, 0x71b4, 0x71b6, 0x71b7, 0x71b8, 0x71ba,
	0x71bb, 0x71bc, 0x71bd, 0x71be, 0x71bf, 0x71c0, 0x71c1, 0x71c2,
	0x71c4, 0x71c5, 0x71c6, 0x71c7, 0x71c8, 0x71c9, 0x71ca, 0x71cb,
	0x71cc, 0x71cd, 0x71cf, 0x71d0, 0x71d1, 0x71d2, 0x71d3, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa040[ 192 ] = {
	0x71d6, 0x71d7, 0x71d8, 0x71d9, 0x71da, 0x71db, 0x71dc, 0x71dd,
	0x71de, 0x71df, 0x71e1, 0x71e2, 0x71e3, 0x71e4, 0x71e6, 0x71e8,
	0x71e9, 0x71ea, 0x71eb, 0x71ec, 0x71ed, 0x71ef, 0x71f0, 0x71f1,
	0x71f2, 0x71f3, 0x71f4, 0x71f5, 0x71f6, 0x71f7, 0x71f8, 0x71fa,
	0x71fb, 0x71fc, 0x71fd, 0x71fe, 0x71ff, 0x7200, 0x7201, 0x7202,
	0x7203, 0x7204, 0x7205, 0x7207, 0x7208, 0x7209, 0x720a, 0x720b,
	0x720c, 0x720d, 0x720e, 0x720f, 0x7210, 0x7211, 0x7212, 0x7213,
	0x7214, 0x7215, 0x7216, 0x7217, 0x7218, 0x7219, 0x721a, 0xfffd,
	0x721b, 0x721c, 0x721e, 0x721f, 0x7220, 0x7221, 0x7222, 0x7223,
	0x7224, 0x7225, 0x7226, 0x7227, 0x7229, 0x722b, 0x722d, 0x722e,
	0x722f, 0x7232, 0x7233, 0x7234, 0x723a, 0x723c, 0x723e, 0x7240,
	0x7241, 0x7242, 0x7243, 0x7244, 0x7245, 0x7246, 0x7249, 0x724a,
	0x724b, 0x724e, 0x724f, 0x7250, 0x7251, 0x7253, 0x7254, 0x7255,
	0x7257, 0x7258, 0x725a, 0x725c, 0x725e, 0x7260, 0x7263, 0x7264,
	0x7265, 0x7268, 0x726a, 0x726b, 0x726c, 0x726d, 0x7270, 0x7271,
	0x7273, 0x7274, 0x7276, 0x7277, 0x7278, 0x727b, 0x727c, 0x727d,
	0x7282, 0x7283, 0x7285, 0x7286, 0x7287, 0x7288, 0x7289, 0x728c,
	0x728e, 0x7290, 0x7291, 0x7293, 0x7294, 0x7295, 0x7296, 0x7297,
	0x7298, 0x7299, 0x729a, 0x729b, 0x729c, 0x729d, 0x729e, 0x72a0,
	0x72a1, 0x72a2, 0x72a3, 0x72a4, 0x72a5, 0x72a6, 0x72a7, 0x72a8,
	0x72a9, 0x72aa, 0x72ab, 0x72ae, 0x72b1, 0x72b2, 0x72b3, 0x72b5,
	0x72ba, 0x72bb, 0x72bc, 0x72bd, 0x72be, 0x72bf, 0x72c0, 0x72c5,
	0x72c6, 0x72c7, 0x72c9, 0x72ca, 0x72cb, 0x72cc, 0x72cf, 0x72d1,
	0x72d3, 0x72d4, 0x72d5, 0x72d6, 0x72d8, 0x72da, 0x72db, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa1a0[ 96 ] = {
	0xfffd, 0x3000, 0x3001, 0x3002, 0x00b7, 0x02c9, 0x02c7, 0x00a8,
	0x3003, 0x3005, 0x2014, 0xff5e, 0x2016, 0x2026, 0x2018, 0x2019,
	0x201c, 0x201d, 0x3014, 0x3015, 0x3008, 0x3009, 0x300a, 0x300b,
	0x300c, 0x300d, 0x300e, 0x300f, 0x3016, 0x3017, 0x3010, 0x3011,
	0x00b1, 0x00d7, 0x00f7, 0x2236, 0x2227, 0x2228, 0x2211, 0x220f,
	0x222a, 0x2229, 0x2208, 0x2237, 0x221a, 0x22a5, 0x2225, 0x2220,
	0x2312, 0x2299, 0x222b, 0x222e, 0x2261, 0x224c, 0x2248, 0x223d,
	0x221d, 0x2260, 0x226e, 0x226f, 0x2264, 0x2265, 0x221e, 0x2235,
	0x2234, 0x2642, 0x2640, 0x00b0, 0x2032, 0x2033, 0x2103, 0xff04,
	0x00a4, 0xffe0, 0xffe1, 0x2030, 0x00a7, 0x2116, 0x2606, 0x2605,
	0x25cb, 0x25cf, 0x25ce, 0x25c7, 0x25c6, 0x25a1, 0x25a0, 0x25b3,
	0x25b2, 0x203b, 0x2192, 0x2190, 0x2191, 0x2193, 0x3013, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa2a0[ 96 ] = {
	0xfffd, 0x2170, 0x2171, 0x2172, 0x2173, 0x2174, 0x2175, 0x2176,
	0x2177, 0x2178, 0x2179, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0x2488, 0x2489, 0x248a, 0x248b, 0x248c, 0x248d, 0x248e,
	0x248f, 0x2490, 0x2491, 0x2492, 0x2493, 0x2494, 0x2495, 0x2496,
	0x2497, 0x2498, 0x2499, 0x249a, 0x249b, 0x2474, 0x2475, 0x2476,
	0x2477, 0x2478, 0x2479, 0x247a, 0x247b, 0x247c, 0x247d, 0x247e,
	0x247f, 0x2480, 0x2481, 0x2482, 0x2483, 0x2484, 0x2485, 0x2486,
	0x2487, 0x2460, 0x2461, 0x2462, 0x2463, 0x2464, 0x2465, 0x2466,
	0x2467, 0x2468, 0x2469, 0xfffd, 0xfffd, 0x3220, 0x3221, 0x3222,
	0x3223, 0x3224, 0x3225, 0x3226, 0x3227, 0x3228, 0x3229, 0xfffd,
	0xfffd, 0x2160, 0x2161, 0x2162, 0x2163, 0x2164, 0x2165, 0x2166,
	0x2167, 0x2168, 0x2169, 0x216a, 0x216b, 0xfffd, 0xfffd, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa3a0[ 96 ] = {
	0xfffd, 0xff01, 0xff02, 0xff03, 0xffe5, 0xff05, 0xff06, 0xff07,
	0xff08, 0xff09, 0xff0a, 0xff0b, 0xff0c, 0xff0d, 0xff0e, 0xff0f,
	0xff10, 0xff11, 0xff12, 0xff13, 0xff14, 0xff15, 0xff16, 0xff17,
	0xff18, 0xff19, 0xff1a, 0xff1b, 0xff1c, 0xff1d, 0xff1e, 0xff1f,
	0xff20, 0xff21, 0xff22, 0xff23, 0xff24, 0xff25, 0xff26, 0xff27,
	0xff28, 0xff29, 0xff2a, 0xff2b, 0xff2c, 0xff2d, 0xff2e, 0xff2f,
	0xff30, 0xff31, 0xff32, 0xff33, 0xff34, 0xff35, 0xff36, 0xff37,
	0xff38, 0xff39, 0xff3a, 0xff3b, 0xff3c, 0xff3d, 0xff3e, 0xff3f,
	0xff40, 0xff41, 0xff42, 0xff43, 0xff44, 0xff45, 0xff46, 0xff47,
	0xff48, 0xff49, 0xff4a, 0xff4b, 0xff4c, 0xff4d, 0xff4e, 0xff4f,
	0xff50, 0xff51, 0xff52, 0xff53, 0xff54, 0xff55, 0xff56, 0xff57,
	0xff58, 0xff59, 0xff5a, 0xff5b, 0xff5c, 0xff5d, 0xffe3, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa4a0[ 88 ] = {
	0xfffd, 0x3041, 0x3042, 0x3043, 0x3044, 0x3045, 0x3046, 0x3047,
	0x3048, 0x3049, 0x304a, 0x304b, 0x304c, 0x304d, 0x304e, 0x304f,
	0x3050, 0x3051, 0x3052, 0x3053, 0x3054, 0x3055, 0x3056, 0x3057,
	0x3058, 0x3059, 0x305a, 0x305b, 0x305c, 0x305d, 0x305e, 0x305f,
	0x3060, 0x3061, 0x3062, 0x3063, 0x3064, 0x3065, 0x3066, 0x3067,
	0x3068, 0x3069, 0x306a, 0x306b, 0x306c, 0x306d, 0x306e, 0x306f,
	0x3070, 0x3071, 0x3072, 0x3073, 0x3074, 0x3075, 0x3076, 0x3077,
	0x3078, 0x3079, 0x307a, 0x307b, 0x307c, 0x307d, 0x307e, 0x307f,
	0x3080, 0x3081, 0x3082, 0x3083, 0x3084, 0x3085, 0x3086, 0x3087,
	0x3088, 0x3089, 0x308a, 0x308b, 0x308c, 0x308d, 0x308e, 0x308f,
	0x3090, 0x3091, 0x3092, 0x3093, 0xfffd, 0xfffd, 0xfffd, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa5a0[ 88 ] = {
	0xfffd, 0x30a1, 0x30a2, 0x30a3, 0x30a4, 0x30a5, 0x30a6, 0x30a7,
	0x30a8, 0x30a9, 0x30aa, 0x30ab, 0x30ac, 0x30ad, 0x30ae, 0x30af,
	0x30b0, 0x30b1, 0x30b2, 0x30b3, 0x30b4, 0x30b5, 0x30b6, 0x30b7,
	0x30b8, 0x30b9, 0x30ba, 0x30bb, 0x30bc, 0x30bd, 0x30be, 0x30bf,
	0x30c0, 0x30c1, 0x30c2, 0x30c3, 0x30c4, 0x30c5, 0x30c6, 0x30c7,
	0x30c8, 0x30c9, 0x30ca, 0x30cb, 0x30cc, 0x30cd, 0x30ce, 0x30cf,
	0x30d0, 0x30d1, 0x30d2, 0x30d3, 0x30d4, 0x30d5, 0x30d6, 0x30d7,
	0x30d8, 0x30d9, 0x30da, 0x30db, 0x30dc, 0x30dd, 0x30de, 0x30df,
	0x30e0, 0x30e1, 0x30e2, 0x30e3, 0x30e4, 0x30e5, 0x30e6, 0x30e7,
	0x30e8, 0x30e9, 0x30ea, 0x30eb, 0x30ec, 0x30ed, 0x30ee, 0x30ef,
	0x30f0, 0x30f1, 0x30f2, 0x30f3, 0x30f4, 0x30f5, 0x30f6, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa6a0[ 88 ] = {
	0xfffd, 0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 0x0397,
	0x0398, 0x0399, 0x039a, 0x039b, 0x039c, 0x039d, 0x039e, 0x039f,
	0x03a0, 0x03a1, 0x03a3, 0x03a4, 0x03a5, 0x03a6, 0x03a7, 0x03a8,
	0x03a9, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0x03b1, 0x03b2, 0x03b3, 0x03b4, 0x03b5, 0x03b6, 0x03b7,
	0x03b8, 0x03b9, 0x03ba, 0x03bb, 0x03bc, 0x03bd, 0x03be, 0x03bf,
	0x03c0, 0x03c1, 0x03c3, 0x03c4, 0x03c5, 0x03c6, 0x03c7, 0x03c8,
	0x03c9, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfe35, 0xfe36, 0xfe39, 0xfe3a, 0xfe3f, 0xfe40, 0xfe3d, 0xfe3e,
	0xfe41, 0xfe42, 0xfe43, 0xfe44, 0xfffd, 0xfffd, 0xfe3b, 0xfe3c,
	0xfe37, 0xfe38, 0xfe31, 0xfffd, 0xfe33, 0xfe34, 0xfffd, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa7a0[ 40 ] = {
	0xfffd, 0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0401,
	0x0416, 0x0417, 0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d,
	0x041e, 0x041f, 0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425,
	0x0426, 0x0427, 0x0428, 0x0429, 0x042a, 0x042b, 0x042c, 0x042d,
	0x042e, 0x042f, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa7d0[ 40 ] = {
	0xfffd, 0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0451,
	0x0436, 0x0437, 0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d,
	0x043e, 0x043f, 0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445,
	0x0446, 0x0447, 0x0448, 0x0449, 0x044a, 0x044b, 0x044c, 0x044d,
	0x044e, 0x044f, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa840[ 88 ] = {
	0x02ca, 0x02cb, 0x02d9, 0x2013, 0x2015, 0x2025, 0x2035, 0x2105,
	0x2109, 0x2196, 0x2197, 0x2198, 0x2199, 0x2215, 0x221f, 0x2223,
	0x2252, 0x2266, 0x2267, 0x22bf, 0x2550, 0x2551, 0x2552, 0x2553,
	0x2554, 0x2555, 0x2556, 0x2557, 0x2558, 0x2559, 0x255a, 0x255b,
	0x255c, 0x255d, 0x255e, 0x255f, 0x2560, 0x2561, 0x2562, 0x2563,
	0x2564, 0x2565, 0x2566, 0x2567, 0x2568, 0x2569, 0x256a, 0x256b,
	0x256c, 0x256d, 0x256e, 0x256f, 0x2570, 0x2571, 0x2572, 0x2573,
	0x2581, 0x2582, 0x2583, 0x2584, 0x2585, 0x2586, 0x2587, 0xfffd,
	0x2588, 0x2589, 0x258a, 0x258b, 0x258c, 0x258d, 0x258e, 0x258f,
	0x2593, 0x2594, 0x2595, 0x25bc, 0x25bd, 0x25e2, 0x25e3, 0x25e4,
	0x25e5, 0x2609, 0x2295, 0x3012, 0x301d, 0x301e, 0xfffd, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa8a0[ 80 ] = {
	0xfffd, 0x0101, 0x00e1, 0x01ce, 0x00e0, 0x0113, 0x00e9, 0x011b,
	0x00e8, 0x012b, 0x00ed, 0x01d0, 0x00ec, 0x014d, 0x00f3, 0x01d2,
	0x00f2, 0x016b, 0x00fa, 0x01d4, 0x00f9, 0x01d6, 0x01d8, 0x01da,
	0x01dc, 0x00fc, 0x00ea, 0x0251, 0xfffd, 0x0144, 0x0148, 0xfffd,
	0x0261, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0x3105, 0x3106, 0x3107,
	0x3108, 0x3109, 0x310a, 0x310b, 0x310c, 0x310d, 0x310e, 0x310f,
	0x3110, 0x3111, 0x3112, 0x3113, 0x3114, 0x3115, 0x3116, 0x3117,
	0x3118, 0x3119, 0x311a, 0x311b, 0x311c, 0x311d, 0x311e, 0x311f,
	0x3120, 0x3121, 0x3122, 0x3123, 0x3124, 0x3125, 0x3126, 0x3127,
	0x3128, 0x3129, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa940[ 72 ] = {
	0x3021, 0x3022, 0x3023, 0x3024, 0x3025, 0x3026, 0x3027, 0x3028,
	0x3029, 0x32a3, 0x338e, 0x338f, 0x339c, 0x339d, 0x339e, 0x33a1,
	0x33c4, 0x33ce, 0x33d1, 0x33d2, 0x33d5, 0xfe30, 0xffe2, 0xffe4,
	0xfffd, 0x2121, 0x3231, 0xfffd, 0x2010, 0xfffd, 0xfffd, 0xfffd,
	0x30fc, 0x309b, 0x309c, 0x30fd, 0x30fe, 0x3006, 0x309d, 0x309e,
	0xfe49, 0xfe4a, 0xfe4b, 0xfe4c, 0xfe4d, 0xfe4e, 0xfe4f, 0xfe50,
	0xfe51, 0xfe52, 0xfe54, 0xfe55, 0xfe56, 0xfe57, 0xfe59, 0xfe5a,
	0xfe5b, 0xfe5c, 0xfe5d, 0xfe5e, 0xfe5f, 0xfe60, 0xfe61, 0xfffd,
	0xfe62, 0xfe63, 0xfe64, 0xfe65, 0xfe66, 0xfe68, 0xfe69, 0xfe6a
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa9a0[ 80 ] = {
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0x2500, 0x2501, 0x2502, 0x2503,
	0x2504, 0x2505, 0x2506, 0x2507, 0x2508, 0x2509, 0x250a, 0x250b,
	0x250c, 0x250d, 0x250e, 0x250f, 0x2510, 0x2511, 0x2512, 0x2513,
	0x2514, 0x2515, 0x2516, 0x2517, 0x2518, 0x2519, 0x251a, 0x251b,
	0x251c, 0x251d, 0x251e, 0x251f, 0x2520, 0x2521, 0x2522, 0x2523,
	0x2524, 0x2525, 0x2526, 0x2527, 0x2528, 0x2529, 0x252a, 0x252b,
	0x252c, 0x252d, 0x252e, 0x252f, 0x2530, 0x2531, 0x2532, 0x2533,
	0x2534, 0x2535, 0x2536, 0x2537, 0x2538, 0x2539, 0x253a, 0x253b,
	0x253c, 0x253d, 0x253e, 0x253f, 0x2540, 0x2541, 0x2542, 0x2543,
	0x2544, 0x2545, 0x2546, 0x2547, 0x2548, 0x2549, 0x254a, 0x254b
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xaa40[ 96 ] = {
	0x72dc, 0x72dd, 0x72df, 0x72e2, 0x72e3, 0x72e4, 0x72e5, 0x72e6,
	0x72e7, 0x72ea, 0x72eb, 0x72f5, 0x72f6, 0x72f9, 0x72fd, 0x72fe,
	0x72ff, 0x7300, 0x7302, 0x7304, 0x7305, 0x7306, 0x7307, 0x7308,
	0x7309, 0x730b, 0x730c, 0x730d, 0x730f, 0x7310, 0x7311, 0x7312,
	0x7314, 0x7318, 0x7319, 0x731a, 0x731f, 0x7320, 0x7323, 0x7324,
	0x7326, 0x7327, 0x7328, 0x732d, 0x732f, 0x7330, 0x7332, 0x7333,
	0x7335, 0x7336, 0x733a, 0x733b, 0x733c, 0x733d, 0x7340, 0x7341,
	0x7342, 0x7343, 0x7344, 0x7345, 0x7346, 0x7347, 0x7348, 0xfffd,
	0x7349, 0x734a, 0x734b, 0x734c, 0x734e, 0x734f, 0x7351, 0x7353,
	0x7354, 0x7355, 0x7356, 0x7358, 0x7359, 0x735a, 0x735b, 0x735c,
	0x735d, 0x735e, 0x735f, 0x7361, 0x7362, 0x7363, 0x7364, 0x7365,
	0x7366, 0x7367, 0x7368, 0x7369, 0x736a, 0x736b, 0x736e, 0x7370
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xab40[ 96 ] = {
	0x7372, 0x7373, 0x7374, 0x7375, 0x7376, 0x7377, 0x7378, 0x7379,
	0x737a, 0x737b, 0x737c, 0x737d, 0x737f, 0x7380, 0x7381, 0x7382,
	0x7383, 0x7385, 0x7386, 0x7388, 0x738a, 0x738c, 0x738d, 0x738f,
	0x7390, 0x7392, 0x7393, 0x7394, 0x7395, 0x7397, 0x7398, 0x7399,
	0x739a, 0x739c, 0x739d, 0x739e, 0x73a0, 0x73a1, 0x73a3, 0x73a4,
	0x73a5, 0x73a6, 0x73a7, 0x73a8, 0x73aa, 0x73ac, 0x73ad, 0x73b1,
	0x73b4, 0x73b5, 0x73b6, 0x73b8, 0x73b9, 0x73bc, 0x73bd, 0x73be,
	0x73bf, 0x73c1, 0x73c3, 0x73c4, 0x73c5, 0x73c6, 0x73c7, 0xfffd,
	0x73cb, 0x73cc, 0x73ce, 0x73d2, 0x73d3, 0x73d4, 0x73d5, 0x73d6,
	0x73d7, 0x73d8, 0x73da, 0x73db, 0x73dc, 0x73dd, 0x73df, 0x73e1,
	0x73e2, 0x73e3, 0x73e4, 0x73e6, 0x73e8, 0x73ea, 0x73eb, 0x73ec,
	0x73ee, 0x73ef, 0x73f0, 0x73f1, 0x73f3, 0x73f4, 0x73f5, 0x73f6
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xac40[ 96 ] = {
	0x73f8, 0x73f9, 0x73fa, 0x73fb, 0x73fc, 0x73fd, 0x73fe, 0x73ff,
	0x7400, 0x7401, 0x7402, 0x7404, 0x7407, 0x7408, 0x740b, 0x740c,
	0x740d, 0x740e, 0x7411, 0x7412, 0x7413, 0x7414, 0x7415, 0x7416,
	0x7417, 0x7418, 0x7419, 0x741c, 0x741d, 0x741e, 0x741f, 0x7420,
	0x7421, 0x7423, 0x7424, 0x7427, 0x7429, 0x742b, 0x742d, 0x742f,
	0x7431, 0x7432, 0x7437, 0x7438, 0x7439, 0x743a, 0x743b, 0x743d,
	0x743e, 0x743f, 0x7440, 0x7442, 0x7443, 0x7444, 0x7445, 0x7446,
	0x7447, 0x7448, 0x7449, 0x744a, 0x744b, 0x744c, 0x744d, 0xfffd,
	0x744e, 0x744f, 0x7450, 0x7451, 0x7452, 0x7453, 0x7454, 0x7456,
	0x7458, 0x745d, 0x7460, 0x7461, 0x7462, 0x7463, 0x7464, 0x7465,
	0x7466, 0x7467, 0x7468, 0x7469, 0x746a, 0x746b, 0x746c, 0x746e,
	0x746f, 0x7471, 0x7472, 0x7473, 0x7474, 0x7475, 0x7478, 0x7479
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xad40[ 96 ] = {
	0x747b, 0x747c, 0x747d, 0x747f, 0x7482, 0x7484, 0x7485, 0x7486,
	0x7488, 0x7489, 0x748a, 0x748c, 0x748d, 0x748f, 0x7491, 0x7492,
	0x7493, 0x7494, 0x7495, 0x7496, 0x7497, 0x7498, 0x7499, 0x749a,
	0x749b, 0x749d, 0x749f, 0x74a0, 0x74a1, 0x74a2, 0x74a3, 0x74a4,
	0x74a5, 0x74a6, 0x74aa, 0x74ab, 0x74ac, 0x74ad, 0x74ae, 0x74af,
	0x74b0, 0x74b1, 0x74b2, 0x74b3, 0x74b4, 0x74b5, 0x74b6, 0x74b7,
	0x74b8, 0x74b9, 0x74bb, 0x74bc, 0x74bd, 0x74be, 0x74bf, 0x74c0,
	0x74c1, 0x74c2, 0x74c3, 0x74c4, 0x74c5, 0x74c6, 0x74c7, 0xfffd,
	0x74c8, 0x74c9, 0x74ca, 0x74cb, 0x74cc, 0x74cd, 0x74ce, 0x74cf,
	0x74d0, 0x74d1, 0x74d3, 0x74d4, 0x74d5, 0x74d6, 0x74d7, 0x74d8,
	0x74d9, 0x74da, 0x74db, 0x74dd, 0x74df, 0x74e1, 0x74e5, 0x74e7,
	0x74e8, 0x74e9, 0x74ea, 0x74eb, 0x74ec, 0x74ed, 0x74f0, 0x74f1
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xae40[ 96 ] = {
	0x74f3, 0x74f5, 0x74f8, 0x74f9, 0x74fa, 0x74fb, 0x74fc, 0x74fd,
	0x74fe, 0x7500, 0x7501, 0x7502, 0x7503, 0x7505, 0x7506, 0x7507,
	0x7508, 0x7509, 0x750a, 0x750b, 0x750c, 0x750e, 0x7510, 0x7512,
	0x7514, 0x7515, 0x7516, 0x7517, 0x751b, 0x751d, 0x751e, 0x7520,
	0x7521, 0x7522, 0x7523, 0x7524, 0x7526, 0x7527, 0x752a, 0x752e,
	0x7534, 0x7536, 0x7539, 0x753c, 0x753d, 0x753f, 0x7541, 0x7542,
	0x7543, 0x7544, 0x7546, 0x7547, 0x7549, 0x754a, 0x754d, 0x7550,
	0x7551, 0x7552, 0x7553, 0x7555, 0x7556, 0x7557, 0x7558, 0xfffd,
	0x755d, 0x755e, 0x755f, 0x7560, 0x7561, 0x7562, 0x7563, 0x7564,
	0x7567, 0x7568, 0x7569, 0x756b, 0x756c, 0x756d, 0x756e, 0x756f,
	0x7570, 0x7571, 0x7573, 0x7575, 0x7576, 0x7577, 0x757a, 0x757b,
	0x757c, 0x757d, 0x757e, 0x7580, 0x7581, 0x7582, 0x7584, 0x7585
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xaf40[ 96 ] = {
	0x7588, 0x7589, 0x758a, 0x758c, 0x758d, 0x758e, 0x7590, 0x7593,
	0x7595, 0x7598, 0x759b, 0x759c, 0x759e, 0x75a2, 0x75a6, 0x75a7,
	0x75a8, 0x75a9, 0x75aa, 0x75ad, 0x75b6, 0x75b7, 0x75ba, 0x75bb,
	0x75bf, 0x75c0, 0x75c1, 0x75c6, 0x75cb, 0x75cc, 0x75ce, 0x75cf,
	0x75d0, 0x75d1, 0x75d3, 0x75d7, 0x75d9, 0x75da, 0x75dc, 0x75dd,
	0x75df, 0x75e0, 0x75e1, 0x75e5, 0x75e9, 0x75ec, 0x75ed, 0x75ee,
	0x75ef, 0x75f2, 0x75f3, 0x75f5, 0x75f6, 0x75f7, 0x75f8, 0x75fa,
	0x75fb, 0x75fd, 0x75fe, 0x7602, 0x7604, 0x7606, 0x7607, 0xfffd,
	0x7608, 0x7609, 0x760b, 0x760d, 0x760e, 0x760f, 0x7611, 0x7612,
	0x7613, 0x7614, 0x7616, 0x761a, 0x761c, 0x761d, 0x761e, 0x7621,
	0x7623, 0x7627, 0x7628, 0x762c, 0x762e, 0x762f, 0x7631, 0x7632,
	0x7636, 0x7637, 0x7639, 0x763a, 0x763b, 0x763d, 0x7641, 0x7642
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb040[ 192 ] = {
	0x7645, 0x7646, 0x7647, 0x7648, 0x7649, 0x764a, 0x764b, 0x764e,
	0x764f, 0x7650, 0x7651, 0x7652, 0x7653, 0x7655, 0x7657, 0x7658,
	0x7659, 0x765a, 0x765b, 0x765d, 0x765f, 0x7660, 0x7661, 0x7662,
	0x7664, 0x7665, 0x7666, 0x7667, 0x7668, 0x7669, 0x766a, 0x766c,
	0x766d, 0x766e, 0x7670, 0x7671, 0x7672, 0x7673, 0x7674, 0x7675,
	0x7676, 0x7677, 0x7679, 0x767a, 0x767c, 0x767f, 0x7680, 0x7681,
	0x7683, 0x7685, 0x7689, 0x768a, 0x768c, 0x768d, 0x768f, 0x7690,
	0x7692, 0x7694, 0x7695, 0x7697, 0x7698, 0x769a, 0x769b, 0xfffd,
	0x769c, 0x769d, 0x769e, 0x769f, 0x76a0, 0x76a1, 0x76a2, 0x76a3,
	0x76a5, 0x76a6, 0x76a7, 0x76a8, 0x76a9, 0x76aa, 0x76ab, 0x76ac,
	0x76ad, 0x76af, 0x76b0, 0x76b3, 0x76b5, 0x76b6, 0x76b7, 0x76b8,
	0x76b9, 0x76ba, 0x76bb, 0x76bc, 0x76bd, 0x76be, 0x76c0, 0x76c1,
	0x76c3, 0x554a, 0x963f, 0x57c3, 0x6328, 0x54ce, 0x5509, 0x54c0,
	0x7691, 0x764c, 0x853c, 0x77ee, 0x827e, 0x788d, 0x7231, 0x9698,
	0x978d, 0x6c28, 0x5b89, 0x4ffa, 0x6309, 0x6697, 0x5cb8, 0x80fa,
	0x6848, 0x80ae, 0x6602, 0x76ce, 0x51f9, 0x6556, 0x71ac, 0x7ff1,
	0x8884, 0x50b2, 0x5965, 0x61ca, 0x6fb3, 0x82ad, 0x634c, 0x6252,
	0x53ed, 0x5427, 0x7b06, 0x516b, 0x75a4, 0x5df4, 0x62d4, 0x8dcb,
	0x9776, 0x628a, 0x8019, 0x575d, 0x9738, 0x7f62, 0x7238, 0x767d,
	0x67cf, 0x767e, 0x6446, 0x4f70, 0x8d25, 0x62dc, 0x7a17, 0x6591,
	0x73ed, 0x642c, 0x6273, 0x822c, 0x9881, 0x677f, 0x7248, 0x626e,
	0x62cc, 0x4f34, 0x74e3, 0x534a, 0x529e, 0x7eca, 0x90a6, 0x5e2e,
	0x6886, 0x699c, 0x8180, 0x7ed1, 0x68d2, 0x78c5, 0x868c, 0x9551,
	0x508d, 0x8c24, 0x82de, 0x80de, 0x5305, 0x8912, 0x5265, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb140[ 192 ] = {
	0x76c4, 0x76c7, 0x76c9, 0x76cb, 0x76cc, 0x76d3, 0x76d5, 0x76d9,
	0x76da, 0x76dc, 0x76dd, 0x76de, 0x76e0, 0x76e1, 0x76e2, 0x76e3,
	0x76e4, 0x76e6, 0x76e7, 0x76e8, 0x76e9, 0x76ea, 0x76eb, 0x76ec,
	0x76ed, 0x76f0, 0x76f3, 0x76f5, 0x76f6, 0x76f7, 0x76fa, 0x76fb,
	0x76fd, 0x76ff, 0x7700, 0x7702, 0x7703, 0x7705, 0x7706, 0x770a,
	0x770c, 0x770e, 0x770f, 0x7710, 0x7711, 0x7712, 0x7713, 0x7714,
	0x7715, 0x7716, 0x7717, 0x7718, 0x771b, 0x771c, 0x771d, 0x771e,
	0x7721, 0x7723, 0x7724, 0x7725, 0x7727, 0x772a, 0x772b, 0xfffd,
	0x772c, 0x772e, 0x7730, 0x7731, 0x7732, 0x7733, 0x7734, 0x7739,
	0x773b, 0x773d, 0x773e, 0x773f, 0x7742, 0x7744, 0x7745, 0x7746,
	0x7748, 0x7749, 0x774a, 0x774b, 0x774c, 0x774d, 0x774e, 0x774f,
	0x7752, 0x7753, 0x7754, 0x7755, 0x7756, 0x7757, 0x7758, 0x7759,
	0x775c, 0x8584, 0x96f9, 0x4fdd, 0x5821, 0x9971, 0x5b9d, 0x62b1,
	0x62a5, 0x66b4, 0x8c79, 0x9c8d, 0x7206, 0x676f, 0x7891, 0x60b2,
	0x5351, 0x5317, 0x8f88, 0x80cc, 0x8d1d, 0x94a1, 0x500d, 0x72c8,
	0x5907, 0x60eb, 0x7119, 0x88ab, 0x5954, 0x82ef, 0x672c, 0x7b28,
	0x5d29, 0x7ef7, 0x752d, 0x6cf5, 0x8e66, 0x8ff8, 0x903c, 0x9f3b,
	0x6bd4, 0x9119, 0x7b14, 0x5f7c, 0x78a7, 0x84d6, 0x853d, 0x6bd5,
	0x6bd9, 0x6bd6, 0x5e01, 0x5e87, 0x75f9, 0x95ed, 0x655d, 0x5f0a,
	0x5fc5, 0x8f9f, 0x58c1, 0x81c2, 0x907f, 0x965b, 0x97ad, 0x8fb9,
	0x7f16, 0x8d2c, 0x6241, 0x4fbf, 0x53d8, 0x535e, 0x8fa8, 0x8fa9,
	0x8fab, 0x904d, 0x6807, 0x5f6a, 0x8198, 0x8868, 0x9cd6, 0x618b,
	0x522b, 0x762a, 0x5f6c, 0x658c, 0x6fd2, 0x6ee8, 0x5bbe, 0x6448,
	0x5175, 0x51b0, 0x67c4, 0x4e19, 0x79c9, 0x997c, 0x70b3, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb240[ 192 ] = {
	0x775d, 0x775e, 0x775f, 0x7760, 0x7764, 0x7767, 0x7769, 0x776a,
	0x776d, 0x776e, 0x776f, 0x7770, 0x7771, 0x7772, 0x7773, 0x7774,
	0x7775, 0x7776, 0x7777, 0x7778, 0x777a, 0x777b, 0x777c, 0x7781,
	0x7782, 0x7783, 0x7786, 0x7787, 0x7788, 0x7789, 0x778a, 0x778b,
	0x778f, 0x7790, 0x7793, 0x7794, 0x7795, 0x7796, 0x7797, 0x7798,
	0x7799, 0x779a, 0x779b, 0x779c, 0x779d, 0x779e, 0x77a1, 0x77a3,
	0x77a4, 0x77a6, 0x77a8, 0x77ab, 0x77ad, 0x77ae, 0x77af, 0x77b1,
	0x77b2, 0x77b4, 0x77b6, 0x77b7, 0x77b8, 0x77b9, 0x77ba, 0xfffd,
	0x77bc, 0x77be, 0x77c0, 0x77c1, 0x77c2, 0x77c3, 0x77c4, 0x77c5,
	0x77c6, 0x77c7, 0x77c8, 0x77c9, 0x77ca, 0x77cb, 0x77cc, 0x77ce,
	0x77cf, 0x77d0, 0x77d1, 0x77d2, 0x77d3, 0x77d4, 0x77d5, 0x77d6,
	0x77d8, 0x77d9, 0x77da, 0x77dd, 0x77de, 0x77df, 0x77e0, 0x77e1,
	0x77e4, 0x75c5, 0x5e76, 0x73bb, 0x83e0, 0x64ad, 0x62e8, 0x94b5,
	0x6ce2, 0x535a, 0x52c3, 0x640f, 0x94c2, 0x7b94, 0x4f2f, 0x5e1b,
	0x8236, 0x8116, 0x818a, 0x6e24, 0x6cca, 0x9a73, 0x6355, 0x535c,
	0x54fa, 0x8865, 0x57e0, 0x4e0d, 0x5e03, 0x6b65, 0x7c3f, 0x90e8,
	0x6016, 0x64e6, 0x731c, 0x88c1, 0x6750, 0x624d, 0x8d22, 0x776c,
	0x8e29, 0x91c7, 0x5f69, 0x83dc, 0x8521, 0x9910, 0x53c2, 0x8695,
	0x6b8b, 0x60ed, 0x60e8, 0x707f, 0x82cd, 0x8231, 0x4ed3, 0x6ca7,
	0x85cf, 0x64cd, 0x7cd9, 0x69fd, 0x66f9, 0x8349, 0x5395, 0x7b56,
	0x4fa7, 0x518c, 0x6d4b, 0x5c42, 0x8e6d, 0x63d2, 0x53c9, 0x832c,
	0x8336, 0x67e5, 0x78b4, 0x643d, 0x5bdf, 0x5c94, 0x5dee, 0x8be7,
	0x62c6, 0x67f4, 0x8c7a, 0x6400, 0x63ba, 0x8749, 0x998b, 0x8c17,
	0x7f20, 0x94f2, 0x4ea7, 0x9610, 0x98a4, 0x660c, 0x7316, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb340[ 192 ] = {
	0x77e6, 0x77e8, 0x77ea, 0x77ef, 0x77f0, 0x77f1, 0x77f2, 0x77f4,
	0x77f5, 0x77f7, 0x77f9, 0x77fa, 0x77fb, 0x77fc, 0x7803, 0x7804,
	0x7805, 0x7806, 0x7807, 0x7808, 0x780a, 0x780b, 0x780e, 0x780f,
	0x7810, 0x7813, 0x7815, 0x7819, 0x781b, 0x781e, 0x7820, 0x7821,
	0x7822, 0x7824, 0x7828, 0x782a, 0x782b, 0x782e, 0x782f, 0x7831,
	0x7832, 0x7833, 0x7835, 0x7836, 0x783d, 0x783f, 0x7841, 0x7842,
	0x7843, 0x7844, 0x7846, 0x7848, 0x7849, 0x784a, 0x784b, 0x784d,
	0x784f, 0x7851, 0x7853, 0x7854, 0x7858, 0x7859, 0x785a, 0xfffd,
	0x785b, 0x785c, 0x785e, 0x785f, 0x7860, 0x7861, 0x7862, 0x7863,
	0x7864, 0x7865, 0x7866, 0x7867, 0x7868, 0x7869, 0x786f, 0x7870,
	0x7871, 0x7872, 0x7873, 0x7874, 0x7875, 0x7876, 0x7878, 0x7879,
	0x787a, 0x787b, 0x787d, 0x787e, 0x787f, 0x7880, 0x7881, 0x7882,
	0x7883, 0x573a, 0x5c1d, 0x5e38, 0x957f, 0x507f, 0x80a0, 0x5382,
	0x655e, 0x7545, 0x5531, 0x5021, 0x8d85, 0x6284, 0x949e, 0x671d,
	0x5632, 0x6f6e, 0x5de2, 0x5435, 0x7092, 0x8f66, 0x626f, 0x64a4,
	0x63a3, 0x5f7b, 0x6f88, 0x90f4, 0x81e3, 0x8fb0, 0x5c18, 0x6668,
	0x5ff1, 0x6c89, 0x9648, 0x8d81, 0x886c, 0x6491, 0x79f0, 0x57ce,
	0x6a59, 0x6210, 0x5448, 0x4e58, 0x7a0b, 0x60e9, 0x6f84, 0x8bda,
	0x627f, 0x901e, 0x9a8b, 0x79e4, 0x5403, 0x75f4, 0x6301, 0x5319,
	0x6c60, 0x8fdf, 0x5f1b, 0x9a70, 0x803b, 0x9f7f, 0x4f88, 0x5c3a,
	0x8d64, 0x7fc5, 0x65a5, 0x70bd, 0x5145, 0x51b2, 0x866b, 0x5d07,
	0x5ba0, 0x62bd, 0x916c, 0x7574, 0x8e0c, 0x7a20, 0x6101, 0x7b79,
	0x4ec7, 0x7ef8, 0x7785, 0x4e11, 0x81ed, 0x521d, 0x51fa, 0x6a71,
	0x53a8, 0x8e87, 0x9504, 0x96cf, 0x6ec1, 0x9664, 0x695a, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb440[ 192 ] = {
	0x7884, 0x7885, 0x7886, 0x7888, 0x788a, 0x788b, 0x788f, 0x7890,
	0x7892, 0x7894, 0x7895, 0x7896, 0x7899, 0x789d, 0x789e, 0x78a0,
	0x78a2, 0x78a4, 0x78a6, 0x78a8, 0x78a9, 0x78aa, 0x78ab, 0x78ac,
	0x78ad, 0x78ae, 0x78af, 0x78b5, 0x78b6, 0x78b7, 0x78b8, 0x78ba,
	0x78bb, 0x78bc, 0x78bd, 0x78bf, 0x78c0, 0x78c2, 0x78c3, 0x78c4,
	0x78c6, 0x78c7, 0x78c8, 0x78cc, 0x78cd, 0x78ce, 0x78cf, 0x78d1,
	0x78d2, 0x78d3, 0x78d6, 0x78d7, 0x78d8, 0x78da, 0x78db, 0x78dc,
	0x78dd, 0x78de, 0x78df, 0x78e0, 0x78e1, 0x78e2, 0x78e3, 0xfffd,
	0x78e4, 0x78e5, 0x78e6, 0x78e7, 0x78e9, 0x78ea, 0x78eb, 0x78ed,
	0x78ee, 0x78ef, 0x78f0, 0x78f1, 0x78f3, 0x78f5, 0x78f6, 0x78f8,
	0x78f9, 0x78fb, 0x78fc, 0x78fd, 0x78fe, 0x78ff, 0x7900, 0x7902,
	0x7903, 0x7904, 0x7906, 0x7907, 0x7908, 0x7909, 0x790a, 0x790b,
	0x790c, 0x7840, 0x50a8, 0x77d7, 0x6410, 0x89e6, 0x5904, 0x63e3,
	0x5ddd, 0x7a7f, 0x693d, 0x4f20, 0x8239, 0x5598, 0x4e32, 0x75ae,
	0x7a97, 0x5e62, 0x5e8a, 0x95ef, 0x521b, 0x5439, 0x708a, 0x6376,
	0x9524, 0x5782, 0x6625, 0x693f, 0x9187, 0x5507, 0x6df3, 0x7eaf,
	0x8822, 0x6233, 0x7ef0, 0x75b5, 0x8328, 0x78c1, 0x96cc, 0x8f9e,
	0x6148, 0x74f7, 0x8bcd, 0x6b64, 0x523a, 0x8d50, 0x6b21, 0x806a,
	0x8471, 0x56f1, 0x5306, 0x4ece, 0x4e1b, 0x51d1, 0x7c97, 0x918b,
	0x7c07, 0x4fc3, 0x8e7f, 0x7be1, 0x7a9c, 0x6467, 0x5d14, 0x50ac,
	0x8106, 0x7601, 0x7cb9, 0x6dec, 0x7fe0, 0x6751, 0x5b58, 0x5bf8,
	0x78cb, 0x64ae, 0x6413, 0x63aa, 0x632b, 0x9519, 0x642d, 0x8fbe,
	0x7b54, 0x7629, 0x6253, 0x5927, 0x5446, 0x6b79, 0x50a3, 0x6234,
	0x5e26, 0x6b86, 0x4ee3, 0x8d37, 0x888b, 0x5f85, 0x902e, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb540[ 192 ] = {
	0x790d, 0x790e, 0x790f, 0x7910, 0x7911, 0x7912, 0x7914, 0x7915,
	0x7916, 0x7917, 0x7918, 0x7919, 0x791a, 0x791b, 0x791c, 0x791d,
	0x791f, 0x7920, 0x7921, 0x7922, 0x7923, 0x7925, 0x7926, 0x7927,
	0x7928, 0x7929, 0x792a, 0x792b, 0x792c, 0x792d, 0x792e, 0x792f,
	0x7930, 0x7931, 0x7932, 0x7933, 0x7935, 0x7936, 0x7937, 0x7938,
	0x7939, 0x793d, 0x793f, 0x7942, 0x7943, 0x7944, 0x7945, 0x7947,
	0x794a, 0x794b, 0x794c, 0x794d, 0x794e, 0x794f, 0x7950, 0x7951,
	0x7952, 0x7954, 0x7955, 0x7958, 0x7959, 0x7961, 0x7963, 0xfffd,
	0x7964, 0x7966, 0x7969, 0x796a, 0x796b, 0x796c, 0x796e, 0x7970,
	0x7971, 0x7972, 0x7973, 0x7974, 0x7975, 0x7976, 0x7979, 0x797b,
	0x797c, 0x797d, 0x797e, 0x797f, 0x7982, 0x7983, 0x7986, 0x7987,
	0x7988, 0x7989, 0x798b, 0x798c, 0x798d, 0x798e, 0x7990, 0x7991,
	0x7992, 0x6020, 0x803d, 0x62c5, 0x4e39, 0x5355, 0x90f8, 0x63b8,
	0x80c6, 0x65e6, 0x6c2e, 0x4f46, 0x60ee, 0x6de1, 0x8bde, 0x5f39,
	0x86cb, 0x5f53, 0x6321, 0x515a, 0x8361, 0x6863, 0x5200, 0x6363,
	0x8e48, 0x5012, 0x5c9b, 0x7977, 0x5bfc, 0x5230, 0x7a3b, 0x60bc,
	0x9053, 0x76d7, 0x5fb7, 0x5f97, 0x7684, 0x8e6c, 0x706f, 0x767b,
	0x7b49, 0x77aa, 0x51f3, 0x9093, 0x5824, 0x4f4e, 0x6ef4, 0x8fea,
	0x654c, 0x7b1b, 0x72c4, 0x6da4, 0x7fdf, 0x5ae1, 0x62b5, 0x5e95,
	0x5730, 0x8482, 0x7b2c, 0x5e1d, 0x5f1f, 0x9012, 0x7f14, 0x98a0,
	0x6382, 0x6ec7, 0x7898, 0x70b9, 0x5178, 0x975b, 0x57ab, 0x7535,
	0x4f43, 0x7538, 0x5e97, 0x60e6, 0x5960, 0x6dc0, 0x6bbf, 0x7889,
	0x53fc, 0x96d5, 0x51cb, 0x5201, 0x6389, 0x540a, 0x9493, 0x8c03,
	0x8dcc, 0x7239, 0x789f, 0x8776, 0x8fed, 0x8c0d, 0x53e0, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb640[ 192 ] = {
	0x7993, 0x7994, 0x7995, 0x7996, 0x7997, 0x7998, 0x7999, 0x799b,
	0x799c, 0x799d, 0x799e, 0x799f, 0x79a0, 0x79a1, 0x79a2, 0x79a3,
	0x79a4, 0x79a5, 0x79a6, 0x79a8, 0x79a9, 0x79aa, 0x79ab, 0x79ac,
	0x79ad, 0x79ae, 0x79af, 0x79b0, 0x79b1, 0x79b2, 0x79b4, 0x79b5,
	0x79b6, 0x79b7, 0x79b8, 0x79bc, 0x79bf, 0x79c2, 0x79c4, 0x79c5,
	0x79c7, 0x79c8, 0x79ca, 0x79cc, 0x79ce, 0x79cf, 0x79d0, 0x79d3,
	0x79d4, 0x79d6, 0x79d7, 0x79d9, 0x79da, 0x79db, 0x79dc, 0x79dd,
	0x79de, 0x79e0, 0x79e1, 0x79e2, 0x79e5, 0x79e8, 0x79ea, 0xfffd,
	0x79ec, 0x79ee, 0x79f1, 0x79f2, 0x79f3, 0x79f4, 0x79f5, 0x79f6,
	0x79f7, 0x79f9, 0x79fa, 0x79fc, 0x79fe, 0x79ff, 0x7a01, 0x7a04,
	0x7a05, 0x7a07, 0x7a08, 0x7a09, 0x7a0a, 0x7a0c, 0x7a0f, 0x7a10,
	0x7a11, 0x7a12, 0x7a13, 0x7a15, 0x7a16, 0x7a18, 0x7a19, 0x7a1b,
	0x7a1c, 0x4e01, 0x76ef, 0x53ee, 0x9489, 0x9876, 0x9f0e, 0x952d,
	0x5b9a, 0x8ba2, 0x4e22, 0x4e1c, 0x51ac, 0x8463, 0x61c2, 0x52a8,
	0x680b, 0x4f97, 0x606b, 0x51bb, 0x6d1e, 0x515c, 0x6296, 0x6597,
	0x9661, 0x8c46, 0x9017, 0x75d8, 0x90fd, 0x7763, 0x6bd2, 0x728a,
	0x72ec, 0x8bfb, 0x5835, 0x7779, 0x8d4c, 0x675c, 0x9540, 0x809a,
	0x5ea6, 0x6e21, 0x5992, 0x7aef, 0x77ed, 0x953b, 0x6bb5, 0x65ad,
	0x7f0e, 0x5806, 0x5151, 0x961f, 0x5bf9, 0x58a9, 0x5428, 0x8e72,
	0x6566, 0x987f, 0x56e4, 0x949d, 0x76fe, 0x9041, 0x6387, 0x54c6,
	0x591a, 0x593a, 0x579b, 0x8eb2, 0x6735, 0x8dfa, 0x8235, 0x5241,
	0x60f0, 0x5815, 0x86fe, 0x5ce8, 0x9e45, 0x4fc4, 0x989d, 0x8bb9,
	0x5a25, 0x6076, 0x5384, 0x627c, 0x904f, 0x9102, 0x997f, 0x6069,
	0x800c, 0x513f, 0x8033, 0x5c14, 0x9975, 0x6d31, 0x4e8c, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb740[ 192 ] = {
	0x7a1d, 0x7a1f, 0x7a21, 0x7a22, 0x7a24, 0x7a25, 0x7a26, 0x7a27,
	0x7a28, 0x7a29, 0x7a2a, 0x7a2b, 0x7a2c, 0x7a2d, 0x7a2e, 0x7a2f,
	0x7a30, 0x7a31, 0x7a32, 0x7a34, 0x7a35, 0x7a36, 0x7a38, 0x7a3a,
	0x7a3e, 0x7a40, 0x7a41, 0x7a42, 0x7a43, 0x7a44, 0x7a45, 0x7a47,
	0x7a48, 0x7a49, 0x7a4a, 0x7a4b, 0x7a4c, 0x7a4d, 0x7a4e, 0x7a4f,
	0x7a50, 0x7a52, 0x7a53, 0x7a54, 0x7a55, 0x7a56, 0x7a58, 0x7a59,
	0x7a5a, 0x7a5b, 0x7a5c, 0x7a5d, 0x7a5e, 0x7a5f, 0x7a60, 0x7a61,
	0x7a62, 0x7a63, 0x7a64, 0x7a65, 0x7a66, 0x7a67, 0x7a68, 0xfffd,
	0x7a69, 0x7a6a, 0x7a6b, 0x7a6c, 0x7a6d, 0x7a6e, 0x7a6f, 0x7a71,
	0x7a72, 0x7a73, 0x7a75, 0x7a7b, 0x7a7c, 0x7a7d, 0x7a7e, 0x7a82,
	0x7a85, 0x7a87, 0x7a89, 0x7a8a, 0x7a8b, 0x7a8c, 0x7a8e, 0x7a8f,
	0x7a90, 0x7a93, 0x7a94, 0x7a99, 0x7a9a, 0x7a9b, 0x7a9e, 0x7aa1,
	0x7aa2, 0x8d30, 0x53d1, 0x7f5a, 0x7b4f, 0x4f10, 0x4e4f, 0x9600,
	0x6cd5, 0x73d0, 0x85e9, 0x5e06, 0x756a, 0x7ffb, 0x6a0a, 0x77fe,
	0x9492, 0x7e41, 0x51e1, 0x70e6, 0x53cd, 0x8fd4, 0x8303, 0x8d29,
	0x72af, 0x996d, 0x6cdb, 0x574a, 0x82b3, 0x65b9, 0x80aa, 0x623f,
	0x9632, 0x59a8, 0x4eff, 0x8bbf, 0x7eba, 0x653e, 0x83f2, 0x975e,
	0x5561, 0x98de, 0x80a5, 0x532a, 0x8bfd, 0x5420, 0x80ba, 0x5e9f,
	0x6cb8, 0x8d39, 0x82ac, 0x915a, 0x5429, 0x6c1b, 0x5206, 0x7eb7,
	0x575f, 0x711a, 0x6c7e, 0x7c89, 0x594b, 0x4efd, 0x5fff, 0x6124,
	0x7caa, 0x4e30, 0x5c01, 0x67ab, 0x8702, 0x5cf0, 0x950b, 0x98ce,
	0x75af, 0x70fd, 0x9022, 0x51af, 0x7f1d, 0x8bbd, 0x5949, 0x51e4,
	0x4f5b, 0x5426, 0x592b, 0x6577, 0x80a4, 0x5b75, 0x6276, 0x62c2,
	0x8f90, 0x5e45, 0x6c1f, 0x7b26, 0x4f0f, 0x4fd8, 0x670d, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb840[ 192 ] = {
	0x7aa3, 0x7aa4, 0x7aa7, 0x7aa9, 0x7aaa, 0x7aab, 0x7aae, 0x7aaf,
	0x7ab0, 0x7ab1, 0x7ab2, 0x7ab4, 0x7ab5, 0x7ab6, 0x7ab7, 0x7ab8,
	0x7ab9, 0x7aba, 0x7abb, 0x7abc, 0x7abd, 0x7abe, 0x7ac0, 0x7ac1,
	0x7ac2, 0x7ac3, 0x7ac4, 0x7ac5, 0x7ac6, 0x7ac7, 0x7ac8, 0x7ac9,
	0x7aca, 0x7acc, 0x7acd, 0x7ace, 0x7acf, 0x7ad0, 0x7ad1, 0x7ad2,
	0x7ad3, 0x7ad4, 0x7ad5, 0x7ad7, 0x7ad8, 0x7ada, 0x7adb, 0x7adc,
	0x7add, 0x7ae1, 0x7ae2, 0x7ae4, 0x7ae7, 0x7ae8, 0x7ae9, 0x7aea,
	0x7aeb, 0x7aec, 0x7aee, 0x7af0, 0x7af1, 0x7af2, 0x7af3, 0xfffd,
	0x7af4, 0x7af5, 0x7af6, 0x7af7, 0x7af8, 0x7afb, 0x7afc, 0x7afe,
	0x7b00, 0x7b01, 0x7b02, 0x7b05, 0x7b07, 0x7b09, 0x7b0c, 0x7b0d,
	0x7b0e, 0x7b10, 0x7b12, 0x7b13, 0x7b16, 0x7b17, 0x7b18, 0x7b1a,
	0x7b1c, 0x7b1d, 0x7b1f, 0x7b21, 0x7b22, 0x7b23, 0x7b27, 0x7b29,
	0x7b2d, 0x6d6e, 0x6daa, 0x798f, 0x88b1, 0x5f17, 0x752b, 0x629a,
	0x8f85, 0x4fef, 0x91dc, 0x65a7, 0x812f, 0x8151, 0x5e9c, 0x8150,
	0x8d74, 0x526f, 0x8986, 0x8d4b, 0x590d, 0x5085, 0x4ed8, 0x961c,
	0x7236, 0x8179, 0x8d1f, 0x5bcc, 0x8ba3, 0x9644, 0x5987, 0x7f1a,
	0x5490, 0x5676, 0x560e, 0x8be5, 0x6539, 0x6982, 0x9499, 0x76d6,
	0x6e89, 0x5e72, 0x7518, 0x6746, 0x67d1, 0x7aff, 0x809d, 0x8d76,
	0x611f, 0x79c6, 0x6562, 0x8d63, 0x5188, 0x521a, 0x94a2, 0x7f38,
	0x809b, 0x7eb2, 0x5c97, 0x6e2f, 0x6760, 0x7bd9, 0x768b, 0x9ad8,
	0x818f, 0x7f94, 0x7cd5, 0x641e, 0x9550, 0x7a3f, 0x544a, 0x54e5,
	0x6b4c, 0x6401, 0x6208, 0x9e3d, 0x80f3, 0x7599, 0x5272, 0x9769,
	0x845b, 0x683c, 0x86e4, 0x9601, 0x9694, 0x94ec, 0x4e2a, 0x5404,
	0x7ed9, 0x6839, 0x8ddf, 0x8015, 0x66f4, 0x5e9a, 0x7fb9, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb940[ 192 ] = {
	0x7b2f, 0x7b30, 0x7b32, 0x7b34, 0x7b35, 0x7b36, 0x7b37, 0x7b39,
	0x7b3b, 0x7b3d, 0x7b3f, 0x7b40, 0x7b41, 0x7b42, 0x7b43, 0x7b44,
	0x7b46, 0x7b48, 0x7b4a, 0x7b4d, 0x7b4e, 0x7b53, 0x7b55, 0x7b57,
	0x7b59, 0x7b5c, 0x7b5e, 0x7b5f, 0x7b61, 0x7b63, 0x7b64, 0x7b65,
	0x7b66, 0x7b67, 0x7b68, 0x7b69, 0x7b6a, 0x7b6b, 0x7b6c, 0x7b6d,
	0x7b6f, 0x7b70, 0x7b73, 0x7b74, 0x7b76, 0x7b78, 0x7b7a, 0x7b7c,
	0x7b7d, 0x7b7f, 0x7b81, 0x7b82, 0x7b83, 0x7b84, 0x7b86, 0x7b87,
	0x7b88, 0x7b89, 0x7b8a, 0x7b8b, 0x7b8c, 0x7b8e, 0x7b8f, 0xfffd,
	0x7b91, 0x7b92, 0x7b93, 0x7b96, 0x7b98, 0x7b99, 0x7b9a, 0x7b9b,
	0x7b9e, 0x7b9f, 0x7ba0, 0x7ba3, 0x7ba4, 0x7ba5, 0x7bae, 0x7baf,
	0x7bb0, 0x7bb2, 0x7bb3, 0x7bb5, 0x7bb6, 0x7bb7, 0x7bb9, 0x7bba,
	0x7bbb, 0x7bbc, 0x7bbd, 0x7bbe, 0x7bbf, 0x7bc0, 0x7bc2, 0x7bc3,
	0x7bc4, 0x57c2, 0x803f, 0x6897, 0x5de5, 0x653b, 0x529f, 0x606d,
	0x9f9a, 0x4f9b, 0x8eac, 0x516c, 0x5bab, 0x5f13, 0x5de9, 0x6c5e,
	0x62f1, 0x8d21, 0x5171, 0x94a9, 0x52fe, 0x6c9f, 0x82df, 0x72d7,
	0x57a2, 0x6784, 0x8d2d, 0x591f, 0x8f9c, 0x83c7, 0x5495, 0x7b8d,
	0x4f30, 0x6cbd, 0x5b64, 0x59d1, 0x9f13, 0x53e4, 0x86ca, 0x9aa8,
	0x8c37, 0x80a1, 0x6545, 0x987e, 0x56fa, 0x96c7, 0x522e, 0x74dc,
	0x5250, 0x5be1, 0x6302, 0x8902, 0x4e56, 0x62d0, 0x602a, 0x68fa,
	0x5173, 0x5b98, 0x51a0, 0x89c2, 0x7ba1, 0x9986, 0x7f50, 0x60ef,
	0x704c, 0x8d2f, 0x5149, 0x5e7f, 0x901b, 0x7470, 0x89c4, 0x572d,
	0x7845, 0x5f52, 0x9f9f, 0x95fa, 0x8f68, 0x9b3c, 0x8be1, 0x7678,
	0x6842, 0x67dc, 0x8dea, 0x8d35, 0x523d, 0x8f8a, 0x6eda, 0x68cd,
	0x9505, 0x90ed, 0x56fd, 0x679c, 0x88f9, 0x8fc7, 0x54c8, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xba40[ 192 ] = {
	0x7bc5, 0x7bc8, 0x7bc9, 0x7bca, 0x7bcb, 0x7bcd, 0x7bce, 0x7bcf,
	0x7bd0, 0x7bd2, 0x7bd4, 0x7bd5, 0x7bd6, 0x7bd7, 0x7bd8, 0x7bdb,
	0x7bdc, 0x7bde, 0x7bdf, 0x7be0, 0x7be2, 0x7be3, 0x7be4, 0x7be7,
	0x7be8, 0x7be9, 0x7beb, 0x7bec, 0x7bed, 0x7bef, 0x7bf0, 0x7bf2,
	0x7bf3, 0x7bf4, 0x7bf5, 0x7bf6, 0x7bf8, 0x7bf9, 0x7bfa, 0x7bfb,
	0x7bfd, 0x7bff, 0x7c00, 0x7c01, 0x7c02, 0x7c03, 0x7c04, 0x7c05,
	0x7c06, 0x7c08, 0x7c09, 0x7c0a, 0x7c0d, 0x7c0e, 0x7c10, 0x7c11,
	0x7c12, 0x7c13, 0x7c14, 0x7c15, 0x7c17, 0x7c18, 0x7c19, 0xfffd,
	0x7c1a, 0x7c1b, 0x7c1c, 0x7c1d, 0x7c1e, 0x7c20, 0x7c21, 0x7c22,
	0x7c23, 0x7c24, 0x7c25, 0x7c28, 0x7c29, 0x7c2b, 0x7c2c, 0x7c2d,
	0x7c2e, 0x7c2f, 0x7c30, 0x7c31, 0x7c32, 0x7c33, 0x7c34, 0x7c35,
	0x7c36, 0x7c37, 0x7c39, 0x7c3a, 0x7c3b, 0x7c3c, 0x7c3d, 0x7c3e,
	0x7c42, 0x9ab8, 0x5b69, 0x6d77, 0x6c26, 0x4ea5, 0x5bb3, 0x9a87,
	0x9163, 0x61a8, 0x90af, 0x97e9, 0x542b, 0x6db5, 0x5bd2, 0x51fd,
	0x558a, 0x7f55, 0x7ff0, 0x64bc, 0x634d, 0x65f1, 0x61be, 0x608d,
	0x710a, 0x6c57, 0x6c49, 0x592f, 0x676d, 0x822a, 0x58d5, 0x568e,
	0x8c6a, 0x6beb, 0x90dd, 0x597d, 0x8017, 0x53f7, 0x6d69, 0x5475,
	0x559d, 0x8377, 0x83cf, 0x6838, 0x79be, 0x548c, 0x4f55, 0x5408,
	0x76d2, 0x8c89, 0x9602, 0x6cb3, 0x6db8, 0x8d6b, 0x8910, 0x9e64,
	0x8d3a, 0x563f, 0x9ed1, 0x75d5, 0x5f88, 0x72e0, 0x6068, 0x54fc,
	0x4ea8, 0x6a2a, 0x8861, 0x6052, 0x8f70, 0x54c4, 0x70d8, 0x8679,
	0x9e3f, 0x6d2a, 0x5b8f, 0x5f18, 0x7ea2, 0x5589, 0x4faf, 0x7334,
	0x543c, 0x539a, 0x5019, 0x540e, 0x547c, 0x4e4e, 0x5ffd, 0x745a,
	0x58f6, 0x846b, 0x80e1, 0x8774, 0x72d0, 0x7cca, 0x6e56, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xbb40[ 192 ] = {
	0x7c43, 0x7c44, 0x7c45, 0x7c46, 0x7c47, 0x7c48, 0x7c49, 0x7c4a,
	0x7c4b, 0x7c4c, 0x7c4e, 0x7c4f, 0x7c50, 0x7c51, 0x7c52, 0x7c53,
	0x7c54, 0x7c55, 0x7c56, 0x7c57, 0x7c58, 0x7c59, 0x7c5a, 0x7c5b,
	0x7c5c, 0x7c5d, 0x7c5e, 0x7c5f, 0x7c60, 0x7c61, 0x7c62, 0x7c63,
	0x7c64, 0x7c65, 0x7c66, 0x7c67, 0x7c68, 0x7c69, 0x7c6a, 0x7c6b,
	0x7c6c, 0x7c6d, 0x7c6e, 0x7c6f, 0x7c70, 0x7c71, 0x7c72, 0x7c75,
	0x7c76, 0x7c77, 0x7c78, 0x7c79, 0x7c7a, 0x7c7e, 0x7c7f, 0x7c80,
	0x7c81, 0x7c82, 0x7c83, 0x7c84, 0x7c85, 0x7c86, 0x7c87, 0xfffd,
	0x7c88, 0x7c8a, 0x7c8b, 0x7c8c, 0x7c8d, 0x7c8e, 0x7c8f, 0x7c90,
	0x7c93, 0x7c94, 0x7c96, 0x7c99, 0x7c9a, 0x7c9b, 0x7ca0, 0x7ca1,
	0x7ca3, 0x7ca6, 0x7ca7, 0x7ca8, 0x7ca9, 0x7cab, 0x7cac, 0x7cad,
	0x7caf, 0x7cb0, 0x7cb4, 0x7cb5, 0x7cb6, 0x7cb7, 0x7cb8, 0x7cba,
	0x7cbb, 0x5f27, 0x864e, 0x552c, 0x62a4, 0x4e92, 0x6caa, 0x6237,
	0x82b1, 0x54d7, 0x534e, 0x733e, 0x6ed1, 0x753b, 0x5212, 0x5316,
	0x8bdd, 0x69d0, 0x5f8a, 0x6000, 0x6dee, 0x574f, 0x6b22, 0x73af,
	0x6853, 0x8fd8, 0x7f13, 0x6362, 0x60a3, 0x5524, 0x75ea, 0x8c62,
	0x7115, 0x6da3, 0x5ba6, 0x5e7b, 0x8352, 0x614c, 0x9ec4, 0x78fa,
	0x8757, 0x7c27, 0x7687, 0x51f0, 0x60f6, 0x714c, 0x6643, 0x5e4c,
	0x604d, 0x8c0e, 0x7070, 0x6325, 0x8f89, 0x5fbd, 0x6062, 0x86d4,
	0x56de, 0x6bc1, 0x6094, 0x6167, 0x5349, 0x60e0, 0x6666, 0x8d3f,
	0x79fd, 0x4f1a, 0x70e9, 0x6c47, 0x8bb3, 0x8bf2, 0x7ed8, 0x8364,
	0x660f, 0x5a5a, 0x9b42, 0x6d51, 0x6df7, 0x8c41, 0x6d3b, 0x4f19,
	0x706b, 0x83b7, 0x6216, 0x60d1, 0x970d, 0x8d27, 0x7978, 0x51fb,
	0x573e, 0x57fa, 0x673a, 0x7578, 0x7a3d, 0x79ef, 0x7b95, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xbc40[ 192 ] = {
	0x7cbf, 0x7cc0, 0x7cc2, 0x7cc3, 0x7cc4, 0x7cc6, 0x7cc9, 0x7ccb,
	0x7cce, 0x7ccf, 0x7cd0, 0x7cd1, 0x7cd2, 0x7cd3, 0x7cd4, 0x7cd8,
	0x7cda, 0x7cdb, 0x7cdd, 0x7cde, 0x7ce1, 0x7ce2, 0x7ce3, 0x7ce4,
	0x7ce5, 0x7ce6, 0x7ce7, 0x7ce9, 0x7cea, 0x7ceb, 0x7cec, 0x7ced,
	0x7cee, 0x7cf0, 0x7cf1, 0x7cf2, 0x7cf3, 0x7cf4, 0x7cf5, 0x7cf6,
	0x7cf7, 0x7cf9, 0x7cfa, 0x7cfc, 0x7cfd, 0x7cfe, 0x7cff, 0x7d00,
	0x7d01, 0x7d02, 0x7d03, 0x7d04, 0x7d05, 0x7d06, 0x7d07, 0x7d08,
	0x7d09, 0x7d0b, 0x7d0c, 0x7d0d, 0x7d0e, 0x7d0f, 0x7d10, 0xfffd,
	0x7d11, 0x7d12, 0x7d13, 0x7d14, 0x7d15, 0x7d16, 0x7d17, 0x7d18,
	0x7d19, 0x7d1a, 0x7d1b, 0x7d1c, 0x7d1d, 0x7d1e, 0x7d1f, 0x7d21,
	0x7d23, 0x7d24, 0x7d25, 0x7d26, 0x7d28, 0x7d29, 0x7d2a, 0x7d2c,
	0x7d2d, 0x7d2e, 0x7d30, 0x7d31, 0x7d32, 0x7d33, 0x7d34, 0x7d35,
	0x7d36, 0x808c, 0x9965, 0x8ff9, 0x6fc0, 0x8ba5, 0x9e21, 0x59ec,
	0x7ee9, 0x7f09, 0x5409, 0x6781, 0x68d8, 0x8f91, 0x7c4d, 0x96c6,
	0x53ca, 0x6025, 0x75be, 0x6c72, 0x5373, 0x5ac9, 0x7ea7, 0x6324,
	0x51e0, 0x810a, 0x5df1, 0x84df, 0x6280, 0x5180, 0x5b63, 0x4f0e,
	0x796d, 0x5242, 0x60b8, 0x6d4e, 0x5bc4, 0x5bc2, 0x8ba1, 0x8bb0,
	0x65e2, 0x5fcc, 0x9645, 0x5993, 0x7ee7, 0x7eaa, 0x5609, 0x67b7,
	0x5939, 0x4f73, 0x5bb6, 0x52a0, 0x835a, 0x988a, 0x8d3e, 0x7532,
	0x94be, 0x5047, 0x7a3c, 0x4ef7, 0x67b6, 0x9a7e, 0x5ac1, 0x6b7c,
	0x76d1, 0x575a, 0x5c16, 0x7b3a, 0x95f4, 0x714e, 0x517c, 0x80a9,
	0x8270, 0x5978, 0x7f04, 0x8327, 0x68c0, 0x67ec, 0x78b1, 0x7877,
	0x62e3, 0x6361, 0x7b80, 0x4fed, 0x526a, 0x51cf, 0x8350, 0x69db,
	0x9274, 0x8df5, 0x8d31, 0x89c1, 0x952e, 0x7bad, 0x4ef6, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xbd40[ 192 ] = {
	0x7d37, 0x7d38, 0x7d39, 0x7d3a, 0x7d3b, 0x7d3c, 0x7d3d, 0x7d3e,
	0x7d3f, 0x7d40, 0x7d41, 0x7d42, 0x7d43, 0x7d44, 0x7d45, 0x7d46,
	0x7d47, 0x7d48, 0x7d49, 0x7d4a, 0x7d4b, 0x7d4c, 0x7d4d, 0x7d4e,
	0x7d4f, 0x7d50, 0x7d51, 0x7d52, 0x7d53, 0x7d54, 0x7d55, 0x7d56,
	0x7d57, 0x7d58, 0x7d59, 0x7d5a, 0x7d5b, 0x7d5c, 0x7d5d, 0x7d5e,
	0x7d5f, 0x7d60, 0x7d61, 0x7d62, 0x7d63, 0x7d64, 0x7d65, 0x7d66,
	0x7d67, 0x7d68, 0x7d69, 0x7d6a, 0x7d6b, 0x7d6c, 0x7d6d, 0x7d6f,
	0x7d70, 0x7d71, 0x7d72, 0x7d73, 0x7d74, 0x7d75, 0x7d76, 0xfffd,
	0x7d78, 0x7d79, 0x7d7a, 0x7d7b, 0x7d7c, 0x7d7d, 0x7d7e, 0x7d7f,
	0x7d80, 0x7d81, 0x7d82, 0x7d83, 0x7d84, 0x7d85, 0x7d86, 0x7d87,
	0x7d88, 0x7d89, 0x7d8a, 0x7d8b, 0x7d8c, 0x7d8d, 0x7d8e, 0x7d8f,
	0x7d90, 0x7d91, 0x7d92, 0x7d93, 0x7d94, 0x7d95, 0x7d96, 0x7d97,
	0x7d98, 0x5065, 0x8230, 0x5251, 0x996f, 0x6e10, 0x6e85, 0x6da7,
	0x5efa, 0x50f5, 0x59dc, 0x5c06, 0x6d46, 0x6c5f, 0x7586, 0x848b,
	0x6868, 0x5956, 0x8bb2, 0x5320, 0x9171, 0x964d, 0x8549, 0x6912,
	0x7901, 0x7126, 0x80f6, 0x4ea4, 0x90ca, 0x6d47, 0x9a84, 0x5a07,
	0x56bc, 0x6405, 0x94f0, 0x77eb, 0x4fa5, 0x811a, 0x72e1, 0x89d2,
	0x997a, 0x7f34, 0x7ede, 0x527f, 0x6559, 0x9175, 0x8f7f, 0x8f83,
	0x53eb, 0x7a96, 0x63ed, 0x63a5, 0x7686, 0x79f8, 0x8857, 0x9636,
	0x622a, 0x52ab, 0x8282, 0x6854, 0x6770, 0x6377, 0x776b, 0x7aed,
	0x6d01, 0x7ed3, 0x89e3, 0x59d0, 0x6212, 0x85c9, 0x82a5, 0x754c,
	0x501f, 0x4ecb, 0x75a5, 0x8beb, 0x5c4a, 0x5dfe, 0x7b4b, 0x65a4,
	0x91d1, 0x4eca, 0x6d25, 0x895f, 0x7d27, 0x9526, 0x4ec5, 0x8c28,
	0x8fdb, 0x9773, 0x664b, 0x7981, 0x8fd1, 0x70ec, 0x6d78, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xbe40[ 192 ] = {
	0x7d99, 0x7d9a, 0x7d9b, 0x7d9c, 0x7d9d, 0x7d9e, 0x7d9f, 0x7da0,
	0x7da1, 0x7da2, 0x7da3, 0x7da4, 0x7da5, 0x7da7, 0x7da8, 0x7da9,
	0x7daa, 0x7dab, 0x7dac, 0x7dad, 0x7daf, 0x7db0, 0x7db1, 0x7db2,
	0x7db3, 0x7db4, 0x7db5, 0x7db6, 0x7db7, 0x7db8, 0x7db9, 0x7dba,
	0x7dbb, 0x7dbc, 0x7dbd, 0x7dbe, 0x7dbf, 0x7dc0, 0x7dc1, 0x7dc2,
	0x7dc3, 0x7dc4, 0x7dc5, 0x7dc6, 0x7dc7, 0x7dc8, 0x7dc9, 0x7dca,
	0x7dcb, 0x7dcc, 0x7dcd, 0x7dce, 0x7dcf, 0x7dd0, 0x7dd1, 0x7dd2,
	0x7dd3, 0x7dd4, 0x7dd5, 0x7dd6, 0x7dd7, 0x7dd8, 0x7dd9, 0xfffd,
	0x7dda, 0x7ddb, 0x7ddc, 0x7ddd, 0x7dde, 0x7ddf, 0x7de0, 0x7de1,
	0x7de2, 0x7de3, 0x7de4, 0x7de5, 0x7de6, 0x7de7, 0x7de8, 0x7de9,
	0x7dea, 0x7deb, 0x7dec, 0x7ded, 0x7dee, 0x7def, 0x7df0, 0x7df1,
	0x7df2, 0x7df3, 0x7df4, 0x7df5, 0x7df6, 0x7df7, 0x7df8, 0x7df9,
	0x7dfa, 0x5c3d, 0x52b2, 0x8346, 0x5162, 0x830e, 0x775b, 0x6676,
	0x9cb8, 0x4eac, 0x60ca, 0x7cbe, 0x7cb3, 0x7ecf, 0x4e95, 0x8b66,
	0x666f, 0x9888, 0x9759, 0x5883, 0x656c, 0x955c, 0x5f84, 0x75c9,
	0x9756, 0x7adf, 0x7ade, 0x51c0, 0x70af, 0x7a98, 0x63ea, 0x7a76,
	0x7ea0, 0x7396, 0x97ed, 0x4e45, 0x7078, 0x4e5d, 0x9152, 0x53a9,
	0x6551, 0x65e7, 0x81fc, 0x8205, 0x548e, 0x5c31, 0x759a, 0x97a0,
	0x62d8, 0x72d9, 0x75bd, 0x5c45, 0x9a79, 0x83ca, 0x5c40, 0x5480,
	0x77e9, 0x4e3e, 0x6cae, 0x805a, 0x62d2, 0x636e, 0x5de8, 0x5177,
	0x8ddd, 0x8e1e, 0x952f, 0x4ff1, 0x53e5, 0x60e7, 0x70ac, 0x5267,
	0x6350, 0x9e43, 0x5a1f, 0x5026, 0x7737, 0x5377, 0x7ee2, 0x6485,
	0x652b, 0x6289, 0x6398, 0x5014, 0x7235, 0x89c9, 0x51b3, 0x8bc0,
	0x7edd, 0x5747, 0x83cc, 0x94a7, 0x519b, 0x541b, 0x5cfb, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xbf40[ 192 ] = {
	0x7dfb, 0x7dfc, 0x7dfd, 0x7dfe, 0x7dff, 0x7e00, 0x7e01, 0x7e02,
	0x7e03, 0x7e04, 0x7e05, 0x7e06, 0x7e07, 0x7e08, 0x7e09, 0x7e0a,
	0x7e0b, 0x7e0c, 0x7e0d, 0x7e0e, 0x7e0f, 0x7e10, 0x7e11, 0x7e12,
	0x7e13, 0x7e14, 0x7e15, 0x7e16, 0x7e17, 0x7e18, 0x7e19, 0x7e1a,
	0x7e1b, 0x7e1c, 0x7e1d, 0x7e1e, 0x7e1f, 0x7e20, 0x7e21, 0x7e22,
	0x7e23, 0x7e24, 0x7e25, 0x7e26, 0x7e27, 0x7e28, 0x7e29, 0x7e2a,
	0x7e2b, 0x7e2c, 0x7e2d, 0x7e2e, 0x7e2f, 0x7e30, 0x7e31, 0x7e32,
	0x7e33, 0x7e34, 0x7e35, 0x7e36, 0x7e37, 0x7e38, 0x7e39, 0xfffd,
	0x7e3a, 0x7e3c, 0x7e3d, 0x7e3e, 0x7e3f, 0x7e40, 0x7e42, 0x7e43,
	0x7e44, 0x7e45, 0x7e46, 0x7e48, 0x7e49, 0x7e4a, 0x7e4b, 0x7e4c,
	0x7e4d, 0x7e4e, 0x7e4f, 0x7e50, 0x7e51, 0x7e52, 0x7e53, 0x7e54,
	0x7e55, 0x7e56, 0x7e57, 0x7e58, 0x7e59, 0x7e5a, 0x7e5b, 0x7e5c,
	0x7e5d, 0x4fca, 0x7ae3, 0x6d5a, 0x90e1, 0x9a8f, 0x5580, 0x5496,
	0x5361, 0x54af, 0x5f00, 0x63e9, 0x6977, 0x51ef, 0x6168, 0x520a,
	0x582a, 0x52d8, 0x574e, 0x780d, 0x770b, 0x5eb7, 0x6177, 0x7ce0,
	0x625b, 0x6297, 0x4ea2, 0x7095, 0x8003, 0x62f7, 0x70e4, 0x9760,
	0x5777, 0x82db, 0x67ef, 0x68f5, 0x78d5, 0x9897, 0x79d1, 0x58f3,
	0x54b3, 0x53ef, 0x6e34, 0x514b, 0x523b, 0x5ba2, 0x8bfe, 0x80af,
	0x5543, 0x57a6, 0x6073, 0x5751, 0x542d, 0x7a7a, 0x6050, 0x5b54,
	0x63a7, 0x62a0, 0x53e3, 0x6263, 0x5bc7, 0x67af, 0x54ed, 0x7a9f,
	0x82e6, 0x9177, 0x5e93, 0x88e4, 0x5938, 0x57ae, 0x630e, 0x8de8,
	0x80ef, 0x5757, 0x7b77, 0x4fa9, 0x5feb, 0x5bbd, 0x6b3e, 0x5321,
	0x7b50, 0x72c2, 0x6846, 0x77ff, 0x7736, 0x65f7, 0x51b5, 0x4e8f,
	0x76d4, 0x5cbf, 0x7aa5, 0x8475, 0x594e, 0x9b41, 0x5080, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc040[ 192 ] = {
	0x7e5e, 0x7e5f, 0x7e60, 0x7e61, 0x7e62, 0x7e63, 0x7e64, 0x7e65,
	0x7e66, 0x7e67, 0x7e68, 0x7e69, 0x7e6a, 0x7e6b, 0x7e6c, 0x7e6d,
	0x7e6e, 0x7e6f, 0x7e70, 0x7e71, 0x7e72, 0x7e73, 0x7e74, 0x7e75,
	0x7e76, 0x7e77, 0x7e78, 0x7e79, 0x7e7a, 0x7e7b, 0x7e7c, 0x7e7d,
	0x7e7e, 0x7e7f, 0x7e80, 0x7e81, 0x7e83, 0x7e84, 0x7e85, 0x7e86,
	0x7e87, 0x7e88, 0x7e89, 0x7e8a, 0x7e8b, 0x7e8c, 0x7e8d, 0x7e8e,
	0x7e8f, 0x7e90, 0x7e91, 0x7e92, 0x7e93, 0x7e94, 0x7e95, 0x7e96,
	0x7e97, 0x7e98, 0x7e99, 0x7e9a, 0x7e9c, 0x7e9d, 0x7e9e, 0xfffd,
	0x7eae, 0x7eb4, 0x7ebb, 0x7ebc, 0x7ed6, 0x7ee4, 0x7eec, 0x7ef9,
	0x7f0a, 0x7f10, 0x7f1e, 0x7f37, 0x7f39, 0x7f3b, 0x7f3c, 0x7f3d,
	0x7f3e, 0x7f3f, 0x7f40, 0x7f41, 0x7f43, 0x7f46, 0x7f47, 0x7f48,
	0x7f49, 0x7f4a, 0x7f4b, 0x7f4c, 0x7f4d, 0x7f4e, 0x7f4f, 0x7f52,
	0x7f53, 0x9988, 0x6127, 0x6e83, 0x5764, 0x6606, 0x6346, 0x56f0,
	0x62ec, 0x6269, 0x5ed3, 0x9614, 0x5783, 0x62c9, 0x5587, 0x8721,
	0x814a, 0x8fa3, 0x5566, 0x83b1, 0x6765, 0x8d56, 0x84dd, 0x5a6a,
	0x680f, 0x62e6, 0x7bee, 0x9611, 0x5170, 0x6f9c, 0x8c30, 0x63fd,
	0x89c8, 0x61d2, 0x7f06, 0x70c2, 0x6ee5, 0x7405, 0x6994, 0x72fc,
	0x5eca, 0x90ce, 0x6717, 0x6d6a, 0x635e, 0x52b3, 0x7262, 0x8001,
	0x4f6c, 0x59e5, 0x916a, 0x70d9, 0x6d9d, 0x52d2, 0x4e50, 0x96f7,
	0x956d, 0x857e, 0x78ca, 0x7d2f, 0x5121, 0x5792, 0x64c2, 0x808b,
	0x7c7b, 0x6cea, 0x68f1, 0x695e, 0x51b7, 0x5398, 0x68a8, 0x7281,
	0x9ece, 0x7bf1, 0x72f8, 0x79bb, 0x6f13, 0x7406, 0x674e, 0x91cc,
	0x9ca4, 0x793c, 0x8389, 0x8354, 0x540f, 0x6817, 0x4e3d, 0x5389,
	0x52b1, 0x783e, 0x5386, 0x5229, 0x5088, 0x4f8b, 0x4fd0, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc140[ 192 ] = {
	0x7f56, 0x7f59, 0x7f5b, 0x7f5c, 0x7f5d, 0x7f5e, 0x7f60, 0x7f63,
	0x7f64, 0x7f65, 0x7f66, 0x7f67, 0x7f6b, 0x7f6c, 0x7f6d, 0x7f6f,
	0x7f70, 0x7f73, 0x7f75, 0x7f76, 0x7f77, 0x7f78, 0x7f7a, 0x7f7b,
	0x7f7c, 0x7f7d, 0x7f7f, 0x7f80, 0x7f82, 0x7f83, 0x7f84, 0x7f85,
	0x7f86, 0x7f87, 0x7f88, 0x7f89, 0x7f8b, 0x7f8d, 0x7f8f, 0x7f90,
	0x7f91, 0x7f92, 0x7f93, 0x7f95, 0x7f96, 0x7f97, 0x7f98, 0x7f99,
	0x7f9b, 0x7f9c, 0x7fa0, 0x7fa2, 0x7fa3, 0x7fa5, 0x7fa6, 0x7fa8,
	0x7fa9, 0x7faa, 0x7fab, 0x7fac, 0x7fad, 0x7fae, 0x7fb1, 0xfffd,
	0x7fb3, 0x7fb4, 0x7fb5, 0x7fb6, 0x7fb7, 0x7fba, 0x7fbb, 0x7fbe,
	0x7fc0, 0x7fc2, 0x7fc3, 0x7fc4, 0x7fc6, 0x7fc7, 0x7fc8, 0x7fc9,
	0x7fcb, 0x7fcd, 0x7fcf, 0x7fd0, 0x7fd1, 0x7fd2, 0x7fd3, 0x7fd6,
	0x7fd7, 0x7fd9, 0x7fda, 0x7fdb, 0x7fdc, 0x7fdd, 0x7fde, 0x7fe2,
	0x7fe3, 0x75e2, 0x7acb, 0x7c92, 0x6ca5, 0x96b6, 0x529b, 0x7483,
	0x54e9, 0x4fe9, 0x8054, 0x83b2, 0x8fde, 0x9570, 0x5ec9, 0x601c,
	0x6d9f, 0x5e18, 0x655b, 0x8138, 0x94fe, 0x604b, 0x70bc, 0x7ec3,
	0x7cae, 0x51c9, 0x6881, 0x7cb1, 0x826f, 0x4e24, 0x8f86, 0x91cf,
	0x667e, 0x4eae, 0x8c05, 0x64a9, 0x804a, 0x50da, 0x7597, 0x71ce,
	0x5be5, 0x8fbd, 0x6f66, 0x4e86, 0x6482, 0x9563, 0x5ed6, 0x6599,
	0x5217, 0x88c2, 0x70c8, 0x52a3, 0x730e, 0x7433, 0x6797, 0x78f7,
	0x9716, 0x4e34, 0x90bb, 0x9cde, 0x6dcb, 0x51db, 0x8d41, 0x541d,
	0x62ce, 0x73b2, 0x83f1, 0x96f6, 0x9f84, 0x94c3, 0x4f36, 0x7f9a,
	0x51cc, 0x7075, 0x9675, 0x5cad, 0x9886, 0x53e6, 0x4ee4, 0x6e9c,
	0x7409, 0x69b4, 0x786b, 0x998f, 0x7559, 0x5218, 0x7624, 0x6d41,
	0x67f3, 0x516d, 0x9f99, 0x804b, 0x5499, 0x7b3c, 0x7abf, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc240[ 192 ] = {
	0x7fe4, 0x7fe7, 0x7fe8, 0x7fea, 0x7feb, 0x7fec, 0x7fed, 0x7fef,
	0x7ff2, 0x7ff4, 0x7ff5, 0x7ff6, 0x7ff7, 0x7ff8, 0x7ff9, 0x7ffa,
	0x7ffd, 0x7ffe, 0x7fff, 0x8002, 0x8007, 0x8008, 0x8009, 0x800a,
	0x800e, 0x800f, 0x8011, 0x8013, 0x801a, 0x801b, 0x801d, 0x801e,
	0x801f, 0x8021, 0x8023, 0x8024, 0x802b, 0x802c, 0x802d, 0x802e,
	0x802f, 0x8030, 0x8032, 0x8034, 0x8039, 0x803a, 0x803c, 0x803e,
	0x8040, 0x8041, 0x8044, 0x8045, 0x8047, 0x8048, 0x8049, 0x804e,
	0x804f, 0x8050, 0x8051, 0x8053, 0x8055, 0x8056, 0x8057, 0xfffd,
	0x8059, 0x805b, 0x805c, 0x805d, 0x805e, 0x805f, 0x8060, 0x8061,
	0x8062, 0x8063, 0x8064, 0x8065, 0x8066, 0x8067, 0x8068, 0x806b,
	0x806c, 0x806d, 0x806e, 0x806f, 0x8070, 0x8072, 0x8073, 0x8074,
	0x8075, 0x8076, 0x8077, 0x8078, 0x8079, 0x807a, 0x807b, 0x807c,
	0x807d, 0x9686, 0x5784, 0x62e2, 0x9647, 0x697c, 0x5a04, 0x6402,
	0x7bd3, 0x6f0f, 0x964b, 0x82a6, 0x5362, 0x9885, 0x5e90, 0x7089,
	0x63b3, 0x5364, 0x864f, 0x9c81, 0x9e93, 0x788c, 0x9732, 0x8def,
	0x8d42, 0x9e7f, 0x6f5e, 0x7984, 0x5f55, 0x9646, 0x622e, 0x9a74,
	0x5415, 0x94dd, 0x4fa3, 0x65c5, 0x5c65, 0x5c61, 0x7f15, 0x8651,
	0x6c2f, 0x5f8b, 0x7387, 0x6ee4, 0x7eff, 0x5ce6, 0x631b, 0x5b6a,
	0x6ee6, 0x5375, 0x4e71, 0x63a0, 0x7565, 0x62a1, 0x8f6e, 0x4f26,
	0x4ed1, 0x6ca6, 0x7eb6, 0x8bba, 0x841d, 0x87ba, 0x7f57, 0x903b,
	0x9523, 0x7ba9, 0x9aa1, 0x88f8, 0x843d, 0x6d1b, 0x9a86, 0x7edc,
	0x5988, 0x9ebb, 0x739b, 0x7801, 0x8682, 0x9a6c, 0x9a82, 0x561b,
	0x5417, 0x57cb, 0x4e70, 0x9ea6, 0x5356, 0x8fc8, 0x8109, 0x7792,
	0x9992, 0x86ee, 0x6ee1, 0x8513, 0x66fc, 0x6162, 0x6f2b, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc340[ 192 ] = {
	0x807e, 0x8081, 0x8082, 0x8085, 0x8088, 0x808a, 0x808d, 0x808e,
	0x808f, 0x8090, 0x8091, 0x8092, 0x8094, 0x8095, 0x8097, 0x8099,
	0x809e, 0x80a3, 0x80a6, 0x80a7, 0x80a8, 0x80ac, 0x80b0, 0x80b3,
	0x80b5, 0x80b6, 0x80b8, 0x80b9, 0x80bb, 0x80c5, 0x80c7, 0x80c8,
	0x80c9, 0x80ca, 0x80cb, 0x80cf, 0x80d0, 0x80d1, 0x80d2, 0x80d3,
	0x80d4, 0x80d5, 0x80d8, 0x80df, 0x80e0, 0x80e2, 0x80e3, 0x80e6,
	0x80ee, 0x80f5, 0x80f7, 0x80f9, 0x80fb, 0x80fe, 0x80ff, 0x8100,
	0x8101, 0x8103, 0x8104, 0x8105, 0x8107, 0x8108, 0x810b, 0xfffd,
	0x810c, 0x8115, 0x8117, 0x8119, 0x811b, 0x811c, 0x811d, 0x811f,
	0x8120, 0x8121, 0x8122, 0x8123, 0x8124, 0x8125, 0x8126, 0x8127,
	0x8128, 0x8129, 0x812a, 0x812b, 0x812d, 0x812e, 0x8130, 0x8133,
	0x8134, 0x8135, 0x8137, 0x8139, 0x813a, 0x813b, 0x813c, 0x813d,
	0x813f, 0x8c29, 0x8292, 0x832b, 0x76f2, 0x6c13, 0x5fd9, 0x83bd,
	0x732b, 0x8305, 0x951a, 0x6bdb, 0x77db, 0x94c6, 0x536f, 0x8302,
	0x5192, 0x5e3d, 0x8c8c, 0x8d38, 0x4e48, 0x73ab, 0x679a, 0x6885,
	0x9176, 0x9709, 0x7164, 0x6ca1, 0x7709, 0x5a92, 0x9541, 0x6bcf,
	0x7f8e, 0x6627, 0x5bd0, 0x59b9, 0x5a9a, 0x95e8, 0x95f7, 0x4eec,
	0x840c, 0x8499, 0x6aac, 0x76df, 0x9530, 0x731b, 0x68a6, 0x5b5f,
	0x772f, 0x919a, 0x9761, 0x7cdc, 0x8ff7, 0x8c1c, 0x5f25, 0x7c73,
	0x79d8, 0x89c5, 0x6ccc, 0x871c, 0x5bc6, 0x5e42, 0x68c9, 0x7720,
	0x7ef5, 0x5195, 0x514d, 0x52c9, 0x5a29, 0x7f05, 0x9762, 0x82d7,
	0x63cf, 0x7784, 0x85d0, 0x79d2, 0x6e3a, 0x5e99, 0x5999, 0x8511,
	0x706d, 0x6c11, 0x62bf, 0x76bf, 0x654f, 0x60af, 0x95fd, 0x660e,
	0x879f, 0x9e23, 0x94ed, 0x540d, 0x547d, 0x8c2c, 0x6478, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc440[ 192 ] = {
	0x8140, 0x8141, 0x8142, 0x8143, 0x8144, 0x8145, 0x8147, 0x8149,
	0x814d, 0x814e, 0x814f, 0x8152, 0x8156, 0x8157, 0x8158, 0x815b,
	0x815c, 0x815d, 0x815e, 0x815f, 0x8161, 0x8162, 0x8163, 0x8164,
	0x8166, 0x8168, 0x816a, 0x816b, 0x816c, 0x816f, 0x8172, 0x8173,
	0x8175, 0x8176, 0x8177, 0x8178, 0x8181, 0x8183, 0x8184, 0x8185,
	0x8186, 0x8187, 0x8189, 0x818b, 0x818c, 0x818d, 0x818e, 0x8190,
	0x8192, 0x8193, 0x8194, 0x8195, 0x8196, 0x8197, 0x8199, 0x819a,
	0x819e, 0x819f, 0x81a0, 0x81a1, 0x81a2, 0x81a4, 0x81a5, 0xfffd,
	0x81a7, 0x81a9, 0x81ab, 0x81ac, 0x81ad, 0x81ae, 0x81af, 0x81b0,
	0x81b1, 0x81b2, 0x81b4, 0x81b5, 0x81b6, 0x81b7, 0x81b8, 0x81b9,
	0x81bc, 0x81bd, 0x81be, 0x81bf, 0x81c4, 0x81c5, 0x81c7, 0x81c8,
	0x81c9, 0x81cb, 0x81cd, 0x81ce, 0x81cf, 0x81d0, 0x81d1, 0x81d2,
	0x81d3, 0x6479, 0x8611, 0x6a21, 0x819c, 0x78e8, 0x6469, 0x9b54,
	0x62b9, 0x672b, 0x83ab, 0x58a8, 0x9ed8, 0x6cab, 0x6f20, 0x5bde,
	0x964c, 0x8c0b, 0x725f, 0x67d0, 0x62c7, 0x7261, 0x4ea9, 0x59c6,
	0x6bcd, 0x5893, 0x66ae, 0x5e55, 0x52df, 0x6155, 0x6728, 0x76ee,
	0x7766, 0x7267, 0x7a46, 0x62ff, 0x54ea, 0x5450, 0x94a0, 0x90a3,
	0x5a1c, 0x7eb3, 0x6c16, 0x4e43, 0x5976, 0x8010, 0x5948, 0x5357,
	0x7537, 0x96be, 0x56ca, 0x6320, 0x8111, 0x607c, 0x95f9, 0x6dd6,
	0x5462, 0x9981, 0x5185, 0x5ae9, 0x80fd, 0x59ae, 0x9713, 0x502a,
	0x6ce5, 0x5c3c, 0x62df, 0x4f60, 0x533f, 0x817b, 0x9006, 0x6eba,
	0x852b, 0x62c8, 0x5e74, 0x78be, 0x64b5, 0x637b, 0x5ff5, 0x5a18,
	0x917f, 0x9e1f, 0x5c3f, 0x634f, 0x8042, 0x5b7d, 0x556e, 0x954a,
	0x954d, 0x6d85, 0x60a8, 0x67e0, 0x72de, 0x51dd, 0x5b81, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc540[ 192 ] = {
	0x81d4, 0x81d5, 0x81d6, 0x81d7, 0x81d8, 0x81d9, 0x81da, 0x81db,
	0x81dc, 0x81dd, 0x81de, 0x81df, 0x81e0, 0x81e1, 0x81e2, 0x81e4,
	0x81e5, 0x81e6, 0x81e8, 0x81e9, 0x81eb, 0x81ee, 0x81ef, 0x81f0,
	0x81f1, 0x81f2, 0x81f5, 0x81f6, 0x81f7, 0x81f8, 0x81f9, 0x81fa,
	0x81fd, 0x81ff, 0x8203, 0x8207, 0x8208, 0x8209, 0x820a, 0x820b,
	0x820e, 0x820f, 0x8211, 0x8213, 0x8215, 0x8216, 0x8217, 0x8218,
	0x8219, 0x821a, 0x821d, 0x8220, 0x8224, 0x8225, 0x8226, 0x8227,
	0x8229, 0x822e, 0x8232, 0x823a, 0x823c, 0x823d, 0x823f, 0xfffd,
	0x8240, 0x8241, 0x8242, 0x8243, 0x8245, 0x8246, 0x8248, 0x824a,
	0x824c, 0x824d, 0x824e, 0x8250, 0x8251, 0x8252, 0x8253, 0x8254,
	0x8255, 0x8256, 0x8257, 0x8259, 0x825b, 0x825c, 0x825d, 0x825e,
	0x8260, 0x8261, 0x8262, 0x8263, 0x8264, 0x8265, 0x8266, 0x8267,
	0x8269, 0x62e7, 0x6cde, 0x725b, 0x626d, 0x94ae, 0x7ebd, 0x8113,
	0x6d53, 0x519c, 0x5f04, 0x5974, 0x52aa, 0x6012, 0x5973, 0x6696,
	0x8650, 0x759f, 0x632a, 0x61e6, 0x7cef, 0x8bfa, 0x54e6, 0x6b27,
	0x9e25, 0x6bb4, 0x85d5, 0x5455, 0x5076, 0x6ca4, 0x556a, 0x8db4,
	0x722c, 0x5e15, 0x6015, 0x7436, 0x62cd, 0x6392, 0x724c, 0x5f98,
	0x6e43, 0x6d3e, 0x6500, 0x6f58, 0x76d8, 0x78d0, 0x76fc, 0x7554,
	0x5224, 0x53db, 0x4e53, 0x5e9e, 0x65c1, 0x802a, 0x80d6, 0x629b,
	0x5486, 0x5228, 0x70ae, 0x888d, 0x8dd1, 0x6ce1, 0x5478, 0x80da,
	0x57f9, 0x88f4, 0x8d54, 0x966a, 0x914d, 0x4f69, 0x6c9b, 0x55b7,
	0x76c6, 0x7830, 0x62a8, 0x70f9, 0x6f8e, 0x5f6d, 0x84ec, 0x68da,
	0x787c, 0x7bf7, 0x81a8, 0x670b, 0x9e4f, 0x6367, 0x78b0, 0x576f,
	0x7812, 0x9739, 0x6279, 0x62ab, 0x5288, 0x7435, 0x6bd7, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc640[ 192 ] = {
	0x826a, 0x826b, 0x826c, 0x826d, 0x8271, 0x8275, 0x8276, 0x8277,
	0x8278, 0x827b, 0x827c, 0x8280, 0x8281, 0x8283, 0x8285, 0x8286,
	0x8287, 0x8289, 0x828c, 0x8290, 0x8293, 0x8294, 0x8295, 0x8296,
	0x829a, 0x829b, 0x829e, 0x82a0, 0x82a2, 0x82a3, 0x82a7, 0x82b2,
	0x82b5, 0x82b6, 0x82ba, 0x82bb, 0x82bc, 0x82bf, 0x82c0, 0x82c2,
	0x82c3, 0x82c5, 0x82c6, 0x82c9, 0x82d0, 0x82d6, 0x82d9, 0x82da,
	0x82dd, 0x82e2, 0x82e7, 0x82e8, 0x82e9, 0x82ea, 0x82ec, 0x82ed,
	0x82ee, 0x82f0, 0x82f2, 0x82f3, 0x82f5, 0x82f6, 0x82f8, 0xfffd,
	0x82fa, 0x82fc, 0x82fd, 0x82fe, 0x82ff, 0x8300, 0x830a, 0x830b,
	0x830d, 0x8310, 0x8312, 0x8313, 0x8316, 0x8318, 0x8319, 0x831d,
	0x831e, 0x831f, 0x8320, 0x8321, 0x8322, 0x8323, 0x8324, 0x8325,
	0x8326, 0x8329, 0x832a, 0x832e, 0x8330, 0x8332, 0x8337, 0x833b,
	0x833d, 0x5564, 0x813e, 0x75b2, 0x76ae, 0x5339, 0x75de, 0x50fb,
	0x5c41, 0x8b6c, 0x7bc7, 0x504f, 0x7247, 0x9a97, 0x98d8, 0x6f02,
	0x74e2, 0x7968, 0x6487, 0x77a5, 0x62fc, 0x9891, 0x8d2b, 0x54c1,
	0x8058, 0x4e52, 0x576a, 0x82f9, 0x840d, 0x5e73, 0x51ed, 0x74f6,
	0x8bc4, 0x5c4f, 0x5761, 0x6cfc, 0x9887, 0x5a46, 0x7834, 0x9b44,
	0x8feb, 0x7c95, 0x5256, 0x6251, 0x94fa, 0x4ec6, 0x8386, 0x8461,
	0x83e9, 0x84b2, 0x57d4, 0x6734, 0x5703, 0x666e, 0x6d66, 0x8c31,
	0x66dd, 0x7011, 0x671f, 0x6b3a, 0x6816, 0x621a, 0x59bb, 0x4e03,
	0x51c4, 0x6f06, 0x67d2, 0x6c8f, 0x5176, 0x68cb, 0x5947, 0x6b67,
	0x7566, 0x5d0e, 0x8110, 0x9f50, 0x65d7, 0x7948, 0x7941, 0x9a91,
	0x8d77, 0x5c82, 0x4e5e, 0x4f01, 0x542f, 0x5951, 0x780c, 0x5668,
	0x6c14, 0x8fc4, 0x5f03, 0x6c7d, 0x6ce3, 0x8bab, 0x6390, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc740[ 192 ] = {
	0x833e, 0x833f, 0x8341, 0x8342, 0x8344, 0x8345, 0x8348, 0x834a,
	0x834b, 0x834c, 0x834d, 0x834e, 0x8353, 0x8355, 0x8356, 0x8357,
	0x8358, 0x8359, 0x835d, 0x8362, 0x8370, 0x8371, 0x8372, 0x8373,
	0x8374, 0x8375, 0x8376, 0x8379, 0x837a, 0x837e, 0x837f, 0x8380,
	0x8381, 0x8382, 0x8383, 0x8384, 0x8387, 0x8388, 0x838a, 0x838b,
	0x838c, 0x838d, 0x838f, 0x8390, 0x8391, 0x8394, 0x8395, 0x8396,
	0x8397, 0x8399, 0x839a, 0x839d, 0x839f, 0x83a1, 0x83a2, 0x83a3,
	0x83a4, 0x83a5, 0x83a6, 0x83a7, 0x83ac, 0x83ad, 0x83ae, 0xfffd,
	0x83af, 0x83b5, 0x83bb, 0x83be, 0x83bf, 0x83c2, 0x83c3, 0x83c4,
	0x83c6, 0x83c8, 0x83c9, 0x83cb, 0x83cd, 0x83ce, 0x83d0, 0x83d1,
	0x83d2, 0x83d3, 0x83d5, 0x83d7, 0x83d9, 0x83da, 0x83db, 0x83de,
	0x83e2, 0x83e3, 0x83e4, 0x83e6, 0x83e7, 0x83e8, 0x83eb, 0x83ec,
	0x83ed, 0x6070, 0x6d3d, 0x7275, 0x6266, 0x948e, 0x94c5, 0x5343,
	0x8fc1, 0x7b7e, 0x4edf, 0x8c26, 0x4e7e, 0x9ed4, 0x94b1, 0x94b3,
	0x524d, 0x6f5c, 0x9063, 0x6d45, 0x8c34, 0x5811, 0x5d4c, 0x6b20,
	0x6b49, 0x67aa, 0x545b, 0x8154, 0x7f8c, 0x5899, 0x8537, 0x5f3a,
	0x62a2, 0x6a47, 0x9539, 0x6572, 0x6084, 0x6865, 0x77a7, 0x4e54,
	0x4fa8, 0x5de7, 0x9798, 0x64ac, 0x7fd8, 0x5ced, 0x4fcf, 0x7a8d,
	0x5207, 0x8304, 0x4e14, 0x602f, 0x7a83, 0x94a6, 0x4fb5, 0x4eb2,
	0x79e6, 0x7434, 0x52e4, 0x82b9, 0x64d2, 0x79bd, 0x5bdd, 0x6c81,
	0x9752, 0x8f7b, 0x6c22, 0x503e, 0x537f, 0x6e05, 0x64ce, 0x6674,
	0x6c30, 0x60c5, 0x9877, 0x8bf7, 0x5e86, 0x743c, 0x7a77, 0x79cb,
	0x4e18, 0x90b1, 0x7403, 0x6c42, 0x56da, 0x914b, 0x6cc5, 0x8d8b,
	0x533a, 0x86c6, 0x66f2, 0x8eaf, 0x5c48, 0x9a71, 0x6e20, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc840[ 192 ] = {
	0x83ee, 0x83ef, 0x83f3, 0x83f4, 0x83f5, 0x83f6, 0x83f7, 0x83fa,
	0x83fb, 0x83fc, 0x83fe, 0x83ff, 0x8400, 0x8402, 0x8405, 0x8407,
	0x8408, 0x8409, 0x840a, 0x8410, 0x8412, 0x8413, 0x8414, 0x8415,
	0x8416, 0x8417, 0x8419, 0x841a, 0x841b, 0x841e, 0x841f, 0x8420,
	0x8421, 0x8422, 0x8423, 0x8429, 0x842a, 0x842b, 0x842c, 0x842d,
	0x842e, 0x842f, 0x8430, 0x8432, 0x8433, 0x8434, 0x8435, 0x8436,
	0x8437, 0x8439, 0x843a, 0x843b, 0x843e, 0x843f, 0x8440, 0x8441,
	0x8442, 0x8443, 0x8444, 0x8445, 0x8447, 0x8448, 0x8449, 0xfffd,
	0x844a, 0x844b, 0x844c, 0x844d, 0x844e, 0x844f, 0x8450, 0x8452,
	0x8453, 0x8454, 0x8455, 0x8456, 0x8458, 0x845d, 0x845e, 0x845f,
	0x8460, 0x8462, 0x8464, 0x8465, 0x8466, 0x8467, 0x8468, 0x846a,
	0x846e, 0x846f, 0x8470, 0x8472, 0x8474, 0x8477, 0x8479, 0x847b,
	0x847c, 0x53d6, 0x5a36, 0x9f8b, 0x8da3, 0x53bb, 0x5708, 0x98a7,
	0x6743, 0x919b, 0x6cc9, 0x5168, 0x75ca, 0x62f3, 0x72ac, 0x5238,
	0x529d, 0x7f3a, 0x7094, 0x7638, 0x5374, 0x9e4a, 0x69b7, 0x786e,
	0x96c0, 0x88d9, 0x7fa4, 0x7136, 0x71c3, 0x5189, 0x67d3, 0x74e4,
	0x58e4, 0x6518, 0x56b7, 0x8ba9, 0x9976, 0x6270, 0x7ed5, 0x60f9,
	0x70ed, 0x58ec, 0x4ec1, 0x4eba, 0x5fcd, 0x97e7, 0x4efb, 0x8ba4,
	0x5203, 0x598a, 0x7eab, 0x6254, 0x4ecd, 0x65e5, 0x620e, 0x8338,
	0x84c9, 0x8363, 0x878d, 0x7194, 0x6eb6, 0x5bb9, 0x7ed2, 0x5197,
	0x63c9, 0x67d4, 0x8089, 0x8339, 0x8815, 0x5112, 0x5b7a, 0x5982,
	0x8fb1, 0x4e73, 0x6c5d, 0x5165, 0x8925, 0x8f6f, 0x962e, 0x854a,
	0x745e, 0x9510, 0x95f0, 0x6da6, 0x82e5, 0x5f31, 0x6492, 0x6d12,
	0x8428, 0x816e, 0x9cc3, 0x585e, 0x8d5b, 0x4e09, 0x53c1, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc940[ 192 ] = {
	0x847d, 0x847e, 0x847f, 0x8480, 0x8481, 0x8483, 0x8484, 0x8485,
	0x8486, 0x848a, 0x848d, 0x848f, 0x8490, 0x8491, 0x8492, 0x8493,
	0x8494, 0x8495, 0x8496, 0x8498, 0x849a, 0x849b, 0x849d, 0x849e,
	0x849f, 0x84a0, 0x84a2, 0x84a3, 0x84a4, 0x84a5, 0x84a6, 0x84a7,
	0x84a8, 0x84a9, 0x84aa, 0x84ab, 0x84ac, 0x84ad, 0x84ae, 0x84b0,
	0x84b1, 0x84b3, 0x84b5, 0x84b6, 0x84b7, 0x84bb, 0x84bc, 0x84be,
	0x84c0, 0x84c2, 0x84c3, 0x84c5, 0x84c6, 0x84c7, 0x84c8, 0x84cb,
	0x84cc, 0x84ce, 0x84cf, 0x84d2, 0x84d4, 0x84d5, 0x84d7, 0xfffd,
	0x84d8, 0x84d9, 0x84da, 0x84db, 0x84dc, 0x84de, 0x84e1, 0x84e2,
	0x84e4, 0x84e7, 0x84e8, 0x84e9, 0x84ea, 0x84eb, 0x84ed, 0x84ee,
	0x84ef, 0x84f1, 0x84f2, 0x84f3, 0x84f4, 0x84f5, 0x84f6, 0x84f7,
	0x84f8, 0x84f9, 0x84fa, 0x84fb, 0x84fd, 0x84fe, 0x8500, 0x8501,
	0x8502, 0x4f1e, 0x6563, 0x6851, 0x55d3, 0x4e27, 0x6414, 0x9a9a,
	0x626b, 0x5ac2, 0x745f, 0x8272, 0x6da9, 0x68ee, 0x50e7, 0x838e,
	0x7802, 0x6740, 0x5239, 0x6c99, 0x7eb1, 0x50bb, 0x5565, 0x715e,
	0x7b5b, 0x6652, 0x73ca, 0x82eb, 0x6749, 0x5c71, 0x5220, 0x717d,
	0x886b, 0x95ea, 0x9655, 0x64c5, 0x8d61, 0x81b3, 0x5584, 0x6c55,
	0x6247, 0x7f2e, 0x5892, 0x4f24, 0x5546, 0x8d4f, 0x664c, 0x4e0a,
	0x5c1a, 0x88f3, 0x68a2, 0x634e, 0x7a0d, 0x70e7, 0x828d, 0x52fa,
	0x97f6, 0x5c11, 0x54e8, 0x90b5, 0x7ecd, 0x5962, 0x8d4a, 0x86c7,
	0x820c, 0x820d, 0x8d66, 0x6444, 0x5c04, 0x6151, 0x6d89, 0x793e,
	0x8bbe, 0x7837, 0x7533, 0x547b, 0x4f38, 0x8eab, 0x6df1, 0x5a20,
	0x7ec5, 0x795e, 0x6c88, 0x5ba1, 0x5a76, 0x751a, 0x80be, 0x614e,
	0x6e17, 0x58f0, 0x751f, 0x7525, 0x7272, 0x5347, 0x7ef3, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xca40[ 192 ] = {
	0x8503, 0x8504, 0x8505, 0x8506, 0x8507, 0x8508, 0x8509, 0x850a,
	0x850b, 0x850d, 0x850e, 0x850f, 0x8510, 0x8512, 0x8514, 0x8515,
	0x8516, 0x8518, 0x8519, 0x851b, 0x851c, 0x851d, 0x851e, 0x8520,
	0x8522, 0x8523, 0x8524, 0x8525, 0x8526, 0x8527, 0x8528, 0x8529,
	0x852a, 0x852d, 0x852e, 0x852f, 0x8530, 0x8531, 0x8532, 0x8533,
	0x8534, 0x8535, 0x8536, 0x853e, 0x853f, 0x8540, 0x8541, 0x8542,
	0x8544, 0x8545, 0x8546, 0x8547, 0x854b, 0x854c, 0x854d, 0x854e,
	0x854f, 0x8550, 0x8551, 0x8552, 0x8553, 0x8554, 0x8555, 0xfffd,
	0x8557, 0x8558, 0x855a, 0x855b, 0x855c, 0x855d, 0x855f, 0x8560,
	0x8561, 0x8562, 0x8563, 0x8565, 0x8566, 0x8567, 0x8569, 0x856a,
	0x856b, 0x856c, 0x856d, 0x856e, 0x856f, 0x8570, 0x8571, 0x8573,
	0x8575, 0x8576, 0x8577, 0x8578, 0x857c, 0x857d, 0x857f, 0x8580,
	0x8581, 0x7701, 0x76db, 0x5269, 0x80dc, 0x5723, 0x5e08, 0x5931,
	0x72ee, 0x65bd, 0x6e7f, 0x8bd7, 0x5c38, 0x8671, 0x5341, 0x77f3,
	0x62fe, 0x65f6, 0x4ec0, 0x98df, 0x8680, 0x5b9e, 0x8bc6, 0x53f2,
	0x77e2, 0x4f7f, 0x5c4e, 0x9a76, 0x59cb, 0x5f0f, 0x793a, 0x58eb,
	0x4e16, 0x67ff, 0x4e8b, 0x62ed, 0x8a93, 0x901d, 0x52bf, 0x662f,
	0x55dc, 0x566c, 0x9002, 0x4ed5, 0x4f8d, 0x91ca, 0x9970, 0x6c0f,
	0x5e02, 0x6043, 0x5ba4, 0x89c6, 0x8bd5, 0x6536, 0x624b, 0x9996,
	0x5b88, 0x5bff, 0x6388, 0x552e, 0x53d7, 0x7626, 0x517d, 0x852c,
	0x67a2, 0x68b3, 0x6b8a, 0x6292, 0x8f93, 0x53d4, 0x8212, 0x6dd1,
	0x758f, 0x4e66, 0x8d4e, 0x5b70, 0x719f, 0x85af, 0x6691, 0x66d9,
	0x7f72, 0x8700, 0x9ecd, 0x9f20, 0x5c5e, 0x672f, 0x8ff0, 0x6811,
	0x675f, 0x620d, 0x7ad6, 0x5885, 0x5eb6, 0x6570, 0x6f31, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xcb40[ 192 ] = {
	0x8582, 0x8583, 0x8586, 0x8588, 0x8589, 0x858a, 0x858b, 0x858c,
	0x858d, 0x858e, 0x8590, 0x8591, 0x8592, 0x8593, 0x8594, 0x8595,
	0x8596, 0x8597, 0x8598, 0x8599, 0x859a, 0x859d, 0x859e, 0x859f,
	0x85a0, 0x85a1, 0x85a2, 0x85a3, 0x85a5, 0x85a6, 0x85a7, 0x85a9,
	0x85ab, 0x85ac, 0x85ad, 0x85b1, 0x85b2, 0x85b3, 0x85b4, 0x85b5,
	0x85b6, 0x85b8, 0x85ba, 0x85bb, 0x85bc, 0x85bd, 0x85be, 0x85bf,
	0x85c0, 0x85c2, 0x85c3, 0x85c4, 0x85c5, 0x85c6, 0x85c7, 0x85c8,
	0x85ca, 0x85cb, 0x85cc, 0x85cd, 0x85ce, 0x85d1, 0x85d2, 0xfffd,
	0x85d4, 0x85d6, 0x85d7, 0x85d8, 0x85d9, 0x85da, 0x85db, 0x85dd,
	0x85de, 0x85df, 0x85e0, 0x85e1, 0x85e2, 0x85e3, 0x85e5, 0x85e6,
	0x85e7, 0x85e8, 0x85ea, 0x85eb, 0x85ec, 0x85ed, 0x85ee, 0x85ef,
	0x85f0, 0x85f1, 0x85f2, 0x85f3, 0x85f4, 0x85f5, 0x85f6, 0x85f7,
	0x85f8, 0x6055, 0x5237, 0x800d, 0x6454, 0x8870, 0x7529, 0x5e05,
	0x6813, 0x62f4, 0x971c, 0x53cc, 0x723d, 0x8c01, 0x6c34, 0x7761,
	0x7a0e, 0x542e, 0x77ac, 0x987a, 0x821c, 0x8bf4, 0x7855, 0x6714,
	0x70c1, 0x65af, 0x6495, 0x5636, 0x601d, 0x79c1, 0x53f8, 0x4e1d,
	0x6b7b, 0x8086, 0x5bfa, 0x55e3, 0x56db, 0x4f3a, 0x4f3c, 0x9972,
	0x5df3, 0x677e, 0x8038, 0x6002, 0x9882, 0x9001, 0x5b8b, 0x8bbc,
	0x8bf5, 0x641c, 0x8258, 0x64de, 0x55fd, 0x82cf, 0x9165, 0x4fd7,
	0x7d20, 0x901f, 0x7c9f, 0x50f3, 0x5851, 0x6eaf, 0x5bbf, 0x8bc9,
	0x8083, 0x9178, 0x849c, 0x7b97, 0x867d, 0x968b, 0x968f, 0x7ee5,
	0x9ad3, 0x788e, 0x5c81, 0x7a57, 0x9042, 0x96a7, 0x795f, 0x5b59,
	0x635f, 0x7b0b, 0x84d1, 0x68ad, 0x5506, 0x7f29, 0x7410, 0x7d22,
	0x9501, 0x6240, 0x584c, 0x4ed6, 0x5b83, 0x5979, 0x5854, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xcc40[ 192 ] = {
	0x85f9, 0x85fa, 0x85fc, 0x85fd, 0x85fe, 0x8600, 0x8601, 0x8602,
	0x8603, 0x8604, 0x8606, 0x8607, 0x8608, 0x8609, 0x860a, 0x860b,
	0x860c, 0x860d, 0x860e, 0x860f, 0x8610, 0x8612, 0x8613, 0x8614,
	0x8615, 0x8617, 0x8618, 0x8619, 0x861a, 0x861b, 0x861c, 0x861d,
	0x861e, 0x861f, 0x8620, 0x8621, 0x8622, 0x8623, 0x8624, 0x8625,
	0x8626, 0x8628, 0x862a, 0x862b, 0x862c, 0x862d, 0x862e, 0x862f,
	0x8630, 0x8631, 0x8632, 0x8633, 0x8634, 0x8635, 0x8636, 0x8637,
	0x8639, 0x863a, 0x863b, 0x863d, 0x863e, 0x863f, 0x8640, 0xfffd,
	0x8641, 0x8642, 0x8643, 0x8644, 0x8645, 0x8646, 0x8647, 0x8648,
	0x8649, 0x864a, 0x864b, 0x864c, 0x8652, 0x8653, 0x8655, 0x8656,
	0x8657, 0x8658, 0x8659, 0x865b, 0x865c, 0x865d, 0x865f, 0x8660,
	0x8661, 0x8663, 0x8664, 0x8665, 0x8666, 0x8667, 0x8668, 0x8669,
	0x866a, 0x736d, 0x631e, 0x8e4b, 0x8e0f, 0x80ce, 0x82d4, 0x62ac,
	0x53f0, 0x6cf0, 0x915e, 0x592a, 0x6001, 0x6c70, 0x574d, 0x644a,
	0x8d2a, 0x762b, 0x6ee9, 0x575b, 0x6a80, 0x75f0, 0x6f6d, 0x8c2d,
	0x8c08, 0x5766, 0x6bef, 0x8892, 0x78b3, 0x63a2, 0x53f9, 0x70ad,
	0x6c64, 0x5858, 0x642a, 0x5802, 0x68e0, 0x819b, 0x5510, 0x7cd6,
	0x5018, 0x8eba, 0x6dcc, 0x8d9f, 0x70eb, 0x638f, 0x6d9b, 0x6ed4,
	0x7ee6, 0x8404, 0x6843, 0x9003, 0x6dd8, 0x9676, 0x8ba8, 0x5957,
	0x7279, 0x85e4, 0x817e, 0x75bc, 0x8a8a, 0x68af, 0x5254, 0x8e22,
	0x9511, 0x63d0, 0x9898, 0x8e44, 0x557c, 0x4f53, 0x66ff, 0x568f,
	0x60d5, 0x6d95, 0x5243, 0x5c49, 0x5929, 0x6dfb, 0x586b, 0x7530,
	0x751c, 0x606c, 0x8214, 0x8146, 0x6311, 0x6761, 0x8fe2, 0x773a,
	0x8df3, 0x8d34, 0x94c1, 0x5e16, 0x5385, 0x542c, 0x70c3, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xcd40[ 192 ] = {
	0x866d, 0x866f, 0x8670, 0x8672, 0x8673, 0x8674, 0x8675, 0x8676,
	0x8677, 0x8678, 0x8683, 0x8684, 0x8685, 0x8686, 0x8687, 0x8688,
	0x8689, 0x868e, 0x868f, 0x8690, 0x8691, 0x8692, 0x8694, 0x8696,
	0x8697, 0x8698, 0x8699, 0x869a, 0x869b, 0x869e, 0x869f, 0x86a0,
	0x86a1, 0x86a2, 0x86a5, 0x86a6, 0x86ab, 0x86ad, 0x86ae, 0x86b2,
	0x86b3, 0x86b7, 0x86b8, 0x86b9, 0x86bb, 0x86bc, 0x86bd, 0x86be,
	0x86bf, 0x86c1, 0x86c2, 0x86c3, 0x86c5, 0x86c8, 0x86cc, 0x86cd,
	0x86d2, 0x86d3, 0x86d5, 0x86d6, 0x86d7, 0x86da, 0x86dc, 0xfffd,
	0x86dd, 0x86e0, 0x86e1, 0x86e2, 0x86e3, 0x86e5, 0x86e6, 0x86e7,
	0x86e8, 0x86ea, 0x86eb, 0x86ec, 0x86ef, 0x86f5, 0x86f6, 0x86f7,
	0x86fa, 0x86fb, 0x86fc, 0x86fd, 0x86ff, 0x8701, 0x8704, 0x8705,
	0x8706, 0x870b, 0x870c, 0x870e, 0x870f, 0x8710, 0x8711, 0x8714,
	0x8716, 0x6c40, 0x5ef7, 0x505c, 0x4ead, 0x5ead, 0x633a, 0x8247,
	0x901a, 0x6850, 0x916e, 0x77b3, 0x540c, 0x94dc, 0x5f64, 0x7ae5,
	0x6876, 0x6345, 0x7b52, 0x7edf, 0x75db, 0x5077, 0x6295, 0x5934,
	0x900f, 0x51f8, 0x79c3, 0x7a81, 0x56fe, 0x5f92, 0x9014, 0x6d82,
	0x5c60, 0x571f, 0x5410, 0x5154, 0x6e4d, 0x56e2, 0x63a8, 0x9893,
	0x817f, 0x8715, 0x892a, 0x9000, 0x541e, 0x5c6f, 0x81c0, 0x62d6,
	0x6258, 0x8131, 0x9e35, 0x9640, 0x9a6e, 0x9a7c, 0x692d, 0x59a5,
	0x62d3, 0x553e, 0x6316, 0x54c7, 0x86d9, 0x6d3c, 0x5a03, 0x74e6,
	0x889c, 0x6b6a, 0x5916, 0x8c4c, 0x5f2f, 0x6e7e, 0x73a9, 0x987d,
	0x4e38, 0x70f7, 0x5b8c, 0x7897, 0x633d, 0x665a, 0x7696, 0x60cb,
	0x5b9b, 0x5a49, 0x4e07, 0x8155, 0x6c6a, 0x738b, 0x4ea1, 0x6789,
	0x7f51, 0x5f80, 0x65fa, 0x671b, 0x5fd8, 0x5984, 0x5a01, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xce40[ 192 ] = {
	0x8719, 0x871b, 0x871d, 0x871f, 0x8720, 0x8724, 0x8726, 0x8727,
	0x8728, 0x872a, 0x872b, 0x872c, 0x872d, 0x872f, 0x8730, 0x8732,
	0x8733, 0x8735, 0x8736, 0x8738, 0x8739, 0x873a, 0x873c, 0x873d,
	0x8740, 0x8741, 0x8742, 0x8743, 0x8744, 0x8745, 0x8746, 0x874a,
	0x874b, 0x874d, 0x874f, 0x8750, 0x8751, 0x8752, 0x8754, 0x8755,
	0x8756, 0x8758, 0x875a, 0x875b, 0x875c, 0x875d, 0x875e, 0x875f,
	0x8761, 0x8762, 0x8766, 0x8767, 0x8768, 0x8769, 0x876a, 0x876b,
	0x876c, 0x876d, 0x876f, 0x8771, 0x8772, 0x8773, 0x8775, 0xfffd,
	0x8777, 0x8778, 0x8779, 0x877a, 0x877f, 0x8780, 0x8781, 0x8784,
	0x8786, 0x8787, 0x8789, 0x878a, 0x878c, 0x878e, 0x878f, 0x8790,
	0x8791, 0x8792, 0x8794, 0x8795, 0x8796, 0x8798, 0x8799, 0x879a,
	0x879b, 0x879c, 0x879d, 0x879e, 0x87a0, 0x87a1, 0x87a2, 0x87a3,
	0x87a4, 0x5dcd, 0x5fae, 0x5371, 0x97e6, 0x8fdd, 0x6845, 0x56f4,
	0x552f, 0x60df, 0x4e3a, 0x6f4d, 0x7ef4, 0x82c7, 0x840e, 0x59d4,
	0x4f1f, 0x4f2a, 0x5c3e, 0x7eac, 0x672a, 0x851a, 0x5473, 0x754f,
	0x80c3, 0x5582, 0x9b4f, 0x4f4d, 0x6e2d, 0x8c13, 0x5c09, 0x6170,
	0x536b, 0x761f, 0x6e29, 0x868a, 0x6587, 0x95fb, 0x7eb9, 0x543b,
	0x7a33, 0x7d0a, 0x95ee, 0x55e1, 0x7fc1, 0x74ee, 0x631d, 0x8717,
	0x6da1, 0x7a9d, 0x6211, 0x65a1, 0x5367, 0x63e1, 0x6c83, 0x5deb,
	0x545c, 0x94a8, 0x4e4c, 0x6c61, 0x8bec, 0x5c4b, 0x65e0, 0x829c,
	0x68a7, 0x543e, 0x5434, 0x6bcb, 0x6b66, 0x4e94, 0x6342, 0x5348,
	0x821e, 0x4f0d, 0x4fae, 0x575e, 0x620a, 0x96fe, 0x6664, 0x7269,
	0x52ff, 0x52a1, 0x609f, 0x8bef, 0x6614, 0x7199, 0x6790, 0x897f,
	0x7852, 0x77fd, 0x6670, 0x563b, 0x5438, 0x9521, 0x727a, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xcf40[ 192 ] = {
	0x87a5, 0x87a6, 0x87a7, 0x87a9, 0x87aa, 0x87ae, 0x87b0, 0x87b1,
	0x87b2, 0x87b4, 0x87b6, 0x87b7, 0x87b8, 0x87b9, 0x87bb, 0x87bc,
	0x87be, 0x87bf, 0x87c1, 0x87c2, 0x87c3, 0x87c4, 0x87c5, 0x87c7,
	0x87c8, 0x87c9, 0x87cc, 0x87cd, 0x87ce, 0x87cf, 0x87d0, 0x87d4,
	0x87d5, 0x87d6, 0x87d7, 0x87d8, 0x87d9, 0x87da, 0x87dc, 0x87dd,
	0x87de, 0x87df, 0x87e1, 0x87e2, 0x87e3, 0x87e4, 0x87e6, 0x87e7,
	0x87e8, 0x87e9, 0x87eb, 0x87ec, 0x87ed, 0x87ef, 0x87f0, 0x87f1,
	0x87f2, 0x87f3, 0x87f4, 0x87f5, 0x87f6, 0x87f7, 0x87f8, 0xfffd,
	0x87fa, 0x87fb, 0x87fc, 0x87fd, 0x87ff, 0x8800, 0x8801, 0x8802,
	0x8804, 0x8805, 0x8806, 0x8807, 0x8808, 0x8809, 0x880b, 0x880c,
	0x880d, 0x880e, 0x880f, 0x8810, 0x8811, 0x8812, 0x8814, 0x8817,
	0x8818, 0x8819, 0x881a, 0x881c, 0x881d, 0x881e, 0x881f, 0x8820,
	0x8823, 0x7a00, 0x606f, 0x5e0c, 0x6089, 0x819d, 0x5915, 0x60dc,
	0x7184, 0x70ef, 0x6eaa, 0x6c50, 0x7280, 0x6a84, 0x88ad, 0x5e2d,
	0x4e60, 0x5ab3, 0x559c, 0x94e3, 0x6d17, 0x7cfb, 0x9699, 0x620f,
	0x7ec6, 0x778e, 0x867e, 0x5323, 0x971e, 0x8f96, 0x6687, 0x5ce1,
	0x4fa0, 0x72ed, 0x4e0b, 0x53a6, 0x590f, 0x5413, 0x6380, 0x9528,
	0x5148, 0x4ed9, 0x9c9c, 0x7ea4, 0x54b8, 0x8d24, 0x8854, 0x8237,
	0x95f2, 0x6d8e, 0x5f26, 0x5acc, 0x663e, 0x9669, 0x73b0, 0x732e,
	0x53bf, 0x817a, 0x9985, 0x7fa1, 0x5baa, 0x9677, 0x9650, 0x7ebf,
	0x76f8, 0x53a2, 0x9576, 0x9999, 0x7bb1, 0x8944, 0x6e58, 0x4e61,
	0x7fd4, 0x7965, 0x8be6, 0x60f3, 0x54cd, 0x4eab, 0x9879, 0x5df7,
	0x6a61, 0x50cf, 0x5411, 0x8c61, 0x8427, 0x785d, 0x9704, 0x524a,
	0x54ee, 0x56a3, 0x9500, 0x6d88, 0x5bb5, 0x6dc6, 0x6653, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd040[ 192 ] = {
	0x8824, 0x8825, 0x8826, 0x8827, 0x8828, 0x8829, 0x882a, 0x882b,
	0x882c, 0x882d, 0x882e, 0x882f, 0x8830, 0x8831, 0x8833, 0x8834,
	0x8835, 0x8836, 0x8837, 0x8838, 0x883a, 0x883b, 0x883d, 0x883e,
	0x883f, 0x8841, 0x8842, 0x8843, 0x8846, 0x8847, 0x8848, 0x8849,
	0x884a, 0x884b, 0x884e, 0x884f, 0x8850, 0x8851, 0x8852, 0x8853,
	0x8855, 0x8856, 0x8858, 0x885a, 0x885b, 0x885c, 0x885d, 0x885e,
	0x885f, 0x8860, 0x8866, 0x8867, 0x886a, 0x886d, 0x886f, 0x8871,
	0x8873, 0x8874, 0x8875, 0x8876, 0x8878, 0x8879, 0x887a, 0xfffd,
	0x887b, 0x887c, 0x8880, 0x8883, 0x8886, 0x8887, 0x8889, 0x888a,
	0x888c, 0x888e, 0x888f, 0x8890, 0x8891, 0x8893, 0x8894, 0x8895,
	0x8897, 0x8898, 0x8899, 0x889a, 0x889b, 0x889d, 0x889e, 0x889f,
	0x88a0, 0x88a1, 0x88a3, 0x88a5, 0x88a6, 0x88a7, 0x88a8, 0x88a9,
	0x88aa, 0x5c0f, 0x5b5d, 0x6821, 0x8096, 0x5578, 0x7b11, 0x6548,
	0x6954, 0x4e9b, 0x6b47, 0x874e, 0x978b, 0x534f, 0x631f, 0x643a,
	0x90aa, 0x659c, 0x80c1, 0x8c10, 0x5199, 0x68b0, 0x5378, 0x87f9,
	0x61c8, 0x6cc4, 0x6cfb, 0x8c22, 0x5c51, 0x85aa, 0x82af, 0x950c,
	0x6b23, 0x8f9b, 0x65b0, 0x5ffb, 0x5fc3, 0x4fe1, 0x8845, 0x661f,
	0x8165, 0x7329, 0x60fa, 0x5174, 0x5211, 0x578b, 0x5f62, 0x90a2,
	0x884c, 0x9192, 0x5e78, 0x674f, 0x6027, 0x59d3, 0x5144, 0x51f6,
	0x80f8, 0x5308, 0x6c79, 0x96c4, 0x718a, 0x4f11, 0x4fee, 0x7f9e,
	0x673d, 0x55c5, 0x9508, 0x79c0, 0x8896, 0x7ee3, 0x589f, 0x620c,
	0x9700, 0x865a, 0x5618, 0x987b, 0x5f90, 0x8bb8, 0x84c4, 0x9157,
	0x53d9, 0x65ed, 0x5e8f, 0x755c, 0x6064, 0x7d6e, 0x5a7f, 0x7eea,
	0x7eed, 0x8f69, 0x55a7, 0x5ba3, 0x60ac, 0x65cb, 0x7384, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd140[ 192 ] = {
	0x88ac, 0x88ae, 0x88af, 0x88b0, 0x88b2, 0x88b3, 0x88b4, 0x88b5,
	0x88b6, 0x88b8, 0x88b9, 0x88ba, 0x88bb, 0x88bd, 0x88be, 0x88bf,
	0x88c0, 0x88c3, 0x88c4, 0x88c7, 0x88c8, 0x88ca, 0x88cb, 0x88cc,
	0x88cd, 0x88cf, 0x88d0, 0x88d1, 0x88d3, 0x88d6, 0x88d7, 0x88da,
	0x88db, 0x88dc, 0x88dd, 0x88de, 0x88e0, 0x88e1, 0x88e6, 0x88e7,
	0x88e9, 0x88ea, 0x88eb, 0x88ec, 0x88ed, 0x88ee, 0x88ef, 0x88f2,
	0x88f5, 0x88f6, 0x88f7, 0x88fa, 0x88fb, 0x88fd, 0x88ff, 0x8900,
	0x8901, 0x8903, 0x8904, 0x8905, 0x8906, 0x8907, 0x8908, 0xfffd,
	0x8909, 0x890b, 0x890c, 0x890d, 0x890e, 0x890f, 0x8911, 0x8914,
	0x8915, 0x8916, 0x8917, 0x8918, 0x891c, 0x891d, 0x891e, 0x891f,
	0x8920, 0x8922, 0x8923, 0x8924, 0x8926, 0x8927, 0x8928, 0x8929,
	0x892c, 0x892d, 0x892e, 0x892f, 0x8931, 0x8932, 0x8933, 0x8935,
	0x8937, 0x9009, 0x7663, 0x7729, 0x7eda, 0x9774, 0x859b, 0x5b66,
	0x7a74, 0x96ea, 0x8840, 0x52cb, 0x718f, 0x5faa, 0x65ec, 0x8be2,
	0x5bfb, 0x9a6f, 0x5de1, 0x6b89, 0x6c5b, 0x8bad, 0x8baf, 0x900a,
	0x8fc5, 0x538b, 0x62bc, 0x9e26, 0x9e2d, 0x5440, 0x4e2b, 0x82bd,
	0x7259, 0x869c, 0x5d16, 0x8859, 0x6daf, 0x96c5, 0x54d1, 0x4e9a,
	0x8bb6, 0x7109, 0x54bd, 0x9609, 0x70df, 0x6df9, 0x76d0, 0x4e25,
	0x7814, 0x8712, 0x5ca9, 0x5ef6, 0x8a00, 0x989c, 0x960e, 0x708e,
	0x6cbf, 0x5944, 0x63a9, 0x773c, 0x884d, 0x6f14, 0x8273, 0x5830,
	0x71d5, 0x538c, 0x781a, 0x96c1, 0x5501, 0x5f66, 0x7130, 0x5bb4,
	0x8c1a, 0x9a8c, 0x6b83, 0x592e, 0x9e2f, 0x79e7, 0x6768, 0x626c,
	0x4f6f, 0x75a1, 0x7f8a, 0x6d0b, 0x9633, 0x6c27, 0x4ef0, 0x75d2,
	0x517b, 0x6837, 0x6f3e, 0x9080, 0x8170, 0x5996, 0x7476, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd240[ 192 ] = {
	0x8938, 0x8939, 0x893a, 0x893b, 0x893c, 0x893d, 0x893e, 0x893f,
	0x8940, 0x8942, 0x8943, 0x8945, 0x8946, 0x8947, 0x8948, 0x8949,
	0x894a, 0x894b, 0x894c, 0x894d, 0x894e, 0x894f, 0x8950, 0x8951,
	0x8952, 0x8953, 0x8954, 0x8955, 0x8956, 0x8957, 0x8958, 0x8959,
	0x895a, 0x895b, 0x895c, 0x895d, 0x8960, 0x8961, 0x8962, 0x8963,
	0x8964, 0x8965, 0x8967, 0x8968, 0x8969, 0x896a, 0x896b, 0x896c,
	0x896d, 0x896e, 0x896f, 0x8970, 0x8971, 0x8972, 0x8973, 0x8974,
	0x8975, 0x8976, 0x8977, 0x8978, 0x8979, 0x897a, 0x897c, 0xfffd,
	0x897d, 0x897e, 0x8980, 0x8982, 0x8984, 0x8985, 0x8987, 0x8988,
	0x8989, 0x898a, 0x898b, 0x898c, 0x898d, 0x898e, 0x898f, 0x8990,
	0x8991, 0x8992, 0x8993, 0x8994, 0x8995, 0x8996, 0x8997, 0x8998,
	0x8999, 0x899a, 0x899b, 0x899c, 0x899d, 0x899e, 0x899f, 0x89a0,
	0x89a1, 0x6447, 0x5c27, 0x9065, 0x7a91, 0x8c23, 0x59da, 0x54ac,
	0x8200, 0x836f, 0x8981, 0x8000, 0x6930, 0x564e, 0x8036, 0x7237,
	0x91ce, 0x51b6, 0x4e5f, 0x9875, 0x6396, 0x4e1a, 0x53f6, 0x66f3,
	0x814b, 0x591c, 0x6db2, 0x4e00, 0x58f9, 0x533b, 0x63d6, 0x94f1,
	0x4f9d, 0x4f0a, 0x8863, 0x9890, 0x5937, 0x9057, 0x79fb, 0x4eea,
	0x80f0, 0x7591, 0x6c82, 0x5b9c, 0x59e8, 0x5f5d, 0x6905, 0x8681,
	0x501a, 0x5df2, 0x4e59, 0x77e3, 0x4ee5, 0x827a, 0x6291, 0x6613,
	0x9091, 0x5c79, 0x4ebf, 0x5f79, 0x81c6, 0x9038, 0x8084, 0x75ab,
	0x4ea6, 0x88d4, 0x610f, 0x6bc5, 0x5fc6, 0x4e49, 0x76ca, 0x6ea2,
	0x8be3, 0x8bae, 0x8c0a, 0x8bd1, 0x5f02, 0x7ffc, 0x7fcc, 0x7ece,
	0x8335, 0x836b, 0x56e0, 0x6bb7, 0x97f3, 0x9634, 0x59fb, 0x541f,
	0x94f6, 0x6deb, 0x5bc5, 0x996e, 0x5c39, 0x5f15, 0x9690, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd340[ 192 ] = {
	0x89a2, 0x89a3, 0x89a4, 0x89a5, 0x89a6, 0x89a7, 0x89a8, 0x89a9,
	0x89aa, 0x89ab, 0x89ac, 0x89ad, 0x89ae, 0x89af, 0x89b0, 0x89b1,
	0x89b2, 0x89b3, 0x89b4, 0x89b5, 0x89b6, 0x89b7, 0x89b8, 0x89b9,
	0x89ba, 0x89bb, 0x89bc, 0x89bd, 0x89be, 0x89bf, 0x89c0, 0x89c3,
	0x89cd, 0x89d3, 0x89d4, 0x89d5, 0x89d7, 0x89d8, 0x89d9, 0x89db,
	0x89dd, 0x89df, 0x89e0, 0x89e1, 0x89e2, 0x89e4, 0x89e7, 0x89e8,
	0x89e9, 0x89ea, 0x89ec, 0x89ed, 0x89ee, 0x89f0, 0x89f1, 0x89f2,
	0x89f4, 0x89f5, 0x89f6, 0x89f7, 0x89f8, 0x89f9, 0x89fa, 0xfffd,
	0x89fb, 0x89fc, 0x89fd, 0x89fe, 0x89ff, 0x8a01, 0x8a02, 0x8a03,
	0x8a04, 0x8a05, 0x8a06, 0x8a08, 0x8a09, 0x8a0a, 0x8a0b, 0x8a0c,
	0x8a0d, 0x8a0e, 0x8a0f, 0x8a10, 0x8a11, 0x8a12, 0x8a13, 0x8a14,
	0x8a15, 0x8a16, 0x8a17, 0x8a18, 0x8a19, 0x8a1a, 0x8a1b, 0x8a1c,
	0x8a1d, 0x5370, 0x82f1, 0x6a31, 0x5a74, 0x9e70, 0x5e94, 0x7f28,
	0x83b9, 0x8424, 0x8425, 0x8367, 0x8747, 0x8fce, 0x8d62, 0x76c8,
	0x5f71, 0x9896, 0x786c, 0x6620, 0x54df, 0x62e5, 0x4f63, 0x81c3,
	0x75c8, 0x5eb8, 0x96cd, 0x8e0a, 0x86f9, 0x548f, 0x6cf3, 0x6d8c,
	0x6c38, 0x607f, 0x52c7, 0x7528, 0x5e7d, 0x4f18, 0x60a0, 0x5fe7,
	0x5c24, 0x7531, 0x90ae, 0x94c0, 0x72b9, 0x6cb9, 0x6e38, 0x9149,
	0x6709, 0x53cb, 0x53f3, 0x4f51, 0x91c9, 0x8bf1, 0x53c8, 0x5e7c,
	0x8fc2, 0x6de4, 0x4e8e, 0x76c2, 0x6986, 0x865e, 0x611a, 0x8206,
	0x4f59, 0x4fde, 0x903e, 0x9c7c, 0x6109, 0x6e1d, 0x6e14, 0x9685,
	0x4e88, 0x5a31, 0x96e8, 0x4e0e, 0x5c7f, 0x79b9, 0x5b87, 0x8bed,
	0x7fbd, 0x7389, 0x57df, 0x828b, 0x90c1, 0x5401, 0x9047, 0x55bb,
	0x5cea, 0x5fa1, 0x6108, 0x6b32, 0x72f1, 0x80b2, 0x8a89, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd440[ 192 ] = {
	0x8a1e, 0x8a1f, 0x8a20, 0x8a21, 0x8a22, 0x8a23, 0x8a24, 0x8a25,
	0x8a26, 0x8a27, 0x8a28, 0x8a29, 0x8a2a, 0x8a2b, 0x8a2c, 0x8a2d,
	0x8a2e, 0x8a2f, 0x8a30, 0x8a31, 0x8a32, 0x8a33, 0x8a34, 0x8a35,
	0x8a36, 0x8a37, 0x8a38, 0x8a39, 0x8a3a, 0x8a3b, 0x8a3c, 0x8a3d,
	0x8a3f, 0x8a40, 0x8a41, 0x8a42, 0x8a43, 0x8a44, 0x8a45, 0x8a46,
	0x8a47, 0x8a49, 0x8a4a, 0x8a4b, 0x8a4c, 0x8a4d, 0x8a4e, 0x8a4f,
	0x8a50, 0x8a51, 0x8a52, 0x8a53, 0x8a54, 0x8a55, 0x8a56, 0x8a57,
	0x8a58, 0x8a59, 0x8a5a, 0x8a5b, 0x8a5c, 0x8a5d, 0x8a5e, 0xfffd,
	0x8a5f, 0x8a60, 0x8a61, 0x8a62, 0x8a63, 0x8a64, 0x8a65, 0x8a66,
	0x8a67, 0x8a68, 0x8a69, 0x8a6a, 0x8a6b, 0x8a6c, 0x8a6d, 0x8a6e,
	0x8a6f, 0x8a70, 0x8a71, 0x8a72, 0x8a73, 0x8a74, 0x8a75, 0x8a76,
	0x8a77, 0x8a78, 0x8a7a, 0x8a7b, 0x8a7c, 0x8a7d, 0x8a7e, 0x8a7f,
	0x8a80, 0x6d74, 0x5bd3, 0x88d5, 0x9884, 0x8c6b, 0x9a6d, 0x9e33,
	0x6e0a, 0x51a4, 0x5143, 0x57a3, 0x8881, 0x539f, 0x63f4, 0x8f95,
	0x56ed, 0x5458, 0x5706, 0x733f, 0x6e90, 0x7f18, 0x8fdc, 0x82d1,
	0x613f, 0x6028, 0x9662, 0x66f0, 0x7ea6, 0x8d8a, 0x8dc3, 0x94a5,
	0x5cb3, 0x7ca4, 0x6708, 0x60a6, 0x9605, 0x8018, 0x4e91, 0x90e7,
	0x5300, 0x9668, 0x5141, 0x8fd0, 0x8574, 0x915d, 0x6655, 0x97f5,
	0x5b55, 0x531d, 0x7838, 0x6742, 0x683d, 0x54c9, 0x707e, 0x5bb0,
	0x8f7d, 0x518d, 0x5728, 0x54b1, 0x6512, 0x6682, 0x8d5e, 0x8d43,
	0x810f, 0x846c, 0x906d, 0x7cdf, 0x51ff, 0x85fb, 0x67a3, 0x65e9,
	0x6fa1, 0x86a4, 0x8e81, 0x566a, 0x9020, 0x7682, 0x7076, 0x71e5,
	0x8d23, 0x62e9, 0x5219, 0x6cfd, 0x8d3c, 0x600e, 0x589e, 0x618e,
	0x66fe, 0x8d60, 0x624e, 0x55b3, 0x6e23, 0x672d, 0x8f67, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd540[ 192 ] = {
	0x8a81, 0x8a82, 0x8a83, 0x8a84, 0x8a85, 0x8a86, 0x8a87, 0x8a88,
	0x8a8b, 0x8a8c, 0x8a8d, 0x8a8e, 0x8a8f, 0x8a90, 0x8a91, 0x8a92,
	0x8a94, 0x8a95, 0x8a96, 0x8a97, 0x8a98, 0x8a99, 0x8a9a, 0x8a9b,
	0x8a9c, 0x8a9d, 0x8a9e, 0x8a9f, 0x8aa0, 0x8aa1, 0x8aa2, 0x8aa3,
	0x8aa4, 0x8aa5, 0x8aa6, 0x8aa7, 0x8aa8, 0x8aa9, 0x8aaa, 0x8aab,
	0x8aac, 0x8aad, 0x8aae, 0x8aaf, 0x8ab0, 0x8ab1, 0x8ab2, 0x8ab3,
	0x8ab4, 0x8ab5, 0x8ab6, 0x8ab7, 0x8ab8, 0x8ab9, 0x8aba, 0x8abb,
	0x8abc, 0x8abd, 0x8abe, 0x8abf, 0x8ac0, 0x8ac1, 0x8ac2, 0xfffd,
	0x8ac3, 0x8ac4, 0x8ac5, 0x8ac6, 0x8ac7, 0x8ac8, 0x8ac9, 0x8aca,
	0x8acb, 0x8acc, 0x8acd, 0x8ace, 0x8acf, 0x8ad0, 0x8ad1, 0x8ad2,
	0x8ad3, 0x8ad4, 0x8ad5, 0x8ad6, 0x8ad7, 0x8ad8, 0x8ad9, 0x8ada,
	0x8adb, 0x8adc, 0x8add, 0x8ade, 0x8adf, 0x8ae0, 0x8ae1, 0x8ae2,
	0x8ae3, 0x94e1, 0x95f8, 0x7728, 0x6805, 0x69a8, 0x548b, 0x4e4d,
	0x70b8, 0x8bc8, 0x6458, 0x658b, 0x5b85, 0x7a84, 0x503a, 0x5be8,
	0x77bb, 0x6be1, 0x8a79, 0x7c98, 0x6cbe, 0x76cf, 0x65a9, 0x8f97,
	0x5d2d, 0x5c55, 0x8638, 0x6808, 0x5360, 0x6218, 0x7ad9, 0x6e5b,
	0x7efd, 0x6a1f, 0x7ae0, 0x5f70, 0x6f33, 0x5f20, 0x638c, 0x6da8,
	0x6756, 0x4e08, 0x5e10, 0x8d26, 0x4ed7, 0x80c0, 0x7634, 0x969c,
	0x62db, 0x662d, 0x627e, 0x6cbc, 0x8d75, 0x7167, 0x7f69, 0x5146,
	0x8087, 0x53ec, 0x906e, 0x6298, 0x54f2, 0x86f0, 0x8f99, 0x8005,
	0x9517, 0x8517, 0x8fd9, 0x6d59, 0x73cd, 0x659f, 0x771f, 0x7504,
	0x7827, 0x81fb, 0x8d1e, 0x9488, 0x4fa6, 0x6795, 0x75b9, 0x8bca,
	0x9707, 0x632f, 0x9547, 0x9635, 0x84b8, 0x6323, 0x7741, 0x5f81,
	0x72f0, 0x4e89, 0x6014, 0x6574, 0x62ef, 0x6b63, 0x653f, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd640[ 192 ] = {
	0x8ae4, 0x8ae5, 0x8ae6, 0x8ae7, 0x8ae8, 0x8ae9, 0x8aea, 0x8aeb,
	0x8aec, 0x8aed, 0x8aee, 0x8aef, 0x8af0, 0x8af1, 0x8af2, 0x8af3,
	0x8af4, 0x8af5, 0x8af6, 0x8af7, 0x8af8, 0x8af9, 0x8afa, 0x8afb,
	0x8afc, 0x8afd, 0x8afe, 0x8aff, 0x8b00, 0x8b01, 0x8b02, 0x8b03,
	0x8b04, 0x8b05, 0x8b06, 0x8b08, 0x8b09, 0x8b0a, 0x8b0b, 0x8b0c,
	0x8b0d, 0x8b0e, 0x8b0f, 0x8b10, 0x8b11, 0x8b12, 0x8b13, 0x8b14,
	0x8b15, 0x8b16, 0x8b17, 0x8b18, 0x8b19, 0x8b1a, 0x8b1b, 0x8b1c,
	0x8b1d, 0x8b1e, 0x8b1f, 0x8b20, 0x8b21, 0x8b22, 0x8b23, 0xfffd,
	0x8b24, 0x8b25, 0x8b27, 0x8b28, 0x8b29, 0x8b2a, 0x8b2b, 0x8b2c,
	0x8b2d, 0x8b2e, 0x8b2f, 0x8b30, 0x8b31, 0x8b32, 0x8b33, 0x8b34,
	0x8b35, 0x8b36, 0x8b37, 0x8b38, 0x8b39, 0x8b3a, 0x8b3b, 0x8b3c,
	0x8b3d, 0x8b3e, 0x8b3f, 0x8b40, 0x8b41, 0x8b42, 0x8b43, 0x8b44,
	0x8b45, 0x5e27, 0x75c7, 0x90d1, 0x8bc1, 0x829d, 0x679d, 0x652f,
	0x5431, 0x8718, 0x77e5, 0x80a2, 0x8102, 0x6c41, 0x4e4b, 0x7ec7,
	0x804c, 0x76f4, 0x690d, 0x6b96, 0x6267, 0x503c, 0x4f84, 0x5740,
	0x6307, 0x6b62, 0x8dbe, 0x53ea, 0x65e8, 0x7eb8, 0x5fd7, 0x631a,
	0x63b7, 0x81f3, 0x81f4, 0x7f6e, 0x5e1c, 0x5cd9, 0x5236, 0x667a,
	0x79e9, 0x7a1a, 0x8d28, 0x7099, 0x75d4, 0x6ede, 0x6cbb, 0x7a92,
	0x4e2d, 0x76c5, 0x5fe0, 0x949f, 0x8877, 0x7ec8, 0x79cd, 0x80bf,
	0x91cd, 0x4ef2, 0x4f17, 0x821f, 0x5468, 0x5dde, 0x6d32, 0x8bcc,
	0x7ca5, 0x8f74, 0x8098, 0x5e1a, 0x5492, 0x76b1, 0x5b99, 0x663c,
	0x9aa4, 0x73e0, 0x682a, 0x86db, 0x6731, 0x732a, 0x8bf8, 0x8bdb,
	0x9010, 0x7af9, 0x70db, 0x716e, 0x62c4, 0x77a9, 0x5631, 0x4e3b,
	0x8457, 0x67f1, 0x52a9, 0x86c0, 0x8d2e, 0x94f8, 0x7b51, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd740[ 192 ] = {
	0x8b46, 0x8b47, 0x8b48, 0x8b49, 0x8b4a, 0x8b4b, 0x8b4c, 0x8b4d,
	0x8b4e, 0x8b4f, 0x8b50, 0x8b51, 0x8b52, 0x8b53, 0x8b54, 0x8b55,
	0x8b56, 0x8b57, 0x8b58, 0x8b59, 0x8b5a, 0x8b5b, 0x8b5c, 0x8b5d,
	0x8b5e, 0x8b5f, 0x8b60, 0x8b61, 0x8b62, 0x8b63, 0x8b64, 0x8b65,
	0x8b67, 0x8b68, 0x8b69, 0x8b6a, 0x8b6b, 0x8b6d, 0x8b6e, 0x8b6f,
	0x8b70, 0x8b71, 0x8b72, 0x8b73, 0x8b74, 0x8b75, 0x8b76, 0x8b77,
	0x8b78, 0x8b79, 0x8b7a, 0x8b7b, 0x8b7c, 0x8b7d, 0x8b7e, 0x8b7f,
	0x8b80, 0x8b81, 0x8b82, 0x8b83, 0x8b84, 0x8b85, 0x8b86, 0xfffd,
	0x8b87, 0x8b88, 0x8b89, 0x8b8a, 0x8b8b, 0x8b8c, 0x8b8d, 0x8b8e,
	0x8b8f, 0x8b90, 0x8b91, 0x8b92, 0x8b93, 0x8b94, 0x8b95, 0x8b96,
	0x8b97, 0x8b98, 0x8b99, 0x8b9a, 0x8b9b, 0x8b9c, 0x8b9d, 0x8b9e,
	0x8b9f, 0x8bac, 0x8bb1, 0x8bbb, 0x8bc7, 0x8bd0, 0x8bea, 0x8c09,
	0x8c1e, 0x4f4f, 0x6ce8, 0x795d, 0x9a7b, 0x6293, 0x722a, 0x62fd,
	0x4e13, 0x7816, 0x8f6c, 0x64b0, 0x8d5a, 0x7bc6, 0x6869, 0x5e84,
	0x88c5, 0x5986, 0x649e, 0x58ee, 0x72b6, 0x690e, 0x9525, 0x8ffd,
	0x8d58, 0x5760, 0x7f00, 0x8c06, 0x51c6, 0x6349, 0x62d9, 0x5353,
	0x684c, 0x7422, 0x8301, 0x914c, 0x5544, 0x7740, 0x707c, 0x6d4a,
	0x5179, 0x54a8, 0x8d44, 0x59ff, 0x6ecb, 0x6dc4, 0x5b5c, 0x7d2b,
	0x4ed4, 0x7c7d, 0x6ed3, 0x5b50, 0x81ea, 0x6e0d, 0x5b57, 0x9b03,
	0x68d5, 0x8e2a, 0x5b97, 0x7efc, 0x603b, 0x7eb5, 0x90b9, 0x8d70,
	0x594f, 0x63cd, 0x79df, 0x8db3, 0x5352, 0x65cf, 0x7956, 0x8bc5,
	0x963b, 0x7ec4, 0x94bb, 0x7e82, 0x5634, 0x9189, 0x6700, 0x7f6a,
	0x5c0a, 0x9075, 0x6628, 0x5de6, 0x4f50, 0x67de, 0x505a, 0x4f5c,
	0x5750, 0x5ea7, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd840[ 192 ] = {
	0x8c38, 0x8c39, 0x8c3a, 0x8c3b, 0x8c3c, 0x8c3d, 0x8c3e, 0x8c3f,
	0x8c40, 0x8c42, 0x8c43, 0x8c44, 0x8c45, 0x8c48, 0x8c4a, 0x8c4b,
	0x8c4d, 0x8c4e, 0x8c4f, 0x8c50, 0x8c51, 0x8c52, 0x8c53, 0x8c54,
	0x8c56, 0x8c57, 0x8c58, 0x8c59, 0x8c5b, 0x8c5c, 0x8c5d, 0x8c5e,
	0x8c5f, 0x8c60, 0x8c63, 0x8c64, 0x8c65, 0x8c66, 0x8c67, 0x8c68,
	0x8c69, 0x8c6c, 0x8c6d, 0x8c6e, 0x8c6f, 0x8c70, 0x8c71, 0x8c72,
	0x8c74, 0x8c75, 0x8c76, 0x8c77, 0x8c7b, 0x8c7c, 0x8c7d, 0x8c7e,
	0x8c7f, 0x8c80, 0x8c81, 0x8c83, 0x8c84, 0x8c86, 0x8c87, 0xfffd,
	0x8c88, 0x8c8b, 0x8c8d, 0x8c8e, 0x8c8f, 0x8c90, 0x8c91, 0x8c92,
	0x8c93, 0x8c95, 0x8c96, 0x8c97, 0x8c99, 0x8c9a, 0x8c9b, 0x8c9c,
	0x8c9d, 0x8c9e, 0x8c9f, 0x8ca0, 0x8ca1, 0x8ca2, 0x8ca3, 0x8ca4,
	0x8ca5, 0x8ca6, 0x8ca7, 0x8ca8, 0x8ca9, 0x8caa, 0x8cab, 0x8cac,
	0x8cad, 0x4e8d, 0x4e0c, 0x5140, 0x4e10, 0x5eff, 0x5345, 0x4e15,
	0x4e98, 0x4e1e, 0x9b32, 0x5b6c, 0x5669, 0x4e28, 0x79ba, 0x4e3f,
	0x5315, 0x4e47, 0x592d, 0x723b, 0x536e, 0x6c10, 0x56df, 0x80e4,
	0x9997, 0x6bd3, 0x777e, 0x9f17, 0x4e36, 0x4e9f, 0x9f10, 0x4e5c,
	0x4e69, 0x4e93, 0x8288, 0x5b5b, 0x556c, 0x560f, 0x4ec4, 0x538d,
	0x539d, 0x53a3, 0x53a5, 0x53ae, 0x9765, 0x8d5d, 0x531a, 0x53f5,
	0x5326, 0x532e, 0x533e, 0x8d5c, 0x5366, 0x5363, 0x5202, 0x5208,
	0x520e, 0x522d, 0x5233, 0x523f, 0x5240, 0x524c, 0x525e, 0x5261,
	0x525c, 0x84af, 0x527d, 0x5282, 0x5281, 0x5290, 0x5293, 0x5182,
	0x7f54, 0x4ebb, 0x4ec3, 0x4ec9, 0x4ec2, 0x4ee8, 0x4ee1, 0x4eeb,
	0x4ede, 0x4f1b, 0x4ef3, 0x4f22, 0x4f64, 0x4ef5, 0x4f25, 0x4f27,
	0x4f09, 0x4f2b, 0x4f5e, 0x4f67, 0x6538, 0x4f5a, 0x4f5d, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd940[ 192 ] = {
	0x8cae, 0x8caf, 0x8cb0, 0x8cb1, 0x8cb2, 0x8cb3, 0x8cb4, 0x8cb5,
	0x8cb6, 0x8cb7, 0x8cb8, 0x8cb9, 0x8cba, 0x8cbb, 0x8cbc, 0x8cbd,
	0x8cbe, 0x8cbf, 0x8cc0, 0x8cc1, 0x8cc2, 0x8cc3, 0x8cc4, 0x8cc5,
	0x8cc6, 0x8cc7, 0x8cc8, 0x8cc9, 0x8cca, 0x8ccb, 0x8ccc, 0x8ccd,
	0x8cce, 0x8ccf, 0x8cd0, 0x8cd1, 0x8cd2, 0x8cd3, 0x8cd4, 0x8cd5,
	0x8cd6, 0x8cd7, 0x8cd8, 0x8cd9, 0x8cda, 0x8cdb, 0x8cdc, 0x8cdd,
	0x8cde, 0x8cdf, 0x8ce0, 0x8ce1, 0x8ce2, 0x8ce3, 0x8ce4, 0x8ce5,
	0x8ce6, 0x8ce7, 0x8ce8, 0x8ce9, 0x8cea, 0x8ceb, 0x8cec, 0xfffd,
	0x8ced, 0x8cee, 0x8cef, 0x8cf0, 0x8cf1, 0x8cf2, 0x8cf3, 0x8cf4,
	0x8cf5, 0x8cf6, 0x8cf7, 0x8cf8, 0x8cf9, 0x8cfa, 0x8cfb, 0x8cfc,
	0x8cfd, 0x8cfe, 0x8cff, 0x8d00, 0x8d01, 0x8d02, 0x8d03, 0x8d04,
	0x8d05, 0x8d06, 0x8d07, 0x8d08, 0x8d09, 0x8d0a, 0x8d0b, 0x8d0c,
	0x8d0d, 0x4f5f, 0x4f57, 0x4f32, 0x4f3d, 0x4f76, 0x4f74, 0x4f91,
	0x4f89, 0x4f83, 0x4f8f, 0x4f7e, 0x4f7b, 0x4faa, 0x4f7c, 0x4fac,
	0x4f94, 0x4fe6, 0x4fe8, 0x4fea, 0x4fc5, 0x4fda, 0x4fe3, 0x4fdc,
	0x4fd1, 0x4fdf, 0x4ff8, 0x5029, 0x504c, 0x4ff3, 0x502c, 0x500f,
	0x502e, 0x502d, 0x4ffe, 0x501c, 0x500c, 0x5025, 0x5028, 0x507e,
	0x5043, 0x5055, 0x5048, 0x504e, 0x506c, 0x507b, 0x50a5, 0x50a7,
	0x50a9, 0x50ba, 0x50d6, 0x5106, 0x50ed, 0x50ec, 0x50e6, 0x50ee,
	0x5107, 0x510b, 0x4edd, 0x6c3d, 0x4f58, 0x4f65, 0x4fce, 0x9fa0,
	0x6c46, 0x7c74, 0x516e, 0x5dfd, 0x9ec9, 0x9998, 0x5181, 0x5914,
	0x52f9, 0x530d, 0x8a07, 0x5310, 0x51eb, 0x5919, 0x5155, 0x4ea0,
	0x5156, 0x4eb3, 0x886e, 0x88a4, 0x4eb5, 0x8114, 0x88d2, 0x7980,
	0x5b34, 0x8803, 0x7fb8, 0x51ab, 0x51b1, 0x51bd, 0x51bc, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xda40[ 192 ] = {
	0x8d0e, 0x8d0f, 0x8d10, 0x8d11, 0x8d12, 0x8d13, 0x8d14, 0x8d15,
	0x8d16, 0x8d17, 0x8d18, 0x8d19, 0x8d1a, 0x8d1b, 0x8d1c, 0x8d20,
	0x8d51, 0x8d52, 0x8d57, 0x8d5f, 0x8d65, 0x8d68, 0x8d69, 0x8d6a,
	0x8d6c, 0x8d6e, 0x8d6f, 0x8d71, 0x8d72, 0x8d78, 0x8d79, 0x8d7a,
	0x8d7b, 0x8d7c, 0x8d7d, 0x8d7e, 0x8d7f, 0x8d80, 0x8d82, 0x8d83,
	0x8d86, 0x8d87, 0x8d88, 0x8d89, 0x8d8c, 0x8d8d, 0x8d8e, 0x8d8f,
	0x8d90, 0x8d92, 0x8d93, 0x8d95, 0x8d96, 0x8d97, 0x8d98, 0x8d99,
	0x8d9a, 0x8d9b, 0x8d9c, 0x8d9d, 0x8d9e, 0x8da0, 0x8da1, 0xfffd,
	0x8da2, 0x8da4, 0x8da5, 0x8da6, 0x8da7, 0x8da8, 0x8da9, 0x8daa,
	0x8dab, 0x8dac, 0x8dad, 0x8dae, 0x8daf, 0x8db0, 0x8db2, 0x8db6,
	0x8db7, 0x8db9, 0x8dbb, 0x8dbd, 0x8dc0, 0x8dc1, 0x8dc2, 0x8dc5,
	0x8dc7, 0x8dc8, 0x8dc9, 0x8dca, 0x8dcd, 0x8dd0, 0x8dd2, 0x8dd3,
	0x8dd4, 0x51c7, 0x5196, 0x51a2, 0x51a5, 0x8ba0, 0x8ba6, 0x8ba7,
	0x8baa, 0x8bb4, 0x8bb5, 0x8bb7, 0x8bc2, 0x8bc3, 0x8bcb, 0x8bcf,
	0x8bce, 0x8bd2, 0x8bd3, 0x8bd4, 0x8bd6, 0x8bd8, 0x8bd9, 0x8bdc,
	0x8bdf, 0x8be0, 0x8be4, 0x8be8, 0x8be9, 0x8bee, 0x8bf0, 0x8bf3,
	0x8bf6, 0x8bf9, 0x8bfc, 0x8bff, 0x8c00, 0x8c02, 0x8c04, 0x8c07,
	0x8c0c, 0x8c0f, 0x8c11, 0x8c12, 0x8c14, 0x8c15, 0x8c16, 0x8c19,
	0x8c1b, 0x8c18, 0x8c1d, 0x8c1f, 0x8c20, 0x8c21, 0x8c25, 0x8c27,
	0x8c2a, 0x8c2b, 0x8c2e, 0x8c2f, 0x8c32, 0x8c33, 0x8c35, 0x8c36,
	0x5369, 0x537a, 0x961d, 0x9622, 0x9621, 0x9631, 0x962a, 0x963d,
	0x963c, 0x9642, 0x9649, 0x9654, 0x965f, 0x9667, 0x966c, 0x9672,
	0x9674, 0x9688, 0x968d, 0x9697, 0x96b0, 0x9097, 0x909b, 0x909d,
	0x9099, 0x90ac, 0x90a1, 0x90b4, 0x90b3, 0x90b6, 0x90ba, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xdb40[ 192 ] = {
	0x8dd5, 0x8dd8, 0x8dd9, 0x8ddc, 0x8de0, 0x8de1, 0x8de2, 0x8de5,
	0x8de6, 0x8de7, 0x8de9, 0x8ded, 0x8dee, 0x8df0, 0x8df1, 0x8df2,
	0x8df4, 0x8df6, 0x8dfc, 0x8dfe, 0x8dff, 0x8e00, 0x8e01, 0x8e02,
	0x8e03, 0x8e04, 0x8e06, 0x8e07, 0x8e08, 0x8e0b, 0x8e0d, 0x8e0e,
	0x8e10, 0x8e11, 0x8e12, 0x8e13, 0x8e15, 0x8e16, 0x8e17, 0x8e18,
	0x8e19, 0x8e1a, 0x8e1b, 0x8e1c, 0x8e20, 0x8e21, 0x8e24, 0x8e25,
	0x8e26, 0x8e27, 0x8e28, 0x8e2b, 0x8e2d, 0x8e30, 0x8e32, 0x8e33,
	0x8e34, 0x8e36, 0x8e37, 0x8e38, 0x8e3b, 0x8e3c, 0x8e3e, 0xfffd,
	0x8e3f, 0x8e43, 0x8e45, 0x8e46, 0x8e4c, 0x8e4d, 0x8e4e, 0x8e4f,
	0x8e50, 0x8e53, 0x8e54, 0x8e55, 0x8e56, 0x8e57, 0x8e58, 0x8e5a,
	0x8e5b, 0x8e5c, 0x8e5d, 0x8e5e, 0x8e5f, 0x8e60, 0x8e61, 0x8e62,
	0x8e63, 0x8e64, 0x8e65, 0x8e67, 0x8e68, 0x8e6a, 0x8e6b, 0x8e6e,
	0x8e71, 0x90b8, 0x90b0, 0x90cf, 0x90c5, 0x90be, 0x90d0, 0x90c4,
	0x90c7, 0x90d3, 0x90e6, 0x90e2, 0x90dc, 0x90d7, 0x90db, 0x90eb,
	0x90ef, 0x90fe, 0x9104, 0x9122, 0x911e, 0x9123, 0x9131, 0x912f,
	0x9139, 0x9143, 0x9146, 0x520d, 0x5942, 0x52a2, 0x52ac, 0x52ad,
	0x52be, 0x54ff, 0x52d0, 0x52d6, 0x52f0, 0x53df, 0x71ee, 0x77cd,
	0x5ef4, 0x51f5, 0x51fc, 0x9b2f, 0x53b6, 0x5f01, 0x755a, 0x5def,
	0x574c, 0x57a9, 0x57a1, 0x587e, 0x58bc, 0x58c5, 0x58d1, 0x5729,
	0x572c, 0x572a, 0x5733, 0x5739, 0x572e, 0x572f, 0x575c, 0x573b,
	0x5742, 0x5769, 0x5785, 0x576b, 0x5786, 0x577c, 0x577b, 0x5768,
	0x576d, 0x5776, 0x5773, 0x57ad, 0x57a4, 0x578c, 0x57b2, 0x57cf,
	0x57a7, 0x57b4, 0x5793, 0x57a0, 0x57d5, 0x57d8, 0x57da, 0x57d9,
	0x57d2, 0x57b8, 0x57f4, 0x57ef, 0x57f8, 0x57e4, 0x57dd, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xdc40[ 192 ] = {
	0x8e73, 0x8e75, 0x8e77, 0x8e78, 0x8e79, 0x8e7a, 0x8e7b, 0x8e7d,
	0x8e7e, 0x8e80, 0x8e82, 0x8e83, 0x8e84, 0x8e86, 0x8e88, 0x8e89,
	0x8e8a, 0x8e8b, 0x8e8c, 0x8e8d, 0x8e8e, 0x8e91, 0x8e92, 0x8e93,
	0x8e95, 0x8e96, 0x8e97, 0x8e98, 0x8e99, 0x8e9a, 0x8e9b, 0x8e9d,
	0x8e9f, 0x8ea0, 0x8ea1, 0x8ea2, 0x8ea3, 0x8ea4, 0x8ea5, 0x8ea6,
	0x8ea7, 0x8ea8, 0x8ea9, 0x8eaa, 0x8ead, 0x8eae, 0x8eb0, 0x8eb1,
	0x8eb3, 0x8eb4, 0x8eb5, 0x8eb6, 0x8eb7, 0x8eb8, 0x8eb9, 0x8ebb,
	0x8ebc, 0x8ebd, 0x8ebe, 0x8ebf, 0x8ec0, 0x8ec1, 0x8ec2, 0xfffd,
	0x8ec3, 0x8ec4, 0x8ec5, 0x8ec6, 0x8ec7, 0x8ec8, 0x8ec9, 0x8eca,
	0x8ecb, 0x8ecc, 0x8ecd, 0x8ecf, 0x8ed0, 0x8ed1, 0x8ed2, 0x8ed3,
	0x8ed4, 0x8ed5, 0x8ed6, 0x8ed7, 0x8ed8, 0x8ed9, 0x8eda, 0x8edb,
	0x8edc, 0x8edd, 0x8ede, 0x8edf, 0x8ee0, 0x8ee1, 0x8ee2, 0x8ee3,
	0x8ee4, 0x580b, 0x580d, 0x57fd, 0x57ed, 0x5800, 0x581e, 0x5819,
	0x5844, 0x5820, 0x5865, 0x586c, 0x5881, 0x5889, 0x589a, 0x5880,
	0x99a8, 0x9f19, 0x61ff, 0x8279, 0x827d, 0x827f, 0x828f, 0x828a,
	0x82a8, 0x8284, 0x828e, 0x8291, 0x8297, 0x8299, 0x82ab, 0x82b8,
	0x82be, 0x82b0, 0x82c8, 0x82ca, 0x82e3, 0x8298, 0x82b7, 0x82ae,
	0x82cb, 0x82cc, 0x82c1, 0x82a9, 0x82b4, 0x82a1, 0x82aa, 0x829f,
	0x82c4, 0x82ce, 0x82a4, 0x82e1, 0x8309, 0x82f7, 0x82e4, 0x830f,
	0x8307, 0x82dc, 0x82f4, 0x82d2, 0x82d8, 0x830c, 0x82fb, 0x82d3,
	0x8311, 0x831a, 0x8306, 0x8314, 0x8315, 0x82e0, 0x82d5, 0x831c,
	0x8351, 0x835b, 0x835c, 0x8308, 0x8392, 0x833c, 0x8334, 0x8331,
	0x839b, 0x835e, 0x832f, 0x834f, 0x8347, 0x8343, 0x835f, 0x8340,
	0x8317, 0x8360, 0x832d, 0x833a, 0x8333, 0x8366, 0x8365, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xdd40[ 192 ] = {
	0x8ee5, 0x8ee6, 0x8ee7, 0x8ee8, 0x8ee9, 0x8eea, 0x8eeb, 0x8eec,
	0x8eed, 0x8eee, 0x8eef, 0x8ef0, 0x8ef1, 0x8ef2, 0x8ef3, 0x8ef4,
	0x8ef5, 0x8ef6, 0x8ef7, 0x8ef8, 0x8ef9, 0x8efa, 0x8efb, 0x8efc,
	0x8efd, 0x8efe, 0x8eff, 0x8f00, 0x8f01, 0x8f02, 0x8f03, 0x8f04,
	0x8f05, 0x8f06, 0x8f07, 0x8f08, 0x8f09, 0x8f0a, 0x8f0b, 0x8f0c,
	0x8f0d, 0x8f0e, 0x8f0f, 0x8f10, 0x8f11, 0x8f12, 0x8f13, 0x8f14,
	0x8f15, 0x8f16, 0x8f17, 0x8f18, 0x8f19, 0x8f1a, 0x8f1b, 0x8f1c,
	0x8f1d, 0x8f1e, 0x8f1f, 0x8f20, 0x8f21, 0x8f22, 0x8f23, 0xfffd,
	0x8f24, 0x8f25, 0x8f26, 0x8f27, 0x8f28, 0x8f29, 0x8f2a, 0x8f2b,
	0x8f2c, 0x8f2d, 0x8f2e, 0x8f2f, 0x8f30, 0x8f31, 0x8f32, 0x8f33,
	0x8f34, 0x8f35, 0x8f36, 0x8f37, 0x8f38, 0x8f39, 0x8f3a, 0x8f3b,
	0x8f3c, 0x8f3d, 0x8f3e, 0x8f3f, 0x8f40, 0x8f41, 0x8f42, 0x8f43,
	0x8f44, 0x8368, 0x831b, 0x8369, 0x836c, 0x836a, 0x836d, 0x836e,
	0x83b0, 0x8378, 0x83b3, 0x83b4, 0x83a0, 0x83aa, 0x8393, 0x839c,
	0x8385, 0x837c, 0x83b6, 0x83a9, 0x837d, 0x83b8, 0x837b, 0x8398,
	0x839e, 0x83a8, 0x83ba, 0x83bc, 0x83c1, 0x8401, 0x83e5, 0x83d8,
	0x5807, 0x8418, 0x840b, 0x83dd, 0x83fd, 0x83d6, 0x841c, 0x8438,
	0x8411, 0x8406, 0x83d4, 0x83df, 0x840f, 0x8403, 0x83f8, 0x83f9,
	0x83ea, 0x83c5, 0x83c0, 0x8426, 0x83f0, 0x83e1, 0x845c, 0x8451,
	0x845a, 0x8459, 0x8473, 0x8487, 0x8488, 0x847a, 0x8489, 0x8478,
	0x843c, 0x8446, 0x8469, 0x8476, 0x848c, 0x848e, 0x8431, 0x846d,
	0x84c1, 0x84cd, 0x84d0, 0x84e6, 0x84bd, 0x84d3, 0x84ca, 0x84bf,
	0x84ba, 0x84e0, 0x84a1, 0x84b9, 0x84b4, 0x8497, 0x84e5, 0x84e3,
	0x850c, 0x750d, 0x8538, 0x84f0, 0x8539, 0x851f, 0x853a, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xde40[ 192 ] = {
	0x8f45, 0x8f46, 0x8f47, 0x8f48, 0x8f49, 0x8f4a, 0x8f4b, 0x8f4c,
	0x8f4d, 0x8f4e, 0x8f4f, 0x8f50, 0x8f51, 0x8f52, 0x8f53, 0x8f54,
	0x8f55, 0x8f56, 0x8f57, 0x8f58, 0x8f59, 0x8f5a, 0x8f5b, 0x8f5c,
	0x8f5d, 0x8f5e, 0x8f5f, 0x8f60, 0x8f61, 0x8f62, 0x8f63, 0x8f64,
	0x8f65, 0x8f6a, 0x8f80, 0x8f8c, 0x8f92, 0x8f9d, 0x8fa0, 0x8fa1,
	0x8fa2, 0x8fa4, 0x8fa5, 0x8fa6, 0x8fa7, 0x8faa, 0x8fac, 0x8fad,
	0x8fae, 0x8faf, 0x8fb2, 0x8fb3, 0x8fb4, 0x8fb5, 0x8fb7, 0x8fb8,
	0x8fba, 0x8fbb, 0x8fbc, 0x8fbf, 0x8fc0, 0x8fc3, 0x8fc6, 0xfffd,
	0x8fc9, 0x8fca, 0x8fcb, 0x8fcc, 0x8fcd, 0x8fcf, 0x8fd2, 0x8fd6,
	0x8fd7, 0x8fda, 0x8fe0, 0x8fe1, 0x8fe3, 0x8fe7, 0x8fec, 0x8fef,
	0x8ff1, 0x8ff2, 0x8ff4, 0x8ff5, 0x8ff6, 0x8ffa, 0x8ffb, 0x8ffc,
	0x8ffe, 0x8fff, 0x9007, 0x9008, 0x900c, 0x900e, 0x9013, 0x9015,
	0x9018, 0x8556, 0x853b, 0x84ff, 0x84fc, 0x8559, 0x8548, 0x8568,
	0x8564, 0x855e, 0x857a, 0x77a2, 0x8543, 0x8572, 0x857b, 0x85a4,
	0x85a8, 0x8587, 0x858f, 0x8579, 0x85ae, 0x859c, 0x8585, 0x85b9,
	0x85b7, 0x85b0, 0x85d3, 0x85c1, 0x85dc, 0x85ff, 0x8627, 0x8605,
	0x8629, 0x8616, 0x863c, 0x5efe, 0x5f08, 0x593c, 0x5941, 0x8037,
	0x5955, 0x595a, 0x5958, 0x530f, 0x5c22, 0x5c25, 0x5c2c, 0x5c34,
	0x624c, 0x626a, 0x629f, 0x62bb, 0x62ca, 0x62da, 0x62d7, 0x62ee,
	0x6322, 0x62f6, 0x6339, 0x634b, 0x6343, 0x63ad, 0x63f6, 0x6371,
	0x637a, 0x638e, 0x63b4, 0x636d, 0x63ac, 0x638a, 0x6369, 0x63ae,
	0x63bc, 0x63f2, 0x63f8, 0x63e0, 0x63ff, 0x63c4, 0x63de, 0x63ce,
	0x6452, 0x63c6, 0x63be, 0x6445, 0x6441, 0x640b, 0x641b, 0x6420,
	0x640c, 0x6426, 0x6421, 0x645e, 0x6484, 0x646d, 0x6496, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xdf40[ 192 ] = {
	0x9019, 0x901c, 0x9023, 0x9024, 0x9025, 0x9027, 0x9028, 0x9029,
	0x902a, 0x902b, 0x902c, 0x9030, 0x9031, 0x9032, 0x9033, 0x9034,
	0x9037, 0x9039, 0x903a, 0x903d, 0x903f, 0x9040, 0x9043, 0x9045,
	0x9046, 0x9048, 0x9049, 0x904a, 0x904b, 0x904c, 0x904e, 0x9054,
	0x9055, 0x9056, 0x9059, 0x905a, 0x905c, 0x905d, 0x905e, 0x905f,
	0x9060, 0x9061, 0x9064, 0x9066, 0x9067, 0x9069, 0x906a, 0x906b,
	0x906c, 0x906f, 0x9070, 0x9071, 0x9072, 0x9073, 0x9076, 0x9077,
	0x9078, 0x9079, 0x907a, 0x907b, 0x907c, 0x907e, 0x9081, 0xfffd,
	0x9084, 0x9085, 0x9086, 0x9087, 0x9089, 0x908a, 0x908c, 0x908d,
	0x908e, 0x908f, 0x9090, 0x9092, 0x9094, 0x9096, 0x9098, 0x909a,
	0x909c, 0x909e, 0x909f, 0x90a0, 0x90a4, 0x90a5, 0x90a7, 0x90a8,
	0x90a9, 0x90ab, 0x90ad, 0x90b2, 0x90b7, 0x90bc, 0x90bd, 0x90bf,
	0x90c0, 0x647a, 0x64b7, 0x64b8, 0x6499, 0x64ba, 0x64c0, 0x64d0,
	0x64d7, 0x64e4, 0x64e2, 0x6509, 0x6525, 0x652e, 0x5f0b, 0x5fd2,
	0x7519, 0x5f11, 0x535f, 0x53f1, 0x53fd, 0x53e9, 0x53e8, 0x53fb,
	0x5412, 0x5416, 0x5406, 0x544b, 0x5452, 0x5453, 0x5454, 0x5456,
	0x5443, 0x5421, 0x5457, 0x5459, 0x5423, 0x5432, 0x5482, 0x5494,
	0x5477, 0x5471, 0x5464, 0x549a, 0x549b, 0x5484, 0x5476, 0x5466,
	0x549d, 0x54d0, 0x54ad, 0x54c2, 0x54b4, 0x54d2, 0x54a7, 0x54a6,
	0x54d3, 0x54d4, 0x5472, 0x54a3, 0x54d5, 0x54bb, 0x54bf, 0x54cc,
	0x54d9, 0x54da, 0x54dc, 0x54a9, 0x54aa, 0x54a4, 0x54dd, 0x54cf,
	0x54de, 0x551b, 0x54e7, 0x5520, 0x54fd, 0x5514, 0x54f3, 0x5522,
	0x5523, 0x550f, 0x5511, 0x5527, 0x552a, 0x5567, 0x558f, 0x55b5,
	0x5549, 0x556d, 0x5541, 0x5555, 0x553f, 0x5550, 0x553c, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe040[ 192 ] = {
	0x90c2, 0x90c3, 0x90c6, 0x90c8, 0x90c9, 0x90cb, 0x90cc, 0x90cd,
	0x90d2, 0x90d4, 0x90d5, 0x90d6, 0x90d8, 0x90d9, 0x90da, 0x90de,
	0x90df, 0x90e0, 0x90e3, 0x90e4, 0x90e5, 0x90e9, 0x90ea, 0x90ec,
	0x90ee, 0x90f0, 0x90f1, 0x90f2, 0x90f3, 0x90f5, 0x90f6, 0x90f7,
	0x90f9, 0x90fa, 0x90fb, 0x90fc, 0x90ff, 0x9100, 0x9101, 0x9103,
	0x9105, 0x9106, 0x9107, 0x9108, 0x9109, 0x910a, 0x910b, 0x910c,
	0x910d, 0x910e, 0x910f, 0x9110, 0x9111, 0x9112, 0x9113, 0x9114,
	0x9115, 0x9116, 0x9117, 0x9118, 0x911a, 0x911b, 0x911c, 0xfffd,
	0x911d, 0x911f, 0x9120, 0x9121, 0x9124, 0x9125, 0x9126, 0x9127,
	0x9128, 0x9129, 0x912a, 0x912b, 0x912c, 0x912d, 0x912e, 0x9130,
	0x9132, 0x9133, 0x9134, 0x9135, 0x9136, 0x9137, 0x9138, 0x913a,
	0x913b, 0x913c, 0x913d, 0x913e, 0x913f, 0x9140, 0x9141, 0x9142,
	0x9144, 0x5537, 0x5556, 0x5575, 0x5576, 0x5577, 0x5533, 0x5530,
	0x555c, 0x558b, 0x55d2, 0x5583, 0x55b1, 0x55b9, 0x5588, 0x5581,
	0x559f, 0x557e, 0x55d6, 0x5591, 0x557b, 0x55df, 0x55bd, 0x55be,
	0x5594, 0x5599, 0x55ea, 0x55f7, 0x55c9, 0x561f, 0x55d1, 0x55eb,
	0x55ec, 0x55d4, 0x55e6, 0x55dd, 0x55c4, 0x55ef, 0x55e5, 0x55f2,
	0x55f3, 0x55cc, 0x55cd, 0x55e8, 0x55f5, 0x55e4, 0x8f94, 0x561e,
	0x5608, 0x560c, 0x5601, 0x5624, 0x5623, 0x55fe, 0x5600, 0x5627,
	0x562d, 0x5658, 0x5639, 0x5657, 0x562c, 0x564d, 0x5662, 0x5659,
	0x565c, 0x564c, 0x5654, 0x5686, 0x5664, 0x5671, 0x566b, 0x567b,
	0x567c, 0x5685, 0x5693, 0x56af, 0x56d4, 0x56d7, 0x56dd, 0x56e1,
	0x56f5, 0x56eb, 0x56f9, 0x56ff, 0x5704, 0x570a, 0x5709, 0x571c,
	0x5e0f, 0x5e19, 0x5e14, 0x5e11, 0x5e31, 0x5e3b, 0x5e3c, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe140[ 192 ] = {
	0x9145, 0x9147, 0x9148, 0x9151, 0x9153, 0x9154, 0x9155, 0x9156,
	0x9158, 0x9159, 0x915b, 0x915c, 0x915f, 0x9160, 0x9166, 0x9167,
	0x9168, 0x916b, 0x916d, 0x9173, 0x917a, 0x917b, 0x917c, 0x9180,
	0x9181, 0x9182, 0x9183, 0x9184, 0x9186, 0x9188, 0x918a, 0x918e,
	0x918f, 0x9193, 0x9194, 0x9195, 0x9196, 0x9197, 0x9198, 0x9199,
	0x919c, 0x919d, 0x919e, 0x919f, 0x91a0, 0x91a1, 0x91a4, 0x91a5,
	0x91a6, 0x91a7, 0x91a8, 0x91a9, 0x91ab, 0x91ac, 0x91b0, 0x91b1,
	0x91b2, 0x91b3, 0x91b6, 0x91b7, 0x91b8, 0x91b9, 0x91bb, 0xfffd,
	0x91bc, 0x91bd, 0x91be, 0x91bf, 0x91c0, 0x91c1, 0x91c2, 0x91c3,
	0x91c4, 0x91c5, 0x91c6, 0x91c8, 0x91cb, 0x91d0, 0x91d2, 0x91d3,
	0x91d4, 0x91d5, 0x91d6, 0x91d7, 0x91d8, 0x91d9, 0x91da, 0x91db,
	0x91dd, 0x91de, 0x91df, 0x91e0, 0x91e1, 0x91e2, 0x91e3, 0x91e4,
	0x91e5, 0x5e37, 0x5e44, 0x5e54, 0x5e5b, 0x5e5e, 0x5e61, 0x5c8c,
	0x5c7a, 0x5c8d, 0x5c90, 0x5c96, 0x5c88, 0x5c98, 0x5c99, 0x5c91,
	0x5c9a, 0x5c9c, 0x5cb5, 0x5ca2, 0x5cbd, 0x5cac, 0x5cab, 0x5cb1,
	0x5ca3, 0x5cc1, 0x5cb7, 0x5cc4, 0x5cd2, 0x5ce4, 0x5ccb, 0x5ce5,
	0x5d02, 0x5d03, 0x5d27, 0x5d26, 0x5d2e, 0x5d24, 0x5d1e, 0x5d06,
	0x5d1b, 0x5d58, 0x5d3e, 0x5d34, 0x5d3d, 0x5d6c, 0x5d5b, 0x5d6f,
	0x5d5d, 0x5d6b, 0x5d4b, 0x5d4a, 0x5d69, 0x5d74, 0x5d82, 0x5d99,
	0x5d9d, 0x8c73, 0x5db7, 0x5dc5, 0x5f73, 0x5f77, 0x5f82, 0x5f87,
	0x5f89, 0x5f8c, 0x5f95, 0x5f99, 0x5f9c, 0x5fa8, 0x5fad, 0x5fb5,
	0x5fbc, 0x8862, 0x5f61, 0x72ad, 0x72b0, 0x72b4, 0x72b7, 0x72b8,
	0x72c3, 0x72c1, 0x72ce, 0x72cd, 0x72d2, 0x72e8, 0x72ef, 0x72e9,
	0x72f2, 0x72f4, 0x72f7, 0x7301, 0x72f3, 0x7303, 0x72fa, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe240[ 192 ] = {
	0x91e6, 0x91e7, 0x91e8, 0x91e9, 0x91ea, 0x91eb, 0x91ec, 0x91ed,
	0x91ee, 0x91ef, 0x91f0, 0x91f1, 0x91f2, 0x91f3, 0x91f4, 0x91f5,
	0x91f6, 0x91f7, 0x91f8, 0x91f9, 0x91fa, 0x91fb, 0x91fc, 0x91fd,
	0x91fe, 0x91ff, 0x9200, 0x9201, 0x9202, 0x9203, 0x9204, 0x9205,
	0x9206, 0x9207, 0x9208, 0x9209, 0x920a, 0x920b, 0x920c, 0x920d,
	0x920e, 0x920f, 0x9210, 0x9211, 0x9212, 0x9213, 0x9214, 0x9215,
	0x9216, 0x9217, 0x9218, 0x9219, 0x921a, 0x921b, 0x921c, 0x921d,
	0x921e, 0x921f, 0x9220, 0x9221, 0x9222, 0x9223, 0x9224, 0xfffd,
	0x9225, 0x9226, 0x9227, 0x9228, 0x9229, 0x922a, 0x922b, 0x922c,
	0x922d, 0x922e, 0x922f, 0x9230, 0x9231, 0x9232, 0x9233, 0x9234,
	0x9235, 0x9236, 0x9237, 0x9238, 0x9239, 0x923a, 0x923b, 0x923c,
	0x923d, 0x923e, 0x923f, 0x9240, 0x9241, 0x9242, 0x9243, 0x9244,
	0x9245, 0x72fb, 0x7317, 0x7313, 0x7321, 0x730a, 0x731e, 0x731d,
	0x7315, 0x7322, 0x7339, 0x7325, 0x732c, 0x7338, 0x7331, 0x7350,
	0x734d, 0x7357, 0x7360, 0x736c, 0x736f, 0x737e, 0x821b, 0x5925,
	0x98e7, 0x5924, 0x5902, 0x9963, 0x9967, 0x9968, 0x9969, 0x996a,
	0x996b, 0x996c, 0x9974, 0x9977, 0x997d, 0x9980, 0x9984, 0x9987,
	0x998a, 0x998d, 0x9990, 0x9991, 0x9993, 0x9994, 0x9995, 0x5e80,
	0x5e91, 0x5e8b, 0x5e96, 0x5ea5, 0x5ea0, 0x5eb9, 0x5eb5, 0x5ebe,
	0x5eb3, 0x8d53, 0x5ed2, 0x5ed1, 0x5edb, 0x5ee8, 0x5eea, 0x81ba,
	0x5fc4, 0x5fc9, 0x5fd6, 0x5fcf, 0x6003, 0x5fee, 0x6004, 0x5fe1,
	0x5fe4, 0x5ffe, 0x6005, 0x6006, 0x5fea, 0x5fed, 0x5ff8, 0x6019,
	0x6035, 0x6026, 0x601b, 0x600f, 0x600d, 0x6029, 0x602b, 0x600a,
	0x603f, 0x6021, 0x6078, 0x6079, 0x607b, 0x607a, 0x6042, 0xfffd,
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe340[ 192 ] = {
	0x9246, 0x9247, 0x9248, 0x9249, 0x924a, 0x924b, 0x924c, 0x924d,
	0x924e, 0x924f, 0x9250, 0x9251, 0x9252, 0x9253, 0x9254, 0x9255,
	0x9256, 0x9257, 0x9258, 0x9259, 0x925a, 0x925b, 0x925c, 0x925d,
	0x925e, 0x925f, 0x9260, 0x9261, 0x9262, 0x9263, 0x9264, 0x9265,
	0x9266, 0x9267, 0x9268, 0x9269, 0x926a, 0x926b, 0x926c, 0x926d,
	0x926e, 0x926f, 0x9270, 0x9271, 0x9272, 0x9273, 0x9275, 0x9276,
	0x9277, 0x9278, 0x9279, 0x927a, 0x927b, 0x927c, 0x927d, 0x927e,
	0x927f, 0x9280, 0x9281, 0x9282, 0x9283, 0x9284, 0x9285, 0xfffd,
	0x9286, 0x9287, 0x9288, 0x9289, 0x928a, 0x928b, 0x928c, 0x928d,
	0x928f, 0x9290, 0x9291, 0x9292, 0x9293, 0x9294, 0x9295, 0x9296,
	0x9297, 0x9298, 0x9299, 0x929a, 0x929b, 0x929c, 0x929d, 0x929e,
	0x929f, 0x92a0, 0x92a1, 0x92a2, 0x92a3, 0x92a4, 0x92a5, 0x92a6,
	0x92a7, 0x606a, 0x607d, 0x6096, 0x609a, 0x60ad, 0x609d, 0x6083,
	0x6092, 0x608c, 0x609b, 0x60ec, 0x60bb, 0x60b1, 0x60dd, 0x60d8,
	0x60c6, 0x60da, 0x60b4, 0x6120, 0x6126, 0x6115, 0x6123, 0x60f4,
	0x6100, 0x610e, 0x612b, 0x614a, 0x6175, 0x61ac, 0x6194, 0x61a7,
	0x61b7, 0x61d4, 0x61f5, 0x5fdd, 0x96b3, 0x95e9, 0x95eb, 0x95f1,
	0x95f3, 0x95f5, 0x95f6, 0x95fc, 0x95fe, 0x9603, 0x9604, 0x9606,
	0x9608, 0x960a, 0x960b, 0x960c, 0x960d, 0x960f, 0x9612, 0x9615,
	0x9616, 0x9617, 0x9619, 0x961a, 0x4e2c, 0x723f, 0x6215, 0x6c35,
	0x6c54, 0x6c5c, 0x6c4a, 0x6ca3, 0x6c85, 0x6c90, 0x6c94, 0x6c8c,
	0x6c68, 0x6c69, 0x6c74, 0x6c76, 0x6c86, 0x6ca9, 0x6cd0, 0x6cd4,
	0x6cad, 0x6cf7, 0x6cf8, 0x6cf1, 0x6cd7, 0x6cb2, 0x6ce0, 0x6cd6,
	0x6cfa, 0x6ceb, 0x6cee, 0x6cb1, 0x6cd3, 0x6cef, 0x6cfe, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe440[ 192 ] = {
	0x92a8, 0x92a9, 0x92aa, 0x92ab, 0x92ac, 0x92ad, 0x92af, 0x92b0,
	0x92b1, 0x92b2, 0x92b3, 0x92b4, 0x92b5, 0x92b6, 0x92b7, 0x92b8,
	0x92b9, 0x92ba, 0x92bb, 0x92bc, 0x92bd, 0x92be, 0x92bf, 0x92c0,
	0x92c1, 0x92c2, 0x92c3, 0x92c4, 0x92c5, 0x92c6, 0x92c7, 0x92c9,
	0x92ca, 0x92cb, 0x92cc, 0x92cd, 0x92ce, 0x92cf, 0x92d0, 0x92d1,
	0x92d2, 0x92d3, 0x92d4, 0x92d5, 0x92d6, 0x92d7, 0x92d8, 0x92d9,
	0x92da, 0x92db, 0x92dc, 0x92dd, 0x92de, 0x92df, 0x92e0, 0x92e1,
	0x92e2, 0x92e3, 0x92e4, 0x92e5, 0x92e6, 0x92e7, 0x92e8, 0xfffd,
	0x92e9, 0x92ea, 0x92eb, 0x92ec, 0x92ed, 0x92ee, 0x92ef, 0x92f0,
	0x92f1, 0x92f2, 0x92f3, 0x92f4, 0x92f5, 0x92f6, 0x92f7, 0x92f8,
	0x92f9, 0x92fa, 0x92fb, 0x92fc, 0x92fd, 0x92fe, 0x92ff, 0x9300,
	0x9301, 0x9302, 0x9303, 0x9304, 0x9305, 0x9306, 0x9307, 0x9308,
	0x9309, 0x6d39, 0x6d27, 0x6d0c, 0x6d43, 0x6d48, 0x6d07, 0x6d04,
	0x6d19, 0x6d0e, 0x6d2b, 0x6d4d, 0x6d2e, 0x6d35, 0x6d1a, 0x6d4f,
	0x6d52, 0x6d54, 0x6d33, 0x6d91, 0x6d6f, 0x6d9e, 0x6da0, 0x6d5e,
	0x6d93, 0x6d94, 0x6d5c, 0x6d60, 0x6d7c, 0x6d63, 0x6e1a, 0x6dc7,
	0x6dc5, 0x6dde, 0x6e0e, 0x6dbf, 0x6de0, 0x6e11, 0x6de6, 0x6ddd,
	0x6dd9, 0x6e16, 0x6dab, 0x6e0c, 0x6dae, 0x6e2b, 0x6e6e, 0x6e4e,
	0x6e6b, 0x6eb2, 0x6e5f, 0x6e86, 0x6e53, 0x6e54, 0x6e32, 0x6e25,
	0x6e44, 0x6edf, 0x6eb1, 0x6e98, 0x6ee0, 0x6f2d, 0x6ee2, 0x6ea5,
	0x6ea7, 0x6ebd, 0x6ebb, 0x6eb7, 0x6ed7, 0x6eb4, 0x6ecf, 0x6e8f,
	0x6ec2, 0x6e9f, 0x6f62, 0x6f46, 0x6f47, 0x6f24, 0x6f15, 0x6ef9,
	0x6f2f, 0x6f36, 0x6f4b, 0x6f74, 0x6f2a, 0x6f09, 0x6f29, 0x6f89,
	0x6f8d, 0x6f8c, 0x6f78, 0x6f72, 0x6f7c, 0x6f7a, 0x6fd1, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe540[ 192 ] = {
	0x930a, 0x930b, 0x930c, 0x930d, 0x930e, 0x930f, 0x9310, 0x9311,
	0x9312, 0x9313, 0x9314, 0x9315, 0x9316, 0x9317, 0x9318, 0x9319,
	0x931a, 0x931b, 0x931c, 0x931d, 0x931e, 0x931f, 0x9320, 0x9321,
	0x9322, 0x9323, 0x9324, 0x9325, 0x9326, 0x9327, 0x9328, 0x9329,
	0x932a, 0x932b, 0x932c, 0x932d, 0x932e, 0x932f, 0x9330, 0x9331,
	0x9332, 0x9333, 0x9334, 0x9335, 0x9336, 0x9337, 0x9338, 0x9339,
	0x933a, 0x933b, 0x933c, 0x933d, 0x933f, 0x9340, 0x9341, 0x9342,
	0x9343, 0x9344, 0x9345, 0x9346, 0x9347, 0x9348, 0x9349, 0xfffd,
	0x934a, 0x934b, 0x934c, 0x934d, 0x934e, 0x934f, 0x9350, 0x9351,
	0x9352, 0x9353, 0x9354, 0x9355, 0x9356, 0x9357, 0x9358, 0x9359,
	0x935a, 0x935b, 0x935c, 0x935d, 0x935e, 0x935f, 0x9360, 0x9361,
	0x9362, 0x9363, 0x9364, 0x9365, 0x9366, 0x9367, 0x9368, 0x9369,
	0x936b, 0x6fc9, 0x6fa7, 0x6fb9, 0x6fb6, 0x6fc2, 0x6fe1, 0x6fee,
	0x6fde, 0x6fe0, 0x6fef, 0x701a, 0x7023, 0x701b, 0x7039, 0x7035,
	0x704f, 0x705e, 0x5b80, 0x5b84, 0x5b95, 0x5b93, 0x5ba5, 0x5bb8,
	0x752f, 0x9a9e, 0x6434, 0x5be4, 0x5bee, 0x8930, 0x5bf0, 0x8e47,
	0x8b07, 0x8fb6, 0x8fd3, 0x8fd5, 0x8fe5, 0x8fee, 0x8fe4, 0x8fe9,
	0x8fe6, 0x8ff3, 0x8fe8, 0x9005, 0x9004, 0x900b, 0x9026, 0x9011,
	0x900d, 0x9016, 0x9021, 0x9035, 0x9036, 0x902d, 0x902f, 0x9044,
	0x9051, 0x9052, 0x9050, 0x9068, 0x9058, 0x9062, 0x905b, 0x66b9,
	0x9074, 0x907d, 0x9082, 0x9088, 0x9083, 0x908b, 0x5f50, 0x5f57,
	0x5f56, 0x5f58, 0x5c3b, 0x54ab, 0x5c50, 0x5c59, 0x5b71, 0x5c63,
	0x5c66, 0x7fbc, 0x5f2a, 0x5f29, 0x5f2d, 0x8274, 0x5f3c, 0x9b3b,
	0x5c6e, 0x5981, 0x5983, 0x598d, 0x59a9, 0x59aa, 0x59a3, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe640[ 192 ] = {
	0x936c, 0x936d, 0x936e, 0x936f, 0x9370, 0x9371, 0x9372, 0x9373,
	0x9374, 0x9375, 0x9376, 0x9377, 0x9378, 0x9379, 0x937a, 0x937b,
	0x937c, 0x937d, 0x937e, 0x937f, 0x9380, 0x9381, 0x9382, 0x9383,
	0x9384, 0x9385, 0x9386, 0x9387, 0x9388, 0x9389, 0x938a, 0x938b,
	0x938c, 0x938d, 0x938e, 0x9390, 0x9391, 0x9392, 0x9393, 0x9394,
	0x9395, 0x9396, 0x9397, 0x9398, 0x9399, 0x939a, 0x939b, 0x939c,
	0x939d, 0x939e, 0x939f, 0x93a0, 0x93a1, 0x93a2, 0x93a3, 0x93a4,
	0x93a5, 0x93a6, 0x93a7, 0x93a8, 0x93a9, 0x93aa, 0x93ab, 0xfffd,
	0x93ac, 0x93ad, 0x93ae, 0x93af, 0x93b0, 0x93b1, 0x93b2, 0x93b3,
	0x93b4, 0x93b5, 0x93b6, 0x93b7, 0x93b8, 0x93b9, 0x93ba, 0x93bb,
	0x93bc, 0x93bd, 0x93be, 0x93bf, 0x93c0, 0x93c1, 0x93c2, 0x93c3,
	0x93c4, 0x93c5, 0x93c6, 0x93c7, 0x93c8, 0x93c9, 0x93cb, 0x93cc,
	0x93cd, 0x5997, 0x59ca, 0x59ab, 0x599e, 0x59a4, 0x59d2, 0x59b2,
	0x59af, 0x59d7, 0x59be, 0x5a05, 0x5a06, 0x59dd, 0x5a08, 0x59e3,
	0x59d8, 0x59f9, 0x5a0c, 0x5a09, 0x5a32, 0x5a34, 0x5a11, 0x5a23,
	0x5a13, 0x5a40, 0x5a67, 0x5a4a, 0x5a55, 0x5a3c, 0x5a62, 0x5a75,
	0x80ec, 0x5aaa, 0x5a9b, 0x5a77, 0x5a7a, 0x5abe, 0x5aeb, 0x5ab2,
	0x5ad2, 0x5ad4, 0x5ab8, 0x5ae0, 0x5ae3, 0x5af1, 0x5ad6, 0x5ae6,
	0x5ad8, 0x5adc, 0x5b09, 0x5b17, 0x5b16, 0x5b32, 0x5b37, 0x5b40,
	0x5c15, 0x5c1c, 0x5b5a, 0x5b65, 0x5b73, 0x5b51, 0x5b53, 0x5b62,
	0x9a75, 0x9a77, 0x9a78, 0x9a7a, 0x9a7f, 0x9a7d, 0x9a80, 0x9a81,
	0x9a85, 0x9a88, 0x9a8a, 0x9a90, 0x9a92, 0x9a93, 0x9a96, 0x9a98,
	0x9a9b, 0x9a9c, 0x9a9d, 0x9a9f, 0x9aa0, 0x9aa2, 0x9aa3, 0x9aa5,
	0x9aa7, 0x7e9f, 0x7ea1, 0x7ea3, 0x7ea5, 0x7ea8, 0x7ea9, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe740[ 192 ] = {
	0x93ce, 0x93cf, 0x93d0, 0x93d1, 0x93d2, 0x93d3, 0x93d4, 0x93d5,
	0x93d7, 0x93d8, 0x93d9, 0x93da, 0x93db, 0x93dc, 0x93dd, 0x93de,
	0x93df, 0x93e0, 0x93e1, 0x93e2, 0x93e3, 0x93e4, 0x93e5, 0x93e6,
	0x93e7, 0x93e8, 0x93e9, 0x93ea, 0x93eb, 0x93ec, 0x93ed, 0x93ee,
	0x93ef, 0x93f0, 0x93f1, 0x93f2, 0x93f3, 0x93f4, 0x93f5, 0x93f6,
	0x93f7, 0x93f8, 0x93f9, 0x93fa, 0x93fb, 0x93fc, 0x93fd, 0x93fe,
	0x93ff, 0x9400, 0x9401, 0x9402, 0x9403, 0x9404, 0x9405, 0x9406,
	0x9407, 0x9408, 0x9409, 0x940a, 0x940b, 0x940c, 0x940d, 0xfffd,
	0x940e, 0x940f, 0x9410, 0x9411, 0x9412, 0x9413, 0x9414, 0x9415,
	0x9416, 0x9417, 0x9418, 0x9419, 0x941a, 0x941b, 0x941c, 0x941d,
	0x941e, 0x941f, 0x9420, 0x9421, 0x9422, 0x9423, 0x9424, 0x9425,
	0x9426, 0x9427, 0x9428, 0x9429, 0x942a, 0x942b, 0x942c, 0x942d,
	0x942e, 0x7ead, 0x7eb0, 0x7ebe, 0x7ec0, 0x7ec1, 0x7ec2, 0x7ec9,
	0x7ecb, 0x7ecc, 0x7ed0, 0x7ed4, 0x7ed7, 0x7edb, 0x7ee0, 0x7ee1,
	0x7ee8, 0x7eeb, 0x7eee, 0x7eef, 0x7ef1, 0x7ef2, 0x7f0d, 0x7ef6,
	0x7efa, 0x7efb, 0x7efe, 0x7f01, 0x7f02, 0x7f03, 0x7f07, 0x7f08,
	0x7f0b, 0x7f0c, 0x7f0f, 0x7f11, 0x7f12, 0x7f17, 0x7f19, 0x7f1c,
	0x7f1b, 0x7f1f, 0x7f21, 0x7f22, 0x7f23, 0x7f24, 0x7f25, 0x7f26,
	0x7f27, 0x7f2a, 0x7f2b, 0x7f2c, 0x7f2d, 0x7f2f, 0x7f30, 0x7f31,
	0x7f32, 0x7f33, 0x7f35, 0x5e7a, 0x757f, 0x5ddb, 0x753e, 0x9095,
	0x738e, 0x7391, 0x73ae, 0x73a2, 0x739f, 0x73cf, 0x73c2, 0x73d1,
	0x73b7, 0x73b3, 0x73c0, 0x73c9, 0x73c8, 0x73e5, 0x73d9, 0x987c,
	0x740a, 0x73e9, 0x73e7, 0x73de, 0x73ba, 0x73f2, 0x740f, 0x742a,
	0x745b, 0x7426, 0x7425, 0x7428, 0x7430, 0x742e, 0x742c, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe840[ 192 ] = {
	0x942f, 0x9430, 0x9431, 0x9432, 0x9433, 0x9434, 0x9435, 0x9436,
	0x9437, 0x9438, 0x9439, 0x943a, 0x943b, 0x943c, 0x943d, 0x943f,
	0x9440, 0x9441, 0x9442, 0x9443, 0x9444, 0x9445, 0x9446, 0x9447,
	0x9448, 0x9449, 0x944a, 0x944b, 0x944c, 0x944d, 0x944e, 0x944f,
	0x9450, 0x9451, 0x9452, 0x9453, 0x9454, 0x9455, 0x9456, 0x9457,
	0x9458, 0x9459, 0x945a, 0x945b, 0x945c, 0x945d, 0x945e, 0x945f,
	0x9460, 0x9461, 0x9462, 0x9463, 0x9464, 0x9465, 0x9466, 0x9467,
	0x9468, 0x9469, 0x946a, 0x946c, 0x946d, 0x946e, 0x946f, 0xfffd,
	0x9470, 0x9471, 0x9472, 0x9473, 0x9474, 0x9475, 0x9476, 0x9477,
	0x9478, 0x9479, 0x947a, 0x947b, 0x947c, 0x947d, 0x947e, 0x947f,
	0x9480, 0x9481, 0x9482, 0x9483, 0x9484, 0x9491, 0x9496, 0x9498,
	0x94c7, 0x94cf, 0x94d3, 0x94d4, 0x94da, 0x94e6, 0x94fb, 0x951c,
	0x9520, 0x741b, 0x741a, 0x7441, 0x745c, 0x7457, 0x7455, 0x7459,
	0x7477, 0x746d, 0x747e, 0x749c, 0x748e, 0x7480, 0x7481, 0x7487,
	0x748b, 0x749e, 0x74a8, 0x74a9, 0x7490, 0x74a7, 0x74d2, 0x74ba,
	0x97ea, 0x97eb, 0x97ec, 0x674c, 0x6753, 0x675e, 0x6748, 0x6769,
	0x67a5, 0x6787, 0x676a, 0x6773, 0x6798, 0x67a7, 0x6775, 0x67a8,
	0x679e, 0x67ad, 0x678b, 0x6777, 0x677c, 0x67f0, 0x6809, 0x67d8,
	0x680a, 0x67e9, 0x67b0, 0x680c, 0x67d9, 0x67b5, 0x67da, 0x67b3,
	0x67dd, 0x6800, 0x67c3, 0x67b8, 0x67e2, 0x680e, 0x67c1, 0x67fd,
	0x6832, 0x6833, 0x6860, 0x6861, 0x684e, 0x6862, 0x6844, 0x6864,
	0x6883, 0x681d, 0x6855, 0x6866, 0x6841, 0x6867, 0x6840, 0x683e,
	0x684a, 0x6849, 0x6829, 0x68b5, 0x688f, 0x6874, 0x6877, 0x6893,
	0x686b, 0x68c2, 0x696e, 0x68fc, 0x691f, 0x6920, 0x68f9, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe940[ 192 ] = {
	0x9527, 0x9533, 0x953d, 0x9543, 0x9548, 0x954b, 0x9555, 0x955a,
	0x9560, 0x956e, 0x9574, 0x9575, 0x9577, 0x9578, 0x9579, 0x957a,
	0x957b, 0x957c, 0x957d, 0x957e, 0x9580, 0x9581, 0x9582, 0x9583,
	0x9584, 0x9585, 0x9586, 0x9587, 0x9588, 0x9589, 0x958a, 0x958b,
	0x958c, 0x958d, 0x958e, 0x958f, 0x9590, 0x9591, 0x9592, 0x9593,
	0x9594, 0x9595, 0x9596, 0x9597, 0x9598, 0x9599, 0x959a, 0x959b,
	0x959c, 0x959d, 0x959e, 0x959f, 0x95a0, 0x95a1, 0x95a2, 0x95a3,
	0x95a4, 0x95a5, 0x95a6, 0x95a7, 0x95a8, 0x95a9, 0x95aa, 0xfffd,
	0x95ab, 0x95ac, 0x95ad, 0x95ae, 0x95af, 0x95b0, 0x95b1, 0x95b2,
	0x95b3, 0x95b4, 0x95b5, 0x95b6, 0x95b7, 0x95b8, 0x95b9, 0x95ba,
	0x95bb, 0x95bc, 0x95bd, 0x95be, 0x95bf, 0x95c0, 0x95c1, 0x95c2,
	0x95c3, 0x95c4, 0x95c5, 0x95c6, 0x95c7, 0x95c8, 0x95c9, 0x95ca,
	0x95cb, 0x6924, 0x68f0, 0x690b, 0x6901, 0x6957, 0x68e3, 0x6910,
	0x6971, 0x6939, 0x6960, 0x6942, 0x695d, 0x6984, 0x696b, 0x6980,
	0x6998, 0x6978, 0x6934, 0x69cc, 0x6987, 0x6988, 0x69ce, 0x6989,
	0x6966, 0x6963, 0x6979, 0x699b, 0x69a7, 0x69bb, 0x69ab, 0x69ad,
	0x69d4, 0x69b1, 0x69c1, 0x69ca, 0x69df, 0x6995, 0x69e0, 0x698d,
	0x69ff, 0x6a2f, 0x69ed, 0x6a17, 0x6a18, 0x6a65, 0x69f2, 0x6a44,
	0x6a3e, 0x6aa0, 0x6a50, 0x6a5b, 0x6a35, 0x6a8e, 0x6a79, 0x6a3d,
	0x6a28, 0x6a58, 0x6a7c, 0x6a91, 0x6a90, 0x6aa9, 0x6a97, 0x6aab,
	0x7337, 0x7352, 0x6b81, 0x6b82, 0x6b87, 0x6b84, 0x6b92, 0x6b93,
	0x6b8d, 0x6b9a, 0x6b9b, 0x6ba1, 0x6baa, 0x8f6b, 0x8f6d, 0x8f71,
	0x8f72, 0x8f73, 0x8f75, 0x8f76, 0x8f78, 0x8f77, 0x8f79, 0x8f7a,
	0x8f7c, 0x8f7e, 0x8f81, 0x8f82, 0x8f84, 0x8f87, 0x8f8b, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xea40[ 192 ] = {
	0x95cc, 0x95cd, 0x95ce, 0x95cf, 0x95d0, 0x95d1, 0x95d2, 0x95d3,
	0x95d4, 0x95d5, 0x95d6, 0x95d7, 0x95d8, 0x95d9, 0x95da, 0x95db,
	0x95dc, 0x95dd, 0x95de, 0x95df, 0x95e0, 0x95e1, 0x95e2, 0x95e3,
	0x95e4, 0x95e5, 0x95e6, 0x95e7, 0x95ec, 0x95ff, 0x9607, 0x9613,
	0x9618, 0x961b, 0x961e, 0x9620, 0x9623, 0x9624, 0x9625, 0x9626,
	0x9627, 0x9628, 0x9629, 0x962b, 0x962c, 0x962d, 0x962f, 0x9630,
	0x9637, 0x9638, 0x9639, 0x963a, 0x963e, 0x9641, 0x9643, 0x964a,
	0x964e, 0x964f, 0x9651, 0x9652, 0x9653, 0x9656, 0x9657, 0xfffd,
	0x9658, 0x9659, 0x965a, 0x965c, 0x965d, 0x965e, 0x9660, 0x9663,
	0x9665, 0x9666, 0x966b, 0x966d, 0x966e, 0x966f, 0x9670, 0x9671,
	0x9673, 0x9678, 0x9679, 0x967a, 0x967b, 0x967c, 0x967d, 0x967e,
	0x967f, 0x9680, 0x9681, 0x9682, 0x9683, 0x9684, 0x9687, 0x9689,
	0x968a, 0x8f8d, 0x8f8e, 0x8f8f, 0x8f98, 0x8f9a, 0x8ece, 0x620b,
	0x6217, 0x621b, 0x621f, 0x6222, 0x6221, 0x6225, 0x6224, 0x622c,
	0x81e7, 0x74ef, 0x74f4, 0x74ff, 0x750f, 0x7511, 0x7513, 0x6534,
	0x65ee, 0x65ef, 0x65f0, 0x660a, 0x6619, 0x6772, 0x6603, 0x6615,
	0x6600, 0x7085, 0x66f7, 0x661d, 0x6634, 0x6631, 0x6636, 0x6635,
	0x8006, 0x665f, 0x6654, 0x6641, 0x664f, 0x6656, 0x6661, 0x6657,
	0x6677, 0x6684, 0x668c, 0x66a7, 0x669d, 0x66be, 0x66db, 0x66dc,
	0x66e6, 0x66e9, 0x8d32, 0x8d33, 0x8d36, 0x8d3b, 0x8d3d, 0x8d40,
	0x8d45, 0x8d46, 0x8d48, 0x8d49, 0x8d47, 0x8d4d, 0x8d55, 0x8d59,
	0x89c7, 0x89ca, 0x89cb, 0x89cc, 0x89ce, 0x89cf, 0x89d0, 0x89d1,
	0x726e, 0x729f, 0x725d, 0x7266, 0x726f, 0x727e, 0x727f, 0x7284,
	0x728b, 0x728d, 0x728f, 0x7292, 0x6308, 0x6332, 0x63b0, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xeb40[ 192 ] = {
	0x968c, 0x968e, 0x9691, 0x9692, 0x9693, 0x9695, 0x9696, 0x969a,
	0x969b, 0x969d, 0x969e, 0x969f, 0x96a0, 0x96a1, 0x96a2, 0x96a3,
	0x96a4, 0x96a5, 0x96a6, 0x96a8, 0x96a9, 0x96aa, 0x96ab, 0x96ac,
	0x96ad, 0x96ae, 0x96af, 0x96b1, 0x96b2, 0x96b4, 0x96b5, 0x96b7,
	0x96b8, 0x96ba, 0x96bb, 0x96bf, 0x96c2, 0x96c3, 0x96c8, 0x96ca,
	0x96cb, 0x96d0, 0x96d1, 0x96d3, 0x96d4, 0x96d6, 0x96d7, 0x96d8,
	0x96d9, 0x96da, 0x96db, 0x96dc, 0x96dd, 0x96de, 0x96df, 0x96e1,
	0x96e2, 0x96e3, 0x96e4, 0x96e5, 0x96e6, 0x96e7, 0x96eb, 0xfffd,
	0x96ec, 0x96ed, 0x96ee, 0x96f0, 0x96f1, 0x96f2, 0x96f4, 0x96f5,
	0x96f8, 0x96fa, 0x96fb, 0x96fc, 0x96fd, 0x96ff, 0x9702, 0x9703,
	0x9705, 0x970a, 0x970b, 0x970c, 0x9710, 0x9711, 0x9712, 0x9714,
	0x9715, 0x9717, 0x9718, 0x9719, 0x971a, 0x971b, 0x971d, 0x971f,
	0x9720, 0x643f, 0x64d8, 0x8004, 0x6bea, 0x6bf3, 0x6bfd, 0x6bf5,
	0x6bf9, 0x6c05, 0x6c07, 0x6c06, 0x6c0d, 0x6c15, 0x6c18, 0x6c19,
	0x6c1a, 0x6c21, 0x6c29, 0x6c24, 0x6c2a, 0x6c32, 0x6535, 0x6555,
	0x656b, 0x724d, 0x7252, 0x7256, 0x7230, 0x8662, 0x5216, 0x809f,
	0x809c, 0x8093, 0x80bc, 0x670a, 0x80bd, 0x80b1, 0x80ab, 0x80ad,
	0x80b4, 0x80b7, 0x80e7, 0x80e8, 0x80e9, 0x80ea, 0x80db, 0x80c2,
	0x80c4, 0x80d9, 0x80cd, 0x80d7, 0x6710, 0x80dd, 0x80eb, 0x80f1,
	0x80f4, 0x80ed, 0x810d, 0x810e, 0x80f2, 0x80fc, 0x6715, 0x8112,
	0x8c5a, 0x8136, 0x811e, 0x812c, 0x8118, 0x8132, 0x8148, 0x814c,
	0x8153, 0x8174, 0x8159, 0x815a, 0x8171, 0x8160, 0x8169, 0x817c,
	0x817d, 0x816d, 0x8167, 0x584d, 0x5ab5, 0x8188, 0x8182, 0x8191,
	0x6ed5, 0x81a3, 0x81aa, 0x81cc, 0x6726, 0x81ca, 0x81bb, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xec40[ 192 ] = {
	0x9721, 0x9722, 0x9723, 0x9724, 0x9725, 0x9726, 0x9727, 0x9728,
	0x9729, 0x972b, 0x972c, 0x972e, 0x972f, 0x9731, 0x9733, 0x9734,
	0x9735, 0x9736, 0x9737, 0x973a, 0x973b, 0x973c, 0x973d, 0x973f,
	0x9740, 0x9741, 0x9742, 0x9743, 0x9744, 0x9745, 0x9746, 0x9747,
	0x9748, 0x9749, 0x974a, 0x974b, 0x974c, 0x974d, 0x974e, 0x974f,
	0x9750, 0x9751, 0x9754, 0x9755, 0x9757, 0x9758, 0x975a, 0x975c,
	0x975d, 0x975f, 0x9763, 0x9764, 0x9766, 0x9767, 0x9768, 0x976a,
	0x976b, 0x976c, 0x976d, 0x976e, 0x976f, 0x9770, 0x9771, 0xfffd,
	0x9772, 0x9775, 0x9777, 0x9778, 0x9779, 0x977a, 0x977b, 0x977d,
	0x977e, 0x977f, 0x9780, 0x9781, 0x9782, 0x9783, 0x9784, 0x9786,
	0x9787, 0x9788, 0x9789, 0x978a, 0x978c, 0x978e, 0x978f, 0x9790,
	0x9793, 0x9795, 0x9796, 0x9797, 0x9799, 0x979a, 0x979b, 0x979c,
	0x979d, 0x81c1, 0x81a6, 0x6b24, 0x6b37, 0x6b39, 0x6b43, 0x6b46,
	0x6b59, 0x98d1, 0x98d2, 0x98d3, 0x98d5, 0x98d9, 0x98da, 0x6bb3,
	0x5f40, 0x6bc2, 0x89f3, 0x6590, 0x9f51, 0x6593, 0x65bc, 0x65c6,
	0x65c4, 0x65c3, 0x65cc, 0x65ce, 0x65d2, 0x65d6, 0x7080, 0x709c,
	0x7096, 0x709d, 0x70bb, 0x70c0, 0x70b7, 0x70ab, 0x70b1, 0x70e8,
	0x70ca, 0x7110, 0x7113, 0x7116, 0x712f, 0x7131, 0x7173, 0x715c,
	0x7168, 0x7145, 0x7172, 0x714a, 0x7178, 0x717a, 0x7198, 0x71b3,
	0x71b5, 0x71a8, 0x71a0, 0x71e0, 0x71d4, 0x71e7, 0x71f9, 0x721d,
	0x7228, 0x706c, 0x7118, 0x7166, 0x71b9, 0x623e, 0x623d, 0x6243,
	0x6248, 0x6249, 0x793b, 0x7940, 0x7946, 0x7949, 0x795b, 0x795c,
	0x7953, 0x795a, 0x7962, 0x7957, 0x7960, 0x796f, 0x7967, 0x797a,
	0x7985, 0x798a, 0x799a, 0x79a7, 0x79b3, 0x5fd1, 0x5fd0, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xed40[ 192 ] = {
	0x979e, 0x979f, 0x97a1, 0x97a2, 0x97a4, 0x97a5, 0x97a6, 0x97a7,
	0x97a8, 0x97a9, 0x97aa, 0x97ac, 0x97ae, 0x97b0, 0x97b1, 0x97b3,
	0x97b5, 0x97b6, 0x97b7, 0x97b8, 0x97b9, 0x97ba, 0x97bb, 0x97bc,
	0x97bd, 0x97be, 0x97bf, 0x97c0, 0x97c1, 0x97c2, 0x97c3, 0x97c4,
	0x97c5, 0x97c6, 0x97c7, 0x97c8, 0x97c9, 0x97ca, 0x97cb, 0x97cc,
	0x97cd, 0x97ce, 0x97cf, 0x97d0, 0x97d1, 0x97d2, 0x97d3, 0x97d4,
	0x97d5, 0x97d6, 0x97d7, 0x97d8, 0x97d9, 0x97da, 0x97db, 0x97dc,
	0x97dd, 0x97de, 0x97df, 0x97e0, 0x97e1, 0x97e2, 0x97e3, 0xfffd,
	0x97e4, 0x97e5, 0x97e8, 0x97ee, 0x97ef, 0x97f0, 0x97f1, 0x97f2,
	0x97f4, 0x97f7, 0x97f8, 0x97f9, 0x97fa, 0x97fb, 0x97fc, 0x97fd,
	0x97fe, 0x97ff, 0x9800, 0x9801, 0x9802, 0x9803, 0x9804, 0x9805,
	0x9806, 0x9807, 0x9808, 0x9809, 0x980a, 0x980b, 0x980c, 0x980d,
	0x980e, 0x603c, 0x605d, 0x605a, 0x6067, 0x6041, 0x6059, 0x6063,
	0x60ab, 0x6106, 0x610d, 0x615d, 0x61a9, 0x619d, 0x61cb, 0x61d1,
	0x6206, 0x8080, 0x807f, 0x6c93, 0x6cf6, 0x6dfc, 0x77f6, 0x77f8,
	0x7800, 0x7809, 0x7817, 0x7818, 0x7811, 0x65ab, 0x782d, 0x781c,
	0x781d, 0x7839, 0x783a, 0x783b, 0x781f, 0x783c, 0x7825, 0x782c,
	0x7823, 0x7829, 0x784e, 0x786d, 0x7856, 0x7857, 0x7826, 0x7850,
	0x7847, 0x784c, 0x786a, 0x789b, 0x7893, 0x789a, 0x7887, 0x789c,
	0x78a1, 0x78a3, 0x78b2, 0x78b9, 0x78a5, 0x78d4, 0x78d9, 0x78c9,
	0x78ec, 0x78f2, 0x7905, 0x78f4, 0x7913, 0x7924, 0x791e, 0x7934,
	0x9f9b, 0x9ef9, 0x9efb, 0x9efc, 0x76f1, 0x7704, 0x770d, 0x76f9,
	0x7707, 0x7708, 0x771a, 0x7722, 0x7719, 0x772d, 0x7726, 0x7735,
	0x7738, 0x7750, 0x7751, 0x7747, 0x7743, 0x775a, 0x7768, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xee40[ 192 ] = {
	0x980f, 0x9810, 0x9811, 0x9812, 0x9813, 0x9814, 0x9815, 0x9816,
	0x9817, 0x9818, 0x9819, 0x981a, 0x981b, 0x981c, 0x981d, 0x981e,
	0x981f, 0x9820, 0x9821, 0x9822, 0x9823, 0x9824, 0x9825, 0x9826,
	0x9827, 0x9828, 0x9829, 0x982a, 0x982b, 0x982c, 0x982d, 0x982e,
	0x982f, 0x9830, 0x9831, 0x9832, 0x9833, 0x9834, 0x9835, 0x9836,
	0x9837, 0x9838, 0x9839, 0x983a, 0x983b, 0x983c, 0x983d, 0x983e,
	0x983f, 0x9840, 0x9841, 0x9842, 0x9843, 0x9844, 0x9845, 0x9846,
	0x9847, 0x9848, 0x9849, 0x984a, 0x984b, 0x984c, 0x984d, 0xfffd,
	0x984e, 0x984f, 0x9850, 0x9851, 0x9852, 0x9853, 0x9854, 0x9855,
	0x9856, 0x9857, 0x9858, 0x9859, 0x985a, 0x985b, 0x985c, 0x985d,
	0x985e, 0x985f, 0x9860, 0x9861, 0x9862, 0x9863, 0x9864, 0x9865,
	0x9866, 0x9867, 0x9868, 0x9869, 0x986a, 0x986b, 0x986c, 0x986d,
	0x986e, 0x7762, 0x7765, 0x777f, 0x778d, 0x777d, 0x7780, 0x778c,
	0x7791, 0x779f, 0x77a0, 0x77b0, 0x77b5, 0x77bd, 0x753a, 0x7540,
	0x754e, 0x754b, 0x7548, 0x755b, 0x7572, 0x7579, 0x7583, 0x7f58,
	0x7f61, 0x7f5f, 0x8a48, 0x7f68, 0x7f74, 0x7f71, 0x7f79, 0x7f81,
	0x7f7e, 0x76cd, 0x76e5, 0x8832, 0x9485, 0x9486, 0x9487, 0x948b,
	0x948a, 0x948c, 0x948d, 0x948f, 0x9490, 0x9494, 0x9497, 0x9495,
	0x949a, 0x949b, 0x949c, 0x94a3, 0x94a4, 0x94ab, 0x94aa, 0x94ad,
	0x94ac, 0x94af, 0x94b0, 0x94b2, 0x94b4, 0x94b6, 0x94b7, 0x94b8,
	0x94b9, 0x94ba, 0x94bc, 0x94bd, 0x94bf, 0x94c4, 0x94c8, 0x94c9,
	0x94ca, 0x94cb, 0x94cc, 0x94cd, 0x94ce, 0x94d0, 0x94d1, 0x94d2,
	0x94d5, 0x94d6, 0x94d7, 0x94d9, 0x94d8, 0x94db, 0x94de, 0x94df,
	0x94e0, 0x94e2, 0x94e4, 0x94e5, 0x94e7, 0x94e8, 0x94ea, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xef40[ 192 ] = {
	0x986f, 0x9870, 0x9871, 0x9872, 0x9873, 0x9874, 0x988b, 0x988e,
	0x9892, 0x9895, 0x9899, 0x98a3, 0x98a8, 0x98a9, 0x98aa, 0x98ab,
	0x98ac, 0x98ad, 0x98ae, 0x98af, 0x98b0, 0x98b1, 0x98b2, 0x98b3,
	0x98b4, 0x98b5, 0x98b6, 0x98b7, 0x98b8, 0x98b9, 0x98ba, 0x98bb,
	0x98bc, 0x98bd, 0x98be, 0x98bf, 0x98c0, 0x98c1, 0x98c2, 0x98c3,
	0x98c4, 0x98c5, 0x98c6, 0x98c7, 0x98c8, 0x98c9, 0x98ca, 0x98cb,
	0x98cc, 0x98cd, 0x98cf, 0x98d0, 0x98d4, 0x98d6, 0x98d7, 0x98db,
	0x98dc, 0x98dd, 0x98e0, 0x98e1, 0x98e2, 0x98e3, 0x98e4, 0xfffd,
	0x98e5, 0x98e6, 0x98e9, 0x98ea, 0x98eb, 0x98ec, 0x98ed, 0x98ee,
	0x98ef, 0x98f0, 0x98f1, 0x98f2, 0x98f3, 0x98f4, 0x98f5, 0x98f6,
	0x98f7, 0x98f8, 0x98f9, 0x98fa, 0x98fb, 0x98fc, 0x98fd, 0x98fe,
	0x98ff, 0x9900, 0x9901, 0x9902, 0x9903, 0x9904, 0x9905, 0x9906,
	0x9907, 0x94e9, 0x94eb, 0x94ee, 0x94ef, 0x94f3, 0x94f4, 0x94f5,
	0x94f7, 0x94f9, 0x94fc, 0x94fd, 0x94ff, 0x9503, 0x9502, 0x9506,
	0x9507, 0x9509, 0x950a, 0x950d, 0x950e, 0x950f, 0x9512, 0x9513,
	0x9514, 0x9515, 0x9516, 0x9518, 0x951b, 0x951d, 0x951e, 0x951f,
	0x9522, 0x952a, 0x952b, 0x9529, 0x952c, 0x9531, 0x9532, 0x9534,
	0x9536, 0x9537, 0x9538, 0x953c, 0x953e, 0x953f, 0x9542, 0x9535,
	0x9544, 0x9545, 0x9546, 0x9549, 0x954c, 0x954e, 0x954f, 0x9552,
	0x9553, 0x9554, 0x9556, 0x9557, 0x9558, 0x9559, 0x955b, 0x955e,
	0x955f, 0x955d, 0x9561, 0x9562, 0x9564, 0x9565, 0x9566, 0x9567,
	0x9568, 0x9569, 0x956a, 0x956b, 0x956c, 0x956f, 0x9571, 0x9572,
	0x9573, 0x953a, 0x77e7, 0x77ec, 0x96c9, 0x79d5, 0x79ed, 0x79e3,
	0x79eb, 0x7a06, 0x5d47, 0x7a03, 0x7a02, 0x7a1e, 0x7a14, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf040[ 192 ] = {
	0x9908, 0x9909, 0x990a, 0x990b, 0x990c, 0x990e, 0x990f, 0x9911,
	0x9912, 0x9913, 0x9914, 0x9915, 0x9916, 0x9917, 0x9918, 0x9919,
	0x991a, 0x991b, 0x991c, 0x991d, 0x991e, 0x991f, 0x9920, 0x9921,
	0x9922, 0x9923, 0x9924, 0x9925, 0x9926, 0x9927, 0x9928, 0x9929,
	0x992a, 0x992b, 0x992c, 0x992d, 0x992f, 0x9930, 0x9931, 0x9932,
	0x9933, 0x9934, 0x9935, 0x9936, 0x9937, 0x9938, 0x9939, 0x993a,
	0x993b, 0x993c, 0x993d, 0x993e, 0x993f, 0x9940, 0x9941, 0x9942,
	0x9943, 0x9944, 0x9945, 0x9946, 0x9947, 0x9948, 0x9949, 0xfffd,
	0x994a, 0x994b, 0x994c, 0x994d, 0x994e, 0x994f, 0x9950, 0x9951,
	0x9952, 0x9953, 0x9956, 0x9957, 0x9958, 0x9959, 0x995a, 0x995b,
	0x995c, 0x995d, 0x995e, 0x995f, 0x9960, 0x9961, 0x9962, 0x9964,
	0x9966, 0x9973, 0x9978, 0x9979, 0x997b, 0x997e, 0x9982, 0x9983,
	0x9989, 0x7a39, 0x7a37, 0x7a51, 0x9ecf, 0x99a5, 0x7a70, 0x7688,
	0x768e, 0x7693, 0x7699, 0x76a4, 0x74de, 0x74e0, 0x752c, 0x9e20,
	0x9e22, 0x9e28, 0x9e29, 0x9e2a, 0x9e2b, 0x9e2c, 0x9e32, 0x9e31,
	0x9e36, 0x9e38, 0x9e37, 0x9e39, 0x9e3a, 0x9e3e, 0x9e41, 0x9e42,
	0x9e44, 0x9e46, 0x9e47, 0x9e48, 0x9e49, 0x9e4b, 0x9e4c, 0x9e4e,
	0x9e51, 0x9e55, 0x9e57, 0x9e5a, 0x9e5b, 0x9e5c, 0x9e5e, 0x9e63,
	0x9e66, 0x9e67, 0x9e68, 0x9e69, 0x9e6a, 0x9e6b, 0x9e6c, 0x9e71,
	0x9e6d, 0x9e73, 0x7592, 0x7594, 0x7596, 0x75a0, 0x759d, 0x75ac,
	0x75a3, 0x75b3, 0x75b4, 0x75b8, 0x75c4, 0x75b1, 0x75b0, 0x75c3,
	0x75c2, 0x75d6, 0x75cd, 0x75e3, 0x75e8, 0x75e6, 0x75e4, 0x75eb,
	0x75e7, 0x7603, 0x75f1, 0x75fc, 0x75ff, 0x7610, 0x7600, 0x7605,
	0x760c, 0x7617, 0x760a, 0x7625, 0x7618, 0x7615, 0x7619, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf140[ 192 ] = {
	0x998c, 0x998e, 0x999a, 0x999b, 0x999c, 0x999d, 0x999e, 0x999f,
	0x99a0, 0x99a1, 0x99a2, 0x99a3, 0x99a4, 0x99a6, 0x99a7, 0x99a9,
	0x99aa, 0x99ab, 0x99ac, 0x99ad, 0x99ae, 0x99af, 0x99b0, 0x99b1,
	0x99b2, 0x99b3, 0x99b4, 0x99b5, 0x99b6, 0x99b7, 0x99b8, 0x99b9,
	0x99ba, 0x99bb, 0x99bc, 0x99bd, 0x99be, 0x99bf, 0x99c0, 0x99c1,
	0x99c2, 0x99c3, 0x99c4, 0x99c5, 0x99c6, 0x99c7, 0x99c8, 0x99c9,
	0x99ca, 0x99cb, 0x99cc, 0x99cd, 0x99ce, 0x99cf, 0x99d0, 0x99d1,
	0x99d2, 0x99d3, 0x99d4, 0x99d5, 0x99d6, 0x99d7, 0x99d8, 0xfffd,
	0x99d9, 0x99da, 0x99db, 0x99dc, 0x99dd, 0x99de, 0x99df, 0x99e0,
	0x99e1, 0x99e2, 0x99e3, 0x99e4, 0x99e5, 0x99e6, 0x99e7, 0x99e8,
	0x99e9, 0x99ea, 0x99eb, 0x99ec, 0x99ed, 0x99ee, 0x99ef, 0x99f0,
	0x99f1, 0x99f2, 0x99f3, 0x99f4, 0x99f5, 0x99f6, 0x99f7, 0x99f8,
	0x99f9, 0x761b, 0x763c, 0x7622, 0x7620, 0x7640, 0x762d, 0x7630,
	0x763f, 0x7635, 0x7643, 0x763e, 0x7633, 0x764d, 0x765e, 0x7654,
	0x765c, 0x7656, 0x766b, 0x766f, 0x7fca, 0x7ae6, 0x7a78, 0x7a79,
	0x7a80, 0x7a86, 0x7a88, 0x7a95, 0x7aa6, 0x7aa0, 0x7aac, 0x7aa8,
	0x7aad, 0x7ab3, 0x8864, 0x8869, 0x8872, 0x887d, 0x887f, 0x8882,
	0x88a2, 0x88c6, 0x88b7, 0x88bc, 0x88c9, 0x88e2, 0x88ce, 0x88e3,
	0x88e5, 0x88f1, 0x891a, 0x88fc, 0x88e8, 0x88fe, 0x88f0, 0x8921,
	0x8919, 0x8913, 0x891b, 0x890a, 0x8934, 0x892b, 0x8936, 0x8941,
	0x8966, 0x897b, 0x758b, 0x80e5, 0x76b2, 0x76b4, 0x77dc, 0x8012,
	0x8014, 0x8016, 0x801c, 0x8020, 0x8022, 0x8025, 0x8026, 0x8027,
	0x8029, 0x8028, 0x8031, 0x800b, 0x8035, 0x8043, 0x8046, 0x804d,
	0x8052, 0x8069, 0x8071, 0x8983, 0x9878, 0x9880, 0x9883, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf240[ 192 ] = {
	0x99fa, 0x99fb, 0x99fc, 0x99fd, 0x99fe, 0x99ff, 0x9a00, 0x9a01,
	0x9a02, 0x9a03, 0x9a04, 0x9a05, 0x9a06, 0x9a07, 0x9a08, 0x9a09,
	0x9a0a, 0x9a0b, 0x9a0c, 0x9a0d, 0x9a0e, 0x9a0f, 0x9a10, 0x9a11,
	0x9a12, 0x9a13, 0x9a14, 0x9a15, 0x9a16, 0x9a17, 0x9a18, 0x9a19,
	0x9a1a, 0x9a1b, 0x9a1c, 0x9a1d, 0x9a1e, 0x9a1f, 0x9a20, 0x9a21,
	0x9a22, 0x9a23, 0x9a24, 0x9a25, 0x9a26, 0x9a27, 0x9a28, 0x9a29,
	0x9a2a, 0x9a2b, 0x9a2c, 0x9a2d, 0x9a2e, 0x9a2f, 0x9a30, 0x9a31,
	0x9a32, 0x9a33, 0x9a34, 0x9a35, 0x9a36, 0x9a37, 0x9a38, 0xfffd,
	0x9a39, 0x9a3a, 0x9a3b, 0x9a3c, 0x9a3d, 0x9a3e, 0x9a3f, 0x9a40,
	0x9a41, 0x9a42, 0x9a43, 0x9a44, 0x9a45, 0x9a46, 0x9a47, 0x9a48,
	0x9a49, 0x9a4a, 0x9a4b, 0x9a4c, 0x9a4d, 0x9a4e, 0x9a4f, 0x9a50,
	0x9a51, 0x9a52, 0x9a53, 0x9a54, 0x9a55, 0x9a56, 0x9a57, 0x9a58,
	0x9a59, 0x9889, 0x988c, 0x988d, 0x988f, 0x9894, 0x989a, 0x989b,
	0x989e, 0x989f, 0x98a1, 0x98a2, 0x98a5, 0x98a6, 0x864d, 0x8654,
	0x866c, 0x866e, 0x867f, 0x867a, 0x867c, 0x867b, 0x86a8, 0x868d,
	0x868b, 0x86ac, 0x869d, 0x86a7, 0x86a3, 0x86aa, 0x8693, 0x86a9,
	0x86b6, 0x86c4, 0x86b5, 0x86ce, 0x86b0, 0x86ba, 0x86b1, 0x86af,
	0x86c9, 0x86cf, 0x86b4, 0x86e9, 0x86f1, 0x86f2, 0x86ed, 0x86f3,
	0x86d0, 0x8713, 0x86de, 0x86f4, 0x86df, 0x86d8, 0x86d1, 0x8703,
	0x8707, 0x86f8, 0x8708, 0x870a, 0x870d, 0x8709, 0x8723, 0x873b,
	0x871e, 0x8725, 0x872e, 0x871a, 0x873e, 0x8748, 0x8734, 0x8731,
	0x8729, 0x8737, 0x873f, 0x8782, 0x8722, 0x877d, 0x877e, 0x877b,
	0x8760, 0x8770, 0x874c, 0x876e, 0x878b, 0x8753, 0x8763, 0x877c,
	0x8764, 0x8759, 0x8765, 0x8793, 0x87af, 0x87a8, 0x87d2, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf340[ 192 ] = {
	0x9a5a, 0x9a5b, 0x9a5c, 0x9a5d, 0x9a5e, 0x9a5f, 0x9a60, 0x9a61,
	0x9a62, 0x9a63, 0x9a64, 0x9a65, 0x9a66, 0x9a67, 0x9a68, 0x9a69,
	0x9a6a, 0x9a6b, 0x9a72, 0x9a83, 0x9a89, 0x9a8d, 0x9a8e, 0x9a94,
	0x9a95, 0x9a99, 0x9aa6, 0x9aa9, 0x9aaa, 0x9aab, 0x9aac, 0x9aad,
	0x9aae, 0x9aaf, 0x9ab2, 0x9ab3, 0x9ab4, 0x9ab5, 0x9ab9, 0x9abb,
	0x9abd, 0x9abe, 0x9abf, 0x9ac3, 0x9ac4, 0x9ac6, 0x9ac7, 0x9ac8,
	0x9ac9, 0x9aca, 0x9acd, 0x9ace, 0x9acf, 0x9ad0, 0x9ad2, 0x9ad4,
	0x9ad5, 0x9ad6, 0x9ad7, 0x9ad9, 0x9ada, 0x9adb, 0x9adc, 0xfffd,
	0x9add, 0x9ade, 0x9ae0, 0x9ae2, 0x9ae3, 0x9ae4, 0x9ae5, 0x9ae7,
	0x9ae8, 0x9ae9, 0x9aea, 0x9aec, 0x9aee, 0x9af0, 0x9af1, 0x9af2,
	0x9af3, 0x9af4, 0x9af5, 0x9af6, 0x9af7, 0x9af8, 0x9afa, 0x9afc,
	0x9afd, 0x9afe, 0x9aff, 0x9b00, 0x9b01, 0x9b02, 0x9b04, 0x9b05,
	0x9b06, 0x87c6, 0x8788, 0x8785, 0x87ad, 0x8797, 0x8783, 0x87ab,
	0x87e5, 0x87ac, 0x87b5, 0x87b3, 0x87cb, 0x87d3, 0x87bd, 0x87d1,
	0x87c0, 0x87ca, 0x87db, 0x87ea, 0x87e0, 0x87ee, 0x8816, 0x8813,
	0x87fe, 0x880a, 0x881b, 0x8821, 0x8839, 0x883c, 0x7f36, 0x7f42,
	0x7f44, 0x7f45, 0x8210, 0x7afa, 0x7afd, 0x7b08, 0x7b03, 0x7b04,
	0x7b15, 0x7b0a, 0x7b2b, 0x7b0f, 0x7b47, 0x7b38, 0x7b2a, 0x7b19,
	0x7b2e, 0x7b31, 0x7b20, 0x7b25, 0x7b24, 0x7b33, 0x7b3e, 0x7b1e,
	0x7b58, 0x7b5a, 0x7b45, 0x7b75, 0x7b4c, 0x7b5d, 0x7b60, 0x7b6e,
	0x7b7b, 0x7b62, 0x7b72, 0x7b71, 0x7b90, 0x7ba6, 0x7ba7, 0x7bb8,
	0x7bac, 0x7b9d, 0x7ba8, 0x7b85, 0x7baa, 0x7b9c, 0x7ba2, 0x7bab,
	0x7bb4, 0x7bd1, 0x7bc1, 0x7bcc, 0x7bdd, 0x7bda, 0x7be5, 0x7be6,
	0x7bea, 0x7c0c, 0x7bfe, 0x7bfc, 0x7c0f, 0x7c16, 0x7c0b, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf440[ 192 ] = {
	0x9b07, 0x9b09, 0x9b0a, 0x9b0b, 0x9b0c, 0x9b0d, 0x9b0e, 0x9b10,
	0x9b11, 0x9b12, 0x9b14, 0x9b15, 0x9b16, 0x9b17, 0x9b18, 0x9b19,
	0x9b1a, 0x9b1b, 0x9b1c, 0x9b1d, 0x9b1e, 0x9b20, 0x9b21, 0x9b22,
	0x9b24, 0x9b25, 0x9b26, 0x9b27, 0x9b28, 0x9b29, 0x9b2a, 0x9b2b,
	0x9b2c, 0x9b2d, 0x9b2e, 0x9b30, 0x9b31, 0x9b33, 0x9b34, 0x9b35,
	0x9b36, 0x9b37, 0x9b38, 0x9b39, 0x9b3a, 0x9b3d, 0x9b3e, 0x9b3f,
	0x9b40, 0x9b46, 0x9b4a, 0x9b4b, 0x9b4c, 0x9b4e, 0x9b50, 0x9b52,
	0x9b53, 0x9b55, 0x9b56, 0x9b57, 0x9b58, 0x9b59, 0x9b5a, 0xfffd,
	0x9b5b, 0x9b5c, 0x9b5d, 0x9b5e, 0x9b5f, 0x9b60, 0x9b61, 0x9b62,
	0x9b63, 0x9b64, 0x9b65, 0x9b66, 0x9b67, 0x9b68, 0x9b69, 0x9b6a,
	0x9b6b, 0x9b6c, 0x9b6d, 0x9b6e, 0x9b6f, 0x9b70, 0x9b71, 0x9b72,
	0x9b73, 0x9b74, 0x9b75, 0x9b76, 0x9b77, 0x9b78, 0x9b79, 0x9b7a,
	0x9b7b, 0x7c1f, 0x7c2a, 0x7c26, 0x7c38, 0x7c41, 0x7c40, 0x81fe,
	0x8201, 0x8202, 0x8204, 0x81ec, 0x8844, 0x8221, 0x8222, 0x8223,
	0x822d, 0x822f, 0x8228, 0x822b, 0x8238, 0x823b, 0x8233, 0x8234,
	0x823e, 0x8244, 0x8249, 0x824b, 0x824f, 0x825a, 0x825f, 0x8268,
	0x887e, 0x8885, 0x8888, 0x88d8, 0x88df, 0x895e, 0x7f9d, 0x7f9f,
	0x7fa7, 0x7faf, 0x7fb0, 0x7fb2, 0x7c7c, 0x6549, 0x7c91, 0x7c9d,
	0x7c9c, 0x7c9e, 0x7ca2, 0x7cb2, 0x7cbc, 0x7cbd, 0x7cc1, 0x7cc7,
	0x7ccc, 0x7ccd, 0x7cc8, 0x7cc5, 0x7cd7, 0x7ce8, 0x826e, 0x66a8,
	0x7fbf, 0x7fce, 0x7fd5, 0x7fe5, 0x7fe1, 0x7fe6, 0x7fe9, 0x7fee,
	0x7ff3, 0x7cf8, 0x7d77, 0x7da6, 0x7dae, 0x7e47, 0x7e9b, 0x9eb8,
	0x9eb4, 0x8d73, 0x8d84, 0x8d94, 0x8d91, 0x8db1, 0x8d67, 0x8d6d,
	0x8c47, 0x8c49, 0x914a, 0x9150, 0x914e, 0x914f, 0x9164, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf540[ 192 ] = {
	0x9b7c, 0x9b7d, 0x9b7e, 0x9b7f, 0x9b80, 0x9b81, 0x9b82, 0x9b83,
	0x9b84, 0x9b85, 0x9b86, 0x9b87, 0x9b88, 0x9b89, 0x9b8a, 0x9b8b,
	0x9b8c, 0x9b8d, 0x9b8e, 0x9b8f, 0x9b90, 0x9b91, 0x9b92, 0x9b93,
	0x9b94, 0x9b95, 0x9b96, 0x9b97, 0x9b98, 0x9b99, 0x9b9a, 0x9b9b,
	0x9b9c, 0x9b9d, 0x9b9e, 0x9b9f, 0x9ba0, 0x9ba1, 0x9ba2, 0x9ba3,
	0x9ba4, 0x9ba5, 0x9ba6, 0x9ba7, 0x9ba8, 0x9ba9, 0x9baa, 0x9bab,
	0x9bac, 0x9bad, 0x9bae, 0x9baf, 0x9bb0, 0x9bb1, 0x9bb2, 0x9bb3,
	0x9bb4, 0x9bb5, 0x9bb6, 0x9bb7, 0x9bb8, 0x9bb9, 0x9bba, 0xfffd,
	0x9bbb, 0x9bbc, 0x9bbd, 0x9bbe, 0x9bbf, 0x9bc0, 0x9bc1, 0x9bc2,
	0x9bc3, 0x9bc4, 0x9bc5, 0x9bc6, 0x9bc7, 0x9bc8, 0x9bc9, 0x9bca,
	0x9bcb, 0x9bcc, 0x9bcd, 0x9bce, 0x9bcf, 0x9bd0, 0x9bd1, 0x9bd2,
	0x9bd3, 0x9bd4, 0x9bd5, 0x9bd6, 0x9bd7, 0x9bd8, 0x9bd9, 0x9bda,
	0x9bdb, 0x9162, 0x9161, 0x9170, 0x9169, 0x916f, 0x917d, 0x917e,
	0x9172, 0x9174, 0x9179, 0x918c, 0x9185, 0x9190, 0x918d, 0x9191,
	0x91a2, 0x91a3, 0x91aa, 0x91ad, 0x91ae, 0x91af, 0x91b5, 0x91b4,
	0x91ba, 0x8c55, 0x9e7e, 0x8db8, 0x8deb, 0x8e05, 0x8e59, 0x8e69,
	0x8db5, 0x8dbf, 0x8dbc, 0x8dba, 0x8dc4, 0x8dd6, 0x8dd7, 0x8dda,
	0x8dde, 0x8dce, 0x8dcf, 0x8ddb, 0x8dc6, 0x8dec, 0x8df7, 0x8df8,
	0x8de3, 0x8df9, 0x8dfb, 0x8de4, 0x8e09, 0x8dfd, 0x8e14, 0x8e1d,
	0x8e1f, 0x8e2c, 0x8e2e, 0x8e23, 0x8e2f, 0x8e3a, 0x8e40, 0x8e39,
	0x8e35, 0x8e3d, 0x8e31, 0x8e49, 0x8e41, 0x8e42, 0x8e51, 0x8e52,
	0x8e4a, 0x8e70, 0x8e76, 0x8e7c, 0x8e6f, 0x8e74, 0x8e85, 0x8e8f,
	0x8e94, 0x8e90, 0x8e9c, 0x8e9e, 0x8c78, 0x8c82, 0x8c8a, 0x8c85,
	0x8c98, 0x8c94, 0x659b, 0x89d6, 0x89de, 0x89da, 0x89dc, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf640[ 192 ] = {
	0x9bdc, 0x9bdd, 0x9bde, 0x9bdf, 0x9be0, 0x9be1, 0x9be2, 0x9be3,
	0x9be4, 0x9be5, 0x9be6, 0x9be7, 0x9be8, 0x9be9, 0x9bea, 0x9beb,
	0x9bec, 0x9bed, 0x9bee, 0x9bef, 0x9bf0, 0x9bf1, 0x9bf2, 0x9bf3,
	0x9bf4, 0x9bf5, 0x9bf6, 0x9bf7, 0x9bf8, 0x9bf9, 0x9bfa, 0x9bfb,
	0x9bfc, 0x9bfd, 0x9bfe, 0x9bff, 0x9c00, 0x9c01, 0x9c02, 0x9c03,
	0x9c04, 0x9c05, 0x9c06, 0x9c07, 0x9c08, 0x9c09, 0x9c0a, 0x9c0b,
	0x9c0c, 0x9c0d, 0x9c0e, 0x9c0f, 0x9c10, 0x9c11, 0x9c12, 0x9c13,
	0x9c14, 0x9c15, 0x9c16, 0x9c17, 0x9c18, 0x9c19, 0x9c1a, 0xfffd,
	0x9c1b, 0x9c1c, 0x9c1d, 0x9c1e, 0x9c1f, 0x9c20, 0x9c21, 0x9c22,
	0x9c23, 0x9c24, 0x9c25, 0x9c26, 0x9c27, 0x9c28, 0x9c29, 0x9c2a,
	0x9c2b, 0x9c2c, 0x9c2d, 0x9c2e, 0x9c2f, 0x9c30, 0x9c31, 0x9c32,
	0x9c33, 0x9c34, 0x9c35, 0x9c36, 0x9c37, 0x9c38, 0x9c39, 0x9c3a,
	0x9c3b, 0x89e5, 0x89eb, 0x89ef, 0x8a3e, 0x8b26, 0x9753, 0x96e9,
	0x96f3, 0x96ef, 0x9706, 0x9701, 0x9708, 0x970f, 0x970e, 0x972a,
	0x972d, 0x9730, 0x973e, 0x9f80, 0x9f83, 0x9f85, 0x9f86, 0x9f87,
	0x9f88, 0x9f89, 0x9f8a, 0x9f8c, 0x9efe, 0x9f0b, 0x9f0d, 0x96b9,
	0x96bc, 0x96bd, 0x96ce, 0x96d2, 0x77bf, 0x96e0, 0x928e, 0x92ae,
	0x92c8, 0x933e, 0x936a, 0x93ca, 0x938f, 0x943e, 0x946b, 0x9c7f,
	0x9c82, 0x9c85, 0x9c86, 0x9c87, 0x9c88, 0x7a23, 0x9c8b, 0x9c8e,
	0x9c90, 0x9c91, 0x9c92, 0x9c94, 0x9c95, 0x9c9a, 0x9c9b, 0x9c9e,
	0x9c9f, 0x9ca0, 0x9ca1, 0x9ca2, 0x9ca3, 0x9ca5, 0x9ca6, 0x9ca7,
	0x9ca8, 0x9ca9, 0x9cab, 0x9cad, 0x9cae, 0x9cb0, 0x9cb1, 0x9cb2,
	0x9cb3, 0x9cb4, 0x9cb5, 0x9cb6, 0x9cb7, 0x9cba, 0x9cbb, 0x9cbc,
	0x9cbd, 0x9cc4, 0x9cc5, 0x9cc6, 0x9cc7, 0x9cca, 0x9ccb, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf740[ 192 ] = {
	0x9c3c, 0x9c3d, 0x9c3e, 0x9c3f, 0x9c40, 0x9c41, 0x9c42, 0x9c43,
	0x9c44, 0x9c45, 0x9c46, 0x9c47, 0x9c48, 0x9c49, 0x9c4a, 0x9c4b,
	0x9c4c, 0x9c4d, 0x9c4e, 0x9c4f, 0x9c50, 0x9c51, 0x9c52, 0x9c53,
	0x9c54, 0x9c55, 0x9c56, 0x9c57, 0x9c58, 0x9c59, 0x9c5a, 0x9c5b,
	0x9c5c, 0x9c5d, 0x9c5e, 0x9c5f, 0x9c60, 0x9c61, 0x9c62, 0x9c63,
	0x9c64, 0x9c65, 0x9c66, 0x9c67, 0x9c68, 0x9c69, 0x9c6a, 0x9c6b,
	0x9c6c, 0x9c6d, 0x9c6e, 0x9c6f, 0x9c70, 0x9c71, 0x9c72, 0x9c73,
	0x9c74, 0x9c75, 0x9c76, 0x9c77, 0x9c78, 0x9c79, 0x9c7a, 0xfffd,
	0x9c7b, 0x9c7d, 0x9c7e, 0x9c80, 0x9c83, 0x9c84, 0x9c89, 0x9c8a,
	0x9c8c, 0x9c8f, 0x9c93, 0x9c96, 0x9c97, 0x9c98, 0x9c99, 0x9c9d,
	0x9caa, 0x9cac, 0x9caf, 0x9cb9, 0x9cbe, 0x9cbf, 0x9cc0, 0x9cc1,
	0x9cc2, 0x9cc8, 0x9cc9, 0x9cd1, 0x9cd2, 0x9cda, 0x9cdb, 0x9ce0,
	0x9ce1, 0x9ccc, 0x9ccd, 0x9cce, 0x9ccf, 0x9cd0, 0x9cd3, 0x9cd4,
	0x9cd5, 0x9cd7, 0x9cd8, 0x9cd9, 0x9cdc, 0x9cdd, 0x9cdf, 0x9ce2,
	0x977c, 0x9785, 0x9791, 0x9792, 0x9794, 0x97af, 0x97ab, 0x97a3,
	0x97b2, 0x97b4, 0x9ab1, 0x9ab0, 0x9ab7, 0x9e58, 0x9ab6, 0x9aba,
	0x9abc, 0x9ac1, 0x9ac0, 0x9ac5, 0x9ac2, 0x9acb, 0x9acc, 0x9ad1,
	0x9b45, 0x9b43, 0x9b47, 0x9b49, 0x9b48, 0x9b4d, 0x9b51, 0x98e8,
	0x990d, 0x992e, 0x9955, 0x9954, 0x9adf, 0x9ae1, 0x9ae6, 0x9aef,
	0x9aeb, 0x9afb, 0x9aed, 0x9af9, 0x9b08, 0x9b0f, 0x9b13, 0x9b1f,
	0x9b23, 0x9ebd, 0x9ebe, 0x7e3b, 0x9e82, 0x9e87, 0x9e88, 0x9e8b,
	0x9e92, 0x93d6, 0x9e9d, 0x9e9f, 0x9edb, 0x9edc, 0x9edd, 0x9ee0,
	0x9edf, 0x9ee2, 0x9ee9, 0x9ee7, 0x9ee5, 0x9eea, 0x9eef, 0x9f22,
	0x9f2c, 0x9f2f, 0x9f39, 0x9f37, 0x9f3d, 0x9f3e, 0x9f44, 0xfffd
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf840[ 96 ] = {
	0x9ce3, 0x9ce4, 0x9ce5, 0x9ce6, 0x9ce7, 0x9ce8, 0x9ce9, 0x9cea,
	0x9ceb, 0x9cec, 0x9ced, 0x9cee, 0x9cef, 0x9cf0, 0x9cf1, 0x9cf2,
	0x9cf3, 0x9cf4, 0x9cf5, 0x9cf6, 0x9cf7, 0x9cf8, 0x9cf9, 0x9cfa,
	0x9cfb, 0x9cfc, 0x9cfd, 0x9cfe, 0x9cff, 0x9d00, 0x9d01, 0x9d02,
	0x9d03, 0x9d04, 0x9d05, 0x9d06, 0x9d07, 0x9d08, 0x9d09, 0x9d0a,
	0x9d0b, 0x9d0c, 0x9d0d, 0x9d0e, 0x9d0f, 0x9d10, 0x9d11, 0x9d12,
	0x9d13, 0x9d14, 0x9d15, 0x9d16, 0x9d17, 0x9d18, 0x9d19, 0x9d1a,
	0x9d1b, 0x9d1c, 0x9d1d, 0x9d1e, 0x9d1f, 0x9d20, 0x9d21, 0xfffd,
	0x9d22, 0x9d23, 0x9d24, 0x9d25, 0x9d26, 0x9d27, 0x9d28, 0x9d29,
	0x9d2a, 0x9d2b, 0x9d2c, 0x9d2d, 0x9d2e, 0x9d2f, 0x9d30, 0x9d31,
	0x9d32, 0x9d33, 0x9d34, 0x9d35, 0x9d36, 0x9d37, 0x9d38, 0x9d39,
	0x9d3a, 0x9d3b, 0x9d3c, 0x9d3d, 0x9d3e, 0x9d3f, 0x9d40, 0x9d41
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf940[ 96 ] = {
	0x9d43, 0x9d44, 0x9d45, 0x9d46, 0x9d47, 0x9d48, 0x9d49, 0x9d4a,
	0x9d4b, 0x9d4c, 0x9d4d, 0x9d4e, 0x9d4f, 0x9d50, 0x9d51, 0x9d52,
	0x9d53, 0x9d54, 0x9d55, 0x9d56, 0x9d57, 0x9d58, 0x9d59, 0x9d5a,
	0x9d5b, 0x9d5c, 0x9d5d, 0x9d5e, 0x9d5f, 0x9d60, 0x9d61, 0x9d62,
	0x9d63, 0x9d64, 0x9d65, 0x9d66, 0x9d67, 0x9d68, 0x9d69, 0x9d6a,
	0x9d6b, 0x9d6c, 0x9d6d, 0x9d6e, 0x9d6f, 0x9d70, 0x9d71, 0x9d72,
	0x9d73, 0x9d74, 0x9d75, 0x9d76, 0x9d77, 0x9d78, 0x9d79, 0x9d7a,
	0x9d7b, 0x9d7c, 0x9d7d, 0x9d7e, 0x9d7f, 0x9d80, 0x9d81, 0xfffd,
	0x9d82, 0x9d83, 0x9d84, 0x9d85, 0x9d86, 0x9d87, 0x9d88, 0x9d89,
	0x9d8a, 0x9d8b, 0x9d8c, 0x9d8d, 0x9d8e, 0x9d8f, 0x9d90, 0x9d91,
	0x9d92, 0x9d93, 0x9d94, 0x9d95, 0x9d96, 0x9d97, 0x9d98, 0x9d99,
	0x9d9a, 0x9d9b, 0x9d9c, 0x9d9d, 0x9d9e, 0x9d9f, 0x9da0, 0x9da1
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xfa40[ 96 ] = {
	0x9da3, 0x9da4, 0x9da5, 0x9da6, 0x9da7, 0x9da8, 0x9da9, 0x9daa,
	0x9dab, 0x9dac, 0x9dad, 0x9dae, 0x9daf, 0x9db0, 0x9db1, 0x9db2,
	0x9db3, 0x9db4, 0x9db5, 0x9db6, 0x9db7, 0x9db8, 0x9db9, 0x9dba,
	0x9dbb, 0x9dbc, 0x9dbd, 0x9dbe, 0x9dbf, 0x9dc0, 0x9dc1, 0x9dc2,
	0x9dc3, 0x9dc4, 0x9dc5, 0x9dc6, 0x9dc7, 0x9dc8, 0x9dc9, 0x9dca,
	0x9dcb, 0x9dcc, 0x9dcd, 0x9dce, 0x9dcf, 0x9dd0, 0x9dd1, 0x9dd2,
	0x9dd3, 0x9dd4, 0x9dd5, 0x9dd6, 0x9dd7, 0x9dd8, 0x9dd9, 0x9dda,
	0x9ddb, 0x9ddc, 0x9ddd, 0x9dde, 0x9ddf, 0x9de0, 0x9de1, 0xfffd,
	0x9de2, 0x9de3, 0x9de4, 0x9de5, 0x9de6, 0x9de7, 0x9de8, 0x9de9,
	0x9dea, 0x9deb, 0x9dec, 0x9ded, 0x9dee, 0x9def, 0x9df0, 0x9df1,
	0x9df2, 0x9df3, 0x9df4, 0x9df5, 0x9df6, 0x9df7, 0x9df8, 0x9df9,
	0x9dfa, 0x9dfb, 0x9dfc, 0x9dfd, 0x9dfe, 0x9dff, 0x9e00, 0x9e01
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xfb40[ 96 ] = {
	0x9e03, 0x9e04, 0x9e05, 0x9e06, 0x9e07, 0x9e08, 0x9e09, 0x9e0a,
	0x9e0b, 0x9e0c, 0x9e0d, 0x9e0e, 0x9e0f, 0x9e10, 0x9e11, 0x9e12,
	0x9e13, 0x9e14, 0x9e15, 0x9e16, 0x9e17, 0x9e18, 0x9e19, 0x9e1a,
	0x9e1b, 0x9e1c, 0x9e1d, 0x9e1e, 0x9e24, 0x9e27, 0x9e2e, 0x9e30,
	0x9e34, 0x9e3b, 0x9e3c, 0x9e40, 0x9e4d, 0x9e50, 0x9e52, 0x9e53,
	0x9e54, 0x9e56, 0x9e59, 0x9e5d, 0x9e5f, 0x9e60, 0x9e61, 0x9e62,
	0x9e65, 0x9e6e, 0x9e6f, 0x9e72, 0x9e74, 0x9e75, 0x9e76, 0x9e77,
	0x9e78, 0x9e79, 0x9e7a, 0x9e7b, 0x9e7c, 0x9e7d, 0x9e80, 0xfffd,
	0x9e81, 0x9e83, 0x9e84, 0x9e85, 0x9e86, 0x9e89, 0x9e8a, 0x9e8c,
	0x9e8d, 0x9e8e, 0x9e8f, 0x9e90, 0x9e91, 0x9e94, 0x9e95, 0x9e96,
	0x9e97, 0x9e98, 0x9e99, 0x9e9a, 0x9e9b, 0x9e9c, 0x9e9e, 0x9ea0,
	0x9ea1, 0x9ea2, 0x9ea3, 0x9ea4, 0x9ea5, 0x9ea7, 0x9ea8, 0x9ea9
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xfc40[ 96 ] = {
	0x9eab, 0x9eac, 0x9ead, 0x9eae, 0x9eaf, 0x9eb0, 0x9eb1, 0x9eb2,
	0x9eb3, 0x9eb5, 0x9eb6, 0x9eb7, 0x9eb9, 0x9eba, 0x9ebc, 0x9ebf,
	0x9ec0, 0x9ec1, 0x9ec2, 0x9ec3, 0x9ec5, 0x9ec6, 0x9ec7, 0x9ec8,
	0x9eca, 0x9ecb, 0x9ecc, 0x9ed0, 0x9ed2, 0x9ed3, 0x9ed5, 0x9ed6,
	0x9ed7, 0x9ed9, 0x9eda, 0x9ede, 0x9ee1, 0x9ee3, 0x9ee4, 0x9ee6,
	0x9ee8, 0x9eeb, 0x9eec, 0x9eed, 0x9eee, 0x9ef0, 0x9ef1, 0x9ef2,
	0x9ef3, 0x9ef4, 0x9ef5, 0x9ef6, 0x9ef7, 0x9ef8, 0x9efa, 0x9efd,
	0x9eff, 0x9f00, 0x9f01, 0x9f02, 0x9f03, 0x9f04, 0x9f05, 0xfffd,
	0x9f06, 0x9f07, 0x9f08, 0x9f09, 0x9f0a, 0x9f0c, 0x9f0f, 0x9f11,
	0x9f12, 0x9f14, 0x9f15, 0x9f16, 0x9f18, 0x9f1a, 0x9f1b, 0x9f1c,
	0x9f1d, 0x9f1e, 0x9f1f, 0x9f21, 0x9f23, 0x9f24, 0x9f25, 0x9f26,
	0x9f27, 0x9f28, 0x9f29, 0x9f2a, 0x9f2b, 0x9f2d, 0x9f2e, 0x9f30
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xfd40[ 96 ] = {
	0x9f32, 0x9f33, 0x9f34, 0x9f35, 0x9f36, 0x9f38, 0x9f3a, 0x9f3c,
	0x9f3f, 0x9f40, 0x9f41, 0x9f42, 0x9f43, 0x9f45, 0x9f46, 0x9f47,
	0x9f48, 0x9f49, 0x9f4a, 0x9f4b, 0x9f4c, 0x9f4d, 0x9f4e, 0x9f4f,
	0x9f52, 0x9f53, 0x9f54, 0x9f55, 0x9f56, 0x9f57, 0x9f58, 0x9f59,
	0x9f5a, 0x9f5b, 0x9f5c, 0x9f5d, 0x9f5e, 0x9f5f, 0x9f60, 0x9f61,
	0x9f62, 0x9f63, 0x9f64, 0x9f65, 0x9f66, 0x9f67, 0x9f68, 0x9f69,
	0x9f6a, 0x9f6b, 0x9f6c, 0x9f6d, 0x9f6e, 0x9f6f, 0x9f70, 0x9f71,
	0x9f72, 0x9f73, 0x9f74, 0x9f75, 0x9f76, 0x9f77, 0x9f78, 0xfffd,
	0x9f79, 0x9f7a, 0x9f7b, 0x9f7c, 0x9f7d, 0x9f7e, 0x9f81, 0x9f82,
	0x9f8d, 0x9f8e, 0x9f8f, 0x9f90, 0x9f91, 0x9f92, 0x9f93, 0x9f94,
	0x9f95, 0x9f96, 0x9f97, 0x9f98, 0x9f9c, 0x9f9d, 0x9f9e, 0x9fa1,
	0x9fa2, 0x9fa3, 0x9fa4, 0x9fa5, 0xf92c, 0xf979, 0xf995, 0xf9e7
};

const uint16_t libuna_codepage_windows_936_byte_stream_to_unicode_base_0xfe40[ 16 ] = {
	0xfa0c, 0xfa0d, 0xfa0e, 0xfa0f, 0xfa11, 0xfa13, 0xfa14, 0xfa18,
	0xfa1f, 0xfa20, 0xfa21, 0xfa23, 0xfa24, 0xfa27, 0xfa28, 0xfa29
};

/* Unicode to ASCII character lookup table for Windows 936 codepage
 * Unknown are filled with the ASCII replacement character 0x1a
 */

const uint16_t libuna_codepage_windows_936_unicode_to_byte_stream_base_0x0080[ 1024 ] = {
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xe8a1, 0x001a, 0x001a, 0xeca1,
	0xa7a1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xe3a1, 0xc0a1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa4a1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc1a1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xa4a8, 0xa2a8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xa8a8, 0xa6a8, 0xbaa8, 0x001a, 0xaca8, 0xaaa8, 0x001a, 0x001a,
	0x001a, 0x001a, 0xb0a8, 0xaea8, 0x001a, 0x001a, 0x001a, 0xc2a1,
	0x001a, 0xb4a8, 0xb2a8, 0x001a, 0xb9a8, 0x001a, 0x001a, 0x001a,
	0x001a, 0xa1a8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xa5a8, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xa7a8, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xa9a8, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xbda8, 0x001a, 0x001a, 0x001a,
	0xbea8, 0x001a, 0x001a, 0x001a, 0x001a, 0xada8, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xb1a8, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa3a8, 0x001a,
	0xaba8, 0x001a, 0xafa8, 0x001a, 0xb3a8, 0x001a, 0xb5a8, 0x001a,
	0xb6a8, 0x001a, 0xb7a8, 0x001a, 0xb8a8, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xbba8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xc0a8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa6a1,
	0x001a, 0xa5a1, 0x40a8, 0x41a8, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x42a8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xa1a6, 0xa2a6, 0xa3a6, 0xa4a6, 0xa5a6, 0xa6a6, 0xa7a6,
	0xa8a6, 0xa9a6, 0xaaa6, 0xaba6, 0xaca6, 0xada6, 0xaea6, 0xafa6,
	0xb0a6, 0xb1a6, 0x001a, 0xb2a6, 0xb3a6, 0xb4a6, 0xb5a6, 0xb6a6,
	0xb7a6, 0xb8a6, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xc1a6, 0xc2a6, 0xc3a6, 0xc4a6, 0xc5a6, 0xc6a6, 0xc7a6,
	0xc8a6, 0xc9a6, 0xcaa6, 0xcba6, 0xcca6, 0xcda6, 0xcea6, 0xcfa6,
	0xd0a6, 0xd1a6, 0x001a, 0xd2a6, 0xd3a6, 0xd4a6, 0xd5a6, 0xd6a6,
	0xd7a6, 0xd8a6, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xa7a7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xa1a7, 0xa2a7, 0xa3a7, 0xa4a7, 0xa5a7, 0xa6a7, 0xa8a7, 0xa9a7,
	0xaaa7, 0xaba7, 0xaca7, 0xada7, 0xaea7, 0xafa7, 0xb0a7, 0xb1a7,
	0xb2a7, 0xb3a7, 0xb4a7, 0xb5a7, 0xb6a7, 0xb7a7, 0xb8a7, 0xb9a7,
	0xbaa7, 0xbba7, 0xbca7, 0xbda7, 0xbea7, 0xbfa7, 0xc0a7, 0xc1a7,
	0xd1a7, 0xd2a7, 0xd3a7, 0xd4a7, 0xd5a7, 0xd6a7, 0xd8a7, 0xd9a7,
	0xdaa7, 0xdba7, 0xdca7, 0xdda7, 0xdea7, 0xdfa7, 0xe0a7, 0xe1a7,
	0xe2a7, 0xe3a7, 0xe4a7, 0xe5a7, 0xe6a7, 0xe7a7, 0xe8a7, 0xe9a7,
	0xeaa7, 0xeba7, 0xeca7, 0xeda7, 0xeea7, 0xefa7, 0xf0a7, 0xf1a7,
	0x001a, 0xd7a7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};

const uint16_t libuna_codepage_windows_936_unicode_to_byte_stream_base_0x2000[ 832 ] = {
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x5ca9, 0x001a, 0x001a, 0x43a8, 0xaaa1, 0x44a8, 0xaca1, 0x001a,
	0xaea1, 0xafa1, 0x001a, 0x001a, 0xb0a1, 0xb1a1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x45a8, 0xada1, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xeba1, 0x001a, 0xe4a1, 0xe5a1, 0x001a, 0x46a8, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xf9a1, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x0080, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xe6a1, 0x001a, 0x47a8, 0x001a, 0x001a,
	0x001a, 0x48a8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xeda1, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x59a9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xf1a2, 0xf2a2, 0xf3a2, 0xf4a2, 0xf5a2, 0xf6a2, 0xf7a2, 0xf8a2,
	0xf9a2, 0xfaa2, 0xfba2, 0xfca2, 0x001a, 0x001a, 0x001a, 0x001a,
	0xa1a2, 0xa2a2, 0xa3a2, 0xa4a2, 0xa5a2, 0xa6a2, 0xa7a2, 0xa8a2,
	0xa9a2, 0xaaa2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xfba1, 0xfca1, 0xfaa1, 0xfda1, 0x001a, 0x001a, 0x49a8, 0x4aa8,
	0x4ba8, 0x4ca8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xcaa1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc7a1,
	0x001a, 0xc6a1, 0x001a, 0x001a, 0x001a, 0x4da8, 0x001a, 0x001a,
	0x001a, 0x001a, 0xcca1, 0x001a, 0x001a, 0xd8a1, 0xdea1, 0x4ea8,
	0xcfa1, 0x001a, 0x001a, 0x4fa8, 0x001a, 0xcea1, 0x001a, 0xc4a1,
	0xc5a1, 0xc9a1, 0xc8a1, 0xd2a1, 0x001a, 0x001a, 0xd3a1, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xe0a1, 0xdfa1, 0xc3a1, 0xcba1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd7a1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xd6a1, 0x001a, 0x001a, 0x001a, 0xd5a1, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x50a8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xd9a1, 0xd4a1, 0x001a, 0x001a, 0xdca1, 0xdda1, 0x51a8, 0x52a8,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xdaa1, 0xdba1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x92a8, 0x001a, 0x001a,
	0x001a, 0xd1a1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xcda1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x53a8,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xd0a1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};

const uint16_t libuna_codepage_windows_936_unicode_to_byte_stream_base_0x2440[ 576 ] = {
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xd9a2, 0xdaa2, 0xdba2, 0xdca2, 0xdda2, 0xdea2, 0xdfa2, 0xe0a2,
	0xe1a2, 0xe2a2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xc5a2, 0xc6a2, 0xc7a2, 0xc8a2,
	0xc9a2, 0xcaa2, 0xcba2, 0xcca2, 0xcda2, 0xcea2, 0xcfa2, 0xd0a2,
	0xd1a2, 0xd2a2, 0xd3a2, 0xd4a2, 0xd5a2, 0xd6a2, 0xd7a2, 0xd8a2,
	0xb1a2, 0xb2a2, 0xb3a2, 0xb4a2, 0xb5a2, 0xb6a2, 0xb7a2, 0xb8a2,
	0xb9a2, 0xbaa2, 0xbba2, 0xbca2, 0xbda2, 0xbea2, 0xbfa2, 0xc0a2,
	0xc1a2, 0xc2a2, 0xc3a2, 0xc4a2, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xa4a9, 0xa5a9, 0xa6a9, 0xa7a9, 0xa8a9, 0xa9a9, 0xaaa9, 0xaba9,
	0xaca9, 0xada9, 0xaea9, 0xafa9, 0xb0a9, 0xb1a9, 0xb2a9, 0xb3a9,
	0xb4a9, 0xb5a9, 0xb6a9, 0xb7a9, 0xb8a9, 0xb9a9, 0xbaa9, 0xbba9,
	0xbca9, 0xbda9, 0xbea9, 0xbfa9, 0xc0a9, 0xc1a9, 0xc2a9, 0xc3a9,
	0xc4a9, 0xc5a9, 0xc6a9, 0xc7a9, 0xc8a9, 0xc9a9, 0xcaa9, 0xcba9,
	0xcca9, 0xcda9, 0xcea9, 0xcfa9, 0xd0a9, 0xd1a9, 0xd2a9, 0xd3a9,
	0xd4a9, 0xd5a9, 0xd6a9, 0xd7a9, 0xd8a9, 0xd9a9, 0xdaa9, 0xdba9,
	0xdca9, 0xdda9, 0xdea9, 0xdfa9, 0xe0a9, 0xe1a9, 0xe2a9, 0xe3a9,
	0xe4a9, 0xe5a9, 0xe6a9, 0xe7a9, 0xe8a9, 0xe9a9, 0xeaa9, 0xeba9,
	0xeca9, 0xeda9, 0xeea9, 0xefa9, 0x001a, 0x001a, 0x001a, 0x001a,
	0x54a8, 0x55a8, 0x56a8, 0x57a8, 0x58a8, 0x59a8, 0x5aa8, 0x5ba8,
	0x5ca8, 0x5da8, 0x5ea8, 0x5fa8, 0x60a8, 0x61a8, 0x62a8, 0x63a8,
	0x64a8, 0x65a8, 0x66a8, 0x67a8, 0x68a8, 0x69a8, 0x6aa8, 0x6ba8,
	0x6ca8, 0x6da8, 0x6ea8, 0x6fa8, 0x70a8, 0x71a8, 0x72a8, 0x73a8,
	0x74a8, 0x75a8, 0x76a8, 0x77a8, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x78a8, 0x79a8, 0x7aa8, 0x7ba8, 0x7ca8, 0x7da8, 0x7ea8,
	0x80a8, 0x81a8, 0x82a8, 0x83a8, 0x84a8, 0x85a8, 0x86a8, 0x87a8,
	0x001a, 0x001a, 0x001a, 0x88a8, 0x89a8, 0x8aa8, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xf6a1, 0xf5a1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xf8a1, 0xf7a1, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x8ba8, 0x8ca8, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf4a1, 0xf3a1,
	0x001a, 0x001a, 0x001a, 0xf0a1, 0x001a, 0x001a, 0xf2a1, 0xf1a1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x8da8, 0x8ea8, 0x8fa8, 0x90a8, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xefa1, 0xeea1, 0x001a,
	0x001a, 0x91a8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xe2a1, 0x001a, 0xe1a1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};

const uint16_t libuna_codepage_windows_936_unicode_to_byte_stream_base_0x3000[ 1024 ] = {
	0xa1a1, 0xa2a1, 0xa3a1, 0xa8a1, 0x001a, 0xa9a1, 0x65a9, 0x96a9,
	0xb4a1, 0xb5a1, 0xb6a1, 0xb7a1, 0xb8a1, 0xb9a1, 0xbaa1, 0xbba1,
	0xbea1, 0xbfa1, 0x93a8, 0xfea1, 0xb2a1, 0xb3a1, 0xbca1, 0xbda1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x94a8, 0x95a8, 0x001a,
	0x001a, 0x40a9, 0x41a9, 0x42a9, 0x43a9, 0x44a9, 0x45a9, 0x46a9,
	0x47a9, 0x48a9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xa1a4, 0xa2a4, 0xa3a4, 0xa4a4, 0xa5a4, 0xa6a4, 0xa7a4,
	0xa8a4, 0xa9a4, 0xaaa4, 0xaba4, 0xaca4, 0xada4, 0xaea4, 0xafa4,
	0xb0a4, 0xb1a4, 0xb2a4, 0xb3a4, 0xb4a4, 0xb5a4, 0xb6a4, 0xb7a4,
	0xb8a4, 0xb9a4, 0xbaa4, 0xbba4, 0xbca4, 0xbda4, 0xbea4, 0xbfa4,
	0xc0a4, 0xc1a4, 0xc2a4, 0xc3a4, 0xc4a4, 0xc5a4, 0xc6a4, 0xc7a4,
	0xc8a4, 0xc9a4, 0xcaa4, 0xcba4, 0xcca4, 0xcda4, 0xcea4, 0xcfa4,
	0xd0a4, 0xd1a4, 0xd2a4, 0xd3a4, 0xd4a4, 0xd5a4, 0xd6a4, 0xd7a4,
	0xd8a4, 0xd9a4, 0xdaa4, 0xdba4, 0xdca4, 0xdda4, 0xdea4, 0xdfa4,
	0xe0a4, 0xe1a4, 0xe2a4, 0xe3a4, 0xe4a4, 0xe5a4, 0xe6a4, 0xe7a4,
	0xe8a4, 0xe9a4, 0xeaa4, 0xeba4, 0xeca4, 0xeda4, 0xeea4, 0xefa4,
	0xf0a4, 0xf1a4, 0xf2a4, 0xf3a4, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x61a9, 0x62a9, 0x66a9, 0x67a9, 0x001a,
	0x001a, 0xa1a5, 0xa2a5, 0xa3a5, 0xa4a5, 0xa5a5, 0xa6a5, 0xa7a5,
	0xa8a5, 0xa9a5, 0xaaa5, 0xaba5, 0xaca5, 0xada5, 0xaea5, 0xafa5,
	0xb0a5, 0xb1a5, 0xb2a5, 0xb3a5, 0xb4a5, 0xb5a5, 0xb6a5, 0xb7a5,
	0xb8a5, 0xb9a5, 0xbaa5, 0xbba5, 0xbca5, 0xbda5, 0xbea5, 0xbfa5,
	0xc0a5, 0xc1a5, 0xc2a5, 0xc3a5, 0xc4a5, 0xc5a5, 0xc6a5, 0xc7a5,
	0xc8a5, 0xc9a5, 0xcaa5, 0xcba5, 0xcca5, 0xcda5, 0xcea5, 0xcfa5,
	0xd0a5, 0xd1a5, 0xd2a5, 0xd3a5, 0xd4a5, 0xd5a5, 0xd6a5, 0xd7a5,
	0xd8a5, 0xd9a5, 0xdaa5, 0xdba5, 0xdca5, 0xdda5, 0xdea5, 0xdfa5,
	0xe0a5, 0xe1a5, 0xe2a5, 0xe3a5, 0xe4a5, 0xe5a5, 0xe6a5, 0xe7a5,
	0xe8a5, 0xe9a5, 0xeaa5, 0xeba5, 0xeca5, 0xeda5, 0xeea5, 0xefa5,
	0xf0a5, 0xf1a5, 0xf2a5, 0xf3a5, 0xf4a5, 0xf5a5, 0xf6a5, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x60a9, 0x63a9, 0x64a9, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc5a8, 0xc6a8, 0xc7a8,
	0xc8a8, 0xc9a8, 0xcaa8, 0xcba8, 0xcca8, 0xcda8, 0xcea8, 0xcfa8,
	0xd0a8, 0xd1a8, 0xd2a8, 0xd3a8, 0xd4a8, 0xd5a8, 0xd6a8, 0xd7a8,
	0xd8a8, 0xd9a8, 0xdaa8, 0xdba8, 0xdca8, 0xdda8, 0xdea8, 0xdfa8,
	0xe0a8, 0xe1a8, 0xe2a8, 0xe3a8, 0xe4a8, 0xe5a8, 0xe6a8, 0xe7a8,
	0xe8a8, 0xe9a8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xe5a2, 0xe6a2, 0xe7a2, 0xe8a2, 0xe9a2, 0xeaa2, 0xeba2, 0xeca2,
	0xeda2, 0xeea2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x5aa9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x49a9, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4aa9, 0x4ba9,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x4ca9, 0x4da9, 0x4ea9, 0x001a,
	0x001a, 0x4fa9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x50a9, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x51a9, 0x001a,
	0x001a, 0x52a9, 0x53a9, 0x001a, 0x001a, 0x54a9, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};

const uint16_t libuna_codepage_windows_936_unicode_to_byte_stream_base_0x4e00[ 20902 ] = {
	0xbbd2, 0xa1b6, 0x4081, 0xdfc6, 0x4181, 0x4281, 0x4381, 0xf2cd,
	0xc9d5, 0xfdc8, 0xcfc9, 0xc2cf, 0xa2d8, 0xbbb2, 0xebd3, 0x4481,
	0xa4d8, 0xf3b3, 0x4581, 0xa8d7, 0xd2c7, 0xa7d8, 0xc0ca, 0x4681,
	0xf0c7, 0xfbb1, 0xb5d2, 0xd4b4, 0xabb6, 0xbfcb, 0xa9d8, 0x4781,
	0x4881, 0x4981, 0xaab6, 0x4a81, 0xbdc1, 0xcfd1, 0x4b81, 0xa5c9,
	0xadd8, 0x4c81, 0xf6b8, 0xbed1, 0xdce3, 0xd0d6, 0x4d81, 0x4e81,
	0xe1b7, 0x4f81, 0xaeb4, 0x5081, 0xd9c1, 0x5181, 0xbcd8, 0x5281,
	0xe8cd, 0xa4b5, 0xaace, 0xf7d6, 0x5381, 0xf6c0, 0xd9be, 0xafd8,
	0x5481, 0x5581, 0x5681, 0xcbc4, 0x5781, 0xc3be, 0x5881, 0xb1d8,
	0xb4c3, 0xe5d2, 0x5981, 0xaed6, 0xdace, 0xa7d5, 0xf5ba, 0xa6b7,
	0xd6c0, 0x5a81, 0xb9c6, 0xd2c5, 0xc7c7, 0x5b81, 0xd4b9, 0x5c81,
	0xcbb3, 0xd2d2, 0x5d81, 0x5e81, 0xbfd8, 0xc5be, 0xf2c6, 0xb2d2,
	0xb0cf, 0xe7cf, 0x5f81, 0x6081, 0x6181, 0x6281, 0xe9ca, 0x6381,
	0x6481, 0xc0d8, 0x6581, 0x6681, 0x6781, 0x6881, 0x6981, 0x6a81,
	0xf2c2, 0xd2c2, 0x6b81, 0xe9c8, 0x6c81, 0x6d81, 0x6e81, 0x6f81,
	0x7081, 0x7181, 0x7281, 0x7381, 0x7481, 0x7581, 0xacc7, 0x7681,
	0x7781, 0x7881, 0x7981, 0x7a81, 0x7b81, 0x7c81, 0xcbc1, 0x7d81,
	0xe8d3, 0xf9d5, 0x7e81, 0xc2ca, 0xfeb6, 0xa1d8, 0xdad3, 0xf7bf,
	0x8081, 0xc6d4, 0xa5bb, 0xc1d8, 0xe5ce, 0xaebe, 0x8181, 0x8281,
	0xa8d8, 0x8381, 0xc7d1, 0xa9d0, 0x8481, 0x8581, 0x8681, 0xbdd8,
	0xefd9, 0xf6cd, 0xbabf, 0x8781, 0xbbbd, 0xa5ba, 0xe0d2, 0xfab2,
	0xe0ba, 0xb6c4, 0x8881, 0xedcf, 0xa9be, 0xa4cd, 0xc1c1, 0x8981,
	0x8a81, 0x8b81, 0xd7c7, 0xf1d9, 0x8c81, 0xf4d9, 0x8d81, 0x8e81,
	0x8f81, 0x9081, 0xcbc8, 0xe9d8, 0x9181, 0x9281, 0x9381, 0xdad2,
	0xb2ca, 0xcac8, 0xecd8, 0xead8, 0xc6d8, 0xf6bd, 0xcdc6, 0xf0b3,
	0x9481, 0xebd8, 0xf1bd, 0xe9bd, 0x9581, 0xd4c8, 0xd3b4, 0x9681,
	0x9781, 0xd8c2, 0x9881, 0xd6b2, 0xd0d7, 0xcbca, 0xfbcb, 0xccd5,
	0xb6b8, 0xc9cf, 0x9981, 0x9a81, 0x9b81, 0xdad9, 0xf0d8, 0xaac7,
	0x9c81, 0xeed8, 0x9d81, 0xfab4, 0xeec1, 0xd4d2, 0x9e81, 0x9f81,
	0xedd8, 0xa081, 0xc7d2, 0xefd8, 0xc7c3, 0xa181, 0xa281, 0xa381,
	0xf6d1, 0xa481, 0xd9d6, 0xf2d8, 0xa581, 0xf5d8, 0xfebc, 0xdbbc,
	0xa681, 0xa781, 0xa881, 0xcec8, 0xa981, 0xddb7, 0xaa81, 0xc2b7,
	0xab81, 0xf3c6, 0xac81, 0xad81, 0xae81, 0xaf81, 0xb081, 0xb181,
	0xb281, 0xf8d8, 0xc1d2, 0xb381, 0xb481, 0xe9ce, 0xbfbc, 0xfcb7,
	0xa5b7, 0xddd0, 0xb581, 0xb681, 0xb781, 0xb881, 0xb981, 0xdad6,
	0xc5d3, 0xefbb, 0xe1bb, 0xf1d8, 0xba81, 0xbb81, 0xa1c9, 0xb0ce,
	0xabb4, 0xbc81, 0xf3d8, 0xbd81, 0xcbc9, 0xf6d8, 0xd7c2, 0xf7d8,
	0xbe81, 0xbf81, 0xb1ce, 0xf9d8, 0xc081, 0xc181, 0xc281, 0xaeb2,
	0xc0b9, 0xc381, 0xa3d9, 0xc481, 0xe9b0, 0xc581, 0xe6c1, 0xc681,
	0xecc9, 0xc781, 0xc5cb, 0xc881, 0xc6cb, 0xa4d9, 0xc981, 0xca81,
	0xcb81, 0xcc81, 0xcd81, 0xe8b5, 0xce81, 0xcf81, 0xabb5, 0xd081,
	0xd181, 0xd281, 0xd381, 0xd481, 0xd581, 0xbbce, 0xcdb5, 0xa1d7,
	0xf4d7, 0xd3d3, 0xd681, 0xe5cc, 0xd781, 0xceba, 0xd881, 0xa2d9,
	0xdcd9, 0xe0d3, 0xfdd8, 0xf0b7, 0xf7d7, 0xfed8, 0xfad8, 0xa1d9,
	0xe3c4, 0xd981, 0xda81, 0xb6d3, 0xf4d8, 0xddd9, 0xdb81, 0xfbd8,
	0xdc81, 0xe5c5, 0xdd81, 0xde81, 0xd0c0, 0xdf81, 0xe081, 0xf0d1,
	0xdbb0, 0xe181, 0xe281, 0xd1bc, 0xa6d9, 0xe381, 0xa5d9, 0xe481,
	0xe581, 0xe681, 0xe781, 0xacd9, 0xaed9, 0xe881, 0xabd9, 0xb9ca,
	0xe981, 0xea81, 0xeb81, 0xa9d9, 0xb6d6, 0xec81, 0xed81, 0xee81,
	0xdeb3, 0xa8d9, 0xef81, 0xfdc0, 0xf081, 0xccca, 0xf181, 0xaad9,
	0xf281, 0xa7d9, 0xf381, 0xf481, 0xb0d9, 0xf581, 0xf681, 0xb1b6,
	0xf781, 0xf881, 0xf981, 0xa9b9, 0xfa81, 0xc0d2, 0xfb81, 0xfc81,
	0xc0cf, 0xfd81, 0xfe81, 0xc2c2, 0x4082, 0xc4bd, 0xecd5, 0xe0b2,
	0xc8c7, 0xebbf, 0xadd9, 0x4182, 0xafd9, 0x4282, 0xeace, 0xeeba,
	0x4382, 0x4482, 0x4582, 0x4682, 0x4782, 0xd6c7, 0x4882, 0x4982,
	0x4a82, 0x4b82, 0x4c82, 0x4d82, 0x4e82, 0x4f82, 0x5082, 0xe3b1,
	0x5182, 0x5282, 0x5382, 0xd9b4, 0xedb6, 0xb4d9, 0x5482, 0x5582,
	0x5682, 0x5782, 0xa1bf, 0x5882, 0x5982, 0x5a82, 0xded9, 0xcec7,
	0xfec0, 0xb8d9, 0x5b82, 0x5c82, 0x5d82, 0x5e82, 0x5f82, 0xd7cb,
	0xfdb7, 0x6082, 0xb5d9, 0x6182, 0xb7d9, 0xa3b1, 0xe1d3, 0xb9d9,
	0x6282, 0xc5d0, 0x6382, 0xb6d9, 0x6482, 0x6582, 0xb1d9, 0x6682,
	0xb2d9, 0xa9c1, 0xb3d9, 0x6782, 0x6882, 0xf3bc, 0xded0, 0xa9b8,
	0x6982, 0xe3be, 0x6a82, 0xbdd9, 0x6b82, 0x6c82, 0x6d82, 0x6e82,
	0xbad9, 0x6f82, 0xb3b0, 0x7082, 0x7182, 0x7282, 0xc2d9, 0x7382,
	0x7482, 0x7582, 0x7682, 0x7782, 0x7882, 0x7982, 0x7a82, 0x7b82,
	0x7c82, 0x7d82, 0x7e82, 0x8082, 0xc4d9, 0xb6b1, 0x8182, 0xbfd9,
	0x8282, 0x8382, 0xb9b5, 0x8482, 0xf3be, 0x8582, 0x8682, 0x8782,
	0xc8cc, 0xf2ba, 0xd0d2, 0x8882, 0xc3d9, 0x8982, 0x8a82, 0xe8bd,
	0x8b82, 0xabb3, 0x8c82, 0x8d82, 0x8e82, 0xc5d9, 0xebbe, 0x8f82,
	0xc6d9, 0xbbd9, 0xdfc4, 0x9082, 0xbed9, 0xc1d9, 0xc0d9, 0x9182,
	0x9282, 0x9382, 0x9482, 0x9582, 0x9682, 0x9782, 0x9882, 0x9982,
	0x9a82, 0x9b82, 0xaed5, 0x9c82, 0xb5d6, 0x9d82, 0xe3c7, 0x9e82,
	0x9f82, 0xa082, 0xa182, 0xc8d9, 0xa282, 0xa382, 0xa482, 0xd9bc,
	0xcad9, 0xa582, 0xa682, 0xa782, 0xbcd9, 0xa882, 0xcbd9, 0xabc6,
	0xa982, 0xaa82, 0xab82, 0xac82, 0xad82, 0xc9d9, 0xae82, 0xaf82,
	0xb082, 0xb182, 0xf6d7, 0xb282, 0xa3cd, 0xb382, 0xb482, 0xb582,
	0xb682, 0xb782, 0xb882, 0xb982, 0xba82, 0xa1bd, 0xbb82, 0xbc82,
	0xbd82, 0xbe82, 0xbf82, 0xc082, 0xccd9, 0xc182, 0xc282, 0xc382,
	0xc482, 0xc582, 0xc682, 0xc782, 0xc882, 0xc982, 0xbcc5, 0xb5cd,
	0xca82, 0xcb82, 0xcc82, 0xcdd9, 0xcd82, 0xce82, 0xc7d9, 0xa5b3,
	0xfebf, 0xcf82, 0xd082, 0xd182, 0xd282, 0xb5b8, 0xd382, 0xd482,
	0xfcc0, 0xd582, 0xd682, 0xd782, 0xd882, 0xf8b0, 0xd982, 0xda82,
	0xdb82, 0xdc82, 0xdd82, 0xde82, 0xdf82, 0xe082, 0xe182, 0xe282,
	0xe382, 0xe482, 0xe582, 0xe682, 0xe782, 0xe882, 0xe982, 0xea82,
	0xeb82, 0xec82, 0xed82, 0xf6b4, 0xee82, 0xced9, 0xef82, 0xcfd9,
	0xa2b4, 0xd0d9, 0xf082, 0xf182, 0xdfb4, 0xf282, 0xf382, 0xf482,
	0xf582, 0xf682, 0xc1b0, 0xf782, 0xf882, 0xf982, 0xfa82, 0xfb82,
	0xfc82, 0xfd82, 0xd1d9, 0xb5c9, 0xfe82, 0x4083, 0x4183, 0x4283,
	0x4383, 0x4483, 0x4583, 0x4683, 0x4783, 0x4883, 0x4983, 0x4a83,
	0x4b83, 0x4c83, 0x4d83, 0x4e83, 0x4f83, 0x5083, 0x5183, 0xf1cf,
	0x5283, 0x5383, 0x5483, 0x5583, 0x5683, 0x5783, 0xd2d9, 0x5883,
	0x5983, 0x5a83, 0xc5c1, 0x5b83, 0x5c83, 0x5d83, 0x5e83, 0x5f83,
	0x6083, 0x6183, 0x6283, 0x6383, 0x6483, 0x6583, 0xd6d9, 0xaec9,
	0x6683, 0x6783, 0x6883, 0x6983, 0xd5d9, 0xd4d9, 0xd7d9, 0x6a83,
	0x6b83, 0x6c83, 0x6d83, 0xdbcb, 0x6e83, 0xa9bd, 0x6f83, 0x7083,
	0x7183, 0x7283, 0x7383, 0xa7c6, 0x7483, 0x7583, 0x7683, 0x7783,
	0x7883, 0x7983, 0x7a83, 0x7b83, 0x7c83, 0x7d83, 0xd3d9, 0xd8d9,
	0x7e83, 0x8083, 0x8183, 0xd9d9, 0x8283, 0x8383, 0x8483, 0x8583,
	0x8683, 0x8783, 0xe5c8, 0x8883, 0x8983, 0x8a83, 0x8b83, 0x8c83,
	0x8d83, 0x8e83, 0x8f83, 0x9083, 0x9183, 0x9283, 0x9383, 0x9483,
	0x9583, 0xdcc0, 0x9683, 0x9783, 0x9883, 0x9983, 0x9a83, 0x9b83,
	0x9c83, 0x9d83, 0x9e83, 0x9f83, 0xa083, 0xa183, 0xa283, 0xa383,
	0xa483, 0xa583, 0xa683, 0xa783, 0xa883, 0xa983, 0xaa83, 0xab83,
	0xac83, 0xad83, 0xae83, 0xaf83, 0xb083, 0xb183, 0xb283, 0xf9b6,
	0xa3d8, 0xcad4, 0xb383, 0xaad4, 0xd6d0, 0xe4b3, 0xd7d5, 0xb483,
	0xc8cf, 0xe2b9, 0xb583, 0xcbbf, 0xb683, 0xe2c3, 0xb783, 0xb883,
	0xb983, 0xd2b6, 0xba83, 0xbb83, 0xc3cd, 0xeed9, 0xf0d9, 0xbc83,
	0xbd83, 0xbe83, 0xb3b5, 0xbf83, 0xb5b6, 0xc083, 0xc183, 0xc283,
	0xc383, 0xc483, 0xa4be, 0xc583, 0xc683, 0xebc8, 0xc783, 0xc883,
	0xabc8, 0xc983, 0xca83, 0xcbb0, 0xabb9, 0xf9c1, 0xe2d9, 0xcb83,
	0xbcc0, 0xb2b9, 0xcc83, 0xd8b9, 0xcbd0, 0xf8b1, 0xe4c6, 0xdfbe,
	0xe4b5, 0xc8d7, 0xcd83, 0xf8d1, 0xe6bc, 0xdeca, 0xce83, 0xcf83,
	0xbdbc, 0xe6d9, 0xe7d8, 0xd083, 0xd183, 0xdac4, 0xd283, 0xd383,
	0xd4b8, 0xbdc8, 0xd483, 0xd583, 0xe1b2, 0xd9d4, 0xd683, 0xd783,
	0xd883, 0xd983, 0xb0c3, 0xda83, 0xdb83, 0xe1c3, 0xa2da, 0xdfc8,
	0xdc83, 0xb4d0, 0xdd83, 0xfcbe, 0xa9c5, 0xde83, 0xdf83, 0xe083,
	0xdab9, 0xe183, 0xa3da, 0xe283, 0xa9d4, 0xa4da, 0xe383, 0xe483,
	0xe583, 0xe683, 0xe783, 0xfbd9, 0xacb6, 0xe883, 0xe983, 0xebb7,
	0xf9b1, 0xfcd9, 0xe5b3, 0xf6be, 0xea83, 0xf6bf, 0xb1d2, 0xe4c0,
	0xeb83, 0xec83, 0xed83, 0xb3b6, 0xfed9, 0xfdd9, 0xee83, 0xef83,
	0xbbbe, 0xf083, 0xf183, 0xf283, 0xe0c6, 0xf383, 0xbcd7, 0xa1da,
	0xf483, 0xb9c1, 0xf583, 0xf2b5, 0xe8c1, 0xf683, 0xf783, 0xf5bc,
	0xf883, 0xd5b4, 0xf983, 0xfa83, 0xfb83, 0xfc83, 0xfd83, 0xfe83,
	0x4084, 0x4184, 0x4284, 0xddc1, 0x4384, 0xfdc4, 0x4484, 0x4584,
	0xb8bc, 0xb2b7, 0x4684, 0x4784, 0xefb7, 0x4884, 0x4984, 0x4a84,
	0x4b84, 0x4c84, 0x4d84, 0xecd9, 0x4e84, 0xbec6, 0x4f84, 0xadbf,
	0xcbbb, 0x5084, 0x5184, 0xcab5, 0x5284, 0xc9db, 0xd7d0, 0x5384,
	0xb9cd, 0xbcb0, 0xf6b3, 0xf7bb, 0xcadb, 0xafba, 0x5484, 0xe4d4,
	0xb6b5, 0xf3b5, 0xd6d8, 0xd0c8, 0x5584, 0x5684, 0xd6b7, 0xd0c7,
	0xd7d8, 0x5784, 0xafbf, 0x5884, 0x5984, 0xbbdb, 0xd8d8, 0x5a84,
	0x5b84, 0xccd0, 0xaebb, 0x5c84, 0x5d84, 0x5e84, 0xbeeb, 0xd0c1,
	0xf5c1, 0xf2d4, 0xd5b8, 0xb4b4, 0x5f84, 0xf5b3, 0x6084, 0x6184,
	0xbec9, 0x6284, 0x6384, 0x6484, 0xd0c5, 0x6584, 0x6684, 0x6784,
	0xd9c5, 0xfbc0, 0x6884, 0xf0b1, 0x6984, 0xd9d8, 0xceb9, 0x6a84,
	0xbdb5, 0x6b84, 0x6c84, 0xdad8, 0x6d84, 0x6e84, 0xc6d6, 0xa2cb,
	0xafc8, 0xb2c9, 0xccb4, 0xccbf, 0x6f84, 0xf4b9, 0x7084, 0xdbd8,
	0xdcd8, 0xe7b6, 0xc1bc, 0xeacc, 0x7184, 0x7284, 0x7384, 0x7484,
	0x7584, 0x7684, 0xf7cf, 0x7784, 0xddd8, 0xb0c7, 0x7884, 0x7984,
	0xd0b9, 0xa3bd, 0x7a84, 0x7b84, 0xdecc, 0x7c84, 0xcac6, 0x7d84,
	0x7e84, 0x8084, 0x8184, 0x8284, 0xe0d8, 0x8384, 0xded8, 0x8484,
	0x8584, 0xdfd8, 0x8684, 0x8784, 0x8884, 0xfeb0, 0x8984, 0xe7be,
	0x8a84, 0xa3ca, 0xf4bc, 0x8b84, 0x8c84, 0x8d84, 0x8e84, 0xb1b8,
	0x8f84, 0x9084, 0xeeb8, 0x9184, 0x9284, 0x9384, 0x9484, 0x9584,
	0x9684, 0x9784, 0x9884, 0x9984, 0x9a84, 0xe2d8, 0x9b84, 0xcbbd,
	0x9c84, 0xe4d8, 0xe3d8, 0x9d84, 0x9e84, 0x9f84, 0xa084, 0xa184,
	0xfcc5, 0xa284, 0xa384, 0xa484, 0xa584, 0xa684, 0xa784, 0xa884,
	0xe5d8, 0xa984, 0xaa84, 0xe6d8, 0xab84, 0xac84, 0xad84, 0xae84,
	0xaf84, 0xb084, 0xb184, 0xa6c1, 0xb284, 0xb0c8, 0xecb0, 0xa6b9,
	0xd3bc, 0xf1ce, 0xbddb, 0xd3c1, 0xb384, 0xb484, 0xb584, 0xb684,
	0xafb6, 0xfad6, 0xacc5, 0xd9bd, 0xbedb, 0xbfdb, 0xb784, 0xb884,
	0xb984, 0xf8c0, 0xa2be, 0xcdc0, 0xba84, 0xbb84, 0xbc84, 0xbd84,
	0xbe84, 0xbf84, 0xc084, 0xc184, 0xc284, 0xc384, 0xc0db, 0xc6ca,
	0xc484, 0xc584, 0xc684, 0xaab2, 0xc784, 0xc884, 0xc984, 0xc2d3,
	0xca84, 0xe3c3, 0xcb84, 0xabd1, 0xcc84, 0xcd84, 0xce84, 0xcf84,
	0xc2db, 0xd084, 0xd5c0, 0xd184, 0xd284, 0xd384, 0xc3db, 0xd484,
	0xb1bf, 0xd584, 0xd684, 0xd784, 0xd884, 0xd984, 0xda84, 0xbcc4,
	0xdb84, 0xdc84, 0xdd84, 0xde84, 0xdac7, 0xdf84, 0xe084, 0xe184,
	0xe284, 0xe384, 0xe484, 0xe584, 0xe684, 0xe784, 0xe884, 0xe984,
	0xc4db, 0xea84, 0xeb84, 0xec84, 0xed84, 0xee84, 0xef84, 0xf084,
	0xf184, 0xe8d9, 0xd7c9, 0xf284, 0xf384, 0xf484, 0xb4b9, 0xf0ce,
	0xc8d4, 0xf584, 0xf684, 0xf784, 0xf884, 0xfcb0, 0xd2b4, 0xf984,
	0xd9d0, 0xfa84, 0xfb84, 0xfc84, 0xfd84, 0xe9d9, 0xfe84, 0xcbde,
	0xebd9, 0x4085, 0x4185, 0x4285, 0x4385, 0xb0d8, 0xafbb, 0xb1b1,
	0x4485, 0xd7b3, 0xced8, 0x4585, 0x4685, 0xd1d4, 0x4785, 0x4885,
	0xb3bd, 0xefbf, 0x4985, 0xbbcf, 0x4a85, 0x4b85, 0xd0d8, 0x4c85,
	0x4d85, 0x4e85, 0xcbb7, 0x4f85, 0x5085, 0x5185, 0xd1d8, 0x5285,
	0x5385, 0x5485, 0x5585, 0x5685, 0x5785, 0x5885, 0x5985, 0x5a85,
	0x5b85, 0xa5c6, 0xf8c7, 0xbdd2, 0x5c85, 0x5d85, 0xd2d8, 0xe4c4,
	0x5e85, 0xaeca, 0x5f85, 0xa7c7, 0x6085, 0xa6d8, 0x6185, 0xfdc9,
	0xe7ce, 0xdcbb, 0xebb0, 0x6285, 0x6385, 0x6485, 0xaabb, 0xadd0,
	0x6585, 0xb0b1, 0xe4d7, 0xbfd7, 0x6685, 0xa5b5, 0xf4c2, 0xcfc4,
	0x6785, 0x6885, 0xa9b2, 0x6985, 0xb7b2, 0x6a85, 0xe5b1, 0xb2df,
	0xbcd5, 0xa8bf, 0xacc2, 0xd5d8, 0xb1c2, 0x6b85, 0xd4d8, 0xd4ce,
	0x6c85, 0xe0da, 0x6d85, 0xc0ce, 0x6e85, 0x6f85, 0xb4d8, 0xaec3,
	0xa1d3, 0xa3ce, 0x7085, 0xb4bc, 0xb4c8, 0xd1c2, 0x7185, 0xedbe,
	0xb6d0, 0x7285, 0xe1da, 0x7385, 0x7485, 0x7585, 0x7685, 0xe4c7,
	0x7785, 0x7885, 0xa7b3, 0x7985, 0xf2b6, 0xfccc, 0xfac0, 0x7a85,
	0x7b85, 0xf7c0, 0x7c85, 0xb9d1, 0xe1d1, 0xc7d8, 0x7d85, 0x7e85,
	0x8085, 0x8185, 0x8285, 0x8385, 0x8485, 0xdeb2, 0x8585, 0x8685,
	0xe5c0, 0x8785, 0xf1ba, 0x8885, 0x8985, 0xc8d8, 0x8a85, 0xadd4,
	0x8b85, 0x8c85, 0xe1cf, 0xc9d8, 0x8d85, 0xcad8, 0xc3cf, 0x8e85,
	0xf8b3, 0xc7be, 0x8f85, 0x9085, 0x9185, 0x9285, 0xcbd8, 0x9385,
	0x9485, 0x9585, 0x9685, 0x9785, 0x9885, 0x9985, 0xccdb, 0x9a85,
	0x9b85, 0x9c85, 0x9d85, 0xa5c8, 0x9e85, 0x9f85, 0xa085, 0xd8cf,
	0xa185, 0xfec8, 0xceb2, 0xa285, 0xa385, 0xa485, 0xa585, 0xa685,
	0xd6d3, 0xe6b2, 0xb0bc, 0xd1d3, 0xabcb, 0xb4b7, 0xa785, 0xa885,
	0xa985, 0xa2b7, 0xaa85, 0xab85, 0xe5ca, 0xac85, 0xa1c8, 0xdcca,
	0xe4b1, 0xf0d0, 0xad85, 0xd1c5, 0xae85, 0xaf85, 0xb085, 0xc5db,
	0xfeb5, 0xb185, 0xb285, 0xdabf, 0xc5b9, 0xe4be, 0xedc1, 0xb385,
	0xb6df, 0xb5df, 0xbbd6, 0xd0bd, 0xd9d5, 0xc8b0, 0xa3b6, 0xc9bf,
	0xa8cc, 0xb3df, 0xb7ca, 0xd2d3, 0xb485, 0xcfd8, 0xb6d2, 0xc5ba,
	0xbecb, 0xbecc, 0xb585, 0xb7df, 0xf0b5, 0xb4df, 0xb685, 0xb785,
	0xb885, 0xf5d3, 0xb985, 0xd4b3, 0xf7b8, 0xba85, 0xbadf, 0xbb85,
	0xcfba, 0xaabc, 0xf5b5, 0xbc85, 0xaccd, 0xfbc3, 0xf3ba, 0xf4c0,
	0xc2cd, 0xf2cf, 0xb8df, 0xc5cf, 0xbd85, 0xc0c2, 0xb9df, 0xf0c2,
	0xbe85, 0xbf85, 0xc085, 0xfdbe, 0xc185, 0xdfc1, 0xcccd, 0xf7d2,
	0xcdb7, 0xc1df, 0xc285, 0xc4df, 0xc385, 0xc485, 0xf1b7, 0xc9b0,
	0xd6b6, 0xd4b7, 0xc585, 0xacba, 0xfdcc, 0xd4bf, 0xb1cb, 0xf4c6,
	0xc685, 0xa8d6, 0xc5df, 0xc785, 0xe2ce, 0xb3b3, 0xc885, 0xc985,
	0xfcce, 0xb5b4, 0xca85, 0xc7ce, 0xf0ba, 0xcb85, 0xe1ce, 0xcc85,
	0xbdd1, 0xcd85, 0xce85, 0xc0df, 0xcf85, 0xd085, 0xf4b4, 0xd185,
	0xcab3, 0xd285, 0xe6b8, 0xbbdf, 0xd385, 0xd485, 0xd585, 0xd685,
	0xc5c4, 0xd785, 0xbcdf, 0xbddf, 0xbedf, 0xbbc5, 0xbfdf, 0xc2df,
	0xb1d4, 0xc3df, 0xd885, 0xbac7, 0xd8ce, 0xd985, 0xda85, 0xdb85,
	0xdc85, 0xdd85, 0xd8c4, 0xde85, 0xcadf, 0xdf85, 0xcfdf, 0xe085,
	0xdcd6, 0xe185, 0xe285, 0xe385, 0xe485, 0xe585, 0xe685, 0xe785,
	0xe885, 0xc9df, 0xdadf, 0xb6ce, 0xe985, 0xc7ba, 0xcedf, 0xc8df,
	0xdec5, 0xea85, 0xeb85, 0xebc9, 0xf4ba, 0xfcc3, 0xec85, 0xed85,
	0xd7be, 0xee85, 0xc6df, 0xef85, 0xcddf, 0xf085, 0xd8c5, 0xf185,
	0xf285, 0xf385, 0xf485, 0xa6d5, 0xcdba, 0xf585, 0xccbe, 0xbdd3,
	0xc0b8, 0xf685, 0xe4d6, 0xf785, 0xc7df, 0xbeb9, 0xa7bf, 0xf885,
	0xf985, 0xfcc1, 0xcbdf, 0xccdf, 0xfa85, 0xd0df, 0xfb85, 0xfc85,
	0xfd85, 0xfe85, 0x4086, 0xdbdf, 0xe5df, 0x4186, 0xd7df, 0xd6df,
	0xc9d7, 0xe3df, 0xe4df, 0xebe5, 0xa7d2, 0xd2df, 0x4286, 0xa9bf,
	0x4386, 0xdbd4, 0x4486, 0xc8bf, 0xd4df, 0x4586, 0x4686, 0x4786,
	0xcccf, 0x4886, 0x4986, 0xdddf, 0x4a86, 0xcad1, 0x4b86, 0xdedf,
	0xa7b0, 0xb7c6, 0xd3df, 0x4c86, 0xe5ba, 0x4d86, 0xdfb6, 0xdbcd,
	0xfeb9, 0xd5d4, 0x4e86, 0x4f86, 0xdfdf, 0xeccf, 0xa5b0, 0xe7df,
	0xd1df, 0xc6d1, 0xd5df, 0xd8df, 0xd9df, 0xdcdf, 0x5086, 0xa9bb,
	0x5186, 0xe0df, 0xe1df, 0x5286, 0xe2df, 0xe6df, 0xe8df, 0xb4d3,
	0x5386, 0x5486, 0x5586, 0x5686, 0x5786, 0xe7b8, 0xb6c5, 0xeadf,
	0xdac9, 0xa8c1, 0xc4c4, 0x5886, 0x5986, 0xdebf, 0xf8cf, 0x5a86,
	0x5b86, 0x5c86, 0xdcd5, 0xeedf, 0x5d86, 0x5e86, 0x5f86, 0x6086,
	0x6186, 0x6286, 0xb8b2, 0x6386, 0xdfba, 0xecdf, 0x6486, 0xc1db,
	0x6586, 0xe4d1, 0x6686, 0x6786, 0x6886, 0x6986, 0xf4cb, 0xbdb4,
	0x6a86, 0xa6b0, 0x6b86, 0x6c86, 0x6d86, 0x6e86, 0x6f86, 0xf1df,
	0xc6cc, 0xf2df, 0x7086, 0x7186, 0xeddf, 0x7286, 0x7386, 0x7486,
	0x7586, 0x7686, 0x7786, 0xe9df, 0x7886, 0x7986, 0x7a86, 0x7b86,
	0xebdf, 0x7c86, 0xefdf, 0xf0df, 0xbdbb, 0x7d86, 0x7e86, 0xf3df,
	0x8086, 0x8186, 0xf4df, 0x8286, 0xa3bb, 0x8386, 0xdbca, 0xa8ce,
	0xa7e0, 0xaab3, 0x8486, 0xa6e0, 0x8586, 0x8686, 0x8786, 0xa1e0,
	0x8886, 0x8986, 0x8a86, 0x8b86, 0xfedf, 0x8c86, 0xd9cd, 0xfcdf,
	0x8d86, 0xfadf, 0x8e86, 0xd0bf, 0xc4d7, 0x8f86, 0xccc9, 0x9086,
	0x9186, 0xf8df, 0xa1b0, 0x9286, 0x9386, 0x9486, 0x9586, 0x9686,
	0xfddf, 0x9786, 0x9886, 0x9986, 0x9a86, 0xfbdf, 0xa2e0, 0x9b86,
	0x9c86, 0x9d86, 0x9e86, 0x9f86, 0xa8e0, 0xa086, 0xa186, 0xa286,
	0xa386, 0xc8b7, 0xa486, 0xa586, 0xa1c6, 0xb6c9, 0xb2c0, 0xf5df,
	0xa686, 0xa786, 0xbec5, 0xa886, 0xc4d8, 0xf9df, 0xf6c4, 0xa986,
	0xaa86, 0xab86, 0xac86, 0xad86, 0xae86, 0xa3e0, 0xa4e0, 0xa5e0,
	0xa5d0, 0xaf86, 0xb086, 0xb4e0, 0xe4cc, 0xb186, 0xb1e0, 0xb286,
	0xa6bf, 0xafe0, 0xb9ce, 0xabe0, 0xc6c9, 0xb386, 0xb486, 0xaec0,
	0xaee0, 0xedba, 0xb0ba, 0xa9e0, 0xb586, 0xb686, 0xb786, 0xf6df,
	0xb886, 0xb3e0, 0xb986, 0xba86, 0xb8e0, 0xbb86, 0xbc86, 0xbd86,
	0xadb4, 0xb9e0, 0xbe86, 0xbf86, 0xb2cf, 0xc8ba, 0xc086, 0xb0e0,
	0xc186, 0xc286, 0xc386, 0xc486, 0xc586, 0xc686, 0xc786, 0xfad0,
	0xc886, 0xc986, 0xca86, 0xcb86, 0xcc86, 0xcd86, 0xce86, 0xcf86,
	0xd086, 0xace0, 0xd186, 0xfbd4, 0xd286, 0xf7df, 0xd386, 0xe7c5,
	0xd486, 0xade0, 0xd586, 0xf7d3, 0xd686, 0xb6e0, 0xb7e0, 0xd786,
	0xd886, 0xd986, 0xda86, 0xdb86, 0xc4e0, 0xe1d0, 0xdc86, 0xdd86,
	0xde86, 0xbce0, 0xdf86, 0xe086, 0xc9e0, 0xcae0, 0xe186, 0xe286,
	0xe386, 0xbee0, 0xaae0, 0xa4c9, 0xc1e0, 0xe486, 0xb2e0, 0xe586,
	0xe686, 0xe786, 0xe886, 0xe986, 0xc8ca, 0xc3e0, 0xea86, 0xb5e0,
	0xeb86, 0xcbce, 0xec86, 0xc3cb, 0xcde0, 0xc6e0, 0xc2e0, 0xed86,
	0xcbe0, 0xee86, 0xbae0, 0xbfe0, 0xc0e0, 0xef86, 0xf086, 0xc5e0,
	0xf186, 0xf286, 0xc7e0, 0xc8e0, 0xf386, 0xcce0, 0xf486, 0xbbe0,
	0xf586, 0xf686, 0xf786, 0xf886, 0xf986, 0xd4cb, 0xd5e0, 0xfa86,
	0xd6e0, 0xd2e0, 0xfb86, 0xfc86, 0xfd86, 0xfe86, 0x4087, 0x4187,
	0xd0e0, 0xcebc, 0x4287, 0x4387, 0xd1e0, 0x4487, 0xc2b8, 0xc5d8,
	0x4587, 0x4687, 0x4787, 0x4887, 0x4987, 0x4a87, 0x4b87, 0x4c87,
	0xead0, 0x4d87, 0x4e87, 0xefc2, 0x4f87, 0x5087, 0xcfe0, 0xbde0,
	0x5187, 0x5287, 0x5387, 0xd4e0, 0xd3e0, 0x5487, 0x5587, 0xd7e0,
	0x5687, 0x5787, 0x5887, 0x5987, 0xdce0, 0xd8e0, 0x5a87, 0x5b87,
	0x5c87, 0xf6d6, 0xb0b3, 0x5d87, 0xecd7, 0x5e87, 0xbbcb, 0x5f87,
	0x6087, 0xdae0, 0x6187, 0xfbce, 0x6287, 0x6387, 0x6487, 0xd9ba,
	0x6587, 0x6687, 0x6787, 0x6887, 0x6987, 0x6a87, 0x6b87, 0x6c87,
	0x6d87, 0x6e87, 0x6f87, 0x7087, 0xe1e0, 0xdde0, 0xadd2, 0x7187,
	0x7287, 0x7387, 0x7487, 0x7587, 0xe2e0, 0x7687, 0x7787, 0xdbe0,
	0xd9e0, 0xdfe0, 0x7887, 0x7987, 0xe0e0, 0x7a87, 0x7b87, 0x7c87,
	0x7d87, 0x7e87, 0xdee0, 0x8087, 0xe4e0, 0x8187, 0x8287, 0x8387,
	0xf7c6, 0xacd8, 0xebd4, 0xe6e0, 0xc9ca, 0x8487, 0x8587, 0x8687,
	0x8787, 0xe5e0, 0x8887, 0x8987, 0x8a87, 0x8b87, 0xc1b8, 0x8c87,
	0x8d87, 0x8e87, 0x8f87, 0xe7e0, 0xe8e0, 0x9087, 0x9187, 0x9287,
	0x9387, 0x9487, 0x9587, 0x9687, 0x9787, 0xe9e0, 0xe3e0, 0x9887,
	0x9987, 0x9a87, 0x9b87, 0x9c87, 0x9d87, 0x9e87, 0xbfba, 0xe7cc,
	0x9f87, 0xa087, 0xa187, 0xeae0, 0xa287, 0xa387, 0xa487, 0xa587,
	0xa687, 0xa787, 0xa887, 0xa987, 0xaa87, 0xab87, 0xac87, 0xad87,
	0xae87, 0xaf87, 0xb087, 0xf9cf, 0xb187, 0xb287, 0xb387, 0xb487,
	0xb587, 0xb687, 0xb787, 0xb887, 0xb987, 0xba87, 0xbb87, 0xebe0,
	0xbc87, 0xbd87, 0xbe87, 0xbf87, 0xc087, 0xc187, 0xc287, 0xc2c8,
	0xc387, 0xc487, 0xc587, 0xc687, 0xc0bd, 0xc787, 0xc887, 0xc987,
	0xca87, 0xcb87, 0xcc87, 0xcd87, 0xce87, 0xcf87, 0xd087, 0xd187,
	0xd287, 0xd387, 0xd2c4, 0xd487, 0xd587, 0xd687, 0xd787, 0xd887,
	0xd987, 0xda87, 0xdb87, 0xdc87, 0xece0, 0xdd87, 0xde87, 0xede0,
	0xdf87, 0xe087, 0xf4c7, 0xc4cb, 0xe187, 0xeee0, 0xd8bb, 0xb6d8,
	0xf2d2, 0xefe0, 0xc5cd, 0xe287, 0xdab6, 0xe387, 0xe487, 0xe587,
	0xe687, 0xe787, 0xe887, 0xf1e0, 0xe987, 0xb0d4, 0xea87, 0xeb87,
	0xa7c0, 0xd1b4, 0xec87, 0xed87, 0xa7ce, 0xf0e0, 0xee87, 0xef87,
	0xf087, 0xf2e0, 0xccb9, 0xf187, 0xf287, 0xfab9, 0xbccd, 0xf3e0,
	0xf387, 0xf487, 0xf587, 0xd4c6, 0xf4e0, 0xf687, 0xb2d4, 0xf787,
	0xa6c8, 0xf6e0, 0xf5e0, 0xf887, 0xf987, 0xfa87, 0xfb87, 0xfc87,
	0xfd87, 0xfe87, 0x4088, 0x4188, 0x4288, 0x4388, 0x4488, 0x4588,
	0x4688, 0x4788, 0x4888, 0x4988, 0xf7e0, 0x4a88, 0x4b88, 0xc1cd,
	0x4c88, 0x4d88, 0x4e88, 0xa5ca, 0x4f88, 0x5088, 0x5188, 0x5288,
	0xdad4, 0xd7db, 0xd9db, 0x5388, 0xd8db, 0xe7b9, 0xdcdb, 0xdddb,
	0xd8b5, 0x5488, 0x5588, 0xdadb, 0x5688, 0x5788, 0x5888, 0x5988,
	0x5a88, 0xdbdb, 0xa1b3, 0xdfdb, 0x5b88, 0x5c88, 0xf8bb, 0x5d88,
	0xb7d6, 0x5e88, 0xe0db, 0x5f88, 0x6088, 0x6188, 0x6288, 0xf9be,
	0x6388, 0x6488, 0xbbb7, 0x6588, 0xd0db, 0xaecc, 0xb2bf, 0xb5bb,
	0xf8d7, 0xd3bf, 0x6688, 0x6788, 0x6888, 0x6988, 0x6a88, 0xe9bf,
	0x6b88, 0x6c88, 0xe1bc, 0xb3cc, 0xdedb, 0xd3b0, 0xebce, 0xd8b7,
	0xb9d7, 0xc2c6, 0x6d88, 0x6e88, 0xa4c0, 0x6f88, 0xb9cc, 0x7088,
	0xe7db, 0xe1db, 0xbac6, 0xe3db, 0x7188, 0xe8db, 0x7288, 0xf7c5,
	0x7388, 0x7488, 0x7588, 0xeadb, 0x7688, 0x7788, 0xe9db, 0xc0bf,
	0x7888, 0x7988, 0x7a88, 0xe6db, 0xe5db, 0x7b88, 0x7c88, 0x7d88,
	0x7e88, 0x8088, 0xb9b4, 0xacc0, 0xa2c2, 0xe2db, 0xe4db, 0x8188,
	0x8288, 0x8388, 0x8488, 0xcdd0, 0xeddb, 0x8588, 0x8688, 0x8788,
	0x8888, 0x8988, 0xddc0, 0xf2db, 0x8a88, 0x8b88, 0x8c88, 0x8d88,
	0x8e88, 0x8f88, 0x9088, 0xe2b6, 0x9188, 0x9288, 0x9388, 0x9488,
	0xf3db, 0xd2db, 0xb8b9, 0xabd4, 0xecdb, 0x9588, 0xd1bf, 0xf0db,
	0x9688, 0xd1db, 0x9788, 0xe6b5, 0x9888, 0xebdb, 0xe5bf, 0x9988,
	0x9a88, 0x9b88, 0xeedb, 0x9c88, 0xf1db, 0x9d88, 0x9e88, 0x9f88,
	0xf9db, 0xa088, 0xa188, 0xa288, 0xa388, 0xa488, 0xa588, 0xa688,
	0xa788, 0xa888, 0xa1b9, 0xa3b0, 0xa988, 0xaa88, 0xab88, 0xac88,
	0xad88, 0xae88, 0xaf88, 0xf1c2, 0xb088, 0xb188, 0xc7b3, 0xefdb,
	0xb288, 0xb388, 0xf8db, 0xb488, 0xd2c6, 0xf4db, 0xb588, 0xb688,
	0xf5db, 0xf7db, 0xf6db, 0xb788, 0xb888, 0xfedb, 0xb988, 0xf2d3,
	0xbab2, 0xba88, 0xbb88, 0xbc88, 0xfddb, 0xbd88, 0xbe88, 0xbf88,
	0xc088, 0xc188, 0xc288, 0xc388, 0xc488, 0xa4dc, 0xc588, 0xfbdb,
	0xc688, 0xc788, 0xc888, 0xc988, 0xfadb, 0xca88, 0xcb88, 0xcc88,
	0xfcdb, 0xe0c5, 0xf9bb, 0xcd88, 0xce88, 0xa3dc, 0xcf88, 0xd088,
	0xa5dc, 0xd188, 0xc3cc, 0xd288, 0xd388, 0xd488, 0xd1b6, 0xc0dd,
	0xd588, 0xd688, 0xd788, 0xa1dc, 0xd888, 0xa2dc, 0xd988, 0xda88,
	0xdb88, 0xb5c7, 0xdc88, 0xdd88, 0xde88, 0xe9b6, 0xdf88, 0xe088,
	0xe188, 0xa7dc, 0xe288, 0xe388, 0xe488, 0xe588, 0xa6dc, 0xe688,
	0xa9dc, 0xa4b1, 0xe788, 0xe888, 0xccb5, 0xe988, 0xea88, 0xeb88,
	0xec88, 0xed88, 0xb0bf, 0xee88, 0xef88, 0xf088, 0xf188, 0xf288,
	0xdfd1, 0xf388, 0xf488, 0xf588, 0xf688, 0xc2b6, 0xf788, 0xf888,
	0xf988, 0xfa88, 0xfb88, 0xfc88, 0xfd88, 0xfe88, 0x4089, 0x4189,
	0x4289, 0x4389, 0x4489, 0x4589, 0xa8dc, 0x4689, 0x4789, 0x4889,
	0x4989, 0x4a89, 0x4b89, 0x4c89, 0xfacb, 0xf3eb, 0x4d89, 0x4e89,
	0x4f89, 0xdccb, 0x5089, 0x5189, 0xfecb, 0x5289, 0x5389, 0x5489,
	0xc1cc, 0x5589, 0x5689, 0x5789, 0x5889, 0x5989, 0xfbc8, 0x5a89,
	0x5b89, 0x5c89, 0x5d89, 0x5e89, 0x5f89, 0xaadc, 0x6089, 0x6189,
	0x6289, 0x6389, 0x6489, 0xeecc, 0xabdc, 0x6589, 0x6689, 0x6789,
	0x6889, 0x6989, 0x6a89, 0x6b89, 0x6c89, 0x6d89, 0x6e89, 0x6f89,
	0x7089, 0x7189, 0x7289, 0x7389, 0x7489, 0x7589, 0xd3db, 0x7689,
	0xafdc, 0xacdc, 0x7789, 0xb3be, 0x7889, 0xfbca, 0x7989, 0x7a89,
	0x7b89, 0xaddc, 0x7c89, 0x7d89, 0x7e89, 0x8089, 0x8189, 0x8289,
	0x8389, 0x8489, 0xcac9, 0xb9c4, 0x8589, 0x8689, 0x8789, 0x8889,
	0x8989, 0xbdc7, 0xaedc, 0x8a89, 0x8b89, 0x8c89, 0xf6d4, 0xe6d0,
	0x8d89, 0x8e89, 0x8f89, 0x9089, 0x9189, 0x9289, 0x9389, 0x9489,
	0xabc4, 0xd5b6, 0x9589, 0x9689, 0x9789, 0x9889, 0x9989, 0x9a89,
	0x9b89, 0x9c89, 0x9d89, 0x9e89, 0x9f89, 0xa089, 0xa189, 0xa289,
	0xa389, 0xa489, 0xa589, 0xa689, 0xd4db, 0xa789, 0xa889, 0xa989,
	0xaa89, 0xdab1, 0xab89, 0xac89, 0xad89, 0xd5db, 0xae89, 0xaf89,
	0xb089, 0xb189, 0xb289, 0xb389, 0xb489, 0xb589, 0xb689, 0xb789,
	0xb889, 0xd6db, 0xb989, 0xba89, 0xbb89, 0xbeba, 0xbc89, 0xbd89,
	0xbe89, 0xbf89, 0xc089, 0xc189, 0xc289, 0xc389, 0xc489, 0xc589,
	0xc689, 0xc789, 0xc889, 0xc989, 0xc0c8, 0xca89, 0xcb89, 0xcc89,
	0xcd89, 0xce89, 0xcf89, 0xbfca, 0xc9c8, 0xd089, 0xb3d7, 0xd189,
	0xf9c9, 0xd289, 0xd389, 0xc7bf, 0xd489, 0xd589, 0xf8ba, 0xd689,
	0xd789, 0xbcd2, 0xd889, 0xd989, 0xda89, 0xdb89, 0xdc89, 0xdd89,
	0xde89, 0xdf89, 0xbae2, 0xe089, 0xa6b4, 0xe189, 0xe289, 0xb8b1,
	0xe389, 0xe489, 0xe589, 0xe689, 0xe789, 0xb4b8, 0xe889, 0xc4cf,
	0xe989, 0xea89, 0xeb89, 0xec89, 0xe7d9, 0xa6cf, 0xe2cd, 0xed89,
	0xee89, 0xedd9, 0xe0b6, 0xef89, 0xb9d2, 0xf089, 0xf189, 0xbbb9,
	0xf289, 0xf389, 0xf489, 0xf589, 0xb9e2, 0xb7e2, 0xf689, 0xf3b4,
	0xf789, 0xeccc, 0xabcc, 0xf2b7, 0xf889, 0xb2d8, 0xebd1, 0xbbba,
	0xf989, 0xa7ca, 0xfa89, 0xfb89, 0xb7cd, 0xfc89, 0xfd89, 0xc4d2,
	0xe4bf, 0xd0bc, 0xe1b6, 0xfe89, 0xc5de, 0x408a, 0x418a, 0x428a,
	0x438a, 0xc6de, 0xbcdb, 0x448a, 0xd9d1, 0x458a, 0x468a, 0xe6c6,
	0xcec4, 0xeeb7, 0x478a, 0xdcb7, 0x488a, 0x498a, 0xfcbf, 0xe0d7,
	0x4a8a, 0xf5c6, 0x4b8a, 0x4c8a, 0xbcb1, 0xc8de, 0xb1bd, 0xd7cc,
	0xcade, 0x4d8a, 0xc9de, 0x4e8a, 0x4f8a, 0x508a, 0x518a, 0x528a,
	0xecb5, 0x538a, 0xddc9, 0x548a, 0x558a, 0xc2b0, 0x568a, 0x578a,
	0x588a, 0x598a, 0x5a8a, 0x5b8a, 0x5c8a, 0x5d8a, 0x5e8a, 0x5f8a,
	0x608a, 0x618a, 0x628a, 0xaec5, 0xabc5, 0x638a, 0xccc4, 0x648a,
	0xe9bc, 0xfdcb, 0x658a, 0x668a, 0x678a, 0xc3ba, 0x688a, 0x698a,
	0x6a8a, 0xf9e5, 0xe7c8, 0xfae5, 0xfdcd, 0x6b8a, 0xb1d7, 0xbeb8,
	0xe8c2, 0x6c8a, 0xd1c8, 0x6d8a, 0x6e8a, 0xfbe5, 0x6f8a, 0x708a,
	0x718a, 0x728a, 0xcab6, 0xcbbc, 0x738a, 0x748a, 0xfdd1, 0xa1e6,
	0x758a, 0xeec3, 0x768a, 0x778a, 0x788a, 0x798a, 0xa4e6, 0x7a8a,
	0x7b8a, 0x7c8a, 0x7d8a, 0xfee5, 0xa5e6, 0xd7cd, 0x7e8a, 0x808a,
	0xc1b7, 0xfce5, 0xfde5, 0xa3e6, 0x818a, 0x828a, 0xddc4, 0xa8e6,
	0x838a, 0x848a, 0xa7e6, 0x858a, 0x868a, 0x878a, 0x888a, 0x898a,
	0x8a8a, 0xc3c3, 0x8b8a, 0xdec6, 0x8c8a, 0x8d8a, 0xaae6, 0x8e8a,
	0x8f8a, 0x908a, 0x918a, 0x928a, 0x938a, 0x948a, 0xb7c4, 0x958a,
	0x968a, 0x978a, 0xa2e6, 0xbcca, 0x988a, 0x998a, 0x9a8a, 0x9b8a,
	0xe3bd, 0xc3b9, 0xa6e6, 0xd5d0, 0xafce, 0x9c8a, 0x9d8a, 0xa9e6,
	0xb0e6, 0x9e8a, 0xa6d2, 0x9f8a, 0xaabd, 0xade6, 0xa08a, 0xa18a,
	0xa28a, 0xa38a, 0xa48a, 0xafe6, 0xa58a, 0xd1c0, 0xa68a, 0xa78a,
	0xccd2, 0xa88a, 0xa98a, 0xaa8a, 0xa7bc, 0xab8a, 0xac8a, 0xad8a,
	0xae8a, 0xaf8a, 0xb08a, 0xb18a, 0xb28a, 0xb38a, 0xb48a, 0xb58a,
	0xb68a, 0xb1e6, 0xb78a, 0xf6d2, 0xb88a, 0xb98a, 0xba8a, 0xcbd7,
	0xbb8a, 0xfecd, 0xbc8a, 0xdecd, 0xa6c2, 0xabe6, 0xace6, 0xbfbd,
	0xaee6, 0xb3e6, 0xbd8a, 0xbe8a, 0xb2e6, 0xbf8a, 0xc08a, 0xc18a,
	0xc28a, 0xb6e6, 0xc38a, 0xb8e6, 0xc48a, 0xc58a, 0xc68a, 0xc78a,
	0xefc4, 0xc88a, 0xc98a, 0xca8a, 0xc8c4, 0xcb8a, 0xcc8a, 0xeabe,
	0xefc9, 0xcd8a, 0xce8a, 0xb7e6, 0xcf8a, 0xf0b6, 0xd08a, 0xd18a,
	0xd28a, 0xe4c3, 0xd38a, 0xd48a, 0xd58a, 0xd68a, 0xd78a, 0xd88a,
	0xd98a, 0xe9d3, 0xb4e6, 0xda8a, 0xb5e6, 0xdb8a, 0xa2c8, 0xdc8a,
	0xdd8a, 0xde8a, 0xdf8a, 0xe08a, 0xbde6, 0xe18a, 0xe28a, 0xe38a,
	0xb9e6, 0xe48a, 0xe58a, 0xe68a, 0xe78a, 0xe88a, 0xc5c6, 0xe98a,
	0xea8a, 0xf1cd, 0xbbe6, 0xeb8a, 0xec8a, 0xed8a, 0xee8a, 0xef8a,
	0xf08a, 0xf18a, 0xf28a, 0xf38a, 0xf48a, 0xbce6, 0xf58a, 0xf68a,
	0xf78a, 0xf88a, 0xe9bb, 0xf98a, 0xfa8a, 0xfb8a, 0xfc8a, 0xfd8a,
	0xfe8a, 0x408b, 0xbee6, 0x418b, 0x428b, 0x438b, 0x448b, 0xbae6,
	0x458b, 0x468b, 0xb7c0, 0x478b, 0x488b, 0x498b, 0x4a8b, 0x4b8b,
	0x4c8b, 0x4d8b, 0x4e8b, 0x4f8b, 0xa4d3, 0xbfe6, 0xf4c9, 0xc3e6,
	0x508b, 0x518b, 0xc4e6, 0x528b, 0x538b, 0x548b, 0x558b, 0xf6d0,
	0x568b, 0x578b, 0x588b, 0x598b, 0x5a8b, 0x5b8b, 0x5c8b, 0x5d8b,
	0x5e8b, 0x5f8b, 0x608b, 0x618b, 0x628b, 0x638b, 0x648b, 0x658b,
	0x668b, 0x678b, 0xbdc3, 0x688b, 0x698b, 0x6a8b, 0x6b8b, 0x6c8b,
	0x6d8b, 0x6e8b, 0xc4c3, 0xc2e6, 0x6f8b, 0x708b, 0x718b, 0x728b,
	0x738b, 0x748b, 0x758b, 0x768b, 0x778b, 0x788b, 0x798b, 0x7a8b,
	0x7b8b, 0x7c8b, 0xc1e6, 0x7d8b, 0x7e8b, 0x808b, 0x818b, 0x828b,
	0x838b, 0x848b, 0xc7e6, 0xb1cf, 0x858b, 0xf4eb, 0x868b, 0x878b,
	0xcae6, 0x888b, 0x898b, 0x8a8b, 0x8b8b, 0x8c8b, 0xc5e6, 0x8d8b,
	0x8e8b, 0xdebc, 0xa9c9, 0x8f8b, 0x908b, 0x918b, 0x928b, 0x938b,
	0x948b, 0xb5bc, 0x958b, 0x968b, 0xd3cf, 0x978b, 0x988b, 0x998b,
	0x9a8b, 0x9b8b, 0xc8e6, 0x9c8b, 0xc9e6, 0x9d8b, 0xcee6, 0x9e8b,
	0xd0e6, 0x9f8b, 0xa08b, 0xa18b, 0xd1e6, 0xa28b, 0xa38b, 0xa48b,
	0xcbe6, 0xd5b5, 0xa58b, 0xcce6, 0xa68b, 0xa78b, 0xcfe6, 0xa88b,
	0xa98b, 0xdbc4, 0xaa8b, 0xc6e6, 0xab8b, 0xac8b, 0xad8b, 0xae8b,
	0xaf8b, 0xcde6, 0xb08b, 0xb18b, 0xb28b, 0xb38b, 0xb48b, 0xb58b,
	0xb68b, 0xb78b, 0xb88b, 0xb98b, 0xba8b, 0xbb8b, 0xbc8b, 0xbd8b,
	0xbe8b, 0xbf8b, 0xc08b, 0xc18b, 0xc28b, 0xc38b, 0xc48b, 0xc58b,
	0xc68b, 0xd2e6, 0xc78b, 0xc88b, 0xc98b, 0xca8b, 0xcb8b, 0xcc8b,
	0xcd8b, 0xce8b, 0xcf8b, 0xd08b, 0xd18b, 0xd28b, 0xd4e6, 0xd3e6,
	0xd38b, 0xd48b, 0xd58b, 0xd68b, 0xd78b, 0xd88b, 0xd98b, 0xda8b,
	0xdb8b, 0xdc8b, 0xdd8b, 0xde8b, 0xdf8b, 0xe08b, 0xe18b, 0xe28b,
	0xe38b, 0xe48b, 0xe58b, 0xe68b, 0xe78b, 0xe88b, 0xe98b, 0xea8b,
	0xeb8b, 0xec8b, 0xd5e6, 0xed8b, 0xf8d9, 0xee8b, 0xef8b, 0xd6e6,
	0xf08b, 0xf18b, 0xf28b, 0xf38b, 0xf48b, 0xf58b, 0xf68b, 0xf78b,
	0xd7e6, 0xf88b, 0xf98b, 0xfa8b, 0xfb8b, 0xfc8b, 0xfd8b, 0xfe8b,
	0x408c, 0x418c, 0x428c, 0x438c, 0x448c, 0x458c, 0x468c, 0x478c,
	0xd3d7, 0xdde6, 0x488c, 0xdee6, 0xd7bf, 0xd0d4, 0x498c, 0xd6d7,
	0xe6b4, 0xefcb, 0xdae6, 0xc3d8, 0xced7, 0xa2d0, 0x4a8c, 0xcfc3,
	0x4b8c, 0x4c8c, 0xdfe6, 0xbebc, 0xc2b9, 0xdbe6, 0xa7d1, 0x4d8c,
	0x4e8c, 0xa2ba, 0xcfc2, 0x4f8c, 0xabd8, 0x508c, 0x518c, 0x528c,
	0xebca, 0xeee5, 0x538c, 0xdce6, 0x548c, 0xf5b7, 0x558c, 0x568c,
	0x578c, 0x588c, 0xe6c8, 0x598c, 0x5a8c, 0xf5c4, 0x5b8c, 0x5c8c,
	0xb2e5, 0xfec4, 0x5d8c, 0xfccb, 0xb3e5, 0xacd5, 0x5e8c, 0xeed3,
	0xd8ca, 0xb2b0, 0x5f8c, 0xcecb, 0xeacd, 0x608c, 0x618c, 0xeaba,
	0x628c, 0x638c, 0x648c, 0xb5e5, 0x658c, 0xb4e5, 0x668c, 0xdad7,
	0xd9b9, 0xe6d6, 0xa8b6, 0xf0cd, 0xcbd2, 0xa6b1, 0xb5ca, 0x678c,
	0xe8b3, 0xf3c9, 0xcdbf, 0xfbd0, 0xd2ca, 0xb6e5, 0xc2bb, 0x688c,
	0x698c, 0x6a8c, 0xdccf, 0xacb9, 0x6b8c, 0x6c8c, 0x6d8c, 0x6e8c,
	0xd7d4, 0x6f8c, 0x708c, 0xa6ba, 0xe7d1, 0xfccf, 0xd2bc, 0x718c,
	0xb7e5, 0xddc8, 0x728c, 0x738c, 0x748c, 0xedbf, 0xf6b1, 0xdecb,
	0x758c, 0x768c, 0xc5bc, 0x778c, 0xc4bc, 0xfad2, 0xdcc3, 0xdcbf,
	0x788c, 0x798c, 0x7a8c, 0x7b8c, 0xbbb8, 0x7c8c, 0x7d8c, 0x7e8c,
	0xc2c3, 0x808c, 0xaeba, 0xa2d4, 0x818c, 0x828c, 0x838c, 0x848c,
	0x858c, 0x868c, 0x878c, 0x888c, 0x898c, 0xdec7, 0xafc4, 0xecb2,
	0x8a8c, 0xd1b9, 0x8b8c, 0x8c8c, 0xbbe5, 0xc8c1, 0x8d8c, 0x8e8c,
	0xafd5, 0x8f8c, 0x908c, 0x918c, 0x928c, 0x938c, 0xbce5, 0x948c,
	0xbee5, 0x958c, 0x968c, 0x978c, 0x988c, 0x998c, 0x9a8c, 0x9b8c,
	0xe7b4, 0xd4b6, 0xc2cb, 0xb0d1, 0xbcb5, 0x9c8c, 0x9d8c, 0xd9ca,
	0x9e8c, 0xe2b7, 0x9f8c, 0xa08c, 0xe4c9, 0xa18c, 0xabbd, 0xa28c,
	0xa38c, 0xbece, 0xf0d7, 0xa48c, 0xa58c, 0xa68c, 0xa78c, 0xa1d0,
	0xa88c, 0xd9c9, 0xa98c, 0xaa8c, 0xfbb6, 0xd8e6, 0xe2bc, 0xab8c,
	0xbeb3, 0xac8c, 0xd0c9, 0xad8c, 0xd9e6, 0xa2b3, 0xae8c, 0xaf8c,
	0xb08c, 0xb18c, 0xccde, 0xb28c, 0xc8d3, 0xcdde, 0xb38c, 0xa2d2,
	0xb48c, 0xb58c, 0xb68c, 0xb78c, 0xcede, 0xb88c, 0xb98c, 0xba8c,
	0xbb8c, 0xcdbe, 0xbc8c, 0xbd8c, 0xcfde, 0xbe8c, 0xbf8c, 0xc08c,
	0xacca, 0xfcd2, 0xdfb3, 0xeae5, 0xe1c4, 0xa1be, 0xb2ce, 0xf2c4,
	0xd6be, 0xa8c6, 0xe3b2, 0xc18c, 0xc28c, 0xd3be, 0xc38c, 0xc48c,
	0xfcc7, 0xebcc, 0xecbd, 0xddce, 0xc58c, 0xc68c, 0xbaca, 0xc1c6,
	0xece5, 0xbcd0, 0xc78c, 0xc88c, 0xc98c, 0xb9d5, 0xca8c, 0xcb8c,
	0xcc8c, 0xede5, 0xcd8c, 0xce8c, 0xcf8c, 0xd08c, 0xf4ca, 0xd18c,
	0xc0cd, 0xc5c2, 0xd28c, 0xefe5, 0xd38c, 0xc4c2, 0xf0e5, 0xd48c,
	0xd58c, 0xd68c, 0xd78c, 0xd88c, 0xd98c, 0xda8c, 0xf8e5, 0xcdcd,
	0xdb8c, 0xbdc9, 0xdc8c, 0xdd8c, 0xde8c, 0xdf8c, 0xe08c, 0xe18c,
	0xe28c, 0xd9d2, 0xa8e1, 0xe38c, 0xe48c, 0xe58c, 0xe68c, 0xecd3,
	0xe78c, 0xeacb, 0xf1c6, 0xe88c, 0xe98c, 0xea8c, 0xeb8c, 0xec8c,
	0xace1, 0xed8c, 0xee8c, 0xef8c, 0xa7e1, 0xa9e1, 0xf08c, 0xf18c,
	0xaae1, 0xafe1, 0xf28c, 0xf38c, 0xedb2, 0xf48c, 0xabe1, 0xdab8,
	0xade1, 0xaee1, 0xb0e1, 0xbab5, 0xb1e1, 0xf58c, 0xf68c, 0xf78c,
	0xf88c, 0xf98c, 0xb3e1, 0xb8e1, 0xfa8c, 0xfb8c, 0xfc8c, 0xfd8c,
	0xfe8c, 0xd2d1, 0x408d, 0xb6e1, 0xb5e1, 0xebc1, 0x418d, 0x428d,
	0x438d, 0xb7e1, 0x448d, 0xc0d4, 0x458d, 0xb2e1, 0x468d, 0xbae1,
	0xb6b0, 0x478d, 0x488d, 0x498d, 0x4a8d, 0xb4e1, 0x4b8d, 0xf9bf,
	0x4c8d, 0xb9e1, 0x4d8d, 0x4e8d, 0xbbe1, 0x4f8d, 0x508d, 0x518d,
	0x528d, 0x538d, 0x548d, 0xbee1, 0x558d, 0x568d, 0x578d, 0x588d,
	0x598d, 0x5a8d, 0xbce1, 0x5b8d, 0x5c8d, 0x5d8d, 0x5e8d, 0x5f8d,
	0x608d, 0xc5d6, 0x618d, 0x628d, 0x638d, 0x648d, 0x658d, 0x668d,
	0x678d, 0xbfcf, 0x688d, 0x698d, 0xbde1, 0xbfe1, 0xcdc2, 0x6a8d,
	0xebb6, 0x6b8d, 0xf8d3, 0x6c8d, 0x6d8d, 0xcdc7, 0x6e8d, 0x6f8d,
	0xe5b7, 0x708d, 0x718d, 0x728d, 0x738d, 0x748d, 0x758d, 0x768d,
	0x778d, 0x788d, 0x798d, 0xfebe, 0x7a8d, 0x7b8d, 0x7c8d, 0x7d8d,
	0x7e8d, 0x808d, 0xc0e1, 0xc1e1, 0x818d, 0x828d, 0xc7e1, 0xe7b3,
	0x838d, 0x848d, 0x858d, 0x868d, 0x878d, 0x888d, 0xe9c6, 0x898d,
	0x8a8d, 0x8b8d, 0x8c8d, 0x8d8d, 0xdeb4, 0x8e8d, 0xc2d1, 0x8f8d,
	0x908d, 0x918d, 0x928d, 0xc8e1, 0x938d, 0x948d, 0xc6e1, 0x958d,
	0x968d, 0x978d, 0x988d, 0x998d, 0xc5e1, 0x9a8d, 0xc3e1, 0xc2e1,
	0x9b8d, 0xc0b1, 0x9c8d, 0x9d8d, 0x9e8d, 0xb8d5, 0xc4e1, 0x9f8d,
	0xa08d, 0xa18d, 0xa28d, 0xa38d, 0xcbe1, 0xa48d, 0xa58d, 0xa68d,
	0xa78d, 0xa88d, 0xa98d, 0xaa8d, 0xab8d, 0xcce1, 0xcae1, 0xac8d,
	0xad8d, 0xae8d, 0xaf8d, 0xb08d, 0xb18d, 0xb28d, 0xb38d, 0xfaef,
	0xb48d, 0xb58d, 0xd3e1, 0xd2e1, 0xb6c7, 0xb68d, 0xb78d, 0xb88d,
	0xb98d, 0xba8d, 0xbb8d, 0xbc8d, 0xbd8d, 0xbe8d, 0xbf8d, 0xc08d,
	0xc9e1, 0xc18d, 0xc28d, 0xcee1, 0xc38d, 0xd0e1, 0xc48d, 0xc58d,
	0xc68d, 0xc78d, 0xc88d, 0xc98d, 0xca8d, 0xcb8d, 0xcc8d, 0xcd8d,
	0xce8d, 0xd4e1, 0xcf8d, 0xd1e1, 0xcde1, 0xd08d, 0xd18d, 0xcfe1,
	0xd28d, 0xd38d, 0xd48d, 0xd58d, 0xd5e1, 0xd68d, 0xd78d, 0xd88d,
	0xd98d, 0xda8d, 0xdb8d, 0xdc8d, 0xdd8d, 0xde8d, 0xdf8d, 0xe08d,
	0xe18d, 0xe28d, 0xd6e1, 0xe38d, 0xe48d, 0xe58d, 0xe68d, 0xe78d,
	0xe88d, 0xe98d, 0xea8d, 0xeb8d, 0xec8d, 0xed8d, 0xee8d, 0xef8d,
	0xf08d, 0xf18d, 0xf28d, 0xf38d, 0xf48d, 0xf58d, 0xf68d, 0xf78d,
	0xf88d, 0xd7e1, 0xf98d, 0xfa8d, 0xfb8d, 0xd8e1, 0xfc8d, 0xfd8d,
	0xfe8d, 0x408e, 0x418e, 0x428e, 0x438e, 0x448e, 0x458e, 0x468e,
	0x478e, 0x488e, 0x498e, 0x4a8e, 0x4b8e, 0x4c8e, 0x4d8e, 0x4e8e,
	0x4f8e, 0x508e, 0x518e, 0x528e, 0x538e, 0x548e, 0x558e, 0xdae1,
	0x568e, 0x578e, 0x588e, 0x598e, 0x5a8e, 0x5b8e, 0x5c8e, 0x5d8e,
	0x5e8e, 0x5f8e, 0x608e, 0x618e, 0x628e, 0xdbe1, 0x638e, 0x648e,
	0x658e, 0x668e, 0x678e, 0x688e, 0x698e, 0xa1ce, 0x6a8e, 0x6b8e,
	0x6c8e, 0x6d8e, 0x6e8e, 0x6f8e, 0x708e, 0x718e, 0x728e, 0x738e,
	0x748e, 0x758e, 0x768e, 0xdde7, 0x778e, 0xa8b4, 0xddd6, 0x788e,
	0x798e, 0xb2d1, 0xb2b3, 0x7a8e, 0x7b8e, 0xa4b9, 0xf3d7, 0xc9c7,
	0xdebe, 0xaeb9, 0x7c8e, 0xd7ce, 0x7d8e, 0x7e8e, 0xeeb2, 0xcfdb,
	0x808e, 0xbabc, 0xd1d2, 0xc8cb, 0xcdb0, 0x818e, 0x828e, 0xefcf,
	0x838e, 0x848e, 0x858e, 0x868e, 0x878e, 0xe3d9, 0xedbd, 0x888e,
	0x898e, 0xd2b1, 0xd0ca, 0xbcb2, 0x8a8e, 0xa7cb, 0xabb7, 0x8b8e,
	0xa6ca, 0x8c8e, 0x8d8e, 0x8e8e, 0xa3cf, 0x8f8e, 0x908e, 0xf8e0,
	0xcad5, 0xfbe0, 0x918e, 0x928e, 0xfae0, 0xc1c5, 0xfbcc, 0x938e,
	0xb1c1, 0xf9e0, 0xe3d6, 0xafb2, 0xc4d6, 0xdbb5, 0x948e, 0x958e,
	0x968e, 0x978e, 0x988e, 0x998e, 0x9a8e, 0x9b8e, 0xf8b4, 0xa1d6,
	0x9c8e, 0x9d8e, 0x9e8e, 0x9f8e, 0xa08e, 0xafcf, 0xefb0, 0xa18e,
	0xa28e, 0xfce0, 0xa38e, 0xa48e, 0xa58e, 0xa68e, 0xa78e, 0xa1e1,
	0xa3b3, 0xa88e, 0xa98e, 0xfde0, 0xfee0, 0xb1c3, 0xaa8e, 0xab8e,
	0xac8e, 0xad8e, 0xddc3, 0xae8e, 0xa2e1, 0xf9b7, 0xaf8e, 0xb08e,
	0xb18e, 0xb28e, 0xb38e, 0xb48e, 0xcfbb, 0xb58e, 0xb68e, 0xb78e,
	0xb88e, 0xb98e, 0xba8e, 0xbb8e, 0xa3e1, 0xbbc4, 0xbc8e, 0xbd8e,
	0xbe8e, 0xbf8e, 0xc08e, 0xa4e1, 0xc18e, 0xc28e, 0xa5e1, 0xc38e,
	0xc48e, 0xa6e1, 0xb1b4, 0xc58e, 0xc68e, 0xc78e, 0xc88e, 0xc98e,
	0xca8e, 0xcb8e, 0xcc8e, 0xcd8e, 0xce8e, 0xcf8e, 0xd08e, 0xd18e,
	0xd28e, 0xd38e, 0xc9b8, 0xbdc6, 0xeac4, 0xd48e, 0xa2b2, 0xd58e,
	0xd2d0, 0xd68e, 0xdbe7, 0xc3bb, 0xd7d3, 0xc4d3, 0xd78e, 0xe3b9,
	0xcfe2, 0xd88e, 0xd98e, 0xda8e, 0xafd7, 0xdb8e, 0xecc7, 0xd3b1,
	0xdc8e, 0xdd8e, 0xb2b4, 0xd1e2, 0xde8e, 0xdf8e, 0xe08e, 0xf2d0,
	0xaec2, 0xd0e2, 0xe18e, 0xe2bf, 0xa6d3, 0xd7b5, 0xd2e2, 0xeab5,
	0xe28e, 0xedc3, 0xfdb8, 0xe38e, 0xaeb8, 0xe48e, 0xd3c5, 0xcfb7,
	0xd4e2, 0xe58e, 0xe68e, 0xe78e, 0xe88e, 0xd3e2, 0xc8b6, 0xf9d7,
	0xe98e, 0xea8e, 0xeb8e, 0xec8e, 0xed8e, 0xa5cd, 0xee8e, 0xef8e,
	0xf08e, 0xf18e, 0xf28e, 0xd8e2, 0xf38e, 0xd6e2, 0xfcca, 0xb5bf,
	0xb9d3, 0xd5e2, 0xf48e, 0xf58e, 0xf68e, 0xf78e, 0xd7e2, 0xf88e,
	0xf98e, 0xfa8e, 0xfb8e, 0xfc8e, 0xfd8e, 0xfe8e, 0x408f, 0x418f,
	0x428f, 0xaec1, 0xc8c0, 0x438f, 0x448f, 0x458f, 0x468f, 0x478f,
	0x488f, 0xdbe2, 0xdae2, 0xaac0, 0x498f, 0x4a8f, 0xcec1, 0x4b8f,
	0x4c8f, 0x4d8f, 0x4e8f, 0xdce2, 0x4f8f, 0x508f, 0x518f, 0x528f,
	0x538f, 0x548f, 0x558f, 0x568f, 0x578f, 0x588f, 0x598f, 0x5a8f,
	0xdde2, 0x5b8f, 0xdee2, 0x5c8f, 0x5d8f, 0x5e8f, 0x5f8f, 0x608f,
	0x618f, 0x628f, 0x638f, 0x648f, 0xc8db, 0x658f, 0xd3d1, 0xa2cd,
	0x668f, 0x678f, 0xa8bd, 0x688f, 0x698f, 0x6a8f, 0xc3de, 0xa5d8,
	0xaabf, 0xcddb, 0xecd2, 0xfac6, 0xaac5, 0x6b8f, 0x6c8f, 0x6d8f,
	0xc4de, 0x6e8f, 0xd7b1, 0xaedf, 0x6f8f, 0x708f, 0x718f, 0xbdca,
	0x728f, 0xb1df, 0x738f, 0xadb9, 0x748f, 0xfdd2, 0x758f, 0xa5b8,
	0xebba, 0x768f, 0x778f, 0xdab3, 0x788f, 0x798f, 0x7a8f, 0xdcb5,
	0xc5d5, 0x7b8f, 0x7c8f, 0x7d8f, 0x7e8f, 0xd6c3, 0xd2cf, 0xa1bb,
	0x808f, 0xf3e5, 0xf2e5, 0x818f, 0x828f, 0xf4e5, 0x838f, 0xe4cd,
	0x848f, 0xf5c8, 0x858f, 0x868f, 0x878f, 0x888f, 0x898f, 0x8a8f,
	0x8b8f, 0xafb5, 0xbfc7, 0x8c8f, 0xf6e5, 0x8d8f, 0x8e8f, 0x8f8f,
	0xb0ec, 0x908f, 0x918f, 0x928f, 0x938f, 0x948f, 0x958f, 0x968f,
	0x978f, 0x988f, 0x998f, 0x9a8f, 0x9b8f, 0x9c8f, 0x9d8f, 0x9e8f,
	0xe6e5, 0x9f8f, 0xe9b9, 0xb1b5, 0xa08f, 0xbcc2, 0xe8e5, 0xe7e5,
	0xe9e5, 0xa18f, 0xa28f, 0xa38f, 0xa48f, 0xcdd2, 0xa58f, 0xa68f,
	0xa78f, 0xeae1, 0xced0, 0xa88f, 0xaecd, 0xa98f, 0xe5d1, 0xaa8f,
	0xab8f, 0xcab2, 0xebb1, 0xac8f, 0xf2b1, 0xedc5, 0xad8f, 0xae8f,
	0xc3d5, 0xb0d3, 0xaf8f, 0xdce1, 0xb08f, 0xb18f, 0xb28f, 0xdde1,
	0xb38f, 0xdbd2, 0xb48f, 0xb9b3, 0xcbb1, 0xb58f, 0xb68f, 0xb78f,
	0xf9cd, 0xf7d5, 0xdee1, 0xb88f, 0xb6be, 0xfdb4, 0xb98f, 0xdfe1,
	0xdcba, 0xe0e1, 0xb2bb, 0xc9c2, 0xe1e1, 0xba8f, 0xbb8f, 0xbc8f,
	0xecd0, 0xbd8f, 0xbdcd, 0xbe8f, 0xbf8f, 0xe2e1, 0xc08f, 0xc3b5,
	0xc7c5, 0xe3e1, 0xc18f, 0xc28f, 0xe4e1, 0xc38f, 0xc48f, 0xc58f,
	0xc68f, 0xf9d3, 0xc78f, 0xc88f, 0xc98f, 0xca8f, 0xcb8f, 0xcc8f,
	0xe5e1, 0xcd8f, 0xadd1, 0xce8f, 0xcf8f, 0xe6e1, 0xa2ce, 0xd08f,
	0xd18f, 0xd28f, 0xd38f, 0xd48f, 0xd58f, 0xe7e1, 0xd68f, 0xc2b5,
	0xd78f, 0xd88f, 0xd98f, 0xda8f, 0xe8e1, 0xd5bb, 0xdb8f, 0xdc8f,
	0xdd8f, 0xde8f, 0xdf8f, 0xc4d0, 0xe0e2, 0xd8b1, 0xe4d2, 0xe08f,
	0xe18f, 0xe1e2, 0xe28f, 0xe38f, 0xc9bc, 0xccc8, 0xe48f, 0xe3e2,
	0xfeec, 0xfdec, 0xafdf, 0xe58f, 0xe68f, 0xe78f, 0xe2e2, 0xbed6,
	0xfccd, 0xa6c3, 0xe88f, 0xe98f, 0xea8f, 0xc3e3, 0xeb8f, 0xec8f,
	0xd2d6, 0xe7e2, 0xed8f, 0xee8f, 0xe8e2, 0xef8f, 0xf08f, 0xc7d3,
	0xf18f, 0xf28f, 0xece2, 0xecbf, 0xf38f, 0xede2, 0xe5e2, 0xf48f,
	0xf58f, 0xc0b3, 0xf68f, 0xf78f, 0xf88f, 0xeec4, 0xf98f, 0xfa8f,
	0xeee2, 0xfb8f, 0xfc8f, 0xc3d0, 0xfd8f, 0xf6ba, 0xe9e2, 0xdeb7,
	0xb3bb, 0xaccc, 0xcbcb, 0xe4e2, 0xe6e2, 0xeae2, 0xebe2, 0xfe8f,
	0x4090, 0x4190, 0xf7e2, 0x4290, 0x4390, 0xf4e2, 0xf5d4, 0xf3e2,
	0x4490, 0x4590, 0xadc5, 0x4690, 0xfad5, 0xc2c5, 0xc0b2, 0x4790,
	0x4890, 0xefe2, 0x4990, 0xf2e2, 0xafc1, 0xbccb, 0x4a90, 0x4b90,
	0xa1b5, 0xf9e2, 0x4c90, 0x4d90, 0x4e90, 0xb1bc, 0xf1e2, 0xd4d0,
	0xb9d4, 0xf5e2, 0xd6b9, 0xf6e2, 0x4f90, 0x5090, 0x5190, 0xd3c7,
	0x5290, 0x5390, 0x5490, 0x5590, 0x5690, 0xf0e2, 0x5790, 0x5890,
	0x5990, 0x5a90, 0x5b90, 0xdcd7, 0xa1ed, 0x5c90, 0x5d90, 0xf8e2,
	0x5e90, 0xa5ed, 0xfee2, 0xd1ca, 0x5f90, 0x6090, 0x6190, 0x6290,
	0x6390, 0x6490, 0x6590, 0xb5c1, 0x6690, 0xd0bb, 0x6790, 0x6890,
	0xd6bf, 0x6990, 0xe3ba, 0x6a90, 0x6b90, 0xa1cb, 0x6c90, 0x6d90,
	0x6e90, 0xa6ed, 0xa3ed, 0x6f90, 0x7090, 0xa2ed, 0x7190, 0x7290,
	0x7390, 0x7490, 0xd6bb, 0xa7ed, 0xf4d0, 0x7590, 0x7690, 0xa4ed,
	0xdeba, 0xf7b6, 0xa1e3, 0xb2b6, 0xf1cc, 0xa7b9, 0x7790, 0xa2cf,
	0xa1c7, 0x7890, 0x7990, 0xd2bf, 0x7a90, 0x7b90, 0xf1b6, 0x7c90,
	0xfae2, 0xfbe2, 0xfde2, 0xfce2, 0xd5c4, 0xa2e3, 0x7d90, 0xc1d3,
	0x7e90, 0x8090, 0x8190, 0xa7e3, 0xc4c7, 0x8290, 0x8390, 0x8490,
	0x8590, 0xa4cf, 0x8690, 0x8790, 0xa9e3, 0xb7ba, 0x8890, 0x8990,
	0x8a90, 0x8b90, 0xa8e3, 0x8c90, 0xdabb, 0x8d90, 0xa3e3, 0x8e90,
	0x8f90, 0x9090, 0xa4e3, 0xaae3, 0x9190, 0xa6e3, 0x9290, 0xf2ce,
	0xc6d3, 0x9390, 0x9490, 0xbcbb, 0x9590, 0x9690, 0xc3d4, 0x9790,
	0xfac4, 0x9890, 0x9990, 0xa8ed, 0xfcd0, 0xa5e3, 0x9a90, 0xf5c3,
	0x9b90, 0xade3, 0xafb1, 0x9c90, 0xb2e3, 0x9d90, 0x9e90, 0x9f90,
	0xc2bc, 0xa090, 0xa190, 0xace3, 0xbfb5, 0xa290, 0xa390, 0xa490,
	0xa590, 0xa690, 0xa790, 0xa890, 0xa990, 0xe9c7, 0xb0e3, 0xaa90,
	0xab90, 0xac90, 0xaabe, 0xefcd, 0xad90, 0xae90, 0xaf90, 0xb090,
	0xb190, 0xf3bb, 0xb290, 0xb390, 0xb490, 0xe8cc, 0xb590, 0xb690,
	0xafe3, 0xb790, 0xb1e3, 0xb890, 0xa7cf, 0xaee3, 0xb990, 0xa9ce,
	0xddbb, 0xba90, 0xbb90, 0xbc90, 0xbd90, 0xbe90, 0xebb5, 0xe5be,
	0xd2b2, 0xcdb3, 0xbf90, 0xb9b1, 0xabe3, 0xd1b2, 0xacb5, 0xdfb9,
	0xe8b6, 0xc090, 0xc190, 0xebcf, 0xb7e3, 0xc290, 0xccbb, 0xc390,
	0xc490, 0xc7c8, 0xcad0, 0xc590, 0xc690, 0xc790, 0xc890, 0xc990,
	0xb8e3, 0xeeb3, 0xca90, 0xcb90, 0xcc90, 0xcd90, 0xa9ed, 0xce90,
	0xfad3, 0xe4d3, 0xcf90, 0xd090, 0xd190, 0xaaed, 0xb9e3, 0xe2d2,
	0xd290, 0xd390, 0xd490, 0xd590, 0xd690, 0xb5e3, 0xd790, 0xd890,
	0xd990, 0xda90, 0xded3, 0xdb90, 0xdc90, 0xdd90, 0xde90, 0xd0b8,
	0xb3e3, 0xdf90, 0xe090, 0xb6e3, 0xdfb7, 0xe190, 0xb4e3, 0xa2c0,
	0xe290, 0xe390, 0xe490, 0xbae3, 0xe590, 0xe690, 0xe790, 0xe890,
	0xe990, 0xea90, 0xeb90, 0xec90, 0xed90, 0xee90, 0xef90, 0xf090,
	0xf190, 0xf290, 0xf390, 0xf490, 0xf590, 0xf690, 0xf790, 0xb8d4,
	0xf890, 0xf990, 0xfa90, 0xfb90, 0xfc90, 0xfd90, 0xfe90, 0x4091,
	0xc8b4, 0x4191, 0xbbe3, 0x4291, 0xc5bb, 0x4391, 0xf7c9, 0x4491,
	0x4591, 0xe5c9, 0x4691, 0x4791, 0x4891, 0xbdc4, 0x4991, 0x4a91,
	0x4b91, 0x4c91, 0x4d91, 0x4e91, 0x4f91, 0xabed, 0x5091, 0x5191,
	0x5291, 0x5391, 0xfdc2, 0x5491, 0x5591, 0x5691, 0x5791, 0xdbbb,
	0xaebf, 0x5891, 0x5991, 0x5a91, 0x5b91, 0x5c91, 0x5d91, 0x5e91,
	0xbfce, 0x5f91, 0x6091, 0x6191, 0x6291, 0xbce3, 0x6391, 0xb6bf,
	0x6491, 0x6591, 0x6691, 0x6791, 0x6891, 0x6991, 0x6a91, 0x6b91,
	0x6c91, 0x6d91, 0x6e91, 0x6f91, 0x7091, 0x7191, 0x7291, 0x7391,
	0x7491, 0x7591, 0x7691, 0xefb1, 0x7791, 0x7891, 0xf7d4, 0x7991,
	0x7a91, 0x7b91, 0x7c91, 0x7d91, 0xbee3, 0x7e91, 0x8091, 0x8191,
	0x8291, 0x8391, 0x8491, 0x8591, 0x8691, 0xaded, 0x8791, 0x8891,
	0x8991, 0x8a91, 0x8b91, 0x8c91, 0x8d91, 0x8e91, 0x8f91, 0xbfe3,
	0xa9ba, 0xaced, 0x9091, 0x9191, 0xbde3, 0x9291, 0x9391, 0x9491,
	0x9591, 0x9691, 0x9791, 0x9891, 0x9991, 0x9a91, 0x9b91, 0xc0e3,
	0x9c91, 0x9d91, 0x9e91, 0x9f91, 0xa091, 0xa191, 0xb6ba, 0xa291,
	0xa391, 0xa491, 0xaeb6, 0xa591, 0xa691, 0xa791, 0xa891, 0xa991,
	0xb8d0, 0xaa91, 0xc3b0, 0xaeed, 0xab91, 0xac91, 0xad91, 0xae91,
	0xaf91, 0xafed, 0xc1c0, 0xb091, 0xc1e3, 0xb191, 0xb291, 0xb391,
	0xb491, 0xb591, 0xb691, 0xb791, 0xb891, 0xb991, 0xba91, 0xbb91,
	0xbc91, 0xbd91, 0xbe91, 0xbf91, 0xc091, 0xc191, 0xb3c5, 0xc291,
	0xc391, 0xc491, 0xc591, 0xc691, 0xc791, 0xc891, 0xc991, 0xca91,
	0xcb91, 0xcc91, 0xcd91, 0xce91, 0xcf91, 0xc2e3, 0xd091, 0xd191,
	0xd291, 0xd391, 0xd491, 0xd591, 0xd691, 0xd791, 0xd891, 0xb2dc,
	0xd991, 0xda91, 0xdb91, 0xdc91, 0xdd91, 0xde91, 0xb0ed, 0xdf91,
	0xeab8, 0xe091, 0xecce, 0xa7ea, 0xe7d0, 0xf9ca, 0xd6c8, 0xb7cf,
	0xc9b3, 0xd2ce, 0xe4bd, 0xe191, 0xe291, 0xdee3, 0xf2bb, 0xa8ea,
	0xbdd5, 0xe391, 0xddc6, 0xa9ea, 0xe491, 0xe591, 0xe691, 0xaaea,
	0xe791, 0xacea, 0xabea, 0xe891, 0xaeea, 0xadea, 0xe991, 0xea91,
	0xeb91, 0xec91, 0xd8bd, 0xed91, 0xafea, 0xee91, 0xbec2, 0xef91,
	0xf091, 0xf191, 0xf291, 0xc1b4, 0xf7b4, 0xf391, 0xf491, 0xa7bb,
	0xf591, 0xf691, 0xf791, 0xf891, 0xf991, 0xe6ec, 0xe5ec, 0xbfb7,
	0xf9cb, 0xe2b1, 0xfa91, 0xe7ec, 0xfb91, 0xfc91, 0xfd91, 0xc8c9,
	0xe8ec, 0xe9ec, 0xfe91, 0xd6ca, 0xd0de, 0xc5b2, 0xfad4, 0x4092,
	0x4192, 0xcbc6, 0xc7b0, 0xf2b4, 0xd3c8, 0x4292, 0x4392, 0x4492,
	0xd0cd, 0x4592, 0x4692, 0xb8bf, 0x4792, 0x4892, 0x4992, 0x4a92,
	0x4b92, 0x4c92, 0x4d92, 0xdbbf, 0x4e92, 0x4f92, 0xa4c7, 0xb4d6,
	0x5092, 0xa9c0, 0xd1de, 0xa8c9, 0xefd1, 0xa4c5, 0xe7b0, 0xb6b3,
	0xc5c8, 0x5192, 0x5292, 0xe2b0, 0x5392, 0x5492, 0xf6b7, 0x5592,
	0x5692, 0xfac5, 0x5792, 0x5892, 0xf3b6, 0x5992, 0xd2d5, 0xd0b3,
	0xbcbc, 0x5a92, 0x5b92, 0x5c92, 0xadb3, 0x5d92, 0x5e92, 0x5f92,
	0x6092, 0xf1be, 0xd1b0, 0x6192, 0x6292, 0x6392, 0x6492, 0x6592,
	0x6692, 0xd6d2, 0xe3ca, 0xa5d7, 0x6792, 0xb6cd, 0xb6b6, 0xb9bf,
	0xdbd5, 0x6892, 0xa7b8, 0xd7c5, 0x6992, 0x6a92, 0x6b92, 0xd2de,
	0xd9bf, 0xd5c2, 0xc0c7, 0x6c92, 0xa4bb, 0xa8b1, 0x6d92, 0x6e92,
	0xeac5, 0x6f92, 0x7092, 0xfbc5, 0xa7cc, 0x7192, 0x7292, 0x7392,
	0x7492, 0xa7b1, 0x7592, 0x7692, 0x7792, 0xd6b5, 0x7892, 0x7992,
	0x7a92, 0xa8c4, 0x7b92, 0xd3de, 0xbad1, 0xe9b3, 0x7c92, 0xf2c3,
	0x7d92, 0x7e92, 0xf7b7, 0x8092, 0xf4d6, 0xa3b5, 0xf0b2, 0xb4c4,
	0xe9c4, 0xadc0, 0xd4de, 0x8192, 0xe8b0, 0xc4c5, 0xe0c1, 0x8292,
	0xd5b9, 0x8392, 0xdcbe, 0xd8cd, 0xceb0, 0x8492, 0xcfcd, 0xd6de,
	0xd0be, 0xbed7, 0xd5de, 0xd0d5, 0xddb0, 0x8592, 0x8692, 0xe2c4,
	0x8792, 0x8892, 0xa3c2, 0xf0bc, 0x8992, 0xb5d3, 0xb9c0, 0xa1c5,
	0xa6b2, 0xf1d4, 0x8a92, 0x8b92, 0xa8c0, 0xc3ca, 0xd7de, 0xfcd5,
	0x8c92, 0xb0b9, 0x8d92, 0xadc8, 0xa9cb, 0x8e92, 0xd9de, 0xbdbf,
	0x8f92, 0x9092, 0x9192, 0x9292, 0xb4c6, 0xa7d7, 0xb0ca, 0xc3c4,
	0x9392, 0xd6b3, 0xd2b9, 0x9492, 0x9592, 0x9692, 0x9792, 0xb8d6,
	0xfcea, 0xb4b0, 0x9892, 0x9992, 0x9a92, 0x9b92, 0xe6bf, 0x9c92,
	0x9d92, 0xf4cc, 0x9e92, 0x9f92, 0xa092, 0xa192, 0xdacd, 0xa292,
	0xa392, 0xa492, 0xbfd6, 0xcec2, 0xa592, 0xcece, 0xa2cc, 0xaed0,
	0xd3c4, 0xb2b5, 0xd8de, 0xf5d5, 0xb7bc, 0xd3bb, 0xa692, 0xa792,
	0xa4b0, 0xa892, 0xb2c5, 0xecb4, 0xa992, 0xaa92, 0xab92, 0xf1d5,
	0xac92, 0xad92, 0xfdea, 0xae92, 0xaf92, 0xb092, 0xb192, 0xb292,
	0xb392, 0xdade, 0xa6cd, 0xb492, 0xb592, 0xeccd, 0xb692, 0xb792,
	0xb892, 0xb992, 0xe6ce, 0xdcde, 0xba92, 0xb1cd, 0xa6c0, 0xbb92,
	0xbc92, 0xbdd7, 0xbd92, 0xdbde, 0xc6b0, 0xb4ba, 0xd3c9, 0xf3c4,
	0xe8be, 0xbe92, 0xbf92, 0xc092, 0xc192, 0xb6b2, 0xc292, 0xc392,
	0xc492, 0xc592, 0xc692, 0xc792, 0xc892, 0xc992, 0xccc0, 0xf0cb,
	0xca92, 0xf1bc, 0xbbbb, 0xb7b5, 0xcb92, 0xcc92, 0xcd92, 0xf5c5,
	0xce92, 0xe6de, 0xcf92, 0xd092, 0xd192, 0xe3de, 0xddbe, 0xd292,
	0xd392, 0xdfde, 0xd492, 0xd592, 0xd692, 0xd792, 0xb7b4, 0xddbd,
	0xd892, 0xd992, 0xe0de, 0xedc4, 0xda92, 0xdb92, 0xdc92, 0xdd92,
	0xc6cf, 0xde92, 0xe0b5, 0xdf92, 0xe092, 0xe192, 0xe292, 0xdeb6,
	0xdaca, 0xf4b5, 0xe5de, 0xe392, 0xc6d5, 0xe492, 0xe1de, 0xcdcc,
	0xfec6, 0xe592, 0xc5c5, 0xe692, 0xe792, 0xe892, 0xb4d2, 0xe992,
	0xf2be, 0xea92, 0xeb92, 0xec92, 0xed92, 0xee92, 0xef92, 0xf092,
	0xd3c2, 0xf192, 0xbdcc, 0xb8b3, 0xf292, 0xd3bd, 0xf392, 0xd8bf,
	0xc6cd, 0xdad1, 0xebb4, 0xf492, 0xe4de, 0xddde, 0xe7de, 0xf592,
	0xfeea, 0xf692, 0xf792, 0xb0c2, 0xe2de, 0xf892, 0xf992, 0xc0d6,
	0xa7b5, 0xfa92, 0xf4b2, 0xfb92, 0xe8de, 0xfc92, 0xf2de, 0xfd92,
	0xfe92, 0x4093, 0x4193, 0x4293, 0xedde, 0x4393, 0xf1de, 0x4493,
	0x4593, 0xe0c8, 0x4693, 0x4793, 0x4893, 0xe1d7, 0xefde, 0xe8c3,
	0xe1cc, 0x4993, 0xe5b2, 0x4a93, 0x4b93, 0x4c93, 0xbed2, 0x4d93,
	0x4e93, 0x4f93, 0x5093, 0x5193, 0x5293, 0x5393, 0xeede, 0x5493,
	0xebde, 0xd5ce, 0x5593, 0xa7b4, 0x5693, 0x5793, 0x5893, 0x5993,
	0x5a93, 0xabbf, 0xbebe, 0x5b93, 0x5c93, 0xd2bd, 0x5d93, 0x5e93,
	0x5f93, 0x6093, 0xe9de, 0x6193, 0xaed4, 0x6293, 0xdede, 0x6393,
	0xeade, 0x6493, 0x6593, 0x6693, 0x6793, 0xbfc0, 0x6893, 0xecde,
	0xf3b2, 0xe9b8, 0xa7c2, 0x6993, 0x6a93, 0xc1bd, 0x6b93, 0x6c93,
	0x6d93, 0x6e93, 0x6f93, 0xf5de, 0xf8de, 0x7093, 0x7193, 0xabb2,
	0xa4b4, 0x7293, 0x7393, 0xeab4, 0xa6c9, 0x7493, 0x7593, 0x7693,
	0x7793, 0x7893, 0x7993, 0xf6de, 0xd1cb, 0x7a93, 0xe3b8, 0x7b93,
	0xf7de, 0xfade, 0x7c93, 0x7d93, 0x7e93, 0x8093, 0xf9de, 0x8193,
	0x8293, 0x8393, 0xc2cc, 0x8493, 0xe1b0, 0xeeb4, 0x8593, 0x8693,
	0x8793, 0x8893, 0x8993, 0x8a93, 0xbae5, 0x8b93, 0x8c93, 0x8d93,
	0x8e93, 0x8f93, 0xafd0, 0x9093, 0x9193, 0xebb2, 0x9293, 0xa1eb,
	0x9393, 0xf4de, 0x9493, 0x9593, 0xe3c9, 0xf3de, 0xdab0, 0xa1d2,
	0xf7b1, 0x9693, 0xafcc, 0x9793, 0x9893, 0x9993, 0x9a93, 0x9b93,
	0x9c93, 0x9d93, 0xf0de, 0x9e93, 0xa4cb, 0x9f93, 0xa093, 0xa193,
	0xaad5, 0xa293, 0xa393, 0xa493, 0xa593, 0xa693, 0xfbde, 0xa793,
	0xa893, 0xa993, 0xaa93, 0xab93, 0xac93, 0xad93, 0xae93, 0xddb4,
	0xaf93, 0xa6c4, 0xb093, 0xb193, 0xb293, 0xfdde, 0xb393, 0xb493,
	0xb593, 0xb693, 0xb793, 0xb893, 0xb993, 0xba93, 0xbb93, 0xbc93,
	0xfec3, 0xa1c4, 0xa1df, 0xbd93, 0xbe93, 0xbf93, 0xc093, 0xc193,
	0xc293, 0xc393, 0xccc1, 0xc493, 0xfcde, 0xefbe, 0xc593, 0xb2c6,
	0xc693, 0xc793, 0xc893, 0xc993, 0xca93, 0xcb93, 0xcc93, 0xcd93,
	0xce93, 0xc5b3, 0xf6c8, 0xcf93, 0xd093, 0xbacb, 0xfede, 0xd193,
	0xd293, 0xa4df, 0xd393, 0xd493, 0xd593, 0xd693, 0xb2d7, 0xd793,
	0xd893, 0xd993, 0xda93, 0xdb93, 0xb7b3, 0xdc93, 0xdd93, 0xde93,
	0xdf93, 0xc3c1, 0xe093, 0xe193, 0xcbc7, 0xa5b2, 0xe9b4, 0xe293,
	0xabd7, 0xe393, 0xe493, 0xe593, 0xe693, 0xecc4, 0xe793, 0xa2df,
	0xa3df, 0xe893, 0xa5df, 0xe993, 0xb3ba, 0xea93, 0xeb93, 0xec93,
	0xa6df, 0xed93, 0xdec0, 0xee93, 0xef93, 0xc3c9, 0xf093, 0xf193,
	0xf293, 0xf393, 0xf493, 0xf593, 0xf693, 0xd9b2, 0xe6c7, 0xf793,
	0xa7df, 0xf893, 0xdcc7, 0xf993, 0xfa93, 0xfb93, 0xfc93, 0xa8df,
	0xa2eb, 0xfd93, 0xfe93, 0x4094, 0x4194, 0x4294, 0xd3cb, 0x4394,
	0x4494, 0x4594, 0xaadf, 0x4694, 0xa9df, 0x4794, 0xc1b2, 0x4894,
	0x4994, 0x4a94, 0x4b94, 0x4c94, 0x4d94, 0x4e94, 0x4f94, 0x5094,
	0x5194, 0x5294, 0x5394, 0x5494, 0x5594, 0x5694, 0x5794, 0x5894,
	0x5994, 0x5a94, 0x5b94, 0x5c94, 0x5d94, 0x5e94, 0x5f94, 0x6094,
	0xcac5, 0x6194, 0x6294, 0x6394, 0x6494, 0x6594, 0x6694, 0x6794,
	0x6894, 0xabdf, 0x6994, 0x6a94, 0x6b94, 0x6c94, 0x6d94, 0x6e94,
	0x6f94, 0x7094, 0xdcd4, 0x7194, 0x7294, 0x7394, 0x7494, 0x7594,
	0xc1c8, 0x7694, 0x7794, 0x7894, 0x7994, 0x7a94, 0x7b94, 0x7c94,
	0x7d94, 0x7e94, 0x8094, 0x8194, 0x8294, 0xacdf, 0x8394, 0x8494,
	0x8594, 0x8694, 0x8794, 0xf0be, 0x8894, 0x8994, 0xaddf, 0xa7d6,
	0x8a94, 0x8b94, 0x8c94, 0x8d94, 0xb7ea, 0xb6eb, 0xd5ca, 0x8e94,
	0xfcd8, 0xc4b8, 0x8f94, 0xa5b9, 0x9094, 0x9194, 0xc5b7, 0xfed5,
	0x9294, 0x9394, 0x9494, 0x9594, 0x9694, 0xcab9, 0x9794, 0x9894,
	0xa7d0, 0xcdf4, 0x9994, 0x9a94, 0xd0b5, 0x9b94, 0x9c94, 0xf4c3,
	0x9d94, 0xc8be, 0x9e94, 0x9f94, 0xa094, 0xb7eb, 0xbdb0, 0xa194,
	0xa294, 0xccbd, 0xa394, 0xb2c1, 0xa494, 0xd6b1, 0xa8b3, 0xa594,
	0xa694, 0xa794, 0xd2b8, 0xa2c9, 0xa894, 0xa994, 0xd8b6, 0xaa94,
	0xab94, 0xac94, 0xad94, 0xb8eb, 0xb4be, 0xae94, 0xaf94, 0xb094,
	0xfdca, 0xb194, 0xc3c7, 0xb294, 0xfbd5, 0xb394, 0xb494, 0xf3b7,
	0xb594, 0xb694, 0xb794, 0xb894, 0xb994, 0xba94, 0xbb94, 0xbc94,
	0xbd94, 0xbe94, 0xbf94, 0xc094, 0xc194, 0xc294, 0xc394, 0xc4ce,
	0xc494, 0xc594, 0xc694, 0xabd5, 0xf3b1, 0xc794, 0xc894, 0xc994,
	0xb3ec, 0xdfb0, 0xca94, 0xb5ec, 0xcb94, 0xcc94, 0xcd94, 0xb7b6,
	0xce94, 0xcfc1, 0xcf94, 0xfaf5, 0xb1d0, 0xd094, 0xd194, 0xe5d5,
	0xd294, 0xd3ce, 0xd394, 0xd494, 0xefbd, 0xe2b3, 0xd594, 0xabb8,
	0xd694, 0xb6d5, 0xd794, 0xbded, 0xd894, 0xcfb6, 0xd994, 0xb9cb,
	0xc2d0, 0xda94, 0xdb94, 0xdc94, 0xdd94, 0xde94, 0xdf94, 0xe094,
	0xe194, 0xbdb7, 0xe294, 0xe394, 0xb6ec, 0xa9ca, 0xe494, 0xe594,
	0xe694, 0xd4c5, 0xe794, 0xb9ec, 0xb8ec, 0xc3c2, 0xb7ec, 0xe894,
	0xe994, 0xea94, 0xeb94, 0xfdd0, 0xbaec, 0xec94, 0xbbec, 0xe5d7,
	0xed94, 0xee94, 0xbcec, 0xef94, 0xf094, 0xf194, 0xbdec, 0xecc6,
	0xf294, 0xf394, 0xf494, 0xf594, 0xf694, 0xf794, 0xf894, 0xf994,
	0xdece, 0xfa94, 0xc8bc, 0xfb94, 0xfc94, 0xd5c8, 0xa9b5, 0xc9be,
	0xbcd6, 0xe7d4, 0xfd94, 0xfe94, 0xaed1, 0xf1d0, 0xb8ea, 0xb9ea,
	0xbaea, 0xb5ba, 0x4095, 0x4195, 0x4295, 0x4395, 0xb1ca, 0xf5bf,
	0x4495, 0x4595, 0xfacd, 0x4695, 0x4795, 0x4895, 0x4995, 0x4a95,
	0xc0ea, 0x4b95, 0xbab0, 0xbeea, 0x4c95, 0x4d95, 0xa5c0, 0x4e95,
	0x4f95, 0x5095, 0xbbea, 0x5195, 0xfdb2, 0x5295, 0xf7c3, 0xe8bb,
	0x5395, 0x5495, 0x5595, 0xd7d2, 0xf4ce, 0xbfea, 0x5695, 0x5795,
	0x5895, 0xbcea, 0x5995, 0x5a95, 0x5b95, 0xc3ea, 0x5c95, 0xc7d0,
	0xb3d3, 0x5d95, 0x5e95, 0x5f95, 0x6095, 0xbab4, 0x6195, 0xc1c3,
	0xf2d7, 0x6295, 0x6395, 0x6495, 0x6595, 0xd1d5, 0x6695, 0xc7ca,
	0x6795, 0xc5ea, 0x6895, 0x6995, 0xc4ea, 0xc7ea, 0xc6ea, 0x6a95,
	0x6b95, 0x6c95, 0x6d95, 0x6e95, 0xe7d6, 0x6f95, 0xd4cf, 0x7095,
	0x7195, 0xcbea, 0x7295, 0xcebb, 0x7395, 0x7495, 0x7595, 0x7695,
	0x7795, 0x7895, 0x7995, 0xfabd, 0xcec9, 0x7a95, 0x7b95, 0xccea,
	0x7c95, 0x7d95, 0xb9c9, 0xfecf, 0xcaea, 0xced4, 0xcdea, 0xcfea,
	0x7e95, 0x8095, 0xedcd, 0x8195, 0x8295, 0x8395, 0x8495, 0xc9ea,
	0x8595, 0xceea, 0x8695, 0x8795, 0xeece, 0x8895, 0xdebb, 0x8995,
	0xbfb3, 0x8a95, 0x8b95, 0x8c95, 0x8d95, 0x8e95, 0xd5c6, 0xb0be,
	0xface, 0x8f95, 0x9095, 0x9195, 0xe7c7, 0x9295, 0xa7be, 0xd0ea,
	0x9395, 0x9495, 0xc7d6, 0x9595, 0x9695, 0x9795, 0xc0c1, 0x9895,
	0x9995, 0x9a95, 0xddd4, 0x9b95, 0xd1ea, 0x9c95, 0x9d95, 0xbecf,
	0x9e95, 0x9f95, 0xa095, 0xa195, 0xd2ea, 0xa295, 0xa395, 0xa495,
	0xa595, 0xeeca, 0xa695, 0xa795, 0xa895, 0xa995, 0xafc5, 0xb5b0,
	0xaa95, 0xab95, 0xac95, 0xad95, 0xae95, 0xd4ea, 0xaf95, 0xb095,
	0xb195, 0xb295, 0xb395, 0xb495, 0xb595, 0xb695, 0xb795, 0xd3ea,
	0xdff4, 0xb895, 0xb995, 0xba95, 0xbb95, 0xbc95, 0xbac4, 0xbd95,
	0xbe95, 0xbf95, 0xc095, 0xc195, 0xa9b1, 0xc295, 0xc395, 0xc495,
	0xc595, 0xdfe5, 0xc695, 0xc795, 0xc895, 0xc995, 0xd5ea, 0xca95,
	0xcb95, 0xcc95, 0xcd95, 0xce95, 0xcf95, 0xd095, 0xd195, 0xd295,
	0xd395, 0xd495, 0xd595, 0xd695, 0xd795, 0xd895, 0xd995, 0xda95,
	0xdb95, 0xdc95, 0xdd95, 0xde95, 0xdf95, 0xe095, 0xe195, 0xe295,
	0xe395, 0xefca, 0xe495, 0xd6ea, 0xd7ea, 0xd8c6, 0xe595, 0xe695,
	0xe795, 0xe895, 0xe995, 0xea95, 0xeb95, 0xec95, 0xd8ea, 0xed95,
	0xee95, 0xd9ea, 0xef95, 0xf095, 0xf195, 0xf295, 0xf395, 0xf495,
	0xbbd4, 0xf595, 0xfac7, 0xb7d2, 0xfcb8, 0xf695, 0xf795, 0xc2ea,
	0xf895, 0xdcb2, 0xf995, 0xfa95, 0xfcc2, 0xfb95, 0xf8d4, 0xe6cc,
	0xeed7, 0xfc95, 0xfd95, 0xfe95, 0x4096, 0x4196, 0x4296, 0x4396,
	0xc2d4, 0xd0d3, 0xc3eb, 0xf3c5, 0x4496, 0xfeb7, 0x4596, 0x4696,
	0xd4eb, 0x4796, 0x4896, 0x4996, 0xb7cb, 0xdeeb, 0x4a96, 0xcac0,
	0x4b96, 0x4c96, 0x4d96, 0xfbcd, 0x4e96, 0xafb3, 0x4f96, 0xdac6,
	0x5096, 0x5196, 0x5296, 0x5396, 0x5496, 0x5596, 0xfceb, 0x5696,
	0xbec4, 0x5796, 0xb4ce, 0xa9c4, 0xbeb1, 0xfdd4, 0x5896, 0xf5ca,
	0x5996, 0xecd6, 0x5a96, 0x5b96, 0xd3c6, 0xe4b6, 0x5c96, 0x5d96,
	0x5e96, 0x5f96, 0xfabb, 0x6096, 0x6196, 0xe0d0, 0x6296, 0x6396,
	0xb1c9, 0x6496, 0xd3d4, 0xa8c8, 0x6596, 0x6696, 0xcbb8, 0x6796,
	0xbee8, 0xbcc9, 0x6896, 0x6996, 0xbbe8, 0x6a96, 0xeec0, 0xd3d0,
	0xc4b2, 0xe5b4, 0x6b96, 0xbce8, 0x6c96, 0x6d96, 0xc8d5, 0x6e96,
	0x6f96, 0x7096, 0x7196, 0x7296, 0xc5b6, 0x7396, 0xbde8, 0xf8ca,
	0xdcb8, 0xf5cc, 0x7496, 0x7596, 0x7696, 0xb4c0, 0x7796, 0x7896,
	0xeed1, 0xbfe8, 0xc2e8, 0x7996, 0x7a96, 0xbcba, 0x7b96, 0xadb1,
	0xdcbd, 0x7c96, 0xbdea, 0xc3e8, 0x7d96, 0xc6e8, 0x7e96, 0xcbe8,
	0x8096, 0x8196, 0x8296, 0x8396, 0xcce8, 0x8496, 0xc9cb, 0xe5b0,
	0x8596, 0xabbc, 0x8696, 0x8796, 0xb9b9, 0x8896, 0x8996, 0xc1e8,
	0x8a96, 0xf7cd, 0x8b96, 0xcae8, 0x8c96, 0x8d96, 0x8e96, 0x8f96,
	0xf6ce, 0x9096, 0x9196, 0x9296, 0x9396, 0xedd5, 0x9496, 0xd6c1,
	0xc4e8, 0x9596, 0xb6c3, 0x9696, 0xfbb9, 0xa6d6, 0xc8e8, 0x9796,
	0x9896, 0x9996, 0xe0ca, 0xe6d4, 0x9a96, 0xc0e8, 0x9b96, 0xc5e8,
	0xc7e8, 0x9c96, 0xb9c7, 0xe3b7, 0x9d96, 0xc9e8, 0x9e96, 0xddbf,
	0xd2e8, 0x9f96, 0xa096, 0xd7e8, 0xa196, 0xd5e8, 0xdcbc, 0xcfbc,
	0xdbe8, 0xa296, 0xa396, 0xa496, 0xa596, 0xa696, 0xa796, 0xa896,
	0xa996, 0xdee8, 0xaa96, 0xdae8, 0xfab1, 0xab96, 0xac96, 0xad96,
	0xae96, 0xaf96, 0xb096, 0xb196, 0xb296, 0xb396, 0xb496, 0xd8b0,
	0xb3c4, 0xccb8, 0xe2c6, 0xbec8, 0xe1c8, 0xb596, 0xb696, 0xb796,
	0xcfe8, 0xd4e8, 0xd6e8, 0xb896, 0xf1b9, 0xd8e8, 0xf5d7, 0xb996,
	0xfbc4, 0xba96, 0xdce8, 0xbb96, 0xbc96, 0xe9b2, 0xbd96, 0xbe96,
	0xbf96, 0xd1e8, 0xc096, 0xc196, 0xedbc, 0xc296, 0xc396, 0xc2bf,
	0xcde8, 0xf9d6, 0xc496, 0xf8c1, 0xf1b2, 0xc596, 0xc696, 0xc796,
	0xc896, 0xc996, 0xca96, 0xcb96, 0xcc96, 0xdfe8, 0xcd96, 0xc1ca,
	0xd9e8, 0xce96, 0xcf96, 0xd096, 0xd196, 0xa4d5, 0xd296, 0xeab1,
	0xbbd5, 0xcee8, 0xd0e8, 0xb0b6, 0xd3e8, 0xd396, 0xdde8, 0xb8c0,
	0xd496, 0xf7ca, 0xd596, 0xa8cb, 0xd696, 0xd796, 0xdcc6, 0xf5c0,
	0xd896, 0xd996, 0xda96, 0xdb96, 0xdc96, 0xe9e8, 0xdd96, 0xde96,
	0xdf96, 0xa3d0, 0xe096, 0xe196, 0xe296, 0xe396, 0xe496, 0xe596,
	0xe696, 0xf2e8, 0xead6, 0xe796, 0xe896, 0xe996, 0xea96, 0xeb96,
	0xec96, 0xed96, 0xe0e8, 0xe1e8, 0xee96, 0xef96, 0xf096, 0xf9d1,
	0xcbba, 0xf9b8, 0xf196, 0xf296, 0xf1b8, 0xd4d4, 0xefe8, 0xf396,
	0xeee8, 0xece8, 0xf0b9, 0xd2cc, 0xe6e8, 0xa6ce, 0xf2bf, 0xf496,
	0xb8b0, 0xf1e8, 0xf0e8, 0xf596, 0xc0d7, 0xf696, 0xe4e8, 0xf796,
	0xa9cd, 0xa3c9, 0xf896, 0xb8bb, 0xdbbd, 0xeae8, 0xf996, 0xfa96,
	0xfb96, 0xfc96, 0xfd96, 0xfe96, 0x4097, 0x4197, 0x4297, 0x4397,
	0xe2e8, 0xe3e8, 0xe5e8, 0xb5b5, 0xe7e8, 0xc5c7, 0xebe8, 0xede8,
	0xb0bd, 0xaed7, 0x4497, 0xf8e8, 0x4597, 0x4697, 0x4797, 0x4897,
	0x4997, 0x4a97, 0x4b97, 0x4c97, 0xf5e8, 0x4d97, 0xb0cd, 0xf6e8,
	0x4e97, 0x4f97, 0x5097, 0x5197, 0x5297, 0x5397, 0x5497, 0x5597,
	0x5697, 0xbac1, 0x5797, 0xe8e8, 0x5897, 0xb7c3, 0xf0b0, 0x5997,
	0x5a97, 0x5b97, 0x5c97, 0x5d97, 0x5e97, 0x5f97, 0x6097, 0xf4e8,
	0x6197, 0x6297, 0x6397, 0xf7e8, 0x6497, 0x6597, 0x6697, 0xa3b9,
	0x6797, 0x6897, 0x6997, 0x6a97, 0x6b97, 0x6c97, 0x6d97, 0x6e97,
	0x6f97, 0x7097, 0xd2c9, 0x7197, 0x7297, 0x7397, 0xcec3, 0xe0ce,
	0xe6c0, 0x7497, 0x7597, 0x7697, 0x7797, 0xf3cb, 0x7897, 0xddcc,
	0xb5d0, 0x7997, 0x7a97, 0xe1ca, 0x7b97, 0xf3e8, 0x7c97, 0x7d97,
	0x7e97, 0x8097, 0x8197, 0x8297, 0x8397, 0x8497, 0x8597, 0x8697,
	0xecbc, 0x8797, 0xf9e8, 0x8897, 0x8997, 0x8a97, 0x8b97, 0x8c97,
	0x8d97, 0xdec3, 0x8e97, 0xe5c6, 0x8f97, 0xf7b9, 0x9097, 0x9197,
	0x9297, 0x9397, 0xf4b0, 0x9497, 0x9597, 0xd8d7, 0x9697, 0x9797,
	0xacbc, 0x9897, 0xefc5, 0x9997, 0x9a97, 0x9b97, 0x9c97, 0x9d97,
	0xc4cc, 0x9e97, 0x9f97, 0xa6e9, 0xa097, 0xa197, 0xa297, 0xa397,
	0xa497, 0xa597, 0xa697, 0xa797, 0xa897, 0xa997, 0xadc9, 0xaa97,
	0xa2e9, 0xe2c0, 0xab97, 0xac97, 0xad97, 0xc3bf, 0xae97, 0xaf97,
	0xb097, 0xfee8, 0xd7b9, 0xb197, 0xfbe8, 0xb297, 0xb397, 0xb497,
	0xb597, 0xa4e9, 0xb697, 0xb797, 0xb897, 0xced2, 0xb997, 0xba97,
	0xbb97, 0xbc97, 0xbd97, 0xa3e9, 0xbe97, 0xb2d6, 0xb5d7, 0xbf97,
	0xa7e9, 0xc097, 0xb7bd, 0xc197, 0xc297, 0xc397, 0xc497, 0xc597,
	0xc697, 0xc797, 0xc897, 0xc997, 0xca97, 0xcb97, 0xcc97, 0xfce8,
	0xfde8, 0xcd97, 0xce97, 0xcf97, 0xa1e9, 0xd097, 0xd197, 0xd297,
	0xd397, 0xd497, 0xd597, 0xd697, 0xd797, 0xd6cd, 0xd897, 0xd997,
	0xacd2, 0xda97, 0xdb97, 0xdc97, 0xb2e9, 0xdd97, 0xde97, 0xdf97,
	0xe097, 0xa9e9, 0xe197, 0xe297, 0xe397, 0xaab4, 0xe497, 0xbbb4,
	0xe597, 0xe697, 0xabe9, 0xe797, 0xe897, 0xe997, 0xea97, 0xeb97,
	0xec97, 0xed97, 0xee97, 0xef97, 0xf097, 0xf197, 0xf297, 0xf397,
	0xf497, 0xf597, 0xf697, 0xf797, 0xa8d0, 0xf897, 0xf997, 0xa5e9,
	0xfa97, 0xfb97, 0xfeb3, 0xfc97, 0xfd97, 0xace9, 0xe3c0, 0xfe97,
	0xaae9, 0x4098, 0x4198, 0xb9e9, 0x4298, 0x4398, 0xb8e9, 0x4498,
	0x4598, 0x4698, 0x4798, 0xaee9, 0x4898, 0x4998, 0xfae8, 0x4a98,
	0x4b98, 0xa8e9, 0x4c98, 0x4d98, 0x4e98, 0x4f98, 0x5098, 0xacbf,
	0xb1e9, 0xbae9, 0x5198, 0x5298, 0xa5c2, 0x5398, 0x5498, 0x5598,
	0xafe9, 0x5698, 0xc5b8, 0x5798, 0xade9, 0x5898, 0xdcd3, 0xb4e9,
	0xb5e9, 0xb7e9, 0x5998, 0x5a98, 0x5b98, 0xc7e9, 0x5c98, 0x5d98,
	0x5e98, 0x5f98, 0x6098, 0x6198, 0xc6c0, 0xc5e9, 0x6298, 0x6398,
	0xb0e9, 0x6498, 0x6598, 0xbbe9, 0xf1b0, 0x6698, 0x6798, 0x6898,
	0x6998, 0x6a98, 0x6b98, 0x6c98, 0x6d98, 0x6e98, 0x6f98, 0xbce9,
	0xa5d5, 0x7098, 0x7198, 0xbee9, 0x7298, 0xbfe9, 0x7398, 0x7498,
	0x7598, 0xc1e9, 0x7698, 0x7798, 0xf1c1, 0x7898, 0x7998, 0xb6c8,
	0x7a98, 0x7b98, 0x7c98, 0xbde9, 0x7d98, 0x7e98, 0x8098, 0x8198,
	0x8298, 0xc2e9, 0x8398, 0x8498, 0x8598, 0x8698, 0x8798, 0x8898,
	0x8998, 0x8a98, 0xc3e9, 0x8b98, 0xb3e9, 0x8c98, 0xb6e9, 0x8d98,
	0xb1bb, 0x8e98, 0x8f98, 0x9098, 0xc0e9, 0x9198, 0x9298, 0x9398,
	0x9498, 0x9598, 0x9698, 0xf7bc, 0x9798, 0x9898, 0x9998, 0xc4e9,
	0xc6e9, 0x9a98, 0x9b98, 0x9c98, 0x9d98, 0x9e98, 0x9f98, 0xa098,
	0xa198, 0xa298, 0xa398, 0xa498, 0xa598, 0xcae9, 0xa698, 0xa798,
	0xa898, 0xa998, 0xcee9, 0xaa98, 0xab98, 0xac98, 0xad98, 0xae98,
	0xaf98, 0xb098, 0xb198, 0xb298, 0xb398, 0xdbb2, 0xb498, 0xc8e9,
	0xb598, 0xb698, 0xb798, 0xb898, 0xb998, 0xba98, 0xbb98, 0xbc98,
	0xbd98, 0xbe98, 0xaeb7, 0xbf98, 0xc098, 0xc198, 0xc298, 0xc398,
	0xc498, 0xc598, 0xc698, 0xc798, 0xc898, 0xc998, 0xca98, 0xcbe9,
	0xcce9, 0xcb98, 0xcc98, 0xcd98, 0xce98, 0xcf98, 0xd098, 0xc1d5,
	0xd198, 0xa3c4, 0xd298, 0xd398, 0xd498, 0xd598, 0xd698, 0xd798,
	0xd8e9, 0xd898, 0xe1ba, 0xd998, 0xda98, 0xdb98, 0xdc98, 0xc9e9,
	0xdd98, 0xa3d3, 0xde98, 0xdf98, 0xe098, 0xd4e9, 0xe198, 0xe298,
	0xe398, 0xe498, 0xe598, 0xe698, 0xe798, 0xd7e9, 0xd0e9, 0xe898,
	0xe998, 0xea98, 0xeb98, 0xec98, 0xcfe9, 0xed98, 0xee98, 0xc1c7,
	0xef98, 0xf098, 0xf198, 0xf298, 0xf398, 0xf498, 0xf598, 0xf698,
	0xd2e9, 0xf798, 0xf898, 0xf998, 0xfa98, 0xfb98, 0xfc98, 0xfd98,
	0xd9e9, 0xc8b3, 0xfe98, 0xd3e9, 0x4099, 0x4199, 0x4299, 0x4399,
	0x4499, 0xf0cf, 0x4599, 0x4699, 0x4799, 0xcde9, 0x4899, 0x4999,
	0x4a99, 0x4b99, 0x4c99, 0x4d99, 0x4e99, 0x4f99, 0x5099, 0x5199,
	0x5299, 0xf7b3, 0x5399, 0x5499, 0x5599, 0x5699, 0x5799, 0x5899,
	0x5999, 0xd6e9, 0x5a99, 0x5b99, 0xdae9, 0x5c99, 0x5d99, 0x5e99,
	0xb4cc, 0x5f99, 0x6099, 0x6199, 0xadcf, 0x6299, 0x6399, 0x6499,
	0x6599, 0x6699, 0x6799, 0x6899, 0x6999, 0x6a99, 0xd5e9, 0x6b99,
	0xdce9, 0xdbe9, 0x6c99, 0x6d99, 0x6e99, 0x6f99, 0x7099, 0xdee9,
	0x7199, 0x7299, 0x7399, 0x7499, 0x7599, 0x7699, 0x7799, 0x7899,
	0xd1e9, 0x7999, 0x7a99, 0x7b99, 0x7c99, 0x7d99, 0x7e99, 0x8099,
	0x8199, 0xdde9, 0x8299, 0xdfe9, 0xcac3, 0x8399, 0x8499, 0x8599,
	0x8699, 0x8799, 0x8899, 0x8999, 0x8a99, 0x8b99, 0x8c99, 0x8d99,
	0x8e99, 0x8f99, 0x9099, 0x9199, 0x9299, 0x9399, 0x9499, 0x9599,
	0x9699, 0x9799, 0x9899, 0x9999, 0x9a99, 0x9b99, 0x9c99, 0x9d99,
	0x9e99, 0x9f99, 0xa099, 0xa199, 0xa299, 0xa399, 0xa499, 0xa599,
	0xa699, 0xa799, 0xa899, 0xa999, 0xaa99, 0xab99, 0xac99, 0xad99,
	0xae99, 0xaf99, 0xb099, 0xb199, 0xb299, 0xb399, 0xb499, 0xb599,
	0xb699, 0xb799, 0xb899, 0xb999, 0xba99, 0xbb99, 0xbc99, 0xbd99,
	0xbe99, 0xbf99, 0xc099, 0xc199, 0xc299, 0xc399, 0xc499, 0xc599,
	0xc699, 0xc799, 0xc899, 0xc999, 0xca99, 0xcb99, 0xcc99, 0xcd99,
	0xce99, 0xcf99, 0xd099, 0xd199, 0xd299, 0xd399, 0xd499, 0xd599,
	0xd699, 0xd799, 0xd899, 0xd999, 0xda99, 0xdb99, 0xdc99, 0xdd99,
	0xde99, 0xdf99, 0xe099, 0xe199, 0xe299, 0xe399, 0xe499, 0xe599,
	0xe699, 0xe799, 0xe899, 0xe999, 0xea99, 0xeb99, 0xec99, 0xed99,
	0xee99, 0xef99, 0xf099, 0xf199, 0xf299, 0xf399, 0xf499, 0xf599,
	0xb7c7, 0xceb4, 0xb6bb, 0xc0d0, 0xa3ec, 0xf699, 0xf799, 0xb7c5,
	0xf899, 0xf999, 0xfa99, 0xfb99, 0xfc99, 0xfd99, 0xfe99, 0x409a,
	0x419a, 0x429a, 0xfbd3, 0x439a, 0x449a, 0x459a, 0x469a, 0xa4ec,
	0x479a, 0xa5ec, 0xdbc6, 0x489a, 0x499a, 0x4a9a, 0xeebf, 0x4b9a,
	0x4c9a, 0x4d9a, 0x4e9a, 0xa6ec, 0x4f9a, 0x509a, 0xa7ec, 0xaad0,
	0x519a, 0xb8c7, 0x529a, 0x539a, 0xe8b8, 0x549a, 0x559a, 0x569a,
	0x579a, 0x589a, 0x599a, 0x5a9a, 0x5b9a, 0x5c9a, 0x5d9a, 0x5e9a,
	0x5f9a, 0xa8ec, 0x609a, 0x619a, 0x629a, 0x639a, 0x649a, 0x659a,
	0x669a, 0x679a, 0xb9d6, 0xfdd5, 0xcbb4, 0xbdb2, 0xe4ce, 0xe7c6,
	0x689a, 0x699a, 0xe1cd, 0x6a9a, 0x6b9a, 0x6c9a, 0x6d9a, 0x6e9a,
	0x6f9a, 0x709a, 0x719a, 0x729a, 0x739a, 0x749a, 0x759a, 0x769a,
	0x779a, 0xf5b4, 0x789a, 0xc0cb, 0xdfbc, 0x799a, 0x7a9a, 0x7b9a,
	0x7c9a, 0xe2e9, 0xe3e9, 0xead1, 0xe5e9, 0x7d9a, 0xf9b4, 0xe4e9,
	0x7e9a, 0xb3d1, 0xe2ca, 0xd0b2, 0x809a, 0xe8e9, 0x819a, 0x829a,
	0x839a, 0x849a, 0xe6e9, 0xe7e9, 0x859a, 0x869a, 0xb3d6, 0x879a,
	0x889a, 0x899a, 0xe9e9, 0xeae9, 0x8a9a, 0x8b9a, 0x8c9a, 0x8d9a,
	0x8e9a, 0xebe9, 0x8f9a, 0x909a, 0x919a, 0x929a, 0x939a, 0x949a,
	0x959a, 0x969a, 0xece9, 0x979a, 0x989a, 0x999a, 0x9a9a, 0x9b9a,
	0x9c9a, 0x9d9a, 0x9e9a, 0xafec, 0xb9c5, 0xceb6, 0x9f9a, 0xf3d2,
	0xa09a, 0xa19a, 0xa29a, 0xa39a, 0xa49a, 0xa59a, 0xa69a, 0xeeb5,
	0xa79a, 0xd9bb, 0xb1ec, 0xa89a, 0xa99a, 0xe3d2, 0xaa9a, 0xab9a,
	0xac9a, 0xad9a, 0xae9a, 0xe3ce, 0xaf9a, 0xb8c4, 0xb09a, 0xbfc3,
	0xb19a, 0xb29a, 0xbeb6, 0xb9d8, 0xc8b1, 0xcfb1, 0xd1b1, 0xfec5,
	0xb39a, 0xd0b1, 0xb49a, 0xabc3, 0xb59a, 0xb69a, 0xb79a, 0xb89a,
	0xb99a, 0xb1d5, 0xba9a, 0xbb9a, 0xbc9a, 0xbd9a, 0xbe9a, 0xbf9a,
	0xc09a, 0xc19a, 0xa4eb, 0xc1ba, 0xc29a, 0xc39a, 0xc49a, 0xbacc,
	0xc59a, 0xc69a, 0xc79a, 0xa5eb, 0xc89a, 0xa7eb, 0xc99a, 0xca9a,
	0xcb9a, 0xa8eb, 0xcc9a, 0xcd9a, 0xce9a, 0xa6eb, 0xcf9a, 0xd09a,
	0xd19a, 0xd29a, 0xd39a, 0xd49a, 0xd59a, 0xa9eb, 0xabeb, 0xaaeb,
	0xd69a, 0xd79a, 0xd89a, 0xd99a, 0xda9a, 0xaceb, 0xdb9a, 0xcfca,
	0xb5d8, 0xf1c3, 0xdc9a, 0xa5c3, 0xf8c6, 0xadeb, 0xcac4, 0xdd9a,
	0xaeeb, 0xafeb, 0xb0eb, 0xd5b7, 0xde9a, 0xdf9a, 0xe09a, 0xfab7,
	0xe19a, 0xb1eb, 0xe2c7, 0xe29a, 0xb3eb, 0xe39a, 0xa4ba, 0xf5d1,
	0xb1b0, 0xb2eb, 0xb4eb, 0xe49a, 0xe59a, 0xe69a, 0xaab5, 0xc8c2,
	0xe8c7, 0xe79a, 0xb5eb, 0xe89a, 0xaecb, 0xdfe3, 0xe99a, 0xea9a,
	0xc0d3, 0xeb9a, 0xec9a, 0xed9a, 0xee9a, 0xdbd9, 0xef9a, 0xf09a,
	0xa1cd, 0xadd6, 0xf3c7, 0xf19a, 0xf29a, 0xf39a, 0xe0d9, 0xe3bb,
	0xf49a, 0xbaba, 0xe2e3, 0xf59a, 0xf69a, 0xf79a, 0xf89a, 0xf99a,
	0xabcf, 0xfa9a, 0xfb9a, 0xfc9a, 0xe0e3, 0xc7c9, 0xfd9a, 0xb9ba,
	0xfe9a, 0x409b, 0x419b, 0xb4d1, 0xe1e3, 0xeac8, 0xafb9, 0xadbd,
	0xd8b3, 0xdbce, 0x429b, 0x439b, 0xc0cc, 0x449b, 0x459b, 0x469b,
	0xe8e3, 0xe9e3, 0xf4cd, 0x479b, 0x489b, 0x499b, 0x4a9b, 0x4b9b,
	0xadcc, 0x4c9b, 0xb3bc, 0x4d9b, 0xeae3, 0x4e9b, 0xebe3, 0x4f9b,
	0x509b, 0xdad0, 0x519b, 0x529b, 0x539b, 0xfbc6, 0xdab7, 0x549b,
	0x559b, 0xdfc7, 0xcad2, 0xd6ce, 0x569b, 0xe4e3, 0xece3, 0x579b,
	0xf2c9, 0xc1b3, 0x589b, 0x599b, 0xe7e3, 0x5a9b, 0x5b9b, 0xe3c6,
	0xe5e3, 0x5c9b, 0x5d9b, 0xb3ed, 0xe6e3, 0x5e9b, 0x5f9b, 0x609b,
	0x619b, 0xb3c9, 0x629b, 0xe6c5, 0x639b, 0x649b, 0x659b, 0xb5b9,
	0x669b, 0xbbc3, 0x679b, 0xe3e3, 0xbdc5, 0xa4c1, 0xd9c2, 0xd7b2,
	0x689b, 0xede3, 0xa6bb, 0xadc4, 0x699b, 0xf0e3, 0xdabe, 0x6a9b,
	0x6b9b, 0xfbe3, 0xf5e3, 0xd3ba, 0x6c9b, 0x6d9b, 0x6e9b, 0x6f9b,
	0xd0b7, 0xcdd3, 0x709b, 0xced6, 0xd3d5, 0xc1b9, 0xb4d5, 0xd8d1,
	0x719b, 0x729b, 0x739b, 0x749b, 0xb9d0, 0xf6c7, 0x759b, 0x769b,
	0x779b, 0xaac8, 0xb4b2, 0x789b, 0xdac3, 0x799b, 0x7a9b, 0x7b9b,
	0xeee3, 0x7c9b, 0x7d9b, 0xfce3, 0xefe3, 0xa8b7, 0xf7e3, 0xf4e3,
	0x7e9b, 0x809b, 0x819b, 0xbab7, 0x829b, 0x839b, 0xa2c5, 0x849b,
	0xf6e3, 0xddc5, 0xa8b2, 0xfcc6, 0x859b, 0xe0c4, 0x869b, 0x879b,
	0xa2d7, 0x889b, 0xe1c0, 0xf9e3, 0x899b, 0x8a9b, 0xfae3, 0xfde3,
	0xa9cc, 0xf3e3, 0x8b9b, 0xbed3, 0x8c9b, 0xc3b1, 0xb4ed, 0xf1e3,
	0xf2e3, 0x8d9b, 0xf8e3, 0xbad0, 0xc3c6, 0xf3d4, 0xfee3, 0x8e9b,
	0x8f9b, 0xe0bd, 0x909b, 0x919b, 0xa7e4, 0x929b, 0x939b, 0xa6e4,
	0x949b, 0x959b, 0x969b, 0xf3d1, 0xa3e4, 0x979b, 0xa9e4, 0x989b,
	0x999b, 0x9a9b, 0xf7c8, 0x9b9b, 0x9c9b, 0x9d9b, 0x9e9b, 0xb4cf,
	0x9f9b, 0xa8e4, 0xaee4, 0xe5c2, 0xa09b, 0xa19b, 0xb4b6, 0xa29b,
	0xa39b, 0xa49b, 0xa59b, 0xa69b, 0xa79b, 0xf2bd, 0xa89b, 0xa2e4,
	0xa99b, 0xaa9b, 0xe9ba, 0xaae4, 0xab9b, 0xac9b, 0xace4, 0xad9b,
	0xae9b, 0xfdb6, 0xded6, 0xb2e4, 0xaf9b, 0xade4, 0xb09b, 0xb19b,
	0xb29b, 0xa1e4, 0xb39b, 0xeebb, 0xddcd, 0xa2c7, 0xc9c5, 0xb49b,
	0xb59b, 0xf7c1, 0xb69b, 0xa4e4, 0xb79b, 0xb3c7, 0xacbd, 0xbdbd,
	0xa5e4, 0xb89b, 0xc7d7, 0xe2b2, 0xb99b, 0xabe4, 0xc3bc, 0xafe4,
	0xba9b, 0xebbb, 0xb0e4, 0xa8c5, 0xb1e4, 0xbb9b, 0xbc9b, 0xbd9b,
	0xbe9b, 0xe3d5, 0xa3bf, 0xbf9b, 0xbae4, 0xc09b, 0xb7e4, 0xc19b,
	0xbbe4, 0xc29b, 0xc39b, 0xbde4, 0xc49b, 0xc59b, 0xd6c6, 0xc69b,
	0xc79b, 0xc6ba, 0xcbc0, 0xc89b, 0xc99b, 0xca9b, 0xa1b8, 0xb4e4,
	0xcb9b, 0xcc9b, 0xcd9b, 0xce9b, 0xa1d4, 0xcf9b, 0xd09b, 0xa3ba,
	0xfebd, 0xd19b, 0xd29b, 0xd39b, 0xbce4, 0xd49b, 0xd59b, 0xd69b,
	0xd79b, 0xd89b, 0xbfcd, 0xd99b, 0xda9b, 0xf9c4, 0xdb9b, 0xdc9b,
	0xfbcf, 0xe6c9, 0xdd9b, 0xde9b, 0xbfd3, 0xdf9b, 0xd1cf, 0xe09b,
	0xe19b, 0xb3e4, 0xe29b, 0xb8e4, 0xb9e4, 0xe9cc, 0xe39b, 0xe49b,
	0xe59b, 0xe69b, 0xe79b, 0xcecc, 0xe89b, 0xd4c0, 0xb5e4, 0xb0c1,
	0xb6e4, 0xd0ce, 0xe99b, 0xc1bb, 0xd3b5, 0xea9b, 0xf3c8, 0xa7bd,
	0xc7d5, 0xacc9, 0xa2b8, 0xcae4, 0xeb9b, 0xec9b, 0xcce4, 0xc4d1,
	0xed9b, 0xee9b, 0xbad2, 0xef9b, 0xf09b, 0xadba, 0xf19b, 0xf29b,
	0xd4ba, 0xf39b, 0xf49b, 0xf59b, 0xf69b, 0xf79b, 0xf89b, 0xc3e4,
	0xedb5, 0xf99b, 0xfa9b, 0xfb9b, 0xcdd7, 0xc0e4, 0xfdcf, 0xbfe4,
	0xfc9b, 0xfd9b, 0xfe9b, 0xdcc1, 0xcacc, 0x409c, 0x419c, 0x429c,
	0x439c, 0xe7ca, 0x449c, 0x459c, 0x469c, 0x479c, 0xd7c4, 0x489c,
	0xd4cc, 0xc8e4, 0x499c, 0x4a9c, 0x4b9c, 0xc7e4, 0xc1e4, 0x4c9c,
	0xc4e4, 0xadb5, 0x4d9c, 0x4e9c, 0xd9d3, 0x4f9c, 0xc6e4, 0x509c,
	0x519c, 0x529c, 0x539c, 0xf9d2, 0xe3b4, 0x549c, 0xb4bb, 0x559c,
	0x569c, 0xeec9, 0x579c, 0xbeb4, 0x589c, 0x599c, 0x5a9c, 0xecbb,
	0x5b9c, 0xcdd1, 0x5c9c, 0xedcc, 0xb5ed, 0x5d9c, 0x5e9c, 0x5f9c,
	0x609c, 0x619c, 0x629c, 0x639c, 0x649c, 0xe5c7, 0x659c, 0x669c,
	0x679c, 0x689c, 0xa8d4, 0x699c, 0xcbe4, 0xd5d7, 0xc2e4, 0x6a9c,
	0xa5bd, 0xc5e4, 0x6b9c, 0x6c9c, 0xe6d3, 0x6d9c, 0xc9e4, 0xf8c9,
	0x6e9c, 0x6f9c, 0xbee4, 0x709c, 0x719c, 0xe5d3, 0x729c, 0x739c,
	0xfec7, 0xc9b6, 0x749c, 0xfcd4, 0xb3b2, 0xd7e4, 0x759c, 0x769c,
	0x779c, 0xc2ce, 0x789c, 0xcde4, 0x799c, 0xbcce, 0x7a9c, 0xdbb8,
	0x7b9c, 0x7c9c, 0xd6e4, 0x7d9c, 0xcabf, 0x7e9c, 0x809c, 0x819c,
	0xced3, 0x829c, 0xecc3, 0x839c, 0x849c, 0x859c, 0x869c, 0x879c,
	0x889c, 0x899c, 0x8a9c, 0xc8c5, 0xd8e4, 0x8b9c, 0x8c9c, 0x8d9c,
	0x8e9c, 0x8f9c, 0x909c, 0x919c, 0x929c, 0xc4cd, 0xcfe4, 0x939c,
	0x949c, 0x959c, 0x969c, 0xd4e4, 0xd5e4, 0x979c, 0xfeba, 0x989c,
	0xe6cf, 0x999c, 0x9a9c, 0xbfd5, 0x9b9c, 0x9c9c, 0x9d9c, 0xd2e4,
	0x9e9c, 0x9f9c, 0xa09c, 0xa19c, 0xa29c, 0xa39c, 0xa49c, 0xa59c,
	0xa69c, 0xa79c, 0xa89c, 0xd0e4, 0xa99c, 0xaa9c, 0xcee4, 0xab9c,
	0xac9c, 0xad9c, 0xae9c, 0xaf9c, 0xb09c, 0xb19c, 0xb29c, 0xb39c,
	0xb49c, 0xb59c, 0xb69c, 0xb79c, 0xb89c, 0xb99c, 0xe5cd, 0xaaca,
	0xba9c, 0xbb9c, 0xbc9c, 0xa3c0, 0xbd9c, 0xa6bd, 0xd3e4, 0xbe9c,
	0xbf9c, 0xc8b8, 0xc09c, 0xc19c, 0xc29c, 0xc39c, 0xc49c, 0xe7e4,
	0xb4d4, 0xc59c, 0xc69c, 0xc79c, 0xc89c, 0xc99c, 0xca9c, 0xcb9c,
	0xdbe4, 0xcc9c, 0xcd9c, 0xce9c, 0xefc1, 0xcf9c, 0xd09c, 0xe9e4,
	0xd19c, 0xd29c, 0xe7d2, 0xd39c, 0xd49c, 0xdfe4, 0xd59c, 0xe0e4,
	0xd69c, 0xd79c, 0xaacf, 0xd89c, 0xd99c, 0xda9c, 0xdb9c, 0xddcb,
	0xdc9c, 0xdae4, 0xd1e4, 0xdd9c, 0xe5e4, 0xde9c, 0xdcc8, 0xe3e4,
	0xdf9c, 0xe09c, 0xe7c4, 0xe2e4, 0xe19c, 0xe1e4, 0xe29c, 0xe39c,
	0xe49c, 0xfcb3, 0xe8e4, 0xe59c, 0xe69c, 0xe79c, 0xe89c, 0xe1b5,
	0xe99c, 0xea9c, 0xeb9c, 0xccd7, 0xec9c, 0xed9c, 0xee9c, 0xe6e4,
	0xef9c, 0xacbb, 0xf09c, 0xd2d7, 0xcfcc, 0xf8eb, 0xf19c, 0xe4e4,
	0xf29c, 0xf39c, 0xf6b9, 0xf49c, 0xf59c, 0xf69c, 0xcdd6, 0xd9e4,
	0xdce4, 0xfac2, 0xdee4, 0xf79c, 0xcbc2, 0xc4c0, 0xd0c2, 0xf89c,
	0xf5b1, 0xb2cc, 0xf99c, 0xfa9c, 0xfb9c, 0xfc9c, 0xfd9c, 0xfe9c,
	0x409d, 0x419d, 0x429d, 0x439d, 0xceb5, 0x449d, 0x459d, 0x469d,
	0x479d, 0xefe4, 0x489d, 0x499d, 0x4a9d, 0x4b9d, 0x4c9d, 0x4d9d,
	0x4e9d, 0x4f9d, 0xafc6, 0x509d, 0x519d, 0x529d, 0xe1c6, 0x539d,
	0x549d, 0xf5e4, 0x559d, 0x569d, 0x579d, 0x589d, 0x599d, 0xa9c2,
	0x5a9d, 0x5b9d, 0x5c9d, 0xecc0, 0xddd1, 0xeee4, 0x5d9d, 0x5e9d,
	0x5f9d, 0x609d, 0x619d, 0x629d, 0x639d, 0x649d, 0x659d, 0x669d,
	0xaec4, 0x679d, 0x689d, 0x699d, 0xede4, 0x6a9d, 0x6b9d, 0x6c9d,
	0x6d9d, 0xf6e4, 0xf4e4, 0xfec2, 0x6e9d, 0xdde4, 0x6f9d, 0xf0e4,
	0x709d, 0xfeca, 0x719d, 0xc4d5, 0x729d, 0x739d, 0xf1e4, 0x749d,
	0x759d, 0x769d, 0x779d, 0x789d, 0x799d, 0x7a9d, 0xfad1, 0x7b9d,
	0x7c9d, 0x7d9d, 0x7e9d, 0x809d, 0x819d, 0x829d, 0xebe4, 0xece4,
	0x839d, 0x849d, 0x859d, 0xf2e4, 0x869d, 0xabce, 0x879d, 0x889d,
	0x899d, 0x8a9d, 0x8b9d, 0x8c9d, 0x8d9d, 0x8e9d, 0x8f9d, 0x909d,
	0xcbc5, 0x919d, 0x929d, 0x939d, 0xb1c7, 0x949d, 0xbac2, 0x959d,
	0x969d, 0x979d, 0xeae4, 0x989d, 0x999d, 0x9a9d, 0xcac1, 0x9b9d,
	0x9c9d, 0x9d9d, 0x9e9d, 0x9f9d, 0xa09d, 0xb6cc, 0xb1b3, 0xa19d,
	0xa29d, 0xa39d, 0xfbe4, 0xa49d, 0xf3e4, 0xa59d, 0xa69d, 0xa79d,
	0xfae4, 0xa89d, 0xfde4, 0xa99d, 0xfce4, 0xaa9d, 0xab9d, 0xac9d,
	0xad9d, 0xae9d, 0xaf9d, 0xb09d, 0xceb3, 0xb19d, 0xb29d, 0xb39d,
	0xbab3, 0xf7e4, 0xb49d, 0xb59d, 0xf9e4, 0xf8e4, 0xecc5, 0xb69d,
	0xb79d, 0xb89d, 0xb99d, 0xba9d, 0xbb9d, 0xbc9d, 0xbd9d, 0xbe9d,
	0xbf9d, 0xc09d, 0xc19d, 0xc29d, 0xbdc0, 0xc39d, 0xc49d, 0xc59d,
	0xc69d, 0xe8d4, 0xc79d, 0xc89d, 0xc99d, 0xca9d, 0xcb9d, 0xa2e5,
	0xcc9d, 0xcd9d, 0xce9d, 0xcf9d, 0xd09d, 0xd19d, 0xd29d, 0xd39d,
	0xd49d, 0xd59d, 0xd69d, 0xc4b0, 0xd79d, 0xd89d, 0xa4e5, 0xd99d,
	0xda9d, 0xa3e5, 0xdb9d, 0xdc9d, 0xdd9d, 0xde9d, 0xdf9d, 0xe09d,
	0xa4bc, 0xe19d, 0xa5e5, 0xe29d, 0xe39d, 0xe49d, 0xe59d, 0xe69d,
	0xe79d, 0xa1e5, 0xe89d, 0xe99d, 0xea9d, 0xeb9d, 0xec9d, 0xed9d,
	0xee9d, 0xfee4, 0xf4b1, 0xef9d, 0xf09d, 0xf19d, 0xf29d, 0xf39d,
	0xf49d, 0xf59d, 0xf69d, 0xf79d, 0xf89d, 0xf99d, 0xa8e5, 0xfa9d,
	0xa9e5, 0xa6e5, 0xfb9d, 0xfc9d, 0xfd9d, 0xfe9d, 0x409e, 0x419e,
	0x429e, 0x439e, 0x449e, 0x459e, 0x469e, 0x479e, 0xa7e5, 0xaae5,
	0x489e, 0x499e, 0x4a9e, 0x4b9e, 0x4c9e, 0x4d9e, 0x4e9e, 0x4f9e,
	0x509e, 0x519e, 0x529e, 0x539e, 0x549e, 0x559e, 0x569e, 0x579e,
	0x589e, 0x599e, 0x5a9e, 0x5b9e, 0x5c9e, 0x5d9e, 0x5e9e, 0x5f9e,
	0x609e, 0x619e, 0x629e, 0x639e, 0x649e, 0x659e, 0x669e, 0x679e,
	0x689e, 0xd9c6, 0x699e, 0x6a9e, 0x6b9e, 0x6c9e, 0x6d9e, 0x6e9e,
	0x6f9e, 0x709e, 0xabe5, 0xade5, 0x719e, 0x729e, 0x739e, 0x749e,
	0x759e, 0x769e, 0x779e, 0xace5, 0x789e, 0x799e, 0x7a9e, 0x7b9e,
	0x7c9e, 0x7d9e, 0x7e9e, 0x809e, 0x819e, 0x829e, 0x839e, 0x849e,
	0x859e, 0x869e, 0x879e, 0x889e, 0x899e, 0xafe5, 0x8a9e, 0x8b9e,
	0x8c9e, 0xaee5, 0x8d9e, 0x8e9e, 0x8f9e, 0x909e, 0x919e, 0x929e,
	0x939e, 0x949e, 0x959e, 0x969e, 0x979e, 0x989e, 0x999e, 0x9a9e,
	0x9b9e, 0x9c9e, 0x9d9e, 0x9e9e, 0xe0b9, 0x9f9e, 0xa09e, 0xb0e5,
	0xa19e, 0xa29e, 0xa39e, 0xa49e, 0xa59e, 0xa69e, 0xa79e, 0xa89e,
	0xa99e, 0xaa9e, 0xab9e, 0xac9e, 0xad9e, 0xae9e, 0xb1e5, 0xaf9e,
	0xb09e, 0xb19e, 0xb29e, 0xb39e, 0xb49e, 0xb59e, 0xb69e, 0xb79e,
	0xb89e, 0xb99e, 0xba9e, 0xf0bb, 0xe1ec, 0xf0c3, 0xbb9e, 0xc6b5,
	0xd2bb, 0xbc9e, 0xbd9e, 0xbe9e, 0xbf9e, 0xe9c1, 0xeed4, 0xc09e,
	0xc4be, 0xc19e, 0xc29e, 0xc39e, 0xc6d7, 0xc49e, 0xd6d4, 0xd3b2,
	0xbeec, 0xc59e, 0xc69e, 0xc79e, 0xc89e, 0xc1ea, 0xc99e, 0xca9e,
	0xcb9e, 0xafc2, 0xb6b4, 0xcc9e, 0xcd9e, 0xce9e, 0xd7d1, 0xcf9e,
	0xd09e, 0xd19e, 0xb4b3, 0xd29e, 0xb2c8, 0xbbbf, 0xc0ec, 0xd39e,
	0xd49e, 0xcbd6, 0xd59e, 0xd69e, 0xbfec, 0xc1ec, 0xd79e, 0xd89e,
	0xd99e, 0xda9e, 0xdb9e, 0xdc9e, 0xdd9e, 0xde9e, 0xdf9e, 0xe09e,
	0xe19e, 0xe29e, 0xe39e, 0xc5ec, 0xe6be, 0xbfcc, 0xdac5, 0xbcbe,
	0xe49e, 0xc6ec, 0xe59e, 0xfeb1, 0xe69e, 0xe79e, 0xe89e, 0xc4ec,
	0xa8d5, 0xe3b5, 0xe99e, 0xc2ec, 0xb6c1, 0xe3b3, 0xea9e, 0xeb9e,
	0xc3ec, 0xb8cb, 0xc3c0, 0xfecc, 0xec9e, 0xed9e, 0xee9e, 0xef9e,
	0xd2c1, 0xf09e, 0xc8ec, 0xf19e, 0xf29e, 0xf39e, 0xf49e, 0xf59e,
	0xf69e, 0xf79e, 0xf89e, 0xf99e, 0xfa9e, 0xfb9e, 0xfc9e, 0xfd9e,
	0xe6ba, 0xd3c0, 0xfe9e, 0xf2d6, 0x409f, 0x419f, 0x429f, 0xccd1,
	0x439f, 0x449f, 0x459f, 0x469f, 0xbebf, 0x479f, 0xb3b7, 0xd5c9,
	0xc7ec, 0xe2bb, 0x489f, 0xcccc, 0xfdbd, 0xc8c8, 0x499f, 0xa9cf,
	0x4a9f, 0x4b9f, 0x4c9f, 0x4d9f, 0x4e9f, 0x4f9f, 0x509f, 0xe9cd,
	0x519f, 0xebc5, 0x529f, 0x539f, 0x549f, 0xe9b7, 0x559f, 0x569f,
	0x579f, 0x589f, 0x599f, 0x5a9f, 0x5b9f, 0x5c9f, 0x5d9f, 0x5e9f,
	0x5f9f, 0xc9d1, 0xb8ba, 0x609f, 0x619f, 0x629f, 0x639f, 0x649f,
	0xc9ec, 0x659f, 0x669f, 0xcaec, 0x679f, 0xc0bb, 0xcbec, 0x689f,
	0xe2ec, 0xbab1, 0xd9b7, 0x699f, 0x6a9f, 0x6b9f, 0x6c9f, 0x6d9f,
	0x6e9f, 0x6f9f, 0x709f, 0x719f, 0x729f, 0x739f, 0xb9bd, 0x749f,
	0x759f, 0x769f, 0x779f, 0x789f, 0x799f, 0x7a9f, 0x7b9f, 0xccec,
	0xe6d1, 0xcdec, 0x7c9f, 0x7d9f, 0x7e9f, 0x809f, 0xbbc8, 0x819f,
	0x829f, 0x839f, 0x849f, 0x859f, 0x869f, 0x879f, 0x889f, 0x899f,
	0x8a9f, 0x8b9f, 0x8c9f, 0x8d9f, 0x8e9f, 0xd1ec, 0x8f9f, 0x909f,
	0x919f, 0x929f, 0xd3ec, 0x939f, 0xcdbb, 0x949f, 0xe5bc, 0x959f,
	0x969f, 0x979f, 0x989f, 0x999f, 0x9a9f, 0x9b9f, 0x9c9f, 0x9d9f,
	0x9e9f, 0x9f9f, 0xa09f, 0xa19f, 0xcfec, 0xa29f, 0xb7c9, 0xa39f,
	0xa49f, 0xa59f, 0xa69f, 0xa79f, 0xbac3, 0xa89f, 0xe3ec, 0xd5d5,
	0xd0ec, 0xa99f, 0xaa9f, 0xab9f, 0xac9f, 0xad9f, 0xf3d6, 0xae9f,
	0xaf9f, 0xb09f, 0xd2ec, 0xceec, 0xb19f, 0xb29f, 0xb39f, 0xb49f,
	0xd4ec, 0xb59f, 0xd5ec, 0xb69f, 0xb79f, 0xbfc9, 0xb89f, 0xb99f,
	0xba9f, 0xbb9f, 0xbc9f, 0xbd9f, 0xa8cf, 0xbe9f, 0xbf9f, 0xc09f,
	0xc19f, 0xc29f, 0xdcd0, 0xc39f, 0xc49f, 0xc59f, 0xc69f, 0xacd1,
	0xc79f, 0xc89f, 0xc99f, 0xca9f, 0xdbc8, 0xcb9f, 0xcc9f, 0xcd9f,
	0xd6ec, 0xf5ce, 0xce9f, 0xcf9f, 0xd09f, 0xd19f, 0xd29f, 0xecca,
	0xdaec, 0xd39f, 0xd49f, 0xd59f, 0xd69f, 0xd79f, 0xd89f, 0xd99f,
	0xd9ec, 0xda9f, 0xdb9f, 0xdc9f, 0xbeb0, 0xdd9f, 0xde9f, 0xdf9f,
	0xe09f, 0xe19f, 0xe29f, 0xd7ec, 0xe39f, 0xd8ec, 0xe49f, 0xe59f,
	0xe69f, 0xe4ec, 0xe79f, 0xe89f, 0xe99f, 0xea9f, 0xeb9f, 0xec9f,
	0xed9f, 0xee9f, 0xef9f, 0xbcc8, 0xf09f, 0xf19f, 0xf29f, 0xf39f,
	0xf49f, 0xf59f, 0xf69f, 0xf79f, 0xf89f, 0xf99f, 0xc7c1, 0xfa9f,
	0xfb9f, 0xfc9f, 0xfd9f, 0xfe9f, 0xdcec, 0xe0d1, 0x40a0, 0x41a0,
	0x42a0, 0x43a0, 0x44a0, 0x45a0, 0x46a0, 0x47a0, 0x48a0, 0x49a0,
	0xdbec, 0x4aa0, 0x4ba0, 0x4ca0, 0x4da0, 0xefd4, 0x4ea0, 0xddec,
	0x4fa0, 0x50a0, 0x51a0, 0x52a0, 0x53a0, 0x54a0, 0xc6db, 0x55a0,
	0x56a0, 0x57a0, 0x58a0, 0x59a0, 0x5aa0, 0x5ba0, 0x5ca0, 0x5da0,
	0x5ea0, 0xdeec, 0x5fa0, 0x60a0, 0x61a0, 0x62a0, 0x63a0, 0x64a0,
	0x65a0, 0x66a0, 0x67a0, 0x68a0, 0x69a0, 0x6aa0, 0xacb1, 0x6ba0,
	0x6ca0, 0x6da0, 0x6ea0, 0x6fa0, 0x70a0, 0x71a0, 0x72a0, 0x73a0,
	0x74a0, 0x75a0, 0x76a0, 0x77a0, 0x78a0, 0x79a0, 0x7aa0, 0x7ba0,
	0x7ca0, 0x7da0, 0x7ea0, 0x80a0, 0x81a0, 0xdfec, 0x82a0, 0x83a0,
	0x84a0, 0x85a0, 0x86a0, 0x87a0, 0x88a0, 0x89a0, 0x8aa0, 0x8ba0,
	0xe0ec, 0x8ca0, 0xa6d7, 0x8da0, 0xc0c5, 0x8ea0, 0x8fa0, 0x90a0,
	0xbceb, 0xaeb0, 0x91a0, 0x92a0, 0x93a0, 0xf4be, 0xb8b8, 0xafd2,
	0xd6b0, 0xf9b5, 0x94a0, 0xb3d8, 0x95a0, 0xaccb, 0x96a0, 0xdde3,
	0x97a0, 0x98a0, 0x99a0, 0x9aa0, 0x9ba0, 0x9ca0, 0x9da0, 0xacc6,
	0xe6b0, 0x9ea0, 0x9fa0, 0xa0a0, 0xc6c5, 0xb9eb, 0xa1a0, 0xa2a0,
	0xa3a0, 0xa4a0, 0xbaeb, 0xa5a0, 0xa6a0, 0xa7a0, 0xbbeb, 0xa8a0,
	0xa9a0, 0xc0d1, 0xaaa0, 0xa3c5, 0xaba0, 0xf2ea, 0xaca0, 0xb2c4,
	0xada0, 0xb5c4, 0xcec0, 0xaea0, 0xafa0, 0xb0a0, 0xf3ea, 0xc1c4,
	0xb1a0, 0xefce, 0xb2a0, 0xb3a0, 0xb4a0, 0xb5a0, 0xf0ea, 0xf4ea,
	0xb6a0, 0xb7a0, 0xfcc9, 0xb8a0, 0xb9a0, 0xa3c7, 0xbaa0, 0xbba0,
	0xbca0, 0xd8cc, 0xfece, 0xbda0, 0xbea0, 0xbfa0, 0xf5ea, 0xf6ea,
	0xaccf, 0xe7c0, 0xc0a0, 0xc1a0, 0xf7ea, 0xc2a0, 0xc3a0, 0xc4a0,
	0xc5a0, 0xc6a0, 0xbfb6, 0xf8ea, 0xc7a0, 0xf9ea, 0xc8a0, 0xfaea,
	0xc9a0, 0xcaa0, 0xfbea, 0xcba0, 0xcca0, 0xcda0, 0xcea0, 0xcfa0,
	0xd0a0, 0xd1a0, 0xd2a0, 0xd3a0, 0xd4a0, 0xd5a0, 0xd6a0, 0xf1ea,
	0xd7a0, 0xd8a0, 0xd9a0, 0xdaa0, 0xdba0, 0xdca0, 0xdda0, 0xdea0,
	0xdfa0, 0xe0a0, 0xe1a0, 0xe2a0, 0xaec8, 0xebe1, 0xe3a0, 0xb8b7,
	0xece1, 0xe4a0, 0xe5a0, 0xe6a0, 0xede1, 0xe7a0, 0xb4d7, 0xeee1,
	0xefe1, 0xccd3, 0xe8a0, 0xe9a0, 0xeaa0, 0xeba0, 0xeca0, 0xeda0,
	0xeea0, 0xf1e1, 0xf1bf, 0xf0e1, 0xd2b5, 0xefa0, 0xf0a0, 0xf1a0,
	0xb7b1, 0xf2a0, 0xf3a0, 0xf4a0, 0xf5a0, 0xf3e1, 0xf2e1, 0xf6a0,
	0xfcba, 0xf7a0, 0xf4e1, 0xf8a0, 0xf9a0, 0xfaa0, 0xfba0, 0xb7b9,
	0xfca0, 0xd1be, 0xfda0, 0xfea0, 0x40aa, 0x41aa, 0xfcc4, 0x42aa,
	0xddba, 0xc6bd, 0x43aa, 0x44aa, 0x45aa, 0x46aa, 0x47aa, 0x48aa,
	0xf5e1, 0xf7e1, 0x49aa, 0x4aaa, 0xc0b6, 0xc1cf, 0xa8ca, 0xf6e1,
	0xf8d5, 0xfcd3, 0xf8e1, 0xfce1, 0xf9e1, 0x4baa, 0x4caa, 0xfae1,
	0xeac0, 0x4daa, 0xfee1, 0xa1e2, 0xc7c0, 0x4eaa, 0x4faa, 0x50aa,
	0x51aa, 0xfbe1, 0x52aa, 0xfde1, 0x53aa, 0x54aa, 0x55aa, 0x56aa,
	0x57aa, 0x58aa, 0xa5e2, 0x59aa, 0x5aaa, 0x5baa, 0xd4c1, 0x5caa,
	0x5daa, 0x5eaa, 0x5faa, 0xa3e2, 0x60aa, 0xa8e2, 0xfeb2, 0xa2e2,
	0x61aa, 0x62aa, 0x63aa, 0xcdc3, 0xc2b2, 0xa7e2, 0xa6e2, 0x64aa,
	0x65aa, 0xa4e2, 0xa9e2, 0x66aa, 0x67aa, 0xabe2, 0x68aa, 0x69aa,
	0x6aaa, 0xc9d0, 0xedd6, 0xa8c3, 0xace2, 0x6baa, 0xd7cf, 0x6caa,
	0x6daa, 0xaee2, 0x6eaa, 0x6faa, 0xefba, 0x70aa, 0x71aa, 0xe0e9,
	0xade2, 0xaae2, 0x72aa, 0x73aa, 0x74aa, 0x75aa, 0xabbb, 0xb3d4,
	0x76aa, 0x77aa, 0x78aa, 0x79aa, 0x7aaa, 0x7baa, 0x7caa, 0x7daa,
	0x7eaa, 0x80aa, 0x81aa, 0x82aa, 0x83aa, 0xb0e2, 0x84aa, 0x85aa,
	0xafe2, 0x86aa, 0xe1e9, 0x87aa, 0x88aa, 0x89aa, 0x8aaa, 0xb1e2,
	0x8baa, 0x8caa, 0x8daa, 0x8eaa, 0x8faa, 0x90aa, 0x91aa, 0x92aa,
	0xb2e2, 0x93aa, 0x94aa, 0x95aa, 0x96aa, 0x97aa, 0x98aa, 0x99aa,
	0x9aaa, 0x9baa, 0x9caa, 0x9daa, 0xb3e2, 0xa1cc, 0x9eaa, 0xb4e2,
	0x9faa, 0xa0aa, 0x40ab, 0x41ab, 0x42ab, 0x43ab, 0x44ab, 0x45ab,
	0x46ab, 0x47ab, 0x48ab, 0x49ab, 0x4aab, 0x4bab, 0xb5e2, 0x4cab,
	0x4dab, 0x4eab, 0x4fab, 0x50ab, 0xfed0, 0x51ab, 0x52ab, 0xcac2,
	0x53ab, 0xf1d3, 0x54ab, 0xf5cd, 0x55ab, 0x56ab, 0xe0e7, 0x57ab,
	0x58ab, 0xe1e7, 0x59ab, 0x5aab, 0x5bab, 0x5cab, 0xc1be, 0x5dab,
	0x5eab, 0x5fab, 0x60ab, 0xeac2, 0x61ab, 0x62ab, 0x63ab, 0xe4e7,
	0x64ab, 0x65ab, 0xe3e7, 0x66ab, 0x67ab, 0x68ab, 0x69ab, 0x6aab,
	0x6bab, 0xe6cd, 0x6cab, 0xb5c3, 0x6dab, 0x6eab, 0xe2e7, 0xb7bb,
	0xd6cf, 0x6fab, 0xe1c1, 0xe9e7, 0x70ab, 0x71ab, 0x72ab, 0xe8e7,
	0x73ab, 0x74ab, 0xf4e7, 0xa3b2, 0x75ab, 0x76ab, 0x77ab, 0x78ab,
	0xeae7, 0x79ab, 0xe6e7, 0x7aab, 0x7bab, 0x7cab, 0x7dab, 0x7eab,
	0xece7, 0xebe7, 0xbac9, 0x80ab, 0x81ab, 0xe4d5, 0x82ab, 0xe5e7,
	0xa9b7, 0xe7e7, 0x83ab, 0x84ab, 0x85ab, 0x86ab, 0x87ab, 0x88ab,
	0x89ab, 0xeee7, 0x8aab, 0x8bab, 0x8cab, 0x8dab, 0xf3e7, 0x8eab,
	0xe9d6, 0x8fab, 0x90ab, 0x91ab, 0x92ab, 0xede7, 0x93ab, 0xf2e7,
	0x94ab, 0xf1e7, 0x95ab, 0x96ab, 0x97ab, 0xe0b0, 0x98ab, 0x99ab,
	0x9aab, 0x9bab, 0xf5e7, 0x9cab, 0x9dab, 0x9eab, 0x9fab, 0xa0ab,
	0x40ac, 0x41ac, 0x42ac, 0x43ac, 0x44ac, 0x45ac, 0x46ac, 0x47ac,
	0x48ac, 0x49ac, 0x4aac, 0xf2c7, 0x4bac, 0xc5c0, 0xedc0, 0x4cac,
	0x4dac, 0xf0c1, 0xf0e7, 0x4eac, 0x4fac, 0x50ac, 0x51ac, 0xf6e7,
	0xf6cb, 0x52ac, 0x53ac, 0x54ac, 0x55ac, 0x56ac, 0x57ac, 0x58ac,
	0x59ac, 0x5aac, 0xa2e8, 0xa1e8, 0x5bac, 0x5cac, 0x5dac, 0x5eac,
	0x5fac, 0x60ac, 0xc1d7, 0x61ac, 0x62ac, 0xfae7, 0xf9e7, 0x63ac,
	0xfbe7, 0x64ac, 0xf7e7, 0x65ac, 0xfee7, 0x66ac, 0xfde7, 0x67ac,
	0xfce7, 0x68ac, 0x69ac, 0xd5c1, 0xd9c7, 0xfdc5, 0xc3c5, 0x6aac,
	0x6bac, 0x6cac, 0x6dac, 0x6eac, 0xedc7, 0x6fac, 0x70ac, 0x71ac,
	0x72ac, 0xa3e8, 0x73ac, 0x74ac, 0x75ac, 0x76ac, 0x77ac, 0x78ac,
	0x79ac, 0x7aac, 0x7bac, 0x7cac, 0x7dac, 0x7eac, 0x80ac, 0x81ac,
	0x82ac, 0x83ac, 0x84ac, 0x85ac, 0x86ac, 0xa6e8, 0x87ac, 0xa5e8,
	0x88ac, 0xa7e8, 0xf7ba, 0xf8e7, 0xa4e8, 0x89ac, 0xf0c8, 0xaac9,
	0x8aac, 0x8bac, 0x8cac, 0x8dac, 0x8eac, 0x8fac, 0x90ac, 0x91ac,
	0x92ac, 0x93ac, 0x94ac, 0x95ac, 0x96ac, 0xa9e8, 0x97ac, 0x98ac,
	0xe5b9, 0x99ac, 0x9aac, 0x9bac, 0x9cac, 0x9dac, 0xfed1, 0xa8e8,
	0x9eac, 0x9fac, 0xa0ac, 0x40ad, 0x41ad, 0x42ad, 0xaae8, 0x43ad,
	0xade8, 0xaee8, 0x44ad, 0xa7c1, 0x45ad, 0x46ad, 0x47ad, 0xafe8,
	0x48ad, 0x49ad, 0x4aad, 0xb0e8, 0x4bad, 0x4cad, 0xace8, 0x4dad,
	0xb4e8, 0x4ead, 0x4fad, 0x50ad, 0x51ad, 0x52ad, 0x53ad, 0x54ad,
	0x55ad, 0x56ad, 0x57ad, 0x58ad, 0xabe8, 0x59ad, 0xb1e8, 0x5aad,
	0x5bad, 0x5cad, 0x5dad, 0x5ead, 0x5fad, 0x60ad, 0x61ad, 0xb5e8,
	0xb2e8, 0xb3e8, 0x62ad, 0x63ad, 0x64ad, 0x65ad, 0x66ad, 0x67ad,
	0x68ad, 0x69ad, 0x6aad, 0x6bad, 0x6cad, 0x6dad, 0x6ead, 0x6fad,
	0x70ad, 0x71ad, 0xb7e8, 0x72ad, 0x73ad, 0x74ad, 0x75ad, 0x76ad,
	0x77ad, 0x78ad, 0x79ad, 0x7aad, 0x7bad, 0x7cad, 0x7dad, 0x7ead,
	0x80ad, 0x81ad, 0x82ad, 0x83ad, 0x84ad, 0x85ad, 0x86ad, 0x87ad,
	0x88ad, 0x89ad, 0xb6e8, 0x8aad, 0x8bad, 0x8cad, 0x8dad, 0x8ead,
	0x8fad, 0x90ad, 0x91ad, 0x92ad, 0xcfb9, 0x93ad, 0xacf0, 0x94ad,
	0xadf0, 0x95ad, 0xb0c6, 0xeab0, 0xbfc8, 0x96ad, 0xdfcd, 0x97ad,
	0x98ad, 0x99ad, 0x9aad, 0x9bad, 0x9cad, 0x9dad, 0xcdce, 0xb1ea,
	0x9ead, 0x9fad, 0xa0ad, 0x40ae, 0xb2ea, 0x41ae, 0xbfc6, 0xc9b4,
	0x42ae, 0x43ae, 0x44ae, 0x45ae, 0x46ae, 0x47ae, 0x48ae, 0xb3ea,
	0x49ae, 0x4aae, 0x4bae, 0x4cae, 0xe7d5, 0x4dae, 0x4eae, 0x4fae,
	0x50ae, 0x51ae, 0x52ae, 0x53ae, 0x54ae, 0xf9dd, 0x55ae, 0xb4ea,
	0x56ae, 0xb5ea, 0x57ae, 0xb6ea, 0x58ae, 0x59ae, 0x5aae, 0x5bae,
	0xcab8, 0xb0df, 0xf5c9, 0x5cae, 0xf0cc, 0x5dae, 0x5eae, 0xfac9,
	0x5fae, 0x60ae, 0x61ae, 0x62ae, 0x63ae, 0xfbc9, 0x64ae, 0x65ae,
	0xc3d3, 0xa6cb, 0x66ae, 0xa6b8, 0xaef0, 0xc2b1, 0x67ae, 0xb8e5,
	0xefcc, 0xc9d3, 0xd7bc, 0xeac9, 0x68ae, 0xe7b5, 0x69ae, 0xd0c4,
	0xe9b5, 0x6aae, 0xaeee, 0xadbb, 0x6bae, 0x6cae, 0xdee7, 0x6dae,
	0xafee, 0x6eae, 0x6fae, 0x70ae, 0x71ae, 0xa9b3, 0x72ae, 0x73ae,
	0xb2ee, 0x74ae, 0x75ae, 0xb1ee, 0xe7bd, 0x76ae, 0xb0ee, 0xb7ce,
	0x77ae, 0x78ae, 0x79ae, 0x7aae, 0xcfc5, 0x7bae, 0x7cae, 0x7dae,
	0x7eae, 0xf4c1, 0xcedb, 0xb3ee, 0xf3d0, 0x80ae, 0x81ae, 0x82ae,
	0x83ae, 0x84ae, 0x85ae, 0x86ae, 0x87ae, 0xd4c2, 0xe8c6, 0x88ae,
	0x89ae, 0x8aae, 0xacb7, 0x8bae, 0x8cae, 0x8dae, 0x8eae, 0x8fae,
	0x90ae, 0x91ae, 0xb4ee, 0x92ae, 0xebb3, 0x93ae, 0x94ae, 0x95ae,
	0xfbbb, 0xb5ee, 0x96ae, 0x97ae, 0x98ae, 0x99ae, 0x9aae, 0xdce7,
	0x9bae, 0x9cae, 0x9dae, 0xb6ee, 0x9eae, 0x9fae, 0xaebd, 0xa0ae,
	0x40af, 0x41af, 0x42af, 0xe2f1, 0x43af, 0x44af, 0x45af, 0xe8ca,
	0x46af, 0xc9d2, 0xdaf0, 0x47af, 0xdbf0, 0x48af, 0xdcf0, 0xc6c1,
	0x49af, 0xedb8, 0xcebe, 0x4aaf, 0x4baf, 0xdef0, 0x4caf, 0xb1c5,
	0xddf0, 0xf1d1, 0x4daf, 0xe0f0, 0xccb0, 0xeabd, 0x4eaf, 0x4faf,
	0x50af, 0x51af, 0x52af, 0xdfd2, 0xdff0, 0x53af, 0xafb4, 0xe8b7,
	0xe6f0, 0xe5f0, 0xa3c6, 0xe1f0, 0xe2f0, 0xc3b4, 0x54af, 0x55af,
	0xe3f0, 0xeed5, 0x56af, 0x57af, 0xdbcc, 0xd2be, 0xb2bc, 0x58af,
	0x59af, 0x5aaf, 0xe8f0, 0xe7f0, 0xe4f0, 0xa1b2, 0x5baf, 0xa2d6,
	0xb8d3, 0xb7be, 0xacc8, 0x5caf, 0x5daf, 0xeaf0, 0x5eaf, 0x5faf,
	0x60af, 0x61af, 0xf7d1, 0x62af, 0xccd6, 0xdbba, 0xe9f0, 0x63af,
	0xbbb6, 0x64af, 0x65af, 0xb4cd, 0x66af, 0x67af, 0xa6c6, 0x68af,
	0x69af, 0x6aaf, 0xa1c1, 0xebf0, 0xeef0, 0x6baf, 0xedf0, 0xf0f0,
	0xecf0, 0x6caf, 0xbebb, 0xeff0, 0x6daf, 0x6eaf, 0x6faf, 0x70af,
	0xb5cc, 0xf2f0, 0x71af, 0x72af, 0xd5b3, 0x73af, 0x74af, 0x75af,
	0x76af, 0xd4b1, 0x77af, 0x78af, 0xf3f0, 0x79af, 0x7aaf, 0xf4f0,
	0xf6f0, 0xe1b4, 0x7baf, 0xf1f0, 0x7caf, 0xf7f0, 0x7daf, 0x7eaf,
	0x80af, 0x81af, 0xfaf0, 0x82af, 0xf8f0, 0x83af, 0x84af, 0x85af,
	0xf5f0, 0x86af, 0x87af, 0x88af, 0x89af, 0xfdf0, 0x8aaf, 0xf9f0,
	0xfcf0, 0xfef0, 0x8baf, 0xa1f1, 0x8caf, 0x8daf, 0x8eaf, 0xc1ce,
	0xa4f1, 0x8faf, 0xa3f1, 0x90af, 0xf6c1, 0xfbf0, 0xddca, 0x91af,
	0x92af, 0xf1b4, 0xf1b1, 0xb1cc, 0x93af, 0xa6f1, 0x94af, 0x95af,
	0xa7f1, 0x96af, 0x97af, 0xacf1, 0xced5, 0xa9f1, 0x98af, 0x99af,
	0xb3c8, 0x9aaf, 0x9baf, 0x9caf, 0xa2f1, 0x9daf, 0xabf1, 0xa8f1,
	0xa5f1, 0x9eaf, 0x9faf, 0xaaf1, 0xa0af, 0x40b0, 0x41b0, 0x42b0,
	0x43b0, 0x44b0, 0x45b0, 0x46b0, 0xa9b0, 0xadf1, 0x47b0, 0x48b0,
	0x49b0, 0x4ab0, 0x4bb0, 0x4cb0, 0xaff1, 0x4db0, 0xb1f1, 0x4eb0,
	0x4fb0, 0x50b0, 0x51b0, 0x52b0, 0xb0f1, 0x53b0, 0xaef1, 0x54b0,
	0x55b0, 0x56b0, 0x57b0, 0xa2d1, 0x58b0, 0x59b0, 0x5ab0, 0x5bb0,
	0x5cb0, 0x5db0, 0x5eb0, 0xb2f1, 0x5fb0, 0x60b0, 0x61b0, 0xb3f1,
	0x62b0, 0x63b0, 0x64b0, 0x65b0, 0x66b0, 0x67b0, 0x68b0, 0x69b0,
	0xefb9, 0x6ab0, 0x6bb0, 0xc7b5, 0x6cb0, 0xd7b0, 0xd9b0, 0x6db0,
	0x6eb0, 0x6fb0, 0xedd4, 0x70b0, 0xc4b5, 0x71b0, 0xd4bd, 0xcabb,
	0xa7f0, 0x72b0, 0x73b0, 0xdeb8, 0x74b0, 0x75b0, 0xa8f0, 0x76b0,
	0x77b0, 0xa8b0, 0x78b0, 0xa9f0, 0x79b0, 0x7ab0, 0xeecd, 0x7bb0,
	0x7cb0, 0xaaf0, 0x7db0, 0x7eb0, 0x80b0, 0x81b0, 0x82b0, 0x83b0,
	0x84b0, 0x85b0, 0x86b0, 0x87b0, 0xabf0, 0x88b0, 0x89b0, 0x8ab0,
	0x8bb0, 0x8cb0, 0x8db0, 0x8eb0, 0x8fb0, 0x90b0, 0xa4c6, 0x91b0,
	0x92b0, 0xe5d6, 0xe4f1, 0x93b0, 0xe5f1, 0x94b0, 0x95b0, 0x96b0,
	0x97b0, 0x98b0, 0x99b0, 0x9ab0, 0x9bb0, 0x9cb0, 0x9db0, 0xf3c3,
	0x9eb0, 0x9fb0, 0xdbd3, 0xa0b0, 0x40b1, 0xd1d6, 0xe8c5, 0x41b1,
	0xafd3, 0x42b1, 0xe6d2, 0x43b1, 0x44b1, 0xc1ee, 0xbbb0, 0xb5d5,
	0xced1, 0xe0bc, 0xd0ba, 0x45b1, 0xf8bf, 0x46b1, 0xc7b8, 0xc1b5,
	0xccc5, 0x47b1, 0x48b1, 0xa2ca, 0x49b1, 0x4ab1, 0x4bb1, 0xcbc3,
	0x4cb1, 0x4db1, 0x4eb1, 0x4fb1, 0x50b1, 0xc2ee, 0x51b1, 0x52b1,
	0x53b1, 0x54b1, 0x55b1, 0x56b1, 0x57b1, 0x58b1, 0xbfc4, 0xa2b6,
	0x59b1, 0xeced, 0xa4c3, 0x5ab1, 0xb1d6, 0x5bb1, 0x5cb1, 0x5db1,
	0xe0cf, 0xefed, 0x5eb1, 0x5fb1, 0xcec5, 0x60b1, 0xdcb6, 0x61b1,
	0x62b1, 0xa1ca, 0x63b1, 0x64b1, 0xeded, 0x65b1, 0x66b1, 0xf0ed,
	0xf1ed, 0xbcc3, 0x67b1, 0xb4bf, 0x68b1, 0xeeed, 0x69b1, 0x6ab1,
	0x6bb1, 0x6cb1, 0x6db1, 0x6eb1, 0x6fb1, 0x70b1, 0x71b1, 0x72b1,
	0x73b1, 0xf4ed, 0xf2ed, 0x74b1, 0x75b1, 0x76b1, 0x77b1, 0xe6d5,
	0xdfc3, 0x78b1, 0xf3ed, 0x79b1, 0x7ab1, 0x7bb1, 0xf6ed, 0x7cb1,
	0xa3d5, 0xa3d1, 0x7db1, 0x7eb1, 0x80b1, 0xf5ed, 0x81b1, 0xd0c3,
	0x82b1, 0x83b1, 0x84b1, 0x85b1, 0x86b1, 0xf7ed, 0xf4bf, 0xecbe,
	0xf8ed, 0x87b1, 0xf7cc, 0x88b1, 0xdbd1, 0x89b1, 0x8ab1, 0x8bb1,
	0xc5d7, 0xf6d5, 0x8cb1, 0xfced, 0x8db1, 0x8eb1, 0x8fb1, 0xfbed,
	0x90b1, 0x91b1, 0x92b1, 0x93b1, 0x94b1, 0x95b1, 0x96b1, 0x97b1,
	0xf9ed, 0xfaed, 0x98b1, 0x99b1, 0x9ab1, 0x9bb1, 0x9cb1, 0x9db1,
	0x9eb1, 0x9fb1, 0xfded, 0xa6be, 0xa0b1, 0x40b2, 0x41b2, 0x42b2,
	0x43b2, 0xafcb, 0xa1ee, 0xbdb6, 0x44b2, 0xa2ee, 0xc0c4, 0x45b2,
	0xfeed, 0x46b2, 0x47b2, 0xdebd, 0xc7b2, 0x48b2, 0x49b2, 0x4ab2,
	0x4bb2, 0x4cb2, 0x4db2, 0x4eb2, 0x4fb2, 0x50b2, 0x51b2, 0x52b2,
	0x53b2, 0xc3b6, 0x54b2, 0x55b2, 0x56b2, 0xa5ee, 0xbad8, 0xa3ee,
	0xa6ee, 0x57b2, 0x58b2, 0x59b2, 0xe9c3, 0xf2b3, 0x5ab2, 0x5bb2,
	0x5cb2, 0x5db2, 0x5eb2, 0x5fb2, 0xa7ee, 0xa4ee, 0xb9cf, 0x60b2,
	0x61b2, 0xa8ee, 0xf7c2, 0x62b2, 0x63b2, 0x64b2, 0x65b2, 0x66b2,
	0x67b2, 0x68b2, 0x69b2, 0x6ab2, 0x6bb2, 0x6cb2, 0x6db2, 0xa9ee,
	0xaaee, 0x6eb2, 0xabde, 0x6fb2, 0x70b2, 0xb3c6, 0x71b2, 0xc6c7,
	0x72b2, 0xf5d6, 0xc9b5, 0x73b2, 0xb2cb, 0x74b2, 0x75b2, 0x76b2,
	0xabee, 0x77b2, 0x78b2, 0xabcd, 0x79b2, 0xacee, 0x7ab2, 0x7bb2,
	0x7cb2, 0x7db2, 0x7eb2, 0xb0d5, 0x80b2, 0xadee, 0x81b2, 0xc4f6,
	0x82b2, 0x83b2, 0x84b2, 0x85b2, 0x86b2, 0x87b2, 0x88b2, 0x89b2,
	0x8ab2, 0x8bb2, 0x8cb2, 0x8db2, 0x8eb2, 0xc7db, 0x8fb2, 0x90b2,
	0x91b2, 0x92b2, 0x93b2, 0x94b2, 0x95b2, 0x96b2, 0x97b2, 0xa3b4,
	0x98b2, 0x99b2, 0x9ab2, 0xacc3, 0xe6f1, 0x9bb2, 0x9cb2, 0x9db2,
	0x9eb2, 0x9fb2, 0xb8ca, 0xd3d2, 0xa0b2, 0xaad6, 0x40b3, 0xf2ef,
	0x41b3, 0xd8be, 0x42b3, 0xc3bd, 0xf3ef, 0xccb6, 0xabb0, 0x43b3,
	0x44b3, 0x45b3, 0x46b3, 0xafca, 0x47b3, 0x48b3, 0xb6ed, 0x49b3,
	0xb7ed, 0x4ab3, 0x4bb3, 0x4cb3, 0x4db3, 0xf9ce, 0xafb7, 0xf3bf,
	0xb8ed, 0xebc2, 0xb0c9, 0x4eb3, 0x4fb3, 0x50b3, 0x51b3, 0x52b3,
	0x53b3, 0xb9ed, 0x54b3, 0x55b3, 0xf6c6, 0xb3bf, 0x56b3, 0x57b3,
	0x58b3, 0xbced, 0xf8c5, 0x59b3, 0xd0d1, 0x5ab3, 0xa9d7, 0xbaed,
	0xbbed, 0x5bb3, 0xe2d1, 0x5cb3, 0xbfed, 0xc0ed, 0x5db3, 0xc4ed,
	0x5eb3, 0x5fb3, 0x60b3, 0xc8ed, 0x61b3, 0xc6ed, 0xceed, 0xe8d5,
	0x62b3, 0xc9ed, 0x63b3, 0x64b3, 0xc7ed, 0xbeed, 0x65b3, 0x66b3,
	0xe9c5, 0x67b3, 0x68b3, 0x69b3, 0xc6c6, 0x6ab3, 0x6bb3, 0xe9c9,
	0xd2d4, 0xc1ed, 0xc2ed, 0xc3ed, 0xc5ed, 0x6cb3, 0xf9c0, 0x6db3,
	0xa1b4, 0x6eb3, 0x6fb3, 0x70b3, 0x71b3, 0xe8b9, 0x72b3, 0xd0ed,
	0x73b3, 0x74b3, 0x75b3, 0x76b3, 0xd1ed, 0x77b3, 0xcaed, 0x78b3,
	0xcfed, 0x79b3, 0xf8ce, 0x7ab3, 0x7bb3, 0xb6cb, 0xcced, 0xcded,
	0x7cb3, 0x7db3, 0x7eb3, 0x80b3, 0x81b3, 0xf5cf, 0x82b3, 0x83b3,
	0x84b3, 0x85b3, 0x86b3, 0x87b3, 0x88b3, 0x89b3, 0x8ab3, 0x8bb3,
	0x8cb3, 0x8db3, 0xd2ed, 0xf2c1, 0xb2d3, 0xcbed, 0xb7c8, 0x8eb3,
	0x8fb3, 0x90b3, 0x91b3, 0x92b3, 0x93b3, 0x94b3, 0x95b3, 0xefbc,
	0x96b3, 0x97b3, 0x98b3, 0x99b3, 0xf0c5, 0x9ab3, 0x9bb3, 0x9cb3,
	0x9db3, 0x9eb3, 0x9fb3, 0xa0b3, 0x40b4, 0x41b4, 0x42b4, 0xd6ed,
	0x43b4, 0xefb5, 0x44b4, 0x45b4, 0xb5c2, 0xadb0, 0xe9cb, 0x46b4,
	0x47b4, 0xaeb1, 0x48b4, 0xd4ed, 0x49b4, 0x4ab4, 0x4bb4, 0xebcd,
	0xe2b5, 0x4cb4, 0xd5ed, 0xd3ed, 0xd7ed, 0x4db4, 0x4eb4, 0xfab5,
	0x4fb4, 0xd8ed, 0x50b4, 0xd9ed, 0x51b4, 0xdced, 0x52b4, 0xccb1,
	0x53b4, 0x54b4, 0x55b4, 0x56b4, 0x57b4, 0x58b4, 0x59b4, 0x5ab4,
	0xf6c5, 0xeebc, 0xdaed, 0xbccc, 0xeab2, 0x5bb4, 0x5cb4, 0x5db4,
	0x5eb4, 0xdbed, 0x5fb4, 0x60b4, 0x61b4, 0x62b4, 0xebc4, 0x63b4,
	0x64b4, 0xc5b4, 0x65b4, 0x66b4, 0x67b4, 0xf5b0, 0x68b4, 0x69b4,
	0x6ab4, 0xdfed, 0xdac0, 0xe8b4, 0x6bb4, 0x6cb4, 0x6db4, 0x6eb4,
	0xcdc5, 0x6fb4, 0x70b4, 0x71b4, 0xdded, 0xc4bf, 0x72b4, 0x73b4,
	0x74b4, 0xdeed, 0x75b4, 0x76b4, 0x77b4, 0x78b4, 0x79b4, 0x7ab4,
	0x7bb4, 0x7cb4, 0x7db4, 0x7eb4, 0x80b4, 0x81b4, 0x82b4, 0x83b4,
	0xa5c4, 0x84b4, 0x85b4, 0x86b4, 0xe0ed, 0x87b4, 0x88b4, 0x89b4,
	0x8ab4, 0x8bb4, 0xe1ed, 0x8cb4, 0xe3ed, 0x8db4, 0x8eb4, 0xd7c1,
	0x8fb4, 0x90b4, 0xc7bb, 0x91b4, 0x92b4, 0x93b4, 0x94b4, 0x95b4,
	0x96b4, 0xb8bd, 0x97b4, 0x98b4, 0x99b4, 0xe2ed, 0x9ab4, 0x9bb4,
	0x9cb4, 0x9db4, 0x9eb4, 0x9fb4, 0xa0b4, 0x40b5, 0x41b5, 0x42b5,
	0x43b5, 0x44b5, 0x45b5, 0xe4ed, 0x46b5, 0x47b5, 0x48b5, 0x49b5,
	0x4ab5, 0x4bb5, 0x4cb5, 0x4db5, 0x4eb5, 0x4fb5, 0xe6ed, 0x50b5,
	0x51b5, 0x52b5, 0x53b5, 0x54b5, 0xe5ed, 0x55b5, 0x56b5, 0x57b5,
	0x58b5, 0x59b5, 0x5ab5, 0x5bb5, 0x5cb5, 0x5db5, 0x5eb5, 0x5fb5,
	0x60b5, 0x61b5, 0x62b5, 0x63b5, 0xe7ed, 0x64b5, 0x65b5, 0x66b5,
	0x67b5, 0x68b5, 0xbeca, 0xeaec, 0xf1c0, 0x69b5, 0xe7c9, 0x6ab5,
	0xebec, 0xeec6, 0x6bb5, 0x6cb5, 0x6db5, 0x6eb5, 0xecec, 0x6fb5,
	0xedc6, 0xedec, 0x70b5, 0x71b5, 0x72b5, 0x73b5, 0x74b5, 0x75b5,
	0x76b5, 0x77b5, 0x78b5, 0xf0ec, 0x79b5, 0x7ab5, 0xe6d7, 0xf3ec,
	0x7bb5, 0x7cb5, 0xf1ec, 0xeeec, 0xefec, 0xa3d7, 0xf1c9, 0xeecb,
	0xf4ec, 0x7db5, 0xf2ec, 0x7eb5, 0x80b5, 0xe9cf, 0x81b5, 0xf6ec,
	0xb1c6, 0x82b5, 0x83b5, 0x84b5, 0x85b5, 0xc0bc, 0x86b5, 0xf5ec,
	0x87b5, 0x88b5, 0x89b5, 0x8ab5, 0x8bb5, 0x8cb5, 0x8db5, 0xbbb5,
	0xf6bb, 0x8eb5, 0xf7ec, 0x8fb5, 0x90b5, 0x91b5, 0x92b5, 0x93b5,
	0xf7d9, 0xfbbd, 0x94b5, 0x95b5, 0xbbc2, 0xf8ec, 0x96b5, 0x97b5,
	0x98b5, 0x99b5, 0xf9ec, 0x9ab5, 0x9bb5, 0x9cb5, 0x9db5, 0xa3b8,
	0x9eb5, 0x9fb5, 0xa0b5, 0x40b6, 0x41b6, 0x42b6, 0x43b6, 0x44b6,
	0x45b6, 0x46b6, 0xfaec, 0x47b6, 0x48b6, 0x49b6, 0x4ab6, 0x4bb6,
	0x4cb6, 0x4db6, 0x4eb6, 0x4fb6, 0x50b6, 0x51b6, 0x52b6, 0xfbec,
	0x53b6, 0x54b6, 0x55b6, 0x56b6, 0x57b6, 0x58b6, 0x59b6, 0x5ab6,
	0x5bb6, 0x5cb6, 0x5db6, 0xfcec, 0x5eb6, 0x5fb6, 0x60b6, 0x61b6,
	0x62b6, 0xedd3, 0xaed8, 0xebc0, 0x63b6, 0xddc7, 0xccba, 0x64b6,
	0xe3d0, 0xbdcb, 0x65b6, 0xbacd, 0x66b6, 0x67b6, 0xd1b8, 0x68b6,
	0x69b6, 0xfcb1, 0x6ab6, 0xefc7, 0x6bb6, 0xd6d6, 0x6cb6, 0x6db6,
	0x6eb6, 0xc6bf, 0xebc3, 0x6fb6, 0x70b6, 0xf5ef, 0x71b6, 0x72b6,
	0xd8c3, 0x73b6, 0x74b6, 0x75b6, 0x76b6, 0x77b6, 0x78b6, 0xe2d7,
	0x79b6, 0x7ab6, 0x7bb6, 0xf7ef, 0xd3b3, 0x7cb6, 0xd8c7, 0xedd1,
	0x7db6, 0xc8d6, 0x7eb6, 0xf8ef, 0x80b6, 0xf6ef, 0x81b6, 0xfdbb,
	0xc6b3, 0x82b6, 0x83b6, 0x84b6, 0x85b6, 0x86b6, 0x87b6, 0x88b6,
	0xd5bd, 0x89b6, 0x8ab6, 0xc6d2, 0x8bb6, 0xe0bb, 0x8cb6, 0x8db6,
	0xa1cf, 0x8eb6, 0xfcef, 0xfbef, 0x8fb6, 0x90b6, 0xf9ef, 0x91b6,
	0x92b6, 0x93b6, 0x94b6, 0xccb3, 0x95b6, 0xd4c9, 0xb0cb, 0x96b6,
	0x97b6, 0x98b6, 0x99b6, 0x9ab6, 0xfeef, 0x9bb6, 0x9cb6, 0xdeb0,
	0x9db6, 0x9eb6, 0xc9d6, 0x9fb6, 0xa0b6, 0x40b7, 0xfdef, 0x41b7,
	0xedb3, 0x42b7, 0x43b7, 0xd5f6, 0x44b7, 0x45b7, 0x46b7, 0x47b7,
	0x48b7, 0x49b7, 0x4ab7, 0x4bb7, 0x4cb7, 0x4db7, 0x4eb7, 0x4fb7,
	0x50b7, 0x51b7, 0x52b7, 0xc8ce, 0x53b7, 0x54b7, 0x55b7, 0xa2f0,
	0x56b7, 0xa1f0, 0x57b7, 0xbeb5, 0xdabc, 0xfcbb, 0x58b7, 0xe5b8,
	0x59b7, 0x5ab7, 0x5bb7, 0x5cb7, 0x5db7, 0x5eb7, 0xc2c4, 0x5fb7,
	0x60b7, 0x61b7, 0x62b7, 0x63b7, 0x64b7, 0x65b7, 0x66b7, 0x67b7,
	0x68b7, 0xa3f0, 0x69b7, 0x6ab7, 0x6bb7, 0x6cb7, 0x6db7, 0xebcb,
	0x6eb7, 0x6fb7, 0x70b7, 0x71b7, 0x72b7, 0x73b7, 0x74b7, 0x75b7,
	0x76b7, 0x77b7, 0x78b7, 0x79b7, 0x7ab7, 0x7bb7, 0x7cb7, 0x7db7,
	0x7eb7, 0x80b7, 0x81b7, 0x82b7, 0x83b7, 0x84b7, 0x85b7, 0x86b7,
	0xa6f0, 0x87b7, 0x88b7, 0x89b7, 0xa8d1, 0x8ab7, 0xbfbe, 0xeec7,
	0xb6f1, 0xb7f1, 0xd5bf, 0x8bb7, 0x8cb7, 0x8db7, 0x8eb7, 0xa9b4,
	0xb8f1, 0xbbcd, 0x8fb7, 0xd4c7, 0xadd5, 0x90b7, 0xb9f1, 0x91b7,
	0xbaf1, 0x92b7, 0x93b7, 0x94b7, 0x95b7, 0xcfc7, 0x96b7, 0x97b7,
	0x98b7, 0xa4d2, 0xcfd6, 0x99b7, 0x9ab7, 0xbbf1, 0xd1bd, 0xb0b4,
	0xbdbe, 0x9bb7, 0x9cb7, 0x9db7, 0xdcb4, 0xd1ce, 0x9eb7, 0xdfbf,
	0xbdf1, 0x9fb7, 0xa0b7, 0x40b8, 0x41b8, 0xfabf, 0xbcf1, 0x42b8,
	0xbff1, 0x43b8, 0x44b8, 0x45b8, 0xbef1, 0xc0f1, 0x46b8, 0x47b8,
	0x48b8, 0x49b8, 0x4ab8, 0xc1f1, 0x4bb8, 0x4cb8, 0x4db8, 0x4eb8,
	0x4fb8, 0x50b8, 0x51b8, 0x52b8, 0x53b8, 0x54b8, 0x55b8, 0xfec1,
	0x56b8, 0x57b8, 0x58b8, 0x59b8, 0x5ab8, 0x5bb8, 0x5cb8, 0x5db8,
	0x5eb8, 0x5fb8, 0x60b8, 0xa2c1, 0x61b8, 0x62b8, 0x63b8, 0x64b8,
	0x65b8, 0x66b8, 0x67b8, 0x68b8, 0x69b8, 0x6ab8, 0xfaca, 0x6bb8,
	0x6cb8, 0xbed5, 0x6db8, 0x6eb8, 0x6fb8, 0x70b8, 0xbabe, 0xb9be,
	0xc2d5, 0x71b8, 0x72b8, 0xa2bf, 0x73b8, 0xafcd, 0xb5f1, 0x74b8,
	0x75b8, 0x76b8, 0x77b8, 0x78b8, 0x79b8, 0xdfbd, 0x7ab8, 0xcbb6,
	0x7bb8, 0x7cb8, 0x7db8, 0x7eb8, 0x80b8, 0x81b8, 0x82b8, 0x83b8,
	0x84b8, 0xf1d6, 0xc3f3, 0x85b8, 0x86b8, 0xc4f3, 0x87b8, 0xcdb8,
	0x88b8, 0x89b8, 0x8ab8, 0xc6f3, 0xc7f3, 0x8bb8, 0xcab0, 0x8cb8,
	0xc5f3, 0x8db8, 0xc9f3, 0xf1cb, 0x8eb8, 0x8fb8, 0x90b8, 0xcbf3,
	0x91b8, 0xa6d0, 0x92b8, 0x93b8, 0xcab1, 0xc8f3, 0x94b8, 0x95b8,
	0x96b8, 0xcff3, 0x97b8, 0xd1b5, 0x98b8, 0x99b8, 0xd7f3, 0x9ab8,
	0xd2f3, 0x9bb8, 0x9cb8, 0x9db8, 0xd4f3, 0xd3f3, 0xfbb7, 0x9eb8,
	0xbfb1, 0x9fb8, 0xcef3, 0xcaf3, 0xdab5, 0xa0b8, 0xd0f3, 0x40b9,
	0x41b9, 0xd1f3, 0x42b9, 0xd5f3, 0x43b9, 0x44b9, 0x45b9, 0x46b9,
	0xcdf3, 0x47b9, 0xe3bc, 0x48b9, 0xfdc1, 0x49b9, 0xd6f3, 0x4ab9,
	0x4bb9, 0x4cb9, 0x4db9, 0x4eb9, 0x4fb9, 0xdaf3, 0x50b9, 0xccf3,
	0x51b9, 0xc8b5, 0x52b9, 0xeebd, 0xdcf3, 0x53b9, 0x54b9, 0xa4b7,
	0xf0bf, 0xfed6, 0xb2cd, 0x55b9, 0xf0b4, 0x56b9, 0xdfb2, 0x57b9,
	0xd8f3, 0x58b9, 0xd9f3, 0xb8c9, 0x59b9, 0xddf3, 0x5ab9, 0x5bb9,
	0xdef3, 0x5cb9, 0xe1f3, 0x5db9, 0x5eb9, 0x5fb9, 0x60b9, 0x61b9,
	0x62b9, 0x63b9, 0x64b9, 0x65b9, 0x66b9, 0x67b9, 0xdff3, 0x68b9,
	0x69b9, 0xe3f3, 0xe2f3, 0x6ab9, 0x6bb9, 0xdbf3, 0x6cb9, 0xeabf,
	0x6db9, 0xefb3, 0x6eb9, 0xe0f3, 0x6fb9, 0x70b9, 0xa9c7, 0x71b9,
	0xf2bc, 0x72b9, 0x73b9, 0x74b9, 0x75b9, 0xebf3, 0x76b9, 0x77b9,
	0x78b9, 0x79b9, 0x7ab9, 0x7bb9, 0x7cb9, 0xbfb9, 0x7db9, 0x7eb9,
	0xe4f3, 0x80b9, 0x81b9, 0x82b9, 0xadb2, 0xfebb, 0x83b9, 0xe3cb,
	0x84b9, 0x85b9, 0x86b9, 0x87b9, 0xedf3, 0xe9f3, 0x88b9, 0x89b9,
	0x8ab9, 0xdcb9, 0xeef3, 0x8bb9, 0x8cb9, 0x8db9, 0xe5f3, 0xe6f3,
	0xeaf3, 0xe1c2, 0xecf3, 0xeff3, 0xe8f3, 0xfdbc, 0x8eb9, 0x8fb9,
	0x90b9, 0xe4cf, 0x91b9, 0x92b9, 0xf0f3, 0x93b9, 0x94b9, 0x95b9,
	0xe7f3, 0x96b9, 0x97b9, 0x98b9, 0x99b9, 0x9ab9, 0x9bb9, 0x9cb9,
	0x9db9, 0xf2f3, 0x9eb9, 0x9fb9, 0xa0b9, 0x40ba, 0xadd7, 0xaac6,
	0x41ba, 0x42ba, 0x43ba, 0x44ba, 0xf3f3, 0x45ba, 0x46ba, 0x47ba,
	0x48ba, 0xf1f3, 0x49ba, 0xa8c2, 0x4aba, 0x4bba, 0x4cba, 0x4dba,
	0x4eba, 0xddb8, 0xf5f3, 0x4fba, 0x50ba, 0xf4f3, 0x51ba, 0x52ba,
	0x53ba, 0xdbb4, 0x54ba, 0x55ba, 0x56ba, 0xf6f3, 0xf7f3, 0x57ba,
	0x58ba, 0x59ba, 0xf8f3, 0x5aba, 0x5bba, 0x5cba, 0xbac0, 0x5dba,
	0x5eba, 0xe9c0, 0x5fba, 0x60ba, 0x61ba, 0x62ba, 0x63ba, 0xf1c5,
	0x64ba, 0x65ba, 0x66ba, 0x67ba, 0xfbf3, 0x68ba, 0xfaf3, 0x69ba,
	0x6aba, 0x6bba, 0x6cba, 0x6dba, 0x6eba, 0x6fba, 0x70ba, 0xd8b4,
	0x71ba, 0x72ba, 0x73ba, 0xfef3, 0xf9f3, 0x74ba, 0x75ba, 0xfcf3,
	0x76ba, 0x77ba, 0x78ba, 0x79ba, 0x7aba, 0x7bba, 0xfdf3, 0x7cba,
	0x7dba, 0x7eba, 0x80ba, 0x81ba, 0x82ba, 0x83ba, 0x84ba, 0xa1f4,
	0x85ba, 0x86ba, 0x87ba, 0x88ba, 0x89ba, 0x8aba, 0xa3f4, 0xc9bb,
	0x8bba, 0x8cba, 0xa2f4, 0x8dba, 0x8eba, 0x8fba, 0x90ba, 0x91ba,
	0x92ba, 0x93ba, 0x94ba, 0x95ba, 0x96ba, 0x97ba, 0x98ba, 0x99ba,
	0xa4f4, 0x9aba, 0x9bba, 0x9cba, 0x9dba, 0x9eba, 0x9fba, 0xbeb2,
	0xa6f4, 0xa5f4, 0xa0ba, 0x40bb, 0x41bb, 0x42bb, 0x43bb, 0x44bb,
	0x45bb, 0x46bb, 0x47bb, 0x48bb, 0x49bb, 0xaebc, 0x4abb, 0x4bbb,
	0x4cbb, 0x4dbb, 0x4ebb, 0x4fbb, 0x50bb, 0x51bb, 0x52bb, 0x53bb,
	0x54bb, 0x55bb, 0x56bb, 0x57bb, 0x58bb, 0x59bb, 0x5abb, 0x5bbb,
	0x5cbb, 0x5dbb, 0x5ebb, 0x5fbb, 0x60bb, 0x61bb, 0x62bb, 0x63bb,
	0x64bb, 0x65bb, 0x66bb, 0x67bb, 0x68bb, 0x69bb, 0x6abb, 0x6bbb,
	0x6cbb, 0x6dbb, 0x6ebb, 0xd7c3, 0xe1d9, 0x6fbb, 0x70bb, 0x71bb,
	0x72bb, 0x73bb, 0x74bb, 0xe0c0, 0xccf4, 0xd1d7, 0x75bb, 0x76bb,
	0x77bb, 0x78bb, 0x79bb, 0x7abb, 0x7bbb, 0x7cbb, 0x7dbb, 0x7ebb,
	0x80bb, 0xdbb7, 0x81bb, 0x82bb, 0x83bb, 0x84bb, 0x85bb, 0x86bb,
	0x87bb, 0xcef4, 0xa3c1, 0x88bb, 0x89bb, 0xc9c6, 0x8abb, 0xd6b4,
	0xb3d5, 0x8bbb, 0x8cbb, 0x8dbb, 0xd0f4, 0xcff4, 0xd1f4, 0xdacb,
	0x8ebb, 0x8fbb, 0xd2f4, 0x90bb, 0xc1d4, 0xe0d6, 0x91bb, 0x92bb,
	0x93bb, 0x94bb, 0xe0b7, 0x95bb, 0x96bb, 0x97bb, 0xb8c1, 0x98bb,
	0x99bb, 0xbbc1, 0xd3f4, 0xacbe, 0x9abb, 0x9bbb, 0x9cbb, 0x9dbb,
	0x9ebb, 0xe2b4, 0x9fbb, 0xa0bb, 0xd4f4, 0xd5f4, 0xabbe, 0x40bc,
	0x41bc, 0xd6f4, 0x42bc, 0x43bc, 0x44bc, 0xdbf4, 0x45bc, 0xd7f4,
	0xdaf4, 0x46bc, 0xfdba, 0x47bc, 0xd8f4, 0xd9f4, 0x48bc, 0x49bc,
	0x4abc, 0x4bbc, 0x4cbc, 0x4dbc, 0x4ebc, 0xe2b8, 0xc7cc, 0xdcf4,
	0x4fbc, 0xdab2, 0x50bc, 0x51bc, 0xd3c3, 0x52bc, 0x53bc, 0xe3d4,
	0xb7bf, 0x54bc, 0x55bc, 0x56bc, 0x57bc, 0x58bc, 0x59bc, 0x5abc,
	0xddf4, 0x5bbc, 0x5cbc, 0x5dbc, 0x5ebc, 0x5fbc, 0x60bc, 0xb4c5,
	0x61bc, 0x62bc, 0x63bc, 0x64bc, 0x65bc, 0x66bc, 0x67bc, 0x68bc,
	0xe9f4, 0x69bc, 0x6abc, 0xb5cf, 0x6bbc, 0x6cbc, 0x6dbc, 0x6ebc,
	0x6fbc, 0x70bc, 0x71bc, 0x72bc, 0x73bc, 0x74bc, 0x75bc, 0x76bc,
	0x77bc, 0x78bc, 0xc9ce, 0x79bc, 0x7abc, 0x7bbc, 0x7cbc, 0x7dbc,
	0x7ebc, 0x80bc, 0x81bc, 0x82bc, 0x83bc, 0x84bc, 0x85bc, 0x86bc,
	0x87bc, 0x88bc, 0x89bc, 0x8abc, 0x8bbc, 0x8cbc, 0x8dbc, 0x8ebc,
	0xd8cb, 0x8fbc, 0xf7cb, 0x90bc, 0x91bc, 0x92bc, 0x93bc, 0xf4bd,
	0x94bc, 0x95bc, 0x96bc, 0xcfd7, 0x97bc, 0x98bc, 0x99bc, 0xdbc0,
	0x9abc, 0x9bbc, 0x9cbc, 0x9dbc, 0x9ebc, 0x9fbc, 0xa0bc, 0x40bd,
	0x41bd, 0x42bd, 0x43bd, 0x44bd, 0x45bd, 0x46bd, 0x47bd, 0x48bd,
	0x49bd, 0x4abd, 0x4bbd, 0x4cbd, 0x4dbd, 0x4ebd, 0x4fbd, 0x50bd,
	0x51bd, 0x52bd, 0x53bd, 0x54bd, 0x55bd, 0x56bd, 0x57bd, 0x58bd,
	0x59bd, 0x5abd, 0x5bbd, 0x5cbd, 0x5dbd, 0x5ebd, 0x5fbd, 0x60bd,
	0x61bd, 0x62bd, 0x63bd, 0x64bd, 0x65bd, 0x66bd, 0x67bd, 0x68bd,
	0x69bd, 0x6abd, 0x6bbd, 0x6cbd, 0x6dbd, 0x6ebd, 0x6fbd, 0x70bd,
	0x71bd, 0x72bd, 0x73bd, 0x74bd, 0x75bd, 0x76bd, 0xf5d0, 0x77bd,
	0x78bd, 0x79bd, 0x7abd, 0x7bbd, 0x7cbd, 0x7dbd, 0x7ebd, 0xeaf4,
	0x80bd, 0x81bd, 0x82bd, 0x83bd, 0x84bd, 0x85bd, 0x86bd, 0x87bd,
	0x88bd, 0x89bd, 0x8abd, 0x8bbd, 0x8cbd, 0x8dbd, 0x8ebd, 0x8fbd,
	0x90bd, 0x91bd, 0x92bd, 0x93bd, 0x94bd, 0x95bd, 0x96bd, 0x97bd,
	0x98bd, 0x99bd, 0x9abd, 0x9bbd, 0x9cbd, 0x9dbd, 0x9ebd, 0x9fbd,
	0xa0bd, 0x40be, 0x41be, 0x42be, 0x43be, 0x44be, 0x45be, 0x46be,
	0x47be, 0x48be, 0x49be, 0x4abe, 0x4bbe, 0x4cbe, 0xebf4, 0x4dbe,
	0x4ebe, 0x4fbe, 0x50be, 0x51be, 0x52be, 0x53be, 0xecf4, 0x54be,
	0x55be, 0x56be, 0x57be, 0x58be, 0x59be, 0x5abe, 0x5bbe, 0x5cbe,
	0x5dbe, 0x5ebe, 0x5fbe, 0x60be, 0x61be, 0x62be, 0x63be, 0x64be,
	0x65be, 0x66be, 0x67be, 0x68be, 0x69be, 0x6abe, 0x6bbe, 0x6cbe,
	0x6dbe, 0x6ebe, 0x6fbe, 0x70be, 0x71be, 0x72be, 0x73be, 0x74be,
	0x75be, 0x76be, 0x77be, 0x78be, 0x79be, 0x7abe, 0x7bbe, 0x7cbe,
	0x7dbe, 0x7ebe, 0x80be, 0x81be, 0x82be, 0x83be, 0x84be, 0x85be,
	0x86be, 0x87be, 0x88be, 0x89be, 0x8abe, 0x8bbe, 0x8cbe, 0x8dbe,
	0x8ebe, 0x8fbe, 0x90be, 0x91be, 0x92be, 0x93be, 0x94be, 0x95be,
	0x96be, 0x97be, 0x98be, 0x99be, 0x9abe, 0x9bbe, 0x9cbe, 0x9dbe,
	0x9ebe, 0x9fbe, 0xa0be, 0x40bf, 0x41bf, 0x42bf, 0x43bf, 0x44bf,
	0x45bf, 0x46bf, 0x47bf, 0x48bf, 0x49bf, 0x4abf, 0x4bbf, 0x4cbf,
	0x4dbf, 0x4ebf, 0x4fbf, 0x50bf, 0x51bf, 0x52bf, 0x53bf, 0x54bf,
	0x55bf, 0x56bf, 0x57bf, 0x58bf, 0x59bf, 0x5abf, 0x5bbf, 0x5cbf,
	0x5dbf, 0x5ebf, 0x5fbf, 0x60bf, 0x61bf, 0x62bf, 0x63bf, 0x64bf,
	0x65bf, 0x66bf, 0x67bf, 0x68bf, 0x69bf, 0x6abf, 0x6bbf, 0x6cbf,
	0x6dbf, 0x6ebf, 0x6fbf, 0x70bf, 0x71bf, 0x72bf, 0x73bf, 0x74bf,
	0x75bf, 0x76bf, 0x77bf, 0x78bf, 0x79bf, 0x7abf, 0x7bbf, 0x7cbf,
	0x7dbf, 0x7ebf, 0x80bf, 0xe3f7, 0x81bf, 0x82bf, 0x83bf, 0x84bf,
	0x85bf, 0xb1b7, 0x86bf, 0x87bf, 0x88bf, 0x89bf, 0x8abf, 0xedf4,
	0x8bbf, 0x8cbf, 0x8dbf, 0x8ebf, 0x8fbf, 0x90bf, 0x91bf, 0x92bf,
	0x93bf, 0x94bf, 0x95bf, 0x96bf, 0x97bf, 0x98bf, 0x99bf, 0x9abf,
	0x9bbf, 0x9cbf, 0x9dbf, 0x9ebf, 0x9fbf, 0xa0bf, 0x40c0, 0x41c0,
	0x42c0, 0x43c0, 0x44c0, 0x45c0, 0x46c0, 0x47c0, 0x48c0, 0x49c0,
	0x4ac0, 0x4bc0, 0x4cc0, 0x4dc0, 0x4ec0, 0x4fc0, 0x50c0, 0x51c0,
	0x52c0, 0x53c0, 0x54c0, 0x55c0, 0x56c0, 0x57c0, 0x58c0, 0x59c0,
	0x5ac0, 0x5bc0, 0x5cc0, 0x5dc0, 0x5ec0, 0x5fc0, 0x60c0, 0x61c0,
	0x62c0, 0x63c0, 0xebd7, 0x64c0, 0x65c0, 0x66c0, 0x67c0, 0x68c0,
	0x69c0, 0x6ac0, 0x6bc0, 0x6cc0, 0x6dc0, 0x6ec0, 0x6fc0, 0x70c0,
	0x71c0, 0x72c0, 0x73c0, 0x74c0, 0x75c0, 0x76c0, 0x77c0, 0x78c0,
	0x79c0, 0x7ac0, 0x7bc0, 0xeef4, 0x7cc0, 0x7dc0, 0x7ec0, 0xf9e6,
	0xc0be, 0xfae6, 0xecba, 0xfbe6, 0xcbcf, 0xfce6, 0xbcd4, 0xb6bc,
	0xfde6, 0xfee6, 0xcdbc, 0xd2c8, 0xb3ce, 0xa1e7, 0x80c0, 0xbfb4,
	0xa2e7, 0xb4c9, 0xd9b8, 0xc9c4, 0x81c0, 0xddd7, 0xdac2, 0xd7b7,
	0xbdd6, 0xc6ce, 0xc4b7, 0x82c0, 0x83c0, 0xa6c5, 0xa3e7, 0xdfcf,
	0xa4e7, 0xa5e7, 0xa6e7, 0xb7c1, 0xe9d7, 0xf0c9, 0xb8cf, 0xafd6,
	0xd5d6, 0xa7e7, 0xedb0, 0xa8e7, 0xa9e7, 0xdcc9, 0xefd2, 0xadbe,
	0xaae7, 0xf3b0, 0xdec8, 0xe1bd, 0xabe7, 0xc6c8, 0x84c0, 0xace7,
	0xe6bb, 0xf8b8, 0xa4d1, 0xade7, 0xe7c2, 0xf8be, 0xcabd, 0xb3cd,
	0xaee7, 0xafe7, 0xeebe, 0xe5d0, 0x85c0, 0xe7cb, 0xd0cc, 0xccbc,
	0xb0e7, 0xa8bc, 0xf7d0, 0xb1e7, 0x86c0, 0xf8d0, 0xb2e7, 0xb3e7,
	0xc2b4, 0xb4e7, 0xb5e7, 0xfec9, 0xacce, 0xe0c3, 0xb7e7, 0xc1b1,
	0xf1b3, 0x87c0, 0xb8e7, 0xb9e7, 0xdbd7, 0xc0d5, 0xbae7, 0xccc2,
	0xbad7, 0xbbe7, 0xbce7, 0xbde7, 0xeabc, 0xe5c3, 0xc2c0, 0xbee7,
	0xbfe7, 0xa9bc, 0x88c0, 0xc0e7, 0xc1e7, 0xb6e7, 0xd0b6, 0xc2e7,
	0x89c0, 0xc3e7, 0xc4e7, 0xbabb, 0xdeb5, 0xc6c2, 0xe0b1, 0xc5e7,
	0xb5d4, 0xc6e7, 0xbfb8, 0xc8e7, 0xc7e7, 0xecb7, 0x8ac0, 0xc9e7,
	0xf8b2, 0xcae7, 0xcbe7, 0xcce7, 0xcde7, 0xcee7, 0xcfe7, 0xd0e7,
	0xa7d3, 0xf5cb, 0xd1e7, 0xd2e7, 0xd3e7, 0xd4e7, 0xc9c9, 0xd5e7,
	0xd6e7, 0xd7e7, 0xd8e7, 0xd9e7, 0xc9bd, 0xdae7, 0xbef3, 0x8bc0,
	0xd7b8, 0x8cc0, 0xb1c8, 0x8dc0, 0x8ec0, 0x8fc0, 0x90c0, 0x91c0,
	0x92c0, 0x93c0, 0xbff3, 0x94c0, 0xc0f3, 0xc1f3, 0x95c0, 0x96c0,
	0x97c0, 0x98c0, 0x99c0, 0x9ac0, 0x9bc0, 0x9cc0, 0x9dc0, 0x9ec0,
	0xdeb9, 0xf8cd, 0x9fc0, 0xa0c0, 0xe8d8, 0xb1ba, 0x40c1, 0xdec2,
	0xb7ee, 0x41c1, 0xa3b7, 0x42c1, 0x43c1, 0x44c1, 0x45c1, 0xb9ee,
	0x46c1, 0xb8ee, 0xd5b0, 0x47c1, 0x48c1, 0x49c1, 0x4ac1, 0x4bc1,
	0xbbee, 0xd6d5, 0xefd7, 0x4cc1, 0x4dc1, 0x4ec1, 0xc3d6, 0x4fc1,
	0x50c1, 0xbdee, 0xf0ca, 0x51c1, 0xbcee, 0x52c1, 0x53c1, 0x54c1,
	0x55c1, 0xbeee, 0x56c1, 0x57c1, 0x58c1, 0x59c1, 0xc0ee, 0x5ac1,
	0x5bc1, 0xbfee, 0x5cc1, 0x5dc1, 0x5ec1, 0x5fc1, 0x60c1, 0x61c1,
	0x62c1, 0x63c1, 0xf2d1, 0x64c1, 0xbcc7, 0x65c1, 0xc0c3, 0x66c1,
	0x67c1, 0x68c1, 0x69c1, 0x6ac1, 0xe1b8, 0x6bc1, 0x6cc1, 0x6dc1,
	0x6ec1, 0x6fc1, 0xe7c1, 0x70c1, 0x71c1, 0xc6f4, 0xdfd0, 0xc7f4,
	0x72c1, 0xdbcf, 0x73c1, 0x74c1, 0xbac8, 0x75c1, 0x76c1, 0xc8f4,
	0x77c1, 0x78c1, 0x79c1, 0x7ac1, 0x7bc1, 0x7cc1, 0x7dc1, 0xc9f4,
	0xcaf4, 0x7ec1, 0xcbf4, 0x80c1, 0x81c1, 0x82c1, 0x83c1, 0x84c1,
	0xfad9, 0xfeb8, 0x85c1, 0x86c1, 0xf1e5, 0xf0d3, 0x87c1, 0xe0f4,
	0x88c1, 0xccce, 0x89c1, 0x8ac1, 0x8bc1, 0xe1b3, 0x8cc1, 0x8dc1,
	0x8ec1, 0x8fc1, 0xb4f1, 0x90c1, 0xeed2, 0x91c1, 0xe1f4, 0x92c1,
	0x93c1, 0x94c1, 0x95c1, 0x96c1, 0xe8cf, 0xe2f4, 0x97c1, 0x98c1,
	0xccc7, 0x99c1, 0x9ac1, 0x9bc1, 0x9cc1, 0x9dc1, 0x9ec1, 0xd4b5,
	0xe4b4, 0xe4f4, 0x9fc1, 0xa0c1, 0x40c2, 0xe3f4, 0xe5f4, 0x41c2,
	0x42c2, 0xe6f4, 0x43c2, 0x44c2, 0x45c2, 0x46c2, 0xe7f4, 0x47c2,
	0xb2ba, 0xbfb0, 0x48c2, 0xe8f4, 0x49c2, 0x4ac2, 0x4bc2, 0x4cc2,
	0x4dc2, 0x4ec2, 0x4fc2, 0xadb7, 0xedd2, 0x50c2, 0x51c2, 0x52c2,
	0xabd2, 0xcfc0, 0x53c2, 0xbcbf, 0xa3eb, 0xdfd5, 0xc8ea, 0x54c2,
	0x55c2, 0x56c2, 0x57c2, 0xf3f1, 0xf8b6, 0xa3cb, 0x58c2, 0x59c2,
	0xcdc4, 0x5ac2, 0xe7f1, 0x5bc2, 0xe8f1, 0xfbb8, 0xe9f1, 0xc4ba,
	0xc5d4, 0xd2b0, 0x5cc2, 0x5dc2, 0xeaf1, 0x5ec2, 0x5fc2, 0x60c2,
	0xebf1, 0x61c2, 0xecf1, 0x62c2, 0x63c2, 0xedf1, 0xeef1, 0xeff1,
	0xf1f1, 0xf0f1, 0xd5c5, 0x64c2, 0x65c2, 0x66c2, 0x67c2, 0x68c2,
	0x69c2, 0xf2f1, 0x6ac2, 0xfab6, 0x6bc2, 0xf4f1, 0xaed2, 0xc7de,
	0xcacb, 0x6cc2, 0x6dc2, 0xdcb3, 0x6ec2, 0xa2b5, 0x6fc2, 0xa2b9,
	0x70c2, 0x71c2, 0xf4c4, 0xf5f1, 0x72c2, 0x73c2, 0xf6f1, 0x74c2,
	0x75c2, 0x76c2, 0xc4c1, 0xfbc1, 0xb0d6, 0xf7f1, 0x77c2, 0x78c2,
	0x79c2, 0x7ac2, 0xf8f1, 0x7bc2, 0xaac1, 0x7cc2, 0x7dc2, 0x7ec2,
	0xb8c6, 0x80c2, 0xdbbe, 0x81c2, 0x82c2, 0x83c2, 0x84c2, 0x85c2,
	0x86c2, 0x87c2, 0x88c2, 0x89c2, 0x8ac2, 0x8bc2, 0x8cc2, 0x8dc2,
	0x8ec2, 0xf9f1, 0xcfb4, 0x8fc2, 0x90c2, 0x91c2, 0x92c2, 0x93c2,
	0x94c2, 0xfaf1, 0x95c2, 0x96c2, 0x97c2, 0x98c2, 0x99c2, 0x9ac2,
	0x9bc2, 0x9cc2, 0x9dc2, 0x9ec2, 0x9fc2, 0xa0c2, 0x40c3, 0xb2ed,
	0xb1ed, 0x41c3, 0x42c3, 0xe0cb, 0xded2, 0x43c3, 0xc1cb, 0xd8d5,
	0x44c3, 0xe2c8, 0x45c3, 0xdfc0, 0xa1bc, 0x46c3, 0x47c3, 0x48c3,
	0x49c3, 0x4ac3, 0x4bc3, 0xc1eb, 0x4cc3, 0x4dc3, 0xa4d0, 0x4ec3,
	0xe2d6, 0x4fc3, 0xc7b6, 0xd8b8, 0xc0eb, 0xceb8, 0x50c3, 0xbfeb,
	0xa6b3, 0xc9b9, 0xabd6, 0x51c3, 0xf4b7, 0xcab7, 0x52c3, 0x53c3,
	0x54c3, 0xe7bc, 0xbeb7, 0xc6eb, 0x55c3, 0xc7eb, 0xb9b0, 0xcfbf,
	0x56c3, 0xc5eb, 0xfdd3, 0x57c3, 0xc8eb, 0x58c3, 0x59c3, 0xc9eb,
	0x5ac3, 0x5bc3, 0xceb7, 0x5cc3, 0xc2eb, 0xc4eb, 0xf6c9, 0xd7d6,
	0xcdd5, 0xb2d0, 0xcfeb, 0xb8ce, 0xd0eb, 0x5dc3, 0xa8b5, 0x5ec3,
	0x5fc3, 0x60c3, 0x61c3, 0x62c3, 0xb3b1, 0xd2eb, 0xa5cc, 0x63c3,
	0x64c3, 0x65c3, 0x66c3, 0x67c3, 0x68c3, 0x69c3, 0xd6c5, 0xd3eb,
	0x6ac3, 0xd1eb, 0xdfc5, 0xceeb, 0xa4ca, 0xd5eb, 0xfbb0, 0x6bc3,
	0x6cc3, 0xfaba, 0x6dc3, 0x6ec3, 0xb7d8, 0xe3f1, 0x6fc3, 0xcaeb,
	0xcbeb, 0xcceb, 0xcdeb, 0xd6eb, 0xc0e6, 0xd9eb, 0x70c3, 0xe8bf,
	0xc8d2, 0xd7eb, 0xdceb, 0xecb8, 0xd8eb, 0x71c3, 0xbabd, 0x72c3,
	0xd8d0, 0x73c3, 0xb7b0, 0x74c3, 0xddeb, 0xdcc4, 0x75c3, 0x76c3,
	0x77c3, 0x78c3, 0xacd6, 0x79c3, 0x7ac3, 0x7bc3, 0xe0b4, 0x7cc3,
	0x7dc3, 0xf6c2, 0xb9bc, 0x7ec3, 0x80c3, 0xdaeb, 0xdbeb, 0xe0d4,
	0xeac6, 0xd4c4, 0xdfeb, 0xa7c5, 0xf5d9, 0x81c3, 0xb1b2, 0x82c3,
	0xe4eb, 0x83c3, 0xc5bd, 0x84c3, 0x85c3, 0x86c3, 0xe2eb, 0x87c3,
	0x88c3, 0x89c3, 0x8ac3, 0x8bc3, 0x8cc3, 0x8dc3, 0x8ec3, 0x8fc3,
	0x90c3, 0x91c3, 0x92c3, 0x93c3, 0xe3eb, 0x94c3, 0x95c3, 0xacb8,
	0x96c3, 0xd1cd, 0xe5eb, 0x97c3, 0x98c3, 0x99c3, 0xe1eb, 0x9ac3,
	0xb3c1, 0x9bc3, 0x9cc3, 0x9dc3, 0x9ec3, 0x9fc3, 0xa2c6, 0xa0c3,
	0x40c4, 0x41c4, 0x42c4, 0x43c4, 0x44c4, 0x45c4, 0xf3cc, 0x46c4,
	0xe6eb, 0x47c4, 0xb0c0, 0xb8d2, 0xe7eb, 0x48c4, 0x49c4, 0x4ac4,
	0xafb8, 0xadb8, 0x4bc4, 0xe8eb, 0xbbc7, 0xf3cd, 0x4cc4, 0x4dc4,
	0x4ec4, 0xeaeb, 0xebeb, 0x4fc4, 0x50c4, 0x51c4, 0x52c4, 0x53c4,
	0xedeb, 0x54c4, 0x55c4, 0x56c4, 0x57c4, 0xc8d0, 0x58c4, 0xf2eb,
	0x59c4, 0xeeeb, 0x5ac4, 0x5bc4, 0x5cc4, 0xf1eb, 0xf9c8, 0x5dc4,
	0xfcd1, 0xeceb, 0x5ec4, 0x5fc4, 0xe9eb, 0x60c4, 0x61c4, 0x62c4,
	0x63c4, 0xb9b8, 0xd9cf, 0xe5c4, 0xefeb, 0xf0eb, 0xdacc, 0xc8cd,
	0xf2b0, 0x64c4, 0xf6eb, 0x65c4, 0x66c4, 0x67c4, 0x68c4, 0x69c4,
	0xf5eb, 0x6ac4, 0xb2b2, 0x6bc4, 0x6cc4, 0x6dc4, 0x6ec4, 0xe0b8,
	0x6fc4, 0xf7eb, 0x70c4, 0x71c4, 0x72c4, 0x73c4, 0x74c4, 0x75c4,
	0xecb1, 0x76c4, 0x77c4, 0xc5cc, 0xa4c4, 0xa5cf, 0x78c4, 0x79c4,
	0x7ac4, 0x7bc4, 0x7cc4, 0xf9eb, 0x7dc4, 0x7ec4, 0xa2ec, 0x80c4,
	0xf2c5, 0x81c4, 0xfaeb, 0x82c4, 0x83c4, 0x84c4, 0x85c4, 0x86c4,
	0x87c4, 0x88c4, 0x89c4, 0xc5c9, 0x8ac4, 0x8bc4, 0x8cc4, 0x8dc4,
	0x8ec4, 0x8fc4, 0xdfe2, 0xfeeb, 0x90c4, 0x91c4, 0x92c4, 0x93c4,
	0xcecd, 0xa1ec, 0xdbb1, 0xb7d3, 0x94c4, 0x95c4, 0xdcd2, 0x96c4,
	0x97c4, 0x98c4, 0xfdeb, 0x99c4, 0xfbeb, 0x9ac4, 0x9bc4, 0x9cc4,
	0x9dc4, 0x9ec4, 0x9fc4, 0xa0c4, 0x40c5, 0x41c5, 0x42c5, 0x43c5,
	0x44c5, 0x45c5, 0x46c5, 0x47c5, 0x48c5, 0x49c5, 0x4ac5, 0x4bc5,
	0x4cc5, 0x4dc5, 0x4ec5, 0xbcb3, 0x4fc5, 0x50c5, 0x51c5, 0xb0ea,
	0x52c5, 0x53c5, 0xd4d7, 0x54c5, 0xabf4, 0xf4b3, 0x55c5, 0x56c5,
	0x57c5, 0x58c5, 0x59c5, 0xc1d6, 0xc2d6, 0x5ac5, 0x5bc5, 0x5cc5,
	0x5dc5, 0x5ec5, 0x5fc5, 0xe9d5, 0xcabe, 0x60c5, 0xa7f4, 0x61c5,
	0xa8d2, 0xa8f4, 0xa9f4, 0x62c5, 0xaaf4, 0xcbbe, 0xdfd3, 0x63c5,
	0x64c5, 0x65c5, 0x66c5, 0x67c5, 0xe0c9, 0xe1c9, 0x68c5, 0x69c5,
	0xc2f3, 0x6ac5, 0xe6ca, 0x6bc5, 0xf2cc, 0x6cc5, 0x6dc5, 0x6ec5,
	0x6fc5, 0x70c5, 0x71c5, 0xb6e2, 0xb4cb, 0x72c5, 0xe8ce, 0xdbd6,
	0x73c5, 0xadf4, 0xaef4, 0xaff4, 0x74c5, 0x75c5, 0x76c5, 0x77c5,
	0xb2f4, 0x78c5, 0xbdba, 0xb3f4, 0xe3b0, 0xb0f4, 0x79c5, 0xb1f4,
	0xa2bd, 0xd5b2, 0x7ac5, 0xb6f4, 0xb7f4, 0xe6b6, 0xb0b2, 0xcfcf,
	0xb4f4, 0xacb4, 0x7bc5, 0xb5f4, 0x7cc5, 0x7dc5, 0xb8f4, 0x7ec5,
	0x80c5, 0x81c5, 0x82c5, 0x83c5, 0xb9f4, 0x84c5, 0x85c5, 0xa7cd,
	0x86c5, 0xbaf4, 0x87c5, 0xbbf4, 0x88c5, 0x89c5, 0x8ac5, 0xbcf4,
	0x8bc5, 0x8cc5, 0x8dc5, 0x8ec5, 0x8fc5, 0x90c5, 0x91c5, 0x92c5,
	0xd2cb, 0x93c5, 0xbdf4, 0x94c5, 0x95c5, 0x96c5, 0x97c5, 0xbef4,
	0x98c5, 0x99c5, 0x9ac5, 0x9bc5, 0x9cc5, 0x9dc5, 0x9ec5, 0x9fc5,
	0xbff4, 0xa0c5, 0x40c6, 0x41c6, 0x42c6, 0x43c6, 0xdef4, 0xbcc1,
	0xe8bc, 0x44c6, 0xabc9, 0xded1, 0xf5e5, 0x45c6, 0x46c6, 0x47c6,
	0x48c6, 0xb3dc, 0xd5d2, 0x49c6, 0x4ac6, 0xb4dc, 0xacb0, 0xb5dc,
	0x4bc6, 0x4cc6, 0xdabd, 0x4dc6, 0xb9dc, 0x4ec6, 0x4fc6, 0x50c6,
	0xc2d8, 0x51c6, 0xb7dc, 0xf3d3, 0x52c6, 0xd6c9, 0xbadc, 0xb6dc,
	0x53c6, 0xbbdc, 0xa2c3, 0x54c6, 0x55c6, 0x56c6, 0x57c6, 0xbcdc,
	0xc5dc, 0xbddc, 0x58c6, 0x59c6, 0xdfce, 0xa5d6, 0x5ac6, 0xcfdc,
	0x5bc6, 0xcddc, 0x5cc6, 0x5dc6, 0xd2dc, 0xe6bd, 0xabc2, 0x5ec6,
	0xb8dc, 0xcbdc, 0xcedc, 0xbedc, 0xd2b7, 0xc5b0, 0xc7dc, 0xbed0,
	0xc1dc, 0xa8bb, 0x5fc6, 0xbcb7, 0xccdc, 0x60c6, 0x61c6, 0xc6dc,
	0xbfdc, 0xdbc7, 0x62c6, 0x63c6, 0x64c6, 0xbfd1, 0xc0dc, 0x65c6,
	0x66c6, 0xcadc, 0x67c6, 0x68c6, 0xd0dc, 0x69c6, 0x6ac6, 0xadce,
	0xc2dc, 0x6bc6, 0xc3dc, 0xc8dc, 0xc9dc, 0xd4b2, 0xd1dc, 0xd5cb,
	0x6cc6, 0xb7d4, 0xdbdc, 0xdfdc, 0xa6cc, 0xe6dc, 0x6dc6, 0xe7c3,
	0xdcdc, 0x6ec6, 0x6fc6, 0xc1bf, 0xd9dc, 0x70c6, 0xfab0, 0xb6b9,
	0xe5dc, 0xd3dc, 0x71c6, 0xc4dc, 0xd6dc, 0xf4c8, 0xe0bf, 0x72c6,
	0x73c6, 0x74c6, 0x75c6, 0xbbc9, 0x76c6, 0x77c6, 0x78c6, 0xbdb1,
	0x79c6, 0xa2d3, 0x7ac6, 0x7bc6, 0xdadc, 0x7cc6, 0x7dc6, 0xd5dc,
	0x7ec6, 0xbbc6, 0x80c6, 0xdedc, 0x81c6, 0x82c6, 0x83c6, 0x84c6,
	0x85c6, 0xc2d7, 0xafc3, 0xb6b7, 0xd1c7, 0xa9c3, 0xe2dc, 0xd8dc,
	0xebdc, 0xd4dc, 0x86c6, 0x87c6, 0xdddc, 0x88c6, 0xa5be, 0xd7dc,
	0x89c6, 0xe0dc, 0x8ac6, 0x8bc6, 0xe3dc, 0xe4dc, 0x8cc6, 0xf8dc,
	0x8dc6, 0x8ec6, 0xe1dc, 0xa2dd, 0xe7dc, 0x8fc6, 0x90c6, 0x91c6,
	0x92c6, 0x93c6, 0x94c6, 0x95c6, 0x96c6, 0x97c6, 0x98c6, 0xebbc,
	0xc4b4, 0x99c6, 0x9ac6, 0xa3c3, 0xe7b2, 0xfadc, 0x9bc6, 0xf2dc,
	0x9cc6, 0xefdc, 0x9dc6, 0xfcdc, 0xeedc, 0xf0d2, 0xe8b2, 0x9ec6,
	0xd7c8, 0xe3c8, 0xfbdc, 0x9fc6, 0xeddc, 0xa0c6, 0x40c7, 0x41c7,
	0xf7dc, 0x42c7, 0x43c7, 0xf5dc, 0x44c7, 0x45c7, 0xa3be, 0xf4dc,
	0x46c7, 0xddb2, 0x47c7, 0x48c7, 0x49c7, 0x4ac7, 0x4bc7, 0xf3dc,
	0xf6bc, 0xe8dc, 0xc4bb, 0x4cc7, 0xf3c0, 0x4dc7, 0x4ec7, 0x4fc7,
	0x50c7, 0x51c7, 0xd4bc, 0xe9dc, 0xeadc, 0x52c7, 0xf1dc, 0xf6dc,
	0xf9dc, 0xb4b5, 0x53c7, 0xd9c8, 0xe7bb, 0xfedc, 0xfddc, 0xabd3,
	0xa1dd, 0xa3dd, 0xa5dd, 0xf1d2, 0xa4dd, 0xa6dd, 0xa7dd, 0xa9d2,
	0x54c7, 0x55c7, 0x56c7, 0x57c7, 0x58c7, 0x59c7, 0x5ac7, 0xc9ba,
	0xa9dd, 0x5bc7, 0x5cc7, 0xb6dd, 0xb1dd, 0xb4dd, 0x5dc7, 0x5ec7,
	0x5fc7, 0x60c7, 0x61c7, 0x62c7, 0x63c7, 0xb0dd, 0xcec6, 0x64c7,
	0x65c7, 0xf2c0, 0x66c7, 0x67c7, 0x68c7, 0x69c7, 0xafc9, 0x6ac7,
	0x6bc7, 0x6cc7, 0xecdc, 0xaedd, 0x6dc7, 0x6ec7, 0x6fc7, 0x70c7,
	0xb7dd, 0x71c7, 0x72c7, 0xf0dc, 0xafdd, 0x73c7, 0xb8dd, 0x74c7,
	0xacdd, 0x75c7, 0x76c7, 0x77c7, 0x78c7, 0x79c7, 0x7ac7, 0x7bc7,
	0xb9dd, 0xb3dd, 0xaddd, 0xaac4, 0x7cc7, 0x7dc7, 0x7ec7, 0x80c7,
	0xa8dd, 0xb3c0, 0xabc1, 0xaadd, 0xabdd, 0x81c7, 0xb2dd, 0xf1bb,
	0xb5dd, 0xa8d3, 0xbadd, 0x82c7, 0xbbdd, 0xa7c3, 0x83c7, 0x84c7,
	0xd2dd, 0xbcdd, 0x85c7, 0x86c7, 0x87c7, 0xd1dd, 0x88c7, 0xbdb9,
	0x89c7, 0x8ac7, 0xd5be, 0x8bc7, 0xfabe, 0x8cc7, 0x8dc7, 0xcaba,
	0x8ec7, 0x8fc7, 0x90c7, 0x91c7, 0xcadd, 0x92c7, 0xc5dd, 0x93c7,
	0xbfdd, 0x94c7, 0x95c7, 0x96c7, 0xcbb2, 0xc3dd, 0x97c7, 0xcbdd,
	0xa4b2, 0xd5dd, 0x98c7, 0x99c7, 0x9ac7, 0xbedd, 0x9bc7, 0x9cc7,
	0x9dc7, 0xd0c6, 0xd0dd, 0x9ec7, 0x9fc7, 0xa0c7, 0x40c8, 0x41c8,
	0xd4dd, 0xe2c1, 0xc6b7, 0x42c8, 0x43c8, 0x44c8, 0x45c8, 0x46c8,
	0xcedd, 0xcfdd, 0x47c8, 0x48c8, 0x49c8, 0xc4dd, 0x4ac8, 0x4bc8,
	0x4cc8, 0xbddd, 0x4dc8, 0xcddd, 0xd1cc, 0x4ec8, 0xc9dd, 0x4fc8,
	0x50c8, 0x51c8, 0x52c8, 0xc2dd, 0xc8c3, 0xbcc6, 0xaece, 0xccdd,
	0x53c8, 0xc8dd, 0x54c8, 0x55c8, 0x56c8, 0x57c8, 0x58c8, 0x59c8,
	0xc1dd, 0x5ac8, 0x5bc8, 0x5cc8, 0xc6dd, 0xdcc2, 0x5dc8, 0x5ec8,
	0x5fc8, 0x60c8, 0x61c8, 0x62c8, 0xa9d3, 0xaad3, 0xd3dd, 0xf4cf,
	0xf8c8, 0x63c8, 0x64c8, 0x65c8, 0x66c8, 0x67c8, 0x68c8, 0x69c8,
	0x6ac8, 0xe6dd, 0x6bc8, 0x6cc8, 0x6dc8, 0x6ec8, 0x6fc8, 0x70c8,
	0xc7dd, 0x71c8, 0x72c8, 0x73c8, 0xe0dd, 0xe4c2, 0x74c8, 0x75c8,
	0x76c8, 0x77c8, 0x78c8, 0x79c8, 0x7ac8, 0x7bc8, 0xe1dd, 0x7cc8,
	0x7dc8, 0x7ec8, 0x80c8, 0x81c8, 0x82c8, 0x83c8, 0x84c8, 0x85c8,
	0x86c8, 0xd7dd, 0x87c8, 0x88c8, 0x89c8, 0x8ac8, 0x8bc8, 0xf8d6,
	0x8cc8, 0xd9dd, 0xd8dd, 0xf0b8, 0xd6dd, 0x8dc8, 0x8ec8, 0x8fc8,
	0x90c8, 0xcfc6, 0x91c8, 0xadb6, 0x92c8, 0x93c8, 0x94c8, 0x95c8,
	0x96c8, 0xe2dd, 0x97c8, 0xf9ba, 0xe1d4, 0xe7dd, 0x98c8, 0x99c8,
	0x9ac8, 0xd0b4, 0x9bc8, 0xdadd, 0x9cc8, 0xfbbf, 0xe3dd, 0x9dc8,
	0xdfdd, 0x9ec8, 0xdddd, 0x9fc8, 0xa0c8, 0x40c9, 0x41c9, 0x42c9,
	0x43c9, 0x44c9, 0xd9b5, 0x45c9, 0x46c9, 0x47c9, 0x48c9, 0xdbdd,
	0xdcdd, 0xdedd, 0x49c9, 0xafbd, 0xe4dd, 0x4ac9, 0xe5dd, 0x4bc9,
	0x4cc9, 0x4dc9, 0x4ec9, 0x4fc9, 0x50c9, 0x51c9, 0x52c9, 0xf5dd,
	0x53c9, 0xc9c3, 0x54c9, 0x55c9, 0xe2cb, 0x56c9, 0x57c9, 0x58c9,
	0x59c9, 0xf2dd, 0x5ac9, 0x5bc9, 0x5cc9, 0x5dc9, 0x5ec9, 0x5fc9,
	0x60c9, 0x61c9, 0x62c9, 0x63c9, 0x64c9, 0x65c9, 0x66c9, 0xe1d8,
	0x67c9, 0x68c9, 0xd1c6, 0x69c9, 0xf4dd, 0x6ac9, 0x6bc9, 0x6cc9,
	0xf4d5, 0xf3dd, 0xf0dd, 0x6dc9, 0x6ec9, 0xecdd, 0x6fc9, 0xefdd,
	0x70c9, 0xe8dd, 0x71c9, 0x72c9, 0xeed0, 0x73c9, 0x74c9, 0x75c9,
	0x76c9, 0xd8c8, 0xeedd, 0x77c9, 0x78c9, 0xe9dd, 0x79c9, 0x7ac9,
	0xeadd, 0xf2cb, 0x7bc9, 0xeddd, 0x7cc9, 0x7dc9, 0xcdb1, 0x7ec9,
	0x80c9, 0x81c9, 0x82c9, 0x83c9, 0x84c9, 0xb6c0, 0x85c9, 0xbbbc,
	0xf1dd, 0x86c9, 0x87c9, 0xf7dd, 0x88c9, 0xf6dd, 0xebdd, 0x89c9,
	0x8ac9, 0x8bc9, 0x8cc9, 0x8dc9, 0xeec5, 0x8ec9, 0x8fc9, 0x90c9,
	0xfbdd, 0x91c9, 0x92c9, 0x93c9, 0x94c9, 0x95c9, 0x96c9, 0x97c9,
	0x98c9, 0x99c9, 0x9ac9, 0x9bc9, 0xa4de, 0x9cc9, 0x9dc9, 0xa3de,
	0x9ec9, 0x9fc9, 0xa0c9, 0x40ca, 0x41ca, 0x42ca, 0x43ca, 0x44ca,
	0x45ca, 0x46ca, 0x47ca, 0x48ca, 0xf8dd, 0x49ca, 0x4aca, 0x4bca,
	0x4cca, 0xefc3, 0x4dca, 0xfbc2, 0x4eca, 0x4fca, 0x50ca, 0xe1d5,
	0x51ca, 0x52ca, 0xb5ce, 0x53ca, 0x54ca, 0x55ca, 0x56ca, 0xfddd,
	0x57ca, 0xccb2, 0x58ca, 0x59ca, 0x5aca, 0x5bca, 0x5cca, 0x5dca,
	0x5eca, 0x5fca, 0x60ca, 0xe8c4, 0xdfca, 0x61ca, 0x62ca, 0x63ca,
	0x64ca, 0x65ca, 0x66ca, 0x67ca, 0x68ca, 0x69ca, 0x6aca, 0xbec7,
	0xfadd, 0xfcdd, 0xfedd, 0xa2de, 0xaab0, 0xceb1, 0x6bca, 0x6cca,
	0x6dca, 0x6eca, 0x6fca, 0xacde, 0x70ca, 0x71ca, 0x72ca, 0x73ca,
	0xa6de, 0xb6bd, 0xefc8, 0x74ca, 0x75ca, 0x76ca, 0x77ca, 0x78ca,
	0x79ca, 0x7aca, 0x7bca, 0x7cca, 0x7dca, 0x7eca, 0xa1de, 0x80ca,
	0x81ca, 0xa5de, 0x82ca, 0x83ca, 0x84ca, 0x85ca, 0xa9de, 0x86ca,
	0x87ca, 0x88ca, 0x89ca, 0x8aca, 0xa8de, 0x8bca, 0x8cca, 0x8dca,
	0xa7de, 0x8eca, 0x8fca, 0x90ca, 0x91ca, 0x92ca, 0x93ca, 0x94ca,
	0x95ca, 0x96ca, 0xadde, 0x97ca, 0xccd4, 0x98ca, 0x99ca, 0x9aca,
	0x9bca, 0xb3de, 0xaade, 0xaede, 0x9cca, 0x9dca, 0xd9c0, 0x9eca,
	0x9fca, 0xa0ca, 0x40cb, 0x41cb, 0xa1b1, 0xb6de, 0x42cb, 0xb1de,
	0x43cb, 0x44cb, 0x45cb, 0x46cb, 0x47cb, 0x48cb, 0x49cb, 0xb2de,
	0x4acb, 0x4bcb, 0x4ccb, 0x4dcb, 0x4ecb, 0x4fcb, 0x50cb, 0x51cb,
	0x52cb, 0x53cb, 0x54cb, 0xa6d1, 0xb5de, 0x55cb, 0x56cb, 0x57cb,
	0x58cb, 0x59cb, 0x5acb, 0x5bcb, 0xafde, 0x5ccb, 0x5dcb, 0x5ecb,
	0xb0de, 0x5fcb, 0xbdd0, 0x60cb, 0x61cb, 0x62cb, 0xb4de, 0xedca,
	0xb9de, 0x63cb, 0x64cb, 0x65cb, 0x66cb, 0x67cb, 0x68cb, 0xb8de,
	0x69cb, 0xb7de, 0x6acb, 0x6bcb, 0x6ccb, 0x6dcb, 0x6ecb, 0x6fcb,
	0x70cb, 0xbbde, 0x71cb, 0x72cb, 0x73cb, 0x74cb, 0x75cb, 0x76cb,
	0x77cb, 0xe5bd, 0x78cb, 0x79cb, 0x7acb, 0x7bcb, 0x7ccb, 0xd8b2,
	0xeac3, 0x7dcb, 0x7ecb, 0xbade, 0x80cb, 0xbac5, 0x81cb, 0x82cb,
	0x83cb, 0x84cb, 0x85cb, 0x86cb, 0xbcde, 0x87cb, 0x88cb, 0x89cb,
	0x8acb, 0x8bcb, 0x8ccb, 0x8dcb, 0xd9cc, 0x8ecb, 0x8fcb, 0x90cb,
	0x91cb, 0xaab7, 0x92cb, 0x93cb, 0x94cb, 0x95cb, 0x96cb, 0x97cb,
	0x98cb, 0x99cb, 0x9acb, 0x9bcb, 0x9ccb, 0x9dcb, 0x9ecb, 0x9fcb,
	0xa0cb, 0x40cc, 0x41cc, 0xe5d4, 0x42cc, 0x43cc, 0x44cc, 0xbdde,
	0x45cc, 0x46cc, 0x47cc, 0x48cc, 0x49cc, 0xbfde, 0x4acc, 0x4bcc,
	0x4ccc, 0x4dcc, 0x4ecc, 0x4fcc, 0x50cc, 0x51cc, 0x52cc, 0x53cc,
	0x54cc, 0xa2c4, 0x55cc, 0x56cc, 0x57cc, 0x58cc, 0xc1de, 0x59cc,
	0x5acc, 0x5bcc, 0x5ccc, 0x5dcc, 0x5ecc, 0x5fcc, 0x60cc, 0x61cc,
	0x62cc, 0x63cc, 0x64cc, 0x65cc, 0x66cc, 0x67cc, 0x68cc, 0xbede,
	0x69cc, 0xc0de, 0x6acc, 0x6bcc, 0x6ccc, 0x6dcc, 0x6ecc, 0x6fcc,
	0x70cc, 0x71cc, 0x72cc, 0x73cc, 0x74cc, 0x75cc, 0x76cc, 0x77cc,
	0xbad5, 0x78cc, 0x79cc, 0x7acc, 0xc2de, 0x7bcc, 0x7ccc, 0x7dcc,
	0x7ecc, 0x80cc, 0x81cc, 0x82cc, 0x83cc, 0x84cc, 0x85cc, 0x86cc,
	0x87cc, 0x88cc, 0x89cc, 0x8acc, 0x8bcc, 0xaef2, 0xa2bb, 0xb2c2,
	0xb0c5, 0xc7c2, 0x8ccc, 0x8dcc, 0xaff2, 0x8ecc, 0x8fcc, 0x90cc,
	0x91cc, 0x92cc, 0xe9d0, 0x93cc, 0x94cc, 0x95cc, 0xddd3, 0x96cc,
	0x97cc, 0x98cc, 0xbdeb, 0x99cc, 0x9acc, 0x9bcc, 0x9ccc, 0x9dcc,
	0x9ecc, 0x9fcc, 0xa0cc, 0xe6b3, 0xb0f2, 0x40cd, 0xb1f2, 0x41cd,
	0x42cd, 0xadca, 0x43cd, 0x44cd, 0x45cd, 0x46cd, 0x47cd, 0x48cd,
	0x49cd, 0xe7ba, 0xb3f2, 0xb5f2, 0xb4f2, 0xe4cb, 0xbacf, 0xb2f2,
	0xb4ca, 0xcfd2, 0xecc2, 0x4acd, 0x4bcd, 0x4ccd, 0x4dcd, 0x4ecd,
	0x4fcd, 0x50cd, 0xc3ce, 0xb8f2, 0xf6b0, 0xb7f2, 0x51cd, 0x52cd,
	0x53cd, 0x54cd, 0x55cd, 0xbef2, 0x56cd, 0xcfb2, 0x57cd, 0x58cd,
	0x59cd, 0x5acd, 0x5bcd, 0x5ccd, 0xc1d1, 0xbaf2, 0x5dcd, 0x5ecd,
	0x5fcd, 0x60cd, 0x61cd, 0xbcf2, 0xe9d4, 0x62cd, 0x63cd, 0xbbf2,
	0xb6f2, 0xbff2, 0xbdf2, 0x64cd, 0xb9f2, 0x65cd, 0x66cd, 0xc7f2,
	0xc4f2, 0xc6f2, 0x67cd, 0x68cd, 0xcaf2, 0xc2f2, 0xc0f2, 0x69cd,
	0x6acd, 0x6bcd, 0xc5f2, 0x6ccd, 0x6dcd, 0x6ecd, 0x6fcd, 0x70cd,
	0xfbd6, 0x71cd, 0x72cd, 0x73cd, 0xc1f2, 0x74cd, 0xf9c7, 0xdfc9,
	0x75cd, 0xc8f2, 0xc6b9, 0xb0b5, 0x76cd, 0x77cd, 0xc3f2, 0xc9f2,
	0xd0f2, 0xd6f2, 0x78cd, 0x79cd, 0xd7bb, 0x7acd, 0x7bcd, 0x7ccd,
	0xd5f2, 0xdccd, 0x7dcd, 0xebd6, 0x7ecd, 0x80cd, 0xd2f2, 0xd4f2,
	0x81cd, 0x82cd, 0x83cd, 0x84cd, 0xf2b8, 0x85cd, 0x86cd, 0x87cd,
	0x88cd, 0xcbf2, 0x89cd, 0x8acd, 0x8bcd, 0xcef2, 0xf9c2, 0x8ccd,
	0xddd5, 0xccf2, 0xcdf2, 0xcff2, 0xd3f2, 0x8dcd, 0x8ecd, 0x8fcd,
	0xd9f2, 0xbcd3, 0x90cd, 0x91cd, 0x92cd, 0x93cd, 0xeab6, 0x94cd,
	0xf1ca, 0x95cd, 0xe4b7, 0xd7f2, 0x96cd, 0x97cd, 0x98cd, 0xd8f2,
	0xdaf2, 0xddf2, 0xdbf2, 0x99cd, 0x9acd, 0xdcf2, 0x9bcd, 0x9ccd,
	0x9dcd, 0x9ecd, 0xd1d1, 0xd1f2, 0x9fcd, 0xc9cd, 0xa0cd, 0xcfce,
	0xa9d6, 0x40ce, 0xe3f2, 0x41ce, 0xdbc3, 0x42ce, 0xe0f2, 0x43ce,
	0x44ce, 0xafc0, 0xecf2, 0xdef2, 0x45ce, 0xe1f2, 0x46ce, 0x47ce,
	0x48ce, 0xe8f2, 0x49ce, 0x4ace, 0x4bce, 0x4cce, 0xe2f2, 0x4dce,
	0x4ece, 0xe7f2, 0x4fce, 0x50ce, 0xe6f2, 0x51ce, 0x52ce, 0xe9f2,
	0x53ce, 0x54ce, 0x55ce, 0xdff2, 0x56ce, 0x57ce, 0xe4f2, 0xeaf2,
	0x58ce, 0x59ce, 0x5ace, 0x5bce, 0x5cce, 0x5dce, 0x5ece, 0xacd3,
	0xe5f2, 0xf5b2, 0x5fce, 0x60ce, 0xf2f2, 0x61ce, 0xabd0, 0x62ce,
	0x63ce, 0x64ce, 0x65ce, 0xf5f2, 0x66ce, 0x67ce, 0x68ce, 0xc8bb,
	0x69ce, 0xf9f2, 0x6ace, 0x6bce, 0x6cce, 0x6dce, 0x6ece, 0x6fce,
	0xf0f2, 0x70ce, 0x71ce, 0xf6f2, 0xf8f2, 0xfaf2, 0x72ce, 0x73ce,
	0x74ce, 0x75ce, 0x76ce, 0x77ce, 0x78ce, 0x79ce, 0xf3f2, 0x7ace,
	0xf1f2, 0x7bce, 0x7cce, 0x7dce, 0xfbba, 0x7ece, 0xfbb5, 0x80ce,
	0x81ce, 0x82ce, 0x83ce, 0xeff2, 0xf7f2, 0xedf2, 0xeef2, 0x84ce,
	0x85ce, 0x86ce, 0xebf2, 0xa6f3, 0x87ce, 0xa3f3, 0x88ce, 0x89ce,
	0xa2f3, 0x8ace, 0x8bce, 0xf4f2, 0x8cce, 0xdac8, 0x8dce, 0x8ece,
	0x8fce, 0x90ce, 0x91ce, 0xfbf2, 0x92ce, 0x93ce, 0x94ce, 0xa5f3,
	0x95ce, 0x96ce, 0x97ce, 0x98ce, 0x99ce, 0x9ace, 0x9bce, 0xf8c3,
	0x9cce, 0x9dce, 0x9ece, 0x9fce, 0xa0ce, 0x40cf, 0x41cf, 0x42cf,
	0xfdf2, 0x43cf, 0x44cf, 0xa7f3, 0xa9f3, 0xa4f3, 0x45cf, 0xfcf2,
	0x46cf, 0x47cf, 0x48cf, 0xabf3, 0x49cf, 0xaaf3, 0x4acf, 0x4bcf,
	0x4ccf, 0x4dcf, 0xddc2, 0x4ecf, 0x4fcf, 0xaef3, 0x50cf, 0x51cf,
	0xb0f3, 0x52cf, 0x53cf, 0x54cf, 0x55cf, 0x56cf, 0xa1f3, 0x57cf,
	0x58cf, 0x59cf, 0xb1f3, 0xacf3, 0x5acf, 0x5bcf, 0x5ccf, 0x5dcf,
	0x5ecf, 0xaff3, 0xfef2, 0xadf3, 0x5fcf, 0x60cf, 0x61cf, 0x62cf,
	0x63cf, 0x64cf, 0x65cf, 0xb2f3, 0x66cf, 0x67cf, 0x68cf, 0x69cf,
	0xb4f3, 0x6acf, 0x6bcf, 0x6ccf, 0x6dcf, 0xa8f3, 0x6ecf, 0x6fcf,
	0x70cf, 0x71cf, 0xb3f3, 0x72cf, 0x73cf, 0x74cf, 0xb5f3, 0x75cf,
	0x76cf, 0x77cf, 0x78cf, 0x79cf, 0x7acf, 0x7bcf, 0x7ccf, 0x7dcf,
	0x7ecf, 0xb7d0, 0x80cf, 0x81cf, 0x82cf, 0x83cf, 0xb8f3, 0x84cf,
	0x85cf, 0x86cf, 0x87cf, 0xf9d9, 0x88cf, 0x89cf, 0x8acf, 0x8bcf,
	0x8ccf, 0x8dcf, 0xb9f3, 0x8ecf, 0x8fcf, 0x90cf, 0x91cf, 0x92cf,
	0x93cf, 0x94cf, 0x95cf, 0xb7f3, 0x96cf, 0xe4c8, 0xb6f3, 0x97cf,
	0x98cf, 0x99cf, 0x9acf, 0xbaf3, 0x9bcf, 0x9ccf, 0x9dcf, 0x9ecf,
	0x9fcf, 0xbbf3, 0xc0b4, 0xa0cf, 0x40d0, 0x41d0, 0x42d0, 0x43d0,
	0x44d0, 0x45d0, 0x46d0, 0x47d0, 0x48d0, 0x49d0, 0x4ad0, 0x4bd0,
	0x4cd0, 0x4dd0, 0xc3ee, 0x4ed0, 0x4fd0, 0x50d0, 0x51d0, 0x52d0,
	0x53d0, 0xbcf3, 0x54d0, 0x55d0, 0xbdf3, 0x56d0, 0x57d0, 0x58d0,
	0xaad1, 0x59d0, 0x5ad0, 0x5bd0, 0xacf4, 0xc6d0, 0x5cd0, 0x5dd0,
	0x5ed0, 0x5fd0, 0x60d0, 0x61d0, 0xd0d0, 0xdcd1, 0x62d0, 0x63d0,
	0x64d0, 0x65d0, 0x66d0, 0x67d0, 0xcecf, 0x68d0, 0x69d0, 0xd6bd,
	0x6ad0, 0xc3d1, 0x6bd0, 0x6cd0, 0x6dd0, 0x6ed0, 0x6fd0, 0x70d0,
	0x71d0, 0xe2ba, 0xe9e1, 0xc2d2, 0xc2f1, 0xb9b2, 0x72d0, 0x73d0,
	0xedb1, 0xc3f1, 0x74d0, 0xc0c9, 0xc4b3, 0x75d0, 0xf2d9, 0x76d0,
	0xa5cb, 0x77d0, 0xc4f1, 0x78d0, 0x79d0, 0x7ad0, 0x7bd0, 0xd4d6,
	0x7cd0, 0x7dd0, 0x7ed0, 0x80d0, 0x81d0, 0xc5f1, 0xc0f4, 0xc6f1,
	0x82d0, 0xacd4, 0xc7f1, 0x83d0, 0xc0b0, 0xc1f4, 0x84d0, 0x85d0,
	0xc2f4, 0x86d0, 0x87d0, 0xfcb4, 0x88d0, 0xdbc5, 0x89d0, 0x8ad0,
	0x8bd0, 0x8cd0, 0xbbcc, 0x8dd0, 0x8ed0, 0x8fd0, 0xe4d0, 0x90d0,
	0x91d0, 0x92d0, 0x93d0, 0x94d0, 0xe0cd, 0x95d0, 0x96d0, 0x97d0,
	0x98d0, 0x99d0, 0xc8f1, 0x9ad0, 0xf3d9, 0x9bd0, 0x9cd0, 0x9dd0,
	0x9ed0, 0x9fd0, 0xa0d0, 0xbbb1, 0x40d1, 0xaecf, 0x41d1, 0x42d1,
	0x43d1, 0xa4b8, 0x44d1, 0x45d1, 0x46d1, 0x47d1, 0x48d1, 0xcaf1,
	0x49d1, 0x4ad1, 0x4bd1, 0x4cd1, 0xcbf1, 0x4dd1, 0x4ed1, 0x4fd1,
	0x50d1, 0xc3b2, 0xd1c1, 0x51d1, 0x52d1, 0xb0d7, 0xc9f1, 0x53d1,
	0x54d1, 0xccf1, 0x55d1, 0x56d1, 0x57d1, 0x58d1, 0xcef1, 0x59d1,
	0x5ad1, 0x5bd1, 0xf6d9, 0x5cd1, 0xe1d2, 0xa3d4, 0x5dd1, 0x5ed1,
	0xc3f4, 0xb9c8, 0x5fd1, 0x60d1, 0x61d1, 0x62d1, 0x63d1, 0xc4f4,
	0x64d1, 0x65d1, 0xcdf1, 0xcff1, 0xe3bf, 0xd0f1, 0x66d1, 0x67d1,
	0xd4f1, 0x68d1, 0x69d1, 0x6ad1, 0x6bd1, 0x6cd1, 0x6dd1, 0x6ed1,
	0xd6f1, 0xd1f1, 0x6fd1, 0xd1c9, 0xe1c5, 0x70d1, 0x71d1, 0x72d1,
	0xe3c2, 0xfcb9, 0x73d1, 0x74d1, 0xd3f1, 0x75d1, 0xd5f1, 0x76d1,
	0x77d1, 0x78d1, 0xd3b9, 0x79d1, 0x7ad1, 0x7bd1, 0x7cd1, 0x7dd1,
	0x7ed1, 0x80d1, 0xdbf1, 0x81d1, 0x82d1, 0x83d1, 0x84d1, 0x85d1,
	0xd6ba, 0x86d1, 0xfdb0, 0xd9f1, 0x87d1, 0x88d1, 0x89d1, 0x8ad1,
	0x8bd1, 0xd8f1, 0xd2f1, 0xdaf1, 0x8cd1, 0x8dd1, 0x8ed1, 0x8fd1,
	0x90d1, 0xd7f1, 0x91d1, 0x92d1, 0x93d1, 0xecc8, 0x94d1, 0x95d1,
	0x96d1, 0x97d1, 0xcacd, 0xddf1, 0x98d1, 0x99d1, 0x9ad1, 0x9bd1,
	0xbde5, 0x9cd1, 0x9dd1, 0x9ed1, 0xdcf1, 0x9fd1, 0xdef1, 0xa0d1,
	0x40d2, 0x41d2, 0x42d2, 0x43d2, 0x44d2, 0x45d2, 0x46d2, 0x47d2,
	0x48d2, 0xdff1, 0x49d2, 0x4ad2, 0xe5cf, 0x4bd2, 0x4cd2, 0x4dd2,
	0x4ed2, 0x4fd2, 0x50d2, 0x51d2, 0x52d2, 0x53d2, 0x54d2, 0x55d2,
	0x56d2, 0x57d2, 0x58d2, 0x59d2, 0x5ad2, 0x5bd2, 0x5cd2, 0x5dd2,
	0x5ed2, 0x5fd2, 0x60d2, 0x61d2, 0x62d2, 0x63d2, 0xc5f4, 0xf3bd,
	0x64d2, 0x65d2, 0x66d2, 0x67d2, 0x68d2, 0x69d2, 0xe0f1, 0x6ad2,
	0x6bd2, 0x6cd2, 0x6dd2, 0x6ed2, 0x6fd2, 0x70d2, 0x71d2, 0x72d2,
	0x73d2, 0x74d2, 0x75d2, 0x76d2, 0x77d2, 0x78d2, 0x79d2, 0x7ad2,
	0x7bd2, 0x7cd2, 0x7dd2, 0xe1f1, 0x7ed2, 0x80d2, 0x81d2, 0xf7ce,
	0x82d2, 0xaad2, 0x83d2, 0xfbf1, 0x84d2, 0x85d2, 0xb2b8, 0x86d2,
	0x87d2, 0x88d2, 0x89d2, 0x8ad2, 0x8bd2, 0x8cd2, 0x8dd2, 0x8ed2,
	0x8fd2, 0x90d2, 0x91d2, 0x92d2, 0x93d2, 0x94d2, 0x95d2, 0x96d2,
	0x97d2, 0x98d2, 0x99d2, 0x9ad2, 0x9bd2, 0x9cd2, 0x9dd2, 0x9ed2,
	0x9fd2, 0xa0d2, 0x40d3, 0x41d3, 0x42d3, 0x43d3, 0x44d3, 0x45d3,
	0x46d3, 0x47d3, 0x48d3, 0x49d3, 0x4ad3, 0x4bd3, 0x4cd3, 0x4dd3,
	0x4ed3, 0x4fd3, 0x50d3, 0x51d3, 0x52d3, 0x53d3, 0x54d3, 0x55d3,
	0x56d3, 0x57d3, 0x58d3, 0x59d3, 0x5ad3, 0x5bd3, 0x5cd3, 0x5dd3,
	0x5ed3, 0xfbbc, 0xdbb9, 0x5fd3, 0xe6b9, 0xd9c3, 0xd3ca, 0xe8ea,
	0xc0c0, 0xf5be, 0xe9ea, 0xeaea, 0xebea, 0x60d3, 0xecea, 0xedea,
	0xeeea, 0xefea, 0xc7bd, 0x61d3, 0x62d3, 0x63d3, 0xfbf5, 0x64d3,
	0x65d3, 0x66d3, 0xfdf5, 0x67d3, 0xfef5, 0x68d3, 0xfcf5, 0x69d3,
	0x6ad3, 0x6bd3, 0x6cd3, 0xe2bd, 0x6dd3, 0xa1f6, 0xa5b4, 0x6ed3,
	0x6fd3, 0x70d3, 0x71d3, 0xa2f6, 0x72d3, 0x73d3, 0x74d3, 0xa3f6,
	0x75d3, 0x76d3, 0x77d3, 0xb2ec, 0x78d3, 0x79d3, 0x7ad3, 0x7bd3,
	0x7cd3, 0x7dd3, 0x7ed3, 0x80d3, 0x81d3, 0x82d3, 0x83d3, 0x84d3,
	0xd4d1, 0x85d3, 0x86d3, 0x87d3, 0x88d3, 0x89d3, 0x8ad3, 0xead9,
	0x8bd3, 0x8cd3, 0x8dd3, 0x8ed3, 0x8fd3, 0x90d3, 0x91d3, 0x92d3,
	0x93d3, 0x94d3, 0x95d3, 0x96d3, 0x97d3, 0x98d3, 0x99d3, 0x9ad3,
	0x9bd3, 0x9cd3, 0x9dd3, 0x9ed3, 0x9fd3, 0xa0d3, 0x40d4, 0x41d4,
	0x42d4, 0x43d4, 0x44d4, 0x45d4, 0x46d4, 0x47d4, 0x48d4, 0x49d4,
	0x4ad4, 0x4bd4, 0x4cd4, 0x4dd4, 0x4ed4, 0x4fd4, 0x50d4, 0x51d4,
	0x52d4, 0x53d4, 0x54d4, 0x55d4, 0x56d4, 0x57d4, 0x58d4, 0x59d4,
	0x5ad4, 0x5bd4, 0x5cd4, 0x5dd4, 0x5ed4, 0x5fd4, 0xa4f6, 0x60d4,
	0x61d4, 0x62d4, 0x63d4, 0x64d4, 0x65d4, 0x66d4, 0x67d4, 0x68d4,
	0xbaee, 0x69d4, 0x6ad4, 0x6bd4, 0x6cd4, 0x6dd4, 0x6ed4, 0x6fd4,
	0x70d4, 0x71d4, 0x72d4, 0x73d4, 0x74d4, 0x75d4, 0x76d4, 0x77d4,
	0x78d4, 0x79d4, 0x7ad4, 0x7bd4, 0x7cd4, 0x7dd4, 0x7ed4, 0x80d4,
	0x81d4, 0x82d4, 0x83d4, 0x84d4, 0x85d4, 0x86d4, 0x87d4, 0x88d4,
	0x89d4, 0x8ad4, 0x8bd4, 0x8cd4, 0x8dd4, 0x8ed4, 0x8fd4, 0x90d4,
	0x91d4, 0x92d4, 0x93d4, 0x94d4, 0x95d4, 0x96d4, 0x97d4, 0x98d4,
	0x99d4, 0xb2d5, 0x9ad4, 0x9bd4, 0x9cd4, 0x9dd4, 0x9ed4, 0x9fd4,
	0xa0d4, 0x40d5, 0x41d5, 0x42d5, 0x43d5, 0x44d5, 0x45d5, 0x46d5,
	0x47d5, 0xfed3, 0xdccc, 0x48d5, 0x49d5, 0x4ad5, 0x4bd5, 0x4cd5,
	0x4dd5, 0x4ed5, 0x4fd5, 0xc4ca, 0x50d5, 0x51d5, 0x52d5, 0x53d5,
	0x54d5, 0x55d5, 0x56d5, 0x57d5, 0x58d5, 0x59d5, 0x5ad5, 0x5bd5,
	0x5cd5, 0x5dd5, 0x5ed5, 0x5fd5, 0x60d5, 0x61d5, 0x62d5, 0x63d5,
	0x64d5, 0x65d5, 0x66d5, 0x67d5, 0x68d5, 0x69d5, 0x6ad5, 0x6bd5,
	0x6cd5, 0x6dd5, 0x6ed5, 0x6fd5, 0x70d5, 0x71d5, 0x72d5, 0x73d5,
	0x74d5, 0x75d5, 0x76d5, 0x77d5, 0x78d5, 0x79d5, 0x7ad5, 0x7bd5,
	0x7cd5, 0x7dd5, 0x7ed5, 0x80d5, 0x81d5, 0x82d5, 0x83d5, 0x84d5,
	0x85d5, 0x86d5, 0x87d5, 0x88d5, 0x89d5, 0x8ad5, 0x8bd5, 0x8cd5,
	0x8dd5, 0x8ed5, 0x8fd5, 0x90d5, 0x91d5, 0x92d5, 0x93d5, 0x94d5,
	0x95d5, 0x96d5, 0x97d5, 0x98d5, 0x99d5, 0x9ad5, 0x9bd5, 0x9cd5,
	0x9dd5, 0x9ed5, 0x9fd5, 0xa0d5, 0x40d6, 0x41d6, 0x42d6, 0x43d6,
	0x44d6, 0x45d6, 0x46d6, 0x47d6, 0x48d6, 0x49d6, 0x4ad6, 0x4bd6,
	0x4cd6, 0x4dd6, 0x4ed6, 0x4fd6, 0x50d6, 0x51d6, 0x52d6, 0x53d6,
	0x54d6, 0x55d6, 0x56d6, 0x57d6, 0x58d6, 0x59d6, 0x5ad6, 0x5bd6,
	0x5cd6, 0x5dd6, 0x5ed6, 0x5fd6, 0x60d6, 0x61d6, 0x62d6, 0xc0e5,
	0x63d6, 0x64d6, 0x65d6, 0x66d6, 0x67d6, 0x68d6, 0x69d6, 0x6ad6,
	0x6bd6, 0x6cd6, 0x6dd6, 0x6ed6, 0x6fd6, 0x70d6, 0x71d6, 0x72d6,
	0x73d6, 0x74d6, 0x75d6, 0x76d6, 0x77d6, 0x78d6, 0x79d6, 0x7ad6,
	0x7bd6, 0x7cd6, 0x7dd6, 0x7ed6, 0x80d6, 0x81d6, 0xa5f6, 0x82d6,
	0x83d6, 0x84d6, 0x85d6, 0x86d6, 0x87d6, 0x88d6, 0x89d6, 0x8ad6,
	0x8bd6, 0x8cd6, 0x8dd6, 0x8ed6, 0x8fd6, 0x90d6, 0x91d6, 0x92d6,
	0x93d6, 0x94d6, 0x95d6, 0x96d6, 0x97d6, 0x98d6, 0x99d6, 0x9ad6,
	0x9bd6, 0x9cd6, 0x9dd6, 0x9ed6, 0x9fd6, 0xa0d6, 0x40d7, 0x41d7,
	0x42d7, 0x43d7, 0x44d7, 0x45d7, 0x46d7, 0x47d7, 0x48d7, 0x49d7,
	0x4ad7, 0x4bd7, 0x4cd7, 0x4dd7, 0x4ed7, 0x4fd7, 0x50d7, 0x51d7,
	0x52d7, 0x53d7, 0x54d7, 0x55d7, 0x56d7, 0x57d7, 0x58d7, 0x59d7,
	0x5ad7, 0x5bd7, 0x5cd7, 0x5dd7, 0x5ed7, 0x5fd7, 0xafbe, 0x60d7,
	0x61d7, 0x62d7, 0x63d7, 0x64d7, 0xa9c6, 0x65d7, 0x66d7, 0x67d7,
	0x68d7, 0x69d7, 0x6ad7, 0x6bd7, 0x6cd7, 0x6dd7, 0x6ed7, 0x6fd7,
	0x70d7, 0x71d7, 0x72d7, 0x73d7, 0x74d7, 0x75d7, 0x76d7, 0x77d7,
	0x78d7, 0x79d7, 0x7ad7, 0x7bd7, 0x7cd7, 0x7dd7, 0x7ed7, 0x80d7,
	0x81d7, 0x82d7, 0x83d7, 0x84d7, 0x85d7, 0x86d7, 0x87d7, 0x88d7,
	0x89d7, 0x8ad7, 0x8bd7, 0x8cd7, 0x8dd7, 0x8ed7, 0x8fd7, 0x90d7,
	0x91d7, 0x92d7, 0x93d7, 0x94d7, 0x95d7, 0x96d7, 0x97d7, 0x98d7,
	0xa5da, 0xc6bc, 0xa9b6, 0xbcb8, 0xcfc8, 0xa5bc, 0xa6da, 0xa7da,
	0xd6cc, 0xc3c8, 0xa8da, 0xfdc6, 0x99d7, 0xb5d1, 0xe9d2, 0xb6d1,
	0xc7bc, 0x9ad7, 0xb2bd, 0xe4bb, 0xa9da, 0xaada, 0xc8d1, 0xabda,
	0xedd0, 0xefb6, 0xdbc2, 0x9bd7, 0xcfcb, 0xedb7, 0xe8c9, 0xc3b7,
	0xf7be, 0xa4d6, 0xacda, 0xadda, 0xc0c6, 0xe7d7, 0xb6ca, 0x9cd7,
	0xa9d5, 0xdfcb, 0xefd5, 0xaeda, 0xdfd6, 0xcab4, 0xb0da, 0xafda,
	0x9dd7, 0xebd2, 0xb1da, 0xb2da, 0xb3da, 0xd4ca, 0xb4da, 0xabca,
	0xb5da, 0xb6da, 0xcfb3, 0xefd6, 0xb7da, 0xb0bb, 0xaeb5, 0xb8da,
	0xb9da, 0xeeb9, 0xafd1, 0xe8d2, 0xbada, 0xc3b8, 0xeacf, 0xefb2,
	0xbbda, 0xbcda, 0x9ed7, 0xebbd, 0xdcce, 0xefd3, 0xbdda, 0xf3ce,
	0xbeda, 0xd5d3, 0xe5bb, 0xbfda, 0xb5cb, 0xd0cb, 0xc0da, 0xebc7,
	0xeed6, 0xc1da, 0xb5c5, 0xc1b6, 0xc2da, 0xccb7, 0xcebf, 0xc3da,
	0xc4da, 0xadcb, 0xc5da, 0xf7b5, 0xc6da, 0xc2c1, 0xbbd7, 0xc7da,
	0xb8cc, 0x9fd7, 0xead2, 0xb1c4, 0xc8da, 0xfdb5, 0xd1bb, 0xc9da,
	0xb3d0, 0xcada, 0xcbda, 0xbdce, 0xccda, 0xcdda, 0xceda, 0xf7b2,
	0xd1da, 0xcfda, 0xe8d1, 0xd0da, 0xd5c3, 0xd2da, 0xa0d7, 0xd3da,
	0xd4da, 0xd5da, 0xbbd0, 0xa5d2, 0xf9b0, 0xd6da, 0xabc7, 0xd7da,
	0xf7bd, 0xa1c3, 0xd8da, 0xd9da, 0xfdc3, 0xb7cc, 0xdada, 0xdbda,
	0xbec0, 0xd7c6, 0xdcda, 0xddda, 0xb4c7, 0xdeda, 0xdfda, 0xc8b9,
	0x40d8, 0x41d8, 0x42d8, 0x43d8, 0x44d8, 0x45d8, 0x46d8, 0x47d8,
	0x48d8, 0xedbb, 0x49d8, 0x4ad8, 0x4bd8, 0x4cd8, 0xb9b6, 0xf8f4,
	0x4dd8, 0xf9f4, 0x4ed8, 0x4fd8, 0xe3cd, 0x50d8, 0x51d8, 0x52d8,
	0x53d8, 0x54d8, 0x55d8, 0x56d8, 0x57d8, 0xb9f5, 0x58d8, 0x59d8,
	0x5ad8, 0x5bd8, 0xe0eb, 0x5cd8, 0x5dd8, 0x5ed8, 0x5fd8, 0x60d8,
	0x61d8, 0xf3cf, 0xbfbb, 0x62d8, 0x63d8, 0x64d8, 0x65d8, 0x66d8,
	0x67d8, 0x68d8, 0xc0ba, 0xa5d4, 0x69d8, 0x6ad8, 0x6bd8, 0x6cd8,
	0x6dd8, 0x6ed8, 0x6fd8, 0xd9e1, 0x70d8, 0x71d8, 0x72d8, 0x73d8,
	0xf4f5, 0xaab1, 0xf2b2, 0x74d8, 0x75d8, 0x76d8, 0x77d8, 0x78d8,
	0x79d8, 0x7ad8, 0xf5f5, 0x7bd8, 0x7cd8, 0xf7f5, 0x7dd8, 0x7ed8,
	0x80d8, 0xd1ba, 0xf6f5, 0x81d8, 0xb2c3, 0x82d8, 0x83d8, 0x84d8,
	0x85d8, 0x86d8, 0x87d8, 0x88d8, 0xf9f5, 0x89d8, 0x8ad8, 0x8bd8,
	0xf8f5, 0x8cd8, 0x8dd8, 0x8ed8, 0x8fd8, 0x90d8, 0x91d8, 0x92d8,
	0x93d8, 0x94d8, 0x95d8, 0x96d8, 0x97d8, 0x98d8, 0x99d8, 0x9ad8,
	0x9bd8, 0x9cd8, 0x9dd8, 0x9ed8, 0x9fd8, 0xa0d8, 0x40d9, 0x41d9,
	0x42d9, 0x43d9, 0x44d9, 0x45d9, 0x46d9, 0x47d9, 0x48d9, 0x49d9,
	0x4ad9, 0x4bd9, 0x4cd9, 0x4dd9, 0x4ed9, 0x4fd9, 0x50d9, 0x51d9,
	0x52d9, 0x53d9, 0x54d9, 0x55d9, 0x56d9, 0x57d9, 0x58d9, 0x59d9,
	0x5ad9, 0x5bd9, 0x5cd9, 0x5dd9, 0x5ed9, 0x5fd9, 0x60d9, 0x61d9,
	0x62d9, 0x63d9, 0x64d9, 0x65d9, 0x66d9, 0x67d9, 0x68d9, 0x69d9,
	0x6ad9, 0x6bd9, 0x6cd9, 0x6dd9, 0x6ed9, 0x6fd9, 0x70d9, 0x71d9,
	0x72d9, 0x73d9, 0x74d9, 0x75d9, 0x76d9, 0x77d9, 0x78d9, 0x79d9,
	0x7ad9, 0x7bd9, 0x7cd9, 0x7dd9, 0x7ed9, 0x80d9, 0x81d9, 0x82d9,
	0x83d9, 0x84d9, 0x85d9, 0x86d9, 0x87d9, 0x88d9, 0x89d9, 0x8ad9,
	0x8bd9, 0x8cd9, 0x8dd9, 0x8ed9, 0x8fd9, 0x90d9, 0x91d9, 0x92d9,
	0x93d9, 0x94d9, 0x95d9, 0x96d9, 0x97d9, 0x98d9, 0x99d9, 0x9ad9,
	0x9bd9, 0x9cd9, 0x9dd9, 0x9ed9, 0x9fd9, 0xa0d9, 0x40da, 0x41da,
	0x42da, 0x43da, 0x44da, 0x45da, 0x46da, 0x47da, 0x48da, 0x49da,
	0x4ada, 0x4bda, 0x4cda, 0x4dda, 0x4eda, 0xb4b1, 0xead5, 0xbab8,
	0x4fda, 0xb1b9, 0xc6b2, 0xf0d4, 0xcdcf, 0xdcb0, 0xcbd5, 0xf5bb,
	0xcad6, 0xb7b7, 0xb0cc, 0xb6c6, 0xe1b1, 0xbab9, 0xfcd6, 0xe1b9,
	0xa1b7, 0xfabc, 0xdaea, 0xdbea, 0xf9cc, 0xf3b9, 0xdcea, 0xfbb4,
	0xb3c3, 0xd1b7, 0xd8ba, 0xddea, 0xf4d4, 0xdeea, 0xd6bc, 0xdfbb,
	0xdfea, 0xdec1, 0xb8c2, 0xdfd4, 0xcad7, 0xe0ea, 0xe1ea, 0xe4ea,
	0xe2ea, 0xe3ea, 0xdec9, 0xb3b8, 0xc4b6, 0xe5ea, 0xeaca, 0xcdc9,
	0xcdb4, 0x50da, 0x51da, 0xd9e2, 0xe2c5, 0xe6ea, 0xb5c0, 0x52da,
	0xb8d7, 0xe7ea, 0xacd7, 0xfcc8, 0xd3d8, 0xcdd8, 0xded4, 0x53da,
	0xf9d4, 0xc4c9, 0xaed3, 0xd3b8, 0xe0b3, 0x54da, 0xe2c9, 0xf6f4,
	0x55da, 0x56da, 0x57da, 0xd5ba, 0x58da, 0xf7f4, 0x59da, 0x5ada,
	0xdfd7, 0x5bda, 0x5cda, 0xf1f4, 0xb0b8, 0xd4d5, 0xcfb8, 0xf0c6,
	0x5dda, 0x5eda, 0x5fda, 0x60da, 0x61da, 0x62da, 0x63da, 0x64da,
	0x65da, 0xc3b3, 0x66da, 0x67da, 0xf2f4, 0xacb3, 0x68da, 0x69da,
	0x6ada, 0x6bda, 0xbdd4, 0xf7c7, 0x6cda, 0x6dda, 0x6eda, 0x6fda,
	0x70da, 0xf4f4, 0x71da, 0x72da, 0xf3f4, 0x73da, 0x74da, 0x75da,
	0x76da, 0x77da, 0x78da, 0x79da, 0x7ada, 0x7bda, 0x7cda, 0xcbcc,
	0x7dda, 0x7eda, 0x80da, 0xa4c8, 0x81da, 0x82da, 0x83da, 0x84da,
	0x85da, 0x86da, 0x87da, 0x88da, 0x89da, 0x8ada, 0x8bda, 0x8cda,
	0x8dda, 0xf5f4, 0x8eda, 0xe3d7, 0xbfc5, 0xc0f5, 0x8fda, 0x90da,
	0xbbf5, 0x91da, 0xc3f5, 0x92da, 0xc2f5, 0x93da, 0xbad6, 0xc1f5,
	0x94da, 0x95da, 0x96da, 0xbed4, 0xc4f5, 0x97da, 0xccf5, 0x98da,
	0x99da, 0x9ada, 0x9bda, 0xcfb0, 0xf8b5, 0x9cda, 0xc9f5, 0xcaf5,
	0x9dda, 0xdcc5, 0x9eda, 0x9fda, 0xa0da, 0x40db, 0xc5f5, 0xc6f5,
	0x41db, 0x42db, 0xc7f5, 0xcbf5, 0x43db, 0xe0be, 0xc8f5, 0xfab8,
	0x44db, 0x45db, 0x46db, 0xd0f5, 0xd3f5, 0x47db, 0x48db, 0x49db,
	0xe7bf, 0x4adb, 0xf2b9, 0xbcf5, 0xcdf5, 0x4bdb, 0x4cdb, 0xb7c2,
	0x4ddb, 0x4edb, 0x4fdb, 0xf8cc, 0x50db, 0xf9bc, 0x51db, 0xcef5,
	0xcff5, 0xd1f5, 0xe5b6, 0xd2f5, 0x52db, 0xd5f5, 0x53db, 0x54db,
	0x55db, 0x56db, 0x57db, 0x58db, 0x59db, 0xbdf5, 0x5adb, 0x5bdb,
	0x5cdb, 0xd4f5, 0xbbd3, 0x5ddb, 0xecb3, 0x5edb, 0x5fdb, 0xa4cc,
	0x60db, 0x61db, 0x62db, 0x63db, 0xd6f5, 0x64db, 0x65db, 0x66db,
	0x67db, 0x68db, 0x69db, 0x6adb, 0x6bdb, 0xd7f5, 0xe1be, 0xd8f5,
	0x6cdb, 0x6ddb, 0xdfcc, 0xdbf5, 0x6edb, 0x6fdb, 0x70db, 0x71db,
	0x72db, 0xc8b2, 0xd9d7, 0x73db, 0xd9f5, 0x74db, 0xdaf5, 0xdcf5,
	0x75db, 0xe2f5, 0x76db, 0x77db, 0x78db, 0xe0f5, 0x79db, 0x7adb,
	0x7bdb, 0xdff5, 0xddf5, 0x7cdb, 0x7ddb, 0xe1f5, 0x7edb, 0x80db,
	0xdef5, 0xe4f5, 0xe5f5, 0x81db, 0xe3cc, 0x82db, 0x83db, 0xbfe5,
	0xb8b5, 0xe3f5, 0xe8f5, 0xa3cc, 0x84db, 0x85db, 0x86db, 0x87db,
	0x88db, 0xe6f5, 0xe7f5, 0x89db, 0x8adb, 0x8bdb, 0x8cdb, 0x8ddb,
	0x8edb, 0xbef5, 0x8fdb, 0x90db, 0x91db, 0x92db, 0x93db, 0x94db,
	0x95db, 0x96db, 0x97db, 0x98db, 0x99db, 0x9adb, 0xc4b1, 0x9bdb,
	0x9cdb, 0xbff5, 0x9ddb, 0x9edb, 0xc5b5, 0xe4b2, 0x9fdb, 0xecf5,
	0xe9f5, 0xa0db, 0xd7b6, 0x40dc, 0xedf5, 0x41dc, 0xeaf5, 0x42dc,
	0x43dc, 0x44dc, 0x45dc, 0x46dc, 0xebf5, 0x47dc, 0x48dc, 0xdab4,
	0x49dc, 0xead4, 0x4adc, 0x4bdc, 0x4cdc, 0xeef5, 0x4ddc, 0xf9b3,
	0x4edc, 0x4fdc, 0x50dc, 0x51dc, 0x52dc, 0x53dc, 0x54dc, 0xeff5,
	0xf1f5, 0x55dc, 0x56dc, 0x57dc, 0xf0f5, 0x58dc, 0x59dc, 0x5adc,
	0x5bdc, 0x5cdc, 0x5ddc, 0x5edc, 0xf2f5, 0x5fdc, 0xf3f5, 0x60dc,
	0x61dc, 0x62dc, 0x63dc, 0x64dc, 0x65dc, 0x66dc, 0x67dc, 0x68dc,
	0x69dc, 0x6adc, 0x6bdc, 0xedc9, 0xaab9, 0x6cdc, 0x6ddc, 0xfbc7,
	0x6edc, 0x6fdc, 0xe3b6, 0x70dc, 0x71dc, 0x72dc, 0x73dc, 0x74dc,
	0x75dc, 0x76dc, 0xc9cc, 0x77dc, 0x78dc, 0x79dc, 0x7adc, 0x7bdc,
	0x7cdc, 0x7ddc, 0x7edc, 0x80dc, 0x81dc, 0x82dc, 0x83dc, 0x84dc,
	0x85dc, 0x86dc, 0x87dc, 0x88dc, 0x89dc, 0x8adc, 0xa6ea, 0x8bdc,
	0x8cdc, 0x8ddc, 0x8edc, 0x8fdc, 0x90dc, 0x91dc, 0x92dc, 0x93dc,
	0x94dc, 0x95dc, 0x96dc, 0x97dc, 0x98dc, 0x99dc, 0x9adc, 0x9bdc,
	0x9cdc, 0x9ddc, 0x9edc, 0x9fdc, 0xa0dc, 0x40dd, 0x41dd, 0x42dd,
	0x43dd, 0x44dd, 0x45dd, 0x46dd, 0x47dd, 0x48dd, 0x49dd, 0x4add,
	0x4bdd, 0x4cdd, 0x4ddd, 0x4edd, 0x4fdd, 0x50dd, 0x51dd, 0x52dd,
	0x53dd, 0x54dd, 0x55dd, 0x56dd, 0x57dd, 0x58dd, 0x59dd, 0x5add,
	0x5bdd, 0x5cdd, 0x5ddd, 0x5edd, 0x5fdd, 0x60dd, 0x61dd, 0x62dd,
	0x63dd, 0x64dd, 0x65dd, 0x66dd, 0x67dd, 0x68dd, 0x69dd, 0x6add,
	0x6bdd, 0x6cdd, 0x6ddd, 0x6edd, 0x6fdd, 0x70dd, 0x71dd, 0x72dd,
	0x73dd, 0x74dd, 0x75dd, 0x76dd, 0x77dd, 0x78dd, 0x79dd, 0x7add,
	0x7bdd, 0x7cdd, 0x7ddd, 0x7edd, 0x80dd, 0x81dd, 0x82dd, 0x83dd,
	0x84dd, 0x85dd, 0x86dd, 0x87dd, 0x88dd, 0x89dd, 0x8add, 0x8bdd,
	0x8cdd, 0x8ddd, 0x8edd, 0x8fdd, 0x90dd, 0x91dd, 0x92dd, 0x93dd,
	0x94dd, 0x95dd, 0x96dd, 0x97dd, 0x98dd, 0x99dd, 0x9add, 0x9bdd,
	0x9cdd, 0x9ddd, 0x9edd, 0x9fdd, 0xa0dd, 0x40de, 0x41de, 0x42de,
	0x43de, 0x44de, 0x45de, 0x46de, 0x47de, 0x48de, 0x49de, 0x4ade,
	0x4bde, 0x4cde, 0x4dde, 0x4ede, 0x4fde, 0x50de, 0x51de, 0x52de,
	0x53de, 0x54de, 0x55de, 0x56de, 0x57de, 0x58de, 0x59de, 0x5ade,
	0x5bde, 0x5cde, 0x5dde, 0x5ede, 0x5fde, 0x60de, 0xb5b3, 0xfed4,
	0xecb9, 0xf9d0, 0x61de, 0xede9, 0xaad7, 0xeee9, 0xd6c2, 0xedc8,
	0xe4ba, 0xefe9, 0xf0e9, 0xf1e9, 0xe1d6, 0xf2e9, 0xf3e9, 0xf5e9,
	0xf4e9, 0xf6e9, 0xf7e9, 0xe1c7, 0xf8e9, 0xd8d4, 0xf9e9, 0xcebd,
	0x62de, 0xfae9, 0xfbe9, 0xcfbd, 0xfce9, 0xa8b8, 0xbec1, 0xfde9,
	0xb2b1, 0xd4bb, 0xf5b9, 0xfee9, 0x63de, 0xa1ea, 0xa2ea, 0xa3ea,
	0xf8b7, 0xadbc, 0x64de, 0xe4ca, 0xcee0, 0xafd4, 0xbdcf, 0xb7d5,
	0xa4ea, 0xded5, 0xa5ea, 0xc1d0, 0xbcb9, 0x65de, 0xc7b4, 0xd9b1,
	0x66de, 0x67de, 0x68de, 0xb1c0, 0x69de, 0x6ade, 0x6bde, 0x6cde,
	0xe6b1, 0xe7b1, 0x6dde, 0xe8b1, 0x6ede, 0x6fde, 0x70de, 0x71de,
	0xbdb3, 0xe8c8, 0x72de, 0x73de, 0x74de, 0x75de, 0xc1e5, 0x76de,
	0x77de, 0xdfb1, 0x78de, 0x79de, 0x7ade, 0xc9c1, 0xefb4, 0x7bde,
	0x7cde, 0xa8c7, 0xd8d3, 0x7dde, 0xf9c6, 0xb8d1, 0x7ede, 0xfdb9,
	0xf5c2, 0x80de, 0x81de, 0x82de, 0x83de, 0x84de, 0xadd3, 0x85de,
	0xcbd4, 0xfcbd, 0x86de, 0xc2e5, 0xb5b7, 0xc3e5, 0x87de, 0x88de,
	0xb9bb, 0xe2d5, 0x89de, 0xf8bd, 0xb6d4, 0xa5ce, 0xacc1, 0xd9b3,
	0x8ade, 0x8bde, 0xf6cc, 0x8cde, 0xc6e5, 0xc4e5, 0xc8e5, 0x8dde,
	0xcae5, 0xc7e5, 0xcfb5, 0xc8c6, 0x8ede, 0xfcb5, 0xc5e5, 0x8fde,
	0xf6ca, 0x90de, 0x91de, 0xc9e5, 0x92de, 0x93de, 0x94de, 0xd4c3,
	0xc5b1, 0xa3bc, 0x95de, 0x96de, 0x97de, 0xb7d7, 0x98de, 0x99de,
	0xcbcd, 0xcdcb, 0xcaca, 0xd3cc, 0xcce5, 0xcbe5, 0xe6c4, 0x9ade,
	0x9bde, 0xa1d1, 0xb7d1, 0xcde5, 0x9cde, 0xd0e5, 0x9dde, 0xb8cd,
	0xf0d6, 0xcfe5, 0xddb5, 0x9ede, 0xbecd, 0x9fde, 0xd1e5, 0xbab6,
	0xa0de, 0x40df, 0xa8cd, 0xe4b9, 0x41df, 0xc5ca, 0xd1b3, 0xd9cb,
	0xecd4, 0xd2e5, 0xeab7, 0x42df, 0x43df, 0x44df, 0xcee5, 0x45df,
	0x46df, 0x47df, 0x48df, 0x49df, 0x4adf, 0xd5e5, 0xfeb4, 0xd6e5,
	0x4bdf, 0x4cdf, 0x4ddf, 0x4edf, 0x4fdf, 0xd3e5, 0xd4e5, 0x50df,
	0xddd2, 0x51df, 0x52df, 0xdfc2, 0xc6b1, 0x53df, 0xe2d3, 0x54df,
	0x55df, 0xddb6, 0xeccb, 0x56df, 0xd7e5, 0x57df, 0x58df, 0xf6d3,
	0x59df, 0x5adf, 0x5bdf, 0x5cdf, 0x5ddf, 0xe9b1, 0x5edf, 0xf4b6,
	0xdae5, 0xd8e5, 0xd9e5, 0xc0b5, 0x5fdf, 0x60df, 0x61df, 0xc5d2,
	0xdce5, 0x62df, 0x63df, 0xdee5, 0x64df, 0x65df, 0x66df, 0x67df,
	0x68df, 0x69df, 0xdde5, 0xb2c7, 0x6adf, 0xa3d2, 0x6bdf, 0x6cdf,
	0xdbe5, 0x6ddf, 0x6edf, 0x6fdf, 0x70df, 0xe2d4, 0xdad5, 0x71df,
	0x72df, 0x73df, 0x74df, 0x75df, 0xe0e5, 0xf1d7, 0x76df, 0x77df,
	0x78df, 0x79df, 0x7adf, 0x7bdf, 0x7cdf, 0xe1e5, 0x7ddf, 0xdcb1,
	0xfbd1, 0x7edf, 0xe2e5, 0xe4e5, 0x80df, 0x81df, 0x82df, 0x83df,
	0xe3e5, 0x84df, 0x85df, 0xe5e5, 0x86df, 0x87df, 0x88df, 0x89df,
	0x8adf, 0xd8d2, 0x8bdf, 0xcbb5, 0x8cdf, 0xdfe7, 0x8ddf, 0xf5da,
	0x8edf, 0xf8da, 0x8fdf, 0xf6da, 0x90df, 0xf7da, 0x91df, 0x92df,
	0x93df, 0xfada, 0xcfd0, 0xc7c4, 0x94df, 0x95df, 0xeeb0, 0x96df,
	0x97df, 0x98df, 0xb0d0, 0x99df, 0xf9da, 0x9adf, 0xcad3, 0xaaba,
	0xa2db, 0xf1c7, 0x9bdf, 0xfcda, 0xfbda, 0xdbc9, 0xfdda, 0x9cdf,
	0xa1db, 0xded7, 0xfeda, 0xdac1, 0x9ddf, 0x9edf, 0xa5db, 0x9fdf,
	0xa0df, 0xf4d3, 0x40e0, 0x41e0, 0xa7db, 0xa4db, 0x42e0, 0xa8db,
	0x43e0, 0x44e0, 0xbcbd, 0x45e0, 0x46e0, 0x47e0, 0xc9c0, 0xa3db,
	0xa6db, 0xa3d6, 0x48e0, 0xa9db, 0x49e0, 0x4ae0, 0x4be0, 0xaddb,
	0x4ce0, 0x4de0, 0x4ee0, 0xaedb, 0xacdb, 0xc2ba, 0x4fe0, 0x50e0,
	0x51e0, 0xa4bf, 0xabdb, 0x52e0, 0x53e0, 0x54e0, 0xaadb, 0xc7d4,
	0xbfb2, 0x55e0, 0x56e0, 0xafdb, 0x57e0, 0xf9b9, 0x58e0, 0xb0db,
	0x59e0, 0x5ae0, 0x5be0, 0x5ce0, 0xbbb3, 0x5de0, 0x5ee0, 0x5fe0,
	0xa6b5, 0x60e0, 0x61e0, 0x62e0, 0x63e0, 0xbcb6, 0xb1db, 0x64e0,
	0x65e0, 0x66e0, 0xf5b6, 0x67e0, 0xb2db, 0x68e0, 0x69e0, 0x6ae0,
	0x6be0, 0x6ce0, 0x6de0, 0x6ee0, 0x6fe0, 0x70e0, 0x71e0, 0x72e0,
	0x73e0, 0x74e0, 0x75e0, 0x76e0, 0x77e0, 0x78e0, 0x79e0, 0x7ae0,
	0x7be0, 0xc9b1, 0x7ce0, 0x7de0, 0x7ee0, 0x80e0, 0xb4db, 0x81e0,
	0x82e0, 0x83e0, 0xb3db, 0xb5db, 0x84e0, 0x85e0, 0x86e0, 0x87e0,
	0x88e0, 0x89e0, 0x8ae0, 0x8be0, 0x8ce0, 0x8de0, 0x8ee0, 0xb7db,
	0x8fe0, 0xb6db, 0x90e0, 0x91e0, 0x92e0, 0x93e0, 0x94e0, 0x95e0,
	0x96e0, 0xb8db, 0x97e0, 0x98e0, 0x99e0, 0x9ae0, 0x9be0, 0x9ce0,
	0x9de0, 0x9ee0, 0x9fe0, 0xb9db, 0xa0e0, 0x40e1, 0xbadb, 0x41e1,
	0x42e1, 0xcfd3, 0xfaf4, 0xf5c7, 0xc3d7, 0xe4c5, 0xfcf4, 0xfdf4,
	0xfbf4, 0x43e1, 0xc6be, 0x44e1, 0x45e1, 0x46e1, 0x47e1, 0xefd0,
	0x48e1, 0x49e1, 0xd3b7, 0x4ae1, 0x4be1, 0xcdd4, 0xaacc, 0x4ce1,
	0x4de1, 0xa2f5, 0xa1f5, 0xa8ba, 0xfef4, 0xd6cb, 0x4ee1, 0x4fe1,
	0x50e1, 0xa4f5, 0xd2c0, 0x51e1, 0xeab3, 0x52e1, 0xaacd, 0xa5f5,
	0xa3f5, 0xb4bd, 0xa8f5, 0x53e1, 0xa9f5, 0xcdbd, 0xb8c3, 0xe1bf,
	0xe1cb, 0xaaf5, 0x54e1, 0x55e1, 0x56e1, 0xa6f5, 0xa7f5, 0xf0c4,
	0x57e1, 0x58e1, 0x59e1, 0x5ae1, 0x5be1, 0xacf5, 0x5ce1, 0xbcb4,
	0x5de1, 0xedd7, 0x5ee1, 0xd7b4, 0xabf5, 0xaef5, 0x5fe1, 0x60e1,
	0xadf5, 0xaff5, 0xd1d0, 0x61e1, 0x62e1, 0x63e1, 0x64e1, 0x65e1,
	0x66e1, 0x67e1, 0xd1c3, 0xa9c8, 0x68e1, 0x69e1, 0x6ae1, 0x6be1,
	0x6ce1, 0x6de1, 0xb0f5, 0xb1f5, 0x6ee1, 0x6fe1, 0x70e1, 0x71e1,
	0x72e1, 0x73e1, 0xb2f5, 0x74e1, 0x75e1, 0xb3f5, 0xb4f5, 0xb5f5,
	0x76e1, 0x77e1, 0x78e1, 0x79e1, 0xb7f5, 0xb6f5, 0x7ae1, 0x7be1,
	0x7ce1, 0x7de1, 0xb8f5, 0x7ee1, 0x80e1, 0x81e1, 0x82e1, 0x83e1,
	0x84e1, 0x85e1, 0x86e1, 0x87e1, 0x88e1, 0x89e1, 0x8ae1, 0xc9b2,
	0x8be1, 0xd4d3, 0xcdca, 0x8ce1, 0xefc0, 0xd8d6, 0xb0d2, 0xbfc1,
	0x8de1, 0xf0bd, 0x8ee1, 0x8fe1, 0x90e1, 0x91e1, 0x92e1, 0x93e1,
	0x94e1, 0x95e1, 0x96e1, 0x97e1, 0xaab8, 0x98e1, 0x99e1, 0x9ae1,
	0x9be1, 0x9ce1, 0x9de1, 0x9ee1, 0x9fe1, 0xa0e1, 0x40e2, 0x41e2,
	0x42e2, 0x43e2, 0x44e2, 0x45e2, 0x46e2, 0x47e2, 0x48e2, 0x49e2,
	0x4ae2, 0x4be2, 0x4ce2, 0x4de2, 0x4ee2, 0x4fe2, 0x50e2, 0x51e2,
	0x52e2, 0x53e2, 0x54e2, 0x55e2, 0x56e2, 0x57e2, 0x58e2, 0x59e2,
	0x5ae2, 0x5be2, 0x5ce2, 0x5de2, 0x5ee2, 0x5fe2, 0x60e2, 0x61e2,
	0x62e2, 0x63e2, 0x64e2, 0x65e2, 0x66e2, 0x67e2, 0x68e2, 0x69e2,
	0x6ae2, 0x6be2, 0x6ce2, 0x6de2, 0x6ee2, 0x6fe2, 0x70e2, 0x71e2,
	0x72e2, 0x73e2, 0x74e2, 0x75e2, 0x76e2, 0x77e2, 0x78e2, 0x79e2,
	0x7ae2, 0x7be2, 0x7ce2, 0x7de2, 0x7ee2, 0x80e2, 0x81e2, 0x82e2,
	0x83e2, 0x84e2, 0x85e2, 0x86e2, 0x87e2, 0x88e2, 0x89e2, 0x8ae2,
	0x8be2, 0x8ce2, 0x8de2, 0x8ee2, 0x8fe2, 0x90e2, 0x91e2, 0x92e2,
	0x93e2, 0x94e2, 0x95e2, 0x96e2, 0x97e2, 0x98e2, 0x99e2, 0x9ae2,
	0x9be2, 0x9ce2, 0x9de2, 0x9ee2, 0x9fe2, 0xa0e2, 0x40e3, 0x41e3,
	0x42e3, 0x43e3, 0x44e3, 0x45e3, 0x46e3, 0x47e3, 0x48e3, 0x49e3,
	0x4ae3, 0x4be3, 0x4ce3, 0x4de3, 0x4ee3, 0x4fe3, 0x50e3, 0x51e3,
	0x52e3, 0x53e3, 0x54e3, 0x55e3, 0x56e3, 0x57e3, 0x58e3, 0x59e3,
	0x5ae3, 0x5be3, 0x5ce3, 0x5de3, 0x5ee3, 0x5fe3, 0x60e3, 0x61e3,
	0x62e3, 0x63e3, 0x64e3, 0x65e3, 0x66e3, 0x67e3, 0x68e3, 0x69e3,
	0x6ae3, 0x6be3, 0x6ce3, 0x6de3, 0xf8bc, 0x6ee3, 0x6fe3, 0x70e3,
	0x71e3, 0x72e3, 0x73e3, 0x74e3, 0x75e3, 0x76e3, 0x77e3, 0x78e3,
	0x79e3, 0x7ae3, 0x7be3, 0x7ce3, 0x7de3, 0x7ee3, 0x80e3, 0x81e3,
	0x82e3, 0x83e3, 0x84e3, 0x85e3, 0x86e3, 0x87e3, 0xc6f6, 0x88e3,
	0x89e3, 0x8ae3, 0x8be3, 0x8ce3, 0x8de3, 0x8ee3, 0x8fe3, 0x90e3,
	0x91e3, 0x92e3, 0x93e3, 0x94e3, 0x95e3, 0x96e3, 0x97e3, 0x98e3,
	0x99e3, 0x9ae3, 0x9be3, 0x9ce3, 0x9de3, 0x9ee3, 0x9fe3, 0xa0e3,
	0x40e4, 0x41e4, 0x42e4, 0x43e4, 0x44e4, 0x45e4, 0xc7f6, 0x46e4,
	0x47e4, 0x48e4, 0x49e4, 0x4ae4, 0x4be4, 0x4ce4, 0x4de4, 0x4ee4,
	0x4fe4, 0x50e4, 0x51e4, 0x52e4, 0x53e4, 0x54e4, 0x55e4, 0x56e4,
	0x57e4, 0x58e4, 0x59e4, 0x5ae4, 0x5be4, 0x5ce4, 0x5de4, 0x5ee4,
	0xc8f6, 0x5fe4, 0x60e4, 0x61e4, 0x62e4, 0x63e4, 0x64e4, 0x65e4,
	0x66e4, 0x67e4, 0x68e4, 0x69e4, 0x6ae4, 0x6be4, 0x6ce4, 0x6de4,
	0x6ee4, 0x6fe4, 0x70e4, 0x71e4, 0x72e4, 0x73e4, 0x74e4, 0x75e4,
	0x76e4, 0x77e4, 0x78e4, 0x79e4, 0x7ae4, 0x7be4, 0x7ce4, 0x7de4,
	0x7ee4, 0x80e4, 0x81e4, 0x82e4, 0x83e4, 0x84e4, 0x85e4, 0x86e4,
	0x87e4, 0x88e4, 0x89e4, 0x8ae4, 0x8be4, 0x8ce4, 0x8de4, 0x8ee4,
	0x8fe4, 0x90e4, 0x91e4, 0x92e4, 0x93e4, 0x94e4, 0x95e4, 0x96e4,
	0x97e4, 0x98e4, 0x99e4, 0x9ae4, 0x9be4, 0x9ce4, 0x9de4, 0x9ee4,
	0x9fe4, 0xa0e4, 0x40e5, 0x41e5, 0x42e5, 0x43e5, 0x44e5, 0x45e5,
	0x46e5, 0x47e5, 0x48e5, 0x49e5, 0x4ae5, 0x4be5, 0x4ce5, 0x4de5,
	0x4ee5, 0x4fe5, 0x50e5, 0x51e5, 0x52e5, 0x53e5, 0x54e5, 0x55e5,
	0x56e5, 0x57e5, 0x58e5, 0x59e5, 0x5ae5, 0x5be5, 0x5ce5, 0x5de5,
	0x5ee5, 0x5fe5, 0x60e5, 0x61e5, 0x62e5, 0x63e5, 0x64e5, 0x65e5,
	0x66e5, 0x67e5, 0x68e5, 0x69e5, 0x6ae5, 0x6be5, 0x6ce5, 0x6de5,
	0x6ee5, 0x6fe5, 0x70e5, 0x71e5, 0x72e5, 0x73e5, 0xc9f6, 0x74e5,
	0x75e5, 0x76e5, 0x77e5, 0x78e5, 0x79e5, 0x7ae5, 0x7be5, 0x7ce5,
	0x7de5, 0x7ee5, 0x80e5, 0x81e5, 0x82e5, 0x83e5, 0x84e5, 0x85e5,
	0x86e5, 0x87e5, 0x88e5, 0x89e5, 0x8ae5, 0x8be5, 0x8ce5, 0x8de5,
	0x8ee5, 0x8fe5, 0x90e5, 0x91e5, 0x92e5, 0x93e5, 0x94e5, 0x95e5,
	0x96e5, 0x97e5, 0x98e5, 0x99e5, 0x9ae5, 0x9be5, 0x9ce5, 0x9de5,
	0x9ee5, 0x9fe5, 0xcaf6, 0xa0e5, 0x40e6, 0x41e6, 0x42e6, 0x43e6,
	0x44e6, 0x45e6, 0x46e6, 0x47e6, 0x48e6, 0x49e6, 0x4ae6, 0x4be6,
	0x4ce6, 0x4de6, 0x4ee6, 0x4fe6, 0x50e6, 0x51e6, 0x52e6, 0x53e6,
	0x54e6, 0x55e6, 0x56e6, 0x57e6, 0x58e6, 0x59e6, 0x5ae6, 0x5be6,
	0x5ce6, 0x5de6, 0x5ee6, 0x5fe6, 0x60e6, 0x61e6, 0x62e6, 0xccf6,
	0x63e6, 0x64e6, 0x65e6, 0x66e6, 0x67e6, 0x68e6, 0x69e6, 0x6ae6,
	0x6be6, 0x6ce6, 0x6de6, 0x6ee6, 0x6fe6, 0x70e6, 0x71e6, 0x72e6,
	0x73e6, 0x74e6, 0x75e6, 0x76e6, 0x77e6, 0x78e6, 0x79e6, 0x7ae6,
	0x7be6, 0x7ce6, 0x7de6, 0x7ee6, 0x80e6, 0x81e6, 0x82e6, 0x83e6,
	0x84e6, 0x85e6, 0x86e6, 0x87e6, 0x88e6, 0x89e6, 0x8ae6, 0x8be6,
	0x8ce6, 0x8de6, 0x8ee6, 0x8fe6, 0x90e6, 0x91e6, 0x92e6, 0x93e6,
	0x94e6, 0x95e6, 0x96e6, 0x97e6, 0x98e6, 0x99e6, 0x9ae6, 0x9be6,
	0x9ce6, 0x9de6, 0xcbf6, 0x9ee6, 0x9fe6, 0xa0e6, 0x40e7, 0x41e7,
	0x42e7, 0x43e7, 0x44e7, 0x45e7, 0x46e7, 0x47e7, 0xe9f7, 0x48e7,
	0x49e7, 0x4ae7, 0x4be7, 0x4ce7, 0x4de7, 0x4ee7, 0x4fe7, 0x50e7,
	0x51e7, 0x52e7, 0x53e7, 0x54e7, 0x55e7, 0x56e7, 0x57e7, 0x58e7,
	0x59e7, 0x5ae7, 0x5be7, 0x5ce7, 0x5de7, 0x5ee7, 0x5fe7, 0x60e7,
	0x61e7, 0x62e7, 0x63e7, 0x64e7, 0x65e7, 0x66e7, 0x67e7, 0x68e7,
	0x69e7, 0x6ae7, 0x6be7, 0x6ce7, 0x6de7, 0x6ee7, 0x6fe7, 0x70e7,
	0x71e7, 0x72e7, 0x73e7, 0x74e7, 0x75e7, 0x76e7, 0x77e7, 0x78e7,
	0x79e7, 0x7ae7, 0x7be7, 0x7ce7, 0x7de7, 0x7ee7, 0x80e7, 0x81e7,
	0x82e7, 0x83e7, 0x84e7, 0x85e7, 0x86e7, 0x87e7, 0x88e7, 0x89e7,
	0x8ae7, 0x8be7, 0x8ce7, 0x8de7, 0x8ee7, 0x8fe7, 0x90e7, 0x91e7,
	0x92e7, 0x93e7, 0x94e7, 0x95e7, 0x96e7, 0x97e7, 0x98e7, 0x99e7,
	0x9ae7, 0x9be7, 0x9ce7, 0x9de7, 0x9ee7, 0x9fe7, 0xa0e7, 0x40e8,
	0x41e8, 0x42e8, 0x43e8, 0x44e8, 0x45e8, 0x46e8, 0x47e8, 0x48e8,
	0x49e8, 0x4ae8, 0x4be8, 0x4ce8, 0x4de8, 0x4ee8, 0xcdf6, 0x4fe8,
	0x50e8, 0x51e8, 0x52e8, 0x53e8, 0x54e8, 0x55e8, 0x56e8, 0x57e8,
	0x58e8, 0x59e8, 0x5ae8, 0x5be8, 0x5ce8, 0x5de8, 0x5ee8, 0x5fe8,
	0x60e8, 0x61e8, 0x62e8, 0x63e8, 0x64e8, 0x65e8, 0x66e8, 0x67e8,
	0x68e8, 0x69e8, 0x6ae8, 0x6be8, 0x6ce8, 0x6de8, 0x6ee8, 0x6fe8,
	0x70e8, 0x71e8, 0x72e8, 0x73e8, 0x74e8, 0x75e8, 0x76e8, 0x77e8,
	0x78e8, 0x79e8, 0x7ae8, 0xcef6, 0x7be8, 0x7ce8, 0x7de8, 0x7ee8,
	0x80e8, 0x81e8, 0x82e8, 0x83e8, 0x84e8, 0x85e8, 0x86e8, 0x87e8,
	0x88e8, 0x89e8, 0x8ae8, 0x8be8, 0x8ce8, 0x8de8, 0x8ee8, 0x8fe8,
	0x90e8, 0x91e8, 0x92e8, 0x93e8, 0x94e8, 0xc4ee, 0xc5ee, 0xc6ee,
	0xebd5, 0xa4b6, 0xc8ee, 0xc7ee, 0xc9ee, 0xcaee, 0xa5c7, 0xcbee,
	0xccee, 0x95e8, 0xb0b7, 0xf6b5, 0xcdee, 0xcfee, 0x96e8, 0xceee,
	0x97e8, 0xc6b8, 0xd0ee, 0xd1ee, 0xd2ee, 0xdbb6, 0xaeb3, 0xd3d6,
	0xc6c4, 0xb5b1, 0xd6b8, 0xd3ee, 0xd4ee, 0xbfd4, 0xd5c7, 0xfbbe,
	0xd9ce, 0xb3b9, 0xd6ee, 0xd5ee, 0xd8ee, 0xd7ee, 0xa5c5, 0xd9ee,
	0xdaee, 0xaec7, 0xdbee, 0xafc7, 0xdcee, 0xa7b2, 0xddee, 0xdeee,
	0xdfee, 0xe0ee, 0xe1ee, 0xead7, 0xe2ee, 0xe3ee, 0xd8bc, 0xe4ee,
	0xcbd3, 0xfacc, 0xacb2, 0xe5c1, 0xe5ee, 0xa6c7, 0xadc3, 0x98e8,
	0xe6ee, 0xe7ee, 0xe8ee, 0xe9ee, 0xeaee, 0xebee, 0xecee, 0x99e8,
	0xedee, 0xeeee, 0xefee, 0x9ae8, 0x9be8, 0xf0ee, 0xf1ee, 0xf2ee,
	0xf4ee, 0xf3ee, 0x9ce8, 0xf5ee, 0xadcd, 0xc1c2, 0xf6ee, 0xf7ee,
	0xf8ee, 0xa1d5, 0xf9ee, 0xb3cf, 0xfaee, 0xfbee, 0x9de8, 0xfcee,
	0xfdee, 0xa1ef, 0xfeee, 0xa2ef, 0xf5b8, 0xfac3, 0xa3ef, 0xa4ef,
	0xc2bd, 0xbfd2, 0xf9b2, 0xa5ef, 0xa6ef, 0xa7ef, 0xf8d2, 0xa8ef,
	0xfdd6, 0xa9ef, 0xccc6, 0x9ee8, 0xaaef, 0xabef, 0xb4c1, 0xacef,
	0xfacf, 0xf8cb, 0xaeef, 0xadef, 0xfab3, 0xf8b9, 0xafef, 0xb0ef,
	0xe2d0, 0xb1ef, 0xb2ef, 0xe6b7, 0xbfd0, 0xb3ef, 0xb4ef, 0xb5ef,
	0xf1c8, 0xe0cc, 0xb6ef, 0xb7ef, 0xb8ef, 0xb9ef, 0xbaef, 0xe0d5,
	0xbbef, 0xedb4, 0xaac3, 0xbcef, 0x9fe8, 0xbdef, 0xbeef, 0xbfef,
	0xa0e8, 0xfdce, 0xc0ef, 0xe0c2, 0xb8b4, 0xb6d7, 0xf5bd, 0x40e9,
	0xc7cf, 0xc3ef, 0xc1ef, 0xc2ef, 0xc4ef, 0xa7b6, 0xfcbc, 0xe2be,
	0xccc3, 0xc5ef, 0xc6ef, 0x41e9, 0xc7ef, 0xcfef, 0xc8ef, 0xc9ef,
	0xcaef, 0xc2c7, 0xf1ef, 0xcdb6, 0xcbef, 0x42e9, 0xccef, 0xcdef,
	0xc6b6, 0xbec3, 0xceef, 0x43e9, 0xd0ef, 0xd1ef, 0xd2ef, 0xf2d5,
	0x44e9, 0xd3ef, 0xf7c4, 0x45e9, 0xd4ef, 0xf8c4, 0xd5ef, 0xd6ef,
	0xe4b8, 0xf7b0, 0xd7ef, 0xd8ef, 0xd9ef, 0x46e9, 0xdaef, 0xdbef,
	0xdcef, 0xddef, 0x47e9, 0xdeef, 0xb5be, 0xe1ef, 0xdfef, 0xe0ef,
	0x48e9, 0xe2ef, 0xe3ef, 0xcdc1, 0xe4ef, 0xe5ef, 0xe6ef, 0xe7ef,
	0xe8ef, 0xe9ef, 0xeaef, 0xebef, 0xecef, 0xd8c0, 0x49e9, 0xedef,
	0xadc1, 0xeeef, 0xefef, 0xf0ef, 0x4ae9, 0x4be9, 0xe2cf, 0x4ce9,
	0x4de9, 0x4ee9, 0x4fe9, 0x50e9, 0x51e9, 0x52e9, 0x53e9, 0xa4b3,
	0x54e9, 0x55e9, 0x56e9, 0x57e9, 0x58e9, 0x59e9, 0x5ae9, 0x5be9,
	0x5ce9, 0x5de9, 0x5ee9, 0x5fe9, 0x60e9, 0x61e9, 0x62e9, 0x63e9,
	0x64e9, 0x65e9, 0x66e9, 0x67e9, 0x68e9, 0x69e9, 0x6ae9, 0x6be9,
	0x6ce9, 0x6de9, 0x6ee9, 0x6fe9, 0x70e9, 0x71e9, 0x72e9, 0x73e9,
	0x74e9, 0x75e9, 0x76e9, 0x77e9, 0x78e9, 0x79e9, 0x7ae9, 0x7be9,
	0x7ce9, 0x7de9, 0x7ee9, 0x80e9, 0x81e9, 0x82e9, 0x83e9, 0x84e9,
	0x85e9, 0x86e9, 0x87e9, 0x88e9, 0x89e9, 0x8ae9, 0x8be9, 0x8ce9,
	0x8de9, 0x8ee9, 0x8fe9, 0x90e9, 0x91e9, 0x92e9, 0x93e9, 0x94e9,
	0x95e9, 0x96e9, 0x97e9, 0x98e9, 0x99e9, 0x9ae9, 0x9be9, 0x9ce9,
	0x9de9, 0x9ee9, 0x9fe9, 0xa0e9, 0x40ea, 0x41ea, 0x42ea, 0x43ea,
	0x44ea, 0x45ea, 0x46ea, 0x47ea, 0x48ea, 0x49ea, 0x4aea, 0x4bea,
	0x4cea, 0x4dea, 0x4eea, 0x4fea, 0x50ea, 0x51ea, 0x52ea, 0x53ea,
	0x54ea, 0x55ea, 0x56ea, 0x57ea, 0x58ea, 0x59ea, 0x5aea, 0x5bea,
	0xc5c3, 0xc5e3, 0xc1c9, 0xc6e3, 0x5cea, 0xd5b1, 0xcace, 0xb3b4,
	0xf2c8, 0xc7e3, 0xd0cf, 0xc8e3, 0xe4bc, 0xc9e3, 0xcae3, 0xc6c3,
	0xa2d5, 0xd6c4, 0xebb9, 0xc5ce, 0xcbe3, 0xf6c3, 0xcce3, 0x5dea,
	0xa7b7, 0xf3b8, 0xd2ba, 0xcde3, 0xcee3, 0xc4d4, 0xcfe3, 0x5eea,
	0xd0e3, 0xcbd1, 0xd1e3, 0xd2e3, 0xd3e3, 0xd4e3, 0xd6d1, 0xd5e3,
	0xfbb2, 0xbbc0, 0xd6e3, 0x5fea, 0xabc0, 0xd7e3, 0xd8e3, 0xd9e3,
	0x60ea, 0xdae3, 0xdbe3, 0x61ea, 0xb7b8, 0xe2da, 0x62ea, 0xd3b6,
	0x63ea, 0xe4da, 0xe3da, 0x64ea, 0x65ea, 0x66ea, 0x67ea, 0x68ea,
	0x69ea, 0x6aea, 0xe6da, 0x6bea, 0x6cea, 0x6dea, 0xeec8, 0x6eea,
	0x6fea, 0xe5da, 0xc0b7, 0xf4d1, 0xf5d2, 0xf3d5, 0xd7bd, 0x70ea,
	0x71ea, 0x72ea, 0x73ea, 0xe8d7, 0xe8da, 0xe7da, 0x74ea, 0xa2b0,
	0xd3cd, 0x75ea, 0xe9da, 0x76ea, 0xbdb8, 0xcabc, 0xbdc2, 0xa4c2,
	0xc2b3, 0xeada, 0x77ea, 0xaac2, 0xb0c4, 0xb5bd, 0x78ea, 0x79ea,
	0xdecf, 0x7aea, 0x7bea, 0x7cea, 0xebda, 0xc2c9, 0x7dea, 0x7eea,
	0x80ea, 0x81ea, 0x82ea, 0xddb1, 0x83ea, 0x84ea, 0x85ea, 0xecda,
	0x86ea, 0xb8b6, 0xbad4, 0x87ea, 0xfdb3, 0x88ea, 0x89ea, 0xedda,
	0xc9d4, 0xd5cf, 0xe3c5, 0x8aea, 0xeeda, 0x8bea, 0x8cea, 0x8dea,
	0x8eea, 0x8fea, 0xefda, 0x90ea, 0xf0da, 0xeac1, 0xd5cc, 0xddcf,
	0x91ea, 0x92ea, 0x93ea, 0x94ea, 0x95ea, 0x96ea, 0x97ea, 0x98ea,
	0x99ea, 0x9aea, 0x9bea, 0x9cea, 0x9dea, 0xe7d3, 0xa1c2, 0x9eea,
	0xf1da, 0x9fea, 0xa0ea, 0xe5cb, 0x40eb, 0xf2da, 0x41eb, 0xe6cb,
	0xfed2, 0x42eb, 0x43eb, 0x44eb, 0xf4b8, 0x45eb, 0x46eb, 0xf3da,
	0xafb0, 0xb6cf, 0x47eb, 0x48eb, 0xcfd5, 0x49eb, 0x4aeb, 0x4beb,
	0x4ceb, 0x4deb, 0x4eeb, 0x4feb, 0x50eb, 0x51eb, 0x52eb, 0xedcb,
	0x53eb, 0x54eb, 0x55eb, 0x56eb, 0x57eb, 0x58eb, 0x59eb, 0x5aeb,
	0xf4da, 0x5beb, 0x5ceb, 0xc4e3, 0x5deb, 0x5eeb, 0xa5c1, 0x5feb,
	0x60eb, 0xbff6, 0x61eb, 0x62eb, 0xc0f6, 0xc1f6, 0xd1c4, 0x63eb,
	0xb8c8, 0xe3d1, 0x64eb, 0x65eb, 0xdbd0, 0xc5d1, 0xafbc, 0xcdb9,
	0x66eb, 0xf4ef, 0x67eb, 0x68eb, 0xc6b4, 0xbad3, 0xc2f6, 0xfbb3,
	0x69eb, 0x6aeb, 0xc3f6, 0x6beb, 0x6ceb, 0xf1b5, 0x6deb, 0x6eeb,
	0x6feb, 0x70eb, 0x71eb, 0x72eb, 0x73eb, 0x74eb, 0x75eb, 0x76eb,
	0xc5f6, 0x77eb, 0x78eb, 0x79eb, 0x7aeb, 0x7beb, 0x7ceb, 0x7deb,
	0xead3, 0xa7f6, 0xa9d1, 0x7eeb, 0x80eb, 0x81eb, 0x82eb, 0xa9f6,
	0x83eb, 0x84eb, 0x85eb, 0xa8f6, 0x86eb, 0x87eb, 0xe3c1, 0xd7c0,
	0x88eb, 0xa2b1, 0x89eb, 0x8aeb, 0x8beb, 0x8ceb, 0xedce, 0x8deb,
	0xe8d0, 0xabf6, 0x8eeb, 0x8feb, 0xf6cf, 0x90eb, 0xaaf6, 0xf0d5,
	0xacf6, 0xb9c3, 0x91eb, 0x92eb, 0x93eb, 0xf4bb, 0xaef6, 0xadf6,
	0x94eb, 0x95eb, 0x96eb, 0xdec4, 0x97eb, 0x98eb, 0xd8c1, 0x99eb,
	0x9aeb, 0x9beb, 0x9ceb, 0x9deb, 0xaacb, 0x9eeb, 0xbccf, 0x9feb,
	0xa0eb, 0x40ec, 0x41ec, 0x42ec, 0x43ec, 0x44ec, 0x45ec, 0x46ec,
	0x47ec, 0x48ec, 0xaff6, 0x49ec, 0x4aec, 0xb0f6, 0x4bec, 0x4cec,
	0xb1f6, 0x4dec, 0xb6c2, 0x4eec, 0x4fec, 0x50ec, 0x51ec, 0x52ec,
	0xd4b0, 0xf9c5, 0x53ec, 0x54ec, 0x55ec, 0x56ec, 0xb2f6, 0x57ec,
	0x58ec, 0x59ec, 0x5aec, 0x5bec, 0x5cec, 0x5dec, 0x5eec, 0x5fec,
	0x60ec, 0x61ec, 0x62ec, 0x63ec, 0x64ec, 0x65ec, 0x66ec, 0x67ec,
	0x68ec, 0x69ec, 0xe0c7, 0xa6f6, 0x6aec, 0x6bec, 0xb8be, 0x6cec,
	0x6dec, 0xb2be, 0x6eec, 0xe5b5, 0x6fec, 0x70ec, 0xc7b7, 0x71ec,
	0xbfbf, 0xd2c3, 0xe6c3, 0x72ec, 0x73ec, 0xccd8, 0x74ec, 0x75ec,
	0x76ec, 0xefb8, 0x77ec, 0x78ec, 0x79ec, 0x7aec, 0x7bec, 0x7cec,
	0x7dec, 0x7eec, 0x80ec, 0xf9bd, 0xa5d1, 0x81ec, 0xd0b0, 0x82ec,
	0x83ec, 0x84ec, 0x85ec, 0x86ec, 0xb0f7, 0x87ec, 0x88ec, 0x89ec,
	0x8aec, 0x8bec, 0x8cec, 0x8dec, 0x8eec, 0xb1f7, 0x8fec, 0x90ec,
	0x91ec, 0x92ec, 0x93ec, 0xacd0, 0x94ec, 0xb0b0, 0x95ec, 0x96ec,
	0x97ec, 0xb2f7, 0xb3f7, 0x98ec, 0xb4f7, 0x99ec, 0x9aec, 0x9bec,
	0xcac7, 0x9cec, 0x9dec, 0x9eec, 0x9fec, 0xa0ec, 0x40ed, 0x41ed,
	0xcfbe, 0x42ed, 0x43ed, 0xb7f7, 0x44ed, 0x45ed, 0x46ed, 0x47ed,
	0x48ed, 0x49ed, 0x4aed, 0xb6f7, 0x4bed, 0xdeb1, 0x4ced, 0xb5f7,
	0x4ded, 0x4eed, 0xb8f7, 0x4fed, 0xb9f7, 0x50ed, 0x51ed, 0x52ed,
	0x53ed, 0x54ed, 0x55ed, 0x56ed, 0x57ed, 0x58ed, 0x59ed, 0x5aed,
	0x5bed, 0x5ced, 0x5ded, 0x5eed, 0x5fed, 0x60ed, 0x61ed, 0x62ed,
	0x63ed, 0x64ed, 0x65ed, 0x66ed, 0x67ed, 0x68ed, 0x69ed, 0x6aed,
	0x6bed, 0x6ced, 0x6ded, 0x6eed, 0x6fed, 0x70ed, 0x71ed, 0x72ed,
	0x73ed, 0x74ed, 0x75ed, 0x76ed, 0x77ed, 0x78ed, 0x79ed, 0x7aed,
	0x7bed, 0x7ced, 0x7ded, 0x7eed, 0x80ed, 0x81ed, 0xa4ce, 0xcdc8,
	0x82ed, 0xabba, 0xb8e8, 0xb9e8, 0xbae8, 0xc2be, 0x83ed, 0x84ed,
	0x85ed, 0x86ed, 0x87ed, 0xf4d2, 0x88ed, 0xcfd4, 0xd8c9, 0x89ed,
	0x8aed, 0x8bed, 0x8ced, 0x8ded, 0x8eed, 0x8fed, 0x90ed, 0x91ed,
	0x92ed, 0x93ed, 0x94ed, 0x95ed, 0x96ed, 0x97ed, 0x98ed, 0x99ed,
	0x9aed, 0x9bed, 0x9ced, 0x9ded, 0x9eed, 0x9fed, 0xa0ed, 0x40ee,
	0x41ee, 0x42ee, 0x43ee, 0x44ee, 0x45ee, 0x46ee, 0x47ee, 0x48ee,
	0x49ee, 0x4aee, 0x4bee, 0x4cee, 0x4dee, 0x4eee, 0x4fee, 0x50ee,
	0x51ee, 0x52ee, 0x53ee, 0x54ee, 0x55ee, 0x56ee, 0x57ee, 0x58ee,
	0x59ee, 0x5aee, 0x5bee, 0x5cee, 0x5dee, 0x5eee, 0x5fee, 0x60ee,
	0x61ee, 0x62ee, 0x63ee, 0x64ee, 0x65ee, 0x66ee, 0x67ee, 0x68ee,
	0x69ee, 0x6aee, 0x6bee, 0x6cee, 0x6dee, 0x6eee, 0x6fee, 0x70ee,
	0x71ee, 0x72ee, 0x73ee, 0x74ee, 0x75ee, 0x76ee, 0x77ee, 0x78ee,
	0x79ee, 0x7aee, 0x7bee, 0x7cee, 0x7dee, 0x7eee, 0x80ee, 0x81ee,
	0x82ee, 0x83ee, 0x84ee, 0x85ee, 0x86ee, 0x87ee, 0x88ee, 0x89ee,
	0x8aee, 0x8bee, 0x8cee, 0x8dee, 0x8eee, 0x8fee, 0x90ee, 0x91ee,
	0x92ee, 0x93ee, 0x94ee, 0x95ee, 0x96ee, 0x97ee, 0x98ee, 0x99ee,
	0x9aee, 0x9bee, 0x9cee, 0x9dee, 0x9eee, 0x9fee, 0xa0ee, 0x40ef,
	0x41ef, 0x42ef, 0x43ef, 0x44ef, 0x45ef, 0xb3d2, 0xa5b6, 0xeac7,
	0xfcf1, 0xeecf, 0xb3cb, 0xebd0, 0xefe7, 0xe7cd, 0xcbb9, 0xd9b6,
	0xfdf1, 0xe4b0, 0xcccb, 0xfef1, 0xa4d4, 0xadc2, 0xecc1, 0xc4c6,
	0xb1be, 0xa1f2, 0xd5bc, 0x46ef, 0xa2f2, 0xa3f2, 0x47ef, 0xa4f2,
	0xc3d2, 0xb5c6, 0x48ef, 0xc7cd, 0xa5f2, 0x49ef, 0xb1d3, 0xc5bf,
	0xe2cc, 0x4aef, 0xa6f2, 0xa7f2, 0xd5d1, 0xeeb6, 0xa8f2, 0xa9f2,
	0xdfb5, 0xaaf2, 0xabf2, 0x4bef, 0xfcb2, 0xacf2, 0xadf2, 0xa7c8,
	0x4cef, 0x4def, 0x4eef, 0x4fef, 0x50ef, 0x51ef, 0x52ef, 0x53ef,
	0x54ef, 0x55ef, 0x56ef, 0x57ef, 0x58ef, 0x59ef, 0x5aef, 0x5bef,
	0x5cef, 0x5def, 0x5eef, 0x5fef, 0x60ef, 0x61ef, 0x62ef, 0x63ef,
	0x64ef, 0x65ef, 0x66ef, 0x67ef, 0x68ef, 0x69ef, 0x6aef, 0x6bef,
	0x6cef, 0x6def, 0x6eef, 0x6fef, 0x70ef, 0x71ef, 0xe7b7, 0x72ef,
	0x73ef, 0xa9ec, 0xaaec, 0xabec, 0x74ef, 0xacec, 0x75ef, 0x76ef,
	0xaec6, 0xadec, 0xaeec, 0x77ef, 0x78ef, 0x79ef, 0xc9b7, 0xb3ca,
	0x7aef, 0x7bef, 0x7cef, 0x7def, 0x7eef, 0x80ef, 0x81ef, 0xb8e2,
	0xcff7, 0x82ef, 0x83ef, 0x84ef, 0x85ef, 0x86ef, 0x87ef, 0x88ef,
	0x89ef, 0x8aef, 0x8bef, 0x8cef, 0x8def, 0x8eef, 0x8fef, 0x90ef,
	0x91ef, 0x92ef, 0x93ef, 0x94ef, 0x95ef, 0x96ef, 0x97ef, 0x98ef,
	0x99ef, 0x9aef, 0x9bef, 0x9cef, 0x9def, 0x9eef, 0x9fef, 0xa0ef,
	0x40f0, 0x41f0, 0x42f0, 0x43f0, 0x44f0, 0xd0f7, 0x45f0, 0x46f0,
	0xcdb2, 0x47f0, 0x48f0, 0x49f0, 0x4af0, 0x4bf0, 0x4cf0, 0x4df0,
	0x4ef0, 0x4ff0, 0x50f0, 0x51f0, 0x52f0, 0x53f0, 0x54f0, 0x55f0,
	0x56f0, 0x57f0, 0x58f0, 0x59f0, 0x5af0, 0x5bf0, 0x5cf0, 0x5df0,
	0x5ef0, 0x5ff0, 0x60f0, 0x61f0, 0x62f0, 0x63f0, 0xd1f7, 0x64f0,
	0x65f0, 0x66f0, 0x67f0, 0x68f0, 0x69f0, 0x6af0, 0x6bf0, 0x6cf0,
	0x6df0, 0x6ef0, 0x6ff0, 0x70f0, 0x71f0, 0x72f0, 0x73f0, 0x74f0,
	0x75f0, 0x76f0, 0x77f0, 0x78f0, 0x79f0, 0x7af0, 0x7bf0, 0x7cf0,
	0x7df0, 0x7ef0, 0x80f0, 0x81f0, 0x82f0, 0x83f0, 0x84f0, 0x85f0,
	0x86f0, 0x87f0, 0x88f0, 0x89f0, 0xd3f7, 0xd2f7, 0x8af0, 0x8bf0,
	0x8cf0, 0x8df0, 0x8ef0, 0x8ff0, 0x90f0, 0x91f0, 0x92f0, 0x93f0,
	0x94f0, 0x95f0, 0x96f0, 0xbbe2, 0x97f0, 0xa2bc, 0x98f0, 0xbce2,
	0xbde2, 0xbee2, 0xbfe2, 0xc0e2, 0xc1e2, 0xb9b7, 0xfbd2, 0xa4bd,
	0xceca, 0xa5b1, 0xc7cb, 0x99f0, 0xc2e2, 0xfcb6, 0xc4c8, 0xc3e2,
	0x9af0, 0x9bf0, 0xc8bd, 0x9cf0, 0xfdb1, 0xc4e2, 0x9df0, 0xf6b6,
	0xc5e2, 0xd9c4, 0x9ef0, 0x9ff0, 0xc6e2, 0xdacf, 0xddb9, 0xc7e2,
	0xa1c0, 0xa0f0, 0xc8e2, 0xf6b2, 0x40f1, 0xc9e2, 0x41f1, 0xf3c1,
	0xcae2, 0xcbe2, 0xf8c2, 0xcce2, 0xcde2, 0xcee2, 0xd7ca, 0xb8d8,
	0xe5d9, 0xe3cf, 0x42f1, 0x43f1, 0x44f1, 0x45f1, 0x46f1, 0x47f1,
	0x48f1, 0x49f1, 0x4af1, 0x4bf1, 0x4cf1, 0xa5f0, 0x4df1, 0x4ef1,
	0xb0dc, 0x4ff1, 0x50f1, 0x51f1, 0x52f1, 0x53f1, 0x54f1, 0x55f1,
	0x56f1, 0x57f1, 0x58f1, 0x59f1, 0x5af1, 0x5bf1, 0x5cf1, 0x5df1,
	0x5ef1, 0x5ff1, 0x60f1, 0x61f1, 0x62f1, 0x63f1, 0x64f1, 0x65f1,
	0x66f1, 0x67f1, 0x68f1, 0x69f1, 0x6af1, 0x6bf1, 0x6cf1, 0x6df1,
	0x6ef1, 0x6ff1, 0x70f1, 0x71f1, 0x72f1, 0x73f1, 0x74f1, 0x75f1,
	0x76f1, 0x77f1, 0x78f1, 0x79f1, 0x7af1, 0x7bf1, 0x7cf1, 0x7df1,
	0x7ef1, 0x80f1, 0x81f1, 0x82f1, 0x83f1, 0x84f1, 0x85f1, 0x86f1,
	0x87f1, 0x88f1, 0x89f1, 0x8af1, 0x8bf1, 0x8cf1, 0x8df1, 0x8ef1,
	0x8ff1, 0x90f1, 0x91f1, 0x92f1, 0x93f1, 0x94f1, 0x95f1, 0x96f1,
	0x97f1, 0x98f1, 0x99f1, 0x9af1, 0x9bf1, 0x9cf1, 0x9df1, 0x9ef1,
	0x9ff1, 0xa0f1, 0x40f2, 0x41f2, 0x42f2, 0x43f2, 0x44f2, 0x45f2,
	0x46f2, 0x47f2, 0x48f2, 0x49f2, 0x4af2, 0x4bf2, 0x4cf2, 0x4df2,
	0x4ef2, 0x4ff2, 0x50f2, 0x51f2, 0x52f2, 0x53f2, 0x54f2, 0x55f2,
	0x56f2, 0x57f2, 0x58f2, 0x59f2, 0x5af2, 0x5bf2, 0x5cf2, 0x5df2,
	0x5ef2, 0x5ff2, 0x60f2, 0x61f2, 0x62f2, 0x63f2, 0x64f2, 0x65f2,
	0x66f2, 0x67f2, 0x68f2, 0x69f2, 0x6af2, 0x6bf2, 0x6cf2, 0x6df2,
	0x6ef2, 0x6ff2, 0x70f2, 0x71f2, 0x72f2, 0x73f2, 0x74f2, 0x75f2,
	0x76f2, 0x77f2, 0x78f2, 0x79f2, 0x7af2, 0x7bf2, 0x7cf2, 0x7df2,
	0x7ef2, 0x80f2, 0x81f2, 0x82f2, 0x83f2, 0x84f2, 0x85f2, 0x86f2,
	0x87f2, 0x88f2, 0x89f2, 0x8af2, 0x8bf2, 0x8cf2, 0x8df2, 0x8ef2,
	0x8ff2, 0x90f2, 0x91f2, 0x92f2, 0x93f2, 0x94f2, 0x95f2, 0x96f2,
	0x97f2, 0x98f2, 0x99f2, 0x9af2, 0x9bf2, 0x9cf2, 0x9df2, 0x9ef2,
	0x9ff2, 0xa0f2, 0x40f3, 0x41f3, 0x42f3, 0x43f3, 0x44f3, 0x45f3,
	0x46f3, 0x47f3, 0x48f3, 0x49f3, 0x4af3, 0x4bf3, 0x4cf3, 0x4df3,
	0x4ef3, 0x4ff3, 0x50f3, 0x51f3, 0xedc2, 0xa6d4, 0xd4cd, 0xb1d1,
	0xdbb3, 0xfdc7, 0x52f3, 0xb5b2, 0xbfc2, 0xe0e6, 0xbbca, 0xe1e6,
	0xe2e6, 0xd4be, 0xe3e6, 0xa4d7, 0xd5cd, 0xe5e6, 0xddbc, 0xe4e6,
	0xe6e6, 0xe7e6, 0xeec2, 0x53f3, 0xbebd, 0xe8e6, 0xe6c2, 0xa7ba,
	0xe9e6, 0x54f3, 0xeae6, 0xd2b3, 0xe9d1, 0x55f3, 0x56f3, 0xa5bf,
	0xebe6, 0xefc6, 0xece6, 0xede6, 0x57f3, 0x58f3, 0xeee6, 0xadc6,
	0xefe6, 0x59f3, 0xa7c9, 0xf0e6, 0xf1e6, 0xf2e6, 0xb9e5, 0xf3e6,
	0xf4e6, 0xe2c2, 0xf5e6, 0xf6e6, 0xe8d6, 0xf7e6, 0x5af3, 0xf8e6,
	0xc7b9, 0x5bf3, 0x5cf3, 0x5df3, 0x5ef3, 0x5ff3, 0x60f3, 0x61f3,
	0xbbf7, 0xbaf7, 0x62f3, 0x63f3, 0x64f3, 0x65f3, 0xbef7, 0xbcf7,
	0xa1ba, 0x66f3, 0xbff7, 0x67f3, 0xc0f7, 0x68f3, 0x69f3, 0x6af3,
	0xc2f7, 0xc1f7, 0xc4f7, 0x6bf3, 0x6cf3, 0xc3f7, 0x6df3, 0x6ef3,
	0x6ff3, 0x70f3, 0x71f3, 0xc5f7, 0xc6f7, 0x72f3, 0x73f3, 0x74f3,
	0x75f3, 0xc7f7, 0x76f3, 0xe8cb, 0x77f3, 0x78f3, 0x79f3, 0x7af3,
	0xdfb8, 0x7bf3, 0x7cf3, 0x7df3, 0x7ef3, 0x80f3, 0x81f3, 0xd4f7,
	0x82f3, 0xd5f7, 0x83f3, 0x84f3, 0x85f3, 0x86f3, 0xd6f7, 0x87f3,
	0x88f3, 0x89f3, 0x8af3, 0xd8f7, 0x8bf3, 0xdaf7, 0x8cf3, 0xd7f7,
	0x8df3, 0x8ef3, 0x8ff3, 0x90f3, 0x91f3, 0x92f3, 0x93f3, 0x94f3,
	0x95f3, 0xdbf7, 0x96f3, 0xd9f7, 0x97f3, 0x98f3, 0x99f3, 0x9af3,
	0x9bf3, 0x9cf3, 0x9df3, 0xd7d7, 0x9ef3, 0x9ff3, 0xa0f3, 0x40f4,
	0xdcf7, 0x41f4, 0x42f4, 0x43f4, 0x44f4, 0x45f4, 0x46f4, 0xddf7,
	0x47f4, 0x48f4, 0x49f4, 0xdef7, 0x4af4, 0x4bf4, 0x4cf4, 0x4df4,
	0x4ef4, 0x4ff4, 0x50f4, 0x51f4, 0x52f4, 0x53f4, 0x54f4, 0xdff7,
	0x55f4, 0x56f4, 0x57f4, 0xe0f7, 0x58f4, 0x59f4, 0x5af4, 0x5bf4,
	0x5cf4, 0x5df4, 0x5ef4, 0x5ff4, 0x60f4, 0x61f4, 0x62f4, 0xcbdb,
	0x63f4, 0x64f4, 0xaad8, 0x65f4, 0x66f4, 0x67f4, 0x68f4, 0x69f4,
	0x6af4, 0x6bf4, 0x6cf4, 0xf7e5, 0xedb9, 0x6df4, 0x6ef4, 0x6ff4,
	0x70f4, 0xfdbf, 0xeabb, 0xc9f7, 0xc7c6, 0xc8f7, 0x71f4, 0xcaf7,
	0xccf7, 0xcbf7, 0x72f4, 0x73f4, 0x74f4, 0xcdf7, 0x75f4, 0xbace,
	0x76f4, 0xcef7, 0x77f4, 0x78f4, 0xa7c4, 0x79f4, 0x7af4, 0x7bf4,
	0x7cf4, 0x7df4, 0x7ef4, 0x80f4, 0x81f4, 0x82f4, 0x83f4, 0x84f4,
	0x85f4, 0x86f4, 0x87f4, 0x88f4, 0x89f4, 0x8af4, 0x8bf4, 0x8cf4,
	0x8df4, 0x8ef4, 0x8ff4, 0x90f4, 0x91f4, 0x92f4, 0x93f4, 0x94f4,
	0x95f4, 0x96f4, 0x97f4, 0x98f4, 0x99f4, 0x9af4, 0x9bf4, 0x9cf4,
	0x9df4, 0x9ef4, 0x9ff4, 0xa0f4, 0x40f5, 0x41f5, 0x42f5, 0x43f5,
	0x44f5, 0x45f5, 0x46f5, 0x47f5, 0x48f5, 0x49f5, 0x4af5, 0x4bf5,
	0x4cf5, 0x4df5, 0x4ef5, 0x4ff5, 0x50f5, 0x51f5, 0x52f5, 0x53f5,
	0x54f5, 0x55f5, 0x56f5, 0x57f5, 0x58f5, 0x59f5, 0x5af5, 0x5bf5,
	0x5cf5, 0x5df5, 0x5ef5, 0x5ff5, 0x60f5, 0x61f5, 0x62f5, 0x63f5,
	0x64f5, 0x65f5, 0x66f5, 0x67f5, 0x68f5, 0x69f5, 0x6af5, 0x6bf5,
	0x6cf5, 0x6df5, 0x6ef5, 0x6ff5, 0x70f5, 0x71f5, 0x72f5, 0x73f5,
	0x74f5, 0x75f5, 0x76f5, 0x77f5, 0x78f5, 0x79f5, 0x7af5, 0x7bf5,
	0x7cf5, 0x7df5, 0x7ef5, 0x80f5, 0x81f5, 0x82f5, 0x83f5, 0x84f5,
	0x85f5, 0x86f5, 0x87f5, 0x88f5, 0x89f5, 0x8af5, 0x8bf5, 0x8cf5,
	0x8df5, 0x8ef5, 0x8ff5, 0x90f5, 0x91f5, 0x92f5, 0x93f5, 0x94f5,
	0x95f5, 0x96f5, 0x97f5, 0x98f5, 0x99f5, 0x9af5, 0x9bf5, 0x9cf5,
	0x9df5, 0x9ef5, 0x9ff5, 0xa0f5, 0x40f6, 0x41f6, 0x42f6, 0x43f6,
	0x44f6, 0x45f6, 0x46f6, 0x47f6, 0x48f6, 0x49f6, 0x4af6, 0x4bf6,
	0x4cf6, 0x4df6, 0x4ef6, 0x4ff6, 0x50f6, 0x51f6, 0x52f6, 0x53f6,
	0x54f6, 0x55f6, 0x56f6, 0x57f6, 0x58f6, 0x59f6, 0x5af6, 0x5bf6,
	0x5cf6, 0x5df6, 0x5ef6, 0x5ff6, 0x60f6, 0x61f6, 0x62f6, 0x63f6,
	0x64f6, 0x65f6, 0x66f6, 0x67f6, 0x68f6, 0x69f6, 0x6af6, 0x6bf6,
	0x6cf6, 0x6df6, 0x6ef6, 0x6ff6, 0x70f6, 0x71f6, 0x72f6, 0x73f6,
	0x74f6, 0x75f6, 0x76f6, 0x77f6, 0x78f6, 0x79f6, 0x7af6, 0x7bf6,
	0x7cf6, 0x7df6, 0x7ef6, 0x80f6, 0x81f6, 0x82f6, 0x83f6, 0x84f6,
	0x85f6, 0x86f6, 0x87f6, 0x88f6, 0x89f6, 0x8af6, 0x8bf6, 0x8cf6,
	0x8df6, 0x8ef6, 0x8ff6, 0x90f6, 0x91f6, 0x92f6, 0x93f6, 0x94f6,
	0x95f6, 0x96f6, 0x97f6, 0x98f6, 0x99f6, 0x9af6, 0x9bf6, 0x9cf6,
	0x9df6, 0x9ef6, 0x9ff6, 0xa0f6, 0x40f7, 0x41f7, 0x42f7, 0x43f7,
	0x44f7, 0x45f7, 0x46f7, 0x47f7, 0x48f7, 0x49f7, 0x4af7, 0x4bf7,
	0x4cf7, 0x4df7, 0x4ef7, 0x4ff7, 0x50f7, 0x51f7, 0x52f7, 0x53f7,
	0x54f7, 0x55f7, 0x56f7, 0x57f7, 0x58f7, 0x59f7, 0x5af7, 0x5bf7,
	0x5cf7, 0x5df7, 0x5ef7, 0x5ff7, 0x60f7, 0x61f7, 0x62f7, 0x63f7,
	0x64f7, 0x65f7, 0x66f7, 0x67f7, 0x68f7, 0x69f7, 0x6af7, 0x6bf7,
	0x6cf7, 0x6df7, 0x6ef7, 0x6ff7, 0x70f7, 0x71f7, 0x72f7, 0x73f7,
	0x74f7, 0x75f7, 0x76f7, 0x77f7, 0x78f7, 0x79f7, 0x7af7, 0x7bf7,
	0x7cf7, 0x7df7, 0x7ef7, 0x80f7, 0xe3d3, 0x81f7, 0x82f7, 0xcff6,
	0x83f7, 0xb3c2, 0xd0f6, 0x84f7, 0x85f7, 0xd1f6, 0xd2f6, 0xd3f6,
	0xd4f6, 0x86f7, 0x87f7, 0xd6f6, 0x88f7, 0xabb1, 0xd7f6, 0x89f7,
	0xd8f6, 0xd9f6, 0xdaf6, 0x8af7, 0xdbf6, 0xdcf6, 0x8bf7, 0x8cf7,
	0x8df7, 0x8ef7, 0xddf6, 0xdef6, 0xcacf, 0x8ff7, 0xdff6, 0xe0f6,
	0xe1f6, 0xe2f6, 0xe3f6, 0xe4f6, 0xf0c0, 0xe5f6, 0xe6f6, 0xe7f6,
	0xe8f6, 0xe9f6, 0x90f7, 0xeaf6, 0x91f7, 0xebf6, 0xecf6, 0x92f7,
	0xedf6, 0xeef6, 0xeff6, 0xf0f6, 0xf1f6, 0xf2f6, 0xf3f6, 0xf4f6,
	0xa8be, 0x93f7, 0xf5f6, 0xf6f6, 0xf7f6, 0xf8f6, 0x94f7, 0x95f7,
	0x96f7, 0x97f7, 0x98f7, 0xfac8, 0xf9f6, 0xfaf6, 0xfbf6, 0xfcf6,
	0x99f7, 0x9af7, 0xfdf6, 0xfef6, 0xa1f7, 0xa2f7, 0xa3f7, 0xa4f7,
	0xa5f7, 0x9bf7, 0x9cf7, 0xa6f7, 0xa7f7, 0xa8f7, 0xeeb1, 0xa9f7,
	0xaaf7, 0xabf7, 0x9df7, 0x9ef7, 0xacf7, 0xadf7, 0xdbc1, 0xaef7,
	0x9ff7, 0xa0f7, 0xaff7, 0x40f8, 0x41f8, 0x42f8, 0x43f8, 0x44f8,
	0x45f8, 0x46f8, 0x47f8, 0x48f8, 0x49f8, 0x4af8, 0x4bf8, 0x4cf8,
	0x4df8, 0x4ef8, 0x4ff8, 0x50f8, 0x51f8, 0x52f8, 0x53f8, 0x54f8,
	0x55f8, 0x56f8, 0x57f8, 0x58f8, 0x59f8, 0x5af8, 0x5bf8, 0x5cf8,
	0x5df8, 0x5ef8, 0x5ff8, 0x60f8, 0x61f8, 0x62f8, 0x63f8, 0x64f8,
	0x65f8, 0x66f8, 0x67f8, 0x68f8, 0x69f8, 0x6af8, 0x6bf8, 0x6cf8,
	0x6df8, 0x6ef8, 0x6ff8, 0x70f8, 0x71f8, 0x72f8, 0x73f8, 0x74f8,
	0x75f8, 0x76f8, 0x77f8, 0x78f8, 0x79f8, 0x7af8, 0x7bf8, 0x7cf8,
	0x7df8, 0x7ef8, 0x80f8, 0x81f8, 0x82f8, 0x83f8, 0x84f8, 0x85f8,
	0x86f8, 0x87f8, 0x88f8, 0x89f8, 0x8af8, 0x8bf8, 0x8cf8, 0x8df8,
	0x8ef8, 0x8ff8, 0x90f8, 0x91f8, 0x92f8, 0x93f8, 0x94f8, 0x95f8,
	0x96f8, 0x97f8, 0x98f8, 0x99f8, 0x9af8, 0x9bf8, 0x9cf8, 0x9df8,
	0x9ef8, 0x9ff8, 0xa0f8, 0x40f9, 0x41f9, 0x42f9, 0x43f9, 0x44f9,
	0x45f9, 0x46f9, 0x47f9, 0x48f9, 0x49f9, 0x4af9, 0x4bf9, 0x4cf9,
	0x4df9, 0x4ef9, 0x4ff9, 0x50f9, 0x51f9, 0x52f9, 0x53f9, 0x54f9,
	0x55f9, 0x56f9, 0x57f9, 0x58f9, 0x59f9, 0x5af9, 0x5bf9, 0x5cf9,
	0x5df9, 0x5ef9, 0x5ff9, 0x60f9, 0x61f9, 0x62f9, 0x63f9, 0x64f9,
	0x65f9, 0x66f9, 0x67f9, 0x68f9, 0x69f9, 0x6af9, 0x6bf9, 0x6cf9,
	0x6df9, 0x6ef9, 0x6ff9, 0x70f9, 0x71f9, 0x72f9, 0x73f9, 0x74f9,
	0x75f9, 0x76f9, 0x77f9, 0x78f9, 0x79f9, 0x7af9, 0x7bf9, 0x7cf9,
	0x7df9, 0x7ef9, 0x80f9, 0x81f9, 0x82f9, 0x83f9, 0x84f9, 0x85f9,
	0x86f9, 0x87f9, 0x88f9, 0x89f9, 0x8af9, 0x8bf9, 0x8cf9, 0x8df9,
	0x8ef9, 0x8ff9, 0x90f9, 0x91f9, 0x92f9, 0x93f9, 0x94f9, 0x95f9,
	0x96f9, 0x97f9, 0x98f9, 0x99f9, 0x9af9, 0x9bf9, 0x9cf9, 0x9df9,
	0x9ef9, 0x9ff9, 0xa0f9, 0x40fa, 0x41fa, 0x42fa, 0x43fa, 0x44fa,
	0x45fa, 0x46fa, 0x47fa, 0x48fa, 0x49fa, 0x4afa, 0x4bfa, 0x4cfa,
	0x4dfa, 0x4efa, 0x4ffa, 0x50fa, 0x51fa, 0x52fa, 0x53fa, 0x54fa,
	0x55fa, 0x56fa, 0x57fa, 0x58fa, 0x59fa, 0x5afa, 0x5bfa, 0x5cfa,
	0x5dfa, 0x5efa, 0x5ffa, 0x60fa, 0x61fa, 0x62fa, 0x63fa, 0x64fa,
	0x65fa, 0x66fa, 0x67fa, 0x68fa, 0x69fa, 0x6afa, 0x6bfa, 0x6cfa,
	0x6dfa, 0x6efa, 0x6ffa, 0x70fa, 0x71fa, 0x72fa, 0x73fa, 0x74fa,
	0x75fa, 0x76fa, 0x77fa, 0x78fa, 0x79fa, 0x7afa, 0x7bfa, 0x7cfa,
	0x7dfa, 0x7efa, 0x80fa, 0x81fa, 0x82fa, 0x83fa, 0x84fa, 0x85fa,
	0x86fa, 0x87fa, 0x88fa, 0x89fa, 0x8afa, 0x8bfa, 0x8cfa, 0x8dfa,
	0x8efa, 0x8ffa, 0x90fa, 0x91fa, 0x92fa, 0x93fa, 0x94fa, 0x95fa,
	0x96fa, 0x97fa, 0x98fa, 0x99fa, 0x9afa, 0x9bfa, 0x9cfa, 0x9dfa,
	0x9efa, 0x9ffa, 0xa0fa, 0x40fb, 0x41fb, 0x42fb, 0x43fb, 0x44fb,
	0x45fb, 0x46fb, 0x47fb, 0x48fb, 0x49fb, 0x4afb, 0x4bfb, 0x4cfb,
	0x4dfb, 0x4efb, 0x4ffb, 0x50fb, 0x51fb, 0x52fb, 0x53fb, 0x54fb,
	0x55fb, 0x56fb, 0x57fb, 0x58fb, 0x59fb, 0x5afb, 0x5bfb, 0xf1c4,
	0xaff0, 0xa6bc, 0xb0f0, 0xf9c3, 0x5cfb, 0xb8c5, 0xbbd1, 0x5dfb,
	0xb1f0, 0xb2f0, 0xb3f0, 0xb4f0, 0xb5f0, 0xbcd1, 0x5efb, 0xecd1,
	0x5ffb, 0xb7f0, 0xb6f0, 0xa7d4, 0x60fb, 0xd2cd, 0xb8f0, 0xbaf0,
	0xb9f0, 0xbbf0, 0xbcf0, 0x61fb, 0x62fb, 0xebb8, 0xbdf0, 0xe8ba,
	0x63fb, 0xbef0, 0xbff0, 0xe9be, 0xc0f0, 0xecb6, 0xc1f0, 0xc2f0,
	0xc3f0, 0xc4f0, 0xb5c8, 0xc5f0, 0xc6f0, 0x64fb, 0xc7f0, 0xf4c5,
	0x65fb, 0xc8f0, 0x66fb, 0x67fb, 0x68fb, 0xc9f0, 0x69fb, 0xcaf0,
	0xbdf7, 0x6afb, 0xcbf0, 0xccf0, 0xcdf0, 0x6bfb, 0xcef0, 0x6cfb,
	0x6dfb, 0x6efb, 0x6ffb, 0xcff0, 0xd7ba, 0x70fb, 0xd0f0, 0xd1f0,
	0xd2f0, 0xd3f0, 0xd4f0, 0xd5f0, 0xd6f0, 0xd8f0, 0x71fb, 0x72fb,
	0xa5d3, 0xd7f0, 0x73fb, 0xd9f0, 0x74fb, 0x75fb, 0x76fb, 0x77fb,
	0x78fb, 0x79fb, 0x7afb, 0x7bfb, 0x7cfb, 0x7dfb, 0xbaf5, 0xb9c2,
	0x7efb, 0x80fb, 0xe4f7, 0x81fb, 0x82fb, 0x83fb, 0x84fb, 0xe5f7,
	0xe6f7, 0x85fb, 0x86fb, 0xe7f7, 0x87fb, 0x88fb, 0x89fb, 0x8afb,
	0x8bfb, 0x8cfb, 0xe8f7, 0xb4c2, 0x8dfb, 0x8efb, 0x8ffb, 0x90fb,
	0x91fb, 0x92fb, 0x93fb, 0x94fb, 0x95fb, 0xeaf7, 0x96fb, 0xebf7,
	0x97fb, 0x98fb, 0x99fb, 0x9afb, 0x9bfb, 0x9cfb, 0xf3c2, 0x9dfb,
	0x9efb, 0x9ffb, 0xa0fb, 0x40fc, 0x41fc, 0x42fc, 0x43fc, 0x44fc,
	0x45fc, 0x46fc, 0x47fc, 0x48fc, 0xf0f4, 0x49fc, 0x4afc, 0x4bfc,
	0xeff4, 0x4cfc, 0x4dfc, 0xe9c2, 0x4efc, 0xe1f7, 0xe2f7, 0x4ffc,
	0x50fc, 0x51fc, 0x52fc, 0x53fc, 0xc6bb, 0x54fc, 0x55fc, 0x56fc,
	0x57fc, 0xe4d9, 0x58fc, 0x59fc, 0x5afc, 0xf2ca, 0xe8c0, 0xa4f0,
	0x5bfc, 0xdaba, 0x5cfc, 0x5dfc, 0xadc7, 0x5efc, 0x5ffc, 0x60fc,
	0xacc4, 0x61fc, 0x62fc, 0xecf7, 0xedf7, 0xeef7, 0x63fc, 0xf0f7,
	0xeff7, 0x64fc, 0xf1f7, 0x65fc, 0x66fc, 0xf4f7, 0x67fc, 0xf3f7,
	0x68fc, 0xf2f7, 0xf5f7, 0x69fc, 0x6afc, 0x6bfc, 0x6cfc, 0xf6f7,
	0x6dfc, 0x6efc, 0x6ffc, 0x70fc, 0x71fc, 0x72fc, 0x73fc, 0x74fc,
	0x75fc, 0xe9ed, 0x76fc, 0xeaed, 0xebed, 0x77fc, 0xbcf6, 0x78fc,
	0x79fc, 0x7afc, 0x7bfc, 0x7cfc, 0x7dfc, 0x7efc, 0x80fc, 0x81fc,
	0x82fc, 0x83fc, 0x84fc, 0xbdf6, 0x85fc, 0xbef6, 0xa6b6, 0x86fc,
	0xbed8, 0x87fc, 0x88fc, 0xc4b9, 0x89fc, 0x8afc, 0x8bfc, 0xbbd8,
	0x8cfc, 0xb1dc, 0x8dfc, 0x8efc, 0x8ffc, 0x90fc, 0x91fc, 0x92fc,
	0xf3ca, 0x93fc, 0xf7f7, 0x94fc, 0x95fc, 0x96fc, 0x97fc, 0x98fc,
	0x99fc, 0x9afc, 0x9bfc, 0x9cfc, 0xf8f7, 0x9dfc, 0x9efc, 0xf9f7,
	0x9ffc, 0xa0fc, 0x40fd, 0x41fd, 0x42fd, 0x43fd, 0x44fd, 0xfbf7,
	0x45fd, 0xfaf7, 0x46fd, 0xc7b1, 0x47fd, 0xfcf7, 0xfdf7, 0x48fd,
	0x49fd, 0x4afd, 0x4bfd, 0x4cfd, 0xfef7, 0x4dfd, 0x4efd, 0x4ffd,
	0x50fd, 0x51fd, 0x52fd, 0x53fd, 0x54fd, 0x55fd, 0x56fd, 0x57fd,
	0xebc6, 0xb4ec, 0x58fd, 0x59fd, 0x5afd, 0x5bfd, 0x5cfd, 0x5dfd,
	0x5efd, 0x5ffd, 0x60fd, 0x61fd, 0x62fd, 0x63fd, 0x64fd, 0x65fd,
	0x66fd, 0x67fd, 0x68fd, 0x69fd, 0x6afd, 0x6bfd, 0x6cfd, 0x6dfd,
	0x6efd, 0x6ffd, 0x70fd, 0x71fd, 0x72fd, 0x73fd, 0x74fd, 0x75fd,
	0x76fd, 0x77fd, 0x78fd, 0x79fd, 0x7afd, 0x7bfd, 0x7cfd, 0x7dfd,
	0x7efd, 0x80fd, 0x81fd, 0x82fd, 0x83fd, 0x84fd, 0x85fd, 0xddb3,
	0xb3f6, 0x86fd, 0x87fd, 0xb4f6, 0xe4c1, 0xb5f6, 0xb6f6, 0xb7f6,
	0xb8f6, 0xb9f6, 0xbaf6, 0xa3c8, 0xbbf6, 0x88fd, 0x89fd, 0x8afd,
	0x8bfd, 0x8cfd, 0x8dfd, 0x8efd, 0x8ffd, 0x90fd, 0x91fd, 0x92fd,
	0x93fd, 0xfac1, 0xa8b9, 0xe8ed, 0x94fd, 0x95fd, 0x96fd, 0xeab9,
	0xdfd9, 0x97fd, 0x98fd, 0x99fd, 0x9afd, 0x9bfd
};

const uint16_t libuna_codepage_windows_936_unicode_to_byte_stream_base_0xf900[ 320 ] = {
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x9cfd, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x9dfd, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9efd, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9ffd,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xa0fd, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x40fe, 0x41fe, 0x42fe, 0x43fe,
	0x001a, 0x44fe, 0x001a, 0x45fe, 0x46fe, 0x001a, 0x001a, 0x001a,
	0x47fe, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x48fe,
	0x49fe, 0x4afe, 0x001a, 0x4bfe, 0x4cfe, 0x001a, 0x001a, 0x4dfe,
	0x4efe, 0x4ffe, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};

const uint16_t libuna_codepage_windows_936_unicode_to_byte_stream_base_0xfe00[ 512 ] = {
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x55a9, 0xf2a6, 0x001a, 0xf4a6, 0xf5a6, 0xe0a6, 0xe1a6, 0xf0a6,
	0xf1a6, 0xe2a6, 0xe3a6, 0xeea6, 0xefa6, 0xe6a6, 0xe7a6, 0xe4a6,
	0xe5a6, 0xe8a6, 0xe9a6, 0xeaa6, 0xeba6, 0x001a, 0x001a, 0x001a,
	0x001a, 0x68a9, 0x69a9, 0x6aa9, 0x6ba9, 0x6ca9, 0x6da9, 0x6ea9,
	0x6fa9, 0x70a9, 0x71a9, 0x001a, 0x72a9, 0x73a9, 0x74a9, 0x75a9,
	0x001a, 0x76a9, 0x77a9, 0x78a9, 0x79a9, 0x7aa9, 0x7ba9, 0x7ca9,
	0x7da9, 0x7ea9, 0x80a9, 0x81a9, 0x82a9, 0x83a9, 0x84a9, 0x001a,
	0x85a9, 0x86a9, 0x87a9, 0x88a9, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xa1a3, 0xa2a3, 0xa3a3, 0xe7a1, 0xa5a3, 0xa6a3, 0xa7a3,
	0xa8a3, 0xa9a3, 0xaaa3, 0xaba3, 0xaca3, 0xada3, 0xaea3, 0xafa3,
	0xb0a3, 0xb1a3, 0xb2a3, 0xb3a3, 0xb4a3, 0xb5a3, 0xb6a3, 0xb7a3,
	0xb8a3, 0xb9a3, 0xbaa3, 0xbba3, 0xbca3, 0xbda3, 0xbea3, 0xbfa3,
	0xc0a3, 0xc1a3, 0xc2a3, 0xc3a3, 0xc4a3, 0xc5a3, 0xc6a3, 0xc7a3,
	0xc8a3, 0xc9a3, 0xcaa3, 0xcba3, 0xcca3, 0xcda3, 0xcea3, 0xcfa3,
	0xd0a3, 0xd1a3, 0xd2a3, 0xd3a3, 0xd4a3, 0xd5a3, 0xd6a3, 0xd7a3,
	0xd8a3, 0xd9a3, 0xdaa3, 0xdba3, 0xdca3, 0xdda3, 0xdea3, 0xdfa3,
	0xe0a3, 0xe1a3, 0xe2a3, 0xe3a3, 0xe4a3, 0xe5a3, 0xe6a3, 0xe7a3,
	0xe8a3, 0xe9a3, 0xeaa3, 0xeba3, 0xeca3, 0xeda3, 0xeea3, 0xefa3,
	0xf0a3, 0xf1a3, 0xf2a3, 0xf3a3, 0xf4a3, 0xf5a3, 0xf6a3, 0xf7a3,
	0xf8a3, 0xf9a3, 0xfaa3, 0xfba3, 0xfca3, 0xfda3, 0xaba1, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xe9a1, 0xeaa1, 0x56a9, 0xfea3, 0x57a9, 0xa4a3, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};

/* Determines the size of a byte stream character from an Unicode character
 * Adds the size to the byte stream character size value
 * Returns 1 if successful or -1 on error
 */
int libuna_codepage_windows_936_unicode_character_size_to_byte_stream(
     libuna_unicode_character_t unicode_character,
     size_t *byte_stream_character_size,
     libcerror_error_t **error )
{
	static char *function      = "libuna_codepage_windows_936_unicode_character_size_to_byte_stream";
	uint16_t byte_stream_value = 0x001a;

	if( byte_stream_character_size == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid byte stream character size.",
		 function );

		return( -1 );
	}
	if( unicode_character < 0x0080 )
	{
		byte_stream_value = (uint16_t) unicode_character;
	}
	else if( ( unicode_character >= 0x0080 )
	      && ( unicode_character < 0x0480 ) )
	{
		unicode_character -= 0x0080;
		byte_stream_value  = libuna_codepage_windows_936_unicode_to_byte_stream_base_0x0080[ unicode_character ];
	}
	else if( ( unicode_character >= 0x2000 )
	      && ( unicode_character < 0x2340 ) )
	{
		unicode_character -= 0x2000;
		byte_stream_value  = libuna_codepage_windows_936_unicode_to_byte_stream_base_0x2000[ unicode_character ];
	}
	else if( ( unicode_character >= 0x2440 )
	      && ( unicode_character < 0x2680 ) )
	{
		unicode_character -= 0x2440;
		byte_stream_value  = libuna_codepage_windows_936_unicode_to_byte_stream_base_0x2440[ unicode_character ];
	}
	else if( ( unicode_character >= 0x3000 )
	      && ( unicode_character < 0x3400 ) )
	{
		unicode_character -= 0x3000;
		byte_stream_value  = libuna_codepage_windows_936_unicode_to_byte_stream_base_0x3000[ unicode_character ];
	}
	else if( ( unicode_character >= 0x4e00 )
	      && ( unicode_character < 0x9fa6 ) )
	{
		unicode_character -= 0x4e00;
		byte_stream_value  = libuna_codepage_windows_936_unicode_to_byte_stream_base_0x4e00[ unicode_character ];
	}
	else if( ( unicode_character >= 0xf900 )
	      && ( unicode_character < 0xfa40 ) )
	{
		unicode_character -= 0xf900;
		byte_stream_value  = libuna_codepage_windows_936_unicode_to_byte_stream_base_0xf900[ unicode_character ];
	}
	else if( ( unicode_character >= 0xfe00 )
	      && ( unicode_character < 0x10000 ) )
	{
		unicode_character -= 0xfe00;
		byte_stream_value  = libuna_codepage_windows_936_unicode_to_byte_stream_base_0xfe00[ unicode_character ];
	}
	byte_stream_value >>= 8;

	if( byte_stream_value != 0 )
	{
		*byte_stream_character_size += 2;
	}
	else
	{
		*byte_stream_character_size += 1;
	}
	return( 1 );
}

/* Copies an Unicode character from a byte stream
 * Returns 1 if successful or -1 on error
 */
int libuna_codepage_windows_936_copy_from_byte_stream(
     libuna_unicode_character_t *unicode_character,
     const uint8_t *byte_stream,
     size_t byte_stream_size,
     size_t *byte_stream_index,
     libcerror_error_t **error )
{
	static char *function         = "libuna_codepage_windows_936_copy_from_byte_stream";
	uint8_t additional_character  = 0;
	uint8_t byte_stream_character = 0;

	if( unicode_character == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid Unicode character.",
		 function );

		return( -1 );
	}
	if( byte_stream == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid byte stream.",
		 function );

		return( -1 );
	}
	if( byte_stream_size > (size_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: invalid byte stream size value exceeds maximum.",
		 function );

		return( -1 );
	}
	if( byte_stream_index == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid byte stream index.",
		 function );

		return( -1 );
	}
	if( *byte_stream_index >= byte_stream_size )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_TOO_SMALL,
		 "%s: byte stream too small.",
		 function );

		return( -1 );
	}
	byte_stream_character = byte_stream[ *byte_stream_index ];

	if( byte_stream_character < 0x80 )
	{
		*unicode_character = byte_stream_character;
	}
	else if( byte_stream_character == 0x80 )
	{
		*unicode_character = 0x20ac;
	}
	else if( ( *byte_stream_index + 1 ) <= byte_stream_size )
	{
		*byte_stream_index += 1;

		additional_character = byte_stream[ *byte_stream_index ];

		if( ( byte_stream_character >= 0x81 )
		 && ( byte_stream_character <= 0xa0 ) )
		{
			if( additional_character >= 0x40 )
			{
				additional_character -= 0x40;

				switch( byte_stream_character )
				{
					case 0x81:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8140[ additional_character ];
						break;

					case 0x82:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8240[ additional_character ];
						break;

					case 0x83:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8340[ additional_character ];
						break;

					case 0x84:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8440[ additional_character ];
						break;

					case 0x85:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8540[ additional_character ];
						break;

					case 0x86:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8640[ additional_character ];
						break;

					case 0x87:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8740[ additional_character ];
						break;

					case 0x88:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8840[ additional_character ];
						break;

					case 0x89:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8940[ additional_character ];
						break;

					case 0x8a:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8a40[ additional_character ];
						break;

					case 0x8b:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8b40[ additional_character ];
						break;

					case 0x8c:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8c40[ additional_character ];
						break;

					case 0x8d:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8d40[ additional_character ];
						break;

					case 0x8e:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8e40[ additional_character ];
						break;

					case 0x8f:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x8f40[ additional_character ];
						break;

					case 0x90:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9040[ additional_character ];
						break;

					case 0x91:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9140[ additional_character ];
						break;

					case 0x92:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9240[ additional_character ];
						break;

					case 0x93:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9340[ additional_character ];
						break;

					case 0x94:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9440[ additional_character ];
						break;

					case 0x95:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9540[ additional_character ];
						break;

					case 0x96:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9640[ additional_character ];
						break;

					case 0x97:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9740[ additional_character ];
						break;

					case 0x98:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9840[ additional_character ];
						break;

					case 0x99:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9940[ additional_character ];
						break;

					case 0x9a:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9a40[ additional_character ];
						break;

					case 0x9b:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9b40[ additional_character ];
						break;

					case 0x9c:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9c40[ additional_character ];
						break;

					case 0x9d:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9d40[ additional_character ];
						break;

					case 0x9e:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9e40[ additional_character ];
						break;

					case 0x9f:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0x9f40[ additional_character ];
						break;

					case 0xa0:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa040[ additional_character ];
						break;
				}
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( ( byte_stream_character >= 0xa1 )
		      && ( byte_stream_character <= 0xa3 ) )
		{
			if( additional_character >= 0xa0 )
			{
				additional_character -= 0xa0;

				switch( byte_stream_character )
				{
					case 0xa1:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa1a0[ additional_character ];
						break;

					case 0xa2:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa2a0[ additional_character ];
						break;

					case 0xa3:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa3a0[ additional_character ];
						break;
				}
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( ( byte_stream_character >= 0xa4 )
		      && ( byte_stream_character <= 0xa6 ) )
		{
			if( ( additional_character >= 0xa0 )
			 && ( additional_character < 0xf8 ) )
			{
				additional_character -= 0xa0;

				switch( byte_stream_character )
				{
					case 0xa4:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa4a0[ additional_character ];
						break;

					case 0xa5:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa5a0[ additional_character ];
						break;

					case 0xa6:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa6a0[ additional_character ];
						break;
				}
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xa7 )
		{
			if( ( additional_character >= 0xa0 )
			 && ( additional_character < 0xc8 ) )
			{
				additional_character -= 0xa0;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa7a0[ additional_character ];
			}
			else if( ( additional_character >= 0xd0 )
			      && ( additional_character < 0xf8 ) )
			{
				additional_character -= 0xd0;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa7d0[ additional_character ];
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xa8 )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0x98 ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa840[ additional_character ];
			}
			else if( ( additional_character >= 0xa0 )
			      && ( additional_character < 0xf0 ) )
			{
				additional_character -= 0xa0;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa8a0[ additional_character ];
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xa9 )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0x88 ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa940[ additional_character ];
			}
			else if( additional_character == 0x88 )
			{
				*unicode_character = 0xfe6b;
			}
			else if( additional_character == 0x96 )
			{
				*unicode_character = 0x3007;
			}
			else if( ( additional_character >= 0xa0 )
			      && ( additional_character < 0xf0 ) )
			{
				additional_character -= 0xa0;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xa9a0[ additional_character ];
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xaa )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xa0 ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xaa40[ additional_character ];
			}
			else if( additional_character == 0xa0 )
			{
				*unicode_character = 0x7371;
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xab )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xa0 ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xab40[ additional_character ];
			}
			else if( additional_character == 0xa0 )
			{
				*unicode_character = 0x73f7;
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xac )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xa0 ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xac40[ additional_character ];
			}
			else if( additional_character == 0xa0 )
			{
				*unicode_character = 0x747a;
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xad )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xa0 ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xad40[ additional_character ];
			}
			else if( additional_character == 0xa0 )
			{
				*unicode_character = 0x74f2;
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xae )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xa0 ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xae40[ additional_character ];
			}
			else if( additional_character == 0xa0 )
			{
				*unicode_character = 0x7587;
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xaf )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xa0 ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xaf40[ additional_character ];
			}
			else if( additional_character == 0xa0 )
			{
				*unicode_character = 0x7644;
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( ( byte_stream_character >= 0xb0 )
		      && ( byte_stream_character <= 0xf7 ) )
		{
			if( additional_character >= 0x40 )
			{
				additional_character -= 0x40;

				switch( byte_stream_character )
				{
					case 0xb0:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb040[ additional_character ];
						break;

					case 0xb1:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb140[ additional_character ];
						break;

					case 0xb2:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb240[ additional_character ];
						break;

					case 0xb3:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb340[ additional_character ];
						break;

					case 0xb4:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb440[ additional_character ];
						break;

					case 0xb5:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb540[ additional_character ];
						break;

					case 0xb6:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb640[ additional_character ];
						break;

					case 0xb7:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb740[ additional_character ];
						break;

					case 0xb8:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb840[ additional_character ];
						break;

					case 0xb9:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xb940[ additional_character ];
						break;

					case 0xba:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xba40[ additional_character ];
						break;

					case 0xbb:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xbb40[ additional_character ];
						break;

					case 0xbc:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xbc40[ additional_character ];
						break;

					case 0xbd:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xbd40[ additional_character ];
						break;

					case 0xbe:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xbe40[ additional_character ];
						break;

					case 0xbf:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xbf40[ additional_character ];
						break;

					case 0xc0:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc040[ additional_character ];
						break;

					case 0xc1:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc140[ additional_character ];
						break;

					case 0xc2:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc240[ additional_character ];
						break;

					case 0xc3:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc340[ additional_character ];
						break;

					case 0xc4:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc440[ additional_character ];
						break;

					case 0xc5:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc540[ additional_character ];
						break;

					case 0xc6:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc640[ additional_character ];
						break;

					case 0xc7:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc740[ additional_character ];
						break;

					case 0xc8:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc840[ additional_character ];
						break;

					case 0xc9:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xc940[ additional_character ];
						break;

					case 0xca:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xca40[ additional_character ];
						break;

					case 0xcb:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xcb40[ additional_character ];
						break;

					case 0xcc:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xcc40[ additional_character ];
						break;

					case 0xcd:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xcd40[ additional_character ];
						break;

					case 0xce:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xce40[ additional_character ];
						break;

					case 0xcf:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xcf40[ additional_character ];
						break;

					case 0xd0:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd040[ additional_character ];
						break;

					case 0xd1:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd140[ additional_character ];
						break;

					case 0xd2:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd240[ additional_character ];
						break;

					case 0xd3:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd340[ additional_character ];
						break;

					case 0xd4:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd440[ additional_character ];
						break;

					case 0xd5:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd540[ additional_character ];
						break;

					case 0xd6:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd640[ additional_character ];
						break;

					case 0xd7:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd740[ additional_character ];
						break;

					case 0xd8:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd840[ additional_character ];
						break;

					case 0xd9:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xd940[ additional_character ];
						break;

					case 0xda:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xda40[ additional_character ];
						break;

					case 0xdb:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xdb40[ additional_character ];
						break;

					case 0xdc:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xdc40[ additional_character ];
						break;

					case 0xdd:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xdd40[ additional_character ];
						break;

					case 0xde:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xde40[ additional_character ];
						break;

					case 0xdf:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xdf40[ additional_character ];
						break;

					case 0xe0:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe040[ additional_character ];
						break;

					case 0xe1:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe140[ additional_character ];
						break;

					case 0xe2:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe240[ additional_character ];
						break;

					case 0xe3:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe340[ additional_character ];
						break;

					case 0xe4:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe440[ additional_character ];
						break;

					case 0xe5:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe540[ additional_character ];
						break;

					case 0xe6:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe640[ additional_character ];
						break;

					case 0xe7:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe740[ additional_character ];
						break;

					case 0xe8:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe840[ additional_character ];
						break;

					case 0xe9:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xe940[ additional_character ];
						break;

					case 0xea:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xea40[ additional_character ];
						break;

					case 0xeb:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xeb40[ additional_character ];
						break;

					case 0xec:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xec40[ additional_character ];
						break;

					case 0xed:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xed40[ additional_character ];
						break;

					case 0xee:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xee40[ additional_character ];
						break;

					case 0xef:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xef40[ additional_character ];
						break;

					case 0xf0:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf040[ additional_character ];
						break;

					case 0xf1:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf140[ additional_character ];
						break;

					case 0xf2:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf240[ additional_character ];
						break;

					case 0xf3:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf340[ additional_character ];
						break;

					case 0xf4:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf440[ additional_character ];
						break;

					case 0xf5:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf540[ additional_character ];
						break;

					case 0xf6:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf640[ additional_character ];
						break;

					case 0xf7:
						*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf740[ additional_character ];
						break;
				}
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xf8 )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xa0 ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf840[ additional_character ];
			}
			else if( additional_character == 0xa0 )
			{
				*unicode_character = 0x9d42;
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xf9 )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xa0 ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xf940[ additional_character ];
			}
			else if( additional_character == 0xa0 )
			{
				*unicode_character = 0x9da2;
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xfa )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xa0 ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xfa40[ additional_character ];
			}
			else if( additional_character == 0xa0 )
			{
				*unicode_character = 0x9e02;
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xfb )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xa0 ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xfb40[ additional_character ];
			}
			else if( additional_character == 0xa0 )
			{
				*unicode_character = 0x9eaa;
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xfc )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xa0 ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xfc40[ additional_character ];
			}
			else if( additional_character == 0xa0 )
			{
				*unicode_character = 0x9f31;
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xfd )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xa0 ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xfd40[ additional_character ];
			}
			else if( additional_character == 0xa0 )
			{
				*unicode_character = 0xf9f1;
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xfe )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0x50 ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_936_byte_stream_to_unicode_base_0xfe40[ additional_character ];
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else
		{
			*unicode_character = 0xfffd;
		}
	}
	else
	{
		*unicode_character = 0xfffd;
	}
	*byte_stream_index += 1;

	return( 1 );
}

/* Copies an Unicode character to a byte stream
 * Returns 1 if successful or -1 on error
 */
int libuna_codepage_windows_936_copy_to_byte_stream(
     libuna_unicode_character_t unicode_character,
     uint8_t *byte_stream,
     size_t byte_stream_size,
     size_t *byte_stream_index,
     libcerror_error_t **error )
{
	static char *function      = "libuna_codepage_windows_936_copy_to_byte_stream";
	uint16_t byte_stream_value = 0x001a;

	if( byte_stream == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid byte stream.",
		 function );

		return( -1 );
	}
	if( byte_stream_size > (size_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: invalid byte stream size value exceeds maximum.",
		 function );

		return( -1 );
	}
	if( byte_stream_index == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid byte stream index.",
		 function );

		return( -1 );
	}
	if( *byte_stream_index >= byte_stream_size )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_TOO_SMALL,
		 "%s: byte stream too small.",
		 function );

		return( -1 );
	}
	if( unicode_character < 0x0080 )
	{
		byte_stream_value = (uint16_t) unicode_character;
	}
	else if( ( unicode_character >= 0x0080 )
	      && ( unicode_character < 0x0480 ) )
	{
		unicode_character -= 0x0080;
		byte_stream_value  = libuna_codepage_windows_936_unicode_to_byte_stream_base_0x0080[ unicode_character ];
	}
	else if( ( unicode_character >= 0x2000 )
	      && ( unicode_character < 0x2340 ) )
	{
		unicode_character -= 0x2000;
		byte_stream_value  = libuna_codepage_windows_936_unicode_to_byte_stream_base_0x2000[ unicode_character ];
	}
	else if( ( unicode_character >= 0x2440 )
	      && ( unicode_character < 0x2680 ) )
	{
		unicode_character -= 0x2440;
		byte_stream_value  = libuna_codepage_windows_936_unicode_to_byte_stream_base_0x2440[ unicode_character ];
	}
	else if( ( unicode_character >= 0x3000 )
	      && ( unicode_character < 0x3400 ) )
	{
		unicode_character -= 0x3000;
		byte_stream_value  = libuna_codepage_windows_936_unicode_to_byte_stream_base_0x3000[ unicode_character ];
	}
	else if( ( unicode_character >= 0x4e00 )
	      && ( unicode_character < 0x9fa6 ) )
	{
		unicode_character -= 0x4e00;
		byte_stream_value  = libuna_codepage_windows_936_unicode_to_byte_stream_base_0x4e00[ unicode_character ];
	}
	else if( ( unicode_character >= 0xf900 )
	      && ( unicode_character < 0xfa40 ) )
	{
		unicode_character -= 0xf900;
		byte_stream_value  = libuna_codepage_windows_936_unicode_to_byte_stream_base_0xf900[ unicode_character ];
	}
	else if( ( unicode_character >= 0xfe00 )
	      && ( unicode_character < 0x10000 ) )
	{
		unicode_character -= 0xfe00;
		byte_stream_value  = libuna_codepage_windows_936_unicode_to_byte_stream_base_0xfe00[ unicode_character ];
	}
	byte_stream[ *byte_stream_index ] = (uint8_t) ( byte_stream_value & 0x00ff );

	byte_stream_value >>= 8;

	if( byte_stream_value != 0 )
	{
		*byte_stream_index += 1;

		byte_stream[ *byte_stream_index ] = (uint8_t) ( byte_stream_value & 0x00ff );
	}
	*byte_stream_index += 1;

	return( 1 );
}

