/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     disc_lagrange_0_2d.c                                           */
/*                                                                          */
/* description:  piecewise constant discontinuous Lagrange elements in 2d   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

static const REAL d_bary0_2d[1][N_LAMBDA] = {{1.0/3.0, 1.0/3.0, 1.0/3.0, 0.0}};

static REAL d_phi0c0_2d(const REAL lambda[N_LAMBDA])
{
  return(1.0);
}

static const REAL *d_grd_phi0c0_2d(const REAL lambda[N_LAMBDA])
{
  static const REAL  grd[N_LAMBDA] = {0};

  return(grd);
}

static const REAL (*d_D2_phi0c0_2d(const REAL *lambda))[N_LAMBDA]
{
  static  const REAL D2[N_LAMBDA][N_LAMBDA];

  return(D2);
}

/*--------------------------------------------------------------------------*/
/*  functions for combining basisfunctions with coefficients                */
/*--------------------------------------------------------------------------*/

static const DOF *d_get_dof_indices0_2d(const EL *el, const DOF_ADMIN *admin,
					DOF *idof)
{
  static DOF  index_vec[1];
  DOF         *rvec = idof ? idof : index_vec;
  int         i = admin->mesh->node[CENTER], n0 = admin->n0_dof[CENTER];
  DOF         **dof = el->dof;

  rvec[0] = dof[i][n0];

  return((const DOF *) rvec);
}

static const S_CHAR *d_get_bound0_2d(const EL_INFO *el_info, S_CHAR *bound)
{
  static S_CHAR  bound_vec[1];
  S_CHAR         *rvec = bound ? bound : bound_vec;

  rvec[0] = INTERIOR;

  return((const S_CHAR *) rvec);
}

static const int *d_get_int_vec0_2d(const EL *el, const DOF_INT_VEC *vec,
				    int *ivec)
{
  FUNCNAME("d_get_int_vec0_2d");
  static int  local_vec[1];
  int         i, n0, *v = nil;
  int         *rvec = ivec ? ivec : local_vec;
  DOF         **dof = el->dof;

  GET_DOF_VEC(v, vec);
  i = vec->fe_space->admin->mesh->node[CENTER];
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  
  rvec[0] = v[dof[i][n0]];

  return((const int *) rvec);
}

static const REAL *d_get_real_vec0_2d(const EL *el, const DOF_REAL_VEC *vec,
				      REAL *Rvec)
{
  FUNCNAME("d_get_real_v0_2d");
  int          i, n0;
  static REAL  local_vec[1];
  REAL         *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF          **dof = el->dof;

  GET_DOF_VEC(v, vec);
  i = vec->fe_space->admin->mesh->node[CENTER];
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  
  rvec[0] = v[dof[i][n0]];

  return((const REAL *) rvec);
}

static const REAL_D *d_get_real_d_vec0_2d(const EL *el,
					  const DOF_REAL_D_VEC *vec,
					  REAL_D *Rvec)
{
  FUNCNAME("d_get_real_d_vec0_2d");
  int           i, k, n0;
  static REAL_D local_vec[1];
  REAL_D        *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF           **dof = el->dof;

  GET_DOF_VEC(v, vec);
  i = vec->fe_space->admin->mesh->node[CENTER];
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  
  for (k = 0; k < DIM_OF_WORLD; k++)
    rvec[0][k] = v[dof[i][n0]][k];

  return((const REAL_D *) rvec);
}

static const U_CHAR *d_get_uchar_vec0_2d(const EL *el, const DOF_UCHAR_VEC *vec,
					 U_CHAR *uvec)
{
  FUNCNAME("d_get_uchar_vec0_2d");
  int           i, n0;
  static U_CHAR local_vec[1];
  U_CHAR        *v = nil, *rvec = uvec ? uvec : local_vec;
  DOF           **dof = el->dof;

  GET_DOF_VEC(v, vec);
  i = vec->fe_space->admin->mesh->node[CENTER];
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  
  rvec[0] = v[dof[i][n0]];

  return((const U_CHAR *) rvec);
}

static const S_CHAR *d_get_schar_vec0_2d(const EL *el, const DOF_SCHAR_VEC *vec,
					 S_CHAR *svec)
{
  FUNCNAME("d_get_schar_vec0_2d");
  int            i, n0;
  static S_CHAR  local_vec[1];
  S_CHAR         *v = nil, *rvec = svec ? svec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);
  i = vec->fe_space->admin->mesh->node[CENTER];
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  
  rvec[0] = v[dof[i][n0]];

  return((const S_CHAR *) rvec);
}

static const REAL *d_interpol0_2d(const EL_INFO *el_info, int no, 
				  const int *b_no, REAL (*f)(const REAL_D),
				  REAL (*f_loc)(const EL_INFO *,
						const REAL [N_LAMBDA]), 
				  REAL *vec)
{
  FUNCNAME("d_interpol0_2d");
  static REAL       inter[1];
  REAL             *rvec = vec ? vec : inter;
  const PARAMETRIC *parametric = el_info->mesh->parametric;

  DEBUG_TEST_EXIT(!b_no || (no == 1), "only for one point!\n");

  if(f_loc)
    rvec[0] = f_loc(el_info, d_bary0_2d[0]);
  else {
    if (parametric) {
      REAL_D     world[1];

      parametric->init_element(el_info, parametric);
      parametric->coord_to_world(el_info, nil, 1, d_bary0_2d, world);

      rvec[0] = f(world[0]);
    }
    else {
      REAL_D world;
	
      DEBUG_TEST_FLAG(FILL_COORDS, el_info);
	
      coord_to_world(el_info, d_bary0_2d[0], world);

      rvec[0] = f(world);
    }
  }

  return((const REAL *) rvec);
}

static const REAL_D *d_interpol_d0_2d(const EL_INFO *el_info, int no, 
				      const int *b_no,
				      const REAL *(*f)(const REAL_D, REAL_D),
				      const REAL *(*f_loc)(const EL_INFO *,
							   const REAL [N_LAMBDA],
							   REAL_D),
				      REAL_D *vec)
{
  FUNCNAME("d_interpol_d0_2d");
  static REAL_D     inter[1];
  REAL_D           *rvec = vec ? vec : inter;
  const PARAMETRIC *parametric = el_info->mesh->parametric;

  DEBUG_TEST_EXIT(!b_no || (no == 1), "only for one point!\n");

  if(f_loc)
    f_loc(el_info, d_bary0_2d[0], rvec[0]);
  else {
    if (parametric) {
      REAL_D world[1];

      parametric->init_element(el_info, parametric);
      parametric->coord_to_world(el_info, nil, 1, d_bary0_2d, world);

      f(world[0], rvec[0]);
    }
    else {
      REAL_D world;
      
      DEBUG_TEST_FLAG(FILL_COORDS, el_info);

      coord_to_world(el_info, d_bary0_2d[0], world);          
      f(world, rvec[0]);                                                   
    }
  }

  return((const REAL_D *) rvec);
}

/*--------------------------------------------------------------------------*/
/*  functions for interpolation/ restriction during refinement/coarsening   */
/*--------------------------------------------------------------------------*/

static void d_real_refine_inter0_2d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("d_real_refine_inter0_2d");
  EL      *el;
  REAL    *vec = nil;
  int     cdof, pdof, node0, n0, i;

  if (n < 1) return;
  GET_DOF_VEC(vec, drv);
  node0 = drv->fe_space->mesh->node[CENTER];        
  n0 = drv->fe_space->admin->n0_dof[CENTER];

  for (i = 0; i < n; i++)
  {
    el = list[i].el_info.el;

    pdof = el->dof[node0][n0];
    cdof = el->child[0]->dof[node0][n0];
    vec[cdof] = vec[pdof];
    cdof = el->child[1]->dof[node0][n0];
    vec[cdof] = vec[pdof];
  }
  return;
}

static void d_real_coarse_inter0_2d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("d_real_coarse_inter0_2d");
  EL      *el;
  REAL    *vec = nil;
  int     cdof0, cdof1, pdof, node0, n0, i;

  if (n < 1) return;
  GET_DOF_VEC(vec, drv);
  node0 = drv->fe_space->mesh->node[CENTER];        
  n0 = drv->fe_space->admin->n0_dof[CENTER];

  for (i = 0; i < n; i++)
  {
    el = list[i].el_info.el;

    pdof = el->dof[node0][n0];
    cdof0 = el->child[0]->dof[node0][n0];
    cdof1 = el->child[1]->dof[node0][n0];

    vec[pdof] = 0.5*(vec[cdof0] + vec[cdof1]);
  }
  return;
}

static void d_real_coarse_restr0_2d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("d_real_coarse_restr0_2d");
  EL      *el;
  REAL    *vec = nil;
  int     cdof0, cdof1, pdof, node0, n0, i;

  if (n < 1) return;
  GET_DOF_VEC(vec, drv);
  node0 = drv->fe_space->mesh->node[CENTER];        
  n0 = drv->fe_space->admin->n0_dof[CENTER];

  for (i = 0; i < n; i++)
  {
    el = list[i].el_info.el;

    pdof = el->dof[node0][n0];
    cdof0 = el->child[0]->dof[node0][n0];
    cdof1 = el->child[1]->dof[node0][n0];

    vec[pdof] = vec[cdof0] + vec[cdof1];
  }
  return;
}

static void d_real_d_refine_inter0_2d(DOF_REAL_D_VEC *drdv, RC_LIST_EL *list,
				      int n)
{
  FUNCNAME("d_real_d_refine_inter0_2d");
  EL      *el;
  REAL_D  *vec = nil;
  int     cdof, pdof, node0, n0, i, k;

  if (n < 1) return;
  GET_DOF_VEC(vec, drdv);
  node0 = drdv->fe_space->mesh->node[CENTER];        
  n0 = drdv->fe_space->admin->n0_dof[CENTER];

  for (i = 0; i < n; i++)
  {
    el = list[i].el_info.el;

    pdof = el->dof[node0][n0];
    cdof = el->child[0]->dof[node0][n0];
    for (k = 0; k < DIM_OF_WORLD; k++)
      vec[cdof][k] = vec[pdof][k];
    cdof = el->child[1]->dof[node0][n0];
    for (k = 0; k < DIM_OF_WORLD; k++)
      vec[cdof][k] = vec[pdof][k];
  }
  return;
}

static void d_real_d_coarse_inter0_2d(DOF_REAL_D_VEC *drdv,
				      RC_LIST_EL *list, int n)
{
  FUNCNAME("d_real_coarse_inter0_2d");
  EL      *el;
  REAL_D  *vec = nil;
  int     cdof0, cdof1, pdof, node0, n0, i, k;

  if (n < 1) return;
  GET_DOF_VEC(vec, drdv);
  node0 = drdv->fe_space->mesh->node[CENTER];        
  n0 = drdv->fe_space->admin->n0_dof[CENTER];

  for (i = 0; i < n; i++)
  {
    el = list[i].el_info.el;

    pdof = el->dof[node0][n0];
    cdof0 = el->child[0]->dof[node0][n0];
    cdof1 = el->child[1]->dof[node0][n0];

    for (k = 0; k < DIM_OF_WORLD; k++)
      vec[pdof][k] = 0.5*(vec[cdof0][k] + vec[cdof1][k]);
  }
  return;
}

static void d_real_d_coarse_restr0_2d(DOF_REAL_D_VEC *drdv,
				      RC_LIST_EL *list, int n)
{
  FUNCNAME("d_real_d_coarse_restr0_2d");
  EL      *el;
  REAL_D  *vec = nil;
  int     cdof0, cdof1, pdof, node0, n0, i, k;

  if (n < 1) return;
  GET_DOF_VEC(vec, drdv);
  node0 = drdv->fe_space->mesh->node[CENTER];        
  n0 = drdv->fe_space->admin->n0_dof[CENTER];

  for (i = 0; i < n; i++)
  {
    el = list[i].el_info.el;

    pdof = el->dof[node0][n0];
    cdof0 = el->child[0]->dof[node0][n0];
    cdof1 = el->child[1]->dof[node0][n0];

    for (k = 0; k < DIM_OF_WORLD; k++)
      vec[pdof] [k]= vec[cdof0][k] + vec[cdof1][k];
  }
  return;
}

static BAS_FCT      *d_phi0_2d[1]     = {d_phi0c0_2d};
static GRD_BAS_FCT  *d_grd_phi0_2d[1] = {d_grd_phi0c0_2d};
static D2_BAS_FCT   *d_D2_phi0_2d[1]  = {d_D2_phi0c0_2d};

static BAS_FCTS disc_lagrange0_2d = {"disc_lagrange0_2d", 2, 1, 0,
				     {0, 1, 0, 0}, 
				     nil, 
				     d_phi0_2d, d_grd_phi0_2d, d_D2_phi0_2d,
				     d_get_dof_indices0_2d, 
				     d_get_bound0_2d,
				     d_interpol0_2d,
				     d_interpol_d0_2d,
				     d_get_int_vec0_2d,
				     d_get_real_vec0_2d,
				     d_get_real_d_vec0_2d,
				     d_get_uchar_vec0_2d,
				     d_get_schar_vec0_2d,
				     d_real_refine_inter0_2d,
				     d_real_coarse_inter0_2d,
				     d_real_coarse_restr0_2d,
				     d_real_d_refine_inter0_2d,
				     d_real_d_coarse_inter0_2d,
				     d_real_d_coarse_restr0_2d,
				     d_bary0_2d, };
