package latexDraw.generators.svg;

import java.awt.Color;
import java.awt.FontMetrics;
import java.awt.geom.Point2D;
import java.util.Vector;

import latexDraw.figures.Figure;
import latexDraw.figures.Grid;
import latexDraw.figures.LaTeXDrawRectangle;
import latexDraw.parsers.svg.CSSColors;
import latexDraw.parsers.svg.SVGAttributes;
import latexDraw.parsers.svg.SVGDocument;
import latexDraw.parsers.svg.SVGTransform;
import latexDraw.parsers.svg.elements.*;
import latexDraw.parsers.svg.parsers.SVGPointsParser;
import latexDraw.util.LaTeXDrawNamespace;
import latexDraw.util.LaTeXDrawPoint2D;

/**
 * Defines a SVG generator for a grid.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 11/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class LGridSVGGenerator extends LShapeSVGGenerator
{

	public LGridSVGGenerator(Grid f)
	{
		super(f);
	}

	
	
	public LGridSVGGenerator(SVGGElement elt)
	{
		this(elt, true);
	}
	
	
	
	/**
	 * Creates a grid from a latexdraw-SVG element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LGridSVGGenerator(SVGGElement elt, boolean withTransformation)
	{
		this(new Grid(new LaTeXDrawPoint2D(), false));

		if(elt==null)
			throw new IllegalArgumentException();
		
		String pref = LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':', v;
		Grid g = (Grid)getShape();
		Vector<Point2D> values;
		
		g.setLinesColor(elt.getStroke());
		
		values = SVGPointsParser.getPoints(elt.getAttribute(pref+LaTeXDrawNamespace.XML_GRID_END));
			
		if(values!=null && values.size()>0) { 
			g.setGridEndX((int)values.elementAt(0).getX());  
			g.setGridEndY((int)values.elementAt(0).getY());  
		}
	
		values = SVGPointsParser.getPoints(elt.getAttribute(pref+LaTeXDrawNamespace.XML_GRID_START));
		
		if(values!=null && values.size()>0) { 
			g.setGridStartX((int)values.elementAt(0).getX());  
			g.setGridStartY((int)values.elementAt(0).getY());  
		}
	
		values = SVGPointsParser.getPoints(elt.getAttribute(pref+LaTeXDrawNamespace.XML_GRID_ORIGIN));

		if(values!=null && values.size()>0) { 
			g.setOriginX((int)values.elementAt(0).getX());  
			g.setOriginY((int)values.elementAt(0).getY());  
		}
		
		v = elt.getAttribute(pref+LaTeXDrawNamespace.XML_GRID_X_SOUTH);
		
		if(v!=null)
			g.setXLabelSouth(Boolean.valueOf(v));
		
		v = elt.getAttribute(pref+LaTeXDrawNamespace.XML_GRID_Y_WEST);
		
		if(v!=null)
			g.setYLabelWest(Boolean.valueOf(v));
		
		SVGElement gridElt = getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_GRID_SUB);
		
		if(gridElt!=null)
		{
			boolean isGridDotted = false;
			v = gridElt.getAttribute(pref+LaTeXDrawNamespace.XML_GRID_DOTS);
			
			if(v!=null)
				try
				{ 
					g.setSubGridDots(Double.valueOf(v).intValue());
					isGridDotted = g.getSubGridDots()>0;
				}
				catch(NumberFormatException e) { /* */ }
				
			v = gridElt.getAttribute(pref+LaTeXDrawNamespace.XML_GRID_SUB_DIV);
			
			if(v!=null)
				try{ g.setSubGridDiv(Double.valueOf(v).intValue());  }
				catch(NumberFormatException e) { /* */ }
			
			v = gridElt.getAttribute(pref+LaTeXDrawNamespace.XML_GRID_WIDTH);
			
			if(isGridDotted)
				g.setSubGridColor(CSSColors.getRGBColour(gridElt.getFill()));
			else
				g.setSubGridColor(gridElt.getStroke());
			
			if(v==null)
			{
				double st = gridElt.getStrokeWidth();
				
				if(!Double.isNaN(st))
					g.setSubGridWidth((float)st);
			}
			else
				try{ g.setSubGridWidth(Double.valueOf(v).floatValue());  }
				catch(NumberFormatException e) { /* */ }
		}
		
		gridElt = getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_GRID);
		
		if(gridElt!=null)
		{
			boolean isGridDotted = false;
			v = gridElt.getAttribute(pref+LaTeXDrawNamespace.XML_GRID_DOTS);
			
			if(v!=null)
				
				try
				{ 
					g.setGridDots(Double.valueOf(v).intValue());
					isGridDotted = g.getGridDots()>0;
				}
				catch(NumberFormatException e) { /* */ }
			
			v = gridElt.getAttribute(pref+LaTeXDrawNamespace.XML_GRID_WIDTH);
			
			if(isGridDotted)
				g.setLinesColor(CSSColors.getRGBColour(gridElt.getFill()));
			else
				g.setLinesColor(gridElt.getStroke());
			
			if(v==null)
			{
				double st = gridElt.getStrokeWidth();
				
				if(!Double.isNaN(st))
					g.setGridWidth((float)st);
			}
			else
				try{ g.setGridWidth(Double.valueOf(v).floatValue());  }
				catch(NumberFormatException e) { /* */ }
		}
		
		gridElt = getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_TEXT);
		
		if(gridElt==null)
			g.setGridLabelsSize(0);
		else
		{
			v = gridElt.getAttribute(gridElt.getUsablePrefix()+SVGAttributes.SVG_FONT_SIZE);
			
			if(v!=null)
				try { g.setGridLabelsSize(Double.valueOf(v).intValue()); }
				catch(NumberFormatException e) { /* */ }
				
			g.setGridLabelsColor(gridElt.getStroke());
		}
		
		setNumber(elt);
		g.updateFonts();
		g.updateBorders(g.getFontMetrics());
		
		if(withTransformation)
			applyTransformations(elt);
	}
	
	

	
	protected void createSVGGrid(SVGElement elt, SVGDocument document)
	{
		if(elt==null || document==null)
			return ;
		
		Grid grid = (Grid)shape;
		grid.updateBorders();
		
		String pref = LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':';
		double unit = grid.getUnit();
		Point2D.Double gridEnd = grid.getGridEnd();
		Point2D.Double gridStart = grid.getGridStart();
		int subGridDiv = grid.getSubGridDiv();
		LaTeXDrawRectangle borders = grid.getBorders();
		double rotationAngle = grid.getRotationAngle();
		double XStep = Figure.PPC*unit, i, j, XSubStep, k, l, m, n;
		double YStep = XStep, YSubStep;
		XStep *= gridEnd.x<gridStart.x ? -1 : 1 ;
		YStep *= gridEnd.y<gridStart.y ? -1 : 1 ;
		XSubStep = XStep/subGridDiv;
		YSubStep = YStep/subGridDiv;
		int subGridDots = grid.getSubGridDots();
		int gridDots = grid.getGridDots();
		int gridLabelsSize = grid.getGridLabelsSize();
		LaTeXDrawPoint2D NW = (LaTeXDrawPoint2D)borders.getTheNWPoint().clone();
		LaTeXDrawPoint2D SE = (LaTeXDrawPoint2D)borders.getTheSEPoint().clone();	
		double minX, maxX, minY, maxY;
		double absStep = Math.abs(XStep);
		Color subGridColor = grid.getSubGridColor();
		Color linesColor = grid.getLinesColor();
		Color gridLabelsColor = grid.getGridLabelsColor();
		float subGridWidth = grid.getSubGridWidth();
		float gridWidth = grid.getGridWidth();
		double posX = Math.min(gridStart.x, gridEnd.x)*Figure.PPC*unit;
		double posY = -Math.min(gridStart.y, gridEnd.y)*Figure.PPC*unit;
		FontMetrics fontMetrics = grid.getFontMetrics();
		Point2D.Double origin = grid.getOrigin();
		boolean isXLabelSouth = grid.isXLabelSouth();
		boolean isYLabelWest = grid.isYLabelWest();
		Point2D.Double position = grid.getPosition();
		
		NW.x -= position.x;
		SE.x -= position.x;
		NW.y -= position.y;
		SE.y -= position.y;
		elt.setAttribute(SVGAttributes.SVG_TRANSFORM, SVGTransform.createTranslation(position.x, position.y).toString());
		
		if(gridEnd.x<gridStart.x)
		{
			minX = gridEnd.x;
			maxX = gridStart.x;
		}else
		{
			minX = gridStart.x;
			maxX = gridEnd.x;
		}
		
		if(gridEnd.y<gridStart.y)
		{
			minY = gridEnd.y;
			maxY = gridStart.y;
		}else
		{
			minY = gridStart.y;
			maxY = gridEnd.y;
		}
		
		// We draw the sub-grid
		if(subGridDots>0)
		{
			double dotStep = (unit*Figure.PPC)/(subGridDots*subGridDiv);
			double nbX = (maxX-minX)*subGridDiv;
			double nbY = (maxY-minY)*subGridDiv;
			SVGElement subgridDots = new SVGGElement(document), dot;
			
			subgridDots.setAttribute(SVGAttributes.SVG_FILL, CSSColors.getColorName(subGridColor, true));
			subgridDots.setAttribute(pref+LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_GRID_SUB);
			subgridDots.setAttribute(pref+LaTeXDrawNamespace.XML_GRID_DOTS, String.valueOf(subGridDots));
			subgridDots.setAttribute(pref+LaTeXDrawNamespace.XML_GRID_SUB_DIV, String.valueOf(subGridDots));
			subgridDots.setAttribute(pref+LaTeXDrawNamespace.XML_GRID_WIDTH, String.valueOf(subGridWidth));
			
			for(i=0, n=NW.x; i<nbX; i++, n+=XSubStep)
				for(j=0, m=NW.y; j<=nbY; j++, m+=YSubStep)
					for(k=0; k<subGridDots; k++)
					{
						dot = new SVGCircleElement(document);
						dot.setAttribute(SVGAttributes.SVG_CX, String.valueOf(n+k*dotStep));
						dot.setAttribute(SVGAttributes.SVG_CY, String.valueOf(m));
						dot.setAttribute(SVGAttributes.SVG_R, String.valueOf(subGridWidth/2.));
						subgridDots.appendChild(dot);
					}
			
			for(j=0, n=NW.y; j<nbY; j++, n+=YSubStep)
				for(i=0, m=NW.x; i<=nbX; i++, m+=XSubStep)
					for(k=0; k<subGridDots; k++)
					{
						dot = new SVGCircleElement(document);
						dot.setAttribute(SVGAttributes.SVG_CX, String.valueOf(m));
						dot.setAttribute(SVGAttributes.SVG_CY, String.valueOf(n+k*dotStep));
						dot.setAttribute(SVGAttributes.SVG_R, String.valueOf(subGridWidth/2.));
						subgridDots.appendChild(dot);
					}
			
			dot = new SVGCircleElement(document);
			dot.setAttribute(SVGAttributes.SVG_CX, String.valueOf(SE.x));
			dot.setAttribute(SVGAttributes.SVG_CY, String.valueOf(SE.y));
			dot.setAttribute(SVGAttributes.SVG_R, String.valueOf(subGridWidth/2.));
			
			elt.appendChild(subgridDots);
		}
		else
			if(subGridDiv>1)
			{
				SVGElement subgrids = new SVGGElement(document), line;
				
				subgrids.setAttribute(SVGAttributes.SVG_STROKE_WIDTH, String.valueOf(subGridWidth));
				subgrids.setAttribute(SVGAttributes.SVG_STROKE, CSSColors.getColorName(subGridColor, true));
				subgrids.setAttribute(SVGAttributes.SVG_STROKE_LINECAP, SVGAttributes.SVG_LINECAP_VALUE_ROUND);
				subgrids.setAttribute(pref+LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_GRID_SUB);
				subgrids.setAttribute(pref+LaTeXDrawNamespace.XML_GRID_DOTS, String.valueOf(subGridDots));
				subgrids.setAttribute(pref+LaTeXDrawNamespace.XML_GRID_SUB_DIV, String.valueOf(subGridDiv));

				for(k=minX, i=posX; k<maxX; i+=XStep, k++)
					for(j=0; j<=subGridDiv; j++)
					{
						line = new SVGLineElement(document);
						line.setAttribute(SVGAttributes.SVG_X1, String.valueOf(i+XSubStep*j));
						line.setAttribute(SVGAttributes.SVG_X2, String.valueOf(i+XSubStep*j));
						line.setAttribute(SVGAttributes.SVG_Y1, String.valueOf(SE.y));
						line.setAttribute(SVGAttributes.SVG_Y2, String.valueOf(NW.y));
						subgrids.appendChild(line);
					}
				
				for(k=minY, i=posY; k<maxY; i-=YStep, k++)
					for(j=0; j<=subGridDiv; j++)
					{
						line = new SVGLineElement(document);
						line.setAttribute(SVGAttributes.SVG_X1, String.valueOf(NW.x));
						line.setAttribute(SVGAttributes.SVG_X2, String.valueOf(SE.x));
						line.setAttribute(SVGAttributes.SVG_Y1, String.valueOf(i-YSubStep*j));
						line.setAttribute(SVGAttributes.SVG_Y2, String.valueOf(i-YSubStep*j));
						subgrids.appendChild(line);
					}
				
				elt.appendChild(subgrids);
			}
		
		if(gridDots>0)
		{
			double dotStep = (unit*Figure.PPC)/gridDots;
			
			SVGElement gridDotsElt = new SVGGElement(document), dot;
			
			gridDotsElt.setAttribute(SVGAttributes.SVG_FILL, CSSColors.getColorName(linesColor, true));
			gridDotsElt.setAttribute(pref+LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_GRID);
			gridDotsElt.setAttribute(pref+LaTeXDrawNamespace.XML_GRID_DOTS, String.valueOf(gridDots));
			gridDotsElt.setAttribute(pref+LaTeXDrawNamespace.XML_GRID_WIDTH, String.valueOf(gridWidth));

			for(k=minX, i=posX; k<=maxX; i+=XStep, k++)
				for(m=NW.y, n=minY; n<maxY; n++, m+=absStep)
					for(l=0, j=m; l<gridDots; l++, j+=dotStep)
					{
						dot = new SVGCircleElement(document);
						dot.setAttribute(SVGAttributes.SVG_CX, String.valueOf(i));
						dot.setAttribute(SVGAttributes.SVG_CY, String.valueOf(j));
						dot.setAttribute(SVGAttributes.SVG_R, String.valueOf(gridWidth/2.));
						gridDotsElt.appendChild(dot);
					}
			
			for(k=minY, i=posY; k<=maxY; i-=YStep, k++)
				for(m=NW.x, n=minX; n<maxX; n++, m+=absStep)
					for(l=0, j=m; l<gridDots; l++, j+=dotStep)
					{
						dot = new SVGCircleElement(document);
						dot.setAttribute(SVGAttributes.SVG_CX, String.valueOf(j));
						dot.setAttribute(SVGAttributes.SVG_CY, String.valueOf(i));
						dot.setAttribute(SVGAttributes.SVG_R, String.valueOf(gridWidth/2.));
						gridDotsElt.appendChild(dot);
					}
			
			dot = new SVGCircleElement(document);
			dot.setAttribute(SVGAttributes.SVG_CX, String.valueOf(SE.x));
			dot.setAttribute(SVGAttributes.SVG_CY, String.valueOf(SE.y));
			dot.setAttribute(SVGAttributes.SVG_R, String.valueOf(gridWidth/2.));
			gridDotsElt.appendChild(dot);
			
			elt.appendChild(gridDotsElt);
		}
		else
		{
			SVGElement grids = new SVGGElement(document), line;
			
			grids.setAttribute(SVGAttributes.SVG_STROKE_WIDTH, String.valueOf(gridWidth));
			grids.setAttribute(SVGAttributes.SVG_STROKE, CSSColors.getColorName(linesColor, true));
			grids.setAttribute(SVGAttributes.SVG_STROKE_LINECAP, SVGAttributes.SVG_LINECAP_VALUE_SQUARE);
			grids.setAttribute(pref+LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_GRID);
			
			for(k=minX, i=posX; k<=maxX; i+=XStep, k++)
			{
				line = new SVGLineElement(document);
				line.setAttribute(SVGAttributes.SVG_X1, String.valueOf(i));
				line.setAttribute(SVGAttributes.SVG_X2, String.valueOf(i));
				line.setAttribute(SVGAttributes.SVG_Y1, String.valueOf(SE.y));
				line.setAttribute(SVGAttributes.SVG_Y2, String.valueOf(NW.y));
				grids.appendChild(line);
			}
			
			for(k=minY, i=posY; k<=maxY; i-=YStep, k++)
			{
				line = new SVGLineElement(document);
				line.setAttribute(SVGAttributes.SVG_X1, String.valueOf(NW.x));
				line.setAttribute(SVGAttributes.SVG_X2, String.valueOf(SE.x));
				line.setAttribute(SVGAttributes.SVG_Y1, String.valueOf(i));
				line.setAttribute(SVGAttributes.SVG_Y2, String.valueOf(i));
				grids.appendChild(line);
			}
			
			elt.appendChild(grids);
		}

		if(gridLabelsSize>0)
		{
			// We draw the labels
			float labelHeight = fontMetrics.getAscent();
			float labelWidth = fontMetrics.stringWidth(String.valueOf((int)maxX));
			String label;
			double xorigin = XStep*origin.x;
			double yorigin = isXLabelSouth  ? YStep*origin.y+labelHeight : YStep*origin.y-2;
			double width=gridWidth/2., tmp = isXLabelSouth ? width : -width;
		
			SVGElement texts = new SVGGElement(document), text;
			
			texts.setAttribute(SVGAttributes.SVG_FONT_SIZE, String.valueOf(grid.getFont().getSize()));
			texts.setAttribute(SVGAttributes.SVG_STROKE, CSSColors.getColorName(gridLabelsColor, true));
			texts.setAttribute(pref+LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_TEXT);
			
			
			for(i=NW.x + (isYLabelWest ? width+gridLabelsSize/4. : -width-labelWidth-gridLabelsSize/4.), j=minX; j<=maxX; i+=absStep, j++)
			{
				text = new SVGTextElement(document);
				text.setAttribute(SVGAttributes.SVG_X, String.valueOf((int)i));
				text.setAttribute(SVGAttributes.SVG_Y, String.valueOf((int)(yorigin+tmp)));
				text.setTextContent(String.valueOf((int)j));
				texts.appendChild(text);
			}
	
			if(isYLabelWest)
				for(i=NW.y + (isXLabelSouth ? -width-gridLabelsSize/4. : width+labelHeight), j=maxY ; j>=minY; i+=absStep, j--)
				{
					label = String.valueOf((int)j);
					text = new SVGTextElement(document);
					text.setAttribute(SVGAttributes.SVG_X, String.valueOf((int)(xorigin-fontMetrics.stringWidth(label)-gridLabelsSize/4.-width)));
					text.setAttribute(SVGAttributes.SVG_Y, String.valueOf((int)i));
					text.setTextContent(label);
					texts.appendChild(text);
				}
			else
				for(i=NW.y + (isXLabelSouth ? -width-gridLabelsSize/4. : width+labelHeight), j=maxY; j>=minY; i+=absStep, j--)
				{
					label = String.valueOf((int)j);
					text = new SVGTextElement(document);
					text.setAttribute(SVGAttributes.SVG_X, String.valueOf((int)(xorigin+gridLabelsSize/4.+width)));
					text.setAttribute(SVGAttributes.SVG_Y, String.valueOf((int)i));
					text.setTextContent(label);
					texts.appendChild(text);
				}
			
			elt.appendChild(texts);
		}

		if(rotationAngle%(Math.PI*2)!=0)
			setSVGRotationAttribute(elt);
		
		grid.updateBorders(fontMetrics);
	}

	
	
	@Override
	public SVGElement toSVG(SVGDocument doc)
	{
		if(doc==null)
			return null;
		
		String pref = LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':';
		SVGElement root = new SVGGElement(doc);
		Grid g = (Grid)getShape();
		
		root.setAttribute(SVGAttributes.SVG_ID, getSVGID());
		root.setAttribute(pref+LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_GRID);
		root.setAttribute(pref+LaTeXDrawNamespace.XML_GRID_X_SOUTH, String.valueOf(g.isXLabelSouth()));
		root.setAttribute(pref+LaTeXDrawNamespace.XML_GRID_Y_WEST, String.valueOf(g.isYLabelWest()));
		root.setAttribute(pref+LaTeXDrawNamespace.XML_GRID_UNIT, String.valueOf(g.getUnit()));
		root.setAttribute(pref+LaTeXDrawNamespace.XML_GRID_END, g.getGridEnd().x + " " + g.getGridEnd().y);//$NON-NLS-1$
		root.setAttribute(pref+LaTeXDrawNamespace.XML_GRID_START, g.getGridStart().x + " " + g.getGridStart().y);//$NON-NLS-1$
		root.setAttribute(pref+LaTeXDrawNamespace.XML_GRID_ORIGIN, g.getOrigin().x + " " + g.getOrigin().y);//$NON-NLS-1$
		createSVGGrid(root, doc);
		
		return root;
	}
}
