/***************************************************************************
*   Copyright (C) 2007-2010 by Thomas Thelliez aka jblud                  *
*   Contact : <admin.kontrol@gmail.com>                                   *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as               *
*   published by the Free Software Foundation; either version 2.0 of      *
*   the License, or (at your option) any later version.                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
*   General Public License for more details.                              *
*                                                                         *
*   You should have received a copy of the GNU General Public             *
*   License along with this program; if not, write to                     *
*   the Free Software Foundation, Inc.,                                   *
*   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA          *
***************************************************************************/

#include "share_files.h"

ShareFiles::ShareFiles(int clientNbr, OsCharacteristic *characteristics, QWidget *parent)
    :nbr(clientNbr), osInfo(characteristics), QFrame(parent)
{
#ifdef WIN32
    slash = QString("\\");
#else
    slash = QString("/");
#endif
    this->setAttribute(
            Qt::WA_TranslucentBackground, false);
    remoteFilesIconList = new QList<QStandardItem *>();
    QGroupBox *remoteGroupBox = new QGroupBox(tr("Remote available files"), this);
    QGroupBox *localGroupBox = new QGroupBox(tr("Local available files"), this);
    QGridLayout *remoteGridBox = new QGridLayout;
    QGridLayout *localGridBox = new QGridLayout;
    QLabel *serverHowTo = new QLabel(tr("Drag and Drop to transfer files from local to remote computer."));
    remotePath = new QLabel("<b>" + osInfo->home_path, this);
    remoteStringPath = osInfo->home_path;
    QPushButton *refreshRemoteFoldersButton = new QPushButton(tr(""), this);
    refreshRemoteFoldersButton->setIcon(QIcon(":/images/refresh.png"));
    refreshRemoteFoldersButton->setToolTip(tr("Refresh remote files and folders."));

    QDir filesDir(QDir::toNativeSeparators(QDir::homePath()));
    localPath = new QLabel("<b>" + QDir::toNativeSeparators(QDir::homePath()) );
    localStringPath = QDir::toNativeSeparators(QDir::homePath());

    localFilesList = filesDir.entryList(QDir::Files, QDir::Name);
    localFoldersList = filesDir.entryList(QDir::Dirs, QDir::Name);
    remoteFilesList = characteristics->file_list;
    remoteFoldersList = characteristics->folder_list;
    remoteList = new DropListView;
    remoteList->setSelectionMode(QAbstractItemView::SingleSelection);
    remoteList->setDragEnabled(true);
    remoteList->setAcceptDrops(true);
    remoteList->setDropIndicatorShown(true);
    remoteList->setDragDropMode(QAbstractItemView::DragDrop);

    localList = new DropListView;
    localList->setSelectionMode(QAbstractItemView::SingleSelection);
    localList->setDragEnabled(true);
    localList->setAcceptDrops(true);
    localList->setDropIndicatorShown(true);
    localList->setDragDropMode(QAbstractItemView::DragDrop);

    remoteGridBox->addWidget(refreshRemoteFoldersButton, 0, 0);
    remoteGridBox->addWidget(remotePath, 1, 0);
    remoteGridBox->addWidget(remoteList, 2, 0);

    localGridBox->addWidget(serverHowTo, 0, 0);
    localGridBox->addWidget(localPath, 1, 0);
    localGridBox->addWidget(localList, 2, 0);
    localGroupBox->setLayout(localGridBox);
    remoteGroupBox->setLayout(remoteGridBox);

    QSplitter *splitter = new QSplitter(this);
    splitter->addWidget(localGroupBox);
    splitter->addWidget(remoteGroupBox);

    QGridLayout *layout = new QGridLayout;
    layout->addWidget(splitter);
    setLayout(layout);

    // Init views.
    if (localFoldersList.contains("."))
        localFoldersList.removeOne(".");
    this->displayRemoteDirectoryContent(remoteFoldersList, remoteFilesList);
    this->displayLocalDirectoryContent(localFoldersList, localFilesList);

    QObject::connect(refreshRemoteFoldersButton, SIGNAL(clicked()), this, SLOT(refresh()));
    QObject::connect(localList, SIGNAL(doubleClicked(QModelIndex)), this, SLOT(doubleClicLocalFolder()));
    QObject::connect(remoteList, SIGNAL(doubleClicked(QModelIndex)), this, SLOT(doubleClicRemoteFolder()));
    QObject::connect(remoteList, SIGNAL(drop_signal()), this, SLOT(drop_remote()));
    QObject::connect(localList, SIGNAL(drop_signal()), this, SLOT(drop_local()));
}

void ShareFiles::drop_remote()
{
    int len = remoteFoldersList->count() + remoteFilesList->count() + 1;
    QStandardItemModel *listModel = (QStandardItemModel*)remoteList->model();
    QStringList *viewList = convert_QStandardItemModel_to_QStringList(listModel, len);
    QStringList *previousRemoteList = new QStringList;
    previousRemoteList->append(*remoteFoldersList);
    previousRemoteList->append(*remoteFilesList);

    QString document = track_delta(previousRemoteList, viewList);
    delete previousRemoteList;
    delete viewList;
    if (remoteFoldersList->contains(document, Qt::CaseSensitive)) {
        this->displayRemoteDirectoryContent(remoteFoldersList, remoteFilesList);
        emit showMessage(tr("Warning"), tr("You cannot move folders."), 4000);
    }
    if (localFoldersList.contains(document, Qt::CaseSensitive)) {
        if (localFilesList.contains(document, Qt::CaseSensitive)) {
            // Ambiguous scheme. File and folder found
            emit showMessage(tr("Warning"), tr("Ambiguous document name. A file and a folder seems to have the exact same name."), 4000);
        } else
            emit showMessage(tr("Warning"), tr("You cannot share folders, only files. Choose a file or double click on folders to change directory."), 4000);
        this->displayRemoteDirectoryContent(remoteFoldersList, remoteFilesList);
    } else if (localFilesList.contains(document), Qt::CaseSensitive) {
        if (remoteFilesList->contains(document, Qt::CaseSensitive)) {
            // Already exists.
            emit showMessage(tr("Warning"), tr("This file already exists. Please delete it with shell commands before to transfer."), 4000);
            this->displayRemoteDirectoryContent(remoteFoldersList, remoteFilesList);
        } else {
            emit send_file_to_server(remoteStringPath + slash, localStringPath, document);
            remoteFilesList->append(document);
        }
    } else {
        this->displayRemoteDirectoryContent(remoteFoldersList, remoteFilesList);
    }
    qDebug(qPrintable("Transfer document : " + document));
}

void ShareFiles::drop_local()
{
    int len = localFoldersList.count() + localFilesList.count() + 1;
    QStandardItemModel *listModel = (QStandardItemModel*)localList->model();
    QStringList *viewList = convert_QStandardItemModel_to_QStringList(listModel, len);
    QStringList *previousLocalList = new QStringList;
    previousLocalList->append(localFoldersList);
    previousLocalList->append(localFilesList);

    QString document = track_delta(previousLocalList, viewList);
    delete previousLocalList;
    delete viewList;
    if (localFoldersList.contains(document, Qt::CaseSensitive)) {
        this->displayLocalDirectoryContent(localFoldersList, localFilesList);
        emit showMessage(tr("Warning"), tr("You cannot move folders."), 4000);
    }

    if (remoteFoldersList->contains(document, Qt::CaseSensitive)) {
        if (remoteFilesList->contains(document, Qt::CaseSensitive)) {
            // Ambiguous scheme. File and folder found
            emit showMessage(tr("Warning"), tr("Ambiguous document name. A file and a folder seems to have the exact same name."), 4000);
        } else
            emit showMessage(tr("Warning"), tr("You cannot share folders, only files. Choose a file or double click on folders to change directory."), 4000);
        this->displayLocalDirectoryContent(localFoldersList, localFilesList);
    } else if (remoteFilesList->contains(document, Qt::CaseSensitive)) {
        if (localFilesList.contains(document, Qt::CaseSensitive)) {
            // Already exists.
            emit showMessage(tr("Warning"), tr("This file already exists. Please delete it with shell commands before to transfer."), 4000);
            this->displayLocalDirectoryContent(localFoldersList, localFilesList);
        } else {
            emit ask_file_to_server(localStringPath + slash, remoteStringPath, document);
            localFilesList.append(document);
        }
    } else {
        this->displayLocalDirectoryContent(localFoldersList, localFilesList);
    }
    qDebug(qPrintable("Transfer document : " + document));
}

QString ShareFiles::track_delta(QStringList *previousList, QStringList *newList)
{
    newList->sort();
    previousList->sort();
    int new_len = newList->size();
    int old_len = previousList->size();
    if (new_len == (old_len + 1)) {
        for (int i = 0; i < new_len; i++) {
            if (!previousList->contains(newList->at(i), Qt::CaseSensitive))
                return newList->at(i);
            else {
                // Same list or same entries. Can't get duplicated entries.
                QStringList list = newList->filter(newList->at(i), Qt::CaseSensitive);
                // Check if contents match exactly newList->at(i) otherwise remove items.
                QStringList *l = new QStringList;
                for (int j = 0; j < list.size(); j++) {
                    if (QString::compare(newList->at(i), list.at(j), Qt::CaseSensitive) == 0) {
                        l->append(list.at(j));
                    }
                }
                if (l->size() == 2)
                    return l->at(1);
            }
        }
    } else {
        return QString("");
    }
}

void ShareFiles::doubleClicLocalFolder()
{
    QItemSelectionModel *selection = localList->selectionModel();
    QModelIndex indexElementSelectionne = selection->currentIndex();
    QStringListModel *listModel = (QStringListModel*)localList->model();
    QVariant elementSelectionne = listModel->data(indexElementSelectionne, Qt::DisplayRole);
    QString selectedDocument = elementSelectionne.toString();
    for (int j = 0; j < localFoldersList.size(); j++) {
        QString folderName = localFoldersList.at(j);
        if (QString::compare(folderName, selectedDocument, Qt::CaseInsensitive) == 0) {
            ChangeDir *changeDir = new ChangeDir(folderName, localStringPath);
            localStringPath = changeDir->cd_Handling(localStringPath);
            QDir filesDir(QDir::toNativeSeparators(localStringPath));
            localPath->setText("<b>" + QDir::toNativeSeparators(localStringPath));
            localFilesList = filesDir.entryList(QDir::Files, QDir::Name);
            localFoldersList = filesDir.entryList(QDir::Dirs, QDir::Name);
            if (localFoldersList.isEmpty() || (!localFoldersList.contains("..") && QString::compare(localStringPath, "/", Qt::CaseInsensitive) != 0))
                localFoldersList.append("..");
            if (localFoldersList.contains("."))
                localFoldersList.removeOne(".");
            if (localFoldersList.contains("..") && QString::compare(localStringPath, "/", Qt::CaseInsensitive) == 0)
                localFoldersList.removeOne("..");
            this->displayLocalDirectoryContent(localFoldersList, localFilesList);
            return;
        }
    }
    for (int j = 0; j < localFilesList.size(); j++) {
        QString fileName = localFilesList.at(j);
        if (QString::compare(fileName, selectedDocument, Qt::CaseInsensitive) == 0) {
#ifdef WIN32
            QFileInfo fileInfo(QDir::toNativeSeparators(localStringPath + "\\" + fileName));
#else
            QFileInfo fileInfo(QDir::toNativeSeparators(localStringPath + "/" + fileName));
#endif
            QDateTime created = fileInfo.created();
            QDateTime lastModified = fileInfo.lastModified();
            bool exec = fileInfo.isExecutable();
            bool read = fileInfo.isReadable();
            QString executable = (exec) ? "true" : "false";
            QString readable = (read) ? "true" : "false";
            QString size = QString::number(fileInfo.size()); // bytes
            QString owner = fileInfo.owner();
            QString dateCreated = created.toString("dd.MM.yyyy");
            QString dateModified = lastModified.toString("dd.MM.yyyy");
            emit showMessage(fileName + tr(" information"), tr("Name : ") + fileName + tr(" \nExecutable : ") + executable + tr(" \nReadable : ") + readable + tr(" \nSize : ") + size + tr(" \nOwner : ") + owner + tr(" \nExtension : ") + fileInfo.completeSuffix() + tr(" \nCreation date : ") + dateCreated + tr(" \nModification date : ") + dateModified, 5000);
            return;
        }
    }
}

void ShareFiles::doubleClicRemoteFolder()
{
    QItemSelectionModel *selection = remoteList->selectionModel();
    QModelIndex indexElementSelectionne = selection->currentIndex();
    QStringListModel *listModel = (QStringListModel*)remoteList->model();
    QVariant elementSelectionne = listModel->data(indexElementSelectionne, Qt::DisplayRole);
    QString selectedDocument = elementSelectionne.toString();
    for (int j = 0; j < remoteFoldersList->size(); j++) {
        QString folderName = remoteFoldersList->at(j);
        if (QString::compare(folderName, selectedDocument, Qt::CaseInsensitive) == 0) {
            QString xml = XmlCreatorClient::create_xml_question(remoteStringPath, folderName, QString(FOLDER));
            emit send_to_server(xml);
            emit update_percent(50, 100);
            return;
        }
    }
    for (int j = 0; j < remoteFilesList->size(); j++) {
        QString fileName = remoteFilesList->at(j);
        if (QString::compare(fileName, selectedDocument, Qt::CaseInsensitive) == 0) {
            QString xml = XmlCreatorClient::create_xml_question(remoteStringPath, fileName, QString(FILE_XML));
            emit send_to_server(xml);
            emit update_percent(50, 100);
            return;
        }
    }
}

void ShareFiles::displayRemoteDirectoryContent(QStringList *folders_list, QStringList *files_list)
{
    if (folders_list->contains("..") && QString::compare(remoteStringPath, QString(CONTENT_DRIVE), Qt::CaseInsensitive) == 0)
        folders_list->removeOne("..");
    remoteFilesIconModel = new QStandardItemModel(this);
    for (int l = 0; l < folders_list->size(); l++) {
        if (osInfo->system.startsWith(QString(LINUX), Qt::CaseSensitive)) {
            QStandardItem *item = new QStandardItem(QIcon(":/images/linux-folder.png") , folders_list->at(l));
            item->setDropEnabled(false);
            item->setEditable (false);
            remoteFilesIconModel->appendRow(item);
        } else if (osInfo->system.startsWith(WINDOWS, Qt::CaseSensitive)) {
            QStandardItem *item = new QStandardItem(QIcon(":/images/windows-folder.png") , folders_list->at(l));
            item->setDropEnabled(false);
            item->setEditable (false);
            remoteFilesIconModel->appendRow(item);
        } else if (osInfo->system.startsWith(MACOSX, Qt::CaseSensitive)) {
            QStandardItem *item = new QStandardItem(QIcon(":/images/mac-folder.png") , folders_list->at(l));
            item->setDropEnabled(false);
            item->setEditable (false);
            remoteFilesIconModel->appendRow(item);
        } else {
            QStandardItem *item = new QStandardItem(QIcon(":/images/linux-folder.png") , folders_list->at(l));
            item->setDropEnabled(false);
            item->setEditable (false);
            remoteFilesIconModel->appendRow(item);
        }
    }
    for (int j = 0; j < files_list->size(); j++) {
        QStandardItem *item = new QStandardItem(QIcon(":/images/file.png"), files_list->at(j));
        item->setDropEnabled(false);
        item->setEditable (false);
        remoteFilesIconModel->appendRow(item);
    }
    remoteList->setModel(remoteFilesIconModel);
    remoteList->setIconSize(QSize(25, 25));
}

void ShareFiles::displayLocalDirectoryContent(QStringList folders_list, QStringList files_list)
{
    QStandardItemModel *fileModel = new QStandardItemModel(this);
    for (int l = 0; l < folders_list.size(); l++) {
        if (QString::compare(folders_list.at(l), QString("."), Qt::CaseInsensitive) != 0) {
#ifdef WIN32
            QStandardItem *item = new QStandardItem(QIcon(":/images/windows-folder.png") , folders_list.at(l));
            item->setEditable (false);
            item->setDropEnabled(false);
            fileModel->appendRow(item);
#elif defined (linux)
            QStandardItem *item = new QStandardItem(QIcon(":/images/linux-folder.png") , folders_list.at(l));
            item->setEditable (false);
            item->setDropEnabled(false);
            fileModel->appendRow(item);
#elif defined __APPLE__
            QStandardItem *item = new QStandardItem(QIcon(":/images/mac-folder.png") , folders_list.at(l));
            item->setEditable (false);
            item->setDropEnabled(false);
            fileModel->appendRow(item);
#else
            QStandardItem *item = new QStandardItem(QIcon(":/images/linux-folder.png") , folders_list.at(l));
            item->setEditable (false);
            item->setDropEnabled(false);
            fileModel->appendRow(item);
#endif
        }
    }
    iconprovider  = new QFileIconProvider();
    for (int k = 0; k < files_list.size(); k++) {
        QStandardItem *item = new QStandardItem(QIcon(":/images/file.png") ,files_list.at(k));
        item->setEditable (false);
        item->setDropEnabled(false);
        fileModel->appendRow(item);
    }
    localList->setModel(fileModel);
    localList->setIconSize(QSize(25, 25));
}

QStringList* ShareFiles::convert_QStandardItemModel_to_QStringList(QStandardItemModel *listModel, int len_2)
{
    QStringList *list = new QStringList;
    for (int i = 0; i < len_2; i++) {
        QStandardItem *item = listModel->item(i, 0);
        list->append(item->text());
    }
    return list;
}

void ShareFiles::refresh_view(Folder *folder)
{
    remoteStringPath = folder->path;
    remotePath->setText("<b>" + folder->path);
    remoteFilesList = folder->file_list;
    remoteFoldersList = folder->folder_list;
    if (remoteFoldersList->isEmpty() || (!remoteFoldersList->contains("..") && QString::compare(folder->path, "/", Qt::CaseInsensitive) != 0))
        remoteFoldersList->append("..");
    this->displayRemoteDirectoryContent(remoteFoldersList, remoteFilesList);
}

void ShareFiles::refresh_local_view()
{
    QDir filesDir(localStringPath);
    localFilesList = filesDir.entryList(QDir::Files, QDir::Name);
    localFoldersList = filesDir.entryList(QDir::Dirs, QDir::Name);
    if (localFoldersList.isEmpty() || (!localFoldersList.contains("..") && QString::compare(localStringPath, "/", Qt::CaseInsensitive) != 0))
        localFoldersList.append("..");
    if (localFoldersList.contains("."))
        localFoldersList.removeOne(".");
    if (localFoldersList.contains("..") && QString::compare(localStringPath, "/", Qt::CaseInsensitive) == 0)
        localFoldersList.removeOne("..");
    this->displayLocalDirectoryContent(localFoldersList, localFilesList);
}

void ShareFiles::remove_from_view(QString file_name, bool is_local)
{
    if (is_local) {
        if (localFilesList.contains(file_name)) {
            localFilesList.removeAt(localFilesList.indexOf(file_name));
            this->displayLocalDirectoryContent(localFoldersList, localFilesList);
        }
    } else {
        refresh();
    }
}

void ShareFiles::refresh()
{
    QString xml = XmlCreatorClient::create_xml_question(remoteStringPath, ".", QString(FOLDER));
    emit send_to_server(xml);
}
