# JerryScript types

## jerry_init_flag_t

Enum that contains the following elements:

 - JERRY_INIT_EMPTY - empty flag set
 - JERRY_INIT_SHOW_OPCODES - dump byte-code to log after parse
 - JERRY_INIT_SHOW_REGEXP_OPCODES - dump regexp byte-code to log after compilation
 - JERRY_INIT_MEM_STATS - dump memory statistics
 - JERRY_INIT_MEM_STATS_SEPARATE - **deprecated**, dump memory statistics and reset peak values after parse
 - JERRY_INIT_DEBUGGER - **deprecated**, an unused placeholder now

*Changed in version 2.0*: JERRY_INIT_MEM_STATS_SEPARATE and JERRY_INIT_DEBUGGER are now deprecated and not used internally.

## jerry_type_t

Enum that contains JerryScript API value types:

 - JERRY_TYPE_NONE - no type information
 - JERRY_TYPE_UNDEFINED - undefined type
 - JERRY_TYPE_NULL - null type
 - JERRY_TYPE_BOOLEAN - boolean type
 - JERRY_TYPE_NUMBER - number type
 - JERRY_TYPE_STRING - string type
 - JERRY_TYPE_OBJECT - object type
 - JERRY_TYPE_FUNCTION - function type
 - JERRY_TYPE_ERROR - error/abort type
 - JERRY_TYPE_SYMBOL - symbol type

*New in version 2.0*.

## jerry_error_t

Possible types of an error:

 - JERRY_ERROR_COMMON - common error
 - JERRY_ERROR_EVAL - eval error
 - JERRY_ERROR_RANGE - range error
 - JERRY_ERROR_REFERENCE - reference error
 - JERRY_ERROR_SYNTAX - syntax error
 - JERRY_ERROR_TYPE - type error
 - JERRY_ERROR_URI - URI error

There is also a special value `JERRY_ERROR_NONE` which is not an error type
this value can only be returned by the [jerry_get_error_type](#jerry_get_error_type).

*Changed in version 2.0*: The `JERRY_ERROR_NONE` was added to be used by the [jerry_get_error_type](#jerry_get_error_type) method.

## jerry_feature_t

Possible compile time enabled feature types:

 - JERRY_FEATURE_CPOINTER_32_BIT - 32 bit compressed pointers
 - JERRY_FEATURE_ERROR_MESSAGES - error messages
 - JERRY_FEATURE_JS_PARSER - js-parser
 - JERRY_FEATURE_MEM_STATS - memory statistics
 - JERRY_FEATURE_PARSER_DUMP - parser byte-code dumps
 - JERRY_FEATURE_REGEXP_DUMP - regexp byte-code dumps
 - JERRY_FEATURE_SNAPSHOT_SAVE - saving snapshot files
 - JERRY_FEATURE_SNAPSHOT_EXEC - executing snapshot files
 - JERRY_FEATURE_DEBUGGER - debugging
 - JERRY_FEATURE_VM_EXEC_STOP - stopping ECMAScript execution
 - JERRY_FEATURE_JSON - JSON support
 - JERRY_FEATURE_PROMISE - promise support
 - JERRY_FEATURE_TYPEDARRAY - Typedarray support
 - JERRY_FEATURE_DATE - Date support
 - JERRY_FEATURE_REGEXP - RegExp support
 - JERRY_FEATURE_LINE_INFO - line info available
 - JERRY_FEATURE_LOGGING - logging
 - JERRY_FEATURE_SYMBOL - symbol support
 - JERRY_FEATURE_DATAVIEW - DataView support
 - JERRY_FEATURE_PROXY - Proxy support
 - JERRY_FEATURE_MAP - Map support
 - JERRY_FEATURE_SET - Set support
 - JERRY_FEATURE_WEAKMAP - WeakMap support
 - JERRY_FEATURE_WEAKSET - WeakSet support

*New in version 2.0*.
*Changed in version 2.3* : Added `JERRY_FEATURE_WEAKMAP`, `JERRY_FEATURE_WEAKSET` values.

## jerry_container_type_t

Container object types:

 - JERRY_CONTAINER_TYPE_INVALID - Invalid container
 - JERRY_CONTAINER_TYPE_MAP - Map type
 - JERRY_CONTAINER_TYPE_SET - Set type
 - JERRY_CONTAINER_TYPE_WEAKMAP - WeakMap type
 - JERRY_CONTAINER_TYPE_WEAKSET - WeakSet type

 *New in version 2.3*.

## jerry_regexp_flags_t

RegExp object optional flags:

  - JERRY_REGEXP_FLAG_GLOBAL - global match; find all matches rather than stopping after the first match
  - JERRY_REGEXP_FLAG_IGNORE_CASE - ignore case
  - JERRY_REGEXP_FLAG_MULTILINE - multiline; treat beginning and end characters (^ and $) as working over
  multiple lines (i.e., match the beginning or end of each line (delimited by \n or \r), not only the
  very beginning or end of the whole input string)

*New in version 2.0*.

## jerry_parse_opts_t

Option bits for [jerry_parse](#jerry_parse) and
[jerry_parse_function](#jerry_parse_function) functions:

 - JERRY_PARSE_NO_OPTS - no options passed
 - JERRY_PARSE_STRICT_MODE - enable strict mode

*New in version 2.0*.

## jerry_gc_mode_t

Set garbage collection operational mode

 - JERRY_GC_PRESSURE_LOW - free unused objects
 - JERRY_GC_PRESSURE_HIGH - free as much memory as possible

The difference between `JERRY_GC_PRESSURE_LOW` and `JERRY_GC_PRESSURE_HIGH`
is that the former keeps memory allocated for performance improvements such
as property hash tables for large objects. The latter frees all possible
memory blocks but the performance may drop after the garbage collection.

*New in version 2.0*.

## jerry_generate_snapshot_opts_t

Flags for [jerry_generate_snapshot](#jerry_generate_snapshot) and
[jerry_generate_function_snapshot](#jerry_generate_function_snapshot) functions:

 - JERRY_SNAPSHOT_SAVE_STATIC - generate static snapshot (see below)
 - JERRY_SNAPSHOT_SAVE_STRICT - strict source code provided

**Generate static snapshots**
Snapshots contain literal pools, and these literal pools contain references
to constant literals (strings, numbers, etc.). When a snapshot is executed,
these literals are converted to jerry values and the literal pool entries
are changed to their corresponding jerry value. To support this conversion,
the literals and literal pools are copied into RAM even if the
`JERRY_SNAPSHOT_EXEC_COPY_DATA` option is passed to
[jerry_exec_snapshot](#jerry_exec_snapshot). This non-negligible memory
consumption can be avoided by using static snapshots. The literals of
these snapshots are limited to magic strings and 28 bit signed integers,
so their constant pools do not need to be loaded into the memory.
Hence these snapshots can be executed from ROM.

***Important note:*** The [jerry_exec_snapshot](#jerry_exec_snapshot)
function rejects static snaphots unless the `JERRY_SNAPSHOT_EXEC_ALLOW_STATIC`
option bit is set. The caller must also ensure that the same magic
strings are set by [jerry_register_magic_strings](#jerry_register_magic_strings)
when the snapshot is generated and executed. Furthermore the
`JERRY_SNAPSHOT_EXEC_COPY_DATA` option is not allowed.

*New in version 2.0*.

## jerry_exec_snapshot_opts_t

Flags for [jerry_exec_snapshot](#jerry_exec_snapshot) and
[jerry_load_function_snapshot](#jerry_load_function_snapshot) functions:

 - JERRY_SNAPSHOT_EXEC_COPY_DATA - copy snapshot data into memory (see below)
 - JERRY_SNAPSHOT_EXEC_ALLOW_STATIC - allow executing static snapshots

**Copy snapshot data into memory**

By default the snapshot buffer is expected to be present in memory until
[jerry_cleanup](#jerry_cleanup) is called. For example `static const` buffers
compiled into the application binary satisfy this requirement.

If the snapshot buffer is freed after [jerry_exec_snapshot](#jerry_exec_snapshot)
is called the `JERRY_SNAPSHOT_EXEC_COPY_DATA` must be passed to copy the necessary
parts of the snapshot buffer into memory.

The `JERRY_SNAPSHOT_EXEC_COPY_DATA` option is not allowed for static snapshots.

*New in version 2.0*.

## jerry_char_t

**Summary**

Jerry's char value

**Prototype**

```c
typedef uint8_t jerry_char_t;
```

## jerry_size_t

**Summary**

Jerry's size

**Prototype**

```c
typedef uint32_t jerry_size_t;
```

## jerry_length_t

**Summary**

Jerry's length

**Prototype**

```c
typedef uint32_t jerry_length_t;
```

## jerry_value_t

**Summary**

JerryScript value can be a boolean, number, null, object, string or undefined. The value has an error flag,
that indicates whether is an error or not. Every type has an error flag not only objects. The error flag should
be cleared before the value is passed as an argument, otherwise it can lead to a type error. The error objects
created by API functions has the error flag set.

Returned and created values by the API functions must be freed with
[jerry_release_value](#jerry_release_value) when they are no longer needed.

**Prototype**

```c
typedef uint32_t jerry_value_t;
```

## jerry_context_data_manager_t

**Summary**

Structure that defines how a context data item will be initialized and deinitialized. JerryScript zeroes out the memory
for the item by default, and if the `init_cb` field is not NULL, it will be called with the pointer to the memory as
an additional custom initializer. The `deinit_cb` (if non-`NULL`) is called during a call to `jerry_cleanup ()` to run
any custom deinitialization *before* the VM has been fully cleaned up. The `finalize_cb` (if non-`NULL`) is also called
during a call to `jerry_cleanup ()` to run any custom deinitialization *after* the VM has been fully cleaned up.
If bytes_needed field is 0, no buffer is allocated for the manager, callback functions are called with NULL pointer.

**Prototype**

```c
typedef struct
{
  /**
   * Callback responsible for initializing a context item, or NULL to zero out the memory. This is called lazily, the
   * first time jerry_get_context_data () is called with this manager.
   *
   * @param [in] data The buffer that JerryScript allocated for the manager. The buffer is zeroed out. The size is
   * determined by the bytes_needed field. The buffer is kept alive until jerry_cleanup () is called.
   */
  void (*init_cb) (void *data);

  /**
   * Callback responsible for deinitializing a context item, or NULL. This is called as part of jerry_cleanup (),
   * right *before* the VM has been cleaned up. This is a good place to release strong references to jerry_value_t's
   * that the manager may be holding.
   * Note: because the VM has not been fully cleaned up yet, jerry_object_native_info_t free_cb's can still get called
   * *after* all deinit_cb's have been run. See finalize_cb for a callback that is guaranteed to run *after* all
   * free_cb's have been run.
   *
   * @param [in] data The buffer that JerryScript allocated for the manager.
   */
  void (*deinit_cb) (void *data);

  /**
   * Callback responsible for finalizing a context item, or NULL. This is called as part of jerry_cleanup (),
   * right *after* the VM has been cleaned up and destroyed and jerry_... APIs cannot be called any more. At this point,
   * all values in the VM have been cleaned up. This is a good place to clean up native state that can only be cleaned
   * up at the very end when there are no more VM values around that may need to access that state.
   *
   * @param [in] data The buffer that JerryScript allocated for the manager. After returning from this callback,
   * the data pointer may no longer be used.
   */
  void (*finalize_cb) (void *data);

  /**
   * Number of bytes to allocate for this manager. This is the size of the buffer that JerryScript will allocate on
   * behalf of the manager. The pointer to this buffer is passed into init_cb, deinit_cb and finalize_cb. It is also
   * returned from the jerry_get_context_data () API.
   */
  size_t bytes_needed;
} jerry_context_data_manager_t;
```

*New in version 2.0*.

## jerry_context_alloc_t

**Summary**

Function type for allocating buffer for JerryScript context.

**Prototype**

```c
typedef void *(*jerry_context_alloc_t) (size_t size, void *cb_data_p);
```

- `size` - allocation size
- `cb_data_p` - pointer to user data

*New in version 2.0*.

## jerry_context_t

**Summary**

An opaque declaration of the JerryScript context structure.

**Prototype**

```c
typedef struct jerry_context_t jerry_context_t;
```

*New in version 2.0*.


## jerry_binary_operation_t

Enum that contains the supported binary operation types
 - JERRY_BIN_OP_EQUAL - equal comparison (==)
 - JERRY_BIN_OP_STRICT_EQUAL - strict equal comparison (===)
 - JERRY_BIN_OP_LESS - less relation (<)
 - JERRY_BIN_OP_LESS_EQUAL - less or equal relation (<=)
 - JERRY_BIN_OP_GREATER - greater relation (>)
 - JERRY_BIN_OP_GREATER_EQUAL - greater or equal relation (>=)
 - JERRY_BIN_OP_INSTANCEOF - instanceof operation
 - JERRY_BIN_OP_ADD - addition operator (+)
 - JERRY_BIN_OP_SUB - subtraction operator (-)
 - JERRY_BIN_OP_MUL - multiplication operator (*)
 - JERRY_BIN_OP_DIV - division operator (/)
 - JERRY_BIN_OP_REM - remainder operator (%)

*New in version 2.0*.

**See also**

- [jerry_binary_operation](#jerry_binary_operation)

## jerry_property_descriptor_t

**Summary**

Description of ECMA property descriptor. This struct can be used
for the [jerry_define_own_property](#jerry_define_own_property) method to
configure how the property should be registered.

The naming scheme is similar to the JavaScript `Object.defineProperty` method.

Fields should be used in pairs. That is if the `is_value_defined` is set to `true`
the `value` field should contain the value for the property.

**Prototype**

```c
typedef struct
{
  /** Is [[Value]] defined? */
  bool is_value_defined;

  /** Is [[Get]] defined? */
  bool is_get_defined;

  /** Is [[Set]] defined? */
  bool is_set_defined;

  /** Is [[Writable]] defined? */
  bool is_writable_defined;

  /** [[Writable]] */
  bool is_writable;

  /** Is [[Enumerable]] defined? */
  bool is_enumerable_defined;

  /** [[Enumerable]] */
  bool is_enumerable;

  /** Is [[Configurable]] defined? */
  bool is_configurable_defined;

  /** [[Configurable]] */
  bool is_configurable;

  /** [[Value]] */
  jerry_value_t value;

  /** [[Get]] */
  jerry_value_t getter;

  /** [[Set]] */
  jerry_value_t setter;
} jerry_property_descriptor_t;
```

**See also**

- [jerry_define_own_property](#jerry_define_own_property)

## jerry_heap_stats_t

**Summary**

Description of JerryScript heap memory stats.
It is for memory profiling.

**Prototype**

```c
typedef struct
{
  size_t version /**< the version of the stats struct */
  size_t size; /**< heap total size */
  size_t allocated_bytes; /**< currently allocated bytes */
  size_t peak_allocated_bytes; /**< peak allocated bytes */
  size_t reserved[4]; /**< padding for future extensions */
} jerry_heap_stats_t;
```

*New in version 2.0*.

**See also**

- [jerry_get_memory_stats](#jerry_get_memory_stats)

## jerry_external_handler_t

**Summary**

Type of an external function handler

**Prototype**

```c
typedef jerry_value_t (*jerry_external_handler_t) (const jerry_value_t function_obj,
                                                   const jerry_value_t this_val,
                                                   const jerry_value_t args_p[],
                                                   const jerry_length_t args_count);
```

- `function_object` - the JavaScript function object which was invoked.
- `this_val` - the `this` value provided for the function call.
- `args_p` - the function arguments, array of JavaScript values.
- `args_count` - the number of arguments.
- return value
  - The function's return value. If there is no return value, use [jerry_create_undefined()](#jerry_create_undefined).

**See also**

- [jerry_create_external_function](#jerry_create_external_function)

## jerry_object_native_free_callback_t

**Summary**

Native free callback of an object. It is used in `jerry_object_native_info_t` and for external Array buffers.

*Note*:
 - Referred values by this method must have at least 1 reference. (Correct API usage satisfies this condition)

**Prototype**

```c
typedef void (*jerry_object_native_free_callback_t) (void *native_p);
```

*New in version 2.0*: Renamed from `jerry_object_free_callback_t`.
*Changed in version 2.2*: API calls are once again allowed. (See note)

**See also**

- [jerry_object_native_info_t](#jerry_object_native_info_t)
- [jerry_create_arraybuffer_external](#jerry_create_arraybuffer_external)

## jerry_object_native_info_t

**Summary**

The type information of the native pointer.
It includes the free callback that will be called when associated JavaScript object is garbage collected. It can be left NULL in case it is not needed.

Typically, one would create a `static const jerry_object_native_info_t` for
each distinct C type for which a pointer is used with
`jerry_set_object_native_pointer ()` and `jerry_get_object_native_pointer ()`.
This way, each `const jerry_object_native_info_t *` pointer address value itself
uniquely identifies the C type of the native pointer.

See [jerry_get_object_native_pointer](#jerry_get_object_native_pointer)
for a best-practice code example.

**Prototype**

```c
typedef struct
{
  jerry_object_native_free_callback_t free_cb;
} jerry_object_native_info_t;
```

*New in version 2.0*.

**See also**

- [jerry_set_object_native_pointer](#jerry_set_object_native_pointer)
- [jerry_get_object_native_pointer](#jerry_get_object_native_pointer)

## jerry_object_property_foreach_t

**Summary**

Function type used as a callback for the [jerry_foreach_object_property](#jerry_foreach_object_property)
method. A function with this type must return "true" to continue the iteration or "false" to finish the
iteration on the object's properties.

**Prototype**

```c
typedef bool (*jerry_object_property_foreach_t) (const jerry_value_t property_name,
                                                 const jerry_value_t property_value,
                                                 void *user_data_p);
```

- `property_name` - a property name, this is not always a string.
- `property_value` - the value for the given property.
- `user_data_p` - optional user data pointer supplied via the (jerry_foreach_object_property)[#jerry_foreach_object_property] method.
- return value
  - true, to continue the iteration
  - false, to stop the iteration

**See also**

- [jerry_foreach_object_property](#jerry_foreach_object_property)

## jerry_objects_foreach_t

**Summary**

Function type used as a callback for the (jerry_objects_foreach)[#jerry_objects_foreach] method.
A function with this type must return "true" to continue the iteration or "false" to finish the
iteration on the object's properties.

**Prototype**

```c
typedef bool (*jerry_objects_foreach_t) (const jerry_value_t object,
                                         void *user_data_p);
```

- `object` - the current JavaScript object in the for-each iteration.
- `user_data_p` - optional user data pointer supplied via the (jerry_objects_foreach)[#jerry_objects_foreach] method.
- return value
  - true, to continue the iteration
  - false, to stop the iteration

*New in version 2.0*.

**See also**

- [jerry_objects_foreach](#jerry_objects_foreach)

## jerry_objects_foreach_by_native_info_t

**Summary**

Function type used as a callback for the (jerry_objects_foreach_by_native_info)[#jerry_objects_foreach_by_native_info]
method. A function with this type must return "true" to continue the iteration or "false" to finish the
iteration on the object's properties.

**Prototype**

```c
typedef bool (*jerry_objects_foreach_by_native_info_t) (const jerry_value_t object,
                                                        void *object_data_p,
                                                        void *user_data_p);
```

- `object` - the current JavaScript object in the for-each iteration.
- `object_data_p` - the current object's native data pointer.
- `user_data_p` - optional user data pointer supplied via the (jerry_objects_foreach_by_native_info)[#jerry_objects_foreach_by_native_info] method.
- return value
  - true, to continue the iteration
  - false, to stop the iteration

*New in version 2.0*.

**See also**

- [jerry_objects_foreach_by_native_info](#jerry_objects_foreach_by_native_info)

## jerry_vm_exec_stop_callback_t

**Summary**

Callback which tells whether the ECMAScript execution should be stopped.
If it returns with undefined value the ECMAScript execution continues.
Otherwise the result is thrown by the engine (if the error flag is not
set for the returned value the engine automatically sets it). The
callback function might be called again even if it threw an error.
In this case the function must throw the same error again.

**Prototype**

```c
typedef jerry_value_t (*jerry_vm_exec_stop_callback_t) (void *user_p);
```

*New in version 2.0*.

**See also**

- [jerry_set_vm_exec_stop_callback](#jerry_set_vm_exec_stop_callback)

## jerry_promise_state_t

Enum which describes the state of a Promise.

Possible values:

 - JERRY_PROMISE_STATE_NONE - Invalid/Unknown state (possibly called on a non-promise object).
 - JERRY_PROMISE_STATE_PENDING - Promise is in "Pending" state.
 - JERRY_PROMISE_STATE_FULFILLED - Promise is in "Fulfilled" state.
 - JERRY_PROMISE_STATE_REJECTED - Promise is in "Rejected" state.

*New in version 2.2*.

**See also**

- [jerry_get_promise_result](#jerry_get_promise_result)

## jerry_typedarray_type_t

Enum which describes the TypedArray types.
Possible values:

 - JERRY_TYPEDARRAY_UINT8 - represents the Uint8Array TypedArray
 - JERRY_TYPEDARRAY_UINT8CLAMPED - represents the Uint8ClampedArray TypedArray
 - JERRY_TYPEDARRAY_INT8 - represents the Int8Array TypedArray
 - JERRY_TYPEDARRAY_UINT16 - represents the Uint16Array TypedArray
 - JERRY_TYPEDARRAY_INT16 - represents the Int16Array TypedArray
 - JERRY_TYPEDARRAY_UINT32 - represents the Uint32Array TypedArray
 - JERRY_TYPEDARRAY_INT32 - represents the Int32Array TypedArray
 - JERRY_TYPEDARRAY_FLOAT32 - represents the Float32Array TypedArray
 - JERRY_TYPEDARRAY_FLOAT64 - represents the Float64Array TypedArray
 - JERRY_TYPEDARRAY_INVALID - represents an invalid TypedArray

API functions can return the `JERRY_TYPEDARRAY_INVALID` value if the
TypedArray support is not in the engine.

*New in version 2.0*.

**See also**

- [jerry_get_typedarray_type](#jerry_get_typedarray_type)


# General engine functions

## jerry_init

**Summary**

Initializes the JerryScript engine, making it possible to run JavaScript code and perform operations
on JavaScript values. This is required for almost all API functions.

**Prototype**

```c
void
jerry_init (jerry_init_flag_t flags)
```

`flags` - combination of various engine configuration flags [jerry_init_flag_t](#jerry_init_flag_t).

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_SHOW_OPCODES | JERRY_INIT_SHOW_REGEXP_OPCODES);

  // ...

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_init_flag_t](#jerry_init_flag_t)
- [jerry_cleanup](#jerry_cleanup)


## jerry_cleanup

**Summary**

Finish JavaScript engine execution, freeing memory and JavaScript values.

*Note*: JavaScript values, received from engine, will be inaccessible after the cleanup.

**Prototype**

```c
void
jerry_cleanup (void);
```

**See also**

- [jerry_init](#jerry_init)


## jerry_get_context_data

**Summary**

Retrieve a pointer to the item stored within the current context by the given manager.

*Note*: Since internally the pointer to a manager's context data item is linked to the next such pointer in a linked
        list, it is inadvisable to invoke too many different managers, because doing so will increase the time it takes
        to retrieve a manager's context data item, degrading performance. For example, try to keep the number of
        managers below five.

**Prototype**

```c
void *
jerry_get_context_data (const jerry_context_data_manager *manager_p);
```

- `manager_p`: the manager of this context data item.
- return value: the item created by `manager_p` when `jerry_get_context_data ()` was first called, or a new item created
  by `manager_p`, which will be stored for future identical calls to `jerry_get_context_data ()`, and which will be
  deinitialized using the `deinit_cb` callback provided by `manager_p` when the context will be destroyed.

*New in version 2.0*.

**Example**

[doctest]: # (test="compile")

```c
#include "jerryscript.h"

typedef struct
{
  int my_data1;
  double my_data2;
  char *my_data3;
} my_context_data_t;

/* Define how context items will be initialized. */
static void
my_context_data_new (void *user_data_p)
{
  my_context_data_t *my_data_p = (my_context_data_t *) user_data_p;

  /*
   * Initialize my_data_p. JerryScript will store it on the current context and return it whenever
   * jerry_get_context_data () is called with a pointer to my_manager as defined below.
   */
}

/* Define how context items will be deinitialized */
static void
my_context_data_free (void *user_data_p)
{
  my_context_data_t *my_data_p = ((my_context_data_t *) user_data_p);

  /* Perform any necessary cleanup on my_data. JerryScript will free the pointer after this function completes. */
}

/* Wrap the creation and destruction functions into a manager */
static const jerry_context_data_manager_t my_manager =
{
  .init_cb = my_context_data_new,
  .deinit_cb = my_context_data_free,
  .bytes_needed = sizeof (my_context_data_t)
};

/*
 * Then, in some function in your code, you can retrieve an item of type my_context_data_t from the currently active
 * context such that JerryScript will create and store such an item if one was not previously created
 */
static void
someplace_in_the_code (void)
{
  my_context_data_t *my_data = (my_context_data_t *) jerry_get_context_data (&my_manager);
  /* Perform useful things using the data found in my_data */
}
```


## jerry_register_magic_strings

**Summary**

Registers an external magic string array.

*Notes*:
  - The strings in the array must be sorted by size at first, then lexicographically.
  - The maximum number of external magic strings is limited to 2147483648 (UINT32_MAX / 2).
    If there are more than 2147483648 external magic strings the extra is cropped.

**Prototype**

```c
void
jerry_register_magic_strings  (const jerry_char_t * const *ex_str_items_p,
                               uint32_t count,
                               const jerry_length_t *str_lengths_p);
```

- `ex_str_items_p` - character arrays, representing external magic strings' contents
- `count` - number of elements in `ext_str_items_p` array
- `str_lengths_p` - array of lengths for each magic string

*Changed in version 2.0*: The first function argument type was changed.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  // must be static, because 'jerry_register_magic_strings' does not copy
  // the items must be sorted by size at first, then lexicographically
  static const jerry_char_t * const magic_string_items[] = {
                                                             (const jerry_char_t *) "magicstring1",
                                                             (const jerry_char_t *) "magicstring2",
                                                             (const jerry_char_t *) "magicstring3"
                                                           };
  uint32_t num_magic_string_items = (uint32_t) (sizeof (magic_string_items) / sizeof (jerry_char_t *));

  // must be static, because 'jerry_register_magic_strings' does not copy
  static const jerry_length_t magic_string_lengths[] = {
                                                         12,
                                                         12,
                                                         12
                                                       };
  jerry_register_magic_strings (magic_string_items, num_magic_string_items, magic_string_lengths);
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_get_literals_from_snapshot](#jerry_get_literals_from_snapshot)


## jerry_get_memory_stats

**Summary**

Get heap memory stats.

**Notes**:
- The engine must be initialized with the `JERRY_INIT_MEM_STATS` option to allow
  heap statistic collections. See [jerry_init](#jerry_init)
- This API depends on a build option (`JERRY_MEM_STATS`) and can be checked
  in runtime with the `JERRY_FEATURE_MEM_STATS` feature enum value,
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).


**Prototype**

```c
bool
jerry_get_memory_stats (jerry_heap_stats_t *out_stats_p);
```

- `out_stats_p` - out parameter, that provides the heap statistics.
- return value
  - true, if stats were written into the `out_stats_p` pointer.
  - false, otherwise. Usually it is because the `JERRY_FEATURE_MEM_STATS` feature is not enabled.

*New in version 2.0*.

**Example**

```c
jerry_init (JERRY_INIT_MEM_STATS);
// ...

jerry_heap_stats_t stats = {0};
bool get_stats_ret = jerry_get_memory_stats (&stats);
```

**See also**

- [jerry_init](#jerry_init)


## jerry_gc

**Summary**

Performs garbage collection.

**Prototype**

```c
void
jerry_gc (jerry_gc_mode_t mode);
```

- `mode` - operational mode, see [jerry_gc_mode_t](#jerry_gc_mode_t)

*Changed in version 2.0*: Added `mode` argument.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t object_value = jerry_create_object ();
  jerry_release_value (object_value);

  jerry_gc (JERRY_GC_PRESSURE_LOW);

  jerry_cleanup ();
}
```

**See also**

- [jerry_gc_mode_t](#jerry_gc_mode_t)
- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)

# Parser and executor functions

Functions to parse and run JavaScript source code.

## jerry_run_simple

**Summary**

The simplest way to run JavaScript.

**Prototype**

```c
bool
jerry_run_simple (const jerry_char_t *script_source_p,
                  size_t script_source_size,
                  jerry_init_flag_t flags);
```

- `script_source_p` - source code, it must be a valid utf8 string.
- `script_source_size` - size of source code buffer, in bytes.
- `jerry_init_flag_t` - combination of various engine configuration flags
- return value
  - true, if run was successful
  - false, otherwise

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  const jerry_char_t script[] = "print ('Hello, World!');";

  jerry_run_simple (script, sizeof (script) - 1, JERRY_INIT_EMPTY);
  return 0;
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_parse](#jerry_parse)
- [jerry_run](#jerry_run)


## jerry_parse

**Summary**

Parse script and construct an EcmaScript function. The lexical environment is
set to the global lexical environment. The resource name can be used by
debugging systems to provide line / backtrace info.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_parse (const jerry_char_t *resource_name_p, /**< resource name (usually a file name) */
             size_t resource_name_length, /**< length of resource name */
             const jerry_char_t *source_p,
             size_t source_size,
             uint32_t parse_opts);
```

- `resource_name_p` - resource name, usually a file name (must be a valid UTF8 string).
- `resource_name_length` - size of the resource name, in bytes.
- `source_p` - string, containing source code to parse (must be a valid UTF8 string).
- `source_size` - size of the string, in bytes.
- `parse_opts` - any combination of [jerry_parse_opts_t](#jerry_parse_opts_t) flags.
- return value
  - function object value, if script was parsed successfully,
  - thrown error, otherwise

*Changed in version 2.0*: Added `resource_name_p`, and `resource_name_length` arguments.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "print ('Hello, World!');";

  jerry_value_t parsed_code = jerry_parse (NULL, 0, script, sizeof (script) - 1, JERRY_PARSE_NO_OPTS);
  jerry_release_value (parsed_code);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_run](#jerry_run)
- [jerry_parse_function](#jerry_parse_function)

## jerry_parse_function

**Summary**

Parse function source code and construct an ECMAScript
function. The function arguments and function body are
passed as separated arguments. The lexical environment
is set to the global lexical environment. The resource
name (usually a file name) is also passed to this function
which is used by the debugger to find the source code.

*Note*: The returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_parse_function (const jerry_char_t *resource_name_p, /**< resource name (usually a file name) */
                      size_t resource_name_length, /**< length of resource name */
                      const jerry_char_t *arg_list_p, /**< script source */
                      size_t arg_list_size, /**< script source size */
                      const jerry_char_t *source_p, /**< script source */
                      size_t source_size, /**< script source size */
                      uint32_t parse_opts) /**< strict mode */
```

- `resource_name_p` - resource name, usually a file name (must be a valid UTF8 string).
- `resource_name_length` - size of the resource name, in bytes.
- `arg_list_p` - argument list of the function (must be a valid UTF8 string).
- `arg_list_size` - size of the argument list, in bytes.
- `source_p` - string, containing source code to parse (must be a valid UTF8 string).
- `source_size` - size of the string, in bytes.
- `parse_opts` - any combination of [jerry_parse_opts_t](#jerry_parse_opts_t) flags.
- return value
  - function object value, if script was parsed successfully,
  - thrown error, otherwise

*New in version 2.0*.

**Example**

[doctest]: # (name="02.API-REFERENCE-parse-func.c")

```c
#include <stdio.h>
#include <string.h>
#include "jerryscript.h"

int
main (void)
{
  int return_value = 1;

  /* Initialize engine */
  jerry_init (JERRY_INIT_EMPTY);

  /* Parse the 'function (a,b) { return a + b; }' function */
  const char function_args[] = "a, b";
  const char function_source[] = "return a + b";

  jerry_value_t parsed_function = jerry_parse_function (NULL,
                                                        0,
                                                        (const jerry_char_t *) function_args,
                                                        strlen (function_args),
                                                        (const jerry_char_t *) function_source,
                                                        strlen (function_source),
                                                        JERRY_PARSE_NO_OPTS);

  if (!jerry_value_is_error (parsed_function))
  {
    /* Run the parsed function */
    jerry_value_t args[] = {
        jerry_create_number (3),
        jerry_create_number (55),
    };
    jerry_size_t argc = sizeof (args) / sizeof (args[0]);
    jerry_value_t ret_value = jerry_call_function (parsed_function,
                                                   jerry_create_undefined(),
                                                   args,
                                                   argc);

    /* Process result value */
    if (jerry_value_is_number (ret_value)) {
        double value = jerry_get_number_value (ret_value);
        printf ("Function result: %lf\n", value);

        return_value = !(value == (3 + 55));
    }

    /* Release the function arguments */
    for (jerry_size_t idx = 0; idx < argc; idx++) {
        jerry_release_value (args[idx]);
    }

    /* Returned value must be freed */
    jerry_release_value (ret_value);
  }

  /* Parsed function must be freed */
  jerry_release_value (parsed_function);

  /* Cleanup engine */
  jerry_cleanup ();

  return return_value;
}
```

**See also**

- [jerry_call_function](#jerry_call_function)


## jerry_run

**Summary**

Run an EcmaScript function created by `jerry_parse`.

*Notes*:
  - The code should be previously parsed with `jerry_parse`.
  - Returned value must be freed with [jerry_release_value](#jerry_release_value)
    when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_run (const jerry_value_t func_val);
```

- `func_val` - function to run
- return value
  - result of bytecode, if run was successful
  - thrown error, otherwise

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  const jerry_char_t script[] = "print ('Hello, World!');";

  /* Initialize engine */
  jerry_init (JERRY_INIT_EMPTY);

  /* Setup Global scope code */
  jerry_value_t parsed_code = jerry_parse (NULL, 0, script, sizeof (script) - 1, JERRY_PARSE_NO_OPTS);

  if (!jerry_value_is_error (parsed_code))
  {
    /* Execute the parsed source code in the Global scope */
    jerry_value_t ret_value = jerry_run (parsed_code);

    /* Returned value must be freed */
    jerry_release_value (ret_value);
  }

  /* Parsed source code must be freed */
  jerry_release_value (parsed_code);

  /* Cleanup engine */
  jerry_cleanup ();
}
```

**See also**

- [jerry_parse](#jerry_parse)


## jerry_eval

**Summary**

Perform JavaScript `eval` function call (ECMA-262 v5.1 sec-15.1.2.1).

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_eval (const jerry_char_t *source_p,
            size_t source_size,
            uint32_t parse_opts);
```

- `source_p` - source code to evaluate, it must be a valid utf8 string.
- `source_size` - length of the source code
- `parse_opts` - any combination of [jerry_parse_opts_t](#jerry_parse_opts_t) flags.
- return value - result of eval, may be an error value.

**Example**

```c
{
  jerry_value_t ret_val = jerry_eval (str_to_eval,
                                      strlen (str_to_eval),
                                      JERRY_PARSE_NO_OPTS);
}
```

**See also**

- [jerry_create_external_function](#jerry_create_external_function)
- [jerry_external_handler_t](#jerry_external_handler_t)

## jerry_run_all_enqueued_jobs

**Summary**

Run enqueued Promise jobs until the first thrown error or until all get executed.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_run_all_enqueued_jobs (void)
```

- return value - result of last executed job, may be error value.

*New in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "new Promise(function(f,r) { f('Hello, World!'); }).then(function(x) { print(x); });";

  jerry_value_t parsed_code = jerry_parse (NULL, 0, script, sizeof (script) - 1, JERRY_PARSE_NO_OPTS);
  jerry_value_t script_value = jerry_run (parsed_code);
  jerry_value_t job_value = jerry_run_all_enqueued_jobs ();

  jerry_release_value (job_value);
  jerry_release_value (script_value);
  jerry_release_value (parsed_code);

  jerry_cleanup ();
  return 0;
}
```


# Get the global context

## jerry_get_global_object

**Summary**

Get the Global object.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_get_global_object (void);
```

- return value - api value of global object

**Example**

```c
{
  jerry_value_t glob_obj_val = jerry_get_global_object ();

  ... // Do something with global object, ex: add properties

  jerry_release_value (glob_obj_val);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_define_own_property](#jerry_define_own_property)


# Checker functions

Functions to check the type of an API value ([jerry_value_t](#jerry_value_t)).

## jerry_value_is_abort

**Summary**

Returns whether the given `jerry_value_t` has the error and abort value set.

**Prototype**

```c
bool
jerry_value_is_abort (const jerry_value_t value);
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` has the error and abort value set
  - false, otherwise

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_abort (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)
- [jerry_value_is_error](#jerry_value_is_error)

## jerry_value_is_array

**Summary**

Returns whether the given `jerry_value_t` is an array.

**Prototype**

```c
bool
jerry_value_is_array (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is an array
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_array (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)

## jerry_value_is_arraybuffer

**Summary**

Returns whether the given `jerry_value_t` is an ArrayBuffer object.

*Notes*:
- This API depends on a build option (`JERRY_BUILTIN_TYPEDARRAY`) and can be checked
  in runtime with the `JERRY_FEATURE_TYPEDARRAY` feature enum value,
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).
- The es.next profile enables this by default.

**Prototype**

```c
bool
jerry_value_is_arraybuffer (const jerry_value_t value)
```

- `value` - api value to check.
- return value
  - true, if the given `jerry_value_t` is an ArrayBuffer object.
  - false, otherwise

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_arraybuffer (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_create_arraybuffer](#jerry_create_arraybuffer)
- [jerry_create_arraybuffer_external](#jerry_create_arraybuffer_external)


## jerry_value_is_boolean

**Summary**

Returns whether the given `jerry_value_t` is a boolean value.

**Prototype**

```c
bool
jerry_value_is_boolean (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a boolean value
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_boolean (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_constructor

**Summary**

Returns whether the given `jerry_value_t` is a constructor function.

**Prototype**

```c
bool
jerry_value_is_constructor (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a constructor
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_constructor (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)

## jerry_value_is_dataview

**Summary**

Returns whether the given `jerry_value_t` is a DataView object value.

*Notes*:
- This API depends on a build option (`JERRY_BUILTIN_DATAVIEW`) and can be checked
  in runtime with the `JERRY_FEATURE_DATAVIEW` feature enum value,
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).
- The es.next profile enables this by default.

**Prototype**

```c
bool
jerry_value_is_dataview (const jerry_value_t value)
```

- `value` - API value
- return value
  - true, if the given `jerry_value_t` is a DataView object
  - false, otherwise

*New in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t arraybuffer = jerry_create_arraybuffer (16);
  jerry_value_t dataview = jerry_create_dataview (arraybuffer, 0, 16);

  if (jerry_value_is_dataview (dataview))
  {
    // usage of dataview
  }

  jerry_release_value (dataview);
  jerry_release_value (arraybuffer);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_create_dataview](#jerry_create_dataview)


## jerry_value_is_error

**Summary**

Returns whether the given `jerry_value_t` is error value.

**Prototype**

```c
bool
jerry_value_is_error (const jerry_value_t value);
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is error value.
  - false, otherwise

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_error (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)
- [jerry_value_is_abort](#jerry_value_is_abort)

## jerry_value_is_function

**Summary**

Returns whether the given `jerry_value_t` is a function.

**Prototype**

```c
bool
jerry_value_is_function (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a function
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_function (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_number

**Summary**

Returns whether the given `jerry_value_t` is a number.

**Prototype**

```c
bool
jerry_value_is_number (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a number
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_number (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_null

**Summary**

Returns whether the given `jerry_value_t` is a null value.

**Prototype**

```c
bool
jerry_value_is_null (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a null
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_null (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_object

**Summary**

Returns whether the given `jerry_value_t` is an object value.

**Prototype**

```c
bool
jerry_value_is_object (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is an object
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_object (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_promise

**Summary**

Returns whether the given `jerry_value_t` is a promise value.

*Notes*:
- This API depends on a build option (`JERRY_BUILTIN_PROMISE`) and can be checked
  in runtime with the `JERRY_FEATURE_PROMISE` feature enum value,
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).
- The es.next profile enables this by default.


**Prototype**

```c
bool
jerry_value_is_promise (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a promise
  - false, otherwise

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_promise (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_create_promise](#jerry_create_promise)


## jerry_value_is_proxy

**Summary**

Returns whether the given `jerry_value_t` is a proxy value.

*Notes*:
- This API depends on a build option (`JERRY_BUILTIN_PROXY`) and can be checked
  in runtime with the `JERRY_FEATURE_PROXY` feature enum value,
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).
- The es.next profile enables this by default.


**Prototype**

```c
bool
jerry_value_is_proxy (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a proxy object
  - false, otherwise

**Example**

*New in version 2.3*.

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_proxy (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_create_proxy](#jerry_create_proxy)


## jerry_value_is_string

**Summary**

Returns whether the given `jerry_value_t` is a string value.

**Prototype**

```c
bool
jerry_value_is_string (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a string
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_string (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_symbol

**Summary**

Returns whether the given `jerry_value_t` is a symbol value.

*Notes*:
- This API depends on a build option (`JERRY_BUILTIN_SYMBOL`) and can be checked
  in runtime with the `JERRY_FEATURE_SYMBOL` feature enum value,
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).
- The es.next profile enables this by default.

**Prototype**

```c
bool
jerry_value_is_symbol (const jerry_value_t value)
```

- `value` - API value
- return value
  - true, if the given `jerry_value_t` is a symbol
  - false, otherwise

*New in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t string_value = jerry_create_string ((const jerry_char_t *) "Symbol description string");
  jerry_value_t symbol_value = jerry_create_symbol (string_value);

  jerry_release_value (string_value);

  if (jerry_value_is_symbol (symbol_value))
  {
    // usage of symbol_value
  }

  jerry_release_value (symbol_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_create_symbol](#jerry_create_symbol)

## jerry_value_is_typedarray

**Summary**

Checks whether the given `jerry_value_t` is a TypedArray object or not.

*Notes*:
- This API depends on a build option (`JERRY_BUILTIN_TYPEDARRAY`) and can be checked
  in runtime with the `JERRY_FEATURE_TYPEDARRAY` feature enum value,
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).
- The es.next profile enables this by default.

**Prototype**

```c
bool
jerry_value_is_typedarray (const jerry_value_t value)
```

- `value` - object to check
- return value
  - true, if the given `jerry_value_t` is a TypedArray object.
  - false, otherwise

*New in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t value = jerry_create_typedarray (JERRY_TYPEDARRAY_UINT16, 15);

  if (jerry_value_is_typedarray (value))
  {
    /* "value" is a typedarray. */
  }

  jerry_release_value (value);

  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_create_typedarray](#jerry_create_typedarray)


## jerry_get_container_type

**Summary**

Checks whether the given `jerry_value_t` is the given `jerry_container_type_t` type container object.

*Notes*
- This API function depends on a build option (`JERRY_BUILTIN_CONTAINER`) and can be checked
  runtime with the `JERRY_FEATURE_MAP, JERRY_FEATURE_SET, JERRY_FEATURE_WEAKMAP, JERRY_FEATURE_WEAKSET`
  feature enum values.
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).
- The es.next profile enables this by default.

*New in version 2.3*.

**Prototype**

```c
jerry_container_type_t
jerry_get_container_type (const jerry_value_t value)
```

- `value` - Container object
- return value
  - The corresponding enum value of `jerry_container_type_t`, or `JERRY_CONTAINER_TYPE_INVALID` if the container
    was not a valid container object.
**Example**

[doctest]: # ()

```c
#include "jerryscript.h"
int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t value = jerry_create_container (JERRY_CONTAINER_TYPE_MAP, NULL, 0);

  if (jerry_get_container_type (value) == JERRY_CONTAINER_TYPE_MAP)
  {
    /* "value" is a map. */
  }

  jerry_release_value (value);

  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_create_container](#jerry_create_container)
- [jerry_container_type_t](#jerry_container_type_t)


## jerry_value_is_undefined

**Summary**

Returns whether the given `jerry_value_t` is an undefined value.

**Prototype**

```c
bool
jerry_value_is_undefined (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is an undefined value
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_undefined (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)

## jerry_value_get_type

**Summary**

Returns the JavaScript type
for a given value as a [jerry_type_t](#jerry_type_t) enum value.

This is a similar operation to the 'typeof' operator
in the standard with an exception that the 'null'
value has its own enum value.

**Prototype**

```c
jerry_type_t
jerry_value_get_type (const jerry_value_t value);
```

- `value` - JavaScript value to check.
- return value
  - One of the [jerry_type_t](#jerry_type_t) value.

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t number = jerry_create_number (3.3);

  jerry_type_t type_info = jerry_value_get_type (number);

  if (type_info == JERRY_TYPE_NUMBER)
  {
    /* ... */
  }

  jerry_release_value (number);
}
```

**See also**

- [jerry_type_t](#jerry_type_t)

## jerry_is_feature_enabled

**Summary**

Returns whether the specified compile time feature is enabled.

**Prototype**

```c
bool
jerry_is_feature_enabled (const jerry_feature_t feature);
```

- `feature` - jerry feature
- return value
  - true, if the given `jerry_feature_t` is enabled
  - false, otherwise

*New in version 2.0*.

**Example**

```c
{
  /* ... */
  jerry_feature_t feature = JERRY_FEATURE_SNAPSHOT_SAVE;

  if (jerry_is_feature_enabled (feature))
  {
    /* ... */
  }

}
```

**See also**

- [jerry_feature_t](#jerry_feature_t)


# Binary operations

## jerry_binary_operation

**Summary**

Perform binary operation on the given operands (==, ===, <, >, etc.).

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_binary_operation (jerry_binary_operation_t op,
                        const jerry_value_t lhs,
                        const jerry_value_t rhs);
```

- `op` - binary operation
- `lhs` - left-hand side operand
- `rhs` - right-hand side operand
- return value
  - error, if argument has an error flag or operation is unsuccessful or unsupported
  - true/false, the result of the binary operation on the given operands otherwise

*New in version 2.0*.

**Example - JERRY_BIN_OP_EQUAL**

```c
{
  jerry_value_t value1;
  jerry_value_t value2;
  ... // create or acquire value
  jerry_value_t result = jerry_binary_operation (JERRY_BIN_OP_EQUAL, value1, value2)

  if (!jerry_value_is_error (result))
  {
    if (jerry_get_boolean_value (result))
    {
       // value1 and value2 are equal
    }
    else
    {
      // value1 and value2 are NOT equal
    }
  }
  else
  {
    ... // handle error
  }

  jerry_release_value (value1);
  jerry_release_value (value2);
  jerry_release_value (result);
}
```

**Example - JERRY_BIN_OP_INSTANCEOF**

[doctest]: # ()

```c
#include "jerryscript.h"

static jerry_value_t
my_constructor (const jerry_value_t func_val,
                const jerry_value_t this_val,
                const jerry_value_t argv[],
                const jerry_length_t argc)
{
  return jerry_create_undefined ();
}

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t base_obj = jerry_create_object ();
  jerry_value_t constructor = jerry_create_external_function (my_constructor);

  /* External functions does not have a prototype by default, so we need to create one */
  jerry_value_t prototype_str = jerry_create_string ((const jerry_char_t *) ("prototype"));
  jerry_release_value (jerry_set_property (constructor, prototype_str, base_obj));
  jerry_release_value (prototype_str);

  /* Construct the instance. */
  jerry_value_t instance_val = jerry_construct_object (constructor, NULL, 0);

  /* Call the API function of 'instanceof'. */
  jerry_value_t is_instance = jerry_binary_operation (JERRY_BIN_OP_INSTANCEOF,
                                                      instance_val,
                                                      constructor);
  if (!jerry_value_is_error (is_instance)
      && jerry_get_boolean_value (is_instance) == true)
  {
    /* ... */
  }

  /* Free all of the jerry values and cleanup the engine. */
  jerry_release_value (base_obj);
  jerry_release_value (constructor);
  jerry_release_value (instance_val);
  jerry_release_value (is_instance);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_binary_operation_t](#jerry_binary_operation_t)


# Error manipulation functions

*Changed in version 2.0*: The error handling and manipulation was modified and the old methods were replaced.

## jerry_create_abort_from_value

**Summary**

Create (api) abort from a value.

This function creates an API abort value from an API value. The second argument defines
whether the input value must be released or not. If it is set to `true`,
then a [`jerry_release_value`](#jerry_release_value) function will be called
for the first argument, so the api value won't be available after the call of
`jerry_create_abort_from_value`. The second argument should be false if both value
and created abort value are needed.

**Prototype**

```c
jerry_value_t
jerry_create_abort_from_value (jerry_value_t value, bool release);
```

- `value` - api value
- `release` - raw boolean, defines whether input value must be released
- return value - abort (api) value

*New in version 2.0*.

**Example 1**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t abort = jerry_create_abort_from_value (value, true);
  // using the 'value' variable after release is invalid.

  jerry_release_value (abort);
}
```

**Example 2**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t abort = jerry_create_abort_from_value (value, false);
  // both 'abort' and 'value' can be used and must be released when they are no longer needed

  jerry_release_value (abort);
  jerry_release_value (value);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)
- [jerry_get_value_from_error](#jerry_get_value_from_error)
- [jerry_create_error_from_value](#jerry_create_error_from_value)

## jerry_create_error_from_value

**Summary**

Create (api) error from a value.

This function creates an API error value from an API value. The second argument defines
whether the input value must be released or not. If it is set to `true`,
then a [`jerry_release_value`](#jerry_release_value) function will be called
for the first argument, so the api value won't be available after the call of
`jerry_create_error_from_value`. The second argument should be false if both value
and created error value are needed.

**Prototype**

```c
jerry_value_t
jerry_create_error_from_value (jerry_value_t value, bool release);
```

- `value` - api value
- `release` - raw boolean, defines whether input value must be released
- return value - error (api) value

*New in version 2.0*.

**Example 1**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t error = jerry_create_error_from_value (value, true);
  // using the 'value' variable after release is invalid.


  jerry_release_value (error);
}
```

**Example 2**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t error = jerry_create_error_from_value (value, false);
  // both 'error' and 'value' can be used and must be released when they are no longer needed

  jerry_release_value (error);
  jerry_release_value (value);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)
- [jerry_get_value_from_error](#jerry_get_value_from_error)
- [jerry_create_abort_from_value](#jerry_create_abort_from_value)

## jerry_get_error_type

**Summary**

Returns the type of the Error object if possible.

If a non-error object is used as the input for the function the method
will return `JERRY_ERROR_NONE` indicating that the value was not
an Error object. However it is still possible that the value contains
error semantics. To correctly detect if a value have error use the
[jerry_value_is_error](#jerry_value_is_error) method.

**Prototype**

```c
jerry_error_t
jerry_get_error_type (const jerry_value_t value);
```

- `value` - api value (possible error object)
- return value
  - JERRY_ERROR_NONE if the input is not an error object
  - one of the [jerry_error_t](#jerry_error_t) value

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t error_obj = jerry_create_error (JERRY_ERROR_RANGE,
                                                (const jerry_char_t *) "error msg");
  jerry_error_t error_type = jerry_get_error_type (error_obj);

  // error_type is now JERRY_ERROR_RANGE.

  jerry_release_value (error_obj);
}
```

**See also**

- [jerry_create_error](#jerry_create_error)
- [jerry_value_is_error](#jerry_value_is_error)

## jerry_get_value_from_error

**Summary**

Get the value from an error.

Many API functions cannot be called with an error value.
This function extracts the API value from an error. The second argument defines
whether the input error value must be released or not. If it is set to `true`,
then a [`jerry_release_value`](#jerry_release_value) function will be called
for the first argument, so the error value won't be available after the call of
`jerry_get_value_from_error`. The second argument should be false if both error
and its represented value are needed.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_get_value_from_error (jerry_value_t value, bool release)
```

- `value` - error (api) value
- `release` - raw boolean, defines whether input value must be released
- return value - api value

*New in version 2.0*.

**Example 1**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t error = jerry_create_error_from_value (value, true);
  jerry_value_t value_from_error = jerry_get_value_from_error (error, true);
  // using the 'error' variable after release is invalid.

  jerry_release_value (value_from_error);
}
```

**Example 2**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t error = jerry_create_error_from_value (value, true);
  jerry_value_t value_from_error = jerry_get_value_from_error (error, false);
  // both 'error' and 'value_from_error' can be used and must be released when they are no longer needed

  jerry_release_value (value_from_error);
  jerry_release_value (error);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)
- [jerry_create_error_from_value](#jerry_create_error_from_value)
- [jerry_create_abort_from_value](#jerry_create_abort_from_value)

# Getter functions of 'jerry_value_t'

Get raw data from API values.

## jerry_get_boolean_value

**Summary**

Gets the raw bool value from a `jerry_value_t`.

**Prototype**

```c
bool
jerry_get_boolean_value (const jerry_value_t value);
```

- `value` - api value
- return value - boolean value represented by the argument.

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_boolean (value))
  {
    bool raw_value = jerry_get_boolean_value (value);

    ... // usage of raw value

  }

  jerry_release_value (value);
}

```

**See also**

- [jerry_value_is_boolean](#jerry_value_is_boolean)
- [jerry_release_value](#jerry_release_value)


## jerry_get_number_value

**Summary**

Gets the number value of the given `jerry_value_t` parameter as a raw double.

If the argument passed is not a number `0.0` will be returned.

**Prototype**

```c
double
jerry_get_number_value (const jerry_value_t value);
```

- `value` - api value
- return value
  - the number value of the given `jerry_value_t` parameter as a raw double.
  - `0.0` if the api value passed is not a number.

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_number (value))
  {
    double raw_value = jerry_get_number_value (value);

    ... // usage of raw value

  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_value_is_number](#jerry_value_is_number)
- [jerry_release_value](#jerry_release_value)


# Functions for string values

## jerry_get_string_size

**Summary**

Get the size of a string. Returns zero, if the value parameter is not a string.
This is effectively the number of bytes required to store the string's characters.

**Prototype**

```c
jerry_size_t
jerry_get_string_size (const jerry_value_t value);
```
- `value` - api value
- return value - number of bytes in the buffer needed to represent the string.

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string = jerry_create_string (char_array);

  jerry_size_t string_size = jerry_get_string_size (string);

  ... // usage of string_size

  jerry_release_value (string);
}
```

**See also**

- [jerry_create_string](#jerry_create_string)
- [jerry_get_string_length](#jerry_get_string_length)
- [jerry_is_valid_cesu8_string](#jerry_is_valid_cesu8_string)


## jerry_get_utf8_string_size

**Summary**

Get the size of an utf8-encoded string. Returns zero, if the value parameter is not a string.
This is effectively the number of bytes required to store the utf8 encoded string's characters.

*Note*: The difference from [jerry_get_string_size](#jerry_get_string_size) is that it returns with utf-8 string size
instead of the cesu-8 string size.

**Prototype**

```c
jerry_size_t
jerry_get_utf8_string_size (const jerry_value_t value);
```
- `value` - api value
- return value - number of bytes in the buffer needed to represent the utf8-encoded string.

*New in version 2.0*.

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string = jerry_create_string (char_array);

  jerry_size_t string_size = jerry_get_utf8_string_size (string);

  ... // usage of string_size

  jerry_release_value (string);
}
```

**See also**

- [jerry_create_string_from_utf8](#jerry_create_string_from_utf8)
- [jerry_get_utf8_string_length](#jerry_get_utf8_string_length)
- [jerry_is_valid_utf8_string](#jerry_is_valid_utf8_string)


## jerry_get_string_length

**Summary**

Get the length of a string. Returns zero, if the value parameter is not a string.

*Notes:*
- The difference from [jerry_get_string_size](#jerry_get_string_size) is that it
  returns the number of bytes used for the string.
- This is **not** the number of bytes required to store the string.

**Prototype**

```c
jerry_length_t
jerry_get_string_length (const jerry_value_t value);
```

- `value` - api value
- return value - number of characters in the string

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string = jerry_create_string (char_array);

  jerry_length_t string_length = jerry_get_string_length (string);

  ... // usage of string_length

  jerry_release_value (string);
}
```

**See also**

- [jerry_create_string](#jerry_create_string)
- [jerry_get_string_size](#jerry_get_string_size)
- [jerry_is_valid_cesu8_string](#jerry_is_valid_cesu8_string)


## jerry_get_utf8_string_length

**Summary**

Get the length of an UTF-8 encoded string. Returns zero, if the value parameter is not a string.

*Notes*:
- The difference from [jerry_get_string_length](#jerry_get_string_length) is that it
  returns with utf-8 string length instead of the cesu-8 string length.
- This is **not** the number of bytes required to store the string.

**Prototype**

```c
jerry_length_t
jerry_get_utf8_string_length (const jerry_value_t value);
```

- `value` - input string value
- return value - number of characters in the string

*New in version 2.0*.

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string = jerry_create_string_from_utf8 (char_array);

  jerry_length_t string_length = jerry_get_utf8_string_length (string);

  ... // usage of string_length

  jerry_release_value (string);
}
```

**See also**

- [jerry_create_string_from_utf8](#jerry_create_string_from_utf8)
- [jerry_get_utf8_string_size](#jerry_get_utf8_string_size)
- [jerry_is_valid_utf8_string](#jerry_is_valid_utf8_string)


## jerry_string_to_char_buffer

**Summary**

Copy the characters of a string into a specified cesu-8 buffer.
The '\0' character could occur in the character buffer. Returns 0,
if the value parameter is not a string or the buffer is not large
enough for the whole string.

*Note*: Does not put '\0' to the end of string, the return value identifies
the number of valid bytes in the output buffer.

*Note*: If the size of the string in jerry value is larger than the size of the
target buffer, the copy will fail. To copy a substring the
[jerry_substring_to_char_buffer](#jerry_substring_to_char_buffer) API function
is recommended instead.


**Prototype**

```c
jerry_size_t
jerry_string_to_char_buffer (const jerry_value_t value,
                             jerry_char_t *buffer_p,
                             jerry_size_t buffer_size);
```

- `value` - input string value
- `buffer_p` - pointer to output buffer
- `buffer_size` - size of the buffer
- return value - number of bytes, actually copied to the buffer

**Example**

[doctest]: # ()

```c
#include <stdio.h>
#include <stdlib.h>
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t value;
  // create or acquire value
  value = jerry_create_string ((const jerry_char_t *) "Demo string");

  // Read the string into a byte buffer.
  jerry_size_t string_size = jerry_get_string_size (value);
  jerry_char_t *string_buffer_p = (jerry_char_t *) malloc (sizeof (jerry_char_t) * (string_size + 1));

  jerry_size_t copied_bytes = jerry_string_to_char_buffer (value, string_buffer_p, string_size);
  string_buffer_p[copied_bytes] = '\0';

  jerry_release_value (value);

  jerry_cleanup ();

  printf ("Test string: %s\n", string_buffer_p);
  free (string_buffer_p);

  return 0;
}
```

**See also**

- [jerry_create_string](#jerry_create_string)
- [jerry_get_string_size](#jerry_get_string_size)
- [jerry_is_valid_cesu8_string](#jerry_is_valid_cesu8_string)
- [jerry_substring_to_char_buffer](#jerry_substring_to_char_buffer)


## jerry_string_to_utf8_char_buffer

**Summary**

Copy the characters of a string into a specified utf-8 buffer.
The '\0' character could occur in character buffer. Returns 0,
if the value parameter is not a string or the buffer is not
large enough for the whole string.

*Note*: Does not put '\0' to the end of string, the return value identifies
the number of valid bytes in the output buffer.

*Note*: If the size of the string in jerry value is larger than the size of the
target buffer, the copy will fail. To copy a substring the
[jerry_substring_to_utf8_char_buffer](#jerry_substring_to_utf8_char_buffer)
API function is recommended instead.

**Prototype**

```c
jerry_size_t
jerry_string_to_utf8_char_buffer (const jerry_value_t value,
                                  jerry_char_t *buffer_p,
                                  jerry_size_t buffer_size);
```

- `value` - input string value
- `buffer_p` - pointer to output buffer
- `buffer_size` - size of the buffer
- return value - number of bytes, actually copied to the buffer

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_size_t req_sz = jerry_get_utf8_string_size (value);
  jerry_char_t str_buf_p[req_sz];

  jerry_size_t bytes_copied = jerry_string_to_utf8_char_buffer (value, str_buf_p, req_sz);

  jerry_release_value (value);
}
```

**See also**

- [jerry_create_string_from_utf8](#jerry_create_string_from_utf8)
- [jerry_get_utf8_string_size](#jerry_get_utf8_string_size)
- [jerry_is_valid_utf8_string](#jerry_is_valid_utf8_string)
- [jerry_substring_to_utf8_char_buffer](#jerry_substring_to_utf8_char_buffer)


## jerry_substring_to_char_buffer

**Summary**

Copy the characters of a cesu-8 encoded substring into a specified buffer.
The '\0' character could occur in character buffer. Returns 0, if the value
parameter is not a string. It will extract the substring between the
specified start position and the end position (or the end of the string,
whichever comes first).

*Note*: Does not put '\0' to the end of string, the return value identifies
the number of valid bytes in the output buffer.

**Prototype**

```c
jerry_size_t
jerry_substring_to_char_buffer (const jerry_value_t value,
                                jerry_length_t start_pos,
                                jerry_length_t end_pos,
                                jerry_char_t *buffer_p,
                                jerry_size_t buffer_size);
```

- `value` - input string value
- `start_pos` - position of the first character
- `end_pos` - position of the last character
- `buffer_p` - pointer to output buffer
- `buffer_size` - size of the buffer
- return value - number of bytes, actually copied to the buffer

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_size_t req_sz = jerry_get_string_size (value);
  jerry_char_t str_buf_p[req_sz];
  jerry_length_t start_pos = 0;
  jerry_length_t end_pos = jerry_get_string_length (value);

  jerry_substring_to_char_buffer (value, start_pos, end_pos, str_buf_p, req_sz);

  jerry_release_value (value);
}
```

**See also**

- [jerry_create_string](#jerry_create_string)
- [jerry_get_string_size](#jerry_get_string_size)
- [jerry_get_string_length](#jerry_get_string_length)
- [jerry_is_valid_cesu8_string](#jerry_is_valid_cesu8_string)


## jerry_substring_to_utf8_char_buffer

**Summary**

Copy the characters of an utf-8 encoded substring into a specified buffer.
The '\0' character could occur in character buffer. Returns 0, if the value
parameter is not a string. It will extract the substring between the specified
start position and the end position (or the end of the string, whichever
comes first).

*Note*: Does not put '\0' to the end of string, the return value identifies
the number of valid bytes in the output buffer.

**Prototype**

```c
jerry_size_t
jerry_substring_to_utf8_char_buffer (const jerry_value_t value,
                                     jerry_length_t start_pos,
                                     jerry_length_t end_pos,
                                     jerry_char_t *buffer_p,
                                     jerry_size_t buffer_size);
```

- `value` - input string value
- `start_pos` - position of the first character
- `end_pos` - position of the last character
- `buffer_p` - pointer to output buffer
- `buffer_size` - size of the buffer
- return value - number of bytes, actually copied to the buffer

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_size_t req_sz = jerry_get_utf8_string_size (value);
  jerry_char_t str_buf_p[req_sz];
  jerry_length_t start_pos = 0;
  jerry_length_t end_pos = jerry_get_utf8_string_length (value);

  jerry_substring_to_utf8_char_buffer (value, start_pos, end_pos, str_buf_p, req_sz);

  jerry_release_value (value);
}
```

**See also**

- [jerry_create_string_from_utf8](#jerry_create_string_from_utf8)
- [jerry_get_utf8_string_size](#jerry_get_utf8_string_size)
- [jerry_get_utf8_string_length](#jerry_get_utf8_string_length)
- [jerry_is_valid_utf8_string](#jerry_is_valid_utf8_string)


# Functions for array object values

## jerry_get_array_length

**Summary**

Get length of an array object. Returns zero, if the given parameter is not an array object.

**Prototype**

```c
uint32_t
jerry_get_array_length (const jerry_value_t value);
```

- `value` - input array value
- return value - length of the given array

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  uint32_t len = jerry_get_array_length (value);

  jerry_release_value (value);
}
```

**See also**

- [jerry_create_array](#jerry_create_array)


# Converters of 'jerry_value_t'

Functions for converting API values to another value type.

## jerry_value_to_boolean

**Summary**

Call ToBoolean operation on the api value.

**Prototype**

```c
bool
jerry_value_to_boolean (const jerry_value_t value);
```

- `value` - api value
- return value
  - true, if the logical value is true
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  bool b = jerry_value_to_boolean (value);

  jerry_release_value (value);
}

```

**See also**

- [jerry_value_to_primitive](#jerry_value_to_primitive)

## jerry_value_to_number

**Summary**

Call ToNumber operation on the api value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_value_to_number (const jerry_value_t value);
```

- `value` - api value
- return value
  - converted number value, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t number_value = jerry_value_to_number (value);

  jerry_release_value (number_value);
  jerry_release_value (value);
}

```

**See also**

- [jerry_value_to_primitive](#jerry_value_to_primitive)

## jerry_value_to_object

**Summary**

Call ToObject operation on the api value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_value_to_object (const jerry_value_t value);
```

- `value` - api value
- return value
  - converted object value, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t object_value = jerry_value_to_object (value);

  jerry_release_value (object_value);
  jerry_release_value (value);
}
```

**See also**

- [jerry_value_to_primitive](#jerry_value_to_primitive)

## jerry_value_to_primitive

**Summary**

Call ToPrimitive operation on the api value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_value_to_primitive (const jerry_value_t value);
```

- `value` - api value
- return value
  - converted primitive value, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t prim_value = jerry_value_to_primitive (value);

  jerry_release_value (prim_value);
  jerry_release_value (value);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)

## jerry_value_to_string

**Summary**

Call the ToString ecma builtin operation on the api value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_value_to_string (const jerry_value_t value);
```

- `value` - api value
- return value
  - converted string value, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t string_value = jerry_value_to_string (value);

  jerry_release_value (string_value);
  jerry_release_value (value);
}
```

**See also**

- [jerry_value_to_primitive](#jerry_value_to_primitive)


# Functions for promise objects

These APIs all depend on the es.next profile (or on some build options).

## jerry_get_promise_result

**Summary**

The function returns the result of a Promise object.

*Notes*:
- Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
  is no longer needed.
- This API depends on a build option (`JERRY_BUILTIN_PROMISE`) and can be checked
  in runtime with the `JERRY_FEATURE_PROMISE` feature enum value,
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).
- The es.next profile enables this by default.


**Prototype**

```c
jerry_value_t
jerry_get_promise_result (const jerry_value_t promise);
```

- `promise` - the input Promise object.
- return
  - The result of the Promise.
  - If the Promise is not resolved yet the result is the 'undefined' value.
  - A TypeError is returned if the input argument was not a Promise object or
    the Promise support was not built into the library.

*New in version 2.2*.

**Example**

[doctest]: # (test="compile")

```c
#include <jerryscript.h>

static void
example (void)
{
  // acquire/create a promise object.
  jerry_value_t promise = jerry_create_promise ();
  {
    // prepare the argumnent for the resolve or reject.
    jerry_value_t argument = jerry_create_number (33);

    jerry_value_t is_ok = jerry_resolve_or_reject_promise (promise,
                                                           argument,
                                                           true);
    // 'is_ok' should be checked if it is an error or not.
    // skipped in this example
    jerry_release_value (is_ok);
    jerry_release_value (argument);
  }

  jerry_value_t promise_result = jerry_get_promise_result (promise);
  // 'promise_result' is now the number 33.

  jerry_release_value (promise_result);
  jerry_release_value (promise);
}
```

**See also**

- [jerry_create_promise](#jerry_create_promise)
- [jerry_promise_state_t](#jerry_promise_state_t)

## jerry_get_promise_state

**Summary**

*Notes*:
- Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
  is no longer needed.
- This API depends on a build option (`JERRY_BUILTIN_PROMISE`) and can be checked
  in runtime with the `JERRY_FEATURE_PROMISE` feature enum value,
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).
- The es.next profile enables this by default.


**Prototype**

```c
jerry_promise_state_t
jerry_get_promise_state (const jerry_value_t promise);
```

- `promise` - the input promise object.
- return
  - [jerry_promise_state_t](#jerry_promise_state_t)
  - `JERRY_PROMISE_STATE_NONE` is returned if the input argument was not a promise object or
    the Promise support was not built into the library.

*New in version 2.2*.

**Example**

[doctest]: # (test="compile")

```c
#include <jerryscript.h>

static void
example (void)
{
  // acquire/create a promise object.
  jerry_value_t promise = jerry_create_promise ();

  jerry_promise_state_t start_state = jerry_get_promise_state (promise);
  // a Promise have a default state of JERRY_PROMISE_STATE_PENDING

  {
    // prepare the argumnent for the resolve or reject.
    jerry_value_t argument = jerry_create_number (33);

    jerry_value_t is_ok = jerry_resolve_or_reject_promise (promise,
                                                           argument,
                                                           true);
    // 'is_ok' should be checked if it is an error or not.
    // skipped in this example
    jerry_release_value (is_ok);
    jerry_release_value (argument);
  }

  jerry_promise_state_t current_state = jerry_get_promise_state (promise);
  // at this point the Promise should be in the JERRY_PROMISE_STATE_FULFILLED state.

  jerry_release_value (promise);
}
```

**See also**

- [jerry_create_promise](#jerry_create_promise)
- [jerry_promise_state_t](#jerry_promise_state_t)

## jerry_resolve_or_reject_promise

**Summary**

Resolve or reject the promise with an argument.

*Note*:
- Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
  is no longer needed.
- This API depends on a build option (`JERRY_BUILTIN_PROMISE`) and can be checked
  in runtime with the `JERRY_FEATURE_PROMISE` feature enum value,
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).
- The es.next profile enables this by default.


**Prototype**

```c
jerry_value_t
jerry_resolve_or_reject_promise (jerry_value_t promise,
                                 jerry_value_t argument,
                                 bool is_resolve)
```

- `promise` - the promise value
- `argument` - the argument for resolve or reject
- `is_resolve` - whether the promise should be resolved or rejected
- return value
  - undefined jerry value - resolve or reject successed
  - jerry value with error flag - otherwise

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t promise = ... // acquire/create a promise object.

  ...

  bool is_resolve = ... // whether the promise should be resolved or rejected
  jerry_value_t argument = ... // prepare the argumnent for the resolve or reject.

  jerry_value_t is_ok = jerry_resolve_or_reject_promise (promise,
                                                         argument,
                                                         is_resolve);

  if (jerry_value_is_error (is_ok))
  {
    // handle the error.
  }

  jerry_release_value (is_ok);
  jerry_release_value (argument);
  jerry_release_value (promise);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_value_is_error](#jerry_value_is_error)

# Functions for symbols

These APIs all depend on the es.next profile (or on build options).

## jerry_get_symbol_descriptive_string

**Summary**

Call the SymbolDescriptiveString ecma builtin operation on the API value.
Based on ECMA 262 v6 19.4.3.2.1 this is in the form of `Symbol(<description>)`.

*Notes*:
- Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
  is no longer needed.
- This API depends on a build option (`JERRY_BUILTIN_SYMBOL`) and can be checked
  in runtime with the `JERRY_FEATURE_SYMBOL` feature enum value,
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).
- The es.next profile enables this by default.
- If the symbol support is not enabled an error will be returned.

**Prototype**

```c
jerry_value_t
jerry_get_symbol_descriptive_string (const jerry_value_t value);
```

- `value` - symbol value
- return value
  - string value containing the symbol's descriptive string - if success
  - thrown error, otherwise

*New in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t string_value = jerry_create_string ((const jerry_char_t *) "foo");
  jerry_value_t symbol_value = jerry_create_symbol (string_value);

  jerry_release_value (string_value);

  jerry_value_t symbol_desc_string = jerry_get_symbol_descriptive_string (symbol_value);

  // usage of symbol_desc_string

  jerry_release_value (symbol_desc_string);
  jerry_release_value (symbol_value);

  jerry_cleanup ();
  return 0;
}
```


# Acquire and release API values

## jerry_acquire_value

**Summary**

Acquires the specified Jerry API value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_acquire_value (jerry_value_t value);
```

- `value` - api value
- return value - acquired value that may be used outside of the engine

**Example**

```c
{
  jerry_value_t object_value = jerry_create_object ();

  jerry_value_t acquired_object = jerry_acquire_value (object_value);

  jerry_release_value (object_value);

  // acquired_object refers to the created object and makes it
  // available after the release of 'object_value'

  jerry_release_value (acquired_object);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_value_t](#jerry_value_t)


## jerry_release_value

**Summary**

Release specified Jerry API value.

**Prototype**

```c
void
jerry_release_value (jerry_value_t value);
```

- `value` - api value

**Example**

```c
{
  jerry_value_t object_value = jerry_create_object ();

  ...

  jerry_release_value (object_value);
}
```


# Create API values

Function for creating [API values](#jerry_value_t).

*Note*: Every created API value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

## jerry_create_array

**Summary**

Create an array object value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_array (uint32_t size);
```

 - `size` - size of array;
 - return value - value of the constructed array object

 **Example**

```c
{
    jerry_value_t array = jerry_create_array (10);

    ...

    jerry_release_value (array);
}
```

**See also**

- [jerry_set_property_by_index](#jerry_set_property_by_index)
- [jerry_get_property_by_index](#jerry_get_property_by_index)


## jerry_create_arraybuffer

**Summary**

Create a jerry_value_t representing an ArrayBuffer object.

*Note*:
  - This API depends on the es.next profile.
  - Returned value must be freed with [jerry_release_value](#jerry_release_value)
    when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_arraybuffer (jerry_length_t size);
```

 - `size` - size of the ArrayBuffer to create **in bytes**
 - return value - the new ArrayBuffer as a `jerry_value_t`

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t buffer_value = jerry_create_arraybuffer (15);

  ... // use the ArrayBuffer

  jerry_release_value (buffer_value);
}
```

**See also**

- [jerry_arraybuffer_read](#jerry_arraybuffer_read)
- [jerry_arraybuffer_write](#jerry_arraybuffer_write)
- [jerry_value_is_arraybuffer](#jerry_value_is_arraybuffer)
- [jerry_release_value](#jerry_release_value)


## jerry_create_arraybuffer_external

**Summary**

Creates a jerry_value_t representing an ArrayBuffer object with
user specified back-buffer.

User must pass a buffer pointer which is at least `size` big.
After the object is not needed the GC will call the `free_cb`
so the user can release the buffer which was provided.

*Note*:
  - This API depends on the es.next profile.
  - Returned value must be freed with [jerry_release_value](#jerry_release_value)
    when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_arraybuffer_external (const jerry_length_t size
                                   uint8_t *buffer_p,
                                   jerry_object_native_free_callback_t free_cb);
```

- `size` - size of the buffer to use **in bytes** (should not be 0)
- `buffer_p` - the buffer used for the Array Buffer object (should not be a null pointer)
- `free_cb` - the callback function called when the object is released
- return value
  - the new ArrayBuffer as a `jerry_value_t`
  - if the `size` is zero or `buffer_p` is a null pointer this will return an empty ArrayBuffer.

*New in version 2.0*.

**Example**

```c
{
  uint8_t buffer_p[15];
  jerry_value_t buffer_value = jerry_create_arraybuffer_external (15, buffer_p, NULL);

  ... // use the array buffer

  jerry_release_value (buffer_value);
}
```

**See also**

- [jerry_get_arraybuffer_pointer](#jerry_get_arraybuffer_pointer)
- [jerry_arraybuffer_read](#jerry_arraybuffer_read)
- [jerry_arraybuffer_write](#jerry_arraybuffer_write)
- [jerry_value_is_arraybuffer](#jerry_value_is_arraybuffer)
- [jerry_release_value](#jerry_release_value)
- [jerry_object_native_free_callback_t](#jerry_object_native_free_callback_t)


## jerry_create_boolean

**Summary**

Create a jerry_value_t representing a boolean value from the given boolean parameter.

**Prototype**

```c
jerry_value_t
jerry_create_boolean (bool value);
```

- `value` - raw boolean value.
- return value - a `jerry_value_t` created from the given boolean argument.

**Example**

```c
{
  jerry_value_t boolean_value = jerry_create_boolean (true);

  ... // usage of the value

  jerry_release_value (boolean_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_create_error

**Summary**

Create new JavaScript error object.

Important! The `error_type` argument *must not be* `JERRY_ERROR_NONE`.
Creating an error with no error type is not valid.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_error (jerry_error_t error_type,
                    const jerry_char_t *message_p);
```

- `error_type` - type of error
- `message_p` - value of 'message' property of constructed error object
- return value - value of the constructed error object

**Example**

```c
{
  jerry_value_t error_obj = jerry_create_error (JERRY_ERROR_TYPE,
                                                (const jerry_char_t *) "error");

  ... // usage of error_obj


  jerry_release_value (error_obj);
}
```

**See also**

- [jerry_value_is_error](#jerry_value_is_error)
- [jerry_get_value_from_error](#jerry_get_value_from_error)
- [jerry_create_error_from_value](#jerry_create_error_from_value)


## jerry_create_error_sz

**Summary**

Create new JavaScript error object.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_error_sz (jerry_error_t error_type,
                       const jerry_char_t *message_p,
                       jerry_size_t message_size);
```

- `error_type` - type of the error
- `message_p` - value of 'message' property of the constructed error object
- `message_size` - size of the message in bytes
- return value - value of the constructed error object

**Example**

```c
{
  const jerry_char_t message[] = "error";
  jerry_value_t error_obj = jerry_create_error_sz (JERRY_ERROR_COMMON,
                                                   message,
                                                   sizeof (message) - 1);

  ... // usage of error_obj

  jerry_release_value (error_obj);
}
```

**See also**

- [jerry_create_error](#jerry_create_error)


## jerry_create_dataview

**Summary**

Create new JavaScript DataView object.

*Note*:
  - This API depends on the es.next profile.
  - Returned value must be freed with [jerry_release_value](#jerry_release_value)
    when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_dataview (const jerry_value_t array_buffer,
                       const jerry_length_t byte_offset,
                       const jerry_length_t byte_length)
```

- `array_buffer` - arrayBuffer to create DataView from
- `byte_offset` - offset in bytes, to the first byte in the buffer
- `byte_length` - number of elements in the byte array
- return value
  - value of the constructed DataView object - if success
  - created error - otherwise

*New in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t arraybuffer = jerry_create_arraybuffer (16);
  jerry_value_t dataview = jerry_create_dataview (arraybuffer, 0, 16);

  // usage of dataview

  jerry_release_value (dataview);
  jerry_release_value (arraybuffer);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_value_is_dataview](#jerry_value_is_dataview)
- [jerry_create_arraybuffer](#jerry_create_arraybuffer)


## jerry_create_external_function

**Summary**

Create an external function object.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_external_function (jerry_external_handler_t handler_p);
```

- `handler_p` - pointer to native handler of the function object
- return value - value of the constructed function object

**Example**

[doctest]: # ()

```c
#include <stdio.h>
#include <string.h>
#include "jerryscript.h"

static jerry_value_t
handler (const jerry_value_t function_obj,
         const jerry_value_t this_val,
         const jerry_value_t args_p[],
         const jerry_length_t args_cnt)
{
  printf ("native handler called!\n");

  return jerry_create_boolean (true);
}

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t func_val = jerry_create_external_function (handler);
  jerry_value_t glob_obj = jerry_get_global_object ();

  // after this, script can invoke the native handler through "handler_field (1, 2, 3);"
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "handler_field");
  // set property and release the return value without any check
  jerry_release_value (jerry_set_property (glob_obj, prop_name, func_val));
  jerry_release_value (prop_name);

  jerry_release_value (func_val);
  jerry_release_value (glob_obj);

  // Test the method by calling it
  const char *test_src = "handler_field ();";
  jerry_value_t ret_val = jerry_eval ((const jerry_char_t *) test_src,
                                      strlen (test_src),
                                      JERRY_PARSE_NO_OPTS);
  // release the eval result
  jerry_release_value (ret_val);
  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_external_handler_t](#jerry_external_handler_t)
- [jerry_set_property](#jerry_set_property)
- [jerry_call_function](#jerry_call_function)


## jerry_create_number

**Summary**

Creates a `jerry_value_t` representing a number value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_number (double value);
```

- `value` - double value from which a `jerry_value_t` will be created
- return value - a `jerry_value_t` created from the given double argument

**Example**

```c
{
  jerry_value_t number_value = jerry_create_number (3.14);

  ... // usage of the value

  jerry_release_value (number_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_create_number_infinity](#jerry_create_number_infinity)
- [jerry_create_number_nan](#jerry_create_number_nan)


## jerry_create_number_infinity

**Summary**

Creates a `jerry_value_t` representing a positive or negative infinity value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_number_infinity (bool sign);
```

- `sign` - true for negative Infinity and false for positive Infinity
- return value - a `jerry_value_t` representing the infinity value

**Example**

```c
{
  jerry_value_t positive_inf_value = jerry_create_number_infinity (false);

  ... // usage of the positive_inf_value

  jerry_release_value (positive_inf_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_create_number](#jerry_create_number)
- [jerry_create_number_nan](#jerry_create_number_nan)


## jerry_create_number_nan

**Summary**

Creates a `jerry_value_t` representing a not-a-number value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_number_nan (void);
```

- return value - a `jerry_value_t` representing the not-a-number value

**Example**

```c
{
  jerry_value_t nan_value = jerry_create_number_nan ();

  ... // usage of the nan_value

  jerry_release_value (nan_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_create_number](#jerry_create_number)
- [jerry_create_number_infinity](#jerry_create_number_infinity)


## jerry_create_null

**Summary**

Creates and returns a `jerry_value_t` with type null object.

**Prototype**

```c
jerry_value_t
jerry_create_null (void);
```

- return value - a `jerry_value_t` representing null.

**Example**

```c
{
  jerry_value_t null_value = jerry_create_null ();

  ... // usage of the value

  jerry_release_value (null_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_create_object

**Summary**

Create new JavaScript object, like with new Object().

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_object (void);
```

- return value - value of the created object

**Example**

```c
{
  jerry_value_t object_value = jerry_create_object ();

  ... // usage of object_value

  jerry_release_value (object_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_create_promise

**Summary**

Create an empty promise object which can be resolved or rejected later
by calling jerry_resolve_or_reject_promise.

*Note*:
  - This API depends on the es.next profile.
  - Returned value must be freed with [jerry_release_value](#jerry_release_value)
    when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_promise (void)
```

- return value - value of the newly created promise

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t p = jerry_create_promise ();

  ...// usage of the promise

  jerry_release_value (p);
}
```

**See also**

- [jerry_resolve_or_reject_promise](#jerry_resolve_or_reject_promise)
- [jerry_release_value](#jerry_release_value)


## jerry_create_proxy

**Summary**

Create a new Proxy object with the given target and handler.

*Note*:
  - This API depends on the es.next profile.
  - Returned value must be freed with [jerry_release_value](#jerry_release_value)
    when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_proxy (const jerry_value_t target,
                    const jerry_value_t handler)
```

- `target` - proxy target
- `handler` - proxy handler
- return thrown error - if the Proxy construction fails
         value of the newly created proxy object - otherwise

**Example**

*New in version 2.3*.

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t target = jerry_create_object ();
  jerry_value_t handler = jerry_create_object ();
  jerry_value_t proxy = jerry_create_proxy (target, handler);

  jerry_release_value (target);
  jerry_release_value (handler);

  // usage of the proxy

  jerry_release_value (proxy);

  jerry_cleanup ();
}
```

**See also**

- [jerry_value_is_proxy](#jerry_value_is_proxy)
- [jerry_release_value](#jerry_release_value)


## jerry_create_string

**Summary**

Create string from a valid CESU8 string.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_string (const jerry_char_t *str_p);
```

- `str_p` - non-null pointer to string
- return value - value of the created string

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string_value  = jerry_create_string (char_array);

  ... // usage of string_value

  jerry_release_value (string_value);
}
```

**See also**

- [jerry_is_valid_cesu8_string](#jerry_is_valid_cesu8_string)
- [jerry_create_string_sz](#jerry_create_string_sz)


## jerry_create_string_sz

**Summary**

Create string from a valid CESU8 string.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_string_sz (const jerry_char_t *str_p,
                        jerry_size_t str_size)
```

- `str_p` - non-null pointer to string
- `str_size` - size of the string
- return value - value of the created string

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string_value  = jerry_create_string_sz (char_array,
                                                        sizeof (char_array) - 1);

  ... // usage of string_value

  jerry_release_value (string_value);
}

```

**See also**

- [jerry_is_valid_cesu8_string](#jerry_is_valid_cesu8_string)
- [jerry_create_string](#jerry_create_string)


## jerry_create_string_from_utf8

**Summary**

Create string from a valid UTF8 string.

*Note*:
 - The difference from [jerry_create_string](#jerry_create_string) is that it accepts utf-8 string instead of cesu-8 string.
 - Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_string_from_utf8 (const jerry_char_t *str_p);
```

- `str_p` - non-null pointer to string
- return value - value of the created string

*New in version 2.0*.

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string_value  = jerry_create_string_from_utf8 (char_array);

  ... // usage of string_value

  jerry_release_value (string_value);
}
```

**See also**

- [jerry_is_valid_utf8_string](#jerry_is_valid_utf8_string)
- [jerry_create_string_sz_from_utf8](#jerry_create_string_sz_from_utf8)


## jerry_create_string_sz_from_utf8

**Summary**

Create string from a valid UTF8 string.

*Note*:
 - The difference from [jerry_create_string_sz](#jerry_create_string_sz) is that it accepts utf-8 string instead of cesu-8 string.
 - Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_string_sz_from_utf8 (const jerry_char_t *str_p,
                                  jerry_size_t str_size)
```

- `str_p` - non-null pointer to string
- `str_size` - size of the string
- return value - value of the created string

*New in version 2.0*.

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string_value  = jerry_create_string_sz_from_utf8 (char_array,
                                                                  sizeof (char_array) - 1);

  ... // usage of string_value

  jerry_release_value (string_value);
}

```

**See also**

- [jerry_is_valid_utf8_string](#jerry_is_valid_utf8_string)
- [jerry_create_string_from_utf8](#jerry_create_string_from_utf8)


## jerry_create_external_string

**Summary**

Create an external string from a valid CESU8 string. The string buffer passed to the function
should not be modified until the free callback is called. This function can be used to avoid
the duplication of large strings.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_external_string (const jerry_char_t *str_p,
                              jerry_object_native_free_callback_t free_cb)
```

- `str_p` - non-null pointer to string
- `free_cb` - optional callback which is called right before the string is freed
- return value - value of the created string

*New in version [[NEXT_RELEASE]]*

**Example**

```c
{
  const char* string_p = "a large and immutable string: this is a story about ....";
  jerry_value_t string_value  = jerry_create_external_string ((const jerry_char_t *) string_p,
                                                              NULL);

  ... // usage of string_value

  jerry_release_value (string_value);
}
```

**See also**

- [jerry_is_valid_cesu8_string](#jerry_is_valid_cesu8_string)
- [jerry_create_external_string_sz](#jerry_create_external_string_sz)


## jerry_create_external_string_sz

**Summary**

Create an external string from a valid CESU8 string. The string buffer passed to the function
should not be modified until the free callback is called. This function can be used to avoid
the duplication of large strings.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_external_string_sz (const jerry_char_t *str_p,
                                 jerry_size_t str_size,
                                 jerry_object_native_free_callback_t free_cb)
```

- `str_p` - non-null pointer to string
- `str_size` - size of the string
- `free_cb` - optional callback which is called right before the string is freed
- return value - value of the created string

*New in version [[NEXT_RELEASE]]*

**Example**

```c
{
  const char* string_p = "a large and immutable string: this is a story about ....";
  jerry_value_t string_value  = jerry_create_external_string_sz ((const jerry_char_t *) string_p,
                                                                 strlen (string_p),
                                                                 NULL);

  ... // usage of string_value

  jerry_release_value (string_value);
}

```

**See also**

- [jerry_is_valid_cesu8_string](#jerry_is_valid_cesu8_string)
- [jerry_create_external_string](#jerry_create_external_string)


## jerry_create_symbol

**Summary**

Create symbol from an API value.

*Note*:
  - The given argument is converted to string. This operation can throw an error.
  - This API depends on the es.next profile.
  - Returned value must be freed with [jerry_release_value](#jerry_release_value)
    when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_create_symbol (const jerry_value_t value)
```

- `value` - API value
- return value
  - value of the created symbol, if success
  - thrown error, otherwise

*New in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t string_value = jerry_create_string ((const jerry_char_t *) "Symbol description string");
  jerry_value_t symbol_value = jerry_create_symbol (string_value);

  // The description value is no longer needed
  jerry_release_value (string_value);

  // usage of symbol_value

  jerry_release_value (symbol_value);

  jerry_cleanup ();
}
```

**See also**

- [jerry_value_is_symbol](#jerry_value_is_symbol)
- [jerry_release_value](#jerry_release_value)


## jerry_create_regexp

**Summary**

Returns a `jerry_value_t` RegExp object or an error, if the construction of the object fails.
Optional flags can be set using [jerry_regexp_flags_t](#jerry_regexp_flags_t).
These flags can be combined together with the binary OR operator or used on their own as enum values.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**
```c
jerry_value_t
jerry_create_regexp (const jerry_char_t *pattern_p, uint16_t flags);
```

- `pattern_p` - the RegExp pattern as a zero-terminated UTF-8 string
- `flags` - optional flags for the RegExp object, see [jerry_regexp_flags_t](#jerry_regexp_flags_t)
- return value - the RegExp object as a `jerry_value_t`

*New in version 2.0*.

**Example**

```c
{
  jerry_char_t pattern_p = "[cgt]gggtaaa|tttaccc[acg]";
  uint16_t pattern_flags = JERRY_REGEXP_FLAG_IGNORE_CASE;

  jerry_value_t regexp = jerry_create_regexp (pattern_p, pattern_flags);

  ...

  jerry_release_value (regexp);
}
```


## jerry_create_regexp_sz

**Summary**

Returns a `jerry_value_t` RegExp object or an error, if the construction of the object fails.
Optional flags can be set using [jerry_regexp_flags_t](#jerry_regexp_flags_t).
These flags can be combined together with the binary OR operator or used on their own as enum values.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**
```c
jerry_value_t
jerry_create_regexp_sz (const jerry_char_t *pattern_p, jerry_size_t pattern_size, uint16_t flags);
```

- `pattern_p` - the RegExp pattern as a zero-terminated UTF-8 string
- `pattern_size` - size of the `pattern`
- `flags` - optional flags for the RegExp object, see [jerry_regexp_flags_t](#jerry_regexp_flags_t)
- return value - the RegExp object as a `jerry_value_t`

*New in version 2.0*.

**Example**

```c
{
  jerry_char_t pattern_p = "[cgt]gggtaaa|tttaccc[acg]";
  jerry_size_t pattern_size = sizeof (pattern_p) - 1;
  uint16_t pattern_flags = JERRY_REGEXP_FLAG_IGNORE_CASE;

  jerry_value_t regexp = jerry_create_regexp_sz (pattern_p, pattern_size, pattern_flags);

  ...

  jerry_release_value (regexp);
}
```


## jerry_create_typedarray

**Summary**

Create a jerry_value_t representing an TypedArray object.

For the new object the type of the TypedArray (see: [jerry_typedarray_type_t](#jerry_typedarray_type_t))
and element count can be specified.

*Notes*:
- Returned value must be freed with [jerry_release_value](#jerry_release_value)
  when it is no longer needed.
- This API depends on a build option (`JERRY_BUILTIN_TYPEDARRAY`) and can be checked
  in runtime with the `JERRY_FEATURE_TYPEDARRAY` feature enum value,
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).
- The es.next profile enables this by default.

**Prototype**

```c
jerry_value_t
jerry_create_typedarray (jerry_typedarray_type_t type_name, jerry_length_t item_count);
```

- `type_name` - type of TypedArray to create
- `item_count` - number of items in the new TypedArray
- return value - the new TypedArray as a `jerry_value_t`

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t array = jerry_create_typedarray (JERRY_TYPEDARRAY_UINT16, 15);

  ... // use the TypedArray

  jerry_release_value (array);
}
```

**See also**

- [jerry_typedarray_type_t](#jerry_typedarray_type_t)
- [jerry_value_is_typedarray](#jerry_value_is_typedarray)
- [jerry_release_value](#jerry_release_value)


## jerry_create_typedarray_for_arraybuffer

**Summary**

Create a jerry_value_t representing an TypedArray object using
an already existing ArrayBuffer object.

For the new object the type of the TypedArray (see: [jerry_typedarray_type_t](#jerry_typedarray_type_t))
and element count can be specified.

The developer must ensure that the ArrayBuffer has the correct length for the given
type of TypedArray otherwise an error is generated.

The JavaScript equivalent of this function is: `new %TypedArray%(arraybuffer)` where `%TypedArray%` is
one of the allowed TypedArray functions.

*Notes*:
- Returned value must be freed with [jerry_release_value](#jerry_release_value)
  when it is no longer needed.
- This API depends on a build option (`JERRY_BUILTIN_TYPEDARRAY`) and can be checked
  in runtime with the `JERRY_FEATURE_TYPEDARRAY` feature enum value,
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).
- The es.next profile enables this by default.

**Prototype**

```c
jerry_value_t
jerry_create_typedarray_for_arraybuffer (jerry_typedarray_type_t type_name,
                                         const jerry_value_t arraybuffer);
```

- `type_name` - type of TypedArray to create
- `arraybuffer` - the ArrayBuffer to use for the new TypedArray
- return value
  - the new TypedArray as a `jerry_value_t`
  - Error if the ArrayBuffer does not have enough space for the given type of TypedArray

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t buffer = jerry_create_array_buffer (12 * 2);
  jerry_value_t array = jerry_create_typedarray_for_arraybuffer (JERRY_TYPEDARRAY_UINT16, buffer);
  jerry_release_value (buffer);

  ... // use the TypedArray

  jerry_release_value (array);
}
```

**See also**

- [jerry_typedarray_type_t](#jerry_typedarray_type_t)
- [jerry_value_is_typedarray](#jerry_value_is_typedarray)
- [jerry_release_value](#jerry_release_value)


## jerry_create_typedarray_for_arraybuffer_sz

**Summary**

Create a jerry_value_t representing an TypedArray object using
an already existing ArrayBuffer object and by specifying the byteOffset, and length properties.

For the new object the type of the TypedArray (see: [jerry_typedarray_type_t](#jerry_typedarray_type_t))
and element count can be specified.

The developer must ensure that the ArrayBuffer has the correct length for the given
type of TypedArray otherwise an error is generated.

The JavaScript equivalent of this function is: `new %TypedArray%(arraybuffer, byteOffset, length)` where `%TypedArray%` is
one of the allowed TypedArray functions.

*Notes*:
- Returned value must be freed with [jerry_release_value](#jerry_release_value)
  when it is no longer needed.
- This API depends on a build option (`JERRY_BUILTIN_TYPEDARRAY`) and can be checked
  in runtime with the `JERRY_FEATURE_TYPEDARRAY` feature enum value,
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).
- The es.next profile enables this by default.

**Prototype**

```c
jerry_value_t
jerry_create_typedarray_for_arraybuffer_sz (jerry_typedarray_type_t type_name,
                                            const jerry_value_t arraybuffer,
                                            jerry_length_t byte_offset,
                                            jerry_length_t length);
```

- `type_name` - type of TypedArray to create
- `arraybuffer` - the ArrayBuffer to use for the new TypedArray
- `byte_offset` - start offset to use for the ArrayBuffer
- `length` - number of elements to used from the ArrayBuffer (this is not the same as the byteLength)
- return value
  - the new TypedArray as a `jerry_value_t`
  - Error if the ArrayBuffer does not have enough space for the given type of TypedArray

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t buffer = jerry_create_array_buffer (12 * 2);
  jerry_value_t array = jerry_create_typedarray_for_arraybuffer_sz (JERRY_TYPEDARRAY_UINT16, buffer, 4, 10);
  jerry_release_value (buffer);

  ... // use the TypedArray

  jerry_release_value (array);
}
```

**See also**

- [jerry_typedarray_type_t](#jerry_typedarray_type_t)
- [jerry_value_is_typedarray](#jerry_value_is_typedarray)
- [jerry_release_value](#jerry_release_value)


## jerry_create_container

**Summary**

Create a jerry_value_t representing a given type container object.

*Notes*:
- This method is expected to work the same way as the JavaScript Map constructor.
- Returned value must be freed with [jerry_release_value](#jerry_release_value)
  when it is no longer needed.
- This API depends on a build option (`JERRY_BUILTIN_CONTAINER`) and can be checked
  in runtime with the `JERRY_FEATURE_MAP, JERRY_FEATURE_SET, JERRY_FEATURE_WEAKMAP, JERRY_FEATURE_WEAKSET`
  feature enum values.
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).
- The es.next profile enables this by default.

*New in version 2.3*.

**Prototype**

```c
jerry_value_t
jerry_create_container (jerry_container_type_t container_type,
                        const jerry_value_t *arguments_list_p,
                        jerry_length_t arguments_list_len);
```

- `container_type` - Type of the container to be created, see `jerry_container_type_t`.
- `arguments_list_p` - The arguments passed to the container constructor to be inserted to the container.
- `arguments_list_len` - The length of the above arguments.
- return value - the new container object as a `jerry_value_t`

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);
  jerry_char_t src[] = "[1,2,3,4].entries()";
  jerry_value_t iterable = jerry_eval (src, sizeof (src) - 1, JERRY_PARSE_NO_OPTS);

  jerry_value_t map = jerry_create_container (JERRY_CONTAINER_TYPE_MAP, &iterable, 1);
  jerry_release_value (iterable);

  // use the Map

  jerry_release_value (map);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_container_type_t](#jerry_container_type_t)
- [jerry_get_container_type](#jerry_get_container_type)


## jerry_create_undefined

**Summary**

Creates a `jerry_value_t` representing an undefined value.

**Prototype**

```c
jerry_value_t
jerry_create_undefined (void);
```

- return value - value of undefined

**Example**

```c
{
  jerry_value_t undefined_value = jerry_create_undefined ();

  ... // usage of the value

  jerry_release_value (undefined_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


# General API functions of JS objects

## jerry_has_property

**Summary**

Checks whether the object or its prototype objects have the given property.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_has_property (const jerry_value_t obj_val,
                    const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value - JavaScript value that evaluates to
  -  raised error - if the operation fail
  -  true/false API value  - depend on whether the property exists

*Changed in version 2.0*: The return value type is now a JavaScript value and not a primitive boolean value.
*Changed in version 2.3*: The return value can be an error value.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_object = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "handler_field");

  jerry_value_t has_prop_js = jerry_has_property (global_object, prop_name);
  bool has_prop = jerry_get_boolean_value (has_prop_js);

  jerry_release_value (has_prop_js);
  jerry_release_value (prop_name);
  jerry_release_value (global_object);

  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_delete_property](#jerry_delete_property)


## jerry_has_own_property

**Summary**

Checks whether the object has the given property.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_has_own_property (const jerry_value_t obj_val,
                        const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value - JavaScript value that evaluates to
  -  raised error - if the operation fails
  -  true/false API value - depend on whether the property exists

*Changed in version 2.0*: The return value type is now a JavaScript value and not a primitive boolean value.
*Changed in version 2.3*: The return value can be an error value.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_object = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "handler_field");

  jerry_value_t has_prop_js = jerry_has_own_property (global_object, prop_name);
  bool has_prop = jerry_get_boolean_value (has_prop_js);

  jerry_release_value (has_prop_js);
  jerry_release_value (prop_name);
  jerry_release_value (global_object);

  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_delete_property](#jerry_delete_property)


## jerry_has_internal_property

**Summary**

Checks whether the object has the given internal property.

*Note*:
 - Properties which were not created with [jerry_set_internal_property](#jerry_set_internal_property) are excluded
   during the operation.
 - Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
bool
jerry_has_internal_property (const jerry_value_t obj_val,
                             const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value
  - true, if the property exists
  - false, otherwise

*New in version 2.2*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_object = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "hidden_property");

  bool has_internal_js_prop = jerry_has_internal_property (global_object, prop_name);

  jerry_release_value (prop_name);
  jerry_release_value (global_object);

  return 0;
}
```

**See also**

- [jerry_delete_internal_property](#jerry_delete_internal_property)
- [jerry_get_internal_property](#jerry_get_internal_property)
- [jerry_set_internal_property](#jerry_set_internal_property)


## jerry_delete_property

**Summary**

Delete a property from an object.

**Prototype**

```c
bool
jerry_delete_property (const jerry_value_t obj_val,
                       const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value
  - true, if property was deleted successfully
  - false, otherwise

**Example**

```c
{
  jerry_value_t global_object = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "my_prop");

  bool delete_result = jerry_delete_property (global_object, prop_name);
  /* use "delete_result" */

  jerry_release_value (prop_name);
  jerry_release_value (global_object);
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_delete_property_by_index](#jerry_delete_property_by_index)
- [jerry_get_property](#jerry_get_property)


## jerry_delete_property_by_index

**Summary**

Delete indexed property from the specified object.

**Prototype**

```c
bool
jerry_delete_property_by_index (const jerry_value_t obj_val,
                                uint32_t index);
```

- `obj_val` - object value
- `index` - index number
- return value
  - true, if property was deleted successfully
  - false, otherwise

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t object;

  ... // create or acquire object

  bool delete_result = jerry_delete_property_by_index (object, 5);

  jerry_release_value (object);
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_delete_property](#jerry_delete_property)
- [jerry_get_property](#jerry_get_property)
- [jerry_set_property](#jerry_set_property)
- [jerry_get_property_by_index](#jerry_get_property_by_index)
- [jerry_set_property_by_index](#jerry_set_property_by_index)

## jerry_delete_internal_property

**Summary**

Delete an internal property from an object.

*Note*: Properties which were not created with [jerry_set_internal_property](#jerry_set_internal_property) are excluded
        during the operation.

**Prototype**

```c
bool
jerry_delete_internal_property (const jerry_value_t obj_val,
                                const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value
  - true, if property was deleted successfully
  - false, otherwise

*New in version 2.2*.

**Example**

```c
{
  jerry_value_t global_object = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "hidden_property");

  bool delete_result = jerry_delete_internal_property (global_object, prop_name);
  /* use "delete_result" */

  jerry_release_value (prop_name);
  jerry_release_value (global_object);
}
```

**See also**

- [jerry_has_internal_property](#jerry_has_internal_property)
- [jerry_get_internal_property](#jerry_get_internal_property)
- [jerry_set_internal_property](#jerry_set_internal_property)


## jerry_get_property

**Summary**

Get value of a property to the specified object with the given name.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_get_property (const jerry_value_t obj_val,
                    const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value
  - value of property, if success
  - thrown error, otherwise

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_object = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "Object");

  jerry_value_t prop_value = jerry_get_property (global_object, prop_name);

  /* use "prop_value" then release it. */

  jerry_release_value (prop_value);
  jerry_release_value (prop_name);
  jerry_release_value (global_object);

  return 0;
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_delete_property](#jerry_delete_property)
- [jerry_delete_property_by_index](#jerry_delete_property_by_index)
- [jerry_set_property](#jerry_set_property)
- [jerry_get_property_by_index](#jerry_get_property_by_index)
- [jerry_set_property_by_index](#jerry_set_property_by_index)


## jerry_get_property_by_index

**Summary**

Get value by an index from the specified object.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_get_property_by_index (const jerry_value_t obj_val,
                             uint32_t index);
```

- `obj_val` - object value
- `index` - index number
- return value
  - stored value on the specified index, if success
  - thrown exception, otherwise.

**Example**

```c
{
  jerry_value_t object;

  ... // create or acquire object

  jerry_value_t value = jerry_get_property_by_index (object, 5);

  ...

  jerry_release_value (value);
  jerry_release_value (object);
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_delete_property](#jerry_delete_property)
- [jerry_delete_property_by_index](#jerry_delete_property_by_index)
- [jerry_get_property](#jerry_get_property)
- [jerry_set_property](#jerry_set_property)
- [jerry_set_property_by_index](#jerry_set_property_by_index)

## jerry_get_internal_property

**Summary**

Get value of an internal property to the specified object with the given name.

*Note*:
 - Properties which were not created with [jerry_set_internal_property](#jerry_set_internal_property) are excluded
   during the operation.
 - Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
   is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_get_internal_property (const jerry_value_t obj_val,
                             const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value
  - value of property, if the internal property exists
  - undefined value, if the, if the internal does not property exists
  - thrown error, otherwise

*New in version 2.2*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_object = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "hidden_property");

  jerry_value_t prop_value = jerry_get_internal_property (global_object, prop_name);

  /* use "prop_value" then release it. */

  jerry_release_value (prop_value);
  jerry_release_value (prop_name);
  jerry_release_value (global_object);

  return 0;
}
```

**See also**

- [jerry_has_internal_property](#jerry_has_internal_property)
- [jerry_delete_internal_property](#jerry_delete_internal_property)
- [jerry_set_internal_property](#jerry_set_internal_property)


## jerry_set_property

**Summary**

Set a property to the specified object with the given name.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_set_property (const jerry_value_t obj_val,
                    const jerry_value_t prop_name_val,
                    const jerry_value_t value_to_set)
```

- `obj_val` - object value
- `prop_name_val` - property name
- `value_to_set` - value to set
- return value
  - true, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t value_to_set;

  ... // create or acquire value to set

  jerry_value_t glob_obj = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "my_prop");

  jerry_value_t set_result = jerry_set_property (glob_obj, prop_name, value_to_set);

  ... // check result of property set call

  jerry_release_value (set_result);
  jerry_release_value (prop_name);

  ...

  jerry_release_value (value_to_set);
  jerry_release_value (glob_obj);
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_delete_property](#jerry_delete_property)
- [jerry_delete_property_by_index](#jerry_delete_property_by_index)
- [jerry_get_property](#jerry_get_property)
- [jerry_get_property_by_index](#jerry_get_property_by_index)
- [jerry_set_property_by_index](#jerry_set_property_by_index)


## jerry_set_property_by_index

**Summary**

Set indexed value in the specified object

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_set_property_by_index (const jerry_value_t obj_val,
                             uint32_t index,
                             const jerry_value_t value_to_set);
```

- `obj_val` - object value
- `index` - index number
- `value_to_set` - value to set
- return value
  - true, if field value was set successfully
  - thrown exception, otherwise

**Example**

```c
{
  jerry_value_t object;
  jerry_value_t value_to_set;

  ... // create or acquire object and value to set

  jerry_value_t ret_val = jerry_set_property_by_index (object, 5, value_to_set);

  ...

  jerry_release_value (value_to_set);
  jerry_release_value (ret_val);
  jerry_release_value (object);
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_delete_property](#jerry_delete_property)
- [jerry_delete_property_by_index](#jerry_delete_property_by_index)
- [jerry_get_property](#jerry_get_property)
- [jerry_set_property](#jerry_set_property)
- [jerry_get_property_by_index](#jerry_get_property_by_index)


## jerry_set_internal_property

**Summary**

Set an internal property to the specified object with the given name.

*Note*:
 - The property cannot be accessed from the JavaScript context, only from the public API.
 - It is different from [jerry_set_object_native_pointer](#jerry_set_object_native_pointer) in that any jerry API value
   can be hidden from the JavaScript context, not only native pointers.

**Prototype**

```c
bool
jerry_set_internal_property (const jerry_value_t obj_val,
                             const jerry_value_t prop_name_val,
                             const jerry_value_t value_to_set)
```

- `obj_val` - object value
- `prop_name_val` - property name
- `value_to_set` - value to set
- return value
  - true, if success
  - thrown error, otherwise

*New in version 2.2*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_object = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "hidden_property");
  jerry_value_t value_to_set = jerry_create_number (5);

  bool set_result = jerry_set_internal_property (global_object, prop_name, value_to_set);

  /* check the result of internal property set call */

  jerry_release_value (value_to_set);
  jerry_release_value (prop_name);
  jerry_release_value (global_object);

  return 0;
}
```

**See also**

- [jerry_has_internal_property](#jerry_has_internal_property)
- [jerry_delete_internal_property](#jerry_delete_internal_property)
- [jerry_get_internal_property](#jerry_get_internal_property)


## jerry_init_property_descriptor_fields

**Summary**

Initialize property descriptor. This means that all fields in the `jerry_property_descriptor_t`
struct will be set to zero or false depending on the field's type.

**Prototype**

```c
void
jerry_init_property_descriptor_fields (jerry_property_descriptor_t *prop_desc_p);
```

- `prop_desc_p` - pointer to property descriptor

**Example**

```c
{
  jerry_property_descriptor_t prop_desc;
  jerry_init_property_descriptor_fields (&prop_desc);

  ... // usage of prop_desc

  jerry_free_property_descriptor_fields (&prop_desc);
}
```

For a more complete example see [jerry_define_own_property](#jerry_define_own_property).

**See also**

- [jerry_property_descriptor_t](#jerry_property_descriptor_t)
- [jerry_define_own_property](#jerry_define_own_property)
- [jerry_get_own_property_descriptor](#jerry_get_own_property_descriptor)
- [jerry_free_property_descriptor_fields](#jerry_free_property_descriptor_fields)


## jerry_define_own_property

**Summary**

Define a property to the specified object with the given name.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_define_own_property (const jerry_value_t obj_val,
                           const jerry_value_t prop_name_val,
                           const jerry_property_descriptor_t *prop_desc_p);
```

- `obj_val` - target object where the property should be registered
- `prop_name_val` - property name
- `prop_desc_p` - pointer to property descriptor
- return value
  - true, if success
  - thrown error, otherwise

**Example**

Registering a simple value property via the `jerry_define_own_property` method:

[doctest]: # (name="02.API-REFERENCE-define-property.c")

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_obj_val = jerry_get_global_object ();

  // configure the property
  jerry_property_descriptor_t prop_desc;
  jerry_init_property_descriptor_fields (&prop_desc);

  jerry_value_t value_to_set;

  // create or acquire value to set
  // For example:
  value_to_set = jerry_create_number (33);

  // set the property descriptor fields:
  // set the "is_value_defined" field to "true" to indicate the "value"
  //  field should be used during the property registration.
  prop_desc.is_value_defined = true;

  // set the "value" field to the number 33
  prop_desc.value = value_to_set;

  // add the property as "my_prop" for the global object
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "my_prop");
  jerry_value_t return_value = jerry_define_own_property (global_obj_val, prop_name, &prop_desc);
  if (jerry_value_is_error (return_value))
  {
    // there was an error
  }

  // if there was no error at this point the global object should have a "my_prop" property

  jerry_release_value (return_value);
  jerry_release_value (prop_name);

  jerry_free_property_descriptor_fields (&prop_desc);
  jerry_release_value (global_obj_val);

  jerry_cleanup ();
  return 0;
}
```


Registering a getter/setter property via the `jerry_define_own_property` method:

[doctest]: # (name="02.API-REFERENCE-define-property-getset.c")

```c
#include <stdio.h>
#include <string.h>
#include "jerryscript.h"

static int counter = 0;

static jerry_value_t
method_getter (const jerry_value_t this_obj,
               const jerry_value_t func_obj,
               const jerry_value_t args[],
               const jerry_length_t argc)
{
  counter++;
  printf("Getter called, returning: %d\n", counter);

  return jerry_create_number (counter);
}

static jerry_value_t
method_setter (const jerry_value_t this_obj,
               const jerry_value_t func_obj,
               const jerry_value_t args[],
               const jerry_length_t argc)
{
  // Note: the arguments count and type should be checked
  // in this example it is ommitted!

  double new_value = jerry_get_number_value (args[0]);
  counter = (int) new_value;

  printf("Setter called, setting: %d\n", counter);

  return jerry_create_undefined ();
}

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_obj_val = jerry_get_global_object ();

  // configure the property
  jerry_property_descriptor_t prop_desc;
  jerry_init_property_descriptor_fields (&prop_desc);

  // set the property descriptor fields:

  prop_desc.is_get_defined = true;
  prop_desc.getter = jerry_create_external_function (method_getter);
  prop_desc.is_set_defined = true;
  prop_desc.setter = jerry_create_external_function (method_setter);

  // add the property as "my_prop" for the global object
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "my_prop");
  jerry_value_t return_value = jerry_define_own_property (global_obj_val, prop_name, &prop_desc);
  if (jerry_value_is_error (return_value))
  {
    // there was an error
  }

  // if there was no error at this point the global object should have a "my_prop" property

  jerry_release_value (return_value);
  jerry_release_value (prop_name);

  jerry_free_property_descriptor_fields (&prop_desc);
  jerry_release_value (global_obj_val);

  // run an example js code to use the getter/setters

  const char *src_p = "this.my_prop; this.my_prop; this.my_prop = 4; this.my_prop";
  jerry_value_t eval_result = jerry_eval ((const jerry_char_t *) src_p, strlen (src_p), JERRY_PARSE_NO_OPTS);

  // "eval_result" is the last result of "this.my_prop" that is "5" currently.
  double result_number = jerry_get_number_value (eval_result);
  printf("output: %lf\n", result_number);

  jerry_cleanup ();

  return result_number != 5.0;
}
```

**See also**

- [jerry_property_descriptor_t](#jerry_property_descriptor_t)
- [jerry_init_property_descriptor_fields](#jerry_init_property_descriptor_fields)
- [jerry_get_own_property_descriptor](#jerry_get_own_property_descriptor)
- [jerry_free_property_descriptor_fields](#jerry_free_property_descriptor_fields)


## jerry_get_own_property_descriptor

**Summary**

Construct property descriptor from specified property.

**Prototype**

```c
bool
jerry_get_own_property_descriptor (const jerry_value_t  obj_val,
                                   const jerry_value_t prop_name_val,
                                   jerry_property_descriptor_t *prop_desc_p);
```

- `obj_val` - object value
- `prop_name_val` - property name
- `prop_desc_p` - pointer to property descriptor
- return value

**Example**

```c
{
  jerry_value_t global_obj_val = jerry_get_global_object ();

  jerry_property_descriptor_t prop_desc;
  jerry_init_property_descriptor_fields (&prop_desc);

  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "my_prop");
  jerry_get_own_property_descriptor (global_obj_val, prop_name, &prop_desc);
  jerry_release_value (prop_name);

  ... // usage of property descriptor

  jerry_free_property_descriptor_fields (&prop_desc);
  jerry_release_value (global_obj_val);
}
```

**See also**

- [jerry_property_descriptor_t](#jerry_property_descriptor_t)
- [jerry_init_property_descriptor_fields](#jerry_init_property_descriptor_fields)
- [jerry_define_own_property](#jerry_define_own_property)
- [jerry_free_property_descriptor_fields](#jerry_free_property_descriptor_fields)


## jerry_free_property_descriptor_fields

**Summary**

Free fields of property descriptor (setter, getter and value).

**Prototype**

```c
void
jerry_free_property_descriptor_fields (const jerry_property_descriptor_t *prop_desc_p);
```

- `prop_desc_p` - pointer to property descriptor

**Example**

```c
{
  jerry_property_descriptor_t prop_desc;
  jerry_init_property_descriptor_fields (&prop_desc);

  ... // usage of property descriptor

  jerry_free_property_descriptor_fields (&prop_desc);
}
```

**See also**

- [jerry_init_property_descriptor_fields](#jerry_init_property_descriptor_fields)
- [jerry_define_own_property](#jerry_define_own_property)
- [jerry_get_own_property_descriptor](#jerry_get_own_property_descriptor)


## jerry_call_function

**Summary**

Call function specified by a function value. Error flag must
not be set for any arguments of this function. Value of `this`
parameter should be set to `undefined` for non-method calls.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_call_function (const jerry_value_t func_obj_val,
                     const jerry_value_t this_val,
                     const jerry_value_t args_p[],
                     jerry_size_t args_count);
```

- `func_obj_val` - the function object to call
- `this_val` - object for 'this' binding
- `args_p` - function's call arguments
- `args_count` - number of arguments
- return value - returned jerry value of the called function

**Example**

```c
{
  jerry_value_t target_function;

  ... // create or get "target_function"

  if (jerry_value_is_function (target_function))
  {
    jerry_value_t this_val = jerry_create_undefined ();
    jerry_value_t ret_val = jerry_call_function (target_function, this_val, NULL, 0);

    if (!jerry_value_is_error (ret_val))
    {
      ... // handle return value
    }

    jerry_release_value (ret_val);
    jerry_release_value (this_val);
  }

  jerry_release_value (target_function);
}
```

**See also**

- [jerry_is_function](#jerry_is_function)
- [jerry_create_external_function](#jerry_create_external_function)


## jerry_construct_object

**Summary**

Construct object, invoking specified function object as constructor.
Error flag must not be set for any arguments of this function.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_construct_object (const jerry_value_t func_obj_val,
                        const jerry_value_t args_p[],
                        jerry_size_t args_count);
```

- `func_obj_val` - function object to call
- `args_p` - function's call arguments
- `args_count` - number of arguments
- return value - returned value of the invoked constructor

**Example**

```c
{
  jerry_value_t val;

  ... // receiving val

  if (jerry_is_constructor (val))
  {
    jerry_value_t ret_val = jerry_construct_object (val, NULL, 0);

    if (!jerry_value_is_error (ret_val))
    {
      ... // handle return value
    }

    jerry_release_value (ret_val);
  }
}
```

**See also**

 - [jerry_is_constructor](#jerry_is_constructor)


## jerry_get_object_keys

**Summary**

Get keys of the specified object value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_get_object_keys (const jerry_value_t obj_val);
```

- `obj_val` - object value
- return value
  - array object value, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t object;
  ... // create or acquire object

  jerry_value_t keys_array = jerry_get_object_keys (object);

  ... // usage of keys_array

  jerry_release_value (keys_array);
}
```

**See also**

- [jerry_get_property](#jerry_get_property)
- [jerry_set_property](#jerry_set_property)


## jerry_get_prototype

**Summary**

Get the prototype of the specified object.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_get_prototype (const jerry_value_t obj_val);
```

- `obj_val` - object value
- return value
  - object value, if success
  - null or thrown error, otherwise

**Example**

```c
{
  jerry_value_t object;
  ... // create or acquire object

  jerry_value_t prototype = jerry_get_prototype (object);

  ... // usage of prototype object

  jerry_release_value (prototype);
  jerry_release_value (object);
}
```

**See also**

- [jerry_set_prototype](#jerry_set_prototype)


## jerry_set_prototype

**Summary**

Set the prototype of the specified object.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_set_prototype (const jerry_value_t obj_val,
                     const jerry_value_t proto_obj_val);
```

- `obj_val` - object value
- `proto_obj_val` - prototype object value
- return value
  - true, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t object;
  jerry_value_t prototype;

  ... // create or acquire object and prototype

  jerry_value_t ret_val = jerry_set_prototype (object, prototype);

  jerry_release_value (ret_val);
  jerry_release_value (prototype);
  jerry_release_value (object);
}
```

**See also**

- [jerry_get_prototype](#jerry_get_prototype)


## jerry_get_object_native_pointer

**Summary**

Get native pointer by the given type information.
The pointer and the type information are previously associated with the object by
[jerry_set_object_native_pointer](#jerry_set_object_native_pointer).

*Note*: `out_native_pointer_p` can be NULL, and it means the
        caller doesn't want to get the native_pointer.

**Prototype**

```c
bool
jerry_get_object_native_pointer (const jerry_value_t obj_val,
                                 void **out_native_pointer_p,
                                 const jerry_object_native_info_t *native_info_p)
```

- `obj_val` - object value to get native pointer from.
- `out_native_pointer_p` - native pointer (output parameter).
- `native_info_p` - native pointer's type information.
- return value
  - true, if there is native pointer associated of the specified object with the given native type info
  - false, otherwise

*New in version 2.0*: Changed from `jerry_get_object_native_handle`.

**Example**

[doctest]: # ()

```c
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "jerryscript.h"

typedef struct
{
  char *data_p;
  unsigned int length;
} buffer_native_object_t;

typedef struct
{
  int area;
  int perimeter;
} shape_native_object_t;

#define SECRET_INFO ((void *) 42)

static void
buffer_native_freecb (void *native_p)
{
  char *data_p = ((buffer_native_object_t*)native_p)->data_p;

  if (data_p != NULL)
  {
    free (data_p);
  }

  free (native_p);
}

static void
shape_native_freecb (void *native_p)
{
  free (native_p);
}

static void
destructor_freecb (void *native_p)
{
   printf("Note: the object has been freed\n");
}

// NOTE: The address (!) of type_info acts as a way to uniquely "identify" the
// C type `buffer_native_object_t *`.
static const jerry_object_native_info_t buffer_obj_type_info =
{
  .free_cb = buffer_native_freecb
};

// NOTE: The address (!) of type_info acts as a way to uniquely "identify" the
// C type `shape_native_object_t *`.
static const jerry_object_native_info_t shape_obj_type_info =
{
  .free_cb = shape_native_freecb
};

// NOTE: The address (!) of type_info is the unique "identifier"
static const jerry_object_native_info_t destructor_obj_type_info =
{
  .free_cb = destructor_freecb
};

static void
print_buffer (char *data_p,
              unsigned int length)
{
  for (unsigned int i = 0; i < length; ++i)
  {
    printf("%c", data_p[i]);
  }

  printf("\n");
}

static void
do_stuff (jerry_value_t object)
{
  void *native_p;
  bool has_p = jerry_get_object_native_pointer (object, &native_p, &buffer_obj_type_info);

  if (!has_p)
  {
    // Process the error
    return;
  }

  // It is safe to cast to buffer_native_object_t * and dereference the pointer:
  buffer_native_object_t *buffer_p = (buffer_native_object_t *) native_p;
  print_buffer (buffer_p->data_p, buffer_p->length); // Usage of buffer_p

  bool need_shape_info = true; // implementation dependent

  if (need_shape_info)
  {
    has_p = jerry_get_object_native_pointer (object, &native_p, &shape_obj_type_info);

    if (!has_p)
    {
      // Process the error
      return;
    }

    // It is safe to cast to shape_native_object_t * and dereference the pointer:
    shape_native_object_t *shape_p = (shape_native_object_t *) native_p;

    printf("Area: %d\tPerimeter: %d\n", shape_p->area, shape_p->perimeter); // Usage of shape_p
  }

  bool need_secret_info = true; // implementation dependent

  if (need_secret_info)
  {
    has_p = jerry_get_object_native_pointer (object, &native_p, NULL);

    if (!has_p)
    {
      // Process the error
      return;
    }

    printf("Secret: %d\n", (int)((uintptr_t) native_p)); // Usage of native_p

    bool deleted = jerry_delete_object_native_pointer (object, NULL);

    if (deleted)
    {
      printf("The secret is no longer available\n");
    }
  }
}

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t object = jerry_create_object ();
  buffer_native_object_t *buffer_p = (buffer_native_object_t *) malloc (sizeof (buffer_native_object_t));
  buffer_p->length = 14;
  buffer_p->data_p = (char *) malloc (buffer_p->length * sizeof (char));
  memcpy (buffer_p->data_p, "My buffer data", buffer_p->length);
  jerry_set_object_native_pointer (object, buffer_p, &buffer_obj_type_info);

  shape_native_object_t *shape_p = (shape_native_object_t *) malloc (sizeof (shape_native_object_t));
  shape_p->area = 6;
  shape_p->perimeter = 12;
  jerry_set_object_native_pointer (object, shape_p, &shape_obj_type_info);

  // The native pointer can be NULL. This gives possibily to get notified via the native type info's
  // free callback when the object has been freed by the GC.
  jerry_set_object_native_pointer (object, NULL, &destructor_obj_type_info);

  // The native type info can be NULL as well. In this case the registered property is simply freed
  // when the object is freed by te GC.
  jerry_set_object_native_pointer (object, SECRET_INFO, NULL);

  do_stuff (object);

  jerry_release_value (object);
  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_create_object](#jerry_create_object)
- [jerry_set_object_native_pointer](#jerry_set_object_native_pointer)
- [jerry_object_native_info_t](#jerry_object_native_info_t)


## jerry_set_object_native_pointer

**Summary**

Set native pointer and an optional type information for the specified object.
You can get them by calling [jerry_get_object_native_pointer](#jerry_get_object_native_pointer) later.

*Notes*:
 - If a native pointer was already set for the object with the same type information, its value is updated.
 - If a non-NULL free callback is specified in the native type information,
   it will be called by the garbage collector when the object is freed.
 - If the object is only referenced via the "global" object (or one of it's "child"),
   the free callback will be invoked during the execution of `jerry_cleanup`.
 - The free callback can invoke API functions.

*Note*: If possible do not store API values in native pointers, rather check
        [jerry_set_internal_property](#jerry_set_internal_property).

**Prototype**

```c
void
jerry_set_object_native_pointer (const jerry_value_t obj_val,
                                 void *native_p,
                                 const jerry_object_native_info_t *info_p)
```

- `obj_val` - object to set native pointer in.
- `native_p` - native pointer.
- `info_p` - native pointer's type information or NULL. When used, this should
             be a long-lived pointer, usually a pointer to a
             `static const jerry_object_native_info_t` makes most sense.

*New in version 2.0*: Changed from `jerry_set_object_native_handle`.

**Example**

See [jerry_get_object_native_pointer](#jerry_get_object_native_pointer) for a
best-practice example.

**See also**

- [jerry_create_object](#jerry_create_object)
- [jerry_get_object_native_pointer](#jerry_get_object_native_pointer)
- [jerry_object_native_info_t](#jerry_object_native_info_t)

## jerry_delete_object_native_pointer

**Summary**

Delete the native pointer of the specified object associated with the given native type info.

*Notes*:
 - If the specified object has no matching native pointer for the given native type info the operation has no effect.
 - The method does not invoke the free callback specified in the type info.
   If the native pointer should be freed then one must get the native pointer first and invoke the free callback manually
   before calling this method.
 - This operation cannot throw an exception.

**Prototype**

```c
bool
jerry_delete_object_native_pointer (const jerry_value_t obj_val,
                                    const jerry_object_native_info_t *info_p)
```

- `obj_val` - object to delete native pointer from.
- `info_p` - native pointer's type information.

*New in version 2.0*.

**Example**

See [jerry_get_object_native_pointer](#jerry_get_object_native_pointer) for a
best-practice example.

**See also**

- [jerry_create_object](#jerry_create_object)
- [jerry_get_object_native_pointer](#jerry_get_object_native_pointer)
- [jerry_get_object_native_pointer](#jerry_set_object_native_pointer)
- [jerry_object_native_info_t](#jerry_object_native_info_t)


## jerry_foreach_object_property

**Summary**

Applies the given function to every enumerable(!) property in the given object.

The "iterator" `foreach_p` method should return `true` value to continue the iteration.
If the method returns `false` the iteration will end.

**Prototype**

```c
bool
jerry_foreach_object_property (jerry_value_t obj_val,
                               jerry_object_property_foreach_t foreach_p,
                               void *user_data_p);
```

- `obj_val` - object value
- `foreach_p` - foreach function, that will be applied for each property
- `user_data_p` - user data for foreach function
- return value
  - true, if object fields traversal was performed successfully, i.e.:
    - no unhandled exceptions were thrown in object fields traversal
    - object fields traversal was stopped on callback that returned false
  - false, otherwise

**Example**


[doctest]: # (name="02.API-REFERENCE-foreach-property.c")

```c
#include <stdio.h>
#include "jerryscript.h"

/* Example structure used as user data for the property iteration. */
struct iteration_data {
  int string_property_count;
};

/*
 * Example foreach function to print out property names.
 */
static bool
foreach_function (const jerry_value_t prop_name,
                  const jerry_value_t prop_value,
                  void *user_data_p)
{
  if (jerry_value_is_string (prop_name)) {
    jerry_char_t string_buffer[128];
    jerry_size_t copied_bytes = jerry_substring_to_char_buffer (prop_name,
                                                                0,
                                                                127,
                                                                string_buffer,
                                                                127);
    string_buffer[copied_bytes] = '\0';

    printf ("Property: %s\n", string_buffer);

    struct iteration_data *data = (struct iteration_data *) user_data_p;
    data->string_property_count++;
  }

  /* return true to continue iteration */
  return true;
}

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  /* Construct an example object with a single property. */
  jerry_value_t object = jerry_create_object ();
  {
    jerry_value_t test_property = jerry_create_string ((const jerry_char_t *) "DemoProp");
    jerry_value_t test_value = jerry_create_number (3);
    /* By default all properties added to an object are enumerable. */
    jerry_value_t set_result = jerry_set_property (object, test_property, test_value);
    /* The `set_result` should be checked if it is an error or not. */
    jerry_release_value (set_result);
    jerry_release_value (test_value);
    jerry_release_value (test_property);
  }

  /* Iterate on the object's properties with the given user data. */
  struct iteration_data user_data = { 0 };

  bool iteration_result = jerry_foreach_object_property (object, foreach_function, &user_data);
  /* Check and process the `iteration_result` if required. */

  jerry_release_value (object);

  jerry_cleanup ();

  return user_data.string_property_count == 0;
}
```

**See also**

- [jerry_object_property_foreach_t](#jerry_object_property_foreach_t)

## jerry_objects_foreach

**Summary**

Iterate over all objects available in the engine.

The "iterator" `foreach_p` method should return `true` value to continue the search.
If the method returns `false` the search for the object is finished.

*Note*: Values obtained in `foreach_p` must be retained using [jerry_acquire_value](#jerry_acquire_value).

**Prototype**

```c
bool
jerry_objects_foreach (jerry_objects_foreach_t foreach_p,
                       void *user_data_p);
```

- `foreach_p` - function that will be invoked for each object.
- `user_data_p` - User data to pass to the function.
- return value
  - `true`, if the search function terminated the traversal by returning `false`
  - `false`, if the end of the list of objects was reached

*New in version 2.0*.

**Example**

[doctest]: # (name="02.API-REFERENCE-objects-foreach.c")

```c
#include <stdio.h>
#include "jerryscript.h"

/* Create a custom structure to guide the search and store the result. */
typedef struct
{
  jerry_value_t property_name;
  jerry_value_t result;
} find_my_object_info_t;

/*
 * Find the first object with the given property.
 */
static bool
find_my_object (const jerry_value_t candidate,
                void *user_data_p)
{
  find_my_object_info_t *info_p = (find_my_object_info_t *) user_data_p;

  /* Check if the given object has the required property. */
  jerry_value_t has_property = jerry_has_property (candidate, info_p->property_name);
  bool object_found = jerry_get_boolean_value (has_property);

  if (object_found)
  {
    /* We found it, so we acquire the value and record it. */
    info_p->result = jerry_acquire_value (candidate);
  }

  jerry_release_value (has_property);

  /* If the object was not found continue the search. */
  return !object_found;
} /* find_my_object */

int
main (void)
{
  int return_value = 0;

  /* Initialize JerryScript engine. */
  jerry_init (JERRY_INIT_EMPTY);

  /* Create the test object. */
  {
    jerry_value_t test_object = jerry_create_object ();

    {
      jerry_value_t test_property = jerry_create_string ((const jerry_char_t *) "DemoProp");
      jerry_value_t test_value = jerry_create_number (3);
      jerry_value_t set_result = jerry_set_property (test_object, test_property, test_value);
      /* The `set_result` should be checked if it is an error or not. */
      jerry_release_value (set_result);
      jerry_release_value (test_value);
      jerry_release_value (test_property);
    }

    {
      /* Register the test object into the global object. */
      jerry_value_t global_object = jerry_get_global_object ();
      jerry_value_t demo_property = jerry_create_string ((const jerry_char_t *) "DemoObject");
      jerry_value_t set_result = jerry_set_property (global_object, demo_property, test_object);
      /* The `set_result` should be checked if it is an error or not. */
      jerry_release_value (set_result);
      jerry_release_value (demo_property);
      jerry_release_value (global_object);
    }

    jerry_release_value (test_object);
  }

  /* Look up the test object base on a property name. */
  find_my_object_info_t search_info =
  {
    .property_name = jerry_create_string ((const jerry_char_t *) "DemoProp")
  };

  if (jerry_objects_foreach (find_my_object, &search_info))
  {
    /* The search was successful. Do something useful with search_info.result. */
    // ...
    printf ("Object found\n");

    /* Release the found object after we're done using it. */
    jerry_release_value (search_info.result);
  }
  else
  {
    /* The search has failed. */
    printf ("Object not found\n");

    return_value = 1;
  }

  jerry_release_value (search_info.property_name);

  /* Engine cleanup */
  jerry_cleanup ();
  return return_value;
}
```

**See also**

- [jerry_objects_foreach_t](#jerry_objects_foreach_t)

## jerry_objects_foreach_by_native_info

**Summary**

Iterate over all objects in the engine matching a certain native data type.

The "iterator" `foreach_p` method should return `true` value to continue the search.
If the method returns `false` the search for the object is finished.

*Note*: Values obtained in `foreach_p` must be retained using [jerry_acquire_value](#jerry_acquire_value).

**Prototype**

```c
bool
jerry_objects_foreach_by_native_info (const jerry_object_native_info_t *native_info_p,
                                      jerry_objects_foreach_by_native_info_t foreach_p,
                                      void *user_data_p);
```

- `native_info_p` - native pointer's type information.
- `foreach_p` - function that will be invoked for each object.
- return value
  - `true`, if the search function terminated the traversal by returning `false`
  - `false`, if the end of the list of objects was reached

*New in version 2.0*.

**Example**

[doctest]: # (name="02.API-REFERENCE-objects-foreach-nativeptr.c")

```c
#include <stdio.h>
#include <stdlib.h>
#include "jerryscript.h"

typedef struct
{
  int foo;
  bool bar;
} native_obj_t;

typedef struct
{
  jerry_value_t found_object;
  native_obj_t *found_native_data_p;

  int match_foo_value;
} find_object_data_t;

static void native_freecb (void *native_p)
{
  /* `native_p` was allocated via malloc. */
  free (native_p);
} /* native_freecb */

/*
 * NOTE: The address (!) of type_info acts as a way to uniquely "identify" the
 * C type `native_obj_t *`.
 */
static const jerry_object_native_info_t native_obj_type_info =
{
  .free_cb = native_freecb
};

/*
 * Function creating JS object that is "backed" by a `native_obj_t`.
 */
static void
add_object_with_nativeptr (int foo_value)
{
  // construct object and native_set value:
  jerry_value_t test_object = jerry_create_object ();
  native_obj_t *native_obj_p = malloc (sizeof (*native_obj_p));
  native_obj_p->foo = foo_value;
  native_obj_p->bar = true;

  jerry_set_object_native_pointer (test_object, native_obj_p, &native_obj_type_info);

  /* Register the test object into the global object. */
  jerry_value_t global_object = jerry_get_global_object ();
  jerry_value_t demo_property = jerry_create_string ((const jerry_char_t *) "DemoObject");
  jerry_value_t set_result = jerry_set_property (global_object, demo_property, test_object);
  /* The `set_result` should be checked if it is an error or not. */
  jerry_release_value (set_result);
  jerry_release_value (demo_property);
  jerry_release_value (global_object);

  jerry_release_value (test_object);
} /* create_object_with_nativeptr */

/*
 * Example native method that searches for a JavaScript object
 * with a `native_obj_type_info` has the correct value.
 */
static bool
find_object (const jerry_value_t candidate, void *data_p, void *user_data_p)
{
  find_object_data_t *find_data_p = (find_object_data_t *) user_data_p;
  native_obj_t *native_obj_p = (native_obj_t *) data_p;

  if (find_data_p->match_foo_value == native_obj_p->foo)
  {
    /* If the object was found, acquire it and store it in the user data. */
    find_data_p->found_object = jerry_acquire_value (candidate);
    find_data_p->found_native_data_p = native_obj_p;

    /* Stop traversing over the objects. */
    return false;
  }

  /* Indicate that the object was not found, so traversal must continue. */
  return true;
} /* find_object */

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  add_object_with_nativeptr (4);
  add_object_with_nativeptr (3);
  add_object_with_nativeptr (2);

  find_object_data_t find_data =
  {
    .match_foo_value = 3,
  };

  if (jerry_objects_foreach_by_native_info (&native_obj_type_info, find_object, &find_data))
  {
    /* The object was found and is now stored in `find_data.found_object`. After using it, it must be released. */
    printf ("Object found, native foo value: %d\n", find_data.found_native_data_p->foo);

    jerry_release_value (find_data.found_object);
  }
  else
  {
    printf ("Object not found\n");
  }

  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_create_object](#jerry_create_object)
- [jerry_set_object_native_pointer](#jerry_set_object_native_pointer)
- [jerry_get_object_native_pointer](#jerry_get_object_native_pointer)
- [jerry_object_native_info_t](#jerry_object_native_info_t)
- [jerry_objects_foreach](#jerry_objects_foreach)


# Input validator functions

## jerry_is_valid_utf8_string

**Summary**

Check if a given character buffer is a valid UTF-8 string.

**Notes**: Calling this method is safe in any time. It can be called
even before engine initialization.

**Prototype**

```c
bool
jerry_is_valid_utf8_string (const jerry_char_t *utf8_buf_p, /**< UTF-8 string */
                            jerry_size_t buf_size) /**< string size */
```

- `utf8_buf_p` - UTF-8 input string buffer.
- `buf_size` - input string buffer size in bytes.
- return value
  - true, if the provided string was a valid UTF-8 string.
  - false, if the string is not valid as an UTF-8 string.

*New in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  const jerry_char_t script[] = "print ('Hello, World!');";
  const jerry_size_t script_size = sizeof (script) - 1;

  if (jerry_is_valid_utf8_string (script, script_size))
  {
    jerry_run_simple (script, script_size, JERRY_INIT_EMPTY);
  }

  return 0;
}
```

**See also**

- [jerry_run_simple](#jerry_run_simple)
- [jerry_create_string_from_utf8](#jerry_create_string_from_utf8)
- [jerry_create_string_sz_from_utf8](#jerry_create_string_sz_from_utf8)
- [jerry_get_utf8_string_size](#jerry_get_utf8_string_size)
- [jerry_get_utf8_string_length](#jerry_get_utf8_string_length)
- [jerry_string_to_utf8_char_buffer](#jerry_string_to_utf8_char_buffer)
- [jerry_substring_to_utf8_char_buffer](#jerry_substring_to_utf8_char_buffer)

## jerry_is_valid_cesu8_string

**Summary**

Check if a given character buffer is a valid CESU-8 string.

**Notes**: Calling this method is safe in any time. It can be called
even before engine initialization.

**Prototype**

```c
bool
jerry_is_valid_cesu8_string (const jerry_char_t *cesu8_buf_p, /**< CESU-8 string */
                             jerry_size_t buf_size) /**< string size */
```

- `cesu8_buf_p` - CESU-8 input string buffer.
- `buf_size` - input string buffer size in bytes.
- return value
  - true, if the provided string was a valid CESU-8 string.
  - false, if the string is not valid as a CESU-8 string.

*New in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "Hello, World!";
  const jerry_size_t script_size = sizeof (script) - 1;

  if (jerry_is_valid_cesu8_string (script, script_size))
  {
    jerry_value_t string_value = jerry_create_string_sz (script,
                                                         script_size);

    // usage of string_value

    jerry_release_value (string_value);
  }

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_create_string](#jerry_create_string)
- [jerry_create_string_sz](#jerry_create_string_sz)
- [jerry_get_string_size](#jerry_get_string_size)
- [jerry_get_string_length](#jerry_get_string_length)
- [jerry_string_to_char_buffer](#jerry_string_to_char_buffer)
- [jerry_substring_to_char_buffer](#jerry_substring_to_char_buffer)


# Dynamic memory management functions

## jerry_heap_alloc

**Summary**

Allocate memory on the engine's heap.

*Note*: This function may take away memory from the executed JavaScript code.
If any other dynamic memory allocation API is available (e.g., libc malloc), it
should be used instead.

**Prototype**

```c
void *jerry_heap_alloc (size_t size);
```

- `size`: size of the memory block.
- return value: non-NULL pointer, if the memory is successfully allocated,
                NULL otherwise.

*New in version 2.0*.

**See also**

- [jerry_heap_free](#jerry_heap_free)

## jerry_heap_free

**Summary**

Free memory allocated on the engine's heap.

**Prototype**

```c
void jerry_heap_free (void *mem_p, size_t size);
```

- `mem_p`: value returned by `jerry_heap_alloc`.
- `size`: same size as passed to `jerry_heap_alloc`.

*New in version 2.0*.

**See also**

- [jerry_heap_alloc](#jerry_heap_alloc)


# External context functions

## jerry_create_context

**Summary**

Create an external JerryScript engine context.

**Prototype**

```c
jerry_context_t *
jerry_create_context (uint32_t heap_size,
                      jerry_context_alloc_t alloc,
                      void *cb_data_p);
```

- `heap_size` - requested heap size of the JerryScript context
- `alloc` - function for allocation
- `cb_data_p` - user data
- return value
  - pointer to the newly created JerryScript context if success
  - NULL otherwise.

*New in version 2.0*.

**Example**

[doctest]: # (test="compile", name="02.API-REFERENCE-create-context.c")

```c
#include <stdlib.h>
#include <pthread.h>

#include "jerryscript.h"
#include "jerryscript-port.h"

/* A different Thread Local Storage variable for each jerry context. */
__thread jerry_context_t *tls_context;

jerry_context_t *
jerry_port_get_current_context (void)
{
  /* Returns the context assigned to the thread. */
  return tls_context;
}

/* Allocate JerryScript heap for each thread. */
static void *
context_alloc_fn (size_t size, void *cb_data)
{
  (void) cb_data;
  return malloc (size);
}

static void *
thread_function (void *param)
{
  tls_context = jerry_create_context (512 * 1024,
                                      context_alloc_fn,
                                      NULL);
  jerry_init (JERRY_INIT_EMPTY);
  /* Run JerryScript in the context (e.g.: jerry_parse & jerry_run) */
  jerry_cleanup ();

  /* Deallocate JerryScript context */
  free (tls_context);

  return NULL;
}

#define NUM_OF_THREADS 8

int
main (void)
{
  pthread_t threads[NUM_OF_THREADS];

  /* Create the threads. */
  for (int i = 0; i < NUM_OF_THREADS; i++)
  {
    pthread_create (&threads[i], NULL, thread_function, (void *) (intptr_t) i);
  }

  /* Wait for the threads to complete, and release their resources. */
  for (int i = 0; i < NUM_OF_THREADS; i++)
  {
    pthread_join (threads[i], NULL);
  }

  return 0;
}
```

**See also**

- [jerry_context_t](#jerry_context_t)
- [jerry_context_alloc_t](#jerry_context_alloc_t)
- [jerry_port_get_current_context](05.PORT-API.md#jerry_port_get_current_context)


# Snapshot functions

## jerry_generate_snapshot

**Summary**

Generate snapshot from the specified source code.

*Notes*:
- Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
  is no longer needed.
- This API depends on a build option (`JERRY_SNAPSHOT_SAVE`) and can be checked in runtime with
  the `JERRY_FEATURE_SNAPSHOT_SAVE` feature enum value, see [jerry_is_feature_enabled](#jerry_is_feature_enabled).
  If the feature is not enabled the function will return an error.

**Prototype**

```c
jerry_value_t
jerry_generate_snapshot (const jerry_char_t *resource_name_p,
                         size_t resource_name_length,
                         const jerry_char_t *source_p,
                         size_t source_size,
                         uint32_t generate_snapshot_opts,
                         uint32_t *buffer_p,
                         size_t buffer_size);
```

- `resource_name_p` - resource (file) name of the source code. Currently unused, the debugger may use it in the future.
- `resource_name_length` - length of resource name.
- `source_p` - script source, it must be a valid utf8 string.
- `source_size` - script source size, in bytes.
- `generate_snapshot_opts` - any combination of [jerry_generate_snapshot_opts_t](#jerry_generate_snapshot_opts_t) flags.
- `buffer_p` - output buffer (aligned to 4 bytes) to save snapshot to.
- `buffer_size` - the output buffer's size in bytes.
- return value
  - the size of the generated snapshot in bytes as number value, if it was generated succesfully (i.e. there
    are no syntax errors in source code, buffer size is sufficient, and snapshot support is enabled in
    current configuration through JERRY_SNAPSHOT_SAVE)
  - thrown error, otherwise.

*New in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  static uint32_t global_mode_snapshot_buffer[256];
  const jerry_char_t script_to_snapshot[] = "(function () { return 'string from snapshot'; }) ();";

  jerry_value_t generate_result;
  generate_result = jerry_generate_snapshot (NULL,
                                             0,
                                             script_to_snapshot,
                                             sizeof (script_to_snapshot) - 1,
                                             0,
                                             global_mode_snapshot_buffer,
                                             sizeof (global_mode_snapshot_buffer) / sizeof (uint32_t));

  if (!jerry_value_is_error (generate_result))
  {
    size_t snapshot_size = (size_t) jerry_get_number_value (generate_result);
  }

  jerry_release_value (generate_result);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_generate_function_snapshot](#jerry_generate_function_snapshot)
- [jerry_exec_snapshot](#jerry_exec_snapshot)


## jerry_generate_function_snapshot

**Summary**

Generate function snapshot from the specified source code
with the given arguments.

The function arguments and function body are
passed as separated arguments.

*Notes*:
- Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
  is no longer needed.
- This API depends on a build option (`JERRY_SNAPSHOT_SAVE`) and can be checked in runtime with
  the `JERRY_FEATURE_SNAPSHOT_SAVE` feature enum value, see [jerry_is_feature_enabled](#jerry_is_feature_enabled).
  If the feature is not enabled the function will return an error.

**Prototype**

```c
jerry_value_t
jerry_generate_function_snapshot (const jerry_char_t *resource_name_p,
                                  size_t resource_name_length,
                                  const jerry_char_t *source_p,
                                  size_t source_size,
                                  const jerry_char_t *args_p,
                                  size_t args_size,
                                  uint32_t generate_snapshot_opts,
                                  uint32_t *buffer_p,
                                  size_t buffer_size)
```

- `resource_name_p` - resource (file) name of the source code. Currently unused, the debugger may use it in the future.
- `resource_name_length` - length of resource name.
- `source_p` - script source, it must be a valid utf8 string.
- `source_size` - script source size, in bytes.
- `args_p` - function arguments, it must be a valid utf8 string.
- `args_size` - function argument size, in bytes.
- `generate_snapshot_opts` - any combination of [jerry_generate_snapshot_opts_t](#jerry_generate_snapshot_opts_t) flags.
- `buffer_p` - buffer (aligned to 4 bytes) to save snapshot to.
- `buffer_size` - the buffer's size in bytes.
- return value
  - the size of the generated snapshot in bytes as number value, if it was generated succesfully (i.e. there
    are no syntax errors in source code, buffer size is sufficient, and snapshot support is enabled in
    current configuration through JERRY_SNAPSHOT_SAVE)
  - thrown error, otherwise.

*New in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  static uint32_t func_snapshot_buffer[256];
  const jerry_char_t args[] = "a, b";
  const jerry_char_t src[] = "return a + b;";

  jerry_value_t generate_result;
  generate_result = jerry_generate_function_snapshot (NULL,
                                                      0,
                                                      src,
                                                      sizeof (src) - 1,
                                                      args,
                                                      sizeof (args) - 1,
                                                      0,
                                                      func_snapshot_buffer,
                                                      sizeof (func_snapshot_buffer) / sizeof (uint32_t));

  if (!jerry_value_is_error (generate_result))
  {
    size_t snapshot_size = (size_t) jerry_get_number_value (generate_result);
  }

  jerry_release_value (generate_result);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_generate_snapshot](#jerry_generate_snapshot)
- [jerry_load_function_snapshot_at](#jerry_load_function_snapshot_at)


## jerry_exec_snapshot

**Summary**

Execute snapshot from the specified buffer.

*Notes*:
- Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
  is no longer needed.
- This API depends on a build option (`JERRY_SNAPSHOT_EXEC`) and can be checked in runtime with
  the `JERRY_FEATURE_SNAPSHOT_EXEC` feature enum value, see [jerry_is_feature_enabled](#jerry_is_feature_enabled).
  If the feature is not enabled the function will return an error.

**Prototype**

```c
jerry_value_t
jerry_exec_snapshot (const uint32_t *snapshot_p,
                     size_t snapshot_size,
                     size_t func_index,
                     uint32_t exec_snapshot_opts);
```

- `snapshot_p` - pointer to snapshot.
- `snapshot_size` - size of snapshot in bytes.
- `func_index` - index of executed function.
- `exec_snapshot_opts` - any combination of [jerry_exec_snapshot_opts_t](#jerry_exec_snapshot_opts_t) flags.
- return value
  - result of bytecode, if run was successful.
  - thrown error, otherwise (an error is reported if the snapshot execution feature is not enabled).

*Changed in version 2.0*: Added `func_index` and `exec_snapshot_opts` arguments. Removed the `copy_bytecode` last argument.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  static uint32_t global_mode_snapshot_buffer[256];
  const jerry_char_t script_to_snapshot[] = "(function () { return 'string from snapshot'; }) ();";

  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t generate_result;
  generate_result = jerry_generate_snapshot (NULL,
                                             0,
                                             script_to_snapshot,
                                             sizeof (script_to_snapshot) - 1,
                                             0,
                                             global_mode_snapshot_buffer,
                                             sizeof (global_mode_snapshot_buffer) / sizeof (uint32_t));
  // generate_result should be checked if it is an error or not

  size_t global_mode_snapshot_size = (size_t) jerry_get_number_value (generate_result);
  jerry_release_value (generate_result);

  jerry_cleanup ();

  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t res = jerry_exec_snapshot (global_mode_snapshot_buffer,
                                           global_mode_snapshot_size,
                                           0,
                                           0);

  // check the `res` value for error and process the result.

  jerry_release_value (res);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_parse_and_save_snapshot](#jerry_parse_and_save_snapshot)


## jerry_load_function_snapshot

**Summary**

Load the selected snapshot function from the specified buffer as a function object.

The lexical environment of the loaded function is always the global lexical environment.

*Notes*:
- Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
  is no longer needed.
- This API depends on a build option (`JERRY_SNAPSHOT_EXEC`) and can be checked in runtime with
  the `JERRY_FEATURE_SNAPSHOT_EXEC` feature enum value, see [jerry_is_feature_enabled](#jerry_is_feature_enabled).
  If the feature is not enabled the function will return an error.

**Prototype**

```c
jerry_value_t
jerry_load_function_snapshot (const uint32_t *snapshot_p,
                              size_t snapshot_size,
                              size_t func_index,
                              uint32_t exec_snapshot_opts);
```

- `snapshot_p` - pointer to snapshot.
- `snapshot_size` - size of snapshot in bytes.
- `func_index` - index of function to load from the snapshot.
- `exec_snapshot_opts` - any combination of [jerry_exec_snapshot_opts_t](#jerry_exec_snapshot_opts_t) flags.
- return value
  - function object built from the snapshot.
  - thrown error, otherwise.

*New in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  static uint32_t snapshot_buffer[256];
  const jerry_char_t func_args[] = "a, b";
  const jerry_char_t func_src[] = "return a + b;";

  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t generate_result;
  generate_result = jerry_generate_function_snapshot (NULL,
                                                      0,
                                                      func_src,
                                                      sizeof (func_src) - 1,
                                                      func_args,
                                                      sizeof (func_args) - 1,
                                                      false,
                                                      snapshot_buffer,
                                                      sizeof (snapshot_buffer) / sizeof (uint32_t));

  size_t snapshot_size = (size_t) jerry_get_number_value (generate_result);
  jerry_release_value (generate_result);

  jerry_cleanup ();

  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t func = jerry_load_function_snapshot (snapshot_buffer,
                                                     snapshot_size,
                                                     0,
                                                     0);
  /* 'func' can be used now as a function object */

  jerry_value_t this_value = jerry_create_undefined ();
  jerry_value_t args[2];
  args[0] = jerry_create_number (1.0);
  args[1] = jerry_create_number (2.0);

  jerry_value_t res = jerry_call_function (func, this_value, args, 2);

  /* 'res' now contains the value 3 as a jerry_value_t */

  jerry_release_value (args[0]);
  jerry_release_value (args[1]);
  jerry_release_value (this_value);
  jerry_release_value (func);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_parse_and_save_function_snapshot](#jerry_parse_and_save_function_snapshot)


## jerry_get_literals_from_snapshot

**Summary**

Collect the used literals from the given snapshot and save them into a buffer in list or C format.
None of these literals are magic strings. In C format only valid identifiers are collected.

*Note*:
- This API depends on a build option (`JERRY_SNAPSHOT_SAVE`) and can be checked in runtime with
  the `JERRY_FEATURE_SNAPSHOT_SAVE` feature enum value, see [jerry_is_feature_enabled](#jerry_is_feature_enabled).
  If the feature is not enabled the function will return zero.

**Prototype**

```c
size_t
jerry_get_literals_from_snapshot (const uint32_t *snapshot_p,
                                  size_t snapshot_size,
                                  jerry_char_t *lit_buf_p,
                                  size_t lit_buf_size,
                                  bool is_c_format);
```

- `snapshot_p` - input snapshot buffer.
- `snapshot_size` - size of snapshot in bytes.
- `lit_buf_p` - buffer to save literals to.
- `lit_buf_size` - the buffer's size.
- `is_c_format` - the output format would be C-style (true) or a simple list (false).
- return value
  - the size of the literal-list, if it was generated succesfully (i.e. the list of literals isn't empty,
    and literal-save support is enabled in current configuration through JERRY_SNAPSHOT_SAVE)
  - 0 otherwise.

*New in version 2.0*.

**Example**

[doctest]: # (test="link")

```c
#include <stdio.h>
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  static jerry_char_t literal_buffer[256];
  static uint32_t snapshot_buffer[256];
  const jerry_char_t script_for_literal_save[] = "var obj = { a:'aa', bb:'Bb' }";

  jerry_value_t generate_result = jerry_generate_snapshot (NULL,
                                                           0,
                                                           script_for_literal_save,
                                                           sizeof (script_for_literal_save) - 1,
                                                           0,
                                                           snapshot_buffer,
                                                           256);
  size_t snapshot_size = (size_t) jerry_get_number_value (generate_result);
  jerry_release_value (generate_result);

  const size_t literal_size = jerry_get_literals_from_snapshot (snapshot_buffer,
                                                                snapshot_size,
                                                                literal_buffer,
                                                                256,
                                                                true);

  if (literal_size != 0)
  {
    FILE *literal_file_p = fopen ("literals.h", "wb");
    fwrite (literal_buffer, sizeof (uint8_t), literal_size, literal_file_p);
    fclose (literal_file_p);
  }

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_register_magic_strings](#jerry_register_magic_strings)


# Miscellaneous functions

## jerry_set_vm_exec_stop_callback

**Summary**

When JERRY_FEATURE_VM_EXEC_STOP is enabled a callback function can be
specified by this function. This callback is periodically called when
JerryScript executes an ECMAScript program.

If the callback returns with undefined value the ECMAScript execution
continues. Otherwise the result is thrown by the engine (if the error
flag is not set for the returned value the engine automatically sets
it). The callback function might be called again even if it threw
an error. In this case the function must throw the same error again.

To reduce the CPU overhead of constantly checking the termination
condition the callback is called when a backward jump is executed
or an exception is caught. Setting the `frequency` to a greater
than `1` value reduces this overhead further. If its value is N
only every Nth event (backward jump, etc.) trigger the next check.


**Prototype**

```c
void
jerry_set_vm_exec_stop_callback (jerry_vm_exec_stop_callback_t stop_cb,
                                 void *user_p,
                                 uint32_t frequency);
```

- `stop_cb` - periodically called callback (passing NULL disables this feature)
- `user_p` - user pointer passed to the `stop_cb` function
- `frequency` - frequency of calling the `stop_cb` function

*New in version 2.0*.

**Example**

[doctest]: # (test="link")

```c
#include "jerryscript.h"

static int countdown = 10;

static jerry_value_t
vm_exec_stop_callback (void *user_p)
{
  while (countdown > 0)
  {
    countdown--;
    return jerry_create_undefined ();
  }

  // The error flag is added automatically.
  return jerry_create_string ((const jerry_char_t *) "Abort script");
}

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_set_vm_exec_stop_callback (vm_exec_stop_callback, &countdown, 16);

  // Inifinte loop.
  const jerry_char_t script[] = "while(true) {}";

  jerry_value_t parsed_code = jerry_parse (NULL, 0, script, sizeof (script) - 1, JERRY_PARSE_NO_OPTS);
  jerry_release_value (jerry_run (parsed_code));
  jerry_release_value (parsed_code);
  jerry_cleanup ();
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_parse](#jerry_parse)
- [jerry_run](#jerry_run)
- [jerry_vm_exec_stop_callback_t](#jerry_vm_exec_stop_callback_t)

## jerry_get_backtrace

**Summary**

Get backtrace. The backtrace is an array of strings where
each string contains the position of the corresponding frame.
The array length is zero if the backtrace is not available.

This function is typically called from native callbacks.

*Notes*:
- Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.
- This feature depends on build option (`JERRY_LINE_INFO`) and can be checked
  in runtime with the `JERRY_FEATURE_LINE_INFO` feature enum value,
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).

**Prototype**

```c
jerry_value_t
jerry_get_backtrace (uint32_t max_depth);
```

- `max_depth` - backtrace collection stops after reaching this value, 0 = unlimited
- return value
  - a newly constructed JS array

*New in version 2.0*.

**Example**

[doctest]: # (name="02.API-REFERENCE-jsbacktrace.c")

```c
#include <stdio.h>
#include <string.h>
#include "jerryscript.h"

static jerry_value_t
backtrace_handler (const jerry_value_t function_obj,
                   const jerry_value_t this_val,
                   const jerry_value_t args_p[],
                   const jerry_length_t args_count)
{
  if (!jerry_is_feature_enabled (JERRY_FEATURE_LINE_INFO))
  {
    printf ("Line info disabled, no backtrace will be printed\n");
  }

  /* If the line info feature is disabled an empty array will be returned. */
  jerry_value_t backtrace_array = jerry_get_backtrace (5);
  uint32_t array_length = jerry_get_array_length (backtrace_array);

  for (uint32_t idx = 0; idx < array_length; idx++)
  {
    jerry_value_t property = jerry_get_property_by_index (backtrace_array, idx);

    jerry_char_t string_buffer[64];
    jerry_size_t copied_bytes = jerry_substring_to_char_buffer (property,
                                                                0,
                                                                63,
                                                                string_buffer,
                                                                63);
    string_buffer[copied_bytes] = '\0';
    printf(" %d: %s\n", idx, string_buffer);

    jerry_release_value (property);
  }

  jerry_release_value (backtrace_array);

  return jerry_create_undefined ();
} /* backtrace_handler */

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global = jerry_get_global_object ();

  /* Register the "dump_backtrace" method. */
  {
    jerry_value_t func = jerry_create_external_function (backtrace_handler);
    jerry_value_t name = jerry_create_string ((const jerry_char_t *) "backtrace");
    jerry_value_t result = jerry_set_property (global, name, func);
    jerry_release_value (result);
    jerry_release_value (name);
    jerry_release_value (func);
  }

  jerry_release_value (global);

  const char *source = ("function f() { g (); }\n"
                        "function g() { h (); }\n"
                        "function h() { backtrace (); }\n"
                        "f ();\n");
  const char *resource = "demo_memoryjs";

  jerry_value_t program = jerry_parse ((const jerry_char_t *) resource,
                                       strlen (resource),
                                       (const jerry_char_t *) source,
                                       strlen (source),
                                       JERRY_PARSE_NO_OPTS);
  if (!jerry_value_is_error (program))
  {
    jerry_value_t run_result = jerry_run (program);
    jerry_release_value (run_result);
  }

  jerry_release_value (program);
  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_create_external_function](#jerry_create_external_function)


## jerry_get_resource_name

**Summary**

Get the resource name (usually a file name) of the currently executed script or the given function object.

This function is typically called from native callbacks.

*Notes*:
- Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.
- This feature depends on build option (`JERRY_LINE_INFO`) and can be checked
  in runtime with the `JERRY_FEATURE_LINE_INFO` feature enum value,
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).

**Prototype**

```c
jerry_value_t
jerry_get_resource_name (jerry_value_t value);
```
- `value` - api value to obtain the resource name from
- return string value constructed from
  - the currently executed function object's resource name, if the given value is undefined
  - resource name of the function object, if the given value is a function object
  - "<anonymous>", otherwise

*New in version 2.2*.

**Example**

[doctest]: # (name="02.API-REFERENCE-jsresourcename.c")

```c
#include <stdio.h>
#include <string.h>
#include "jerryscript.h"

static jerry_value_t
resource_name_handler (const jerry_value_t function_obj,
                       const jerry_value_t this_val,
                       const jerry_value_t args_p[],
                       const jerry_length_t args_count)
{
  jerry_value_t undefined_value = jerry_create_undefined ();
  jerry_value_t resource_name = jerry_get_resource_name (args_count > 0 ? args_p[0] : undefined_value);
  jerry_release_value (undefined_value);

  return resource_name;
} /* resource_name_handler */

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global = jerry_get_global_object ();

  /* Register the "resourceName" method. */
  {
    jerry_value_t func = jerry_create_external_function (resource_name_handler);
    jerry_value_t name = jerry_create_string ((const jerry_char_t *) "resourceName");
    jerry_value_t result = jerry_set_property (global, name, func);
    jerry_release_value (result);
    jerry_release_value (name);
    jerry_release_value (func);
  }

  jerry_release_value (global);

  const jerry_char_t source[] = "function myFunction() { return resourceName() }; myFunction()";
  const jerry_char_t resource[] = "demo.js";

  jerry_value_t program = jerry_parse (resource,
                                       sizeof (resource) - 1,
                                       source,
                                       sizeof (source) - 1,
                                       JERRY_PARSE_NO_OPTS);

  if (!jerry_value_is_error (program))
  {
    /* `run_result` contains "demo.js" */
    jerry_value_t run_result = jerry_run (program);

    /* usage of `run_result` */

    jerry_release_value (run_result);
  }

  jerry_release_value (program);
  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_create_external_function](#jerry_create_external_function)

## jerry_get_new_target

**Summary**

Returns the current "new.target" JavaScript function at the call site.

If used outside of a native C function it will return "undefined" value.

*Notes*:
- Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.
- This feature depends on build option (`JERRY_ESNEXT`) and can be checked
  in runtime with the `JERRY_FEATURE_SYMBOL` feature enum value (as symbols are enabled in case of ES.next),
  see: [jerry_is_feature_enabled](#jerry_is_feature_enabled).
- If the ES.next mode is not enabled this method will always return the "undefined" value.

**Prototype**

```c
jerry_value_t
jerry_get_new_target (void);
```
- return
  - "undefined" - if at the call site it was not a constructor call.
  - function object - if the current call site is in a constructor call.

*New in version 2.2*.

**Example 1**

[doctest]: # (name="02.API-REFERENCE-jsnewtarget-01.c")

```c
#include <stdio.h>
#include <string.h>
#include <jerryscript.h>

static jerry_value_t
demo_handler (const jerry_value_t func_obj_val,
              const jerry_value_t this_val,
              const jerry_value_t args_p[],
              const jerry_length_t args_cnt)
{
  jerry_value_t new_target = jerry_get_new_target ();

  /* new_target is the "demo" JS function object */
  if (jerry_value_get_type (new_target) == JERRY_TYPE_FUNCTION)
  {
    printf ("This is a construct call\r\n");
  }

  jerry_release_value (new_target);

  return jerry_create_undefined ();
}

int
main (int argc, char** argv)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t function_val = jerry_create_external_function (demo_handler);

  jerry_value_t ret_val = jerry_construct_object (function_val, NULL, 0);

  jerry_release_value (ret_val);
  jerry_release_value (function_val);

  jerry_cleanup ();
  return 0;
}
```

**Example 2**

[doctest]: # (name="02.API-REFERENCE-jsnewtarget-02.c")

```c
#include <stdio.h>
#include <string.h>
#include <jerryscript.h>

static jerry_value_t
demo_handler (const jerry_value_t func_obj_val,
              const jerry_value_t this_val,
              const jerry_value_t args_p[],
              const jerry_length_t args_cnt)
{
  jerry_value_t new_target = jerry_get_new_target ();

  /* new_target is a JS function object */
  if (jerry_value_get_type (new_target) == JERRY_TYPE_FUNCTION)
  {
    printf ("This is a construct call\r\n");
  }

  jerry_release_value (new_target);

  return jerry_create_undefined ();
}

int
main (int argc, char** argv)
{
  jerry_init (JERRY_INIT_EMPTY);

  /* register C method */
  jerry_value_t global_obj_val = jerry_get_global_object ();
  jerry_value_t function_val = jerry_create_external_function (demo_handler);
  jerry_value_t function_name_val = jerry_create_string ((const jerry_char_t *) "demo");
  jerry_value_t result_val = jerry_set_property (global_obj_val, function_name_val, function_val);
  jerry_release_value (result_val);
  jerry_release_value (function_name_val);
  jerry_release_value (function_val);
  jerry_release_value (global_obj_val);

  /* Invoke C method via JS */
  const char *src = "new demo ()";
  jerry_value_t ret_val = jerry_eval ((const jerry_char_t *) src,
                                      strlen (src),
                                      JERRY_PARSE_NO_OPTS);

  jerry_release_value (ret_val);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_construct_object](#jerry_construct_object)

# ArrayBuffer and TypedArray functions

These APIs all depend on the es.next profile.

## jerry_get_arraybuffer_byte_length

**Summary**

Get the byte length property of the ArrayBuffer. This is the
same value which was passed to the ArrayBuffer constructor call.

**Prototype**

```c
jerry_length_t
jerry_get_arraybuffer_byte_length (const jerry_value_t value);
```

- `value` - ArrayBuffer object
- return value
  - size of the ArrayBuffer in bytes
  - 0 if the `value` parameter is not an ArrayBuffer

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t buffer = jerry_create_arraybuffer (15);
  jerry_length_t length = jerry_get_arraybuffer_byte_length (buffer);
  // length should be 15

  jerry_release_value (buffer);
}
```

**See also**
- [jerry_create_arraybuffer](#jerry_create_arraybuffer)


## jerry_arraybuffer_read

**Summary**

Copy the portion of the ArrayBuffer into a user provided buffer.
The start offset of the read operation can be specified.

The number bytes to be read can be specified via the `buf_size`
parameter. It is not possible to read more than the length of
the ArrayBuffer.

Function returns the number of bytes read from the ArrayBuffer
(and written to the buffer parameter). This value is
calculated in the following way: `min(array buffer length - offset, buf_size)`.

**Prototype**

```c
jerry_length_t
jerry_arraybuffer_read (const jerry_value_t value,
                        jerry_length_t offset,
                        uint8_t *buf_p,
                        jerry_length_t buf_size);
```

- `value` - ArrayBuffer to read from
- `offset` - start offset of the read operation
- `buf_p` - buffer to read the data to
- `buf_size` - maximum number of bytes to read into the buffer
- return value
  - number of bytes written into the buffer (read from the ArrayBuffer)
  - 0 if the `value` is not an ArrayBuffer object
  - 0 if the `buf_size` is zero or there is nothing to read

*New in version 2.0*.

**Example**

```c
{
  uint8_t data[20];
  jerry_value_t buffer;
  // ... create the ArrayBuffer or acuiqre it from somewhere.

  jerry_value_t bytes_read;

  // read 10 bytes from the start of the ArrayBuffer.
  bytes_read = jerry_arraybuffer_read (buffer, 0, data, 10);
  // read the next 10 bytes
  bytes_read += jerry_arraybuffer_read (buffer, bytes_read, data + bytes_read, 10);

  // process the data variable

  jerry_release_value (buffer);
}
```

**See also**

- [jerry_create_arraybuffer](#jerry_create_arraybuffer)
- [jerry_arraybuffer_write](#jerry_arraybuffer_write)
- [jerry_get_arraybuffer_byte_length](#jerry_get_arraybuffer_byte_length)


## jerry_arraybuffer_write

**Summary**

Copy the contents of a buffer into the ArrayBuffer.
The start offset of the write operation can be specified.

The number bytes to be written can be specified via the `buf_size`
parameter. It is not possible to write more than the length of
the ArrayBuffer.

Function returns the number of bytes written into the ArrayBuffer
(and read from the buffer parameter). This value is
calculated in the following way: `min(array buffer length - offset, buf_size)`.

**Prototype**

```c
jerry_length_t
jerry_arraybuffer_write (const jerry_value_t value,
                         jerry_length_t offset,
                         const uint8_t *buf_p,
                         jerry_length_t buf_size);
```

- `value` - ArrayBuffer to write to
- `offset` - start offset of the write operation
- `buf_p` - buffer to read the data from
- `buf_size` - maximum number of bytes to write into the ArrayBuffer
- return value
  - number of bytes written into the ArrayBuffer (read from the buffer parameter)
  - 0 if the `value` is not an ArrayBuffer object
  - 0 if the `buf_size` is zero or there is nothing to write

*New in version 2.0*.

**Example**

```c
{
  uint8_t data[20];

  // fill the data with values
  for (int i = 0; i < 20; i++)
  {
    data[i] = (uint8_t) (i * 2);
  }

  jerry_value_t buffer;
  // ... create the ArrayBuffer or acquire it from somewhere.

  jerry_value_t bytes_written;

  // write 10 bytes from to the start of the ArrayBuffer.
  bytes_written = jerry_arraybuffer_write (buffer, 0, data, 10);
  // read the next 10 bytes
  bytes_written += jerry_arraybuffer_write (buffer, bytes_written, data + bytes_written, 10);

  // use the ArrayBuffer

  jerry_release_value (buffer);
}
```

**See also**

- [jerry_create_arraybuffer](#jerry_create_arraybuffer)
- [jerry_arraybuffer_write](#jerry_arraybuffer_write)
- [jerry_get_arraybuffer_byte_length](#jerry_get_arraybuffer_byte_length)


## jerry_get_arraybuffer_pointer

**Summary**

The function allows access to the contents of the Array Buffer directly.

**WARNING!** This operation is for expert use only! The programmer must
ensure that the returned memory area is used correctly. That is
there is no out of bounds reads or writes. The lifetime of the underlying
data buffer is managed by the ArrayBuffer value. Make sure to acquire the
value with [`jerry_acquire_value`](#jerry_acquire_value) if the data
buffer is needed later.

**Prototype**

```c
uint8_t *
jerry_get_arraybuffer_pointer (const jerry_value_t value);
```

- `value` - Array Buffer object.
- return value
  - pointer to the Array Buffer's data area.
  - NULL if the `value` is not an Array Buffer object.

*New in version 2.0*.

**Example**

```c
{
  // create the ArrayBuffer
  jerry_value_t buffer = jerry_create_arraybuffer (16);

  uint8_t *const data = jerry_get_arraybuffer_pointer (buffer);

  for (int i = 0; i < 16; i++)
  {
    data[i] = (uint8_t) (i + 4);
  }

  // use the Array Buffer

  // release buffer as it is not needed after this point
  jerry_release_value (buffer);
}
```

**See also**

- [jerry_create_arraybuffer_external](#jerry_create_arraybuffer_external)

## jerry_is_arraybuffer_detachable

**Summary**

Get if the ArrayBuffer is detachable.

**Prototype**

```c
jerry_value_t
jerry_is_arraybuffer_detachable (const jerry_value_t value);
```

- `value` - ArrayBuffer to be detached
- return
  - boolean value if success
  - Error otherwise

*New in version 2.2*.

**Example**

```c
{
  // create the ArrayBuffer
  jerry_value_t buffer = jerry_create_arraybuffer (16);

  jerry_value_t res = jerry_is_arraybuffer_detachable (buffer);
  bool is_detachable = jerry_get_boolean_value (res);

  // release buffer as it is not needed after this point
  jerry_release_value (res);
  jerry_release_value (buffer);
}
```

**See also**

- [jerry_detach_arraybuffer](#jerry_detach_arraybuffer)

## jerry_detach_arraybuffer

**Summary**

Detach the underlying data block from ArrayBuffer and set its bytelength to 0.

This operation requires the ArrayBuffer to be external that created by
`jerry_create_arraybuffer_external`.

**Prototype**

```c
jerry_value_t
jerry_detach_arraybuffer (const jerry_value_t value);
```

- `value` - ArrayBuffer to be detached
- return
  - null value if success
  - Error otherwise

*New in version 2.2*.

**Example**

```c
{
  uint8_t buf[1];
  jerry_size_t length = 1;
  // create the ArrayBuffer
  jerry_value_t buffer = jerry_create_arraybuffer (length, buf, NULL);

  jerry_value_t res = jerry_detach_arraybuffer (buffer);

  // release buffer as it is not needed after this point
  jerry_release_value (res);
  jerry_release_value (buffer);
}
```

**See also**

- [jerry_is_arraybuffer_detachable](#jerry_is_arraybuffer_detachable)

## jerry_get_dataview_buffer

**Summary**

Get the ArrayBuffer object used by a DataView object.
Additionally returns the byteLength and byteOffset properties
of the DataView object.

For the returned ArrayBuffer the [jerry_release_value](#jerry_release_value)
must be called when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_get_dataview_buffer (const jerry_value_t value,
                           jerry_length_t *byteOffset,
                           jerry_length_t *byteLength);
```

- `value` - DataView to get the ArrayBuffer from
- `byteOffset` - (Optional) returns the start offset of the ArrayBuffer for the DataView
- `byteLength` - (Optional) returns the number of bytes used from the ArrayBuffer for the DataView
- return
  - DataView object's underlying ArrayBuffer object
  - TypeError if the `value` is not a DataView object

*New in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t arraybuffer = jerry_create_arraybuffer (16);
  jerry_value_t dataview = jerry_create_dataview (arraybuffer, 0, 16);
  jerry_length_t byteOffset = 0;
  jerry_length_t byteLength = 0;
  jerry_value_t buffer = jerry_get_dataview_buffer (dataview, &byteOffset, &byteLength);

  // buffer is an ArrayBuffer object and ArrayBuffer operations can be performed on it
  // byteOffset is 0
  // byteLength is 16

  // usage of buffer

  jerry_release_value (buffer);
  jerry_release_value (dataview);
  jerry_release_value (arraybuffer);

  jerry_cleanup ();
}
```

**See also**

- [jerry_create_dataview](#jerry_create_dataview)


## jerry_get_typedarray_type

**Summary**

Get the type of the TypedArray.

The returned type is one of the [jerry_typedarray_type_t](#jerry_typedarray_type_t)
enum value.

**Prototype**

```c
jerry_typedarray_type_t
jerry_get_typedarray_type (jerry_value_t value);
```

- `value` - TypedArray object to query for type.
- return
  - the type of the TypedArray
  - JERRY_TYPEDARRAY_INVALID if the object was not a TypedArray

*New in version 2.0*.

**Example**

```c
{
  jerry_typedarray_type_t expected_type = JERRY_TYPEDARRAY_UINT32;
  jerry_value_t typedarray = jerry_create_typedarray (expected_klass, 25);

  jerry_typedarray_type_t type = jerry_get_typedarray_type (typedarray);

  // 'type' is now JERRY_TYPEDARRAY_UINT32

  jerry_release_value (typedarray);
}
```

**See also**

- [jerry_create_typedarray](#jerry_create_typedarray)
- [jerry_typedarray_type_t](#jerry_typedarray_type_t)


## jerry_get_typedarray_length

**Summary**

Get the element count of the TypedArray as specified during creation.

This is not the same as the byteLength property of a TypedArray object.

**Prototype**

```
jerry_length_t
jerry_get_typedarray_length (jerry_value_t value);
```

- `value` - TypedArray object to query
- return
  - length (element count) of the TypedArray object
  - 0 if the object is not a TypedArray

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t array = jerry_create_typedarray (JERRY_TYPEDARRAY_INT32, 21);

  jerry_length_t element_count = jerry_get_typedarray_length (array);

  // element_count is now 21.

  jerry_release_value (array);
}
```

**See also**

- [jerry_create_typedarray](#jerry_create_typedarray)


## jerry_get_typedarray_buffer

**Summary**

Get the ArrayBuffer object used by a TypedArray object.
Additionally returns the byteLength and byteOffset properties
of the TypedArray object.

For the returned ArrayBuffer the [jerry_release_value](#jerry_release_value)
must be called.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_get_typedarray_buffer (jerry_value_t value,
                             jerry_length_t *byteOffset,
                             jerry_length_t *byteLength);
```

- `value` - TypedArray to get the ArrayBuffer from
- `byteOffset` - (Optional) returns the start offset of the ArrayBuffer for the TypedArray
- `byteLength` - (Optional) returns the number of bytes used from the ArrayBuffer for the TypedArray
- return
  - TypedArray object's underlying ArrayBuffer object
  - TypeError if the `value` is not a TypedArray object

*New in version 2.0*.

**Example**

```c
{
  jerry_value_t array = jerry_create_typedarray (JERRY_TYPEDARRAY_INT16, 11);

  jerry_length_t byteLength = 0;
  jerry_length_t byteOffset = 0;
  jerry_value_t buffer = jerry_get_typedarray_buffer (array, &byteOffset, &byteLength);

  // buffer is an ArrayBuffer object and ArrayBuffer operations can be performed on it
  // byteLength is 11 * 2  (2 as the TypedArray stores Int16 that is 2 byte elements)
  // byteOffset is 0

  jerry_release_value (buffer);
  jerry_release_value (array);
}
```

**See also**

- [jerry_create_typedarray](#jerry_create_typedarray)

# JSON functions

## jerry_json_parse

**Summary**

Parses a JSON string creating a JavaScript value. The behaviour is equivalent with
the "JSON.parse(string)" JS call.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_json_parse (const jerry_char_t *string_p,
                  jerry_size_t string_size);
```

- `string_p` - pointer to a JSON string.
- `string_size` - size of the string.
- return
  - `jerry_value_t` containing a JavaScript value.
  - Error value in case of any parse error.

*New in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  /* Initialize engine */
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t data[] = "{\"name\": \"John\", \"age\": 5}";
  jerry_value_t obj = jerry_json_parse (data, sizeof (data) - 1);

  /* "obj" now conatins and object created from the "data" JSON string. */

  jerry_release_value (obj);

  /* Cleanup engine */
  jerry_cleanup ();

  return 0;
}
```

## jerry_json_stringify

**Summary**

Create a JSON string value from a JavaScript value. The behaviour is equivalent with
the "JSON.stringify(input_value)" JS call.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_json_stringify (const jerry_value_t input_value);
```

- `input_value` - a `jerry_value_t` to stringify.
- return
  - `jerry_value_t` containing a JSON string.
  - Error value in case of any stringification error.

*New in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  /* Initialize engine */
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t obj = jerry_create_object ();
  {
    jerry_value_t key = jerry_create_string ((const jerry_char_t *) "name");
    jerry_value_t value = jerry_create_string ((const jerry_char_t *) "John");
    jerry_release_value (jerry_set_property (obj, key, value));
    jerry_release_value (key);
    jerry_release_value (value);
  }

  jerry_value_t stringified = jerry_json_stringify (obj);

  /* "stringified" now contains a JSON string */

  jerry_release_value (stringified);
  jerry_release_value (obj);

  /* Cleanup engine */
  jerry_cleanup ();

  return 0;
}
```
