/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file bluetooth.c
 * \brief Interaction with bluetooth headset
 */

#include <ffgtk.h>

#ifdef HAVE_DBUS
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>

extern gchar *softphoneGetHeadset( struct sProfile * );

/** headset proxy */
static DBusGProxy *psHeadset = NULL;
/** bluetooth device list */
static GList *psBluetoothDeviceList = NULL;
/** dbus connection */
static DBusGConnection *psConnection = NULL;

/** Bluetooth device information structure */
struct sBluetoothDeviceInfo {
	char *pnName;
	char *pnPath;
};

/**
 * \brief Fill bluetooth treeview
 * \param psWidget combobox widget
 */
void bluetoothSetList( GtkWidget *psWidget ) {
	GList *psList = NULL;
	int nIndex = 0;
	gchar *pnName = softphoneGetHeadset( getActiveProfile() );
	
	for ( psList = psBluetoothDeviceList; psList != NULL && psList -> data != NULL; psList = psList -> next  ) {
		struct sBluetoothDeviceInfo *psInfo;

		psInfo = psList -> data;

		Debug( KERN_INFO, "Name: %s\t\tPath: %s\n", psInfo -> pnName, psInfo -> pnPath );
		gtk_combo_box_append_text( GTK_COMBO_BOX( psWidget ), psInfo -> pnName );
		if ( pnName != NULL && !strcmp( psInfo -> pnName, pnName ) ) {
			gtk_combo_box_set_active( GTK_COMBO_BOX( psWidget ), nIndex );
		}

		nIndex++;
	}
}

/**
 * \brief Find and return bluetooth device by name
 * \param pnDevice device name
 * \return bluetooth device information structure or NULL on error
 */
static struct sBluetoothDeviceInfo *findBluetoothDevice( gchar *pnDevice ) {
	GList *psList = NULL;

	for ( psList = psBluetoothDeviceList; psList != NULL && psList -> data != NULL; psList = psList -> next  ) {
		struct sBluetoothDeviceInfo *psInfo;

		psInfo = psList -> data;
		if ( strcmp( psInfo -> pnName, pnDevice ) == 0 ) {
			return psInfo;
		}
	}

	return NULL;
}

/**
 * \brief Get bluetooth device name
 * \param psDevice device proxy
 * \return device name or NULL
 */
static gchar *bluetoothGetName( DBusGProxy *psDevice ) {
	GHashTable *psHash;

	if ( dbus_g_proxy_call( psDevice, "GetProperties", NULL, G_TYPE_INVALID, dbus_g_type_get_map( "GHashTable", G_TYPE_STRING, G_TYPE_VALUE ), &psHash, G_TYPE_INVALID ) != FALSE ) {
		GValue *psValue;
		char *pnName;

		psValue = g_hash_table_lookup( psHash, "Name" );
		pnName = psValue ? g_value_dup_string( psValue ) : NULL;
		g_hash_table_destroy( psHash );
		return pnName;
	}

	return NULL;
}

/**
 * \brief Scan all bluetooth adapters for device and create list
 * \return error code
 */
static int bluetoothCreateList( void ) {
	GError *psError = NULL;
	DBusGProxy *psManager = NULL;
	DBusGProxy *psAdapter = NULL;
	DBusGProxy *psDevice = NULL;
	GPtrArray *psAdapters = NULL;
	GPtrArray *psDevices = NULL;
	struct sBluetoothDeviceInfo *psBluetoothDeviceInfo = NULL;
	char *pnDevicePath = NULL;
	int nAdapterIndex = 0;
	int nDeviceIndex = 0;

	/* Get dbus bluez manager */
	psManager = dbus_g_proxy_new_for_name( psConnection, "org.bluez", "/", "org.bluez.Manager" );
	if ( psManager == NULL ) {
		return -1;
	}

	/* Get adapter list */
	dbus_g_proxy_call( psManager, "ListAdapters", &psError, G_TYPE_INVALID, dbus_g_type_get_collection( "GPtrArray", DBUS_TYPE_G_OBJECT_PATH ), &psAdapters, G_TYPE_INVALID );
	if ( psError != NULL ) {
		Debug( KERN_WARNING, "Couldn't get adapter list: %s\n", psError -> message );
		g_error_free( psError );
		g_object_unref( psManager );
		return -2;
	}
	g_object_unref( psManager );

	for ( nAdapterIndex = 0; nAdapterIndex < psAdapters -> len; nAdapterIndex++ ) {
		/* Create adapter proxy */
		psAdapter = dbus_g_proxy_new_for_name( psConnection, "org.bluez", g_ptr_array_index( psAdapters, nAdapterIndex ), "org.bluez.Adapter" );
		if ( psAdapter == NULL ) {
			Debug( KERN_WARNING, "Couldn't get adapter proxy!!\n" );
			continue;
		}

		/* List all devices on this adapter */
		dbus_g_proxy_call( psAdapter, "ListDevices", &psError, G_TYPE_INVALID, dbus_g_type_get_collection( "GPtrArray", DBUS_TYPE_G_OBJECT_PATH ), &psDevices, G_TYPE_INVALID );
		if ( psError != NULL ) {
			Debug( KERN_WARNING, "Couldn't get device list: %s\n", psError -> message );
			g_error_free( psError );
			g_object_unref( psAdapter );
			return -3;
		}
		
		for ( nDeviceIndex = 0; nDeviceIndex < psDevices -> len; nDeviceIndex++ ) {
			pnDevicePath = g_ptr_array_index( psDevices, nDeviceIndex );
			psDevice = dbus_g_proxy_new_for_name( psConnection, "org.bluez", pnDevicePath, "org.bluez.Device" );

			psBluetoothDeviceInfo = g_malloc0( sizeof( struct sBluetoothDeviceInfo ) );
			if ( psBluetoothDeviceInfo != NULL ) {
				psBluetoothDeviceInfo -> pnName = bluetoothGetName( psDevice );
				psBluetoothDeviceInfo -> pnPath = g_strdup( pnDevicePath );

				psBluetoothDeviceList = g_list_append( psBluetoothDeviceList, psBluetoothDeviceInfo );
			}

			g_object_unref( psDevice );
		}
		g_object_unref( psAdapter );
	}

	g_ptr_array_free( psAdapters, TRUE );

	return 0;
}

#if 0
/**
 * \brief Indicate call on headset
 * \return 0 on success, otherwise error
 */
int bluetoothIndicateCall( void ) {
	GError *psError = NULL;

	dbus_g_proxy_call( psHeadset, "IndicateCall", &psError, G_TYPE_INVALID, G_TYPE_INVALID );
	if ( psError != NULL ) {
		Debug( KERN_WARNING, "Couldn't call IndicateCall: %s\n", psError -> message );
		g_error_free( psError );
		return -1;
	}

	return 0;
}
#endif

/**
 * \brief Cancel an incoming call indication
 * \return 0 on success, otherwise error
 */
static int bluetoothCancelCall( void ) {
	GError *psError = NULL;

	dbus_g_proxy_call( psHeadset, "CancelCall", &psError, G_TYPE_INVALID, G_TYPE_INVALID );
	if ( psError != NULL ) {
		Debug( KERN_WARNING, "Couldn't call CancelCall: %s\n", psError -> message );
		g_error_free( psError );
		return -1;
	}

	return 0;
}

/**
 * \brief Callback function called when bluetooth headset answer button is pressed
 * \param psObject current dbus object proxy
 * \param pUserData UNUSED
 */
static void bluetoothAnswerRequested( DBusGProxy *psObject, gpointer pUserData ) {
	Debug( KERN_INFO, "Got Signal: AnswerRequested!\n" );
	bluetoothCancelCall();

	softphoneHandleBluetooth();
}

/**
 * \brief Set active bluetooth headset
 * \param pnName bluetooth device name
 * \return 0 on success, otherwise error
 */
int bluetoothSetHeadset( gchar *pnName ) {
	struct sBluetoothDeviceInfo *psInfo;

	if ( pnName == NULL ) {
		return 0;
	}

	psInfo = findBluetoothDevice( pnName );
	if ( psInfo == NULL ) {
		return -1;
	}

	psHeadset = dbus_g_proxy_new_for_name( psConnection, "org.bluez", psInfo -> pnPath, "org.bluez.Headset" );
	if ( psHeadset == NULL ) {
		Debug( KERN_WARNING, "Could not create bluetooth headset proxy!\n" );
		return -1;
	}
	Debug( KERN_INFO, "Selected %s\n", psInfo -> pnName );

	dbus_g_proxy_add_signal( psHeadset, "AnswerRequested", G_TYPE_INVALID );
	dbus_g_proxy_connect_signal( psHeadset, "AnswerRequested", G_CALLBACK( bluetoothAnswerRequested ), psConnection, NULL );

	return 0;
}

/**
 * \brief Initialize bluetooth connection and create device list
 * \return error code
 */
int InitBluetooth( void ) {
	GError *psError = NULL;
	GList *psList = NULL;
	struct sBluetoothDeviceInfo *psInfo;

	/* Get dbus connection */
	psConnection = dbus_g_bus_get( DBUS_BUS_SYSTEM, &psError );
	if ( psConnection == NULL ) {
		g_printerr( "Error: %s\n", psError -> message );
		g_error_free( psError );
		return -1;
	}

	/* Create device list */
	bluetoothCreateList();

	if ( psBluetoothDeviceList == NULL ) {
		Debug( KERN_INFO, "No bluetooth devices detected!\n" );
		return 0;
	}

	/* Dump found devices on screen */
	for ( psList = psBluetoothDeviceList; psList != NULL && psList -> data != NULL; psList = psList -> next  ) {
		psInfo = psList -> data;

		Debug( KERN_INFO, "Name: %s\t\tPath: %s\n", psInfo -> pnName, psInfo -> pnPath );
	}

	/* Set headset */
	bluetoothSetHeadset( softphoneGetHeadset( getActiveProfile() ) );

	return 0;
}
#endif
