/* $Id: kmo_dark-test.c,v 1.10 2013-10-08 11:18:56 aagudo Exp $
 *
 * This file is part of the KMOS Library
 * Copyright (C) 2002-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: aagudo $
 * $Date: 2013-10-08 11:18:56 $
 * $Revision: 1.10 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <sys/stat.h>
#include <math.h>

#include <cpl.h>

#include "kmclipm_functions.h"
#include "kmclipm_constants.h"

#include "kmo_dfs.h"

const char  *test_global_path_test_data;
float   test_global_seed_data, test_global_seed_noise;
int     test_global_size_x, test_global_size_y, test_global_size_z, test_global_nr_frames;
void kmo_get_pipe_command(char*, const char*, const char*, int);
int kmo_test_file_exists(const char*);
void kmo_test_verbose_off();
void kmo_test_verbose_on();
double kmo_test_esorex_data(const char*, int);
const char* kmo_test_cat_strings(char*, const char*, const char*);
int  kmo_test_create_RAW_data(int, int, const char*, char**, char**, int*, int, char**, char**, int*, int, float, float);
int  kmo_test_create_RAW_data_zero(int, int, const char*);
int  kmo_test_create_F2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2D_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data_noise(int, int, int, const char*, char *, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_infinite(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data2(int, int, int, int, int*, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_noise(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_B2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);

const char      *path_recipe        = "kmo_dark/";

const char      *valid_files[]      = {"v_raw_123_600sec.fits",
                                       "v_raw_234_600sec.fits",
                                       "v_raw_345_600sec.fits",
                                       "dark_123.fits",
                                       "dark_231.fits",
                                       "dark_312.fits"};

const char      *invalid_files[]    = {"i_data_F2D.fits"};

/**
    @defgroup kmo_dark   kmo_dark unit tests

    @{
*/

/**
 * @brief test with data and mask
 */
static void test_dark(const char *path_op1,
                      const char *path_op2,
                      const char *path_op3,
                      int ret_val,
                      const char* extra_pars)
{
    // create sof-file
    const char *sof_path = "test_dark.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh,
             "%s   DARK\n%s   DARK\n%s   DARK\n",
             path_op1, path_op2, path_op3);
    fclose(fh);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_dark %s %s", extra_pars, sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_dark.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

static void test_dark2(const char *path_op1,
                      const char *path_op2,
                      int ret_val,
                      const char* extra_pars)
{
    // create sof-file
    const char *sof_path = "test_dark2.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh,
             "%s   DARK\n%s   DARK\n",
             path_op1, path_op2);
    fclose(fh);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_dark %s %s", extra_pars, sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_dark.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

static void test_dark1(const char *path_op1,
                      int ret_val,
                      const char* extra_pars)
{
    // create sof-file
    const char *sof_path = "test_dark1.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh, "%s   DARK\n", path_op1);
    fclose(fh);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_dark %s %s", extra_pars, sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_dark.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

void check_headers_dark(const char *path, int ext, double tol, int offset,
                        int nr_badpix, double dark, double ron,
                        double dark_median, double ron_median, double darkcur)
{
    cpl_propertylist *h = kmclipm_propertylist_load(path, ext);

    if (cpl_propertylist_get_int(h, QC_NR_BAD_PIX) == nr_badpix) {
        // Linux
        cpl_test_eq(cpl_propertylist_get_int(h, QC_NR_BAD_PIX),
                    nr_badpix);
    } else {
        // Mac
        cpl_test_eq(cpl_propertylist_get_int(h, QC_NR_BAD_PIX),
                    nr_badpix+19);
    }
    cpl_test_abs(cpl_propertylist_get_double(h, QC_READNOISE),
                 ron, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_DARK),
                 dark+offset, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_DARK_MEDIAN),
                 dark_median+offset, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_READNOISE_MEDIAN),
                 ron_median, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_DARK_CURRENT),
                 fabs(darkcur), tol);

    cpl_propertylist_delete(h); h = NULL;
}


/**
    @brief
        Generates test data for kmo_dark.

    @param path Path where the generated test date should be saved to.
*/
static int kmo_generate_test_data_dark(const char *path)
{
    char        file_path[256];
    int         size_x              = 300,
                size_y              = 509;
    int         nr_primary_keys     = 3,
                nr_sub_keys         = 1;
    char        *primary_keys[nr_primary_keys],
                *primary_vals[nr_primary_keys],
                *sub_keys[nr_sub_keys],
                *sub_vals[nr_sub_keys];
    int         primary_types[nr_primary_keys],
                sub_types[nr_primary_keys];

    /* ----- valid test data ----- */
    /* DARK, 600 sec */
    primary_keys[0] = MINDIT;
    primary_types[0] = CPL_TYPE_DOUBLE;
    primary_vals[0] = "1.0";

    primary_keys[1] = EXPTIME;
    primary_types[1] = CPL_TYPE_DOUBLE;
    primary_vals[1] = "600.0";

    primary_keys[2] = NDIT;
    primary_types[2] = CPL_TYPE_INT;
    primary_vals[2] = "1.0";

    sub_keys[0] = EXPTIME;
    sub_types[0] = CPL_TYPE_DOUBLE;
    sub_vals[0] = "600.0";

    cpl_test_eq(0,
        kmo_test_create_RAW_data(size_x,
                                 size_y,
                               kmo_test_cat_strings(file_path, path, valid_files[0]),
                                 primary_keys,
                                 primary_vals,
                                 primary_types,
                                 nr_primary_keys,
                                 sub_keys,
                                 sub_vals,
                                 sub_types,
                                 nr_sub_keys,
                                 1.1, 0.0));

    cpl_test_eq(0,
        kmo_test_create_RAW_data(size_x,
                                 size_y,
                               kmo_test_cat_strings(file_path, path, valid_files[1]),
                                 primary_keys,
                                 primary_vals,
                                 primary_types,
                                 nr_primary_keys,
                                 sub_keys,
                                 sub_vals,
                                 sub_types,
                                 nr_sub_keys,
                                 1.3, 0.0));

    cpl_test_eq(0,
        kmo_test_create_RAW_data(size_x,
                                 size_y,
                               kmo_test_cat_strings(file_path, path, valid_files[2]),
                                 primary_keys,
                                 primary_vals,
                                 primary_types,
                                 nr_primary_keys,
                                 sub_keys,
                                 sub_vals,
                                 sub_types,
                                 nr_sub_keys,
                                 1.5, 0.0));

    /* ----- invalid test data ----- */
    /* F2D */
    cpl_test_eq(0,
                kmo_test_create_F2D_data(test_global_size_x, test_global_size_y, FALSE,
                        kmo_test_cat_strings(file_path, path, invalid_files[0]),
                        NULL, NULL, NULL, 0,
                        NULL, NULL, NULL, 0));

    return 0;
}


/**
  @brief    Test of kmo_dark recipe.
  @param    argc   the number of parameters
  @param    argv   the parameter list

  Call @c kmo_dark-test @c --generate when only the test data for this
  recipe should be generated.
  Call @c kmo_dark-test @c --no-gen when only the tests should be
  executed and no data should be generated.

 */
int main(int argc, char *argv[])
{
    float   tol     = 0.01;

    char    test_path[1024],
            op1_file[1024],
            op2_file[1024],
            op3_file[1024],
            file_path[1024],
            tmp[1024];

    int nr_badpix = 0;

    double dark         = 0.0,
           ron          = 0.0,
           dark_median  = 0.0,
           ron_median   = 0.0,
           dark_cur     = 0.0;

    cpl_test_init("<kmos-spark@mpe.mpg.de>", CPL_MSG_WARNING);

    strcpy(test_path, test_global_path_test_data);
    strcat(test_path, path_recipe);

    if (kmo_test_file_exists(test_path) == FALSE) {
        mkdir(test_path, 0777);
    }

    if ((argc == 1) || ((argc > 1) && (strcmp(argv[1], "--no-gen") != 0))) {
        cpl_test_zero(kmo_generate_test_data_dark(test_path));
    }

    if ((argc > 1) && (strcmp(argv[1], "--generate") == 0)) {
        // just generate the data
        return cpl_test_end(0);
    }

    /* ----- run the tests ----- */
    FILE *fd = fopen("log_kmo_dark.txt", "w");
    fprintf(fd, "%s\n", " ");
    fclose(fd);

    /* ----- invalid tests ----- */
    cpl_msg_info(cpl_func, "Testing kmo_dark with invalid data.");

    // invalid data (F2D)
    kmo_test_cat_strings(op1_file, test_path, valid_files[0]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[1]);
    kmo_test_cat_strings(op3_file, test_path, invalid_files[0]);
    test_dark(op1_file, op2_file, op3_file, -1, "");

    /* ----- valid tests ----- */
    cpl_msg_info(cpl_func, "Testing kmo_dark with valid data.");
    kmo_test_cat_strings(op1_file, test_path, valid_files[0]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[1]);
    kmo_test_cat_strings(op3_file, test_path, valid_files[2]);

    //
    // test with just one frame
    //
    test_dark1(op1_file, 0, "");

    nr_badpix    = 0;
    dark         = 84026.834;
    ron          = 47746.096;
    dark_median  = 84048.96875;
    ron_median   = 47746.101;
    dark_cur     = 140.046;

    // check MASTER_DARK
    cpl_test_abs(kmo_test_esorex_data(MASTER_DARK, 0), 395322, tol*100);

    strcpy(tmp, MASTER_DARK);
    sprintf(file_path, "%s.fits", kmo_strlower(tmp));

    int i = 0;
    for (i = 1; i <= 2*KMOS_NR_DETECTORS; i+=2) {
        check_headers_dark(file_path, i, tol, (i-1)/2,
                           nr_badpix, dark, ron, dark_median, ron_median, dark_cur);
    }

    // check BADPIXEL_DARK
    cpl_test_abs(kmo_test_esorex_data(BADPIXEL_DARK, 0), 2.874105, tol);
    strcpy(tmp, BADPIXEL_DARK);
    sprintf(file_path, "%s.fits", kmo_strlower(tmp));
    for (i = 1; i <= KMOS_NR_DETECTORS; i++) {
        check_headers_dark(file_path, i, tol, i-1,
                           nr_badpix, dark, ron, dark_median, ron_median, dark_cur);
    }

    //
    // test with just two frames
    //
    test_dark2(op1_file, op2_file, 0, "");

    nr_badpix    = 0;
    dark         = 91598.4;
    ron          = 8564.77;
    dark_median  =  91624.9921875;
    ron_median   = 8564.77;
    dark_cur     = 152.667;

    // check MASTER_DARK
    cpl_test_abs(kmo_test_esorex_data(MASTER_DARK, 0), 306922, tol*100);

    strcpy(tmp, MASTER_DARK);
    sprintf(file_path, "%s.fits", kmo_strlower(tmp));

    for (i = 1; i <= 2*KMOS_NR_DETECTORS; i+=2) {
        check_headers_dark(file_path, i, tol, (i-1)/2,
                           nr_badpix, dark, ron, dark_median, ron_median, dark_cur);
    }

    // check BADPIXEL_DARK
    cpl_test_abs(kmo_test_esorex_data(BADPIXEL_DARK, 0), 2.87411, tol);
    strcpy(tmp, BADPIXEL_DARK);
    sprintf(file_path, "%s.fits", kmo_strlower(tmp));
    for (i = 1; i <= KMOS_NR_DETECTORS; i++) {
        check_headers_dark(file_path, i, tol, i-1,
                           nr_badpix, dark, ron, dark_median, ron_median, dark_cur);
    }

    //
    // tests with three frames
    //
    test_dark(op1_file, op2_file, op3_file, 0, "");

    // expected values
    nr_badpix    = 0;
    dark         = 99257.281;
    ron          = 8798.0377;
    dark_median  = 99260.0234375;
    ron_median   = 8797.673828125;

    // expected values
    dark_cur = 7.27596e-14;

    // check MASTER_DARK
    cpl_test_abs(kmo_test_esorex_data(MASTER_DARK, 0), 324136, tol*100);
    strcpy(tmp, MASTER_DARK);
    sprintf(file_path, "%s.fits", kmo_strlower(tmp));
    for (i = 1; i <= 2*KMOS_NR_DETECTORS; i+=2) {
        check_headers_dark(file_path, i, tol/1e-10, (i-1)/2,
                           nr_badpix, dark, ron, dark_median, ron_median, dark_cur);
    }

    // check BADPIXEL_DARK
    cpl_test_abs(kmo_test_esorex_data(BADPIXEL_DARK, 0), 2.87411, tol);
    strcpy(tmp, BADPIXEL_DARK);
    sprintf(file_path, "%s.fits", kmo_strlower(tmp));
    for (i = 1; i <= KMOS_NR_DETECTORS; i++) {
        check_headers_dark(file_path, i, tol/1e-10, i-1,
                           nr_badpix, dark, ron, dark_median, ron_median, dark_cur);
    }

    // keep 'small' badpixel_dark.fits for kmo_flat-test
    strcpy(test_path, "mv badpixel_dark.fits badpixel_dark_small.fits");
    system(test_path);

    //
    // PIPELINE TEST (must be the last test,since the following recipes
    //   rely on this output)
    //
    cpl_msg_info(cpl_func, "Testing kmo_dark with real pipeline data.");
    sprintf(test_path, "%s%s", test_global_path_test_data, "pipeline/");

    //
    // test MASTER_DARK pipeline
    //
    kmo_test_cat_strings(op1_file, test_path, valid_files[3]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[4]);
    kmo_test_cat_strings(op3_file, test_path, valid_files[5]);
    test_dark(op1_file, op2_file, op3_file, 0, " --pos_b=11.0 --neg_b=11.0 ");

    // expected values
    dark_cur    = 5.3829e-18;
    nr_badpix   = 33237;

    // check MASTER_DARK
    cpl_test_abs(kmo_test_esorex_data(MASTER_DARK, 0), 29.4568, tol);
    strcpy(tmp, MASTER_DARK);
    sprintf(file_path, "%s.fits", kmo_strlower(tmp));
    for (i = 1; i <= 2*KMOS_NR_DETECTORS; i++) {
        check_headers_dark(file_path, i, tol/1e-10, (i-1)/2,
                           nr_badpix, dark, ron, dark_median, ron_median, dark_cur);
    }

    //check BADPIXEL_DARK
    cpl_test_abs(kmo_test_esorex_data(BADPIXEL_DARK, 0), 2.95284, tol);
    strcpy(tmp, BADPIXEL_DARK);
    sprintf(file_path, "%s.fits", kmo_strlower(tmp));
    for (i = 1; i <= KMOS_NR_DETECTORS; i++) {
        check_headers_dark(file_path, i, tol/1e-10, i-1,
                nr_badpix, dark, ron, dark_median, ron_median, dark_cur);
    }

    return cpl_test_end(0);
}

/**@}*/
