/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#define _BSD_SOURCE /* strdup() */

#include <string.h>
#include <errno.h>
#include <stdlib.h>

#include "cdw_fs.h"
#include "cdw_file_picker.h"
#include "cdw_string.h"
#include "cdw_ncurses.h"
#include "cdw_widgets.h"
#include "cdw_window.h"
#include "cdw_debug.h"
#include "cdw_sys.h"
#include "cdw_fs_browser.h"
#include "gettext.h"
#include "canonicalize.h"

static struct cdw_file_picker {
	/* file system browser for traversing file system tree;
	   I could have put it somehow in 'browser_window' struct,
	   but current layout seems to be better */
	cdw_fs_browser_t *fs_browser;

	/* window of whole picker */
	struct {
		int n_cols;
		int n_lines;
		int begin_y;
		int begin_x;
		WINDOW *win;
	} main_window;

	/* subwindow for file system browser */
	struct {
		int n_cols;
		int n_lines;
		int begin_y;
		int begin_x;
		WINDOW *win;
	} browser_window;

	/* one line input form; here user can enter path manually */
	struct {
		CDW_INPUT_LINE *input_line;
		int begin_y;
		int begin_x;
		int n_cols;
	} input;

	/* OK/Cancel buttons at the bottom of picker */
	struct {
		CDW_BUTTON *ok;
		CDW_BUTTON *cancel;
		int row;
	} buttons;

	int expected_file_type;
	int expected_permissions;
	int expected_new_or_existing;

	char *fullpath;

	enum {
		PICKER_BROWSER,
		PICKER_INPUT,
		PICKER_OK_BUTTON,
		PICKER_CANCEL_BUTTON
	} focus;


} file_picker;



static void     cdw_fs_ui_file_picker_init(struct cdw_file_picker *picker);
static cdw_rv_t cdw_fs_ui_file_picker_build(struct cdw_file_picker *picker, const char *title, const char *message, const char *initial_fullpath);
static cdw_rv_t cdw_fs_ui_file_picker_driver(struct cdw_file_picker *picker);
static int      cdw_fs_ui_file_picker_driver_on_enter(struct cdw_file_picker *picker);
static int      cdw_fs_ui_file_picker_driver_on_enter_policy(struct cdw_file_picker *picker, int rv);
static void     cdw_fs_ui_file_picker_driver_print_debug(struct cdw_file_picker *picker, int key);
static void     cdw_fs_ui_file_picker_destroy(struct cdw_file_picker *picker);


/* WARNING: file name without specified directory is a file in current location */




/**
   \p caller_fullpath will be modified or not (depending on keys pressed
   in the picker)
*/
cdw_rv_t cdw_fs_ui_file_picker(const char *title, const char *message, char **caller_fullpath, int expected_file_type, int expected_permissions, int expected_new_or_existing)
{
	cdw_fs_ui_file_picker_init(&file_picker);
	file_picker.expected_file_type = expected_file_type;
	file_picker.expected_permissions = expected_permissions;
	file_picker.expected_new_or_existing = expected_new_or_existing;

	cdw_rv_t crv = cdw_fs_ui_file_picker_build(&file_picker, title, message, *caller_fullpath);

	cdw_rv_t retval = CDW_OK;

	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to build picker\n");
		retval = CDW_ERROR;
	} else {
		crv = cdw_fs_ui_file_picker_driver(&file_picker);
		if (crv == CDW_OK) {
			crv = cdw_string_set(caller_fullpath, file_picker.fullpath);
			if (crv == CDW_OK) {
				retval = CDW_OK;
			} else {
				cdw_vdm ("ERROR: failed to set caller's fullpath at return\n");
				retval =  CDW_ERROR;
			}
		} else if (crv == CDW_CANCEL) {
			retval = CDW_CANCEL;
		} else {
			retval = CDW_ERROR;
		}
	}

	cdw_fs_ui_file_picker_destroy(&file_picker);
	return retval;
}




void cdw_fs_ui_file_picker_init(struct cdw_file_picker *picker)
{
	picker->expected_file_type = 0;
	picker->expected_permissions = 0;
	picker->expected_new_or_existing = 0;

	picker->main_window.n_lines = LINES - 2;
	picker->main_window.n_cols = COLS - 10;
	picker->main_window.begin_y = (LINES - picker->main_window.n_lines) / 2;
	picker->main_window.begin_x = (COLS - picker->main_window.n_cols) / 2;

	picker->browser_window.n_lines = picker->main_window.n_lines - 8;
	picker->browser_window.n_cols = picker->main_window.n_cols - 6;
	picker->browser_window.begin_y = 4;
	picker->browser_window.begin_x = 2;

	picker->input.begin_y = picker->main_window.n_lines - 4;
	picker->input.begin_x = 2;
	picker->input.n_cols = picker->main_window.n_cols - 6;

	picker->buttons.row = picker->main_window.n_lines - 2;

	picker->fullpath = (char *) NULL;

	return;
}





cdw_rv_t cdw_fs_ui_file_picker_build(struct cdw_file_picker *picker, const char *title, const char *message, const char *initial_fullpath)
{
	if (initial_fullpath == (char *) NULL) {
		cdw_vdm ("ERROR: invalid initial fullpath \"%s\"\n", initial_fullpath);
		return CDW_ERROR;
	}
	/* *** main window + fs browser window *** */

	picker->main_window.win = cdw_window_new((WINDOW *) NULL,
						 picker->main_window.n_lines, picker->main_window.n_cols,
						 picker->main_window.begin_y, picker->main_window.begin_x,
						 CDW_COLORS_DIALOG, title, (char *) NULL);
	if (picker->main_window.win == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create main window for file picker\n");
		return CDW_ERROR;
	}

	picker->browser_window.win = cdw_window_new(picker->main_window.win,
						    picker->browser_window.n_lines, picker->browser_window.n_cols,
						    picker->browser_window.begin_y, picker->browser_window.begin_x,
						    CDW_COLORS_DIALOG, (char *) NULL, (char *) NULL);
	if (picker->browser_window.win == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create fs browser window for file picker\n");
		return CDW_ERROR;
	}

	cdw_window_add_strings(picker->browser_window.win, "", "");
	cdw_window_add_help(picker->browser_window.win);

	picker->fullpath = strdup(initial_fullpath);
	if (picker->fullpath == (char *) NULL) {
		cdw_vdm ("ERROR: failed to strdup() initial fullpath \"%s\"\n", picker->fullpath);
		return CDW_ERROR;
	}

	/* *** fs browser *** */

	picker->fs_browser = cdw_fs_browser_new(picker->browser_window.win, picker->fullpath);
	if (picker->fs_browser == (cdw_fs_browser_t *) NULL) {
		cdw_vdm ("ERROR: failed to create fs browser\n");
		return CDW_ERROR;
	}
	cdw_list_display_add_return_keys(picker->fs_browser->display, CDW_KEY_TAB,
					 KEY_DOWN, KEY_UP,
					 KEY_NPAGE, KEY_PPAGE,
					 KEY_HOME, KEY_END, 0);

	/* *** buttons *** */

	picker->buttons.ok = cdw_button_new(picker->main_window.win,
					    /* 2TRANS: this is label of button */
					    picker->buttons.row, 4, _("OK"), CDW_COLORS_DIALOG);
	picker->buttons.cancel = cdw_button_new(picker->main_window.win,
						/* 2TRANS: this is label of button */
						picker->buttons.row, 15, _("Cancel"), CDW_COLORS_DIALOG);
	if (picker->buttons.ok == (CDW_BUTTON *) NULL || picker->buttons.cancel == (CDW_BUTTON *) NULL) {
		cdw_vdm ("ERROR: failed to create one of buttons\n");
		return CDW_ERROR;
	}

	/* *** input line *** */

	picker->input.input_line = cdw_input_line_new(picker->main_window.win,
						      picker->input.begin_y, picker->input.begin_x,
						      picker->input.n_cols,
						      picker->fullpath,
						      0, 0); /* input type and length limit */
	if (picker->input.input_line == (CDW_INPUT_LINE *) NULL) {
		cdw_vdm ("ERROR: failed to create input line\n");
		return CDW_ERROR;
	}
	/* content of input field buffer will be stored to provided data buffer
	   every time user presses TAB key to move away from input line */
	picker->input.input_line->save_on_tab = true;

	/* *** message on top of window *** */
	mvwprintw(picker->main_window.win, 2, 2, message);
	wrefresh(picker->main_window.win);

	return CDW_OK;
}





cdw_rv_t cdw_fs_ui_file_picker_driver(struct cdw_file_picker *picker)
{
	picker->focus = PICKER_BROWSER; /* initial focus is on file system browser */
	int key = 'a'; /* safe initial value that doesn't trigger any action */
	while (key != CDW_KEY_ESCAPE && key != CDW_KEY_ENTER) {

		if (key == CDW_KEY_TAB) {
			if (picker->focus == PICKER_BROWSER) {
				curs_set(1);
				picker->focus = PICKER_INPUT;
				/* TODO: accessing form directly, maybe some wrapper? */
				form_driver(picker->input.input_line->form, REQ_END_LINE);
			} else if (picker->focus == PICKER_INPUT) {
				curs_set(0);
				cdw_button_focus(picker->buttons.ok);
				picker->focus = PICKER_OK_BUTTON;
			} else if (picker->focus == PICKER_OK_BUTTON) {
				cdw_button_unfocus(picker->buttons.ok);
				cdw_button_focus(picker->buttons.cancel);
				picker->focus = PICKER_CANCEL_BUTTON;
			} else if (picker->focus == PICKER_CANCEL_BUTTON) {
				cdw_button_unfocus(picker->buttons.cancel);
				picker->focus = PICKER_BROWSER;
			} else {
				cdw_vdm ("ERROR: incorrect focus\n");
			}
		}
		redrawwin(picker->main_window.win);
		wrefresh(picker->main_window.win);

		//key = 'a'; /* dummy value that doesn't trigger any behavior */
		if (picker->focus == PICKER_BROWSER) {
#ifndef NDEBUG
			char *fp = cdw_fs_browser_get_current_fullpath(picker->fs_browser);
			cdw_vdm ("INFO: focus on browser, fs browser fullpath = \"%s\"\n", fp);
			free(fp);
			fp = (char *) NULL;
#endif
			while (1) {
				/* this driver function receives and processes
				   keys (e.g. causes fs browser to change a
				   dir on enter or backspace), but it also
				   returns a (meaningful) key value, so that
				   file picker can also act upon the key */
				key = cdw_fs_browser_driver(picker->fs_browser);
				cdw_fs_ui_file_picker_driver_print_debug(picker, key);

				if (key == CDW_KEY_ESCAPE || key == 'q' || key == 'Q') {
					key = CDW_KEY_ESCAPE;
					break;
				} else if (key == CDW_KEY_TAB || cdw_fs_browser_is_navigation_key(key)) {
					/* *** synchronization point *** */
					/* synchronization means that picker->fullpath,
					   fs_browser->fullpath and input_line->buffer must
					   have the same value, this time the value must
					   be taken from fs_browser */
					cdw_file_t *file = cdw_fs_browser_get_current_file(picker->fs_browser);
					if (file->is_ref_to_parent_dir) {
						/* Don't put
						   "/some/path/.." in
						   input line, because
						   it is annoying. */
						char *cut = strdup(file->fullpath);
						if (cut) {
							/* Just to be 100% sure that me and the code base agree on contents of the fullpath string. */
							cdw_assert (cut[file->name_start] == '.'
								    && cut[file->name_start + 1] == '.',
								    "ERROR: fullpath to parent dir does not end with '..': \"%s\"\n", file->fullpath);

							cut[file->name_start] = '\0';
							cdw_input_line_refresh_with_string(picker->input.input_line, cut);
							cdw_string_set(&picker->fullpath, cut);
							free(cut);
							cut = (char *) NULL;
						} else {
							cdw_input_line_refresh_with_string(picker->input.input_line, file->fullpath);
							cdw_string_set(&picker->fullpath, file->fullpath);
						}
					} else {
						cdw_input_line_refresh_with_string(picker->input.input_line, file->fullpath);
						cdw_string_set(&picker->fullpath, file->fullpath);

					}
					cdw_vdm ("INFO: SYNCHRONIZATION POINT: non-escape key \"%s\" in fs browser\n", cdw_ncurses_key_label(key));
					cdw_vdm ("INFO: fs browser fullpath at synchronization point: \"%s\"\n", picker->fullpath);

					if (key == CDW_KEY_TAB) {
						break;
					}
				} else if (key == KEY_BACKSPACE || key == '~') {
					cdw_file_t *file = cdw_fs_browser_get_current_file(picker->fs_browser);

					cdw_input_line_refresh_with_string(picker->input.input_line, file->fullpath);
					cdw_string_set(&picker->fullpath, file->fullpath);
					cdw_vdm ("INFO: SYNCHRONIZATION POINT: non-escape key \"%s\" in fs browser\n", cdw_ncurses_key_label(key));
					cdw_vdm ("INFO: fs browser fullpath at synchronization point: \"%s\"\n", file->fullpath);
				} else if (key == '?') {
					cdw_fs_browser_help_window(picker->fs_browser);
					redrawwin(picker->main_window.win);
					wrefresh(picker->main_window.win);
				} else {
					; /* some other, alphanumeric key */
				}
			}
		} else if (picker->focus == PICKER_INPUT) {
			cdw_sdm ("INFO: focus on input\n");

			/* 5 attempts to enter safe string */
			key = cdw_input_line_driver(picker->input.input_line, &(picker->fullpath), 5);
			/* here synchronization point is implicit:
			   picker->fullpath and input_line->buffer are always
			   in sync, because they are basically the same variable
			   (second argument to cdw_input_line_driver()), so now
			   we need to sync it with fs_browser; this is done in
			   cdw_fs_ui_file_picker_driver_on_enter() */
			if (key == CDW_KEY_TAB) {
				/* Tab key will move focus to next element
				   in file picker window */
				cdw_vdm ("INFO: picker input, tab, picker fullpath = \"%s\"\n", picker->fullpath);
			} else if (key == CDW_KEY_ENTER) {
				key = cdw_fs_ui_file_picker_driver_on_enter(picker);
			} else if (key == CDW_KEY_ESCAPE) {
				; /* will break loop */
			} else {
				/* some error in input line driver */
				cdw_vdm ("ERROR in input line driver\n");
				key = CDW_KEY_ESCAPE;
			}

		} else if (picker->focus == PICKER_OK_BUTTON) {
			cdw_sdm ("INFO: focus on ok button\n");
			key = wgetch(picker->main_window.win);
			cdw_sdm ("INFO: got %d / \"%s\" key\n", key, cdw_ncurses_key_label(key));
			if (key == CDW_KEY_ENTER) {
				key = cdw_fs_ui_file_picker_driver_on_enter(picker);

				cdw_input_line_refresh_with_string(picker->input.input_line, picker->fullpath);
			} else {
				/* escape, tab, or whatever - will be handled by loop condition */
			}
		} else if (picker->focus == PICKER_CANCEL_BUTTON) {
			cdw_sdm ("INFO: focus on cancel button\n");
			key = wgetch(picker->main_window.win);
			cdw_sdm ("INFO: got %d / \"%s\" key\n", key, cdw_ncurses_key_label(key));
		} else {
			cdw_vdm ("ERROR: incorrect focus\n");
		}
	}

	if (key == CDW_KEY_ENTER) {
		if (picker->focus == PICKER_OK_BUTTON || picker->focus == PICKER_INPUT) {
			return CDW_OK;
		} else if (picker->focus == PICKER_CANCEL_BUTTON) {
			return CDW_CANCEL;
		} else {
			cdw_vdm ("ERROR: got ENTER in unknown focus %d\n", picker->focus);
			return CDW_ERROR;
		}
	} else if (key == CDW_KEY_ESCAPE) {
		return CDW_CANCEL;
	} else {
		cdw_vdm ("ERROR: got unknown key at return: %d / \"%s\"\n", key, cdw_ncurses_key_label(key));
		return CDW_ERROR;
	}
}





void cdw_fs_ui_file_picker_driver_print_debug(struct cdw_file_picker *picker, int key)
{
	if (picker->focus == PICKER_BROWSER) {
		char *fp = cdw_fs_browser_get_current_fullpath(picker->fs_browser);
		cdw_vdm ("INFO: fs browser: file_i = %zd, fullpath = \"%s\"\n",
			 picker->fs_browser->display->current_item_ind, fp);
		if (key == CDW_KEY_ESCAPE) {
			cdw_vdm ("INFO: got Escape in fs browser, moving away from browser\n");
		} else if (key == 'q' || key == 'Q') {
			cdw_vdm ("INFO: got 'Q' in fs browser, moving away from browser\n");
		} else if (key == ' ') {
			cdw_vdm ("INFO: got Space in fs browser, selected file = \"%s\"\n", fp);
		} else if (key == CDW_KEY_TAB) {
			cdw_vdm ("INFO: got Tab in fs browser, moving away from browser\n");
		} else {
			cdw_sdm ("INFO: got key %d / \"%s\" in fs browser, looping\n", key, cdw_ncurses_key_label(key));
		}
		free(fp);
		fp = (char *) NULL;
	}
	return;
}





void cdw_fs_ui_file_picker_destroy(struct cdw_file_picker *picker)
{
	cdw_input_line_delete(&picker->input.input_line);
	cdw_fs_browser_delete(&picker->fs_browser);
	cdw_button_delete(&picker->buttons.ok);
	cdw_button_delete(&picker->buttons.cancel);

	if (picker->browser_window.win != (WINDOW *) NULL) {
		delwin(picker->browser_window.win);
		picker->browser_window.win = (WINDOW *) NULL;
	}

	if (picker->main_window.win != (WINDOW *) NULL) {
		delwin(picker->main_window.win);
		picker->main_window.win = (WINDOW *) NULL;
	}

	if (picker->fullpath != (char *) NULL) {
		free(picker->fullpath);
		picker->fullpath = (char *) NULL;
	}

	return;
}





/**

   \return 'a' - safe value, not triggering any action in fs browser, current fullpath should not be considered as correct value returned to user
   \return CDW_KEY_ENTER - current fullpath can be considered as correct value that can be returned to user
   \return CDW_KEY_ESCAPE - close file picker, don't return any path
 */
int cdw_fs_ui_file_picker_driver_on_enter(struct cdw_file_picker *picker)
{
	char *canonical_fullpath = canonicalize_filename_mode(picker->fullpath, CAN_MISSING);
	if (canonical_fullpath == (char *) NULL) {
		/* canonicalization of path failed */
		cdw_assert (canonical_fullpath == (char *) NULL, "ERROR: cdw_fs_canonicalize_path() has modified result\n");
		return 'a';
	}

	cdw_vdm ("INFO: canonicalized fullpath is \"%s\"\n", canonical_fullpath);

	int fs_check = cdw_fs_check_fullpath(canonical_fullpath, picker->expected_file_type, picker->expected_permissions, picker->expected_new_or_existing);
	int key = cdw_fs_ui_file_picker_driver_on_enter_policy(picker, fs_check);

	if (key == 1) {
		/* current path is correct, browse to it */
		cdw_vdm ("INFO: catch b, attempting to browse to \"%s\"\n", canonical_fullpath);
		cdw_rv_t crv = cdw_fs_browser_browse_to(picker->fs_browser, canonical_fullpath);
		cdw_assert (crv == CDW_OK || crv == CDW_CANCEL,
			    "ERROR: failed to browse to \"%s\" when I was told to do so by b\n",
			    canonical_fullpath);


		crv = cdw_string_set(&picker->fullpath, canonical_fullpath);
		cdw_assert (crv == CDW_OK, "ERROR: failed to set fullpath on ENTER\n");

		key = 'a';

	} else if (key == 0) {
		/* current path is correct, file picker can return it to user */
		if (picker->expected_file_type == CDW_FS_FILE) {
			/* remove ending slashes; in theory there should be no
			   more than one such char at the end of canonical_fullpath */
			size_t len = strlen(canonical_fullpath);
			cdw_assert (len > 0, "ERROR: len of canonical fullpath == 0: \"%s\"\n", canonical_fullpath);
			if (len > 1) {
				if (canonical_fullpath[len - 1] == '/') {
					canonical_fullpath[len - 1] = '\0';
				}
				if (len > 2) {
					cdw_assert (canonical_fullpath[len - 1] != '/',
						    "ERROR: multiple slashes at the end of canonical fullpath \"%s\" (last one already removed)\n",
						    canonical_fullpath);
				}
			}
		}
		cdw_rv_t crv = cdw_string_set(&picker->fullpath, canonical_fullpath);
		cdw_assert (crv == CDW_OK, "ERROR: failed to set fullpath on ENTER with canonical fullpath = \"%s\",\n",
			    canonical_fullpath);

		key = CDW_KEY_ENTER;
	} else if (key == -2) {
		/* close file picker, don't return any path */
		key = CDW_KEY_ESCAPE;
	} else {
		/* -1, path is invalid, don't change current dir, wait for user's actions */
		key = 'a'; /* safe value, not triggering any action */
	}

	free(canonical_fullpath);
	canonical_fullpath = (char *) NULL;

	return key;
}





/**
   \brief Definition of policy of behavior of file picker

   Function resolves behavior of file picker for given expected file
   constraints (type, perms, existence) and for given result of file system
   check (is file/dir, exists/doesn't exist, has correct/wrong permissions).

   Note that \p fs_check doesn't have to be related to picker->fullpath,
   it can be a result of call of fs check function for completely different
   fullpath. \p picker is here just a pointer to struct with values of
   'expected_*' fields, nothing more. picker->fullpath is not used by this
   function.

   \param picker - picker variable with valid values of 'expected_*' fields
   \param fs_check - result of file system checks for a fullpath (not necessarily picker->fullpath)

   \return -2 if file picker should be closed without returning any path
   \return -1 if given path is invalid for some reasons, and file picker should do nothing and wait for further user's actions
   \return  1 if given path is valid, and existing, but instead of returning it, fs browser should browse to given file and wait for further user's actions (don't accept correct path, but browse to it)
   \return  0 if file picker can return given path (the one for which \p fs_check was returned) as a valid return value
*/
int cdw_fs_ui_file_picker_driver_on_enter_policy(struct cdw_file_picker *picker, int fs_check)
{
	if (fs_check & CDW_FS_CHECK_SYS_ERROR) {
		cdw_vdm ("ERROR: fullpath check returns \"sys error\"\n");
	        return -1;
	} else if (fs_check & CDW_FS_CHECK_WRONG_PERM) {
		/* file specified by given fullpath has incorrect permissions,
		   or file doesn't exist + non-existent file is allowed +
		   parent dir doesn't have 'write' permissions */
		cdw_fs_errno_handler(EPERM);
		cdw_vdm ("ERROR: fullpath check returns \"wrong perms\"\n");
		return -1;
	} else if (fs_check & CDW_FS_CHECK_DOESNT_EXIST) {
		cdw_assert (! (fs_check & CDW_FS_CHECK_EXISTS), "ERROR: file exists and doesn't exist at the same time\n");
		if (picker->expected_new_or_existing & CDW_FS_NEW) {
			if (picker->expected_file_type & CDW_FS_CHECK_IS_FILE) {
				/* caller allows/expects non-existing file */
				if (fs_check & CDW_FS_CHECK_NO_PARENT) {
					/* file doesn't exist, and parent dir
					   in given fullpath doesn't exist
					   either */
					cdw_fs_errno_handler(ENOENT);
					return -1;
				} else {
					/* file doesn't exist, but its parent dir does, and
					   the dir has 'write' perms (since 'fs_check & CDW_FS_CHECK_WRONG_PERM'
					   condition at the beginning was not met) */
					return 0;
				}
			} else {
				/* non-existing dir not acceptable,
				   cdw can't create dirs (yet) */
				cdw_fs_errno_handler(ENOENT);
				return -1;
			}
		} else {
			/* file doesn't exist, and such situation is
			   not acceptable regardless if we were
			   searching for dir or file */
			cdw_fs_errno_handler(ENOENT);
			return -1;
		}
	} else if (fs_check & CDW_FS_CHECK_EXISTS) {
		cdw_assert (! (fs_check & CDW_FS_CHECK_DOESNT_EXIST), "ERROR: file exists and doesn't exist at the same time\n");
		/* file exists */
		/* in this branch we first have to check if this file or dir */
		if ( ((picker->expected_file_type == CDW_FS_FILE) && (fs_check & CDW_FS_CHECK_IS_DIR))
		     || ((picker->expected_file_type == CDW_FS_DIR) && (fs_check & CDW_FS_CHECK_IS_FILE)) ) {
			/* file type mismatch; don't prompt, just silently return to browsing */
			cdw_vdm ("INFO: file types mismatch\n");
			return 1;  /* don't accept correct path, but browse to it */
		} else {
			cdw_vdm ("INFO: file types match\n");
			; /* file types match, pass to checking file types */
		}

		/* is existing file type acceptable? should we ask before overwriting? */

		if (picker->expected_new_or_existing == (CDW_FS_NEW | CDW_FS_EXISTING)) {
			/* existing file acceptable, but ask for overwriting */
			cdw_rv_t crv = cdw_fs_errno_handler(EEXIST);
			if (crv == CDW_OK) {
				; /* file exists and may be overwritten;
				     go to checking permissions */
			} else if (crv == CDW_NO) {
				/* file exists, but don't overwrite */
				return 1; /* don't accept correct path, but Browse to it */
			} else if (crv == CDW_CANCEL) {
				/* file exists, and user don't want to
				   overwrite nor enter new path; quit */
				return -2;
			} else {
				cdw_vdm ("ERROR: cdw_fs_errno_handler() returns %d\n", crv);
				return -2;
			}
		} else if (picker->expected_new_or_existing == CDW_FS_NEW) {
			/* file exists, we were expecting only new file;
			   silently return to browsing */
			return 1;  /* don't accept correct path, but browse to it */
		} else if (picker->expected_new_or_existing == CDW_FS_EXISTING) {
			; /* silently accept file, pass to checking permissions */
		} else {
			cdw_vdm ("ERROR: incorrect value of \"expected_new_or_existing\": %d\n",
				 picker->expected_new_or_existing);
			return -1;
		}

		/* check if permissions are correct */
		if (fs_check & CDW_FS_CHECK_WRONG_PERM) {
			/* wasn't this handled at the beginning of the function? */
			cdw_fs_errno_handler(EPERM);
			return -1;
		} else {
			return 0;
		}
	} else {
		/* no more values to handle at this level, other values were
		   checked after checking for "exists" / "doesn't exist" first */
		cdw_assert (0, "ERROR: file neither exists, nor doesn't exist\n");
		return -1;
	}
}
