/*
 * COPYRIGHT AND PERMISSION NOTICE
 * 
 * Copyright (c) 2003 G.J. Andruk
 * 
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY
 * SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 * 
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include "base64.h"
#include "sha1.h"
#include "hmac_sha1.h"
#include "canlock.h"

#define BUFFSIZE 40

/* 
 * Return a stripped cancel lock, that is, with the xx: prefix
 * removed, else NULL on failure.
 * type is set to the lock type, else zero on failure.
 */
char *
lock_strip_alpha(const char *key, char *type)
{
    char
        *c,
        *typetext,
        *mykey = (char *)key;
    size_t
        ttpos = 0,
        ttlen = 256;

    typetext = (char *) malloc(ttlen);
    if (!typetext)
        return NULL;
    *typetext = 0;

    while (*mykey && *mykey != ':') {
        if (ttpos >= ttlen) {
            ttlen += 256;
            typetext = (char *) realloc( (void *)typetext, ttlen);
            if (!typetext)
                return NULL;
        }
        typetext[ttpos++] = *mykey++;
    }
    if (! *mykey)
        type = NULL;
    else {
        mykey++;
        for (c = mykey; *c; c++) {
            if (*c == ':')
                *c = '\0';
        }
        strcpy(type, typetext);
        for (c = type; *c; ++c)
            *c = tolower(*c);
    }
    return (mykey);
}


char *
lock_strip(const char *key, char *type)
{
    return lock_strip_alpha(key, type);
}

/* 
 * Generate an SHA1 cancel key.
 * Returns a malloc()'d buffer that the caller will need to free().
 */
char *
sha_key(const unsigned char *secret, size_t seclen,
        const unsigned char *message, size_t msglen)
{
    char
        *cankey[1];
    unsigned char
        *hmacbuff;
    size_t
        keysize;

    hmacbuff = hmac_sha1(secret, seclen, message, msglen);
    if (!hmacbuff)
        return NULL;
    keysize = base64_encode(hmacbuff, SHA1HashSize, cankey);
    free ((void *) hmacbuff);
    if (!keysize)
        return NULL;
    *cankey = (char *) realloc((void *) *cankey, keysize + 6);
    if (!*cankey)
        return NULL;
    memmove((void *) (*cankey + 5), (void *) *cankey, keysize + 1);
    strncpy(*cankey, "sha1:", 5);
    return (*cankey);
}

/* 
 * Generate an SHA1 cancel lock.
 * Returns a malloc()'d buffer that the caller will need to free().
 */
char *
sha_lock(const unsigned char *secret, size_t seclen,
         const unsigned char *message, size_t msglen)
{
    char
        *canlock[1],
        junk[SHA1HashSize];
    unsigned char
        *cankey,
        hmacbuff[SHA1HashSize];
    size_t
        locksize;
    SHA1Context
        hash_ctx;

    cankey = (unsigned char *) lock_strip_alpha(
                            sha_key(secret, seclen, message, msglen), junk);
    if (!cankey)
        return NULL;
    if (SHA1Reset(&hash_ctx))
        return NULL;
    if (SHA1Input(&hash_ctx, cankey, strlen((char *) cankey)))
        return NULL;
    if (SHA1Result(&hash_ctx, hmacbuff))
        return NULL;
    locksize = base64_encode(hmacbuff, SHA1HashSize, canlock);
    if (!locksize)
        return NULL;
    *canlock = (char *) realloc((void *) *canlock, locksize + 6);
    if (!*canlock)
        return NULL;
    memmove((void *) (*canlock + 5), (void *) *canlock, locksize + 1);
    strncpy(*canlock, "sha1:", 5);
    return (*canlock);
}


/* 
 * Verify an SHA cancel key against a cancel lock.
 * Returns 0 on success, nonzero on failure.
 */
int
sha_verify(const char *key, const char *lock)
{
    unsigned char
        binkey[SHA1HashSize + 4],
        hmacbuff[SHA1HashSize];
    char
        *templock[1];
    size_t
        keysize,
        locksize;
    SHA1Context
        hash_ctx;


    /* Convert the key back into binary */
    keysize = base64_decode(key, (void *) &binkey);
    if (!keysize)
        return -1;

    if (SHA1Reset(&hash_ctx))
        return -1;
    if (SHA1Input(&hash_ctx, (unsigned char *)key, strlen(key)))
        return -1;
    if (SHA1Result(&hash_ctx, hmacbuff))
        return -1;

    locksize = base64_encode(hmacbuff, SHA1HashSize, templock);
    if (!locksize)
        return -1;

    return strcmp(*templock, lock);
}
