/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2016 Univ. Grenoble Alpes, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef LOADSMANAGER_H
#define LOADSMANAGER_H

#include <QString>
#include <QSet>
#include <vector>

#include "PMComponentAPI.h"

class QDockWidget;

class QWidget;

// class used in this file
class Loads;
class Load;
class AtomDecoration;
class PhysicalModel;
class Loads;
class PMManagerDC;
class LoadsMovie;
class LoadsEditor;
class LoadsSimulation;
class Atom;

namespace std {
/** definition of a couple (=STL pair) [Atom *, double]
  * this associates an atom to a given data (scalar value)
  */
typedef std::pair<Atom *, double> AtomDataPair;
/// a vector of atom data
typedef std::vector<std::AtomDataPair> AtomDataVector;
}

/** This class manages the applied all loads (see LML), deformation add-on,
  * atom data monitoring and atom decoration.
  *
  * Thanks to contribution from: Pascale Ducloux, Aude Flourens, Jocelyne Mekontso, Christine Oddes.
  *
  *

  **/

class PHYSICALMODEL_COMPONENT_API LoadsManager {

public:
    /// Constructor (give it the PMManagerDC it is linked with)
    LoadsManager(PMManagerDC *);

    /// Destructor
    ~LoadsManager();

    /** @name Load Management
     *  input/output of load for the simulation
     */
    ///@{
    /** Load a load file and its loads' list
     * The filename should have the .lml extension (impload file)
     * @return true if the file was succesfully read,
        *  @return false otherwise (problem or unknown extension).
        */
    bool open(const QString&);

    /// save the current load list in the current file name
    void save();

    /// save the current load list in a new file
    void saveAs(const QString&);

    /// get the current load list
    Loads * getLoads();

    /// set the current load list (delete previous if existing)
    void setLoads(Loads *);

    /// delete all current loads
    void deleteAllLoads();

    /** close the current load list (means do not save, put the current filename to ""
      * and physicalmodel ptr to NULL (but of course don not delete it!)
      */
    void close();

    /// add a load to the list
    void addLoad(Load *);

    /// show the edit dialog
    void editLoads();

    /// show the preview dialog
    void previewLoads();

    /// show the add load dialog (add loads on all currently selected AtomDC atoms)
    void addLoad();
    ///@}

    /** @name Simulation
     *  methods that deals with the simulation
     */
    ///@{
    /// get the current LoadsSimulation
    LoadsSimulation * getLoadsSimulation();

    /// show the animation motor add-on dialog
    void simulation();

    /// add another widget in the simulation tab
    void addSimulationTab(QWidget*);

    /// stop the current simulation or movie
    void pause();

    /// rewind the current simulation or movie
    void rewind();
    ///@}

    /** @name 3D Display
     *  methods in charge of change in the 3D display
     */
    ///@{

    /// To update the display of current time using the animation motor add-on (if it exists) actual time
    void updateTime();

    /// show/unshow the load in 3D
    void setDisplayLoads(bool);

    /// update the current 3D representation of the loads (arrows...)
    void updateLoadsDisplay();

    /// return true only if the loads are currently being displayed in 3D
    bool displayLoads() const;
    ///@}

    /** @name Atom Data Display
     *  methods in charge of the atom display (color for each node, interpolated if the representation is surfacic)
     */
    ///@{
    /// type of atom data display (exclusive)
    enum AtomDataDisplayType {
        NONE,                           ///< do not display any atom data
        ADD_ON,                         ///< display add-on atom data (i.e. atom data computed by the add-on)
        DISPLACEMENTS,                  ///< display total displacements from initial position
        DISTANCES,                      ///< display distances from the reference PML
        RELATIVE_ENERGY_NORM_ERROR      ///< display the relative energy norm error (as a percentage)
    };

    /// set the type of atom data display you want to see, for ADD_ON please use setAtomData
    void setAtomDataDisplay(AtomDataDisplayType);

    /// get current atom display type
    AtomDataDisplayType getAtomDataDisplay() const;

    /// set the file name for the reference Physical Model
    void setReferencePML(const QString&);

    /// return the reference physical model to use for some computations
    PhysicalModel * getReferencePM() const;

    /** set the new atom data (one scalar per atom), and update display
     * @param values values to show
     * @param name for add-ons, this is the specific atom data name to show on screen
     */
    void setAtomData(std::AtomDataVector &values, QString name = "Add-On Monitor");

    /// get the current list of all the atom data
    std::AtomDataVector & getAtomData();

    /// update the color scale of the atom data (min and max values)
    void updateAtomDataScale(double min, double max);
    //@}

    /** @name Other (Utilities)
     *  other methods (generaly utilities)
     */
    //@{
    /// return the current state regarding modification since the last save
    bool isModified() const;

    ///set the Loads changed flag to false;
    void setChangedFlagOff();

    ///return the loads changed flag value;
    bool getLoadsChangedFlag();

    /// ask the user if (s)he wants to save the last changed
    bool userWantsToSave();

    /// get the current physical model
    PMManagerDC * getPMManagerDC();

    /// get the animation motor add-onlocation (.dll / .so / .dylib)
    QString getAnimationMotorAddonLocation();

    /// set the animation motor add-on location
    void setAnimationMotorAddonLocation(QString);

    /// set the data scale state (user constrained or driven by min/max in the simulution)
    void userConstrainedAtomDataScale ( bool constrained);

    /// get the current state of the data scale
    bool getUserConstrainedAtomDataScale();

    /** get the user constrained values of the data scale (or current min/max in the data scale is not user constrained)
    * @param min the min value of the data scale
    * @param max the max value of the data scale
    */
    void getAtomDataScale ( double* min , double* max);

    //@}

private:

    /// the PM manager DC
    PMManagerDC *myPMManagerDC;

    /// List of arrow and sphere representing the loads in 3D
    QSet <AtomDecoration *> representation3D;

    /// List of atom data values (one value per atom)
    std::AtomDataVector atomData;

    /// the current name of managed file
    QString fileName;

    /// all the managed loads
    Loads * myLoads;

    /// had the load list been changed since the last time it was saved
    bool changed;

    /// has the load list been changed since the last simulation
    bool loadsChangedFlag;

    /// the dialog where all the loads are shown, and can be modified
    LoadsEditor *loadsEditor;

    /// the preview (simulation animation dialog)
    LoadsMovie *previewDialog;

    /// the simulation dialog
    LoadsSimulation *simulationDialog;

    /// the dock widget where the simulation dialog is
    QDockWidget *simDock;

    /// the current animation motor add-on location
    QString motorAddonLocation;

    /// are the load currently displayed in 3D?
    bool showLoads;

    /// which atom data are to display: none, displacements, distances, ...
    AtomDataDisplayType atomDataDisplay;

    /// the physical model to use as the reference for some atom data display
    PhysicalModel *referencePM;

    /// name of the add-on atom data
    QString atomDataName;

    /** add a decoration to atom.
      * @param a the atom on which the decoration will be added
      * @param ld the decoration type is dependent of the type of ld (translation, force, null translation...)
      * @param time the current time
      * @param defaultSize size by default
      * @param min minimal value (can be updated)
      * @param max maximal value (can be updated)
      * @param val current value (always updated)
      */
    void addDecoration(Atom *a, Load *ld, double time, double defaultSize, double *max, double *min, double *val);

    /// current state of the data scale
    bool constrainedAtomDataScale;

    /// current minimum value of the data scale
    double min;

    /// current maximum value of the data scale
    double max;
};

inline PMManagerDC * LoadsManager::getPMManagerDC() {
    return myPMManagerDC;
}

inline LoadsSimulation * LoadsManager::getLoadsSimulation() {
    return simulationDialog;
}

inline bool LoadsManager::isModified() const {
    return changed;
}

inline bool LoadsManager::displayLoads() const {
    return showLoads;
}

inline PhysicalModel * LoadsManager::getReferencePM() const {
    return referencePM;
}

inline LoadsManager::AtomDataDisplayType LoadsManager::getAtomDataDisplay() const {
    return atomDataDisplay;
}

inline void LoadsManager::setChangedFlagOff() {
    loadsChangedFlag = false;
}

inline bool LoadsManager::getLoadsChangedFlag() {
    return loadsChangedFlag;
}

inline std::AtomDataVector & LoadsManager::getAtomData() {
    return atomData;
}

#endif
