#ifndef DEBTAGS_ENVIRONMENT_H
#define DEBTAGS_ENVIRONMENT_H

/*
 * Copyright (C) 2003--2010  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#pragma interface

#include <sigc++/sigc++.h>
#include <buffy/config/config.h>
#include <buffy/mailfolder.h>

class Environment
{
public:
	// void child_exited(int pid, int status);
	typedef sigc::signal2<void, int, int> type_signal_child_exit;

protected:
	static Environment* instance;

	// True when operations should be verbose
	bool _verbose;

	// True when operations should be very verbose
	bool _debug;

	// Configuration access
	buffy::config::Config _config;

	type_signal_child_exit _signal_child_exit;

	void on_sigchld();

	Environment();

public:
	// editable: if true, allow modifications to metadata (where implemented)
	// facet_only: if true, remove all tags without facets; if false, add a
	//   facet-name tag for each faceted tag
	static void init();
	static Environment& get() throw () { return *instance; }

	// Get the configuration access object
	buffy::config::Config& config() throw () { return _config; }

	// Scan folder locations for folders and send what is found to `cons'
	void enumerateFolders(buffy::MailFolderConsumer& cons);
	
	// Fork a process to run the command specified
	void spawnCommand(const std::string& command);

	// Logging output control functions

	bool verbose() const throw () { return _verbose; }
	bool verbose(bool verbose) throw () { return _verbose = verbose; }

	bool debug() const throw () { return _debug; }
	bool debug(bool debug) throw ()
	{
		// Debug implies verbose
		if (debug)
			_verbose = true;
		return _debug = debug;
	}

	type_signal_child_exit signal_child_exit() throw ()
		{ return _signal_child_exit; }
};


// Commodity output functions

#ifndef ATTR_PRINTF
 #ifdef GCC
  #define ATTR_PRINTF(string, first) __attribute__((format (printf, string, first)))
 #else
  #define ATTR_PRINTF(string, first)
 #endif
#endif

void fatal_error(const char* fmt, ...) throw() ATTR_PRINTF(1, 2);
void error(const char* fmt, ...) throw() ATTR_PRINTF(1, 2);
void warning(const char* fmt, ...) throw() ATTR_PRINTF(1, 2);
void verbose(const char* fmt, ...) throw() ATTR_PRINTF(1, 2);
void debug(const char* fmt, ...) throw() ATTR_PRINTF(1, 2);
void feedback(const char* fmt, ...) throw() ATTR_PRINTF(1, 2);

// vim:set ts=4 sw=4:
#endif
