#! /usr/bin/python3
# -*- coding: utf-8 -*-

# Copyright(C) 2013 Mark Tully <markjtully@gmail.com>
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

from gi.repository import GLib, Gio
from gi.repository import Unity
import gettext
import urllib.parse
import dbus
import hashlib
import unicodedata
import os
import shutil
import sqlite3

APP_NAME = 'unity-scope-guayadeque'
LOCAL_PATH = '/usr/share/locale/'
gettext.bindtextdomain(APP_NAME, LOCAL_PATH)
gettext.textdomain(APP_NAME)
_ = gettext.gettext

GROUP_NAME = 'com.canonical.Unity.Scope.Music.Guayadeque'
UNIQUE_PATH = '/com/canonical/unity/scope/music/guayadeque'

SEARCH_HINT = _('Search Guayadeque')
NO_RESULTS_HINT = _('Sorry, there are no Guayadeque results that match your search.')
PROVIDER_CREDITS = _('')
SVG_DIR = '/usr/share/icons/unity-icon-theme/places/svg/'
PROVIDER_ICON = SVG_DIR + 'service-guayadeque.svg'
DEFAULT_RESULT_ICON = SVG_DIR + 'result-guayadeque.svg'
DEFAULT_RESULT_MIMETYPE = 'taglib/mp3'
DEFAULT_RESULT_TYPE = Unity.ResultType.PERSONAL
GUAYADEQUE_DBFILE = os.getenv("HOME") + "/.guayadeque/Collections"

c1 = {'id': 'songs',
      'name': _('Songs'),
      'icon': SVG_DIR + 'group-installed.svg',
      'renderer': Unity.CategoryRenderer.VERTICAL_TILE}
c2 = {'id': 'albums',
      'name': _('Albums'),
      'icon': SVG_DIR + 'group-installed.svg',
      'renderer': Unity.CategoryRenderer.VERTICAL_TILE}
CATEGORIES = [c1, c2]

FILTERS = []

m1 = {'id': 'album',
      'type': 's',
      'field': Unity.SchemaFieldType.OPTIONAL}
m2 = {'id': 'artist',
      'type': 's',
      'field': Unity.SchemaFieldType.OPTIONAL}
m3 = {'id': 'genre',
      'type': 's',
      'field': Unity.SchemaFieldType.OPTIONAL}
m4 = {'id': 'year',
      'type': 'i',
      'field': Unity.SchemaFieldType.OPTIONAL}
m5 = {'id': 'track_length',
      'type': 'i',
      'field': Unity.SchemaFieldType.OPTIONAL}
m6 = {'id': 'track_number',
      'type': 'i',
      'field': Unity.SchemaFieldType.OPTIONAL}
EXTRA_METADATA = [m1, m2, m3, m4, m5, m6]

SEARCH_SQL = '''SELECT songs.song_name, songs.song_path, songs.song_filename, songs.song_artist, songs.song_album, songs.song_albumartist, covers.cover_path, songs.song_year, songs.song_genre, songs.song_number, songs.song_length
                FROM songs, covers
                WHERE songs.song_coverid = covers.cover_id
                AND (songs.song_name LIKE '%%%s%%' OR songs.song_album like '%%%s%%' OR songs.song_artist LIKE '%%%s%%')
                ORDER BY song_number'''

ALBUM_SQL = '''SELECT songs.song_name, songs.song_path, songs.song_filename, songs.song_artist, songs.song_album, songs.song_albumartist, covers.cover_path, songs.song_year, songs.song_genre, songs.song_number, songs.song_length
                FROM songs, covers
                WHERE songs.song_coverid = covers.cover_id
                AND (songs.song_album like '%%%s%%' AND songs.song_artist LIKE '%%%s%%')
                ORDER BY song_number'''


def get_music_from_guayadeque(query):
    """Parses Guayadeque's collections into a form we can use"""
    tracks = []
    # Copy guayadeque's database to a backup so we can run searches on that rather than the main database
    for collection in os.listdir(GUAYADEQUE_DBFILE):
        collection_tracks = []
        guayadeque_dbfile = GUAYADEQUE_DBFILE + "/" + collection + "/guayadeque.db"
        if not os.path.exists(guayadeque_dbfile):
            continue
        guayadeque_backupfile = GUAYADEQUE_DBFILE + "/" + collection + "/guayadeque-scope-backup.db"

        # Only make a backup of guayadeque's database if the database has been modified since
        # the last backup copy was made (or if the backup doesn't exist) 
        if os.path.exists(guayadeque_backupfile):
            if os.path.getmtime(guayadeque_dbfile) > os.path.getmtime(guayadeque_backupfile):
                shutil.copy2(guayadeque_dbfile, guayadeque_backupfile)
        else:
            shutil.copy2(guayadeque_dbfile, guayadeque_backupfile)

        # Grab all the data we need from the backup
        conn = sqlite3.connect(guayadeque_backupfile)
        cursor = conn.cursor()
        cursor.execute(query)
        collection_tracks = cursor.fetchall()
        cursor.close()
        for track in collection_tracks:
            tracks.append(track)
    return tracks


def get_album_art(track):
    """Finds album art for the selected album
    """
    if not track[6] is None:
        return track[6]

    # If that fails, thumbnail any embedded album art and use that
    hashname = unicodedata.normalize("NFKD", unicode(track[4])) + "\t" + unicodedata.normalize("NFKD", unicode(track[3]))
    file_hash = hashlib.md5(hashname).hexdigest()
    tb_filename = os.path.join(os.path.expanduser("~/.cache/media-art"), ("album-" + file_hash)) + ".jpg"
    if os.path.exists(tb_filename):
        return tb_filename
    else:
        try:
            from mutagen import File
            audio = File(track[1])
            if "APIC:" in audio:
                artwork = audio.tags["APIC:"].data
                if not os.path.exists(os.path.expanduser("~/.cache/media-art")):
                    os.makedirs(os.path.expanduser("~/.cache/media-art"))
                with open(tb_filename, "wb") as img:
                    img.write(artwork)
                return tb_filename
            else:
                return "audio-x-generic"
        except:
            # Otherwise, return a generic audio icon
            return "audio-x-generic"


def search(search, filters):
    '''
    Search for help documents matching the search string
    '''
    results = []
    tracks = get_music_from_guayadeque(SEARCH_SQL % (search, search, search))
    albumresults = []
    for track in tracks:
        title = "" if track[0] is None else track[0]
        uri = "" if track[1] is None else track[1] + track[2]
        artist = "" if track[2] is None else track[3]
        album = "" if track[3] is None else track[4]
        albumartist = "" if track[5] is None else track[5]
        year = 0 if track[7] is None else track[7]
        genre = "" if track[8] is None else track[8]
        track_length = 0 if track[10] is None else track[10]
        track_number = 0 if track[9] is None else track[9]
        try:
            albumart = get_album_art(track)
        except:
            albumart = ""
        albumuri = "album://" + albumartist + "/" + album
        results.append({'uri': uri,
                        'icon': albumart,
                        'category': 0,
                        'title': title,
                        'album': GLib.Variant('s', album),
                        'artist': GLib.Variant('s', artist),
                        'genre': GLib.Variant('s', genre),
                        'year': GLib.Variant('i', year),
                        'track_length': GLib.Variant('i', track_length),
                        'track_number': GLib.Variant('i', track_number)})

        if album not in albumresults:
            results.append({'uri': albumuri,
                            'icon': albumart,
                            'category': 1,
                            'title': album,
                            'album': GLib.Variant('s', album),
                            'artist': GLib.Variant('s', artist),
                            'genre': GLib.Variant('s', genre),
                            'year': GLib.Variant('i', year),
                            'track_length': GLib.Variant('i', track_length),
                            'track_number': GLib.Variant('i', track_number)})
            albumresults.append(album)
    return results


class Preview(Unity.ResultPreviewer):

    def do_run(self):
        album = self.result.metadata['album'].get_string()
        artist = self.result.metadata['artist'].get_string()
        preview = Unity.MusicPreview.new(self.result.title, '', None)
        preview.props.image_source_uri = 'file://%s' % self.result.icon_hint
        preview.props.subtitle = self.result.metadata['artist'].get_string()
        if self.result.uri.startswith("album://"):
            tracks = get_music_from_guayadeque(ALBUM_SQL % (album, artist))
            for track in tracks:
                track = Unity.TrackMetadata.full('file://%s%s' % (track[1], track[2]),
                                                 track[9],
                                                 track[0],
                                                 track[3],
                                                 track[4],
                                                 track[10])
                preview.add_track(track)
        else:
            track = Unity.TrackMetadata.full('file://%s' % self.result.uri,
                                             self.result.metadata['track_number'].get_int32(),
                                             self.result.title,
                                             self.result.metadata['artist'].get_string(),
                                             self.result.metadata['album'].get_string(),
                                             self.result.metadata['track_length'].get_int32())
            preview.add_track(track)

        view_action = Unity.PreviewAction.new("play", _("Play"), None)
        preview.add_action(view_action)
        show_action = Unity.PreviewAction.new("show", _("Show in Folder"), None)
        preview.add_action(show_action)
        return preview


# Classes below this point establish communication
# with Unity, you probably shouldn't modify them.


class MySearch(Unity.ScopeSearchBase):
    def __init__(self, search_context):
        super(MySearch, self).__init__()
        self.set_search_context(search_context)

    def do_run(self):
        '''
        Adds results to the model
        '''
        try:
            result_set = self.search_context.result_set
            for i in search(self.search_context.search_query,
                            self.search_context.filter_state):
                if not 'uri' in i or not i['uri'] or i['uri'] == '':
                    continue
                if not 'icon' in i or not i['icon'] or i['icon'] == '':
                    i['icon'] = DEFAULT_RESULT_ICON
                if not 'mimetype' in i or not i['mimetype'] or i['mimetype'] == '':
                    i['mimetype'] = DEFAULT_RESULT_MIMETYPE
                if not 'result_type' in i or not i['result_type'] or i['result_type'] == '':
                    i['result_type'] = DEFAULT_RESULT_TYPE
                if not 'category' in i or not i['category'] or i['category'] == '':
                    i['category'] = 0
                if not 'title' in i or not i['title']:
                    i['title'] = ''
                if not 'comment' in i or not i['comment']:
                    i['comment'] = ''
                if not 'dnd_uri' in i or not i['dnd_uri'] or i['dnd_uri'] == '':
                    i['dnd_uri'] = i['uri']
                i['provider_credits'] = GLib.Variant('s', PROVIDER_CREDITS)
                result_set.add_result(**i)
        except Exception as error:
            print(error)


class Scope(Unity.AbstractScope):
    def __init__(self):
        Unity.AbstractScope.__init__(self)

    def do_get_search_hint(self):
        return SEARCH_HINT

    def do_get_schema(self):
        '''
        Adds specific metadata fields
        '''
        schema = Unity.Schema.new()
        if EXTRA_METADATA:
            for m in EXTRA_METADATA:
                schema.add_field(m['id'], m['type'], m['field'])
        #FIXME should be REQUIRED for credits
        schema.add_field('provider_credits', 's', Unity.SchemaFieldType.OPTIONAL)
        return schema

    def do_get_categories(self):
        '''
        Adds categories
        '''
        cs = Unity.CategorySet.new()
        if CATEGORIES:
            for c in CATEGORIES:
                cat = Unity.Category.new(c['id'], c['name'],
                                         Gio.ThemedIcon.new(c['icon']),
                                         c['renderer'])
                cs.add(cat)
        return cs

    def do_get_filters(self):
        '''
        Adds filters
        '''
        fs = Unity.FilterSet.new()
        #if FILTERS:
        #
        return fs

    def do_get_group_name(self):
        return GROUP_NAME

    def do_get_unique_name(self):
        return UNIQUE_PATH

    def do_create_search_for_query(self, search_context):
        se = MySearch(search_context)
        return se

    def do_activate(self, result, metadata, id):
        album = result.metadata['album'].get_string()
        artist = result.metadata['artist'].get_string()

        if id == 'show':
            if result.uri.startswith("album://"):
                tracks = get_music_from_guayadeque(ALBUM_SQL % (album, artist))
                filename = tracks[0][1].decode('utf-8')
            else:
                filename = result.uri
            dirname = os.path.dirname(filename)
            os.system("xdg-open '%s'" % str(dirname))
        else:
            albumtracks = ''
            if result.uri.startswith('album://'):
                tracks = get_music_from_guayadeque(ALBUM_SQL % (album, artist))
                for track in tracks:
                    albumtracks = albumtracks + ' \'%s\'' % (urllib.parse.unquote(str(track[1]) + str(track[2])))
            else:
                albumtracks = '\'%s\'' % result.uri
            print(albumtracks)
            os.system('guayadeque %s' % albumtracks)

        try:
            session_bus = dbus.SessionBus()
            player = session_bus.get_object('org.mpris.guayadeque', '/Player')
            iface = dbus.Interface(player, dbus_interface='org.freedesktop.MediaPlayer')
            iface.Play()
        except Exception as e:
            print('Error playing file: ' % e)

        return Unity.ActivationResponse(handled=Unity.HandledType.HIDE_DASH, goto_uri=None)

    def do_create_previewer(self, result, metadata):
        rp = Preview()
        rp.set_scope_result(result)
        rp.set_search_metadata(metadata)
        return rp


def load_scope():
    return Scope()
