/*
 *  XSMC-CALC: Smith Chart Calculator program for X
 * 
 *
 *      by Lapo Pieri (IK5NAX)  2000-2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Send bugs reports, comments, critique, etc, to ik5nax@amsat.org
 */

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>
#include <X11/Xlib.h>
#include <X11/cursorfont.h>
#include <X11/keysym.h>
#include "smc.h"

extern Display       *display;
extern GC            gc;
extern Pixmap        schart;
extern XFontStruct   *fontdialogname;

extern unsigned long numcolors, colorBLACK, colorWHITE, colorBG,
  colorZ, colorY, colorRHO, colorQ, colorSI, colorDW;
extern unsigned int wdim, scdim;
extern SMCDATA smcdata;
extern float tunestep;
extern int Zcirc, Ycirc, RHOcirc, Qcirc;

void drawchart(void) {
  int i, n, step;
  float R, X, G, B, sa, ea, a;
  float rer, imr, rer1, imr1, rer2, imr2;
 
 XSetForeground(display, gc, colorBG);
 XFillRectangle(display, schart, gc, 0, 0, wdim, wdim);


 /* Lines between Smith chart and number */
 XSetForeground(display, gc, colorBLACK);
 XDrawLine(display, schart, gc, 0.02*wdim+scdim, 0, 0.02*wdim+scdim, wdim);
 XDrawLine(display, schart, gc, 0, 0.02*wdim+scdim, wdim, 0.02*wdim+scdim);

 /* Dialog window */
 XSetForeground(display, gc, colorDW);
 XFillRectangle(display, schart, gc, 0.02*wdim+scdim+1, 0.02*wdim+scdim+1,
		0.98*wdim-scdim-1, 0.98*wdim-scdim-1);

 /* and dialog entry point */
 XSetForeground(display, gc, colorWHITE);
 XFillRectangle(display, schart, gc, 0.04*wdim+scdim+1, 0.05*wdim+scdim+1,
		0.8*(0.98*wdim-scdim-1), 0.2*(0.98*wdim-scdim-1));
 XFillRectangle(display, schart, gc, 0.04*wdim+scdim+1, 
		0.05*wdim+scdim+1+0.3*(0.98*wdim-scdim-1),
		0.8*(0.98*wdim-scdim-1), 0.2*(0.98*wdim-scdim-1));
 XFillRectangle(display, schart, gc, 0.04*wdim+scdim+1, 
		0.05*wdim+scdim+1+0.6*(0.98*wdim-scdim-1),
		0.8*(0.98*wdim-scdim-1), 0.2*(0.98*wdim-scdim-1));


 XSetForeground(display, gc, colorBLACK);
 XDrawArc(display, schart, gc,                               /* Main circle */
	  0.01*wdim, wdim*0.01, scdim, scdim, 0, 360*64);
 XDrawArc(display, schart, gc,                               /* Central dot */
	  wdim*0.01+scdim/2-1, wdim*0.01+scdim/2-1, 2, 2, 0, 360*64);


 if(Zcirc) {
   drawRk(display, schart, gc, 0.5);
   drawRk(display, schart, gc, 1.);
   drawRk(display, schart, gc, 2.);
   drawXk(display, schart, gc, 0.);
   drawXk(display, schart, gc, -1.);
   drawXk(display, schart, gc, 1.);
   drawXk(display, schart, gc, -0.5);
   drawXk(display, schart, gc, 0.5);
   drawXk(display, schart, gc, -2.);
   drawXk(display, schart, gc, 2.);
 }

 if(Ycirc) {
   drawGk(display, schart, gc, 0.5);
   drawGk(display, schart, gc, 1.);
   drawGk(display, schart, gc, 2.);
   drawBk(display, schart, gc, 0.);
   drawBk(display, schart, gc, -1.);
   drawBk(display, schart, gc, 1.);
   drawBk(display, schart, gc, -0.5);
   drawBk(display, schart, gc, 0.5);
   drawBk(display, schart, gc, -2.);
   drawBk(display, schart, gc, 2.);
 }

 if(RHOcirc) {
   drawRHOk(display, schart, gc, 0.2);
   drawRHOk(display, schart, gc, 0.5);
   drawRHOk(display, schart, gc, 0.75);
 }

 if(Qcirc) {
   drawQk(display, schart, gc, 0.);
   drawQk(display, schart, gc, 1.);
   drawQk(display, schart, gc, 2.);
   drawQk(display, schart, gc, 0.5);
   drawQk(display, schart, gc, 5.);
 }

 drawSP(display, schart, gc, smcdata.rerhoSP, smcdata.imrhoSP);

 /* Draw arcs */
 for(i=1; i<=smcdata.ne; i++) {

   /* Series placed element, that could be an R, L, or C */
     switch(smcdata.ELplace[i]) {
     case 's':
       rho2z(smcdata.rerhoIP[i], smcdata.imrhoIP[i], &R, &X);
       if(smcdata.ELtype[i]=='r') {
	 if(fabs(X)>QZ) {
	   sa=thetaonX(smcdata.rerhoIP[i-1], smcdata.imrhoIP[i-1], X)*180/M_PI;
	   ea=thetaonX(smcdata.rerhoIP[i], smcdata.imrhoIP[i], X)*180/M_PI;
	   if(ea<0.) ea+=360; if(sa<0.) sa+=360;
	   drawX(display, schart, gc, X, sa, ea-sa);
	 }
	 else
	   drawX(display, schart, gc, X, 
		 smcdata.rerhoIP[i-1], smcdata.rerhoIP[i]);
       }
       else if(smcdata.ELtype[i]=='l' || smcdata.ELtype[i]=='c') {
	 sa=thetaonR(smcdata.rerhoIP[i-1], smcdata.imrhoIP[i-1], R)*180/M_PI;
	 ea=thetaonR(smcdata.rerhoIP[i], smcdata.imrhoIP[i], R)*180/M_PI;
	 if(ea<0.) ea+=360; if(sa<0.) sa+=360;
	 drawR(display, schart, gc, R, sa, ea-sa);
       }
       break;

       /* Parallel placed element, that could be an R, L, or C */
       case'p':
	 rho2y(smcdata.rerhoIP[i], smcdata.imrhoIP[i], &G, &B);
       if(smcdata.ELtype[i]=='r') {
	 if(fabs(B)>QZ) {
	   sa=thetaonB(smcdata.rerhoIP[i-1], smcdata.imrhoIP[i-1], B)*180/M_PI;
	   ea=thetaonB(smcdata.rerhoIP[i], smcdata.imrhoIP[i], B)*180/M_PI;
	   drawB(display, schart, gc, B, sa, ea-sa);
	 }
	 else 
	   drawB(display, schart, gc, B, 
		 smcdata.rerhoIP[i-1], smcdata.rerhoIP[i]);
	 
       }
       else if(smcdata.ELtype[i]=='l' || smcdata.ELtype[i]=='c') {
	 sa=thetaonG(smcdata.rerhoIP[i-1], smcdata.imrhoIP[i-1], G)*180/M_PI;
	 ea=thetaonG(smcdata.rerhoIP[i], smcdata.imrhoIP[i], G)*180/M_PI;
	 
	 drawG(display, schart, gc, G, sa, ea-sa);
       }
       break;
       
     /* The rest are for transmission line */
     case 't':
       rer1=smcdata.rerhoIP[i-1]; imr1=smcdata.imrhoIP[i-1];
       step=36*(1+(int)(smcdata.ELval0[i]/360));
       a=smcdata.ELval0[i]/180.*M_PI/step;
       for(n=0; n<step; n++) {
	 rhoz1z2(&rer1, &imr1, smcdata.z0, smcdata.ELval1[i]);
	 rer=rer1; imr=imr1;
	 rer2=rer*cos(2.*a)+imr*sin(2.*a);
	 imr2=-rer*sin(2.*a)+imr*cos(2.*a);
	 rhoz1z2(&rer1, &imr1, smcdata.ELval1[i], smcdata.z0);
	 rhoz1z2(&rer2, &imr2, smcdata.ELval1[i], smcdata.z0);
	 drawLINE(display, schart, gc, rer1, imr1, rer2, imr2);
	 rer1=rer2; imr1=imr2;
       }
       break;
     case 'c': 
     case 'o':
	 rho2y(smcdata.rerhoIP[i], smcdata.imrhoIP[i], &G, &B);       
	 sa=thetaonG(smcdata.rerhoIP[i-1], smcdata.imrhoIP[i-1], G)*180/M_PI;
	 ea=thetaonG(smcdata.rerhoIP[i], smcdata.imrhoIP[i], G)*180/M_PI;
	 drawG(display, schart, gc, G, sa, ea-sa);
       break;
     case 'l':
     case 'a':
       rho2z(smcdata.rerhoIP[i], smcdata.imrhoIP[i], &R, &X);
       sa=thetaonR(smcdata.rerhoIP[i-1], smcdata.imrhoIP[i-1], R)*180/M_PI;
       ea=thetaonR(smcdata.rerhoIP[i], smcdata.imrhoIP[i], R)*180/M_PI;
       if(ea<0.) ea+=360; if(sa<0.) sa+=360;
       drawR(display, schart, gc, R, sa, ea-sa);
       break;
       
     default:
       fprintf(stderr, "Unallowed connection\n");
     }
 }
 
 drawdata();
}

void drawRk(Display *display, Drawable drawable, GC gc, float R){

  float xc, yc, r;
  int x, y, w, h;

  XSetForeground(display, gc, colorZ);

  xc=R/(1.+R); yc=0; r=1./(1.+R);
  x=(int)((xc+1.)/2.*scdim); y=(int)((1.-yc)/2.*scdim);
  w=(int)(r*scdim); h=(int)(r*scdim);

  XDrawArc(display, drawable, gc,
	   x-w/2+wdim*0.01, y-h/2+wdim*0.01, w, h, 0, 360*64);
}


void drawXk(Display *display, Drawable drawable, GC gc, float X){

  float xc, yc, r;
  int x, y, w, h;

  XSetForeground(display, gc, colorZ);
  if(X!=0.) {
    xc=1.; yc=1./X; r=1./fabs(X);
    x=(int)((xc+1.)/2.*scdim); y=(int)((1.-yc)/2.*scdim);
    w=(int)(r*scdim); h=(int)(r*scdim);
    if(X<0.)
      XDrawArc(display, drawable, gc,
	       x-w/2+wdim*0.01, y-h/2+wdim*0.01, w, h, 
	       90*64, -64.*2.*180./M_PI*atan(X));
    else
      XDrawArc(display, drawable, gc,
	       x-w/2+wdim*0.01, y-h/2+wdim*0.01, w, h, 
	       -90*64, -64.*2.*180./M_PI*atan(X));
  }
  else 
    XDrawLine(display, drawable, gc, wdim*0.01, scdim/2+wdim*0.01, 
	      wdim*0.01+scdim, scdim/2+wdim*0.01);
}

void drawGk(Display *display, Drawable drawable, GC gc, float G){

  float xc, yc, r;
  int x, y, w, h;

  XSetForeground(display, gc, colorY);

  xc=-G/(1.+G); yc=0; r=1./(1.+G);
  x=(int)((xc+1.)/2.*scdim); y=(int)((1.-yc)/2.*scdim);
  w=(int)(r*scdim); h=(int)(r*scdim);

  XDrawArc(display, drawable, gc,
	   x-w/2+wdim*0.01, y-h/2+wdim*0.01, w, h, 0, 360*64);
}


void drawBk(Display *display, Drawable drawable, GC gc, float B){

  float xc, yc, r;
  int x, y, w, h;

  XSetForeground(display, gc, colorY);

  if(B!=0.) {
    xc=-1.; yc=1./B; r=1./fabs(B);
    x=(int)((xc+1.)/2.*scdim); y=(int)((1.-yc)/2.*scdim);
    w=(int)(r*scdim); h=(int)(r*scdim);
    
    if(B<0.)
      XDrawArc(display, drawable, gc,
	       x-w/2+wdim*0.01, y-h/2+wdim*0.01, w, h,
	       90*64, 64.*2.*180./M_PI*atan(B));
    else
      XDrawArc(display, drawable, gc,
	       x-w/2+wdim*0.01, y-h/2+wdim*0.01, w, h,
	       -90*64, 64.*2.*180./M_PI*atan(B));
  }
  else 
    XDrawLine(display, drawable, gc, wdim*0.01, scdim/2+wdim*0.01, 
	      wdim*0.01+scdim, scdim/2+wdim*0.01);
}


void drawRHOk(Display *display, Drawable drawable, GC gc, float RHO){

  float xc, yc, r;
  int x, y, w, h;

  XSetForeground(display, gc, colorRHO);

  xc=0; yc=0; r=RHO;
  x=(int)((xc+1.)/2.*scdim); y=(int)((yc+1.)/2.*scdim);
  w=(int)(r*scdim); h=(int)(r*scdim);

  XDrawArc(display, drawable, gc,
	   x-w/2+wdim*0.01, y-h/2+wdim*0.01, w, h, 0, 360*64);
}

void drawQk(Display *display, Drawable drawable, GC gc, float Q){
  float xc, yc, r;
  int x, y, w, h;

  XSetForeground(display, gc, colorQ);


  xc=0.; yc=1./Q; r=sqrt(1.+1./Q/Q);
  x=(int)((xc+1.)/2.*scdim); y=(int)((1.-yc)/2.*scdim);
  w=(int)(r*scdim); h=(int)(r*scdim);
  XDrawArc(display, drawable, gc,
	   x-w/2+wdim*0.01, y-h/2+wdim*0.01, w, h,
	   64*(-90+180/M_PI*atan(Q)), -64*(180/M_PI*2*atan(Q)));

  xc=0.; yc=-1./Q; r=sqrt(1.+1./Q/Q);
  x=(int)((xc+1.)/2.*scdim); y=(int)((1.-yc)/2.*scdim);
  w=(int)(r*scdim); h=(int)(r*scdim);
  XDrawArc(display, drawable, gc,
	   x-w/2+wdim*0.01, y-h/2+wdim*0.01, w, h,
	   64*(90-180/M_PI*atan(Q)), 64*(180/M_PI*2*atan(Q)));
}


void drawSP(Display *display, Drawable drawable, GC gc,
	    float rerho, float imrho) {
  int x, y;
  
  XSetForeground(display, gc, colorSI);

  x=(int)((rerho+1.)/2.*scdim); y=(int)((1.-imrho)/2.*scdim);

  XFillArc(display, drawable, gc, x-2+wdim*0.01, y-2+wdim*0.01, 4, 4,
	   0, 64*360);
}


void drawR(Display *display, Drawable drawable, GC gc, float R,
	   float sa, float al) {

  float xc, yc, r;
  int x, y, w, h;

  XSetForeground(display, gc, colorBLACK);
  XSetLineAttributes(display, gc, 2, LineSolid, CapRound, JoinRound);

  xc=R/(1.+R); yc=0; r=1./(1.+R);
  x=(int)((xc+1.)/2.*scdim); y=(int)((1.-yc)/2.*scdim);
  w=(int)(r*scdim); h=(int)(r*scdim);

  XDrawArc(display, drawable, gc,
	   x-w/2+wdim*0.01, y-h/2+wdim*0.01, w, h, sa*64, al*64);

  XSetLineAttributes(display, gc, 1, LineSolid, CapRound, JoinRound);
}


void drawX(Display *display, Drawable drawable, GC gc, float X,
	   float sa, float al) {

  float xc, yc, r;
  int x, y, w, h;

  XSetForeground(display, gc, colorBLACK);
  XSetLineAttributes(display, gc, 2, LineSolid, CapRound, JoinRound);

  if(fabs(X)>QZ) {
    xc=1.; yc=1./X; r=1./fabs(X);
    x=(int)((xc+1.)/2.*scdim); y=(int)((1.-yc)/2.*scdim);
    w=(int)(r*scdim); h=(int)(r*scdim);

    XDrawArc(display, drawable, gc,
	     x-w/2+wdim*0.01, y-h/2+wdim*0.01, w, h, sa*64, al*64);
  }

  else {
    yc=X/2.; y=(int)((1.-yc)/2.*scdim);
    yc=0; h=(int)((1.-yc)/2.*scdim);
    xc=sa; x=(int)((xc+1.)/2.*scdim);
    xc=al; w=(int)((xc+1.)/2.*scdim);
    XDrawLine(display, drawable, gc, 
	      x+wdim*0.01, y+wdim*0.01, w+wdim*0.01, h+wdim*0.01);
  }

  XSetLineAttributes(display, gc, 1, LineSolid, CapRound, JoinRound);
}

void drawG(Display *display, Drawable drawable, GC gc, float G,
	   float sa, float al){

  float xc, yc, r;
  int x, y, w, h;

  XSetForeground(display, gc, colorBLACK);
  XSetLineAttributes(display, gc, 2, LineSolid, CapRound, JoinRound);

  xc=-G/(1.+G); yc=0; r=1./(1.+G);
  x=(int)((xc+1.)/2.*scdim); y=(int)((1.-yc)/2.*scdim);
  w=(int)(r*scdim); h=(int)(r*scdim);

  XDrawArc(display, drawable, gc,
	   x-w/2+wdim*0.01, y-h/2+wdim*0.01, w, h, 
	   sa*64, al*64);

  XSetLineAttributes(display, gc, 1, LineSolid, CapRound, JoinRound);
}


void drawB(Display *display, Drawable drawable, GC gc, float B,
	   float sa, float al) {

  float xc, yc, r;
  int x, y, w, h;

  XSetForeground(display, gc, colorBLACK);
  XSetLineAttributes(display, gc, 2, LineSolid, CapRound, JoinRound);

  if(fabs(B)>QZ) {
    xc=-1.; yc=-1./B; r=1./fabs(B);
    x=(int)((xc+1.)/2.*scdim); y=(int)((1.-yc)/2.*scdim);
    w=(int)(r*scdim); h=(int)(r*scdim);
    
    XDrawArc(display, drawable, gc,
	     x-w/2+wdim*0.01, y-h/2+wdim*0.01, w, h, sa*64, al*64);
  }

  else {
    yc=-B/2.; y=(int)((1.-yc)/2.*scdim);
    yc=0; h=(int)((1.-yc)/2.*scdim);
    xc=sa; x=(int)((xc+1.)/2.*scdim);
    xc=al; w=(int)((xc+1.)/2.*scdim);
    XDrawLine(display, drawable, gc, 
	      x+wdim*0.01, y+wdim*0.01, w+wdim*0.01, h+wdim*0.01);
  }

  XSetLineAttributes(display, gc, 1, LineSolid, CapRound, JoinRound);
}


void drawRHO(Display *display, Drawable drawable, GC gc, float RHO,
	     float sa, float ea) {

  float xc, yc, r;
  int x, y, w, h;

  XSetForeground(display, gc, colorBLACK);
  XSetLineAttributes(display, gc, 2, LineSolid, CapRound, JoinRound);

  xc=0; yc=0; r=RHO;
  x=(int)((xc+1.)/2.*scdim); y=(int)((yc+1.)/2.*scdim);
  w=(int)(r*scdim); h=(int)(r*scdim);

  XDrawArc(display, drawable, gc,
	   x-w/2+wdim*0.01, y-h/2+wdim*0.01, w, h, sa*64, ea*64);

  XSetLineAttributes(display, gc, 1, LineSolid, CapRound, JoinRound);
}

void drawLINE(Display *display, Drawable drawable, GC gc,
	      float rerhoi, float imrhoi, float rerhof, float imrhof) {

  int x1, y1, x2, y2;

  XSetForeground(display, gc, colorBLACK);
  XSetLineAttributes(display, gc, 2, LineSolid, CapRound, JoinRound);


  x1=(int)((rerhoi+1.)/2.*scdim); y1=(int)((1.-imrhoi)/2.*scdim);
  x2=(int)((rerhof+1.)/2.*scdim); y2=(int)((1.-imrhof)/2.*scdim);

  XDrawLine(display, drawable, gc,
	    x1+wdim*0.01, y1+wdim*0.01, x2+wdim*0.01, y2+wdim*0.01);

  XSetLineAttributes(display, gc, 1, LineSolid, CapRound, JoinRound);
}


void drawdata(void) {
  char b[80], b2[80];
  int chrh, ls, i;
  float r, j;

  /* Setup font and get some dimension in pixel */
  XSetFont(display, gc, fontdialogname->fid);
  chrh=fontdialogname->ascent+fontdialogname->descent; ls=chrh*1.2;


  /* Write the vertical bar */

  /* Clear first */
  XSetForeground(display, gc, colorWHITE);
  XFillRectangle(display, schart, gc, 0.02*wdim+scdim+1, 0,
		 scdim-1-0.02*wdim, scdim+0.02*wdim);

  /* Display general data */
  XSetForeground(display, gc, colorBLACK);
  sprintf(b, "Z0=%4.0f", smcdata.z0);
  XDrawString(display, schart, gc, 0.02*wdim+scdim+8, 
	      ls, b, strlen(b));
  sprintf(b, "f0= "); float2prefs(smcdata.f0, b2, 0); 
  strcat(b, b2); strcat(b, "Hz");
  XDrawString(display, schart, gc, 0.02*wdim+scdim+8, 
	      2*ls, b, strlen(b));

  /* Start point */
  rho2z(smcdata.rerhoIP[0], smcdata.imrhoIP[0], &r, &j);
  r*=smcdata.z0; j*=smcdata.z0;
  sprintf(b, "zSP="); float2prefs(r, b2, 0); strcat(b, b2);
  sprintf(b2, "%cj", j>0. ? '+' : '-'); strcat(b, b2);
  float2prefs(fabs(j), b2, 0); strcat(b, b2);
  XDrawString(display, schart, gc, 0.02*wdim+scdim+8, 
	      4*ls, b, strlen(b));

  rho2y(smcdata.rerhoIP[0], smcdata.imrhoIP[0], &r, &j);
  r/=smcdata.z0; j/=smcdata.z0;
  sprintf(b, "ySP="); float2prefs(r, b2, 0); strcat(b, b2);
  sprintf(b2, "%cj", j>0. ? '+' : '-'); strcat(b, b2);
  float2prefs(fabs(j), b2, 0); strcat(b, b2);
  XDrawString(display, schart, gc, 0.02*wdim+scdim+8, 
	      5*ls, b, strlen(b));

  r=smcdata.rerhoIP[0]; j=smcdata.imrhoIP[0];
  sprintf(b, "rhoSP="); float2prefs(r, b2, 0); strcat(b, b2);
  sprintf(b2, "%cj", j>0. ? '+' : '-'); strcat(b, b2);
  float2prefs(fabs(j), b2, 0); strcat(b, b2);
  XDrawString(display, schart, gc, 0.02*wdim+scdim+8, 
	      6*ls, b, strlen(b));

  r=sqrt((smcdata.rerhoIP[0]*smcdata.rerhoIP[0])+
	 (smcdata.imrhoIP[0]*smcdata.imrhoIP[0]));
  sprintf(b, "SWR= %5.2f", (1.+r)/(1.-r));
  XDrawString(display, schart, gc, 0.02*wdim+scdim+8, 
	      7*ls, b, strlen(b));
  

  /* End point */
  rho2z(smcdata.rerhoIP[smcdata.ne], smcdata.imrhoIP[smcdata.ne], &r, &j);
  r*=smcdata.z0; j*=smcdata.z0;
  sprintf(b, "zEP="); float2prefs(r, b2, 0); strcat(b, b2);
  sprintf(b2, "%cj", j>0. ? '+' : '-'); strcat(b, b2);
  float2prefs(fabs(j), b2, 0); strcat(b, b2);
  XDrawString(display, schart, gc, 0.02*wdim+scdim+8, 
	      9*ls, b, strlen(b));

  rho2y(smcdata.rerhoIP[smcdata.ne], smcdata.imrhoIP[smcdata.ne], &r, &j);
  r/=smcdata.z0; j/=smcdata.z0;
  sprintf(b, "yEP="); float2prefs(r, b2, 0); strcat(b, b2);
  sprintf(b2, "%cj", j>0. ? '+' : '-'); strcat(b, b2);
  float2prefs(fabs(j), b2, 0); strcat(b, b2);
  XDrawString(display, schart, gc, 0.02*wdim+scdim+8, 
	      10*ls, b, strlen(b));

  r=smcdata.rerhoIP[smcdata.ne]; j=smcdata.imrhoIP[smcdata.ne];
  sprintf(b, "rhoEP="); float2prefs(r, b2, 0); strcat(b, b2);
  sprintf(b2, "%cj", j>0. ? '+' : '-'); strcat(b, b2);
  float2prefs(fabs(j), b2, 0); strcat(b, b2);
  XDrawString(display, schart, gc, 0.02*wdim+scdim+8, 
	      11*ls, b, strlen(b));

  r=sqrt((smcdata.rerhoIP[smcdata.ne]*smcdata.rerhoIP[smcdata.ne])+
	 (smcdata.imrhoIP[smcdata.ne]*smcdata.imrhoIP[smcdata.ne]));
  sprintf(b, "SWR= %5.2f", (1.+r)/(1.-r));
  XDrawString(display, schart, gc, 0.02*wdim+scdim+8, 
	      12*ls, b, strlen(b));

  /* Other data */
  sprintf(b, "tunestep %3.0f%%", (tunestep-1.)*100.);
  XDrawString(display, schart, gc, 0.02*wdim+scdim+8, 
	      15*ls, b, strlen(b));


  /* End point */
  rho2z(smcdata.rerhoCU, smcdata.imrhoCU, &r, &j);
  r*=smcdata.z0; j*=smcdata.z0;
  sprintf(b, "zCU="); float2prefs(r, b2, 0); strcat(b, b2);
  sprintf(b2, "%cj", j>0. ? '+' : '-'); strcat(b, b2);
  float2prefs(fabs(j), b2, 0); strcat(b, b2);
  XDrawString(display, schart, gc, 0.02*wdim+scdim+8, 
	      18*ls, b, strlen(b));

  rho2y(smcdata.rerhoCU, smcdata.imrhoCU, &r, &j);
  r/=smcdata.z0; j/=smcdata.z0;
  sprintf(b, "yCU="); float2prefs(r, b2, 0); strcat(b, b2);
  sprintf(b2, "%cj", j>0. ? '+' : '-'); strcat(b, b2);
  float2prefs(fabs(j), b2, 0); strcat(b, b2);
  XDrawString(display, schart, gc, 0.02*wdim+scdim+8, 
	      19*ls, b, strlen(b));

  r=smcdata.rerhoCU; j=smcdata.imrhoCU;
  sprintf(b, "rhoCU="); float2prefs(r, b2, 0); strcat(b, b2);
  sprintf(b2, "%cj", j>0. ? '+' : '-'); strcat(b, b2);
  float2prefs(fabs(j), b2, 0); strcat(b, b2);
  XDrawString(display, schart, gc, 0.02*wdim+scdim+8, 
	      20*ls, b, strlen(b));

  r=sqrt((smcdata.rerhoCU*smcdata.rerhoCU)+
	 (smcdata.imrhoCU*smcdata.imrhoCU));
  sprintf(b, "SWR= %5.2f", (1.+r)/(1.-r));
  XDrawString(display, schart, gc, 0.02*wdim+scdim+8, 
	      21*ls, b, strlen(b));


  /* Write the horizontal bar */

  /* Clear first */
  XSetForeground(display, gc, colorWHITE);
  XFillRectangle(display, schart, gc, 1, 0.02*wdim+scdim+1,
		 scdim-1+0.02*wdim, wdim);

  /* Display network description */
  XSetForeground(display, gc, colorBLACK);

  for(i=1; i<=smcdata.ne; i++) {

    switch(smcdata.ELtype[i]) {
    case 'r':
    case 'l':
    case 'c':
      if(i==smcdata.neidx) {
	XSetForeground(display, gc, colorDW);
	XFillRectangle(display, schart, gc, 
		       ls/4, 0.02*wdim+scdim+ls*(i-1)+ls/4, 5/4*ls, ls);
	XSetForeground(display, gc, colorBLACK);
      }
      sprintf(b, "#%d:", i); 
      XDrawString(display, schart, gc, ls/4, 0.02*wdim+scdim+ls*i,
		  b, strlen(b));

      sprintf(b, "%c", toupper(smcdata.ELtype[i]));
      XDrawString(display, schart, gc, ls/4+2*ls, 0.02*wdim+scdim+ls*i,
		  b, strlen(b));
      sprintf(b, "%c", smcdata.ELplace[i]);
      XDrawString(display, schart, gc, ls/4+4*ls, 0.02*wdim+scdim+ls*i,
		  b, strlen(b));
      float2prefs(smcdata.ELval0[i], b, 0);
      XDrawString(display, schart, gc, ls/4+6*ls, 0.02*wdim+scdim+ls*i,
		  b, strlen(b));
      break;
    case 't':
      if(i==smcdata.neidx) {
	XSetForeground(display, gc, colorDW);
	XFillRectangle(display, schart, gc, 
		       ls/4, 0.02*wdim+scdim+ls*(i-1)+ls/4, 5/4*ls, ls);
	XSetForeground(display, gc, colorBLACK);
      }
      sprintf(b, "#%d:", i); 
      XDrawString(display, schart, gc, ls/4, 0.02*wdim+scdim+ls*i,
		  b, strlen(b));

      sprintf(b, "%c", toupper(smcdata.ELtype[i]));
      XDrawString(display, schart, gc, ls/4+2*ls, 0.02*wdim+scdim+ls*i,
		  b, strlen(b));

      sprintf(b, "%c", smcdata.ELplace[i]);
      XDrawString(display, schart, gc, ls/4+4*ls, 0.02*wdim+scdim+ls*i,
		  b, strlen(b));

      float2prefs(smcdata.ELval0[i], b, 0);
      XDrawString(display, schart, gc, ls/4+6*ls, 0.02*wdim+scdim+ls*i,
		  b, strlen(b));

      float2prefs(smcdata.ELval1[i], b, 0);
      XDrawString(display, schart, gc, ls/4+11*ls, 0.02*wdim+scdim+ls*i,
		  b, strlen(b));

      break;

    default:
    }

  }

}

