/*
 * Copyright (C) 2000-2024 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>

#include "_xitk.h"
#include "button.h"
#include "backend.h"

typedef struct {
  xitk_widget_t       w;

  xitk_hv_t           img_state_list[XITK_IMG_STATE_LIST_SIZE];
  xitk_part_image_t   skin;

  /* callback function (active_widget, user_data) */
  xitk_simple_callback_t  callback;
  xitk_state_callback_t   state_callback;
} _button_private_t;

/**
 *
 */
static void _button_paint (_button_private_t *wp, const widget_event_t *event) {
  XITK_HV_INIT;
#ifdef XITK_PAINT_DEBUG
  printf ("xitk.button.paint (%d, %d, %d, %d).\n", event->x, event->y, event->width, event->height);
#endif
  if (wp->w.state & XITK_WIDGET_STATE_VISIBLE) {
    xitk_img_state_t state = xitk_image_find_state (wp->skin.num_states - 1, wp->w.state);

    xitk_part_image_draw (wp->w.wl, &wp->skin, NULL,
      (int)XITK_HV_H (wp->img_state_list[1 + state]) + event->x - XITK_HV_H (wp->w.pos),
      (int)XITK_HV_V (wp->img_state_list[1 + state]) + event->y - XITK_HV_V (wp->w.pos),
      event->width, event->height,
      event->x, event->y);
  }
}

static void _button_read_skin (_button_private_t *wp, xitk_skin_config_t *skonfig) {
  const xitk_skin_element_info_t *s = xitk_skin_get_info (skonfig, wp->w.skin_element_name);
  XITK_HV_INIT;

  xitk_widget_state_from_info (&wp->w, s);
  if (s) {
    wp->skin = s->pixmap_img;
    xitk_part_image_states (&wp->skin, wp->img_state_list, 3);
    XITK_HV_H (wp->w.pos) = s->x;
    XITK_HV_V (wp->w.pos) = s->y;
  } else {
    wp->skin.x      = 0;
    wp->skin.y      = 0;
    wp->skin.width  = 0;
    wp->skin.height = 0;
    wp->skin.image  = NULL;
    wp->w.pos.w     = 0;
  }
}

/*
 *
 */
static void _button_new_skin (_button_private_t *wp, xitk_skin_config_t *skonfig) {
  XITK_HV_INIT;

  if (wp->w.skin_element_name[0]) {
    _button_read_skin (wp, skonfig);
    wp->w.size.w = wp->img_state_list[0].w;
    xitk_set_widget_pos (&wp->w, XITK_HV_H (wp->w.pos), XITK_HV_V (wp->w.pos));
  }
}

static int _button_input (_button_private_t *wp, const widget_event_t *event) {
  uint32_t fire;

  if (!(wp->w.state & XITK_WIDGET_STATE_FOCUS))
    return 0;

  fire = event->pressed ? 0 : ~0u;
  wp->w.state &= ~XITK_WIDGET_STATE_CLICK;
  wp->w.state |= ~fire & XITK_WIDGET_STATE_CLICK;
  fire = (wp->w.state ^ fire) & XITK_WIDGET_STATE_IMMEDIATE;
  if (fire && (wp->w.state & XITK_WIDGET_STATE_TOGGLE))
    wp->w.state ^= XITK_WIDGET_STATE_ON;

  {
    widget_event_t ev2;
    XITK_HV_INIT;
    ev2.x = XITK_HV_H (wp->w.pos);
    ev2.y = XITK_HV_V (wp->w.pos);
    ev2.width = XITK_HV_H (wp->w.size);
    ev2.height = XITK_HV_V (wp->w.size);
    _button_paint (wp, &ev2);
    wp->w.shown_state = wp->w.state;
  }

  if (fire) {
    if (wp->state_callback)
      wp->state_callback (&wp->w, wp->w.userdata, !!(wp->w.state & XITK_WIDGET_STATE_ON));
    else if (wp->callback)
      wp->callback (&wp->w, wp->w.userdata);
  }
  return 1;
}

static int button_event (xitk_widget_t *w, const widget_event_t *event) {
  _button_private_t *wp;

  xitk_container (wp, w, w);
  if (!wp || ! event)
    return 0;
  if (((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_BUTTON)
    && ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_CHECKBOX))
    return 0;

  switch (event->type) {
    case WIDGET_EVENT_PAINT:
      _button_paint (wp, event);
      break;
    case WIDGET_EVENT_KEY:
      {
        const char k[] = {
          ' ', 0,
          XITK_CTRL_KEY_PREFIX, XITK_KEY_RETURN,
        };
        if (event->modifier & ~(MODIFIER_SHIFT | MODIFIER_NUML))
          break;
        if (!event->string)
          break;
        if ( !memcmp (event->string, k + 0, 2)
          || !memcmp (event->string, k + 2, 2))
          goto _input;
      }
      break;
    case WIDGET_EVENT_CLICK:
      if (event->button != 1)
        break;
    _input:
      return _button_input (wp, event);
    case WIDGET_EVENT_INSIDE:
      if (wp->w.state & XITK_WIDGET_STATE_VISIBLE) {
        XITK_HV_INIT;
        return xitk_image_inside (wp->skin.image,
          event->x + wp->skin.x - XITK_HV_H (wp->w.pos),
          event->y + wp->skin.y - XITK_HV_V (wp->w.pos)) ? 1 : 2;
      }
      break;
    case WIDGET_EVENT_CHANGE_SKIN:
      _button_new_skin (wp, event->skonfig);
      break;
    case WIDGET_EVENT_DESTROY:
      if (!wp->w.skin_element_name[0])
        xitk_image_free_image (&wp->skin.image);
      break;
    case WIDGET_EVENT_GET_SKIN:
      if (event->image) {
        *event->image = (event->skin_layer == FOREGROUND_SKIN) ? wp->skin.image : NULL;
        return 1;
      }
      break;
    default: ;
  }
  return 0;
}

/*
 *
 */
static xitk_widget_t *_xitk_button_create (_button_private_t *wp, const xitk_button_widget_t *b) {
  ABORT_IF_NULL (wp->w.wl);

  wp->callback          = b->callback;
  wp->state_callback    = b->state_callback;

  wp->w.size.w          = wp->img_state_list[0].w;
  wp->w.type            = (wp->state_callback ? WIDGET_TYPE_CHECKBOX : WIDGET_TYPE_BUTTON)
                        | WIDGET_CLICKABLE | WIDGET_FOCUSABLE | WIDGET_TABABLE
                        | WIDGET_KEYABLE | WIDGET_PARTIAL_PAINTABLE;
  wp->w.state          |= wp->state_callback ? XITK_WIDGET_STATE_TOGGLE : 0;
  wp->w.event           = button_event;

  _xitk_new_widget_apply (&b->nw, &wp->w);

  return &wp->w;
}

/*
 *
 */
xitk_widget_t *xitk_button_create (const xitk_button_widget_t *b, xitk_skin_config_t *skonfig) {
  _button_private_t *wp;

  wp = (_button_private_t *)xitk_widget_new (&b->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  _button_read_skin (wp, skonfig);

  return _xitk_button_create (wp, b);
}

/*
 *
 */
xitk_widget_t *xitk_noskin_button_create (const xitk_button_widget_t *b, int x, int y, int width, int height) {
  XITK_HV_INIT;
  _button_private_t *wp;
  xitk_button_widget_t _b;
  xitk_image_t *i;
  int states;

  wp = (_button_private_t *)xitk_widget_new (&b->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  _b = *b;
  _b.nw.skin_element_name = NULL;

  if (_b.symbol == XITK_SYMBOL_USER) {
    i = xitk_image_new (wp->w.wl->xitk, NULL, 0, width * 3, height);
    xitk_image_draw_bevel_style (i, XITK_DRAW_BEVEL | _b.style);
    states = 3;
  } else {
    char name[16] = "xitk_button_00";

    name[12] = '0' + _b.symbol;
    name[13] = '0' + ((_b.style >> 16) & 7);
    states = 6;
    if (xitk_shared_image (wp->w.wl, name, width * 6, height, &i) == 1) {
      i->last_state = XITK_IMG_STATE_DISABLED_SELECTED;
      switch (_b.symbol) {
        case XITK_SYMBOL_LEFT:
        case XITK_SYMBOL_RIGHT:
        case XITK_SYMBOL_UP:
        case XITK_SYMBOL_DOWN:
        case XITK_SYMBOL_PLUS:
        case XITK_SYMBOL_MINUS:
          xitk_image_draw_bevel_style (i, XITK_DRAW_BEVEL | _b.style);
          xitk_image_draw_symbol (i, _b.symbol);
          break;
        case XITK_SYMBOL_CHECK:
          xitk_image_draw_checkbox_check (i);
          break;
        /* case XITK_SYMBOL_NONE: */
        default:
          xitk_image_draw_bevel_style (i, XITK_DRAW_BEVEL | _b.style);
      }
    }
  }

  XITK_HV_H (wp->w.pos) = x;
  XITK_HV_V (wp->w.pos) = y;

  wp->w.skin_element_name[0] = 0;
  wp->skin.image = i;
  wp->skin.x        = 0;
  wp->skin.y        = 0;
  wp->skin.width    = i->width;
  wp->skin.height   = i->height;
  xitk_part_image_states (&wp->skin, wp->img_state_list, states);
  wp->w.state &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);

  return _xitk_button_create (wp, &_b);
}
