/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#include "nsCycleCollectorUtils.h"

#include "prthread.h"
#include "nsCOMPtr.h"
#include "nsServiceManagerUtils.h"
#include "nsXPCOMCIDInternal.h"

#include "nsIThreadManager.h"

#if defined(XP_WIN)
#include "windows.h"
#endif

#ifndef MOZILLA_INTERNAL_API

bool
NS_IsCycleCollectorThread()
{
  bool result = false;
  nsCOMPtr<nsIThreadManager> mgr =
    do_GetService(NS_THREADMANAGER_CONTRACTID);
  if (mgr)
    mgr->GetIsCycleCollectorThread(&result);
  return bool(result);
}

#elif defined(XP_WIN)

// Defined in nsThreadManager.cpp.
extern DWORD gTLSThreadIDIndex;

bool
NS_IsCycleCollectorThread()
{
  return TlsGetValue(gTLSThreadIDIndex) ==
    (void*) mozilla::threads::CycleCollector;
}

#elif !defined(NS_TLS)

// Defined in nsCycleCollector.cpp
extern PRThread* gCycleCollectorThread;

bool
NS_IsCycleCollectorThread()
{
  return PR_GetCurrentThread() == gCycleCollectorThread;
}

#endif

#ifdef WINE_GECKO_SRC

#include <new>
#include "nsCycleCollectionParticipant.h"

struct CCObjCallback {
    void (NS_STDCALL *unmark_if_purple)(void*);
    nsresult (NS_STDCALL *traverse)(nsCycleCollectionParticipant*,void*,nsCycleCollectionTraversalCallback&);
    nsresult (NS_STDCALL *unlink)(void*);
};

extern "C" {
    nsrefcnt __declspec(dllexport) ccref_incr(nsCycleCollectingAutoRefCnt *ref, nsISupports *base)
    {
        return ref->incr(base);
    }

    nsrefcnt __declspec(dllexport) ccref_decr(nsCycleCollectingAutoRefCnt *ref, nsISupports *base)
    {
        nsrefcnt count = ref->decr(base);
        if(!count)
            ref->stabilizeForDeletion();
        return count;
    }

    void __declspec(dllexport) ccref_init(nsCycleCollectingAutoRefCnt *ref, nsrefcnt refcnt)
    {
        static_assert(sizeof(nsCycleCollectingAutoRefCnt) == sizeof(void*), "Unexpected nsCycleCollectingAutoRefCnt size");
        new (ref) nsCycleCollectingAutoRefCnt(refcnt);
    }

    void __declspec(dllexport) ccref_unmark_if_purple(nsCycleCollectingAutoRefCnt *ref)
    {
        if(ref->HasPurpleBufferEntry())
            ref->ReleasePurpleBufferEntry();
    }

    void __declspec(dllexport) ccp_init(nsXPCOMCycleCollectionParticipant *ccp, const CCObjCallback *callback)
    {
        static_assert(sizeof(nsXPCOMCycleCollectionParticipant) == 9*sizeof(void*), "unexpected CCParticipant size");

        new (ccp) nsXPCOMCycleCollectionParticipant();

        ccp->TraverseReal = callback->traverse;
        ccp->Root = nsXPCOMCycleCollectionParticipant::RootImpl;
        ccp->Unlink = callback->unlink;
        ccp->Unroot = nsXPCOMCycleCollectionParticipant::UnrootImpl;
        ccp->UnmarkIfPurple = callback->unmark_if_purple;
        ccp->CanSkipReal = nullptr;
        ccp->CanSkipInCCReal = nullptr;
        ccp->CanSkipThisReal = nullptr;
        ccp->Trace = nsXPCOMCycleCollectionParticipant::TraceImpl;
    }

    void __declspec(dllexport) describe_cc_node(nsCycleCollectingAutoRefCnt *ref, const char *obj_name,
                                                nsCycleCollectionTraversalCallback *cb)
    {
        cb->DescribeRefCountedNode(ref->get(), obj_name);
    }

    void __declspec(dllexport) note_cc_edge(nsISupports *child, const char *name, nsCycleCollectionTraversalCallback *cb)
    {
        if(cb->WantDebugInfo())
            cb->NoteNextEdgeName(name);
        cb->NoteXPCOMChild(child);
    }
}

#endif
