// SPDX-License-Identifier: Apache-2.0
// SPDX-FileCopyrightText: 2019-2024 Second State INC

#include "common/configure.h"
#include "vm/vm.h"

#include <gtest/gtest.h>

#include <vector>

namespace {

using namespace WasmEdge;

TEST(Component, LoadAndRun_SimpleBinary) {
  Configure Conf;
  Conf.addProposal(WasmEdge::Proposal::Component);
  VM::VM VM(Conf);

  std::vector<uint8_t> Vec = {
      0x00, 0x61, 0x73, 0x6d, 0x0d, 0x00, 0x01, 0x00, 0x01, 0x33, 0x00, 0x61,
      0x73, 0x6d, 0x01, 0x00, 0x00, 0x00, 0x01, 0x06, 0x01, 0x60, 0x01, 0x7e,
      0x01, 0x7e, 0x03, 0x02, 0x01, 0x00, 0x07, 0x07, 0x01, 0x03, 0x64, 0x75,
      0x70, 0x00, 0x00, 0x0a, 0x09, 0x01, 0x07, 0x00, 0x20, 0x00, 0x20, 0x00,
      0x7c, 0x0b, 0x00, 0x09, 0x04, 0x6e, 0x61, 0x6d, 0x65, 0x00, 0x02, 0x01,
      0x4d, 0x02, 0x04, 0x01, 0x00, 0x00, 0x00, 0x07, 0x08, 0x01, 0x40, 0x01,
      0x01, 0x61, 0x78, 0x00, 0x78, 0x06, 0x09, 0x01, 0x00, 0x00, 0x01, 0x00,
      0x03, 0x64, 0x75, 0x70, 0x08, 0x06, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x0b, 0x0a, 0x01, 0x00, 0x04, 0x6d, 0x64, 0x75, 0x70, 0x01, 0x00, 0x00,
      0x00, 0x28, 0x0e, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x6e, 0x65, 0x6e, 0x74,
      0x2d, 0x6e, 0x61, 0x6d, 0x65, 0x01, 0x06, 0x00, 0x11, 0x01, 0x00, 0x01,
      0x4d, 0x01, 0x06, 0x00, 0x12, 0x01, 0x00, 0x01, 0x6d, 0x01, 0x07, 0x01,
      0x01, 0x00, 0x03, 0x72, 0x75, 0x6e,
  };
  ASSERT_TRUE(VM.loadWasm(Vec));
  ASSERT_TRUE(VM.validate());
  ASSERT_TRUE(VM.instantiate());

  uint64_t V = 100;
  auto Res = VM.executeComponent("mdup", {ComponentValVariant(ValVariant(V))},
                                 {ComponentValType(ComponentTypeCode::U64)});
  ASSERT_TRUE(Res);
  std::vector<std::pair<ComponentValVariant, ComponentValType>> Result = *Res;
  auto Ret = std::get<ValVariant>(Result[0].first).get<uint64_t>();
  EXPECT_EQ(Ret, 200);
}

TEST(Component, Load_HttpBinary) {
  Configure Conf;
  Conf.addProposal(WasmEdge::Proposal::Component);
  VM::VM VM(Conf);

  std::vector<uint8_t> Vec = {
      0x00, 0x61, 0x73, 0x6d, 0x0d, 0x00, 0x01, 0x00, 0x01, 0x81, 0x02, 0x00,
      0x61, 0x73, 0x6d, 0x01, 0x00, 0x00, 0x00, 0x01, 0x09, 0x01, 0x60, 0x04,
      0x7f, 0x7f, 0x7f, 0x7f, 0x01, 0x7f, 0x03, 0x02, 0x01, 0x00, 0x05, 0x03,
      0x01, 0x00, 0x01, 0x06, 0x06, 0x01, 0x7f, 0x01, 0x41, 0x08, 0x0b, 0x07,
      0x11, 0x02, 0x03, 0x6d, 0x65, 0x6d, 0x02, 0x00, 0x07, 0x72, 0x65, 0x61,
      0x6c, 0x6c, 0x6f, 0x63, 0x00, 0x00, 0x0a, 0x6c, 0x01, 0x6a, 0x01, 0x01,
      0x7f, 0x20, 0x00, 0x04, 0x40, 0x20, 0x01, 0x20, 0x03, 0x4b, 0x04, 0x40,
      0x20, 0x00, 0x0f, 0x0b, 0x0b, 0x23, 0x00, 0x20, 0x02, 0x41, 0x7f, 0x6a,
      0x6a, 0x20, 0x02, 0x41, 0x7f, 0x6a, 0x41, 0x7f, 0x73, 0x71, 0x24, 0x00,
      0x23, 0x00, 0x21, 0x04, 0x23, 0x00, 0x20, 0x03, 0x6a, 0x24, 0x00, 0x03,
      0x40, 0x3f, 0x00, 0x41, 0x80, 0x80, 0x04, 0x6c, 0x23, 0x00, 0x49, 0x04,
      0x40, 0x41, 0x01, 0x40, 0x00, 0x41, 0x7f, 0x46, 0x04, 0x40, 0x00, 0x0b,
      0x0c, 0x01, 0x0b, 0x0b, 0x20, 0x04, 0x41, 0xde, 0x01, 0x20, 0x03, 0xfc,
      0x0b, 0x00, 0x20, 0x00, 0x04, 0x40, 0x20, 0x04, 0x20, 0x00, 0x20, 0x01,
      0xfc, 0x0a, 0x00, 0x00, 0x0b, 0x20, 0x04, 0x0b, 0x00, 0x5a, 0x04, 0x6e,
      0x61, 0x6d, 0x65, 0x00, 0x05, 0x04, 0x4c, 0x69, 0x62, 0x63, 0x01, 0x0a,
      0x01, 0x00, 0x07, 0x72, 0x65, 0x61, 0x6c, 0x6c, 0x6f, 0x63, 0x02, 0x2c,
      0x01, 0x00, 0x05, 0x00, 0x07, 0x6f, 0x6c, 0x64, 0x5f, 0x70, 0x74, 0x72,
      0x01, 0x08, 0x6f, 0x6c, 0x64, 0x5f, 0x73, 0x69, 0x7a, 0x65, 0x02, 0x05,
      0x61, 0x6c, 0x69, 0x67, 0x6e, 0x03, 0x08, 0x6e, 0x65, 0x77, 0x5f, 0x73,
      0x69, 0x7a, 0x65, 0x04, 0x03, 0x72, 0x65, 0x74, 0x03, 0x09, 0x01, 0x00,
      0x01, 0x02, 0x04, 0x6c, 0x6f, 0x6f, 0x70, 0x07, 0x07, 0x01, 0x00, 0x04,
      0x6c, 0x61, 0x73, 0x74, 0x02, 0x04, 0x01, 0x00, 0x00, 0x00, 0x07, 0x2f,
      0x01, 0x42, 0x04, 0x01, 0x40, 0x01, 0x03, 0x75, 0x72, 0x69, 0x73, 0x00,
      0x73, 0x04, 0x00, 0x08, 0x68, 0x74, 0x74, 0x70, 0x2d, 0x67, 0x65, 0x74,
      0x01, 0x00, 0x01, 0x40, 0x01, 0x04, 0x74, 0x65, 0x78, 0x74, 0x73, 0x01,
      0x00, 0x04, 0x00, 0x05, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x01, 0x01, 0x0a,
      0x13, 0x01, 0x00, 0x0e, 0x77, 0x61, 0x73, 0x69, 0x3a, 0x68, 0x74, 0x74,
      0x70, 0x2f, 0x74, 0x65, 0x73, 0x74, 0x05, 0x00, 0x06, 0x21, 0x03, 0x01,
      0x00, 0x00, 0x08, 0x68, 0x74, 0x74, 0x70, 0x2d, 0x67, 0x65, 0x74, 0x00,
      0x02, 0x01, 0x00, 0x03, 0x6d, 0x65, 0x6d, 0x00, 0x00, 0x01, 0x00, 0x07,
      0x72, 0x65, 0x61, 0x6c, 0x6c, 0x6f, 0x63, 0x08, 0x09, 0x01, 0x01, 0x00,
      0x00, 0x02, 0x03, 0x00, 0x04, 0x00, 0x06, 0x1e, 0x03, 0x01, 0x00, 0x00,
      0x05, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x00, 0x02, 0x01, 0x00, 0x03, 0x6d,
      0x65, 0x6d, 0x00, 0x00, 0x01, 0x00, 0x07, 0x72, 0x65, 0x61, 0x6c, 0x6c,
      0x6f, 0x63, 0x08, 0x09, 0x01, 0x01, 0x00, 0x01, 0x02, 0x03, 0x01, 0x04,
      0x02, 0x01, 0x8f, 0x01, 0x00, 0x61, 0x73, 0x6d, 0x01, 0x00, 0x00, 0x00,
      0x01, 0x15, 0x03, 0x60, 0x04, 0x7f, 0x7f, 0x7f, 0x7f, 0x01, 0x7f, 0x60,
      0x02, 0x7f, 0x7f, 0x02, 0x7f, 0x7f, 0x60, 0x02, 0x7f, 0x7f, 0x00, 0x02,
      0x39, 0x04, 0x04, 0x6c, 0x69, 0x62, 0x63, 0x03, 0x6d, 0x65, 0x6d, 0x02,
      0x00, 0x01, 0x04, 0x6c, 0x69, 0x62, 0x63, 0x07, 0x72, 0x65, 0x61, 0x6c,
      0x6c, 0x6f, 0x63, 0x00, 0x00, 0x04, 0x68, 0x74, 0x74, 0x70, 0x08, 0x68,
      0x74, 0x74, 0x70, 0x2d, 0x67, 0x65, 0x74, 0x00, 0x01, 0x04, 0x68, 0x74,
      0x74, 0x70, 0x05, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x00, 0x02, 0x03, 0x02,
      0x01, 0x02, 0x07, 0x07, 0x01, 0x03, 0x72, 0x75, 0x6e, 0x00, 0x03, 0x0a,
      0x0c, 0x01, 0x0a, 0x00, 0x20, 0x00, 0x20, 0x01, 0x10, 0x01, 0x10, 0x02,
      0x0b, 0x00, 0x18, 0x04, 0x6e, 0x61, 0x6d, 0x65, 0x00, 0x02, 0x01, 0x4d,
      0x01, 0x0d, 0x02, 0x01, 0x03, 0x67, 0x65, 0x74, 0x02, 0x05, 0x70, 0x72,
      0x69, 0x6e, 0x74, 0x02, 0x27, 0x02, 0x01, 0x02, 0x08, 0x68, 0x74, 0x74,
      0x70, 0x2d, 0x67, 0x65, 0x74, 0x00, 0x01, 0x05, 0x70, 0x72, 0x69, 0x6e,
      0x74, 0x00, 0x03, 0x00, 0x01, 0x02, 0x04, 0x6c, 0x69, 0x62, 0x63, 0x12,
      0x00, 0x04, 0x68, 0x74, 0x74, 0x70, 0x12, 0x01, 0x07, 0x0a, 0x01, 0x40,
      0x01, 0x03, 0x75, 0x72, 0x69, 0x73, 0x01, 0x00, 0x06, 0x1d, 0x03, 0x00,
      0x00, 0x01, 0x02, 0x03, 0x72, 0x75, 0x6e, 0x00, 0x02, 0x01, 0x00, 0x03,
      0x6d, 0x65, 0x6d, 0x00, 0x00, 0x01, 0x00, 0x07, 0x72, 0x65, 0x61, 0x6c,
      0x6c, 0x6f, 0x63, 0x08, 0x0a, 0x01, 0x00, 0x00, 0x04, 0x02, 0x03, 0x02,
      0x04, 0x05, 0x01, 0x0b, 0x09, 0x01, 0x00, 0x03, 0x72, 0x75, 0x6e, 0x01,
      0x02, 0x00, 0x00, 0x57, 0x0e, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x6e, 0x65,
      0x6e, 0x74, 0x2d, 0x6e, 0x61, 0x6d, 0x65, 0x01, 0x14, 0x00, 0x00, 0x02,
      0x01, 0x08, 0x68, 0x74, 0x74, 0x70, 0x2d, 0x67, 0x65, 0x74, 0x03, 0x05,
      0x70, 0x72, 0x69, 0x6e, 0x74, 0x01, 0x0c, 0x00, 0x11, 0x02, 0x00, 0x04,
      0x4c, 0x69, 0x62, 0x63, 0x01, 0x01, 0x4d, 0x01, 0x0f, 0x00, 0x12, 0x02,
      0x00, 0x04, 0x6c, 0x69, 0x62, 0x63, 0x02, 0x04, 0x6d, 0x61, 0x69, 0x6e,
      0x01, 0x07, 0x01, 0x01, 0x02, 0x03, 0x72, 0x75, 0x6e, 0x01, 0x08, 0x05,
      0x01, 0x00, 0x04, 0x68, 0x74, 0x74, 0x70,
  };
  ASSERT_TRUE(VM.loadWasm(Vec));

  // TODO: Fix this for the validator.
  // ASSERT_TRUE(VM.validate());
}

TEST(Component, LoadAndRun_MultiComponentBinary) {
  Configure Conf;
  Conf.addProposal(WasmEdge::Proposal::Component);
  VM::VM VM(Conf);

  // clang-format off
  std::vector<uint8_t> Vec = {
      // Top component header
      0x00, 0x61, 0x73, 0x6d, 0x0d, 0x00, 0x01, 0x00,

      // Top component component section, size = 1157
      0x04, 0x85, 0x09,
      // { component $C
      // component $C header
      0x00, 0x61, 0x73, 0x6d, 0x0d, 0x00, 0x01, 0x00,

      // $C core module section, size = 104
      0x01, 0x68,
      // { core module $Indirect
      // module $Indirect header
      0x00, 0x61, 0x73, 0x6d, 0x01, 0x00, 0x00, 0x00,
      // $Indirect type section vec<1>: func{i32}->{}
      0x01, 0x05, 0x01, 0x60, 0x01, 0x7f, 0x00,
      // $Indirect func section vec<2>: 0, 0
      0x03, 0x03, 0x02, 0x00, 0x00,
      // $Indirect table section vec<1>: {funcref, lim{0,2}}
      0x04, 0x04, 0x01, 0x70, 0x00, 0x02,
      // $Indirect export section vec<3>, size = 28
      0x07, 0x1c, 0x03,
      // export[0]: "ftbl" table index 0
      0x04, 0x66, 0x74, 0x62, 0x6c, 0x01, 0x00,
      // export[1]: "R1-dtor" func index 0
      0x07, 0x52, 0x31, 0x2d, 0x64, 0x74, 0x6f, 0x72, 0x00, 0x00,
      // export[2]: "R2-dtor" func index 1
      0x07, 0x52, 0x32, 0x2d, 0x64, 0x74, 0x6f, 0x72, 0x00, 0x01,
      // $Indirect code section vec<2>, size = 21
      0x0a, 0x15, 0x02, 0x09, 0x00, 0x20, 0x00, 0x41, 0x00, 0x11, 0x00, 0x00,
      0x0b, 0x09, 0x00, 0x20, 0x00, 0x41, 0x01, 0x11, 0x00, 0x00, 0x0b,
      // $Indirect custom section, size = 23
      0x00, 0x17, 0x04, 0x6e, 0x61, 0x6d, 0x65, 0x00, 0x09, 0x08, 0x49, 0x6e,
      0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x04, 0x05, 0x01, 0x00, 0x02, 0x46,
      0x54,
      // } core module $Indirect

      // $C core instance section, size = 4, instantiate module 0 with arg {}
      0x02, 0x04, 0x01, 0x00, 0x00, 0x00,
      // $C alias section vec<1>, size = 13
      0x06, 0x0d, 0x01,
      // alias[0]: core:sort:func, target: core:export index:0 name:"R1-dtor"
      0x00, 0x00, 0x01, 0x00, 0x07, 0x52, 0x31, 0x2d, 0x64, 0x74, 0x6f, 0x72,
      // $C type section vec<1>, size = 5
      0x07, 0x05, 0x01,
      // type[0] = (resource (rep i32) (dtor index:0))
      0x3f, 0x7f, 0x01, 0x00,
      // $C alias section vec<1>, size = 13
      0x06, 0x0d, 0x01,
      // alias[0]: core:sort:func, target: core:export index:0 name:"R2-dtor"
      0x00, 0x00, 0x01, 0x00, 0x07, 0x52, 0x32, 0x2d, 0x64, 0x74, 0x6f, 0x72,
      // $C type section vec<1>, size = 5
      0x07, 0x05, 0x01,
      // type[0] = (resource (rep i32) (dtor index:1))
      0x3f, 0x7f, 0x01, 0x01,
      // $C export section vec<2>, size = 15
      0x0b, 0x0f, 0x02,
      // export[0] = "R1", sort:type, index:0
      0x00, 0x02, 0x52, 0x31, 0x03, 0x00, 0x00,
      // export[0] = "R2", sort:type, index:1
      0x00, 0x02, 0x52, 0x32, 0x03, 0x01, 0x00,
      // $C canon section vec<2>, size = 5: resource.new:0, resource.new:1
      0x08, 0x05, 0x02, 0x02, 0x00, 0x02, 0x01,

      // $C core module section, size = 492
      0x01, 0xec, 0x03,
      // { core module $CM
      // module $CM header
      0x00, 0x61, 0x73, 0x6d, 0x01, 0x00, 0x00, 0x00,
      // $CM type section vec<3>
      0x01, 0x0e, 0x03,
      // type[0]: func{i32}->{i32}
      0x60, 0x01, 0x7f, 0x01, 0x7f,
      // type[1]: func{}->{i32}
      0x60, 0x00, 0x01, 0x7f,
      // type[2]: func{i32}->{}
      0x60, 0x01, 0x7f, 0x00,
      // $CM import section vec<3>, size = 49
      0x02, 0x31, 0x03,
      // import[0]: "" "ftbl" table type {funcref, lim{0,1}}
      0x00, 0x04, 0x66, 0x74, 0x62, 0x6c, 0x01, 0x70, 0x00, 0x01,
      // import[1]: "" "R1.resource.new" func type index 0
      0x00, 0x0f, 0x52, 0x31, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63,
      0x65, 0x2e, 0x6e, 0x65, 0x77, 0x00, 0x00,
      // import[1]: "" "R2.resource.new" func type index 0
      0x00, 0x0f, 0x52, 0x32, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63,
      0x65, 0x2e, 0x6e, 0x65, 0x77, 0x00, 0x00,
      // $CM func section vec<7>: 1, 1, 0, 0, 2, 2, 1
      0x03, 0x08, 0x07, 0x01, 0x01, 0x00, 0x00, 0x02, 0x02, 0x01,
      // $CM memory section vec<1>: {lim{0,1}}
      0x05, 0x03, 0x01, 0x00, 0x01,
      // $CM global section vec<2>, size = 11
      0x06, 0x0b, 0x02,
      // global[0]: mut i32.const 0
      0x7f, 0x01, 0x41, 0x00, 0x0b,
      // global[1]: mut i32.const 0
      0x7f, 0x01, 0x41, 0x00, 0x0b,
      // $CM export section vec<5>, size = 58
      0x07, 0x3a, 0x05,
      // export[0]: "make-R1" func index 2
      0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x31, 0x00, 0x02,
      // export[1]: "make-R2" func index 3
      0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x32, 0x00, 0x03,
      // export[2]: "get-rep-R1" func index 4
      0x0a, 0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52, 0x31, 0x00,
      0x04,
      // export[3]: "get-rep-R1" func index 5
      0x0a, 0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52, 0x32, 0x00,
      0x05,
      // export[4]: "num-live" func index 8
      0x08, 0x6e, 0x75, 0x6d, 0x2d, 0x6c, 0x69, 0x76, 0x65, 0x00, 0x08,
      // $CM element section vec<1>: 0ffset=0 funcindex{6, 7}
      0x09, 0x08, 0x01, 0x00, 0x41, 0x00, 0x0b, 0x02, 0x06, 0x07,
      // $CM code section vec<7>, size = 127
      0x0a, 0x7f, 0x07, 0x13, 0x01, 0x01, 0x7f, 0x23, 0x00, 0x41, 0x01, 0x6a,
      0x24, 0x00, 0x41, 0xc0, 0x00, 0x23, 0x00, 0x6a, 0x10, 0x00, 0x0b, 0x13,
      0x01, 0x01, 0x7f, 0x23, 0x01, 0x41, 0x01, 0x6a, 0x24, 0x01, 0x41, 0x80,
      0x01, 0x23, 0x01, 0x6a, 0x10, 0x01, 0x0b, 0x04, 0x00, 0x20, 0x00, 0x0b,
      0x04, 0x00, 0x20, 0x00, 0x0b, 0x21, 0x00, 0x20, 0x00, 0x41, 0xc1, 0x00,
      0x49, 0x20, 0x00, 0x41, 0xc2, 0x00, 0x4b, 0x72, 0x04, 0x40, 0x00, 0x0b,
      0x23, 0x00, 0x45, 0x04, 0x40, 0x00, 0x0b, 0x23, 0x00, 0x41, 0x01, 0x6b,
      0x24, 0x00, 0x0b, 0x21, 0x00, 0x20, 0x00, 0x41, 0x81, 0x01, 0x49, 0x20,
      0x00, 0x41, 0x82, 0x01, 0x4b, 0x72, 0x04, 0x40, 0x00, 0x0b, 0x23, 0x01,
      0x45, 0x04, 0x40, 0x00, 0x0b, 0x23, 0x01, 0x41, 0x01, 0x6b, 0x24, 0x01,
      0x0b, 0x07, 0x00, 0x23, 0x00, 0x23, 0x01, 0x6a, 0x0b,
      // $CM custom section, size = 187
      0x00, 0xbb, 0x01, 0x04, 0x6e, 0x61, 0x6d, 0x65, 0x00, 0x03, 0x02, 0x43,
      0x4d, 0x01, 0x69, 0x09, 0x00, 0x0f, 0x52, 0x31, 0x2e, 0x72, 0x65, 0x73,
      0x6f, 0x75, 0x72, 0x63, 0x65, 0x2e, 0x6e, 0x65, 0x77, 0x01, 0x0f, 0x52,
      0x32, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x2e, 0x6e,
      0x65, 0x77, 0x02, 0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x31, 0x03,
      0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x32, 0x04, 0x0a, 0x67, 0x65,
      0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52, 0x31, 0x05, 0x0a, 0x67, 0x65,
      0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52, 0x32, 0x06, 0x07, 0x52, 0x31,
      0x2d, 0x64, 0x74, 0x6f, 0x72, 0x07, 0x07, 0x52, 0x32, 0x2d, 0x64, 0x74,
      0x6f, 0x72, 0x08, 0x08, 0x6e, 0x75, 0x6d, 0x2d, 0x6c, 0x69, 0x76, 0x65,
      0x02, 0x27, 0x06, 0x02, 0x01, 0x00, 0x01, 0x68, 0x03, 0x01, 0x00, 0x01,
      0x68, 0x04, 0x01, 0x00, 0x03, 0x72, 0x65, 0x70, 0x05, 0x01, 0x00, 0x03,
      0x72, 0x65, 0x70, 0x06, 0x01, 0x00, 0x03, 0x72, 0x65, 0x70, 0x07, 0x01,
      0x00, 0x03, 0x72, 0x65, 0x70, 0x07, 0x1b, 0x02, 0x00, 0x0b, 0x6e, 0x75,
      0x6d, 0x2d, 0x6c, 0x69, 0x76, 0x65, 0x2d, 0x52, 0x31, 0x01, 0x0b, 0x6e,
      0x75, 0x6d, 0x2d, 0x6c, 0x69, 0x76, 0x65, 0x2d, 0x52, 0x32,
      // } core module $CM

      // $C alias section vec<1>, size = 10
      0x06, 0x0a, 0x01,
      // alias[0]: core:sort:table, target: core:export index:0 name:"ftbl"
      0x00, 0x01, 0x01, 0x00, 0x04, 0x66, 0x74, 0x62, 0x6c,
      // $C core instance section vec<2>, size = 52
      0x02, 0x34, 0x02,
      // instance[0]: inline export vec<3>
      0x01, 0x03,
      // instance[0] inline export[0]: "ftbl", table:0
      0x04, 0x66, 0x74, 0x62, 0x6c, 0x01, 0x00,
      // instance[0] inline export[1]: "R1.resource.new", func:2
      0x0f, 0x52, 0x31, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65,
      0x2e, 0x6e, 0x65, 0x77, 0x00, 0x02,
      // instance[0] inline export[1]: "R2.resource.new", func:3
      0x0f, 0x52, 0x32, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65,
      0x2e, 0x6e, 0x65, 0x77, 0x00, 0x03,
      // instance[1]: instantiate module:1 with name:"" instance:1
      0x00, 0x01, 0x01, 0x00, 0x12, 0x01,

      // $C type section vec<2>, size = 7
      0x07, 0x07, 0x02,
      // type[0] = (own typeindex:2)
      0x69, 0x02,
      // type[1] = (func {}->{typeindex:4})
      0x40, 0x00, 0x00, 0x04,
      // $C alias section vec<1>, size = 13
      0x06, 0x0d, 0x01,
      // alias[0]: core:sort:func, target: core:export index:2 name:"make-R1"
      0x00, 0x00, 0x01, 0x02, 0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x31,
      // $C canon section vec<1>: canon lift (core:func:4 typeindex:5)
      0x08, 0x06, 0x01, 0x00, 0x00, 0x04, 0x00, 0x05,

      // $C type section vec<2>, size = 7
      0x07, 0x07, 0x02,
      // type[0] = (own typeindex:3)
      0x69, 0x03,
      // type[1] = (func {}->{typeindex:6})
      0x40, 0x00, 0x00, 0x06,
      // $C alias section vec<1>, size = 13
      0x06, 0x0d, 0x01,
      // alias[0]: core:sort:func, target: core:export index:2 name:"make-R2"
      0x00, 0x00, 0x01, 0x02, 0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x32,
      // $C canon section vec<1>: canon lift (core:func:5 typeindex:7)
      0x08, 0x06, 0x01, 0x00, 0x00, 0x05, 0x00, 0x07,

      // $C type section vec<2>, size = 10
      0x07, 0x0a, 0x02,
      // type[0] = (borrow typeindex:2)
      0x68, 0x02,
      // type[1] = (func {"r" borrow typeindex:8}->{u32})
      0x40, 0x01, 0x01, 0x72, 0x08, 0x00, 0x79,
      // $C alias section vec<1>, size = 16
      0x06, 0x10, 0x01,
      // alias[0]: core:sort:func, target: core:export index:2
      0x00, 0x00, 0x01, 0x02,
      // alias[0]: name:"get-rep-R1"
      0x0a, 0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52, 0x31,
      // $C canon section vec<1>: canon lift (core:func:6 typeindex:9)
      0x08, 0x06, 0x01, 0x00, 0x00, 0x06, 0x00, 0x09,

      // $C type section vec<2>, size = 10
      0x07, 0x0a, 0x02,
      // type[0] = (borrow typeindex:3)
      0x68, 0x03,
      // type[1] = (func {"r" borrow typeindex:10}->{u32})
      0x40, 0x01, 0x01, 0x72, 0x0a, 0x00, 0x79,
      // $C alias section vec<1>, size = 16
      0x06, 0x10, 0x01,
      // alias[0]: core:sort:func, target: core:export index:2
      0x00, 0x00, 0x01, 0x02,
      // alias[0]: name:"get-rep-R2"
      0x0a, 0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52, 0x32,
      // $C canon section vec<1>: canon lift (core:func:7 typeindex:11)
      0x08, 0x06, 0x01, 0x00, 0x00, 0x07, 0x00, 0x0b,

      // $C type section vec<1>, size = 5: (func {}->{u32})
      0x07, 0x05, 0x01, 0x40, 0x00, 0x00, 0x79,
      // $C alias section vec<1>, size = 14
      0x06, 0x0e, 0x01,
      // alias[0]: core:sort:func, target: core:export index:2
      0x00, 0x00, 0x01, 0x02,
      // alias[0]: name:"num-live"
      0x08, 0x6e, 0x75, 0x6d, 0x2d, 0x6c, 0x69, 0x76, 0x65,
      // $C canon section vec<1>: canon lift (core:func:8 typeindex:12)
      0x08, 0x06, 0x01, 0x00, 0x00, 0x08, 0x00, 0x0c,

      // $C export section vec<5>, size = 68
      0x0b, 0x44, 0x05,
      // export[0]: name:"make-R1", sort:func, index:0
      0x00, 0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x31, 0x01, 0x00, 0x00,
      // export[1]: name:"make-R2", sort:func, index:1
      0x00, 0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x32, 0x01, 0x01, 0x00,
      // export[2]: name:"get-rep-R1", sort:func, index:2
      0x00, 0x0a, 0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52, 0x31,
      0x01, 0x02, 0x00,
      // export[3]: name:"get-rep-R2", sort:func, index:3
      0x00, 0x0a, 0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52, 0x32,
      0x01, 0x03, 0x00,
      // export[4]: name:"num-live", sort:func, index:4
      0x00, 0x08, 0x6e, 0x75, 0x6d, 0x2d, 0x6c, 0x69, 0x76, 0x65, 0x01, 0x04,
      0x00,

      // $C custom section, size = 164
      0x00, 0xa4, 0x01, 0x0e, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x6e, 0x65, 0x6e,
      0x74, 0x2d, 0x6e, 0x61, 0x6d, 0x65, 0x00, 0x02, 0x01, 0x43, 0x01, 0x25,
      0x00, 0x00, 0x02, 0x02, 0x0f, 0x52, 0x31, 0x2e, 0x72, 0x65, 0x73, 0x6f,
      0x75, 0x72, 0x63, 0x65, 0x2e, 0x6e, 0x65, 0x77, 0x03, 0x0f, 0x52, 0x32,
      0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x2e, 0x6e, 0x65,
      0x77, 0x01, 0x11, 0x00, 0x11, 0x02, 0x00, 0x08, 0x49, 0x6e, 0x64, 0x69,
      0x72, 0x65, 0x63, 0x74, 0x01, 0x02, 0x43, 0x4d, 0x01, 0x11, 0x00, 0x12,
      0x02, 0x00, 0x08, 0x69, 0x6e, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x02,
      0x02, 0x63, 0x6d, 0x01, 0x2c, 0x01, 0x04, 0x00, 0x07, 0x6d, 0x61, 0x6b,
      0x65, 0x2d, 0x52, 0x31, 0x01, 0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52,
      0x32, 0x02, 0x0a, 0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52,
      0x31, 0x03, 0x0a, 0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52,
      0x32, 0x01, 0x14, 0x03, 0x04, 0x00, 0x03, 0x52, 0x31, 0x27, 0x01, 0x03,
      0x52, 0x32, 0x27, 0x02, 0x02, 0x52, 0x31, 0x03, 0x02, 0x52, 0x32,
      // } component $C

      // Top component component section, size = 1010
      0x04, 0xf2, 0x07,
      // { component $D
      // component $D header
      0x00, 0x61, 0x73, 0x6d, 0x0d, 0x00, 0x01, 0x00,

      // $D type section vec<1>, size = 127
      0x07, 0x7f, 0x01,
      // type[0]: instance type vec<16>
      0x42, 0x10,
      // type[0] instancedecl[0]: exportdecl "R1" type:(sub resource)
      0x04, 0x00, 0x02, 0x52, 0x31, 0x03, 0x01,
      // type[0] instancedecl[1]: exportdecl "R2" type:(sub resource)
      0x04, 0x00, 0x02, 0x52, 0x32, 0x03, 0x01,
      // type[0] instancedecl[2]: type:(own typeindex:0)
      0x01, 0x69, 0x00,
      // type[0] instancedecl[3]: func:{}->{typeindex:2}
      0x01, 0x40, 0x00, 0x00, 0x02,
      // type[0] instancedecl[4]: exportdecl "make-R1" func:(type index:3)
      0x04, 0x00, 0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x31, 0x01, 0x03,
      // type[0] instancedecl[5]: type:(own typeindex:1)
      0x01, 0x69, 0x01,
      // type[0] instancedecl[6]: func:{}->{typeindex:4}
      0x01, 0x40, 0x00, 0x00, 0x04,
      // type[0] instancedecl[7]: exportdecl "make-R2" func:(type index:5)
      0x04, 0x00, 0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x32, 0x01, 0x05,
      // type[0] instancedecl[8]: type:(borrow typeindex:1)
      0x01, 0x68, 0x00,
      // type[0] instancedecl[9]: func:{"r" typeindex:6}->{u32}
      0x01, 0x40, 0x01, 0x01, 0x72, 0x06, 0x00, 0x79,
      // type[0] instancedecl[10]: exportdecl "get-rep-R1" func:(type index:7)
      0x04, 0x00, 0x0a, 0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52,
      0x31, 0x01, 0x07,
      // type[0] instancedecl[11]: type:(borrow typeindex:1)
      0x01, 0x68, 0x01,
      // type[0] instancedecl[12]: func:{"r" typeindex:8}->{u32}
      0x01, 0x40, 0x01, 0x01, 0x72, 0x08, 0x00, 0x79,
      // type[0] instancedecl[13]: exportdecl "get-rep-R2" func:(type index:9)
      0x04, 0x00, 0x0a, 0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52,
      0x32, 0x01, 0x09,
      // type[0] instancedecl[14]: func:{}->{u32}
      0x01, 0x40, 0x00, 0x00, 0x79,
      // type[0] instancedecl[15]: exportdecl "num-live" func:(type index:10)
      0x04, 0x00, 0x08, 0x6e, 0x75, 0x6d, 0x2d, 0x6c, 0x69, 0x76, 0x65, 0x01,
      0x0a,

      // $D import section vec<1>, size = 6: "c" instance:typeindex:0
      0x0a, 0x06, 0x01, 0x00, 0x01, 0x63, 0x05, 0x00,

      // $D core module section, size = 428
      0x01, 0xac, 0x03,
      // { core module $DM
      // module $DM header
      0x00, 0x61, 0x73, 0x6d, 0x01, 0x00, 0x00, 0x00,
      // $DM type section vec<3>
      0x01, 0x0e, 0x03,
      // type[0]: func{i32}->{}
      0x60, 0x01, 0x7f, 0x00,
      // type[1]: func{}->{i32}
      0x60, 0x00, 0x01, 0x7f,
      // type[2]: func{i32}->{}
      0x60, 0x01, 0x7f, 0x01, 0x7f,
      // $DM type section vec<7>, size = 103
      0x02, 0x67, 0x07,
      // import[0]: "" "R1.resource.drop" func type index 0
      0x00, 0x10, 0x52, 0x31, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63,
      0x65, 0x2e, 0x64, 0x72, 0x6f, 0x70, 0x00, 0x00,
      // import[1]: "" "R2.resource.drop" func type index 0
      0x00, 0x10, 0x52, 0x32, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63,
      0x65, 0x2e, 0x64, 0x72, 0x6f, 0x70, 0x00, 0x00,
      // import[2]: "" "make-R1" func type index 1
      0x00, 0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x31, 0x00, 0x01,
      // import[3]: "" "make-R2" func type index 1
      0x00, 0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x32, 0x00, 0x01,
      // import[4]: "" "get-rep-R1" func type index 2
      0x00, 0x0a, 0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52, 0x31,
      0x00, 0x02,
      // import[5]: "" "get-rep-R2" func type index 2
      0x00, 0x0a, 0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52, 0x32,
      0x00, 0x02,
      // import[5]: "" "num-live" func type index 1
      0x00, 0x08, 0x6e, 0x75, 0x6d, 0x2d, 0x6c, 0x69, 0x76, 0x65, 0x00, 0x01,
      // $DM func section vec<1>: 1
      0x03, 0x02, 0x01, 0x01,
      // $DM memory section vec<1>: {lim{0,1}}
      0x05, 0x03, 0x01, 0x00, 0x01,
      // $DM export section vec<1>, size = 7
      0x07, 0x07, 0x01,
      // export[0]: "run" func index 7
      0x03, 0x72, 0x75, 0x6e, 0x00, 0x07,
      // $DM code section vec<1>, size = 143
      0x0a, 0x8f, 0x01, 0x01, 0x8c, 0x01, 0x01, 0x05, 0x7f, 0x10, 0x02, 0x21,
      0x01, 0x41, 0x01, 0x20, 0x01, 0x47, 0x04, 0x40, 0x00, 0x0b, 0x10, 0x03,
      0x21, 0x02, 0x41, 0x02, 0x20, 0x02, 0x47, 0x04, 0x40, 0x00, 0x0b, 0x10,
      0x02, 0x21, 0x03, 0x41, 0x03, 0x20, 0x03, 0x47, 0x04, 0x40, 0x00, 0x0b,
      0x10, 0x03, 0x21, 0x04, 0x41, 0x04, 0x20, 0x04, 0x47, 0x04, 0x40, 0x00,
      0x0b, 0x41, 0x04, 0x10, 0x06, 0x47, 0x04, 0x40, 0x00, 0x0b, 0x41, 0x81,
      0x01, 0x20, 0x02, 0x10, 0x05, 0x47, 0x04, 0x40, 0x00, 0x0b, 0x20, 0x02,
      0x10, 0x01, 0x41, 0xc1, 0x00, 0x20, 0x01, 0x10, 0x04, 0x47, 0x04, 0x40,
      0x00, 0x0b, 0x20, 0x01, 0x10, 0x00, 0x41, 0x82, 0x01, 0x20, 0x04, 0x10,
      0x05, 0x47, 0x04, 0x40, 0x00, 0x0b, 0x20, 0x04, 0x10, 0x01, 0x41, 0xc2,
      0x00, 0x20, 0x03, 0x10, 0x04, 0x47, 0x04, 0x40, 0x00, 0x0b, 0x20, 0x03,
      0x10, 0x00, 0x41, 0x00, 0x10, 0x06, 0x47, 0x04, 0x40, 0x00, 0x0b, 0x41,
      0x2a, 0x0b,
      // $DM custom section, size = 132
      0x00, 0x84, 0x01, 0x04, 0x6e, 0x61, 0x6d, 0x65, 0x00, 0x03, 0x02, 0x44,
      0x4d, 0x01, 0x5e, 0x08, 0x00, 0x10, 0x52, 0x31, 0x2e, 0x72, 0x65, 0x73,
      0x6f, 0x75, 0x72, 0x63, 0x65, 0x2e, 0x64, 0x72, 0x6f, 0x70, 0x01, 0x10,
      0x52, 0x32, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x2e,
      0x64, 0x72, 0x6f, 0x70, 0x02, 0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52,
      0x31, 0x03, 0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x32, 0x04, 0x0a,
      0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52, 0x31, 0x05, 0x0a,
      0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52, 0x32, 0x06, 0x08,
      0x6e, 0x75, 0x6d, 0x2d, 0x6c, 0x69, 0x76, 0x65, 0x07, 0x03, 0x72, 0x75,
      0x6e, 0x02, 0x18, 0x01, 0x07, 0x05, 0x00, 0x03, 0x72, 0x65, 0x74, 0x01,
      0x02, 0x68, 0x31, 0x02, 0x02, 0x68, 0x32, 0x03, 0x02, 0x68, 0x33, 0x04,
      0x02, 0x68, 0x34,
      // } core module $DM

      // $D alias section vec<2>, size = 13
      0x06, 0x0d, 0x02,
      // alias[0]: sort:type, target: export index:0 name:"R1"
      0x03, 0x00, 0x00, 0x02, 0x52, 0x31,
      // alias[1]: sort:type, target: export index:0 name:"R2"
      0x03, 0x00, 0x00, 0x02, 0x52, 0x32,
      // $D canon section vec<2>, size = 5: resource.drop:0, resource.drop:1
      0x08, 0x05, 0x02, 0x03, 0x01, 0x03, 0x02,
      // $D alias section vec<1>, size = 12
      0x06, 0x0c, 0x01,
      // alias[0]: sort:func, target: export index:0 name:"make-R1"
      0x01, 0x00, 0x00, 0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x31,
      // $D canon section vec<1>, size = 5: canon lower (func:0)
      0x08, 0x05, 0x01, 0x01, 0x00, 0x00, 0x00,
      // $D alias section vec<1>, size = 12
      0x06, 0x0c, 0x01,
      // alias[0]: sort:func, target: export index:0 name:"make-R2"
      0x01, 0x00, 0x00, 0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x32,
      // $D canon section vec<1>, size = 5: canon lower (func:1)
      0x08, 0x05, 0x01, 0x01, 0x00, 0x01, 0x00,
      // $D alias section vec<1>, size = 15
      0x06, 0x0f, 0x01,
      // alias[0]: sort:func, target: export index:0 name:"get-rep-R1"
      0x01, 0x00, 0x00, 0x0a, 0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d,
      0x52, 0x31,
      // $D canon section vec<1>, size = 5: canon lower (func:2)
      0x08, 0x05, 0x01, 0x01, 0x00, 0x02, 0x00,
      // $D alias section vec<1>, size = 15
      0x06, 0x0f, 0x01,
      // alias[0]: sort:func, target: export index:0 name:"get-rep-R2"
      0x01, 0x00, 0x00, 0x0a, 0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d,
      0x52, 0x32,
      // $D canon section vec<1>, size = 5: canon lower (func:3)
      0x08, 0x05, 0x01, 0x01, 0x00, 0x03, 0x00,
      // $D alias section vec<1>, size = 13
      0x06, 0x0d, 0x01,
      // alias[0]: sort:func, target: export index:0 name:"num-live"
      0x01, 0x00, 0x00, 0x08, 0x6e, 0x75, 0x6d, 0x2d, 0x6c, 0x69, 0x76, 0x65,
      // $D canon section vec<1>, size = 5: canon lower (func:4)
      0x08, 0x05, 0x01, 0x01, 0x00, 0x04, 0x00,

      // $D core instance section vec<2>, size = 104
      0x02, 0x68, 0x02,
      // instance[0]: inline export vec<7>
      0x01, 0x07,
      // instance[0] inline export[0]: "R1.resource.drop" core:func:0
      0x10, 0x52, 0x31, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65,
      0x2e, 0x64, 0x72, 0x6f, 0x70, 0x00, 0x00,
      // instance[0] inline export[1]: "R2.resource.drop" core:func:1
      0x10, 0x52, 0x32, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65,
      0x2e, 0x64, 0x72, 0x6f, 0x70, 0x00, 0x01,
      // instance[0] inline export[2]: "make-R1" core:func:2
      0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x31, 0x00, 0x02,
      // instance[0] inline export[3]: "make-R2" core:func:3
      0x07, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x32, 0x00, 0x03,
      // instance[0] inline export[4]: "get-rep-R1" core:func:4
      0x0a, 0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52, 0x31, 0x00,
      0x04,
      // instance[0] inline export[5]: "get-rep-R2" core:func:5
      0x0a, 0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52, 0x32, 0x00,
      0x05,
      // instance[0] inline export[6]: "num-live" core:func:6
      0x08, 0x6e, 0x75, 0x6d, 0x2d, 0x6c, 0x69, 0x76, 0x65, 0x00, 0x06,
      // instance[1]: instantiate module:0 with name:"" instance:0
      0x00, 0x00, 0x01, 0x00, 0x12, 0x00,

      // $D type section vec<1>, size = 5: (func {}->{u32})
      0x07, 0x05, 0x01, 0x40, 0x00, 0x00, 0x79,
      // $D alias section vec<1>, size = 9
      0x06, 0x09, 0x01,
      // alias[0]: sort:core:func, target: core:export index:0 name:"run"
      0x00, 0x00, 0x01, 0x01, 0x03, 0x72, 0x75, 0x6e,
      // $D canon section vec<1>, size = 6: canon lift (core:func:7 typeindex:3)
      0x08, 0x06, 0x01, 0x00, 0x00, 0x07, 0x00, 0x03,
      // $D export section vec<1>, size = 9
      0x0b, 0x09, 0x01,
      // export[0]: name:"run", sort:func, index:5
      0x00, 0x03, 0x72, 0x75, 0x6e, 0x01, 0x05, 0x00,

      // $D custom section, size = 154
      0x00, 0x9a, 0x01, 0x0e, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x6e, 0x65, 0x6e,
      0x74, 0x2d, 0x6e, 0x61, 0x6d, 0x65, 0x00, 0x02, 0x01, 0x44, 0x01, 0x60,
      0x00, 0x00, 0x07, 0x00, 0x10, 0x52, 0x31, 0x2e, 0x72, 0x65, 0x73, 0x6f,
      0x75, 0x72, 0x63, 0x65, 0x2e, 0x64, 0x72, 0x6f, 0x70, 0x01, 0x10, 0x52,
      0x32, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x2e, 0x64,
      0x72, 0x6f, 0x70, 0x02, 0x08, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x31,
      0x27, 0x03, 0x08, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x52, 0x32, 0x27, 0x04,
      0x0b, 0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52, 0x31, 0x27,
      0x05, 0x0b, 0x67, 0x65, 0x74, 0x2d, 0x72, 0x65, 0x70, 0x2d, 0x52, 0x32,
      0x27, 0x06, 0x09, 0x6e, 0x75, 0x6d, 0x2d, 0x6c, 0x69, 0x76, 0x65, 0x27,
      0x01, 0x07, 0x00, 0x11, 0x01, 0x00, 0x02, 0x44, 0x4d, 0x01, 0x07, 0x00,
      0x12, 0x01, 0x01, 0x02, 0x64, 0x6d, 0x01, 0x0a, 0x03, 0x02, 0x01, 0x02,
      0x52, 0x31, 0x02, 0x02, 0x52, 0x32, 0x01, 0x05, 0x05, 0x01, 0x00, 0x01,
      0x63,
      // } component $D

      // Top component instance section, vec<2>, size = 11
      0x05, 0x0b, 0x02,
      // instance[0]: instantiate component:0 with arg {}
      0x00, 0x00, 0x00,
      // instance[1]: instantiate component:1 with arg {"c" instance:0}
      0x00, 0x01, 0x01, 0x01, 0x63, 0x05, 0x00,
      // Top component alias section vec<1>, size = 8
      0x06, 0x08, 0x01,
      // alias sort: sort:func, target: export index:1 name:"run"
      0x01, 0x00, 0x01, 0x03, 0x72, 0x75, 0x6e,
      // Top component export section vec<1>, size = 9
      0x0b, 0x09, 0x01,
      // export[0]: name:"run", sort:func, index:0
      0x00, 0x03, 0x72, 0x75, 0x6e, 0x01, 0x00, 0x00,

      // Top component custom section, size = 35
      0x00, 0x23, 0x0e, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x6e, 0x65, 0x6e, 0x74,
      0x2d, 0x6e, 0x61, 0x6d, 0x65, 0x01, 0x08, 0x04, 0x02, 0x00, 0x01, 0x43,
      0x01, 0x01, 0x44, 0x01, 0x08, 0x05, 0x02, 0x00, 0x01, 0x63, 0x01, 0x01,
      0x64,
  };
  // clang-format on

  ASSERT_TRUE(VM.loadWasm(Vec));
  // TODO: Fix this for the validator.
  /*
  ASSERT_TRUE(VM.validate());
  ASSERT_TRUE(VM.instantiate());
  */
}

} // namespace
