/*
 * unity-webapps-repository-binding.c
 * Copyright (C) Canonical LTD 2012
 * 
 * Author: Alexandre Abreu <alexandre.abreu@canonical.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include <glib.h>
#include  <gio/gdesktopappinfo.h>

#include <string.h>

#include <unity-webapps-application-repository.h>

#include "npapi-headers/headers/npapi.h"
#include "npapi-headers/headers/npfunctions.h"
#include "npapi-headers/headers/npruntime.h"

#include "wrapped-ptr-type.h"
#include "wrapped-callback-type.h"
#include "unity-webapps-binding-test.h"
#include "unity-npapi-binding-utils.h"
#include "unity-webapps-scriptable-object.h"
#include "unity-npapi-plugin.h"
#include "unity-webapps-base-js-object.h"

gchar*
unity_webapps_desktop_infos_build_desktop_basename(const gchar* name,
						   const gchar* domain);

void
unity_webapps_desktop_infos_generate_desktop_for_url_launch(const gchar* name,
							    const gchar* domain,
							    const gchar* icon_url,
							    const gchar* url);

static gboolean
unity_webapps_binding_does_desktop_file_exists(const gchar *name, const gchar* domain)
{
  GDesktopAppInfo *appinfo;
  gchar *desktop_basename;
  gchar *desktop_id;
  gboolean exists;

  exists = FALSE;
  g_return_val_if_fail(NULL != name, exists);
  g_return_val_if_fail(NULL != domain, exists);

  desktop_basename =
    unity_webapps_desktop_infos_build_desktop_basename(name, domain);
  desktop_id =
    g_strdup_printf("%s.desktop", desktop_basename);

  // Make sure that the desktop file exists
  appinfo = g_desktop_app_info_new(desktop_id);
  if (NULL == appinfo)
    {
      // local search
      gchar *local_desktop_filename =
	g_build_filename (g_get_user_data_dir(),
			  "applications",
			  desktop_id,
			  NULL);

      appinfo = g_desktop_app_info_new_from_filename(local_desktop_filename);

      g_free(local_desktop_filename);
    }
  g_free(desktop_id);
  g_free(desktop_basename);

  exists = (appinfo != NULL);
  return exists;
}

NPVariant
unity_webapps_binding_application_repository_generate_desktop_for_url_launch (NPP instance
									      , NPObject * object
									      , const NPVariant *args
									      , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 4)
    {
      NPN_SetException(object, "Invalid argument count in NPAPI function call");
      return result;
    }

  if ( ! NPVARIANT_IS_STRING(args[0])
      || ! NPVARIANT_IS_STRING(args[1])
      || ! NPVARIANT_IS_STRING(args[2])
      || ! NPVARIANT_IS_STRING(args[3]))
    {
      NPN_SetException(object, "Invalid argument type in NPAPI function"
                       " call: generate_desktop_for_url_launch");
      return result;
    }

  gchar * name = create_safe_string_for (&args[0]);
  gchar * domain = create_safe_string_for (&args[1]);
  gchar * url = create_safe_string_for (&args[2]);
  gchar * icon_url = create_safe_string_for (&args[3]);

  if ( ! unity_webapps_binding_does_desktop_file_exists(name, domain))
       unity_webapps_desktop_infos_generate_desktop_for_url_launch(name, domain, icon_url, url);

  g_free(name);
  g_free(domain);
  g_free(url);
  g_free(icon_url);

  return result;
}

NPVariant
unity_webapps_binding_application_repository_add_desktop_to_launcher (NPP instance
								      , NPObject * object
								      , const NPVariant *args
								      , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 2)
    {
      NPN_SetException(object, "Invalid argument count in NPAPI function call");
      return result;
    }

  if ( ! NPVARIANT_IS_STRING(args[0])
       || ! NPVARIANT_IS_STRING(args[1]))
    {
      NPN_SetException(object, "Invalid argument type in NPAPI function"
                       " call: application_repository_add_desktop_to_launcher");
      return result;
    }

  gchar * name = create_safe_string_for (&args[0]);
  gchar * domain = create_safe_string_for (&args[1]);
  if (unity_webapps_binding_does_desktop_file_exists(name, domain))
    {
      gchar *desktop_basename =
	unity_webapps_desktop_infos_build_desktop_basename(name, domain);
      gchar *desktop_id =
	g_strdup_printf("application://%s.desktop", desktop_basename);

      unity_webapps_application_repository_add_desktop_to_launcher(desktop_id);

      g_free(desktop_id);
      g_free(desktop_basename);
    }
  else
      g_message("Could not find desktop file for webapp: %s", name);

  g_free(name);
  g_free(domain);

  return result;
}

NPVariant
unity_webapps_binding_application_repository_new_default (NPP instance
                                                          , NPObject * npobject
                                                          , const NPVariant *args
                                                          , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 0)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function "
                       "call: application_repository_new_default");
      return result;
    }

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  UnityWebappsApplicationRepository * repository =
    unity_webapps_application_repository_new_default ();

  NPObject * object = create_wrapped_ptr_object_for (instance, repository);
  if (NULL == object)
    {
      NPN_SetException(object, "Unable to get a new repository object");
      return result;
    }

  OBJECT_TO_NPVARIANT(object, result);

  return result;
}

NPVariant
unity_webapps_binding_application_repository_prepare (NPP instance
                                                      , NPObject * npobject
                                                      , const NPVariant *args
                                                      , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 1)
    {
      NPN_SetException(npobject,
                       "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if (! NPVARIANT_IS_OBJECT(args[0]))
    {
      NPN_SetException(npobject,
                       "Invalid argument type in NPAPI function "
                       "call: application_repository_prepare");
      return result;
    }

  UnityWebappsApplicationRepository * repo =
    (UnityWebappsApplicationRepository *) ((wrapped_void_ptr_t *) NPVARIANT_TO_OBJECT(args[0]))->pWrapped;

  BOOLEAN_TO_NPVARIANT ((bool) unity_webapps_application_repository_prepare(repo), result);

  return result;
}

static NPVariant
_string_to_npvariant(const gchar * s)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  // TODO strlen -> glib utf8
  gint size = strlen(s) + 1;
  gchar * ret = NPN_MemAlloc (size);
  if (NULL == ret)
    return result;

  memset (ret, 0, size);
  g_strlcpy (ret, s, size);

  STRINGZ_TO_NPVARIANT(ret, result);
  return result;
}

NPVariant
unity_webapps_binding_application_repository_resolve_url_as_json (NPP instance
                                                          , NPObject * npobject
                                                          , const NPVariant *args
                                                          , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  // TODO fix that, automatically infer the mandatory argument count, etc.
  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if ( ! NPVARIANT_IS_OBJECT(args[0])
      || ! NPVARIANT_IS_STRING(args[1]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function"
                       " call: application_repository_resolve_url");
      return result;
    }

  UnityWebappsApplicationRepository * repo =
    (UnityWebappsApplicationRepository *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  gchar * url = create_safe_string_for (&args[1]);

  const gchar * names_as_json =
    unity_webapps_application_repository_resolve_url_as_json(repo, url);
  if (NULL != names_as_json)
    {
      result = _string_to_npvariant(names_as_json);
      g_free(names_as_json);
    }
  g_free (url);

  return result;
}

NPVariant
unity_webapps_binding_application_repository_get_resolved_application_status (NPP instance
                                                                              , NPObject * npobject
                                                                              , const NPVariant *args
                                                                              , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  // TODO fix that, automatically infer the mandatory argument count, etc.
  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if ( ! NPVARIANT_IS_OBJECT(args[0])
      || ! NPVARIANT_IS_STRING(args[1]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function"
                       " call: application_repository_get_resolved_application_status");
      return result;
    }

  UnityWebappsApplicationRepository * repo =
    (UnityWebappsApplicationRepository *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  gchar * application = create_safe_string_for (&args[1]);

  UnityWebappsApplicationStatus status =
    unity_webapps_application_repository_get_resolved_application_status(repo,
                                                                       application);
  g_free(application);

  INT32_TO_NPVARIANT(status, result);

  return result;
}

static void
UnityWebappsApplicationRepositoryInstallCallback_dispatcher (UnityWebappsApplicationRepository *repository
                                                             , const gchar *name
                                                             , UnityWebappsApplicationStatus status
                                                             , gpointer user_data)
{
  // not really safe ... 
  wrapped_callback_t * pCallbackObject = (wrapped_callback_t *) user_data;

  // fill out the arguments
  NPVariant args [4];

  // TODO call w/ meaningful values
  NULL_TO_NPVARIANT (args[0]); //repo
  STRINGZ_TO_NPVARIANT (name, args[1]); //name
  INT32_TO_NPVARIANT (status, args[2]); //status
  NULL_TO_NPVARIANT (args[3]); //user data

  NPVariant response;

  NPN_InvokeDefault (pCallbackObject->instance,
		     pCallbackObject->wrapped_callback,
		     args,
		     G_N_ELEMENTS(args),
		     &response);

  NPN_ReleaseVariantValue(&response);
}


NPVariant
unity_webapps_binding_application_repository_install_application (NPP instance
                                                                  , NPObject * npobject
                                                                  , const NPVariant *args
                                                                  , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  // TODO fix that, automatically infer the mandatory argument count, etc.
  if (argCount != 4)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if ( ! NPVARIANT_IS_OBJECT(args[0])
      || ! NPVARIANT_IS_STRING(args[1])
      || ! NPVARIANT_IS_OBJECT(args[2])
      )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function"
                       " call: application_repository_install_application");
      return result;
    }

  UnityWebappsApplicationRepository * repo =
    (UnityWebappsApplicationRepository *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  gchar * name = create_safe_string_for (&args[1]);

  NPObject * callback = NPVARIANT_TO_OBJECT(args[2]);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, repo, callback);

  NPObject * wrappedCallback =
    create_wrapped_callback_object_for (instance, callback);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, repo, wrappedCallback);

  unity_webapps_application_repository_install_application (repo,
                                                            name,
                                                            UnityWebappsApplicationRepositoryInstallCallback_dispatcher,
                                                            wrappedCallback);
  g_free(name);

  return result;
}

NPVariant
unity_webapps_binding_application_repository_get_userscript_contents (NPP instance
                                                                      , NPObject * npobject
                                                                      , const NPVariant *args
                                                                      , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  // TODO fix that, automatically infer the mandatory argument count, etc.
  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if ( ! NPVARIANT_IS_OBJECT(args[0])
      || ! NPVARIANT_IS_STRING(args[1]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function"
                       " call: application_repository_get_userscript_contents");
      return result;
    }

  UnityWebappsApplicationRepository * repo =
    (UnityWebappsApplicationRepository *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  gchar * name = create_safe_string_for (&args[1]);
  gchar * content =
    unity_webapps_application_repository_get_userscript_contents(repo, name);
  if (NULL != content)
    {
      result = _string_to_npvariant(content);
    }
  g_free(name);

  return result;
}

NPVariant
unity_webapps_binding_application_repository_get_resolved_application_domain (NPP instance
                                                                              , NPObject * npobject
                                                                              , const NPVariant *args
                                                                              , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  // TODO fix that, automatically infer the mandatory argument count, etc.
  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if ( ! NPVARIANT_IS_OBJECT(args[0])
      || ! NPVARIANT_IS_STRING(args[1]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function"
                       " call: application_repository_get_resolved_application_domain");
      return result;
    }

  UnityWebappsApplicationRepository * repo =
    (UnityWebappsApplicationRepository *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  gchar * application = create_safe_string_for (&args[1]);

  const gchar * domain =
    unity_webapps_application_repository_get_resolved_application_domain(repo, application);
  if (NULL != domain)
    {
      result = _string_to_npvariant(domain);
    }
  g_free (application);

  return result;
}

NPVariant
unity_webapps_binding_application_repository_get_resolved_application_name (NPP instance
                                                                            , NPObject * npobject
                                                                            , const NPVariant *args
                                                                            , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  // TODO fix that, automatically infer the mandatory argument count, etc.
  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if ( ! NPVARIANT_IS_OBJECT(args[0])
      || ! NPVARIANT_IS_STRING(args[1]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function"
                       " call: application_repository_get_resolved_application_name");
      return result;
    }

  UnityWebappsApplicationRepository * repo =
    (UnityWebappsApplicationRepository *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  gchar * application = create_safe_string_for (&args[1]);

  const gchar * name =
    unity_webapps_application_repository_get_resolved_application_name(repo, application);
  if (NULL != name)
    {
      result = _string_to_npvariant(name);
    }
  g_free (application);

  return result;
}

