#include "ArithmeticDecoder.h"




// Arithmetic decoder based on US patent 4791403.

static void InitDec(WinZipJPEGArithmeticDecoder *self);

static unsigned int LogDecoder(WinZipJPEGArithmeticDecoder *self,WinZipJPEGContext *context);

static void UpdateMPS(WinZipJPEGArithmeticDecoder *self,WinZipJPEGContext *context);
static void QSmaller(WinZipJPEGArithmeticDecoder *self,WinZipJPEGContext *context);

static void UpdateLPS(WinZipJPEGArithmeticDecoder *self,WinZipJPEGContext *context);
static void QBigger(WinZipJPEGArithmeticDecoder *self,WinZipJPEGContext *context);
static void IncrIndex(int *i,int *incrsv);
static void DblIndex(int *i,int *incrsv);

static void LRMBig(WinZipJPEGArithmeticDecoder *self);
static void Renorm(WinZipJPEGArithmeticDecoder *self);
static uint8_t ByteIn(WinZipJPEGArithmeticDecoder *self);

static uint32_t AntilogX(int16_t lr);
static int16_t LogX(uint32_t x);




static uint16_t logp[]; // log p at index i
static uint16_t lqp[]; // log q - log p
static uint16_t nmaxlp[]; // nmax * lp
static uint8_t halfi[]; // pointer to q halved
static uint8_t dbli[]; // pointer to q doubled

static uint16_t alogtbl[];
static uint16_t logtbl[];
static uint8_t chartbl[];




void InitializeWinZipJPEGArithmeticDecoder(WinZipJPEGArithmeticDecoder *self,WinZipJPEGReadFunction *readfunc, void *inputcontext)
{
	self->readfunc=readfunc;
	self->inputcontext=inputcontext;
	self->eof=false;

	InitDec(self);
}

void InitializeWinZipJPEGContext(WinZipJPEGContext *self)
{
	self->dlrm=nmaxlp[0];
	self->i=0;
	self->k=0;
	self->mps=0;
}

void InitializeWinZipJPEGContexts(WinZipJPEGContext *first,size_t bytes)
{
	for(int i=0;i<bytes/sizeof(WinZipJPEGContext);i++) InitializeWinZipJPEGContext(first+i);
}

void InitializeFixedWinZipJPEGContext(WinZipJPEGContext *self)
{
	self->dlrm=nmaxlp[0];
	self->i=48;
	self->k=0;
	self->mps=0;
}

int NextBitFromWinZipJPEGArithmeticDecoder(WinZipJPEGArithmeticDecoder *self,WinZipJPEGContext *context)
{
self->dx=0; // Otherwise tests don't pass.
	int bit=LogDecoder(self,context);
self->lp=logp[context->i];
	return bit;
}

void FlushWinZipJPEGArithmeticDecoder(WinZipJPEGArithmeticDecoder *self)
{
	Renorm(self);

	if(self->currbyte==0xff && self->lastbyte==0xff) ByteIn(self);
}


static void InitDec(WinZipJPEGArithmeticDecoder *self)
{
	self->kmin2=0;
	self->kmin1=1;
	self->kmin=5;
	self->kmax=11;

	uint8_t b1=ByteIn(self);
	uint8_t b2=ByteIn(self);
	self->x=(b1<<8)|b2;

	self->lr=0x1001;
	self->lrm=self->lr;
	self->lx=LogX(self->x);

	if(self->x==0xffff) ByteIn(self);
}




static unsigned int LogDecoder(WinZipJPEGArithmeticDecoder *self,WinZipJPEGContext *context)
{
	self->lrm=self->lr+context->dlrm;
	LRMBig(self);

	self->lr+=logp[context->i];

	unsigned int bit=context->mps;

	int lrt;
	if(self->lx<self->lrm) lrt=self->lx;
	else lrt=self->lrm;

	if(self->lr>=lrt)
	{
		if(self->lr<self->lx)
		{
			UpdateMPS(self,context);
		}
		else
		{
			Renorm(self);
			if(self->lr<self->lx)
			{
				if(self->lr>=self->lrm) UpdateMPS(self,context);
			}
			else
			{
				bit^=1;

				context->k++;

				uint32_t dx=AntilogX(self->lr);
				self->x-=dx;
				self->lx=LogX(self->x);
self->dx=dx; // for tests

				UpdateLPS(self,context);
			}
		}
	}

	context->dlrm=self->lrm-self->lr;

	return bit;
}




static void UpdateMPS(WinZipJPEGArithmeticDecoder *self,WinZipJPEGContext *context)
{
	if(context->k<=self->kmin) QSmaller(self,context);
	context->k=0;
	self->lrm=self->lr+nmaxlp[context->i];
	LRMBig(self);
}

static void QSmaller(WinZipJPEGArithmeticDecoder *self,WinZipJPEGContext *context)
{
	if(context->i>=47) return; // WinZip modification.

	context->i++;

	if(context->k<=self->kmin1)
	{
		context->i+=halfi[context->i];
		if(context->k<=self->kmin2)
		{
			context->i+=halfi[context->i];
		}
	}
}




static void UpdateLPS(WinZipJPEGArithmeticDecoder *self,WinZipJPEGContext *context)
{
	self->lr+=lqp[context->i];
	self->lrm+=lqp[context->i];

	if(context->k>=self->kmax)
	{
		QBigger(self,context);
		context->k=0;
		self->lrm=self->lr+nmaxlp[context->i];
	}
	else
	{
		if(self->lrm<self->lr) self->lrm=self->lr;
	}
}

static void QBigger(WinZipJPEGArithmeticDecoder *self,WinZipJPEGContext *context)
{
	if(context->i>=48) return; // WinZip modification.

	int32_t dlrm=self->lrm-self->lr;
	int incrsv=0;

	if(dlrm>=nmaxlp[context->i]/2)
	{
		dlrm=nmaxlp[context->i]-dlrm;
		if(dlrm<=nmaxlp[context->i]/4) DblIndex(&context->i,&incrsv);
		DblIndex(&context->i,&incrsv);
	}
	else
	{
		if(dlrm>=nmaxlp[context->i]/4) IncrIndex(&context->i,&incrsv);
		IncrIndex(&context->i,&incrsv);
	}

	if(context->i<=0)
	{
		context->i=incrsv;
		context->mps=context->mps^1;
	}

	self->lrm=self->lr+dlrm;
}

static void IncrIndex(int *i,int *incrsv)
{
	if(*i>0) (*i)--;
	else (*incrsv)++;
}

static void DblIndex(int *i,int *incrsv)
{
	if(*i>0) *i-=dbli[*i];
	else *incrsv+=dbli[*i];
}




static void LRMBig(WinZipJPEGArithmeticDecoder *self)
{
	if(self->lrm>0x7ff) Renorm(self);
}

static void Renorm(WinZipJPEGArithmeticDecoder *self)
{
	while(self->lr>0x1fff)
	{
		if(self->currbyte==0xff && self->lastbyte==0xff)
		{
			self->x+=ByteIn(self);
		}
		self->x=(self->x<<8)|ByteIn(self);
		self->lr-=0x2000;
		self->lrm-=0x2000;
	}

	self->lx=LogX(self->x);
}

static uint8_t ByteIn(WinZipJPEGArithmeticDecoder *self)
{
	self->lastbyte=self->currbyte;

	if(self->readfunc(self->inputcontext,&self->currbyte,1)!=1)
	{
		self->eof=true;
		self->currbyte=0;
	}

	return self->currbyte;
}


static int16_t LogX(uint32_t x)
{
	unsigned int highbits=x>>12;
	if(highbits==0) return 0x2000;

	int whole;
	if(highbits<512) whole=chartbl[highbits];
	else whole=0;

	int shift=8-whole;

	int negfraction;
	if(shift>=0) negfraction=logtbl[(x>>shift)&0xfff];
	else negfraction=logtbl[(x<<-shift)&0xfff]; // Is this necessary? No idea.

	return (whole<<10)-negfraction;
}

static uint32_t AntilogX(int16_t lr)
{
	int whole=lr>>10;
	unsigned int fraction=lr&0x3ff;

	int shift=7-whole;
	if(shift>=0) return alogtbl[fraction]<<shift;
	else return alogtbl[fraction]>>-shift; // Is this necessary? No idea.
}


static uint16_t logp[49]= // log p at index i
{
	1024,
	895,
	795,
	706,
	628,
	559,
	493,
	437,
	379,
	331,
	287,
	247,
	212,
	186,
	158,
	143,
	127,
	110,
	98,
	84,
	72,
	65,
	59,
	53,
	48,
	45,
	42,
	40,
	37,
	35,
	33,
	30,
	28,
	26,
	23,
	21,
	19,
	17,
	15,
	13,
	11,
	9,
	7,
	5,
	4,
	3,
	2,
	1,
	1024,
};

static uint16_t lqp[49]= // log q - log p
{
	0,
	272,
	502,
	726,
	941,
	1150,
	1371,
	1578,
	1819,
	2044,
	2278,
	2521,
	2765,
	2971,
	3227,
	3382,
	3566,
	3788,
	3965,
	4200,
	4435,
	4590,
	4737,
	4899,
	5050,
	5147,
	5250,
	5325,
	5441,
	5527,
	5617,
	5758,
	5863,
	5976,
	6157,
	6295,
	6447,
	6616,
	6806,
	7024,
	7278,
	7585,
	7972,
	8495,
	8884,
	9309,
	10065,
	11689,
	0,
};

static uint16_t nmaxlp[49]= // nmax * lp
{
	16384,
	16110,
	15105,
	14826,
	14444,
	13975,
	13804,
	13547,
	13265,
	13240,
	12915,
	12844,
	12720,
	12648,
	12482,
	12441,
	12319,
	12320,
	12250,
	12180,
	12168,
	12155,
	12154,
	12084,
	12096,
	12105,
	12096,
	12080,
	12062,
	12075,
	12078,
	12060,
	12068,
	12090,
	12075,
	12075,
	12103,
	12121,
	12150,
	12181,
	12221,
	12294,
	12411,
	12615,
	13120,
	13113,
	14574,
	21860,
	0,
};

static uint8_t halfi[49]= // pointer to q halved
{
	8,
	8,
	7,
	7,
	7,
	6,
	6,
	6,
	6,
	6,
	6,
	6,
	6,
	6,
	6,
	6,
	6,
	6,
	6,
	8,
	9,
	10,
	10,
	10,
	10,
	10,
	10,
	9,
	9,
	8,
	8,
	7,
	7,
	6,
	6,
	6,
	5,
	5,
	4,
	4,
	3,
	3,
	3,
	3,
	2,
	2,
	1,
	0,
	0,
};

static uint8_t dbli[49]= // pointer to q doubled
{
	0,
	1,
	2,
	3,
	4,
	5,
	6,
	7,
	8,
	7,
	7,
	6,
	6,
	6,
	6,
	6,
	6,
	6,
	6,
	6,
	6,
	6,
	6,
	6,
	6,
	7,
	8,
	8,
	9,
	9,
	10,
	10,
	10,
	10,
	10,
	10,
	9,
	8,
	7,
	6,
	6,
	5,
	4,
	3,
	3,
	3,
	2,
	1,
	0,
};

static uint16_t alogtbl[1024]=
{
	0x2000,0x1ffd,0x1ff7,0x1ff1,0x1fec,0x1fe6,0x1fe1,0x1fdb,
	0x1fd6,0x1fd0,0x1fcb,0x1fc5,0x1fc0,0x1fba,0x1fb5,0x1faf,
	0x1faa,0x1fa4,0x1f9f,0x1f99,0x1f94,0x1f8e,0x1f89,0x1f83,
	0x1f7e,0x1f79,0x1f73,0x1f6e,0x1f68,0x1f63,0x1f5d,0x1f58,
	0x1f53,0x1f4d,0x1f48,0x1f42,0x1f3d,0x1f37,0x1f32,0x1f2d,
	0x1f27,0x1f22,0x1f1c,0x1f17,0x1f12,0x1f0c,0x1f07,0x1f02,
	0x1efc,0x1ef7,0x1ef1,0x1eec,0x1ee7,0x1ee1,0x1edc,0x1ed7,
	0x1ed1,0x1ecc,0x1ec7,0x1ec1,0x1ebc,0x1eb7,0x1eb1,0x1eac,
	0x1ea7,0x1ea1,0x1e9c,0x1e97,0x1e92,0x1e8c,0x1e87,0x1e82,
	0x1e7c,0x1e77,0x1e72,0x1e6d,0x1e67,0x1e62,0x1e5d,0x1e58,
	0x1e52,0x1e4d,0x1e48,0x1e43,0x1e3d,0x1e38,0x1e33,0x1e2e,
	0x1e28,0x1e23,0x1e1e,0x1e19,0x1e14,0x1e0e,0x1e09,0x1e04,
	0x1dff,0x1dfa,0x1df4,0x1def,0x1dea,0x1de5,0x1de0,0x1dda,
	0x1dd5,0x1dd0,0x1dcb,0x1dc6,0x1dc1,0x1dbc,0x1db6,0x1db1,
	0x1dac,0x1da7,0x1da2,0x1d9d,0x1d98,0x1d93,0x1d8d,0x1d88,
	0x1d83,0x1d7e,0x1d79,0x1d74,0x1d6f,0x1d6a,0x1d65,0x1d5f,
	0x1d5a,0x1d55,0x1d50,0x1d4b,0x1d46,0x1d41,0x1d3c,0x1d37,
	0x1d32,0x1d2d,0x1d28,0x1d23,0x1d1e,0x1d19,0x1d14,0x1d0e,
	0x1d09,0x1d04,0x1cff,0x1cfa,0x1cf5,0x1cf0,0x1ceb,0x1ce6,
	0x1ce1,0x1cdc,0x1cd7,0x1cd2,0x1ccd,0x1cc8,0x1cc3,0x1cbf,
	0x1cba,0x1cb5,0x1cb0,0x1cab,0x1ca6,0x1ca1,0x1c9c,0x1c97,
	0x1c92,0x1c8d,0x1c88,0x1c83,0x1c7e,0x1c79,0x1c74,0x1c6f,
	0x1c6a,0x1c65,0x1c61,0x1c5c,0x1c57,0x1c52,0x1c4d,0x1c48,
	0x1c43,0x1c3e,0x1c39,0x1c34,0x1c2f,0x1c2b,0x1c26,0x1c21,
	0x1c1c,0x1c17,0x1c12,0x1c0d,0x1c09,0x1c04,0x1bff,0x1bfa,
	0x1bf5,0x1bf0,0x1bec,0x1be7,0x1be2,0x1bdd,0x1bd8,0x1bd3,
	0x1bcf,0x1bca,0x1bc5,0x1bc0,0x1bbb,0x1bb7,0x1bb2,0x1bad,
	0x1ba8,0x1ba3,0x1b9f,0x1b9a,0x1b95,0x1b90,0x1b8c,0x1b87,
	0x1b82,0x1b7d,0x1b78,0x1b74,0x1b6f,0x1b6a,0x1b66,0x1b61,
	0x1b5c,0x1b57,0x1b53,0x1b4e,0x1b49,0x1b44,0x1b40,0x1b3b,
	0x1b36,0x1b32,0x1b2d,0x1b28,0x1b23,0x1b1f,0x1b1a,0x1b15,
	0x1b11,0x1b0c,0x1b07,0x1b03,0x1afe,0x1af9,0x1af5,0x1af0,
	0x1aeb,0x1ae7,0x1ae2,0x1add,0x1ad9,0x1ad4,0x1acf,0x1acb,
	0x1ac6,0x1ac1,0x1abd,0x1ab8,0x1ab4,0x1aaf,0x1aaa,0x1aa6,
	0x1aa1,0x1a9c,0x1a98,0x1a93,0x1a8f,0x1a8a,0x1a85,0x1a81,
	0x1a7c,0x1a78,0x1a73,0x1a6f,0x1a6a,0x1a65,0x1a61,0x1a5c,
	0x1a58,0x1a53,0x1a4f,0x1a4a,0x1a46,0x1a41,0x1a3c,0x1a38,
	0x1a33,0x1a2f,0x1a2a,0x1a26,0x1a21,0x1a1d,0x1a18,0x1a14,
	0x1a0f,0x1a0b,0x1a06,0x1a02,0x19fd,0x19f9,0x19f4,0x19f0,
	0x19eb,0x19e7,0x19e2,0x19de,0x19d9,0x19d5,0x19d0,0x19cc,
	0x19c7,0x19c3,0x19be,0x19ba,0x19b6,0x19b1,0x19ad,0x19a8,
	0x19a4,0x199f,0x199b,0x1996,0x1992,0x198e,0x1989,0x1985,
	0x1980,0x197c,0x1978,0x1973,0x196f,0x196a,0x1966,0x1962,
	0x195d,0x1959,0x1954,0x1950,0x194c,0x1947,0x1943,0x193e,
	0x193a,0x1936,0x1931,0x192d,0x1929,0x1924,0x1920,0x191c,
	0x1917,0x1913,0x190f,0x190a,0x1906,0x1902,0x18fd,0x18f9,
	0x18f5,0x18f0,0x18ec,0x18e8,0x18e3,0x18df,0x18db,0x18d6,
	0x18d2,0x18ce,0x18ca,0x18c5,0x18c1,0x18bd,0x18b8,0x18b4,
	0x18b0,0x18ac,0x18a7,0x18a3,0x189f,0x189b,0x1896,0x1892,
	0x188e,0x188a,0x1885,0x1881,0x187d,0x1879,0x1874,0x1870,
	0x186c,0x1868,0x1863,0x185f,0x185b,0x1857,0x1853,0x184e,
	0x184a,0x1846,0x1842,0x183e,0x1839,0x1835,0x1831,0x182d,
	0x1829,0x1824,0x1820,0x181c,0x1818,0x1814,0x180f,0x180b,
	0x1807,0x1803,0x17ff,0x17fb,0x17f7,0x17f2,0x17ee,0x17ea,
	0x17e6,0x17e2,0x17de,0x17da,0x17d5,0x17d1,0x17cd,0x17c9,
	0x17c5,0x17c1,0x17bd,0x17b9,0x17b5,0x17b0,0x17ac,0x17a8,
	0x17a4,0x17a0,0x179c,0x1798,0x1794,0x1790,0x178c,0x1788,
	0x1784,0x177f,0x177b,0x1777,0x1773,0x176f,0x176b,0x1767,
	0x1763,0x175f,0x175b,0x1757,0x1753,0x174f,0x174b,0x1747,
	0x1743,0x173f,0x173b,0x1737,0x1733,0x172f,0x172b,0x1727,
	0x1723,0x171f,0x171b,0x1717,0x1713,0x170f,0x170b,0x1707,
	0x1703,0x16ff,0x16fb,0x16f7,0x16f3,0x16ef,0x16eb,0x16e7,
	0x16e3,0x16df,0x16db,0x16d7,0x16d3,0x16cf,0x16cb,0x16c7,
	0x16c3,0x16bf,0x16bb,0x16b7,0x16b3,0x16b0,0x16ac,0x16a8,
	0x16a4,0x16a0,0x169c,0x1698,0x1694,0x1690,0x168c,0x1688,
	0x1685,0x1681,0x167d,0x1679,0x1675,0x1671,0x166d,0x1669,
	0x1665,0x1662,0x165e,0x165a,0x1656,0x1652,0x164e,0x164a,
	0x1647,0x1643,0x163f,0x163b,0x1637,0x1633,0x162f,0x162c,
	0x1628,0x1624,0x1620,0x161c,0x1618,0x1615,0x1611,0x160d,
	0x1609,0x1605,0x1602,0x15fe,0x15fa,0x15f6,0x15f2,0x15ee,
	0x15eb,0x15e7,0x15e3,0x15df,0x15dc,0x15d8,0x15d4,0x15d0,
	0x15cc,0x15c9,0x15c5,0x15c1,0x15bd,0x15b9,0x15b6,0x15b2,
	0x15ae,0x15aa,0x15a7,0x15a3,0x159f,0x159c,0x1598,0x1594,
	0x1590,0x158d,0x1589,0x1585,0x1581,0x157e,0x157a,0x1576,
	0x1573,0x156f,0x156b,0x1567,0x1564,0x1560,0x155c,0x1559,
	0x1555,0x1551,0x154e,0x154a,0x1546,0x1542,0x153f,0x153b,
	0x1537,0x1534,0x1530,0x152c,0x1529,0x1525,0x1521,0x151e,
	0x151a,0x1516,0x1513,0x150f,0x150b,0x1508,0x1504,0x1501,
	0x14fd,0x14f9,0x14f6,0x14f2,0x14ee,0x14eb,0x14e7,0x14e4,
	0x14e0,0x14dc,0x14d9,0x14d5,0x14d2,0x14ce,0x14ca,0x14c7,
	0x14c3,0x14c0,0x14bc,0x14b8,0x14b5,0x14b1,0x14ae,0x14aa,
	0x14a6,0x14a3,0x149f,0x149c,0x1498,0x1495,0x1491,0x148d,
	0x148a,0x1486,0x1483,0x147f,0x147c,0x1478,0x1475,0x1471,
	0x146e,0x146a,0x1466,0x1463,0x145f,0x145c,0x1458,0x1455,
	0x1451,0x144e,0x144a,0x1447,0x1443,0x1440,0x143c,0x1439,
	0x1435,0x1432,0x142e,0x142b,0x1427,0x1424,0x1420,0x141d,
	0x1419,0x1416,0x1412,0x140f,0x140b,0x1408,0x1405,0x1401,
	0x13fd,0x13fa,0x13f7,0x13f3,0x13f0,0x13ec,0x13e9,0x13e5,
	0x13e2,0x13de,0x13db,0x13d8,0x13d4,0x13d1,0x13cd,0x13ca,
	0x13c7,0x13c3,0x13c0,0x13bc,0x13b9,0x13b5,0x13b2,0x13ae,
	0x13ab,0x13a8,0x13a4,0x13a1,0x139e,0x139a,0x1397,0x1393,
	0x1390,0x138d,0x1389,0x1386,0x1382,0x137f,0x137c,0x1378,
	0x1375,0x1372,0x136e,0x136b,0x1367,0x1364,0x1361,0x135d,
	0x135a,0x1357,0x1353,0x1350,0x134d,0x1349,0x1346,0x1343,
	0x133f,0x133c,0x1339,0x1335,0x1332,0x132f,0x132b,0x1328,
	0x1325,0x1321,0x131e,0x131b,0x1317,0x1314,0x1311,0x130e,
	0x130a,0x1307,0x1304,0x1300,0x12fd,0x12fa,0x12f7,0x12f3,
	0x12f0,0x12ed,0x12e9,0x12e6,0x12e3,0x12df,0x12dc,0x12d9,
	0x12d6,0x12d2,0x12cf,0x12cc,0x12c9,0x12c5,0x12c2,0x12bf,
	0x12bc,0x12b8,0x12b5,0x12b2,0x12af,0x12ac,0x12a8,0x12a5,
	0x12a2,0x129f,0x129b,0x1298,0x1295,0x1292,0x128e,0x128b,
	0x1288,0x1285,0x1282,0x127e,0x127b,0x1278,0x1275,0x1272,
	0x126e,0x126b,0x1268,0x1265,0x1262,0x125e,0x125b,0x1258,
	0x1255,0x1252,0x124f,0x124b,0x1248,0x1245,0x1242,0x123f,
	0x123c,0x1238,0x1235,0x1232,0x122f,0x122c,0x1229,0x1226,
	0x1222,0x121f,0x121c,0x1219,0x1216,0x1213,0x1210,0x120c,
	0x1209,0x1206,0x1203,0x1200,0x11fd,0x11fa,0x11f7,0x11f4,
	0x11f0,0x11ed,0x11ea,0x11e7,0x11e4,0x11e1,0x11de,0x11db,
	0x11d8,0x11d5,0x11d1,0x11ce,0x11cb,0x11c8,0x11c5,0x11c2,
	0x11bf,0x11bc,0x11b9,0x11b6,0x11b3,0x11b0,0x11ad,0x11a9,
	0x11a6,0x11a3,0x11a0,0x119d,0x119a,0x1197,0x1194,0x1191,
	0x118e,0x118b,0x1188,0x1185,0x1182,0x117f,0x117c,0x1179,
	0x1176,0x1173,0x1170,0x116d,0x116a,0x1167,0x1164,0x1161,
	0x115e,0x115b,0x1158,0x1155,0x1152,0x114f,0x114c,0x1149,
	0x1146,0x1143,0x1140,0x113d,0x113a,0x1137,0x1134,0x1131,
	0x112e,0x112b,0x1128,0x1125,0x1122,0x111f,0x111c,0x1119,
	0x1116,0x1113,0x1110,0x110d,0x110a,0x1107,0x1104,0x1101,
	0x10fe,0x10fb,0x10f8,0x10f5,0x10f2,0x10ef,0x10ec,0x10e9,
	0x10e6,0x10e3,0x10e0,0x10de,0x10db,0x10d8,0x10d5,0x10d2,
	0x10cf,0x10cc,0x10c9,0x10c6,0x10c3,0x10c0,0x10bd,0x10ba,
	0x10b8,0x10b5,0x10b2,0x10af,0x10ac,0x10a9,0x10a6,0x10a3,
	0x10a0,0x109e,0x109b,0x1098,0x1095,0x1092,0x108f,0x108c,
	0x1089,0x1086,0x1084,0x1081,0x107e,0x107b,0x1078,0x1075,
	0x1072,0x1070,0x106d,0x106a,0x1067,0x1064,0x1061,0x105e,
	0x105c,0x1059,0x1056,0x1053,0x1050,0x104d,0x104a,0x1048,
	0x1045,0x1042,0x103f,0x103c,0x1039,0x1037,0x1034,0x1031,
	0x102e,0x102b,0x1028,0x1026,0x1023,0x1020,0x101d,0x101a,
	0x1018,0x1015,0x1012,0x100f,0x100c,0x100a,0x1007,0x1004,
};

static uint16_t logtbl[4096]=
{
	0x000,0x000,0x000,0x000,0x001,0x001,0x001,0x002,
	0x002,0x002,0x003,0x003,0x004,0x004,0x004,0x005,
	0x005,0x005,0x006,0x006,0x006,0x007,0x007,0x007,
	0x008,0x008,0x009,0x009,0x009,0x00a,0x00a,0x00a,
	0x00b,0x00b,0x00b,0x00c,0x00c,0x00c,0x00d,0x00d,
	0x00e,0x00e,0x00e,0x00f,0x00f,0x00f,0x010,0x010,
	0x010,0x011,0x011,0x011,0x012,0x012,0x012,0x013,
	0x013,0x014,0x014,0x014,0x015,0x015,0x015,0x016,
	0x016,0x016,0x017,0x017,0x017,0x018,0x018,0x018,
	0x019,0x019,0x01a,0x01a,0x01a,0x01b,0x01b,0x01b,
	0x01c,0x01c,0x01c,0x01d,0x01d,0x01d,0x01e,0x01e,
	0x01e,0x01f,0x01f,0x01f,0x020,0x020,0x021,0x021,
	0x021,0x022,0x022,0x022,0x023,0x023,0x023,0x024,
	0x024,0x024,0x025,0x025,0x025,0x026,0x026,0x026,
	0x027,0x027,0x028,0x028,0x028,0x029,0x029,0x029,
	0x02a,0x02a,0x02a,0x02b,0x02b,0x02b,0x02c,0x02c,
	0x02c,0x02d,0x02d,0x02d,0x02e,0x02e,0x02f,0x02f,
	0x02f,0x030,0x030,0x030,0x031,0x031,0x031,0x032,
	0x032,0x032,0x033,0x033,0x033,0x034,0x034,0x034,
	0x035,0x035,0x035,0x036,0x036,0x036,0x037,0x037,
	0x038,0x038,0x038,0x039,0x039,0x039,0x03a,0x03a,
	0x03a,0x03b,0x03b,0x03b,0x03c,0x03c,0x03c,0x03d,
	0x03d,0x03d,0x03e,0x03e,0x03e,0x03f,0x03f,0x03f,
	0x040,0x040,0x041,0x041,0x041,0x042,0x042,0x042,
	0x043,0x043,0x043,0x044,0x044,0x044,0x045,0x045,
	0x045,0x046,0x046,0x046,0x047,0x047,0x047,0x048,
	0x048,0x048,0x049,0x049,0x049,0x04a,0x04a,0x04a,
	0x04b,0x04b,0x04b,0x04c,0x04c,0x04c,0x04d,0x04d,
	0x04e,0x04e,0x04e,0x04f,0x04f,0x04f,0x050,0x050,
	0x050,0x051,0x051,0x051,0x052,0x052,0x052,0x053,
	0x053,0x053,0x054,0x054,0x054,0x055,0x055,0x055,
	0x056,0x056,0x056,0x057,0x057,0x057,0x058,0x058,
	0x058,0x059,0x059,0x059,0x05a,0x05a,0x05a,0x05b,
	0x05b,0x05b,0x05c,0x05c,0x05c,0x05d,0x05d,0x05d,
	0x05e,0x05e,0x05e,0x05f,0x05f,0x05f,0x060,0x060,
	0x060,0x061,0x061,0x061,0x062,0x062,0x062,0x063,
	0x063,0x063,0x064,0x064,0x064,0x065,0x065,0x065,
	0x066,0x066,0x066,0x067,0x067,0x067,0x068,0x068,
	0x068,0x069,0x069,0x069,0x06a,0x06a,0x06a,0x06b,
	0x06b,0x06b,0x06c,0x06c,0x06c,0x06d,0x06d,0x06d,
	0x06e,0x06e,0x06e,0x06f,0x06f,0x06f,0x070,0x070,
	0x070,0x071,0x071,0x071,0x072,0x072,0x072,0x073,
	0x073,0x073,0x074,0x074,0x074,0x075,0x075,0x075,
	0x076,0x076,0x076,0x077,0x077,0x077,0x078,0x078,
	0x078,0x079,0x079,0x079,0x07a,0x07a,0x07a,0x07b,
	0x07b,0x07b,0x07c,0x07c,0x07c,0x07d,0x07d,0x07d,
	0x07e,0x07e,0x07e,0x07f,0x07f,0x07f,0x080,0x080,
	0x080,0x081,0x081,0x081,0x082,0x082,0x082,0x083,
	0x083,0x083,0x084,0x084,0x084,0x085,0x085,0x085,
	0x086,0x086,0x086,0x087,0x087,0x087,0x088,0x088,
	0x088,0x089,0x089,0x089,0x08a,0x08a,0x08a,0x08b,
	0x08b,0x08b,0x08c,0x08c,0x08c,0x08d,0x08d,0x08d,
	0x08e,0x08e,0x08e,0x08f,0x08f,0x08f,0x090,0x090,
	0x090,0x091,0x091,0x091,0x091,0x092,0x092,0x092,
	0x093,0x093,0x093,0x094,0x094,0x094,0x095,0x095,
	0x095,0x096,0x096,0x096,0x097,0x097,0x097,0x098,
	0x098,0x098,0x099,0x099,0x099,0x09a,0x09a,0x09a,
	0x09b,0x09b,0x09b,0x09c,0x09c,0x09c,0x09d,0x09d,
	0x09d,0x09e,0x09e,0x09e,0x09e,0x09f,0x09f,0x09f,
	0x0a0,0x0a0,0x0a0,0x0a1,0x0a1,0x0a1,0x0a2,0x0a2,
	0x0a2,0x0a3,0x0a3,0x0a3,0x0a4,0x0a4,0x0a4,0x0a5,
	0x0a5,0x0a5,0x0a6,0x0a6,0x0a6,0x0a7,0x0a7,0x0a7,
	0x0a8,0x0a8,0x0a8,0x0a8,0x0a9,0x0a9,0x0a9,0x0aa,
	0x0aa,0x0aa,0x0ab,0x0ab,0x0ab,0x0ac,0x0ac,0x0ac,
	0x0ad,0x0ad,0x0ad,0x0ae,0x0ae,0x0ae,0x0af,0x0af,
	0x0af,0x0b0,0x0b0,0x0b0,0x0b1,0x0b1,0x0b1,0x0b1,
	0x0b2,0x0b2,0x0b2,0x0b3,0x0b3,0x0b3,0x0b4,0x0b4,
	0x0b4,0x0b5,0x0b5,0x0b5,0x0b6,0x0b6,0x0b6,0x0b7,
	0x0b7,0x0b7,0x0b8,0x0b8,0x0b8,0x0b8,0x0b9,0x0b9,
	0x0b9,0x0ba,0x0ba,0x0ba,0x0bb,0x0bb,0x0bb,0x0bc,
	0x0bc,0x0bc,0x0bd,0x0bd,0x0bd,0x0be,0x0be,0x0be,
	0x0bf,0x0bf,0x0bf,0x0bf,0x0c0,0x0c0,0x0c0,0x0c1,
	0x0c1,0x0c1,0x0c2,0x0c2,0x0c2,0x0c3,0x0c3,0x0c3,
	0x0c4,0x0c4,0x0c4,0x0c5,0x0c5,0x0c5,0x0c5,0x0c6,
	0x0c6,0x0c6,0x0c7,0x0c7,0x0c7,0x0c8,0x0c8,0x0c8,
	0x0c9,0x0c9,0x0c9,0x0ca,0x0ca,0x0ca,0x0cb,0x0cb,
	0x0cb,0x0cb,0x0cc,0x0cc,0x0cc,0x0cd,0x0cd,0x0cd,
	0x0ce,0x0ce,0x0ce,0x0cf,0x0cf,0x0cf,0x0d0,0x0d0,
	0x0d0,0x0d0,0x0d1,0x0d1,0x0d1,0x0d2,0x0d2,0x0d2,
	0x0d3,0x0d3,0x0d3,0x0d4,0x0d4,0x0d4,0x0d5,0x0d5,
	0x0d5,0x0d5,0x0d6,0x0d6,0x0d6,0x0d7,0x0d7,0x0d7,
	0x0d8,0x0d8,0x0d8,0x0d9,0x0d9,0x0d9,0x0da,0x0da,
	0x0da,0x0da,0x0db,0x0db,0x0db,0x0dc,0x0dc,0x0dc,
	0x0dd,0x0dd,0x0dd,0x0de,0x0de,0x0de,0x0de,0x0df,
	0x0df,0x0df,0x0e0,0x0e0,0x0e0,0x0e1,0x0e1,0x0e1,
	0x0e2,0x0e2,0x0e2,0x0e2,0x0e3,0x0e3,0x0e3,0x0e4,
	0x0e4,0x0e4,0x0e5,0x0e5,0x0e5,0x0e6,0x0e6,0x0e6,
	0x0e7,0x0e7,0x0e7,0x0e7,0x0e8,0x0e8,0x0e8,0x0e9,
	0x0e9,0x0e9,0x0ea,0x0ea,0x0ea,0x0eb,0x0eb,0x0eb,
	0x0eb,0x0ec,0x0ec,0x0ec,0x0ed,0x0ed,0x0ed,0x0ee,
	0x0ee,0x0ee,0x0ef,0x0ef,0x0ef,0x0ef,0x0f0,0x0f0,
	0x0f0,0x0f1,0x0f1,0x0f1,0x0f2,0x0f2,0x0f2,0x0f3,
	0x0f3,0x0f3,0x0f3,0x0f4,0x0f4,0x0f4,0x0f5,0x0f5,
	0x0f5,0x0f6,0x0f6,0x0f6,0x0f6,0x0f7,0x0f7,0x0f7,
	0x0f8,0x0f8,0x0f8,0x0f9,0x0f9,0x0f9,0x0f9,0x0fa,
	0x0fa,0x0fa,0x0fb,0x0fb,0x0fb,0x0fc,0x0fc,0x0fc,
	0x0fd,0x0fd,0x0fd,0x0fd,0x0fe,0x0fe,0x0fe,0x0ff,
	0x0ff,0x0ff,0x100,0x100,0x100,0x100,0x101,0x101,
	0x101,0x102,0x102,0x102,0x103,0x103,0x103,0x104,
	0x104,0x104,0x104,0x105,0x105,0x105,0x106,0x106,
	0x106,0x107,0x107,0x107,0x107,0x108,0x108,0x108,
	0x109,0x109,0x109,0x10a,0x10a,0x10a,0x10a,0x10b,
	0x10b,0x10b,0x10c,0x10c,0x10c,0x10d,0x10d,0x10d,
	0x10d,0x10e,0x10e,0x10e,0x10f,0x10f,0x10f,0x110,
	0x110,0x110,0x110,0x111,0x111,0x111,0x112,0x112,
	0x112,0x113,0x113,0x113,0x113,0x114,0x114,0x114,
	0x115,0x115,0x115,0x116,0x116,0x116,0x116,0x117,
	0x117,0x117,0x118,0x118,0x118,0x119,0x119,0x119,
	0x119,0x11a,0x11a,0x11a,0x11b,0x11b,0x11b,0x11c,
	0x11c,0x11c,0x11c,0x11d,0x11d,0x11d,0x11e,0x11e,
	0x11e,0x11e,0x11f,0x11f,0x11f,0x120,0x120,0x120,
	0x121,0x121,0x121,0x121,0x122,0x122,0x122,0x123,
	0x123,0x123,0x124,0x124,0x124,0x124,0x125,0x125,
	0x125,0x126,0x126,0x126,0x126,0x127,0x127,0x127,
	0x128,0x128,0x128,0x129,0x129,0x129,0x129,0x12a,
	0x12a,0x12a,0x12b,0x12b,0x12b,0x12b,0x12c,0x12c,
	0x12c,0x12d,0x12d,0x12d,0x12e,0x12e,0x12e,0x12e,
	0x12f,0x12f,0x12f,0x130,0x130,0x130,0x131,0x131,
	0x131,0x131,0x132,0x132,0x132,0x133,0x133,0x133,
	0x133,0x134,0x134,0x134,0x135,0x135,0x135,0x135,
	0x136,0x136,0x136,0x137,0x137,0x137,0x137,0x138,
	0x138,0x138,0x139,0x139,0x139,0x13a,0x13a,0x13a,
	0x13a,0x13b,0x13b,0x13b,0x13c,0x13c,0x13c,0x13c,
	0x13d,0x13d,0x13d,0x13e,0x13e,0x13e,0x13f,0x13f,
	0x13f,0x13f,0x140,0x140,0x140,0x141,0x141,0x141,
	0x141,0x142,0x142,0x142,0x143,0x143,0x143,0x143,
	0x144,0x144,0x144,0x145,0x145,0x145,0x145,0x146,
	0x146,0x146,0x147,0x147,0x147,0x148,0x148,0x148,
	0x148,0x149,0x149,0x149,0x149,0x14a,0x14a,0x14a,
	0x14b,0x14b,0x14b,0x14c,0x14c,0x14c,0x14c,0x14d,
	0x14d,0x14d,0x14e,0x14e,0x14e,0x14e,0x14f,0x14f,
	0x14f,0x150,0x150,0x150,0x150,0x151,0x151,0x151,
	0x152,0x152,0x152,0x152,0x153,0x153,0x153,0x154,
	0x154,0x154,0x154,0x155,0x155,0x155,0x156,0x156,
	0x156,0x156,0x157,0x157,0x157,0x158,0x158,0x158,
	0x158,0x159,0x159,0x159,0x15a,0x15a,0x15a,0x15a,
	0x15b,0x15b,0x15b,0x15c,0x15c,0x15c,0x15c,0x15d,
	0x15d,0x15d,0x15e,0x15e,0x15e,0x15e,0x15f,0x15f,
	0x15f,0x160,0x160,0x160,0x160,0x161,0x161,0x161,
	0x162,0x162,0x162,0x162,0x163,0x163,0x163,0x164,
	0x164,0x164,0x164,0x165,0x165,0x165,0x166,0x166,
	0x166,0x166,0x167,0x167,0x167,0x167,0x168,0x168,
	0x168,0x169,0x169,0x169,0x169,0x16a,0x16a,0x16a,
	0x16b,0x16b,0x16b,0x16b,0x16c,0x16c,0x16c,0x16d,
	0x16d,0x16d,0x16d,0x16e,0x16e,0x16e,0x16f,0x16f,
	0x16f,0x16f,0x170,0x170,0x170,0x171,0x171,0x171,
	0x171,0x172,0x172,0x172,0x172,0x173,0x173,0x173,
	0x174,0x174,0x174,0x174,0x175,0x175,0x175,0x176,
	0x176,0x176,0x176,0x177,0x177,0x177,0x178,0x178,
	0x178,0x178,0x179,0x179,0x179,0x179,0x17a,0x17a,
	0x17a,0x17b,0x17b,0x17b,0x17b,0x17c,0x17c,0x17c,
	0x17d,0x17d,0x17d,0x17d,0x17e,0x17e,0x17e,0x17e,
	0x17f,0x17f,0x17f,0x180,0x180,0x180,0x180,0x181,
	0x181,0x181,0x182,0x182,0x182,0x182,0x183,0x183,
	0x183,0x183,0x184,0x184,0x184,0x185,0x185,0x185,
	0x185,0x186,0x186,0x186,0x187,0x187,0x187,0x187,
	0x188,0x188,0x188,0x188,0x189,0x189,0x189,0x18a,
	0x18a,0x18a,0x18a,0x18b,0x18b,0x18b,0x18c,0x18c,
	0x18c,0x18c,0x18d,0x18d,0x18d,0x18d,0x18e,0x18e,
	0x18e,0x18f,0x18f,0x18f,0x18f,0x190,0x190,0x190,
	0x190,0x191,0x191,0x191,0x192,0x192,0x192,0x192,
	0x193,0x193,0x193,0x194,0x194,0x194,0x194,0x195,
	0x195,0x195,0x195,0x196,0x196,0x196,0x197,0x197,
	0x197,0x197,0x198,0x198,0x198,0x198,0x199,0x199,
	0x199,0x19a,0x19a,0x19a,0x19a,0x19b,0x19b,0x19b,
	0x19b,0x19c,0x19c,0x19c,0x19d,0x19d,0x19d,0x19d,
	0x19e,0x19e,0x19e,0x19e,0x19f,0x19f,0x19f,0x1a0,
	0x1a0,0x1a0,0x1a0,0x1a1,0x1a1,0x1a1,0x1a1,0x1a2,
	0x1a2,0x1a2,0x1a3,0x1a3,0x1a3,0x1a3,0x1a4,0x1a4,
	0x1a4,0x1a4,0x1a5,0x1a5,0x1a5,0x1a5,0x1a6,0x1a6,
	0x1a6,0x1a7,0x1a7,0x1a7,0x1a7,0x1a8,0x1a8,0x1a8,
	0x1a8,0x1a9,0x1a9,0x1a9,0x1aa,0x1aa,0x1aa,0x1aa,
	0x1ab,0x1ab,0x1ab,0x1ab,0x1ac,0x1ac,0x1ac,0x1ad,
	0x1ad,0x1ad,0x1ad,0x1ae,0x1ae,0x1ae,0x1ae,0x1af,
	0x1af,0x1af,0x1af,0x1b0,0x1b0,0x1b0,0x1b1,0x1b1,
	0x1b1,0x1b1,0x1b2,0x1b2,0x1b2,0x1b2,0x1b3,0x1b3,
	0x1b3,0x1b4,0x1b4,0x1b4,0x1b4,0x1b5,0x1b5,0x1b5,
	0x1b5,0x1b6,0x1b6,0x1b6,0x1b6,0x1b7,0x1b7,0x1b7,
	0x1b8,0x1b8,0x1b8,0x1b8,0x1b9,0x1b9,0x1b9,0x1b9,
	0x1ba,0x1ba,0x1ba,0x1ba,0x1bb,0x1bb,0x1bb,0x1bc,
	0x1bc,0x1bc,0x1bc,0x1bd,0x1bd,0x1bd,0x1bd,0x1be,
	0x1be,0x1be,0x1bf,0x1bf,0x1bf,0x1bf,0x1c0,0x1c0,
	0x1c0,0x1c0,0x1c1,0x1c1,0x1c1,0x1c1,0x1c2,0x1c2,
	0x1c2,0x1c3,0x1c3,0x1c3,0x1c3,0x1c4,0x1c4,0x1c4,
	0x1c4,0x1c5,0x1c5,0x1c5,0x1c5,0x1c6,0x1c6,0x1c6,
	0x1c6,0x1c7,0x1c7,0x1c7,0x1c8,0x1c8,0x1c8,0x1c8,
	0x1c9,0x1c9,0x1c9,0x1c9,0x1ca,0x1ca,0x1ca,0x1ca,
	0x1cb,0x1cb,0x1cb,0x1cb,0x1cc,0x1cc,0x1cc,0x1cd,
	0x1cd,0x1cd,0x1cd,0x1ce,0x1ce,0x1ce,0x1ce,0x1cf,
	0x1cf,0x1cf,0x1cf,0x1d0,0x1d0,0x1d0,0x1d1,0x1d1,
	0x1d1,0x1d1,0x1d2,0x1d2,0x1d2,0x1d2,0x1d3,0x1d3,
	0x1d3,0x1d3,0x1d4,0x1d4,0x1d4,0x1d4,0x1d5,0x1d5,
	0x1d5,0x1d5,0x1d6,0x1d6,0x1d6,0x1d7,0x1d7,0x1d7,
	0x1d7,0x1d8,0x1d8,0x1d8,0x1d8,0x1d9,0x1d9,0x1d9,
	0x1d9,0x1da,0x1da,0x1da,0x1da,0x1db,0x1db,0x1db,
	0x1dc,0x1dc,0x1dc,0x1dc,0x1dd,0x1dd,0x1dd,0x1dd,
	0x1de,0x1de,0x1de,0x1de,0x1df,0x1df,0x1df,0x1df,
	0x1e0,0x1e0,0x1e0,0x1e0,0x1e1,0x1e1,0x1e1,0x1e2,
	0x1e2,0x1e2,0x1e2,0x1e3,0x1e3,0x1e3,0x1e3,0x1e4,
	0x1e4,0x1e4,0x1e4,0x1e5,0x1e5,0x1e5,0x1e5,0x1e6,
	0x1e6,0x1e6,0x1e6,0x1e7,0x1e7,0x1e7,0x1e7,0x1e8,
	0x1e8,0x1e8,0x1e9,0x1e9,0x1e9,0x1e9,0x1ea,0x1ea,
	0x1ea,0x1ea,0x1eb,0x1eb,0x1eb,0x1eb,0x1ec,0x1ec,
	0x1ec,0x1ec,0x1ed,0x1ed,0x1ed,0x1ed,0x1ee,0x1ee,
	0x1ee,0x1ee,0x1ef,0x1ef,0x1ef,0x1f0,0x1f0,0x1f0,
	0x1f0,0x1f1,0x1f1,0x1f1,0x1f1,0x1f2,0x1f2,0x1f2,
	0x1f2,0x1f3,0x1f3,0x1f3,0x1f3,0x1f4,0x1f4,0x1f4,
	0x1f4,0x1f5,0x1f5,0x1f5,0x1f5,0x1f6,0x1f6,0x1f6,
	0x1f6,0x1f7,0x1f7,0x1f7,0x1f7,0x1f8,0x1f8,0x1f8,
	0x1f9,0x1f9,0x1f9,0x1f9,0x1fa,0x1fa,0x1fa,0x1fa,
	0x1fb,0x1fb,0x1fb,0x1fb,0x1fc,0x1fc,0x1fc,0x1fc,
	0x1fd,0x1fd,0x1fd,0x1fd,0x1fe,0x1fe,0x1fe,0x1fe,
	0x1ff,0x1ff,0x1ff,0x1ff,0x200,0x200,0x200,0x200,
	0x201,0x201,0x201,0x201,0x202,0x202,0x202,0x202,
	0x203,0x203,0x203,0x204,0x204,0x204,0x204,0x205,
	0x205,0x205,0x205,0x206,0x206,0x206,0x206,0x207,
	0x207,0x207,0x207,0x208,0x208,0x208,0x208,0x209,
	0x209,0x209,0x209,0x20a,0x20a,0x20a,0x20a,0x20b,
	0x20b,0x20b,0x20b,0x20c,0x20c,0x20c,0x20c,0x20d,
	0x20d,0x20d,0x20d,0x20e,0x20e,0x20e,0x20e,0x20f,
	0x20f,0x20f,0x20f,0x210,0x210,0x210,0x210,0x211,
	0x211,0x211,0x211,0x212,0x212,0x212,0x212,0x213,
	0x213,0x213,0x213,0x214,0x214,0x214,0x214,0x215,
	0x215,0x215,0x215,0x216,0x216,0x216,0x216,0x217,
	0x217,0x217,0x217,0x218,0x218,0x218,0x218,0x219,
	0x219,0x219,0x219,0x21a,0x21a,0x21a,0x21a,0x21b,
	0x21b,0x21b,0x21b,0x21c,0x21c,0x21c,0x21c,0x21d,
	0x21d,0x21d,0x21d,0x21e,0x21e,0x21e,0x21e,0x21f,
	0x21f,0x21f,0x21f,0x220,0x220,0x220,0x220,0x221,
	0x221,0x221,0x221,0x222,0x222,0x222,0x222,0x223,
	0x223,0x223,0x223,0x224,0x224,0x224,0x224,0x225,
	0x225,0x225,0x225,0x226,0x226,0x226,0x226,0x227,
	0x227,0x227,0x227,0x228,0x228,0x228,0x228,0x229,
	0x229,0x229,0x229,0x22a,0x22a,0x22a,0x22a,0x22b,
	0x22b,0x22b,0x22b,0x22c,0x22c,0x22c,0x22c,0x22d,
	0x22d,0x22d,0x22d,0x22e,0x22e,0x22e,0x22e,0x22f,
	0x22f,0x22f,0x22f,0x230,0x230,0x230,0x230,0x231,
	0x231,0x231,0x231,0x232,0x232,0x232,0x232,0x233,
	0x233,0x233,0x233,0x234,0x234,0x234,0x234,0x235,
	0x235,0x235,0x235,0x236,0x236,0x236,0x236,0x237,
	0x237,0x237,0x237,0x237,0x238,0x238,0x238,0x238,
	0x239,0x239,0x239,0x239,0x23a,0x23a,0x23a,0x23a,
	0x23b,0x23b,0x23b,0x23b,0x23c,0x23c,0x23c,0x23c,
	0x23d,0x23d,0x23d,0x23d,0x23e,0x23e,0x23e,0x23e,
	0x23f,0x23f,0x23f,0x23f,0x240,0x240,0x240,0x240,
	0x241,0x241,0x241,0x241,0x242,0x242,0x242,0x242,
	0x243,0x243,0x243,0x243,0x243,0x244,0x244,0x244,
	0x244,0x245,0x245,0x245,0x245,0x246,0x246,0x246,
	0x246,0x247,0x247,0x247,0x247,0x248,0x248,0x248,
	0x248,0x249,0x249,0x249,0x249,0x24a,0x24a,0x24a,
	0x24a,0x24b,0x24b,0x24b,0x24b,0x24c,0x24c,0x24c,
	0x24c,0x24c,0x24d,0x24d,0x24d,0x24d,0x24e,0x24e,
	0x24e,0x24e,0x24f,0x24f,0x24f,0x24f,0x250,0x250,
	0x250,0x250,0x251,0x251,0x251,0x251,0x252,0x252,
	0x252,0x252,0x253,0x253,0x253,0x253,0x253,0x254,
	0x254,0x254,0x254,0x255,0x255,0x255,0x255,0x256,
	0x256,0x256,0x256,0x257,0x257,0x257,0x257,0x258,
	0x258,0x258,0x258,0x259,0x259,0x259,0x259,0x25a,
	0x25a,0x25a,0x25a,0x25a,0x25b,0x25b,0x25b,0x25b,
	0x25c,0x25c,0x25c,0x25c,0x25d,0x25d,0x25d,0x25d,
	0x25e,0x25e,0x25e,0x25e,0x25f,0x25f,0x25f,0x25f,
	0x25f,0x260,0x260,0x260,0x260,0x261,0x261,0x261,
	0x261,0x262,0x262,0x262,0x262,0x263,0x263,0x263,
	0x263,0x264,0x264,0x264,0x264,0x264,0x265,0x265,
	0x265,0x265,0x266,0x266,0x266,0x266,0x267,0x267,
	0x267,0x267,0x268,0x268,0x268,0x268,0x269,0x269,
	0x269,0x269,0x269,0x26a,0x26a,0x26a,0x26a,0x26b,
	0x26b,0x26b,0x26b,0x26c,0x26c,0x26c,0x26c,0x26d,
	0x26d,0x26d,0x26d,0x26e,0x26e,0x26e,0x26e,0x26e,
	0x26f,0x26f,0x26f,0x26f,0x270,0x270,0x270,0x270,
	0x271,0x271,0x271,0x271,0x272,0x272,0x272,0x272,
	0x272,0x273,0x273,0x273,0x273,0x274,0x274,0x274,
	0x274,0x275,0x275,0x275,0x275,0x276,0x276,0x276,
	0x276,0x276,0x277,0x277,0x277,0x277,0x278,0x278,
	0x278,0x278,0x279,0x279,0x279,0x279,0x27a,0x27a,
	0x27a,0x27a,0x27a,0x27b,0x27b,0x27b,0x27b,0x27c,
	0x27c,0x27c,0x27c,0x27d,0x27d,0x27d,0x27d,0x27e,
	0x27e,0x27e,0x27e,0x27e,0x27f,0x27f,0x27f,0x27f,
	0x280,0x280,0x280,0x280,0x281,0x281,0x281,0x281,
	0x282,0x282,0x282,0x282,0x282,0x283,0x283,0x283,
	0x283,0x284,0x284,0x284,0x284,0x285,0x285,0x285,
	0x285,0x285,0x286,0x286,0x286,0x286,0x287,0x287,
	0x287,0x287,0x288,0x288,0x288,0x288,0x289,0x289,
	0x289,0x289,0x289,0x28a,0x28a,0x28a,0x28a,0x28b,
	0x28b,0x28b,0x28b,0x28c,0x28c,0x28c,0x28c,0x28c,
	0x28d,0x28d,0x28d,0x28d,0x28e,0x28e,0x28e,0x28e,
	0x28f,0x28f,0x28f,0x28f,0x28f,0x290,0x290,0x290,
	0x290,0x291,0x291,0x291,0x291,0x292,0x292,0x292,
	0x292,0x292,0x293,0x293,0x293,0x293,0x294,0x294,
	0x294,0x294,0x295,0x295,0x295,0x295,0x295,0x296,
	0x296,0x296,0x296,0x297,0x297,0x297,0x297,0x298,
	0x298,0x298,0x298,0x298,0x299,0x299,0x299,0x299,
	0x29a,0x29a,0x29a,0x29a,0x29b,0x29b,0x29b,0x29b,
	0x29b,0x29c,0x29c,0x29c,0x29c,0x29d,0x29d,0x29d,
	0x29d,0x29e,0x29e,0x29e,0x29e,0x29e,0x29f,0x29f,
	0x29f,0x29f,0x2a0,0x2a0,0x2a0,0x2a0,0x2a1,0x2a1,
	0x2a1,0x2a1,0x2a1,0x2a2,0x2a2,0x2a2,0x2a2,0x2a3,
	0x2a3,0x2a3,0x2a3,0x2a3,0x2a4,0x2a4,0x2a4,0x2a4,
	0x2a5,0x2a5,0x2a5,0x2a5,0x2a6,0x2a6,0x2a6,0x2a6,
	0x2a6,0x2a7,0x2a7,0x2a7,0x2a7,0x2a8,0x2a8,0x2a8,
	0x2a8,0x2a8,0x2a9,0x2a9,0x2a9,0x2a9,0x2aa,0x2aa,
	0x2aa,0x2aa,0x2ab,0x2ab,0x2ab,0x2ab,0x2ab,0x2ac,
	0x2ac,0x2ac,0x2ac,0x2ad,0x2ad,0x2ad,0x2ad,0x2ad,
	0x2ae,0x2ae,0x2ae,0x2ae,0x2af,0x2af,0x2af,0x2af,
	0x2b0,0x2b0,0x2b0,0x2b0,0x2b0,0x2b1,0x2b1,0x2b1,
	0x2b1,0x2b2,0x2b2,0x2b2,0x2b2,0x2b2,0x2b3,0x2b3,
	0x2b3,0x2b3,0x2b4,0x2b4,0x2b4,0x2b4,0x2b5,0x2b5,
	0x2b5,0x2b5,0x2b5,0x2b6,0x2b6,0x2b6,0x2b6,0x2b7,
	0x2b7,0x2b7,0x2b7,0x2b7,0x2b8,0x2b8,0x2b8,0x2b8,
	0x2b9,0x2b9,0x2b9,0x2b9,0x2b9,0x2ba,0x2ba,0x2ba,
	0x2ba,0x2bb,0x2bb,0x2bb,0x2bb,0x2bb,0x2bc,0x2bc,
	0x2bc,0x2bc,0x2bd,0x2bd,0x2bd,0x2bd,0x2be,0x2be,
	0x2be,0x2be,0x2be,0x2bf,0x2bf,0x2bf,0x2bf,0x2c0,
	0x2c0,0x2c0,0x2c0,0x2c0,0x2c1,0x2c1,0x2c1,0x2c1,
	0x2c2,0x2c2,0x2c2,0x2c2,0x2c2,0x2c3,0x2c3,0x2c3,
	0x2c3,0x2c4,0x2c4,0x2c4,0x2c4,0x2c4,0x2c5,0x2c5,
	0x2c5,0x2c5,0x2c6,0x2c6,0x2c6,0x2c6,0x2c6,0x2c7,
	0x2c7,0x2c7,0x2c7,0x2c8,0x2c8,0x2c8,0x2c8,0x2c8,
	0x2c9,0x2c9,0x2c9,0x2c9,0x2ca,0x2ca,0x2ca,0x2ca,
	0x2ca,0x2cb,0x2cb,0x2cb,0x2cb,0x2cc,0x2cc,0x2cc,
	0x2cc,0x2cc,0x2cd,0x2cd,0x2cd,0x2cd,0x2ce,0x2ce,
	0x2ce,0x2ce,0x2ce,0x2cf,0x2cf,0x2cf,0x2cf,0x2d0,
	0x2d0,0x2d0,0x2d0,0x2d0,0x2d1,0x2d1,0x2d1,0x2d1,
	0x2d2,0x2d2,0x2d2,0x2d2,0x2d2,0x2d3,0x2d3,0x2d3,
	0x2d3,0x2d4,0x2d4,0x2d4,0x2d4,0x2d4,0x2d5,0x2d5,
	0x2d5,0x2d5,0x2d6,0x2d6,0x2d6,0x2d6,0x2d6,0x2d7,
	0x2d7,0x2d7,0x2d7,0x2d8,0x2d8,0x2d8,0x2d8,0x2d8,
	0x2d9,0x2d9,0x2d9,0x2d9,0x2da,0x2da,0x2da,0x2da,
	0x2da,0x2db,0x2db,0x2db,0x2db,0x2db,0x2dc,0x2dc,
	0x2dc,0x2dc,0x2dd,0x2dd,0x2dd,0x2dd,0x2dd,0x2de,
	0x2de,0x2de,0x2de,0x2df,0x2df,0x2df,0x2df,0x2df,
	0x2e0,0x2e0,0x2e0,0x2e0,0x2e1,0x2e1,0x2e1,0x2e1,
	0x2e1,0x2e2,0x2e2,0x2e2,0x2e2,0x2e3,0x2e3,0x2e3,
	0x2e3,0x2e3,0x2e4,0x2e4,0x2e4,0x2e4,0x2e4,0x2e5,
	0x2e5,0x2e5,0x2e5,0x2e6,0x2e6,0x2e6,0x2e6,0x2e6,
	0x2e7,0x2e7,0x2e7,0x2e7,0x2e8,0x2e8,0x2e8,0x2e8,
	0x2e8,0x2e9,0x2e9,0x2e9,0x2e9,0x2ea,0x2ea,0x2ea,
	0x2ea,0x2ea,0x2eb,0x2eb,0x2eb,0x2eb,0x2eb,0x2ec,
	0x2ec,0x2ec,0x2ec,0x2ed,0x2ed,0x2ed,0x2ed,0x2ed,
	0x2ee,0x2ee,0x2ee,0x2ee,0x2ef,0x2ef,0x2ef,0x2ef,
	0x2ef,0x2f0,0x2f0,0x2f0,0x2f0,0x2f0,0x2f1,0x2f1,
	0x2f1,0x2f1,0x2f2,0x2f2,0x2f2,0x2f2,0x2f2,0x2f3,
	0x2f3,0x2f3,0x2f3,0x2f3,0x2f4,0x2f4,0x2f4,0x2f4,
	0x2f5,0x2f5,0x2f5,0x2f5,0x2f5,0x2f6,0x2f6,0x2f6,
	0x2f6,0x2f7,0x2f7,0x2f7,0x2f7,0x2f7,0x2f8,0x2f8,
	0x2f8,0x2f8,0x2f8,0x2f9,0x2f9,0x2f9,0x2f9,0x2fa,
	0x2fa,0x2fa,0x2fa,0x2fa,0x2fb,0x2fb,0x2fb,0x2fb,
	0x2fb,0x2fc,0x2fc,0x2fc,0x2fc,0x2fd,0x2fd,0x2fd,
	0x2fd,0x2fd,0x2fe,0x2fe,0x2fe,0x2fe,0x2fe,0x2ff,
	0x2ff,0x2ff,0x2ff,0x300,0x300,0x300,0x300,0x300,
	0x301,0x301,0x301,0x301,0x301,0x302,0x302,0x302,
	0x302,0x303,0x303,0x303,0x303,0x303,0x304,0x304,
	0x304,0x304,0x304,0x305,0x305,0x305,0x305,0x306,
	0x306,0x306,0x306,0x306,0x307,0x307,0x307,0x307,
	0x307,0x308,0x308,0x308,0x308,0x309,0x309,0x309,
	0x309,0x309,0x30a,0x30a,0x30a,0x30a,0x30a,0x30b,
	0x30b,0x30b,0x30b,0x30c,0x30c,0x30c,0x30c,0x30c,
	0x30d,0x30d,0x30d,0x30d,0x30d,0x30e,0x30e,0x30e,
	0x30e,0x30e,0x30f,0x30f,0x30f,0x30f,0x310,0x310,
	0x310,0x310,0x310,0x311,0x311,0x311,0x311,0x311,
	0x312,0x312,0x312,0x312,0x313,0x313,0x313,0x313,
	0x313,0x314,0x314,0x314,0x314,0x314,0x315,0x315,
	0x315,0x315,0x315,0x316,0x316,0x316,0x316,0x317,
	0x317,0x317,0x317,0x317,0x318,0x318,0x318,0x318,
	0x318,0x319,0x319,0x319,0x319,0x319,0x31a,0x31a,
	0x31a,0x31a,0x31b,0x31b,0x31b,0x31b,0x31b,0x31c,
	0x31c,0x31c,0x31c,0x31c,0x31d,0x31d,0x31d,0x31d,
	0x31e,0x31e,0x31e,0x31e,0x31e,0x31f,0x31f,0x31f,
	0x31f,0x31f,0x320,0x320,0x320,0x320,0x320,0x321,
	0x321,0x321,0x321,0x321,0x322,0x322,0x322,0x322,
	0x323,0x323,0x323,0x323,0x323,0x324,0x324,0x324,
	0x324,0x324,0x325,0x325,0x325,0x325,0x325,0x326,
	0x326,0x326,0x326,0x327,0x327,0x327,0x327,0x327,
	0x328,0x328,0x328,0x328,0x328,0x329,0x329,0x329,
	0x329,0x329,0x32a,0x32a,0x32a,0x32a,0x32a,0x32b,
	0x32b,0x32b,0x32b,0x32c,0x32c,0x32c,0x32c,0x32c,
	0x32d,0x32d,0x32d,0x32d,0x32d,0x32e,0x32e,0x32e,
	0x32e,0x32e,0x32f,0x32f,0x32f,0x32f,0x32f,0x330,
	0x330,0x330,0x330,0x331,0x331,0x331,0x331,0x331,
	0x332,0x332,0x332,0x332,0x332,0x333,0x333,0x333,
	0x333,0x333,0x334,0x334,0x334,0x334,0x334,0x335,
	0x335,0x335,0x335,0x335,0x336,0x336,0x336,0x336,
	0x337,0x337,0x337,0x337,0x337,0x338,0x338,0x338,
	0x338,0x338,0x339,0x339,0x339,0x339,0x339,0x33a,
	0x33a,0x33a,0x33a,0x33a,0x33b,0x33b,0x33b,0x33b,
	0x33b,0x33c,0x33c,0x33c,0x33c,0x33d,0x33d,0x33d,
	0x33d,0x33d,0x33e,0x33e,0x33e,0x33e,0x33e,0x33f,
	0x33f,0x33f,0x33f,0x33f,0x340,0x340,0x340,0x340,
	0x340,0x341,0x341,0x341,0x341,0x341,0x342,0x342,
	0x342,0x342,0x342,0x343,0x343,0x343,0x343,0x344,
	0x344,0x344,0x344,0x344,0x345,0x345,0x345,0x345,
	0x345,0x346,0x346,0x346,0x346,0x346,0x347,0x347,
	0x347,0x347,0x347,0x348,0x348,0x348,0x348,0x348,
	0x349,0x349,0x349,0x349,0x349,0x34a,0x34a,0x34a,
	0x34a,0x34a,0x34b,0x34b,0x34b,0x34b,0x34b,0x34c,
	0x34c,0x34c,0x34c,0x34c,0x34d,0x34d,0x34d,0x34d,
	0x34d,0x34e,0x34e,0x34e,0x34e,0x34f,0x34f,0x34f,
	0x34f,0x34f,0x350,0x350,0x350,0x350,0x350,0x351,
	0x351,0x351,0x351,0x351,0x352,0x352,0x352,0x352,
	0x352,0x353,0x353,0x353,0x353,0x353,0x354,0x354,
	0x354,0x354,0x354,0x355,0x355,0x355,0x355,0x355,
	0x356,0x356,0x356,0x356,0x356,0x357,0x357,0x357,
	0x357,0x357,0x358,0x358,0x358,0x358,0x358,0x359,
	0x359,0x359,0x359,0x359,0x35a,0x35a,0x35a,0x35a,
	0x35a,0x35b,0x35b,0x35b,0x35b,0x35b,0x35c,0x35c,
	0x35c,0x35c,0x35c,0x35d,0x35d,0x35d,0x35d,0x35d,
	0x35e,0x35e,0x35e,0x35e,0x35e,0x35f,0x35f,0x35f,
	0x35f,0x35f,0x360,0x360,0x360,0x360,0x360,0x361,
	0x361,0x361,0x361,0x362,0x362,0x362,0x362,0x362,
	0x363,0x363,0x363,0x363,0x363,0x364,0x364,0x364,
	0x364,0x364,0x365,0x365,0x365,0x365,0x365,0x366,
	0x366,0x366,0x366,0x366,0x367,0x367,0x367,0x367,
	0x367,0x368,0x368,0x368,0x368,0x368,0x369,0x369,
	0x369,0x369,0x369,0x36a,0x36a,0x36a,0x36a,0x36a,
	0x36b,0x36b,0x36b,0x36b,0x36b,0x36c,0x36c,0x36c,
	0x36c,0x36c,0x36d,0x36d,0x36d,0x36d,0x36d,0x36e,
	0x36e,0x36e,0x36e,0x36e,0x36f,0x36f,0x36f,0x36f,
	0x36f,0x370,0x370,0x370,0x370,0x370,0x371,0x371,
	0x371,0x371,0x371,0x371,0x372,0x372,0x372,0x372,
	0x372,0x373,0x373,0x373,0x373,0x373,0x374,0x374,
	0x374,0x374,0x374,0x375,0x375,0x375,0x375,0x375,
	0x376,0x376,0x376,0x376,0x376,0x377,0x377,0x377,
	0x377,0x377,0x378,0x378,0x378,0x378,0x378,0x379,
	0x379,0x379,0x379,0x379,0x37a,0x37a,0x37a,0x37a,
	0x37a,0x37b,0x37b,0x37b,0x37b,0x37b,0x37c,0x37c,
	0x37c,0x37c,0x37c,0x37d,0x37d,0x37d,0x37d,0x37d,
	0x37e,0x37e,0x37e,0x37e,0x37e,0x37f,0x37f,0x37f,
	0x37f,0x37f,0x380,0x380,0x380,0x380,0x380,0x381,
	0x381,0x381,0x381,0x381,0x381,0x382,0x382,0x382,
	0x382,0x382,0x383,0x383,0x383,0x383,0x383,0x384,
	0x384,0x384,0x384,0x384,0x385,0x385,0x385,0x385,
	0x385,0x386,0x386,0x386,0x386,0x386,0x387,0x387,
	0x387,0x387,0x387,0x388,0x388,0x388,0x388,0x388,
	0x389,0x389,0x389,0x389,0x389,0x38a,0x38a,0x38a,
	0x38a,0x38a,0x38a,0x38b,0x38b,0x38b,0x38b,0x38b,
	0x38c,0x38c,0x38c,0x38c,0x38c,0x38d,0x38d,0x38d,
	0x38d,0x38d,0x38e,0x38e,0x38e,0x38e,0x38e,0x38f,
	0x38f,0x38f,0x38f,0x38f,0x390,0x390,0x390,0x390,
	0x390,0x391,0x391,0x391,0x391,0x391,0x392,0x392,
	0x392,0x392,0x392,0x392,0x393,0x393,0x393,0x393,
	0x393,0x394,0x394,0x394,0x394,0x394,0x395,0x395,
	0x395,0x395,0x395,0x396,0x396,0x396,0x396,0x396,
	0x397,0x397,0x397,0x397,0x397,0x398,0x398,0x398,
	0x398,0x398,0x399,0x399,0x399,0x399,0x399,0x399,
	0x39a,0x39a,0x39a,0x39a,0x39a,0x39b,0x39b,0x39b,
	0x39b,0x39b,0x39c,0x39c,0x39c,0x39c,0x39c,0x39d,
	0x39d,0x39d,0x39d,0x39d,0x39e,0x39e,0x39e,0x39e,
	0x39e,0x39e,0x39f,0x39f,0x39f,0x39f,0x39f,0x3a0,
	0x3a0,0x3a0,0x3a0,0x3a0,0x3a1,0x3a1,0x3a1,0x3a1,
	0x3a1,0x3a2,0x3a2,0x3a2,0x3a2,0x3a2,0x3a3,0x3a3,
	0x3a3,0x3a3,0x3a3,0x3a3,0x3a4,0x3a4,0x3a4,0x3a4,
	0x3a4,0x3a5,0x3a5,0x3a5,0x3a5,0x3a5,0x3a6,0x3a6,
	0x3a6,0x3a6,0x3a6,0x3a7,0x3a7,0x3a7,0x3a7,0x3a7,
	0x3a8,0x3a8,0x3a8,0x3a8,0x3a8,0x3a8,0x3a9,0x3a9,
	0x3a9,0x3a9,0x3a9,0x3aa,0x3aa,0x3aa,0x3aa,0x3aa,
	0x3ab,0x3ab,0x3ab,0x3ab,0x3ab,0x3ac,0x3ac,0x3ac,
	0x3ac,0x3ac,0x3ac,0x3ad,0x3ad,0x3ad,0x3ad,0x3ad,
	0x3ae,0x3ae,0x3ae,0x3ae,0x3ae,0x3af,0x3af,0x3af,
	0x3af,0x3af,0x3b0,0x3b0,0x3b0,0x3b0,0x3b0,0x3b0,
	0x3b1,0x3b1,0x3b1,0x3b1,0x3b1,0x3b2,0x3b2,0x3b2,
	0x3b2,0x3b2,0x3b3,0x3b3,0x3b3,0x3b3,0x3b3,0x3b4,
	0x3b4,0x3b4,0x3b4,0x3b4,0x3b4,0x3b5,0x3b5,0x3b5,
	0x3b5,0x3b5,0x3b6,0x3b6,0x3b6,0x3b6,0x3b6,0x3b7,
	0x3b7,0x3b7,0x3b7,0x3b7,0x3b8,0x3b8,0x3b8,0x3b8,
	0x3b8,0x3b8,0x3b9,0x3b9,0x3b9,0x3b9,0x3b9,0x3ba,
	0x3ba,0x3ba,0x3ba,0x3ba,0x3bb,0x3bb,0x3bb,0x3bb,
	0x3bb,0x3bb,0x3bc,0x3bc,0x3bc,0x3bc,0x3bc,0x3bd,
	0x3bd,0x3bd,0x3bd,0x3bd,0x3be,0x3be,0x3be,0x3be,
	0x3be,0x3bf,0x3bf,0x3bf,0x3bf,0x3bf,0x3bf,0x3c0,
	0x3c0,0x3c0,0x3c0,0x3c0,0x3c1,0x3c1,0x3c1,0x3c1,
	0x3c1,0x3c2,0x3c2,0x3c2,0x3c2,0x3c2,0x3c2,0x3c3,
	0x3c3,0x3c3,0x3c3,0x3c3,0x3c4,0x3c4,0x3c4,0x3c4,
	0x3c4,0x3c5,0x3c5,0x3c5,0x3c5,0x3c5,0x3c5,0x3c6,
	0x3c6,0x3c6,0x3c6,0x3c6,0x3c7,0x3c7,0x3c7,0x3c7,
	0x3c7,0x3c8,0x3c8,0x3c8,0x3c8,0x3c8,0x3c8,0x3c9,
	0x3c9,0x3c9,0x3c9,0x3c9,0x3ca,0x3ca,0x3ca,0x3ca,
	0x3ca,0x3cb,0x3cb,0x3cb,0x3cb,0x3cb,0x3cb,0x3cc,
	0x3cc,0x3cc,0x3cc,0x3cc,0x3cd,0x3cd,0x3cd,0x3cd,
	0x3cd,0x3ce,0x3ce,0x3ce,0x3ce,0x3ce,0x3ce,0x3cf,
	0x3cf,0x3cf,0x3cf,0x3cf,0x3d0,0x3d0,0x3d0,0x3d0,
	0x3d0,0x3d0,0x3d1,0x3d1,0x3d1,0x3d1,0x3d1,0x3d2,
	0x3d2,0x3d2,0x3d2,0x3d2,0x3d3,0x3d3,0x3d3,0x3d3,
	0x3d3,0x3d3,0x3d4,0x3d4,0x3d4,0x3d4,0x3d4,0x3d5,
	0x3d5,0x3d5,0x3d5,0x3d5,0x3d6,0x3d6,0x3d6,0x3d6,
	0x3d6,0x3d6,0x3d7,0x3d7,0x3d7,0x3d7,0x3d7,0x3d8,
	0x3d8,0x3d8,0x3d8,0x3d8,0x3d8,0x3d9,0x3d9,0x3d9,
	0x3d9,0x3d9,0x3da,0x3da,0x3da,0x3da,0x3da,0x3db,
	0x3db,0x3db,0x3db,0x3db,0x3db,0x3dc,0x3dc,0x3dc,
	0x3dc,0x3dc,0x3dd,0x3dd,0x3dd,0x3dd,0x3dd,0x3dd,
	0x3de,0x3de,0x3de,0x3de,0x3de,0x3df,0x3df,0x3df,
	0x3df,0x3df,0x3df,0x3e0,0x3e0,0x3e0,0x3e0,0x3e0,
	0x3e1,0x3e1,0x3e1,0x3e1,0x3e1,0x3e2,0x3e2,0x3e2,
	0x3e2,0x3e2,0x3e2,0x3e3,0x3e3,0x3e3,0x3e3,0x3e3,
	0x3e4,0x3e4,0x3e4,0x3e4,0x3e4,0x3e4,0x3e5,0x3e5,
	0x3e5,0x3e5,0x3e5,0x3e6,0x3e6,0x3e6,0x3e6,0x3e6,
	0x3e6,0x3e7,0x3e7,0x3e7,0x3e7,0x3e7,0x3e8,0x3e8,
	0x3e8,0x3e8,0x3e8,0x3e9,0x3e9,0x3e9,0x3e9,0x3e9,
	0x3e9,0x3ea,0x3ea,0x3ea,0x3ea,0x3ea,0x3eb,0x3eb,
	0x3eb,0x3eb,0x3eb,0x3eb,0x3ec,0x3ec,0x3ec,0x3ec,
	0x3ec,0x3ed,0x3ed,0x3ed,0x3ed,0x3ed,0x3ed,0x3ee,
	0x3ee,0x3ee,0x3ee,0x3ee,0x3ef,0x3ef,0x3ef,0x3ef,
	0x3ef,0x3ef,0x3f0,0x3f0,0x3f0,0x3f0,0x3f0,0x3f1,
	0x3f1,0x3f1,0x3f1,0x3f1,0x3f1,0x3f2,0x3f2,0x3f2,
	0x3f2,0x3f2,0x3f3,0x3f3,0x3f3,0x3f3,0x3f3,0x3f3,
	0x3f4,0x3f4,0x3f4,0x3f4,0x3f4,0x3f5,0x3f5,0x3f5,
	0x3f5,0x3f5,0x3f5,0x3f6,0x3f6,0x3f6,0x3f6,0x3f6,
	0x3f7,0x3f7,0x3f7,0x3f7,0x3f7,0x3f7,0x3f8,0x3f8,
	0x3f8,0x3f8,0x3f8,0x3f9,0x3f9,0x3f9,0x3f9,0x3f9,
	0x3f9,0x3fa,0x3fa,0x3fa,0x3fa,0x3fa,0x3fb,0x3fb,
	0x3fb,0x3fb,0x3fb,0x3fb,0x3fc,0x3fc,0x3fc,0x3fc,
	0x3fc,0x3fd,0x3fd,0x3fd,0x3fd,0x3fd,0x3fd,0x3fe,
	0x3fe,0x3fe,0x3fe,0x3fe,0x3fe,0x3ff,0x3ff,0x3ff,
};

static uint8_t chartbl[512]=
{
	9,8,7,7, 6,6,6,6, 5,5,5,5, 5,5,5,5,
	4,4,4,4, 4,4,4,4, 4,4,4,4, 4,4,4,4,
	3,3,3,3, 3,3,3,3, 3,3,3,3, 3,3,3,3,
	3,3,3,3, 3,3,3,3, 3,3,3,3, 3,3,3,3,
	2,2,2,2, 2,2,2,2, 2,2,2,2, 2,2,2,2,
	2,2,2,2, 2,2,2,2, 2,2,2,2, 2,2,2,2,
	2,2,2,2, 2,2,2,2, 2,2,2,2, 2,2,2,2,
	2,2,2,2, 2,2,2,2, 2,2,2,2, 2,2,2,2,
	1,1,1,1, 1,1,1,1, 1,1,1,1, 1,1,1,1,
	1,1,1,1, 1,1,1,1, 1,1,1,1, 1,1,1,1,
	1,1,1,1, 1,1,1,1, 1,1,1,1, 1,1,1,1,
	1,1,1,1, 1,1,1,1, 1,1,1,1, 1,1,1,1,
	1,1,1,1, 1,1,1,1, 1,1,1,1, 1,1,1,1,
	1,1,1,1, 1,1,1,1, 1,1,1,1, 1,1,1,1,
	1,1,1,1, 1,1,1,1, 1,1,1,1, 1,1,1,1,
	1,1,1,1, 1,1,1,1, 1,1,1,1, 1,1,1,1,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
	0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
};

