/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#ifndef INDII_IMAGE_IMAGEMANIPULATION_HPP
#define INDII_IMAGE_IMAGEMANIPULATION_HPP

#include "Image.hpp"

#include "wx/image.h"

#include "boost/numeric/ublas/matrix.hpp"
#include "boost/numeric/ublas/matrix_sparse.hpp"

namespace indii {
/**
 * Image channel element type.
 */
typedef float channel_element;

/**
 * Dense image channel.
 */
typedef boost::numeric::ublas::matrix<channel_element> channel;

/**
 * Sparse image channel.
 */
typedef boost::numeric::ublas::mapped_matrix<channel_element> sparse_channel;

/**
 * Image mask element type.
 */
typedef bool mask_element;

/**
 * Dense image mask.
 */
typedef boost::numeric::ublas::matrix<mask_element> mask;

/**
 * Sparse image channel.
 */
typedef boost::numeric::ublas::mapped_matrix<mask_element> sparse_mask;

/**
 * Threshold value.
 *
 * @param x Value to threshold.
 * @param a Threshold point.
 * @param edge Edge softness.
 *
 * @return Thresholded value.
 */
float threshold(const float x, const float a, const float edge);

/**
 * Calculate saturation.
 *
 * @param r Red value.
 * @param g Green value.
 * @param b Blue value.
 *
 * @return Saturation.
 */
float saturation(const float r, const float g, const float b);

/**
 * Round and cast float to unsigned char.
 */
static unsigned char uround(const float val);

/**
 * Round and cast float to unsigned char.
 */
static int intround(const float val);

/**
 * Bound value above and below.
 *
 * @param lower Lower bound.
 * @param value Value.
 * @param upper Upper bound.
 */
float bound(const float lower, const float value, const float upper);

/**
 * Bound value above .
 *
 * @param value Value.
 * @param upper Upper bound.
 */
float upper_bound(const float value, const float upper);

/**
 * Bound value below.
 *
 * @param lower Lower bound.
 * @param value Value.
 */
float lower_bound(const float lower, const float value);

/**
 * Scale channel or mask.
 *
 * @tparam T Channel or mask type.
 *
 * @param src Source channel or mask.
 * @param dst Destination channel or mask.
 *
 * @p src is sampled to build @p dst.
 */
template<class T>
void scale(const T& src, T& dst);

/**
 * Hide image.
 *
 * @param image Image.
 *
 * The alpha channel across the entire image is set to transparent.
 */
void hide(wxImage& image);

/**
 * Hide image, with mask.
 *
 * @tparam M Mask type.
 *
 * @param image Image.
 * @param m Mask.
 *
 * The alpha channel of the area indicated by @p m is made transparent.
 */
template<class M>
void hide(wxImage& image, const M& m);

/**
 * Show image.
 *
 * @param image Image.
 *
 * The alpha channel across the entire image is set to opaque.
 */
void show(wxImage& image);

/**
 * Show image, with mask.
 *
 * @tparam M Mask type.
 *
 * @param image Image.
 * @param m Mask.
 *
 * The alpha channel of the area indicated by @p m is made opaque.
 */
template<class M>
void show(wxImage& image, const M& m);

/**
 * Show image, with mask.
 *
 * @tparam M Mask type.
 *
 * @param image Image.
 * @param rect Rectangle to show.
 *
 * The alpha channel of the area indicated by @p rect is made opaque.
 */
void show(wxImage& image, const wxRect& rect);

/**
 * Mask channel.
 *
 * @tparam C1 channel type.
 * @tparam C2 channel type.
 * @tparam M mask type.
 *
 * @param c The channel.
 * @param m The mask.
 * @param result The result.
 */
template<class C1, class C2, class M>
void maskChannel(const C1& c, const M& m, C2& result);

/**
 * Update alpha of image, with channel.
 *
 * @tparam C Channel type.
 *
 * @param image Image to update.
 * @param c Alpha channel.
 *
 * The alpha channel is updated according to the alpha values provided by
 * @p c.
 */
template<class C>
void applyAlpha(wxImage& image, const C& c);

/**
 * Update alpha channel of image, with channel and mask.
 *
 * @tparam M Mask type.
 * @tparam C Channel type.
 *
 * @param image Image to update.
 * @param m Mask defining update region.
 * @param c Alpha channel.
 *
 * The alpha channel of the area indicated by @p m is updated according to
 * the alpha values provided by @p c.
 */
template<class M, class C>
void applyAlpha(wxImage& image, const M& m, const C& c);

/**
 * Convert an image to a wxImage for writing or display.
 *
 * @param in Input image.
 * @param out Output image.
 */
void convert(const Image& in, wxImage& out);

/**
 * Convert a wxImage to an image.
 *
 * @param in Input image.
 * @param out Output image.
 */
void convert(const wxImage& in, Image& out);

/**
 * Return a list of file types and extensions for use with wxFileDialog.
 *
 * @param save Is this for a save dialog?
 */
wxString wildcards(const bool save = false);

}

#ifdef __WXMSW__
/* defined as macros in windows.h, which clashes with std::min and std::max,
 * so undef */
#undef min
#undef max
#endif

inline float indii::saturation(const float r, const float g, const float b) {
  float mn, mx, s;

  mn = std::min(std::min(r, g), b);
  mx = std::max(std::max(r, g), b);
  if (mx > 0.0f) {
    s = 1.0f - mn / mx;
  } else {
    s = 0.0f;
  }

  return s;
}

inline unsigned char indii::uround(const float val) {
  if (val >= 1.0f) {
    return 255;
  } else if (val <= 0.0) {
    return 0;
  } else {
    return static_cast<unsigned char>(255.0f*val);
  }
}

inline int indii::intround(const float val) {
  if (val >= 0.0f) {
    return static_cast<int>(val + 0.5f);
  } else {
    return static_cast<int>(val - 0.5f);
  }
}

inline float indii::bound(const float lower, const float value,
    const float upper) {
  if (!(value >= lower)) { // handles nan
    return lower;
  } else if (value > upper) {
    return upper;
  } else {
    return value;
  }
}

inline float indii::upper_bound(const float value,
    const float upper) {
  return (value > upper) ? upper : value;
}

inline float indii::lower_bound(const float lower,
    const float value) {
  return (value < lower) ? lower : value;
}

inline float indii::threshold(const float x, const float a, const float edge) {
  /* pre-condition */
  assert(x >= 0.0f && x <= 1.0f);
  assert(a >= 0.0f && a <= 1.0f);
  assert(edge >= 0.0 && edge <= 1.0);

  if (edge == 0.0f) {
    /* limit case */
    return (x >= a) ? 1.0f : 0.0f;
  } else {
    //return 1.0f / (1.0f + expf(-logf(255.0f)*(x - a)));
    return 1.0f / (1.0f + std::pow(255.0f, (a - x) / edge));
  }
}

template<class T>
void indii::scale(const T& src, T& dst) {
  int srcX, srcY, dstX, dstY;
  int srcWidth, srcHeight, dstWidth, dstHeight;
  double widthFactor, heightFactor;

  srcWidth = src.size2();
  srcHeight = src.size1();

  dstWidth = dst.size2();
  dstHeight = dst.size1();

  widthFactor = (double) srcWidth / dstWidth;
  heightFactor = (double) srcHeight / dstHeight;

  dst.clear();

  for (dstY = 0; dstY < dstHeight; dstY++) {
    for (dstX = 0; dstX < dstWidth; dstX++) {
      srcX = static_cast<int>(widthFactor * dstX);
      srcY = static_cast<int>(heightFactor * dstY);
      if (src(srcY, srcX) > 0.0) {
        dst(dstY, dstX) = src(srcY, srcX);
      }
    }
  }
}

template<class M>
void indii::hide(wxImage& image, const M& m) {
  /* pre-conditions */
  assert((unsigned int )image.GetWidth() == m.size2());
  assert((unsigned int )image.GetHeight() == m.size1());

  int x, y;
  typename M::const_iterator1 iter1;
  typename M::const_iterator2 iter2;

  if (!image.HasAlpha()) {
    image.InitAlpha();
  }

  for (iter1 = m.begin1(); iter1 != m.end1(); iter1++) {
    for (iter2 = iter1.begin(); iter2 != iter1.end(); iter2++) {
      x = iter2.index2();
      y = iter2.index1();
      if (m(y, x)) {
        image.SetAlpha(x, y, (unsigned char) 0);
      }
    }
  }
}

template<class M>
void indii::show(wxImage& image, const M& m) {
  /* pre-conditions */
  assert((unsigned int )image.GetWidth() == m.size2());
  assert((unsigned int )image.GetHeight() == m.size1());

  int x, y;
  typename M::const_iterator1 iter1;
  typename M::const_iterator2 iter2;

  if (!image.HasAlpha()) {
    image.InitAlpha();
  }

  for (iter1 = m.begin1(); iter1 != m.end1(); iter1++) {
    for (iter2 = iter1.begin(); iter2 != iter1.end(); iter2++) {
      x = static_cast<int>(iter2.index2());
      y = static_cast<int>(iter2.index1());
      if (m(y, x)) {
        image.SetAlpha(x, y, (unsigned char) 255);
      }
    }
  }
}

template<class C1, class C2, class M>
void indii::maskChannel(const C1& c, const M& m, C2& result) {
  /* pre-conditions */
  assert(result.size1() == c.size1());
  assert(result.size2() == c.size2());
  assert(result.size1() == m.size1());
  assert(result.size2() == m.size2());

  unsigned int x, y;
  typename M::const_iterator1 iter1;
  typename M::const_iterator2 iter2;

  for (iter1 = m.begin1(); iter1 != m.end1(); iter1++) {
    for (iter2 = iter1.begin(); iter2 != iter1.end(); iter2++) {
      x = iter2.index2();
      y = iter2.index1();
      if (m(y, x)) {
        result(y, x) = c(y, x);
      }
    }
  }
}

template<class C>
void indii::applyAlpha(wxImage& image, const C& c) {
  /* pre-conditions */
  assert((unsigned int)image.GetWidth() == c.size2());
  assert((unsigned int)image.GetHeight() == c.size1());

  int x, y;
  const int width = image.GetWidth();
  const int height = image.GetHeight();

  if (!image.HasAlpha()) {
    image.InitAlpha();
  }

  for (y = 0; y < height; y++) {
    for (x = 0; x < width; x++) {
      image.SetAlpha(x, y, uround(255.0f*c(y, x)));
    }
  }
}

template<class M, class C>
void indii::applyAlpha(wxImage& image, const M& m, const C& c) {
  /* pre-conditions */
  assert((unsigned int)image.GetWidth() == m.size2());
  assert((unsigned int)image.GetHeight() == m.size1());
  assert((unsigned int)image.GetWidth() == c.size2());
  assert((unsigned int)image.GetHeight() == c.size1());

  int x, y;
  typename M::const_iterator1 iter1;
  typename M::const_iterator2 iter2;

  if (!image.HasAlpha()) {
    image.InitAlpha();
  }

  for (iter1 = m.begin1(); iter1 != m.end1(); iter1++) {
    for (iter2 = iter1.begin(); iter2 != iter1.end(); iter2++) {
      x = iter2.index2();
      y = iter2.index1();
      if (m(y, x)) {
        image.SetAlpha(x, y, uround(255.0f*c(y, x)));
      }
    }
  }
}

#endif
