/*
 *  Copyright (C) 2008 Nicolas Vion <nico@picapo.net>
 *
 *   This file is part of swac-get.
 *
 *   Swac-get is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   Swac-get is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with swac-get; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <string>
#include <iostream>

#include <bzlib.h> 

#include "parser_dom.hh"
#include "basic.hh"
#include "global.hh"

/* Init libxml */     
Tdom_parser::Tdom_parser() {
	xmlInitParser();
	LIBXML_TEST_VERSION
}

Tdom_parser::~Tdom_parser() {
	xmlCleanupParser();
	xmlMemoryDump();
}

void Tdom_parser::clear_dom() {
	xmlFreeDoc(doc); 
}

std::string Tdom_parser::get_node_data(xmlNodePtr node) {
	if (node != NULL and (node->type == XML_ELEMENT_NODE or node->type == XML_ATTRIBUTE_NODE)) {
		xmlChar *content = xmlNodeListGetString(doc, node->xmlChildrenNode, 1);
		std::string result = (char*) content;
		xmlFree(content);
		return result;
	}
	else 
		return "";
}

int streamReadCallback(void *context, char *buffer, int len) {
	int bzerror;
	int read = BZ2_bzRead(&bzerror, (BZFILE*) context, buffer, len);
	return (bzerror == BZ_OK or bzerror == BZ_STREAM_END) ? read : -1;
}

int streamCloseCallback (void *context) {
	int bzerror;
	BZ2_bzReadClose(&bzerror, (BZFILE*) context);
	return 0;
}

bool Tdom_parser::createFromStreamBz2(FILE *f, char const *filename) {
	BZFILE* bzfile;
	int bzerror;

	bzfile = BZ2_bzReadOpen(&bzerror, f, 0, 0, NULL, 0 );
	if (bzerror != BZ_OK) {
		BZ2_bzReadClose(&bzerror, bzfile);
		return false;
	}

	doc = xmlReadIO(streamReadCallback, streamCloseCallback, bzfile, filename, NULL, 0);
	return (doc != NULL);
}

bool Tdom_parser::createFromFile(char const *filename) {
	doc = xmlReadFile(filename, NULL, 0);
	return (doc != NULL);
}

xmlNodePtr Tdom_parser::xpath_get_node(std::string expr, xmlXPathContextPtr xpath_ctxt) {
	if (xpath_ctxt == NULL)
		return NULL;

	xmlXPathObjectPtr xpathObj = xmlXPathEvalExpression(BAD_CAST expr.c_str(), xpath_ctxt);

	if(xpathObj == NULL) {
		print_error(std::string(_("Unable to evaluate xpath expression")) + " \"" + expr + "\"");
		return NULL;
	}
	else {
		xmlNodeSetPtr nodes = xpathObj->nodesetval;
		xmlNodePtr result = NULL;	
		if (nodes != NULL and nodes->nodeNr != 0)
			result = nodes->nodeTab[0];
		xmlXPathFreeObject(xpathObj);
		return result;
	}
}

std::string Tdom_parser::xpath_get_value(std::string expr, xmlXPathContextPtr xpath_ctxt) {
	xmlNodePtr node = xpath_get_node(expr, xpath_ctxt);
	if (node != NULL)
		return get_node_data(node); 
	else
		return "";
}

