/*===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/
#ifndef _h_align_writer_reference_
#define _h_align_writer_reference_

#ifndef _h_align_extern_
#include <align/extern.h>
#endif

#include <insdc/insdc.h>
#include <align/writer-cmn.h>
#include <align/writer-alignment.h>
#include <align/reader-refseq.h>

#ifdef __cplusplus
extern "C" {
#endif

enum EReference_Options {
    ewrefmgr_co_allREADs = 0x01, /* always write READ */
    ewrefmgr_co_Coverage = 0x02  /* use coverage data, by default not used */
};

typedef struct ReferenceMgr ReferenceMgr;

/* Writes REFERENCE table with in given db
    
    vmgr [IN] - needed only if conf is not NULL
    conf [IN] - optional path to tab separated file to resolve reference names
                from source files to standard refseq accessions;
                standard refseq dictionary location is obtained from kfg files.
    path {IN] - optional path points to a directory with 'local' references
                (not found in conf file), default is '.'
    max_seq_len [IN] - 0 - use default
    cache [IN] - db cache size per single!!! reference (vdb cursor cache)
    num_open [IN] - number of refseq kept open, 0 - no limit
 */
ALIGN_EXTERN rc_t CC ReferenceMgr_Make(const ReferenceMgr** cself, VDatabase* db, const VDBManager* vmgr,
                                       const uint32_t options, const char* conf, const char* path, uint32_t max_seq_len,
                                       size_t cache, uint32_t num_open);

ALIGN_EXTERN rc_t CC ReferenceMgr_Release(const ReferenceMgr* cself, bool commit, uint64_t* rows, bool build_coverage);

typedef struct ReferenceSeq ReferenceSeq;

/* id: chr12 or NC_000001.3 */
ALIGN_EXTERN rc_t CC ReferenceMgr_GetSeq(const ReferenceMgr* cself, const ReferenceSeq** seq, const char* id);

ALIGN_EXTERN rc_t CC ReferenceMgr_Verify(const ReferenceMgr* cself, const char* id, uint64_t length, const uint8_t md5[16]);

enum EReference_CompressionOptions {
    ewrefmgr_cmp_Binary = 0x01, /* binary cigar on input (BAM format: 28+4 bits) */
    ewrefmgr_cmp_NoMismatch = 0x02  /* do not fill out MISMATCH data */
};

ALIGN_EXTERN rc_t CC ReferenceMgr_Compress(const ReferenceMgr* cself, uint32_t options,
                                           const char* id, int64_t offset,
                                           const char* seq, INSDC_coord_len seq_len,
                                           const void* cigar, uint32_t cigar_len,
                                           TableWriterAlgnData* data);

/* Read refseq chunk of 'len' bases into provided 'buffer' (must be enough big for len);
   ref_len - on return has number of bases written to the buffer
 */
ALIGN_EXTERN rc_t CC ReferenceSeq_Read(const ReferenceSeq* cself, int64_t offset, INSDC_coord_len len,
                                       INSDC_dna_text* buffer, INSDC_coord_len* ref_len);

ALIGN_EXTERN rc_t CC ReferenceSeq_Get1stRow(const ReferenceSeq* cself, int64_t* row_id);

ALIGN_EXTERN rc_t CC ReferenceSeq_Compress(const ReferenceSeq* cself, uint32_t options,
                                           int64_t offset,
                                           const char* seq, INSDC_coord_len seq_len,
                                           const void* cigar, uint32_t cigar_len,
                                           TableWriterAlgnData* data);


typedef struct ReferenceSeqCoverage_struct {
    uint8_t high;
    uint8_t low;
    uint32_t mismatches;
    uint32_t indels;
    TableWriterData primary_ids;
    TableWriterData secondary_ids;
} ReferenceSeqCoverage;

ALIGN_EXTERN rc_t CC ReferenceSeq_AddCoverage(const ReferenceSeq* cself, int64_t offset, const ReferenceSeqCoverage* data);

ALIGN_EXTERN rc_t CC ReferenceSeq_Release(const ReferenceSeq* cself);

#ifdef __cplusplus
}
#endif

#endif /* _h_align_writer_reference_ */
