/*--------------------------------------------------------------------*//*:Ignore this sentence.
Copyright (C) 1999, 2001, 2005 SIL International. All rights reserved.

Distributable under the terms of either the Common Public License or the
GNU Lesser General Public License, as specified in the LICENSING.txt file.

File: XftFont.h
Responsibility: Keith Stribley
Last reviewed: Not yet.

Description:
    A Font is an object that represents a font-family + bold + italic setting, that contains
	Graphite tables.
----------------------------------------------------------------------------------------------*/
#ifndef XFTGRFONT_INCLUDED
#define XFTGRFONT_INCLUDED
#include "Font.h"
#include <X11/Xlib.h>
#include <X11/Xft/Xft.h>

#include <map>
//:End Ignore


namespace gr
{

typedef std::map<int, FT_Glyph_Metrics> GlyphMetricHash;

enum 
{
	kMaxFaceLength = 32
};

class FontTableCache;

/*-----------------------------------------------------------------------------
	Stock implementation of an Xft font class. 
------------------------------------------------------------------------------*/
/** 
* An implementation of the Graphite gr::Font class for Xft fonts for use on
* systems using X. Freetype is used to retrieve font information.
* If you are using Freetype directly, without Xft, you can pass in a FT_Face
* directly.
*/
class XftGrFont : public Font
{
public:
  /**
  * The main user constructor. Constructs the font from the supplied XftFont.
  * @param xftFont the XftFont pointer to initialise from (must not be null)
  */
  XftGrFont(XftFont * xftFont, Display * disp = NULL, Screen * scrn = NULL);
  
  /**
  * Alternative user constructor. Constructs the font from the supplied FT_Face.
  * @param ftFace the FT_Face handle to initialise from (must not be null)
  */
  XftGrFont(FT_Face ftFace, Display * disp = NULL, Screen * scrn = NULL);
  
  
  /**
  * Destructor - cleans up the tables that it has allocated if all other copies
  * of this font have been deleted.
  */
  virtual ~XftGrFont();
  
  /**
  * Returns a copy of the recipient. Specifically needed to store the Font 
  * in a segment. 
  * @internal
  */
  virtual Font * copyThis();
  
  /**
  * Copy constructor - note the tables are shared between the copy and the
  * original font for efficiency reasons. 
  * The last copy to be deleted will delete the tables.
  * @param font the XftGrFont object to copy from.
  * @internal
  */
  XftGrFont(XftGrFont & font);

  //virtual FontErrorCode isValidForGraphite(int * pnVersion = NULL, int * pnSubVersion = NULL);
  /**
  * Returns a pointer to the start of a table in the font. The ftblid32 type 
  * is a 32-bit unsigned integer.
  * If the Font class cannot easily determine the length of the table, it may 
  * set 0 as the length (while returning a non-NULL pointer to the table). This
  * means that certain kinds of error checking cannot be done by the Graphite
  * engine.
  * Throws an exception if there is some other error in reading the table, or if
  * the table asked for is not one of the expected ones (cmap, head, name, Sile,
  * Silf, Feat, Gloc, Glat). [If we want to be able to read “non-standard” 
  * tables, then the caller needs to be responsible for freeing the memory.]
  */
  virtual const void * getTable(fontTableId32 tableID, size_t * pcbSize);
  
  /**
  * Returns the basic metrics of the font. It corresponds to the current
  * GrGraphics::get_FontAscent, get_FontDescent and GetFontEmSquare methods.
  * @param pAscent pointer to hold font ascent
  * @param pDescent pointer to hold font descent 
  * @param pEmSquare pointer to hold font EM square
  */
  virtual void getFontMetrics(float * pAscent = NULL, float * pDescent = NULL,
          float * pEmSquare = NULL);
          
  /**  Converts the point number of a glyph’s on-curve point to a pair of x/y
  * coordinates in pixels. The default implementation will read the curve
  * information directly from the font and perform a simple transformation to
  * pixels. Some subclasses (e.g., WinFont) will use a system-level API call to
  * return hinted metrics.
  * Note that the coordinates returned are floating point values, using a 
  * special floating-point point class.
  * @param gid glyph id
  * @param pointNum within glyph
  * @param xyReturn reference to a Point object to hold the x,y result
  */
  virtual void getGlyphPoint(gid16 gid, unsigned int pointNum, gr::Point & xyReturn);
  
  /**
  * Returns the metrics of a glyph in the font as given by FreeType. 
  * Note that the coordinates returned are floating point values, using special 
  * floating-point rectangle and point classes.
  * @param glyphID
  * @param boundingBox reference to gr::Rect to hold bounding box of glyph
  * @param advances refererence to gr::Point to hold the horizontal / vertical
  *                 advances
  */
  virtual void getGlyphMetrics(gid16 glyphID, gr::Rect & boundingBox, gr::Point & advances);
  
  /**
  * Returns true if the given arguments specify a font that contains Graphite
  * tables—specfically an “Silf” table.
  * @param xftFont XftFont pointer
  * @return true if this font has the silf table
  */
  static bool FontHasGraphiteTables(XftFont * xftFont);
  
  /**
  * Returns true if the given arguments specify a font that contains Graphite
  * tables—specfically an “Silf” table.
  * @param face FT_Face handle
  * @return true if this font has the silf table
  */
  static bool FontHasGraphiteTables(FT_Face face);

  // Temporary, until interface gets thoroughly reworked:
  //GrEngine * GraphiteEngine();
        
  /**
  * Returns the font ascent in pixels. Using floating point allows us to 
  * avoid rounding errors. 
  * Value is the same as that returned by getFontMetrics()
  * @return the font ascent
  */
  virtual float ascent()
  {
    float pixAscent;
    getFontMetrics(&pixAscent);
    return pixAscent;
  }
  /**
  * Returns the font descent in pixels with a positive sign.
  * Value is the same as that returned by getFontMetrics()
  * @return the font descent in pixels.
  */
  virtual float descent()
  {
    float pixDescent;
    getFontMetrics(NULL, &pixDescent);
    return pixDescent;
  }
  /**
  * Returns the total height of the font. 
  * Equivalent to ascent() + descent().
  * @return font height
  */
  virtual float height()
  {
    float pixAscent;
    float pixDescent;
    getFontMetrics(&pixAscent, &pixDescent);
    return (pixAscent + pixDescent);
  }
  /**
  * @return true if the font is a bold styled one. 
  */
  virtual bool bold()
  {
    return m_fBold;
  }
  /**
  * @return true if the font is italic
  */
  virtual bool italic()
  {
    return m_fItalic;
  }
  /** @return the DPI for the x-axis.
  */
  virtual unsigned int getDPIx() 
  {
    return m_dpiX;
  }
  /** @return Returns the DPI for the y-axis.
  */
  virtual unsigned int getDPIy()
  {
    return m_dpiY;
  }
  // Specific to XftGrFont:
  /**
  * Gives the face name of the font. This may be returned by an ITextSource
  * implementation.
  * @return the face name of the font
  */
  //virtual std::wstring getFaceName(void) { return m_fpropDef.szFaceName; };
  //GrResult SetInternalFont();
	//GrResult SetInternalFont(unsigned long clrFore, unsigned long clrBack);

  /** locks the face if the XftGrFont object was constructed from an XftFont
  * object. unlockFace() must be called to release the face.
  * This is public so that SegmentPainters can access it if necessary.
  * Only 1 lock should be in place at any one time.
  * @return FT_Face a locked face handle
  */
  virtual FT_Face lockFace(void);
  
  /** Unlocks the face if it was previously locked by lockFace() */
  virtual void unlockFace(void);
  
  /** @return set the DPI for the x-axis.
  */
  virtual void setDPIx(unsigned int dpi) 
  {
    m_dpiX = dpi;
  }
  /** @return set the DPI for the y-axis.
  */
  virtual void setDPIy(unsigned int dpi)
  {
    m_dpiY = dpi;
  }
protected:
  /** Default constructor is not used */
  XftGrFont(Display * disp = NULL, Screen * scrn = NULL);
  /**
  * Common initialisation between the XftFont and FT_Face constructors.
  */
  void initializeDPI(Display * disp, Screen * scrn);
  void initializeFromFace(void);
  // Member variables:
  XftFont *m_hfont;
  FT_Face m_ftFace;

  unsigned long m_clrFore;
  unsigned long m_clrBack;
  bool m_fBold;
  bool m_fItalic;
  float m_pixHeight;
  // font table caches:
  FontTableCache * m_pTableCache;
  GlyphMetricHash m_glyphMetricHash;

  // KRS: I think these should be cached otherwise Segment::LineContextSegment doesn't work
  float m_ascent;
  float m_descent;
  float m_emSquare;
  int m_dpiX;
  int m_dpiY;
  bool m_locked;
  std::wstring m_faceName;
};


} // namespace gr

#endif


