/*
 *                            COPYRIGHT
 *
 *  sch-rnd - modular/flexible schematics editor - Tango netlist export
 *  Copyright (C) 2025 Tibor 'Igor2' Palinkas
 *  Copyright (C) 2025 Aron Barath
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 31 Milk Street, # 960789 Boston, MA 02196 USA.
 *
 *  Contact:
 *    Project page: http://repo.hu/projects/sch-rnd
 *    contact lead developer: http://www.repo.hu/projects/sch-rnd/contact.html
 *    mailing list: http://www.repo.hu/projects/sch-rnd/contact.html
 */


#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <librnd/core/compat_misc.h>
#include <librnd/core/safe_fs.h>
#include <librnd/core/plugins.h>
#include <librnd/core/error.h>
#include <libcschem/config.h>
#include <libcschem/plug_io.h>

#include <plugins/lib_netlist_exp/lib_netlist_exp.h>

static csch_plug_io_t etango_net;

static int tango_export_prio(const char *fn, const char *fmt, csch_plug_io_type_t type)
{
	if (type != CSCH_IOTYP_NETLIST)
		return 0;
	if (rnd_strcasecmp(fmt, "tango") == 0)
		return 100;
	return 0;
}

static void tango_print_value(FILE* const f, const char* value)
{
	if(value)
	{
		if(strpbrk(value, "()[]-,"))
		{
			/* Tango does not escape, let the user know the output is broken */
			rnd_message(RND_MSG_ERROR, "tango: Broken output! Value contains comma (,), dash (-), parenthesis or brackets, but it cannot be escaped: %s\n", value);
		}

		fprintf(f, "%s", value);
	}
}

/* Export abstract components; exports refdes, footprint, value and device */
static void tango_export_comps(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e;
	for(e = htsp_first(&abs->comps); e != NULL; e = htsp_next(&abs->comps, e)) {
		csch_acomp_t *comp = e->value;
		const char *refdes = sch_nle_get_refdes(comp);
		const char *fp, *dev, *val;

		if (refdes == NULL)
			continue;

		if (comp->hdr.omit)
			continue;

		/* Get main symbol attributes the safe way, considering alternate names;
		   these should be always exported and are usually hardwiared in the
		   netlist format. */
		fp = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_FOOTPRINT);
		val = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_VALUE);
		dev = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_DEVICE);

		/* The reference designator may be up to 16 alphanumeric characters long. */
		if(!memchr(refdes, 0, 16+1))
		{
			rnd_message(RND_MSG_ERROR, "tango: Broken output! Refdes is too long: %s\n", refdes);
		}

		fprintf(f, "[\r\n");
		tango_print_value(f, refdes);
		fprintf(f, "\r\n");
		tango_print_value(f, fp);
		fprintf(f, "\r\n");
		tango_print_value(f, dev ? dev : val);
		fprintf(f, "\r\n");
		tango_print_value(f, val);
		fprintf(f, "\r\n");
		fprintf(f, "\r\n\r\n]\r\n");
	}
}

/* Export abstract nets; exports net's name and a list of refdes-pinnum */
/* pairs connected */
static void tango_export_nets(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e;
	long n;

	for(e = htsp_first(&abs->nets); e != NULL; e = htsp_next(&abs->nets, e)) {
		csch_anet_t *net = e->value;
		const char *netname = sch_nle_get_netname(net);
		int net_exported = 0; /* net had at least one connection so it was written */

		if (net->hdr.omit) continue;

		for(n = 0; n < net->conns.used; n++) {
			csch_aport_t *port = net->conns.array[n];
			const char *refdes = NULL, *pinnums;

			if (port->hdr.type != CSCH_ATYPE_PORT) {
				rnd_message(RND_MSG_ERROR, "tango: invalid connection (object type)\n");
				continue;
			}

			pinnums = sch_nle_get_pinnum(port);
			if (pinnums == NULL) {
				rnd_message(RND_MSG_ERROR, "tango: can't determine port's pin number\n");
				continue;
			}

			if (port->parent != NULL) {
				refdes = sch_nle_get_refdes(port->parent);
				if (port->parent->hdr.omit)
					continue; /* omit component */
			}
			if (refdes == NULL) {
				/* This is not an error: no refdes means: do not export (e.g. gnd) */
/*				rnd_message(RND_MSG_ERROR, "tango: can't determine port's parent component refdes\n");*/
				continue;
			}

			/* split up pinnum at space and create one or more conn lines connecting
			   each pin to the given net */
			SCH_NLE_FOREACH_PINNUM(pinnums, my_num,
				{
					if(!net_exported)
					{
						net_exported = 1;
						fprintf(f, "(\r\n");
						tango_print_value(f, netname);
						fprintf(f, "\r\n");
					}

					tango_print_value(f, refdes);
					fputc(',', f);
					tango_print_value(f, my_num);
					fprintf(f, "\r\n");
				}
			);

		}

		/* If the net got exported, all the vital attributes are already printed;
		   now print misc attributes the user wanted to see in the export */
		if(net_exported)
		{
			fprintf(f, ")\r\n");
		}
	}
}


/* Export netlist from the abstract model */
static int tango_export_project_abst(const char *fn, const char *fmt, csch_abstract_t *abs, rnd_hid_attr_val_t *options)
{
	TODO("get hidlib as an arg")
	rnd_design_t *hidlib = NULL;
	FILE *f = rnd_fopen(hidlib, fn, "w");
	if (f == NULL)
		return -1;

	fprintf(f, "%-45s Revised: Thuesday, September 16, 2025\r\n", "");
	fprintf(f, "%-45s Revision: \r\n", fn);
	fprintf(f, "\r\n\r\n\r\n\r\n\r\n\r\n");

	tango_export_comps(f, abs);
	tango_export_nets(f, abs);

	fclose(f);
	return 0;
}

#include "hid_impl.c"

int pplg_check_ver_export_tango(int ver_needed) { return 0; }

void pplg_uninit_export_tango(void)
{
	csch_plug_io_unregister(&etango_net);
	rnd_export_remove_opts_by_cookie(tango_cookie);
	rnd_hid_remove_hid(&tango_hid);
}

int pplg_init_export_tango(void)
{
	RND_API_CHK_VER;

	etango_net.name = "export to Tango";
	etango_net.export_prio = tango_export_prio;
	etango_net.export_project_abst = tango_export_project_abst;
	etango_net.ext_export_project = ".net";
	csch_plug_io_register(&etango_net);


	rnd_hid_nogui_init(&tango_hid);

	tango_hid.struct_size = sizeof(rnd_hid_t);
	tango_hid.name = "tango";
	tango_hid.description = "Exports project's Tango netlist";
	tango_hid.exporter = 1;

	tango_hid.get_export_options = tango_get_export_options;
	tango_hid.do_export = tango_do_export;
	tango_hid.parse_arguments = tango_parse_arguments;
	tango_hid.argument_array = tango_values;

	tango_hid.usage = tango_usage;

	rnd_hid_register_hid(&tango_hid);
	rnd_hid_load_defaults(&tango_hid, tango_options, NUM_OPTIONS);


	return 0;
}

