/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "QtJabberSettings.h"

#include "ui_JabberSettings.h"

#include <model/connect/ConnectHandler.h>
#include <model/profile/UserProfile.h>

#include <util/Logger.h>
#include <util/SafeDelete.h>

#include <qtutil/SafeConnect.h>

#include <QtGui/QtGui>

#include <buildconfig.h>

#include <QtCore/QObject>
#include <QtGui/QPalette>
#include <QtGui/QComboBox>

#define toolTipText "user@jabber.server.com"

QtJabberSettings::QtJabberSettings(UserProfile & userProfile, IMAccount & imAccount, QDialog * parent)
	: QtIMAccountPlugin(userProfile, imAccount, parent) {

	init();
}

QtJabberSettings::~QtJabberSettings() {
	OWSAFE_DELETE(_ui);
}

void QtJabberSettings::init() {
	_IMSettingsWidget = new QWidget(_parentDialog);

	_ui = new Ui::JabberSettings();
	_ui->setupUi(_IMSettingsWidget);
	
	_ui->loginLineEdit->installEventFilter(this);
	_originalPalette = _ui->loginLineEdit->palette();
	_greyPalette.setColor(QPalette::Text, Qt::gray);
	_toolTip = tr(toolTipText);
	_toolTipShown = false;
	IMAccountParameters & params = _imAccount.getIMAccountParameters();

	QString fullLogin = QString::fromUtf8(_imAccount.getLogin().c_str());
	// Hackish way to edit the resource in a more explicit way. There really
	// should be another Jabber param in IMAccountParameters to store it.
	QString login = fullLogin.section('/', 0, 0);
	QString resource = fullLogin.section('/', 1);
	if (resource.isEmpty()) {
		resource = BINARY_NAME;
	}
	
	if (login.isEmpty())
		showToolTip();
	else
		_ui->loginLineEdit->setText(login);
	_ui->passwordLineEdit->setText(QString::fromUtf8(_imAccount.getPassword().c_str()));
	_ui->useTLSCheckBox->setChecked(params.isJabberTLSUsed());
	_ui->requireTLSCheckBox->setChecked(params.isJabberTLSRequired());
	_ui->forceOldSSLCheckBox->setChecked(params.isJabberOldSSLUsed());
	_ui->allowPlainTextAuthenticationCheckBox->setChecked(params.isJabberAuthPlainInClearUsed());
	_ui->connectServerLineEdit->setText(QString::fromUtf8(params.getJabberConnectionServer().c_str()));
	_ui->portLineEdit->setText(QString("%1").arg(params.getJabberServerPort()));
	_ui->resourceLineEdit->setText(resource);
}

bool QtJabberSettings::isValid() const {
	
	if (_toolTipShown)
		return false;
	return !_ui->loginLineEdit->text().isEmpty();
}

void QtJabberSettings::setIMAccount() {
	std::string login(_ui->loginLineEdit->text().toUtf8());
	std::string resource(_ui->resourceLineEdit->text().toUtf8());
	if (!resource.empty()) {
		login += '/' + resource;
	}
	std::string password(_ui->passwordLineEdit->text().toUtf8());

	_imAccount.setLogin(login);
	_imAccount.setPassword(password);

	IMAccountParameters & params = _imAccount.getIMAccountParameters();
	params.set(IMAccountParameters::JABBER_USE_TLS_KEY, _ui->useTLSCheckBox->isChecked());
	params.set(IMAccountParameters::JABBER_REQUIRE_TLS_KEY, _ui->requireTLSCheckBox->isChecked());
	params.set(IMAccountParameters::JABBER_USE_OLD_SSL_KEY, _ui->forceOldSSLCheckBox->isChecked());
	params.set(IMAccountParameters::JABBER_AUTH_PLAIN_IN_CLEAR_KEY, _ui->allowPlainTextAuthenticationCheckBox->isChecked());
	params.set(IMAccountParameters::JABBER_CONNECTION_SERVER_KEY, std::string(_ui->connectServerLineEdit->text().toUtf8()));
	params.set(IMAccountParameters::JABBER_PORT_KEY, _ui->portLineEdit->text().toInt());
}

bool QtJabberSettings::eventFilter(QObject *obj, QEvent *ev)
{
	if (obj == _ui->loginLineEdit)
	{
		if (ev->type() == QEvent::FocusIn)
		{
			if (_toolTipShown)
				hideToolTip();
		}
		if (ev->type() == QEvent::FocusOut)
		{
			if (_ui->loginLineEdit->text().isEmpty())
				showToolTip();
		}
	}
	return false;
}

void QtJabberSettings::showToolTip()
{
	_ui->loginLineEdit->setPalette(_greyPalette);
	_ui->loginLineEdit->setText(_toolTip);
	_toolTipShown = true;
}

void QtJabberSettings::hideToolTip()
{
	_ui->loginLineEdit->setPalette(_originalPalette);
	_ui->loginLineEdit->clear();
	_toolTipShown = false;
}
