/*
 * Copyright (c) 2001, 2018, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package nsk.jdi.Event._itself_;

import com.sun.jdi.Field;
import com.sun.jdi.ReferenceType;
import com.sun.jdi.VirtualMachine;
import com.sun.jdi.request.AccessWatchpointRequest;
import com.sun.jdi.request.EventRequestManager;
import com.sun.jdi.VMDisconnectedException;
import com.sun.jdi.VMMismatchException;
import com.sun.jdi.event.*;

import com.sun.jdi.request.EventRequest;
import java.util.Iterator;
import java.util.ArrayList;
import java.util.List;
import java.io.*;

import nsk.share.*;
import nsk.share.jpda.*;
import nsk.share.jdi.*;

/**
 * The test checks on the JDI interface
 * <b>com.sun.jdi.event.Event.</b>
 * A debugger part
 * of the test creates several <code>AccessWatchpointRequest</code>s, and
 * then it verifies that appropriate events will be generated by a target
 * VM. After that, the debugger removes the list of the event requests via
 * <code>deleteEventRequests()</code> call. Finally, the test checks
 * that the events which corresponding requests have been deleted, are
 * not generated by the target VM.
 */
public class event002 {
    public static final int PASSED = 0;
    public static final int FAILED = 2;
    public static final int JCK_STATUS_BASE = 95;
    static final String DEBUGGEE_CLASS =
        "nsk.jdi.Event._itself_.event002t";
    static final String COMMAND_READY = "ready";
    static final String COMMAND_QUIT = "quit";

    static final int FLDS_NUM = 9; // number of debuggee's fields
    static final int DEL_REQS = 5; // number of requests to delete
    static final String COMMAND_RUN[] = {
        "run1", "run2", "run3", "run4",
        "run5", "run6", "run7", "run8", "run9"
    };
    static final String DEBUGGEE_FLDS[][] = {
        {"byte", "byteFld", "DELETE"},
        {"short", "shortFld", "NO"},
        {"int", "intFld", "DELETE"},
        {"long", "longFld", "NO"},
        {"float", "floatFld", "DELETE"},
        {"double", "doubleFld", "DELETE"},
        {"char", "charFld", "NO"},
        {"boolean", "booleanFld", "NO"},
        {"java.lang.String", "strFld", "DELETE"}
    };
    private volatile int accFCount[] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0
    };

    private ArgumentHandler argHandler;
    private Log log;
    private IOPipe pipe;
    private Debugee debuggee;
    private VirtualMachine vm;
    private EventListener elThread;
    private volatile AccessWatchpointRequest awpRequest[];
    private List<EventRequest> delRequests; // list of requests to delete
    private volatile int tot_res = PASSED;

// for notification a main thread about received events
    private Object gotEvent = new Object();

    public static void main (String argv[]) {
        System.exit(run(argv,System.out) + JCK_STATUS_BASE);
    }

    public static int run(String argv[], PrintStream out) {
        return new event002().runIt(argv, out);
    }

    private int runIt(String args[], PrintStream out) {
        argHandler = new ArgumentHandler(args);
        log = new Log(out, argHandler);
        Binder binder = new Binder(argHandler, log);
        ReferenceType rType;
        List fields;
        String cmd;

        debuggee = binder.bindToDebugee(DEBUGGEE_CLASS);
        pipe = debuggee.createIOPipe();
        debuggee.redirectStderr(log, "event002t.err> ");
        vm = debuggee.VM();
        EventRequestManager erManager = vm.eventRequestManager();
        debuggee.resume();
        cmd = pipe.readln();
        if (!cmd.equals(COMMAND_READY)) {
            log.complain("TEST BUG: unknown debuggee's command: "
                + cmd);
            tot_res = FAILED;
            return quitDebuggee();
        }

        if ( !vm.canWatchFieldAccess() ) {
            log.display("  TEST CANCELLED due to:  vm.canWatchFieldAccess() == false");
            return quitDebuggee();
        }

// Create testing requests
        delRequests = new ArrayList<EventRequest>();
        if ((rType = debuggee.classByName(DEBUGGEE_CLASS)) == null) {
            log.complain("TEST FAILURE: Method Debugee.classByName() returned null");
            tot_res = FAILED;
            return quitDebuggee();
        }
        try {
            fields = rType.allFields();
        } catch (Exception e) {
            log.complain("TEST FAILURE: allFields: caught " + e);
            tot_res = FAILED;
            return quitDebuggee();
        }
        if (createRequests(erManager, fields) == FAILED) {
            tot_res = FAILED;
            return quitDebuggee();
        }

// Create a thread listening JDI events
        elThread = new EventListener();
        elThread.setPriority(Thread.NORM_PRIORITY + 2);
        synchronized(gotEvent) {
            elThread.start();

// Check all requested AccessWatchpointRequest events
            for (int i=0; i<FLDS_NUM; i++) {
                awpRequest[i].enable();
            }
            log.display("\na) Getting all requested AccessWatchpointEvents...");
            if (checkEvents(null) != PASSED)
                return FAILED;

            log.display("\nb) Getting AccessWatchpointEvents without "
                + DEL_REQS + " deleted requests...");
// Remove a list of AccessWatchpointRequests
            if (removeRequests(erManager, delRequests) == FAILED) {
                tot_res = FAILED;
                return quitDebuggee();
            }

// Check remained AccessWatchpointRequests
            checkEvents(delRequests);

// Finish the test
            for (int i=0; i<FLDS_NUM; i++) {
/* NOTE: for the jdk1.4: an InvalidRequestStateException could be
         checked for deleted request, but for the jdk1.3.x
         the InvalidRequestStateException is unspecified */
                if (DEBUGGEE_FLDS[i][2].equals("NO"))
                    awpRequest[i].disable();
            }

            quitDebuggee();

            for (int i=0; i<FLDS_NUM; i++) {
                if (accFCount[i] > 0 && DEBUGGEE_FLDS[i][2].equals("DELETE")) {
                    log.complain("TEST FAILED: got AccessWatchpointEvent for disabled field "
                        + DEBUGGEE_FLDS[i][0]);
                    return FAILED;
                }
            }
            log.display("TEST PASSED: Got no AccessWatchpointEvents for disabled fields");

        }
        return tot_res;
    }

    private int createRequests(EventRequestManager erManager,
            List fields) {
        Field fld = null;
        int i = 0;

        awpRequest =
            new AccessWatchpointRequest[FLDS_NUM];
        for (i=0; i<FLDS_NUM; i++) {
            boolean notFound = true;
            Iterator iter = fields.iterator();
            while (iter.hasNext()) {
                fld = (Field) iter.next();
                if (fld.name().equals(DEBUGGEE_FLDS[i][1]) &&
                    fld.typeName().equals(DEBUGGEE_FLDS[i][0])) {
                    log.display("\nCreating AccessWatchpointRequest #"
                        + i + " for the debuggee's field:\n\t"
                        + fld.typeName() + " " + fld.name());
                    try {
                        awpRequest[i] =
                            erManager.createAccessWatchpointRequest(fld);
                    } catch (Exception e) {
                        log.complain("TEST FAILED: createAccessWatchpointRequest: caught "
                            + e);
                        return FAILED;
                    }
                    if (DEBUGGEE_FLDS[i][2].equals("DELETE")) {
                        log.display("Adding this AccessWatchpointRequest to the list of requests to delete");
                        delRequests.add(awpRequest[i]);
                    }
                    notFound = false;
                    break;
                }
            }
            if (notFound) {
                log.complain("TEST FAILED: found unexpected debuggee's field:\n\t"
                    + fld.typeName() + " " + fld.name());
                return FAILED;
            }
        }
        return PASSED;
    }

    private int removeRequests(EventRequestManager erManager,
            List delReqs) {
        log.display("\nRemoving the following list of AccessWatchpointRequests:");
        Iterator iter = delReqs.iterator();
        while (iter.hasNext()) {
            AccessWatchpointRequest awpReq =
                (AccessWatchpointRequest) iter.next();
            Field fld = awpReq.field();
            log.display("\tAccessWatchpointRequest for the debuggee's field: "
                + fld.typeName() + " " + fld.name());
        }
        try {
            erManager.deleteEventRequests(delRequests);
        } catch (VMMismatchException e) {
            log.complain("TEST FAILED: EventRequestManager.deleteEventRequests: caught"
                + e);
            e.printStackTrace();
            return FAILED;
        }
        return PASSED;
    }

    private int runTestCase(int i, boolean waitForEvent) {
        String token = null;

        log.display("\n" + (i+1) + ") Sending the command \""
            + COMMAND_RUN[i] + "\" to a debuggee");
        pipe.println(COMMAND_RUN[i]);

// wait for a requested event
        if (waitForEvent) {
            try {
                gotEvent.wait(argHandler.getWaitTime() * 60 * 1000);
            } catch (InterruptedException e) {
                log.complain("TEST FAILURE: waiting for a requested AccessWatchpointEvent #"
                    + i + ": caught " + e);
                e.printStackTrace();
                tot_res = FAILED;
                return quitDebuggee();
            }
            log.display("Notification about the AccessWatchpointEvent #"
                + i + " received,\n\tor time has elapsed");
        }

        if ((token = pipe.readln()) == null) {
            log.complain("TEST FAILURE: debuggee's reply is empty, probably due to the VM crash");
            tot_res = FAILED;
            return quitDebuggee();
        }
        if (!token.equals(COMMAND_READY)) {
            log.complain("TEST BUG: unknown debuggee's command: " + token);
            tot_res = FAILED;
            return quitDebuggee();
        }
        else log.display("debuggee's reply received: "
            + token);
        return PASSED;
    }

    private int checkEvents(List delReqs) {
        for (int i=0; i<FLDS_NUM; i++) {
            accFCount[i] = 0;
            boolean waitForEvent = true;
            if (delReqs != null && DEBUGGEE_FLDS[i][2].equals("DELETE")) {
                waitForEvent = false;
            }
            if (runTestCase(i, waitForEvent) == FAILED) {
                return FAILED;
            }
            if (delReqs != null) {
                if (DEBUGGEE_FLDS[i][2].equals("DELETE")) {
                    if (accFCount[i] != 0) {
                        log.complain("TEST FAILED: got AccessWatchpointEvent for the following field:\n\t"
                            + DEBUGGEE_FLDS[i][0] + " " + DEBUGGEE_FLDS[i][1]
                            + "\n\tbut corresponding request has been deleted via EventRequestManager.deleteEventRequests()");
                        tot_res = FAILED;
                    } else
                        log.display("\nTEST PASSED: no event for the deleted AccessWatchpointRequest #"
                            + i);
                }
            } else {
                log.display("Got expected AccessWatchpointEvent for the following debuggee's field:\n\t"
                    + DEBUGGEE_FLDS[i][0] + " " + DEBUGGEE_FLDS[i][1]);
            }
        }
        return PASSED;
    }

    private int quitDebuggee() {
        if (elThread != null) {
            elThread.isConnected = false;
            try {
                if (elThread.isAlive())
                    elThread.join();
            } catch (InterruptedException e) {
                log.complain("TEST INCOMPLETE: caught InterruptedException "
                    + e);
                tot_res = FAILED;
            }
        }

        pipe.println(COMMAND_QUIT);
        debuggee.waitFor();
        int debStat = debuggee.getStatus();
        if (debStat != (JCK_STATUS_BASE + PASSED)) {
            log.complain("TEST FAILED: Debuggee's process finished with status: "
                + debStat);
            tot_res = FAILED;
        } else
            log.display("Debuggee's process finished with status: "
                + debStat);

        return tot_res;
    }

    class EventListener extends Thread {
        public volatile boolean isConnected = true;

        public void run() {
            try {
                do {
                    EventSet eventSet = vm.eventQueue().remove(10);
                    if (eventSet != null) { // there is not a timeout
                        EventIterator it = eventSet.eventIterator();
                        while (it.hasNext()) {
                            Event event = it.nextEvent();
                            if (event instanceof VMDeathEvent) {
                                tot_res = FAILED;
                                isConnected = false;
                                log.complain("TEST FAILED: caught unexpected VMDeathEvent");
                            } else if (event instanceof VMDisconnectEvent) {
                                tot_res = FAILED;
                                isConnected = false;
                                log.complain("TEST FAILED: caught unexpected VMDisconnectEvent");
                            } else {
                                log.display("EventListener: following JDI event occured: "
                                    + event.toString());

                                if (event instanceof AccessWatchpointEvent) {
                                    AccessWatchpointEvent awpEvent =
                                        (AccessWatchpointEvent) event;
                                    Field fld = awpEvent.field();
                                    boolean notFound = true;
                                    for (int i=0; i<FLDS_NUM; i++) {
                                        if (awpRequest[i].equals(event.request())) {
                                            log.display("EventListener: AccessWatchpointEvent for the debuggee's field #"
                                                + i + ":\n\t" + fld.typeName()
                                                + " " + fld.name());
                                            accFCount[i] += 1;
                                            notFound = false;
                                            log.display("EventListener: notifying about the received event #"
                                                + i);
                                            synchronized(gotEvent) {
                                                gotEvent.notify(); // notify the main thread
                                            }
                                            break;
                                        }
                                    }
                                    if (notFound) {
                                        log.complain("TEST FAILED: found in the received AccessWatchpointEvent\n\tunexpected debuggee's field "
                                            + fld.typeName() + " " + fld.name());
                                        tot_res = FAILED;
                                    }
                                }

                            }
                        }
                        if (isConnected) {
                            eventSet.resume();
                        }
                    }
                } while (isConnected);
            } catch (InterruptedException e) {
                tot_res = FAILED;
                log.complain("FAILURE in EventListener: caught unexpected "
                    + e);
            } catch (VMDisconnectedException e) {
                tot_res = FAILED;
                log.complain("FAILURE in EventListener: caught unexpected "
                    + e);
                e.printStackTrace();
            }
            log.display("EventListener: exiting");
        }
    }
}
