//  ---------------------------------------------------------------------------
//
//  @file       TwFonts.cpp
//  @author     Philippe Decaudin - http://www.antisphere.com
//  @license    This file is part of the AntTweakBar library.
//              For conditions of distribution and use, see License.txt
//
//  ---------------------------------------------------------------------------


#include "TwPrecomp.h"
#include "TwMgr.h"
#include "TwFonts.h"

// Fedora patch: memset()
using std::memset;

//  ---------------------------------------------------------------------------

CTexFont::CTexFont()
{
    for( int i=0; i<256; ++i )
    {
        m_CharU0[i] = 0;
        m_CharU1[i] = 0;
        m_CharV0[i] = 0;
        m_CharV1[i] = 0;
        m_CharWidth[i] = 0;
    }
    m_TexWidth = 0;
    m_TexHeight = 0;
    m_TexBytes = NULL;
    m_NbCharRead = 0;
    m_CharHeight = 0;
}

//  ---------------------------------------------------------------------------

CTexFont::~CTexFont()
{
    if( m_TexBytes )
        delete[] m_TexBytes;
    m_TexBytes = NULL;
    m_TexWidth = 0;
    m_TexHeight = 0;
    m_NbCharRead = 0;
}

//  ---------------------------------------------------------------------------

static int NextPow2(int _n)
{
    int r = 1;
    while( r<_n )
        r *= 2;
    return r;
}

//  ---------------------------------------------------------------------------

const char *g_ErrBadFontHeight = "Cannot determine font height while reading font bitmap (check first pixel column)";

CTexFont *TwGenerateFont(const unsigned char *_Bitmap, int _BmWidth, int _BmHeight)
{
    // find height of the font
    int x, y;
    int h = 0, hh = 0;
    int r, NbRow = 0;
    for( y=0; y<_BmHeight; ++y )
        if( _Bitmap[y*_BmWidth]==0 )
        {
            if( (hh<=0 && h<=0) || (h!=hh && h>0 && hh>0) )
            {
                g_TwMgr->SetLastError(g_ErrBadFontHeight);
                return NULL;
            }
            else if( h<=0 )
                h = hh;
            else if( hh<=0 )
                break;
            hh = 0;
            ++NbRow;
        }
        else
            ++hh;

    // find width and position of each character
    int w = 0;
    int x0[224], y0[224], x1[224], y1[224];
    int ch = 32;
    int start;
    for( r=0; r<NbRow; ++r )
    {
        start = 1;
        for( x=1; x<_BmWidth; ++x )
            if( _Bitmap[(r*(h+1)+h)*_BmWidth+x]==0 || x==_BmWidth-1 )
            {
                if( x==start )
                    break;  // next row
                if( ch<256 )
                {
                    x0[ch-32] = start;
                    x1[ch-32] = x;
                    y0[ch-32] = r*(h+1);
                    y1[ch-32] = r*(h+1)+h-1;
                    w += x-start+1;
                    start = x+1;
                }
                ++ch;
            }
    }
    for( x=ch-32; x<224; ++x )
    {
        x0[ch] = 0;
        x1[ch] = 0;
        y0[ch] = 0;
        y1[ch] = 0;
    }

    // Repack: build 14 rows of 16 characters.
    // - First, find the largest row
    int l, lmax = 1;
    for( r=0; r<14; ++r )
    {
        l = 0;
        for( x=0; x<16; ++x )
            l += x1[x+r*16]-x0[x+r*16]+1;
        if( l>lmax )
            lmax = l;
    }
    // A little empty margin is added between chars to avoid artefact when antialiasing is on
    const int MARGIN_X = 2; 
    const int MARGIN_Y = 2;
    lmax += 16*MARGIN_X;
    // - Second, build the texture
    CTexFont *TexFont = new CTexFont;
    TexFont->m_NbCharRead = ch-32;
    TexFont->m_CharHeight = h;
    TexFont->m_TexWidth = NextPow2(lmax);
    TexFont->m_TexHeight = NextPow2(14*(h+MARGIN_Y));
    TexFont->m_TexBytes = new unsigned char[TexFont->m_TexWidth*TexFont->m_TexHeight];
    memset(TexFont->m_TexBytes, 0, TexFont->m_TexWidth*TexFont->m_TexHeight);
    int xx;
    float du = 0.4f;
    float dv = 0.4f;
    assert( g_TwMgr!=NULL );
    if( g_TwMgr )
    {
        if( g_TwMgr->m_GraphAPI==TW_OPENGL || g_TwMgr->m_GraphAPI==TW_OPENGL_CORE )
        {
            du = 0;
            dv = 0;
        }
        else    // texel alignement for D3D
        {
            du = 0.5f;
            dv = 0.5f;
        }
    }
    float alpha;
    for( r=0; r<14; ++r )
        for( xx=0, ch=r*16; ch<(r+1)*16; ++ch )
            if( y1[ch]-y0[ch]==h-1 )
            {
                for( y=0; y<h; ++y )
                    for( x=x0[ch]; x<=x1[ch]; ++x )
                    {
                        alpha = ((float)(_Bitmap[x+(y0[ch]+y)*_BmWidth]))/256.0f;
                        //alpha = alpha*sqrtf(alpha); // powf(alpha, 1.5f);   // some gamma correction
                        TexFont->m_TexBytes[(xx+x-x0[ch])+(r*(h+MARGIN_Y)+y)*TexFont->m_TexWidth] = (unsigned char)(alpha*256.0f);
                    }
                TexFont->m_CharU0[ch+32] = (float(xx)+du)/float(TexFont->m_TexWidth);
                xx += x1[ch]-x0[ch]+1;
                TexFont->m_CharU1[ch+32] = (float(xx)+du)/float(TexFont->m_TexWidth);
                TexFont->m_CharV0[ch+32] = (float(r*(h+MARGIN_Y))+dv)/float(TexFont->m_TexHeight);
                TexFont->m_CharV1[ch+32] = (float(r*(h+MARGIN_Y)+h)+dv)/float(TexFont->m_TexHeight);
                TexFont->m_CharWidth[ch+32] = x1[ch]-x0[ch]+1;
                xx += MARGIN_X;
            }

    const unsigned char Undef = 127; // default character used as for undifined ones (having ascii codes from 0 to 31)
    for( ch=0; ch<32; ++ch )
    {
        TexFont->m_CharU0[ch] = TexFont->m_CharU0[Undef];
        TexFont->m_CharU1[ch] = TexFont->m_CharU1[Undef];
        TexFont->m_CharV0[ch] = TexFont->m_CharV0[Undef];
        TexFont->m_CharV1[ch] = TexFont->m_CharV1[Undef];
        TexFont->m_CharWidth[ch] = TexFont->m_CharWidth[Undef]/2;
    }

    return TexFont;
}

//  ---------------------------------------------------------------------------

CTexFont *g_DefaultSmallFont = NULL;
CTexFont *g_DefaultNormalFont = NULL;
CTexFont *g_DefaultLargeFont = NULL;

// Small font
const int FONT0_BM_W = 211;
const int FONT0_BM_H = 84;
static const unsigned char s_Font0[] = 
{
    127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,
    0,0,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,255,0,0,0,255,255,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,
    0,0,0,0,255,0,255,0,255,0,0,0,255,0,0,255,0,0,0,255,0,0,0,0,255,255,0,0,
    0,255,0,0,0,0,255,255,0,0,0,255,0,0,255,0,0,255,0,0,255,0,255,0,255,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,255,0,0,0,255,0,0,255,255,
    255,0,0,255,255,255,0,0,0,0,0,255,0,255,255,255,255,0,0,255,255,0,0,255,
    255,255,255,0,0,255,255,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,127,0,0,0,0,255,0,255,0,255,0,0,0,255,0,0,255,0,0,255,255,255,255,
    0,255,0,0,255,0,255,0,0,0,0,255,0,0,255,0,0,255,0,255,0,0,0,0,255,0,0,255,
    255,255,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,255,0,255,
    255,0,0,0,0,0,255,0,0,0,0,255,0,0,0,255,255,0,255,0,0,0,0,255,0,0,0,0,0,
    0,0,255,0,255,0,0,255,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,
    0,0,0,0,255,0,0,0,0,0,0,255,255,255,255,255,255,255,0,255,0,0,0,255,0,0,
    255,0,255,0,0,0,0,255,0,0,255,0,0,0,0,255,0,0,0,0,255,0,255,0,255,0,255,
    0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,255,0,0,255,0,0,0,
    0,0,255,0,0,0,0,255,0,0,255,0,255,0,255,0,0,0,0,255,0,0,0,0,0,0,255,0,0,
    255,0,0,255,0,255,0,0,255,0,0,255,0,0,255,0,0,0,0,0,255,255,0,0,0,0,0,0,
    0,0,0,255,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    127,0,0,0,0,255,0,0,0,0,0,0,0,255,0,255,0,0,0,255,255,0,0,0,0,255,255,0,
    255,0,255,255,0,0,0,255,255,0,255,0,0,0,255,0,0,0,0,255,0,0,0,255,0,0,0,
    0,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,255,0,0,255,
    0,0,0,0,255,0,0,0,255,255,0,0,255,0,0,255,0,255,255,255,0,0,255,255,255,
    0,0,0,0,255,0,0,0,255,255,0,0,0,255,255,255,0,0,255,0,0,255,0,0,0,255,255,
    0,0,0,255,255,255,255,255,255,0,0,0,0,255,255,0,0,0,0,255,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,255,0,0,0,0,0,255,255,255,255,255,
    255,0,0,0,255,255,0,0,0,0,0,255,0,255,0,0,255,0,255,0,0,255,0,0,0,0,255,
    0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,255,0,0,0,0,0,255,0,
    0,255,0,0,255,0,0,255,0,0,0,255,0,0,0,0,0,0,255,0,255,255,255,255,255,0,
    0,0,255,0,255,0,0,255,0,0,255,0,0,0,255,0,0,255,0,0,0,0,255,0,0,0,0,0,0,
    0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,
    255,0,255,0,0,0,0,255,0,255,0,0,255,0,255,0,0,255,255,0,0,0,255,0,0,0,0,
    255,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,255,0,0,255,0,0,255,0,
    0,255,0,0,255,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,255,0,255,0,
    0,255,0,0,255,0,0,0,255,0,0,255,0,0,0,0,255,0,0,255,0,0,255,0,0,0,255,255,
    0,0,0,255,255,255,255,255,255,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,255,0,0,0,0,0,0,255,0,0,255,0,0,255,
    255,255,255,0,0,0,0,255,0,0,0,255,255,0,0,0,255,255,0,0,255,0,0,255,0,0,
    0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,255,0,0,0,0,255,
    255,0,0,255,255,255,0,255,255,255,255,0,255,255,255,0,0,0,0,0,255,0,255,
    255,255,0,0,0,255,255,0,0,255,0,0,0,0,0,255,255,0,0,0,255,255,0,0,0,255,
    0,0,255,0,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,255,255,0,0,0,0,0,255,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,127,127,0,127,0,127,127,127,
    0,127,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,127,
    127,127,127,0,127,127,127,127,127,0,127,0,127,127,0,127,127,127,0,127,127,
    127,127,127,0,127,127,127,127,127,127,0,127,127,0,127,127,127,0,127,0,127,
    127,127,0,127,127,127,127,0,127,127,127,0,127,127,127,127,0,127,127,127,
    127,0,127,127,127,127,0,127,127,127,127,0,127,127,127,127,0,127,127,127,
    127,0,127,127,127,127,0,127,127,127,127,0,127,127,0,127,127,0,127,127,127,
    127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,
    127,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,0,255,0,0,0,255,255,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,255,255,255,0,0,0,0,0,
    255,255,0,0,0,255,255,255,0,0,0,0,0,255,255,255,0,0,255,255,255,255,0,0,
    0,255,255,255,255,255,0,255,255,255,255,255,0,0,0,255,255,255,0,0,255,0,
    0,0,0,255,0,255,255,255,0,0,255,255,0,255,0,0,0,255,0,255,0,0,0,255,255,
    0,0,0,255,255,0,255,0,0,0,0,255,0,0,0,255,255,255,0,0,0,255,255,255,255,
    0,0,0,0,255,255,255,0,0,0,255,255,255,255,0,0,0,255,255,255,255,0,255,255,
    255,255,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,255,0,0,255,0,
    255,0,0,255,0,255,0,0,0,255,0,255,255,255,255,0,255,0,0,255,0,0,0,0,255,
    0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,255,0,
    0,0,255,0,0,0,0,255,255,0,0,0,255,0,0,255,0,0,0,255,0,0,0,255,0,255,0,0,
    0,255,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,255,0,255,0,0,0,0,255,
    0,0,255,0,0,0,0,255,0,255,0,0,255,0,0,255,0,0,0,255,255,0,0,0,255,255,0,
    255,255,0,0,0,255,0,0,255,0,0,0,255,0,0,255,0,0,0,255,0,0,255,0,0,0,255,
    0,0,255,0,0,0,255,0,255,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,255,0,
    0,0,0,255,0,255,0,0,255,0,0,255,0,255,0,0,255,0,0,255,0,255,0,0,0,0,0,255,
    0,255,0,0,0,255,0,0,0,255,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,127,255,0,0,255,255,0,255,0,0,255,0,0,255,0,0,255,0,0,255,
    0,0,255,0,0,0,0,0,0,255,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,
    0,0,0,0,0,255,0,0,0,0,255,0,0,255,0,0,0,0,255,0,255,0,255,0,0,0,255,0,0,
    0,255,0,255,0,255,0,255,0,255,0,255,0,0,255,0,255,0,0,0,0,0,255,0,255,0,
    0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,255,0,255,0,0,0,0,0,0,0,255,0,0,
    0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,255,0,255,0,255,0,0,255,255,
    0,0,0,255,0,255,0,0,0,0,255,0,0,255,0,0,0,255,0,0,0,255,0,255,0,0,0,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,255,0,255,0,255,0,255,
    0,0,255,0,0,255,0,0,255,255,255,255,0,0,255,0,0,0,0,0,0,255,0,0,0,0,255,
    0,255,255,255,255,0,0,255,255,255,255,0,0,255,0,0,255,255,255,0,255,255,
    255,255,255,255,0,0,255,0,0,0,0,255,0,255,255,0,0,0,0,255,0,0,0,255,0,255,
    0,255,0,255,0,255,0,0,255,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,255,0,255,
    0,0,0,0,0,255,0,255,255,255,255,0,0,0,255,255,255,0,0,0,0,255,0,0,0,255,
    0,0,0,0,255,0,0,255,0,0,255,0,0,255,0,255,0,255,0,255,0,0,255,255,0,0,0,
    0,255,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,255,0,255,0,255,0,255,0,255,255,255,
    255,255,255,0,255,0,0,0,255,0,255,0,0,0,0,0,0,255,0,0,0,0,255,0,255,0,0,
    0,0,0,255,0,0,0,0,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,0,255,0,0,0,0,255,
    0,255,0,255,0,0,0,255,0,0,0,255,0,0,255,0,0,255,0,255,0,0,0,255,255,0,255,
    0,0,0,0,0,255,0,255,255,255,255,0,0,255,0,0,0,0,0,255,0,255,0,255,0,0,0,
    0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,255,0,0,255,0,0,255,0,255,
    0,255,0,255,0,0,255,255,0,0,0,0,255,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,
    0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,255,
    0,0,255,255,255,0,0,255,0,0,0,0,255,0,255,0,0,0,255,0,0,255,0,0,0,255,0,
    255,0,0,0,255,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,255,0,255,0,0,
    0,0,255,0,0,255,0,0,0,0,255,0,255,0,0,255,0,0,255,0,0,0,255,0,0,255,0,0,
    255,0,255,0,0,0,0,255,0,0,255,0,0,0,255,0,0,255,0,0,0,0,0,0,255,0,0,0,255,
    0,0,255,0,0,255,0,0,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,0,255,
    255,0,0,0,0,255,0,0,0,255,0,0,255,0,0,255,0,0,0,255,0,0,0,255,0,0,0,0,255,
    0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,127,0,255,0,0,0,0,0,0,255,0,0,0,0,255,0,255,255,255,255,0,0,0,0,255,
    255,255,0,0,255,255,255,255,0,0,0,255,255,255,255,255,0,255,0,0,0,0,0,0,
    0,255,255,255,255,0,255,0,0,0,0,255,0,255,255,255,0,255,255,0,0,255,0,0,
    0,255,0,255,255,255,255,255,0,0,0,0,0,255,0,255,0,0,0,0,255,0,0,0,255,255,
    255,0,0,0,255,0,0,0,0,0,0,0,255,255,255,0,0,0,255,0,0,0,255,0,255,255,255,
    255,0,0,0,0,255,0,0,0,0,255,255,255,255,0,0,0,0,255,255,0,0,0,0,255,0,0,
    0,255,0,0,255,0,0,255,0,0,0,255,0,0,0,255,255,255,255,0,255,0,0,0,0,255,
    0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,
    0,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,
    0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,
    0,0,0,0,0,255,255,0,0,0,0,0,0,0,255,255,255,255,255,255,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,
    0,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,
    0,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,127,0,
    127,127,127,127,127,127,0,127,127,127,0,127,127,127,0,127,127,127,127,127,
    0,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,0,
    127,127,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,
    127,127,0,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,
    0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,
    127,127,0,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,0,127,
    127,0,127,127,127,0,127,127,0,127,127,127,127,127,0,127,127,127,127,127,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,127,255,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,
    0,0,255,0,0,0,0,0,0,0,255,255,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,255,0,0,0,
    0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,255,0,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,255,0,0,0,0,0,0,0,255,0,0,
    0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,
    255,0,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,4,4,
    4,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,
    0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,
    0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,0,0,0,
    0,0,0,0,4,4,4,4,12,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    127,0,0,0,0,0,255,255,0,0,255,255,255,0,0,0,255,255,0,0,255,255,255,0,0,
    255,255,0,0,255,255,255,0,255,255,255,0,255,255,255,0,0,255,0,255,255,0,
    255,0,0,255,0,255,0,255,255,255,0,255,255,0,0,255,255,255,0,0,0,255,255,
    0,0,255,255,255,0,0,0,255,255,255,0,255,0,255,255,255,255,0,255,255,0,255,
    0,0,255,0,255,0,0,0,255,0,255,0,0,255,0,0,255,0,255,0,255,0,255,0,0,0,255,
    0,255,255,255,0,0,255,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,4,4,4,4,0,
    255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,
    255,0,255,0,0,255,0,255,0,0,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,
    0,255,0,255,0,0,255,0,255,0,0,255,0,255,0,255,0,0,255,0,255,0,0,255,0,0,
    255,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,255,0,
    255,0,0,0,255,0,0,255,0,0,255,0,0,255,0,255,0,0,255,0,0,255,0,0,255,0,0,
    255,0,0,0,255,0,255,0,0,0,0,255,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,255,
    0,0,255,0,0,255,4,4,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,127,0,0,0,0,0,255,255,255,0,255,0,0,255,0,255,0,0,0,255,0,0,255,0,
    255,255,255,255,0,255,0,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,
    255,0,0,0,255,0,255,0,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,0,
    0,255,0,255,0,0,255,0,255,0,0,0,255,0,0,255,0,0,255,0,0,255,0,0,255,0,255,
    0,0,255,0,255,0,255,0,255,0,0,255,0,0,0,255,0,255,0,0,0,255,0,0,255,0,0,
    0,0,255,0,0,0,0,255,0,255,0,0,255,255,0,0,0,255,255,4,255,255,0,4,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,255,0,0,255,0,255,0,
    0,255,0,255,0,0,0,255,0,0,255,0,255,0,0,0,0,255,0,0,255,0,0,255,0,255,0,
    0,255,0,255,0,0,255,0,255,0,255,0,0,255,0,255,0,0,255,0,0,255,0,255,0,0,
    255,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,0,0,0,0,255,0,255,0,
    0,255,0,0,255,0,0,255,0,255,0,0,0,255,255,0,255,255,0,0,0,255,0,0,0,255,
    0,255,0,0,255,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,255,
    255,255,0,4,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,
    0,255,255,255,0,255,255,255,0,0,0,255,255,0,0,255,255,255,0,0,255,255,255,
    0,255,0,0,0,255,255,255,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,
    0,255,0,0,255,0,0,255,0,255,0,0,255,0,0,255,255,0,0,255,255,255,0,0,0,255,
    255,255,0,255,0,0,255,255,255,0,0,255,0,0,255,255,255,0,0,0,255,0,0,0,0,
    255,0,0,0,255,0,0,255,0,255,0,0,0,255,0,0,0,255,255,255,0,0,255,0,0,0,255,
    0,0,0,255,0,0,0,0,0,0,0,0,0,0,20,0,255,0,4,4,4,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,255,
    0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,4,0,0,0,4,4,4,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,
    0,0,0,0,255,0,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,127,127,0,127,127,127,127,0,
    127,127,127,127,0,127,127,127,0,127,127,127,127,0,127,127,127,127,0,127,
    127,0,127,127,127,127,0,127,127,127,127,0,127,0,127,127,0,127,127,127,127,
    0,127,0,127,127,127,127,127,127,127,0,127,127,127,127,0,127,127,127,127,
    0,127,127,127,127,0,127,127,127,127,0,127,127,0,127,127,127,0,127,127,0,
    127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,
    127,127,0,127,127,127,127,127,0,127,127,127,0,127,127,127,0,127,127,127,
    0,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,255,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,
    0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,
    255,255,0,255,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,
    0,255,0,0,0,0,0,0,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,
    0,255,255,255,0,0,255,255,255,255,255,255,255,0,0,0,0,0,0,255,255,0,0,0,
    0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,255,0,255,0,0,255,255,0,0,
    0,255,0,0,0,0,0,0,0,255,255,255,255,0,0,0,0,0,255,255,255,255,255,255,255,
    0,0,255,255,255,255,255,255,255,0,255,255,255,255,0,0,255,255,255,255,255,
    255,255,0,0,255,255,255,255,255,255,255,0,255,0,0,255,255,0,255,0,0,255,
    0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,255,255,255,255,255,0,255,
    0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,255,255,255,0,
    0,255,0,0,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,
    255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,
    255,255,255,255,255,0,255,255,255,255,255,0,0,0,0,0,0,255,0,0,255,0,255,
    0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,
    255,0,0,0,0,255,0,0,255,0,0,0,0,0,255,0,0,255,0,0,0,0,0,255,0,0,255,255,
    0,0,255,0,255,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    255,0,0,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,255,
    0,0,0,0,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,
    255,255,255,255,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,0,255,0,0,255,0,255,0,0,0,0,0,
    0,0,255,0,0,0,0,0,0,255,0,255,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,
    0,255,0,0,0,255,0,0,0,0,0,255,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,255,0,
    255,255,255,0,255,0,0,0,255,255,0,255,255,0,0,0,255,0,0,0,0,0,255,0,255,
    255,255,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,
    255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,255,255,255,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,255,0,0,0,255,255,255,255,255,0,0,0,0,0,0,0,255,255,0,255,0,255,
    255,0,255,255,0,0,0,255,255,255,0,0,255,0,0,255,0,0,0,255,255,255,255,0,
    0,255,0,0,0,0,0,255,0,0,255,0,0,0,0,255,0,0,0,0,0,255,0,0,255,0,0,0,0,0,
    255,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,255,0,0,255,0,0,255,0,0,255,0,
    0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,127,255,255,255,255,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,255,0,255,
    0,0,255,0,0,255,0,0,0,0,0,255,0,255,0,0,255,0,0,0,255,0,0,0,0,0,255,0,0,
    0,0,0,255,0,0,255,0,0,0,0,255,0,0,0,0,0,255,0,0,255,0,0,0,0,0,255,0,0,0,
    0,0,0,0,0,0,0,0,0,0,255,255,255,0,255,255,255,255,0,255,255,255,255,255,
    255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,255,0,255,0,0,255,255,
    255,255,0,0,255,0,0,0,0,0,255,0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,255,
    0,0,255,0,0,0,255,0,255,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,
    0,0,0,0,0,255,0,255,0,0,255,0,0,255,0,0,0,0,0,255,0,0,255,0,255,0,0,0,255,
    0,0,0,0,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,255,0,
    0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,255,0,0,255,0,0,255,0,0,0,0,0,
    255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,127,0,255,255,255,0,0,255,255,255,255,255,255,255,0,0,255,
    0,255,0,0,0,0,255,0,255,0,255,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,255,0,0,0,255,255,0,255,255,0,0,255,255,255,255,0,0,0,0,0,0,255,
    255,255,255,255,255,255,0,0,255,255,255,255,255,255,255,0,255,255,255,255,
    0,0,255,255,255,255,255,255,255,0,0,255,255,255,255,255,255,255,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,255,255,255,0,0,0,0,0,255,255,0,255,255,255,0,0,255,255,255,255,255,
    255,255,0,255,255,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,255,255,0,0,0,255,0,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,0,127,127,
    127,127,0,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,0,
    127,127,127,127,127,0,127,127,127,127,0,127,127,127,127,127,127,127,127,
    127,127,127,127,0,127,127,127,127,127,0,127,127,0,127,127,127,127,127,127,
    127,127,0,127,127,127,127,127,127,127,127,0,127,127,127,127,0,127,127,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,0,127,0,127,
    127,127,0,127,127,127,0,127,127,127,0,127,127,127,127,0,127,127,127,127,
    127,127,127,0,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,
    0,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,
    0,127,127,127,0,127,127,127,127,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,127,0,0,0,0,0,255,0,0,0,255,0,0,0,0,255,255,0,0,0,0,0,0,0,255,0,
    0,0,255,0,255,0,0,255,255,255,0,0,255,0,255,0,0,0,255,255,255,255,0,0,255,
    255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,0,0,0,0,0,0,0,
    0,255,0,0,0,0,0,0,0,0,0,0,255,255,255,0,255,255,255,0,0,255,0,0,0,0,0,0,
    0,0,255,255,255,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,255,0,0,0,0,0,0,0,0,255,
    0,0,255,0,0,0,0,0,255,0,0,0,255,0,0,0,255,255,255,0,0,255,0,0,0,0,255,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,
    0,255,0,255,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,255,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,255,0,255,
    0,0,0,0,255,0,0,0,0,0,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,255,255,0,255,
    0,0,0,0,0,0,0,0,0,0,255,255,0,255,0,0,255,0,0,0,0,0,0,255,255,0,255,0,0,
    0,0,0,255,255,0,0,255,0,0,0,0,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    127,0,0,0,0,0,255,0,0,255,255,255,0,0,255,0,0,0,0,255,255,255,0,0,0,255,
    0,255,0,0,255,0,255,0,0,0,0,0,0,0,0,0,255,0,0,255,255,0,0,255,255,0,255,
    0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,255,0,255,255,255,0,0,255,0,0,0,0,0,
    0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,255,0,0,255,0,255,
    255,0,255,0,0,0,0,0,0,0,0,0,0,0,255,0,255,0,0,255,0,255,0,255,0,0,0,255,
    0,255,0,0,0,0,0,0,255,0,0,255,0,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,127,0,0,0,0,0,255,0,255,0,255,0,0,255,255,255,0,0,0,255,0,255,
    0,0,0,0,255,0,0,0,0,0,255,255,255,0,0,0,0,0,0,0,255,0,255,0,0,0,0,255,0,
    255,255,0,255,0,255,0,0,0,255,255,255,255,255,0,0,0,0,255,0,255,0,0,255,
    0,255,0,0,0,0,0,0,0,0,0,0,255,255,255,255,255,0,0,255,255,255,0,255,255,
    0,0,0,0,0,0,255,0,0,255,0,255,255,0,255,0,0,255,0,0,0,0,0,0,0,255,255,255,
    0,255,255,0,0,0,255,0,255,0,0,255,255,0,0,255,255,0,0,0,255,0,255,0,255,
    255,0,0,255,255,0,255,0,255,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,
    255,0,255,0,255,0,0,0,255,0,0,0,0,255,255,255,0,0,0,255,255,255,0,0,0,0,
    255,0,0,255,0,0,0,0,0,0,255,0,255,0,0,0,0,255,0,0,0,0,255,0,255,0,0,0,0,
    0,0,0,255,0,255,255,0,255,0,255,255,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,
    255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,255,0,255,0,0,255,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,255,0,0,0,255,0,255,0,255,0,0,0,0,255,
    0,0,0,0,255,0,0,0,255,0,255,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,
    0,255,0,255,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,
    255,255,255,0,0,0,0,0,0,255,0,0,255,255,0,0,255,0,0,0,0,0,255,0,255,0,0,
    0,0,0,0,255,0,0,0,0,255,0,255,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,127,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,255,0,255,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,255,0,255,0,0,0,0,255,0,255,255,255,255,0,0,0,255,
    0,0,0,255,0,0,0,0,255,0,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,
    0,0,0,255,0,0,255,255,255,0,255,255,255,255,0,0,0,0,0,0,0,0,0,255,0,0,0,
    255,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,0,0,255,0,255,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,255,0,0,0,255,255,
    255,0,0,255,0,0,0,0,255,0,0,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,
    0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,
    0,0,0,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,
    255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,127,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,
    0,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,127,127,127,127,0,127,0,127,127,127,127,0,127,127,127,
    127,0,127,127,127,127,127,0,127,127,127,127,127,0,127,0,127,127,127,127,
    0,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,0,127,127,
    127,127,0,127,127,127,127,127,127,0,127,127,0,127,127,127,127,127,127,127,
    0,127,127,127,127,0,127,127,127,127,0,127,127,127,127,127,0,127,127,127,
    127,0,127,127,127,0,127,127,127,0,127,127,127,127,0,127,127,127,127,0,127,
    127,127,0,127,127,127,0,127,127,127,0,127,127,127,127,0,127,127,127,127,
    0,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,
    127,127,127,127,127,127,0,127,127,127,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,255,0,
    0,0,0,0,255,0,0,0,0,0,0,255,0,255,0,0,0,255,0,255,255,0,0,0,255,0,0,255,
    0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,
    0,0,0,0,255,0,255,0,0,0,255,0,255,0,0,0,255,0,0,0,255,0,0,255,0,255,0,255,
    0,255,0,0,0,0,0,0,0,0,0,255,0,255,255,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,
    0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,255,0,0,0,0,255,0,255,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,255,0,255,
    0,0,0,255,0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,0,0,127,0,0,255,255,
    0,0,0,0,0,255,255,0,0,0,0,0,255,255,0,0,0,0,0,255,255,0,0,0,0,0,255,255,
    0,0,0,0,0,255,255,0,0,0,0,0,255,255,255,255,255,0,0,0,255,255,255,0,0,255,
    255,255,255,255,0,255,255,255,255,255,0,255,255,255,255,255,0,255,255,255,
    255,255,0,255,255,255,0,255,255,255,0,255,255,255,0,255,255,255,0,255,255,
    255,255,0,0,0,255,0,0,0,0,255,0,0,0,255,255,255,0,0,0,0,0,255,255,255,0,
    0,0,0,0,255,255,255,0,0,0,0,0,255,255,255,0,0,0,0,0,255,255,255,0,0,0,0,
    0,0,0,0,0,0,0,0,255,255,255,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,
    255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,255,0,255,0,0,0,0,255,0,0,
    255,0,127,0,0,255,255,0,0,0,0,0,255,255,0,0,0,0,0,255,255,0,0,0,0,0,255,
    255,0,0,0,0,0,255,255,0,0,0,0,0,255,255,0,0,0,0,255,0,255,0,0,0,0,0,255,
    0,0,0,255,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,255,
    0,0,0,255,0,0,0,255,0,0,0,255,0,0,255,0,0,0,255,0,0,255,255,0,0,0,255,0,
    0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,
    255,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,255,0,0,0,255,0,0,255,0,0,0,
    0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,0,255,0,255,
    0,0,255,255,255,0,0,255,0,0,255,0,127,0,255,0,0,255,0,0,0,255,0,0,255,0,
    0,0,255,0,0,255,0,0,0,255,0,0,255,0,0,0,255,0,0,255,0,0,0,255,0,0,255,0,
    0,0,255,0,255,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,
    0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,255,0,0,0,
    0,255,0,255,0,255,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,
    0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,255,0,0,0,255,
    0,255,0,0,0,255,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,
    255,0,255,0,0,0,0,255,0,0,255,0,255,0,0,255,0,0,255,0,255,0,255,0,0,127,
    0,255,0,0,255,0,0,0,255,0,0,255,0,0,0,255,0,0,255,0,0,0,255,0,0,255,0,0,
    0,255,0,0,255,0,0,0,255,0,0,255,0,0,0,255,0,255,255,255,255,0,255,0,0,0,
    0,0,0,255,255,255,255,0,0,255,255,255,255,0,0,255,255,255,255,0,0,255,255,
    255,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,255,255,255,0,0,255,
    0,255,0,0,255,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,
    0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,0,255,0,255,0,0,255,
    0,0,255,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,
    255,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,255,0,255,0,0,255,0,127,255,255,
    255,255,255,255,0,255,255,255,255,255,255,0,255,255,255,255,255,255,0,255,
    255,255,255,255,255,0,255,255,255,255,255,255,0,255,255,255,255,255,255,
    0,0,255,255,255,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,
    0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,255,0,0,
    0,0,255,0,255,0,0,0,255,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,
    0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,0,0,255,0,0,0,
    255,0,255,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,
    0,255,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,255,0,255,0,0,255,0,127,255,0,
    0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,
    0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,255,0,0,0,0,0,255,0,0,0,255,0,255,
    0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,255,0,0,
    0,255,0,0,0,255,0,0,255,0,0,0,255,0,0,255,0,0,0,0,255,0,0,255,0,0,0,255,
    0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,
    0,0,255,0,0,0,0,255,0,255,0,0,0,255,0,0,0,255,0,0,255,0,0,0,0,255,0,255,
    0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,0,0,255,0,0,0,255,255,
    255,0,0,255,0,0,255,0,127,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,
    0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,255,
    255,255,255,0,0,0,255,255,255,0,0,255,255,255,255,255,0,255,255,255,255,
    255,0,255,255,255,255,255,0,255,255,255,255,255,0,255,255,255,0,255,255,
    255,0,255,255,255,0,255,255,255,0,255,255,255,255,0,0,0,255,0,0,0,0,255,
    0,0,0,255,255,255,0,0,0,0,0,255,255,255,0,0,0,0,0,255,255,255,0,0,0,0,0,
    255,255,255,0,0,0,0,0,255,255,255,0,0,0,0,255,0,0,0,255,0,255,0,255,255,
    255,0,0,0,0,255,255,255,255,0,0,0,255,255,255,255,0,0,0,255,255,255,255,
    0,0,0,255,255,255,255,0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,255,0,0,127,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,127,127,127,
    127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,
    127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,
    127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,0,127,
    127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,
    0,127,127,127,0,127,127,127,0,127,127,127,0,127,127,127,127,127,127,0,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,
    127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,
    127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,
    127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,
    127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,
    127,0,127,127,127,127,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,127,0,255,0,0,0,0,0,0,255,0,0,0,255,0,0,255,255,0,255,0,
    0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,
    0,255,0,0,0,0,0,0,0,255,0,0,0,255,0,255,0,0,0,0,0,0,0,0,0,255,255,0,255,
    0,0,255,0,0,0,0,0,0,255,0,0,0,255,0,0,255,255,0,255,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,255,
    0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,255,
    0,0,0,0,255,0,0,0,255,0,255,0,255,0,255,255,0,0,255,0,255,0,0,255,0,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,255,0,255,0,0,255,0,
    255,0,0,255,0,255,0,255,0,255,255,0,255,0,0,255,255,0,255,0,255,255,0,0,
    0,255,0,0,0,0,255,0,0,0,255,0,255,0,255,0,255,255,0,255,0,0,255,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,255,0,255,0,0,255,0,255,0,
    0,0,255,0,0,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,127,0,255,255,0,0,0,255,255,0,0,0,255,255,0,0,0,255,255,0,0,0,255,
    255,0,0,0,255,255,0,0,255,255,255,0,255,255,0,0,0,255,255,0,0,255,255,0,
    0,0,255,255,0,0,0,255,255,0,0,0,255,255,0,0,0,255,0,255,0,0,255,0,0,255,
    0,0,0,0,255,0,255,255,255,0,0,0,255,255,0,0,0,255,255,0,0,0,255,255,0,0,
    0,255,255,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,255,255,255,0,0,255,0,0,255,
    0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,0,0,0,255,0,255,255,255,
    0,0,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,255,0,0,0,
    0,255,0,0,0,0,255,0,0,0,0,255,0,0,0,0,255,0,0,0,0,255,0,0,0,0,255,0,0,255,
    0,255,0,0,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,0,255,
    0,255,0,0,255,0,0,255,0,0,255,255,255,0,255,0,0,255,0,255,0,0,255,0,255,
    0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,0,255,255,255,255,255,
    0,255,0,0,255,255,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,
    0,0,255,0,255,0,0,255,0,0,255,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,127,0,255,255,255,0,0,255,255,255,0,0,255,255,255,0,0,255,255,255,
    0,0,255,255,255,0,0,255,255,255,0,0,255,255,255,255,255,255,0,255,0,0,0,
    255,255,255,255,0,255,255,255,255,0,255,255,255,255,0,255,255,255,255,0,
    0,255,0,255,0,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,
    255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,0,0,0,0,0,0,0,255,
    0,255,0,255,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,0,
    255,0,255,0,0,255,0,0,255,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,127,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,0,0,
    255,0,255,0,0,255,0,255,0,0,255,0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,0,
    0,255,0,0,0,0,255,0,0,0,0,0,255,0,255,0,0,255,0,0,255,0,255,0,0,255,0,255,
    0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,0,
    0,255,0,0,0,0,255,0,0,0,255,255,0,0,255,0,255,0,0,255,0,255,0,0,255,0,255,
    0,0,255,0,255,0,0,255,0,0,255,0,255,0,0,255,0,0,255,0,0,255,0,255,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,255,255,255,0,0,255,255,255,0,0,255,
    255,255,0,0,255,255,255,0,0,255,255,255,0,0,255,255,255,0,0,255,255,0,255,
    255,255,0,0,255,255,0,0,255,255,255,0,0,255,255,255,0,0,255,255,255,0,0,
    255,255,255,0,0,255,0,255,0,0,255,0,0,255,0,0,255,255,0,0,255,0,0,255,0,
    0,255,255,0,0,0,255,255,0,0,0,255,255,0,0,0,255,255,0,0,0,255,255,0,0,0,
    0,0,0,0,0,0,0,255,255,255,0,0,0,255,255,255,0,0,255,255,255,0,0,255,255,
    255,0,0,255,255,255,0,0,0,255,0,0,0,255,255,255,0,0,0,0,255,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,127,127,127,0,127,127,127,127,0,127,127,
    127,127,0,127,127,127,127,0,127,127,127,127,0,127,127,127,127,0,127,127,
    127,127,127,127,127,0,127,127,127,0,127,127,127,127,0,127,127,127,127,0,
    127,127,127,127,0,127,127,127,127,0,127,127,0,127,0,127,127,0,127,127,0,
    127,127,127,127,0,127,127,127,127,0,127,127,127,127,0,127,127,127,127,0,
    127,127,127,127,0,127,127,127,127,0,127,127,127,127,0,127,127,127,127,127,
    127,0,127,127,127,127,127,0,127,127,127,127,0,127,127,127,127,0,127,127,
    127,127,0,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,0,127,
    127,127,127,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
};


// Normal font
const int FONT1_BM_W = 253;
const int FONT1_BM_H = 106;
static const unsigned char s_Font1[] = 
{
    127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,255,
    0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,255,0,0,0,255,0,255,0,0,0,0,
    0,255,0,255,0,0,0,0,0,255,0,0,0,0,255,255,0,0,0,0,255,0,0,0,0,0,255,255,
    255,0,0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,0,255,0,255,0,255,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,255,255,255,0,0,0,0,0,
    255,0,0,0,0,255,255,255,255,0,0,0,255,255,255,255,0,0,0,0,0,0,255,0,0,255,
    255,255,255,255,255,0,0,0,255,255,255,0,0,255,255,255,255,255,255,0,0,255,
    255,255,255,0,0,0,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,127,0,0,0,0,0,0,255,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,255,255,
    255,0,0,255,0,0,255,0,0,255,0,0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,0,255,
    0,0,0,0,255,0,0,0,0,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,255,0,0,255,0,0,0,0,255,0,0,255,255,255,0,0,0,255,0,0,0,0,255,0,
    255,0,0,0,0,255,0,0,0,0,255,255,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,
    0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,127,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,255,255,255,255,255,255,
    0,0,255,0,255,0,255,0,255,0,0,255,0,0,255,0,0,0,0,0,255,0,0,0,255,0,0,0,
    0,0,0,0,255,0,0,0,0,0,0,255,0,0,255,0,255,0,255,0,0,0,0,255,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,
    0,0,255,0,0,0,0,0,0,255,0,0,0,255,0,255,0,0,255,0,0,0,0,0,0,255,0,0,0,0,
    0,0,0,0,0,0,255,0,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,0,255,0,0,0,255,
    0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,255,0,255,
    0,0,0,0,255,0,255,0,0,0,255,0,0,255,0,255,0,0,0,0,0,0,0,255,0,255,0,0,0,
    0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,
    0,255,0,0,0,0,0,0,255,0,0,255,0,0,255,0,0,255,255,255,255,255,0,0,255,255,
    255,255,255,0,0,0,0,0,0,255,0,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,0,255,
    0,0,0,255,0,0,0,0,0,0,255,255,0,0,0,0,255,255,255,255,255,255,0,0,0,0,255,
    255,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,255,0,0,
    0,0,0,0,0,0,0,0,255,0,255,0,0,0,0,0,255,255,0,0,0,0,255,255,0,0,255,0,0,
    255,255,0,0,0,255,255,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,
    0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,
    255,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,255,255,255,0,0,255,0,0,0,255,
    0,0,0,0,0,0,0,255,0,255,0,0,0,0,255,0,0,0,0,255,0,0,0,0,255,255,255,255,
    0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,255,255,0,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,
    0,0,0,0,255,0,0,0,0,0,0,0,0,255,255,255,255,255,255,0,0,0,0,0,255,255,0,
    0,0,0,0,0,0,255,0,255,0,0,255,0,255,0,0,255,0,0,255,0,0,0,0,0,255,0,0,0,
    0,0,0,255,0,0,0,0,0,0,0,0,255,255,255,255,255,255,255,0,0,0,0,0,255,255,
    255,0,0,0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,255,255,
    0,0,0,0,0,0,0,0,255,0,255,255,255,255,255,255,0,0,0,0,0,0,255,0,255,0,0,
    0,0,255,0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,255,255,255,255,255,0,0,0,
    0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,255,255,255,255,255,255,0,0,0,0,0,0,0,
    0,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,255,0,0,0,0,
    0,0,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,255,0,0,255,
    0,255,0,0,0,255,255,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,
    0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,0,
    0,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,255,
    0,255,0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,255,0,0,
    0,0,0,0,0,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,0,
    0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,255,0,255,0,0,0,0,0,255,0,255,0,255,0,0,0,0,0,255,0,0,255,0,0,255,0,255,
    0,0,0,255,255,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,255,
    0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,0,0,
    255,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,255,0,0,255,0,0,0,0,
    255,0,255,0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,255,0,
    0,0,255,0,0,0,255,0,0,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,255,0,0,0,0,0,
    0,0,0,0,255,0,255,0,0,0,0,0,0,255,255,255,0,0,0,0,0,255,0,0,0,0,255,255,
    0,0,0,255,255,255,0,0,255,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,
    0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,0,255,255,255,
    255,0,0,0,255,255,255,255,255,0,255,255,255,255,255,255,0,0,255,255,255,
    255,0,0,0,0,0,0,255,0,0,0,255,255,255,255,0,0,0,255,255,255,255,0,0,0,255,
    0,0,0,0,0,0,255,255,255,255,0,0,0,255,255,255,0,0,0,0,255,0,0,0,255,0,0,
    0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,255,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,
    0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,
    0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,127,127,127,0,
    127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,0,127,127,127,127,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,127,0,127,127,127,0,127,127,127,0,127,127,127,127,0,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,0,127,127,127,
    127,0,127,127,127,0,127,127,127,127,0,127,127,127,127,127,127,0,127,127,
    127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,
    127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,
    0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,
    127,0,127,127,0,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,
    127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,
    127,127,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,0,255,0,0,0,0,0,255,255,255,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,255,
    255,255,255,0,0,0,0,0,0,0,255,0,0,0,0,255,255,255,255,255,0,0,0,0,255,255,
    255,255,0,255,255,255,255,255,0,0,0,255,255,255,255,255,255,0,255,255,255,
    255,255,0,0,0,255,255,255,255,0,0,255,0,0,0,0,0,255,0,255,255,255,0,0,255,
    255,255,0,255,0,0,0,0,255,0,255,0,0,0,0,0,255,255,0,0,0,0,255,255,0,255,
    255,0,0,0,0,255,0,0,0,255,255,255,255,0,0,0,255,255,255,255,255,0,0,0,0,
    255,255,255,255,0,0,0,255,255,255,255,0,0,0,0,255,255,255,255,0,0,255,255,
    255,255,255,255,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,
    0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,255,255,255,
    255,255,0,0,255,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,255,255,0,0,0,0,255,255,0,0,
    0,0,255,0,255,0,0,0,255,0,0,0,0,255,0,0,255,0,0,0,0,0,255,0,0,0,0,255,0,
    0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,255,0,255,0,0,0,0,0,255,0,
    0,255,0,0,0,0,0,255,0,255,0,0,0,255,0,0,255,0,0,0,0,0,255,255,0,0,0,0,255,
    255,0,255,255,0,0,0,0,255,0,0,255,0,0,0,0,255,0,0,255,0,0,0,0,255,0,0,255,
    0,0,0,0,255,0,0,255,0,0,0,255,0,0,255,0,0,0,0,255,0,0,0,0,255,0,0,0,0,255,
    0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,255,0,0,0,0,255,0,0,255,
    0,0,255,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,255,0,0,0,
    0,0,0,255,0,0,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,127,0,255,0,255,255,255,255,0,255,0,0,0,0,255,0,255,0,0,0,255,0,0,
    0,0,255,0,255,0,0,0,0,0,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,0,0,
    0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,255,0,0,0,0,0,255,0,255,0,0,
    255,0,0,0,255,0,0,0,0,0,255,0,255,0,0,255,0,255,0,255,0,255,0,0,0,255,0,
    255,0,0,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,255,0,0,0,
    255,0,0,255,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,255,0,0,255,0,0,
    0,255,0,0,0,255,0,0,255,0,255,0,0,255,0,0,0,255,0,0,255,0,0,0,255,0,0,0,
    255,0,0,0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,255,0,0,
    0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,255,0,255,0,0,
    0,255,0,0,255,0,0,0,255,0,255,0,0,0,255,0,0,0,0,255,0,255,0,0,0,0,0,0,255,
    0,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,
    0,0,0,255,0,0,255,0,0,0,0,0,255,0,255,0,255,0,0,0,0,255,0,0,0,0,0,255,0,
    255,0,0,255,0,255,0,255,0,255,0,0,0,255,0,255,0,0,0,0,0,0,255,0,255,0,0,
    0,0,255,0,255,0,0,0,0,0,0,255,0,255,0,0,0,255,0,0,255,0,0,0,0,0,0,0,0,0,
    255,0,0,0,0,255,0,0,0,0,0,255,0,0,255,0,0,0,255,0,0,0,255,0,0,255,0,255,
    0,0,255,0,0,0,0,255,255,0,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,0,0,0,255,
    0,0,0,0,255,0,0,0,0,0,0,255,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,127,255,0,255,0,0,0,255,0,0,255,0,0,255,0,0,0,255,
    0,0,255,255,255,255,255,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,255,0,255,255,
    255,255,255,255,0,255,255,255,255,255,0,255,0,0,0,255,255,255,0,255,255,
    255,255,255,255,255,0,0,255,0,0,0,0,0,255,0,255,255,0,0,0,0,0,255,0,0,0,
    0,0,255,0,255,0,0,255,0,255,0,255,0,0,255,0,0,255,0,255,0,0,0,0,0,0,255,
    0,255,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,255,255,255,255,0,0,0,0,255,255,
    255,255,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,255,0,0,255,0,0,0,255,0,0,0,
    255,0,0,255,0,255,0,0,255,0,0,0,0,255,255,0,0,0,0,0,0,255,0,0,0,0,0,0,255,
    0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,255,0,255,0,0,0,255,0,0,255,0,0,255,
    0,0,0,255,0,0,255,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,0,0,0,0,255,0,255,
    0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,255,
    0,0,0,0,0,255,0,255,0,255,0,0,0,0,255,0,0,0,0,0,255,0,0,255,255,0,0,255,
    0,255,0,0,0,255,0,255,0,255,0,0,0,0,0,0,255,0,255,255,255,255,255,0,0,255,
    0,0,0,0,0,0,255,0,255,0,0,255,0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,255,
    0,0,0,0,0,255,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,255,0,255,0,0,0,0,
    255,255,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,
    0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,127,255,0,255,0,0,0,255,0,0,255,0,0,255,255,255,255,255,0,0,255,0,0,0,
    0,255,0,255,0,0,0,0,0,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,0,0,0,
    0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,255,0,0,0,0,0,255,0,255,0,0,
    255,0,0,0,255,0,0,0,0,0,255,0,0,255,255,0,0,255,0,255,0,0,0,255,0,255,0,
    255,0,0,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,255,0,0,0,255,
    0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,255,0,0,0,255,0,255,
    0,0,0,0,0,255,255,0,0,0,255,255,0,0,0,0,255,0,0,255,0,0,0,0,0,255,0,0,0,
    0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,255,0,255,255,255,255,
    255,255,0,0,255,0,0,0,0,0,255,0,255,0,0,0,0,255,0,0,255,0,0,0,0,0,255,0,
    0,0,0,255,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,255,0,255,0,0,
    0,0,0,255,0,0,255,0,0,0,0,0,255,0,255,0,0,0,255,0,0,255,0,0,0,0,0,255,0,
    0,0,0,0,0,255,0,255,0,0,0,0,255,255,0,0,255,0,0,0,0,255,0,0,255,0,0,0,0,
    0,0,0,255,0,0,0,0,255,0,0,255,0,0,0,255,0,0,255,0,0,0,0,255,0,0,0,0,255,
    0,0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,
    0,0,0,255,0,0,255,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,
    255,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,127,0,255,255,0,0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,255,255,255,
    255,255,0,0,0,0,255,255,255,255,0,255,255,255,255,255,0,0,0,255,255,255,
    255,255,255,0,255,0,0,0,0,0,0,0,255,255,255,255,255,0,255,0,0,0,0,0,255,
    0,255,255,255,0,255,255,255,0,0,255,0,0,0,0,255,0,255,255,255,255,255,0,
    255,0,0,0,0,0,0,255,0,255,0,0,0,0,255,255,0,0,0,255,255,255,255,0,0,0,255,
    0,0,0,0,0,0,0,0,255,255,255,255,0,0,0,255,0,0,0,0,255,0,0,255,255,255,255,
    0,0,0,0,0,255,0,0,0,0,0,0,255,255,255,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,
    0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,0,0,255,0,0,0,0,255,255,255,255,255,
    255,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,255,255,255,255,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,0,0,0,0,255,0,0,
    255,255,255,0,0,0,0,0,0,0,0,0,0,255,255,255,255,255,255,255,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,127,127,127,127,127,127,127,127,127,127,0,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,
    127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,
    0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,
    0,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,0,127,
    127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,
    127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,
    127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,
    127,0,127,127,127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,
    127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,
    127,0,127,127,127,127,0,127,127,127,127,0,127,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,255,
    0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,
    0,0,0,0,0,0,255,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,
    0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,0,0,0,255,
    0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,127,0,255,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,255,0,0,0,
    0,0,0,255,0,0,255,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    255,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,
    0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,
    0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,
    0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,255,
    0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,255,255,255,0,0,255,0,255,255,
    255,0,0,0,255,255,255,255,0,0,255,255,255,255,255,0,0,255,255,255,255,0,
    0,255,255,255,255,0,255,255,255,255,255,0,255,0,255,255,255,0,0,255,0,255,
    255,0,255,0,0,0,255,0,255,0,255,255,255,255,0,255,255,255,0,0,255,0,255,
    255,255,0,0,0,255,255,255,255,0,0,255,0,255,255,255,0,0,0,255,255,255,255,
    255,0,255,0,255,0,0,255,255,255,0,255,255,255,255,0,255,0,0,0,0,255,0,255,
    0,0,0,255,0,255,0,0,0,255,0,0,0,255,0,255,0,0,0,255,0,255,0,0,0,255,0,255,
    255,255,255,0,0,0,255,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,
    0,0,0,0,0,0,255,0,255,255,0,0,0,255,0,255,0,0,0,0,0,255,0,0,0,0,255,0,255,
    0,0,0,0,255,0,0,255,0,0,255,0,0,0,0,255,0,255,255,0,0,0,255,0,255,0,0,255,
    0,255,0,0,255,0,0,255,0,255,0,0,0,255,0,0,0,255,0,255,255,0,0,0,255,0,255,
    0,0,0,0,255,0,255,255,0,0,0,255,0,255,0,0,0,0,255,0,255,255,0,0,255,0,0,
    0,0,0,255,0,0,0,255,0,0,0,0,255,0,255,0,0,0,255,0,255,0,0,0,255,0,0,0,255,
    0,0,255,0,255,0,0,255,0,0,0,255,0,0,0,0,255,0,0,0,255,0,0,0,0,0,255,0,0,
    0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,255,0,255,0,0,0,0,255,0,255,
    0,0,0,0,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,0,255,0,0,255,0,0,0,0,255,
    0,255,0,0,0,0,255,0,255,0,0,255,0,255,0,255,0,0,0,255,0,255,0,0,0,255,0,
    0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,
    0,0,255,0,255,0,0,0,255,0,0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,255,0,255,
    0,0,0,255,0,255,0,255,0,255,0,0,0,255,0,255,0,0,0,255,0,255,0,0,0,0,255,
    0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,255,255,0,0,0,255,0,0,0,
    255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,
    0,255,255,255,255,0,255,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,0,0,255,0,255,
    255,255,255,255,255,0,0,255,0,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,
    0,0,255,0,255,255,0,0,0,0,255,0,255,0,0,0,255,0,0,0,255,0,255,0,0,0,0,255,
    0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,
    255,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,255,0,255,0,0,0,255,0,255,0,255,
    0,255,0,0,0,0,255,0,0,0,0,255,0,255,0,0,0,255,0,0,0,255,255,0,0,0,0,0,0,
    255,0,0,0,0,0,255,255,0,0,255,0,0,255,0,0,255,0,0,0,255,0,0,0,0,0,0,0,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,255,0,0,0,255,0,255,0,
    0,0,0,255,0,255,0,0,0,0,0,255,0,0,0,0,255,0,255,0,0,0,0,0,0,0,255,0,0,255,
    0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,255,0,255,0,255,0,0,0,255,0,255,
    0,0,0,255,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,
    0,255,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,0,255,0,0,0,255,0,0,0,0,255,0,
    0,255,0,255,0,0,0,255,0,255,0,255,0,255,0,0,0,255,0,255,0,0,0,255,0,255,
    0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,255,0,0,0,255,
    255,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,
    0,0,0,0,0,0,255,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,0,255,
    255,0,255,0,0,0,0,255,0,0,255,0,0,255,0,0,0,255,255,0,255,0,0,0,0,255,0,
    255,0,0,255,0,255,0,0,255,0,0,255,0,255,0,0,0,255,0,0,0,255,0,255,0,0,0,
    0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,255,255,0,255,0,0,
    0,0,0,0,255,0,0,255,0,0,0,255,0,0,0,255,255,0,0,0,255,0,0,0,0,0,255,0,0,
    0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,
    255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,255,255,255,255,0,255,255,
    255,255,255,0,0,0,255,255,255,255,0,0,255,255,255,0,255,0,0,255,255,255,
    255,0,0,0,255,0,0,0,255,255,255,0,255,0,255,0,0,0,0,255,0,255,0,0,255,0,
    255,0,0,0,255,0,255,0,255,0,0,0,255,0,0,0,255,0,255,0,0,0,0,255,0,0,255,
    255,255,255,0,0,255,255,255,255,255,0,0,0,255,255,255,0,255,0,255,0,0,0,
    255,255,255,0,0,0,0,255,255,0,0,255,255,255,0,255,0,0,0,255,0,0,0,0,0,255,
    0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,255,255,255,0,0,0,255,
    0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,255,
    255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,
    0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,255,255,0,0,0,
    255,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,127,127,127,127,0,127,127,127,127,127,0,
    127,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,127,
    0,127,127,127,127,127,127,0,127,127,127,0,127,127,127,127,127,127,0,127,
    127,127,127,127,127,0,127,0,127,127,0,127,127,127,127,127,0,127,0,127,127,
    127,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,
    127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,
    0,127,127,127,127,0,127,127,127,127,0,127,127,127,127,127,127,0,127,127,
    127,127,127,0,127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,
    0,127,127,127,127,127,0,127,127,127,127,0,127,127,127,127,127,0,127,127,
    127,127,0,127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,0,127,
    127,127,127,127,127,127,127,127,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,255,0,0,255,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,
    0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,255,0,255,0,255,
    0,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,
    255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,0,0,0,0,0,127,0,0,255,
    255,255,255,0,0,0,255,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,255,
    255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,
    255,0,0,255,0,0,0,255,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,255,255,255,255,
    0,0,0,0,0,0,0,0,0,255,255,255,255,255,255,255,255,255,0,0,0,255,255,255,
    255,255,255,255,255,255,0,255,255,255,255,255,255,0,0,0,255,255,255,255,
    255,255,255,255,255,0,0,0,255,255,255,255,255,255,255,255,255,0,255,0,0,
    0,255,0,255,0,255,0,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,255,0,255,255,0,0,0,255,255,255,0,255,0,0,0,255,0,0,255,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,255,255,255,255,
    255,0,0,255,255,0,0,255,0,0,0,0,0,255,0,127,0,255,0,0,0,0,0,0,0,255,0,0,
    0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,
    255,255,255,255,0,0,255,255,255,255,255,0,0,0,0,0,0,0,0,255,0,0,255,0,0,
    255,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,255,0,
    0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,255,0,0,0,0,0,0,
    0,255,0,0,0,255,0,0,0,0,0,0,0,255,0,0,255,0,255,0,0,0,255,0,255,0,255,0,
    255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    255,0,0,255,255,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,255,0,0,127,255,0,0,0,0,
    0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,
    0,255,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,255,0,255,0,0,0,0,0,255,
    0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,
    0,0,255,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,
    0,255,0,255,0,0,255,255,255,0,0,255,0,0,0,0,255,255,255,255,0,255,255,255,
    0,0,0,0,255,0,0,0,0,0,0,0,255,0,255,255,255,255,0,0,255,0,0,0,255,0,0,127,
    255,255,255,255,255,255,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,
    0,0,0,0,255,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,255,
    0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,255,0,
    0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,255,0,0,255,0,0,0,255,0,255,0,0,0,0,0,0,255,0,0,255,0,0,
    0,0,255,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,255,0,255,
    0,0,0,127,255,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,255,
    255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,
    0,0,0,0,0,0,0,0,255,255,0,0,255,0,0,255,255,0,0,0,255,255,0,0,0,255,255,
    255,255,0,0,0,255,0,0,0,255,0,0,0,0,0,255,255,255,255,255,0,0,0,255,0,0,
    0,0,0,0,0,255,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,255,0,0,
    0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,
    0,0,0,0,0,0,255,0,255,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,255,
    0,0,0,255,0,0,0,0,0,255,0,0,0,0,127,255,255,255,255,255,255,0,0,0,255,0,
    0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,255,0,0,0,0,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,255,
    0,0,255,0,255,0,0,255,0,0,0,0,0,0,255,0,0,255,0,0,0,255,0,0,0,0,0,255,0,
    0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,
    0,255,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,
    255,255,255,0,255,255,255,255,255,0,255,255,255,255,255,255,255,255,255,
    255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,0,0,0,0,0,255,
    0,255,0,0,0,0,255,255,255,255,255,0,0,0,255,0,0,0,0,0,0,0,255,0,0,255,0,
    0,0,0,0,0,255,0,0,0,0,127,255,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,
    0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,
    255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,255,0,255,0,0,255,0,0,0,
    0,0,0,255,0,0,0,255,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,
    0,255,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,0,0,0,
    0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,255,
    0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,
    0,0,255,0,0,0,0,127,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,255,0,
    0,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,
    255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,255,0,255,0,0,255,0,255,
    0,0,0,0,255,0,0,0,0,255,0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,
    0,0,255,0,255,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,0,0,
    0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,
    0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,255,0,0,0,0,0,
    0,0,255,0,0,0,0,127,0,0,255,255,255,255,0,0,0,255,255,255,255,255,255,255,
    255,255,0,0,255,0,0,0,255,0,0,0,0,0,255,0,255,0,0,255,0,0,255,0,0,255,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,255,0,
    0,0,255,255,0,0,0,255,255,255,255,0,0,0,0,0,0,0,0,0,255,255,255,255,255,
    255,255,255,255,0,0,0,255,255,255,255,255,255,255,255,255,0,255,255,255,
    255,255,255,0,0,0,255,255,255,255,255,255,255,255,255,0,0,0,255,255,255,
    255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,
    255,255,0,0,0,0,0,0,0,0,255,255,255,255,0,255,255,255,255,0,0,0,255,255,
    255,255,255,255,255,255,255,0,255,255,255,255,0,0,0,0,255,0,0,0,0,127,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,255,0,0,0,0,0,255,0,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,
    127,127,127,0,127,127,0,127,127,127,127,127,127,0,127,127,127,127,0,127,
    127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,
    127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,
    127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,
    127,0,127,127,127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,
    127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,
    127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,127,127,0,
    127,127,0,127,127,0,127,127,127,127,0,127,127,127,127,0,127,127,127,127,
    0,127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,127,127,0,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,127,0,127,127,
    127,127,0,127,127,127,127,0,127,127,127,127,127,127,127,127,127,127,0,127,
    127,127,127,127,127,127,127,127,127,127,0,127,127,127,127,0,127,127,127,
    127,127,127,127,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,255,0,0,0,
    0,0,255,0,0,0,0,0,255,255,255,255,0,0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,
    255,0,0,0,0,255,255,255,255,0,0,255,0,0,255,0,0,0,0,0,255,255,255,255,0,
    0,0,0,0,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,
    255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,255,
    255,255,0,0,0,255,255,255,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,255,255,255,
    255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,255,255,255,0,0,0,0,0,0,
    0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,255,255,
    255,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,
    0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,
    0,0,0,0,255,255,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,255,255,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,255,
    0,0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,
    255,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,0,0,255,0,0,0,255,0,0,
    0,0,0,0,0,0,0,255,255,0,0,0,255,0,0,0,0,0,0,255,255,0,0,0,0,255,0,0,0,0,
    0,0,0,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,127,0,0,0,0,0,255,0,0,0,0,255,255,255,255,0,0,255,0,0,0,0,
    0,255,0,0,0,0,255,0,0,0,255,0,255,0,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,0,
    0,0,0,0,0,255,0,0,255,255,0,0,255,0,0,0,0,255,255,255,0,0,0,255,0,0,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,255,255,255,0,0,255,0,0,0,0,0,0,0,0,
    0,0,255,0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,255,255,0,0,0,0,0,
    0,0,0,0,255,0,0,0,0,255,0,255,255,255,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,255,0,0,255,0,0,0,255,0,255,0,0,255,0,0,0,0,0,255,0,0,0,255,0,0,0,
    0,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,255,255,0,0,0,255,0,0,0,0,0,0,0,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,255,0,0,
    0,255,0,255,0,0,0,0,255,0,0,0,0,0,0,255,255,255,255,0,0,0,0,255,0,255,0,
    0,0,0,255,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,255,0,0,255,
    0,0,255,0,0,255,0,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,
    0,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,255,255,0,0,0,0,0,255,0,0,0,0,0,0,255,
    0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,255,255,255,255,0,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,0,255,0,0,255,0,0,255,0,0,
    0,0,255,0,0,255,0,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,0,0,255,0,255,
    0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,
    0,0,0,0,0,255,0,0,0,255,0,255,0,0,0,0,255,0,0,0,0,0,0,255,0,0,255,0,0,0,
    0,0,255,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,
    0,0,0,255,0,0,255,0,0,255,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    255,0,0,255,0,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,
    255,255,255,0,0,255,255,255,255,0,0,255,255,255,0,0,0,0,0,0,0,0,0,255,0,
    0,0,0,255,0,0,255,255,255,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,255,255,255,
    0,255,0,0,0,255,0,0,0,255,0,0,255,0,0,0,255,0,0,255,0,0,255,255,0,0,0,0,
    255,0,0,255,0,255,255,255,0,0,0,0,0,0,255,0,255,0,0,255,255,0,0,0,255,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,255,0,0,
    0,255,0,255,0,0,0,255,255,255,255,255,0,0,0,255,0,0,255,0,0,0,255,255,255,
    255,255,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,
    0,255,0,0,0,255,255,255,0,255,0,0,255,0,0,0,0,255,255,255,255,255,255,255,
    0,255,255,255,0,255,0,0,255,255,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,
    0,0,0,0,255,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,0,
    0,0,0,255,0,0,255,0,0,0,0,0,255,0,0,255,0,255,0,0,0,0,0,0,0,255,0,0,0,0,
    255,0,0,255,255,255,0,255,0,0,255,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,255,0,0,0,255,0,255,0,0,0,0,
    255,0,0,0,0,0,0,255,255,255,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,255,
    0,0,255,0,0,0,0,0,0,0,0,255,0,0,255,0,0,255,0,0,255,0,0,0,0,0,0,0,0,255,
    0,0,255,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,255,0,0,255,0,0,255,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,255,0,0,0,0,255,0,0,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,0,255,0,0,255,0,0,255,0,0,0,0,0,
    0,255,0,0,0,0,255,0,0,0,0,0,0,255,0,0,255,0,0,255,0,255,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,255,0,0,0,255,0,255,
    0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,
    0,255,255,0,0,0,0,0,0,0,0,0,0,255,0,0,255,255,0,0,255,0,0,0,0,0,0,0,0,0,
    0,255,0,0,255,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,255,0,0,0,255,255,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,255,255,0,0,0,0,255,0,255,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,0,0,255,0,0,255,
    255,255,255,255,0,0,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,255,0,0,255,255,255,
    255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,
    0,0,0,0,0,255,0,0,0,0,255,255,255,255,0,255,255,255,255,255,255,0,0,0,0,
    0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,255,
    255,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,
    0,0,0,0,255,255,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,0,255,0,0,0,
    0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,255,255,255,255,0,0,0,0,255,
    0,0,0,0,0,0,255,0,0,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,255,255,
    255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,255,0,255,0,0,0,
    0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,255,255,255,
    255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,
    0,0,255,0,255,0,0,0,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,
    127,127,0,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,
    0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,0,
    127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,
    127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,127,0,127,127,127,0,127,127,127,127,127,127,127,
    127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,
    127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,
    127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,
    127,0,127,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,
    0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,
    127,127,127,127,127,0,127,127,127,127,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,127,0,0,255,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,255,
    0,0,0,0,0,255,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,255,0,0,0,0,255,255,0,
    0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,255,255,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,
    255,0,0,0,0,0,0,0,255,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,255,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,127,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,255,0,0,0,
    255,0,255,255,0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,255,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,255,0,0,255,0,
    0,0,255,0,0,255,0,0,0,255,0,0,0,255,0,0,255,0,0,255,255,0,255,0,0,0,0,0,
    0,0,0,0,0,255,0,255,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,
    0,255,0,0,255,0,0,0,0,0,255,0,255,255,0,0,0,0,0,255,0,0,255,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,255,
    0,0,255,0,0,0,255,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,255,0,0,0,0,0,0,0,255,0,
    0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,
    0,0,0,0,0,0,255,255,255,255,255,255,255,0,0,0,255,255,255,255,0,255,255,
    255,255,255,255,0,255,255,255,255,255,255,0,255,255,255,255,255,255,0,255,
    255,255,255,255,255,0,255,255,255,0,255,255,255,0,255,255,255,0,255,255,
    255,0,0,255,255,255,255,0,0,0,255,255,0,0,0,0,255,0,0,0,255,255,255,255,
    0,0,0,0,0,255,255,255,255,0,0,0,0,0,255,255,255,255,0,0,0,0,0,255,255,255,
    255,0,0,0,0,0,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,
    0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,
    0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,0,0,255,0,0,0,
    0,0,0,0,0,0,0,0,127,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,
    0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,0,
    255,0,255,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,255,
    0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,
    0,0,0,255,0,0,255,255,0,0,0,0,255,0,0,255,0,0,0,0,255,0,0,0,255,0,0,0,0,
    255,0,0,0,255,0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,
    0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,255,0,0,0,0,0,255,0,255,0,0,0,
    0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,255,0,0,0,255,0,0,255,
    0,0,0,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,127,0,0,255,0,255,0,0,0,
    0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,
    0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,255,
    0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,
    255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,0,255,0,255,0,255,0,0,0,255,0,255,
    0,0,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,255,0,0,0,
    0,0,0,255,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,0,255,
    0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,
    0,0,255,0,0,255,0,0,0,255,0,0,255,255,255,255,255,0,0,255,0,0,0,255,0,0,
    0,0,0,0,0,0,0,0,0,127,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,
    255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,
    0,255,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,
    255,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,
    0,255,0,0,0,0,255,0,255,0,255,0,0,0,255,0,255,0,0,0,0,0,0,255,0,255,0,0,
    0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,255,0,0,0,0,0,
    0,255,0,0,0,255,0,0,0,255,0,0,255,0,0,0,255,0,0,255,0,255,0,0,0,0,0,255,
    0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,0,255,0,
    255,0,0,0,255,0,0,0,0,255,0,255,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,127,0,
    255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,
    0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,255,255,255,255,
    255,0,255,0,0,0,0,0,0,255,255,255,255,255,255,0,255,255,255,255,255,255,
    0,255,255,255,255,255,255,0,255,255,255,255,255,255,0,0,255,0,0,0,255,0,
    0,0,255,0,0,0,255,0,0,255,255,255,255,0,0,255,0,255,0,0,255,0,0,255,0,255,
    0,0,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,255,0,0,0,
    0,0,0,255,0,255,0,0,0,0,0,0,255,0,0,0,0,255,0,255,0,0,0,255,0,0,255,0,0,
    0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,
    0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,255,0,0,0,0,255,0,255,0,0,0,255,0,0,0,
    0,0,0,0,0,0,0,0,127,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,
    255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,
    255,255,255,255,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,
    0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,
    0,0,255,0,0,0,0,255,0,255,0,0,0,255,0,255,0,255,0,0,0,0,0,0,255,0,255,0,
    0,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,255,0,0,0,0,
    0,0,255,0,0,0,0,0,255,0,0,0,0,255,0,0,255,0,0,0,255,0,255,0,0,0,0,0,255,
    0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,0,0,255,
    0,0,0,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,127,0,255,
    255,255,255,255,0,0,0,255,255,255,255,255,0,0,0,255,255,255,255,255,0,0,
    0,255,255,255,255,255,0,0,0,255,255,255,255,255,0,0,0,255,255,255,255,255,
    0,0,0,255,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,
    0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,
    0,0,0,255,0,0,0,0,255,0,255,0,0,0,255,0,255,0,255,0,0,0,0,0,0,255,0,255,
    0,0,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,255,0,0,0,0,0,0,255,0,255,0,0,0,
    0,0,0,255,0,0,0,0,255,0,255,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,0,
    255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,0,0,
    255,0,0,0,0,255,255,255,255,255,0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,
    127,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,
    0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,255,0,0,0,255,0,0,
    0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,255,
    0,0,0,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,
    255,0,0,0,0,255,255,0,0,255,0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,0,255,
    0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,0,255,0,0,0,0,255,0,0,0,0,255,0,0,
    0,255,0,0,0,255,0,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,
    0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,
    0,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,127,255,0,0,0,0,0,255,0,255,0,0,0,
    0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,255,
    0,0,0,0,0,255,0,255,0,0,0,0,255,255,255,255,255,0,0,0,255,255,255,255,0,
    255,255,255,255,255,255,0,255,255,255,255,255,255,0,255,255,255,255,255,
    255,0,255,255,255,255,255,255,0,255,255,255,0,255,255,255,0,255,255,255,
    0,255,255,255,0,0,255,255,255,255,0,0,0,255,0,0,0,0,255,255,0,0,0,255,255,
    255,255,0,0,0,0,0,255,255,255,255,0,0,0,0,0,255,255,255,255,0,0,0,0,0,255,
    255,255,255,0,0,0,0,0,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,
    255,255,255,0,0,0,0,0,255,255,255,0,0,0,0,0,255,255,255,0,0,0,0,0,255,255,
    255,0,0,0,0,0,255,255,255,0,0,0,0,0,0,255,0,0,0,0,255,0,0,0,0,0,0,255,0,
    255,255,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,127,127,127,127,127,
    127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,
    127,127,0,127,127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,
    127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,
    127,127,0,127,127,127,127,127,127,0,127,127,127,0,127,127,127,0,127,127,
    127,0,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,
    127,0,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,
    127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,
    127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,
    127,127,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,255,0,0,0,0,0,0,255,0,0,
    0,0,255,255,0,0,0,255,255,0,255,0,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,255,255,0,0,
    0,0,0,0,0,0,0,0,255,0,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,0,
    255,0,0,255,0,0,0,0,0,0,0,0,0,255,0,0,0,0,255,255,0,0,0,0,0,255,255,0,255,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,
    0,255,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,255,0,0,0,0,255,0,0,0,0,255,
    0,0,255,0,255,0,255,255,0,0,0,255,0,0,255,0,0,255,0,0,255,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,255,0,0,
    0,255,0,0,255,0,0,0,255,0,255,0,255,0,255,255,0,255,0,0,255,0,255,0,0,0,
    255,0,255,255,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,255,0,0,255,0,0,0,255,
    0,255,255,0,0,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    255,0,0,0,0,0,0,255,0,0,0,0,255,0,0,255,0,0,0,255,0,0,255,0,0,0,0,255,0,
    0,0,255,0,0,0,0,0,0,0,255,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    127,0,255,255,255,0,0,0,255,255,255,0,0,0,255,255,255,0,0,0,255,255,255,
    0,0,0,255,255,255,0,0,0,255,255,255,0,0,0,255,255,255,0,0,255,255,0,0,0,
    0,255,255,255,255,0,0,255,255,255,255,0,0,0,255,255,255,255,0,0,0,255,255,
    255,255,0,0,0,255,255,255,255,0,0,0,255,0,255,0,0,255,0,0,255,0,0,255,255,
    0,255,0,0,255,0,255,255,255,0,0,0,255,255,255,255,0,0,0,255,255,255,255,
    0,0,0,255,255,255,255,0,0,0,255,255,255,255,0,0,0,255,255,255,255,0,0,0,
    0,0,0,255,0,0,0,0,0,0,255,255,255,255,0,0,255,0,0,0,0,255,0,255,0,0,0,0,
    255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,255,0,255,0,255,255,
    255,0,0,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,255,0,0,0,0,0,
    255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,
    255,0,0,255,0,0,255,0,0,0,0,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,
    0,0,0,255,0,255,0,0,0,0,255,0,0,255,0,255,0,0,255,0,0,255,0,0,0,0,0,0,255,
    0,255,255,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,
    0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,255,
    0,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,
    0,255,0,0,0,255,0,255,255,0,0,0,255,0,255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,127,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,
    255,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,255,0,255,0,0,0,0,0,255,0,0,0,0,
    255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,0,255,0,255,
    0,0,255,0,0,255,0,0,255,255,255,255,255,0,255,0,0,0,0,255,0,255,0,0,0,0,
    255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,
    255,0,0,0,0,0,0,0,0,0,0,255,0,0,0,255,0,255,0,255,0,0,0,0,255,0,255,0,0,
    0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,0,255,0,255,0,0,255,0,0,0,
    0,255,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,255,255,255,255,0,0,
    255,255,255,255,0,0,255,255,255,255,0,0,255,255,255,255,0,0,255,255,255,
    255,0,0,255,255,255,255,0,0,255,255,255,255,255,255,255,255,255,0,255,0,
    0,0,0,0,255,255,255,255,255,255,0,255,255,255,255,255,255,0,255,255,255,
    255,255,255,0,255,255,255,255,255,255,0,0,255,0,255,0,0,255,0,0,255,0,255,
    0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,
    0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,0,255,255,255,255,255,
    255,255,0,255,0,0,255,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,
    0,0,0,0,255,0,255,0,0,0,0,255,0,0,255,0,255,0,0,255,0,0,0,0,255,0,0,255,
    0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,255,0,0,0,255,0,255,0,0,0,255,0,255,
    0,0,0,255,0,255,0,0,0,255,0,255,0,0,0,255,0,255,0,0,0,255,0,255,0,0,0,255,
    0,0,0,0,0,0,255,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,0,255,0,0,0,0,0,
    0,255,0,0,0,0,0,0,0,255,0,255,0,0,255,0,0,255,0,255,0,0,0,0,255,0,255,0,
    0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,
    0,0,0,255,0,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,255,0,255,0,0,0,255,0,255,
    0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,0,255,
    0,255,0,0,255,0,0,0,0,255,0,0,255,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,255,
    0,0,0,255,0,255,0,0,0,255,0,255,0,0,0,255,0,255,0,0,0,255,0,255,0,0,0,255,
    0,255,0,0,0,255,0,255,0,0,0,255,255,0,0,0,255,0,255,0,0,0,0,0,255,0,0,0,
    0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,0,255,0,255,
    0,0,255,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,
    255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,255,0,0,0,0,255,0,
    0,0,0,0,255,0,0,0,0,0,255,0,0,0,255,0,0,255,0,0,0,255,255,0,255,0,0,0,255,
    255,0,255,0,0,0,255,255,0,255,0,0,0,255,255,0,0,0,255,0,0,0,255,0,0,0,0,
    255,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,255,255,255,255,0,0,255,
    255,255,255,0,0,255,255,255,255,0,0,255,255,255,255,0,0,255,255,255,255,
    0,0,255,255,255,255,0,0,255,255,255,0,0,255,255,255,0,0,0,255,255,255,255,
    0,0,255,255,255,255,0,0,0,255,255,255,255,0,0,0,255,255,255,255,0,0,0,255,
    255,255,255,0,0,0,255,0,255,0,0,255,0,0,255,0,0,255,255,255,255,0,0,255,
    0,0,0,0,255,0,0,255,255,255,255,0,0,0,255,255,255,255,0,0,0,255,255,255,
    255,0,0,0,255,255,255,255,0,0,0,255,255,255,255,0,0,0,0,0,0,255,0,0,0,0,
    0,255,255,255,255,0,0,0,0,255,255,255,0,255,0,0,255,255,255,0,255,0,0,255,
    255,255,0,255,0,0,255,255,255,0,255,0,0,0,255,0,0,0,255,255,255,255,255,
    0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,
    0,255,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,255,0,0,0,0,255,0,0,0,0,0,0,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,
    127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,
    127,127,0,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,
    127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,127,0,127,
    127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,
    0,127,127,0,127,0,127,127,0,127,127,0,127,127,127,127,127,127,0,127,127,
    127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,
    127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,
    0,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,
    127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,127,0,127,
    127,127,127,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
};

// Normal font anti-aliased
const int FONT1AA_BM_W = 264;
const int FONT1AA_BM_H = 106;
static const unsigned char s_Font1AA[] = 
{
    127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,0,0,0,0,0,0,0,0,0,
    0,0,0,4,4,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,4,4,0,0,0,0,0,0,0,0,0,0,0,0,4,4,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,
    59,241,97,206,166,0,0,0,0,0,0,0,0,0,0,0,0,0,168,34,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,251,89,0,0,89,255,125,89,255,125,0,0,0,0,
    7,199,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,166,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,0,0,0,
    0,0,0,0,0,0,0,0,0,4,4,0,0,0,0,0,0,0,0,127,0,0,0,0,0,138,225,21,59,238,42,
    206,125,0,0,0,0,7,199,34,89,166,0,0,0,0,168,34,0,0,0,175,255,255,166,0,
    0,7,202,89,0,0,0,0,59,245,255,251,89,0,0,0,59,238,34,0,12,232,89,0,0,89,
    247,34,0,59,245,206,199,124,255,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,7,202,89,0,12,235,255,247,34,0,0,0,0,12,232,89,0,0,12,235,
    255,255,251,89,0,7,206,255,255,255,125,0,0,0,0,138,251,89,0,0,59,245,255,
    255,255,251,89,0,0,89,255,255,166,0,89,255,255,255,255,255,201,0,0,59,245,
    255,255,125,0,0,12,235,255,247,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,255,255,255,247,34,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,0,0,0,0,0,0,0,0,0,0,0,0,4,4,0,0,0,
    0,0,0,0,0,127,0,0,0,0,0,138,225,21,59,238,34,175,125,0,0,0,0,59,192,0,172,
    89,0,0,59,245,255,255,251,89,89,247,34,12,228,34,0,138,166,0,0,0,0,12,235,
    125,0,175,225,21,0,0,59,238,34,0,138,201,0,0,0,0,175,166,0,0,0,89,255,201,
    0,0,0,0,0,0,7,202,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,215,21,0,175,
    166,0,138,201,0,0,7,206,255,251,89,0,0,59,192,0,0,138,247,34,59,192,0,0,
    89,251,89,0,0,59,245,251,89,0,0,59,241,89,0,0,0,0,0,89,247,34,0,0,0,0,0,
    0,0,7,206,166,0,7,206,125,0,89,247,34,7,206,166,0,138,225,21,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,12,232,89,0,0,0,0,0,0,0,0,0,0,0,175,166,0,0,0,0,0,
    0,0,89,125,0,0,175,201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,0,
    0,0,0,0,0,0,0,0,0,0,0,4,4,0,0,0,0,0,0,0,0,127,0,0,0,0,0,138,225,21,12,206,
    21,175,125,0,0,89,255,255,255,255,255,255,166,59,241,89,168,34,138,125,
    89,225,21,7,202,89,12,228,34,0,0,0,0,12,232,89,0,138,201,0,0,0,12,206,21,
    7,202,89,0,0,0,0,59,215,21,59,245,206,199,124,255,125,0,0,0,0,7,202,89,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,166,0,12,232,89,0,59,238,34,0,0,
    0,59,241,89,0,0,0,0,0,0,59,241,89,0,0,0,0,59,241,89,0,12,232,132,241,89,
    0,0,59,241,89,0,0,0,0,7,206,125,0,0,0,0,0,0,0,0,89,247,34,0,12,232,89,0,
    12,232,89,59,241,89,0,59,241,89,0,138,247,34,0,0,138,247,34,0,0,0,0,0,12,
    235,247,34,0,0,0,0,0,0,0,0,0,0,0,0,0,138,255,166,0,0,0,0,0,0,0,0,0,138,
    225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,0,0,0,0,0,0,0,0,0,0,
    0,0,4,4,0,0,0,0,0,0,0,0,127,0,0,0,0,0,138,225,21,0,0,0,0,0,0,0,0,0,172,
    89,59,192,0,0,59,238,34,168,34,0,0,89,247,34,12,228,34,138,166,0,0,0,0,
    0,0,138,251,159,247,34,0,0,0,0,0,0,59,238,34,0,0,0,0,7,202,89,0,0,7,199,
    34,0,0,0,0,0,0,7,202,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,202,89,0,59,
    241,89,0,59,241,89,0,0,0,59,241,89,0,0,0,0,0,0,89,247,34,0,0,0,0,138,201,
    0,7,206,125,59,241,89,0,0,59,245,255,255,251,89,0,12,235,255,255,255,125,
    0,0,0,0,7,206,166,0,0,0,175,251,89,138,201,0,59,241,89,0,12,235,125,0,138,
    247,34,0,0,138,247,34,0,0,0,59,245,247,34,0,0,0,0,7,206,255,255,255,255,
    255,255,125,0,0,0,0,138,255,201,0,0,0,0,0,0,89,251,89,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,4,4,0,0,0,0,0,0,0,0,0,0,0,0,4,4,0,0,0,0,0,0,0,0,
    127,0,0,0,0,0,138,201,0,0,0,0,0,0,0,0,0,12,206,21,138,125,0,0,7,206,255,
    247,34,0,0,0,175,255,255,166,59,215,21,175,255,255,125,0,0,138,171,206,
    166,0,175,201,0,0,0,0,89,201,0,0,0,0,0,0,175,125,0,0,0,0,0,0,0,0,12,235,
    255,255,255,255,255,255,125,0,0,0,0,138,255,255,251,89,0,0,0,0,0,59,215,
    21,0,59,241,89,0,59,241,89,0,0,0,59,241,89,0,0,0,0,0,12,235,166,0,0,0,138,
    255,255,125,0,175,201,0,59,241,89,0,0,0,0,0,0,175,247,34,59,241,89,0,89,
    247,34,0,0,0,89,247,34,0,0,0,89,255,255,255,125,0,12,235,166,0,59,245,125,
    0,0,0,0,0,0,0,0,0,0,0,175,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,89,251,89,0,0,7,206,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,4,4,0,0,0,0,0,0,0,0,0,0,0,0,4,4,0,0,0,0,0,0,0,0,127,0,0,0,0,0,89,201,
    0,0,0,0,0,0,0,12,235,255,255,255,255,255,225,21,0,0,0,175,255,251,89,0,
    0,0,0,0,175,125,89,225,21,59,238,34,89,225,21,12,235,166,175,166,0,0,0,
    0,89,201,0,0,0,0,0,0,175,125,0,0,0,0,0,0,0,0,0,0,0,7,202,89,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,138,166,0,0,59,241,89,0,59,241,89,0,0,0,59,241,89,
    0,0,0,0,12,235,166,0,0,0,0,0,0,59,241,97,206,255,255,255,255,255,125,0,
    0,0,0,0,59,241,89,59,238,34,0,12,235,125,0,0,12,235,125,0,0,0,12,232,89,
    0,59,245,125,0,89,255,255,232,241,89,0,0,0,0,0,0,0,0,0,0,0,0,59,245,247,
    34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,255,201,0,0,0,0,7,206,125,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,0,0,0,0,0,0,0,0,0,0,0,0,4,
    4,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,125,12,206,21,
    0,0,0,0,0,168,34,175,166,0,0,0,0,59,215,21,138,201,0,12,228,34,138,225,
    21,0,12,235,251,89,0,0,0,0,59,215,21,0,0,0,0,12,232,89,0,0,0,0,0,0,0,0,
    0,0,0,7,202,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,202,89,0,0,12,232,89,0,
    59,238,34,0,0,0,59,241,89,0,0,0,12,235,166,0,0,0,0,0,0,0,12,235,125,0,0,
    0,59,241,89,0,0,0,0,0,0,59,241,89,12,232,89,0,12,232,89,0,0,138,225,21,
    0,0,0,59,238,34,0,7,206,166,0,0,0,0,89,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,12,235,247,34,0,0,7,206,255,255,255,255,255,255,125,0,0,138,255,166,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,0,0,0,
    0,0,0,0,0,0,0,0,0,4,4,0,0,0,0,0,0,0,0,127,0,0,0,0,0,138,225,21,0,0,0,0,
    0,0,0,0,172,89,89,166,0,0,0,89,166,0,168,42,206,125,0,0,0,7,202,89,0,89,
    225,21,59,238,34,89,251,89,0,0,175,255,201,0,0,0,0,7,202,89,0,0,0,0,59,
    215,21,0,0,0,0,0,0,0,0,0,0,0,7,202,89,0,0,0,0,138,247,34,0,0,0,0,0,7,206,
    201,0,12,228,34,0,0,0,175,166,0,138,201,0,0,0,0,59,241,89,0,0,12,235,166,
    0,0,0,0,89,166,0,0,89,251,89,0,0,0,59,241,89,0,0,59,192,0,0,175,225,21,
    0,175,201,0,138,225,21,0,12,235,125,0,0,0,0,12,235,166,0,59,241,89,0,0,
    0,7,206,166,0,0,138,247,34,0,0,59,245,125,0,0,0,0,0,0,0,12,232,89,0,0,0,
    0,0,0,0,0,0,0,0,175,166,0,0,0,0,0,0,0,0,7,206,166,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,4,4,0,0,0,0,0,0,0,0,0,0,0,0,4,4,0,0,0,0,0,0,0,0,
    127,0,0,0,0,0,138,225,21,0,0,0,0,0,0,0,12,206,21,138,125,0,0,0,12,235,255,
    255,255,166,0,0,0,0,138,201,0,0,0,175,255,255,125,0,0,138,255,255,255,125,
    12,235,247,0,0,0,0,138,201,0,0,0,0,175,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,7,206,166,0,0,0,0,0,0,7,206,201,0,89,201,0,0,0,0,12,235,255,247,
    34,0,0,7,206,255,255,255,225,21,89,255,255,255,255,255,166,59,245,255,255,
    251,89,0,0,0,0,59,241,89,0,0,12,235,255,255,225,21,0,0,12,235,255,251,89,
    0,0,175,225,21,0,0,0,0,0,59,245,255,255,125,0,0,89,255,255,166,0,0,0,138,
    247,34,0,0,138,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,7,206,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,
    0,0,0,0,0,0,0,0,0,0,0,0,4,4,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,168,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,12,232,89,0,0,59,238,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,59,241,89,0,0,0,0,0,0,0,0,0,0,175,125,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,125,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,0,0,0,0,0,0,0,0,0,0,0,0,
    4,4,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,168,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,
    255,125,89,255,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,201,0,0,0,0,
    0,0,0,0,0,0,12,228,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,228,34,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,0,0,0,0,0,0,0,0,0,
    127,127,127,127,0,127,127,0,127,127,127,127,127,0,127,127,127,127,127,127,
    127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,
    127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,0,127,127,127,127,
    0,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,
    127,127,0,127,127,127,0,127,127,127,127,0,127,127,127,0,127,127,127,127,
    0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,
    127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,0,
    127,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,127,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,127,127,
    127,127,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,255,255,125,138,166,0,0,0,89,255,255,247,
    34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,89,255,255,
    255,255,166,0,0,0,0,0,12,235,225,21,0,0,59,245,255,255,255,251,89,0,0,0,
    59,245,255,255,251,89,59,245,255,255,255,247,34,0,0,59,245,255,255,255,
    255,127,81,245,255,255,255,255,127,0,0,59,245,255,255,255,166,0,59,241,
    89,0,0,0,59,241,89,89,255,255,255,125,7,206,255,251,89,59,241,89,0,0,89,
    255,166,59,241,89,0,0,0,0,59,245,225,21,0,0,7,206,251,89,59,245,247,34,
    0,0,59,241,89,0,0,138,255,255,255,166,0,0,59,245,255,255,255,225,21,0,0,
    0,138,255,255,255,166,0,0,59,245,255,255,255,251,89,0,0,0,59,245,255,255,
    201,89,255,255,255,255,255,255,255,125,59,241,89,0,0,0,59,241,97,206,166,
    0,0,0,0,175,201,175,201,0,0,7,206,201,0,0,0,175,171,206,225,21,0,0,59,245,
    166,245,125,0,0,0,89,251,89,89,255,255,255,255,255,127,0,228,34,0,0,59,
    215,21,0,0,0,0,12,228,34,0,0,0,59,245,201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,127,0,0,175,225,21,0,0,0,175,225,21,0,0,0,89,232,241,89,0,0,59,
    241,89,0,0,138,225,21,0,89,255,125,0,0,59,192,59,241,89,0,0,175,251,89,
    0,59,241,89,0,0,0,0,59,241,89,0,0,0,0,0,89,255,125,0,0,7,199,34,59,241,
    89,0,0,0,59,241,89,0,59,241,89,0,0,0,59,241,89,59,241,89,0,59,241,89,0,
    59,241,89,0,0,0,0,59,245,255,125,0,0,89,255,251,89,59,245,255,201,0,0,59,
    241,89,0,138,251,89,0,12,235,166,0,59,241,89,0,7,206,225,21,0,138,251,89,
    0,12,235,166,0,59,241,89,0,0,138,247,34,0,12,235,125,0,7,176,21,0,0,59,
    241,89,0,0,0,59,241,89,0,0,0,59,241,89,138,225,21,0,0,12,235,125,89,225,
    21,0,59,245,247,34,0,12,232,89,12,235,166,0,7,206,166,0,89,247,34,0,7,206,
    125,0,0,0,0,0,7,206,166,12,228,34,0,0,7,202,89,0,0,0,0,12,228,34,0,0,12,
    235,133,206,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,138,201,0,138,
    255,255,255,125,138,166,0,0,7,206,166,175,166,0,0,59,241,89,0,0,89,247,
    34,7,206,166,0,0,0,0,0,59,241,89,0,0,0,175,225,21,59,241,89,0,0,0,0,59,
    241,89,0,0,0,0,7,206,166,0,0,0,0,0,0,59,241,89,0,0,0,59,241,89,0,59,241,
    89,0,0,0,59,241,89,59,241,89,59,241,89,0,0,59,241,89,0,0,0,0,59,241,159,
    225,21,0,175,166,241,89,59,241,132,241,89,0,59,241,89,12,235,166,0,0,0,
    89,247,34,59,241,89,0,0,89,247,34,12,235,166,0,0,0,89,247,34,59,241,89,
    0,0,59,241,89,0,59,238,34,0,0,0,0,0,0,59,241,89,0,0,0,59,241,89,0,0,0,59,
    241,89,59,241,89,0,0,89,225,21,59,241,89,0,89,206,202,89,0,59,238,34,0,
    89,251,89,138,225,21,0,0,175,201,0,138,225,21,0,0,0,0,0,175,225,21,12,228,
    34,0,0,0,138,166,0,0,0,0,12,228,34,0,7,206,166,0,12,235,125,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,127,7,202,89,89,225,21,7,206,125,12,206,21,0,59,238,
    34,89,247,34,0,59,241,89,0,0,175,201,0,59,241,89,0,0,0,0,0,59,241,89,0,
    0,0,59,241,89,59,241,89,0,0,0,0,59,241,89,0,0,0,0,59,241,89,0,0,0,0,0,0,
    59,241,89,0,0,0,59,241,89,0,59,241,89,0,0,0,59,241,89,59,241,102,232,89,
    0,0,0,59,241,89,0,0,0,0,59,241,102,232,89,59,215,81,241,89,59,241,89,138,
    225,21,59,241,89,59,241,89,0,0,0,59,241,89,59,241,89,0,7,206,201,0,59,241,
    89,0,0,0,59,241,89,59,241,89,0,0,175,201,0,0,12,235,166,0,0,0,0,0,0,59,
    241,89,0,0,0,59,241,89,0,0,0,59,241,89,7,206,166,0,0,175,166,0,7,206,125,
    0,175,125,175,166,0,138,201,0,0,0,175,255,251,89,0,0,0,59,245,166,241,89,
    0,0,0,0,0,89,247,34,0,12,228,34,0,0,0,89,201,0,0,0,0,12,228,34,12,235,201,
    0,0,0,59,245,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,59,215,21,175,125,
    0,7,206,125,7,199,34,0,138,201,0,12,235,125,0,59,245,255,255,255,247,34,
    0,59,241,89,0,0,0,0,0,59,241,89,0,0,0,59,241,89,59,245,255,255,255,255,
    127,59,245,255,255,255,255,127,59,241,89,0,0,0,0,0,0,59,245,255,255,255,
    255,255,251,89,0,59,241,89,0,0,0,59,241,89,59,245,255,247,34,0,0,0,59,241,
    89,0,0,0,0,59,241,89,138,201,175,166,59,241,89,59,241,89,12,235,125,59,
    241,89,59,241,89,0,0,0,12,235,125,59,245,255,255,255,201,0,0,59,241,89,
    0,0,0,12,235,125,59,245,255,255,255,125,0,0,0,0,59,245,255,255,125,0,0,
    0,59,241,89,0,0,0,59,241,89,0,0,0,59,241,89,0,138,225,21,59,241,89,0,0,
    175,201,7,202,89,89,201,0,175,166,0,0,0,12,235,166,0,0,0,0,0,138,255,166,
    0,0,0,0,0,59,245,125,0,0,12,228,34,0,0,0,12,228,34,0,0,0,12,228,34,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,59,215,21,175,125,0,7,
    206,125,7,199,34,7,206,125,0,0,175,201,0,59,241,89,0,0,89,247,34,59,241,
    89,0,0,0,0,0,59,241,89,0,0,0,59,241,89,59,241,89,0,0,0,0,59,241,89,0,0,
    0,0,59,241,89,0,59,245,255,251,89,59,241,89,0,0,0,59,241,89,0,59,241,89,
    0,0,0,59,241,89,59,241,89,175,225,21,0,0,59,241,89,0,0,0,0,59,241,89,12,
    235,247,34,59,241,89,59,241,89,0,89,247,94,241,89,59,241,89,0,0,0,59,241,
    89,59,241,89,0,0,0,0,0,59,241,89,0,0,0,59,241,89,59,241,89,12,235,166,0,
    0,0,0,0,0,0,138,251,89,0,0,59,241,89,0,0,0,59,241,89,0,0,0,59,241,89,0,
    12,232,89,138,225,21,0,0,89,225,81,215,21,12,228,47,232,89,0,0,0,175,255,
    251,89,0,0,0,0,59,241,89,0,0,0,0,7,206,201,0,0,0,12,228,34,0,0,0,0,175,
    125,0,0,0,12,228,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,
    12,228,34,89,201,0,7,206,125,59,215,21,59,245,255,255,255,255,247,34,59,
    241,89,0,0,59,241,89,7,206,166,0,0,0,0,0,59,241,89,0,0,0,138,225,21,59,
    241,89,0,0,0,0,59,241,89,0,0,0,0,7,206,166,0,0,0,59,241,89,59,241,89,0,
    0,0,59,241,89,0,59,241,89,0,0,0,59,241,89,59,241,89,7,206,201,0,0,59,241,
    89,0,0,0,0,59,241,89,0,175,166,0,59,241,89,59,241,89,0,7,206,200,241,89,
    12,235,166,0,0,0,89,247,34,59,241,89,0,0,0,0,0,12,235,166,0,0,0,89,247,
    34,59,241,89,0,59,245,125,0,0,0,0,0,0,12,232,89,0,0,59,241,89,0,0,0,12,
    232,89,0,0,0,59,238,34,0,0,175,171,206,166,0,0,0,12,232,159,201,0,7,202,
    132,215,21,0,0,89,247,34,175,225,21,0,0,0,59,241,89,0,0,0,0,138,225,21,
    0,0,0,12,228,34,0,0,0,0,89,201,0,0,0,12,228,34,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,138,201,7,206,255,251,226,255,255,166,0,
    138,201,0,0,0,12,235,125,59,241,89,0,0,138,247,34,0,89,255,125,0,0,59,192,
    59,241,89,0,0,138,251,89,0,59,241,89,0,0,0,0,59,241,89,0,0,0,0,0,89,255,
    125,0,0,59,241,89,59,241,89,0,0,0,59,241,89,0,59,241,89,0,0,0,89,247,34,
    59,241,89,0,12,235,166,0,59,241,89,0,0,0,0,59,241,89,0,0,0,0,59,241,89,
    59,241,89,0,0,59,245,251,89,0,138,251,89,0,59,245,166,0,59,241,89,0,0,0,
    0,0,0,138,251,89,0,59,245,166,0,59,241,89,0,0,138,251,89,0,89,166,0,0,89,
    247,34,0,0,59,241,89,0,0,0,0,138,225,21,0,7,206,166,0,0,0,89,255,251,89,
    0,0,0,7,206,255,125,0,0,138,255,201,0,0,12,235,125,0,12,235,166,0,0,0,59,
    241,89,0,0,0,89,251,89,0,0,0,0,12,228,34,0,0,0,0,12,228,34,0,0,12,228,34,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,7,206,225,21,0,
    0,0,0,0,0,7,206,125,0,0,0,0,175,201,59,245,255,255,255,247,34,0,0,0,59,
    245,255,255,251,89,59,245,255,255,255,225,21,0,0,59,245,255,255,255,255,
    127,81,241,89,0,0,0,0,0,0,59,245,255,255,255,201,0,59,241,89,0,0,0,59,241,
    89,89,255,255,255,138,235,255,255,125,0,59,241,89,0,0,89,255,201,59,245,
    255,255,255,255,166,59,241,89,0,0,0,0,59,241,89,59,241,89,0,0,0,175,251,
    89,0,0,138,255,255,255,166,0,0,59,241,89,0,0,0,0,0,0,0,138,255,255,255,
    166,0,0,59,241,89,0,0,0,175,251,89,12,235,255,255,251,89,0,0,0,59,241,89,
    0,0,0,0,0,59,245,255,251,89,0,0,0,0,12,235,201,0,0,0,0,0,138,251,89,0,0,
    89,255,125,0,7,206,225,21,0,0,89,255,125,0,0,59,241,89,0,0,0,175,255,255,
    255,255,255,127,0,228,34,0,0,0,0,0,175,125,0,0,12,228,34,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,89,255,255,255,255,125,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,241,89,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,
    228,34,0,0,0,0,0,89,201,0,0,12,228,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,138,255,255,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,255,255,125,0,0,0,12,228,
    124,255,255,247,34,0,0,0,0,0,0,0,0,0,245,255,255,255,255,255,255,0,0,0,
    0,0,0,0,0,0,0,127,127,127,127,127,127,127,127,127,127,127,0,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,
    127,0,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,
    127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,127,127,127,
    127,127,127,0,127,127,127,127,0,127,127,127,127,0,127,127,127,127,127,127,
    127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,0,127,
    127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,
    127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,0,127,
    127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,4,4,4,4,4,4,4,4,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,127,0,89,255,125,0,0,0,0,0,0,0,0,0,0,59,241,89,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,59,241,89,0,0,0,0,0,0,0,0,89,255,255,166,0,0,0,0,0,
    0,0,59,241,89,0,0,0,0,0,0,0,0,0,0,0,59,241,89,0,0,0,0,59,241,89,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,255,201,0,12,228,
    34,0,0,89,255,247,34,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,4,4,4,4,4,116,116,
    4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,127,0,0,59,241,89,0,0,0,0,0,0,0,0,0,59,241,89,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,59,241,89,0,0,0,0,0,0,0,12,235,125,0,0,0,0,0,0,
    0,0,0,59,241,89,0,0,0,0,59,241,89,0,89,251,89,59,241,89,0,0,0,0,59,241,
    89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,241,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,225,
    21,0,0,12,228,34,0,0,0,0,138,201,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,4,4,
    4,4,28,244,252,52,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,
    241,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,241,89,0,0,0,0,0,0,0,59,241,89,0,
    0,0,0,0,0,0,0,0,59,241,89,0,0,0,0,0,0,0,0,0,0,0,59,241,89,0,0,0,0,59,241,
    89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,241,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,166,
    0,0,0,12,228,34,0,0,0,0,89,225,21,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,4,4,4,
    4,180,252,164,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,7,206,255,255,255,
    125,0,59,241,194,255,251,89,0,0,7,206,255,255,201,0,12,235,255,255,251,
    89,0,12,235,255,251,89,7,206,255,255,247,34,0,12,235,255,255,251,89,59,
    241,194,255,255,125,0,59,241,89,89,255,251,89,59,241,89,0,138,251,89,59,
    241,89,59,241,159,255,255,125,89,255,255,166,0,59,241,194,255,255,125,0,
    0,0,12,235,255,247,34,0,59,241,194,255,255,125,0,0,12,235,255,255,251,89,
    59,241,159,255,201,0,138,255,255,247,34,206,255,255,255,166,59,241,89,0,
    59,241,97,206,166,0,0,12,235,125,175,201,0,7,206,166,0,7,206,133,206,225,
    21,0,89,255,255,166,0,0,12,235,125,138,255,255,255,255,166,0,0,138,166,
    0,0,0,12,228,34,0,0,0,0,89,225,21,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,4,4,4,
    76,252,244,20,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,89,247,34,
    59,245,166,0,138,225,21,7,206,201,0,0,0,7,206,166,0,59,241,89,7,206,125,
    0,89,225,21,59,241,89,0,0,7,206,166,0,59,241,89,59,245,166,0,89,247,34,
    59,241,89,0,59,241,89,59,241,89,138,225,21,0,59,241,89,59,245,201,0,89,
    255,201,0,89,247,34,59,245,166,0,89,247,34,0,7,206,166,0,138,225,21,59,
    245,166,0,138,247,34,7,206,166,0,59,241,89,59,245,201,0,0,59,238,34,0,130,
    34,59,241,89,0,0,59,241,89,0,59,241,89,89,247,34,0,89,247,34,138,225,21,
    12,235,225,21,12,232,89,7,206,166,12,235,125,89,247,34,0,89,247,34,0,0,
    0,89,247,34,0,0,138,166,0,0,0,12,228,34,0,0,0,0,89,225,21,0,0,7,206,247,
    34,0,0,89,201,0,0,4,4,68,12,4,4,4,220,252,108,4,4,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,
    0,0,0,0,0,0,0,0,0,0,0,59,241,89,59,241,89,0,59,241,89,59,241,89,0,0,0,59,
    241,89,0,59,241,89,59,238,34,0,59,238,34,59,241,89,0,0,59,241,89,0,59,241,
    89,59,241,89,0,59,241,89,59,241,89,0,59,241,89,59,241,159,201,0,0,0,59,
    241,89,59,241,89,0,59,241,89,0,59,241,89,59,241,89,0,59,241,89,0,59,241,
    89,0,59,241,89,59,241,89,0,59,241,89,59,241,89,0,59,241,89,59,241,89,0,
    0,59,241,89,0,0,0,59,241,89,0,0,59,241,89,0,59,241,89,12,235,125,0,175,
    166,0,59,238,34,89,171,202,89,89,225,21,0,59,241,226,201,0,12,235,125,0,
    175,166,0,0,0,12,235,125,0,0,59,238,34,0,0,0,12,228,34,0,0,0,0,7,206,125,
    0,7,202,89,12,235,166,0,175,125,0,0,4,60,244,172,4,4,132,252,212,4,4,4,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,89,255,255,255,251,89,59,241,89,
    0,59,241,89,59,238,34,0,0,0,59,238,34,0,59,241,89,59,245,255,255,255,251,
    0,59,241,89,0,0,59,238,34,0,59,241,89,59,241,89,0,59,241,89,59,241,89,0,
    59,241,89,59,245,255,225,21,0,0,59,241,89,59,241,89,0,59,241,89,0,59,241,
    89,59,241,89,0,59,241,89,0,59,238,34,0,12,232,89,59,241,89,0,12,232,89,
    59,238,34,0,59,241,89,59,241,89,0,0,0,175,255,255,201,0,59,241,89,0,0,59,
    241,89,0,59,241,89,0,175,201,12,232,89,0,7,206,125,172,89,138,166,138,201,
    0,0,0,138,247,34,0,0,175,201,12,232,89,0,0,7,206,166,0,0,175,225,21,0,0,
    0,0,12,228,34,0,0,0,0,0,0,175,225,34,206,21,0,0,175,255,166,0,0,0,4,52,
    244,252,140,36,244,252,60,4,4,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,59,
    241,89,0,59,241,89,59,241,89,0,59,238,34,59,241,89,0,0,0,59,241,89,0,59,
    241,89,59,238,34,0,0,0,0,59,241,89,0,0,59,241,89,0,59,241,89,59,241,89,
    0,59,241,89,59,241,89,0,59,241,89,59,241,97,206,201,0,0,59,241,89,59,241,
    89,0,59,241,89,0,59,241,89,59,241,89,0,59,241,89,0,59,241,89,0,59,241,89,
    59,241,89,0,59,241,89,59,241,89,0,59,241,89,59,241,89,0,0,0,0,0,59,245,
    125,59,241,89,0,0,59,241,89,0,59,241,89,0,59,238,124,225,21,0,0,175,176,
    206,21,59,215,187,125,0,0,59,245,255,201,0,0,89,247,124,225,21,0,0,138,
    225,21,0,0,0,59,241,89,0,0,0,12,228,34,0,0,0,0,12,235,125,0,0,0,0,0,0,0,
    0,0,0,0,0,4,4,76,252,252,220,252,164,4,4,4,4,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,
    0,0,0,0,0,0,89,247,34,0,89,251,89,59,241,89,0,175,201,0,7,206,201,0,0,0,
    7,206,166,0,138,251,89,7,206,166,0,7,199,34,59,241,89,0,0,7,206,166,0,138,
    251,89,59,241,89,0,59,241,89,59,241,89,0,59,241,89,59,241,89,12,235,166,
    0,59,241,89,59,241,89,0,59,241,89,0,59,241,89,59,241,89,0,59,241,89,0,7,
    206,166,0,138,225,21,59,241,89,0,138,225,21,7,206,166,0,89,251,89,59,241,
    89,0,0,89,125,0,12,232,89,12,232,89,0,12,12,235,125,0,175,251,89,0,7,206,
    255,125,0,0,0,89,255,201,0,7,206,247,34,0,7,206,166,59,245,125,0,7,206,
    255,125,0,0,59,241,89,0,0,0,0,0,138,166,0,0,0,12,228,34,0,0,0,0,89,225,
    21,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,100,252,252,244,28,4,4,4,4,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,127,0,0,0,0,0,0,0,0,175,255,255,232,241,89,59,245,255,255,247,34,
    0,0,12,235,255,255,201,0,59,245,255,200,241,89,0,12,235,255,251,89,0,59,
    241,89,0,0,0,12,235,255,200,241,89,59,241,89,0,59,241,89,59,241,89,0,59,
    241,89,59,241,89,0,59,245,201,59,241,89,59,241,89,0,59,241,89,0,59,241,
    89,59,241,89,0,59,241,89,0,0,12,235,255,247,34,0,59,245,166,255,247,34,
    0,0,59,245,255,166,241,89,59,241,89,0,0,59,245,255,255,166,0,0,138,255,
    255,125,0,89,255,255,166,241,89,0,0,138,247,34,0,0,0,59,245,125,0,0,138,
    225,21,7,206,225,21,0,138,251,0,0,138,247,34,0,0,175,255,255,255,255,166,
    0,0,138,166,0,0,0,12,228,34,0,0,0,0,89,225,21,0,0,0,0,0,0,0,0,0,0,0,0,4,
    4,4,4,132,252,108,4,4,4,4,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,138,225,21,0,0,0,0,0,0,0,0,0,0,0,89,247,34,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,241,89,0,0,0,0,0,
    0,0,0,59,241,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,232,89,0,0,0,0,0,0,0,0,0,0,0,138,
    201,0,0,0,12,228,34,0,0,0,0,138,201,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,4,
    116,4,4,4,4,4,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,166,255,255,
    247,34,0,0,0,0,0,0,0,0,0,0,0,255,255,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,241,89,0,0,0,0,0,0,0,0,59,
    241,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,225,21,0,0,0,0,0,0,0,0,0,0,0,7,206,255,
    201,0,12,228,34,0,0,89,255,251,89,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,4,4,
    4,4,4,4,4,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,127,127,127,127,127,0,127,127,127,
    127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,
    127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,0,127,127,127,127,
    127,127,0,127,127,127,127,127,127,0,127,127,0,127,127,127,0,127,127,127,
    127,127,127,0,127,127,0,127,127,127,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,
    0,127,127,127,127,127,127,0,127,127,127,127,0,127,127,127,127,127,0,127,
    127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,
    127,127,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,
    127,0,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,0,127,
    127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,127,
    127,127,127,127,127,127,127,127,127,127,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,89,247,34,138,201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,125,59,238,34,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,255,
    225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    12,235,251,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,238,34,138,201,0,0,0,0,0,
    0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,199,34,0,0,0,0,0,7,199,34,0,0,0,0,138,255,
    201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,251,89,0,0,138,255,251,97,206,201,0,0,138,
    251,102,235,201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,255,
    201,12,228,34,0,0,0,0,0,0,0,0,0,0,0,0,7,206,166,12,232,89,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,166,12,235,127,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,127,0,0,175,255,255,255,225,21,59,245,255,255,255,
    255,255,125,0,0,0,0,0,0,0,7,206,255,247,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,7,199,34,0,0,0,0,0,7,199,34,0,0,0,138,225,21,175,166,0,0,175,255,255,
    166,0,0,7,202,89,0,0,0,0,0,0,0,0,0,0,59,245,255,255,201,0,0,0,0,0,0,0,0,
    59,245,255,255,255,255,255,255,255,255,125,0,59,245,255,255,255,255,255,
    125,0,0,89,255,255,255,255,255,225,21,59,245,255,255,255,255,255,125,0,
    0,0,59,245,255,255,255,255,255,125,7,206,166,0,0,175,171,206,166,89,247,
    34,0,175,201,59,241,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    12,228,34,175,255,125,0,0,89,255,255,255,125,175,251,89,89,255,125,0,7,
    206,255,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,245,255,255,255,
    255,255,125,0,0,7,206,255,125,59,245,125,0,0,0,89,251,89,0,0,0,0,0,0,0,
    127,7,206,225,21,0,0,0,0,59,115,0,0,0,0,59,115,0,0,0,0,0,0,0,175,201,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,245,255,255,255,255,125,0,59,245,255,
    255,255,255,125,0,0,0,0,0,0,0,89,247,34,12,228,34,0,138,166,0,0,0,0,0,0,
    0,0,0,0,12,235,125,0,7,176,21,0,0,0,0,0,0,138,251,89,0,0,138,201,0,0,0,
    0,0,0,59,115,0,0,0,0,59,115,0,0,0,0,0,0,7,206,166,0,59,115,0,0,0,0,59,115,
    0,0,0,59,115,0,0,0,0,59,115,0,89,201,0,12,232,89,89,201,7,202,89,12,232,
    89,138,201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,7,199,34,0,172,132,196,199,163,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,59,115,0,0,0,0,59,115,0,0,0,0,0,0,0,89,247,34,0,7,
    206,125,0,0,0,0,0,0,0,0,127,89,247,34,0,0,0,0,0,59,115,0,0,0,0,59,115,0,
    0,0,0,0,0,7,206,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,199,34,0,
    0,0,0,0,7,199,34,0,0,0,0,0,0,0,0,0,89,225,21,7,202,89,12,228,34,0,0,0,0,
    0,0,0,0,0,0,59,238,34,0,0,0,0,0,0,0,130,34,59,241,89,0,0,0,138,201,0,0,
    0,0,0,0,59,115,0,0,0,0,59,115,0,0,0,0,0,0,175,225,21,0,59,115,0,0,0,0,59,
    115,0,0,0,59,115,0,0,0,0,59,115,0,12,228,34,59,192,0,12,228,34,138,166,
    59,215,21,175,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,7,199,34,0,172,89,175,166,138,125,0,138,255,255,247,34,12,
    146,0,0,0,0,89,255,255,255,125,12,235,255,255,125,0,0,0,59,115,0,0,0,0,
    59,115,0,138,255,255,255,255,127,0,175,201,0,138,225,21,0,0,0,0,0,0,0,0,
    127,245,255,255,255,255,255,125,0,59,115,0,0,0,0,59,115,0,0,0,0,0,0,12,
    232,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,199,34,0,0,0,0,0,7,199,
    34,0,0,0,0,0,0,0,0,0,89,247,34,12,228,34,138,166,0,0,0,0,0,0,0,0,0,0,0,
    12,235,166,0,0,0,0,0,0,175,225,21,138,225,21,0,0,0,138,201,0,0,0,0,0,0,
    59,115,0,0,0,0,59,115,0,0,0,0,0,89,247,34,0,0,59,115,0,0,0,0,59,115,0,0,
    0,59,115,0,0,0,0,59,115,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,255,166,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,199,34,0,172,
    89,0,0,138,125,59,238,34,0,130,34,7,206,201,0,0,59,241,89,0,12,235,255,
    125,0,59,241,89,0,0,59,115,0,0,0,0,59,115,0,0,0,0,89,247,34,0,59,245,166,
    241,89,0,0,0,0,0,0,0,0,0,127,138,225,21,0,0,0,0,0,59,115,0,0,0,0,59,115,
    0,0,0,0,0,89,255,255,255,247,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,199,
    34,0,0,0,0,0,7,199,34,0,0,0,0,0,0,0,0,0,0,175,255,255,166,59,215,21,175,
    255,255,125,0,89,255,255,201,0,0,0,59,245,255,255,125,0,12,235,166,0,0,
    138,225,21,0,0,0,138,255,255,255,255,247,34,0,59,115,0,0,0,0,59,115,0,0,
    0,0,59,245,125,0,0,0,59,115,0,0,0,0,59,115,0,0,0,59,115,0,0,0,0,59,115,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,245,255,251,102,0,255,255,255,255,
    255,0,245,255,255,255,255,255,255,255,255,255,255,127,21,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,59,241,89,0,0,0,0,0,138,247,34,138,201,0,0,0,175,
    201,0,0,0,175,166,0,0,59,115,0,0,0,0,59,115,0,0,0,12,235,125,0,0,0,138,
    255,166,0,0,0,0,0,0,0,0,0,0,127,245,255,255,255,255,225,21,0,59,115,0,0,
    0,0,59,115,0,0,0,0,0,0,89,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,7,199,34,0,0,0,59,245,255,255,255,255,125,0,0,0,0,0,0,0,0,0,0,0,0,175,
    125,89,225,21,59,238,47,232,89,7,206,125,0,0,0,0,0,138,251,89,12,235,166,
    0,0,138,225,21,0,0,0,138,201,0,0,0,0,0,0,59,115,0,0,0,0,59,115,0,0,0,7,
    206,201,0,0,0,0,59,115,0,0,0,0,59,115,0,0,0,59,115,0,0,0,0,59,115,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,245,255,251,89,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,255,255,201,0,
    0,0,138,247,34,175,201,0,0,0,138,255,255,255,255,255,166,0,0,59,115,0,0,
    0,0,59,115,0,0,7,206,166,0,0,0,0,59,241,89,0,0,0,0,0,0,0,0,0,0,127,89,251,
    89,0,0,0,0,0,59,115,0,0,0,0,59,115,0,0,0,0,0,0,138,201,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,7,199,34,0,0,0,0,0,7,199,34,0,0,0,0,0,0,0,0,0,
    0,0,0,0,59,215,21,138,201,0,12,228,47,228,34,0,175,166,0,0,0,0,0,12,232,
    89,0,0,175,225,21,59,241,89,0,0,0,138,201,0,0,0,0,0,0,59,115,0,0,0,0,59,
    115,0,0,0,138,225,21,0,0,0,0,59,115,0,0,0,0,59,115,0,0,0,59,115,0,0,0,0,
    59,115,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,255,166,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,
    245,125,7,206,201,0,0,138,201,0,0,0,175,201,0,0,0,0,0,0,0,59,115,0,0,0,
    0,59,115,0,0,138,225,21,0,0,0,0,59,241,89,0,0,0,0,0,0,0,0,0,0,127,7,206,
    247,34,0,0,0,0,59,115,0,0,0,0,59,115,0,59,245,125,0,0,175,166,0,0,0,59,
    245,125,175,225,29,206,166,0,89,247,34,7,206,166,0,0,0,7,199,34,0,0,0,0,
    0,7,199,34,0,0,0,0,0,0,0,0,0,0,0,0,7,202,89,0,89,225,21,59,238,47,232,89,
    7,206,125,0,89,166,0,0,89,247,34,0,0,0,130,34,0,138,255,125,0,0,138,201,
    0,0,0,0,0,0,59,115,0,0,0,0,59,115,0,0,89,251,89,0,0,0,0,0,59,115,0,0,0,
    0,59,115,0,0,0,59,115,0,0,0,0,59,115,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,89,125,0,12,232,89,12,146,0,0,0,59,241,89,0,12,235,247,
    34,0,0,89,125,0,0,59,115,0,0,0,0,59,115,0,59,241,89,0,0,0,0,0,59,241,89,
    0,0,0,0,0,0,0,0,0,0,127,0,0,175,255,255,255,225,21,59,245,255,255,255,255,
    255,125,0,138,225,21,0,12,235,125,0,0,0,138,225,34,235,125,7,206,166,0,
    89,247,34,7,206,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    138,201,0,0,0,175,255,255,125,0,89,255,255,201,0,0,12,235,255,255,251,89,
    0,0,0,0,0,0,0,0,89,255,255,255,255,255,255,255,255,255,125,0,59,245,255,
    255,255,255,255,125,0,0,175,255,255,255,255,255,247,34,59,245,255,255,255,
    255,255,125,0,0,0,59,245,255,255,255,255,255,125,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,245,255,255,166,0,0,0,0,0,0,0,89,255,255,
    255,125,59,245,255,255,201,0,0,0,59,245,255,255,255,255,255,125,0,175,255,
    255,255,255,127,0,0,59,241,89,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,175,166,0,255,255,201,0,0,0,0,175,166,59,238,34,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,228,34,0,0,
    0,0,0,0,0,12,228,34,138,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,0,127,127,127,
    127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,
    127,127,0,127,127,127,127,127,127,127,127,127,127,127,127,127,127,127,127,
    127,127,0,127,127,127,127,127,127,127,0,127,127,127,0,127,127,127,127,127,
    127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,0,
    127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,0,127,
    127,127,127,127,127,127,127,127,0,127,127,0,127,127,0,127,127,127,127,0,
    127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,0,127,127,
    127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,
    127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,
    127,0,127,127,127,127,127,127,127,127,127,127,127,127,0,127,127,127,127,
    127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,127,127,
    0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,245,255,255,255,255,255,
    251,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,12,228,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,255,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,138,225,21,
    0,0,0,138,125,0,0,0,0,59,245,255,255,125,0,0,0,0,0,0,0,0,138,225,21,0,0,
    175,166,0,12,228,34,0,0,59,245,255,255,247,34,0,89,225,29,206,166,0,0,0,
    0,0,89,255,255,255,255,125,0,0,0,7,206,255,255,247,34,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,255,255,255,255,125,0,0,0,0,0,0,0,
    0,0,0,0,0,138,255,255,166,0,0,0,0,7,202,89,0,0,0,0,0,12,235,255,125,0,0,
    175,255,255,225,21,0,0,0,12,235,125,0,0,0,0,0,0,0,0,0,0,0,138,255,255,255,
    255,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,228,34,0,0,89,255,255,225,21,0,0,
    0,0,0,0,0,0,0,0,0,138,166,0,0,0,89,225,21,0,0,0,0,0,138,166,0,0,0,89,225,
    21,0,0,0,12,235,255,255,166,0,0,7,206,125,0,0,0,0,0,0,89,247,34,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,138,225,21,0,0,0,138,125,0,0,0,12,
    235,125,0,59,115,0,0,0,0,0,0,0,0,7,206,125,0,59,215,21,0,12,228,34,0,12,
    235,125,0,0,168,34,0,0,0,0,0,0,0,0,0,0,175,225,21,0,0,0,175,225,21,0,0,
    0,0,0,138,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,225,
    21,0,0,0,175,225,21,0,0,0,0,0,0,0,0,0,59,238,34,7,206,125,0,0,0,7,202,89,
    0,0,0,0,7,199,34,59,238,34,0,0,0,7,202,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,89,255,255,255,125,175,125,0,0,0,0,0,0,0,0,0,0,0,0,0,138,255,247,34,0,
    59,241,89,0,175,201,0,0,0,0,0,0,0,0,0,7,206,255,166,0,0,12,232,89,0,0,0,
    0,7,206,255,166,0,0,12,232,89,0,0,0,0,0,0,0,59,215,21,0,89,201,0,0,0,0,
    0,0,0,89,247,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,7,
    206,255,255,251,89,0,59,241,89,0,0,0,138,201,0,0,0,138,201,0,0,89,225,21,
    175,125,0,0,12,228,34,0,12,235,125,0,0,0,0,0,0,0,0,0,0,0,0,0,138,166,0,
    89,255,255,247,34,89,201,0,0,89,255,255,255,166,0,0,0,0,168,34,7,151,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,166,7,206,255,255,225,21,89,201,0,0,0,
    0,0,0,0,0,0,89,166,0,0,138,166,0,0,0,7,202,89,0,0,0,0,0,0,0,59,238,34,0,
    7,206,255,125,0,0,0,0,0,0,0,0,0,59,238,34,0,0,175,166,0,175,255,255,255,
    125,175,125,0,138,247,34,0,0,0,0,0,0,0,0,0,0,12,228,34,0,89,201,0,0,89,
    225,0,81,115,0,134,89,0,0,0,0,0,138,166,0,0,138,166,0,0,0,0,0,0,0,138,166,
    0,0,138,166,0,0,0,0,0,0,59,245,247,34,0,12,232,89,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,89,201,0,7,206,201,138,125,
    138,125,0,59,241,89,0,0,0,0,175,255,255,255,225,21,0,0,7,206,166,215,21,
    0,0,12,228,34,0,0,138,255,255,251,89,0,0,0,0,0,0,0,0,0,12,206,21,59,241,
    89,0,134,89,0,172,89,59,238,34,0,138,166,0,0,7,206,201,12,235,125,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,12,206,21,7,202,89,12,235,125,0,172,89,0,0,0,0,
    0,0,0,0,59,238,34,7,206,125,12,235,255,255,255,255,255,255,125,0,0,0,12,
    235,125,0,0,0,0,7,206,125,0,0,0,0,0,0,0,0,59,238,34,0,0,175,166,0,175,255,
    255,255,125,175,125,0,138,247,34,0,0,0,0,0,0,0,0,0,0,12,228,34,0,89,201,
    0,0,89,225,0,29,206,166,59,245,125,0,0,0,0,138,166,0,12,228,34,0,175,225,
    21,0,0,0,138,166,0,12,228,42,206,255,255,166,0,0,0,0,12,228,34,138,166,
    0,89,247,34,0,0,0,0,59,238,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,
    0,0,138,201,0,59,241,89,138,125,0,0,59,245,255,255,255,125,0,0,138,166,
    0,89,201,0,0,0,0,89,255,125,0,0,0,0,0,0,0,7,206,125,0,138,251,89,0,0,0,
    0,0,0,0,0,89,166,0,138,201,0,0,0,0,0,89,166,59,215,21,0,175,166,0,59,245,
    125,89,251,89,0,0,12,235,255,255,255,255,255,255,125,138,255,255,251,89,
    127,166,0,7,202,89,12,232,89,0,89,166,0,0,0,0,0,0,0,0,0,138,255,255,166,
    0,0,0,0,7,202,89,0,0,0,0,0,59,241,89,0,0,0,0,0,7,206,125,0,0,0,0,0,0,0,
    0,59,238,34,0,0,175,166,0,89,255,255,255,125,175,125,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,12,228,34,0,59,241,89,0,175,201,0,0,0,175,201,7,206,201,0,0,0,
    138,166,0,175,166,0,138,200,215,21,0,0,0,138,166,0,175,166,7,151,0,89,247,
    34,0,0,0,59,238,47,228,34,59,219,209,34,0,0,0,89,255,125,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,138,225,21,59,238,34,138,125,0,0,0,59,
    241,89,0,0,0,0,138,166,0,89,201,0,0,59,245,255,255,255,255,125,0,0,0,0,
    0,59,238,34,0,7,206,125,0,0,0,0,0,0,0,0,89,166,0,138,201,0,0,0,0,0,89,166,
    0,175,255,255,223,166,0,12,235,125,59,241,89,0,0,0,0,0,0,0,0,0,175,125,
    0,0,0,0,0,138,125,0,7,206,255,255,125,0,0,59,157,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,7,202,89,0,0,0,0,7,206,255,255,255,166,7,206,255,255,201,0,
    0,0,0,0,0,0,0,0,59,238,34,0,0,175,166,0,0,89,255,255,125,175,125,0,0,0,
    0,0,0,0,0,0,0,0,0,0,89,255,255,251,89,0,89,255,255,225,21,0,0,0,175,225,
    29,206,166,0,0,0,138,166,59,215,21,59,215,81,215,21,0,0,0,138,166,59,215,
    21,0,0,0,89,225,21,59,245,255,255,125,138,166,7,202,97,199,34,0,0,89,251,
    89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,138,225,21,59,241,
    89,138,125,0,0,0,89,247,34,0,0,0,0,175,255,255,255,225,21,0,0,0,12,232,
    89,0,0,0,12,228,34,0,12,235,225,21,59,215,21,0,0,0,0,0,0,0,0,12,206,21,
    59,241,89,0,134,89,0,172,89,0,0,0,0,0,0,0,0,7,206,201,12,235,125,0,0,0,
    0,0,0,0,0,175,125,0,0,0,0,0,12,206,21,7,202,89,7,206,125,0,172,89,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,202,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,59,238,34,0,0,175,166,0,0,0,0,175,125,175,125,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,166,59,245,125,0,0,0,0,0,0,
    175,125,12,228,34,59,215,21,0,0,0,0,0,175,125,0,0,0,12,232,89,0,0,0,0,0,
    59,238,34,175,125,7,199,34,0,0,175,201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,127,0,0,0,0,0,138,225,21,7,206,201,138,125,138,125,7,202,89,0,0,0,
    0,138,201,0,0,0,138,166,0,0,0,12,232,89,0,0,0,12,228,34,0,0,0,175,255,255,
    166,0,0,0,0,0,0,0,0,0,0,138,166,0,89,255,255,247,34,89,201,0,0,0,0,0,0,
    0,0,0,0,0,168,34,7,151,0,0,0,0,0,0,0,0,175,125,0,0,0,0,0,0,138,166,7,202,
    89,0,89,247,124,201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,255,255,255,255,
    255,255,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,245,125,0,7,206,
    166,0,0,0,0,175,125,175,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,59,115,0,134,89,0,0,0,0,0,0,59,215,21,59,245,255,255,255,225,21,0,
    0,0,59,215,21,0,0,59,238,34,0,0,0,0,0,0,175,125,7,206,255,255,255,251,89,
    0,138,247,34,0,59,157,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,138,225,
    21,0,7,206,255,255,251,89,138,255,255,255,255,255,166,0,0,0,0,0,0,0,0,0,
    0,12,232,89,0,0,0,12,228,34,0,0,0,0,0,59,241,89,0,0,0,0,0,0,0,0,0,0,175,
    225,21,0,0,0,175,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,175,225,21,0,0,0,175,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,238,198,
    255,251,194,166,0,0,0,0,175,125,175,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,125,0,0,0,0,89,225,21,0,0,0,
    7,206,125,0,0,12,235,255,255,255,166,0,0,0,89,225,21,0,0,0,12,228,34,0,
    0,0,175,255,255,255,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,
    0,0,0,0,138,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,228,
    34,0,7,176,21,0,89,247,34,0,0,0,0,0,0,0,0,0,0,0,89,255,255,255,255,125,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,
    255,255,255,255,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,238,34,0,0,0,0,0,0,0,0,175,
    125,175,125,0,0,0,0,0,0,0,0,59,215,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,
    0,0,0,0,0,0,0,138,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    12,228,34,0,7,206,255,255,251,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,238,34,0,0,0,0,0,0,0,0,175,125,175,125,
    0,0,0,0,0,0,59,245,251,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,127,127,0,127,127,
    127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,
    127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,0,127,127,127,127,
    127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,
    127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,
    127,127,127,127,127,127,127,0,127,127,127,127,0,127,127,127,127,127,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,
    0,127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,
    127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,
    127,127,127,127,127,127,0,127,127,127,127,0,127,127,127,127,127,127,0,127,
    127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,
    127,127,127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,127,127,127,0,
    127,127,127,127,127,127,127,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,89,
    255,125,0,0,0,0,0,0,12,235,201,0,0,0,0,12,235,251,89,0,0,0,0,175,255,125,
    89,201,0,0,0,0,0,0,0,0,0,0,0,59,245,247,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,89,255,125,0,0,0,0,0,0,138,251,89,0,0,0,12,235,251,
    89,0,0,0,0,0,0,0,0,7,206,225,21,0,0,0,89,255,125,0,89,255,225,21,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,247,34,172,89,0,0,0,0,7,206,225,21,0,0,
    0,0,0,0,0,89,255,125,0,0,0,0,0,0,89,255,225,21,0,0,0,0,12,235,247,34,172,
    89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,
    255,125,0,0,0,0,0,0,0,7,206,225,21,0,0,0,0,89,255,225,21,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,89,255,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,
    0,89,247,34,0,0,0,0,0,175,166,0,0,0,0,7,206,125,59,241,89,0,0,89,201,12,
    235,247,34,0,0,7,206,166,59,241,89,0,0,12,228,34,59,215,21,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,238,34,0,0,0,0,89,247,34,0,0,0,7,
    206,125,59,241,89,0,7,206,166,59,238,34,0,0,175,166,0,0,59,241,89,0,89,
    247,34,138,201,59,238,34,138,201,0,0,0,0,0,0,0,0,0,0,0,175,125,89,255,201,
    0,0,0,0,0,0,0,175,201,0,0,0,0,0,0,12,232,89,0,0,0,0,0,0,59,238,34,138,225,
    21,0,0,0,175,125,89,255,201,0,0,0,0,59,238,34,138,201,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,247,34,0,0,0,0,0,0,138,201,0,0,0,0,
    0,59,238,34,138,225,21,0,0,0,59,238,34,138,201,0,0,0,0,12,232,89,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,228,34,59,215,21,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,255,255,125,0,
    0,0,0,127,0,0,12,235,225,21,0,0,0,0,12,235,225,21,0,0,0,0,12,235,225,21,
    0,0,0,0,12,235,225,21,0,0,0,0,12,235,225,21,0,0,0,0,12,235,225,21,0,0,0,
    0,0,175,255,255,255,255,255,255,255,166,0,0,138,255,255,255,251,89,59,245,
    255,255,255,255,127,81,245,255,255,255,255,225,21,59,245,255,255,255,255,
    127,81,245,255,255,255,255,127,111,255,255,255,125,89,255,255,255,125,89,
    255,255,255,125,89,255,255,255,125,7,206,255,255,255,255,125,0,0,59,245,
    247,34,0,0,59,241,89,0,0,0,138,255,255,255,166,0,0,0,0,138,255,255,255,
    166,0,0,0,0,0,138,255,255,255,166,0,0,0,0,138,255,255,255,166,0,0,0,0,138,
    255,255,255,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,255,255,255,210,235,166,
    59,241,89,0,0,0,59,241,89,59,241,89,0,0,0,59,241,89,59,241,89,0,0,0,59,
    241,89,59,241,89,0,0,0,59,241,132,245,125,0,0,0,89,251,89,12,232,89,0,0,
    0,0,7,206,166,0,89,251,89,0,0,0,127,0,0,89,232,241,89,0,0,0,0,89,232,241,
    89,0,0,0,0,89,232,241,89,0,0,0,0,89,232,241,89,0,0,0,0,89,232,241,89,0,
    0,0,0,89,232,241,89,0,0,0,0,12,232,89,89,225,21,0,0,0,0,0,175,247,34,0,
    0,59,192,59,241,89,0,0,0,0,59,241,89,0,0,0,0,0,59,241,89,0,0,0,0,59,241,
    89,0,0,0,0,0,59,241,89,0,0,59,241,89,0,0,59,241,89,0,0,59,241,89,0,7,206,
    166,0,0,59,245,201,0,59,245,255,201,0,0,59,241,89,0,0,138,251,89,0,12,235,
    166,0,0,138,251,89,0,12,235,166,0,0,0,138,251,89,0,12,235,166,0,0,138,251,
    89,0,12,235,166,0,0,138,251,89,0,12,235,166,0,0,0,138,166,0,0,0,12,228,
    34,0,0,175,247,34,0,0,175,225,21,59,241,89,0,0,0,59,241,89,59,241,89,0,
    0,0,59,241,89,59,241,89,0,0,0,59,241,89,59,241,89,0,0,0,59,241,89,89,247,
    34,0,7,206,125,0,12,232,89,0,0,0,0,59,241,89,0,12,232,89,0,0,0,127,0,7,
    206,166,175,166,0,0,0,7,206,166,175,166,0,0,0,7,206,166,175,166,0,0,0,7,
    206,166,175,166,0,0,0,7,206,166,175,166,0,0,0,0,175,166,175,166,0,0,0,0,
    138,225,21,89,225,21,0,0,0,0,59,241,89,0,0,0,0,0,59,241,89,0,0,0,0,59,241,
    89,0,0,0,0,0,59,241,89,0,0,0,0,59,241,89,0,0,0,0,0,59,241,89,0,0,59,241,
    89,0,0,59,241,89,0,0,59,241,89,0,7,206,166,0,0,0,59,245,125,59,241,132,
    241,89,0,59,241,89,0,12,235,166,0,0,0,89,247,34,12,235,166,0,0,0,89,247,
    34,0,12,235,166,0,0,0,89,247,34,12,235,166,0,0,0,89,247,34,12,235,166,0,
    0,0,89,247,34,0,0,12,235,125,0,12,235,125,0,0,59,241,89,0,0,138,176,235,
    166,59,241,89,0,0,0,59,241,89,59,241,89,0,0,0,59,241,89,59,241,89,0,0,0,
    59,241,89,59,241,89,0,0,0,59,241,89,0,175,201,0,138,225,21,0,12,235,255,
    255,255,225,21,59,238,34,0,138,225,21,0,0,0,127,0,59,238,34,89,247,34,0,
    0,59,238,34,89,247,34,0,0,59,238,34,89,247,34,0,0,59,238,34,89,247,34,0,
    0,59,238,34,89,247,34,0,0,59,241,89,89,225,21,0,0,7,206,125,0,89,225,21,
    0,0,0,0,138,225,21,0,0,0,0,0,59,241,89,0,0,0,0,59,241,89,0,0,0,0,0,59,241,
    89,0,0,0,0,59,241,89,0,0,0,0,0,59,241,89,0,0,59,241,89,0,0,59,241,89,0,
    0,59,241,89,0,7,206,166,0,0,0,7,206,166,59,241,89,138,225,21,59,241,89,
    0,59,241,89,0,0,0,59,241,89,59,241,89,0,0,0,59,241,89,0,59,241,89,0,0,0,
    59,241,89,59,241,89,0,0,0,59,241,89,59,241,89,0,0,0,59,241,89,0,0,0,12,
    235,138,235,125,0,0,0,138,225,21,0,59,215,21,175,201,59,241,89,0,0,0,59,
    241,89,59,241,89,0,0,0,59,241,89,59,241,89,0,0,0,59,241,89,59,241,89,0,
    0,0,59,241,89,0,59,245,166,241,89,0,0,12,232,89,0,0,175,225,59,238,47,235,
    225,21,0,0,0,0,127,0,138,201,0,12,235,125,0,0,138,201,0,12,235,125,0,0,
    138,201,0,12,235,125,0,0,138,201,0,12,235,125,0,0,138,201,0,12,235,125,
    0,0,138,225,21,12,235,125,0,0,89,247,34,0,89,255,255,255,255,251,89,138,
    225,21,0,0,0,0,0,59,245,255,255,255,255,127,59,245,255,255,255,255,166,
    0,59,245,255,255,255,255,127,59,245,255,255,255,255,127,0,59,241,89,0,0,
    59,241,89,0,0,59,241,89,0,0,59,241,89,7,206,255,255,255,166,0,0,175,201,
    59,241,89,12,235,125,59,241,89,0,59,241,89,0,0,0,12,235,125,59,241,89,0,
    0,0,12,235,125,0,59,241,89,0,0,0,12,235,125,59,241,89,0,0,0,12,235,125,
    59,241,89,0,0,0,12,235,125,0,0,0,0,12,235,125,0,0,0,0,138,225,21,7,199,
    34,0,138,225,81,241,89,0,0,0,59,241,89,59,241,89,0,0,0,59,241,89,59,241,
    89,0,0,0,59,241,89,59,241,89,0,0,0,59,241,89,0,0,138,255,166,0,0,0,12,232,
    89,0,0,89,247,59,238,34,0,59,245,125,0,0,0,127,7,206,125,0,0,175,201,0,
    7,206,125,0,0,175,201,0,7,206,125,0,0,175,201,0,7,206,125,0,0,175,201,0,
    7,206,125,0,0,175,201,0,7,206,125,0,0,175,201,0,7,206,255,255,255,255,225,
    21,0,0,0,0,138,225,21,0,0,0,0,0,59,241,89,0,0,0,0,59,241,89,0,0,0,0,0,59,
    241,89,0,0,0,0,59,241,89,0,0,0,0,0,59,241,89,0,0,59,241,89,0,0,59,241,89,
    0,0,59,241,89,0,7,206,166,0,0,0,7,206,166,59,241,89,0,89,247,94,241,89,
    0,59,241,89,0,0,0,59,241,89,59,241,89,0,0,0,59,241,89,0,59,241,89,0,0,0,
    59,241,89,59,241,89,0,0,0,59,241,89,59,241,89,0,0,0,59,241,89,0,0,0,12,
    235,138,235,125,0,0,0,138,225,21,175,125,0,0,175,201,59,241,89,0,0,0,59,
    241,89,59,241,89,0,0,0,59,241,89,59,241,89,0,0,0,59,241,89,59,241,89,0,
    0,0,59,241,89,0,0,59,241,89,0,0,0,12,232,89,0,7,206,201,59,238,34,0,0,138,
    201,0,0,0,127,59,245,255,255,255,255,247,34,59,245,255,255,255,255,247,
    34,59,245,255,255,255,255,247,34,59,245,255,255,255,255,247,34,59,245,255,
    255,255,255,247,34,59,245,255,255,255,255,247,34,59,241,89,0,0,89,225,21,
    0,0,0,0,59,241,89,0,0,0,0,0,59,241,89,0,0,0,0,59,241,89,0,0,0,0,0,59,241,
    89,0,0,0,0,59,241,89,0,0,0,0,0,59,241,89,0,0,59,241,89,0,0,59,241,89,0,
    0,59,241,89,0,7,206,166,0,0,0,59,241,89,59,241,89,0,7,206,200,241,89,0,
    12,235,166,0,0,0,89,247,34,12,235,166,0,0,0,89,247,34,0,12,235,166,0,0,
    0,89,247,34,12,235,166,0,0,0,89,247,34,12,235,166,0,0,0,89,247,34,0,0,12,
    235,125,0,12,235,125,0,0,59,241,159,166,0,0,12,235,166,12,232,89,0,0,0,
    59,238,34,12,232,89,0,0,0,59,238,34,12,232,89,0,0,0,59,238,34,12,232,89,
    0,0,0,59,238,34,0,0,59,241,89,0,0,0,12,235,255,255,255,201,0,59,238,34,
    0,0,138,201,0,0,0,127,138,201,0,0,0,12,235,125,138,201,0,0,0,12,235,125,
    138,201,0,0,0,12,235,125,138,201,0,0,0,12,235,125,138,201,0,0,0,12,235,
    125,138,201,0,0,0,12,235,125,175,201,0,0,0,89,225,21,0,0,0,0,0,175,247,
    34,0,0,59,192,59,241,89,0,0,0,0,59,241,89,0,0,0,0,0,59,241,89,0,0,0,0,59,
    241,89,0,0,0,0,0,59,241,89,0,0,59,241,89,0,0,59,241,89,0,0,59,241,89,0,
    7,206,166,0,0,59,245,201,0,59,241,89,0,0,59,245,251,89,0,0,138,251,89,0,
    59,245,166,0,0,138,251,89,0,59,245,166,0,0,0,138,251,89,0,59,245,166,0,
    0,138,251,89,0,59,245,166,0,0,138,251,89,0,59,245,166,0,0,0,138,166,0,0,
    0,12,228,34,0,0,175,247,34,0,7,206,225,21,0,138,225,21,0,7,206,166,0,0,
    138,225,21,0,7,206,166,0,0,138,225,21,0,7,206,166,0,0,138,225,21,0,7,206,
    166,0,0,0,59,241,89,0,0,0,12,232,89,0,0,0,0,59,238,34,0,12,235,125,0,0,
    0,127,206,125,0,0,0,0,175,206,206,125,0,0,0,0,175,206,206,125,0,0,0,0,175,
    206,206,125,0,0,0,0,175,206,206,125,0,0,0,0,175,206,206,125,0,0,0,0,175,
    232,245,125,0,0,0,89,255,255,255,255,255,166,0,0,138,255,255,255,251,89,
    59,245,255,255,255,255,127,81,245,255,255,255,255,225,21,59,245,255,255,
    255,255,127,81,245,255,255,255,255,127,111,255,255,255,125,89,255,255,255,
    125,89,255,255,255,125,89,255,255,255,125,7,206,255,255,255,255,125,0,0,
    59,241,89,0,0,0,175,251,89,0,0,0,138,255,255,255,166,0,0,0,0,138,255,255,
    255,166,0,0,0,0,0,138,255,255,255,166,0,0,0,0,138,255,255,255,166,0,0,0,
    0,138,255,255,255,166,0,0,0,0,0,0,0,0,0,0,0,0,0,7,202,194,255,255,255,201,
    0,0,0,0,59,245,255,251,89,0,0,0,0,59,245,255,251,89,0,0,0,0,59,245,255,
    251,89,0,0,0,0,59,245,255,251,89,0,0,0,0,59,241,89,0,0,0,12,232,89,0,0,
    0,0,59,238,47,235,255,166,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,12,228,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,
    166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,255,166,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,127,127,127,127,127,127,0,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,
    127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,
    127,0,127,127,127,127,127,127,0,127,127,127,127,0,127,127,127,127,0,127,
    127,127,127,0,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,127,127,
    127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,0,
    0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,255,201,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,7,206,
    225,21,0,0,0,0,0,12,235,201,0,0,0,138,255,201,0,0,0,59,245,225,29,202,89,
    0,0,0,0,0,0,0,0,138,166,7,202,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    7,206,225,21,0,0,0,0,0,12,235,201,0,0,0,89,255,225,21,0,0,0,0,0,0,0,0,175,
    247,34,0,12,235,255,255,166,0,0,0,0,0,0,0,0,0,0,0,0,59,245,225,29,202,89,
    0,0,138,251,89,0,0,0,0,0,7,206,225,21,0,0,0,89,255,225,21,0,0,59,245,225,
    29,202,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,225,
    21,0,0,0,0,0,89,255,125,0,0,0,89,255,225,21,0,0,0,0,0,0,0,0,0,0,0,0,138,
    251,89,0,59,238,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,175,
    166,0,0,0,0,7,206,166,0,0,0,89,225,21,175,201,0,7,202,89,138,255,166,0,
    0,89,247,34,175,166,0,0,138,166,7,202,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,7,206,166,0,0,0,0,0,175,166,0,0,0,89,247,34,138,201,0,0,89,247,
    34,175,201,0,0,138,201,0,175,200,215,34,235,247,47,232,0,138,255,225,111,
    225,21,0,0,172,89,138,255,166,0,0,0,0,89,225,21,0,0,0,0,175,201,0,0,0,0,
    89,247,34,138,201,0,0,172,89,138,255,166,0,0,59,238,34,138,201,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,166,0,0,0,0,59,241,89,0,0,0,89,
    247,34,138,201,0,0,0,59,238,34,138,201,0,0,0,89,247,34,0,0,59,238,34,0,
    0,0,0,0,138,225,29,206,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,255,201,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,225,21,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,12,235,166,0,0,0,0,0,0,0,0,89,166,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,238,34,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,7,206,255,255,255,125,0,7,206,255,255,255,
    125,0,7,206,255,255,255,125,0,7,206,255,255,255,125,0,7,206,255,255,255,
    125,0,12,235,255,255,251,89,0,12,235,255,255,251,89,59,245,255,166,0,0,
    59,245,255,255,201,0,12,235,255,251,89,0,0,12,235,255,251,89,0,0,12,235,
    255,251,89,0,0,12,235,255,251,89,0,59,238,34,59,238,34,59,238,34,59,238,
    34,0,59,241,89,175,225,21,0,59,241,194,255,255,125,0,0,12,235,255,247,34,
    0,0,12,235,255,247,34,0,0,0,12,235,255,247,34,0,0,12,235,255,247,34,0,0,
    12,235,255,247,34,0,0,0,0,0,12,235,166,0,0,0,0,7,206,255,255,225,21,0,59,
    241,89,0,59,241,89,59,241,89,0,59,241,89,59,241,89,0,59,241,89,0,59,241,
    89,0,59,241,97,206,166,0,0,12,235,125,59,238,163,255,255,201,7,206,166,
    0,0,12,235,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,89,247,34,0,0,0,0,89,247,
    34,0,0,0,0,89,247,34,0,0,0,0,89,247,34,0,0,0,0,89,247,34,0,0,0,0,138,225,
    21,0,0,0,0,89,255,225,21,0,138,201,59,245,125,0,0,0,7,206,125,0,89,225,
    21,7,206,125,0,89,225,21,7,206,125,0,89,225,21,7,206,125,0,89,225,21,59,
    238,34,59,238,34,59,238,34,59,238,34,0,0,0,0,12,235,125,0,59,245,166,0,
    89,247,34,7,206,166,0,138,225,21,7,206,166,0,138,225,21,0,7,206,166,0,138,
    225,21,7,206,166,0,138,225,21,7,206,166,0,138,225,21,0,0,0,0,0,0,0,0,0,
    0,12,232,89,0,138,251,89,0,59,241,89,0,59,241,89,59,241,89,0,59,241,89,
    59,241,89,0,59,241,89,0,59,241,89,0,59,241,89,89,247,34,0,89,247,34,59,
    245,166,0,7,206,166,89,247,34,0,89,247,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,
    59,241,89,0,0,0,0,59,241,89,0,0,0,0,59,241,89,0,0,0,0,59,241,89,0,0,0,0,
    59,241,89,0,0,0,0,59,238,34,0,0,0,0,12,232,89,0,0,59,238,127,225,21,0,0,
    0,59,238,34,0,59,238,34,59,238,34,0,59,238,34,59,238,34,0,59,238,34,59,
    238,34,0,59,238,34,59,238,34,59,238,34,59,238,34,59,238,34,0,138,255,255,
    255,255,201,0,59,241,89,0,59,241,89,59,241,89,0,59,241,89,59,241,89,0,59,
    241,89,0,59,241,89,0,59,241,89,59,241,89,0,59,241,89,59,241,89,0,59,241,
    89,0,12,235,255,255,255,255,255,255,166,138,201,0,59,157,175,201,0,59,241,
    89,0,59,241,89,59,241,89,0,59,241,89,59,241,89,0,59,241,89,0,59,241,89,
    0,59,241,89,12,235,125,0,175,166,0,59,238,34,0,0,138,225,34,235,125,0,175,
    166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,127,0,89,255,255,255,251,89,0,89,255,255,255,251,
    89,0,89,255,255,255,251,89,0,89,255,255,255,251,89,0,89,255,255,255,251,
    89,0,138,255,255,255,247,34,0,175,255,255,255,255,255,255,255,255,251,127,
    201,0,0,0,0,59,245,255,255,255,251,89,59,245,255,255,255,251,89,59,245,
    255,255,255,251,89,59,245,255,255,255,251,89,59,238,34,59,238,34,59,238,
    34,59,238,34,138,247,34,0,0,138,201,0,59,241,89,0,59,241,89,59,238,34,0,
    12,232,89,59,238,34,0,12,232,89,0,59,238,34,0,12,232,89,59,238,34,0,12,
    232,89,59,238,34,0,12,232,89,0,0,0,0,0,0,0,0,0,0,175,201,7,176,21,138,201,
    0,59,241,89,0,59,241,89,59,241,89,0,59,241,89,59,241,89,0,59,241,89,0,59,
    241,89,0,59,241,89,0,175,201,12,232,89,0,59,238,34,0,0,138,225,21,175,201,
    12,232,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,59,241,89,0,59,241,89,59,241,89,0,59,
    241,89,59,241,89,0,59,241,89,59,241,89,0,59,241,89,59,241,89,0,59,241,89,
    138,247,34,0,59,238,34,138,225,21,0,12,232,89,0,0,0,0,138,201,0,0,0,0,59,
    238,34,0,0,0,0,59,238,34,0,0,0,0,59,238,34,0,0,0,0,59,238,34,0,0,0,0,59,
    238,34,59,238,34,59,238,34,59,238,34,175,201,0,0,0,138,166,0,59,241,89,
    0,59,241,89,59,241,89,0,59,241,89,59,241,89,0,59,241,89,0,59,241,89,0,59,
    241,89,59,241,89,0,59,241,89,59,241,89,0,59,241,89,0,0,0,0,12,235,166,0,
    0,0,138,201,134,89,0,175,166,0,59,241,89,0,59,241,89,59,241,89,0,59,241,
    89,59,241,89,0,59,241,89,0,59,241,89,0,59,241,89,0,89,247,124,225,21,0,
    59,238,34,0,0,138,201,0,89,247,124,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,89,247,
    34,0,89,251,89,89,247,34,0,89,251,89,89,247,34,0,89,251,89,89,247,34,0,
    89,251,89,89,247,34,0,89,251,89,175,201,0,0,175,247,34,175,201,0,0,59,245,
    225,21,0,7,199,94,245,125,0,0,0,7,206,166,0,7,199,34,7,206,166,0,7,199,
    34,7,206,166,0,7,199,34,7,206,166,0,7,199,34,59,238,34,59,238,34,59,238,
    34,59,238,34,138,247,34,0,12,232,89,0,59,241,89,0,59,241,89,7,206,166,0,
    138,225,21,7,206,166,0,138,225,21,0,7,206,166,0,138,225,21,7,206,166,0,
    138,225,21,7,206,166,0,138,225,21,0,0,0,0,12,235,166,0,0,0,59,245,166,0,
    59,241,89,0,12,235,125,0,175,251,89,12,235,125,0,175,251,89,12,235,125,
    0,175,251,89,0,12,235,125,0,175,251,89,0,7,206,255,125,0,0,59,238,34,0,
    12,235,125,0,7,206,255,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,175,255,255,232,
    241,89,0,175,255,255,232,241,89,0,175,255,255,232,241,89,0,175,255,255,
    232,241,89,0,175,255,255,232,241,89,12,235,255,255,166,238,34,12,235,255,
    255,225,21,89,255,255,251,89,0,89,255,255,255,201,0,12,235,255,251,89,0,
    0,12,235,255,251,89,0,0,12,235,255,251,89,0,0,12,235,255,251,89,0,59,238,
    34,59,238,34,59,238,34,59,238,34,0,138,255,255,255,125,0,0,59,241,89,0,
    59,241,89,0,12,235,255,247,34,0,0,12,235,255,247,34,0,0,0,12,235,255,247,
    34,0,0,12,235,255,247,34,0,0,12,235,255,247,34,0,0,0,0,0,0,0,0,0,0,0,7,
    206,255,255,225,21,0,0,0,89,255,255,166,241,89,0,89,255,255,166,241,89,
    0,89,255,255,166,241,89,0,0,89,255,255,166,241,89,0,0,138,247,34,0,0,59,
    245,166,255,255,166,0,0,0,138,247,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,215,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,232,89,0,0,0,59,238,34,0,0,0,0,0,
    12,232,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,12,235,255,125,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,138,225,21,0,0,0,59,238,34,0,0,0,0,0,138,225,21,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,
    127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,127,0,127,127,
    127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,
    127,127,127,127,0,127,127,127,127,127,127,0,127,127,0,127,127,0,127,127,
    0,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,
    127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,
    127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,
    127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,
    127,127,127,127,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
};


// Large font anti-aliased
const int FONT2AA_BM_W = 276;
const int FONT2AA_BM_H = 120;
static const unsigned char s_Font2AA[] = 
{
    127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,
    4,4,4,0,0,0,0,0,0,0,0,0,0,0,4,4,4,0,4,4,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,0,0,0,0,0,0,0,0,
    0,0,0,4,4,4,0,4,4,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,59,245,125,175,225,21,
    0,0,0,0,0,0,0,0,0,0,0,0,0,138,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,59,241,89,0,0,12,235,201,89,255,166,0,0,0,0,0,172,89,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,225,21,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,0,
    0,0,0,0,0,0,0,0,0,0,4,4,4,0,4,4,0,0,0,0,0,127,0,0,0,0,0,138,247,34,0,12,
    232,89,138,225,21,0,0,0,0,138,125,7,199,34,0,0,0,0,138,125,0,0,0,0,138,
    255,255,201,0,0,0,59,215,21,0,0,0,0,59,245,255,255,166,0,0,0,59,241,89,
    0,7,206,201,0,0,89,251,89,0,59,215,21,172,89,59,192,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,166,0,0,138,255,251,89,0,0,0,0,0,
    138,201,0,0,0,7,206,255,255,255,166,0,0,7,206,255,255,255,201,0,0,0,0,0,
    0,138,251,89,0,0,175,255,255,255,255,225,21,0,0,12,235,255,255,125,89,255,
    255,255,255,255,251,89,0,12,235,255,255,225,21,0,0,59,245,255,255,166,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,59,245,255,255,251,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,
    4,4,4,0,0,0,0,0,0,0,0,0,0,0,4,4,4,0,4,4,0,0,0,0,0,127,0,0,0,0,0,89,247,
    34,0,12,232,89,138,201,0,0,0,0,7,202,89,59,215,21,0,0,12,235,255,255,255,
    166,0,59,241,89,12,235,125,0,0,172,89,0,0,0,0,7,206,166,0,89,251,89,0,0,
    12,228,34,0,89,247,34,0,0,0,175,201,0,0,89,251,191,194,247,34,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,232,89,0,175,201,0,12,235,
    125,0,0,138,255,255,201,0,0,0,12,182,0,0,59,245,125,0,12,206,21,0,12,235,
    166,0,0,0,0,89,255,251,89,0,0,175,201,0,0,0,0,0,0,89,255,125,0,0,0,0,0,
    0,0,0,89,251,89,12,235,166,0,7,206,201,0,59,245,125,0,12,235,166,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,89,166,0,0,138,251,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,0,
    0,0,0,0,0,0,0,0,0,0,4,4,4,0,4,4,0,0,0,0,0,127,0,0,0,0,0,89,247,34,0,12,
    228,34,89,201,0,0,0,0,12,206,21,89,166,0,0,12,235,125,138,125,59,192,0,
    89,247,34,7,206,166,0,89,201,0,0,0,0,0,12,235,125,0,12,232,89,0,0,12,228,
    34,0,175,201,0,0,0,0,59,241,89,0,0,7,206,166,0,0,0,0,0,0,0,138,166,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,225,21,59,241,89,0,0,138,225,21,0,
    0,0,175,201,0,0,0,0,0,0,0,7,206,201,0,0,0,0,0,0,175,201,0,0,0,59,241,132,
    241,89,0,0,175,201,0,0,0,0,0,7,206,166,0,0,0,0,0,0,0,0,7,206,201,0,59,241,
    89,0,0,138,225,21,138,225,21,0,0,138,225,21,89,255,125,0,0,89,255,125,0,
    0,0,0,0,0,0,0,138,225,21,0,0,0,0,0,0,0,0,0,0,0,138,201,0,0,0,0,0,0,0,0,
    0,0,0,59,241,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,0,0,0,0,0,
    0,0,0,0,0,0,4,4,4,0,4,4,0,0,0,0,0,127,0,0,0,0,0,89,247,34,0,0,0,0,0,0,0,
    0,89,255,255,255,255,255,255,255,125,59,238,34,138,125,0,0,0,89,247,34,
    7,206,166,7,202,89,0,0,0,0,0,0,175,225,21,138,225,21,0,0,0,0,0,12,235,125,
    0,0,0,0,7,206,125,0,89,251,191,194,247,34,0,0,0,0,0,138,166,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,138,166,0,89,247,34,0,0,89,247,34,0,0,0,175,201,
    0,0,0,0,0,0,0,12,235,166,0,0,0,0,0,59,245,125,0,0,12,235,125,59,241,89,
    0,0,175,201,0,0,0,0,0,59,241,89,0,0,0,0,0,0,0,0,89,247,34,0,12,235,201,
    0,0,175,201,0,138,225,21,0,0,89,247,34,89,255,125,0,0,89,255,125,0,0,0,
    0,0,12,235,255,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,175,255,251,89,0,0,0,0,
    0,0,0,0,138,247,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,0,0,0,
    0,0,0,0,0,0,0,0,4,4,4,0,4,4,0,0,0,0,0,127,0,0,0,0,0,89,247,34,0,0,0,0,0,
    0,0,0,0,0,175,125,7,199,34,0,0,12,235,166,138,125,0,0,0,59,241,89,12,235,
    125,89,201,12,235,255,251,89,0,0,7,206,255,166,0,59,241,89,0,0,0,59,238,
    34,0,0,0,0,0,175,166,59,215,21,172,89,59,192,0,0,0,0,0,138,166,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,125,0,138,247,34,0,0,89,247,34,0,0,0,
    175,201,0,0,0,0,0,0,0,89,251,89,0,0,0,89,255,247,34,0,0,7,206,166,0,59,
    241,89,0,0,175,255,255,255,225,21,0,89,251,226,255,255,247,34,0,0,0,7,206,
    166,0,0,0,12,235,255,255,201,0,0,89,255,125,0,0,138,247,34,0,0,0,0,0,0,
    0,0,0,0,0,89,255,255,166,0,0,0,0,0,175,255,255,255,255,255,255,225,21,0,
    0,0,0,59,245,255,201,0,0,0,0,0,175,251,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,4,4,4,4,0,0,0,0,0,0,0,0,0,0,0,4,4,4,0,4,4,0,0,0,0,0,127,0,0,0,
    0,0,89,225,21,0,0,0,0,0,0,0,0,0,7,199,34,59,215,21,0,0,0,59,245,255,255,
    201,0,0,0,138,255,255,201,12,228,34,175,166,0,138,201,0,12,235,125,89,255,
    125,59,241,89,0,0,0,59,238,34,0,0,0,0,0,138,201,0,0,0,172,89,0,0,0,7,206,
    255,255,255,255,255,255,247,34,0,0,0,0,89,255,255,255,166,0,0,0,0,0,59,
    238,34,0,138,247,34,0,0,89,247,34,0,0,0,175,201,0,0,0,0,0,0,59,245,166,
    0,0,0,0,0,0,12,235,166,0,138,201,0,0,59,241,89,0,0,0,0,0,12,235,201,0,138,
    251,89,0,0,175,225,21,0,0,89,247,34,0,0,7,206,166,0,175,255,166,0,0,89,
    255,255,255,223,247,34,0,0,0,0,0,0,0,0,0,0,175,247,34,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,225,21,0,0,175,225,21,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,0,0,0,0,0,0,0,0,0,0,0,4,4,4,0,4,4,0,0,0,
    0,0,127,0,0,0,0,0,59,215,21,0,0,0,0,0,0,0,12,235,255,255,255,255,255,255,
    166,0,0,0,0,138,125,175,225,21,0,0,0,0,0,138,166,7,206,125,0,89,247,34,
    138,225,21,0,89,255,166,215,21,0,0,0,59,238,34,0,0,0,0,0,138,201,0,0,0,
    0,0,0,0,0,0,0,0,0,138,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,201,0,
    0,89,247,34,0,0,89,247,34,0,0,0,175,201,0,0,0,0,0,12,235,201,0,0,0,0,0,
    0,0,0,138,225,21,175,255,255,255,255,255,255,125,0,0,0,0,0,138,247,34,89,
    247,34,0,0,59,241,89,0,7,206,166,0,0,0,138,247,34,0,0,138,247,34,0,0,0,
    0,0,138,225,21,0,0,0,0,0,0,0,0,0,0,0,89,255,255,166,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,59,245,255,201,0,0,0,0,175,201,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,4,4,4,4,0,0,0,0,0,0,0,0,0,0,0,4,4,4,0,4,4,0,0,0,0,0,
    127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,166,0,175,125,0,0,0,0,0,0,138,
    125,89,247,34,0,0,0,0,12,228,34,7,206,125,0,89,247,34,138,247,34,0,0,89,
    255,166,0,0,0,0,59,238,34,0,0,0,0,0,175,166,0,0,0,0,0,0,0,0,0,0,0,0,138,
    166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,125,0,0,59,241,89,0,0,138,225,
    21,0,0,0,175,201,0,0,0,0,12,235,201,0,0,0,0,0,0,0,0,0,138,225,21,0,0,0,
    0,59,241,89,0,0,0,0,0,0,138,225,21,59,241,89,0,0,59,241,89,0,89,247,34,
    0,0,0,138,247,34,0,0,89,251,89,0,0,0,0,7,206,166,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,12,235,255,225,21,0,0,175,255,255,255,255,255,255,225,21,0,0,175,
    255,251,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    4,4,4,4,0,0,0,0,0,0,0,0,0,0,0,4,4,4,0,4,4,0,0,0,0,0,127,0,0,0,0,0,89,247,
    34,0,0,0,0,0,0,0,0,0,175,125,7,199,34,0,0,0,89,201,0,138,125,175,201,0,
    0,0,0,0,138,166,0,0,175,166,0,138,201,0,89,255,166,0,0,89,255,255,125,0,
    0,0,12,235,125,0,0,0,0,7,206,125,0,0,0,0,0,0,0,0,0,0,0,0,138,166,0,0,0,
    0,0,138,255,125,0,0,0,0,0,0,175,247,34,59,238,34,0,0,0,175,201,0,12,235,
    125,0,0,0,0,175,201,0,0,0,12,235,166,0,0,0,0,0,89,166,0,0,59,245,166,0,
    0,0,0,0,59,241,89,0,59,215,21,0,12,235,166,0,7,206,201,0,0,175,225,21,7,
    206,166,0,0,0,0,59,245,166,0,7,206,225,21,0,0,0,0,175,225,21,0,89,255,125,
    0,0,12,235,201,0,0,0,0,0,0,0,0,138,225,21,0,0,0,0,0,0,0,0,0,0,0,138,201,
    0,0,0,0,0,0,0,0,0,175,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    4,4,4,4,0,0,0,0,0,0,0,0,0,0,0,4,4,4,0,4,4,0,0,0,0,0,127,0,0,0,0,0,89,247,
    34,0,0,0,0,0,0,0,0,7,199,34,59,215,21,0,0,0,12,235,255,255,255,201,0,0,
    0,0,0,59,215,21,0,0,12,235,255,251,89,0,0,89,255,255,255,201,0,89,255,0,
    0,0,0,175,201,0,0,0,0,59,238,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,
    201,0,0,0,0,0,0,0,175,247,34,138,201,0,0,0,0,0,138,255,251,89,0,0,0,138,
    255,255,255,255,166,0,89,255,255,255,255,255,247,34,12,235,255,255,255,
    166,0,0,0,0,0,0,59,241,89,0,12,235,255,255,255,166,0,0,0,7,206,255,255,
    225,21,0,138,247,34,0,0,0,0,0,59,245,255,255,201,0,0,0,175,255,255,201,
    0,0,0,89,255,125,0,0,89,251,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,175,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,4,4,4,4,4,4,4,4,52,4,4,4,4,4,4,4,4,4,0,4,4,0,0,0,0,0,127,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,125,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,247,34,0,0,0,175,201,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,232,89,0,0,0,0,0,0,0,0,0,0,175,125,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,0,4,4,0,0,0,0,0,127,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,125,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,201,0,0,89,251,89,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,201,0,0,0,0,0,0,0,0,0,0,12,232,89,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,232,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,0,4,4,0,0,0,0,0,127,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,201,0,0,201,201,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,
    4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,0,0,0,0,0,0,127,127,127,0,127,127,127,
    127,0,127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,127,127,0,127,127,0,127,127,127,127,0,127,127,
    127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,
    127,0,127,127,127,0,127,127,127,127,0,127,127,127,0,127,127,127,127,0,127,
    127,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,
    127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,
    127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,
    0,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,
    4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,255,255,
    201,0,138,201,0,0,0,0,89,255,255,255,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,127,0,0,0,0,175,255,255,255,247,34,0,0,0,0,0,7,206,251,89,0,0,12,0,
    235,255,255,255,255,201,0,0,0,0,59,245,255,255,255,201,12,0,235,255,255,
    255,255,166,0,0,0,12,235,255,255,255,255,255,127,12,235,255,255,255,255,
    251,89,0,0,12,235,255,255,255,251,89,12,235,166,0,0,0,12,235,125,89,255,
    255,255,201,0,0,175,255,255,225,21,12,235,166,0,0,7,206,251,102,0,235,166,
    0,0,0,0,12,235,251,89,0,0,0,89,255,225,21,12,235,251,89,0,0,12,235,125,
    0,0,0,138,255,255,166,0,0,0,12,235,255,255,255,251,89,0,0,0,0,175,255,255,
    201,0,0,0,12,235,255,255,255,251,89,0,0,0,12,235,255,255,255,247,47,235,
    255,255,255,255,255,255,255,138,0,235,125,0,0,0,59,245,133,206,166,0,0,
    0,0,59,245,255,133,201,0,0,0,138,251,89,0,0,12,235,133,206,247,34,0,0,0,
    175,229,216,225,21,0,0,0,138,247,124,255,255,255,255,255,255,125,7,206,
    125,0,0,0,59,238,34,0,0,0,0,0,12,235,125,0,0,0,0,175,247,34,0,0,0,0,0,0,
    0,0,0,0,127,0,0,59,245,166,0,0,0,59,245,166,0,0,0,0,59,245,255,166,0,0,
    12,0,235,166,0,0,59,245,125,0,0,138,255,125,0,0,7,202,102,0,235,166,0,0,
    59,245,225,21,0,12,235,166,0,0,0,0,0,12,235,166,0,0,0,0,0,0,89,255,166,
    0,0,0,89,127,12,235,166,0,0,0,12,235,125,0,12,235,125,0,0,0,0,0,138,225,
    21,12,235,166,0,7,206,225,21,12,0,235,166,0,0,0,0,12,235,255,166,0,0,7,
    206,255,225,21,12,235,255,201,0,0,12,235,125,0,59,245,166,0,0,138,251,89,
    0,12,235,166,0,0,138,251,89,0,89,255,125,0,0,89,255,125,0,12,235,166,0,
    0,138,251,89,0,12,235,166,0,0,7,202,89,0,0,0,138,225,21,0,0,12,0,235,125,
    0,0,0,59,245,125,138,225,21,0,0,0,138,225,151,34,247,34,0,0,175,255,125,
    0,0,89,247,34,12,235,166,0,0,89,247,34,59,245,125,0,0,59,245,125,0,0,0,
    0,0,138,247,34,7,206,125,0,0,0,7,206,125,0,0,0,0,0,12,235,125,0,0,0,138,
    225,187,201,0,0,0,0,0,0,0,0,0,0,127,0,12,232,89,0,0,0,0,0,12,232,89,0,0,
    0,138,225,151,225,21,0,12,0,235,166,0,0,12,235,166,0,12,235,166,0,0,0,0,
    0,12,0,235,166,0,0,0,12,235,166,0,12,235,166,0,0,0,0,0,12,235,166,0,0,0,
    0,0,12,235,166,0,0,0,0,0,0,12,235,166,0,0,0,12,235,125,0,12,235,125,0,0,
    0,0,0,138,225,21,12,235,166,0,175,225,21,0,12,0,235,166,0,0,0,0,12,235,
    166,238,34,0,59,215,187,225,21,12,235,166,245,125,0,12,235,125,12,235,125,
    0,0,0,0,138,247,34,12,235,166,0,0,12,235,166,12,235,125,0,0,0,0,138,247,
    34,12,235,166,0,0,12,235,166,0,89,247,34,0,0,0,0,0,0,0,0,138,225,21,0,0,
    12,0,235,125,0,0,0,59,245,125,59,241,89,0,0,7,206,166,59,0,241,89,0,12,
    232,194,201,0,0,138,225,21,0,89,251,89,12,235,166,0,0,138,247,34,7,206,
    201,0,0,0,0,0,59,245,125,0,7,206,125,0,0,0,0,138,201,0,0,0,0,0,12,235,125,
    0,0,59,241,89,12,235,166,0,0,0,0,0,0,0,0,0,127,0,175,166,0,59,245,255,255,
    247,34,138,201,0,0,7,206,166,59,241,89,0,12,0,235,166,0,0,89,251,89,0,89,
    247,34,0,0,0,0,0,12,0,235,166,0,0,0,0,138,225,21,12,235,166,0,0,0,0,0,12,
    235,166,0,0,0,0,0,89,247,34,0,0,0,0,0,0,12,235,166,0,0,0,12,235,125,0,12,
    235,125,0,0,0,0,0,138,225,21,12,235,166,175,247,34,0,0,12,0,235,166,0,0,
    0,0,12,235,133,206,166,0,175,166,175,225,21,12,235,125,138,225,21,12,235,
    125,89,247,34,0,0,0,0,59,245,125,12,235,166,0,0,12,235,166,89,247,34,0,
    0,0,0,59,245,125,12,235,166,0,0,12,235,125,0,89,255,125,0,0,0,0,0,0,0,0,
    138,225,21,0,0,12,0,235,125,0,0,0,59,245,125,7,206,201,0,0,59,241,89,7,
    0,206,166,0,59,215,111,225,21,7,206,166,0,0,0,175,225,187,225,21,0,0,12,
    235,166,89,247,34,0,0,0,0,7,206,201,0,0,7,206,125,0,0,0,0,89,225,21,0,0,
    0,0,12,235,125,0,12,235,166,0,0,59,241,89,0,0,0,0,0,0,0,0,127,0,202,89,
    12,235,125,0,12,228,34,59,215,0,0,59,241,89,7,206,166,0,12,0,235,255,255,
    255,255,166,0,0,138,225,21,0,0,0,0,0,12,0,235,166,0,0,0,0,89,247,34,12,
    235,255,255,255,255,247,34,12,235,255,255,255,255,247,0,163,225,21,0,0,
    0,0,0,0,12,235,255,255,255,255,255,255,125,0,12,235,125,0,0,0,0,0,138,225,
    21,12,235,255,247,34,0,0,0,12,0,235,166,0,0,0,0,12,235,125,89,225,34,228,
    34,175,225,21,12,235,125,12,235,125,12,235,125,138,225,21,0,0,0,0,12,235,
    166,12,235,166,0,0,175,247,34,138,225,21,0,0,0,0,12,235,166,12,235,166,
    0,0,175,225,21,0,0,175,255,255,225,21,0,0,0,0,0,138,225,21,0,0,12,0,235,
    125,0,0,0,59,245,125,0,138,247,34,0,138,225,21,0,0,175,201,0,138,201,12,
    232,89,12,235,125,0,0,0,12,235,251,89,0,0,0,0,89,255,255,125,0,0,0,0,0,
    138,247,34,0,0,7,206,125,0,0,0,0,12,232,89,0,0,0,0,12,235,125,7,206,201,
    0,0,0,0,138,251,89,0,0,0,0,0,0,0,127,7,228,34,89,225,21,0,12,228,34,12,
    228,0,0,138,225,21,0,138,225,21,12,0,235,166,0,0,12,235,201,0,138,225,21,
    0,0,0,0,0,12,0,235,166,0,0,0,0,89,247,34,12,235,166,0,0,0,0,0,12,235,166,
    0,0,0,0,0,138,225,21,0,12,235,255,255,127,12,235,166,0,0,0,12,235,125,0,
    12,235,125,0,0,0,0,0,138,225,21,12,235,229,216,225,21,0,0,12,0,235,166,
    0,0,0,0,12,235,125,12,235,223,201,0,175,225,21,12,235,125,0,138,225,34,
    235,125,138,225,21,0,0,0,0,12,235,166,12,235,255,255,255,247,34,0,138,225,
    21,0,0,0,0,12,235,166,12,235,255,255,255,166,0,0,0,0,0,0,89,255,255,247,
    34,0,0,0,138,225,21,0,0,12,0,235,125,0,0,0,59,245,125,0,59,245,125,7,206,
    166,0,0,0,89,247,34,175,125,7,206,125,89,247,34,0,0,0,12,235,251,89,0,0,
    0,0,7,206,225,21,0,0,0,0,59,245,125,0,0,0,7,206,125,0,0,0,0,0,175,166,0,
    0,0,0,12,235,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,12,228,34,89,225,
    21,0,12,228,34,59,215,0,7,206,255,255,255,255,251,89,12,0,235,166,0,0,0,
    138,247,0,124,247,34,0,0,0,0,0,12,0,235,166,0,0,0,0,138,225,21,12,235,166,
    0,0,0,0,0,12,235,166,0,0,0,0,0,89,247,34,0,0,0,0,175,127,12,235,166,0,0,
    0,12,235,125,0,12,235,125,0,0,0,0,0,138,225,21,12,235,166,59,245,201,0,
    0,12,0,235,166,0,0,0,0,12,235,125,0,138,251,89,0,175,225,21,12,235,125,
    0,12,235,138,235,125,89,247,34,0,0,0,0,59,245,125,12,235,166,0,0,0,0,0,
    89,247,34,0,0,0,0,59,245,125,12,235,166,0,175,247,34,0,0,0,0,0,0,0,59,245,
    166,0,0,0,138,225,21,0,0,12,0,235,125,0,0,0,59,241,89,0,7,206,201,59,241,
    89,0,0,0,59,241,102,232,89,0,138,201,138,225,21,0,0,0,175,201,175,225,21,
    0,0,0,0,175,225,21,0,0,0,7,206,201,0,0,0,0,7,206,125,0,0,0,0,0,89,225,21,
    0,0,0,12,235,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,12,232,89,59,241,
    89,0,89,247,34,89,201,0,59,241,89,0,0,7,206,166,12,0,235,166,0,0,0,138,
    225,0,81,245,166,0,0,0,0,0,12,0,235,166,0,0,0,12,235,166,0,12,235,166,0,
    0,0,0,0,12,235,166,0,0,0,0,0,12,235,166,0,0,0,0,175,127,12,235,166,0,0,
    0,12,235,125,0,12,235,125,0,0,0,0,0,138,225,21,12,235,166,0,89,255,166,
    0,12,0,235,166,0,0,0,0,12,235,125,0,12,182,0,0,175,225,21,12,235,125,0,
    0,138,232,245,125,12,235,125,0,0,0,0,138,247,34,12,235,166,0,0,0,0,0,12,
    235,125,0,0,0,0,138,247,34,12,235,166,0,7,206,225,21,0,0,0,0,0,0,12,235,
    166,0,0,0,138,225,21,0,0,12,0,235,166,0,0,0,89,251,89,0,0,138,247,163,225,
    21,0,0,0,7,206,200,215,21,0,89,225,187,166,0,0,0,89,251,89,12,235,166,0,
    0,0,0,175,225,21,0,0,0,138,247,34,0,0,0,0,7,206,125,0,0,0,0,0,12,232,89,
    0,0,0,12,235,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,12,175,166,0,89,
    255,255,210,235,255,255,125,0,138,225,21,0,0,0,138,247,47,0,235,166,0,0,
    59,245,166,0,0,138,255,125,0,0,7,202,102,0,235,166,0,0,12,235,225,21,0,
    12,235,166,0,0,0,0,0,12,235,166,0,0,0,0,0,0,138,255,125,0,0,0,175,127,12,
    235,166,0,0,0,12,235,125,0,12,235,125,0,0,0,0,7,206,201,0,12,235,166,0,
    0,138,255,125,12,0,235,166,0,0,0,12,0,235,125,0,0,0,0,0,175,225,21,12,235,
    125,0,0,12,235,255,125,0,89,255,125,0,0,89,251,89,0,12,235,166,0,0,0,0,
    0,0,89,255,125,0,0,89,255,125,0,12,235,166,0,0,12,235,201,0,138,166,0,0,
    0,138,251,89,0,0,0,138,225,21,0,0,0,0,138,247,34,0,7,206,225,21,0,0,12,
    235,255,166,0,0,0,0,0,175,255,201,0,0,12,235,255,125,0,0,12,235,166,0,0,
    138,251,89,0,0,0,175,225,21,0,0,89,251,89,0,0,0,0,0,7,206,125,0,0,0,0,0,
    0,175,166,0,0,0,12,235,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,59,241,
    89,0,0,0,0,0,0,0,0,7,206,166,0,0,0,0,59,245,138,0,235,255,255,255,255,125,
    0,0,0,0,59,245,255,255,255,201,12,0,235,255,255,255,255,166,0,0,0,12,235,
    255,255,255,255,255,127,12,235,166,0,0,0,0,0,0,0,59,245,255,255,255,225,
    21,12,235,166,0,0,0,12,235,125,89,255,255,255,210,127,235,255,255,225,21,
    0,12,235,166,0,0,0,175,255,127,0,235,255,255,255,247,47,0,235,125,0,0,0,
    0,0,175,225,21,12,235,125,0,0,0,138,255,125,0,0,0,175,255,255,201,0,0,0,
    12,235,166,0,0,0,0,0,0,0,0,175,255,255,201,0,0,0,12,235,166,0,0,0,89,255,
    225,34,235,255,255,255,247,34,0,0,0,0,138,225,21,0,0,0,0,0,138,255,255,
    255,201,0,0,0,0,0,175,251,89,0,0,0,0,0,89,255,166,0,0,7,206,247,34,0,7,
    206,225,21,0,0,7,206,225,21,0,0,175,225,21,0,0,138,255,255,255,255,255,
    255,166,7,206,125,0,0,0,0,0,0,138,201,0,0,0,12,235,125,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,127,0,0,89,255,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,201,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,7,206,125,0,0,0,0,0,0,59,238,34,0,0,12,235,125,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,127,0,0,0,7,206,255,255,255,225,21,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,251,89,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,7,206,125,0,0,0,0,0,0,7,206,125,0,0,12,235,125,0,0,0,
    0,0,0,0,0,0,0,245,255,255,255,255,255,255,127,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,
    255,255,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,255,255,201,0,0,0,0,0,0,89,89,255,255,
    255,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,127,127,127,127,127,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,
    127,0,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,
    0,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,
    127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,0,127,
    127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,127,127,127,127,127,127,0,127,
    127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,
    127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,
    127,0,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,59,245,166,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,89,247,34,0,0,0,0,0,0,0,0,0,0,12,235,
    125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,225,21,0,0,0,0,0,0,0,0,0,12,235,
    255,247,0,0,0,0,0,0,0,12,12,235,125,0,0,0,0,0,0,0,0,0,0,0,0,12,235,125,
    0,0,0,0,0,12,235,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,255,251,89,0,7,206,125,0,89,255,251,89,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,4,4,4,4,4,84,84,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,7,206,
    125,0,0,0,0,0,0,0,0,0,0,12,235,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,
    225,21,0,0,0,0,0,0,0,0,0,175,201,0,0,0,0,0,0,0,0,0,12,12,235,125,0,0,0,
    0,0,59,245,102,0,89,247,34,12,235,125,0,0,0,0,0,12,235,125,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    59,241,89,0,0,0,7,206,125,0,0,0,138,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    4,4,4,4,4,4,4,4,100,252,252,84,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,
    235,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,225,21,0,0,0,0,0,0,0,12,0,
    235,125,0,0,0,0,0,0,0,0,0,0,12,235,125,0,0,0,0,0,0,0,0,0,0,0,0,12,235,125,
    0,0,0,0,0,12,235,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,
    125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,247,34,0,0,0,7,206,125,0,0,0,59,238,
    34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,4,4,4,20,236,252,164,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,
    0,0,0,0,0,0,0,12,235,255,255,255,166,0,12,235,166,245,255,247,34,0,0,12,
    235,255,255,247,34,0,34,235,255,255,255,225,21,0,12,235,255,255,225,29,
    0,206,255,255,255,127,0,12,235,255,255,255,225,21,12,235,138,235,255,247,
    34,0,12,235,102,175,255,247,34,12,235,125,0,59,245,201,0,12,235,125,12,
    0,235,166,245,255,225,29,206,255,251,89,0,12,235,138,235,255,247,34,0,0,
    12,235,255,255,201,0,0,12,235,166,245,255,251,89,0,0,12,235,255,255,255,
    225,21,12,235,138,235,247,127,34,138,255,255,255,206,0,206,255,255,255,
    201,59,241,89,0,0,89,247,42,206,201,0,0,0,138,225,187,201,0,0,138,225,21,
    0,59,241,187,226,247,34,0,7,206,206,206,201,0,0,0,138,225,151,255,255,255,
    255,247,0,0,89,247,34,0,0,0,7,206,125,0,0,0,59,238,34,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,4,4,4,4,4,4,4,148,252,236,20,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,12,206,
    21,0,59,245,125,12,235,247,34,0,138,225,21,12,235,166,0,0,134,102,0,235,
    166,0,0,138,225,21,12,235,125,0,0,175,201,12,0,235,125,0,0,12,235,166,0,
    0,138,225,21,12,235,247,34,0,175,201,0,12,235,102,0,89,247,34,12,235,125,
    12,235,166,0,0,12,235,125,12,0,235,225,21,12,235,251,89,0,175,201,0,12,
    235,247,34,0,175,201,0,12,235,166,0,7,206,201,0,12,235,225,21,0,175,225,
    21,12,235,166,0,0,138,225,21,12,235,247,34,0,0,89,247,34,0,12,206,34,0,
    235,125,0,0,59,241,89,0,0,89,247,34,89,247,34,0,7,206,166,138,225,21,7,
    206,251,89,0,89,225,138,34,235,201,0,138,225,21,89,247,34,0,7,206,166,0,
    0,0,7,206,166,0,0,89,225,21,0,0,0,7,206,125,0,0,0,59,241,89,0,0,0,0,138,
    251,89,0,0,7,202,89,0,0,4,4,4,4,4,4,52,252,252,108,4,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,
    0,0,0,0,0,0,0,0,7,206,102,12,235,125,0,0,59,241,89,138,225,21,0,0,0,34,
    89,225,21,0,0,138,225,21,89,225,21,0,0,89,247,47,0,235,125,0,0,89,225,21,
    0,0,138,225,21,12,235,125,0,0,89,247,34,12,235,102,0,89,247,34,12,235,138,
    235,166,0,0,0,12,235,125,12,0,235,125,0,7,206,166,0,0,138,225,21,12,235,
    125,0,0,89,247,34,138,225,21,0,0,59,238,34,12,235,125,0,0,59,241,89,89,
    225,21,0,0,138,225,21,12,235,125,0,0,0,138,225,21,0,0,0,12,0,235,125,0,
    0,59,241,89,0,0,89,247,34,12,235,125,0,59,241,89,59,238,34,12,228,198,166,
    0,175,166,59,0,89,251,132,241,89,0,12,235,125,0,59,238,34,0,0,0,138,225,
    21,0,12,235,166,0,0,0,0,7,206,125,0,0,0,0,175,201,0,0,0,138,166,12,235,
    166,0,12,232,89,0,0,12,84,4,4,4,4,204,252,204,4,4,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,
    0,0,0,59,245,255,255,255,102,12,235,125,0,0,12,235,125,175,201,0,0,0,0,
    0,175,201,0,0,0,138,225,21,175,255,255,255,255,255,247,47,0,235,125,0,0,
    175,201,0,0,0,138,225,21,12,235,125,0,0,89,247,34,12,235,102,0,89,247,34,
    12,235,255,225,21,0,0,0,12,235,125,12,0,235,125,0,7,206,166,0,0,138,225,
    21,12,235,125,0,0,89,247,34,175,201,0,0,0,12,232,89,12,235,125,0,0,12,235,
    125,175,201,0,0,0,138,225,21,12,235,125,0,0,0,59,245,255,247,34,0,12,0,
    235,125,0,0,59,241,89,0,0,89,247,34,0,175,201,0,138,201,0,12,235,125,89,
    201,89,225,29,206,125,12,0,0,175,255,166,0,0,0,175,201,0,138,201,0,0,0,
    89,251,89,0,138,247,34,0,0,0,0,0,7,206,125,0,0,0,0,0,89,255,125,7,202,89,
    0,89,251,89,89,201,0,0,0,172,252,84,4,4,100,252,252,60,4,4,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,
    0,0,0,0,0,0,0,89,255,166,0,7,206,102,12,235,125,0,0,12,235,125,175,201,
    0,0,0,0,0,175,201,0,0,0,138,225,21,175,201,0,0,0,0,0,12,0,235,125,0,0,175,
    201,0,0,0,138,225,21,12,235,125,0,0,89,247,34,12,235,102,0,89,247,34,12,
    235,138,235,201,0,0,0,12,235,125,12,0,235,125,0,7,206,166,0,0,138,225,21,
    12,235,125,0,0,89,247,34,175,201,0,0,0,12,232,89,12,235,125,0,0,12,235,
    125,175,201,0,0,0,138,225,21,12,235,125,0,0,0,0,0,138,255,255,201,12,0,
    235,125,0,0,59,241,89,0,0,89,247,34,0,89,247,42,206,125,0,0,175,166,175,
    125,12,232,102,232,89,0,0,0,175,255,201,0,0,0,89,247,47,235,125,0,0,12,
    235,166,0,0,0,12,235,125,0,0,0,0,7,206,125,0,0,0,0,138,201,0,0,12,232,89,
    0,0,59,245,225,21,0,0,0,196,252,244,60,20,236,252,156,4,4,4,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,
    0,0,0,0,0,0,0,175,201,0,0,7,206,102,12,235,125,0,0,59,241,89,138,225,21,
    0,0,0,34,89,225,21,0,0,138,225,21,138,247,34,0,0,0,0,12,0,235,125,0,0,138,
    225,21,0,0,138,225,21,12,235,125,0,0,89,247,34,12,235,102,0,89,247,34,12,
    235,125,59,245,125,0,0,12,235,125,12,0,235,125,0,7,206,166,0,0,138,225,
    21,12,235,125,0,0,89,247,34,138,225,21,0,0,89,247,34,12,235,125,0,0,59,
    241,89,138,225,21,0,0,138,225,21,12,235,125,0,0,0,0,0,0,0,89,247,47,0,235,
    125,0,0,59,241,89,0,0,89,247,34,0,12,235,166,238,34,0,0,138,210,228,34,
    0,175,166,215,21,0,0,89,251,159,251,89,0,0,12,235,191,247,34,0,0,175,225,
    21,0,0,0,0,138,225,21,0,0,0,7,206,125,0,0,0,12,232,89,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,20,220,252,236,180,252,244,28,4,4,4,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,
    138,225,21,0,138,255,102,12,235,125,0,7,206,201,0,12,235,166,0,0,134,132,
    0,245,125,0,59,245,225,21,12,235,201,0,0,12,206,34,0,235,125,0,0,59,245,
    125,0,12,235,225,21,12,235,125,0,0,89,247,34,12,235,102,0,89,247,34,12,
    235,125,0,138,251,89,0,12,235,125,12,0,235,125,0,7,206,166,0,0,138,225,
    21,12,235,125,0,0,89,247,34,12,235,166,0,7,206,201,0,12,235,125,0,7,206,
    201,0,59,245,125,0,12,235,225,21,12,235,125,0,0,0,138,125,0,0,138,225,29,
    0,206,166,0,0,7,206,166,0,59,245,247,34,0,0,175,255,201,0,0,0,59,245,225,
    21,0,89,255,201,0,0,12,235,166,0,175,225,21,0,0,138,255,166,0,0,89,251,
    89,0,0,0,0,0,89,247,34,0,0,0,7,206,125,0,0,0,59,238,34,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,4,36,236,252,252,252,108,4,4,4,4,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,
    7,206,255,255,171,206,102,12,232,226,255,255,225,21,0,0,12,235,255,255,
    247,34,0,89,255,255,247,163,225,21,0,7,206,255,255,247,34,12,0,235,125,
    0,0,0,89,255,255,247,163,225,21,12,235,125,0,0,89,247,34,12,235,102,0,89,
    247,34,12,235,125,0,0,175,251,34,0,235,125,12,0,235,125,0,7,206,166,0,0,
    138,225,21,12,235,125,0,0,89,247,34,0,12,235,255,255,201,0,0,12,235,255,
    255,255,225,21,0,0,89,255,255,247,163,225,21,12,235,125,0,0,0,89,255,255,
    255,247,34,0,0,89,255,255,127,0,59,245,255,225,111,247,34,0,0,59,245,125,
    0,0,0,12,235,166,0,0,59,245,125,7,0,206,225,21,0,12,235,201,0,0,59,241,
    89,0,0,175,255,255,255,255,247,0,0,89,247,34,0,0,0,7,206,125,0,0,0,59,238,
    34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,60,252,252,204,4,4,4,4,4,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,201,0,0,0,0,0,0,0,0,0,0,0,0,0,
    89,247,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,12,235,125,0,0,0,0,0,0,0,0,0,0,138,225,21,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,138,225,21,0,0,0,0,0,0,0,0,0,0,89,247,34,0,0,0,7,
    206,125,0,0,0,59,238,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,76,252,60,4,4,
    4,4,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,199,34,0,12,232,89,0,0,
    0,0,0,0,0,0,0,0,0,0,0,138,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,125,0,0,0,0,0,0,0,0,0,
    0,138,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,166,0,0,0,0,0,0,0,0,
    0,0,0,12,235,125,0,0,0,7,206,125,0,0,0,138,225,21,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,76,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    12,235,255,247,34,0,0,0,0,0,0,0,0,0,0,0,0,0,255,251,89,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,
    125,0,0,0,0,0,0,0,0,0,0,138,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,
    247,34,0,0,0,0,0,0,0,0,0,0,0,0,89,255,251,89,0,7,206,125,0,89,255,247,34,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,
    127,127,127,0,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,
    127,127,127,127,0,127,127,0,127,127,127,0,127,127,127,127,127,127,127,0,
    127,127,127,0,127,127,127,127,127,127,127,127,127,127,127,0,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,
    127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,
    127,127,0,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,
    127,127,0,127,127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,
    127,0,127,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,
    127,127,0,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,127,127,127,127,127,127,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,125,0,175,166,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,206,125,0,175,166,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,245,225,21,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,255,125,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,255,125,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,166,0,138,201,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,166,0,138,201,0,7,206,166,12,235,
    125,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,125,0,0,0,0,0,175,125,
    0,0,0,0,0,175,171,206,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,255,125,0,
    31,206,130,255,166,175,247,34,0,0,89,255,125,175,247,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,245,247,34,138,166,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,59,241,132,238,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,59,241,132,238,34,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,
    59,245,255,255,255,125,0,12,235,255,255,255,255,255,225,21,0,0,0,0,0,0,
    0,0,175,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,125,0,0,0,
    0,0,175,125,0,0,0,0,89,225,21,59,238,34,0,0,138,255,255,201,0,0,0,59,215,
    21,0,0,0,0,0,0,0,0,0,12,235,255,255,255,247,34,0,0,0,0,0,0,0,12,235,255,
    255,255,255,255,255,255,255,251,89,0,12,235,255,255,255,255,255,225,21,
    0,89,255,255,255,255,255,255,125,0,12,235,255,255,255,255,255,225,21,0,
    0,12,235,255,255,255,255,255,225,21,7,206,201,0,50,206,56,255,201,12,235,
    125,0,0,138,225,29,206,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,7,202,89,89,255,225,21,0,89,255,255,255,225,81,245,201,0,138,251,
    89,0,0,138,255,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,255,255,
    255,255,255,225,21,0,0,0,138,255,166,7,206,225,21,0,0,0,138,247,34,0,0,
    0,0,127,0,89,255,125,0,0,0,0,0,12,146,0,0,0,0,0,144,21,0,0,0,0,0,0,0,89,
    247,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,125,0,0,0,0,0,175,
    125,0,0,0,0,0,0,0,0,0,0,0,59,241,89,12,235,125,0,0,172,89,0,0,0,0,0,0,0,
    0,0,12,235,166,0,0,7,202,89,0,0,0,0,0,0,89,255,201,0,0,12,235,125,0,0,0,
    0,0,0,12,146,0,0,0,0,0,144,21,0,0,0,0,0,0,138,247,34,0,12,146,0,0,0,0,0,
    144,21,0,0,12,146,0,0,0,0,0,144,21,0,89,225,21,71,157,22,191,225,21,175,
    201,0,7,206,125,59,238,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,175,125,0,59,196,199,47,206,184,89,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,146,0,0,0,0,0,144,21,0,0,0,0,
    0,0,0,59,245,125,0,0,59,245,125,0,0,0,0,0,127,12,235,166,0,0,0,0,0,0,12,
    146,0,0,0,0,0,144,21,0,0,0,0,0,0,0,175,201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,12,235,255,255,255,255,127,34,235,255,255,255,255,225,21,0,
    0,0,0,0,0,0,0,89,247,34,7,206,166,0,89,201,0,0,0,0,0,0,0,0,0,0,89,247,34,
    0,0,0,0,0,0,0,0,59,115,12,235,166,0,0,0,12,235,125,0,0,0,0,0,0,12,146,0,
    0,0,0,0,144,21,0,0,0,0,0,59,245,125,0,0,12,146,0,0,0,0,0,144,21,0,0,12,
    146,0,0,0,0,0,144,21,0,7,202,89,117,104,0,29,202,89,59,215,21,59,215,21,
    138,201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,175,125,0,59,192,89,223,125,172,89,0,138,255,255,255,201,12,182,
    0,0,0,0,0,175,255,255,125,0,89,255,255,247,34,0,0,12,146,0,0,0,0,0,144,
    21,0,138,255,255,255,255,247,34,138,247,34,7,206,201,0,0,0,0,0,0,127,89,
    251,89,0,0,0,0,0,0,12,146,0,0,0,0,0,144,21,0,0,0,0,0,0,7,206,166,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,125,0,0,0,0,0,175,125,0,0,0,
    0,0,0,0,0,0,0,0,89,247,34,7,206,166,7,202,89,0,0,0,0,0,0,0,0,0,0,89,255,
    125,0,0,0,0,0,0,0,89,255,125,89,247,34,0,0,0,12,235,125,0,0,0,0,0,0,12,
    146,0,0,0,0,0,144,21,0,0,0,0,7,206,201,0,0,0,12,146,0,0,0,0,0,144,21,0,
    0,12,146,0,0,0,0,0,144,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,245,
    255,201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,
    125,0,59,192,12,228,34,172,89,89,247,34,0,12,206,29,206,201,0,0,7,206,166,
    0,7,206,255,225,21,0,89,247,34,0,12,146,0,0,0,0,0,144,21,0,0,0,0,7,206,
    166,0,12,235,166,89,247,34,0,0,0,0,0,0,127,245,255,255,255,255,255,201,
    0,0,12,146,0,0,0,0,0,144,21,0,0,0,0,0,59,245,255,255,255,127,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,125,0,0,0,0,0,175,125,0,0,0,0,0,0,0,
    0,0,0,0,59,241,89,12,235,125,89,201,12,235,255,251,89,0,89,255,255,225,
    21,0,175,255,255,225,21,0,0,0,89,251,89,0,138,225,21,0,0,0,12,235,255,255,
    255,255,225,21,0,12,146,0,0,0,0,0,144,21,0,0,0,0,138,247,34,0,0,0,12,146,
    0,0,0,0,0,144,21,0,0,12,146,0,0,0,0,0,144,21,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,12,235,255,255,255,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,175,125,0,59,192,0,0,0,172,89,138,225,21,0,0,0,
    0,7,206,225,21,138,225,21,0,0,89,251,89,0,0,12,235,125,0,12,146,0,0,0,0,
    0,144,21,0,0,0,0,138,225,21,0,0,89,255,255,125,0,0,0,0,0,0,0,127,138,225,
    21,0,0,0,0,0,0,12,146,0,0,0,0,0,144,21,0,0,0,0,0,0,59,241,89,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,125,0,0,12,235,255,255,255,255,
    225,21,0,0,0,0,0,0,0,0,0,138,255,255,201,12,228,34,175,166,0,138,201,7,
    206,125,7,206,166,0,0,0,89,255,255,247,34,59,241,89,0,0,138,225,21,0,0,
    0,12,235,125,0,0,0,0,0,0,12,146,0,0,0,0,0,144,21,0,0,0,59,245,125,0,0,0,
    0,12,146,0,0,0,0,0,144,21,0,0,12,146,0,0,0,0,0,144,21,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,59,245,255,255,255,207,235,255,255,255,255,255,255,
    207,235,255,255,255,255,255,255,255,255,255,255,255,225,21,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,59,245,255,247,34,0,0,0,0,175,166,175,201,0,0,0,
    59,245,255,255,255,255,255,125,0,12,146,0,0,0,0,0,144,21,0,0,0,89,251,89,
    0,0,0,7,206,225,21,0,0,0,0,0,0,0,127,245,255,255,255,255,255,125,0,0,12,
    146,0,0,0,0,0,144,21,0,0,0,0,0,0,59,241,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,175,125,0,0,0,0,0,175,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,138,166,7,206,125,0,89,247,94,241,89,0,138,201,0,0,0,0,0,59,245,166,
    0,89,251,89,0,89,247,34,0,0,0,12,235,125,0,0,0,0,0,0,12,146,0,0,0,0,0,144,
    21,0,0,7,206,201,0,0,0,0,0,12,146,0,0,0,0,0,144,21,0,0,12,146,0,0,0,0,0,
    144,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,255,255,255,166,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,138,255,255,201,0,7,206,225,21,175,201,0,0,0,59,241,89,0,0,0,0,
    0,0,12,146,0,0,0,0,0,144,21,0,0,12,235,166,0,0,0,0,0,175,225,21,0,0,0,0,
    0,0,0,127,89,255,125,0,0,0,0,0,0,12,146,0,0,0,0,0,144,21,0,0,0,0,0,0,89,
    247,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,125,0,0,0,0,0,
    175,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,228,34,7,206,125,0,89,247,94,241,
    89,0,138,201,0,0,0,0,0,12,235,166,0,0,89,255,125,12,235,166,0,0,0,12,235,
    125,0,0,0,0,0,0,12,146,0,0,0,0,0,144,21,0,0,138,247,34,0,0,0,0,0,12,146,
    0,0,0,0,0,144,21,0,0,12,146,0,0,0,0,0,144,21,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,59,245,255,201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,247,42,206,201,0,0,89,
    225,21,0,0,89,255,125,0,0,0,0,0,0,12,146,0,0,0,0,0,144,21,0,0,175,225,21,
    0,0,0,0,0,175,225,21,0,0,0,0,0,0,0,127,0,175,251,89,0,0,0,0,0,12,146,0,
    0,0,0,0,144,21,0,59,245,166,0,0,138,225,21,0,0,0,59,245,166,138,251,89,
    7,206,201,0,12,235,125,0,59,241,89,0,0,0,175,125,0,0,0,0,0,175,125,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,138,166,0,0,175,166,0,138,201,7,206,125,7,206,
    166,138,166,0,0,0,138,251,89,0,0,0,59,115,0,89,255,201,0,0,12,235,125,0,
    0,0,0,0,0,12,146,0,0,0,0,0,144,21,0,89,251,89,0,0,0,0,0,0,12,146,0,0,0,
    0,0,144,21,0,0,12,146,0,0,0,0,0,144,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,138,125,0,0,138,225,34,182,0,0,0,7,206,166,0,
    7,206,255,247,34,0,0,175,125,0,12,146,0,0,0,0,0,144,21,0,89,251,89,0,0,
    0,0,0,0,175,225,21,0,0,0,0,0,0,0,127,0,0,138,255,255,255,255,125,0,12,235,
    255,255,255,255,255,225,21,0,138,247,34,0,7,206,166,0,0,0,0,89,247,34,175,
    201,0,7,206,201,0,12,235,125,0,59,241,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,59,215,21,0,0,12,235,255,251,89,0,89,255,255,225,
    21,12,235,255,255,255,247,34,0,0,0,0,0,0,0,0,12,235,255,255,255,255,255,
    255,255,255,251,89,0,12,235,255,255,255,255,255,225,21,0,138,255,255,255,
    255,255,255,166,0,12,235,255,255,255,255,255,225,21,0,0,12,235,255,255,
    255,255,255,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,89,255,255,255,247,34,0,0,0,0,0,0,0,175,255,255,125,0,138,255,255,
    255,125,0,0,12,235,255,255,255,255,255,225,21,0,175,255,255,255,255,247,
    0,0,0,175,225,21,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,175,166,0,255,255,201,0,0,0,0,0,175,166,12,232,89,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,
    228,34,0,0,0,0,0,0,0,0,12,232,89,59,215,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,
    127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,
    127,127,127,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,
    127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,
    127,127,127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,
    0,127,127,127,127,0,127,127,127,127,127,127,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,
    127,0,127,127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,
    127,127,0,127,127,127,0,127,127,0,127,127,127,127,127,0,127,127,127,127,
    127,0,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,
    127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,
    127,127,127,127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,
    0,127,127,127,127,127,127,127,127,127,127,127,127,127,0,127,127,127,127,
    127,127,127,127,127,0,127,127,127,127,127,0,127,127,127,127,127,127,127,
    127,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,245,255,255,255,255,255,255,225,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,225,21,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,7,206,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,247,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,89,247,
    34,0,0,0,0,59,192,0,0,0,0,0,7,206,255,255,225,21,0,0,0,0,0,0,0,0,138,247,
    34,0,0,89,251,89,0,7,206,125,0,0,7,206,255,255,255,166,0,89,251,89,138,
    247,34,0,0,0,0,7,206,255,255,255,247,34,0,0,0,0,175,255,255,251,89,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,255,255,255,247,
    34,0,0,0,0,0,0,0,0,0,0,0,0,89,255,255,247,34,0,0,0,0,0,0,0,0,0,0,0,0,12,
    235,255,247,34,0,0,7,206,255,251,89,0,0,7,206,125,0,0,0,0,0,0,0,0,0,0,0,
    0,89,255,255,255,255,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,202,89,0,0,0,
    59,245,255,247,34,0,0,0,0,0,0,0,0,0,0,0,89,201,0,0,0,0,175,166,0,0,0,0,
    0,0,89,201,0,0,0,0,175,166,0,0,0,0,0,59,245,255,201,0,0,0,59,241,89,0,0,
    0,0,0,59,245,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,89,247,34,
    0,0,0,0,59,192,0,0,0,0,0,175,201,0,0,144,21,0,0,0,0,0,0,0,0,7,206,166,0,
    7,206,166,0,0,7,206,125,0,7,206,201,0,0,89,166,0,0,0,0,0,0,0,0,0,0,89,255,
    125,0,0,0,59,245,166,0,0,0,0,0,0,12,206,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,89,255,125,0,0,0,59,245,166,0,0,0,0,0,0,0,0,0,0,59,
    241,89,0,138,201,0,0,0,0,0,138,166,0,0,0,0,0,168,34,7,206,166,0,0,172,89,
    0,175,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,255,255,255,166,89,225,21,
    0,0,0,0,0,0,0,0,0,0,0,0,0,89,255,251,89,0,0,12,235,125,0,138,225,21,0,0,
    0,0,0,0,0,0,7,206,255,201,0,0,0,89,225,21,0,0,0,0,7,206,255,201,0,0,0,89,
    225,21,0,0,0,0,12,206,21,12,235,125,0,0,175,166,0,0,0,0,0,0,59,245,125,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,175,255,255,255,
    166,0,0,12,235,125,0,0,0,0,89,225,21,0,0,12,232,89,0,89,247,34,89,247,34,
    0,0,7,206,125,0,12,235,125,0,0,0,0,0,0,0,0,0,0,0,0,0,89,225,21,0,0,0,0,
    0,7,206,125,0,0,7,206,255,255,247,34,0,0,0,85,89,0,85,89,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,89,225,21,0,0,0,0,0,7,206,125,0,0,0,0,0,0,0,0,0,89,201,
    0,0,12,228,34,0,0,0,0,138,166,0,0,0,0,0,0,0,7,206,125,0,0,7,206,255,166,
    0,0,0,0,0,0,0,0,0,12,235,125,0,0,89,247,34,175,255,255,255,166,89,225,21,
    0,89,255,125,0,0,0,0,0,0,0,0,0,0,7,202,89,0,0,89,225,21,0,12,232,89,59,
    115,0,59,115,0,0,0,0,0,89,201,0,0,7,206,125,0,0,0,0,0,0,0,89,201,0,0,7,
    206,125,0,0,0,0,0,0,0,0,12,232,89,0,59,238,34,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,89,225,21,0,0,138,247,94,192,12,182,
    0,0,12,235,125,0,0,0,0,0,175,255,255,255,255,166,0,0,7,206,171,206,166,
    0,0,0,7,206,125,0,7,206,251,89,0,0,0,0,0,0,0,0,0,0,0,7,202,89,0,59,245,
    255,255,201,0,12,228,34,12,235,166,0,12,228,34,0,0,138,251,89,138,247,34,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,202,89,0,138,255,255,255,125,0,12,228,34,
    0,0,0,0,0,0,0,0,59,241,89,0,138,201,0,0,0,0,0,138,166,0,0,0,0,0,0,0,175,
    201,0,0,0,0,0,0,175,201,0,0,0,0,0,0,0,0,12,235,125,0,0,89,247,34,175,255,
    255,255,166,89,225,21,0,89,255,125,0,0,0,0,0,0,0,0,0,0,7,202,89,0,0,138,
    225,21,0,12,235,125,12,235,166,59,245,166,0,0,0,0,89,201,0,0,89,225,21,
    0,0,0,0,0,0,0,89,201,0,0,89,225,21,0,0,0,0,0,0,12,235,255,125,0,0,175,125,
    0,0,0,0,0,0,0,12,235,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,89,
    225,21,0,12,235,125,59,192,0,0,0,0,12,235,125,0,0,0,0,0,59,215,21,59,238,
    34,0,0,0,89,255,247,34,0,0,0,7,206,125,0,0,7,206,255,255,247,34,0,0,0,0,
    0,0,0,0,59,192,0,12,235,166,0,7,176,21,0,175,125,59,238,34,0,12,228,34,
    0,138,247,34,138,247,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,192,0,0,138,
    201,0,89,247,34,0,175,125,0,0,0,0,0,0,0,0,0,89,255,255,225,21,0,7,206,255,
    255,255,255,255,255,247,34,0,12,235,125,0,0,0,7,176,21,0,175,201,0,0,0,
    0,0,0,0,0,12,235,125,0,0,89,247,34,89,255,255,255,166,89,225,21,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,7,202,89,0,0,89,225,21,0,12,232,89,0,12,235,166,12,
    235,166,0,0,0,89,201,0,7,206,125,0,12,235,166,0,0,0,0,89,201,0,7,206,125,
    89,255,255,255,125,0,0,0,0,7,206,125,89,225,21,0,138,225,21,0,0,0,138,255,
    125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,89,247,34,0,59,241,89,59,
    192,0,0,0,12,235,255,255,255,225,21,0,0,138,166,0,7,202,89,0,0,0,7,206,
    166,0,0,0,0,0,0,0,0,7,206,125,0,12,235,201,0,0,0,0,0,0,0,0,89,166,0,89,
    247,34,0,0,0,0,0,89,166,12,232,89,0,138,247,34,89,247,34,59,238,34,0,0,
    12,235,255,255,255,255,255,255,247,34,89,255,255,255,166,89,166,0,0,138,
    201,0,138,225,21,0,89,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,166,
    0,0,0,0,7,206,255,255,255,247,34,0,59,245,255,247,34,0,0,0,0,0,0,0,0,12,
    235,125,0,0,89,247,34,0,89,255,255,166,89,225,21,0,0,0,0,0,0,0,0,0,0,0,
    0,0,89,255,255,255,166,0,12,235,125,0,138,225,21,0,0,12,235,125,12,235,
    125,0,0,89,201,0,89,201,0,7,206,223,166,0,0,0,0,89,201,0,89,201,0,89,125,
    0,138,225,21,12,182,0,7,206,133,206,125,0,89,232,215,21,0,7,206,247,34,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,89,247,34,0,59,241,89,59,
    192,0,0,0,0,12,235,125,0,0,0,0,0,59,215,21,59,238,34,0,59,245,255,255,255,
    255,225,21,0,0,0,0,0,59,241,89,0,0,138,225,21,0,0,0,0,0,0,0,89,166,0,89,
    247,34,0,0,0,0,0,89,166,0,138,255,255,176,228,34,0,138,247,34,138,247,34,
    0,0,0,0,0,0,0,0,59,238,34,0,0,0,0,0,89,166,0,0,138,255,255,225,21,0,0,89,
    166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,166,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,125,0,0,89,247,34,0,0,0,138,166,89,225,
    21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,245,255,247,34,0,0,12,235,
    166,12,235,166,0,0,0,0,0,12,232,89,0,175,166,138,166,0,0,0,0,0,0,12,232,
    89,0,0,0,0,138,201,0,0,89,255,255,201,89,225,21,89,225,81,215,21,0,138,
    247,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,89,247,34,0,12,235,
    125,59,192,0,0,0,0,59,241,89,0,0,0,0,0,175,255,255,255,255,166,0,0,0,7,
    206,166,0,0,0,0,7,206,125,0,12,235,201,0,7,206,166,0,0,0,0,0,0,0,0,59,192,
    0,12,235,166,0,7,176,21,0,175,125,0,0,0,0,0,0,0,0,0,138,251,89,138,247,
    34,0,0,0,0,0,0,0,59,238,34,0,0,0,0,0,59,192,0,0,138,201,59,245,166,0,0,
    175,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,166,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,125,0,0,89,247,34,0,0,0,138,166,
    89,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,166,
    59,245,166,0,0,0,0,0,0,138,201,0,138,201,0,138,166,0,0,0,0,0,0,138,201,
    0,0,0,0,89,247,34,0,0,0,0,0,7,206,125,59,238,34,59,215,21,0,175,225,21,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,89,247,34,0,0,175,225,81,
    192,12,182,0,7,206,125,0,0,0,0,0,89,225,21,0,0,12,232,89,0,0,7,206,166,
    0,0,0,0,7,206,125,0,0,59,245,255,255,166,0,0,0,0,0,0,0,0,0,7,202,89,0,59,
    245,255,255,166,0,12,228,34,0,0,0,0,0,0,0,0,0,0,85,89,0,85,89,0,0,0,0,0,
    0,0,59,238,34,0,0,0,0,0,7,202,89,0,138,201,0,59,245,225,34,228,34,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,7,206,255,255,255,255,255,255,247,34,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,201,0,0,175,247,34,0,0,0,138,166,
    89,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,59,115,0,
    59,115,0,0,0,0,0,0,12,232,89,0,175,255,255,255,255,201,0,0,0,0,12,232,89,
    0,0,0,138,201,0,0,0,0,0,0,0,89,201,0,89,255,255,255,255,247,34,138,251,
    89,0,7,176,21,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,138,247,34,0,0,0,
    175,255,255,255,166,0,89,255,255,255,255,255,247,34,0,0,0,0,0,0,0,0,0,0,
    7,206,166,0,0,0,0,7,206,125,0,0,0,0,0,138,255,166,0,0,0,0,0,0,0,0,0,89,
    225,21,0,0,0,0,0,7,206,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,59,238,34,0,0,0,0,0,0,89,225,21,0,0,0,0,0,7,206,125,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,12,235,191,255,255,166,238,34,0,0,0,138,166,89,225,21,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,201,0,0,
    0,0,0,138,166,0,0,0,0,0,175,201,0,0,0,89,255,255,255,255,125,0,0,0,12,232,
    89,0,0,0,0,59,215,21,0,0,138,255,255,255,225,21,0,0,0,0,0,0,0,0,0,0,0,0,
    0,127,0,0,0,0,0,0,0,0,0,0,0,0,59,192,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,7,206,125,0,0,0,0,0,0,175,201,0,0,0,0,0,0,0,0,0,0,
    89,255,125,0,0,0,59,245,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,89,255,125,0,0,0,59,245,166,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,12,235,125,0,0,0,0,0,0,0,0,138,166,89,225,21,0,0,0,0,0,0,0,0,0,175,
    125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,59,192,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,125,0,7,199,34,
    0,12,235,125,0,0,0,0,0,0,0,0,0,0,0,7,206,255,255,255,247,34,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,255,255,
    255,247,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,125,0,0,0,0,0,0,0,0,138,166,
    89,225,21,0,0,0,0,0,0,0,0,7,202,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,7,206,125,0,7,206,255,255,255,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,125,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,12,235,255,201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,
    127,127,0,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,
    127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,
    127,0,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,
    0,127,127,127,127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,
    127,0,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,
    0,127,127,127,127,0,127,127,127,127,127,127,127,127,127,127,127,127,0,127,
    127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,0,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,
    127,127,0,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,
    127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,
    127,127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,127,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,138,225,
    21,0,0,0,0,0,12,235,125,0,0,0,0,19,172,255,190,11,0,0,0,0,138,255,201,7,
    202,89,0,0,0,0,0,0,0,0,0,0,7,206,255,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,59,138,225,21,0,0,0,0,0,0,59,245,201,0,0,0,19,172,
    255,190,11,0,0,0,0,0,0,0,0,0,7,206,225,21,0,0,0,59,245,201,19,172,255,190,
    11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,251,89,89,201,0,0,0,0,0,175,
    201,0,0,0,0,0,0,0,0,7,206,225,21,0,0,0,0,0,19,172,255,190,11,0,0,0,0,0,
    175,255,166,12,228,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,89,255,125,0,0,0,0,0,0,0,12,175,247,34,0,0,0,19,172,255,
    190,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,247,34,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,127,0,0,0,7,206,125,0,0,0,0,0,138,201,0,0,0,0,0,136,190,
    45,196,145,0,0,0,59,215,21,175,255,166,0,0,0,175,225,29,206,166,0,0,7,202,
    89,7,202,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,209,125,
    0,0,0,0,0,0,138,225,21,0,0,0,136,190,45,196,145,0,0,0,175,225,29,206,166,
    0,0,12,235,125,0,0,12,138,225,21,136,190,45,196,145,159,251,89,138,247,
    34,0,0,0,0,0,0,0,0,0,0,0,175,125,59,245,247,34,0,0,0,0,0,12,232,89,0,0,
    0,0,0,0,0,175,166,0,0,0,0,0,0,12,136,190,45,196,145,0,0,0,0,138,166,12,
    235,255,125,0,0,0,0,7,206,166,12,235,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,12,232,89,0,0,0,0,0,0,138,201,0,0,0,0,0,136,190,45,
    196,145,34,0,0,0,89,251,89,138,247,34,0,0,0,0,0,138,201,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,202,89,7,202,89,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,255,
    255,125,0,0,0,0,127,0,0,7,206,251,89,0,0,0,0,7,206,251,89,0,0,0,0,7,206,
    251,89,0,0,0,0,0,7,206,251,89,0,0,0,0,7,206,251,89,0,0,0,0,12,235,255,125,
    0,0,0,0,0,89,255,255,255,255,255,255,255,255,125,0,0,0,59,245,255,255,255,
    201,12,235,255,255,255,255,255,125,12,235,255,255,255,255,255,125,12,235,
    255,255,255,255,255,125,12,235,255,255,255,255,255,125,89,255,255,255,201,
    89,255,255,255,201,89,255,255,255,201,89,255,255,255,201,0,175,255,255,
    255,255,201,0,0,0,12,235,251,89,0,0,12,235,125,0,0,0,138,255,255,166,0,
    0,0,0,0,0,138,255,255,166,0,0,0,0,0,0,138,255,255,166,0,0,0,0,0,0,138,255,
    255,166,0,0,0,0,0,0,138,255,255,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,
    255,255,201,89,251,89,12,235,125,0,0,0,59,245,125,12,235,125,0,0,0,59,245,
    125,12,235,125,0,0,0,59,245,125,12,235,125,0,0,0,59,245,125,7,206,225,21,
    0,0,0,138,247,0,235,166,0,0,0,0,0,0,138,225,21,7,206,166,0,0,0,127,0,0,
    59,245,255,166,0,0,0,0,59,245,255,166,0,0,0,0,59,245,255,166,0,0,0,0,0,
    59,245,255,166,0,0,0,0,59,245,255,166,0,0,0,0,59,245,255,166,0,0,0,0,0,
    175,201,7,206,166,0,0,0,0,0,0,0,138,255,125,0,0,7,202,102,235,166,0,0,0,
    0,0,12,235,166,0,0,0,0,0,12,235,166,0,0,0,0,0,12,235,166,0,0,0,0,0,0,12,
    235,125,0,0,12,235,125,0,0,12,235,125,0,0,12,235,125,0,0,175,201,0,0,7,
    206,251,89,0,12,235,255,201,0,0,12,235,125,0,59,245,166,0,0,138,251,89,
    0,0,59,245,166,0,0,138,251,89,0,0,59,245,166,0,0,138,251,89,0,0,59,245,
    166,0,0,138,251,89,0,0,59,245,166,0,0,138,251,89,0,0,0,0,0,0,0,0,0,0,0,
    0,59,245,166,0,0,89,255,166,0,12,235,125,0,0,0,59,245,125,12,235,125,0,
    0,0,59,245,125,12,235,125,0,0,0,59,245,125,12,235,125,0,0,0,59,245,125,
    0,59,245,125,0,0,59,245,125,12,235,166,0,0,0,0,0,12,235,125,0,0,175,201,
    0,0,0,127,0,0,138,225,151,225,21,0,0,0,138,225,151,225,21,0,0,0,138,225,
    151,225,21,0,0,0,0,138,225,151,225,21,0,0,0,138,225,151,225,21,0,0,0,138,
    225,151,225,21,0,0,0,59,241,89,7,206,166,0,0,0,0,0,0,12,235,166,0,0,0,0,
    0,12,235,166,0,0,0,0,0,12,235,166,0,0,0,0,0,12,235,166,0,0,0,0,0,12,235,
    166,0,0,0,0,0,0,12,235,125,0,0,12,235,125,0,0,12,235,125,0,0,12,235,125,
    0,0,175,201,0,0,0,0,175,225,21,12,235,166,245,125,0,12,235,125,12,235,125,
    0,0,0,0,138,247,34,12,235,125,0,0,0,0,138,247,34,12,235,125,0,0,0,0,138,
    247,34,12,235,125,0,0,0,0,138,247,34,12,235,125,0,0,0,0,138,247,34,0,138,
    225,21,0,0,0,175,201,0,12,235,125,0,0,7,202,159,247,34,12,235,125,0,0,0,
    59,245,125,12,235,125,0,0,0,59,245,125,12,235,125,0,0,0,59,245,125,12,235,
    125,0,0,0,59,245,125,0,0,138,247,34,7,206,201,0,12,235,255,255,255,251,
    89,0,12,235,125,0,12,235,125,0,0,0,127,0,7,206,166,59,241,89,0,0,7,206,
    166,59,241,89,0,0,7,206,166,59,241,89,0,0,0,7,206,166,59,241,89,0,0,7,206,
    166,59,241,89,0,0,7,206,166,59,241,89,0,0,0,138,225,21,7,206,166,0,0,0,
    0,0,0,89,247,34,0,0,0,0,0,12,235,166,0,0,0,0,0,12,235,166,0,0,0,0,0,12,
    235,166,0,0,0,0,0,12,235,166,0,0,0,0,0,0,12,235,125,0,0,12,235,125,0,0,
    12,235,125,0,0,12,235,125,0,0,175,201,0,0,0,0,59,241,89,12,235,125,138,
    225,21,12,235,125,89,247,34,0,0,0,0,59,245,125,89,247,34,0,0,0,0,59,245,
    125,89,247,34,0,0,0,0,59,245,125,89,247,34,0,0,0,0,59,245,125,89,247,34,
    0,0,0,0,59,245,125,0,0,175,225,21,0,175,225,21,0,89,247,34,0,0,138,166,
    12,235,125,12,235,125,0,0,0,59,245,125,12,235,125,0,0,0,59,245,125,12,235,
    125,0,0,0,59,245,125,12,235,125,0,0,0,59,245,125,0,0,12,235,166,89,247,
    34,0,12,235,166,0,0,138,251,89,12,235,133,206,255,125,0,0,0,0,127,0,59,
    241,89,7,206,166,0,0,59,241,89,7,206,166,0,0,59,241,89,7,206,166,0,0,0,
    59,241,89,7,206,166,0,0,59,241,89,7,206,166,0,0,59,241,89,7,206,166,0,0,
    12,235,125,0,7,206,255,255,255,255,247,34,0,138,225,21,0,0,0,0,0,12,235,
    255,255,255,255,247,34,12,235,255,255,255,255,247,34,12,235,255,255,255,
    255,247,34,12,235,255,255,255,255,247,34,0,12,235,125,0,0,12,235,125,0,
    0,12,235,125,0,0,12,235,125,0,206,255,255,255,247,34,0,12,235,125,12,235,
    125,12,235,125,12,235,125,138,225,21,0,0,0,0,12,235,166,138,225,21,0,0,
    0,0,12,235,166,138,225,21,0,0,0,0,12,235,166,138,225,21,0,0,0,0,12,235,
    166,138,225,21,0,0,0,0,12,235,166,0,0,0,175,225,187,225,21,0,0,138,225,
    21,0,59,215,21,7,206,166,12,235,125,0,0,0,59,245,125,12,235,125,0,0,0,59,
    245,125,12,235,125,0,0,0,59,245,125,12,235,125,0,0,0,59,245,125,0,0,0,89,
    255,255,125,0,0,12,235,166,0,0,12,235,166,12,235,125,0,7,206,201,0,0,0,
    127,0,138,225,21,0,138,225,21,0,138,225,21,0,138,225,21,0,138,225,21,0,
    138,225,21,0,0,138,225,21,0,138,225,21,0,138,225,21,0,138,225,21,0,138,
    225,21,0,138,225,21,0,89,255,255,255,255,255,166,0,0,0,0,0,0,138,225,21,
    0,0,0,0,0,12,235,166,0,0,0,0,0,12,235,166,0,0,0,0,0,12,235,166,0,0,0,0,
    0,12,235,166,0,0,0,0,0,0,12,235,125,0,0,12,235,125,0,0,12,235,125,0,0,12,
    235,125,0,0,175,201,0,0,0,0,12,235,125,12,235,125,0,138,225,34,235,125,
    138,225,21,0,0,0,0,12,235,166,138,225,21,0,0,0,0,12,235,166,138,225,21,
    0,0,0,0,12,235,166,138,225,21,0,0,0,0,12,235,166,138,225,21,0,0,0,0,12,
    235,166,0,0,0,0,175,225,21,0,0,0,138,225,21,7,202,89,0,7,206,166,12,235,
    125,0,0,0,59,245,125,12,235,125,0,0,0,59,245,125,12,235,125,0,0,0,59,245,
    125,12,235,125,0,0,0,59,245,125,0,0,0,7,206,225,21,0,0,12,235,166,0,0,12,
    235,166,12,235,125,0,0,59,241,89,0,0,127,7,206,255,255,255,255,251,89,7,
    206,255,255,255,255,251,89,7,206,255,255,255,255,251,89,0,7,206,255,255,
    255,255,251,89,7,206,255,255,255,255,251,89,7,206,255,255,255,255,251,89,
    7,206,166,0,0,7,206,166,0,0,0,0,0,0,89,247,34,0,0,0,0,0,12,235,166,0,0,
    0,0,0,12,235,166,0,0,0,0,0,12,235,166,0,0,0,0,0,12,235,166,0,0,0,0,0,0,
    12,235,125,0,0,12,235,125,0,0,12,235,125,0,0,12,235,125,0,0,175,201,0,0,
    0,0,59,241,89,12,235,125,0,12,235,138,235,125,89,247,34,0,0,0,0,59,245,
    125,89,247,34,0,0,0,0,59,245,125,89,247,34,0,0,0,0,59,245,125,89,247,34,
    0,0,0,0,59,245,125,89,247,34,0,0,0,0,59,245,125,0,0,0,175,225,187,225,21,
    0,0,138,247,34,175,125,0,0,12,235,125,12,235,125,0,0,0,59,241,89,12,235,
    125,0,0,0,59,241,89,12,235,125,0,0,0,59,241,89,12,235,125,0,0,0,59,241,
    89,0,0,0,0,175,225,21,0,0,12,235,166,0,0,175,247,34,12,235,125,0,0,12,235,
    125,0,0,127,59,241,89,0,0,7,206,166,59,241,89,0,0,7,206,166,59,241,89,0,
    0,7,206,166,0,59,241,89,0,0,7,206,166,59,241,89,0,0,7,206,166,59,241,89,
    0,0,7,206,166,59,241,89,0,0,7,206,166,0,0,0,0,0,0,59,245,166,0,0,0,0,0,
    12,235,166,0,0,0,0,0,12,235,166,0,0,0,0,0,12,235,166,0,0,0,0,0,12,235,166,
    0,0,0,0,0,0,12,235,125,0,0,12,235,125,0,0,12,235,125,0,0,12,235,125,0,0,
    175,201,0,0,0,0,175,225,21,12,235,125,0,0,138,232,245,125,12,235,125,0,
    0,0,0,138,247,34,12,235,125,0,0,0,0,138,247,34,12,235,125,0,0,0,0,138,247,
    34,12,235,125,0,0,0,0,138,247,34,12,235,125,0,0,0,0,138,247,34,0,0,175,
    225,21,0,175,225,21,0,59,245,191,201,0,0,0,89,225,21,12,235,166,0,0,0,89,
    251,89,12,235,166,0,0,0,89,251,89,12,235,166,0,0,0,89,251,89,12,235,166,
    0,0,0,89,251,89,0,0,0,0,175,225,21,0,0,12,235,255,255,255,247,34,0,12,235,
    125,0,0,59,241,89,0,0,127,138,225,21,0,0,0,138,247,163,225,21,0,0,0,138,
    247,163,225,21,0,0,0,138,247,34,138,225,21,0,0,0,138,247,163,225,21,0,0,
    0,138,247,163,225,21,0,0,0,138,247,198,225,21,0,0,7,206,166,0,0,0,0,0,0,
    0,138,255,125,0,0,7,202,102,235,166,0,0,0,0,0,12,235,166,0,0,0,0,0,12,235,
    166,0,0,0,0,0,12,235,166,0,0,0,0,0,0,12,235,125,0,0,12,235,125,0,0,12,235,
    125,0,0,12,235,125,0,0,175,201,0,0,7,206,251,89,0,12,235,125,0,0,12,235,
    255,125,0,89,255,125,0,0,89,251,89,0,0,89,255,125,0,0,89,251,89,0,0,89,
    255,125,0,0,89,251,89,0,0,89,255,125,0,0,89,251,89,0,0,89,255,125,0,0,89,
    251,89,0,0,138,225,21,0,0,0,175,201,0,0,138,251,89,0,0,89,251,89,0,0,138,
    247,34,0,7,206,225,21,0,138,247,34,0,7,206,225,21,0,138,247,34,0,7,206,
    225,21,0,138,247,34,0,7,206,225,21,0,0,0,0,175,225,21,0,0,12,235,166,0,
    0,0,0,0,12,235,125,0,0,175,225,21,0,0,127,206,166,0,0,0,0,59,245,255,166,
    0,0,0,0,59,245,255,166,0,0,0,0,59,245,133,206,166,0,0,0,0,59,245,255,166,
    0,0,0,0,59,245,255,166,0,0,0,0,59,245,255,125,0,0,0,7,206,255,255,255,255,
    255,125,0,0,0,59,245,255,255,255,201,12,235,255,255,255,255,255,125,12,
    235,255,255,255,255,255,125,12,235,255,255,255,255,255,125,12,235,255,255,
    255,255,255,125,89,255,255,255,201,89,255,255,255,201,89,255,255,255,201,
    89,255,255,255,201,0,175,255,255,255,255,225,21,0,0,12,235,125,0,0,0,138,
    255,125,0,0,0,175,255,255,201,0,0,0,0,0,0,175,255,255,201,0,0,0,0,0,0,175,
    255,255,201,0,0,0,0,0,0,175,255,255,201,0,0,0,0,0,0,175,255,255,201,0,0,
    0,0,0,0,0,0,0,0,0,0,0,7,202,97,206,255,255,201,0,0,0,0,0,138,255,255,255,
    201,0,0,0,0,138,255,255,255,201,0,0,0,0,138,255,255,255,201,0,0,0,0,138,
    255,255,255,201,0,0,0,0,0,0,175,225,21,0,0,12,235,166,0,0,0,0,0,12,235,
    133,206,255,225,21,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,138,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,138,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,175,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,255,225,21,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,127,127,127,127,
    127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,127,127,127,127,127,127,127,0,
    127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,
    127,0,127,127,127,127,0,127,127,127,127,0,127,127,127,127,0,127,127,127,
    127,0,127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,
    127,0,127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,
    127,127,0,127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,
    127,127,127,0,127,127,127,127,127,127,127,127,127,0,127,127,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,127,127,127,0,127,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,127,127,127,
    127,127,127,0,127,127,127,127,127,127,127,127,0,127,127,127,127,127,127,
    127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,
    0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,7,206,225,21,
    0,0,0,0,0,12,235,225,21,0,0,89,255,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,7,206,255,247,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,247,
    34,0,0,0,0,0,0,0,138,251,89,0,0,59,245,247,34,0,0,0,0,0,0,0,0,0,175,247,
    34,0,0,175,0,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,225,21,
    0,0,0,0,0,0,0,138,255,125,0,0,0,12,235,251,89,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,251,89,0,0,0,0,0,0,
    7,206,225,21,0,0,0,7,206,251,89,0,0,0,0,0,0,0,0,0,0,0,0,0,59,245,166,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,59,241,89,0,0,0,0,
    0,89,247,34,0,0,7,206,138,235,125,0,0,89,255,225,21,175,125,0,0,0,0,0,0,
    0,0,0,138,201,0,138,201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,
    235,125,0,0,0,0,0,0,12,235,125,0,0,0,175,171,206,166,0,0,0,0,0,0,0,0,0,
    7,206,166,0,59,245,255,166,238,0,0,0,0,0,0,0,0,0,0,0,0,7,206,255,125,59,
    215,21,0,0,59,241,89,0,0,0,0,0,0,7,206,166,0,0,0,0,138,201,175,201,0,0,
    0,12,235,251,89,89,201,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,7,206,166,0,0,0,0,0,0,89,247,34,0,0,0,0,89,225,151,201,0,0,0,0,
    0,0,0,0,0,0,0,0,0,175,201,0,12,235,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,127,0,0,0,138,201,0,0,0,0,7,206,125,0,0,0,138,201,0,89,225,21,
    12,228,34,138,255,201,0,0,0,138,247,34,175,225,21,0,138,201,0,138,201,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,225,21,0,0,0,0,0,89,225,
    21,0,0,89,247,34,59,241,89,0,59,241,89,89,247,34,0,0,89,225,21,175,127,
    215,21,206,247,42,206,0,138,255,247,42,206,125,0,0,138,166,12,235,251,89,
    0,0,0,0,138,201,0,0,0,0,0,0,89,225,21,0,0,0,59,241,89,12,235,125,0,0,175,
    125,59,245,247,34,0,0,12,235,125,89,251,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,59,238,34,0,0,0,0,0,175,166,0,0,0,0,12,232,89,7,206,125,0,
    0,12,235,166,59,245,125,0,0,0,59,238,34,0,12,235,125,0,0,0,0,0,0,89,247,
    34,138,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,255,247,34,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,175,251,89,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,199,34,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,235,125,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,12,235,255,255,255,166,0,12,235,255,
    255,255,166,0,12,235,255,255,255,166,0,12,235,255,255,255,166,0,0,12,235,
    255,255,255,166,0,12,235,255,255,255,166,0,12,235,255,255,255,166,0,175,
    255,255,125,0,0,12,235,255,255,125,0,0,12,235,255,255,225,21,0,0,12,235,
    255,255,225,21,0,12,235,255,255,225,21,0,12,235,255,255,225,21,0,12,235,
    125,12,235,125,12,235,125,12,235,125,0,12,235,125,89,251,89,0,12,235,138,
    235,255,247,34,0,0,12,235,255,255,201,0,0,0,12,235,255,255,201,0,0,0,12,
    235,255,255,201,0,0,0,12,235,255,255,201,0,0,0,12,235,255,255,201,0,0,0,
    0,0,0,175,247,34,0,0,0,12,235,255,255,255,166,0,59,241,89,0,0,89,247,34,
    59,241,89,0,0,89,247,34,59,241,89,0,0,89,247,34,59,241,89,0,0,89,247,42,
    206,201,0,0,0,138,232,245,166,245,255,251,89,7,206,201,0,0,0,138,225,21,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,127,12,206,21,0,59,245,125,12,206,21,0,59,245,125,12,206,
    21,0,59,245,125,12,206,21,0,59,245,125,0,12,206,21,0,59,245,125,12,206,
    21,0,59,245,125,12,206,21,0,12,235,255,125,0,7,206,166,12,235,166,0,0,172,
    102,0,235,125,0,0,175,201,0,12,235,125,0,0,175,201,12,235,125,0,0,175,201,
    12,235,125,0,0,175,201,0,12,235,125,12,235,125,12,235,125,12,235,125,0,
    0,0,0,0,175,201,0,12,235,247,34,0,175,201,0,12,235,166,0,7,206,201,0,12,
    235,166,0,7,206,201,0,12,235,166,0,7,206,201,0,12,235,166,0,7,206,201,0,
    12,235,166,0,7,206,201,0,0,0,0,0,175,247,34,0,0,12,235,166,0,12,235,201,
    0,59,241,89,0,0,89,247,34,59,241,89,0,0,89,247,34,59,241,89,0,0,89,247,
    34,59,241,89,0,0,89,247,34,89,247,34,0,7,206,176,235,225,21,0,175,225,21,
    89,247,34,0,7,206,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,7,206,166,0,0,0,0,7,
    206,166,0,0,0,0,7,206,166,0,0,0,0,7,206,166,0,0,0,0,0,7,206,166,0,0,0,0,
    7,206,166,0,0,0,0,0,175,201,0,0,0,89,225,138,225,21,0,0,0,0,89,225,21,0,
    0,89,247,34,89,225,21,0,0,89,247,124,225,21,0,0,89,247,124,225,21,0,0,89,
    247,34,12,235,125,12,235,125,12,235,125,12,235,125,0,89,255,255,255,255,
    247,34,12,235,125,0,0,89,247,34,138,225,21,0,0,59,238,34,138,225,21,0,0,
    59,238,34,138,225,21,0,0,59,238,34,138,225,21,0,0,59,238,34,138,225,21,
    0,0,59,238,34,0,0,0,0,0,0,0,0,0,138,225,21,0,172,132,238,34,59,241,89,0,
    0,89,247,34,59,241,89,0,0,89,247,34,59,241,89,0,0,89,247,34,59,241,89,0,
    0,89,247,34,12,235,125,0,59,238,47,235,125,0,0,59,241,89,12,235,125,0,59,
    238,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,59,245,255,255,255,166,0,59,245,255,255,
    255,166,0,59,245,255,255,255,166,0,59,245,255,255,255,166,0,0,59,245,255,
    255,255,166,0,59,245,255,255,255,166,0,89,255,255,255,255,255,255,255,255,
    255,247,175,201,0,0,0,0,0,175,255,255,255,255,255,247,34,175,255,255,255,
    255,255,247,198,255,255,255,255,255,247,198,255,255,255,255,255,247,34,
    12,235,125,12,235,125,12,235,125,12,235,125,89,251,89,0,0,59,241,89,12,
    235,125,0,0,89,247,34,175,201,0,0,0,12,232,89,175,201,0,0,0,12,232,89,175,
    201,0,0,0,12,232,89,175,201,0,0,0,12,232,89,175,201,0,0,0,12,232,89,7,206,
    255,255,255,255,255,255,251,226,201,0,89,166,12,232,89,59,241,89,0,0,89,
    247,34,59,241,89,0,0,89,247,34,59,241,89,0,0,89,247,34,59,241,89,0,0,89,
    247,34,0,175,201,0,138,201,12,235,125,0,0,12,235,125,0,175,201,0,138,201,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,127,89,255,166,0,7,206,166,89,255,166,0,7,206,166,89,
    255,166,0,7,206,166,89,255,166,0,7,206,166,0,89,255,166,0,7,206,166,89,
    255,166,0,7,206,166,138,255,125,0,0,175,201,0,0,0,0,0,175,201,0,0,0,0,0,
    175,201,0,0,0,0,0,0,175,201,0,0,0,0,0,175,201,0,0,0,0,0,175,201,0,0,0,0,
    0,0,12,235,125,12,235,125,12,235,125,12,235,125,175,201,0,0,0,59,241,89,
    12,235,125,0,0,89,247,34,175,201,0,0,0,12,232,89,175,201,0,0,0,12,232,89,
    175,201,0,0,0,12,232,89,175,201,0,0,0,12,232,89,175,201,0,0,0,12,232,89,
    0,0,0,0,0,0,0,0,0,175,201,7,176,21,12,232,89,59,241,89,0,0,89,247,34,59,
    241,89,0,0,89,247,34,59,241,89,0,0,89,247,34,59,241,89,0,0,89,247,34,0,
    89,247,47,235,125,12,235,125,0,0,12,235,125,0,89,247,47,235,125,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,127,175,201,0,0,7,206,166,175,201,0,0,7,206,166,175,201,0,0,
    7,206,166,175,201,0,0,7,206,166,0,175,201,0,0,7,206,166,175,201,0,0,7,206,
    166,175,201,0,0,0,138,225,21,0,0,0,0,138,225,21,0,0,0,0,138,247,34,0,0,
    0,0,0,138,247,34,0,0,0,0,138,247,34,0,0,0,0,138,247,34,0,0,0,0,0,12,235,
    125,12,235,125,12,235,125,12,235,125,175,201,0,0,0,89,247,34,12,235,125,
    0,0,89,247,34,138,225,21,0,0,89,247,34,138,225,21,0,0,89,247,34,138,225,
    21,0,0,89,247,34,138,225,21,0,0,89,247,34,138,225,21,0,0,89,247,34,0,0,
    0,0,175,247,34,0,0,138,225,151,125,0,89,247,34,59,241,89,0,0,89,247,34,
    59,241,89,0,0,89,247,34,59,241,89,0,0,89,247,34,59,241,89,0,0,89,247,34,
    0,12,235,191,247,34,12,235,125,0,0,59,241,89,0,12,235,191,247,34,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,127,138,225,21,0,138,255,166,138,225,21,0,138,255,166,138,
    225,21,0,138,255,166,138,225,21,0,138,255,166,0,138,225,21,0,138,255,166,
    138,225,21,0,138,255,166,89,247,34,0,89,255,255,166,0,0,12,206,12,235,166,
    0,0,127,102,0,235,201,0,0,12,206,21,12,235,201,0,0,12,206,34,235,201,0,
    0,12,206,34,235,201,0,0,12,206,21,12,235,125,12,235,125,12,235,125,12,235,
    125,89,255,125,0,7,206,166,0,12,235,125,0,0,89,247,34,12,235,166,0,7,206,
    201,0,12,235,166,0,7,206,201,0,12,235,166,0,7,206,201,0,12,235,166,0,7,
    206,201,0,12,235,166,0,7,206,201,0,0,0,0,0,175,247,34,0,0,12,235,201,0,
    7,206,201,0,7,206,166,0,59,245,247,34,7,206,166,0,59,245,247,34,7,206,166,
    0,59,245,247,34,7,206,166,0,59,245,247,34,0,0,138,255,166,0,12,235,125,
    0,7,206,201,0,0,0,138,255,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,7,206,255,255,171,
    206,166,7,206,255,255,171,206,166,7,206,255,255,171,206,166,7,206,255,255,
    171,206,166,0,7,206,255,255,171,206,166,7,206,255,255,171,206,166,0,89,
    255,255,201,0,0,175,255,255,247,34,0,12,235,255,255,166,0,0,7,206,255,255,
    247,34,0,0,7,206,255,255,247,34,0,7,206,255,255,247,34,0,7,206,255,255,
    247,34,0,12,235,125,12,235,125,12,235,125,12,235,125,0,89,255,255,255,201,
    0,0,12,235,125,0,0,89,247,34,0,12,235,255,255,201,0,0,0,12,235,255,255,
    201,0,0,0,12,235,255,255,201,0,0,0,12,235,255,255,201,0,0,0,12,235,255,
    255,201,0,0,0,0,0,0,0,0,0,0,0,7,206,255,255,255,201,0,0,0,59,245,255,225,
    111,247,34,0,59,245,255,225,111,247,34,0,59,245,255,225,111,247,34,0,59,
    245,255,225,111,247,34,0,0,59,241,89,0,12,235,255,255,255,225,21,0,0,0,
    59,241,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,175,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,138,125,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,138,225,21,0,12,235,125,0,0,0,0,0,0,0,
    138,225,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,7,202,89,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,206,166,0,0,12,235,125,0,0,0,0,0,0,7,206,
    166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    12,235,255,166,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,247,34,0,0,12,235,125,0,0,0,0,0,0,89,247,
    34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,127,127,127,127,127,127,0,127,127,127,127,
    127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,127,
    127,127,0,127,127,0,127,127,0,127,127,0,127,127,0,127,127,127,127,127,127,
    127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,
    127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,
    127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,127,
    0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,
    127,127,127,127,0,127,127,127,127,127,127,127,0,127,127,127,127,127,127,
    127,0,127,127,127,127,127,127,0,127,127,127,127,127,127,0,127,127,127,127,
    127,127,127,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0
};


void TwGenerateDefaultFonts()
{
    g_DefaultSmallFont = TwGenerateFont(s_Font0, FONT0_BM_W, FONT0_BM_H);
    assert(g_DefaultSmallFont && g_DefaultSmallFont->m_NbCharRead==224);
    g_DefaultNormalFont = TwGenerateFont(s_Font1AA, FONT1AA_BM_W, FONT1AA_BM_H);
    assert(g_DefaultNormalFont && g_DefaultNormalFont->m_NbCharRead==224);
    g_DefaultLargeFont = TwGenerateFont(s_Font2AA, FONT2AA_BM_W, FONT2AA_BM_H);
    assert(g_DefaultLargeFont && g_DefaultLargeFont->m_NbCharRead==224);
}

//  ---------------------------------------------------------------------------

void TwDeleteDefaultFonts()
{
    delete g_DefaultSmallFont;
    g_DefaultSmallFont = NULL;
    delete g_DefaultNormalFont;
    g_DefaultNormalFont = NULL;
    delete g_DefaultLargeFont;
    g_DefaultLargeFont = NULL;
}

//  ---------------------------------------------------------------------------
