/* NVTV data -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: data.c,v 1.27 2003/10/15 10:57:58 dthierbach Exp $
 *
 * Contents:
 *
 * Common routines for data access
 *
 */

#include "local.h" /* before everything else */

#include <string.h>
#include <stdlib.h>

#include "data.h"
#include "data_bt.h"
#include "data_cx.h"
#include "data_ch.h"
#include "data_ph.h"
#include "data_nv.h"
#include "data_tdfx.h"
#include "data_i810.h"
#include "data_xbox.h"

/* -------- -------- */

inline double dsqr (double arg)
{
  return arg * arg;
}

/* -------- -------- */

int data_clamp (int val, int size, int ticks[])
{
  int i;

  for (i = 0; i <= size; i++) 
    if (i >= size || val <= ticks[i]) break;
  if (i <= 0) return ticks[0];
  if (i >= size) return ticks[size-1];
  if (val <= ticks[i-1] + (ticks[i] - ticks[i-1]) / 2) {
    return ticks[i-1];
  } else {
    return ticks[i];
  }
}

int data_pick (int val, int size, int ticks[])
{
  int i;

  for (i = 0; i < size; i++) if (ticks[i] == val) return i;
  return 0;
}

/* -------- -------- */

/*
 * List all modes
 */

int data_listModes (TVMode *modes, TVSystem system, TVMode *(list[]))
{
  TVMode *m, *r;
  int c;
  if (!modes) return 0;

  c = 0;
  for (m = modes; m->spec.system != TV_SYSTEM_NONE; m++) {
    if (system == TV_SYSTEM_NONE || system == m->spec.system) c++;
  }
  r = xcalloc (sizeof (TVMode), c);
  *list = r;
  for (m = modes; m->spec.system != TV_SYSTEM_NONE; m++) {
    if (system == TV_SYSTEM_NONE || system == m->spec.system) *r++ = *m;
  }
  return c;
}

/*
 * find mode by size
 */

TVMode *data_find (TVMode *modes, TVSystem system, 
  int res_x, int res_y, char *size)
{
  if (!modes) return NULL;
  while (modes->spec.system != TV_SYSTEM_NONE) {
    if ((res_x < 0 || modes->spec.res_x == res_x) && 
	(res_y < 0 || modes->spec.res_y == res_y) &&
	(system == modes->spec.system) &&
        strcasecmp (modes->spec.size, size) == 0) break;
    modes++;
  }
  if (modes->spec.system != TV_SYSTEM_NONE) {
    RAISE (MSG_DEBUG, "found mode");
    return modes;
  } else {
    RAISE (MSG_DEBUG, "found no mode");
    return NULL;
  }
}

/*
 * find nearest mode by overscan (don't calculate new mode for BT/CX)
 */

TVMode *data_findNearest (TVMode *modes, TVSystem system, 
  int res_x, int res_y, double hoc, double voc)
{
  TVMode *best;
  double best_dist, dist;

  if (!modes) return NULL;
  best = NULL;
  best_dist = 0.0;
  while (modes->spec.system != TV_SYSTEM_NONE) {
    if ((res_x < 0 || modes->spec.res_x == res_x) && 
	(res_y < 0 || modes->spec.res_y == res_y) &&
	(system == modes->spec.system)) 
    {
      dist = dsqr (modes->spec.hoc - hoc) + dsqr (modes->spec.voc - voc);
      if (!best || best_dist > dist)
      {
	best_dist = dist;
	best = modes;
      }
    }
    modes++;
  }
  return best;
}

/* -------- Null data object -------- */

TVMode *data_modes_null (void)
{
  return NULL;
}

void data_default_null (TVSettings *s)
{
}

void data_clamp_null (TVSettings *s, TVRegs *r)
{
}

void data_setup_null (TVSettings *s, TVRegs *r)
{
}

void data_calc_null (TVSystem system, int hres, int vres, 
   double hoc, double voc, TVRegs *r)
{
}

DataFunc null_func = {
  modes:    data_modes_null,
  defaults: data_default_null,
  setup:    data_setup_null, 
  clamp:    data_clamp_null,
  calc:     data_calc_null,
};

/* -------- -------- */

void data_make_null (int hdisplay, int hsyncstart, int hsyncend, 
  int htotal, int vdisplay, int vsyncstart, int vsyncend, int vtotal, 
  int dotclock, TVCrtcRegs *crt)
{
}

DataCardFunc null_card_func = {
  make: data_make_null,
};

/* -------- -------- */

DataFunc *data_func (CardType card, TVChip chip)
{
  switch ((chip & TV_ENCODER) | (card << CARD_SHIFT)) {
    case (CARD_NVIDIA << CARD_SHIFT) | TV_BROOKTREE: 
      return &data_nv_bt_func;
    case (CARD_NVIDIA << CARD_SHIFT) | TV_CONEXANT: 
      return &data_nv_cx_func;
    case (CARD_NVIDIA << CARD_SHIFT) | TV_CHRONTEL_MODEL1:  
      return &data_nv_ch1_func; 
    case (CARD_NVIDIA << CARD_SHIFT) | TV_CHRONTEL_MODEL2:  
      return &data_nv_ch2_func; 
    case (CARD_NVIDIA << CARD_SHIFT) | TV_PHILIPS_MODEL1:  
      return &data_nv_ph1_func; 
    case (CARD_NVIDIA << CARD_SHIFT) | TV_PHILIPS_MODEL2:  
      return &data_nv_ph2_func; 
    case (CARD_NVIDIA << CARD_SHIFT) | TV_NVIDIA:  
      return &data_nv_nx_func; 
    case (CARD_XBOX   << CARD_SHIFT) | TV_CONEXANT: 
      return &data_xbox_cx_func;
    case (CARD_TDFX   << CARD_SHIFT) | TV_BROOKTREE: 
      return &data_tdfx_bt_func;
    case (CARD_I810   << CARD_SHIFT) | TV_CHRONTEL_MODEL1:  
      return &data_i810_ch1_func; 
    case (CARD_I810   << CARD_SHIFT) | TV_CHRONTEL_MODEL2:  
      return &data_i810_ch2_func; 
    default: 
      return &null_func;
  }
}

DataCardFunc *data_card_func (CardType card)
{
  switch (card) {
    case CARD_NVIDIA:
      return &data_nv_func;
    case CARD_XBOX:
      return &data_xbox_func;
    case CARD_TDFX:
      return &data_tdfx_func;
    case CARD_I810:
      return &data_i810_func;
    default: 
      return &null_card_func;
  }
}

