#############################################################################
# $Id: CMake.NCBIptb.cmake 581270 2019-02-27 14:52:17Z gouriano $
#############################################################################
#############################################################################
##
##  NCBI CMake wrapper
##    Author: Andrei Gourianov, gouriano@ncbi
##
##---------------------------------------------------------------------------
##  in CMakeLists.txt:
##    NCBI_add_subdirectory( list of subdirectories)
##    NCBI_add_library(      list of libraries)
##    NCBI_add_app(          list of apps)
##---------------------------------------------------------------------------
##  the following calls in CMakeLists.txt affect all projects in current dir and all subdirs
##    NCBI_headers( list of headers)
##    NCBI_disable_pch() / NCBI_enable_pch()
##    NCBI_requires( list of components)
##    NCBI_optional_components(  list of components)
##    NCBI_add_definitions(         list of compiler definitions)
##    NCBI_add_include_directories( list of directories)
##    NCBI_uses_toolkit_libraries(  list of libraries)
##    NCBI_uses_external_libraries( list of libraries)
##    NCBI_project_tags(list of tags)
##    
##---------------------------------------------------------------------------
##  in CMakeLists.xxx.[lib|app].txt - all calls affect current project only
##    NCBI_begin_lib(name) or NCBI_begin_app(name)
##
##      NCBI_sources(   list of source files)
##      NCBI_generated_sources(   list of source files) - file extension is mandatory
##      NCBI_headers(   list of header files) - only relative paths and masks are allowed
##      NCBI_resources( list of resource files) - file extension is mandatory
##      NCBI_dataspecs( list of data specs - ASN, DTD, XSD etc) - file extension is mandatory
##
##      NCBI_requires( list of components)
##      NCBI_optional_components(  list of components)
##
##      NCBI_enable_pch() / NCBI_disable_pch() - no arguments
##      NCBI_disable_pch_for( list of source files)
##      NCBI_set_pch_header( header file)
##      NCBI_set_pch_define( macro to define)
##
##      NCBI_uses_toolkit_libraries(  list of libraries)
##      NCBI_uses_external_libraries( list of libraries)
##      NCBI_add_definitions(         list of compiler definitions)
##      NCBI_add_include_directories( list of directories)
##
##      NCBI_hosts_projects(list of parts) - used to assemble composite shared libraries
##
##      NCBI_project_tags(     list of tags)
##      NCBI_project_watchers( list of watchers)
##
##      Testing:
##      short form
##          NCBI_add_test( test command) - empty command means run the app with no arguments
##      long form
##          NCBI_begin_test(name) - name is optional
##              NCBI_set_test_command(command, maybe with arguments)
##           OR NCBI_set_test_arguments(arguments only)
##              NCBI_set_test_assets(list of assets) - required files and directories
##              NCBI_set_test_timeout(seconds)
##              NCBI_set_test_requires(list of components)
##          NCBI_end_test()
##
##    NCBI_end_lib(result) or NCBI_end_app(result) - argument 'result' is optional
##
##---------------------------------------------------------------------------
##  custom targets
##  in CMakeLists.txt:
##    NCBI_add_target( list of targets)
##
##  in CMakeLists.xxx.txt - add function which defines target, then add target:
##    function(xxx_definition)      - the function name must be unique
##      ...
##      add_custom_target(name ...) - the same name as in NCBI_begin_custom_target
##    endfunction()
##
##    NCBI_begin_custom_target(name)
##      NCBI_requires( list of components)
##      NCBI_custom_target_dependencies(list of toolkit libraries or apps)
##      NCBI_custom_target_definition( xxx_definition) - function name defined above
##    NCBI_end_custom_target(result) - argument 'result' is optional
##  
#############################################################################
function(NCBI_add_root_subdirectory)

    set(NCBI_CURRENT_SOURCE_DIR ${CMAKE_CURRENT_SOURCE_DIR})
    if (NCBI_PTB_HAS_ROOT)
        message(FATAL_ERROR "NCBI_add_root_subdirectory() should be called once only (${CMAKE_CURRENT_SOURCE_DIR})")
        return()
    endif()
    set(NCBI_PTB_HAS_ROOT TRUE PARENT_SCOPE)
    set(NCBI_PTB_HAS_ROOT TRUE)
    set(NCBI_PTBMODE_COLLECT_DEPS OFF)
    if (NCBI_PTBCFG_ENABLE_COLLECTOR)
        message("Analyzing source tree...")
        set(NCBI_PTBMODE_COLLECT_DEPS ON)
        set_property(GLOBAL PROPERTY NCBI_PTBPROP_ALL_PROJECTS "")
        set_property(GLOBAL PROPERTY NCBI_PTBPROP_ALLOWED_PROJECTS "")

        NCBI_add_subdirectory(${NCBI_PTBCFG_COMPOSITE_DLL} ${ARGV})

        set(NCBI_PTBMODE_COLLECT_DEPS OFF)
        get_property(_allprojects     GLOBAL PROPERTY NCBI_PTBPROP_ALL_PROJECTS)
        get_property(_allowedprojects GLOBAL PROPERTY NCBI_PTBPROP_ALLOWED_PROJECTS)

        if(OFF)
            message("NCBI_PTBPROP_ALL_PROJECTS: ${_allprojects}")
            foreach(_prj IN LISTS _allprojects)
                get_property(_prjdeps GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_prj})
                message("NCBI_PTBPROP_DEPS_${_prj}: ${_prjdeps}")
            endforeach()
            message("NCBI_PTBPROP_ALLOWED_PROJECTS: ${_allowedprojects}")
        endif()

        if(NOT "${_allowedprojects}" STREQUAL "")
            message("Collecting projects...")
            if (NOT IS_ABSOLUTE ${NCBI_DATATOOL})
                list(APPEND _allowedprojects ${NCBI_DATATOOL})
            endif()
            list(REMOVE_DUPLICATES _allowedprojects)
            foreach(_prj IN LISTS _allowedprojects)
                NCBI_internal_collect_dependencies(${_prj})
                get_property(_prjdeps GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_prj})
                set(NCBI_PTB_ALLOWED_PROJECTS ${NCBI_PTB_ALLOWED_PROJECTS} ${_prj} ${_prjdeps})
            endforeach()
            list(SORT NCBI_PTB_ALLOWED_PROJECTS)
            list(REMOVE_DUPLICATES NCBI_PTB_ALLOWED_PROJECTS)
            foreach(_prj IN LISTS NCBI_PTB_ALLOWED_PROJECTS)
                NCBI_internal_collect_requires(${_prj})
            endforeach()
#message("NCBI_PTB_ALLOWED_PROJECTS: ${NCBI_PTB_ALLOWED_PROJECTS}")
            foreach(_prj IN LISTS NCBI_PTB_ALLOWED_PROJECTS)
                if (NCBI_VERBOSE_PROJECT_${_prj})
                    NCBI_internal_print_project_info(${_prj})
                endif()
            endforeach()
#        elseif("${_allprojects}" STREQUAL "")
        else()
            message(FATAL_ERROR "List of projects is empty")
            return()
        endif()
        message("Configuring projects...")
    endif()

    set_property(GLOBAL PROPERTY NCBI_PTBPROP_COUNT_STATIC 0)
    set_property(GLOBAL PROPERTY NCBI_PTBPROP_COUNT_SHARED 0)
    set_property(GLOBAL PROPERTY NCBI_PTBPROP_COUNT_CONSOLEAPP 0)
    set_property(GLOBAL PROPERTY NCBI_PTBPROP_COUNT_CUSTOM 0)

    NCBI_add_subdirectory(${NCBI_PTBCFG_COMPOSITE_DLL} ${ARGV})
    if (NCBI_PTBCFG_DOINSTALL)
        NCBI_internal_install_root(${NCBI_PTBCFG_COMPOSITE_DLL} ${ARGV})
    endif()

    get_property(_app GLOBAL PROPERTY NCBI_PTBPROP_COUNT_CONSOLEAPP)
    get_property(_lib GLOBAL PROPERTY NCBI_PTBPROP_COUNT_STATIC)
    get_property(_dll GLOBAL PROPERTY NCBI_PTBPROP_COUNT_SHARED)
    get_property(_cust GLOBAL PROPERTY NCBI_PTBPROP_COUNT_CUSTOM)
    if(BUILD_SHARED_LIBS)
        message("Added successfully: ${_app} console apps, ${_dll} shared libs, ${_lib} static libs, ${_cust} custom targets")
    else()
        message("Added successfully: ${_app} console apps, ${_lib} static libs, ${_cust} custom targets")
    endif()
endfunction()

#############################################################################
function(NCBI_add_subdirectory)
    if(NCBI_EXPERIMENTAL_CFG)
        if (NOT NCBI_PTB_HAS_ROOT)
            message(FATAL_ERROR "In source tree root, NCBI_add_root_subdirectory() must be used instead of NCBI_add_subdirectory() (${CMAKE_CURRENT_SOURCE_DIR})")
            return()
        endif()
    endif()
    if(NCBI_PTBMODE_PARTS)
        return()
    endif()
if(OFF)
# this requires that ALL subdirs use this new API
    set(_curdir ${NCBI_CURRENT_SOURCE_DIR})
    foreach(_sub IN LISTS ARGV)
        set(NCBI_CURRENT_SOURCE_DIR ${_curdir}/${_sub})
        if (EXISTS "${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.txt")
            if (NCBI_PTBMODE_COLLECT_DEPS)
                NCBI_internal_include("${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.txt")
            else()
                add_subdirectory(${_sub})
            endif()
        else()
            message("WARNING: directory not found: ${NCBI_CURRENT_SOURCE_DIR}")
        endif()
    endforeach()
else()
  if(NCBI_PTBMODE_COLLECT_DEPS)
    set(_curdir ${NCBI_CURRENT_SOURCE_DIR})
    foreach(_sub IN LISTS ARGV)
      if (EXISTS "${_curdir}/${_sub}/CMakeLists.txt")
        set(NCBI_CURRENT_SOURCE_DIR ${_curdir}/${_sub})
        NCBI_internal_include("${_curdir}/${_sub}/CMakeLists.txt")
      else()
        message("WARNING: directory not found: ${_curdir}/${_sub}")
      endif()
    endforeach()
  else()
    foreach(_sub IN LISTS ARGV)
      set(NCBI_CURRENT_SOURCE_DIR ${CMAKE_CURRENT_SOURCE_DIR}/${_sub})
      if (EXISTS "${CMAKE_CURRENT_SOURCE_DIR}/${_sub}")
        add_subdirectory(${_sub})
      else()
        message("WARNING: directory not found: ${CMAKE_CURRENT_SOURCE_DIR}/${_sub}")
      endif()
    endforeach()
  endif()
endif()
endfunction()

#############################################################################
function(NCBI_add_library)
    if(NCBI_PTBMODE_PARTS)
        return()
    endif()
if(OFF)
# this requires that ALL subdirs use this new API
    foreach(_lib IN LISTS ARGV)
        if (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.lib.txt)
            NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.lib.txt)
        elseif (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.asn.txt)
            NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.asn.txt)
        elseif (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/${_lib} AND NOT IS_DIRECTORY ${NCBI_CURRENT_SOURCE_DIR}/${_lib})
            NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/${_lib})
        else()
            message("WARNING: Library project not found: ${_lib} (${NCBI_CURRENT_SOURCE_DIR})")
        endif()
    endforeach()
else()
  if(NCBI_PTBMODE_COLLECT_DEPS)
    foreach(_lib IN LISTS ARGV)
      if (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.lib.txt)
        NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.lib.txt)
      elseif (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.asn.txt)
        NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.asn.txt)
      elseif (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/${_lib} AND NOT IS_DIRECTORY ${NCBI_CURRENT_SOURCE_DIR}/${_lib})
        NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/${_lib})
      else()
        message("WARNING: Library project not found: ${_lib} (${NCBI_CURRENT_SOURCE_DIR})")
      endif()
    endforeach()
  else()
    set(NCBI_CURRENT_SOURCE_DIR ${CMAKE_CURRENT_SOURCE_DIR})
    foreach(_lib IN LISTS ARGV)
      if (EXISTS ${CMAKE_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.lib.txt)
        NCBI_internal_include(CMakeLists.${_lib}.lib.txt)
      elseif (EXISTS ${CMAKE_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.asn.txt)
        NCBI_internal_include(CMakeLists.${_lib}.asn.txt)
      elseif (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/${_lib} AND NOT IS_DIRECTORY ${NCBI_CURRENT_SOURCE_DIR}/${_lib})
        NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/${_lib})
      else()
        message("WARNING: Library project not found: ${_lib} (${NCBI_CURRENT_SOURCE_DIR})")
      endif()
    endforeach()
  endif()
endif()
endfunction()

#############################################################################
function(NCBI_add_app)
    if(NCBI_PTBMODE_PARTS)
        return()
    endif()
if(OFF)
# this requires that ALL subdirs use this new API
    foreach(_app IN LISTS ARGV)
        if (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_app}.app.txt)
            NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_app}.app.txt)
        elseif (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/${_app} AND NOT IS_DIRECTORY ${NCBI_CURRENT_SOURCE_DIR}/${_app})
            NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/${_app})
        else()
            message("WARNING: App project not found: ${_app} (${NCBI_CURRENT_SOURCE_DIR})")
        endif()
    endforeach()
else()
  if(NCBI_PTBMODE_COLLECT_DEPS)
    foreach(_app IN LISTS ARGV)
      if (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_app}.app.txt)
        NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_app}.app.txt)
      elseif (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/${_app} AND NOT IS_DIRECTORY ${NCBI_CURRENT_SOURCE_DIR}/${_app})
        NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/${_app})
      else()
        message("WARNING: App project not found: ${_app} (${NCBI_CURRENT_SOURCE_DIR})")
      endif()
    endforeach()
  else()
    set(NCBI_CURRENT_SOURCE_DIR ${CMAKE_CURRENT_SOURCE_DIR})
    foreach(_app IN LISTS ARGV)
      if (EXISTS ${CMAKE_CURRENT_SOURCE_DIR}/CMakeLists.${_app}.app.txt)
        NCBI_internal_include(CMakeLists.${_app}.app.txt)
      elseif (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/${_app} AND NOT IS_DIRECTORY ${NCBI_CURRENT_SOURCE_DIR}/${_app})
        NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/${_app})
      else()
        message("WARNING: App project not found: ${_app} (${NCBI_CURRENT_SOURCE_DIR})")
      endif()
    endforeach()
  endif()
endif()
endfunction()

#############################################################################
function(NCBI_add_target)
    if(NCBI_PTBMODE_PARTS)
        return()
    endif()
    if(NOT NCBI_EXPERIMENTAL_CFG)
        set(NCBI_CURRENT_SOURCE_DIR ${CMAKE_CURRENT_SOURCE_DIR})
    endif()
    foreach(_prj IN LISTS ARGV)
        if (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_prj}.txt)
            NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_prj}.txt)
        elseif (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/${_prj} AND NOT IS_DIRECTORY ${NCBI_CURRENT_SOURCE_DIR}/${_prj})
            NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/${_prj})
        else()
            message("WARNING: Target not found: ${_prj} (${NCBI_CURRENT_SOURCE_DIR})")
        endif()
    endforeach()
endfunction()

#############################################################################
macro(NCBI_begin_lib _name)
    if(NOT NCBI_EXPERIMENTAL_CFG)
        set(NCBI_CURRENT_SOURCE_DIR ${CMAKE_CURRENT_SOURCE_DIR})
    endif()
    if(NCBI_PTBMODE_PARTS)
        set(_libname ${_name}.part)
    else()
        set(_libname ${_name})
    endif()
    if (DEFINED NCBI_EXTERNAL_TREE_ROOT AND TARGET ${_name})
        set(_libname ${_libname}.local)
        if (NCBI_PTBMODE_COLLECT_DEPS)
            set_property(GLOBAL PROPERTY NCBI_PTBPROP_LOCAL_${_name} "${_libname}")
            message(STATUS "Imported target ${_name} will be replaced with local ${_libname} (${NCBI_CURRENT_SOURCE_DIR})")
        endif()
    endif()
    set(NCBI_PROJECT_lib ${_libname})
    if(NCBI_EXPERIMENTAL_CFG)
        set(NCBI_PROJECT ${_libname})
        set(NCBI_${NCBI_PROJECT}_OUTPUT ${_name})
    else()
        if(${_name} STREQUAL "general")
            set(NCBI_PROJECT ${_libname}-lib)
            set(NCBI_${NCBI_PROJECT}_OUTPUT ${_name})
        else()
            set(NCBI_PROJECT ${_libname})
            set(NCBI_${NCBI_PROJECT}_OUTPUT ${_name})
        endif()
    endif()
    if ("${ARGC}" GREATER "1")
        if(BUILD_SHARED_LIBS)
            set(NCBI_${NCBI_PROJECT}_TYPE ${ARGV1})
        else()
            set(NCBI_${NCBI_PROJECT}_TYPE STATIC)
        endif()
    else()
#CMake global flag
        if(BUILD_SHARED_LIBS)
            if(WIN32 OR XCODE)
                set(NCBI_${NCBI_PROJECT}_TYPE STATIC)
            else()
                set(NCBI_${NCBI_PROJECT}_TYPE SHARED)
            endif()
        else()
            set(NCBI_${NCBI_PROJECT}_TYPE STATIC)
        endif()
    endif()
    set(NCBI_PROJECT_PARTNAME ${_name})
    set(NCBI_PROJECT_ID ${_name}.${NCBI_${NCBI_PROJECT}_TYPE})
endmacro()

#############################################################################
macro(NCBI_end_lib)
    if(NOT DEFINED NCBI_PROJECT_lib)
        message(SEND_ERROR "${NCBI_CURRENT_SOURCE_DIR}/${NCBI_PROJECT}: Unexpected NCBI_end_lib call")
    endif()
    NCBI_internal_add_project(${ARGV})
    unset(NCBI_PROJECT)
endmacro()

#############################################################################
macro(NCBI_begin_app _name)
    if(NOT NCBI_EXPERIMENTAL_CFG)
        set(NCBI_CURRENT_SOURCE_DIR ${CMAKE_CURRENT_SOURCE_DIR})
    endif()
    if(NCBI_PTBMODE_PARTS)
        set(_appname ${_name}.part)
    else()
        set(_appname ${_name})
    endif()
    if (DEFINED NCBI_EXTERNAL_TREE_ROOT AND TARGET ${_name})
        set(_appname ${_appname}.local)
        if (NCBI_PTBMODE_COLLECT_DEPS)
            set_property(GLOBAL PROPERTY NCBI_PTBPROP_LOCAL_${_name} "${_appname}")
            message(STATUS "Imported target ${_name} will be replaced with local ${_appname} (${NCBI_CURRENT_SOURCE_DIR})")
        endif()
    endif()
    set(NCBI_PROJECT_app ${_appname})
    if(NCBI_EXPERIMENTAL_CFG)
        if (NOT DEFINED NCBI_EXTERNAL_TREE_ROOT)
            get_property(_dir GLOBAL PROPERTY NCBI_PTBPROP_DIR_${_name})
            if (NOT ${_dir} STREQUAL "" AND NOT ${_dir} STREQUAL ${NCBI_CURRENT_SOURCE_DIR})
                set(_appname ${_appname}-app)
            endif()
        endif()
        set(NCBI_PROJECT ${_appname})
        set(NCBI_${NCBI_PROJECT}_OUTPUT ${_name})
    else()
        set(NCBI_PROJECT ${_appname}-app)
        set(NCBI_${NCBI_PROJECT}_OUTPUT ${_name})
    endif()
    set(NCBI_${NCBI_PROJECT}_TYPE CONSOLEAPP)
    set(NCBI_PROJECT_PARTNAME ${_name})
    set(NCBI_PROJECT_ID ${_name}.${NCBI_${NCBI_PROJECT}_TYPE})
endmacro()

#############################################################################
macro(NCBI_end_app)
    if(NOT DEFINED NCBI_PROJECT_app)
        message(SEND_ERROR "${NCBI_CURRENT_SOURCE_DIR}/${NCBI_PROJECT}: Unexpected NCBI_end_app call")
    endif()
    NCBI_internal_add_project(${ARGV})
    unset(NCBI_PROJECT)
endmacro()

#############################################################################
macro(NCBI_begin_custom_target _name)
    if(NOT NCBI_EXPERIMENTAL_CFG)
        set(NCBI_CURRENT_SOURCE_DIR ${CMAKE_CURRENT_SOURCE_DIR})
    endif()
    set(NCBI_PROJECT_custom ${_name})
    set(NCBI_PROJECT ${_name})
    set(NCBI_${NCBI_PROJECT}_TYPE CUSTOM)
    set(NCBI_PROJECT_ID ${_name}.${NCBI_${NCBI_PROJECT}_TYPE})
endmacro()

#############################################################################
macro(NCBI_end_custom_target)
    if(NOT DEFINED NCBI_PROJECT_custom)
        message(SEND_ERROR "${NCBI_CURRENT_SOURCE_DIR}/${NCBI_PROJECT}: Unexpected NCBI_end_custom_target call")
    endif()
    if(NOT DEFINED NCBI_${NCBI_PROJECT}_DEFINITION)
        message(FATAL_ERROR "${NCBI_PROJECT} (${NCBI_CURRENT_SOURCE_DIR}): Custom project definition not provided")
    endif()
    NCBI_internal_add_project(${ARGV})
    unset(NCBI_PROJECT)
endmacro()

#############################################################################
macro(NCBI_custom_target_definition _def)
    set(NCBI_${NCBI_PROJECT}_DEFINITION ${_def})
    set(NCBI_${NCBI_PROJECT}_CALLBACK 0)
endmacro()

#############################################################################
macro(NCBI_custom_target_dependencies)
    set(NCBI_${NCBI_PROJECT}_NCBILIB ${NCBI_${NCBI_PROJECT}_NCBILIB} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_sources)
    set(NCBI_${NCBI_PROJECT}_SOURCES ${NCBI_${NCBI_PROJECT}_SOURCES} "${ARGV}")
endmacro()
macro(NCBI_generated_sources)
    set(NCBI_${NCBI_PROJECT}_GENERATED_SOURCES ${NCBI_${NCBI_PROJECT}_GENERATED_SOURCES} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_headers)
    set(NCBI_${NCBI_PROJECT}_HEADERS ${NCBI_${NCBI_PROJECT}_HEADERS} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_resources)
    set(NCBI_${NCBI_PROJECT}_RESOURCES ${NCBI_${NCBI_PROJECT}_RESOURCES} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_dataspecs)
    set(NCBI_${NCBI_PROJECT}_DATASPEC ${NCBI_${NCBI_PROJECT}_DATASPEC} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_hosts_projects)
    set(NCBI_${NCBI_PROJECT}_PARTS ${NCBI_${NCBI_PROJECT}_PARTS} "${ARGV}")
endmacro()

macro(NCBI_hosts_virtual_projects)
    set(NCBI_${NCBI_PROJECT}_VIRTPARTS ${NCBI_${NCBI_PROJECT}_VIRTPARTS} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_requires)
    set(NCBI_${NCBI_PROJECT}_REQUIRES ${NCBI_${NCBI_PROJECT}_REQUIRES} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_optional_components)
    set(NCBI_${NCBI_PROJECT}_COMPONENTS ${NCBI_${NCBI_PROJECT}_COMPONENTS} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_enable_pch)
    set(NCBI_${NCBI_PROJECT}_USEPCH ${NCBI_DEFAULT_USEPCH})
endmacro()
macro(NCBI_disable_pch)
    set(NCBI_${NCBI_PROJECT}_USEPCH OFF)
endmacro()

#############################################################################
macro(NCBI_disable_pch_for)
    set(NCBI_${NCBI_PROJECT}_NOPCH ${NCBI_${NCBI_PROJECT}_NOPCH} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_set_pch_header)
    set(NCBI_${NCBI_PROJECT}_PCH "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_set_pch_define)
    set(NCBI_${NCBI_PROJECT}_PCH_DEFINE "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_uses_toolkit_libraries)
    set(NCBI_${NCBI_PROJECT}_NCBILIB ${NCBI_${NCBI_PROJECT}_NCBILIB} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_uses_external_libraries)
    set(NCBI_${NCBI_PROJECT}_EXTLIB ${NCBI_${NCBI_PROJECT}_EXTLIB} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_add_include_directories)
    set(NCBI_${NCBI_PROJECT}_INCLUDES ${NCBI_${NCBI_PROJECT}_INCLUDES} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_add_definitions)
    set(NCBI_${NCBI_PROJECT}_DEFINES ${NCBI_${NCBI_PROJECT}_DEFINES} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_project_tags)
    set(NCBI_${NCBI_PROJECT}_PROJTAG ${NCBI_${NCBI_PROJECT}_PROJTAG} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_project_watchers)
    set(NCBI_${NCBI_PROJECT}_WATCHER ${NCBI_${NCBI_PROJECT}_WATCHER} "${ARGV}")
endmacro()

#############################################################################
#############################################################################
#############################################################################
macro(NCBI_begin_test)
    if(DEFINED NCBI_TEST)
        message(SEND_ERROR "${NCBI_CURRENT_SOURCE_DIR}/${NCBI_PROJECT}: Unexpected NCBI_begin_test call")
    endif()
    if (DEFINED NCBITEST_${NCBI_PROJECT}_NUM)
        math(EXPR NCBITEST_${NCBI_PROJECT}_NUM "${NCBITEST_${NCBI_PROJECT}_NUM} + 1")
    endif()
    if ("${ARGC}" GREATER "0")
        set(_testname "${ARGV}")
        if ( "${_testname}" STREQUAL "")
            set(_testname "${NCBI_PROJECT}${NCBITEST_${NCBI_PROJECT}_NUM}")
        endif()
    else()
        set(_testname "${NCBI_PROJECT}${NCBITEST_${NCBI_PROJECT}_NUM}")
    endif()
    set(NCBI_TEST ${_testname})
    if (NOT DEFINED NCBITEST_${NCBI_PROJECT}_NUM)
        set(NCBITEST_${NCBI_PROJECT}_NUM "1")
    endif()
endmacro()

##############################################################################
macro(NCBI_set_test_command)
    set( _args ${ARGV})
    list(GET _args 0 _cmd)
    list(REMOVE_AT _args 0) 
    if ( "${_cmd}" STREQUAL "${NCBI_PROJECT}")
        set(_cmd "${NCBI_${NCBI_PROJECT}_OUTPUT}")
        if (${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "CONSOLEAPP")
           set(_cmd "${_cmd}${CMAKE_EXECUTABLE_SUFFIX}")
        endif()
    endif()
    if (NOT NCBI_EXPERIMENTAL_CFG)
        set(_cmd "${EXECUTABLE_OUTPUT_PATH}/${_cmd}")
    endif()
    set(NCBITEST_${NCBI_TEST}_CMD "${_cmd}")
    set(NCBITEST_${NCBI_TEST}_ARG "${_args}")
endmacro()

##############################################################################
macro(NCBI_set_test_arguments)
    set(NCBITEST_${NCBI_TEST}_ARG "${ARGV}")
endmacro()

##############################################################################
macro(NCBI_set_test_assets)
    set(NCBITEST_${NCBI_TEST}_ASSETS "${ARGV}")
endmacro()

##############################################################################
macro(NCBI_set_test_timeout)
    set(NCBITEST_${NCBI_TEST}_TIMEOUT "${ARGV}")
endmacro()

##############################################################################
macro(NCBI_set_test_requires)
    set(NCBITEST_${NCBI_TEST}_REQUIRES "${ARGV}")
endmacro()

##############################################################################
macro(NCBI_end_test)
    set(NCBI_ALLTESTS ${NCBI_ALLTESTS} ${NCBI_TEST})
    unset(NCBI_TEST)
endmacro()

##############################################################################
macro(NCBI_add_test)
    if ("${ARGC}" GREATER "0")
        set(_cmd "${ARGV}")
        if ( "${_cmd}" STREQUAL "")
            set(_cmd "${NCBI_PROJECT}")
        endif()
    else()
        set(_cmd "${NCBI_PROJECT}")
    endif()
    NCBI_begin_test()
    NCBI_set_test_command(${_cmd})
    NCBI_end_test()
endmacro()


##############################################################################
##############################################################################
##############################################################################

function(NCBI_internal_collect_dependencies _project)
    get_property(_collected GLOBAL PROPERTY NCBI_PTBPROP_COLLECTED_DEPS)
    if( ${_project} IN_LIST _collected)
        return()
    endif()
    set_property(GLOBAL PROPERTY NCBI_PTBPROP_COLLECTED_DEPS ${_collected} ${_project})

    get_property(_prjdeps GLOBAL PROPERTY NCBI_PTBPROP_DIRECT_DEPS_${_project} SET)
    if (NOT _prjdeps)
        message("ERROR: project ${_project} not found")
        return()
    endif()
    get_property(_prjdeps GLOBAL PROPERTY NCBI_PTBPROP_DIRECT_DEPS_${_project})
    foreach( _value IN LISTS _prjdeps)
        NCBI_internal_recur_collect_dependencies( ${_project} ${_value})
    endforeach()
    get_property(_prjdeps GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_project})
    foreach( _value IN LISTS _prjdeps)
        get_property(_host GLOBAL PROPERTY NCBI_PTBPROP_HOST_${_value})
        if ("${_host}" STREQUAL "")
            list(APPEND _deps ${_value})
        else()
            list(APPEND _deps ${_host})
        endif()
    endforeach()
    if (NOT "${_deps}" STREQUAL "")
        list(SORT _deps)
        list(REMOVE_DUPLICATES _deps)
    endif()
    set_property(GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_project} "${_deps}")
endfunction()

##############################################################################
function(NCBI_internal_recur_collect_dependencies _project _dep)
    get_property(_prjdeps GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_project})
    get_property(_depdeps GLOBAL PROPERTY NCBI_PTBPROP_DIRECT_DEPS_${_dep} SET)
    if (NOT _depdeps)
        if(NOT TARGET ${_dep})
            message("ERROR: project ${_dep} not found (${_project} requires it)")
        endif()
        return()
    endif()
    get_property(_depdeps GLOBAL PROPERTY NCBI_PTBPROP_DIRECT_DEPS_${_dep})
    foreach( _value IN LISTS _dep _depdeps)
        NCBI_internal_collect_dependencies(${_value}) 
        get_property(_valuedeps GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_value})
        set(_prjdeps ${_prjdeps} ${_value} ${_valuedeps})
    endforeach()
    if (NOT "${_prjdeps}" STREQUAL "")
        list(SORT _prjdeps)
        list(REMOVE_DUPLICATES _prjdeps)
    endif()
    set_property(GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_project} ${_prjdeps})
endfunction()

##############################################################################
function(NCBI_internal_collect_requires _project)
    get_property(_deps GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_project})
    if (NOT "${_deps}" STREQUAL "")
        list(REMOVE_DUPLICATES _deps)
        set(_implreq "")
        foreach(_dep IN LISTS _deps)
            get_property(_y GLOBAL PROPERTY NCBI_PTBPROP_REQUIRES_${_dep})
            list(APPEND _implreq "${_y}")
        endforeach()
        if (NOT "${_implreq}" STREQUAL "")
            list(SORT _implreq)
            list(REMOVE_DUPLICATES _implreq)
            set_property(GLOBAL PROPERTY NCBI_PTBPROP_IMPLREQ_${_project} ${_implreq})
        endif()
    endif()
endfunction()

##############################################################################
function(NCBI_internal_include)
    include(${ARGV0})
endfunction()

##############################################################################
function(NCBI_internal_collect_sources)
    set(_dir ${NCBI_CURRENT_SOURCE_DIR})
    if(NCBI_PTBMODE_PARTS)
        set(_prefix "Hosted Libraries\\${NCBI_PROJECT_PARTNAME}\\")
    else()
        set(_prefix "")
    endif()

    foreach(_file IN LISTS NCBI_${NCBI_PROJECT}_NOPCH)
        if(EXISTS ${_dir}/${_file}.cpp)
            list(APPEND _nopch "${_dir}/${_file}.cpp")
        elseif(EXISTS ${_dir}/${_file}.c)
            list(APPEND _nopch   "${_dir}/${_file}.c")
        elseif(EXISTS ${_dir}/${_file})
            list(APPEND _nopch   "${_dir}/${_file}")
        else()
            list(APPEND _nopch   "${_file}")
        endif()
    endforeach()

    foreach(_file IN LISTS NCBI_${NCBI_PROJECT}_SOURCES)
        if(EXISTS ${_dir}/${_file}.cpp)
            list(APPEND _sources "${_dir}/${_file}.cpp")
        elseif(EXISTS ${_dir}/${_file}.c)
            list(APPEND _sources "${_dir}/${_file}.c")
            list(APPEND _nopch   "${_dir}/${_file}.c")
        else()
            get_filename_component(_ext ${_file} EXT)
            if(EXISTS ${_file} OR IS_ABSOLUTE ${_file})
                list(APPEND _sources "${_file}")
                if("${_ext}" STREQUAL ".c")
                    list(APPEND _nopch "${_file}")
                endif()
            else()
                list(APPEND _sources "${_dir}/${_file}")
                if("${_ext}" STREQUAL ".c")
                    list(APPEND _nopch "${_dir}/${_file}")
                endif()
            endif()
        endif()
    endforeach()

    foreach(_file IN LISTS NCBI_${NCBI_PROJECT}_GENERATED_SOURCES)
        if(NOT IS_ABSOLUTE ${_file})
            set(_file "${_dir}/${_file}")
        endif()
        list(APPEND _sources "${_file}")
        set_source_files_properties(${_file} PROPERTIES GENERATED TRUE)
        get_filename_component(_ext ${_file} EXT)
        if("${_ext}" STREQUAL ".c")
            list(APPEND _nopch "${_file}")
        endif()
    endforeach()

    if (NOT NCBI_PTBMODE_PARTS AND DEFINED NCBI_DEFAULT_DLLENTRY)
        if (${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "SHARED")
            list(APPEND _sources ${NCBI_DEFAULT_DLLENTRY})
        endif()
    endif()

    if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
        message("create source group  ${_prefix}Source Files")
        message("${_sources}")
    endif()
    source_group("${_prefix}Source Files"   FILES ${_sources})
    set(NCBITMP_PROJECT_SOURCES ${NCBITMP_PROJECT_SOURCES} ${_sources} PARENT_SCOPE)
    set(NCBITMP_PROJECT_NOPCH   ${NCBITMP_PROJECT_NOPCH}   ${_nopch}   PARENT_SCOPE)
endfunction()

##############################################################################
function(NCBI_internal_collect_headers)

    file(RELATIVE_PATH _rel "${NCBI_SRC_ROOT}" "${NCBI_CURRENT_SOURCE_DIR}")
    set(_inc_dir ${NCBI_INC_ROOT}/${_rel})
    if(NCBI_PTBMODE_PARTS)
        set(_prefix "Hosted Libraries\\${NCBI_PROJECT_PARTNAME}\\")
    else()
        set(_prefix "")
    endif()
    if (DEFINED NCBI_${NCBI_PROJECT}_HEADERS)
        set(_headers ${NCBI_${NCBI_PROJECT}_HEADERS})
    elseif(DEFINED NCBI__HEADERS)
        set(_headers ${NCBI__HEADERS})
    else()
        set(_headers *.h* *impl/*.h* *.inl *impl/*.inl)
    endif()

    if (MSVC OR XCODE)
        set(_priv "")
        set(_pub "")
        set(_inl "")
        if ( NOT "${_headers}" STREQUAL "")
            foreach(_mask ${_headers})
                file(GLOB _files LIST_DIRECTORIES false "${NCBI_CURRENT_SOURCE_DIR}/${_mask}")
                set(_priv ${_priv} ${_files})
                if (${_mask} MATCHES ".inl$")
                    file(GLOB _files LIST_DIRECTORIES false "${_inc_dir}/${_mask}")
                    set(_inl ${_inl} ${_files})
                else()
                    file(GLOB _files LIST_DIRECTORIES false "${_inc_dir}/${_mask}")
                    set(_pub ${_pub} ${_files})
                endif()
            endforeach()

            if ( NOT "${_priv}" STREQUAL "")
                source_group("${_prefix}Private Headers" FILES ${_priv})
            endif()
            if ( NOT "${_pub}" STREQUAL "")
                source_group("${_prefix}Header Files"    FILES ${_pub})
            endif()
            if ( NOT "${_inl}" STREQUAL "")
                source_group("${_prefix}Inline Files"    FILES ${_inl})
            endif()
        endif()
        set(NCBITMP_PROJECT_HEADERS ${NCBITMP_PROJECT_HEADERS} ${_priv} ${_pub} ${_inl} PARENT_SCOPE)
    else()
        set(NCBITMP_PROJECT_HEADERS "" PARENT_SCOPE)
    endif()
endfunction()

##############################################################################
function(NCBI_internal_add_resources) 
    if (WIN32)
        if (${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "CONSOLEAPP")
            if (DEFINED NCBI_${NCBI_PROJECT}_RESOURCES)
                set(_res ${NCBI_${NCBI_PROJECT}_RESOURCES})
            elseif (DEFINED NCBI__RESOURCES)
                set(_res ${NCBI__RESOURCES})
            else()
                set(_res ${NCBI_DEFAULT_RESOURCES})
            endif()
        else()
            if (DEFINED NCBI_${NCBI_PROJECT}_RESOURCES)
                set(_res ${NCBI_${NCBI_PROJECT}_RESOURCES})
            endif()
        endif()

        if(NCBI_PTBMODE_PARTS)
            set(_prefix "Hosted Libraries\\${NCBI_PROJECT_PARTNAME}\\")
        else()
            set(_prefix "")
        endif()
        if ( NOT "${_res}" STREQUAL "")
            source_group("${_prefix}Resource Files" FILES ${_res})
        endif()
        set(NCBITMP_PROJECT_RESOURCES ${_res} PARENT_SCOPE)
    else()
        set(NCBITMP_PROJECT_RESOURCES "" PARENT_SCOPE)
    endif()
endfunction()

##############################################################################
function(NCBI_internal_collect_dataspec) 
    if (NOT DEFINED NCBI_${NCBI_PROJECT}_DATASPEC)
        return()
    endif()

    if(NCBI_PTBMODE_PARTS)
        set(_prefix "Hosted Libraries\\${NCBI_PROJECT_PARTNAME}\\")
    else()
        set(_prefix "")
    endif()

    foreach(_dataspec IN LISTS NCBI_${NCBI_PROJECT}_DATASPEC)
        if (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/${_dataspec})
            get_filename_component(_basename ${_dataspec} NAME_WE)
            get_filename_component(_ext ${_dataspec} EXT)
            get_filename_component(_path ${NCBI_CURRENT_SOURCE_DIR}/${_dataspec} DIRECTORY)
            set(_specfiles ${_specfiles}  ${NCBI_CURRENT_SOURCE_DIR}/${_dataspec})
            if ("${_ext}" STREQUAL ".proto")
                if ("PROTOBUF" IN_LIST NCBITMP_REQUIRE)
                    set_source_files_properties(${_path}/${_basename}.pb.cc PROPERTIES GENERATED TRUE)
                    set(_srcfiles ${_srcfiles}  ${_path}/${_basename}.pb.cc)
                endif()
                if ("GRPC" IN_LIST NCBITMP_REQUIRE)
                    set_source_files_properties(${_path}/${_basename}.grpc.pb.cc PROPERTIES GENERATED TRUE)
                    set(_srcfiles ${_srcfiles}  ${_path}/${_basename}.grpc.pb.cc)
                endif()
            else()
                set_source_files_properties(${_path}/${_basename}__.cpp ${_path}/${_basename}___.cpp PROPERTIES GENERATED TRUE)
                set(_srcfiles ${_srcfiles}  ${_path}/${_basename}__.cpp ${_path}/${_basename}___.cpp)
            endif()
        else()
            message(WARNING "ERROR: file not found: ${NCBI_CURRENT_SOURCE_DIR}/${_dataspec}")
        endif()
    endforeach()

    source_group("${_prefix}Source Files"   FILES ${_srcfiles})
    set(NCBITMP_PROJECT_SOURCES ${NCBITMP_PROJECT_SOURCES} ${_srcfiles} PARENT_SCOPE)
    source_group("${_prefix}DataSpec Files" FILES ${_specfiles})
    set(NCBITMP_PROJECT_DATASPEC ${NCBITMP_PROJECT_DATASPEC} ${_specfiles} PARENT_SCOPE)
endfunction()

##############################################################################
function(NCBI_internal_add_dataspec) 
    if (NOT DEFINED NCBITMP_PROJECT_DATASPEC)
        return()
    endif()

    foreach(_dataspec IN LISTS NCBITMP_PROJECT_DATASPEC)
        get_filename_component(_basename ${_dataspec} NAME_WE)
        get_filename_component(_ext ${_dataspec} EXT)
        set(_filepath ${_dataspec})
        get_filename_component(_path ${_filepath} DIRECTORY)
        file(RELATIVE_PATH _relpath ${NCBI_SRC_ROOT} ${_path})

        if ("${_ext}" STREQUAL ".proto")
            if ("PROTOBUF" IN_LIST NCBITMP_REQUIRE)
                set(_cmd ${NCBI_PROTOC_APP} --cpp_out=${_path} --proto_path=${_path} ${_filepath})
                add_custom_command(
                    OUTPUT ${_path}/${_basename}.pb.cc
                    COMMAND ${_cmd} VERBATIM
                    COMMAND ${CMAKE_COMMAND} -E copy_if_different ${_path}/${_basename}.pb.h ${NCBI_INC_ROOT}/${_relpath} VERBATIM
                    COMMAND ${CMAKE_COMMAND} -E remove -f ${_path}/${_basename}.pb.h VERBATIM
                    WORKING_DIRECTORY ${top_src_dir}
                    COMMENT "Generate PROTOC C++ classes from ${_filepath}"
                    DEPENDS ${_filepath}
                    VERBATIM
                )
            endif()
            if ("GRPC" IN_LIST NCBITMP_REQUIRE)
                set(_cmd ${NCBI_PROTOC_APP} --grpc_out=${_path} --proto_path=${_path} --plugin=protoc-gen-grpc=${NCBI_GRPC_PLUGIN}  ${_filepath})
                add_custom_command(
                    OUTPUT ${_path}/${_basename}.grpc.pb.cc
                    COMMAND ${_cmd} VERBATIM
                    COMMAND ${CMAKE_COMMAND} -E copy_if_different ${_path}/${_basename}.grpc.pb.h ${NCBI_INC_ROOT}/${_relpath} VERBATIM
                    COMMAND ${CMAKE_COMMAND} -E remove -f ${_path}/${_basename}.grpc.pb.h VERBATIM
                    WORKING_DIRECTORY ${top_src_dir}
                    COMMENT "Generate GRPC C++ classes from ${_filepath}"
                    DEPENDS ${_filepath}
                    VERBATIM
                )
            endif()
            set(NCBITMP_INCLUDES ${NCBITMP_INCLUDES} ${NCBI_INC_ROOT}/${_relpath} PARENT_SCOPE)
        else()
            set(_module_imports "")
            set(_imports "")
            if(EXISTS "${_path}/${_basename}.module")
                FILE(READ "${_path}/${_basename}.module" _module_contents)
                STRING(REGEX MATCH "MODULE_IMPORT *=[^\n]*[^ \n]" _tmp "${_module_contents}")
                STRING(REGEX REPLACE "MODULE_IMPORT *= *" "" _tmp "${_tmp}")
                STRING(REGEX REPLACE "  *$" "" _imp_list "${_tmp}")
                STRING(REGEX REPLACE " " ";" _imp_list "${_imp_list}")

                foreach(_module IN LISTS _imp_list)
                    set(_module_imports "${_module_imports} ${_module}${_ext}")
                endforeach()
                if (NOT "${_module_imports}" STREQUAL "")
                    set(_imports -M ${_module_imports})
                endif()
            endif()

            set(_oc ${_basename})
            if (NOT "${NCBI_DEFAULT_PCH}" STREQUAL "")
                set(_pch -pch ${NCBI_DEFAULT_PCH})
            endif()
            set(_od ${_path}/${_basename}.def)
            set(_oex -oex " ")
            set(_cmd ${NCBI_DATATOOL} ${_oex} ${_pch} -m ${_filepath} -oA -oc ${_oc} -od ${_od} -odi -ocvs -or ${_relpath} -oR ${top_src_dir} ${_imports})
            set(_depends ${NCBI_DATATOOL} ${_filepath})
            if(EXISTS ${_od})
                set(_depends ${_depends} ${_od})
            endif()
            add_custom_command(
                OUTPUT ${_path}/${_basename}__.cpp ${_path}/${_basename}___.cpp
                COMMAND ${_cmd} VERBATIM
                WORKING_DIRECTORY ${top_src_dir}
                COMMENT "Generate C++ classes from ${_filepath}"
                DEPENDS ${_depends}
                VERBATIM
            )
        endif()
    endforeach()
endfunction()

##############################################################################
function(NCBI_internal_define_precompiled_header_usage)

    if (DEFINED NCBI_${NCBI_PROJECT}_USEPCH)
        set(_usepch ${NCBI_${NCBI_PROJECT}_USEPCH})
    elseif (DEFINED NCBI__USEPCH)
        set(_usepch ${NCBI__USEPCH})
    else()
        set(_usepch ${NCBI_DEFAULT_USEPCH})
    endif()

    if (DEFINED NCBI_${NCBI_PROJECT}_PCH)
        set(_pch ${NCBI_${NCBI_PROJECT}_PCH})
    elseif (DEFINED NCBI__PCH)
        set(_pch ${NCBI__PCH})
    else()
        set(_pch ${NCBI_DEFAULT_PCH})
    endif()

    if (DEFINED NCBI_${NCBI_PROJECT}_PCH_DEFINE)
        set(_pchdef ${NCBI_${NCBI_PROJECT}_PCH_DEFINE})
    elseif (DEFINED NCBI__PCH_DEFINE)
        set(_pchdef ${NCBI__PCH_DEFINE})
    else()
        set(_pchdef ${NCBI_DEFAULT_PCH_DEFINE})
    endif()

if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
    message("_usepch ${_usepch}:  prj=${NCBI_${NCBI_PROJECT}_USEPCH} dir=${NCBI__USEPCH} def=${NCBI_DEFAULT_USEPCH}")
    message("_pch ${_pch}:  prj=${NCBI_${NCBI_PROJECT}_PCH} dir=${NCBI__PCH} def=${NCBI_DEFAULT_PCH}")
    message("_pchdef ${_pchdef}")
endif()

    if (NOT DEFINED NCBITMP_PROJECT_SOURCES)
        set(_usepch OFF)
    endif()

    if (_usepch)
        if (MSVC)
            set_target_properties(${NCBI_PROJECT} PROPERTIES COMPILE_FLAGS "/Yu${_pch}")
            set(_files ${NCBITMP_PROJECT_SOURCES})
            if (DEFINED NCBITMP_PROJECT_NOPCH)
                foreach(_file ${NCBITMP_PROJECT_NOPCH})
                    list(FIND _files ${_file} _found)
                    if (${_found} LESS "0")
                        list(FIND _files ${_file}.cpp _found)
                        if (${_found} LESS "0")
                            list(FIND _files ${_file}.c _found)
                            if (${_found} LESS "0")
                                continue()
                            endif()
                            set(_file ${_file}.c)
                        else()
                            set(_file ${_file}.cpp)
                        endif()
                    endif()
                    list(REMOVE_ITEM _files ${_file})
                    set_source_files_properties(${_file} PROPERTIES COMPILE_FLAGS "/Y-")
                endforeach()
            endif()

            list(GET _files 0 _pchfile)
            get_filename_component(_pchname ${_pchfile} NAME)
            foreach(_file ${_files})
                get_filename_component(_name ${_file} NAME)
                if (${_name} STRLESS ${_pchname})
                    set(_pchname ${_name})
                    set(_pchfile ${_file})
                endif()
                set_source_files_properties(${_file} PROPERTIES COMPILE_DEFINITIONS ${_pchdef})
            endforeach()
            set_source_files_properties(${_pchfile} PROPERTIES COMPILE_FLAGS "/Yc${_pch}")

        elseif (XCODE)

            if (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/${_pch})
                set(_pch ${NCBI_CURRENT_SOURCE_DIR}/${_pch})
            elseif (EXISTS ${NCBI_INC_ROOT}/${_pch})
                set(_pch ${NCBI_INC_ROOT}/${_pch})
            else()
                file(RELATIVE_PATH _rel "${NCBI_SRC_ROOT}" "${NCBI_CURRENT_SOURCE_DIR}")
                if (EXISTS "${NCBI_INC_ROOT}/${_rel}/${_pch}")
                    set(_pch "${NCBI_INC_ROOT}/${_rel}/${_pch}")
                else()
#                    message(WARNING "ERROR: in project ${NCBI_CURRENT_SOURCE_DIR}/${NCBI_PROJECT}, precompiled header ${_pch} was not found.")
                    set(_usepch NO)
                endif()
            endif()
            if (_usepch)
                set_target_properties(${NCBI_PROJECT} PROPERTIES XCODE_ATTRIBUTE_GCC_PRECOMPILE_PREFIX_HEADER "YES")
                set_target_properties(${NCBI_PROJECT} PROPERTIES XCODE_ATTRIBUTE_GCC_PREFIX_HEADER "${_pch}")
            endif()
        endif()
    endif()
endfunction()

##############################################################################
function(NCBI_internal_install_component_files _comp)
    if (NOT DEFINED NCBI_COMPONENT_${_comp}_BINPATH)
        return()
    endif()

    foreach(_cfg ${CMAKE_CONFIGURATION_TYPES})
        if (DEFINED NCBI_COMPONENT_${_comp}_BINPATH_${_cfg})
            set(_src ${NCBI_COMPONENT_${_comp}_BINPATH_${_cfg}})
        elseif (DEFINED NCBI_COMPONENT_${_comp}_BINPATH)
            if (IS_DIRECTORY ${NCBI_COMPONENT_${_comp}_BINPATH}/${_cfg})
                set(_src ${NCBI_COMPONENT_${_comp}_BINPATH}/${_cfg})
            else()
                set(_src ${NCBI_COMPONENT_${_comp}_BINPATH})
            endif()
        else()
            continue()
        endif()
        if (IS_DIRECTORY ${_src})
            file(GLOB _files LIST_DIRECTORIES false "${_src}/*${CMAKE_SHARED_LIBRARY_SUFFIX}")
            foreach(_file IN LISTS _files)
#message("============ COPY ${_file}  DESTINATION ${CMAKE_RUNTIME_OUTPUT_DIRECTORY}/${_cfg}")
                file(COPY ${_file}  DESTINATION ${CMAKE_RUNTIME_OUTPUT_DIRECTORY}/${_cfg})
            endforeach()
        endif()
    endforeach()
endfunction()

##############################################################################
macro(NCBI_internal_parse_sign _input _value _negative)
    string(SUBSTRING ${_input} 0 1 _sign)
    if ("${_sign}" STREQUAL "-")
        string(SUBSTRING ${_input} 1 -1 ${_value})
        set(${_negative} ON)
    else()
        set(${_value} ${_input})
        set(${_negative} OFF)
    endif()
endmacro()

##############################################################################
macro(NCBI_internal_process_project_requires)
    set(NCBITMP_REQUIRE_NOTFOUND "")
    if(NCBI_PTBMODE_PARTS)
        set(_all ${NCBI__REQUIRES} ${NCBI_${NCBI_PROJECT}_REQUIRES})
    else()
        set(_all ${NCBI__REQUIRES} ${NCBI_${NCBI_PROJECT}_REQUIRES} ${NCBITMP_PROJECT_REQUIRES})
    endif()
    if (NOT "${_all}" STREQUAL "")
        list(REMOVE_DUPLICATES _all)
    endif()

    set(NCBITMP_REQUIRE ${_all})
    if(NCBI_PTBMODE_COLLECT_DEPS)
        set_property(GLOBAL PROPERTY NCBI_PTBPROP_REQUIRES_${NCBI_PROJECT} ${_all})
    endif()

    foreach(_req IN LISTS _all)
        NCBI_internal_parse_sign(${_req} _value _negate)
        if (NCBI_REQUIRE_${_value}_FOUND OR NCBI_COMPONENT_${_value}_FOUND)
            if (_negate)
                set(NCBITMP_REQUIRE_NOTFOUND ${NCBITMP_REQUIRE_NOTFOUND} ${_req})
            else()
                set(NCBITMP_INCLUDES ${NCBITMP_INCLUDES} ${NCBI_COMPONENT_${_value}_INCLUDE})
                set(NCBITMP_DEFINES  ${NCBITMP_DEFINES}  ${NCBI_COMPONENT_${_value}_DEFINES})
                set(NCBITMP_EXTLIB   ${NCBITMP_EXTLIB}   ${NCBI_COMPONENT_${_value}_LIBS})
                foreach(_lib IN LISTS NCBI_COMPONENT_${_value}_NCBILIB)
                    if(NOT ${_lib} STREQUAL ${NCBI_${NCBI_PROJECT}_OUTPUT})
                        set(NCBITMP_NCBILIB  ${NCBITMP_NCBILIB}  ${_lib})
                    endif()
                endforeach()
                if (WIN32 AND NOT NCBI_PTBMODE_COLLECT_DEPS)
                    NCBI_internal_install_component_files(${_value})
                endif()
            endif()
        else()
            if (_negate)
# no-no, this is a mistake
#        set(NCBITMP_INCLUDES ${NCBITMP_INCLUDES} ${NCBI_COMPONENT_${_value}_INCLUDE})
#        set(NCBITMP_DEFINES  ${NCBITMP_DEFINES}  ${NCBI_COMPONENT_${_value}_DEFINES})
#        set(NCBITMP_EXTLIB     ${NCBITMP_EXTLIB}     ${NCBI_COMPONENT_${_value}_LIBS})
            else()
                set(NCBITMP_REQUIRE_NOTFOUND ${NCBITMP_REQUIRE_NOTFOUND} ${_req})
            endif()
        endif()     
    endforeach()

    if (NOT NCBI_PTBMODE_COLLECT_DEPS AND NOT ${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "STATIC")
        get_property(_all GLOBAL PROPERTY NCBI_PTBPROP_IMPLREQ_${NCBI_PROJECT})
        foreach(_req IN LISTS _all)
            NCBI_internal_parse_sign(${_req} _value _negate)
            if (NCBI_REQUIRE_${_value}_FOUND OR NCBI_COMPONENT_${_value}_FOUND)
                if (_negate)
                    set(NCBITMP_REQUIRE_NOTFOUND ${NCBITMP_REQUIRE_NOTFOUND} ${_req})
                endif()
            else()
                if (NOT _negate)
                    set(NCBITMP_REQUIRE_NOTFOUND ${NCBITMP_REQUIRE_NOTFOUND} ${_req})
                endif()
            endif()     
        endforeach()
    endif()

    if (NOT "${NCBITMP_REQUIRE_NOTFOUND}" STREQUAL "")
        list(REMOVE_DUPLICATES NCBITMP_REQUIRE_NOTFOUND)
    endif()
endmacro()

##############################################################################
macro(NCBI_internal_process_project_components)
    set(NCBITMP_COMPONENT_NOTFOUND "")
    if(NCBI_PTBMODE_PARTS)
        set(_all ${NCBI__COMPONENTS} ${NCBI_${NCBI_PROJECT}_COMPONENTS})
    else()
        set(_all ${NCBI__COMPONENTS} ${NCBI_${NCBI_PROJECT}_COMPONENTS} ${NCBITMP_PROJECT_COMPONENTS})
    endif()
    if (NOT "${_all}" STREQUAL "")
        list(REMOVE_DUPLICATES _all)
    endif()

    if(NCBI_PTBMODE_COLLECT_DEPS)
        set_property(GLOBAL PROPERTY NCBI_PTBPROP_COMPONENTS_${NCBI_PROJECT} ${_all})
    endif()

    foreach(_value IN LISTS _all)
        if (NCBI_REQUIRE_${_value}_FOUND OR NCBI_COMPONENT_${_value}_FOUND)
            set(NCBITMP_INCLUDES ${NCBITMP_INCLUDES} ${NCBI_COMPONENT_${_value}_INCLUDE})
            set(NCBITMP_DEFINES  ${NCBITMP_DEFINES}  ${NCBI_COMPONENT_${_value}_DEFINES})
            set(NCBITMP_EXTLIB   ${NCBITMP_EXTLIB}   ${NCBI_COMPONENT_${_value}_LIBS})
            foreach(_lib IN LISTS NCBI_COMPONENT_${_value}_NCBILIB)
                if(NOT ${_lib} STREQUAL ${NCBI_${NCBI_PROJECT}_OUTPUT})
                    set(NCBITMP_NCBILIB  ${NCBITMP_NCBILIB}  ${_lib})
                endif()
            endforeach()
            if (WIN32 AND NOT NCBI_PTBMODE_COLLECT_DEPS)
                NCBI_internal_install_component_files(${_value})
            endif()
        else()
            set(NCBITMP_COMPONENT_NOTFOUND ${NCBITMP_COMPONENT_NOTFOUND} ${_value})
        endif()
    endforeach()
endmacro()

##############################################################################
function(NCBI_internal_collect_parts)
    set(NCBI_PTBMODE_PARTS ON)
    set(_hostproject ${NCBI_PROJECT})

    foreach(_part IN LISTS NCBI_${_hostproject}_PARTS)
        set(_filepath ${NCBI_SRC_ROOT}/${_part})
        get_filename_component(_path ${_filepath} DIRECTORY)
        get_filename_component(_lib ${_filepath} NAME)

        unset(NCBI_PROJECT)
        unset(NCBI_PROJECT_ID)
        unset(NCBI__USEPCH)
        unset(NCBI__REQUIRES)
        unset(NCBI__COMPONENTS)
        unset(NCBI__INCLUDES)
        unset(NCBI__DEFINES)
        unset(NCBI__NCBILIB)
        unset(NCBI__EXTLIB)

        file(RELATIVE_PATH _relpath ${NCBI_SRC_ROOT} ${_path})
        string(REPLACE "/" ";" _dirlist ${_relpath})
        set(NCBI_CURRENT_SOURCE_DIR ${NCBI_SRC_ROOT})
        foreach(_dir ${_dirlist})
            set(NCBI_CURRENT_SOURCE_DIR ${NCBI_CURRENT_SOURCE_DIR}/${_dir})
            include(${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.txt)
        endforeach()
        set(NCBI_CURRENT_SOURCE_DIR ${_path})

        if (EXISTS ${_path}/CMakeLists.${_lib}.txt)
            include(${_path}/CMakeLists.${_lib}.txt)
        elseif (EXISTS ${_path}/CMakeLists.${_lib}.lib.txt)
            include(${_path}/CMakeLists.${_lib}.lib.txt)
        elseif (EXISTS ${_path}/CMakeLists.${_lib}.asn.txt)
            include(${_path}/CMakeLists.${_lib}.asn.txt)
        else()
            message(WARNING "ERROR: part of ${_hostproject} project not found: ${NCBI_SRC_ROOT}/${_part}")
        endif()
    endforeach()

    set(NCBITMP_PROJECT_PART_IDS    ${NCBITMP_PROJECT_PART_IDS}   PARENT_SCOPE)
    set(NCBITMP_PROJECT_PARTS       ${NCBITMP_PROJECT_PARTS}      PARENT_SCOPE)
    set(NCBITMP_PROJECT_REQUIRES    ${NCBITMP_PROJECT_REQUIRES}   PARENT_SCOPE)
    set(NCBITMP_PROJECT_COMPONENTS  ${NCBITMP_PROJECT_COMPONENTS} PARENT_SCOPE)
    set(NCBITMP_PROJECT_INCLUDES    ${NCBITMP_PROJECT_INCLUDES}   PARENT_SCOPE)
    set(NCBITMP_PROJECT_DEFINES     ${NCBITMP_PROJECT_DEFINES}    PARENT_SCOPE)
    set(NCBITMP_PROJECT_NCBILIB     ${NCBITMP_PROJECT_NCBILIB}    PARENT_SCOPE)
    set(NCBITMP_PROJECT_EXTLIB      ${NCBITMP_PROJECT_EXTLIB}     PARENT_SCOPE)
    set(NCBITMP_PROJECT_SOURCES     ${NCBITMP_PROJECT_SOURCES}    PARENT_SCOPE)
    set(NCBITMP_PROJECT_NOPCH       ${NCBITMP_PROJECT_NOPCH}      PARENT_SCOPE)
    set(NCBITMP_PROJECT_DATASPEC    ${NCBITMP_PROJECT_DATASPEC}   PARENT_SCOPE)
    set(NCBITMP_PROJECT_HEADERS     ${NCBITMP_PROJECT_HEADERS}    PARENT_SCOPE)
endfunction()

##############################################################################
macro(NCBI_internal_process_parts)

    set(NCBITMP_PROJECT_PART_IDS "")
    set(NCBITMP_PROJECT_PARTS "")
    NCBI_internal_collect_parts()

    if (NCBI_PTBMODE_COLLECT_DEPS)
#set_property(GLOBAL PROPERTY NCBI_PTBPROP_PARTS_${NCBI_PROJECT_ID} ${NCBITMP_PROJECT_PART_IDS})
        foreach(_part IN LISTS NCBITMP_PROJECT_PART_IDS)
            set_property(GLOBAL PROPERTY NCBI_PTBPROP_HOSTID_${_part} ${NCBI_PROJECT_ID})
        endforeach()
        foreach(_part IN LISTS NCBITMP_PROJECT_PARTS NCBI_${NCBI_PROJECT}_VIRTPARTS)
            if (NOT "${_part}" STREQUAL ${NCBI_PROJECT})
                set_property(GLOBAL PROPERTY NCBI_PTBPROP_HOST_${_part} ${NCBI_PROJECT})
            endif()
        endforeach()
    endif()
endmacro()

##############################################################################
function(NCBI_internal_process_interface_libraries _lib)
    if (NOT TARGET ${_lib})
        return()
    endif()
    get_target_property(_deps ${_lib} INTERFACE_LINK_LIBRARIES)
    if ( NOT "${_deps}" STREQUAL "")
        set(_value "")
        foreach(_prj IN LISTS _deps)
            get_property(_local GLOBAL PROPERTY NCBI_PTBPROP_LOCAL_${_prj})
            if ("${_local}" STREQUAL "")
                set(_value ${_value} ${_prj})
                NCBI_internal_process_interface_libraries(${_prj})
            else()
                set(_value ${_value} ${_local})
            endif()
        endforeach()
        set_target_properties(${_lib} PROPERTIES INTERFACE_LINK_LIBRARIES "${_value}")
    endif()
endfunction()

##############################################################################
function(NCBI_internal_verify_ncbilibs)
    set(_res "")
    set(_exclude "")
    foreach(_prj IN LISTS NCBITMP_NCBILIB)
        NCBI_internal_parse_sign(${_prj} _value _negate)
        list(APPEND _res ${_value})
        if (_negate)
            list(APPEND _exclude ${_value})
        endif()
    endforeach()
    foreach(_prj IN LISTS _exclude)
        list(REMOVE_ITEM _res ${_prj})
    endforeach()
    set(NCBITMP_NCBILIB ${_res} PARENT_SCOPE)
endfunction()

##############################################################################
function(NCBI_internal_verify_libs)
    set(_optimize NO)
    if (WIN32 AND NCBI_PTBCFG_ENABLE_COLLECTOR AND NOT DEFINED NCBI_EXTERNAL_TREE_ROOT AND NOT DEFINED NCBI_PTBCFG_DOINSTALL)
        if (${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "STATIC")
            set(_ncbilib ${NCBITMP_NCBILIB})
            set(_optimize YES)
        else()
            get_property(_ncbilib GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${NCBI_PROJECT})
            if ("${_ncbilib}" STREQUAL "")
                set(_ncbilib ${NCBITMP_NCBILIB})
            endif()
        endif()
    else()
        set(_ncbilib ${NCBITMP_NCBILIB})
    endif()
    if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
        message("NCBI_internal_verify_libs ${NCBI_PROJECT_ID}:  on enter = ${_ncbilib}")
    endif()
    set(_value "")
    if ( NOT "${_ncbilib}" STREQUAL "")
        list(REMOVE_DUPLICATES _ncbilib)
        foreach(_prj IN LISTS _ncbilib)
            get_property(_host GLOBAL PROPERTY NCBI_PTBPROP_HOST_${_prj})
            if ("${_host}" STREQUAL "")
                if (_optimize)
                    get_property(_hasspec GLOBAL PROPERTY NCBI_PTBPROP_DATASPEC_${_prj})
                    if (_hasspec)
                        set(_value ${_value} ${_prj})
                    endif()
                else()
                    set(_value ${_value} ${_prj})
                endif()
            else()
                set(_value ${_value} ${_host})
            endif()
        endforeach()
    endif()
    if ( NOT "${_value}" STREQUAL "")
        list(REMOVE_DUPLICATES _value)
        list(REMOVE_ITEM _value ${NCBI_PROJECT})
        set(_ncbilib ${_value})
    endif()

    if ( NOT "${_ncbilib}" STREQUAL "" AND DEFINED NCBI_EXTERNAL_TREE_ROOT)
        set(_value "")
        foreach(_prj IN LISTS _ncbilib)
            get_property(_local GLOBAL PROPERTY NCBI_PTBPROP_LOCAL_${_prj})
            if ("${_local}" STREQUAL "")
                set(_value ${_value} ${_prj})
                NCBI_internal_process_interface_libraries(${_prj})
            else()
                set(_value ${_value} ${_local})
            endif()
        endforeach()
    endif()

    set(_tk_libs "${_value}")
    set(_value "")
    foreach(_tk_lib IN LISTS _tk_libs)
        if(${_tk_lib} STREQUAL "general")
            if(NCBI_EXPERIMENTAL_CFG)
                set(_value ${_value} \$<1:general>)
            else()
                set(_value ${_value} general-lib)
            endif()
        else()
            set(_value ${_value} ${_tk_lib})
        endif()
    endforeach()

    if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
        message("NCBI_internal_verify_libs ${NCBI_PROJECT_ID}:  on exit = ${_value}")
    endif()
    set(NCBITMP_NCBILIB ${_value} PARENT_SCOPE)
endfunction()

##############################################################################
function(NCBI_internal_process_project_filters _result)
    if(NOT "${NCBI_PTBCFG_PROJECT_TARGETS}" STREQUAL "")
        set(_is_good FALSE)
        foreach(_prj IN LISTS NCBI_PTBCFG_PROJECT_TARGETS)
            if (${NCBI_PROJECT} STREQUAL ${_prj})
                set(${_result} TRUE PARENT_SCOPE)
                return()
            endif()
            if (${NCBI_PROJECT} MATCHES ${_prj})
                set(_is_good TRUE)
            endif()
        endforeach()
        if (NOT _is_good)
            set(${_result} FALSE PARENT_SCOPE)
            return()
        endif()
    endif()
    if(NOT "${NCBI_PTBCFG_PROJECT_LIST}" STREQUAL "")
        set(_is_good FALSE)
        foreach(_dir IN LISTS NCBI_PTBCFG_PROJECT_LIST)
            if(${NCBI_CURRENT_SOURCE_DIR} MATCHES ${NCBI_SRC_ROOT}/${_dir})
                set(_is_good TRUE)
            endif()
        endforeach()
        if (NOT _is_good)
            set(${_result} FALSE PARENT_SCOPE)
            return()
        endif()
    endif()
    if(NOT "${NCBI_PTBCFG_PROJECT_TAGS}" STREQUAL "")
        set(_all ${NCBI__PROJTAG} ${NCBI_${NCBI_PROJECT}_PROJTAG})

        foreach(_tag IN LISTS _all)
            list(FIND NCBI_PTBCFG_PROJECT_TAGS "-${_tag}" _found)
            if (${_found} GREATER_EQUAL "0")
                set(${_result} FALSE PARENT_SCOPE)
                return()
            endif()

            list(FIND NCBI_PTBCFG_PROJECT_TAGS "${_tag}" _found)
            if (${_found} LESS "0")
                list(FIND NCBI_PTBCFG_PROJECT_TAGS "*" _found)
                if (${_found} LESS "0")
                    set(${_result} FALSE PARENT_SCOPE)
                    return()
                endif()
            endif()
        endforeach()
    endif()
    set(${_result} TRUE PARENT_SCOPE)
endfunction()

##############################################################################
macro(NCBI_internal_process_test_requires _test)
    set(NCBITEST_REQUIRE_NOTFOUND "")
    set(_all ${NCBITEST__REQUIRES} ${NCBITEST_${_test}_REQUIRES})
    if (NOT "${_all}" STREQUAL "")
        list(REMOVE_DUPLICATES _all)
    endif()

    foreach(_req IN LISTS _all)
        NCBI_internal_parse_sign( ${_req} _value _negate)
        if (NCBI_REQUIRE_${_value}_FOUND OR NCBI_COMPONENT_${_value}_FOUND)
            if (_negate)
                set(NCBITEST_REQUIRE_NOTFOUND ${NCBITEST_REQUIRE_NOTFOUND} ${_req})
            endif()
        else()
            if (NOT _negate)
                set(NCBITEST_REQUIRE_NOTFOUND ${NCBITEST_REQUIRE_NOTFOUND} ${_req})
            endif()
        endif()     
    endforeach()
endmacro()

##############################################################################
function(NCBI_internal_add_test _test)
    if( NOT DEFINED NCBITEST_${_test}_CMD)
        set(NCBITEST_${_test}_CMD ${NCBI_${NCBI_PROJECT}_OUTPUT})
    endif()
    get_filename_component(_ext ${NCBITEST_${_test}_CMD} EXT)
    if("${_ext}" STREQUAL ".sh")
        set(NCBITEST_${_test}_REQUIRES ${NCBITEST_${_test}_REQUIRES} -MSWin)
        set(NCBITEST_${_test}_ASSETS ${NCBITEST__ASSETS} ${NCBITEST_${_test}_ASSETS} ${NCBITEST_${_test}_CMD})
    endif()

    NCBI_internal_process_test_requires(${_test})
    if ( NOT "${NCBITEST_REQUIRE_NOTFOUND}" STREQUAL "")
        if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
            message("${NCBI_PROJECT} (${NCBI_CURRENT_SOURCE_DIR}): Test ${_test} of project ${NCBI_PROJECT} is excluded because of unmet requirements: ${NCBITEST_REQUIRE_NOTFOUND}")
        endif()
        return()
    endif()

    if (DEFINED NCBITEST_${_test}_ASSETS)
        set(_assets ${NCBITEST_${_test}_ASSETS})
    elseif(DEFINED NCBITEST__ASSETS)
        set(_assets ${NCBITEST__ASSETS})
    endif()
    if (DEFINED NCBITEST_${_test}_TIMEOUT)
        set(_timeout ${NCBITEST_${_test}_TIMEOUT})
    elseif(DEFINED NCBITEST__TIMEOUT)
        set(_timeout ${NCBITEST__TIMEOUT})
    else()
        set(_timeout 86400)
    endif()
    string(REPLACE ";" " " _args    "${NCBITEST_${_test}_ARG}")
    string(REPLACE ";" " " _assets   "${_assets}")

    if (XCODE)
        set(_extra -DXCODE=TRUE)
    endif()

    file(RELATIVE_PATH _outdir "${NCBI_SRC_ROOT}" "${NCBI_CURRENT_SOURCE_DIR}")
    if (WIN32 OR XCODE)
        set(_outdir ${NCBI_BUILD_ROOT}/${NCBI_DIRNAME_TESTING}/$<CONFIG>/${_outdir})
    else()
        set(_outdir ${NCBI_BUILD_ROOT}/${NCBI_DIRNAME_TESTING}/${_outdir})
    endif()

    add_test(NAME ${_test} COMMAND ${CMAKE_COMMAND}
        -DNCBITEST_NAME=${_test}
        -DNCBITEST_CONFIG=$<CONFIG>
        -DNCBITEST_COMMAND=${NCBITEST_${_test}_CMD}
        -DNCBITEST_ARGS=${_args}
        -DNCBITEST_TIMEOUT=${_timeout}
        -DNCBITEST_BINDIR=${CMAKE_RUNTIME_OUTPUT_DIRECTORY}
        -DNCBITEST_SOURCEDIR=${NCBI_CURRENT_SOURCE_DIR}
        -DNCBITEST_OUTDIR=${_outdir}
        -DNCBITEST_ASSETS=${_assets}
        ${_extra}
        -P "${NCBITEST_DRIVER}")
endfunction()

##############################################################################
function(NCBI_internal_export_hostinfo _file)
    if(EXISTS ${_file})
        file(REMOVE ${_file})
    endif()
    get_property(_allprojects     GLOBAL PROPERTY NCBI_PTBPROP_ALL_PROJECTS)
    if (NOT "${_allprojects}" STREQUAL "")
        set(_hostinfo)
        foreach(_prj IN LISTS _allprojects)
            get_property(_prjhost GLOBAL PROPERTY NCBI_PTBPROP_HOST_${_prj})
            if (NOT "${_prjhost}" STREQUAL "")
                list(APPEND _hostinfo "${_prj} ${_prjhost}\n")

            endif()
        endforeach()
        if (NOT "${_hostinfo}" STREQUAL "")
            file(WRITE ${_file} ${_hostinfo})
        endif()
    endif()
endfunction()

##############################################################################
function(NCBI_internal_import_hostinfo _file)
    if(NOT EXISTS ${_file})
        return()
    endif()
    file(STRINGS ${_file} _hostinfo)
    if (NOT "${_hostinfo}" STREQUAL "")
        foreach( _item IN LISTS _hostinfo)
            string(REPLACE " " ";" _item ${_item})
            if (NOT "${_item}" STREQUAL "")
                list(GET _item 0 _prj)
                list(GET _item 1 _host)
                set_property(GLOBAL PROPERTY NCBI_PTBPROP_HOST_${_prj} ${_host})
            endif()
        endforeach()
    endif()
endfunction()

##############################################################################
function(NCBI_internal_install_root)

    file(RELATIVE_PATH _dest "${NCBI_TREE_ROOT}" "${NCBI_BUILD_ROOT}")

    set(_hostinfo ${NCBI_BUILD_ROOT}/${NCBI_DIRNAME_BUILD}/${CMAKE_PROJECT_NAME}.hostinfo)
    NCBI_internal_export_hostinfo(${_hostinfo})
    if (EXISTS ${_hostinfo})
        install( FILES ${_hostinfo} DESTINATION ${_dest}/${NCBI_DIRNAME_EXPORT} RENAME ${NCBI_PTBCFG_INSTALL_EXPORT}.hostinfo)
    endif()

    if (WIN32 OR XCODE)
        foreach(_cfg ${CMAKE_CONFIGURATION_TYPES})
            install(EXPORT ${NCBI_PTBCFG_INSTALL_EXPORT}${_cfg}
                CONFIGURATIONS ${_cfg}
                DESTINATION ${_dest}/${NCBI_DIRNAME_EXPORT}
                FILE ${NCBI_PTBCFG_INSTALL_EXPORT}.cmake
            )
        endforeach()
    else()
        install(EXPORT ${NCBI_PTBCFG_INSTALL_EXPORT}
            DESTINATION ${_dest}/${NCBI_DIRNAME_EXPORT}
            FILE ${NCBI_PTBCFG_INSTALL_EXPORT}.cmake
        )
    endif()

# install headers
    get_property(_all_subdirs GLOBAL PROPERTY NCBI_PTBPROP_ROOT_SUBDIR)
    list(APPEND _all_subdirs ${NCBI_DIRNAME_COMMON_INCLUDE})
    foreach(_dir IN LISTS _all_subdirs)
        if (EXISTS ${NCBI_INC_ROOT}/${_dir})
            install( DIRECTORY ${NCBI_INC_ROOT}/${_dir} DESTINATION ${NCBI_DIRNAME_INCLUDE}
                REGEX "/[.].*$" EXCLUDE)
        endif()
    endforeach()
    file(GLOB _files LIST_DIRECTORIES false "${NCBI_INC_ROOT}/*")
    install( FILES ${_files} DESTINATION ${NCBI_DIRNAME_INCLUDE})

# install sources?
    # TODO

    file(GLOB _files LIST_DIRECTORIES false "${NCBI_TREE_BUILDCFG}/*")
    install( FILES ${_files} DESTINATION ${NCBI_DIRNAME_BUILDCFG})
    install( DIRECTORY ${NCBI_TREE_CMAKECFG} DESTINATION ${NCBI_DIRNAME_BUILDCFG}
            USE_SOURCE_PERMISSIONS REGEX "/[.].*$" EXCLUDE)

    install( DIRECTORY ${NCBI_TREE_ROOT}/${NCBI_DIRNAME_COMMON_SCRIPTS} DESTINATION ${NCBI_DIRNAME_SCRIPTS}
            USE_SOURCE_PERMISSIONS REGEX "/[.].*$" EXCLUDE)

    file(RELATIVE_PATH _dest "${NCBI_TREE_ROOT}" "${NCBI_BUILD_ROOT}")
    install( DIRECTORY ${NCBI_CFGINC_ROOT} DESTINATION "${_dest}"
            REGEX "/[.].*$" EXCLUDE)
endfunction()

##############################################################################
function(NCBI_internal_install_target)

    if (${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "STATIC")
        set(_haspdb NO)
        file(RELATIVE_PATH _dest "${NCBI_TREE_ROOT}" "${CMAKE_ARCHIVE_OUTPUT_DIRECTORY}")
    elseif (${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "SHARED")
        set(_haspdb YES)
        if (WIN32)
            file(RELATIVE_PATH _dest    "${NCBI_TREE_ROOT}" "${CMAKE_LIBRARY_OUTPUT_DIRECTORY}")
            file(RELATIVE_PATH _dest_ar "${NCBI_TREE_ROOT}" "${CMAKE_ARCHIVE_OUTPUT_DIRECTORY}")
        else()
            file(RELATIVE_PATH _dest "${NCBI_TREE_ROOT}" "${CMAKE_ARCHIVE_OUTPUT_DIRECTORY}")
        endif()
    elseif (${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "CONSOLEAPP")
        set(_haspdb YES)
        file(RELATIVE_PATH _dest "${NCBI_TREE_ROOT}" "${CMAKE_RUNTIME_OUTPUT_DIRECTORY}")
        if (DEFINED NCBI_PTBCFG_INSTALL_TAGS)
            set(_alltags ${NCBI__PROJTAG} ${NCBI_${NCBI_PROJECT}_PROJTAG})
            foreach(_tag IN LISTS _alltags)
                list(FIND NCBI_PTBCFG_INSTALL_TAGS "-${_tag}" _found)
                if (${_found} GREATER_EQUAL "0")
                    if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
                        message("${NCBI_PROJECT} will not be installed because of tag ${_tag}")
                    endif()
                    return()
                endif()
                list(FIND NCBI_PTBCFG_INSTALL_TAGS "${_tag}" _found)
                if (${_found} LESS "0")
                    list(FIND NCBI_PTBCFG_INSTALL_TAGS "*" _found)
                    if (${_found} LESS "0")
                        if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
                            message("${NCBI_PROJECT} will not be installed because of tag ${_tag}")
                        endif()
                        return()
                    endif()
                endif()
            endforeach()
        endif()
    else()
        return()
    endif()
    if ("${_dest}" STREQUAL "")
        return()
    endif()

# not sure about this part
    file(RELATIVE_PATH _rel "${NCBI_SRC_ROOT}" "${NCBI_CURRENT_SOURCE_DIR}")
    string(REPLACE "/" ";" _rel ${_rel})
    list(GET _rel 0 _dir)
    get_property(_all_subdirs GLOBAL PROPERTY NCBI_PTBPROP_ROOT_SUBDIR)
    list(APPEND _all_subdirs ${_dir})
    if (DEFINED NCBI_${NCBI_PROJECT}_PARTS)
        foreach(_rel IN LISTS NCBI_${NCBI_PROJECT}_PARTS)
            string(REPLACE "/" ";" _rel ${_rel})
            list(GET _rel 0 _dir)
            list(APPEND _all_subdirs ${_dir})
        endforeach()
    endif()
    list(REMOVE_DUPLICATES _all_subdirs)
    set_property(GLOBAL PROPERTY NCBI_PTBPROP_ROOT_SUBDIR ${_all_subdirs})

    if (WIN32 OR XCODE)
        foreach(_cfg ${CMAKE_CONFIGURATION_TYPES})
            if (DEFINED _dest_ar)
                install(
                    TARGETS ${NCBI_PROJECT}
                    EXPORT ${NCBI_PTBCFG_INSTALL_EXPORT}${_cfg}
                    RUNTIME DESTINATION ${_dest}/${_cfg}
                    CONFIGURATIONS ${_cfg}
                    ARCHIVE DESTINATION ${_dest_ar}/${_cfg}
                    CONFIGURATIONS ${_cfg}
                )
            else()
                install(
                    TARGETS ${NCBI_PROJECT}
                    EXPORT ${NCBI_PTBCFG_INSTALL_EXPORT}${_cfg}
                    DESTINATION ${_dest}/${_cfg}
                    CONFIGURATIONS ${_cfg}
                )
            endif()
            if (WIN32 AND _haspdb)
                install(FILES $<TARGET_PDB_FILE:${NCBI_PROJECT}>
                        DESTINATION ${_dest}/${_cfg} OPTIONAL
                        CONFIGURATIONS ${_cfg})
            endif()
        endforeach()
    else()
        install(
            TARGETS ${NCBI_PROJECT}
            EXPORT ${NCBI_PTBCFG_INSTALL_EXPORT}
            DESTINATION ${_dest}
        )
    endif()
endfunction()

##############################################################################
function(NCBI_internal_print_project_info _prj)
    message("=============================================================================")
    get_property(_dir GLOBAL PROPERTY NCBI_PTBPROP_DIR_${_prj})
    message("PROJECT: ${_prj} (${_dir})")
    get_property(_x GLOBAL PROPERTY NCBI_PTBPROP_REQUIRES_${_prj})
    if(NOT "${_x}" STREQUAL "")
        list(SORT _x)
        message("  REQUIRES:  ${_x}")
    endif()
    get_property(_x GLOBAL PROPERTY NCBI_PTBPROP_COMPONENTS_${_prj})
    if(NOT "${_x}" STREQUAL "")
        list(SORT _x)
        message("  OPTIONAL COMPONENTS:  ${_x}")
    endif()
    get_property(_x GLOBAL PROPERTY NCBI_PTBPROP_DIRECT_DEPS_${_prj})
    if(NOT "${_x}" STREQUAL "")
        message("  DEPENDENCIES:  ${_x}")
    endif()
    get_property(_x GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_prj})
    if(NOT "${_x}" STREQUAL "")
        list(SORT _x)
        message("-----------------------------------------------------------------------------")
        message("  ALL DEPENDENCIES:  ${_x}")
        foreach(_dep IN LISTS _x)
            get_property(_dir GLOBAL PROPERTY NCBI_PTBPROP_DIR_${_dep})
            message("  ${_dep} (${_dir})")
            get_property(_y GLOBAL PROPERTY NCBI_PTBPROP_REQUIRES_${_dep})
            if(NOT "${_y}" STREQUAL "")
                list(SORT _y)
                message("    REQUIRES:  ${_y}")
            endif()
            get_property(_y GLOBAL PROPERTY NCBI_PTBPROP_COMPONENTS_${_dep})
            if(NOT "${_y}" STREQUAL "")
                list(SORT _y)
                message("    OPTIONAL COMPONENTS:  ${_y}")
            endif()
            get_property(_y GLOBAL PROPERTY NCBI_PTBPROP_DIRECT_DEPS_${_dep})
            if(NOT "${_y}" STREQUAL "")
                list(SORT _y)
                message("    DEPENDENCIES:  ${_y}")
            endif()
        endforeach()
    endif()
    get_property(_x GLOBAL PROPERTY NCBI_PTBPROP_IMPLREQ_${_prj})
    if(NOT "${_x}" STREQUAL "")
        message("-----------------------------------------------------------------------------")
        message("  ALL REQUIRED COMPONENTS: ${_x}")
        foreach(_req IN LISTS _x)
            if (NCBI_REQUIRE_${_req}_FOUND)
            elseif (NCBI_COMPONENT_${_req}_FOUND)
                message("  ${_req}")
                if (DEFINED NCBI_COMPONENT_${_req}_INCLUDE)
                    message("    INCLUDE: ${NCBI_COMPONENT_${_req}_INCLUDE}")
                endif()
                if (DEFINED NCBI_COMPONENT_${_req}_DEFINES)
                    message("    DEFINES: ${NCBI_COMPONENT_${_req}_DEFINES}")
                endif()
                if (DEFINED NCBI_COMPONENT_${_req}_LIBS)
                    message("    LIBS: ${NCBI_COMPONENT_${_req}_LIBS}")
                endif()
                if (DEFINED NCBI_COMPONENT_${_req}_BINPATH)
                    message("    BINPATH: ${NCBI_COMPONENT_${_req}_BINPATH}")
                endif()
            else()
                message("  ${_req} not found")
            endif()
        endforeach()
    endif()
    message("=============================================================================")
endfunction()

##############################################################################
function(NCBI_internal_add_project)

    if (NOT NCBI_PTBMODE_PARTS AND NOT NCBI_PTBMODE_COLLECT_DEPS AND NCBI_PTBCFG_ENABLE_COLLECTOR)
        if(DEFINED NCBI_PTB_ALLOWED_PROJECTS)
            list(FIND NCBI_PTB_ALLOWED_PROJECTS ${NCBI_PROJECT} _found)
            if (${_found} LESS "0")
                if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
                    message("${NCBI_PROJECT} (${NCBI_CURRENT_SOURCE_DIR}) is excluded by user's request")
                endif()
                if ("${ARGC}" GREATER "0")
                    set(${ARGV0} FALSE PARENT_SCOPE)
                endif()
                return()
            endif()
        endif()
        get_property(_hosted GLOBAL PROPERTY NCBI_PTBPROP_HOSTID_${NCBI_PROJECT_ID})
        if ("${_hosted}" STREQUAL "")
            get_property(_hosted GLOBAL PROPERTY NCBI_PTBPROP_HOST_${NCBI_PROJECT})
        endif()

        if (NOT "${_hosted}" STREQUAL "")
            if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
                message("${NCBI_PROJECT_ID} (${NCBI_CURRENT_SOURCE_DIR}) is excluded because it is part of ${_hosted}")
            endif()
            if ("${ARGC}" GREATER "0")
                set(${ARGV0} FALSE PARENT_SCOPE)
            endif()
            return()
        endif()
    endif()

if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
message("-----------------------------------")
message("NCBI_PROJECT = ${NCBI_PROJECT}")
message("  TYPE = ${NCBI_${NCBI_PROJECT}_TYPE}")
message("  SOURCES = ${NCBI_${NCBI_PROJECT}_SOURCES} ${NCBI_${NCBI_PROJECT}_GENERATED_SOURCES}")
message("  RESOURCES = ${NCBI_${NCBI_PROJECT}_RESOURCES}")
message("  HEADERS = ${NCBI_${NCBI_PROJECT}_HEADERS}")
message("  REQUIRES = ${NCBI_${NCBI_PROJECT}_REQUIRES}")
message("  COMPONENTS = ${NCBI_${NCBI_PROJECT}_COMPONENTS}")
message("  NOPCH = ${NCBI_${NCBI_PROJECT}_NOPCH}")
message("  NCBILIB = ${NCBI_${NCBI_PROJECT}_NCBILIB}")
message("  EXTLIB = ${NCBI_${NCBI_PROJECT}_EXTLIB}")
message("  INCLUDES = ${NCBI_${NCBI_PROJECT}_INCLUDES}")
message("  DEFINES = ${NCBI_${NCBI_PROJECT}_DEFINES}")
endif()

    if (DEFINED NCBI_${NCBI_PROJECT}_PARTS)
        NCBI_internal_process_parts()
    endif()

    set(NCBITMP_INCLUDES ${NCBITMP_PROJECT_INCLUDES} ${NCBI__INCLUDES} ${NCBI_${NCBI_PROJECT}_INCLUDES})
    set(NCBITMP_DEFINES  ${NCBITMP_PROJECT_DEFINES}  ${NCBI__DEFINES}  ${NCBI_${NCBI_PROJECT}_DEFINES})
    set(NCBITMP_NCBILIB  ${NCBITMP_PROJECT_NCBILIB}  ${NCBI__NCBILIB}  ${NCBI_${NCBI_PROJECT}_NCBILIB})
    set(NCBITMP_EXTLIB   ${NCBITMP_PROJECT_EXTLIB}   ${NCBI__EXTLIB}   ${NCBI_${NCBI_PROJECT}_EXTLIB})

    NCBI_internal_process_project_requires()
    if ( NOT "${NCBITMP_REQUIRE_NOTFOUND}" STREQUAL "" AND NOT NCBI_PTBMODE_COLLECT_DEPS)
        message("${NCBI_PROJECT} (${NCBI_CURRENT_SOURCE_DIR}) is excluded because of unmet requirements: ${NCBITMP_REQUIRE_NOTFOUND}")
        if (NOT NCBI_PTBMODE_PARTS)
            if ("${ARGC}" GREATER "0")
                set(${ARGV0} FALSE PARENT_SCOPE)
            endif()
            return()
        endif()
    endif()

    NCBI_internal_process_project_components()
    if ( NOT "${NCBITMP_COMPONENT_NOTFOUND}" STREQUAL "" AND NOT NCBI_PTBMODE_COLLECT_DEPS)
        message("${NCBI_PROJECT} (${NCBI_CURRENT_SOURCE_DIR}): cannot find optional component: ${NCBITMP_COMPONENT_NOTFOUND}")
    endif()
    if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
        message("  NCBITMP_INCLUDES = ${NCBITMP_INCLUDES}")
        message("  NCBITMP_DEFINES = ${NCBITMP_DEFINES}")
        message("  NCBITMP_NCBILIB = ${NCBITMP_NCBILIB}")
        message("  NCBITMP_EXTLIB = ${NCBITMP_EXTLIB}")
    endif()
    NCBI_internal_verify_ncbilibs()

    if (NCBI_PTBMODE_COLLECT_DEPS)
        if (NCBI_PTBMODE_PARTS)
            set(NCBITMP_PROJECT_PART_IDS  ${NCBITMP_PROJECT_PART_IDS}  ${NCBI_PROJECT_ID}       PARENT_SCOPE )
            set(NCBITMP_PROJECT_PARTS     ${NCBITMP_PROJECT_PARTS}     ${NCBI_PROJECT_PARTNAME} PARENT_SCOPE )
        endif()
        set_property(GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${NCBI_PROJECT} "${NCBITMP_NCBILIB}")
        set_property(GLOBAL PROPERTY NCBI_PTBPROP_DIRECT_DEPS_${NCBI_PROJECT} "${NCBITMP_NCBILIB}")
        set_property(GLOBAL PROPERTY NCBI_PTBPROP_DIR_${NCBI_PROJECT} "${NCBI_CURRENT_SOURCE_DIR}")
        if (DEFINED NCBI_${NCBI_PROJECT}_DATASPEC)
            set_property(GLOBAL PROPERTY NCBI_PTBPROP_DATASPEC_${NCBI_PROJECT} YES)
        endif()

        if(NOT NCBI_PTBMODE_PARTS)
            get_property(_allprojects GLOBAL PROPERTY NCBI_PTBPROP_ALL_PROJECTS)
            set(_allprojects ${_allprojects} ${NCBI_PROJECT})
            set_property(GLOBAL PROPERTY NCBI_PTBPROP_ALL_PROJECTS ${_allprojects})

            NCBI_internal_process_project_filters(_allowed)
            if (_allowed)
                get_property(_allowedprojects GLOBAL PROPERTY NCBI_PTBPROP_ALLOWED_PROJECTS)
                set(_allowedprojects ${_allowedprojects} ${NCBI_PROJECT})
                set_property(GLOBAL PROPERTY NCBI_PTBPROP_ALLOWED_PROJECTS ${_allowedprojects})
            endif()
        endif()
        if ("${ARGC}" GREATER "0")
            set(${ARGV0} FALSE PARENT_SCOPE)
        endif()
        return()
    elseif (${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "CONSOLEAPP")
        NCBI_internal_process_project_filters(_allowed)
        if (NOT _allowed)
            if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
                message("${NCBI_PROJECT} (${NCBI_CURRENT_SOURCE_DIR}) is excluded by user's request")
            endif()
            return()
        endif()
    endif()

#message("processing ${NCBI_PROJECT_ID}")
    NCBI_internal_collect_sources() 
    NCBI_internal_collect_dataspec()
    NCBI_internal_collect_headers() 
    if (NCBI_PTBMODE_PARTS)
        set(NCBITMP_PROJECT_REQUIRES    ${NCBITMP_PROJECT_REQUIRES}   ${NCBI__REQUIRES}   ${NCBI_${NCBI_PROJECT}_REQUIRES}    PARENT_SCOPE)
        set(NCBITMP_PROJECT_COMPONENTS  ${NCBITMP_PROJECT_COMPONENTS} ${NCBI__COMPONENTS} ${NCBI_${NCBI_PROJECT}_COMPONENTS}  PARENT_SCOPE)
        set(NCBITMP_PROJECT_INCLUDES    ${NCBITMP_PROJECT_INCLUDES}   ${NCBI__INCLUDES}   ${NCBI_${NCBI_PROJECT}_INCLUDES}    PARENT_SCOPE)
        set(NCBITMP_PROJECT_DEFINES     ${NCBITMP_PROJECT_DEFINES}    ${NCBI__DEFINES}    ${NCBI_${NCBI_PROJECT}_DEFINES}     PARENT_SCOPE)
        set(NCBITMP_PROJECT_NCBILIB     ${NCBITMP_PROJECT_NCBILIB}    ${NCBI__NCBILIB}    ${NCBI_${NCBI_PROJECT}_NCBILIB}     PARENT_SCOPE)
        set(NCBITMP_PROJECT_EXTLIB      ${NCBITMP_PROJECT_EXTLIB}     ${NCBI__EXTLIB}     ${NCBI_${NCBI_PROJECT}_EXTLIB}      PARENT_SCOPE)
        set(NCBITMP_PROJECT_SOURCES     ${NCBITMP_PROJECT_SOURCES}    PARENT_SCOPE)
        set(NCBITMP_PROJECT_NOPCH       ${NCBITMP_PROJECT_NOPCH}      PARENT_SCOPE)
        set(NCBITMP_PROJECT_DATASPEC    ${NCBITMP_PROJECT_DATASPEC}   PARENT_SCOPE)
        set(NCBITMP_PROJECT_HEADERS     ${NCBITMP_PROJECT_HEADERS}    PARENT_SCOPE)
        if ("${ARGC}" GREATER "0")
            set(${ARGV0} FALSE PARENT_SCOPE)
        endif()
        return()
    endif()
    NCBI_internal_add_resources()
    NCBI_internal_add_dataspec()
    NCBI_internal_verify_libs()

#----------------------------------------------------------------------------
    if (${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "STATIC")

        if (WIN32)
            set(NCBITMP_DEFINES  ${NCBITMP_DEFINES} "_LIB")
        endif()
#message("add static library(${NCBI_PROJECT} STATIC ${NCBITMP_PROJECT_SOURCES} ${NCBITMP_PROJECT_HEADERS} ${NCBITMP_PROJECT_RESOURCES} ${NCBITMP_PROJECT_DATASPEC})")
#message("add static library ${NCBI_PROJECT}")
        add_library(${NCBI_PROJECT} STATIC ${NCBITMP_PROJECT_SOURCES} ${NCBITMP_PROJECT_HEADERS} ${NCBITMP_PROJECT_RESOURCES} ${NCBITMP_PROJECT_DATASPEC})
        set(_suffix ${CMAKE_STATIC_LIBRARY_SUFFIX})
        if (NOT WIN32 AND BUILD_SHARED_LIBS )
            target_compile_options(${NCBI_PROJECT} PRIVATE -fPIC)
        endif()

#----------------------------------------------------------------------------
    elseif (${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "SHARED")

        if (WIN32)
            set(NCBITMP_DEFINES  ${NCBITMP_DEFINES} "_USRDLL")
        endif()
#message("add shared library(${NCBI_PROJECT} SHARED ${NCBITMP_PROJECT_SOURCES} ${NCBITMP_PROJECT_HEADERS} ${NCBITMP_PROJECT_RESOURCES} ${NCBITMP_PROJECT_DATASPEC})")
#message("add shared library ${NCBI_PROJECT}")
        add_library(${NCBI_PROJECT} SHARED ${NCBITMP_PROJECT_SOURCES} ${NCBITMP_PROJECT_HEADERS} ${NCBITMP_PROJECT_RESOURCES} ${NCBITMP_PROJECT_DATASPEC})
        set(_suffix ${CMAKE_SHARED_LIBRARY_SUFFIX})

#----------------------------------------------------------------------------
    elseif (${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "CONSOLEAPP")

        if(NCBI_EXPERIMENTAL_CFG)
            if (WIN32)
                set(NCBITMP_DEFINES  ${NCBITMP_DEFINES} "_CONSOLE")
            endif()
#message("add_executable(${NCBI_PROJECT} ${NCBITMP_PROJECT_SOURCES} ${NCBITMP_PROJECT_HEADERS} ${NCBITMP_PROJECT_RESOURCES} ${NCBITMP_PROJECT_DATASPEC})")
#message("add executable ${NCBI_PROJECT}")
            add_executable(${NCBI_PROJECT} ${NCBITMP_PROJECT_SOURCES} ${NCBITMP_PROJECT_HEADERS} ${NCBITMP_PROJECT_RESOURCES} ${NCBITMP_PROJECT_DATASPEC})
            set(_suffix ${CMAKE_EXECUTABLE_SUFFIX})
        else()
#message("add_executable(${NCBI_PROJECT} ${NCBITMP_PROJECT_SOURCES})")
            add_executable(${NCBI_PROJECT} ${NCBITMP_PROJECT_SOURCES})
        endif()

#----------------------------------------------------------------------------
    elseif (${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "CUSTOM")

        variable_watch(NCBI_${NCBI_PROJECT}_CALLBACK ${NCBI_${NCBI_PROJECT}_DEFINITION})
        set(NCBI_${NCBI_PROJECT}_CALLBACK 1)

#----------------------------------------------------------------------------
    else()

        message("${NCBI_PROJECT} (${NCBI_CURRENT_SOURCE_DIR}) unsupported project type ${NCBI_${NCBI_PROJECT}_TYPE}")
        if ("${ARGC}" GREATER "0")
            set(${ARGV0} FALSE PARENT_SCOPE)
        endif()
        return()

    endif()
    if (DEFINED NCBI_${NCBI_PROJECT}_OUTPUT)
        if(NCBI_EXPERIMENTAL_CFG AND NOT DEFINED NCBI_EXTERNAL_TREE_ROOT AND NOT ${NCBI_PROJECT} STREQUAL ${NCBI_${NCBI_PROJECT}_OUTPUT})
            get_property(_dir GLOBAL PROPERTY NCBI_PTBPROP_DIR_${NCBI_${NCBI_PROJECT}_OUTPUT})
            if (NOT ${_dir} STREQUAL "" AND NOT ${_dir} STREQUAL ${NCBI_CURRENT_SOURCE_DIR})
                message("WARNING: App target ${NCBI_${NCBI_PROJECT}_OUTPUT} (${NCBI_CURRENT_SOURCE_DIR}) cannot be created")
                message("         because there is already a target with the same name in ${_dir}")
                message("         App target ${NCBI_${NCBI_PROJECT}_OUTPUT} will be renamed into ${NCBI_PROJECT}")
            endif()
        endif()
        set_target_properties(${NCBI_PROJECT} PROPERTIES OUTPUT_NAME ${NCBI_${NCBI_PROJECT}_OUTPUT})
    endif()

    get_property(_count  GLOBAL PROPERTY NCBI_PTBPROP_COUNT_${NCBI_${NCBI_PROJECT}_TYPE})
    math(EXPR _count "${_count} + 1")
    set_property(GLOBAL PROPERTY NCBI_PTBPROP_COUNT_${NCBI_${NCBI_PROJECT}_TYPE} ${_count})

if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
message("  ADDED: ${NCBI_CURRENT_SOURCE_DIR}/${NCBI_PROJECT_ID}")
message("  NCBITMP_PROJECT_SOURCES ${NCBITMP_PROJECT_SOURCES}")
message("  NCBITMP_PROJECT_HEADERS ${NCBITMP_PROJECT_HEADERS}")
message("  NCBITMP_PROJECT_RESOURCES ${NCBITMP_PROJECT_RESOURCES}")
endif()

    if (${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "CUSTOM")

        get_property(_prjdeps GLOBAL PROPERTY NCBI_PTBPROP_DIRECT_DEPS_${NCBI_PROJECT})
        if (NOT "${_prjdeps}" STREQUAL "")
            add_dependencies(${NCBI_PROJECT} ${_prjdeps})
        endif()

    else()
        target_include_directories(${NCBI_PROJECT} PRIVATE ${NCBITMP_INCLUDES})
        target_compile_definitions(${NCBI_PROJECT} PRIVATE ${NCBITMP_DEFINES})

if(OFF)
# this does not seem to have any effect
        if (XCODE AND NOT BUILD_SHARED_LIBS)
            set_target_properties(${NCBI_PROJECT} PROPERTIES XCODE_ATTRIBUTE_STANDARD_C_PLUS_PLUS_LIBRARY_TYPE "static")
            set_target_properties(${NCBI_PROJECT} PROPERTIES XCODE_ATTRIBUTE_GCC_SYMBOLS_PRIVATE_EXTERN "YES")
        endif()
endif()

#message("target_link_libraries: ${NCBI_PROJECT}     ${NCBITMP_NCBILIB} ${NCBITMP_EXTLIB}")
        target_link_libraries(     ${NCBI_PROJECT}         ${NCBITMP_NCBILIB} ${NCBITMP_EXTLIB})

        if (DEFINED _suffix)
            set_target_properties( ${NCBI_PROJECT} PROPERTIES PROJECT_LABEL ${NCBI_PROJECT}${_suffix})
        endif()
        NCBI_internal_define_precompiled_header_usage()
    endif()

    if (DEFINED NCBI_ALLTESTS)
        foreach(_test IN LISTS NCBI_ALLTESTS)
            NCBI_internal_add_test(${_test})
        endforeach()
        unset(NCBI_ALLTESTS PARENT_SCOPE)
    endif()

    if (NCBI_PTBCFG_DOINSTALL)
        NCBI_internal_install_target()
    endif()

if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
message("-----------------------------------")
endif()

    if ("${ARGC}" GREATER "0")
        set(${ARGV0} TRUE PARENT_SCOPE)
    endif()
endfunction()
