/*
 *  Copyright (c) 2008-2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the license.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ParametersWidgetBase_p.h"

#include <GTLCore/Value.h>
#include <GTLCore/Metadata/Entry.h>
#include <GTLCore/Metadata/Group.h>
#include <GTLCore/Metadata/ParameterEntry.h>

#include <QDebug>
#include <QCheckBox>
#include <QDoubleSpinBox>
#include <QGridLayout>
#include <QLabel>
#include <QTabWidget>
#include <QSlider>

#include "ColorButton_p.h"
#include "Proxy_p.h"
#include "SpinBoxSliderConnector_p.h"
#include "VectorEditor_p.h"
#include <GTLCore/Type.h>
#include "Utils_p.h"

using namespace QtShiva;

void ParametersWidgetBase::Private::regenerateWidget()
{
  delete currentWidget;
  currentWidget = 0;
  proxies.clear();
  if( parametrisation->parameters() )
  {
    const GTLCore::Metadata::Group* parameters = parametrisation->parameters();
    // Is tabWidget ?
    QTabWidget* tabWidget = 0;
    QWidget* widget = new QWidget();
    foreach( const GTLCore::Metadata::Entry* entry, parameters->entries() )
    {
      if( entry->asGroup() and not entry->asParameterEntry() )
      {
        if(not tabWidget )
        {
          tabWidget = new QTabWidget( self );
          widget = new QWidget( );
          tabWidget->addTab( widget, parametrisation->name().c_str() );
        }
        break;
      }
    }
    // Set the currentWidget
    if( tabWidget )
    {
      currentWidget = tabWidget;
    } else {
      currentWidget = widget;
    }
    
    // Create widgets
    QGridLayout* widgetGridLayout = new QGridLayout(widget);
    int posInGrid = 0;
    foreach( const GTLCore::Metadata::Entry* entry, parameters->entries() )
    {
      if( const GTLCore::Metadata::ParameterEntry* pe = entry->asParameterEntry() )
      {
        createParameterEntryWidget( widget, widgetGridLayout, posInGrid, pe );
        ++posInGrid;
      } else if( const GTLCore::Metadata::Group* gr = entry->asGroup() )
      {
        Q_ASSERT( tabWidget );
        QWidget* groupWidget = new QWidget( );
        QGridLayout* groupWidgetGridLayout = new QGridLayout(groupWidget);
        int groupPosInGrid = 0;
        foreach( const GTLCore::Metadata::Entry* groupEntry, gr->entries() )
        {
          if( const GTLCore::Metadata::ParameterEntry* gpe = groupEntry->asParameterEntry() )
          {
            createParameterEntryWidget( groupWidget, groupWidgetGridLayout, groupPosInGrid, gpe );
            ++groupPosInGrid;
          } else {
            qDebug() << "Non parameter entry: " << groupEntry->name().c_str() ;
          }
        }
        groupWidgetGridLayout->addItem(new QSpacerItem(0,0,QSizePolicy::Minimum, QSizePolicy::Expanding), groupPosInGrid, 0);
        QString caption = ( ( gr->label() == "" ) ? gr->name() : gr->label() ).c_str();
        tabWidget->addTab( groupWidget, caption );
      }
    }
    widgetGridLayout->addItem(new QSpacerItem(0,0,QSizePolicy::Minimum, QSizePolicy::Expanding), posInGrid, 0);
  } else {
    currentWidget = new QLabel( QObject::tr( "No configuration" ), self );
  }
  
  currentWidget->setVisible( true );
  selfLayout->addWidget(currentWidget, 0, 0, 1, 1);
  
}

void ParametersWidgetBase::Private::createParameterEntryWidget( QWidget* _parent, QGridLayout* _gridLayout, int _layoutIndex, const GTLCore::Metadata::ParameterEntry* _parameterEntry )
{
  // Create the Label
  QString caption = ( ( _parameterEntry->label() == "" ) ? _parameterEntry->name() : _parameterEntry->label() ).c_str();
  if( _parameterEntry->type() != GTLCore::Type::Boolean)
  { // Checkbox widgets integrate their own labels
    QLabel* label = new QLabel( caption + ":", _parent);
    _gridLayout->addWidget(label, _layoutIndex, 0, 1, 1);
  }
  Proxy* proxy = new Proxy( _parent, _parameterEntry->name(), parametrisation, _parameterEntry->type() );
  switch( _parameterEntry->type()->dataType() )
  {
    case GTLCore::Type::BOOLEAN:
    {
      // Checkbox
      QCheckBox* checkBox = new QCheckBox(caption, _parent);
      _gridLayout->addWidget(checkBox, _layoutIndex, 0, 1, 2);
      connect( checkBox, SIGNAL(toggled(bool)), proxy, SLOT(setBoolValue(bool)));
      connect( proxy, SIGNAL(boolValueChanged(bool)), checkBox, SLOT(setChecked(bool)));
      break;
    }
    case GTLCore::Type::INTEGER32:
    {
      // Integer spinbox
      QSpinBox* spinBox = new QSpinBox(_parent);
      _gridLayout->addWidget(spinBox, _layoutIndex, 1, 1, 1);
      
      // Slider
      QSlider* horizontalSlider = new QSlider(_parent);
      horizontalSlider->setOrientation(Qt::Horizontal);
      _gridLayout->addWidget(horizontalSlider, _layoutIndex, 2, 1, 1);
      
      connect( spinBox, SIGNAL(valueChanged(int)), horizontalSlider, SLOT(setValue(int)));
      connect( horizontalSlider, SIGNAL(valueChanged(int)), spinBox, SLOT(setValue(int)));
      
      connect( spinBox, SIGNAL(valueChanged(int)), proxy, SLOT(setIntValue(int)));
      connect( proxy, SIGNAL(intValueChanged(int)), spinBox, SLOT(setValue(int)));
      
      int min = _parameterEntry->minimumValue().asInt32();
      int max = _parameterEntry->maximumValue().asInt32();
      
      spinBox->setMinimum( min );
      spinBox->setMaximum( max );
      horizontalSlider->setMinimum( min );
      horizontalSlider->setMaximum( max );
    }
      break;
    case GTLCore::Type::FLOAT32:
    {
      
      // Double spinbox
      QDoubleSpinBox* doubleSpinBox = new QDoubleSpinBox(_parent);
      _gridLayout->addWidget(doubleSpinBox, _layoutIndex, 1, 1, 1);
      doubleSpinBox->setSingleStep( 0.1 );
      
      // Slider
      QSlider* horizontalSlider = new QSlider(_parent);
      horizontalSlider->setOrientation(Qt::Horizontal);
      _gridLayout->addWidget(horizontalSlider, _layoutIndex, 2, 1, 1);
      horizontalSlider->setMinimum( 0 );
      horizontalSlider->setMaximum( 1000 );
      
      SpinBoxSliderConnector* connector = new SpinBoxSliderConnector( _parent, doubleSpinBox, horizontalSlider);
      connect( connector, SIGNAL(valueChanged(double)), proxy, SLOT(setDoubleValue(double)));
      connect( proxy, SIGNAL(doubleValueChanged(double)), doubleSpinBox, SLOT(setValue(double)));
      
      // Set parameters
      doubleSpinBox->setMinimum( _parameterEntry->minimumValue().asFloat32() );
      doubleSpinBox->setMaximum( _parameterEntry->maximumValue().asFloat32() );
    }
      break;
    case GTLCore::Type::VECTOR:
    {
      switch(_parameterEntry->type()->embeddedType()->dataType())
      {
        case GTLCore::Type::FLOAT32:
        {
          VectorEditor* vectorEditor = new VectorEditor(_parent, _parameterEntry->type()->vectorSize());
          _gridLayout->addWidget(vectorEditor, _layoutIndex, 1, 1, 2);
          
          connect( vectorEditor, SIGNAL(valuesChanged(QList<double>)), proxy, SLOT(setVector(QList<double>)));
          connect( proxy, SIGNAL(vectorValueChanged(QList<double>)), vectorEditor, SLOT(setValues(QList<double>)));
          
          vectorEditor->setMaximum( valueToList( _parameterEntry->maximumValue() ) );
          vectorEditor->setMinimum( valueToList( _parameterEntry->minimumValue() ) );
          break;
        }
      }
    }
      break;
    case GTLCore::Type::STRUCTURE:
    {
      if(_parameterEntry->type() == GTLCore::Type::Color)
      {
        ColorButton* colorButton = new ColorButton( _parent, true );
        _gridLayout->addWidget(colorButton, _layoutIndex, 1, 1, 2);
        connect( colorButton, SIGNAL(colorChanged(const QColor&)), proxy, SLOT(setRgba(const QColor&)));
        connect( proxy, SIGNAL(rgbaValueChanged(QColor)), colorButton, SLOT(setCurrentColor(QColor)));
      }
    }
      break;
  }
  connect(proxy, SIGNAL(valueChanged()), self, SIGNAL(configurationChanged()));
  proxy->setValue(_parameterEntry->defaultValue());
  proxies[ _parameterEntry->name() ] = proxy;
}

#include "ParametersWidgetBase_p.moc"
