#include "test_helpers.hxx"

using namespace PGSTD;
using namespace pqxx;


namespace
{
void TestPipeline(pipeline &P, int numqueries)
{
  const string Q("SELECT count(*) FROM pg_tables");
  const result Empty;
  PQXX_CHECK(Empty.empty(), "Default-constructed result is not empty.");
  PQXX_CHECK(Empty.query().empty(), "Default-constructed result has query");

  P.retain();
  for (int i=numqueries; i; --i) P.insert(Q);
  P.resume();

  PQXX_CHECK(!numqueries || !P.empty(), "pipeline::empty() is broken.");

  int res = 0;
  result Prev;
  PQXX_CHECK_EQUAL(Prev, Empty, "Default-constructed results are not equal.");

  for (int i=numqueries; i; --i)
  {
    PQXX_CHECK(!P.empty(), "Got no results from pipeline.");

    pair<pipeline::query_id, result> R = P.retrieve();

    PQXX_CHECK_NOT_EQUAL(R.second, Empty, "Got empty result.");
    if (Prev != Empty)
      PQXX_CHECK_EQUAL(R.second, Prev, "Results to same query are different.");

    Prev = R.second;
    PQXX_CHECK_EQUAL(Prev, R.second, "Assignment breaks result equality.");
    PQXX_CHECK_EQUAL(R.second.query(), Q, "Result is for unexpected query.");

    if (res)
      PQXX_CHECK_EQUAL(Prev[0][0].as<int>(), res, "Bad result from pipeline.");

    res = Prev[0][0].as<int>();
  }

  PQXX_CHECK(P.empty(), "Pipeline was not empty after retrieval.");
}


// Test program for libpqxx.  Issue a query repeatedly through a pipeline, and
// compare results.  Use retain() and resume() for performance.
void test_070(transaction_base &W)
{
  pipeline P(W);

  PQXX_CHECK(P.empty(), "Pipeline is not empty initially.");

  // Try to confuse the pipeline by feeding it a query and flushing
  P.retain();
  const string Q = "SELECT * FROM pg_tables";
  P.insert(Q);
  P.flush();

  PQXX_CHECK(P.empty(), "Pipeline was not empty after flush().");

  // See if complete() breaks retain() as it should
  P.retain();
  P.insert(Q);
  PQXX_CHECK(!P.empty(), "Pipeline was empty after insert().");
  P.complete();
  PQXX_CHECK(!P.empty(), "complete() emptied pipeline.");

  PQXX_CHECK_EQUAL(
	P.retrieve().second.query(),
	Q,
	"Result is for wrong query.");

  PQXX_CHECK(P.empty(), "Pipeline not empty after retrieve().");

  // See if retrieve() breaks retain() when it needs to
  P.retain();
  P.insert(Q);
  PQXX_CHECK_EQUAL(
	P.retrieve().second.query(),
	Q,
	"Got result for wrong query.");

  // See if regular retain()/resume() works
  for (int i=0; i<5; ++i) TestPipeline(P, i);

  // See if retrieve() fails on an empty pipeline, as it should
  quiet_errorhandler d(W.conn());
  PQXX_CHECK_THROWS(
	P.retrieve(),
	exception,
	"Empty pipeline allows retrieve().");
}
} // namespace

PQXX_REGISTER_TEST_C(test_070, asyncconnection)
