/*
    libmaus2
    Copyright (C) 2009-2013 German Tischler
    Copyright (C) 2011-2013 Genome Research Limited

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#if ! defined(BUFFEREDOUTPUT_HPP)
#define BUFFEREDOUTPUT_HPP

#include <libmaus2/autoarray/AutoArray.hpp>
#include <libmaus2/sorting/InPlaceParallelSort.hpp>

namespace libmaus2
{
	namespace aio
	{
		/**
		 * base class for block buffer output
		 **/
		template<typename _data_type>
		struct BufferedOutputBase
		{
			//! data type
			typedef _data_type data_type;
			//! this type
			typedef BufferedOutputBase<data_type> this_type;
			//! unique pointer type
			typedef typename ::libmaus2::util::unique_ptr<this_type>::type unique_ptr_type;

			//! block buffer
			::libmaus2::autoarray::AutoArray<data_type> B;
			//! start of buffer pointer
			data_type * const pa;
			//! current pointer
			data_type * pc;
			//! end of buffer pointer
			data_type * const pe;

			//! total bytes written
			uint64_t totalwrittenbytes;
			//! total words written
			uint64_t totalwrittenwords;

			/**
			 * construct buffer of size bufsize
			 *
			 * @param bufsize buffer size (in elements)
			 **/
			BufferedOutputBase(uint64_t const bufsize)
			: B(bufsize,false), pa(B.begin()), pc(pa), pe(B.end()), totalwrittenbytes(0), totalwrittenwords(0)
			{

			}
			/**
			 * destructor
			 **/
			virtual ~BufferedOutputBase() {}

			/**
			 * flush buffer
			 **/
			void flush()
			{
				writeBuffer();
			}

			/**
			 * pure virtual buffer writing function
			 **/
			virtual void writeBuffer() = 0;

			/**
			 * put one element
			 *
			 * @param v element to be put in buffer
			 * @return true if buffer is full after putting the element
			 **/
			bool put(data_type const & v)
			{
				*(pc++) = v;
				if ( pc == pe )
				{
					flush();
					return true;
				}
				else
				{
					return false;
				}
			}

			/**
			 * put a list of elements
			 *
			 * @param A list start pointer
			 * @param n number of elements to be put in buffer
			 **/
			void put(data_type const * A, uint64_t n)
			{
				while ( n )
				{
					uint64_t const towrite = std::min(n,static_cast<uint64_t>(pe-pc));
					std::copy(A,A+towrite,pc);
					pc += towrite;
					A += towrite;
					n -= towrite;
					if ( pc == pe )
						flush();
				}
			}

			/**
			 * put an element without checking whether buffer fills up
			 *
			 * @param v element to be put in buffer
			 **/
			void uncheckedPut(data_type const & v)
			{
				*(pc++) = v;
			}

			/**
			 * @return space available in buffer (in elements)
			 **/
			uint64_t spaceLeft() const
			{
				return pe-pc;
			}

			/**
			 * @return number of words written (including those still in the buffer)
			 **/
			uint64_t getWrittenWords() const
			{
				return totalwrittenwords + (pc-pa);
			}

			/**
			 * @return number of bytes written (including those still in the buffer)
			 **/
			uint64_t getWrittenBytes() const
			{
				return getWrittenWords() * sizeof(data_type);
			}

			/**
			 * increase written accus by number of elements curently in buffer
			 **/
			void increaseWritten()
			{
				totalwrittenwords += (pc-pa);
				totalwrittenbytes += (pc-pa)*sizeof(data_type);
			}

			/**
			 * @return number of elements currently in buffer
			 **/
			uint64_t fill()
			{
				return pc-pa;
			}

			/**
			 * reset buffer
			 **/
			void reset()
			{
				increaseWritten();
				pc = pa;
			}
		};

		/**
		 * class for buffer file output
		 **/
		template<typename _data_type>
		struct BufferedOutput : public BufferedOutputBase<_data_type>
		{
			//! data type
			typedef _data_type data_type;
			//! this type
			typedef BufferedOutput<data_type> this_type;
			//! base class type
			typedef BufferedOutputBase<data_type> base_type;
			//! unique pointer type
			typedef typename ::libmaus2::util::unique_ptr<this_type>::type unique_ptr_type;

			private:
			std::ostream & out;

			public:
			/**
			 * constructor
			 *
			 * @param rout output stream
			 * @param bufsize size of buffer in elements
			 **/
			BufferedOutput(std::ostream & rout, uint64_t const bufsize)
			: BufferedOutputBase<_data_type>(bufsize), out(rout)
			{

			}
			/**
			 * destructor
			 **/
			~BufferedOutput()
			{
				base_type::flush();
			}

			/**
			 * write buffer to output stream
			 **/
			virtual void writeBuffer()
			{
				if ( base_type::fill() )
				{
					out.write ( reinterpret_cast<char const *>(base_type::pa), (base_type::pc-base_type::pa)*sizeof(data_type) );

					if ( ! out )
					{
						::libmaus2::exception::LibMausException se;
						se.getStream() << "Failed to write " << (base_type::pc-base_type::pa)*sizeof(data_type) << " bytes." << std::endl;
						se.finish();
						throw se;
					}

					base_type::reset();
				}
			}
		};

		/**
		 * null write output buffer. data put in buffer is written nowhere.
		 * this is useful for code directly using the buffer base.
		 **/
		template<typename _data_type>
		struct BufferedOutputNull : public BufferedOutputBase<_data_type>
		{
			//! data type
			typedef _data_type data_type;
			//! this type
			typedef BufferedOutputNull<data_type> this_type;
			//! base class type
			typedef BufferedOutputBase<data_type> base_type;
			//! unique pointer type
			typedef typename ::libmaus2::util::unique_ptr<this_type>::type unique_ptr_type;

			/**
			 * constructor
			 *
			 * @param bufsize size of buffer
			 **/
			BufferedOutputNull(uint64_t const bufsize)
			: BufferedOutputBase<_data_type>(bufsize)
			{

			}
			/**
			 * destructor
			 **/
			~BufferedOutputNull()
			{
				base_type::flush();
			}
			/**
			 * null writeBuffer function (does nothing)
			 **/
			virtual void writeBuffer()
			{
			}
		};

		/**
		 * class for sorting buffer file output
		 **/
		template<typename _data_type, typename _order_type = std::less<_data_type> >
		struct SortingBufferedOutput : public BufferedOutputBase<_data_type>
		{
			//! data type
			typedef _data_type data_type;
			//! order type
			typedef _order_type order_type;
			//! this type
			typedef SortingBufferedOutput<data_type,order_type> this_type;
			//! base class type
			typedef BufferedOutputBase<data_type> base_type;
			//! unique pointer type
			typedef typename ::libmaus2::util::unique_ptr<this_type>::type unique_ptr_type;
			//! order pointer type
			typedef typename ::libmaus2::util::unique_ptr<order_type>::type order_ptr_type;

			private:
			std::ostream & out;
			order_ptr_type Porder;
			order_type & order;
			std::vector<uint64_t> blocksizes;

			public:
			/**
			 * constructor
			 *
			 * @param rout output stream
			 * @param bufsize size of buffer in elements
			 **/
			SortingBufferedOutput(std::ostream & rout, uint64_t const bufsize)
			: BufferedOutputBase<_data_type>(bufsize), out(rout), Porder(new order_type), order(*Porder), blocksizes()
			{

			}

			/**
			 * constructor
			 *
			 * @param rout output stream
			 * @param bufsize size of buffer in elements
			 * @param rorder sort order
			 **/
			SortingBufferedOutput(std::ostream & rout, uint64_t const bufsize, order_type & rorder)
			: BufferedOutputBase<_data_type>(bufsize), out(rout), Porder(), order(rorder), blocksizes()
			{

			}

			/**
			 * destructor
			 **/
			~SortingBufferedOutput()
			{
				flush();
			}

			/**
			 * flush the buffer
			 **/
			void flush()
			{
				base_type::flush();
				out.flush();
			}

			/**
			 * write buffer to output stream
			 **/
			virtual void writeBuffer()
			{
				if ( base_type::fill() )
				{
					std::sort(base_type::pa,base_type::pc,order);

					out.write ( reinterpret_cast<char const *>(base_type::pa), base_type::fill()*sizeof(data_type) );

					if ( ! out )
					{
						::libmaus2::exception::LibMausException se;
						se.getStream() << "Failed to write " << base_type::fill()*sizeof(data_type) << " bytes." << std::endl;
						se.finish();
						throw se;
					}

					blocksizes.push_back(base_type::fill());

					base_type::reset();
				}
			}

			/**
			 * get block sizes (number of elements written in each block)
			 *
			 * @return block size vector
			 **/
			std::vector<uint64_t> const & getBlockSizes() const
			{
				return blocksizes;
			}
		};

		/**
		 * class for sorting buffer file output
		 **/
		template<typename _data_type, typename _order_type = std::less<_data_type> >
		struct SerialisingSortingBufferedOutput : public BufferedOutputBase<_data_type>
		{
			//! data type
			typedef _data_type data_type;
			//! order type
			typedef _order_type order_type;
			//! this type
			typedef SerialisingSortingBufferedOutput<data_type,order_type> this_type;
			//! base class type
			typedef BufferedOutputBase<data_type> base_type;
			//! unique pointer type
			typedef typename ::libmaus2::util::unique_ptr<this_type>::type unique_ptr_type;
			//! order pointer type
			typedef typename ::libmaus2::util::unique_ptr<order_type>::type order_ptr_type;

			struct BlockDescriptor
			{
				uint64_t numel;
				uint64_t fileptr;

				BlockDescriptor() {}
				BlockDescriptor(uint64_t const rnumel, uint64_t const rfileptr) : numel(rnumel), fileptr(rfileptr) {}
			};

			private:
			std::ostream & out;
			order_ptr_type Porder;
			order_type & order;
			std::vector<BlockDescriptor> blocksizes;
			uint64_t const numthreads;

			public:
			/**
			 * constructor
			 *
			 * @param rout output stream
			 * @param bufsize size of buffer in elements
			 **/
			SerialisingSortingBufferedOutput(std::ostream & rout, uint64_t const bufsize, uint64_t const rnumthreads = 1)
			: BufferedOutputBase<_data_type>(bufsize), out(rout), Porder(new order_type), order(*Porder), blocksizes(), numthreads(rnumthreads)
			{

			}

			/**
			 * constructor
			 *
			 * @param rout output stream
			 * @param bufsize size of buffer in elements
			 * @param rorder sort order
			 **/
			SerialisingSortingBufferedOutput(std::ostream & rout, uint64_t const bufsize, order_type & rorder, uint64_t const rnumthreads = 1)
			: BufferedOutputBase<_data_type>(bufsize), out(rout), Porder(), order(rorder), blocksizes(), numthreads(rnumthreads)
			{

			}

			/**
			 * destructor
			 **/
			~SerialisingSortingBufferedOutput()
			{
				flush();
			}

			/**
			 * flush the buffer
			 **/
			void flush()
			{
				base_type::flush();
				out.flush();
			}

			/**
			 * write buffer to output stream
			 **/
			virtual void writeBuffer()
			{
				if ( base_type::fill() )
				{
					if ( numthreads <= 1 )
						std::sort(base_type::pa,base_type::pc,order);
					else
					{
						libmaus2::sorting::InPlaceParallelSort::inplacesort2<data_type *,order_type>(
							base_type::pa,base_type::pc,
							numthreads,
							order
						);
					}

					uint64_t const fileptr = out.tellp();

					for ( data_type const * pi = base_type::pa; pi != base_type::pc; ++pi )
					{
						pi->serialise(out);

						if ( ! out )
						{
							::libmaus2::exception::LibMausException se;
							se.getStream() << "SerialisingSortingBufferedOutput: Failed to write" << std::endl;
							se.finish();
							throw se;
						}
					}

					blocksizes.push_back(BlockDescriptor(base_type::fill(),fileptr));

					base_type::reset();
				}
			}

			/**
			 * get block sizes (number of elements written in each block)
			 *
			 * @return block size vector
			 **/
			std::vector<BlockDescriptor> const & getBlockSizes() const
			{
				return blocksizes;
			}
		};
	}
}
#endif
