/*
 * @(#)PostCompileClassUTest.java
 *
 * Copyright (C) 2002,2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.compiler;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import net.sourceforge.groboutils.autodoc.v1.AutoDoc;
import net.sourceforge.groboutils.codecoverage.v2.BytecodeLoaderUtil;
import net.sourceforge.groboutils.codecoverage.v2.CCCreatorUtil;
import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModule;
import net.sourceforge.groboutils.codecoverage.v2.datastore.ClassRecord;
import net.sourceforge.groboutils.codecoverage.v2.datastore.IClassMetaDataWriter;
import net.sourceforge.groboutils.codecoverage.v2.datastore.IMetaDataWriter;
import net.sourceforge.groboutils.codecoverage.v2.logger.TestLogger;


/**
 * Tests the PostCompileClass class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:28 $
 * @since     December 28, 2002
 */
public class PostCompileClassUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = PostCompileClassUTest.class;
    private static final AutoDoc DOC = new AutoDoc( THIS_CLASS );
    
    public PostCompileClassUTest( String name )
    {
        super( name );
    }


    //-------------------------------------------------------------------------
    // Tests
    
    public void testConstructor1()
    {
        try
        {
            new PostCompileClass( null, null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception?
        }
    }
    
    public void testConstructor2()
    {
        try
        {
            new PostCompileClass( createIMetaDataWriter(), null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception?
        }
    }
    
    public void testConstructor3()
    {
        try
        {
            new PostCompileClass( null, createAnalysisModules() );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception?
        }
    }
    
    public void testConstructor4()
    {
        try
        {
            new PostCompileClass( null, new IAnalysisModule[1] );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception?
        }
    }
    
    public void testConstructor5()
    {
        try
        {
            new PostCompileClass( createIMetaDataWriter(),
                new IAnalysisModule[1] );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception?
        }
    }
    
    public void testConstructor6()
    {
        new PostCompileClass( createIMetaDataWriter(),
            createAnalysisModules() );
    }
    
    //-------------
    
    public void testConstructor7()
    {
        try
        {
            new PostCompileClass( null, null, null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception?
        }
    }
    
    public void testConstructor8()
    {
        try
        {
            new PostCompileClass( null, createIMetaDataWriter(), null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception?
        }
    }
    
    public void testConstructor9()
    {
        try
        {
            new PostCompileClass( null, null, createAnalysisModules() );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception?
        }
    }
    
    public void testConstructor10()
    {
        try
        {
            new PostCompileClass( null, null, new IAnalysisModule[1] );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception?
        }
    }
    
    public void testConstructor11()
    {
        try
        {
            new PostCompileClass( null, createIMetaDataWriter(),
                new IAnalysisModule[1] );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception?
        }
    }
    
    public void testConstructor12()
    {
        try
        {
            new PostCompileClass( null, createIMetaDataWriter(),
                createAnalysisModules() );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception
        }
    }
    
    public void testConstructor13()
    {
        try
        {
            new PostCompileClass( createParseCoverageLogger(),
                createIMetaDataWriter(), null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception?
        }
    }
    
    public void testConstructor14()
    {
        try
        {
            new PostCompileClass( createParseCoverageLogger(),
                null, createAnalysisModules() );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception?
        }
    }
    
    public void testConstructor15()
    {
        try
        {
            new PostCompileClass( createParseCoverageLogger(),
                null, new IAnalysisModule[1] );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception?
        }
    }
    
    public void testConstructor16()
    {
        try
        {
            new PostCompileClass( createParseCoverageLogger(),
                createIMetaDataWriter(), new IAnalysisModule[1] );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception?
        }
    }
    
    public void testConstructor17()
    {
        new PostCompileClass( createParseCoverageLogger(),
            createIMetaDataWriter(), createAnalysisModules() );
    }
    
    //----------
    
    private static class Sample1 {}
    public void testPostCompile1()
            throws Exception
    {
        PostCompileClass pcc = new PostCompileClass(
            createParseCoverageLogger(), createIMetaDataWriter(),
            createAnalysisModules() );
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        String filename = getClassFilename( Sample1.class );
        byte[] bytecode = loadBytecode( filename );
        
        pcc.postCompile( filename, bytecode, baos );
        assertTrue(
            "Output bytecode is empty.",
            baos.toByteArray().length > 0 );
    }
    
    
    public void testPostCompile2()
            throws Exception
    {
        PostCompileClass pcc = new PostCompileClass(
            createIMetaDataWriter(), createAnalysisModules() );
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        String filename = getClassFilename( Sample1.class );
        byte[] bytecode = loadBytecode( filename );
        
        pcc.postCompile( filename, bytecode, baos );
        assertTrue(
            "Output bytecode is empty.",
            baos.toByteArray().length > 0 );
    }
    
    
    private static abstract class Sample3
    {
        public abstract void a();
    }
    public void testPostCompile3()
            throws Exception
    {
        PostCompileClass pcc = new PostCompileClass(
            createIMetaDataWriter(), createAnalysisModules() );
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        String filename = getClassFilename( Sample3.class );
        byte[] bytecode = loadBytecode( filename );
        
        pcc.postCompile( filename, bytecode, baos );
        assertTrue(
            "Output bytecode is empty.",
            baos.toByteArray().length > 0 );
    }
    
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    
    public static class MyClassMetaDataWriter implements IClassMetaDataWriter
    {
        private List records;
        public MyClassMetaDataWriter( List l )
        {
            this.records = l;
        }
        public void writeClassRecord( ClassRecord cr )
        {
            this.records.add( cr );
        }
        public void close()
        {
            this.records = null;
        }
    }
    
    
    public static class MyMetaDataWriter implements IMetaDataWriter
    {
        public Map modulesToClassRecordList = new HashMap(); 
        
        public IClassMetaDataWriter getClassWriter( IAnalysisModule module )
        {
            List list = (List)this.modulesToClassRecordList.get( module );
            if (list == null)
            {
                list = new LinkedList();
                this.modulesToClassRecordList.put( module, list );
            }
            return new MyClassMetaDataWriter( list );
        }
        public void close() {}
    }
    
    //------------------------------------------------------------------------

    protected ParseCoverageLogger createParseCoverageLogger()
    {
        return TestLogger.createPCL();
    }

    protected MyMetaDataWriter createIMetaDataWriter()
    {
        return new MyMetaDataWriter();
    }
    

    protected IAnalysisModule[] createAnalysisModules()
    {
        return new IAnalysisModule[] {
            CCCreatorUtil.createIAnalysisModule( 2 ),
            CCCreatorUtil.createIAnalysisModule( 3 ),
            CCCreatorUtil.createIAnalysisModule( 4 ),
        };
    }

    protected static String getClassFilename( Class c )
    {
        return BytecodeLoaderUtil.getClassFilename( c );
    }
    
    protected static byte[] loadBytecode( String filename ) throws IOException
    {
        return BytecodeLoaderUtil.loadBytecode( filename );
    }

    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

