/*
 * @(#)StdoutChannelLogger.java
 *
 * Copyright (C) 2004 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.logger;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

import net.sourceforge.groboutils.codecoverage.v2.IChannelLogger;


/**
 * Logs coverage reports to a directory of logs.  The directories are split
 * by the channel index, and the directory contains one log per class file
 * analyzed.  The contents are in CSV format:
 * <PRE>
 *    <i>method index</i> , <i>mark index</i> EOL
 * </PRE>
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/05/11 23:22:33 $
 * @since     May 11, 2004
 */
public class StdoutChannelLogger implements IChannelLogger
{
    private short channelIndex;
    public StdoutChannelLogger( short index )
    {
        this.channelIndex = index;
    }
    
    
    /**
     * Records a coverage of a marked bytecode instruction.  This method should
     * never throw an exception.
     *
     * @param classSignature a signature of the class file being covered.
     *        this signature includes the fully-qualified name of the class,
     *        along with a checksum to uniquely identify it.
     * @param methodIndex index for a method within the class.  The meta-data
     *        store will know how to translate the index to a method signature.
     * @param markIndex the index of the bytecode instruction mark for this
     *        particular channel.
     */
    public void cover( String classSignature, short methodIndex,
            short markIndex )
    {
        String text = createCoverString( classSignature, methodIndex,
            markIndex );
        System.err.println(text);
    }
    
    
    /**
     * Make static final so that the invocation time is minimized.
     */
    protected final String createCoverString( String classSig,
            short methodIndex, short markIndex )
    {
        // make the string too big, so we don't need to resize it during
        // execution.
        StringBuffer sb = new StringBuffer( "[Cover String: " );
        sb.append( this.channelIndex ).append( ',' ).
            append( classSig ).append( ',' ).
            append( methodIndex ).append( ',' ).
            append( markIndex ).append( ']' );
        
        String out = sb.toString();
        return out;
    }
}

