/*
 * Linux IEEE 802.15.4 userspace tools
 *
 * Copyright (C) 2008, 2009 Siemens AG
 *
 * Written-by: Dmitry Eremin-Solenikov
 * Written-by: Sergey Lapin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
#include <netlink/netlink.h>
#include <netlink/genl/genl.h>
#include <netlink/genl/ctrl.h>
#include <errno.h>

#include <ieee802154.h>
#include <nl802154.h>

static int parse_cb(struct nl_msg *msg, void *arg)
{
	struct nlmsghdr *nlh = nlmsg_hdr(msg);
	struct nlattr *attrs[IEEE802154_ATTR_MAX+1];
        struct genlmsghdr *ghdr;


	// Validate message and parse attributes
	genlmsg_parse(nlh, 0, attrs, IEEE802154_ATTR_MAX, ieee802154_policy);

        ghdr = nlmsg_data(nlh);

	printf("Received command %d (%d)\n", ghdr->cmd, ghdr->version);
	if (!attrs[IEEE802154_ATTR_DEV_NAME] || !attrs[IEEE802154_ATTR_HW_ADDR])
		return -EINVAL;

	uint64_t addr = nla_get_u64(attrs[IEEE802154_ATTR_HW_ADDR]);
	uint8_t buf[8];
	memcpy(buf, &addr, 8);

	printf("Addr for %s is %02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x\n",
			nla_get_string(attrs[IEEE802154_ATTR_DEV_NAME]),
			buf[0], buf[1],	buf[2], buf[3],
			buf[4], buf[5],	buf[6], buf[7]);

	return 0;
}


int main(void) {

	struct nl_sock *nl = nl_socket_alloc();
	int err;

	if (!nl) {
		nl_perror(NLE_NOMEM, "nl_sock_alloc");
		return 1;
	}

	err = genl_connect(nl);
	nl_perror(err, "genl_connect");

	int family = genl_ctrl_resolve(nl, "802.15.4 MAC");
	nl_perror(family, "genl_ctrl_resolve");

	struct nl_msg *msg = nlmsg_alloc();
	if (!msg) {
		nl_perror(NLE_NOMEM, "nlmsg_alloc");
		return 1;
	}
	genlmsg_put(msg, NL_AUTO_PID, NL_AUTO_SEQ, family, 0, NLM_F_ECHO, /* cmd */ IEEE802154_LIST_IFACE, /* vers */ 1);
	nla_put_string(msg, IEEE802154_ATTR_DEV_NAME, "wpan0");

	err = nl_send_auto_complete(nl, msg);
	nl_perror(err, "nl_send_auto_complete");

	nlmsg_free(msg);

	nl_socket_modify_cb(nl, NL_CB_VALID, NL_CB_CUSTOM, parse_cb, NULL);

	// Wait for the answer and receive it
	nl_recvmsgs_default(nl);


	nl_close(nl);

	return 0;
}
