#ifndef CHILD_PROCESS_H
#define CHILD_PROCESS_H

#include <sys/types.h>
#include <Exception.h>

class Process
{
protected:
	void detachFromTTY() throw (SystemException);

public:
	// TODO: since the destructor is called twice (one in the parent and one in
	// the child), it could be useful to add a bool isChild() method to let the
	// destructor and other functions know where they are operating.  The value
	// returned can be set by ChildProcess::fork.
	virtual int main() throw () = 0;

	/// Pretty-print the return value of a process into a string
	static std::string formatStatus(int status) throw ();
	
	/// Change working directory
	static void chdir(const std::string& dir) throw (SystemException);

	/// Change root directory
	static void chroot(const std::string& dir) throw (SystemException);

	/// Change umask (always succeeds and returns the previous umask)
	static mode_t umask(mode_t mask) throw ();

	/// Set user and group permissions
	static void setPerms(const std::string& user)
		throw (ConsistencyCheckException, SystemException);
	static void setPerms(const std::string& user, const std::string& group)
		throw (ConsistencyCheckException, SystemException);
	static void setPerms(uid_t user)
		throw (ConsistencyCheckException, SystemException);
	static void setPerms(uid_t user, gid_t group)
		throw (ConsistencyCheckException, SystemException);

	/// Get current resource limits; store also maximum resource limits in max
	/// if nonzero
	static int getCPUTimeLimit(int* max = 0) throw (SystemException);
	static int getFileSizeLimit(int* max = 0) throw (SystemException);
	static int getDataMemoryLimit(int* max = 0) throw (SystemException);
	static int getChildrenLimit(int* max = 0) throw (SystemException);
	static int getOpenFilesLimit(int* max = 0) throw (SystemException);
	static int getCoreSizeLimit(int* max = 0) throw (SystemException);

	/// Set resource limits
	static void setCPUTimeLimit(int value) throw (SystemException);
	static void setFileSizeLimit(int value) throw (SystemException);
	static void setDataMemoryLimit(int value) throw (SystemException);
	static void setChildrenLimit(int value) throw (SystemException);
	static void setOpenFilesLimit(int value) throw (SystemException);
	static void setCoreSizeLimit(int value) throw (SystemException);
};

struct rusage;

class ChildProcess
{
protected:
	Process* _proc;
	pid_t _pid;
	
public:
	ChildProcess(Process* proc) throw () : _proc(proc), _pid(-1) {}
	~ChildProcess() throw () {}

	/// Fork a subprocess to run proc.  If one of the std*fd variables is
	/// non-null, create a pipe connected to the corresponding file descriptor
	/// of the child process and store the parent end in the std*fd variable.
	/// After this function exits successfully, `proc' can safely be deleted.
	pid_t fork(int* stdinfd = 0, int* stdoutfd = 0, int* stderrfd = 0) throw (SystemException);
	
	/// Get the pid of the child process or (pid_t)-1 if no child is running
	pid_t pid() const throw () { return _pid; }

	/// Wait for the child to finish, returing its exit status.  Return -1 if
	/// no child is running.
	/// TODO: gracefully handle the EINTR error code
	int wait() throw (SystemException, InterruptedException);

	/// Wait for the child to finish, returing its exit status and storing
	/// resource usage informations in `ru'.  Return -1 if no child is running.
	/// TODO: gracefully handle the EINTR error code
	int wait(struct rusage* ru) throw (SystemException, InterruptedException);
};

// vim:set ts=4 sw=4:
#endif
