package latexDraw.parsers.svg.elements;

import java.io.File;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.ParseException;

import latexDraw.parsers.svg.MalformedSVGDocument;
import latexDraw.parsers.svg.SVGAttributes;
import latexDraw.parsers.svg.SVGDocument;
import latexDraw.parsers.svg.SVGElements;
import latexDraw.parsers.svg.parsers.SVGLengthParser;

import org.w3c.dom.Node;

public class SVGImageElement extends SVGElement
{

	public SVGImageElement(Node n, SVGElement p) throws MalformedSVGDocument
	{
		super(n, p);
	}


	
	/**
	 * Creates an SVG image.
	 * @param doc The owner document.
	 * @param pathSource The path of the picture.
	 * @since 0.1
	 */
	public SVGImageElement(SVGDocument doc, String pathSource)
	{
		super(doc);
		
		if(pathSource==null)
			throw new IllegalArgumentException();
		
		setNodeName(SVGElements.SVG_IMAGE);
		setAttribute("xlink:href", pathSource);//$NON-NLS-1$
	}



	/**
	 * @return The value of the X attribute (0 if there it does not exist or it is not a length).
	 * @since 0.1
	 */
	public double getX()
	{
		String v = getAttribute(getUsablePrefix()+SVGAttributes.SVG_X);
		double x;
		
		try { x = v==null ? 0 : new SVGLengthParser(v).parseCoordinate().getValue(); }
		catch(ParseException e) { x = 0; }
		
		return x;
	}
	
	
	
	
	/**
	 * @return The value of the Y attribute (0 if there it does not exist or it is not a length).
	 * @since 0.1
	 */
	public double getY()
	{
		String v = getAttribute(getUsablePrefix()+SVGAttributes.SVG_Y);
		double y;
		
		try { y = v==null ? 0 : new SVGLengthParser(v).parseCoordinate().getValue(); }
		catch(ParseException e) { y = 0; }
		
		return y;
	}
	
	
	/**
	 * @return The value of the <code>width</code> attribute.
	 * @since 0.1
	 */
	public double getWidth()
	{
		String v = getAttribute(getUsablePrefix()+SVGAttributes.SVG_WIDTH);
		double width;
		
		try { width = v==null ? Double.NaN : new SVGLengthParser(v).parseLength().getValue(); }
		catch(ParseException e) { width = Double.NaN; }
		
		return width;
	}
	
	
	
	/**
	 * @return The value of the <code>height</code> attribute.
	 * @since 0.1
	 */
	public double getHeight()
	{
		String v = getAttribute(getUsablePrefix()+SVGAttributes.SVG_HEIGHT);
		double height;
		
		try { height = v==null ? Double.NaN : new SVGLengthParser(v).parseLength().getValue(); }
		catch(ParseException e) { height = Double.NaN; }
		
		return height;
	}
	
	
	
	/**
	 * @return The URI reference.
	 * @since 0.1
	 */
	public String getURI()
	{
		return getAttribute("xlink:href");//$NON-NLS-1$
	}
	
	

	@Override
	public boolean checkAttributes()
	{
		double vWidth	= getWidth();
		double vHeight	= getHeight();
		
		if(Double.isNaN(vWidth) || Double.isNaN(vHeight) || vWidth<0 || vHeight<0)
			return false;
		
		return true;
	}




	@Override
	public boolean enableRendering()
	{
		if(getWidth()==0 || getHeight()==0 || getURI()==null)
			return false;
		
		try
		{
			URI uri = new URI(getURI());
			File f = new File(uri.getPath());
			
			if(!f.exists() || !f.canRead())
				return false;
		}
		catch(URISyntaxException e) { return false; }
		
		return true;
	}

}
