/***************************************************************************
*   Copyright (C) 2007-2010 by Thomas Thelliez aka jblud                  *
*   Contact : <admin.kontrol@gmail.com>                                   *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as               *
*   published by the Free Software Foundation; either version 2.0 of      *
*   the License, or (at your option) any later version.                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
*   General Public License for more details.                              *
*                                                                         *
*   You should have received a copy of the GNU General Public             *
*   License along with this program; if not, write to                     *
*   the Free Software Foundation, Inc.,                                   *
*   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA          *
***************************************************************************/

#include "remote_computer.h"

RemoteComputer::RemoteComputer(QString remote_ip, QString remote_port, QString remote_port_file, QString remote_password, int nbr)
    :  ip(remote_ip), port(remote_port), portFile(remote_port_file), password(remote_password), client_nbr(nbr)
{
    client_tab = new QTabWidget(this);
    settings = new QSettings("OpenNetwork", "KontrolPack", this);
    user_mode = settings->value("user/mode", "NONE").toString();

    client_tab->hide();
    client_tab->setMovable(true);
    isScreenshot = true;
#ifdef WIN32
    client_tab->setDocumentMode(true);
#elif defined __APPLE__
#else
    client_tab->setDocumentMode(true);
#endif
    client_tab->setTabPosition(QTabWidget::West);
    label = new QLabel(tr("Connecting..."), this);
    progress = new QProgressBar(this);
    progress->setMinimumHeight(150);
    QVBoxLayout* layout = new QVBoxLayout();
    layout->addWidget(client_tab);
    layout->addWidget(label);
    layout->addWidget(progress);
    setLayout(layout);
    controller = new SocketController(ip.toUtf8().data(), port.toInt(), portFile.toInt(), password);
    QObject::connect(controller->master_tcp_client, SIGNAL(disconnected_client(int)), this, SLOT(disconnected(int)));
    QObject::connect(controller, SIGNAL(client_characteristics(OsCharacteristic*)), this, SLOT(client_characteristics_slot(OsCharacteristic*)));
    QObject::connect(controller->master_tcp_client, SIGNAL(emit_tcp_state(int)), this, SLOT(tcp_state(int)));
    QObject::connect(controller->master_tcp_client, SIGNAL(wrong_password()), this, SLOT(wrong_password_slot()));
    QObject::connect(controller->master_tcp_client, SIGNAL(emit_error()), this, SLOT(error()));
    QObject::connect(controller->file_tcp_client, SIGNAL(emit_error()), this, SLOT(error()));
    QObject::connect(controller->file_tcp_client, SIGNAL(disconnected_client(int)), this, SLOT(disconnected(int)));
    QObject::connect(controller, SIGNAL(showMessage(QString,QString,int)), this, SLOT(showMessageSlot(QString,QString,int)));
    QObject::connect(controller->file_manager, SIGNAL(showMessage(QString,QString,int)), this, SLOT(showMessageSlot(QString,QString,int)));
    QObject::connect(controller->file_manager, SIGNAL(file_received(File*)), this, SLOT(file_received_slot(File*)));
    QObject::connect(controller, SIGNAL(file_uploaded(File*)), this, SLOT(file_uploaded_slot(File*)));
    QObject::connect(controller->file_tcp_client, SIGNAL(update_progress_size(int,int)), this, SLOT(update_percent_slot(int,int)));
    QObject::connect(controller->file_manager, SIGNAL(update_percent(int,int)), this, SLOT(update_percent_slot(int,int)));
    QObject::connect(controller, SIGNAL(update_percent(int,int)), this, SLOT(update_percent_slot(int,int)));
    QObject::connect(controller, SIGNAL(showLabelSignal(QString)), this, SLOT(showLabel(QString)));
    QObject::connect(controller->file_manager, SIGNAL(showLabelSignal(QString)), this, SLOT(showLabel(QString)));
}

void RemoteComputer::error()
{
    emit showMessage(tr("Error"), tr("An error occured during the authentication process. Please try again..."), 5000);
    emit remove_tab_signal(client_nbr);
}

void RemoteComputer::wrong_password_slot()
{
    QMessageBox::information(this, tr("Wrong password"), tr("Login failed : Wrong password"));
    emit remove_tab_signal(client_nbr);
}

void RemoteComputer::client_characteristics_slot(OsCharacteristic *characteristics)
{
    os = characteristics;
    QLabel *image = new QLabel(this);
    if (characteristics->system.startsWith(QString(LINUX), Qt::CaseSensitive)) {
        image->setPixmap(QPixmap(":/images/linux-computer.png"));
    } else if (characteristics->system.startsWith(WINDOWS, Qt::CaseSensitive)) {
        image->setPixmap(QPixmap(":/images/windows-computer.png"));
    }else if (characteristics->system.startsWith(MACOSX, Qt::CaseSensitive)) {
        image->setPixmap(QPixmap(":/images/mac-computer.png"));
    }
    // Instantiate 3 widgets corresponding to sub tab client widget.
    identity = new IdentityClient(client_nbr, image, characteristics, this);
    client_tab->addTab(identity, QIcon(":/images/connection.png"), tr("Identity card"));

    shareFiles = new ShareFiles(client_nbr, characteristics, this);
    client_tab->addTab(shareFiles, QIcon(":/images/explorer.png"), tr("Share files"));
    QString user_mode = settings->value("user/mode", "NONE").toString();

    // Command shell execution only for advanced user.
    if (!user_mode.startsWith("END", Qt::CaseInsensitive)) {
        execShell = new ExecShell(client_nbr, characteristics, this);
        client_tab->addTab(execShell, QIcon(":/images/terminal.png"), tr("Shell command"));
        QObject::connect(execShell, SIGNAL(sendCommand(QString,QString)), controller, SLOT(executeCommand(QString,QString)));
        QObject::connect(execShell, SIGNAL(add_command_to_database(Command*)), this, SLOT(add_command_to_database_slot(Command*)));
        QObject::connect(controller, SIGNAL(command_results(QString)), execShell, SLOT(set_command_results(QString)));
    }
    this->setStyleSheet("QFrame {background : white}");

    QObject::connect(shareFiles, SIGNAL(showMessage(QString,QString,int)), this, SLOT(showMessageSlot(QString,QString,int)));
    QObject::connect(shareFiles, SIGNAL(send_to_server(QString)), controller, SLOT(send_to_server(QString)));
    QObject::connect(shareFiles, SIGNAL(ask_file_to_server(QString,QString,QString)), controller, SLOT(ask_file_to_server(QString,QString,QString)));
    QObject::connect(shareFiles, SIGNAL(send_file_to_server(QString,QString,QString)), controller, SLOT(send_file_to_server(QString,QString,QString)));
    QObject::connect(shareFiles, SIGNAL(update_percent(int,int)), this, SLOT(update_percent_slot(int,int)));
    QObject::connect(controller, SIGNAL(remove_from_view(QString, bool)), shareFiles, SLOT(remove_from_view(QString, bool)));
    QObject::connect(this, SIGNAL(refresh_view(Folder*)), shareFiles, SLOT(refresh_view(Folder*)));
    QObject::connect(controller, SIGNAL(refresh_directory_view(Folder*)), this, SLOT(refresh_view_slot(Folder*)));
    QObject::connect(identity, SIGNAL(refreshScreenshotSignal(QString,QString,QString)), this, SLOT(ask_screenshot_to_server(QString,QString,QString)));

    label->hide();
    progress->setMinimumHeight(30);
    client_tab->show();
    emit update_os_information(characteristics->system, characteristics->hostname, client_nbr);
}

void RemoteComputer::refresh_view_slot(Folder *folder_info)
{
    emit refresh_view(folder_info);
    // Command shell execution only for advanced user.
    if (!user_mode.startsWith("END", Qt::CaseInsensitive)) {
        execShell->set_remote_path(folder_info->path);
    }
}

void RemoteComputer::tcp_state(int state)
{
    switch (state)
    {
    case TOKEN_WELCOME : {
            label->setText("Connecting to remote server...");
            break;
        }
    case RSA_PUBLIC_KEY_RECEPTION : {
            label->setText("RSA asymetric encryption started...");
            break;
        }
    case PASSWORD_VALIDATION : {
            label->setText("Password validation process...");
            break;
        }
    case AES_KEY_RECEPTION : {
            label->setText("AES symetric encryption started...");
            break;
        }
    case AUTH_SUCCESS : {
            label->setText("Authentification succeded. AES symetric encryption started...");
            progress->hide();
            label->hide();
            break;
        }
    default:
        ;
    }
}

void RemoteComputer::disconnected(int error)
{
    switch (error) {
    case QAbstractSocket::RemoteHostClosedError:
        emit showMessage(tr("Network error"), tr("Remote Host Closed Error"), 5000);
        break;
    case QAbstractSocket::HostNotFoundError:
        emit showMessage(tr("Network error"), tr("Host Not Found Error"), 5000);
        break;
    case QAbstractSocket::ConnectionRefusedError:
        emit showMessage(tr("Network error"), tr("Connection Refused Error"), 5000);
        break;
    case QAbstractSocket::SocketAccessError:
        emit showMessage(tr("Network error"), tr("Socket Access Error"), 5000);
        break;
    case QAbstractSocket::SocketResourceError:
        emit showMessage(tr("Network error"), tr("Socket Resource Error"), 5000);
        break;
    case QAbstractSocket::SocketTimeoutError:
        emit showMessage(tr("Network error"), tr("Socket Timeout Error"), 5000);
        break;
    case QAbstractSocket::DatagramTooLargeError:
        emit showMessage(tr("Network error"), tr("Datagram Too Large Error"), 5000);
        break;
    case QAbstractSocket::NetworkError:
        emit showMessage(tr("Network error"), tr("Network Error"), 5000);
        break;
    case QAbstractSocket::AddressInUseError:
        emit showMessage(tr("Network error"), tr("Address In Use Error"), 5000);
        break;
    case QAbstractSocket::SocketAddressNotAvailableError:
        emit showMessage(tr("Network error"), tr("Socket Address Not Available Error"), 5000);
        break;
    case QAbstractSocket::UnsupportedSocketOperationError:
        emit showMessage(tr("Network error"), tr("Unsupported Socket Operation Error"), 5000);
        break;
    case QAbstractSocket::ProxyAuthenticationRequiredError:
        emit showMessage(tr("Network error"), tr("Proxy Authentication Required Error"), 5000);
        break;
    case QAbstractSocket::SslHandshakeFailedError:
        emit showMessage(tr("Network error"), tr("Ssl Handshake Failed Error"), 5000);
        break;
    case QAbstractSocket::UnfinishedSocketOperationError:
        emit showMessage(tr("Network error"), tr("Unfinished Socket Operation Error"), 5000);
        break;
    case QAbstractSocket::ProxyConnectionRefusedError:
        emit showMessage(tr("Network error"), tr("Proxy Connection Refused Error"), 5000);
        break;
    case QAbstractSocket::ProxyConnectionClosedError:
        emit showMessage(tr("Network error"), tr("Proxy Connection Closed Error"), 5000);
        break;
    case QAbstractSocket::ProxyConnectionTimeoutError:
        emit showMessage(tr("Network error"), tr("Proxy Connection Timeout Error"), 5000);
        break;
    case QAbstractSocket::ProxyNotFoundError:
        emit showMessage(tr("Network error"), tr("Proxy Not Found Error"), 5000);
        break;
    case QAbstractSocket::ProxyProtocolError:
        emit showMessage(tr("Network error"), tr("Proxy Protocol Error"), 5000);
        break;
    case QAbstractSocket::UnknownSocketError:
        emit showMessage(tr("Network error"), tr("Unknown Socket Error"), 5000);
        break;
    default:
        ;
    }
    emit remove_tab_signal(client_nbr);
}


void RemoteComputer::showMessageSlot(QString title, QString content, int duration)
{
    emit showMessage(title, content, duration);
}

void RemoteComputer::file_received_slot(File *file)
{
    progress->hide();
    label->hide();
    if (isScreenshot) {
        isScreenshot = false;
        identity->blocked = false;
        if (!controller->file_queue->isEmpty())
            controller->file_queue->dequeue();
        identity->screenshotLabel->setPixmap(QPixmap(file->to + file->name).scaled(identity->screenshotLabel->size(), Qt::KeepAspectRatio, Qt::SmoothTransformation));
    } else {
        emit showMessage(tr("File download"), file->name + tr(" received."), 4500);
        shareFiles->refresh_local_view();
    }
    // Download next file if the file queue if not empty...
    if (!controller->file_queue->isEmpty())
        controller->file_queue->dequeue();
    controller->process_queue();
}

void RemoteComputer::file_uploaded_slot(File *file)
{
    progress->hide();
    label->hide();
    shareFiles->refresh();
}

void RemoteComputer::update_percent_slot(int current, int total)
{
    if (current > 0) {
        progress->show();
        progress->setMinimum(0);
        progress->setMaximum(total); // Set to 0 when uploading...
        progress->setValue(progress->value() + current);
    } else if (current == 0)
        progress->hide();
}

void RemoteComputer::ask_screenshot_to_server(QString to, QString from, QString file_name)
{
    identity->blocked = true;
    isScreenshot = true;
    controller->ask_file_to_server(to, from, file_name);
}

void RemoteComputer::showLabel(QString text)
{
    label->setText(text);
    label->show();
}

RemoteComputer::~RemoteComputer()
{
    delete this->controller->master_tcp_client;
    delete this->controller->file_tcp_client;
    delete this->controller->file_manager->file_queue;
    delete this->controller->file_manager;
    delete this->controller->file_queue;
    delete this->controller;
}

void RemoteComputer::add_command_to_database_slot(Command *command)
{
    emit add_command_to_database(command);
}
