/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <QtGui/QMessageBox>
#include <QtGui/QPushButton>

#include "connection.h"
#include "record.h"

#include "dialog.h"

using namespace Form;

Dialog::Dialog( Page *page, const QString &id, PagePolicy::State state, QWidget *parent )
  : QDialog( parent ),
    mPage( page ), mId( id ), mState( state )
{
  // set window title
  QString title;
  if ( state == PagePolicy::Add )
    title = tr( "Add: " );
  else if ( state == PagePolicy::Modify )
    title = tr( "Modify: " );
  else
    title = tr( "Delete: " );

  title += page->title();
  setWindowTitle( title );

  // create gui
  mContainer = new Container( page->name(), this );

  QGridLayout *layout = new QGridLayout( this );
  layout->addWidget( mContainer, 0, 0, 1, 3 );

  QPushButton *button = new QPushButton( tr( "&Ok" ), this );
  connect( button, SIGNAL( clicked() ), this, SLOT( accept() ) );
  layout->addWidget( button, 1, 1 );

  button = new QPushButton( tr( "&Cancel" ), this );
  connect( button, SIGNAL( clicked() ), this, SLOT( reject() ) );
  layout->addWidget( button, 1, 2 );

  Description *description = mPage->formDescription();
  if ( description ) {
    mContainer->build( description, mPage->formPolicy(), mState );
  }

  Record *record = mPage->formRecord();
  if ( record ) {
    Entry entry = record->loadEntry( mId, mState );
    mContainer->loadEntry( entry );
  }
}

void Dialog::accept()
{
  QString msg;

  if ( !mContainer->isValid( msg, mState ) ) {
    QMessageBox::warning( this, tr( "Form not valid" ), msg );
    return;
  }

  Entry entry;
  entry.setId( mId );

  /**
   * Fill the entry with the original values in Modify state.
   */
  if ( mState == PagePolicy::Modify ) {
    entry = mPage->formRecord()->loadEntry( mId, mState );
    entry.dump();
  }

  mContainer->saveEntry( entry );

  Record *record = mPage->formRecord();
  if ( record ) {
    QString errorMsg;
    if ( !record->saveEntry( entry, mState, errorMsg ) ) {
      QMessageBox::critical( this, QString(), errorMsg );
      return;
    }
  }

  done( Accepted );
}
