/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "iggdialogpaletteeditor.h"


#include "icontrolmodule.h"
#include "ierror.h"
#include "iimagefactory.h"
#include "ifile.h"
#include "ipalette.h"
#include "ipaletteset.h"
#include "ipiecewisefunction.h"
#include "ishell.h"


#include "iggframepaletteselectionbase.h"
#include "iggframepiecewisefunctioncontrols.h"
#include "iggmainwindow.h"
#include "iggwidgetarea.h"
#include "iggwidgetotherbutton.h"

#include "ibgwidgetentrysubject.h"

#include "iggsubjectfactory.h"


using namespace iParameter;
using namespace iParameter;


namespace iggDialogPaletteEditor_Private
{
	//
	//  Load and save palettes to files
	//
	class FileLoadButton : public iggWidgetSimpleButton
	{

	public:

		FileLoadButton(iggDialogPaletteEditor *dialog, iggFrame *parent) : iggWidgetSimpleButton("Load from file",parent)
		{
			mDialog = dialog;
			this->SetBaloonHelp("Load palette from file","Load a palette from a file. The file must have been previously created by IFrIT Palette Editor.");
		}

	protected:

		virtual void Execute()
		{
			iString fn;
			if(mOldFileName.IsEmpty()) mOldFileName = this->GetShell()->GetEnvironment(Environment::Palette);
			fn = this->GetMainWindow()->GetFileName("Open palette file",mOldFileName,"Palette file (*.ipf)");
			if(!fn.IsEmpty() && fn.Part(-4).Lower()!=".ipf") fn += ".ipf";

			iFile f(fn);
			if(!f.Open(iFile::_Read,iFile::_Text))
			{
				this->GetMainWindow()->PopupWindow(mDialog->GetFrame(),"Failed to open a palette file.",PopupWindow::Error);
				return;
			}

			iString ws;
			if(!f.ReadLine(ws) || !iPaletteSet::Default()->UnPackValuePalette(ws,mDialog->GetCurrentPalette()))
			{
				this->GetMainWindow()->PopupWindow(mDialog->GetFrame(),"Palette file is corrupted.",PopupWindow::Error);
			}
			else
			{
				mOldFileName = fn;
				mDialog->Apply(false);
				mDialog->UpdateDialog();
			}
			
			f.Close();
		}

		iString mOldFileName;
		iggDialogPaletteEditor *mDialog;
	};


	class FileSaveButton : public iggWidgetSimpleButton
	{

	public:

		FileSaveButton(iggDialogPaletteEditor *dialog, iggFrame *parent) : iggWidgetSimpleButton("Save to file",parent)
		{
			mDialog = dialog;
			this->SetBaloonHelp("Save palette to file","Save the current palette to a file. If the file exists, it will be overwritten and all its contents erased.");
		}

	protected:

		virtual void Execute()
		{
			iString fn;
			if(mOldFileName.IsEmpty()) mOldFileName = this->GetShell()->GetEnvironment(Environment::Palette);
			fn = this->GetMainWindow()->GetFileName("Save palette file",mOldFileName,"Palette file (*.ipf)",false);
			if(!fn.IsEmpty() && fn.Part(-4).Lower()!=".ipf") fn += ".ipf";

			iFile f(fn);
			if(!f.Open(iFile::_Write,iFile::_Text))
			{
				this->GetMainWindow()->PopupWindow(mDialog->GetFrame(),"Failed to create a palette file.",PopupWindow::Error);
				return;
			}

			iString ws;
			iPaletteSet::Default()->PackValuePalette(ws,mDialog->GetCurrentPalette());
			if(!f.WriteLine(ws))
			{
				this->GetMainWindow()->PopupWindow(mDialog->GetFrame(),"Unable to write into a palette file.",PopupWindow::Error);
			}
			else
			{
				mOldFileName = fn;
				mDialog->UpdateDialog();
			}

			f.Close();
		}

		iString mOldFileName;
		iggDialogPaletteEditor *mDialog;
	};


	class PaletteSelection : public iggFramePaletteSelectionBase
	{

	public:

		PaletteSelection(iggDialogPaletteEditor *editor, iggFrame *parent) : iggFramePaletteSelectionBase(false,false,parent)
		{
			mEditor = editor;
		}

		virtual iPalette* GetCurrentPalette() const
		{
			return mEditor->GetCurrentPalette();
		}

	protected:

		virtual int GetPaletteId() const
		{
			return mEditor->GetCurrentPaletteIndex() + 1;
		}

		virtual void SetPaletteId(int n)
		{
			mEditor->SetCurrentPaletteIndex(n-1);
		}

		iggDialogPaletteEditor *mEditor;
	};


	class PiecewiseFunctionControls : public iggFramePiecewiseFunctionControls
	{

	public:

		PiecewiseFunctionControls(int type, iggDialogPaletteEditor *editor, iggFrame *parent): iggFramePiecewiseFunctionControls(false,false,parent)
		{
			mType = type;
			mEditor = editor;

			this->SetBaloonHelp("Interactively controls palette components. Press Shift+F1 for more help.","Each of the three palette component can be represented as a piecewise function. The piecewise function can be controlled interactively by clicking on a control point and dragging it around with the mouse. Buttons below the interactive area allow you to add or remove a control point and to reset the function to the default one.");
		}

		virtual const iHistogram* GetHistogram() const
		{
			//
			//  No background histogram
			//
			return 0;
		}

		virtual iPiecewiseFunction* GetFunction()
		{
			return mEditor->GetCurrentPalette()->GetComponent(mType);
		}

		virtual void OnRender()
		{
			//
			//  No rendering
			//
		}

		virtual void OnChange()
		{
			mEditor->SetChanged(true);
			mEditor->GetCurrentPalette()->Update();
			this->UpdateDependents();
		}

	protected:

		int mType;
		iggDialogPaletteEditor *mEditor;
	};
	//
	//  Palette operations
	//
	class ApplyButton : public iggWidgetSimpleButton
	{

	public:

		ApplyButton(iggDialogPaletteEditor *dialog, iggFrame *parent) : iggWidgetSimpleButton("Apply",parent)
		{
			mDialog = dialog;
			this->SetBaloonHelp("Apply the changes","Apply all the changes made to palettes and render all visualization scenes.");
		}

	protected:

		virtual void Execute()
		{
			mDialog->Apply(false);
			this->Enable(false);
		}

		iggDialogPaletteEditor *mDialog;
	};


	class NameLineEdit : public iggWidget
	{

	public:

		NameLineEdit(iggDialogPaletteEditor *dialog, iggFrame *parent) : iggWidget(parent)
		{
			mDialog = dialog;
			mSubject = iggSubjectFactory::CreateWidgetEntrySubject(this,false,0,"Name");
			this->SetBaloonHelp("Set the name","Change the name of the current palette. The new name may contain white spaces but should not contain symbols '*'.");
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			mSubject->SetText(mDialog->GetCurrentPalette()->GetName());
		}

		virtual void OnString1Body(const iString &s)
		{
			if(s.Find("*") >= 0)
			{
				this->GetMainWindow()->PopupWindow(mDialog->GetFrame(),"A palette name may not contain symbols '*'",PopupWindow::Error);
				mSubject->SetText(mDialog->GetCurrentPalette()->GetName());
			}
			else
			{
				mDialog->GetCurrentPalette()->SetName(s);
				mDialog->SetChanged(true);
			}
		}

		ibgWidgetEntrySubject *mSubject;
		iggDialogPaletteEditor *mDialog;
	};


	class CreateNewButton : public iggWidgetSimpleButton
	{

	public:

		CreateNewButton(iggDialogPaletteEditor *dialog, iggFrame *parent) : iggWidgetSimpleButton("Create new palette",parent)
		{
			mDialog = dialog;
			this->SetBaloonHelp("Create new palette","Add a new palette to the global palette list.");
		}

	protected:

		virtual void Execute()
		{
			mDialog->Apply(true);
			if(!iPaletteSet::Default()->AddEmptyPalette())
			{
				this->GetMainWindow()->PopupWindow(mDialog->GetFrame(),"Unable to create a new palette.\nPerhaps, there is not enough free memory.",PopupWindow::Error);
			}
			else
			{
				mDialog->SetCurrentPaletteIndex(iPaletteSet::Default()->GetNumberOfPalettes()-1);
			}
		}

		iggDialogPaletteEditor *mDialog;
	};
};


using namespace iggDialogPaletteEditor_Private;


iggDialogPaletteEditor::iggDialogPaletteEditor(iggMainWindow *parent) : iggDialog(parent,0U,iImageFactory::FindIcon("paled.png"),"Palette Editor","sr.gg.de",2)
{
	mCurrentPalette = new iPalette; IERROR_ASSERT(mCurrentPalette);
	mCurrentPalette->Copy(iPaletteSet::Default()->GetPalette(0));
	mCurrentPaletteIndex = 0;

	if(this->ImmediateConstruction()) this->CompleteConstruction();
}


void iggDialogPaletteEditor::CompleteConstructionBody()
{
	iggFrame *cpb = new iggFrame(mFrame,2);
	PaletteSelection *ps = new PaletteSelection(this,cpb);
	cpb->AddLine(ps);
	cpb->SetColStretch(0,0);
	cpb->SetColStretch(1,10);

	iggFrame *f2 = new iggFrame("File operations",mFrame);
	f2->AddLine(new FileLoadButton(this,f2));
	f2->AddLine(new FileSaveButton(this,f2));

	mFrame->AddLine(cpb,f2);

	iggFrame *spb = new iggFrame("Sculpt Palette",mFrame,3);
	spb->AddLine(new iggWidgetTextArea("%bRed",spb),new iggWidgetTextArea("%bGreen",spb),new iggWidgetTextArea("%bBlue",spb));

	iggWidget *pf1, *pf2, *pf3;
	pf1 = new PiecewiseFunctionControls(0,this,spb);
	ps->AddBuddy(pf1);
	pf2 = new PiecewiseFunctionControls(1,this,spb);
	ps->AddBuddy(pf2);
	pf3 = new PiecewiseFunctionControls(2,this,spb);
	ps->AddBuddy(pf3);
	spb->AddLine(pf1,pf2,pf3);

	spb->SetRowStretch(0,0);
	spb->SetRowStretch(1,10);

	iggFrame *po = new iggFrame("Palette operations",mFrame);

	pf1 = new NameLineEdit(this,po);
	ps->AddDependent(pf1);
	po->AddLine(pf1);
	pf2 = new CreateNewButton(this,po);
	po->AddLine(pf2);

	po->AddSpace(10);

	mApplyButton = new ApplyButton(this,po);
	mApplyButton->Enable(false);
	po->AddLine(mApplyButton);

	mFrame->AddLine(spb,po);
	mFrame->SetColStretch(0,10);
	mFrame->SetColStretch(1,0);
	mFrame->SetRowStretch(0,0);
	mFrame->SetRowStretch(1,10);
}


iggDialogPaletteEditor::~iggDialogPaletteEditor()
{
	if(mCurrentPalette != 0) delete mCurrentPalette;
}


const iString& iggDialogPaletteEditor::GetToolTip() const
{
	static const iString tmp = "Create new palettes and edit existing ones.";
	return tmp;
}


void iggDialogPaletteEditor::SetChanged(bool s)
{
	this->CompleteConstruction();
	mApplyButton->Enable(s);
}


void iggDialogPaletteEditor::SetCurrentPaletteIndex(int n)
{
	if(n>=0 && n<iPaletteSet::Default()->GetNumberOfPalettes())
	{
		this->CompleteConstruction();
		if(mApplyButton->IsEnabled()) this->Apply(true);
		mCurrentPalette->Copy(iPaletteSet::Default()->GetPalette(n));
		mCurrentPaletteIndex = n;
		this->UpdateDialog();
	}
}


void iggDialogPaletteEditor::Apply(bool ask)
{
	this->CompleteConstruction();
	if(mApplyButton->IsEnabled() && (!ask || this->GetMainWindow()->AskForConfirmation("Do you want to apply changes?","Apply")))
	{
		iPaletteSet::Default()->GetPalette(mCurrentPaletteIndex)->Copy(mCurrentPalette);
		iggFramePaletteSelectionBase::UpdateAll();
		this->GetShell()->GetControlModule()->Render(RenderOption::All);
		mApplyButton->Enable(false);
	}
}


bool iggDialogPaletteEditor::CanBeClosed()
{
	this->Apply(true);
	return true;
}

#endif
