/* $Id: main.c,v 1.102 2003/06/17 10:19:12 sjoerd Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <locale.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <fcntl.h>
#include <errno.h>
#include <assert.h>

#include <SDL.h>
#include <SDL_ttf.h>

#include "global.h"
#include "model.h"
#include "graphics.h"
#include "localio.h"
#include "network.h"
#include "comm.h"
#include "audio.h"
#include "levelloader.h"
#include "configuration.h"
#include "vector.h"

int quit = 0;

FILE *output = NULL;
FILE *input = NULL;
int debugmask = 0 ;
int do_fullscreen = FALSE;
int do_music = FALSE;
int do_sfx = TRUE;

#define DEFAULT_TICK_INTERVAL 40
int TICK_INTERVAL = DEFAULT_TICK_INTERVAL;

#define MDEBUG(...) DEBUG(DMAIN,"Main",__VA_ARGS__)

int slavemode = 0;
#define SLAVE(...) if (slavemode) { \
                     printf("SLAVE:"); \
                     printf(__VA_ARGS__); \
                     printf("\n"); \
                   }

#define PORT 4321
#define SHIFT(x)  (1 << x)

typedef struct {
  /* Name of the player */
  char *name;
  /* Current events of a player */
  uint32_t event;
  /* Previous events of a player */
  uint32_t pevent;
  /* indicates if a player is local or not */
  int is_local;
  int throws;
  Model_object *obj;
} _Player;

typedef struct {
  char *name;
  Level_types type;
  union {
    int *i;
    Vector *v;
    int32_t *i32;
  } target;
} settingstranslation;

void
init_nls(void) {
#ifdef ENABLE_NLS
  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE, LOCALEDIR);
  textdomain(PACKAGE);
#endif
}

static _Player **
new_players(int nrplayers) {
  _Player **players;
  _Player *p;
  int x;

  players = malloc(sizeof(_Player *) * (nrplayers + 1));
  assert(players != NULL);
  for (x = 0; x < nrplayers ; x++) {
    p = malloc(sizeof(_Player) * nrplayers);
    p->name = NULL;
    p->event = 0;
    p->pevent = 0;
    p->is_local = FALSE;
    p->throws = 0;
    p->obj = NULL;
    players[x] = p;
  }
  players[x] = NULL;
  return players;
}

static void
set_players_name(_Player **players,int player,char *name) {
  free(players[player]->name); 
  players[player]->name = strdup(name);
}

static void
del_players(_Player **players) {
  int x;
  for (x = 0 ; players[x] != NULL ; x++) {
    free(players[x]->name);
    free(players[x]);
  }
  free(players);
}

char *
read_level_file(char *pathname) {
#define BUFSIZE 1024
  int fd;
  char *result = NULL;
  char buf[BUFSIZE];
  int size = 0,ret;

  MDEBUG("Reading level from %s",pathname);
  fd = open(pathname,O_RDONLY); 
  if (fd == -1) {
    MDEBUG("Couldn't read level file %s",strerror(errno));
    return NULL;
  }
  while ((ret = read(fd,buf,BUFSIZE)) > 0) {
    result = realloc(result,size+ret); 
    memcpy(result+size,buf,ret);
    size += ret;
  }
  if (ret < 0) {
    MDEBUG("Reading from file %s failed %s",pathname,strerror(errno));
    free(result);
    return NULL;
  }
  result = realloc(result,size+1);
  result[size] = '\0';
  close(fd);
  return result;
}

char *
load_level_file(char *levelname) {
  char file[MAXPATHLEN];
  char *xmlstring = NULL;

  if (levelname == NULL) {
    levelname = "ffrenzy_default";
  }
  snprintf(file,MAXPATHLEN,"%s/ffrenzy/levels/%s.xml",DATADIR, levelname);
  xmlstring = read_level_file(file);
  if (xmlstring == NULL) {
    /* try to load level in relative path */
    snprintf(file,MAXPATHLEN,"%s",levelname);
    xmlstring = read_level_file(file);
  }

  if (xmlstring == NULL) {
    WARN(_("Couldn't open level file"));
  }
  return xmlstring;
}

void 
init_program(void) {
  init_nls();
  INFO(_("Starting feeding frenzy"));
  INFO(_("Released under the GNU Public License\n"));
  /* initialize sdl */
  if (SDL_Init(SDL_INIT_VIDEO|SDL_INIT_AUDIO|SDL_INIT_NOPARACHUTE) < 0) {
    WARN(_("SDL initialisation failed: %s"),SDL_GetError());
    exit(-1);
  }
  /* initialize libsdl-ttf2 */
  if (TTF_Init() < 0) {
    WARN(_("SDL-ttf2 init failed: %s"),SDL_GetError());
    SDL_Quit();
    exit(-1);
  }
}

void 
deinit_program(int exitstatus) {
  SDL_Quit();
  TTF_Quit();
  exit(exitstatus);
}

void usage(char *progname, char *error) {
  if (error != NULL) {
    WARN(_("error: %s"),error);
  }
  WARN(_("Usage: %s [OPTIONS]"),progname);
  WARN(_("\t-p players    \tNumber of players on the local machine"));
  WARN(_("\t-l levelname  \tLoad levelname as the level to play"));
  WARN(_("\t-s nrplayers  \tStart network server for nrplayers players"));
  WARN(_("\t-c server     \tConnect to network server server"));
  WARN(_("\t-C configfile \tAlternate config file to use"));
  WARN(_("\t-f            \tPlay fullscreen"));
  WARN(_("\t-m            \tTurn on the music"));
  WARN(_("\t-q            \tTurn off the sound effect (quiet)"));
  WARN(_("\t-i inputfile  \tUse input from inputfile"));
  WARN(_("\t-o outputfile \tPut all events in the outputfile"));
  WARN(_("\t-d debugmask  \tDebug mask to use"));
  WARN(_("\t\tGraphics:     \t0x1"));
  WARN(_("\t\tModel:        \t0x2"));
  WARN(_("\t\tNetwork:      \t0x4"));
  WARN(_("\t\tLocalio:      \t0x8"));
  WARN(_("\t\tMain:         \t0x10"));
  WARN(_("\t\tCommunication:\t0x20"));
}

Uint32 TimeLeft(void) {
  static Uint32 next_time = 0;
  Uint32 now,res;
  static Uint32 tick = 0;
  static int i = 0;
  now = SDL_GetTicks();
  if (i == 0) {
    if (tick != 0)
      INFO(_("FPS: %d"),(30 * 1000)/(now-tick));
    tick = now;
    i = 30;
  }
  i--;
  if (next_time <= now) {
    next_time = now + TICK_INTERVAL;
    return 0;
  }
  res = next_time - now;
  next_time += TICK_INTERVAL;
  return res;
}

void 
do_events(Comm_game *game,Model *m,_Player **players) {
  int x;
  _Player *p;
  for (x = 0 ; players[x] != NULL ; x++) {
    p = players[x];
    p->event = comm_game_get_event(game,x);
    if (output != NULL) {
      fprintf(output,"%d: %#x\n",x,p->event);
      fflush(output);
    }
    if (p->event & SHIFT(CONFIG_ACTION_THROW)) {
      p->throws --;
    } else { 
      p->throws = 0;
    }
    if (p->throws > 0) {
      p->event &= ~SHIFT(CONFIG_ACTION_THROW);
    }
    if (p->event == p->pevent) continue;

    /* left/right action */
    if (p->event & SHIFT(CONFIG_ACTION_LEFT) && 
        p->event & SHIFT(CONFIG_ACTION_RIGHT)) {
      /* skip */;
    } else if (p->event & SHIFT(CONFIG_ACTION_LEFT)) {
      model_object_move_x(p->obj,-1);
    } else if (p->event & SHIFT(CONFIG_ACTION_RIGHT)) {
      model_object_move_x(p->obj,1);
    } else {
      model_object_move_x(p->obj,0);
    }

    /* jumping */
    if (p->event & SHIFT(CONFIG_ACTION_JUMP)) {
      model_object_move_y(p->obj,1);
    } else {
      model_object_move_y(p->obj,0);
    }

    /* throwing */
    if (p->event & SHIFT(CONFIG_ACTION_THROW)) {
      model_object_throw(p->obj);
    }
    p->pevent = p->event;
  }
}

int
get_local_events(Comm_game *game, Local_io *io,int *playermapping) {
  int x;
  char *p,buf[128];
  if (input != NULL) {
    for (x = 0 ; playermapping[x] != -1 ; x++) {
      if (fgets(buf,128,input) == NULL) {
        WARN(_("End of input file"));
        return FALSE;
      }
      if (atoi(buf) != x) {
        WARN(_("Expecting %d but got %d"),x,atoi(buf));
        return FALSE;
      }
      for (p = buf; *p != '\0' && *p != ':'; p++) 
        ;
      if (*p == '\0') {
        WARN(_("Delimeter : not found in string %s"),buf);
        return FALSE;
      }
      p++;
      comm_game_send_event(game,playermapping[x],strtol(p,NULL,16));
    }
  } else {
    for (x = 0 ; playermapping[x] != -1 ; x++) {
      comm_game_send_event(game,playermapping[x],local_io_get_events(io,x));
    }
  }
  return TRUE;
}

static int
event_filter(const SDL_Event *event) {

  if (event->type == SDL_QUIT ||
      (event->type == SDL_KEYDOWN && event->key.keysym.sym == SDLK_ESCAPE)) {
    quit = 1;
  }
  if (input != NULL && event->key.keysym.sym == SDLK_UP) {
    if (event->type == SDL_KEYDOWN) {
      TICK_INTERVAL = DEFAULT_TICK_INTERVAL/4;
    } else {
      TICK_INTERVAL = DEFAULT_TICK_INTERVAL;
    }
  }
  return 1;
}

void
do_event_loop(Comm_game *game,Model *m, Graphics *g, Local_io *io, 
              _Player **players,int *playermapping) {
  Audio *au = NULL;
  if (do_sfx) {
    au = new_audio(do_music);
  }

  SDL_SetEventFilter(event_filter); 

  while (model_tick(m) && !quit) {
    if (au != NULL) audio_update(au,m);
    graphics_update(g,m);
    local_io_update(io);
    if (!get_local_events(game,io,playermapping)) 
      break;
    if (!comm_game_update(game)) {
      break;
    }
    do_events(game,m,players);
    SDL_Delay(TimeLeft());
  }
  quit = FALSE;
  if (au != NULL) {
      audio_update(au,m);
      del_audio(au);
  }
}

int
do_setup_player_io(void *data, void *user_data) {
  Config_key *key = (Config_key *)data;
  void **arg = (void **)user_data;
  Local_io *io = arg[0];
  int *player = arg[1];

  local_io_register_key(io,key->key,*player,SHIFT(key->action));

  return TRUE;
}

Local_io *
setup_player_io(int nrplayers, Config *c) {
  int x;
  List *keys;
  void *arg[2];
  Local_io *io;

  io = new_local_io(nrplayers);
  assert(io != NULL);

  arg[0] = io;
  for (x = 0 ; x < nrplayers; x++) {
    arg[1] = &x;
    keys = config_get_keys(c,x);
    if (keys != NULL)  {
      list_foreach(keys,do_setup_player_io,arg);
    }
  }
  return io;
}

static void
initialise_model_settings(Level *level, Model_settings *ms) {
#define INT_SETTING(x)    { #x, LEVEL_INT,      {i:    &ms->x}}
#define INT32_SETTING(x)  { #x, LEVEL_INT32_T,  {i32: &ms->x}}
#define VECTOR_SETTING(x) { #x, LEVEL_VECTOR,   {v:    ms->x }}
  Level_setting *sobj;
  int x;
  int nrsettings;
  settingstranslation settings_table[] = { 
       INT_SETTING    (spawn_powerups),
       INT_SETTING    (spawn_constant),
       INT_SETTING    (spawn_deliver),
       INT_SETTING    (spawn_hit),
       INT_SETTING    (spawn_pickup_powerup),
       INT32_SETTING  (init_food_spawn),
       INT32_SETTING  (spawn_delay),
       INT32_SETTING  (spawn_constant_delay),
       INT32_SETTING  (random_seed),
       VECTOR_SETTING (gravity),
       VECTOR_SETTING (bounciness),
       INT32_SETTING  (player_max_life),
       INT32_SETTING  (player_ground_friction),
       INT32_SETTING  (player_air_friction),
       INT32_SETTING  (player_ground_accel),
       INT32_SETTING  (player_air_accel),
       VECTOR_SETTING (player_max_speed),
       INT32_SETTING  (foodstack_size),
       INT32_SETTING  (powerup_length),
       INT32_SETTING  (unconc_length),
       INT32_SETTING  (food_ground_friction),  
       INT32_SETTING  (food_air_friction),
       VECTOR_SETTING (food_max_speed),
       INT32_SETTING  (food_deliver_life),
       INT32_SETTING  (min_food),
       INT32_SETTING  (max_food),
       VECTOR_SETTING (food_throw_speed),
       VECTOR_SETTING (player_jump_speed),
       INT32_SETTING  (starvation_delay),
       INT32_SETTING  (starvation_amount)
     };

  nrsettings = sizeof(settings_table)/sizeof(settingstranslation);

  while(((sobj = level_next_setting(level))) != NULL) {
    for ( x = 0 ; x < nrsettings; x++) {
      if (!strcmp(settings_table[x].name,sobj->name)) {
        if (sobj->type != settings_table[x].type) {
          WARN("Invalid type for setting %s",sobj->name);
        } else {
          switch (sobj->type) {
            case LEVEL_VECTOR:
              settings_table[x].target.v->x = sobj->data.vval->x;
              settings_table[x].target.v->y = sobj->data.vval->y;
              MDEBUG("Setting %s to (%d,%d)",sobj->name,
                                            sobj->data.vval->x,
                                            sobj->data.vval->y);
              break;
            case LEVEL_INT:
              *(settings_table[x].target.i) = sobj->data.ival;
              MDEBUG("Setting %s to %d",sobj->name,sobj->data.ival);
              break;
            case LEVEL_INT32_T:
              *(settings_table[x].target.i32) = sobj->data.i32val;
              MDEBUG("Setting %s to %d",sobj->name,sobj->data.i32val);
              break;
            default:
              assert (FALSE && "Unknown settings type");
          }

        }
        break;
      }
    }
  }
}

static int
setup_model_graphics(Level* level,Config *config, _Player **players,
                     Model **model, Graphics **graphics) { 
  Model * m;
  Graphics *g;
  Level_object *obj;
  Level_player *pobj;
  Level_type   *tobj;
  Model_object *mobj;
  int x;
  Vector *v;
  Model_settings *ms = new_model_settings();

  initialise_model_settings(level,ms);
  m = new_model(level->size->x,level->size->y,ms);
  *model = m;
  del_model_settings(ms);

  g = new_graphics(config->resolution->x,config->resolution->y,
                  level->size->x,
                  level->size->y,do_fullscreen);
  *graphics = g;

  level_reset_object(level);
  while ((obj = level_next_object(level)) != NULL) {
    mobj = model_add_object(m,obj->type,NULL,obj->pos->x,obj->pos->y,
                     obj->size->x,obj->size->y);
    if (obj->svgseries != NULL) {
      graphics_add_object(g,mobj,obj->svgseries);
    }
  }

  for (x = 0 ; players[x] != NULL; x++) {
    pobj = level_get_player(level,x);
    mobj = players[x]->obj = model_add_object(m,OT_PLAYER,NULL,
                                        pobj->pos->x,pobj->pos->y,
                                        pobj->size->x,pobj->size->y);
    graphics_add_player(g,mobj,players[x]->name,players[x]->is_local);
    if (pobj->svgseries != NULL) {
      graphics_add_object(g,mobj,pobj->svgseries);
    }
    mobj = model_add_object(m,OT_HOME,players[x]->obj,
                                        pobj->home_pos->x,pobj->home_pos->y,
                                        pobj->home_size->x,pobj->home_size->y);
    if (pobj->home_svgseries != NULL) {
      graphics_add_object(g,mobj,pobj->home_svgseries);
    }
  }

  for (x = OT_FIRST; x < OT_NRTYPES; x++) {
    v = level_get_type_size(level,x);
    if (v == NULL) {
      model_register_type(m,x,15,15);
    } else {
      model_register_type(m,x,v->x,v->y);
    }
    tobj = level_get_type(level,x);
    if (tobj != NULL && tobj->svgseries != NULL) {
      graphics_register_type(g,x,tobj->svgseries);
    }
  }
  
  if (level->background != NULL) {
    graphics_set_background(g,level->background);
  } else {
    WARN(_("Couldn't set background"));
  }
  
  return TRUE;
}

int *
new_player_mapping(int nrplayers) {
  int *result;

  result = malloc(sizeof(int) * (nrplayers + 1));
  assert(result != NULL);
  memset(result,-1,sizeof(int) * (nrplayers +1));
  return result;
}

inline void
set_player_mapping(int *mapping, int localnr, int gamenr) {
  mapping[localnr] = gamenr;

}

void
del_player_mapping(int *playermapping) {
  free(playermapping);
}

/* starts up the game and cleans up afterward */
void
start_game(Comm_game *game,Level *level,Config *config,
           _Player **players, int *playermapping,int localplayers) {
  Model *m;
  Graphics *g;
  Local_io *io;
  int x;

  /* setup local io for our players */
  io = setup_player_io(localplayers,config);

  setup_model_graphics(level,config,players,&m,&g);
  del_level(level);
  /* startup the main loop */
  do_event_loop(game,m,g,io,players,playermapping);
  /* first remove the communication, so the other processes know immediately we
   * have exit'ed the main loop */
  del_comm_game(game);

  /* Do one final drawing and show the scores */
  graphics_update(g,m);
  graphics_showscores(g);
  for (x = 100; x && !quit;  x--) {
    SDL_PumpEvents();
    SDL_Delay(100);
  }

  /* print the scores on stdout too */
  INFO(_("\t-----------------------Player scores------------------"));
  for (x = 0; players[x] != NULL ; x++) {
    INFO(_("\t%c %-30s: %8d"),
         IS_DEAD(players[x]->obj) ? 'X' : ' ',
         players[x]->name,
         players[x]->obj->score);
  }
  INFO(N_("\t-----------------------------------------------------"));
  INFO(_("Game finished"));

  /* final cleaning up */
  del_graphics(g,m);
  del_model(m);
  del_local_io(io);
  del_players(players);
  del_player_mapping(playermapping);
  del_config(config);
}

void 
do_single_game(int nrplayers,char *levelname, Config *config) {
  int *playermapping;
  _Player **players;
  int x;
  char *levelstr;
  Level *level;
  Comm_game *game;

  INFO(_("Single machine, players: %d, level: %s"),nrplayers,levelname);
  levelstr = load_level_file(levelname);
  if (levelstr == NULL) return;

  level = new_level(levelstr);
  if (level == NULL) {
    WARN(_("Loading level failed")); 
    return;
  }
  free(levelstr);
  INFO(_("Level loaded for %d players"),level->numplayers);
  if (nrplayers > level->numplayers) { 
    WARN(_("Level can only support %d players"),level->numplayers);
    nrplayers = level->numplayers;
  }
  playermapping = new_player_mapping(nrplayers);
  players = new_players(nrplayers);

  for (x = 0 ; x < nrplayers ; x++ ) {
    set_player_mapping(playermapping,x,x);
    players[x]->is_local = TRUE;
    set_players_name(players,x,config_get_pname(config,x));
  }

  game = new_comm_game(nrplayers);
  start_game(game,level,config,players,playermapping,nrplayers);
}

int 
do_network_server(Comm_server *serv, char *levelstr,
                  Config *config, int localplayers,int *playermapping) {
  int x,y;

  for (x=0; x < localplayers; x++) {
    y = comm_server_register_player(serv,config_get_pname(config,x));
    if (y == -2) {
      WARN("Registering player failed");
      return  FALSE;
    }
    playermapping[x] = y;
  }
  if (!comm_server_wait_for_others(serv)) {
    INFO(_("Waiting for other players failed"));
    return FALSE;
  }
  INFO(_("All needed players have arrived"));
  if (!comm_server_send_connections(serv)) {
    INFO(_("Sending connections failed"));
    return FALSE;
  }
  INFO(_("Networking setup done, sending network player info"));
  if (!comm_server_send_player_info(serv)) {
    WARN("Sending player info to others failed");
    return FALSE;
  }
  INFO(_("Sending level"));
  if (!comm_server_send_level(serv,levelstr)) {
    WARN("Sending level  to others failed");
    return FALSE;
  }
  return TRUE;
}

void
do_server_game(int nrplayers,int totalplayers, 
               char *levelname, Config *config) {
  int *playermapping;
  _Player **players;
  char *levelstr;
  Level *level;
  int x;
  Comm_server *serv;
  Comm_game *game;

  INFO("multi machine server , local players: %d, players: %d, level: %s"
      ,nrplayers,totalplayers,levelname);
  levelstr = load_level_file(levelname);
  if (levelstr == NULL) return;

  level = new_level(levelstr);
  if (level == NULL) {
    WARN(_("Loading level failed")); 
    return;
  }
  INFO(_("Level loaded for %d players"),level->numplayers);
  if (totalplayers > level->numplayers) { 
    WARN(_("Level can only support %d players"),level->numplayers);
    totalplayers = level->numplayers;
    nrplayers = MIN(totalplayers,nrplayers);
  }

  serv = new_comm_server(PORT,totalplayers);
  if (serv == NULL) {
    WARN(_("Setting up ffrenzy server failed"));
    return;
  }
  playermapping = new_player_mapping(nrplayers);
  players = new_players(totalplayers);

  if (!do_network_server(serv,levelstr,config,nrplayers,playermapping)) {
    del_player_mapping(playermapping);
    del_players(players);
    del_config(config);
    del_comm_server(serv);
    return;
  }
  free(levelstr);
  for (x = 0 ; playermapping[x] != -1; x++) {
    players[x]->is_local = TRUE;
  }
  for (x = 0 ; players[x] != NULL; x++) {
    players[x]->name = strdup(comm_server_get_name(serv,x));
  }
  game = comm_server_finalize(serv);
  start_game(game,level,config,players,playermapping,nrplayers);
}

int
do_network_client(Comm_client *client,Config *config,
                  int *playermapping, int localplayers) {
  int x,y;

  for (x=0; x < localplayers; x++) {
     y = comm_client_register_player(client,config_get_pname(config,x));
     if (y == -1) {
       WARN(_("Couldn't register more then %d player(s)"),x);
       break;
     } else if (y == -2) {
       WARN("Network error during player registration");
       return FALSE;
     }
     playermapping[x] = y;
  }
  INFO(_("Players registered to server"));
  if (!comm_client_register_done(client)) {
    WARN(_("Network error"));
    return FALSE;
  }

  if (!comm_client_wait_for_connections(client,config->hostname,4322)) {
    WARN(_("Network error"));
    return FALSE;
  }
  INFO(_("Waiting for player info"));
  if (!comm_client_get_player_info(client)) {
    WARN(_("Network error"));
    return FALSE;
  }
  INFO(_("Waiting for level info"));
  if (!comm_client_get_level(client)) {
    WARN(_("Network error"));
    return FALSE;
  }
  return TRUE;
}

void 
do_client_game(int nrplayers, char *servername, Config *config) {
  int *playermapping;
  _Player **players;
  Level *level;
  int x;
  Comm_client *client;
  Comm_game *game;

  INFO("multi machine client, local players: %d, server: %s"
      ,nrplayers,servername);

  client = new_comm_client(servername,PORT);
  if (client == NULL) {
    WARN(_("Connecting to  ffrenzy server failed"));
    return;
  }
  playermapping = new_player_mapping(nrplayers);

  if (!do_network_client(client,config,playermapping,nrplayers)) {
    del_player_mapping(playermapping);
    del_config(config);
    del_comm_client(client); 
    return;
  }
  level = new_level(comm_client_get_levelstr(client));
  if (level == NULL) {
    WARN(_("Level parsing failed"));
    del_player_mapping(playermapping);
    del_config(config);
    del_comm_client(client); 
    return;
  }
  players = new_players(comm_client_get_nrplayers(client));
  for (x = 0 ; playermapping[x] != -1; x++) {
    players[playermapping[x]]->is_local = TRUE;
  }
  for (x = 0 ; players[x] != NULL; x++) {
    set_players_name(players,x,(comm_client_get_name(client,x)));
  }
  game = comm_client_finalize(client);
  start_game(game,level,config,players,playermapping,nrplayers);
}

int
parse_number(char *number) {
  if (*number == '\0') return 0;
  if (*(number+1) == 'x') {
    /* hex */
    return strtol(number+2,(char **) NULL,16);
  } else {
    return atoi(number);
  }
}

FILE *
open_inputfile(char *file) {
  FILE *result;
  result = fopen(file,"r");
  if (result == NULL) {
    WARN(_("Opening output file failed: %s"),strerror(errno));
  }
  return result;
}

FILE *
open_outputfile(char *file) {
  FILE *result;
  result = fopen(file,"w");
  if (result == NULL) {
    WARN(_("Opening output file failed: %s"),strerror(errno));
  }
  return result;
}

int
main(int argc, char **argv) {
  int op;
  /* nr of players on this machine */
  int nrplayers = -1;
  /* total number of players playing */
  int totalplayers = -1;
  char *levelname = NULL;
  char *serverhost = NULL;
  char *outputfile = NULL;
  char *inputfile = NULL;
  Config *config;
  char *configfile = NULL;
  
  init_program();

  while ((op = getopt(argc,argv,"C:c:d:p:l:s:o:i:fmq")) > 0) {
    switch (op) {
      case 'C':
        configfile = optarg;
        break;
      case 'c':
        serverhost = optarg;
        break;
      case 'd':
        debugmask = parse_number(optarg);
        break;
      case 'p':
        nrplayers = atoi(optarg);
        break;
      case 'l':
        levelname = optarg;
        break;
      case 's':
        totalplayers = atoi(optarg);
        break;
      case 'o':
        outputfile = optarg;
        break;
      case 'i':
        inputfile = optarg;
        break;
      case 'f':
        do_fullscreen = TRUE;
        break;
      case 'm':
        do_music = TRUE;
        break;
      case 'q':
        do_sfx = FALSE;
        break;
      default:
        usage(argv[0],NULL);
        deinit_program(-1);
    }
  }
  if (argc - optind != 0) {
    usage(argv[0],"Too many arguments");
    deinit_program(-1);
  }

  if (inputfile != NULL) {
    input = open_inputfile(inputfile);
  }
  if (outputfile != NULL) {
    output = open_outputfile(outputfile);
  }
  config = new_config(configfile);
  if (serverhost == NULL && totalplayers == -1) {
    /* single machine game */
    if (nrplayers == -1) nrplayers = 2;
    do_single_game(nrplayers,levelname,config);
  } else if (totalplayers != -1 && serverhost == NULL) {
    /* multiplayer game, current is server */
    if (nrplayers == -1) nrplayers = 1;
    do_server_game(nrplayers,totalplayers,levelname,config);
  } else if (totalplayers == -1 && serverhost != NULL) {
    /* multiplayer game, current is client */
    if (nrplayers == -1) nrplayers = 1;
    do_client_game(nrplayers,serverhost,config);
  } else {
    /* invalid arguments */
    usage(argv[0],_("Can't be both server and client"));
    deinit_program(-1);
  }
  if (output != NULL) {
    fclose(output);
  }
  if (input != NULL) {
    fclose(input);
  }
  deinit_program(0);
  return 0;
}

