/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "ProcessingOptionsInfo.h"

#include <QApplication>
#include <QMutexLocker>
#include <QImage>

#include <KDebug>

#include <threadweaver/ThreadWeaver.h>

#include "JobPreview.h"
#include "PreviewInfo.h"
#include "ProcessingOptions.h"
#include "Icons.h"

struct ProcessingOptionsInfo::Private {
  QString name;
  ProcessingOptions processingOptions;
  QImage preview;
  QImage previewImage;
  JobPreview* jobPreview;
  QMutex mutex;
  bool previewUptodate;
  void updatePreview();
};

void ProcessingOptionsInfo::Private::updatePreview()
{
  if(previewUptodate) return;
  preview = previewImage.scaled(100,100, Qt::KeepAspectRatio);
  previewUptodate = true;
}

ProcessingOptionsInfo::ProcessingOptionsInfo() : d(new Private)
{
  d->preview = Icons::dcraw();
  d->previewUptodate = true;
}

ProcessingOptionsInfo::ProcessingOptionsInfo( const QString& _name, const ProcessingOptions& _options, const QImage& _preview) : d(new Private)
{
  d->name = _name;
  d->processingOptions = _options;
  d->preview = _preview.scaled ( 100, 100, Qt::KeepAspectRatio);
  d->jobPreview = 0;
  d->previewUptodate = true;
}

const QString& ProcessingOptionsInfo::name() const
{
  return d->name;
}

void ProcessingOptionsInfo::setName( const QString& _name)
{
  d->name = _name;
}

const ProcessingOptions& ProcessingOptionsInfo::processingOptions() const
{
  return d->processingOptions;
}

void ProcessingOptionsInfo::changeRawImageInfo(RawImageInfoSP _info ) const
{
  kDebug() << "changeRawImageInfo ================================================";
  {
    QMutexLocker l(&d->mutex);
    if(JobPreview::weaver()->dequeue( d->jobPreview ))
    {
      delete d->jobPreview;
      d->jobPreview = 0;
    }
  }
  while( d->jobPreview )
  {
    QApplication::processEvents();
  }
  QMutexLocker l(&d->mutex);
  d->jobPreview = new JobPreview( _info, 0 );
  connect( d->jobPreview, SIGNAL(imageFinished(const PreviewInfo*)), this, SLOT(setPreviewInfo(const  PreviewInfo* ) ) );
  connect( d->jobPreview, SIGNAL(done(ThreadWeaver::Job *)), this, SLOT(jobDone()));
  JobPreview::weaver()->enqueue( d->jobPreview );
}

const QImage& ProcessingOptionsInfo::preview() const
{
  d->updatePreview();
  return d->preview;
}

void ProcessingOptionsInfo::setPreviewInfo( const PreviewInfo* previewInfo )
{
  QMutexLocker l(&d->mutex);
  d->previewUptodate = false;
  d->previewImage = previewInfo->asQImage();
  emit previewChanged( );
}

void ProcessingOptionsInfo::jobDone()
{
  kDebug() << "Job done ================================================";
  QMutexLocker l(&d->mutex);
  delete d->jobPreview;
  d->jobPreview = 0;
}
