/*
 * jdapistd.c
 *
 * Copyright (C) 1994-1998, Thomas G. Lane.
 * Modifided for multibit by Bruce Barton of BITS Limited (shameless plug), 2009, (GPL).
 * This file is part of the Independent JPEG Group's software.
 * For conditions of distribution and use, see the accompanying README file.
 *
 * This file contains application interface code for the decompression half
 * of the JPEG library.  These are the "standard" API routines that are
 * used in the normal full-decompression case.  They are not used by a
 * transcoding-only application.  Note that if an application links in
 * jpeg_start_decompress, it will end up linking in the entire decompressor.
 * We thus must separate this file from jdapimin.c to avoid linking the
 * whole decompression library into a transcoder.
 */

#define JPEG_INTERNALS
#include "jinclude.h"
#include "jpeglib.h"


/* Forward declarations */
LOCAL(boolean) output_pass_setup JPP((j_decompress_ptr cinfo));


/*
 * Decompression initialization.
 * jpeg_read_header must be completed before calling this.
 *
 * If a multipass operating mode was selected, this will do all but the
 * last pass, and thus may take a great deal of time.
 *
 * Returns FALSE if suspended.  The return value need be inspected only if
 * a suspending data source is used.
 */

GLOBAL(boolean)
jpeg_start_decompress (j_decompress_ptr cinfo)
{
  /* Fix scaling here */
  if (cinfo->data_precision == cinfo->data_precision_other)
    cinfo->shft = 0;
  /* Warn now if scaled */
  if (cinfo->shft != 0) WARNMS2(cinfo, JWRN_SCALED,
    cinfo->data_precision, cinfo->data_precision_other);

  if (cinfo->global_state == DSTATE_READY) {
    /* First call: initialize master control, select active modules */
    jinit_master_decompress(cinfo);
    if (cinfo->buffered_image) {
      /* No more work here; expecting jpeg_start_output next */
      cinfo->global_state = DSTATE_BUFIMAGE;
      return TRUE;
    }
    cinfo->global_state = DSTATE_PRELOAD;
  }
  if (cinfo->global_state == DSTATE_PRELOAD) {
    /* If file has multiple scans, absorb them all into the coef buffer */
    if (cinfo->inputctl->has_multiple_scans) {
#ifdef D_MULTISCAN_FILES_SUPPORTED
      for (;;) {
	int retcode;
	/* Call progress monitor hook if present */
	if (cinfo->progress != NULL)
	  (*cinfo->progress->progress_monitor) ((j_common_ptr) cinfo);
	/* Absorb some more input */
	retcode = (*cinfo->inputctl->consume_input) (cinfo);
	if (retcode == JPEG_SUSPENDED)
	  return FALSE;
	if (retcode == JPEG_REACHED_EOI)
	  break;
	/* Advance progress counter if appropriate */
	if (cinfo->progress != NULL &&
	    (retcode == JPEG_ROW_COMPLETED || retcode == JPEG_REACHED_SOS)) {
	  if (++cinfo->progress->pass_counter >= cinfo->progress->pass_limit) {
	    /* jdmaster underestimated number of scans; ratchet up one scan */
	    cinfo->progress->pass_limit += (long) cinfo->total_iMCU_rows;
	  }
	}
      }
#else
      ERREXIT(cinfo, JERR_NOT_COMPILED);
#endif /* D_MULTISCAN_FILES_SUPPORTED */
    }
    cinfo->output_scan_number = cinfo->input_scan_number;
  } else if (cinfo->global_state != DSTATE_PRESCAN)
    ERREXIT1(cinfo, JERR_BAD_STATE, cinfo->global_state);
  /* Perform any dummy output passes, and set up for the final pass */
  return output_pass_setup(cinfo);
}


/*
 * Set up for an output pass, and perform any dummy pass(es) needed.
 * Common subroutine for jpeg_start_decompress and jpeg_start_output.
 * Entry: global_state = DSTATE_PRESCAN only if previously suspended.
 * Exit: If done, returns TRUE and sets global_state for proper output mode.
 *       If suspended, returns FALSE and sets global_state = DSTATE_PRESCAN.
 */

LOCAL(boolean)
output_pass_setup (j_decompress_ptr cinfo)
{
  if (cinfo->global_state != DSTATE_PRESCAN) {
    /* First call: do pass setup */
    (*cinfo->master->prepare_for_output_pass) (cinfo);
    cinfo->output_scanline = 0;
    cinfo->global_state = DSTATE_PRESCAN;
  }
  /* Loop over any required dummy passes */
  while (cinfo->master->is_dummy_pass) {
#ifdef QUANT_2PASS_SUPPORTED
    /* Crank through the dummy pass */
    while (cinfo->output_scanline < cinfo->output_height) {
      JDIMENSION last_scanline;
      /* Call progress monitor hook if present */
      if (cinfo->progress != NULL) {
	cinfo->progress->pass_counter = (long) cinfo->output_scanline;
	cinfo->progress->pass_limit = (long) cinfo->output_height;
	(*cinfo->progress->progress_monitor) ((j_common_ptr) cinfo);
      }
      /* Process some data */
      last_scanline = cinfo->output_scanline;
      (*cinfo->mainp->process_data) (cinfo, (JSAMPARRAY16) NULL,
				    &cinfo->output_scanline, (JDIMENSION) 0);
      if (cinfo->output_scanline == last_scanline)
	return FALSE;		/* No progress made, must suspend */
    }
    /* Finish up dummy pass, and set up for another one */
    (*cinfo->master->finish_output_pass) (cinfo);
    (*cinfo->master->prepare_for_output_pass) (cinfo);
    cinfo->output_scanline = 0;
#else
    ERREXIT(cinfo, JERR_NOT_COMPILED);
#endif /* QUANT_2PASS_SUPPORTED */
  }
  /* Ready for application to drive output pass through
   * jpeg_read_scanlines or jpeg_read_raw_data.
   */
  cinfo->global_state = cinfo->raw_data_out ? DSTATE_RAW_OK : DSTATE_SCANNING;
  return TRUE;
}


/*
 * Read some scanlines of data from the JPEG decompressor.
 *
 * The return value will be the number of lines actually read.
 * This may be less than the number requested in several cases,
 * including bottom of image, data source suspension, and operating
 * modes that emit multiple scanlines at a time.
 *
 * Note: we warn about excess calls to jpeg_read_scanlines() since
 * this likely signals an application programmer error.  However,
 * an oversize buffer (max_lines > scanlines remaining) is not an error.
 */

GLOBAL(JDIMENSION)
jpeg_read_scanlines (j_decompress_ptr cinfo, JSAMPARRAY scanlines,
		     JDIMENSION max_lines)
{
  JDIMENSION row_ctr;
  JSAMPARRAY16 scanlines16;
  JDIMENSION max_lines_buffer;
  register JDIMENSION col, width = 0;
  register JSAMPLE16 *temp_buf_ptr;
  register JSAMPLE16 *out_buf16_row_ptr;
  register JSAMPLE *out_buf_row_ptr;
  register int rshft;
  int line;

  if (cinfo->global_state != DSTATE_SCANNING)
    ERREXIT1(cinfo, JERR_BAD_STATE, cinfo->global_state);
  if (cinfo->output_scanline >= cinfo->output_height) {
    WARNMS(cinfo, JWRN_TOO_MUCH_DATA);
    return 0;
  }
  
  /* Deal with 8 bit outputs and scaling. */
  max_lines_buffer = max_lines;
  if ((cinfo->shft != 0)  || (cinfo->buffer_size_char)) {
    /* Create a data buffer <= DCTSIZE */
    if (max_lines_buffer > DCTSIZE) max_lines_buffer = DCTSIZE;
    width = cinfo->output_width * cinfo->output_components;
    scanlines16 = (JSAMPARRAY16)(cinfo->mem->alloc_sarray)
      ((j_common_ptr) cinfo, JPOOL_IMAGE_BUF, (JDIMENSION) width * 
      SIZEOF(JSAMPLE16), max_lines_buffer);
    
  } else { /* Short buffer, no scalling, just cast. */
    scanlines16 = (JSAMPARRAY16)scanlines;
  }

  /* Call progress monitor hook if present */
  if (cinfo->progress != NULL) {
    cinfo->progress->pass_counter = (long) cinfo->output_scanline;
    cinfo->progress->pass_limit = (long) cinfo->output_height;
    (*cinfo->progress->progress_monitor) ((j_common_ptr) cinfo);
  }

  /* Process some data */
  row_ctr = 0;
  (*cinfo->mainp->process_data) (cinfo, scanlines16, &row_ctr, max_lines_buffer);
  cinfo->output_scanline += row_ctr;
  
  /* Copy the buffer if needed. */
  rshft = cinfo->shft;
  if (cinfo->buffer_size_char) {
    for (line = 0; line < row_ctr; line++) {
      temp_buf_ptr = scanlines16[line];
      out_buf_row_ptr = scanlines[line];
      if (rshft == 0 ) { /* Faster */
        for (col = 0; col < width; col++) {
          *out_buf_row_ptr++ = *temp_buf_ptr++;
        }
      } else if (rshft > 0) { /* Shift data. */
        for (col = 0; col < width; col++) {
           *out_buf_row_ptr++ = (*temp_buf_ptr++ >> rshft);
        }    
      } else { /* Shift data the other way. */
        rshft = -rshft;
        for (col = 0; col < width; col++) {
           *out_buf_row_ptr++ = (*temp_buf_ptr++ << rshft);
        }
      }
    }
  } else if ( rshft != 0) { /* 16 bit shifted. */
   for (line = 0; line < row_ctr; line++) {
      temp_buf_ptr = scanlines16[line];
      out_buf16_row_ptr = (JSAMPLE16 *)scanlines[line];
      if (rshft > 0) { /* Shift data. */
        for (col = 0; col < width; col++) {
          *temp_buf_ptr++ = (*out_buf16_row_ptr++ >> rshft);
        }
      } else { /* Shift data the other way. */
        rshft = -rshft;
        for (col = 0; col < width; col++) {
          *temp_buf_ptr++ = (*out_buf16_row_ptr++ << rshft);
        }
      }
    }
  }
  
   /* Free the buffer if used. */
  if ( scanlines16 != (JSAMPARRAY16)scanlines ) {
    cinfo->mem->free_pool ((j_common_ptr)cinfo, JPOOL_IMAGE_BUF);
    scanlines16 = NULL;
  }
          
  return row_ctr;
}


/*
 * Alternate entry point to read raw data.
 * Processes exactly one iMCU row per call, unless suspended.
 */

GLOBAL(JDIMENSION)
jpeg_read_raw_data (j_decompress_ptr cinfo, JSAMPIMAGE data,
		    JDIMENSION max_lines)
{
  JDIMENSION lines_per_iMCU_row;
  JSAMPIMAGE16 data16;
  register JDIMENSION col, row;
  register int component;
  register JSAMPLE *out_buf_row_ptr;
  register JSAMPLE16 *in_buf_row_ptr;

  if (cinfo->global_state != DSTATE_RAW_OK)
    ERREXIT1(cinfo, JERR_BAD_STATE, cinfo->global_state);
  if (cinfo->output_scanline >= cinfo->output_height) {
    WARNMS(cinfo, JWRN_TOO_MUCH_DATA);
    return 0;
  }

  /* Call progress monitor hook if present */
  if (cinfo->progress != NULL) {
    cinfo->progress->pass_counter = (long) cinfo->output_scanline;
    cinfo->progress->pass_limit = (long) cinfo->output_height;
    (*cinfo->progress->progress_monitor) ((j_common_ptr) cinfo);
  }

  /* Verify that at least one iMCU row can be returned. */
  lines_per_iMCU_row = cinfo->max_v_samp_factor * cinfo->min_codec_data_unit;
  if (max_lines < lines_per_iMCU_row)
    ERREXIT(cinfo, JERR_BUFFER_SIZE);

  /* Check user's buffer size. */
  if (cinfo->buffer_size_char && (cinfo->data_precision_other <= 8)) { /* Defaults are true */
    /* Make a 16 bit image buffer */
    data16 = (JSAMPIMAGE16)(*cinfo->mem->alloc_small) ((j_common_ptr) cinfo, JPOOL_IMAGE_BUF,
           cinfo->output_components * SIZEOF(JSAMPARRAY16));
    for (component = 0; component < cinfo->output_components; component++) {
      data16[component] = (JSAMPARRAY16)(cinfo->mem->alloc_sarray)
      ((j_common_ptr) cinfo, JPOOL_IMAGE_BUF, (JDIMENSION) cinfo->output_width * 
      SIZEOF(JSAMPLE16), max_lines);
    }
  } else data16 = (JSAMPIMAGE16)data; /* Is a 16 bit buffer */

  /* Decompress directly into user's buffer. */
  if (! (*cinfo->codec->decompress_data) (cinfo, (JSAMPIMAGE16)data16)) {
    if (data16 != (JSAMPIMAGE16)data) {
      cinfo->mem->free_pool ((j_common_ptr)cinfo, JPOOL_IMAGE_BUF);
      data16 = NULL;
    }
    return 0;			/* suspension forced, can do nothing more */
  }

  /* OK, we processed one iMCU row. */
  /* If needed, copy the data. */
  if (data16 != (JSAMPIMAGE16)data) {
    for (component = 0; component < cinfo->output_components; component++) {
      for (row = 0; row < lines_per_iMCU_row; row++) {
        out_buf_row_ptr = data[component][row];
        in_buf_row_ptr = data16[component][row];        
        for (col = 0; col < cinfo->output_width; col++) {
          *out_buf_row_ptr++ = (JSAMPLE)(*in_buf_row_ptr++ & 0xFF);
        }
      }
    }
    cinfo->mem->free_pool ((j_common_ptr)cinfo, JPOOL_IMAGE_BUF);
    data16 = NULL;
  }
  cinfo->output_scanline += lines_per_iMCU_row;
  return lines_per_iMCU_row;
}


/* Additional entry points for buffered-image mode. */

#ifdef D_MULTISCAN_FILES_SUPPORTED

/*
 * Initialize for an output pass in buffered-image mode.
 */

GLOBAL(boolean)
jpeg_start_output (j_decompress_ptr cinfo, int scan_number)
{
  if (cinfo->global_state != DSTATE_BUFIMAGE &&
      cinfo->global_state != DSTATE_PRESCAN)
    ERREXIT1(cinfo, JERR_BAD_STATE, cinfo->global_state);
  /* Limit scan number to valid range */
  if (scan_number <= 0)
    scan_number = 1;
  if (cinfo->inputctl->eoi_reached &&
      scan_number > cinfo->input_scan_number)
    scan_number = cinfo->input_scan_number;
  cinfo->output_scan_number = scan_number;
  /* Perform any dummy output passes, and set up for the real pass */
  return output_pass_setup(cinfo);
}


/*
 * Finish up after an output pass in buffered-image mode.
 *
 * Returns FALSE if suspended.  The return value need be inspected only if
 * a suspending data source is used.
 */

GLOBAL(boolean)
jpeg_finish_output (j_decompress_ptr cinfo)
{
  if ((cinfo->global_state == DSTATE_SCANNING ||
       cinfo->global_state == DSTATE_RAW_OK) && cinfo->buffered_image) {
    /* Terminate this pass. */
    /* We do not require the whole pass to have been completed. */
    (*cinfo->master->finish_output_pass) (cinfo);
    cinfo->global_state = DSTATE_BUFPOST;
  } else if (cinfo->global_state != DSTATE_BUFPOST) {
    /* BUFPOST = repeat call after a suspension, anything else is error */
    ERREXIT1(cinfo, JERR_BAD_STATE, cinfo->global_state);
  }
  /* Read markers looking for SOS or EOI */
  while (cinfo->input_scan_number <= cinfo->output_scan_number &&
	 ! cinfo->inputctl->eoi_reached) {
    if ((*cinfo->inputctl->consume_input) (cinfo) == JPEG_SUSPENDED)
      return FALSE;		/* Suspend, come back later */
  }
  cinfo->global_state = DSTATE_BUFIMAGE;
  return TRUE;
}

#endif /* D_MULTISCAN_FILES_SUPPORTED */
