/*
 * This file is part of buteo-syncfw package
 *
 * Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: Sateesh Kavuri <sateesh.kavuri@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include "ClientPluginRunner.h"
#include "ClientThread.h"
#include "ClientPlugin.h"
#include "LogMacros.h"
#include "PluginManager.h"

using namespace Buteo;

// Maximum time in milliseconds for a plugin to finish sync
static const unsigned long long MAX_PLUGIN_SYNC_TIME = 1800000; //30 mins

ClientPluginRunner::ClientPluginRunner(const QString &aPluginName,
                                       SyncProfile *aProfile, PluginManager *aPluginMgr,
                                       PluginCbInterface *aPluginCbIf, QObject *aParent)
    :   PluginRunner(PLUGIN_CLIENT, aPluginName, aPluginMgr, aPluginCbIf, aParent),
        iProfile(aProfile),
        iPlugin(0),
        iThread(0)
{
    FUNCTION_CALL_TRACE(lcButeoTrace);
}

ClientPluginRunner::~ClientPluginRunner()
{
    FUNCTION_CALL_TRACE(lcButeoTrace);

    stop();
    disconnect();

    if (iPlugin != 0 && iPluginMgr != 0) {
        iPluginMgr->destroyClient(iPlugin);
        iPlugin = 0;
    }

    delete iThread;
    iThread = 0;
}

bool ClientPluginRunner::init()
{
    FUNCTION_CALL_TRACE(lcButeoTrace);

    if (iInitialized)
        return true;

    if (iPluginMgr == 0 || iPluginCbIf == 0 || iProfile == 0) {
        qCWarning(lcButeoMsyncd) << "Invalid members, failed to initialize";
        return false;
    }

    iPlugin = iPluginMgr->createClient(iPluginName, *iProfile, iPluginCbIf);
    if (iPlugin == 0) {
        qCWarning(lcButeoMsyncd) << "Failed to create client plug-in:" << iPluginName;
        return false;
    }

    iThread = new ClientThread();

    // Pass connectivity state change signal to the plug-in.
    connect(this, SIGNAL(connectivityStateChanged(Sync::ConnectivityType, bool)),
            iPlugin, SLOT(connectivityStateChanged(Sync::ConnectivityType, bool)));

    // Connect signals from the plug-in.

    connect(iPlugin, SIGNAL(transferProgress(const QString &, Sync::TransferDatabase, Sync::TransferType, const QString &,
                                             int)),
            this, SLOT(onTransferProgress(const QString &, Sync::TransferDatabase, Sync::TransferType, const QString &, int)));

    connect(iPlugin, &ClientPlugin::error, this, &ClientPluginRunner::onError);

    connect(iPlugin, SIGNAL(success(const QString &, const QString &)),
            this, SLOT(onSuccess(const QString &, const QString &)));

    connect(iPlugin, SIGNAL(accquiredStorage(const QString &)),
            this, SLOT(onStorageAccquired(const QString &)));

    connect(iPlugin, SIGNAL(syncProgressDetail(const QString &, int)),
            this, SLOT(onSyncProgressDetail(const QString &, int)));

    // Connect signals from the thread.
    connect(iThread, &ClientThread::initError, this, &ClientPluginRunner::onError);

    connect(iThread, SIGNAL(finished()), this, SLOT(onThreadExit()));

    iInitialized = true;

    return true;
}

bool ClientPluginRunner::start()
{
    FUNCTION_CALL_TRACE(lcButeoTrace);

    bool rv = false;
    if (iInitialized && iThread != 0) {
        // Set a timer after which the sync session should stop
        QTimer::singleShot(MAX_PLUGIN_SYNC_TIME, this, SLOT(pluginTimeout()));
        rv = iThread->startThread(iPlugin);
        qCDebug(lcButeoMsyncd) << "ClientPluginRunner started thread for plugin:" << iPlugin->getProfileName() << ", returning:" << rv;
    }

    return rv;
}

void ClientPluginRunner::stop()
{
    FUNCTION_CALL_TRACE(lcButeoTrace);

    if (iThread != 0) {
        iThread->stopThread();
        iThread->wait();
    }
}

void ClientPluginRunner::abort(Sync::SyncStatus aStatus)
{
    FUNCTION_CALL_TRACE(lcButeoTrace);

    if (iPlugin != 0) {
        iPlugin->abortSync(aStatus);
    }
}

SyncPluginBase *ClientPluginRunner::plugin()
{
    FUNCTION_CALL_TRACE(lcButeoTrace);

    return iPlugin;
}

SyncResults ClientPluginRunner::syncResults()
{
    FUNCTION_CALL_TRACE(lcButeoTrace);

    if (iPlugin != 0) {
        return iPlugin->getSyncResults();
    } else {
        return SyncResults();
    }
}

bool ClientPluginRunner::cleanUp()
{
    FUNCTION_CALL_TRACE(lcButeoTrace);

    bool retval = false;
    if (iPlugin != 0) {
        retval = iPlugin->cleanUp();
    }
    return retval;
}

void ClientPluginRunner::onTransferProgress(const QString &aProfileName,
                                            Sync::TransferDatabase aDatabase, Sync::TransferType aType,
                                            const QString &aMimeType, int aCommittedItems)
{
    FUNCTION_CALL_TRACE(lcButeoTrace);

    emit transferProgress(aProfileName, aDatabase, aType, aMimeType, aCommittedItems);
}

void ClientPluginRunner::onError(const QString &aProfileName,
                                 const QString &aMessage, SyncResults::MinorCode aErrorCode)
{
    FUNCTION_CALL_TRACE(lcButeoTrace);

    emit error(aProfileName, aMessage, aErrorCode);
    stop();
}

void ClientPluginRunner::onSuccess(const QString &aProfileName,
                                   const QString &aMessage)
{
    FUNCTION_CALL_TRACE(lcButeoTrace);

    emit success(aProfileName, aMessage);
    stop();

}

void ClientPluginRunner::onStorageAccquired(const QString &aMimeType)
{
    FUNCTION_CALL_TRACE(lcButeoTrace);

    emit storageAccquired(aMimeType);
}

void ClientPluginRunner::onSyncProgressDetail(const QString &aProfileName, int aProgressDetail)
{
    FUNCTION_CALL_TRACE(lcButeoTrace);

    emit syncProgressDetail(aProfileName, aProgressDetail);
}


void ClientPluginRunner::onThreadExit()
{
    FUNCTION_CALL_TRACE(lcButeoTrace);

    emit done();
}

void ClientPluginRunner::pluginTimeout()
{
    FUNCTION_CALL_TRACE(lcButeoTrace);

    emit error(iProfile->name(), "Plugin timeout occurred", SyncResults::PLUGIN_TIMEOUT);
    stop();
}
