//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Sample/HardParticle/Spheroid.cpp
//! @brief     Implements class Spheroid.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "Sample/HardParticle/Spheroid.h"
#include "Base/Math/Functions.h"
#include "Base/Util/Assert.h"
#include "Sample/HardParticle/TruncatedSpheroid.h"
#include "Sample/Shapes/TruncatedEllipsoidNet.h"
#include <limits>
#include <numbers>
using std::numbers::pi;

Spheroid::Spheroid(const std::vector<double> P)
    : IFormFactor(P)
    , m_radius(m_P[0])
    , m_height(m_P[1])
{
    validateOrThrow();
    m_shape3D =
        std::make_unique<TruncatedEllipsoidNet>(m_radius, m_radius, m_height / 2.0, m_height, 0.0);
}

Spheroid::Spheroid(double radius, double height)
    : Spheroid(std::vector<double>{radius, height})
{
}

complex_t Spheroid::formfactor(C3 q) const
{
    ASSERT(m_validated);
    double h = m_height / 2;
    double R = m_radius;

    // complex length of q (not a sesquilinear dot product!),
    // xy components multiplied with R, z component multiplied with h
    complex_t qR = sqrt(R * R * (q.x() * q.x() + q.y() * q.y()) + h * h * q.z() * q.z());

    complex_t zFactor = exp_I(h * q.z());

    if (std::abs(qR) < 1e-4)
        // expand sin(qR)-qR*cos(qR) up to qR^5
        return 4 * pi / 3 * R * R * h * (1. - 0.1 * pow(qR, 2)) * zFactor;

    return 4 * pi / pow(qR, 3) * R * R * h * (sin(qR) - qR * cos(qR)) * zFactor;
}

std::string Spheroid::validate() const
{
    std::vector<std::string> errs;
    requestGt0(errs, m_radius, "radius");
    requestGt0(errs, m_height, "height");
    if (!errs.empty())
        return jointError(errs);
    m_validated = true;
    return "";
}
