// ---------------------------------------------------------------------------
// - TlsState.cpp                                                            -
// - afnix:tls service - tls state class implementation                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2015 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Byte.hpp"
#include "Vector.hpp"
#include "Boolean.hpp"
#include "TlsState.hpp"
#include "TlsTypes.hxx"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // the state name and info
  static const String PN_STA_SFLG = "TLS-STA-SFLG";
  static const String PI_STA_SFLG = "TLS STATE SERVER FLAG";
  static const String PN_STA_VMAJ = "TLS-STA-VMAJ";
  static const String PI_STA_VMAJ = "TLS STATE MAJOR VERSION";
  static const String PN_STA_VMIN = "TLS-STA-VMIN";
  static const String PI_STA_VMIN = "TLS STATE MINOR VERSION";

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a default tls state

  TlsState::TlsState (void) {
    reset ();
  }

  // create a tls state by flag

  TlsState::TlsState (const bool sflg) {
    reset ();
    d_sflg = sflg;
  }

  // create a tls state by flag and parameters

  TlsState::TlsState (const bool sflg, const TlsParams& prms) {
    reset ();
    d_sflg = sflg;
    d_vmaj = prms.getvmaj ();
    d_vmin = prms.getvmin ();
  }

  // copy construct this tls state

  TlsState::TlsState (const TlsState& that) {
    that.rdlock ();
    try {
      d_sflg = that.d_sflg;
      d_vmaj = that.d_vmaj;
      d_vmin = that.d_vmin;
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }

  // assign a tls parameters to this one

  TlsState& TlsState::operator = (const TlsState& that) {
    // check for self assignation
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.rdlock ();
    try {
      d_sflg = that.d_sflg;
      d_vmaj = that.d_vmaj;
      d_vmin = that.d_vmin;
      unlock ();
      that.unlock ();
      return *this;
    } catch (...) {
      unlock ();
      that.unlock ();
      throw;
    }
  }

  // return the class name
  
  String TlsState::repr (void) const {
    return "TlsState";
  }

  // reset the tls state object

  void TlsState::reset (void) {
    wrlock ();
    try {
      d_vmaj = TLS_VMAJ_DEF;
      d_vmin = TLS_VMIN_DEF;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the connection info list

  Plist TlsState::getinfo (void) const {
    rdlock ();
    try {
      // create a result plist
      Plist plst;
      // set the result plist
      plst.add (PN_STA_SFLG, PI_STA_SFLG, d_sflg);
      plst.add (PN_STA_VMAJ, PI_STA_VMAJ, (t_long) d_vmaj);
      plst.add (PN_STA_VMIN, PI_STA_VMIN, (t_long) d_vmin);
      unlock ();
      return plst;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the server flag

  bool TlsState::getsflg (void) const {
    rdlock ();
    try {
      bool result = d_sflg;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the tls major version

  t_byte TlsState::getvmaj (void) const {
    rdlock ();
    try {
      t_byte result = d_vmaj;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the tls minor version

  t_byte TlsState::getvmin (void) const {
    rdlock ();
    try {
      t_byte result = d_vmin;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set the tls version at once

  void TlsState::setvers (const t_byte vmaj, const t_byte vmin) {
    wrlock ();
    try {
      // check version
      if (tls_vers_isvalid (vmaj, vmin) == false) {
	String vers = tls_vers_tostring (vmaj, vmin);
	throw Exception ("tls-error", "invalid tls version", vers);
      }
      // update major version
      if (vmaj < d_vmaj) {
	d_vmaj = vmaj;
	d_vmin = vmin;
      } else {
	if (vmin < d_vmin) d_vmin = vmin;
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set the selected cipher code

  void TlsState::setcifr (const t_word cifr) {
    wrlock ();
    try {
      d_cifr = cifr;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the selected cipher

  t_word TlsState::getcifr (void) const {
    rdlock ();
    try {
      t_word result = d_cifr;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 3;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_SERVP   = zone.intern ("server-p");
  static const long QUARK_GETVMAJ = zone.intern ("get-major-version");
  static const long QUARK_GETVMIN = zone.intern ("get-minor-version");

  // create a new object in a generic way

  Object* TlsState::mknew (Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 0 argument
    if (argc == 0) return new TlsState;
    // check for 1 argument
    if (argc == 1) {
      bool sflg = argv->getbool (0);
      return new TlsState (sflg);
    }
    // too many arguments
    throw Exception ("argument-error", 
                     "too many argument with tls state constructor");
  }

  // return true if the given quark is defined

  bool TlsState::isquark (const long quark, const bool hflg) const {
    rdlock ();
    try {
      if (zone.exists (quark) == true) {
	unlock ();
	return true;
      }
      bool result = hflg ? TlsInfos::isquark (quark, hflg) : false;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // apply this object with a set of arguments and a quark
  
  Object* TlsState::apply (Runnable* robj, Nameset* nset, const long quark,
			   Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // dispatch 0 arguments
    if (argc == 0) {
      if (quark == QUARK_SERVP)   return new Boolean (getsflg ());
      if (quark == QUARK_GETVMAJ) return new Byte (getvmaj ());
      if (quark == QUARK_GETVMIN) return new Byte (getvmin ());
    }
    // call the info method
    return TlsInfos::apply (robj, nset, quark, argv);
  }
}
