/*###############################################################################
# Linux Management Providers (LMP), BIOS provider package
# Copyright (C) 2010 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr>
#
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
#
# Modified by <Author(s)>, <Affiliation>, <Year>
###############################################################################*/

#include <sys/io.h>
#include "BIOS_Common.h"

#if defined(__linux__)||defined(__FreeBSD__)||defined(__NetBSD__)||defined(__CYGWIN32__)||defined(__MINGW32__)
static __inline__ void outportb(uint16_t port,uint8_t value)
{
	__asm__ volatile ("outb %0,%1"::"a" ((char) value), "d"((uint16_t) port));
}

static __inline__ uint8_t inportb(uint16_t port)
{
	uint8_t _v;
	__asm__ volatile ("inb %1,%0":"=a" (_v):"d"((uint16_t) port));
	return _v;
}
#endif

int set_permissions(string& errorMessage)
{
#if defined(__linux__)
	if (ioperm(PORT_CMOS_0,4*2,IO_READ))
	{
		errorMessage="Need to be run as root to access the Cmos.";
		return FAILED;
	}
#else
	errorMessage="Linux only";
	return FAILED;
#endif
}

int unset_permissions(string& errorMessage)
{
#ifdef __linux__
	ioperm(PORT_CMOS_0,4*2,0);
	return OK;
#else
	errorMessage="Linux only";
	return FAILED;
#endif
}

uint8_t read_cmos(const unsigned int cell)
{
	if(cell<128)
	{
		outportb(PORT_CMOS_0,cell);
		return inportb(PORT_CMOS_1);
	}
	if(cell<2*128)
	{
		outportb(PORT_CMOS_2,cell);
		return inportb(PORT_CMOS_3);
	}
#if !defined(WIN32) || defined(__MINGW32__)
	if(cell<3*128)
	{
		outportb(PORT_CMOS_4,cell);
		return inportb(PORT_CMOS_5);
	}
	if(cell<4*128)
	{
		outportb(PORT_CMOS_6,cell);
		return inportb(PORT_CMOS_7);
	}
#endif
	return 0;
}

int CF_loadCMOS(uint8_t *cmos, const int cmos_size, string& errorMessage)
{
	CF_assert(set_permissions(errorMessage));
	
	for (unsigned int i=0;i<cmos_size;i++)
		cmos[i]=read_cmos(i);
		
	CF_assert(unset_permissions(errorMessage));
	
	return OK;
}

int SMBIOS_getBIOSInformation(vector<bios_information>& bios,vector< vector<string> >& dmi_strings,string& errorMessage)
{
	vector<void*> _bios;
	void *start_ptr;
	char *tableAddress;
	_smbios_entry_point *anchor;
	
	if((start_ptr=SMBIOS_getRawData(_SMBIOS_ANCHOR_SEARCH_OFFSET,_SMBIOS_ANCHOR_SEARCH_RANGE,errorMessage))==NULL)
		return FAILED;
	
	anchor = SMBIOS_getEntryPoint(start_ptr);
	
	if((tableAddress=(char*)SMBIOS_getRawData(anchor->structure_table_address,anchor->structure_table_length,errorMessage))==NULL)
	{
		free(start_ptr);
		return FAILED;
	}
	
	SMBIOS_getStructure(_bios,tableAddress,anchor-> number_of_smbios_structure,0);
	
	for(unsigned int i=0;i<_bios.size();i++)
	{
		vector<string> _dmi_strings;
		vector<int> indexes;
		int index_max;
		
		bios.push_back(*(bios_information*)_bios[i]);
		
		indexes.push_back((int)bios[i].vendor);
		indexes.push_back((int)bios[i].bios_version);
		indexes.push_back((int)bios[i].bios_release_date);
		
		index_max=*max_element(indexes.begin(), indexes.end());
		
		for(unsigned int j=0;j<=index_max;j++)
			_dmi_strings.push_back(SMBIOS_getDmiString((_smbios_structure_header*)_bios[i],j));
		
		dmi_strings.push_back(_dmi_strings);
	}
	
	free(tableAddress);
	free(start_ptr);
	
	return OK;
}

// FIXME: Need to be improved
int CF_getBIOSManufacturer(const string& vendor)
{
	string vendor_touppper=vendor;
	transform(vendor_touppper.begin(), vendor_touppper.end(),vendor_touppper.begin(),::toupper);
	
	if (vendor_touppper.find("ACER")!=string::npos) return ACER;                      // ACER
	else if (vendor_touppper.find("AMI")!=string::npos) return AMI;                   // AMI
	else if (vendor_touppper.find("AWARD")!=string::npos) return AWARD;               // AWARD
	else if (vendor_touppper.find("COMPAQ")!=string::npos) return COMPAQ;             // COMPAQ
	else if (vendor_touppper.find("DTK")!=string::npos) return DTK;                   // DTK
	else if (vendor_touppper.find("IBM")!=string::npos) return IBM;                   // IBM
	else if (vendor_touppper.find("PACKARD BELL")!=string::npos) return PACKARD_BELL; // PACKARD_BELL
	else if (vendor_touppper.find("PHOENIX")!=string::npos) return PHOENIX;           // PHOENIX
	else if (vendor_touppper.find("GATEWAY")!=string::npos) return GATEWAY_PHOENIX;   // GATEWAY_PHOENIX
	else if (vendor_touppper.find("SAMSUNG")!=string::npos) return SAMSUNG;           // SAMSUNG
	else if (vendor_touppper.find("SONY")!=string::npos) return SONY;                 // SONY
	else if (vendor_touppper.find("TOSHIBA")!=string::npos) return TOSHIBA;           // TOSHIBA
	else if (vendor_touppper.find("ZENITH")!=string::npos) return ZENITH_AMI;         // ZENITH_AMI
	else return DEFAULT;                                                              // DEFAULT
}

